// Includes
#include <stdio.h>
#include <string.h>
#include <gccore.h>

#include "sysCheckGX/SysTitle.h"
#include "sysCheckGX/Tools.h"
#include "sysCheckGX/NAND.h"

// Macros

// Get upper or lower half of a title ID
#define TITLE_ID(titleId) ((u32)((titleId) >> 32))

// Turn upper and lower into a full title ID
#define FULL_TITLE_ID(titleId) ((u32)(titleId))

int SysTitle::GetTMD(u64 titleId, tmd **ptmd)
{
	int 		ret = 0;
	signed_blob *stmd = NULL;

	ret = GetTMD(titleId, &stmd);

	if (ret < 0)
		goto end;
	
	if (!stmd)
	{
		ret = -1;

		goto end;
	}

	*ptmd = (tmd*)AllocateMemory(SIGNED_TMD_SIZE(stmd));
	
	memcpy(*ptmd, SIGNATURE_PAYLOAD(stmd), SIGNED_TMD_SIZE(stmd));

end:
	delete stmd;
	
	stmd = NULL;

	return ret;
}

int SysTitle::GetTMD(u64 titleId, signed_blob **stmd)
{
	*stmd = NULL;	
	
	u32 size;

	// Get TMD size
	int ret = ES_GetStoredTMDSize(titleId, &size);

	// If we get an error try getting it from the NAND directly
	if (ret < 0)
	{
		printf("\n>> ES_GetTMDSize Failed\n");

		// Attempting to get TMD from NAND
		char fileName[256];

		sprintf(fileName, "/title/%08x/%08x/content/title.tmd", TITLE_ID(titleId), FULL_TITLE_ID(titleId));		
		ret = Nand::Read(fileName, (u8**)stmd);

		memset(fileName, 0, 256);
	}
	else 
	{
		// Allocate memory
		*stmd = (signed_blob*)AllocateMemory(size);

		if (!*stmd)
			return -1;

		// Read TMD
		ret = ES_GetStoredTMD(titleId, *stmd, size);

		if (ret < 0)
		{
			delete stmd;

			stmd = NULL;
		}

		ret = (int)size;
	}

	return ret;
}

int SysTitle::GetTMDView(u64 titleId, TmdView **tmdv)
{
	*tmdv = NULL;
	
	u32 size = 0;
	
	int ret = ES_GetTMDViewSize(titleId, &size);

	if (ret < 0)
		return ret;

	*tmdv = (TmdView*)AllocateMemory(size);

	if (!*tmdv)
		return -1;

	ret = ES_GetTMDView(titleId, (u8*)*tmdv, size);

	if (ret < 0)
	{
		delete *tmdv;
		
		*tmdv = NULL;
	}

	return ret;
}

u16 SysTitle::GetVersion32(u32 titleId)
{
	u16 result;

	GetVersion(0x0000000100000000ULL | titleId, &result);

	return result;
}

int SysTitle::GetVersion(u64 titleId, u16 *version)
{
	*version = 0;
	
	TmdView *view = NULL;

	int ret = GetTMDView(titleId, &view);

	if (ret < 0)
		goto end;

	*version = view->TitleVersion;

end:
	delete view;
	
	view = NULL;
	
	return ret;
}

string SysTitle::GetBootFilename(u64 titleId)
{
	int    ret = 0;
	tmd    *ptmd = NULL;
	string fileName;

	ret = GetTMD(titleId, &ptmd);

	if (ret < 0 || ptmd == NULL)
		goto end;

	for (u32 i = 0; i < ptmd->num_contents; i++)
	{
		if (ptmd->contents[i].index == ptmd->boot_index)
		{
			char tmp[256];

			sprintf(tmp, "/title/%08x/%08x/content/%08x.app" , TITLE_ID(titleId), FULL_TITLE_ID(titleId), ptmd->contents[i].cid);
			fileName = tmp;
			
			memset(tmp, 0, sizeof(tmp));

			break;
		}
	}

end:
	delete ptmd;
	
	ptmd = NULL;
	
	return fileName;
}
