// Includes
#include <string.h>
#include <gccore.h>
#include <malloc.h>

#include "sysCheckGX/Check.h"
#include "sysCheckGX/Identify.h"
#include "sysCheckGX/Tools.h"
#include "sysCheckGX/NAND.h"
#include "sysCheckGX/SysTitle.h"

#include "Ticket_dat.h"
#include "Tmd_dat.h"

// Macros

// Turn upper and lower into a full title ID
#define FULL_TITLE_ID(titleId1, titleId2) (((u64)(titleId1) << 32) | (titleId2))

// Variables
static u8 certs[0xA00] ATTRIBUTE_ALIGN(32); // Certificates

bool GetCertificates()
{
	// Initializing the Internal Storage File System (ISFS)
	s32 ret = ISFS_Initialize();

	if (ret < 0)
		return false;

	int fd = ISFS_Open("/sys/cert.sys", ISFS_OPEN_READ);

	if (fd < 0)
		return false;

	if (ISFS_Read(fd, certs, sizeof(certs)) != sizeof(certs))
		return false;

	ISFS_Close(fd);

	return true;
}

// Get the system menu version from TMD
u32 GetSysMenuVersion()
{
	static u64 TitleID ATTRIBUTE_ALIGN(32) = 0x0000000100000002LL;
	static u32 tmdSize ATTRIBUTE_ALIGN(32);

	// Get the stored TMD size for the system menu
	if (ES_GetStoredTMDSize(TitleID, &tmdSize) < 0)
		return false;

	signed_blob *TMD = (signed_blob *)memalign(32, (tmdSize + 32)&(~31));
	memset(TMD, 0, tmdSize);

	// Get the stored TMD for the system menu
	if (ES_GetStoredTMD(TitleID, TMD, tmdSize) < 0)
		return false;

	// Get the system menu version from TMD
	tmd *rTMD = (tmd *)(TMD + (0x140 / sizeof(tmd *)));
	u32 version = rTMD->title_version;

	free(TMD);

	// Return the system menu version
	return version;
}

// Get the MIOS version from TMD
s32 GetMIOSVersion()
{
	static u64 TitleID ATTRIBUTE_ALIGN(32) = 0x0000000100000101LL;
	static u32 tmdSize ATTRIBUTE_ALIGN(32);

	// Get the stored TMD size for the MIOS
	if (ES_GetStoredTMDSize(TitleID, &tmdSize) < 0)
		return false;

	signed_blob *TMD = (signed_blob *)memalign(32, (tmdSize + 32)&(~31));
	memset(TMD, 0, tmdSize);

	// Get the stored TMD for the MIOS
	if (ES_GetStoredTMD(TitleID, TMD, tmdSize) < 0)
		return false;

	// Get the MIOS version from TMD
	tmd *rTMD = (tmd *)(TMD + (0x140 / sizeof(tmd *)));
	u32 version = rTMD->title_version;

	free(TMD);

	// Return the MIOS version
	return version;
}

// Get the BC version from TMD
s32 GetBCVersion()
{
	static u64 TitleID ATTRIBUTE_ALIGN(32) = 0x0000000100000100LL;
	static u32 tmdSize ATTRIBUTE_ALIGN(32);

	// Get the stored TMD size for the BC
	if (ES_GetStoredTMDSize(TitleID, &tmdSize) < 0)
		return false;

	signed_blob *TMD = (signed_blob *)memalign(32, (tmdSize + 32)&(~31));
	memset(TMD, 0, tmdSize);

	// Get the stored TMD for the BC
	if (ES_GetStoredTMD(TitleID, TMD, tmdSize) < 0)
		return false;

	// Get the BC version from TMD
	tmd *rTMD = (tmd *)(TMD + (0x140 / sizeof(tmd *)));
	u32 version = rTMD->title_version;

	free(TMD);

	// Return the BC version
	return version;
}

// Get the boot2 version
u32 GetBoot2Version()
{
	u32 boot2version;

	if (ES_GetBoot2Version(&boot2version) < 0)
		boot2version = 0;

	return boot2version;
}

// Get the console ID
u32 GetConsoleID()
{
	u32 deviceId;

	if (ES_GetDeviceID(&deviceId) < 0)
		deviceId = 0;
	
	return deviceId;
}

// Remove Bogus Ticket
int RemoveBogusTicket(void)
{
	tikview	*viewdata = NULL;
	u64 	titleId = 0x100000000LL;
	u32 	cnt, views;
	s32 	ret;

	// Get number of ticket views
	ret = ES_GetNumTicketViews(titleId, &views);

	if (ret < 0)
		return ret;

	if (!views)
		return 1;
	else
	{
		if (views > 16)
			return -1;
	}
	
	// Get ticket views
	viewdata = (tikview*)memalign(32, sizeof(tikview) * views);
	ret = ES_GetTicketViews(titleId, viewdata, views);

	if (ret < 0)
		return ret;

	// Remove tickets
	for (cnt = 0; cnt < views; cnt++)
	{
		ret = ES_DeleteTicket(&viewdata[cnt]);

		if (ret < 0)
			return ret;
	}

	return ret;
}

// Check fake signatures (aka Trucha Bug)
bool CheckFakeSignature(void)
{
	int ret = ES_AddTicket((signed_blob *)Ticket_dat, Ticket_dat_size, (signed_blob *)certs, sizeof(certs), 0, 0);

	if (ret > -1)
		RemoveBogusTicket();

	if (ret > -1 || ret == -1028)
		return true;

	return false;
}

bool CheckESIdentify(void)
{
	int ret = Identify::AsSuperUser();
	
	return (ret > -1);
}

// Check flash access (NAND flash memory)
bool CheckFlashAccess(void)
{
	int ret = IOS_Open("/dev/flash", 1);

	IOS_Close(ret);

	return (ret >= 0);
}

// Check NAND access
bool CheckNANDAccess(void)
{
	int ret = IOS_Open("/title/00000001/00000002/content/title.tmd", 1);

	IOS_Close(ret);

	return (ret >= 0);
}

// Check boot2 access
bool CheckBoot2Access(void)
{
	int ret = IOS_Open("/dev/boot2", 1);

	IOS_Close(ret);

	return (ret >= 0);
}

// Check USB 2.0 module
bool CheckUSB2(u32 titleID)
{
	// Nintendo's IOS58 and Hermes' IOS supports USB 2.0 module
	switch (titleID)
	{
		case 58:
			return true;
			break;

		case 202:
			return true;
			break;

		case 222:
			return true;
			break;

		case 223:
			return true;
			break;

		case 224:
			return true;
			break;
	}

	// Open USB 2.0 module
	int ret = IOS_Open("/dev/usb2", 1);

	// If fail, try old USB 2.0 module (EHCI)
	if (ret < 0)
		ret = IOS_Open("/dev/usb/ehc", 1);

	// If fail, try USB 2.0 module from cIOS Hermes v5.1
	if (ret < 0)
		ret = IOS_Open("/dev/usb123", 1);

	IOS_Close(ret);

	return (ret >= 0);
}

// Check if this is an IOS stub (according to WiiBrew.org)
bool IsKnownStub(u32 noIOS, s32 noRevision)
{
	if (noIOS == 3 && noRevision == 65280)
		return true;
		
	if (noIOS == 4 && noRevision == 65280)
		return true;

	if (noIOS == 10 && noRevision == 768)
		return true;

	if (noIOS == 11 && noRevision == 256)
		return true;

	if (noIOS == 16 && noRevision == 512)
		return true;

	if (noIOS == 20 && noRevision == 256)
		return true;

	if (noIOS == 30 && noRevision == 2816)
		return true;

	if (noIOS == 40 && noRevision == 3072)
		return true;
		
	if (noIOS == 50 && noRevision == 5120)
		return true;

	if (noIOS == 51 && noRevision == 4864)
		return true;

	if (noIOS == 52 && noRevision == 5888)
		return true;

	if (noIOS == 60 && noRevision == 6400)
		return true;

	if (noIOS == 70 && noRevision == 6912)
		return true;

	if (noIOS == 222 && noRevision == 65280)
		return true;

	if (noIOS == 223 && noRevision == 65280)
		return true;

	if (noIOS == 249 && noRevision == 65280)
		return true;

	if (noIOS == 250 && noRevision == 65280)
		return true;

	if (noIOS == 254 && noRevision == 2)
		return true;

	if (noIOS == 254 && noRevision == 3)
		return true;

	if (noIOS == 254 && noRevision == 260)
		return true;

	if (noIOS == 254 && noRevision == 65280)
		return true;

	return false;
}

// Check if Priiloader is installed
bool IsPriiloaderInstalled()
{
	bool 		retValue = false;
	int 		size = 0;
	u8 			*buffer = NULL;
	const char	*checkStr = "priiloader";
	string 		bootfile = SysTitle::GetBootFilename(FULL_TITLE_ID(1, 2));

	if (bootfile.size() == 0)
		goto end;

	size = Nand::Read(bootfile.c_str(), &buffer);

	if (size < 0)
		goto end;
	
	for (u32 i = 0; i < size - strlen(checkStr); i++)
	{
		if (!strncmp((char*)buffer + i, checkStr, strlen(checkStr)))		
		{
			retValue = true;
			
			break;
		}
	}
	
end:
	bootfile.clear();

	delete buffer;

	buffer = NULL;

	return retValue;
}
