/////////////////////////////////////////////////////////
// Copyright (C) 2009
// by Dimok
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications and to alter it and
// redistribute it freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you
//    must not claim that you wrote the original software. If you use
//    this software in a product, an acknowledgment in the product
//    documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such and
//    must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
///////////////////////////////////////////////////////

// Includes
#include "Text/TextEditor.h"
#include "Main.h"

TextEditor::TextEditor(char *inText, int lines, char *fileName)
{
	focus = true; // Allow focus
	triggerDisabled = false;
	linesToDraw = lines;
	currentLine = 0;

	// Sounds
	btnSoundOver = new GuiSound(ButtonOver_pcm, ButtonOver_pcm_size, SOUND_PCM, cfg.SFXVolume);
	btnSoundClick = new GuiSound(ButtonClick_pcm, ButtonClick_pcm_size, SOUND_PCM, cfg.SFXVolume);

	// Triggers
	triggerA = new GuiTrigger;
	triggerA->SetSimpleTrigger(-1, WPAD_BUTTON_A | WPAD_CLASSIC_BUTTON_A, PAD_BUTTON_A);

	triggerHeldA = new GuiTrigger;
	triggerHeldA->SetHeldTrigger(-1, WPAD_BUTTON_A | WPAD_CLASSIC_BUTTON_A, PAD_BUTTON_A);

	triggerB = new GuiTrigger;
	triggerB->SetButtonOnlyTrigger(-1, WPAD_BUTTON_B | WPAD_CLASSIC_BUTTON_B, PAD_BUTTON_B);

	triggerHome = new GuiTrigger;
	triggerHome->SetButtonOnlyTrigger(-1, WPAD_BUTTON_HOME | WPAD_CLASSIC_BUTTON_HOME, 0);
	
	// Objects	
	bgTexteditorData = new GuiImageData(TextReaderBox_png);
	bgTexteditorImg = new GuiImage(bgTexteditorData);

	arrowDown = new GuiImageData(ScrollbarArrowDown_png);
	arrowDownImg = new GuiImage(arrowDown);
	
	arrowDownOver = new GuiImageData(ScrollbarArrowDownOver_png);
	arrowDownOverImg = new GuiImage(arrowDownOver);
	
	arrowUp = new GuiImageData(ScrollbarArrowUp_png);
	arrowUpImg = new GuiImage(arrowUp);
	
	arrowUpOver = new GuiImageData(ScrollbarArrowUpOver_png);
	arrowUpOverImg = new GuiImage(arrowUpOver);
	
	scrollbarBox = new GuiImageData(ScrollbarBox_png);
	scrollbarBoxImg = new GuiImage(scrollbarBox);
	
	scrollbarBoxOver = new GuiImageData(ScrollbarBoxOver_png);
	scrollbarBoxOverImg = new GuiImage(scrollbarBoxOver);
	
	closeImgData = new GuiImageData(Close_png);
	closeImg = new GuiImage(closeImgData);

	closeImgOverData = new GuiImageData(CloseOver_png);
    closeImgOver = new GuiImage(closeImgOverData);
	
	arrowUpBtn = new GuiButton(arrowUpImg->GetWidth(), arrowUpImg->GetHeight());
	arrowUpBtn->SetImage(arrowUpImg);
	arrowUpBtn->SetImageOver(arrowUpOverImg);
	arrowUpBtn->SetAlignment(ALIGN_RIGHT, ALIGN_TOP);
	arrowUpBtn->SetPosition(-5, 55);
	arrowUpBtn->SetSelectable(false);
	arrowUpBtn->SetClickable(false);
	arrowUpBtn->SetHoldable(true);
	arrowUpBtn->SetTrigger(triggerHeldA);
	arrowUpBtn->SetSoundOver(btnSoundOver);
	arrowUpBtn->SetSoundClick(btnSoundClick);
	arrowUpBtn->SetEffectGrow();

	arrowDownBtn = new GuiButton(arrowDownImg->GetWidth(), arrowDownImg->GetHeight());
	arrowDownBtn->SetImage(arrowDownImg);
	arrowDownBtn->SetImageOver(arrowDownOverImg);
	arrowDownBtn->SetAlignment(ALIGN_RIGHT, ALIGN_BOTTOM);
	arrowDownBtn->SetPosition(-5, -46);
	arrowDownBtn->SetSelectable(false);
	arrowDownBtn->SetClickable(false);
	arrowDownBtn->SetHoldable(true);
	arrowDownBtn->SetTrigger(triggerHeldA);
	arrowDownBtn->SetSoundOver(btnSoundOver);
	arrowDownBtn->SetSoundClick(btnSoundClick);
	arrowDownBtn->SetEffectGrow();

	scrollbarBoxBtn = new GuiButton(scrollbarBoxImg->GetWidth(), scrollbarBoxImg->GetHeight());
	scrollbarBoxBtn->SetImage(scrollbarBoxImg);
	scrollbarBoxBtn->SetImageOver(scrollbarBoxOverImg);
	scrollbarBoxBtn->SetAlignment(ALIGN_RIGHT, ALIGN_TOP);
	scrollbarBoxBtn->SetPosition(-5, 36 + 50);
	scrollbarBoxBtn->SetMinY(0);
	scrollbarBoxBtn->SetMaxY(112);
	scrollbarBoxBtn->SetSelectable(false);
	scrollbarBoxBtn->SetClickable(false);
	scrollbarBoxBtn->SetHoldable(true);
	scrollbarBoxBtn->SetTrigger(triggerHeldA);
	scrollbarBoxBtn->SetEffectGrow();

    closeBtn = new GuiButton(closeImg->GetWidth(), closeImg->GetHeight());
    closeBtn->SetImage(closeImg);
    closeBtn->SetImageOver(closeImgOver);
    closeBtn->SetAlignment(ALIGN_RIGHT, ALIGN_TOP);
	closeBtn->SetPosition(-10, 24);
    closeBtn->SetSoundOver(btnSoundOver);
    closeBtn->SetSoundClick(btnSoundClick);
    closeBtn->SetTrigger(triggerA);
    closeBtn->SetTrigger(triggerB);
    closeBtn->SetTrigger(triggerHome);
    closeBtn->SetEffectGrow();

    titleTxt = new GuiText(fileName, 20, (GXColor){0, 0, 0, 255});
    titleTxt->SetAlignment(ALIGN_CENTER, ALIGN_TOP);
    titleTxt->SetPosition(-20, 25);
    titleTxt->SetMaxWidth(340, GuiText::SCROLL);

    fileContentTxt = new GuiText(inText, 15, (GXColor){0, 0, 0, 255});
    fileContentTxt->SetAlignment(ALIGN_LEFT, ALIGN_TOP);
	fileContentTxt->SetPosition(33, 65);
	fileContentTxt->SetMaxWidth(470, GuiText::LONGTEXT);
    fileContentTxt->SetFirstLine(0);
    fileContentTxt->SetLinesToDraw(linesToDraw);

    totalLines = fileContentTxt->GetTotalLines();

	if (totalLines < linesToDraw)
	{
		linesToDraw = totalLines;
		fileContentTxt->SetLinesToDraw(linesToDraw);
	}

	// Set Widescreen
	if (cfg.Widescreen == Widescreen_On)
	{
		bgTexteditorImg->SetWidescreen(true);
		arrowDownImg->SetWidescreen(true);
		arrowDownOverImg->SetWidescreen(true);
		arrowUpImg->SetWidescreen(true);
		arrowUpOverImg->SetWidescreen(true);
		scrollbarBoxImg->SetWidescreen(true);
		scrollbarBoxOverImg->SetWidescreen(true);
		closeImg->SetWidescreen(true);
		closeImgOver->SetWidescreen(true);
	}
	
	Window = new GuiWindow(bgTexteditorImg->GetWidth(), bgTexteditorImg->GetHeight());
	
	Window->SetParent(this);

	// Append objects
    Window->Append(bgTexteditorImg);
    Window->Append(titleTxt);
    Window->Append(fileContentTxt);
    Window->Append(scrollbarBoxBtn);
    Window->Append(arrowUpBtn);
    Window->Append(arrowDownBtn);
    Window->Append(closeBtn);
}

// Class destructor
TextEditor::~TextEditor()
{
    Window->RemoveAll();

	delete arrowUpBtn;
	delete arrowDownBtn;
	delete scrollbarBoxBtn;
	delete closeBtn;

	delete bgTexteditorImg;
	delete arrowDownImg;
	delete arrowDownOverImg;
	delete arrowUpImg;
	delete arrowUpOverImg;
	delete scrollbarBoxImg;
	delete scrollbarBoxOverImg;
	delete closeImg;
	delete closeImgOver;
	
	delete bgTexteditorData;
	delete arrowDown;
	delete arrowDownOver;
	delete arrowUp;
	delete arrowUpOver;
	delete scrollbarBox;
	delete scrollbarBoxOver;
	delete closeImgData;
	delete closeImgOverData;
	
	delete btnSoundOver;
	delete btnSoundClick;

	delete triggerHeldA;
	delete triggerHome;
	delete triggerA;
	delete triggerB;

    delete titleTxt;
    delete fileContentTxt;

    delete Window;
}

void TextEditor::SetFocus(bool f)
{
    focus = f;
}

void TextEditor::SetText(char *inText)
{
    if (fileContentTxt)
        delete fileContentTxt;

	fileContentTxt = new GuiText(inText, 15, (GXColor){0, 0, 0, 255});
	fileContentTxt->SetAlignment(ALIGN_LEFT, ALIGN_TOP);
	fileContentTxt->SetPosition(33, 65);
	fileContentTxt->SetMaxWidth(470, GuiText::LONGTEXT);
	fileContentTxt->SetFirstLine(0);
	fileContentTxt->SetLinesToDraw(linesToDraw);
}

void TextEditor::DisableTriggerUpdate(bool set)
{
    triggerDisabled = set;
}

void TextEditor::SetAlignment(int h, int v)
{
    GuiElement::SetAlignment(h, v);
	Window->SetAlignment(h, v);
}

void TextEditor::ResetState()
{
    state = STATE_DEFAULT;
	stateChan = -1;

	arrowUpBtn->ResetState();
	arrowDownBtn->ResetState();
	scrollbarBoxBtn->ResetState();
	closeBtn->ResetState();
}

// Draw the window
void TextEditor::Draw()
{
	if (!this->IsVisible())
		return;

	Window->Draw();

	this->UpdateEffects();
}

void TextEditor::Update(GuiTrigger *t)
{
	if (state == STATE_DISABLED || !t || triggerDisabled)
		return;

	int position = 0;
	int positionWiimote = 0;
	int oldCurrentLine = currentLine;

	arrowUpBtn->Update(t);
	arrowDownBtn->Update(t);
	scrollbarBoxBtn->Update(t);
	closeBtn->Update(t);

	if (closeBtn->GetState() == STATE_CLICKED)
        this->SetState(STATE_CLICKED);

	if (totalLines <= linesToDraw)
        return;
	
	if (scrollbarBoxBtn->GetState() == STATE_HELD && scrollbarBoxBtn->GetStateChan() == t->chan && t->wPad->ir.valid)
	{
		scrollbarBoxBtn->SetPosition(-5, 50);
		positionWiimote = t->wPad->ir.y - 60 - scrollbarBoxBtn->GetTop();

		if (positionWiimote < scrollbarBoxBtn->GetMinY())
			positionWiimote = scrollbarBoxBtn->GetMinY();
		else if (positionWiimote > scrollbarBoxBtn->GetMaxY())
			positionWiimote = scrollbarBoxBtn->GetMaxY();

		currentLine = (positionWiimote * totalLines) / 112.0;

        if (currentLine + linesToDraw > totalLines)
            currentLine = totalLines - linesToDraw;

        if (currentLine < 0)
            currentLine = 0;

		if (oldCurrentLine != currentLine)
			fileContentTxt->SetFirstLine(currentLine);
	}

	if (arrowDownBtn->GetState() == STATE_HELD && arrowDownBtn->GetStateChan() == t->chan && currentLine < (totalLines - linesToDraw))
	{
        currentLine++;

        if (currentLine + linesToDraw > totalLines)
            currentLine = totalLines - linesToDraw;

        if (currentLine < 0)
            currentLine = 0;

        fileContentTxt->SetFirstLine(currentLine);
	}
	else if (arrowUpBtn->GetState() == STATE_HELD && arrowUpBtn->GetStateChan() == t->chan && currentLine > 0)
	{
		currentLine--;

        if (currentLine < 0)
            currentLine = 0;

        fileContentTxt->SetFirstLine(currentLine);
	}

	if (t->Right() && currentLine < (totalLines - linesToDraw))
	{
		currentLine += linesToDraw;

        if (currentLine + linesToDraw > totalLines)
            currentLine = totalLines - linesToDraw;

        if (currentLine < 0)
            currentLine = 0;

        fileContentTxt->SetFirstLine(currentLine);
	}
	else if (t->Left() && currentLine > 0)
	{
		currentLine -= linesToDraw;

        if (currentLine < 0)
            currentLine = 0;

        fileContentTxt->SetFirstLine(currentLine);
	}
	else if (t->Down() && currentLine < (totalLines - linesToDraw))
	{
		currentLine++;

        if (currentLine + linesToDraw > totalLines)
            currentLine = totalLines - linesToDraw;

        if (currentLine < 0)
            currentLine = 0;

        fileContentTxt->SetFirstLine(currentLine);
	}
	else if (t->Up() && currentLine > 0)
	{
		currentLine--;

        if (currentLine < 0)
            currentLine = 0;
		
        fileContentTxt->SetFirstLine(currentLine);
	}

	// Update the location of the scroll box based on the position in the file list
	if (positionWiimote > 0)
		position = positionWiimote; // Follow Wiimote cursor
	else
	{
		position = 112 * (currentLine + linesToDraw / 2.0) / (totalLines * 1.0);

		if (currentLine < 1)
			position = 0;
		else if ((currentLine + linesToDraw) >= totalLines)
			position = 112;
	}

	scrollbarBoxBtn->SetPosition(-5, position + 36 + 50);

	if (updateCB)
		updateCB(this);
}
