/////////////////////////////////////////////////////////
// Copyright 2009
// by Tantric
/////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////
//Video routines
/////////////////////////////////////////////////////////

// Includes
#include "System/Input.h"
#include "Libs/LibWiiGUI/GUI.h"

// Constants
#define DEFAULT_FIFO_SIZE 256 * 1024

// variables
static unsigned int		*xFB[2] = {NULL, NULL};							// Double buffered
static int 				whichFB = 0; 										// Switch
static GXRModeObj 		*vMode; 											// Menu video mode
static unsigned char	gpFIFO[DEFAULT_FIFO_SIZE] ATTRIBUTE_ALIGN(32);
static Mtx 				GXModelView2D;

int ScreenHeight;
int ScreenWidth;
u32 FrameTimer = 0;

// Reset the video/rendering mode for the menu
void ResetVideo_Menu()
{
	Mtx44	p;
	f32		yScale;
	u32		xFBHeight;

	VIDEO_Configure(vMode);
	VIDEO_Flush();
	VIDEO_WaitVSync();
	
	if (vMode->viTVMode & VI_NON_INTERLACE)
		VIDEO_WaitVSync();
	else
		while (VIDEO_GetNextField())
			VIDEO_WaitVSync();

	// Clears the background and clears the z buffer
	GXColor background = {0, 0, 0, 255};
	GX_SetCopyClear(background, 0x00ffffff);

	yScale = GX_GetYScaleFactor(vMode->efbHeight, vMode->xfbHeight);
	xFBHeight = GX_SetDispCopyYScale(yScale);

	GX_SetScissor(0, 0, vMode->fbWidth, vMode->efbHeight);
	GX_SetDispCopySrc(0, 0, vMode->fbWidth, vMode->efbHeight);
	GX_SetDispCopyDst(vMode->fbWidth, xFBHeight);
	GX_SetCopyFilter(vMode->aa, vMode->sample_pattern, GX_TRUE, vMode->vfilter);
	GX_SetFieldMode(vMode->field_rendering, ((vMode->viHeight == 2 * vMode->xfbHeight) ? GX_ENABLE:GX_DISABLE));

	if (vMode->aa)
		GX_SetPixelFmt(GX_PF_RGB565_Z16, GX_ZC_LINEAR);
	else
		GX_SetPixelFmt(GX_PF_RGB8_Z24, GX_ZC_LINEAR);

	// Setup the vertex descriptor
	GX_ClearVtxDesc();
	GX_InvVtxCache();
	GX_InvalidateTexAll();

	GX_SetVtxDesc(GX_VA_TEX0, GX_NONE);
	GX_SetVtxDesc(GX_VA_POS, GX_DIRECT);
	GX_SetVtxDesc(GX_VA_CLR0, GX_DIRECT);

	GX_SetVtxAttrFmt(GX_VTXFMT0, GX_VA_POS, GX_POS_XYZ, GX_F32, 0);
	GX_SetVtxAttrFmt(GX_VTXFMT0, GX_VA_CLR0, GX_CLR_RGBA, GX_RGBA8, 0);
	GX_SetVtxAttrFmt(GX_VTXFMT0, GX_VA_TEX0, GX_TEX_ST, GX_F32, 0);
	GX_SetZMode(GX_FALSE, GX_LEQUAL, GX_TRUE);

	GX_SetNumChans(1);
	GX_SetNumTexGens(1);
	GX_SetTevOp(GX_TEVSTAGE0, GX_PASSCLR);
	GX_SetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD0, GX_TEXMAP0, GX_COLOR0A0);
	GX_SetTexCoordGen(GX_TEXCOORD0, GX_TG_MTX2x4, GX_TG_TEX0, GX_IDENTITY);

	guMtxIdentity(GXModelView2D);
	guMtxTransApply(GXModelView2D, GXModelView2D, 0.0F, 0.0F, -50.0F);
	GX_LoadPosMtxImm(GXModelView2D, GX_PNMTX0);

	guOrtho(p, 0, 479, 0, 639, 0, 300);
	GX_LoadProjectionMtx(p, GX_ORTHOGRAPHIC);

	GX_SetViewport(0, 0, vMode->fbWidth, vMode->efbHeight, 0, 1);
	GX_SetBlendMode(GX_BM_BLEND, GX_BL_SRCALPHA, GX_BL_INVSRCALPHA, GX_LO_CLEAR);
	GX_SetAlphaUpdate(GX_TRUE);
}

// Initialize Video
void InitVideo()
{
	VIDEO_Init();

	vMode = VIDEO_GetPreferredMode(NULL); // Get the default video mode

	// Widescreen fix
	if(CONF_GetAspectRatio() == CONF_ASPECT_16_9)
		vMode->viWidth = VI_MAX_WIDTH_PAL;

	VIDEO_Configure(vMode);

	ScreenHeight = 480;
	ScreenWidth = vMode->fbWidth;

	// Allocate the video buffers
	xFB[0] = (u32 *)MEM_K0_TO_K1(SYS_AllocateFramebuffer(vMode));
	xFB[1] = (u32 *)MEM_K0_TO_K1(SYS_AllocateFramebuffer(vMode));

	console_init(xFB[0], 20, 64, vMode->fbWidth, vMode->xfbHeight, vMode->fbWidth * 2);

	// Clear framebuffers
	VIDEO_ClearFrameBuffer(vMode, xFB[0], COLOR_BLACK);
	VIDEO_ClearFrameBuffer(vMode, xFB[1], COLOR_BLACK);
	VIDEO_SetNextFramebuffer(xFB[0]);

	VIDEO_SetBlack(FALSE);
	VIDEO_Flush();
	VIDEO_WaitVSync();
	
	if (vMode->viTVMode & VI_NON_INTERLACE)
		VIDEO_WaitVSync();

	// Initialize GX
	GXColor background = {0, 0, 0, 0xff};

	memset(&gpFIFO, 0, DEFAULT_FIFO_SIZE);

	GX_Init(&gpFIFO, DEFAULT_FIFO_SIZE);
	GX_SetCopyClear(background, 0x00ffffff);
	GX_SetDispCopyGamma(GX_GM_1_0);
	GX_SetCullMode(GX_CULL_NONE);
	
	ResetVideo_Menu();
}

// Stops GX (when exiting)
void StopGX()
{
	GX_AbortFrame();
	GX_Flush();

	VIDEO_SetBlack(TRUE);
	VIDEO_Flush();
}

// Renders everything current sent to GX and flushes video
void Menu_Render()
{
	whichFB ^= 1; // Flip framebuffer

	GX_SetZMode(GX_TRUE, GX_LEQUAL, GX_TRUE);
	GX_SetColorUpdate(GX_TRUE);
	GX_CopyDisp(xFB[whichFB], GX_TRUE);
	GX_DrawDone();

	VIDEO_SetNextFramebuffer(xFB[whichFB]);
	VIDEO_Flush();
	VIDEO_WaitVSync();

	FrameTimer++;
}

//  Draws the specified image on screen using GX
void Menu_DrawImg(f32 xPos, f32 yPos, u16 width, u16 height, u8 data[], f32 degrees, f32 scaleX, f32 scaleY, u8 alpha)
{
	if (data == NULL)
		return;

	GXTexObj texObj;

	GX_InitTexObj(&texObj, data, width, height, GX_TF_RGBA8, GX_CLAMP, GX_CLAMP, GX_FALSE);
	GX_LoadTexObj(&texObj, GX_TEXMAP0);
	GX_InvalidateTexAll();

	GX_SetTevOp(GX_TEVSTAGE0, GX_MODULATE);
	GX_SetVtxDesc(GX_VA_TEX0, GX_DIRECT);

	Mtx m, m1, m2, mv;
	
	width *= .5;
	height *= .5;

	guMtxIdentity(m1);
	guMtxScaleApply(m1, m1, scaleX, scaleY, 1.0);
	
	guVector axis = (guVector) {0 , 0, 1};

	guMtxRotAxisDeg(m2, &axis, degrees);
	guMtxConcat(m2, m1, m);

	guMtxTransApply(m, m, xPos + width, yPos + height, 0);
	guMtxConcat(GXModelView2D, m, mv);
	GX_LoadPosMtxImm(mv, GX_PNMTX0);

	GX_Begin(GX_QUADS, GX_VTXFMT0, 4);
	GX_Position3f32(-width, -height,  0);
	GX_Color4u8(0xFF, 0xFF, 0xFF, alpha);
	GX_TexCoord2f32(0, 0);

	GX_Position3f32(width, -height,  0);
	GX_Color4u8(0xFF, 0xFF, 0xFF, alpha);
	GX_TexCoord2f32(1, 0);

	GX_Position3f32(width, height,  0);
	GX_Color4u8(0xFF, 0xFF, 0xFF, alpha);
	GX_TexCoord2f32(1, 1);

	GX_Position3f32(-width, height,  0);
	GX_Color4u8(0xFF, 0xFF, 0xFF, alpha);
	GX_TexCoord2f32(0, 1);
	
	GX_End();

	GX_LoadPosMtxImm(GXModelView2D, GX_PNMTX0);
	GX_SetTevOp(GX_TEVSTAGE0, GX_PASSCLR);
	GX_SetVtxDesc(GX_VA_TEX0, GX_NONE);
}

// Draws a rectangle at the specified coordinates using GX
void Menu_DrawRectangle(f32 x, f32 y, f32 width, f32 height, GXColor color, u8 filled)
{
	u8			fmt;
	long		n;
	int			i;
	f32			x2 = x + width;
	f32			y2 = y + height;
	guVector 	v[] = {{x, y, 0.0f}, {x2, y, 0.0f}, {x2, y2, 0.0f}, {x, y2, 0.0f}, {x, y, 0.0f}};

	if(!filled)
	{
		fmt = GX_LINESTRIP;
		n = 5;
	}
	else
	{
		fmt = GX_TRIANGLEFAN;
		n = 4;
	}

	GX_Begin(fmt, GX_VTXFMT0, n);
	
	for (i = 0; i < n; i++)
	{
		GX_Position3f32(v[i].x, v[i].y,  v[i].z);
		GX_Color4u8(color.r, color.g, color.b, color.a);
	}
	
	GX_End();
}

void Menu_DrawTPLImg(f32 xPos, f32 yPos, f32 zpos, f32 width, f32 height, GXTexObj *texObj, f32 degrees, f32 scaleX, f32 scaleY, u8 alpha)
{
    GX_LoadTexObj(texObj, GX_TEXMAP0);
    GX_InvalidateTexAll();

    GX_SetTevOp(GX_TEVSTAGE0, GX_REPLACE);
    GX_SetVtxDesc(GX_VA_TEX0, GX_DIRECT);

    Mtx m, m1, m2, mv;
	
    width *= .5;
    height *= .5;
	
    guMtxIdentity(m1);
    guMtxScaleApply(m1, m1, scaleX, scaleY, 1.0);
    guVector axis = (guVector) {0 , 0, 1};
    guMtxRotAxisDeg(m2, &axis, degrees);
    guMtxConcat(m1, m2, m);

    guMtxTransApply(m, m, xPos + width + 0.5, yPos + height + 0.5, zpos);
    guMtxConcat(GXModelView2D, m, mv);
    GX_LoadPosMtxImm(mv, GX_PNMTX0);

    GX_Begin(GX_QUADS, GX_VTXFMT0, 4);
    GX_Position3f32(-width , -height,  0);
    GX_Color4u8(0xFF, 0xFF, 0xFF, alpha);
    GX_TexCoord2f32(0, 0);

    GX_Position3f32(width, -height,  0);
    GX_Color4u8(0xFF, 0xFF, 0xFF, alpha);
    GX_TexCoord2f32(1, 0);

    GX_Position3f32(width, height,  0);
    GX_Color4u8(0xFF, 0xFF, 0xFF, alpha);
    GX_TexCoord2f32(1, 1);

    GX_Position3f32(-width, height,  0);
    GX_Color4u8(0xFF, 0xFF, 0xFF, alpha);
    GX_TexCoord2f32(0, 1);

    GX_End();
	
    GX_LoadPosMtxImm(GXModelView2D, GX_PNMTX0);

    GX_SetTevOp(GX_TEVSTAGE0, GX_PASSCLR);
    GX_SetVtxDesc(GX_VA_TEX0, GX_NONE);
}
