///////////////////////////////////////////////////////
// Copyright 2009
// by Tantric
///////////////////////////////////////////////////////

///////////////////////////////////////////////////////
// Wii/GameCube controller management
///////////////////////////////////////////////////////

// Includes
#include "Menu/Menu.h"
#include "Libs/LibWiiGUI/GUI.h"
#include "Main.h"

// Variables
GuiTrigger userInput[4];
int 	   rumbleRequest[4] = {0, 0, 0, 0};

static int rumbleCount[4] = {0, 0, 0, 0};

// Extern variables
extern u8 shutdown;

//Scans PAD and WPAD
void UpdatePads()
{
	WPAD_ScanPads();
	PAD_ScanPads();

	for (int i = 3; i >= 0; i--)
	{
		userInput[i].pad.btns_d = PAD_ButtonsDown(i);
		userInput[i].pad.btns_u = PAD_ButtonsUp(i);
		userInput[i].pad.btns_h = PAD_ButtonsHeld(i);
		userInput[i].pad.stickX = PAD_StickX(i);
		userInput[i].pad.stickY = PAD_StickY(i);
		userInput[i].pad.substickX = PAD_SubStickX(i);
		userInput[i].pad.substickY = PAD_SubStickY(i);
		userInput[i].pad.triggerL = PAD_TriggerL(i);
		userInput[i].pad.triggerR = PAD_TriggerR(i);
	}
}

// Sets up user input triggers for use
void SetupPads()
{
	PAD_Init();
	WPAD_Init();

	// Read wiimote accelerometer and IR data
	WPAD_SetDataFormat(WPAD_CHAN_ALL,WPAD_FMT_BTNS_ACC_IR);
	WPAD_SetVRes(WPAD_CHAN_ALL, ScreenWidth, ScreenHeight);

	for (int i = 0; i < 4; i++)
	{
		userInput[i].chan = i;
		userInput[i].wPad = WPAD_Data(i);
	}
}

// Shutoff rumble
void ShutoffRumble()
{
	for (int i = 0; i < 4; i++)
	{
		WPAD_Rumble(i, 0);
		rumbleCount[i] = 0;
	}
}

// Rumble
void DoRumble(int i)
{
	if (cfg.Rumble == Rumble_On)
	{
		if (rumbleRequest[i] && rumbleCount[i] < 3)
		{
			WPAD_Rumble(i, 1); // Rumble on
			rumbleCount[i]++;
		}
		else if (rumbleRequest[i])
		{
			rumbleCount[i] = 12;
			rumbleRequest[i] = 0;
		}
		else
		{
			if (rumbleCount[i])
				rumbleCount[i]--;

			WPAD_Rumble(i, 0); // Rumble off
		}
	}
}

// Get X/Y value from Wii Joystick (Classic Controller, Nunchuk) input
s8 WPAD_Stick(u8 chan, u8 right, int axis)
{
	float 	 mag = 0.0;
	float 	 ang = 0.0;
	WPADData *data = WPAD_Data(chan);

	switch (data->exp.type)
	{
		case WPAD_EXP_NUNCHUK:

		case WPAD_EXP_GUITARHERO3:
			if (right == 0)
			{
				mag = data->exp.nunchuk.js.mag;
				ang = data->exp.nunchuk.js.ang;
			}

			break;

		case WPAD_EXP_CLASSIC:
			if (right == 0)
			{
				mag = data->exp.classic.ljs.mag;
				ang = data->exp.classic.ljs.ang;
			}
			else
			{
				mag = data->exp.classic.rjs.mag;
				ang = data->exp.classic.rjs.ang;
			}

			break;

		default:

			break;
	}

	// calculate x/y value (angle need to be converted into radian)
	if (mag > 1.0)
		mag = 1.0;
	else if (mag < -1.0)
		mag = -1.0;

	double val;

	if (axis == 0) // X axis
		val = mag * sin((PI * ang)/180.0f);
	else // Y axis
		val = mag * cos((PI * ang)/180.0f);

	return (s8)(val * 128.0f);
}

// Initialize Stuff
void __Wpad_PowerCallback(s32 chan)
{
	shutdown = 1;
}

int Wpad_Init(void)
{
	int ret;

	ret = WPAD_Init();

	if (ret < 0)
		return ret;

	WPAD_SetPowerButtonCallback(__Wpad_PowerCallback);

	return ret;
}

void WPad_SetIdleTime(u32 seconds)
{
    // Set idle time for Wiimote
    WPAD_SetIdleTimeout(seconds);
}

bool IsWpadConnected()
{
    int i = 0;
    u32 test = 0;
    int notconnected = 0;

    for (i = 3; i >= 0; i--)
	{
        if (WPAD_Probe(i, &test) == WPAD_ERR_NO_CONTROLLER)
            notconnected++;
	}

    if (notconnected < 4)
        return true;
    else
        return false;
}

int HowManyWpadAreConnected()
{
    int i = 0;
    u32 test = 0;
    int notconnected = 0;

    for (i = 3; i >= 0; i--)
	{
        if (WPAD_Probe(i, &test) == WPAD_ERR_NO_CONTROLLER)
            notconnected++;
	}
    
	i = 4-notconnected;
	
	return i;
}

u32 ButtonsHold(void)
{
    int i;
    u32 buttons = 0;

    WPAD_ScanPads();
    PAD_ScanPads();

    for (i = 3; i >= 0; i--)
	{
        buttons |= PAD_ButtonsHeld(i);
        buttons |= WPAD_ButtonsHeld(i);
	}

    return buttons;
}

u32 ButtonsPressed(void)
{
    int i;
    u32 buttons = 0;

    WPAD_ScanPads();
    PAD_ScanPads();

    for (i = 3; i >= 0; i--)
	{
        buttons |= PAD_ButtonsDown(i);
        buttons |= WPAD_ButtonsDown(i);
	}

    return buttons;
}

