///////////////////////////////////////////////////////
// FreeTypeGX is a wrapper class for libFreeType which renders a compiled
// FreeType parsable font into a GX texture for Wii homebrew development.
// Copyright (C) 2008 Armin Tamzarian
// Modified by Tantric, 2009
// 
// This file is part of FreeTypeGX.
// 
// FreeTypeGX is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// FreeTypeGX is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with FreeTypeGX.  If not, see <http://www.gnu.org/licenses/>.
// 
// If you have any suggestions, questions, or comments regarding this library feel free to e-mail me at tamzarian1989@gmail.com.
///////////////////////////////////////////////////////

#ifndef FREETYPEGX_H_
	#define FREETYPEGX_H_

	// Includes
	#include <gccore.h>
	#include <ft2build.h>
	#include <malloc.h>
	#include <string.h>
	#include <map>
	
	#include "Metaphrasis.h"

	#include FT_FREETYPE_H
	#include FT_BITMAP_H

	// Forward declaration of private structures
	typedef struct ftgxCharData_   ftgxCharData;
	typedef struct ftgxDataOffset_ ftgxDataOffset;

	// Macros (UNICODE helper)
	#define _TEXT(t) L ## t

	// Constants
	#define FTGX_NULL					0x0000
	#define FTGX_JUSTIFY_LEFT			0x0001
	#define FTGX_JUSTIFY_CENTER			0x0002
	#define FTGX_JUSTIFY_RIGHT			0x0003
	#define FTGX_JUSTIFY_MASK			0x000f

	#define FTGX_ALIGN_TOP				0x0010
	#define FTGX_ALIGN_MIDDLE			0x0020
	#define FTGX_ALIGN_BOTTOM			0x0030
	#define FTGX_ALIGN_BASELINE			0x0040
	#define FTGX_ALIGN_GLYPH_TOP		0x0050
	#define FTGX_ALIGN_GLYPH_MIDDLE		0x0060
	#define FTGX_ALIGN_GLYPH_BOTTOM		0x0070
	#define FTGX_ALIGN_MASK				0x00f0

	#define FTGX_STYLE_UNDERLINE		0x0100
	#define FTGX_STYLE_STRIKE			0x0200
	#define FTGX_STYLE_MASK				0x0f00

	#define FTGX_COMPATIBILITY_DEFAULT_TEVOP_GX_MODULATE	0X0001
	#define FTGX_COMPATIBILITY_DEFAULT_TEVOP_GX_DECAL		0X0002
	#define FTGX_COMPATIBILITY_DEFAULT_TEVOP_GX_BLEND		0X0004
	#define FTGX_COMPATIBILITY_DEFAULT_TEVOP_GX_REPLACE		0X0008
	#define FTGX_COMPATIBILITY_DEFAULT_TEVOP_GX_PASSCLR		0X0010

	#define FTGX_COMPATIBILITY_DEFAULT_VTXDESC_GX_NONE		0X0100
	#define FTGX_COMPATIBILITY_DEFAULT_VTXDESC_GX_DIRECT	0X0200
	#define FTGX_COMPATIBILITY_DEFAULT_VTXDESC_GX_INDEX8	0X0400
	#define FTGX_COMPATIBILITY_DEFAULT_VTXDESC_GX_INDEX16	0X0800

	#define FTGX_COMPATIBILITY_NONE							0x0000
	#define FTGX_COMPATIBILITY_GRRLIB						FTGX_COMPATIBILITY_DEFAULT_TEVOP_GX_PASSCLR | FTGX_COMPATIBILITY_DEFAULT_VTXDESC_GX_NONE
	#define FTGX_COMPATIBILITY_LIBWIISPRITE					FTGX_COMPATIBILITY_DEFAULT_TEVOP_GX_MODULATE | FTGX_COMPATIBILITY_DEFAULT_VTXDESC_GX_DIRECT

	// Constant color value used only to sanitize Doxygen documentation
	const GXColor ftgxWhite = (GXColor)
	{
		0xff, 0xff, 0xff, 0xff
	};

	typedef std::map<wchar_t, ftgxCharData> FTGX_Cache;

	// Classes
	class FreeTypeGX
	{
		// Private members
		private:
			FT_Library 		ftLibrary;			// FreeType FT_Library instance
			FT_Face 		ftFace;				// FreeType reusable FT_Face typographic object
			FT_Byte 		*ftFace_fromFile;
			FT_GlyphSlot	ftSlot;				// FreeType reusable FT_GlyphSlot glyph container object
			FT_UInt 		ftPointSize_v;		// Requested size of the rendered font
			FT_UInt 		ftPointSize_h;		// Requested size of the rendered font
			bool 			ftKerningEnabled;	// Flag indicating the availability of font kerning data

			uint8_t 		textureFormat;		// Defined texture format of the target EFB
			uint8_t 		vertexIndex;		// Vertex format descriptor index
			uint32_t 		compatibilityMode;	// Compatibility mode for default tev operations and vertex descriptors

			std::map<uint16_t, FTGX_Cache> fontDatas;

			static uint16_t	AdjustTextureWidth(uint16_t, uint8_t);
			static uint16_t AdjustTextureHeight(uint16_t, uint8_t);
			static int16_t	GetStyleOffsetWidth(uint16_t, uint16_t);
			static int16_t	GetStyleOffsetHeight(ftgxDataOffset *, uint16_t);
			void 		 	UnloadFont(void);
			void 		 	ClearGlyphData(void);
			ftgxCharData 	*CacheGlyphData(wchar_t);
			ftgxCharData 	*CacheGlyphData(wchar_t, FTGX_Cache &);
			uint16_t 	 	CacheGlyphDataComplete(void);
			void 		 	LoadGlyphData(FT_Bitmap *, ftgxCharData *);
			void 		 	SetDefaultMode(void);
			void 		 	DrawTextFeature(int16_t, int16_t, uint16_t, ftgxDataOffset *, uint16_t, GXColor);
			void 		 	CopyTextureToFramebuffer(GXTexObj *, f32, f32, int16_t, int16_t, GXColor);
			void 		 	CopyFeatureToFramebuffer(f32, f32, int16_t, int16_t,  GXColor);

		// Public members
		public:
			FreeTypeGX(uint8_t textureFormat = GX_TF_RGBA8, uint8_t vertexIndex = GX_VTXFMT1, uint32_t compatibilityMode = FTGX_COMPATIBILITY_NONE);
			~FreeTypeGX(void);

			static wchar_t	*CharToWideChar(const char*);
			void 			SetVertexFormat(uint8_t);
			void 			SetCompatibilityMode(uint32_t);
			uint16_t 		LoadFont(const char*, const uint8_t*, FT_Long, FT_UInt, bool cacheAll = false);
			void 			ChangeSize(FT_UInt vPointSize, FT_UInt hPointSize = 0);
			uint16_t 		DrawText(int16_t, int16_t, const wchar_t *, GXColor color = ftgxWhite, uint16_t textStyling = FTGX_NULL);
			uint16_t 		GetWidth(const wchar_t *);
			uint16_t 		GetHeight(const wchar_t *);
			ftgxDataOffset	*GetOffset(const wchar_t *, ftgxDataOffset*);
	};
#endif
