///////////////////////////////////////////////////////
// Copyright (C) 2009
// by Dimok
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications and to alter it and
// redistribute it freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you
//    must not claim that you wrote the original software. If you use
//    this software in a product, an acknowledgment in the product
//    documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such and
//    must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
///////////////////////////////////////////////////////

///////////////////////////////////////////////////////
// Handling all the needed file operations
///////////////////////////////////////////////////////

// Includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <unistd.h>

#include "Prompts/PromptWindows.h"
#include "System/FileOps.h"

// Constants
#define BLOCKSIZE 50*1024 // 50KB

typedef struct
{
    char *name;
} SubDirList;

// Extern variables
extern bool actioncanceled;

// Check if file is existing
bool CheckFile(const char *filePath)
{
    FILE *f;

    f = fopen(filePath, "rb");

    if (f)
	{
        fclose(f);
        return true;
	}

    fclose(f);

    return false;
}

// Get fileSize in bytes. u64 for files bigger than 4GB
int FileSize(const char *filePath)
{
  struct stat fileStat;

  if (stat(filePath, &fileStat) != 0)
    return 0;

  return fileStat.st_size;
}

// Load up the file into a block of memory
int LoadFileToMem(const char *filePath, u8 **inbuffer, u64 *size)
{
    int  ret = -1;
    u64  fileSize = FileSize(filePath);
	char *fileName = strrchr(filePath, '/');

	if (fileName)
        fileName++;

    FILE *file = fopen(filePath, "rb");

    if (file == NULL)
    {
        *inbuffer = NULL;
        *size = 0;

        return -1;
	}

    u8 *buffer = (u8 *) malloc(fileSize);

    if (buffer == NULL)
    {
        fclose(file);
		
        *inbuffer = NULL;
        *size = 0;

        return -2;
	}

    u64 done = 0;
    u32 blockSize = BLOCKSIZE;

    do
    {
        if (actioncanceled)
        {
            free(buffer);
            fclose(file);
			
            *inbuffer = NULL;
            *size = 0;

            return -10;
		}

        if (blockSize > fileSize - done)
            blockSize = fileSize - done;

        ret = fread(buffer + done, 1, blockSize, file);

        if (ret < 0)
        {
            free(buffer);
            fclose(file);
			
            *inbuffer = NULL;
            *size = 0;
			
            return -3;
		}
        else if (ret == 0)
            break;

        done += ret;

	} while (done < fileSize);

    fclose(file);

    if (done != fileSize)
    {
        free(buffer);
		
        *inbuffer = NULL;
        *size = 0;

        return -3;
	}

    *inbuffer = buffer;
    *size = fileSize;

    return 1;
}

// Create recursive all subfolders to the given path
bool CreateSubfolder(const char *fullPath)
{
    // Check or create subfolders
    struct stat st;
    char   dirNoSlash[MAXPATHLEN];

    snprintf(dirNoSlash, strlen(fullPath), "%s", fullPath);

    if (stat(fullPath, &st) != 0)
	{
        char dircheck[MAXPATHLEN];
        char *pch = NULL;
        u32  cnt = 0;

        pch = strrchr(dirNoSlash, '/');
        cnt = pch - dirNoSlash;
        
		snprintf(dircheck, cnt + 2, "%s", dirNoSlash);
		
        CreateSubfolder(dircheck);
	}

    if (mkdir(dirNoSlash, 0777) == -1)
        return false;

    return true;
}

// Compare the file type of the path with a type
bool VerifyFileType(char *filePath, const char *type)
{
	char temp[200], value[200];

	memcpy(temp, filePath, strlen(filePath) + 1);

    char *eq = strrchr(temp, '.');

    if (!eq)
		return false;

    *eq = 0;

    TrimFileTypePath(value, eq + 1, sizeof(value));
	
	return (strcasecmp(value, type) == 0);
}

void TrimFileTypePath(char *dest, char *src, int size)
{
	int len;

	while (*src == ' ')
		src++;

	len = strlen(src);

	while (len > 0 && strchr(" \r\n", src[len - 1]))
		len--;

	if (len >= size)
		len = size - 1;

	strncpy(dest, src, len);
	dest[len] = 0;
}
