///////////////////////////////////////////////////////
// Copyright (C) 2009
// by Dimok
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications and to alter it and
// redistribute it freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you
//    must not claim that you wrote the original software. If you use
//    this software in a product, an acknowledgment in the product
//    documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such and
//    must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
///////////////////////////////////////////////////////

///////////////////////////////////////////////////////
// Progress Window
///////////////////////////////////////////////////////

// Includes
#include "Libs/LibWiiGUI/GUI.h"
#include "Prompts/ProgressWindow.h"
#include "System/FileOps.h"
#include "Menu/Menu.h"
#include "Main.h"

// Variables used only in this file
static lwp_t	progressthread = LWP_THREAD_NULL;
static char		progressTitle[100];
static char		progressMsg[200];
static short	showProgress = 0;
static u64		progressDone = 0;
static u64		progressTotal = 100;
static time_t	start = 0;
static bool		changed = false;
static int		stateManage = 1;

// Variables used outside of this file
bool actioncanceled = false;

// Extern variables
extern GuiWindow *mainWindow;
extern GuiSound  *backgroundMusic;

// Extern methods
extern void ResumeGui();
extern void HaltGui();

// Opens a window, which displays a progress bar
void ProgressWindow()
{
	if (!showProgress)
		return;

	// Sounds
	GuiSound btnSoundOver(ButtonOver_pcm, ButtonOver_pcm_size, SOUND_PCM, cfg.SFXVolume);
    GuiSound btnClick(ButtonClick_pcm, ButtonClick_pcm_size, SOUND_PCM, cfg.SFXVolume);
    
	// Triggers
	GuiTrigger triggerA;
	triggerA.SetSimpleTrigger(-1, WPAD_BUTTON_A | WPAD_CLASSIC_BUTTON_A, PAD_BUTTON_A);

	GuiTrigger triggerB;
	triggerB.SetSimpleTrigger(-1, WPAD_BUTTON_B | WPAD_CLASSIC_BUTTON_B, PAD_BUTTON_B);

	// Objects
	GuiWindow promptWindow(472, 320);
	promptWindow.SetAlignment(ALIGN_CENTER, ALIGN_MIDDLE);
	promptWindow.SetPosition(0, -10);

	GuiImageData dialogBox(DialogueBox_png);
	GuiImage dialogBoxImg(&dialogBox);

	GuiImageData progressbarOutline(ProgressbarOutline_png);
	GuiImage progressbarOutlineImg(&progressbarOutline);
	progressbarOutlineImg.SetAlignment(ALIGN_LEFT, ALIGN_MIDDLE);
	progressbarOutlineImg.SetPosition(35, 15);

	GuiImageData progressbarEmpty(ProgressbarEmpty_png);
	GuiImage progressbarEmptyImg(&progressbarEmpty);
	progressbarEmptyImg.SetAlignment(ALIGN_LEFT, ALIGN_MIDDLE);
	progressbarEmptyImg.SetPosition(35, 15);
	progressbarEmptyImg.SetTile(100);

    GuiImageData progressbar(Progressbar_png);
    GuiImage progressbarImg(&progressbar);
    progressbarImg.SetAlignment(ALIGN_LEFT, ALIGN_MIDDLE);
	progressbarImg.SetPosition(35, 15);

	GuiText titleTxt(progressTitle, 24, (GXColor){0, 0, 0, 255});
	titleTxt.SetAlignment(ALIGN_CENTER, ALIGN_TOP);
	titleTxt.SetPosition(-20, 25);
	titleTxt.SetMaxWidth(430, GuiText::DOTTED);

    GuiText msgTxt(NULL, 22, (GXColor){0, 0, 0, 255});
	msgTxt.SetAlignment(ALIGN_CENTER, ALIGN_TOP);
	msgTxt.SetPosition(0, 110);
	msgTxt.SetMaxWidth(430, GuiText::DOTTED);

    GuiText prTxt(NULL, 24, (GXColor){0, 0, 0, 255});
	prTxt.SetAlignment(ALIGN_LEFT, ALIGN_MIDDLE);
	prTxt.SetPosition(210, 15);

	GuiText prsTxt("%", 24, (GXColor){0, 0, 0, 255});
	prsTxt.SetAlignment(ALIGN_RIGHT, ALIGN_MIDDLE);
	prsTxt.SetPosition(-178, 15);

	GuiImageData btnClose(Close_png);
	GuiImage closeImg(&btnClose);
	GuiImageData btnCloseOver(CloseOver_png);
	GuiImage closeImgOver(&btnCloseOver);
	GuiButton closeBtn(closeImg.GetWidth(), closeImg.GetHeight());

	closeBtn.SetAlignment(ALIGN_RIGHT, ALIGN_TOP);
	closeBtn.SetPosition(-22, 26);
	closeBtn.SetImage(&closeImg);
	closeBtn.SetImageOver(&closeImgOver);
	closeBtn.SetSoundOver(&btnSoundOver);
	closeBtn.SetSoundClick(&btnClick);
	closeBtn.SetTrigger(&triggerA);
	closeBtn.SetEffectGrow();

    GuiImageData btnOutline(Button_png);
	GuiImage buttonImg(&btnOutline);
	
    GuiText abortTxt(GetText("Cancel"), 22, (GXColor){0, 0, 0, 255});
	GuiButton abortBtn(buttonImg.GetWidth(), buttonImg.GetHeight());

	abortBtn.SetAlignment(ALIGN_CENTER, ALIGN_BOTTOM);
	abortBtn.SetPosition(0, -48);
	abortBtn.SetLabel(&abortTxt);
	abortBtn.SetImage(&buttonImg);
	abortBtn.SetSoundOver(&btnSoundOver);
    abortBtn.SetSoundClick(&btnClick);
    abortBtn.SetTrigger(&triggerA);
    abortBtn.SetTrigger(&triggerB);
	abortBtn.SetEffectGrow();

	// Set Widescreen
	if (cfg.Widescreen == Widescreen_On)
	{
		dialogBoxImg.SetWidescreen(true);
		progressbarOutlineImg.SetWidescreen(true);
		progressbarEmptyImg.SetWidescreen(true);
		progressbarImg.SetWidescreen(true);
		titleTxt.SetWidescreen(true);
		msgTxt.SetWidescreen(true);
		prTxt.SetWidescreen(true);
		prsTxt.SetWidescreen(true);
		closeImg.SetWidescreen(true);
		closeImgOver.SetWidescreen(true);
		buttonImg.SetWidescreen(true);
		abortTxt.SetWidescreen(true);
	}

	// Append objects
	promptWindow.Append(&dialogBoxImg);
	promptWindow.Append(&titleTxt);
	promptWindow.Append(&closeBtn);
	promptWindow.Append(&msgTxt);
	promptWindow.Append(&abortBtn);
	promptWindow.Append(&progressbarEmptyImg);
	promptWindow.Append(&progressbarImg);
	promptWindow.Append(&progressbarOutlineImg);
	promptWindow.Append(&prTxt);
	promptWindow.Append(&prsTxt);

	HaltGui();
	promptWindow.SetEffect(EFFECT_SLIDE_TOP | EFFECT_SLIDE_IN, 50);

	if (stateManage == 1)
		mainWindow->SetState(STATE_DISABLED);

	mainWindow->Append(&promptWindow);
	mainWindow->ChangeFocus(&promptWindow);
	ResumeGui();

    while (promptWindow.GetEffect() > 0)
		usleep(100);

	while (showProgress != 0)
	{
	    usleep(THREAD_SLEEP);

	    if (changed)
		{
            msgTxt.SetText(progressMsg);

			progressbarImg.SetTile(100 * progressDone / progressTotal);
			prTxt.SetTextf("%0.1f", (float) (100.0f * progressDone / progressTotal));

			changed = false;
		}
        else if (abortBtn.GetState() == STATE_CLICKED)
		{
            actioncanceled = true;
            abortBtn.ResetState();
		}
        else if (closeBtn.GetState() == STATE_CLICKED)
            actioncanceled = true;
	}

	promptWindow.SetEffect(EFFECT_SLIDE_BOTTOM | EFFECT_SLIDE_OUT, 50);

	while (promptWindow.GetEffect() > 0)
		usleep(100);

	HaltGui();
	mainWindow->Remove(&promptWindow);

	if (stateManage == 1)
		mainWindow->SetState(STATE_DEFAULT);

	ResumeGui();

	actioncanceled = false;
}

static void *ProgressThread(void *arg)
{
	while (1)
	{
		if (!showProgress)
			LWP_SuspendThread(progressthread);

        ProgressWindow();

        usleep(THREAD_SLEEP);
	}

	return NULL;
}

void StopProgress()
{
	showProgress = 0;

	// Wait for thread to finish
	while (!LWP_ThreadIsSuspended(progressthread))
		usleep(THREAD_SLEEP);
}

void StartProgress(const char *title, int windowStateManage)
{
    strncpy(progressTitle, title, sizeof(progressTitle));

	showProgress = 1;
	stateManage = windowStateManage;
	
	LWP_ResumeThread(progressthread);
}

// Callback function for updating the progress values
void UpdateProgress(const char *msg, u64 done, u64 total)
{
	if (total <= 0)
		return;

	if (done > total)
		done = total;

    if (!done)
        start = time(0);

	// Set new message
	if (msg)
		strncpy(progressMsg, msg, sizeof(progressMsg));
	else
		sprintf(progressMsg, " ");

    progressDone = done;
    progressTotal = total;

    changed = true;
}

// Startup Progress thread in idle priority
void InitProgressThread()
{
	LWP_CreateThread(&progressthread, ProgressThread, NULL, NULL, 0, 70);
}

// Shutdown Progress thread
void ExitProgressThread()
{
	LWP_JoinThread(progressthread, NULL);
	progressthread = LWP_THREAD_NULL;
}
