///////////////////////////////////////////////////////
// Handles all menu logic
///////////////////////////////////////////////////////

// Includes
#include "Menu/Menu.h"
#include "Main.h"
#include "Menu/HomeMenu.h"
#include "Libs/LibWiiGUI/GUI.h"
#include "Prompts/PromptWindows.h"
#include "Prompts/ProgressWindow.h"
#include "Settings/SettingsPrompts.h"
#include "Text/TextViewer.h"
#include "Report/Report.h"
#include "System/Storage/Mounter.h"

// Extern variables
extern u8   		shutdown;
extern u8   		reset;
extern bool 		actioncanceled;
extern Settings		Settings;
extern GuiWindow	*mainWindow;
extern bool			sysCheckStatus;

// Variables
GuiSound			*backgroundMusic = NULL;
bool				reportDisplayed;

static GuiImageData *pointer[4];
static lwp_t 		guiThread = LWP_THREAD_NULL;
static bool 		guiHalt = true;

// Namespaces
using namespace std;

// Signals the GUI thread to start and resumes the thread
// This is called after finishing the removal/insertion of new elements and after initial GUI setup
void ResumeGui()
{
	guiHalt = false;
	LWP_ResumeThread(guiThread);
}

// Signals the GUI thread to stop and waits for GUI thread to stop
// This is necessary whenever removing/inserting new elements into the GUI
// This eliminates the possibility that the GUI is in the middle of accessing an element that is being changed
void HaltGui()
{
	guiHalt = true;

	// Wait for thread to finish
	while (!LWP_ThreadIsSuspended(guiThread))
		usleep(THREAD_SLEEP);
}

// Primary thread to allow GUI to respond to state changes and draws GUI
void *UpdateGUI(void *arg)
{
	int i;

	while (1)
	{
		if (guiHalt)
			LWP_SuspendThread(guiThread);
		
		if (shutdown == 1)
			Sys_Shutdown();

		else if (reset == 1)
			Sys_Reboot();
		
		else
		{
			if (!ExitRequested)
			{
				UpdatePads();
				mainWindow->Draw();

				if (mainWindow->GetState() != STATE_DISABLED)
					mainWindow->DrawTooltip();
				
				for (i = 3; i >= 0; i--) // So that player 1's cursor appears on top!
				{
					if (userInput[i].wPad->ir.valid)
						Menu_DrawImg(userInput[i].wPad->ir.x-48, userInput[i].wPad->ir.y-48, 96, 96, pointer[i]->GetImage(), userInput[i].wPad->ir.angle, cfg.Widescreen? 0.8 : 1, 1, 255);

					if (cfg.Rumble == Rumble_On)
						DoRumble(i);
				}
				
				Menu_Render();
				
				for (i = 3; i >= 0; i--)
					mainWindow->Update(&userInput[i]);
			}
			
			if (ExitRequested)
			{
				ShutoffRumble();
				
                for (i = 0; i < 255; i += 15)
				{
					mainWindow->Draw();
					Menu_DrawRectangle(0, 0, ScreenWidth, ScreenHeight, (GXColor){0, 0, 0, i}, 1);
					Menu_Render();
				}
				
				mainWindow->RemoveAll();
				
                return 0;
			}
		}
	}

	return NULL;
}

// Startup GUI threads
void InitGUIThreads()
{
	// Initialize main GUI handling thread
	LWP_CreateThread(&guiThread, UpdateGUI, NULL, NULL, 0, 70);
	
	// Initalize the progress thread
	InitProgressThread();
	StopProgress();
}

void ExitGUIThreads()
{
    ExitRequested = 1;
    LWP_JoinThread(guiThread, NULL);
    guiThread = LWP_THREAD_NULL;
}

static int ShowMainMenu()
{
	ShutoffRumble();

	// Background image
	GuiImageData bgImgData(cfg.Widescreen? BackgroundMainWide_png : BackgroundMain_png);
	GuiImage bgImg(&bgImgData);
	
	mainWindow->Append(&bgImg);

	// Sounds
	GuiSound btnSoundOver(ButtonOver_pcm, ButtonOver_pcm_size, SOUND_PCM, cfg.SFXVolume);
	GuiSound btnSoundClick(ButtonClick_pcm, ButtonClick_pcm_size, SOUND_PCM, cfg.SFXVolume);
	
	// Triggers
	GuiTrigger triggerA;
	triggerA.SetSimpleTrigger(-1, WPAD_BUTTON_A | WPAD_CLASSIC_BUTTON_A, PAD_BUTTON_A);

	GuiTrigger triggerHome;
    triggerHome.SetButtonOnlyTrigger(-1, WPAD_BUTTON_HOME | WPAD_CLASSIC_BUTTON_HOME, 0);

	// Tooltips
    GuiTooltip settingsBtnTT(GetText("Settings"));
    settingsBtnTT.SetAlpha(255);
	
    GuiTooltip homeBtnTT(GetText("HOME Menu"));
    homeBtnTT.SetAlpha(255);

    char revision[50];
	
    snprintf(revision,sizeof(revision), "%s v%s", APP_TITLE, APP_VERSION);

    GuiTooltip sysCheckGXBtnTT(revision);
    sysCheckGXBtnTT.SetAlpha(255);

	// Objects
	GuiImageData settings(SettingsButton_png);
	GuiImage settingsBtnImg(&settings);

	GuiImageData settingsOver(SettingsButtonOver_png);
	GuiImage settingsBtnImgOver(&settingsOver);

	GuiImageData home(Close_png);
	GuiImage homeBtnImg(&home);

	GuiImageData homeOver(CloseOver_png);
	GuiImage homeBtnImgOver(&homeOver);
	
	GuiImageData sysCheckGX(sysCheckGX_png);
	GuiImage sysCheckGXBtnImg(&sysCheckGX);

	GuiImageData sysCheckGXOver(sysCheckGXOver_png);
	GuiImage sysCheckGXBtnImgOver(&sysCheckGXOver);

	GuiButton settingsBtn(settings.GetWidth(), settings.GetHeight());
	settingsBtn.SetAlignment(ALIGN_LEFT, ALIGN_TOP);
	settingsBtn.SetPosition(64, 371);
	settingsBtn.SetImage(&settingsBtnImg);
	settingsBtn.SetImageOver(&settingsBtnImgOver);
	settingsBtn.SetSoundOver(&btnSoundOver);
	settingsBtn.SetSoundClick(&btnSoundClick);
	settingsBtn.SetTrigger(&triggerA);
	settingsBtn.SetEffectGrow();

	GuiButton sysCheckGXBtn(sysCheckGX.GetWidth(), sysCheckGX.GetHeight());
	sysCheckGXBtn.SetAlignment(ALIGN_CENTER, ALIGN_TOP);
	sysCheckGXBtn.SetPosition(0, 17);
	sysCheckGXBtn.SetImage(&sysCheckGXBtnImg);
	sysCheckGXBtn.SetImageOver(&sysCheckGXBtnImgOver);
	sysCheckGXBtn.SetEffectGrow();

	GuiButton homeBtn(home.GetWidth(), home.GetHeight());
	homeBtn.SetAlignment(ALIGN_LEFT, ALIGN_TOP);
	homeBtn.SetPosition(590, 22);
	homeBtn.SetImage(&homeBtnImg);
	homeBtn.SetImageOver(&homeBtnImgOver);
	homeBtn.SetSoundOver(&btnSoundOver);
	homeBtn.SetSoundClick(&btnSoundClick);
	homeBtn.SetTrigger(&triggerA);
	homeBtn.SetTrigger(&triggerHome);
	homeBtn.SetEffectGrow();
	
	settingsBtn.SetToolTip(&settingsBtnTT, 65, -30, 0, 5);
	homeBtn.SetToolTip(&homeBtnTT, -24, 30, 1, 5);
	sysCheckGXBtn.SetToolTip(&sysCheckGXBtnTT, 65, 30, 0, 5);
	
	GuiWindow buttonWindow(ScreenWidth, ScreenHeight);

	// Set Widescreen
	if (cfg.Widescreen == Widescreen_On)
	{
		settingsBtnImg.SetWidescreen(true);
		settingsBtnImgOver.SetWidescreen(true);
		homeBtnImg.SetWidescreen(true);
		homeBtnImgOver.SetWidescreen(true);
		sysCheckGXBtnImg.SetWidescreen(true);
		sysCheckGXBtnImgOver.SetWidescreen(true);
		settingsBtnTT.SetWidescreen(true);
		homeBtnTT.SetWidescreen(true);
		sysCheckGXBtnTT.SetWidescreen(true);
	}

	// Append objects
	buttonWindow.Append(&settingsBtn);
	buttonWindow.Append(&homeBtn);
	buttonWindow.Append(&sysCheckGXBtn);

	HaltGui();
	
	mainWindow->Append(&buttonWindow);
	
	ResumeGui();	

	// Check if the report has been displayed
	if (!reportDisplayed)
	{
		reportDisplayed = true;

		if (cfg.DisplayReport == DisplayReport_On)
		{
			bool   status = false;
			Report sysCheckReport;
			
			if (cfg.DetailedReport == DetailedReport_On)
				status = sysCheckReport.Show(true);
			else
				status = sysCheckReport.Show(false);

			// Check if AutoShutdown is on
			if ((status) && (cfg.AutoShutdownHomebrew == AutoShutdownHomebrew_On) && (cfg.ShutdownMode != ShutdownMode_Off))
			{
				const int seconds = 5;
				char 	  message[256];
				
				// Check the shutdown mode
				switch (cfg.ShutdownMode)
				{
					case ShutdownMode_Shutdown:
						snprintf(message, sizeof(message), GetText("Shutdown in %d seconds..."), seconds);

						break;

					case ShutdownMode_SystemMenu:
						snprintf(message, sizeof(message), GetText("Return to the System Menu in %d seconds..."), seconds);

						break;

					case ShutdownMode_HBC:
						snprintf(message, sizeof(message), GetText("Return to the HBC in %d seconds..."), seconds);

						break;
				}

				int ret = TimedMessageBox(seconds, "", message, "Cancel");

				if (ret == MESSAGEBOX_TIMEOUT)
				{
					// Check the shutdown mode
					switch (cfg.ShutdownMode)
					{
						case ShutdownMode_Shutdown:
							Sys_ShutdownToStandby();

							break;

						case ShutdownMode_SystemMenu:
							Sys_LoadMenu();

							break;

						case ShutdownMode_HBC:
							Sys_BackToLoader();
					}
				}
			}
		}
	}

	int choice = 0;
	int menu = MENU_NONE;
	
	while (menu == MENU_NONE)
	{
		usleep(THREAD_SLEEP);
		
		if (homeBtn.GetState() == STATE_CLICKED)
		{
			backgroundMusic->Pause();
			
            choice = WindowExitPrompt();

			switch (choice)
			{
//				case CHOICE_PRIILOADER:
//					Sys_ExitToPriiloader();
//
//					break;

				case CHOICE_IDLE_SHUTDOWN:
					Sys_ShutdownToIdle();

					break;

				case CHOICE_FULL_SHUTDOWN:
					Sys_ShutdownToStandby();

					break;

				case CHOICE_SYSTEM_MENU:
					Sys_LoadMenu();

					break;

				case CHOICE_HBC:
					Sys_BackToLoader();

					break;

				default:
					mainWindow->SetState(STATE_DEFAULT);
					backgroundMusic->Resume();
					backgroundMusic->Play();
					homeBtn.ResetState();
					
					menu = MENU_MAIN;

					break;
			}
		}
		else if (settingsBtn.GetState() == STATE_CLICKED)
		{
			menu = MENU_SETTINGS;
			settingsBtn.ResetState();
		}
	}

	HaltGui();

	mainWindow->Remove(&buttonWindow);

	return menu;
}

// Main menu
void ShowMenu(int menu)
{
	int currentMenu = menu;

	// Initialize input
	SetupPads();
	
	// Initialize audio
	InitAudio();

	if (!DEBUG_MODE)
		SDMount(); // Mount SD card

	// Initialize GUI
	InitGUIThreads();

#ifdef HW_RVL
	pointer[0] = new GuiImageData(Pointer1_png);
	pointer[1] = new GuiImageData(Pointer2_png);
	pointer[2] = new GuiImageData(Pointer3_png);
	pointer[3] = new GuiImageData(Pointer4_png);
#endif

	ResumeGui();

	if (!sysCheckStatus)
	{
		char message[256];

		snprintf(message, sizeof(message), GetText("System check failed to complete! Please consult sd:/%s for more details."), LOG_FILE_NAME);
	
        MessageBox(ERROR, false, GetText("Error"), message, GetText("OK"));
	}
	
	// Check if AutoShutdown is on
	if ((cfg.AutoShutdownHomebrew == AutoShutdownHomebrew_On) && (cfg.ShutdownMode != ShutdownMode_Off) && (cfg.DisplayReport == DisplayReport_Off))
	{
		const int seconds = 5;
		char 	  message[256];
		
		// Check the shutdown mode
		switch (cfg.ShutdownMode)
		{
			case ShutdownMode_Shutdown:
				snprintf(message, sizeof(message), GetText("Shutdown in %d seconds..."), seconds);

				break;

			case ShutdownMode_SystemMenu:
				snprintf(message, sizeof(message), GetText("Return to the System Menu in %d seconds..."), seconds);

				break;

			case ShutdownMode_HBC:
				snprintf(message, sizeof(message), GetText("Return to the HBC in %d seconds..."), seconds);

				break;
		}

		int ret = TimedMessageBox(seconds, "", message, "Cancel");

		if (ret == MESSAGEBOX_TIMEOUT)
		{
			// Check the shutdown mode
			switch (cfg.ShutdownMode)
			{
				case ShutdownMode_Shutdown:
					Sys_ShutdownToStandby();

					break;

				case ShutdownMode_SystemMenu:
					Sys_LoadMenu();

					break;

				case ShutdownMode_HBC:
					Sys_BackToLoader();
			}
		}
	}
	
	// Music
	backgroundMusic = new GuiSound(BackgroundMainMusic_ogg, BackgroundMainMusic_ogg_size, SOUND_OGG, cfg.MusicVolume);
	backgroundMusic->SetLoop(true);
	backgroundMusic->Play();

	while (currentMenu != MENU_EXIT)
	{
		switch (currentMenu)
		{
			case MENU_SETTINGS:
				currentMenu = ShowMenuSettings();

				break;

			default:
				currentMenu = ShowMainMenu();

				break;
		}
	}

	ResumeGui();
	AppExit();
}
