///////////////////////////////////////////////////////
// sysCheck GX v1.0.0
//
// Homebrew application which does several checks on installed IOS:
// 	- IOS Stub;
// 	- Fake signature (aka Trucha Bug);
// 	- ES_DiVerify (aka ES_Identify);
// 	- Flash Access;
// 	- NAND Access;
// 	- Boot2 Access;
// 	- USB 2.0.
///////////////////////////////////////////////////////

// Includes
#include <sstream>

#include "Main.h"
#include "Menu/Menu.h"
#include "Libs/LibWiiGUI/GUI.h"
#include "System/Storage/Mounter.h"
#include "sysCheckGX/Extra.h"
#include "sysCheckGX/Check.h"
#include "sysCheckGX/Light.h"
#include "Prompts/PromptWindows.h"

// Variables
FreeTypeGX 		*fontSystem;
GuiWindow    	*mainWindow = NULL;
Settings     	Settings;
int    			ExitRequested = 0;
bool			sysCheckStatus = true;

// Namespaces
using namespace std;

bool Demo()
{
	return true;
}

// Save the log file
bool SaveLog(const void *str, int size)
{
	ostringstream log; // Log Stream

	// Mount SD card
	SDMount();

	// Initialize the FAT file system
	if (fatInitDefault())
	{
		char logPath[MAXPATHLEN];

		snprintf(logPath, sizeof(logPath), "%s%s", "sd:/", LOG_FILE_NAME);

		// Open the log file
		FILE *logFile = fopen(logPath, "wb");

		if (logFile)
		{
			// Write content to the log file
			fwrite(str, 1, size, logFile);
			fwrite(log.str().c_str(), 1, log.str().size(), logFile);

			// Close the log file
			fclose(logFile);

			return true;
		}
		else
			return false;
	}
	else
		return false;
}

// System Checker
bool RunSysCheck(bool detailedReport)
{
	ostringstream rpt; // Report Stream
	ostringstream log; // Log Stream

	GuiImageData dialogBox(DialogueBox_png);
	GuiImage dialogBoxImg(&dialogBox);
	dialogBoxImg.SetAlignment(ALIGN_CENTER, ALIGN_MIDDLE);
	dialogBoxImg.SetPosition(0, -10);

	GuiImageData close(Close_png);
	GuiImage closeImg(&close);
	closeImg.SetAlignment(ALIGN_RIGHT, ALIGN_TOP);
	closeImg.SetPosition(-106, 96);

	GuiText progressBarMessage(NULL, 22, (GXColor){0, 0, 0, 255});
	progressBarMessage.SetAlignment(ALIGN_CENTER, ALIGN_TOP);
	progressBarMessage.SetPosition(0, 190);
	progressBarMessage.SetMaxWidth(430, GuiText::WRAP);

	GuiImageData progressbarEmpty(ProgressbarEmpty_png);
	GuiImage progressbarEmptyImg(&progressbarEmpty);
	progressbarEmptyImg.SetAlignment(ALIGN_LEFT, ALIGN_MIDDLE);
	progressbarEmptyImg.SetPosition(120, 15);
	progressbarEmptyImg.SetTile(100);

    GuiImageData progressbar(Progressbar_png);
    GuiImage progressbarImg(&progressbar);
    progressbarImg.SetAlignment(ALIGN_LEFT, ALIGN_MIDDLE);
	progressbarImg.SetPosition(120, 15);

	GuiImageData progressbarOutline(ProgressbarOutline_png);
	GuiImage progressbarOutlineImg(&progressbarOutline);
	dialogBoxImg.SetAlignment(ALIGN_CENTER, ALIGN_MIDDLE);
	progressbarOutlineImg.SetAlignment(ALIGN_LEFT, ALIGN_MIDDLE);
	progressbarOutlineImg.SetPosition(120, 15);

	// Set Widescreen
	if (cfg.Widescreen == Widescreen_On)
	{
		dialogBoxImg.SetWidescreen(true);
		closeImg.SetWidescreen(true);
		progressBarMessage.SetWidescreen(true);
		progressbarEmptyImg.SetWidescreen(true);
		progressbarImg.SetWidescreen(true);
		progressbarOutlineImg.SetWidescreen(true);
	}

	// Append objects
	mainWindow->Append(&dialogBoxImg);
	mainWindow->Append(&closeImg);
	mainWindow->Append(&progressBarMessage);
	mainWindow->Append(&progressbarEmptyImg);
	mainWindow->Append(&progressbarImg);
	mainWindow->Append(&progressbarOutlineImg);

	// Show objects
	mainWindow->Draw();
	Menu_Render();

	char message[256];

	snprintf(message, sizeof(message), GetText("%s starting up..."), APP_TITLE);

	log << message << endl << endl;

	// Get the console region
	log << GetText("Getting the console region...") << " ";

	int regionSelection = CONF_GetRegion();

	// Validating the region code
	if ((regionSelection != CONF_REGION_US) && (regionSelection != CONF_REGION_EU) && (regionSelection != CONF_REGION_JP) && (regionSelection != CONF_REGION_KR))
	{
		log << GetText("Error!") << endl;

		SaveLog(log.str().c_str(), log.str().size());

		return false;
	}
	else
		log << GetText("Success.") << endl;

	// Get the system menu version
	log << GetText("Getting the system menu version...") << " ";

	u32 sysVersion = GetSysMenuVersion();

	// Validating the system menu version
	if (!sysVersion)
	{
		log << GetText("Error!") << endl;

		SaveLog(log.str().c_str(), log.str().size());

		return false;
	}
	else
		log << GetText("Success.") << endl;

	// Get the running IOS version and revision
	log << GetText("Getting the running IOS version and revision...") << " ";

	u32 runningIOS = IOS_GetVersion();
	u32 runningIOSRevision = IOS_GetRevision();

	// Validating the running IOS version and revision
	if (!runningIOS || !runningIOSRevision)
	{
		log << GetText("Error!") << endl;

		SaveLog(log.str().c_str(), log.str().size());

		return false;
	}
	else
		log << GetText("Success.") << endl;

	// Get the console ID
	log << GetText("Getting the console ID...") << " ";

	u32 deviceID = GetConsoleID();

	// Validating the console ID
	if (!deviceID)
	{
		log << GetText("Error!") << endl;

		SaveLog(log.str().c_str(), log.str().size());

		return false;
	}
	else
		log << GetText("Success.") << endl;

	// Get the boot2 version
	log << GetText("Getting the boot2 version...") << " ";

	u32 boot2version = GetBoot2Version();

	// Validating the boot2 version
	if (boot2version)
		log << GetText("Success.") << endl;
	else
		log << GetText("Could not get boot2 version. It's possible your Wii is a boot2v4+ Wii, maybe not.") << endl;

	// Get number of titles
	log << GetText("Getting number of titles...") << " ";

	u32 nbTitles;

	if (ES_GetNumTitles(&nbTitles) >= 0)
	{
		snprintf(message, sizeof(message), GetText("Found %d titles."), nbTitles);

		log << message << endl;
	}
	else
	{
		log << GetText("Error!") << endl;

		SaveLog(log.str().c_str(), log.str().size());

		return false;
	}

	// Allocate the memory for titles
	u64 *titles = (u64*)memalign(32, nbTitles * sizeof(u64));

	if (titles == NULL)
	{
		snprintf(message, sizeof(message), GetText("Unable to allocate the memory for %d titles!"), nbTitles);

		log << message << endl;

		SaveLog(log.str().c_str(), log.str().size());

		return false;
	}

	// Get list of titles
	log << GetText("Getting list of titles...") << " ";

	if (ES_GetTitles(titles, nbTitles) < 0)
	{
		log << GetText("Error!") << endl;

		SaveLog(log.str().c_str(), log.str().size());

		return false;
	}
	else
		log << GetText("Success.") << endl;

	int  countIOS = 0;		// Number of IOS
	int  countStubs = 0;	// Number of IOS Stubs
	u32  titleID = 0;
	s32  infoRevision[nbTitles];
	bool infoStub[nbTitles];

	// BC information (Backward Compatibility)
	s32 versionBC = 0;

	// MIOS information
	// MIOS is a special version of IOS that runs when the Wii enters GameCube mode. The same MIOS is used for all GameCube games and software.
	s32 versionMIOS = 0;

	// For each titles found
	for (u32 i = 0; i < nbTitles; i++)
	{
		// Skip non-system titles
		if (titles[i] >> 32 != 1)
			continue;

		titleID = titles[i] & 0xFFFFFFFF;

		// Skip the system menu
		if (titleID == 2)
			continue;

		// Skip possible other non-IOS titles
		if ((titleID > 0xFF) && (titleID != 0x100) && (titleID != 0x101))
			continue;
		
		// Skip the running IOS
		if (titleID == 0)
			continue;

		// Check if this title is an IOS stub
		u32 tmdSize;
		tmd *iosTMD ATTRIBUTE_ALIGN(32);

		// Get the stored TMD size for the title
		if (ES_GetStoredTMDSize(0x0000000100000000ULL | titleID, &tmdSize) < 0)
		{
			snprintf(message, sizeof(message), GetText("Failed to get the stored TMD size for IOS%d!"), titleID);

			log << message << endl;

			SaveLog(log.str().c_str(), log.str().size());

			return false;
		}

		signed_blob *iosTMDBuffer = (signed_blob *)memalign(32, (tmdSize + 32)&(~31));

		memset(iosTMDBuffer, 0, tmdSize);

		// Get the stored TMD for the title
		if (ES_GetStoredTMD(0x0000000100000000ULL | titleID, iosTMDBuffer, tmdSize) < 0)
		{
			snprintf(message, sizeof(message), GetText("Failed to get the stored TMD for IOS%d!"), titleID);

			log << message << endl;
	
			SaveLog(log.str().c_str(), log.str().size());

			return false;
		}

		iosTMD = (tmd *)SIGNATURE_PAYLOAD(iosTMDBuffer);
		
		free(iosTMDBuffer);

		// Get the title version
		u8 noVersion = iosTMD->title_version;
		bool isStub = false;
		
		// Check if this is an IOS stub (according to WiiBrew.org)
		if (IsKnownStub(titleID, iosTMD->title_version))
			isStub = true;
		else
		{
			// If the version is 00, it's probably a stub
			//
			// Stubs have these things in common:
			//	- Title version is mostly 65280, or even better, the last 2 hexadecimal digits are 0;
			// 	- Stub have one app of their own (type 0x1) and 2 shared apps (type 0x8001).
			if (noVersion == 0)
				isStub = ((iosTMD->num_contents == 3) && (iosTMD->contents[0].type == 1 && iosTMD->contents[1].type == 0x8001 && iosTMD->contents[2].type == 0x8001));
			else		
				isStub = false;
		}

		// Check if this is the BC
		if (titleID == 0x100)
		{
			// Get BC version
			versionBC = iosTMD->title_version;
			
			continue;
		}

		// Check if this is the MIOS
		if (titleID == 0x101)
		{
			// Get MIOS version
			versionMIOS = iosTMD->title_version;
			
			continue;
		}
		
		// Add IOS to the list
		titles[countIOS] = titles[i];
		infoRevision[countIOS] = iosTMD->title_version;
		infoStub[countIOS] = isStub;

		countIOS++;

		if (isStub)
			countStubs++;
	}

	titles = (u64*)realloc(titles, countIOS * sizeof(u64));

	// Create the report
	snprintf(message, sizeof(message), "%s v%s", APP_TITLE, APP_VERSION);

	rpt << message << " " << GetReportText("by", detailedReport) << " Erik Spyder" << endl << endl;

	snprintf(message, sizeof(message), GetReportText("%s is running under IOS%d (rev %d).", detailedReport), APP_TITLE, runningIOS, runningIOSRevision);

	rpt << message << endl << endl;

	// Display the console ID
	rpt << GetReportText("Console ID", detailedReport) << ": " << dec << deviceID << "." << endl;

	// Display the console region
	rpt << GetReportText("Region", detailedReport);
	
	switch (regionSelection)
	{
		case CONF_REGION_US:
			rpt << ": NTSC-U.";
			break;

		case CONF_REGION_EU:
			rpt << ": PAL.";
			break;

		case CONF_REGION_JP:
			rpt << ": NTSC-J.";
			break;

		case CONF_REGION_KR:
			rpt << ": KOR.";
			break;

		default:
			rpt << GetReportText("unknown!", detailedReport);
			break;
	}

	// Display Hollywood version
	rpt << endl << "Hollywood v0x" << hex << SYS_GetHollywoodRevision() << "." << endl;

	// Display the boot2 version
	if (boot2version)
		rpt << "Boot2 v" << dec << boot2version << "." << endl;
	else
		rpt << "Boot2 v" << dec << "?" << ". " << GetText("Could not get boot2 version. It's possible your Wii is a boot2v4+ Wii, maybe not.") << endl;

	// Display the system menu version
	rpt << GetReportText("System Menu", detailedReport) << " v" << dec << sysVersion << " (" << SystemMenuVersion::ToString(sysVersion) << ")." << endl;

	// Display the BC version
	rpt << "BC v" << dec << versionBC << "." << endl;

	// Display the MIOS version
	rpt << "MIOS v" << dec << versionMIOS << ". " << endl;

	switch (versionMIOS)
	{
		case 4:
			rpt << GetReportText("First release.", detailedReport);
			break;

		case 5:
			rpt << GetReportText("Blocks the GameCube Action Replay.", detailedReport);
			break;

		case 8:
			rpt << GetReportText("Overwrites memory to prevent the Tweezer Attack.)", detailedReport);
			break;
	}

	rpt << endl << endl;

	snprintf(message, sizeof(message), GetReportText("Found %d titles.", detailedReport), nbTitles);

	rpt << message << endl;

	int countRealIOS = countIOS-countStubs;

	if (countRealIOS == 0 || countStubs == 0)
	{
		if (countRealIOS == 0)
			rpt << GetReportText("No IOS on this console!", detailedReport) << endl;
		else
		{
			snprintf(message, sizeof(message), GetReportText("Found %d IOS on this console.", detailedReport), countRealIOS);

			rpt << message << endl;
		}

		if (countStubs == 0)
			rpt << GetReportText("No IOS Stub on this console!", detailedReport) << endl;
		else
		{
			snprintf(message, sizeof(message), GetReportText("Found %d IOS Stubs on this console.", detailedReport), countStubs);

			rpt << message << endl;
		}
	}
	else
	{
		snprintf(message, sizeof(message), GetReportText("Found %d IOS and %d IOS Stubs on this console.", detailedReport), countRealIOS, countStubs);

		rpt << message << endl;
	}
	
	// Check if Priiloader is installed
	if (IsPriiloaderInstalled())
		rpt << endl << GetReportText("Priiloader is installed.", detailedReport) << endl;

	rpt << endl;

	nbTitles = countIOS;

	if (nbTitles == 0)
	{
		rpt << GetReportText("Error!", detailedReport) << " " << GetReportText("No title on this console!", detailedReport) << endl;

		SaveLog(log.str().c_str(), log.str().size());

		return false;
	}

	// Sort IOS titles
	log << GetText("Sorting titles...") << " ";

	u64  lowestTitle;
	s32  lowestRevision;
	bool lowestStub;
	int  lowestID;

	for (u32 i = 0; i < nbTitles; i++)
	{
		lowestTitle = titles[i];
		lowestRevision = infoRevision[i];
		lowestStub = infoStub[i];
		lowestID = i;
		
		for (u32 j = i + 1; j < nbTitles; j++)
		{
			if (titles[j] < lowestTitle)
			{
				lowestTitle = titles[j];
				lowestRevision = infoRevision[j];
				lowestStub = infoStub[j];
				lowestID = j;
			}
		}

		// Swap IOS title
		titles[lowestID] = titles[i];
		titles[i] = lowestTitle;

		// Swap IOS revision
		infoRevision[lowestID] = infoRevision[i];
		infoRevision[i] = lowestRevision;

		// Swap IOS stub
		infoStub[lowestID] = infoStub[i];
		infoStub[i] = lowestStub;
	}

	log << GetText("Success.") << endl;

	// Get the certificates from the NAND
	log << GetText("Getting the certificates from the NAND...") << " ";

	if (!GetCertificates())
	{
		log << GetText("Error!") << endl;

		SaveLog(log.str().c_str(), log.str().size());

		return false;
	}
	else
		log << GetText("Success.") << endl;
	
	// Test vulnerabilities in IOS
	char progressText[50];
	bool infoFakeSignature[nbTitles];
	bool infoESIdentify[nbTitles];
	bool infoFlashAccess[nbTitles];
	bool infoNANDAccess[nbTitles];
	bool infoBoot2Access[nbTitles];
	bool infoUSB2[nbTitles];

	for (u32 i = 0; i < nbTitles; i++)
	{
		// Get the title ID
		titleID = titles[i] & 0xFFFFFFFF;

		snprintf(progressText, sizeof(progressText), GetText("Scanning IOS%i..."), titleID);

		progressbarImg.SetTile(100 * i / (nbTitles - 1));
		progressBarMessage.SetText(progressText);

		// Refresh objects
		mainWindow->Draw();
		Menu_Render();

		usleep(40000);

		// BootMii As IOS is installed on IOS254
		if (infoStub[i] || titleID == 254)
		{
			infoFakeSignature[i] = false;
			infoESIdentify[i] = false;
			infoFlashAccess[i] = false;
			infoNANDAccess[i] = false;
			infoBoot2Access[i] = false;
			infoUSB2[i] = false;
		}
		else
		{
			// Reload IOS
			IOS_ReloadIOS(titleID);	

			// Test fake signature
			infoFakeSignature[i] = (CheckFakeSignature());

			// Test ES Identify
			infoESIdentify[i] = (CheckESIdentify());

			// Test Flash Access
			infoFlashAccess[i] = (CheckFlashAccess());

			// Test NAND Access
			infoNANDAccess[i] = (CheckNANDAccess());

			// Test Boot2 Access
			infoBoot2Access[i] = (CheckBoot2Access());

			// Test USB 2.0
			infoUSB2[i] = (CheckUSB2(titleID));
		}
	}

	// Initialize the Wii Light
	WIILIGHT_Init();

	// Turn on the Wii Light
	WiiLightControl(WIILIGHT_ON);

	snprintf(message, sizeof(message), GetText("Reloading IOS%d (rev %d)..."), runningIOS, runningIOSRevision);

	log  << endl << message << " ";

	// Reload the running IOS
	IOS_ReloadIOS(runningIOS);	

	log << GetText("Success.") << endl;

	// Generate the report
	log << endl << GetText("Generating the report...") << " ";

	// Get the field separator
	char sep[2];

	if (cfg.FieldSeparator == FieldSeparator_Comma)
		snprintf(sep, sizeof(sep), ",");
	else
		snprintf(sep, sizeof(sep), ";");

	if (detailedReport)
	{
		rpt << endl;

		// Display the header
		rpt << GetReportText("IOS (revision)", detailedReport) << sep << GetReportText("IOS Stub", detailedReport) << sep << GetReportText("Trucha Bug", detailedReport) << sep << GetReportText("ES_Identify", detailedReport) << sep;
		rpt << GetReportText("Flash Access", detailedReport) << sep << GetReportText("NAND Access", detailedReport) << sep << GetReportText("Boot2 Access", detailedReport) << sep << GetReportText("USB 2.0", detailedReport) << sep << GetReportText("Information", detailedReport) << endl;
	}
	else
		rpt << endl;

	// Display IOS vulnerabilities
	for (u32 i = 0; i < nbTitles; i++)
	{
		titleID = titles[i] & 0xFFFFFFFF;

		snprintf(message, sizeof(message), GetReportText("IOS%3d (rev %5d)", detailedReport), titleID, infoRevision[i]);

		rpt << message;

		if (detailedReport)
		{
			rpt << sep;

			if (titleID == 254)
				rpt << GetReportText("No", detailedReport) << sep << "?" << sep << "?" << sep << "?" << sep << "?" << sep << "?" << sep << "?" << sep;
			else
			{
				if (infoStub[i])
					rpt << GetReportText("Yes", detailedReport) << sep << "?" << sep << "?" << sep << "?" << sep << "?" << sep << "?" << sep << "?" << sep;
				else
				{
					rpt << GetReportText("No", detailedReport) << sep;

					rpt << (infoFakeSignature[i] ? GetReportText("Enabled", detailedReport) : GetReportText("Disabled", detailedReport)) << sep;
					rpt << (infoESIdentify[i]    ? GetReportText("Enabled", detailedReport) : GetReportText("Disabled", detailedReport)) << sep;
					rpt << (infoFlashAccess[i]   ? GetReportText("Enabled", detailedReport) : GetReportText("Disabled", detailedReport)) << sep;
					rpt << (infoNANDAccess[i]    ? GetReportText("Enabled", detailedReport) : GetReportText("Disabled", detailedReport)) << sep;
					rpt << (infoBoot2Access[i]   ? GetReportText("Enabled", detailedReport) : GetReportText("Disabled", detailedReport)) << sep;
					rpt << (infoUSB2[i]          ? GetReportText("Enabled", detailedReport) : GetReportText("Disabled", detailedReport)) << sep;
				}
			}

			// Display some details on IOS
			snprintf(message, sizeof(message), "\"%s\"", History::ToString(titleID, infoStub[i], infoRevision[i]));

			rpt << message;
		}
		else
		{
			bool displayedText;

			displayedText = false;

			// Check BootMii As IOS, BootMii As IOS is installed on IOS254
			if ((titleID != 254) && (infoStub[i]))
			{
				rpt << ": " << GetReportText("Stub", detailedReport);
				displayedText = true;
			}

			if (infoFakeSignature[i])
			{
				if (displayedText)
					rpt << sep;
				else
					rpt << ":";

				rpt << " " << GetReportText("Trucha Bug", detailedReport);
				displayedText = true;
			}

			if (infoESIdentify[i])
			{
				if (displayedText)
					rpt << sep;
				else
					rpt << ":";

				rpt << " " << GetReportText("ES_Identify", detailedReport);
				displayedText = true;
			}

			if (infoFlashAccess[i])
			{
				if (displayedText)
					rpt << sep;
				else
					rpt << ":";

				rpt << " " << GetReportText("FLASH Access", detailedReport);
				displayedText = true;
			}

			if (infoNANDAccess[i])
			{
				if (displayedText)
					rpt << sep;
				else
					rpt << ":";

				rpt << " " << GetReportText("NAND Access", detailedReport);
				displayedText = true;
			}

			if (infoBoot2Access[i])
			{
				if (displayedText)
					rpt << sep;
				else
					rpt << ":";

				rpt << " " << GetReportText("Boot2 Access", detailedReport);
				displayedText = true;
			}

			if (infoUSB2[i])
			{
				if (displayedText)
					rpt << sep;
				else
					rpt << ":";
				
				rpt << " " << GetReportText("USB 2.0", detailedReport);
			}
		}

		rpt << endl;
	}

	log << GetText("Success.") << endl;

	// Get and display the current date
	struct tm today;
	time_t    rawtime;

	time(&rawtime);
	today = *localtime(&rawtime);

	snprintf(message, sizeof(message), GetReportText("Report generated on %4.4d-%2.2d-%2.2d.", detailedReport), today.tm_year + 1900, today.tm_mon + 1, today.tm_mday);

	rpt << endl << message << endl;

	log << GetText("Mounting SD card...") << " ";

	// Mount SD card
	if (SDMount())
		log << GetText("Success.") << endl;
	else
		log << GetText("Error!") << endl;

	// Initialize the FAT file system
	log << GetText("Initializing the FAT File System...") << " ";

	if (fatInitDefault())
		log << GetText("Success.") << endl;
	else
	{
		log << GetText("Error!") << endl;

		SaveLog(log.str().c_str(), log.str().size());

		return false;
	}
	
	// Save the report
	char reportPath[MAXPATHLEN];

	if (detailedReport)
		snprintf(reportPath, sizeof(reportPath), "sd:/%s", FULL_REPORT_FILE_NAME);
	else
		snprintf(reportPath, sizeof(reportPath), "sd:/%s", SIMPLE_REPORT_FILE_NAME);

	snprintf(message, sizeof(message), GetText("Creating report to %s..."), reportPath);

	log << message << " ";

	FILE *report = fopen(reportPath, "wb");
	
	if (report)
		log << GetText("Success.") << endl;
	else
	{
		log << GetText("Unable to save the report!") << endl;

		SaveLog(log.str().c_str(), log.str().size());

		return false;
	}

	fwrite(rpt.str().c_str(), 1, rpt.str().size(), report);

	// Close the report
	fclose(report);

	log << endl << GetText("Report saved successfully.") << endl;

	// Wait 2 seconds
	sleep(1);

	// Turn off the Wii Light
	WiiLightControl(WIILIGHT_OFF);

	// Save the log file
	SaveLog(log.str().c_str(), log.str().size());

	return true;
}

// Main program
int main(int argc, char *argv[])
{
	// Initialize Wii button call back
	Sys_Init(); 

	// Initialize video
	InitVideo();

	mainWindow = new GuiWindow(ScreenWidth, ScreenHeight);

	// Background image
	GuiImageData bgImgData(BackgroundIntro_png);
	GuiImage bgImg(&bgImgData);

	mainWindow->Append(&bgImg);

	// Show objects
	mainWindow->Draw();
	Menu_Render();

	// Initialize font system
    fontSystem = new FreeTypeGX();
    fontSystem->LoadFont((char*) NULL, Horatio_ttf, Horatio_ttf_size, 0);
    fontSystem->SetCompatibilityMode(FTGX_COMPATIBILITY_DEFAULT_TEVOP_GX_PASSCLR | FTGX_COMPATIBILITY_DEFAULT_VTXDESC_GX_NONE);

	if (DEBUG_MODE)
	{
		Settings.SetDefault();
		Demo();
	}
	else
	{
		// Mount SD card
		SDMount();

		// Initalize settings
		Settings.Init();

		// Load language file
		if (cfg.UseLanguageFile == UseLanguageFile_On)
			Settings.LoadLanguage();

		// Check vulnerabilities
		if (cfg.SystemCheck == SystemCheck_On)
			sysCheckStatus = RunSysCheck((cfg.DetailedReport == DetailedReport_On));
	}

	// Remove objects
	mainWindow->RemoveAll();

	// Manage main menu
	ShowMenu(MENU_MAIN);
}
