///////////////////////////////////////////////////////
// Copyright 2009
// by Tantric
///////////////////////////////////////////////////////

///////////////////////////////////////////////////////
// GUI window class definitions
// Allows GuiElements to be grouped together into a  Window 
///////////////////////////////////////////////////////

// Includes
#include "GUI.h"

GuiWindow::GuiWindow()
{
	width = 0;
	height = 0;
	focus = 0; // Allow focus
}

GuiWindow::GuiWindow(int w, int h)
{
	width = w;
	height = h;
	focus = 0; // Allow focus
}

// Class destructor
GuiWindow::~GuiWindow()
{
}

// Appends a GuiElement to the GuiWindow
void GuiWindow::Append(GuiElement *e)
{
	if (e == NULL)
		return;

	Remove(e);
	AllElements.push_back(e);
	e->SetParent(this);
}

// Inserts a GuiElement into the GuiWindow at the specified index
void GuiWindow::Insert(GuiElement *e, u32 index)
{
	if (e == NULL || index > (AllElements.size() - 1))
		return;

	Remove(e);
	AllElements.insert(AllElements.begin() + index, e);
	e->SetParent(this);
}

// Removes the specified GuiElement from the GuiWindow
void GuiWindow::Remove(GuiElement *e)
{
	if (e == NULL)
		return;

	for (u8 i = 0; i < AllElements.size(); i++)
	{
		if (e == AllElements.at(i))
		{
			AllElements.erase(AllElements.begin() + i);
			
			break;
		}
	}
}

// Removes all GuiElements
void GuiWindow::RemoveAll()
{
	AllElements.clear();
}

// Returns the GuiElement at the specified index
GuiElement *GuiWindow::GetGuiElementAt(u32 index) const
{
	if (index >= AllElements.size())
		return NULL;
		
	return AllElements.at(index);
}

// Returns the size of the list of elements
u32 GuiWindow::GetSize()
{
	return AllElements.size();
}

// Draws all the elements in this GuiWindow
void GuiWindow::Draw()
{
	if (AllElements.size() == 0 || !this->IsVisible())
		return;

	for (u8 i = 0; i < AllElements.size(); i++)
	{
		try
		{
			AllElements.at(i)->Draw();
		}
		catch (const std::exception& e)
		{
		}
	}

	this->UpdateEffects();

	if (parentElement && state == STATE_DISABLED)
		Menu_DrawRectangle(0, 0, ScreenWidth, ScreenHeight, (GXColor){0, 0, 0, 0x70}, 1);
}

// Draw Tooltip
void GuiWindow::DrawTooltip()
{
	if (AllElements.size() == 0 || !this->IsVisible())
		return;

	for (u8 i = 0; i < AllElements.size(); i++)
	{
		try
		{
			AllElements.at(i)->DrawTooltip();
		}
		catch (const std::exception& e)
		{
		}
	}
}

// Resets the window's state to STATE_DEFAULT
void GuiWindow::ResetState()
{
	if (state != STATE_DISABLED)
		state = STATE_DEFAULT;

	for (u8 i = 0; i < AllElements.size(); i++)
	{
		try
		{
			AllElements.at(i)->ResetState();
		}
		catch (const std::exception& e)
		{
		}
	}
}

// Sets the window's state
void GuiWindow::SetState(int s)
{
	state = s;

	for (u8 i = 0; i < AllElements.size(); i++)
	{
		try
		{
			AllElements.at(i)->SetState(s);
		}
		catch (const std::exception& e)
		{
		}
	}
}

// Sets the visibility of the window
void GuiWindow::SetVisible(bool v)
{
	visible = v;

	for (u8 i = 0; i < AllElements.size(); i++)
	{
		try
		{
			AllElements.at(i)->SetVisible(v);
		}
		catch (const std::exception& e)
		{
		}
	}
}

// Sets the window focus
void GuiWindow::SetFocus(int f)
{
	focus = f;

	if (f == 1)
		this->MoveSelectionVert(1);
	else
		this->ResetState();
}

// Change the focus to the specified element
// (This is intended for the primary GuiWindow only)
void GuiWindow::ChangeFocus(GuiElement *e)
{
	if (parentElement)
		return; // This is only intended for the main window

	for (u8 i = 0; i < AllElements.size(); i++)
	{
		if (e == AllElements.at(i))
			AllElements.at(i)->SetFocus(1);
		else if (AllElements.at(i)->IsFocused() == 1)
			AllElements.at(i)->SetFocus(0);
	}
}

// Changes window focus to the next focusable window or element
// If no element is in focus, changes focus to the first available element
// If B or 1 button is pressed, changes focus to the next available element
// (This is intended for the primary GuiWindow only)
void GuiWindow::ToggleFocus(GuiTrigger *t)
{
	if (parentElement)
		return; // This is only intended for the main window

	int found = -1;
	int newfocus = -1;
	u8  i;

	// 	Look for currently in focus element
	for (i = 0; i < AllElements.size(); i++)
	{
		try
		{
			if (AllElements.at(i)->IsFocused() == 1)
			{
				found = i;
				
				break;
			}
		}
		catch (const std::exception& e)
		{
		}
	}

	// Element with focus not found, try to give focus
	if (found == -1)
	{
		for (i = 0; i < AllElements.size(); i++)
		{
			try
			{
				if (AllElements.at(i)->IsFocused() == 0 && AllElements.at(i)->GetState() != STATE_DISABLED) // Focus is possible (but not set)
				{
					AllElements.at(i)->SetFocus(1); // Give this element focus
					
					break;
				}
			}
			catch (const std::exception& e)
			{
			}
		}
	}
	// Change focus
	else if (t->wPad->btns_d & (WPAD_BUTTON_1 | WPAD_BUTTON_B | WPAD_CLASSIC_BUTTON_B) || t->pad.btns_d & PAD_BUTTON_B)
	{
		for (i = found; i < AllElements.size(); i++)
		{
			try
			{
				if (AllElements.at(i)->IsFocused() == 0 && AllElements.at(i)->GetState() != STATE_DISABLED) // Focus is possible (but not set)
				{
					newfocus = i;
					AllElements.at(i)->SetFocus(1); 		// Give this element focus
					AllElements.at(found)->SetFocus(0); 	// Disable focus on other element

					break;
				}
			}
			catch (const std::exception& e)
			{
			}
		}

		if (newfocus == -1)
		{
			for (i = 0; i < found; i++)
			{
				try
				{
					if (AllElements.at(i)->IsFocused() == 0 && AllElements.at(i)->GetState() != STATE_DISABLED) // Focus is possible (but not set)
					{
						AllElements.at(i)->SetFocus(1); 		// Give this element focus
						AllElements.at(found)->SetFocus(0); 	// Disable focus on other element

						break;
					}
				}
				catch (const std::exception& e)
				{
				}
			}
		}
	}
}

// Gets the index of the GuiElement inside the window that is currently selected
int GuiWindow::GetSelected()
{
	// Find selected element
	int found = -1;

	for (u8 i = 0; i < AllElements.size(); i++)
	{
		try
		{
			if (AllElements.at(i)->GetState() == STATE_SELECTED)
			{
				found = i;
				break;
			}
		}
		catch (const std::exception& e)
		{
		}
	}

	return found;
}

// Moves the selected element to the element to the left or right
void GuiWindow::MoveSelectionHor(int dir)
{
	int found = -1;
	u16 left = 0;
	u16 top = 0;
	u8  i = 0;

	int selected = this->GetSelected();

	if (selected >= 0)
	{
		left = AllElements.at(selected)->GetLeft();
		top = AllElements.at(selected)->GetTop();
	}

	// Look for a button on the same row, to the left/right
	for (i = 0; i < AllElements.size(); i++)
	{
		try
		{
			if (AllElements.at(i)->IsSelectable())
			{
				if (AllElements.at(i)->GetLeft() * dir > left * dir && AllElements.at(i)->GetTop() == top)
				{
					if (found == -1)
						found = i;
					else if (AllElements.at(i)->GetLeft() * dir < AllElements.at(found)->GetLeft() * dir)
						found = i; // This is a better match
				}
			}
		}
		catch (const std::exception& e)
		{
		}
	}

	if (found >= 0)
		goto matchfound;

	// Match still not found, let's try the first button in the next row
	for (i = 0; i < AllElements.size(); i++)
	{
		try
		{
			if (AllElements.at(i)->IsSelectable())
			{
				if (AllElements.at(i)->GetTop() * dir > top * dir)
				{
					if (found == -1)
						found = i;
					else if (AllElements.at(i)->GetTop() * dir < AllElements.at(found)->GetTop() * dir)
						found = i; // This is a better match
					else if (AllElements.at(i)->GetTop() * dir == AllElements.at(found)->GetTop() * dir && AllElements.at(i)->GetLeft() * dir < AllElements.at(found)->GetLeft() * dir)
						found = i; // This is a better match
				}
			}
		}
		catch (const std::exception& e)
		{
		}
	}

	// Match found
	matchfound:

	if (found >= 0)
	{
		AllElements.at(found)->SetState(STATE_SELECTED);

		if (selected >= 0)
			AllElements.at(selected)->ResetState();
	}
}

// Moves the selected element to the element above or below
void GuiWindow::MoveSelectionVert(int dir)
{
	int found = -1;
	u16 left = 0;
	u16 top = 0;
	u8  i = 0;

	int selected = this->GetSelected();

	if (selected >= 0)
	{
		left = AllElements.at(selected)->GetLeft();
		top = AllElements.at(selected)->GetTop();
	}

	// Look for a button above/below, with the least horizontal difference
	for (i = 0; i < AllElements.size(); i++)
	{
		try
		{
			if (AllElements.at(i)->IsSelectable())
			{
				if (AllElements.at(i)->GetTop() * dir > top * dir)
				{
					if (found == -1)
						found = i;
					else if (AllElements.at(i)->GetTop() * dir < AllElements.at(found)->GetTop() * dir)
						found = i; // This is a better match
					else if (AllElements.at(i)->GetTop() * dir == AllElements.at(found)->GetTop() * dir && abs(AllElements.at(i)->GetLeft() - left) < abs(AllElements.at(found)->GetLeft() - left))
						found = i;
				}
			}
		}
		catch (const std::exception& e)
		{
		}
	}

	if (found >= 0)
		goto matchfound;

	// Match found
	matchfound:
	
	if (found >= 0)
	{
		AllElements.at(found)->SetState(STATE_SELECTED);

		if (selected >= 0)
			AllElements.at(selected)->ResetState();
	}
}

// Updates the window and all elements contains within
// Allows the GuiWindow and all elements to respond to the input data specified
void GuiWindow::Update(GuiTrigger *t)
{
	if (AllElements.size() == 0 || (state == STATE_DISABLED && parentElement))
		return;

	for (u8 i = 0; i < AllElements.size(); i++)
	{
		try
		{
			AllElements.at(i)->Update(t);
		}
		catch (const std::exception& e)
		{
		}
	}

	this->ToggleFocus(t);

	if (focus) // Only send actions to this window if it's in focus
	{
		// PAD/Joystick navigation
		if (t->Right())
			this->MoveSelectionHor(1);
		else if (t->Left())
			this->MoveSelectionHor(-1);
		else if (t->Down())
			this->MoveSelectionVert(1);
		else if (t->Up())
			this->MoveSelectionVert(-1);
	}

	if (updateCB)
		updateCB(this);
}
