///////////////////////////////////////////////////////
// LibWiiGUI Copyright 2009
// by Tantric
///////////////////////////////////////////////////////

///////////////////////////////////////////////////////
// GUI text class definitions
// Display, manage and manipulate text in the GUI
///////////////////////////////////////////////////////

// Includes
#include "GUI.h"

// Variables
static int     presetSize = 0;
static int     presetMaxWidth = 0;
static int     presetAlignmentHor = 0;
static int     presetAlignmentVert = 0;
static int     presetWrapMode = GuiText::WRAP;
static u16     presetStyle = FTGX_NULL;
static GXColor presetColor = (GXColor){255, 255, 255, 255};

// Class constructor
GuiText::GuiText(const char *t, int s, GXColor c)
{
	text = NULL;
	size = s;
	color = c;
	alpha = c.a;
	style = FTGX_JUSTIFY_CENTER | FTGX_ALIGN_MIDDLE;
	maxWidth = 0;
	wrapMode = GuiText::WRAP;
	scrollPos1 = 0;
	scrollPos2 = 0;
	scrollDelay = 0;
	lineBreak = NULL;
	font = NULL;
	widescreen = 0;
	firstLine = 1;
	numLines = -1;
	totalLines = 1;
	passChar = 0;
	alignmentHor = ALIGN_CENTER;
	alignmentVert = ALIGN_MIDDLE;

	for (int i = 0; i < MAX_LINES_TO_DRAW; i++)
        tmpText[i] = NULL;
	
	if (t)
		text = FreeTypeGX::CharToWideChar((char *)t);
}

// Class constructor
// (Assumes SetPresets() has been called to setup preferred text attributes)
GuiText::GuiText(const char *t)
{
	text = NULL;
	size = presetSize;
	color = presetColor;
	alpha = presetColor.a;
	style = presetStyle;
	maxWidth = presetMaxWidth;
	wrapMode = presetWrapMode;
	scrollPos1 = 0;
	scrollPos2 = 0;
	scrollDelay = 0;
	lineBreak = NULL;
	font = NULL;
	widescreen = 0;
	firstLine = 1;
	numLines = -1;
	totalLines = 1;

	alignmentHor = presetAlignmentHor;
	alignmentVert = presetAlignmentVert;

	for (int i = 0; i < MAX_LINES_TO_DRAW; i++)
        tmpText[i] = NULL;
	
	if (t)
		text = FreeTypeGX::CharToWideChar((char *)t);
}

// Class destructor
GuiText::~GuiText()
{
	if (text)
	{
		delete [] text;

		text = NULL;
	}
	
	if (lineBreak)
	{
        free(lineBreak);

        lineBreak = NULL;
	}
	
	for (int i = 0; i < MAX_LINES_TO_DRAW; i++)
	{
        if (tmpText[i])
	    {
            delete [] tmpText[i];
			
			tmpText[i] = NULL;			
		}
	}
}

// Sets the text of the GuiText element
void GuiText::SetText(const char *t)
{
	if (text)
		delete [] text;
		
	text = NULL;

	for (int i = 0; i < MAX_LINES_TO_DRAW; i++)
	{
        if (tmpText[i])
	    {
            delete [] tmpText[i];
			
			tmpText[i] = NULL;			
		}
	}
	
	if (t)
	{
		text = FreeTypeGX::CharToWideChar((char *)t);
		
		if (passChar != 0)
		{
			for (u8 i = 0; i < wcslen(text); i++)
				text[i] = passChar;
		}
	}

	scrollPos2 = 0;
	scrollDelay = 0;
}

// Sets the text of the GuiText element
void GuiText::SetTextf(const char *format, ...)
{
	char 	*tmp = 0;
	va_list va;
	
	va_start(va, format);

	if ((vasprintf(&tmp, format, va) >= 0) && tmp)
	{
		this->SetText(tmp);

		free(tmp);
	}

	va_end(va);
}

// Sets the text of the GuiText element
void GuiText::SetText(const wchar_t *t)
{
	if (text)
		delete [] text;
		
	text = NULL;
	
	for (int i = 0; i < MAX_LINES_TO_DRAW; i++)
	{
        if (tmpText[i])
	    {
            delete [] tmpText[i];
			
			tmpText[i] = NULL;			
		}
	}
	
	if (t)
	{
		int len = wcslen(t);
		
		text = new wchar_t[len + 1];
		
		if (text)
			wcscpy(text, t);
		
		if (passChar != 0)
		{
			for (u8 i = 0; i < wcslen(text); i++)
				text[i] = passChar;
		}
	}

	scrollPos2 = 0;
	scrollDelay = 0;
}

// Sets up preset values to be used by GuiText(t)
// (Useful when printing multiple text elements, all with the same attributes set)
void GuiText::SetPresets(int sz, GXColor c, int w, int wrap, u16 s, int h, int v)
{
	presetSize = sz;
	presetColor = c;
	presetMaxWidth = w;
	presetWrapMode = wrap;
	presetStyle = s;
	presetAlignmentHor = h;
	presetAlignmentVert = v;
}

// Sets the font
void GuiText::SetFont(FreeTypeGX *f)
{
	font = f;
}

// Sets the font size
void GuiText::SetFontSize(int s)
{
	size = s;
}

// Set a password character
void GuiText::SetPassChar(wchar_t p)
{
	passChar = p;
}

//  not NULL set horizontal scale to 0.75 //added
void GuiText::SetWidescreen(bool w)
{
	widescreen = w;
}

// Sets the maximum width of the drawn texture image
// If the text exceeds this, it is wrapped to the next line
void GuiText::SetMaxWidth(int w, short m)
{
	maxWidth = w;
	wrapMode = m;
	
	if (wrapMode == GuiText::LONGTEXT)
	{
        int newSize = size * this->GetScale();
		
		(font ? font : fontSystem)->ChangeSize(newSize, widescreen ? newSize * 0.8 : 0);
		
		int strlen = wcslen(text);
        int i = 0;
        int ch = 0;
        int linenum = 0;
        int lastSpace = -1;
        int lastSpaceIndex = -1;
		
        wchar_t *setText = new wchar_t[maxWidth];
		
        lineBreak = (u32 *)malloc(sizeof(u32));
		
        memset(&(lineBreak[linenum]), 0, sizeof(u32));
		
        lineBreak[linenum] = 0;
        linenum++;
		
        while (ch < strlen)
        {
            setText[i] = text[ch];
            setText[i + 1] = 0;
			
            if (ch == strlen - 1 || (font ? font : fontSystem)->GetWidth(setText) >= maxWidth)
            {
                if ((font ? font : fontSystem)->GetWidth(setText) >= maxWidth)
                {
                    if (lastSpace >= 0)
                    {
                        setText[lastSpaceIndex] = 0;	// Discard space and everything after
                        ch = lastSpace; 				// Go backwards to the last space
                        lastSpace = -1; 				// We have used this space
                        lastSpaceIndex = -1;
                    }

                    lineBreak = (u32 *)realloc(lineBreak, (linenum + 1) * sizeof(u32));
					
                    memset(&(lineBreak[linenum]), 0, sizeof(u32));
					
                    lineBreak[linenum] = ch;
                    linenum++;

                    i = -1;
                }
            }

            if ((text[ch] == ' ' || text[ch] == ',') && i >= 0)
            {
                lastSpace = ch + 1;
                lastSpaceIndex = i;
            }

            if (text[ch] == '\n')
            {
                lineBreak = (u32 *)realloc(lineBreak, (linenum + 1) * sizeof(u32));
				
                memset(&(lineBreak[linenum]), 0, sizeof(u32));
				
                lineBreak[linenum] = ch + 1;
                linenum++;
                i = -1;
                lastSpace = -1;
                lastSpaceIndex = -1;
            }

            ch++;
            i++;
			
            if (ch == strlen)
            {
                lineBreak = (u32 *)realloc(lineBreak, (linenum + 1) * sizeof(u32));
				
                memset(&(lineBreak[linenum]), 0, sizeof(u32));
				
                lineBreak[linenum] = ch;
				
                if (text[ch - 1] != '\n')
					linenum++;
                
				break;
            }
        }
		
        delete [] setText;
		
		setText = NULL;
        totalLines = linenum;
	}
	
	for (int i = 0; i < MAX_LINES_TO_DRAW; i++)
	{
	    if (tmpText[i])
	    {
            delete [] tmpText[i];
			
            tmpText[i] = NULL;
	    }
	}
}

// Sets the font color
void GuiText::SetColor(GXColor c)
{
	color = c;
	alpha = c.a;
}

// Sets the FreeTypeGX style attributes
void GuiText::SetStyle(u16 s, u16 m)
{
	style &= ~m;
	style |= s & m;
}

// Sets the text alignment
void GuiText::SetAlignment(int hor, int vert)
{
	style = FTGX_NULL;

	switch (hor)
	{
		case ALIGN_LEFT:
			style |= FTGX_JUSTIFY_LEFT;
			
			break;
			
		case ALIGN_RIGHT:
			style |= FTGX_JUSTIFY_RIGHT;
			
			break;
			
		default:
			style |= FTGX_JUSTIFY_CENTER;
			
			break;
	}
	
	switch (vert)
	{
		case ALIGN_TOP:
			style |= FTGX_ALIGN_TOP;
			
			break;
			
		case ALIGN_BOTTOM:
			style |= FTGX_ALIGN_BOTTOM;
			
			break;
			
		default:
			style |= FTGX_ALIGN_MIDDLE;
			
			break;
	}

	alignmentHor = hor;
	alignmentVert = vert;
}

// Sets maximum lines to draw
void GuiText::SetLinesToDraw(int l)
{
    linesToDraw = l;
}

// Sets the number of lines drawn
void GuiText::SetNumLines(int n)
{	
	numLines = n;
}

// Sets the first line
void GuiText::SetFirstLine(int n)
{
	firstLine = n;
	longListChanged = true;
}

//  Return the line variables for this text
int GuiText::GetNumLines()
{	
	return numLines;
}

// Returns the first line
int GuiText::GetFirstLine()
{
	return firstLine;
}

// Returns Total number of lines
int GuiText::GetTotalLines()
{
	return totalLines;
}

// Get the Horizontal Size of Text
int GuiText::GetTextWidth()
{
	if (!text)
		return 0;
	
	int newSize = size * this->GetScale();

	(font ? font : fontSystem)->ChangeSize(newSize, widescreen ? newSize * 0.8 : 0);
	
	return (font ? font : fontSystem)->GetWidth(text);
}

// Returns the height of the number of lines
// (including spacing if wrap mode is on)
int GuiText::GetLineHeight(int n)
{	
	int newSize = size * this->GetScale();
	int lineheight = newSize + 6;
	
	if (numLines < 0)
		return totalLines * lineheight + newSize;
	else
		return numLines * lineheight + newSize;
}

// Draw the text on screen
void GuiText::Draw()
{
	if (!text)
		return;

	if (!this->IsVisible())
		return;

	GXColor c = color;
	
	c.a = this->GetAlpha();

	int newSize = size * this->GetScale();

	(font ? font : fontSystem)->ChangeSize(newSize, widescreen ? newSize * 0.8 : 0);

	int voffset = 0;

	if (wrapMode == LONGTEXT) // Text wrapping
	{
		int lineheight = newSize + 6;

		if (longListChanged)
		{
			for (int i = 0; i < MAX_LINES_TO_DRAW; i++)
			{
				if (tmpText[i])
				{
					delete [] tmpText[i];
					
					tmpText[i] = NULL;
				}
			}
			
			if (!tmpText[0])
			{
				int index = 0;
				u32 strlen = (u32)wcslen(text);
				int linenum = firstLine;
				int lineIndex = 0;
				u32 ch = lineBreak[linenum];

				u32 lastch = lineBreak[linenum + linesToDraw] + 1;

				tmpText[lineIndex] = new wchar_t[maxWidth];
				tmpText[lineIndex][index] = 0;

				while ((ch < lastch) && (ch < strlen + 1))
				{
					if (ch == lineBreak[linenum + 1])
					{
						(font ? font : fontSystem)->DrawText(this->GetLeft(), this->GetTop() + lineIndex * lineheight, tmpText[lineIndex], c, style);
						linenum++;
						lineIndex++;
						index = 0;
						
						if (!tmpText[lineIndex])
							tmpText[lineIndex] = new wchar_t[maxWidth];
						
						tmpText[lineIndex][index] = 0;
					}

					tmpText[lineIndex][index] = text[ch];
					tmpText[lineIndex][index + 1] = 0;

					index++;
					ch++;
				}
			}

			longListChanged = false;
		}
		
		for (int i = 0; i < linesToDraw; i++)
		{
			if (tmpText[i])
				(font ? font : fontSystem)->DrawText(this->GetLeft(), this->GetTop() + i * lineheight, tmpText[i], c, style);
		}
	}
	else if (wrapMode == GuiText::SCROLLFULL) // Text full scroller
	{
		wchar_t save;
	
		if (scrollPos2 == 0 || FrameTimer > scrollDelay + 5)
		{
			scrollPos1 = 0;
			scrollOffset = maxWidth;
			scrollDelay = FrameTimer + 50; // Wait 50 frames before beginning with scrolling
			scrollPos2 = 1;
		}
		else if (scrollPos2 > 0 && FrameTimer >= scrollDelay)
		{
			if (--scrollOffset <= 0)
			{
				wchar_t tmp[] = {text[scrollPos1], text[scrollPos1 + 1], 0};

				scrollOffset += (font ? font : fontSystem)->GetWidth(tmp) - (font ? font : fontSystem)->GetWidth(tmp + 1);
				scrollPos1++;
			}
			
			int strlen = wcslen(text);

			for (; scrollPos2 < strlen; scrollPos2++)
			{
				save = text[scrollPos2 + 1]; // Save Pos2
				text[scrollPos2 + 1] = 0;

				int textWidth = (font ? font : fontSystem)->GetWidth(&text[scrollPos1]);

				text[scrollPos2 + 1] = save; // Restore Pos2

				if (textWidth + scrollOffset > maxWidth)
					break;
			}

			if (scrollPos1 == strlen)
				scrollPos2 = 0;
			else
				scrollDelay = FrameTimer + 1; // Wait 1 frame
		}
		
		uint16_t drawStyle = style;
		uint16_t drawX = this->GetLeft() + scrollOffset;
		
		if ((drawStyle & FTGX_JUSTIFY_MASK) == FTGX_JUSTIFY_CENTER)
		{
			drawStyle = (drawStyle & ~FTGX_JUSTIFY_MASK) | FTGX_JUSTIFY_LEFT;
			drawX -= maxWidth >> 1;
		}
		else if ((drawStyle & FTGX_JUSTIFY_MASK) == FTGX_JUSTIFY_RIGHT)
		{
			drawStyle = (drawStyle & ~FTGX_JUSTIFY_MASK) | FTGX_JUSTIFY_LEFT;
			drawX -= maxWidth;
		}
		
		save = text[abs(scrollPos2)]; // Save Pos2
		text[abs(scrollPos2)] = 0;

		if (scrollPos2 > 1)
			(font ? font : fontSystem)->DrawText(drawX, this->GetTop() + voffset, &text[scrollPos1], c, drawStyle);

		text[abs(scrollPos2)] = save; // Restore Pos2
	}
	else if (maxWidth > 0 && (font ? font : fontSystem)->GetWidth(text) > maxWidth)
	{
		if (wrapMode == GuiText::WRAP) // Text wrapping
		{
			int 	lineheight = newSize + 6;
			int 	strlen = wcslen(text);
			int 	i = 0;
			int 	ch = 0;
			int 	linenum = 0;
			int 	linemax = 200;
			int 	lastSpace = -1;
			int 	lastSpaceIndex = -1;
			wchar_t *wraptext[linemax];
			
			totalLines = 0;

			while (ch < strlen)
			{
				if (i == 0)
				{
					if (linenum <= linemax)
						wraptext[linenum] = new wchar_t[strlen + 1];
					else 
						break;
				}
					
				wraptext[linenum][i] = text[ch];
				wraptext[linenum][i + 1] = 0;

				if ((font ? font : fontSystem)->GetWidth(wraptext[linenum]) >= maxWidth)
				{
					if (lastSpace >= 0)
					{
						wraptext[linenum][lastSpaceIndex] = 0;	// Discard space and everything after
						ch = lastSpace;							// Go backwards to the last space
						lastSpace = -1;							// We have used this space
						lastSpaceIndex = -1;
					}

					linenum++;
					
					if (linenum == numLines && ch + 1 < strlen)
					{
						wraptext[linenum - 1][i - 3] = '.';
						wraptext[linenum - 1][i - 2] = '.';
						wraptext[linenum - 1][i - 1] = '.';
						wraptext[linenum - 1][i] = 0;

						break;
					}
					
					i = -1;
				}
				else if (ch == strlen - 1)
					linenum++;
				
				if ((text[ch] == ' ' || text[ch] == ',') && i >= 0)
				{
					lastSpace = ch;
					lastSpaceIndex = i;
				}

				if (text[ch] == '\n' && ch != strlen - 1 && i >= 0)
				{
					linenum++;
					i = -1;
				}

				ch++;
				i++;
			}
			
			totalLines = linenum;
			
			if (alignmentVert == ALIGN_MIDDLE)
				voffset = voffset - (lineheight * linenum) / 2 + lineheight / 2;

			if (numLines < 0)
			{
				for (i = 0; i < linenum; i++)
				{
					(font ? font : fontSystem)->DrawText(this->GetLeft(), this->GetTop() + voffset + i * lineheight, wraptext[i], c, style);
					
					delete wraptext[i]; 
				}
			}
			else
			{
				// Put in for txt vertical txt scrolling
				for (i = 0; i < numLines; i++)
				{
					if (i < linenum)
						(font ? font : fontSystem)->DrawText(this->GetLeft(), this->GetTop() + voffset + i * lineheight, wraptext[i], c, style);
				}

				for (i = 0; i < linenum; i++) 
					delete wraptext[i];
			}
		}
		else if (wrapMode == GuiText::DOTTED) // Text dotted
		{
			wchar_t save[4];
			int 	strlen = wcslen(text);
			int 	dotPos=strlen - 3;
			int 	i;
			bool 	drawed = false;

			while (dotPos > 0 && !drawed)
			{
				for (i = 0; i < 4; i++) // Save Text for "..."
				{
					save[i] = text[dotPos + i];
					text[dotPos + i] = (i != 3 ? _TEXT('.') : 0);
				}

				if (((font ? font : fontSystem)->GetWidth(text)) <= maxWidth)
				{
					(font ? font : fontSystem)->DrawText(this->GetLeft(), this->GetTop() + voffset, text, c, style);
					drawed = true;
				}

				for (i = 0; i < 4; i++) // Write saved Text back
					text[dotPos + i] = save[i];

				dotPos--;
			}

			if (!drawed)
				(font ? font : fontSystem)->DrawText(this->GetLeft(), this->GetTop() + voffset, text, c, style);
		}
		else if (wrapMode == GuiText::SCROLL) // Text scroller
		{
			wchar_t save;
		
			if (scrollPos2 == 0 || FrameTimer > scrollDelay + 5)
			{
				scrollPos1 = 0;
				scrollOffset = 0;

				for (scrollPos2 = wcslen(text); scrollPos2 > 1; scrollPos2--)
				{
					save = text[scrollPos2]; // Save Pos2
					text[scrollPos2] = 0;
					
					int textWidth = (font ? font : fontSystem)->GetWidth(text);
					
					text[scrollPos2] = save; // Restore Pos2
					
					if (textWidth <= maxWidth)
						break;
				}
				scrollDelay = FrameTimer + 50; // Wait 50 frames before beginning with scrolling
			}
			else if (scrollPos2 > 0 && FrameTimer >= scrollDelay)
			{
				
				if (--scrollOffset < 0)
				{
					wchar_t tmp[] = {text[scrollPos1], text[scrollPos1 + 1], 0};
					
					scrollOffset += (font ? font : fontSystem)->GetWidth(tmp) - (font ? font : fontSystem)->GetWidth(tmp + 1);
					scrollPos1++;
				}
				
				int strlen = wcslen(text);

				for (; scrollPos2 < strlen; scrollPos2++)
				{
					save = text[scrollPos2 + 1]; // Save Pos2
					text[scrollPos2 + 1] = 0;

					int textWidth = (font ? font : fontSystem)->GetWidth(&text[scrollPos1]);

					text[scrollPos2 + 1] = save; // Restore Pos2

					if (textWidth + scrollOffset > maxWidth)
						break;
				}

				if (scrollPos2 == strlen)
				{
					scrollPos2 = -scrollPos2;
					scrollDelay = FrameTimer + 25; // When dir-change wait 25 frames
				}
				else
					scrollDelay = FrameTimer + 1; // Wait 1 frame
			}
			else if (FrameTimer >= scrollDelay)
			{
				scrollPos2 = -scrollPos2;
				
				scrollOffset++;
				
				wchar_t tmp[] = {text[scrollPos1 - 1], text[scrollPos1], 0};
				int 	tmpOffset = (font ? font : fontSystem)->GetWidth(tmp) - (font ? font : fontSystem)->GetWidth(tmp + 1);
				
				if (scrollOffset >= tmpOffset)
				{
					scrollOffset -= tmpOffset;
					scrollPos1--;
				}

				for (; scrollPos2 > scrollPos1; scrollPos2--)
				{
					save = text[scrollPos2]; // Save Pos2
					text[scrollPos2] = 0;

					int textWidth = (font ? font : fontSystem)->GetWidth(&text[scrollPos1]);

					text[scrollPos2] = save; // Restore Pos2

					if (textWidth + scrollOffset <= maxWidth)
						break;
				}
				
				if (scrollPos1 == 0)
				{
					scrollPos2 = -scrollPos2;
					scrollDelay = FrameTimer + 25; // When dir-change wait 25 frames
				}
				else
					scrollDelay = FrameTimer + 1; // Wait 10 frames

				scrollPos2 = -scrollPos2;
			}
			
			uint16_t drawStyle = style;
			uint16_t drawX = this->GetLeft() + scrollOffset;
			
			if ((drawStyle & FTGX_JUSTIFY_MASK) == FTGX_JUSTIFY_CENTER)
			{
				drawStyle = (drawStyle & ~FTGX_JUSTIFY_MASK) | FTGX_JUSTIFY_LEFT;
				drawX -= maxWidth >> 1;
			}
			else if ((drawStyle & FTGX_JUSTIFY_MASK) == FTGX_JUSTIFY_RIGHT)
			{
				drawStyle = (drawStyle & ~FTGX_JUSTIFY_MASK) | FTGX_JUSTIFY_LEFT;
				drawX -= maxWidth;
			}
			
			save = text[abs(scrollPos2)]; // Save Pos2
			text[abs(scrollPos2)] = 0;
			(font ? font : fontSystem)->DrawText(drawX, this->GetTop() + voffset, &text[scrollPos1], c, drawStyle);
			text[abs(scrollPos2)] = save; // Restore Pos2
		}
		else
			(font ? font : fontSystem)->DrawText(this->GetLeft(), this->GetTop() + voffset, text, c, style);
	}
	else
		(font ? font : fontSystem)->DrawText(this->GetLeft(), this->GetTop() + voffset, text, c, style);

	this->UpdateEffects();
}
