///////////////////////////////////////////////////////
// LibWiiGUI Copyright 2009
// by Tantric
///////////////////////////////////////////////////////

///////////////////////////////////////////////////////
// GUI option browser class definitions
// Display a list of menu options
///////////////////////////////////////////////////////

// Includes
#include <unistd.h>

#include "GUI.h"
#include "Main.h"

// Class constructor
GuiOptionBrowser::GuiOptionBrowser(int w, int h, OptionList *l)
{
	width = w;
	height = h;
	options = l;
	selectable = true;
	listOffset = this->FindMenuItem(-1, 1);
	listChanged = true; // Trigger an initial list update
	selectedItem = 0;
	focus = 0; // Allow focus

	// Sounds
	btnSoundOver = new GuiSound(ButtonOver_pcm, ButtonOver_pcm_size, SOUND_PCM, cfg.SFXVolume);
	btnSoundClick = new GuiSound(ButtonClick_pcm, ButtonClick_pcm_size, SOUND_PCM, cfg.SFXVolume);

	// Triggers
	triggerA = new GuiTrigger;
	triggerA->SetSimpleTrigger(-1, WPAD_BUTTON_A | WPAD_CLASSIC_BUTTON_A, PAD_BUTTON_A);

	triggerHeldA = new GuiTrigger;
	triggerHeldA->SetHeldTrigger(-1, WPAD_BUTTON_A | WPAD_CLASSIC_BUTTON_A, PAD_BUTTON_A);

	// Objects
	bgOptions = new GuiImageData(BackgroundOptionsSettings_png);
	bgOptionsImg = new GuiImage(bgOptions);
	bgOptionsImg->SetParent(this);
	bgOptionsImg->SetAlignment(ALIGN_CENTER, ALIGN_MIDDLE);

	bgOptionsEntry = new GuiImageData(BackgroundOptionsEntry_png);

	for (int i = 0; i < PAGESIZE; i++)
	{
		optionBg[i] = new GuiImage(bgOptionsEntry);

		optionTxt[i] = new GuiText((char*) NULL, 20, (GXColor){0, 0, 0, 0xff});
		optionTxt[i]->SetAlignment(ALIGN_LEFT, ALIGN_MIDDLE);
		optionTxt[i]->SetPosition(25, 0);
		optionTxt[i]->SetMaxWidth(210, GuiText::SCROLL);

		optionVal[i] = new GuiText((char*) NULL, 20, (GXColor){0, 0, 0, 0xff});
		optionVal[i]->SetAlignment(ALIGN_LEFT, ALIGN_MIDDLE);
		optionVal[i]->SetPosition(240, 0);
		optionVal[i]->SetMaxWidth(235, GuiText::SCROLL);

		optionBtn[i] = new GuiButton(512, 30);
		optionBtn[i]->SetParent(this);
		optionBtn[i]->SetLabel(optionTxt[i], 0);
		optionBtn[i]->SetLabel(optionVal[i], 1);
		optionBtn[i]->SetImageOver(optionBg[i]);
		optionBtn[i]->SetPosition(-55, 32 * i + 12);
		optionBtn[i]->SetTrigger(triggerA);
		optionBtn[i]->SetSoundClick(btnSoundClick);
		optionBtn[i]->SetRumble(false);
	}
}

// Class destructor
GuiOptionBrowser::~GuiOptionBrowser()
{
	delete bgOptionsImg;
	delete bgOptions;
	delete bgOptionsEntry;
	delete triggerA;
	delete triggerHeldA;
	delete btnSoundOver;
	delete btnSoundClick;

	for (int i = 0; i < PAGESIZE; i++)
	{
		delete optionTxt[i];
		delete optionVal[i];
		delete optionBg[i];
		delete optionBtn[i];
	}
}

void GuiOptionBrowser::SetFocus(int f)
{
	focus = f;

	for (int i = 0; i < PAGESIZE; i++)
		optionBtn[i]->ResetState();

	if (f == 1)
		optionBtn[selectedItem]->SetState(STATE_SELECTED);
}

void GuiOptionBrowser::ResetState()
{
	if (state != STATE_DISABLED)
	{
		state = STATE_DEFAULT;
		stateChan = -1;
	}

	for (int i = 0; i < PAGESIZE; i++)
		optionBtn[i]->ResetState();
}

int GuiOptionBrowser::GetClickedOption()
{
	int found = -1;
	
	for (int i = 0; i < PAGESIZE; i++)
	{
		if (optionBtn[i]->GetState() == STATE_CLICKED)
		{
			optionBtn[i]->SetState(STATE_SELECTED);
			found = optionIndex[i];

			break;
		}
	}

	return found;
}

int GuiOptionBrowser::GetSelectedOption()
{
	int found = -1;

	for (int i = 0; i < PAGESIZE; i++)
	{
		if (optionBtn[i]->GetState() == STATE_SELECTED)
		{
			optionBtn[i]->SetState(STATE_SELECTED);
			found = optionIndex[i];
			
			break;
		}
	}
	
	return found;
}

// Help function to find the next visible menu item on the list
int GuiOptionBrowser::FindMenuItem(int currentItem, int direction)
{
	int nextItem = currentItem + direction;

	if (nextItem < 0 || nextItem >= options->length)
		return -1;

	if (strlen(options->name[nextItem]) > 0)
		return nextItem;
	else
		return FindMenuItem(nextItem, direction);
}

// Draw the option browser on screen
void GuiOptionBrowser::Draw()
{
	if (!this->IsVisible())
		return;

	bgOptionsImg->Draw();

	int next = listOffset;

	for (int i = 0; i < PAGESIZE; i++)
	{
		if (next >= 0)
		{
			optionBtn[i]->Draw();
			next = this->FindMenuItem(next, 1);
		}
		else
			break;
	}
	
	this->UpdateEffects();
}

void GuiOptionBrowser::TriggerUpdate()
{
	listChanged = true;
}

void GuiOptionBrowser::Update(GuiTrigger *t)
{
	int next, prev, lang = options->length;
	int startat = 0, loaded = 0;

	if ((loaded == 0) && (startat > 0))
	{
		if (startat > (lang - 8))
		{
			listOffset= (lang - 8);
			selectedItem = startat;
			optionBtn[selectedItem]->SetState(STATE_SELECTED, t->chan);
		}
		else if (startat < 8)
		{
			selectedItem = startat;
			optionBtn[selectedItem]->SetState(STATE_SELECTED, t->chan);
		}
		else
		{
			listOffset = (startat - 4);
			selectedItem = startat;
			optionBtn[selectedItem]->SetState(STATE_SELECTED, t->chan);
		}
		
		this->SetFocus(1);
		
		loaded = 1;
	}
	
	if (state == STATE_DISABLED || !t)
		return;
	
	next = listOffset;

	if (listChanged)
	{
		listChanged = false;
		
		for (int i = 0; i < PAGESIZE; i++)
		{
			if (next >= 0)
			{
				if (optionBtn[i]->GetState() == STATE_DISABLED)
				{
					optionBtn[i]->SetVisible(true);
					optionBtn[i]->SetState(STATE_DEFAULT);
				}

				optionTxt[i]->SetText(options->name[next]);
				optionVal[i]->SetText(options->value[next]);
				optionIndex[i] = next;

				next = this->FindMenuItem(next, 1);
			}
			else
			{
				optionBtn[i]->SetVisible(false);
				optionBtn[i]->SetState(STATE_DISABLED);
			}
		}
	}

	for (int i = 0; i < PAGESIZE; i++)
	{
		if (focus)
		{
			if (i != selectedItem && optionBtn[i]->GetState() == STATE_SELECTED)
				optionBtn[i]->ResetState();
			else if (i == selectedItem && optionBtn[i]->GetState() == STATE_DEFAULT)
				optionBtn[selectedItem]->SetState(STATE_SELECTED, t->chan);
		}

		optionBtn[i]->Update(t);
		
		if (optionBtn[i]->GetState() == STATE_SELECTED)
			selectedItem = i;
	}

	// PAD/Joystick navigation
	if (!focus)
		return; // Skip navigation

	if (t->Down())
	{
		next = this->FindMenuItem(optionIndex[selectedItem], 1);

		if (next >= 0)
		{
			if (selectedItem == PAGESIZE - 1)
			{
				listOffset = this->FindMenuItem(listOffset, 1); // Move list down by 1
				listChanged = true;
			}
			else if (optionBtn[selectedItem + 1]->IsVisible())
			{
				optionBtn[selectedItem]->ResetState();
				optionBtn[selectedItem + 1]->SetState(STATE_SELECTED, t->chan);

				selectedItem++;
			}
		}
	}
	else if (t->Up())
	{
		prev = this->FindMenuItem(optionIndex[selectedItem], -1);

		if (prev >= 0)
		{
			if (selectedItem == 0)
			{
				listOffset = prev; // Move list up by 1
				listChanged = true;
			}
			else
			{
				optionBtn[selectedItem]->ResetState();
				optionBtn[selectedItem - 1]->SetState(STATE_SELECTED, t->chan);

				selectedItem--;
			}
		}
	}

	if (updateCB)
		updateCB(this);
}
