///////////////////////////////////////////////////////
// LibWiiGUI Copyright 2009
// by Tantric
///////////////////////////////////////////////////////

///////////////////////////////////////////////////////
// GUI keyboard class definitions
// On-screen keyboard
///////////////////////////////////////////////////////

// Includes
#include "GUI.h"
#include "Main.h"

// Variables
static char tmptxt[MAX_KEYBOARD_DISPLAY];

static char *GetDisplayText(char *t)
{
	if (!t)
		return NULL;

	int len = strlen(t);

	if (len < MAX_KEYBOARD_DISPLAY)
		return t;

	int n = 0;
	
    for (int i = len - MAX_KEYBOARD_DISPLAY + 1; i < len; i++)
    {
        tmptxt[n] = t[i];
        tmptxt[n+1] = '\0';
        n++;
    }

	return tmptxt;
}

// Class constructor
GuiKeyboard::GuiKeyboard(char *t, u32 max)
{
	width = 540;
	height = 400;
	shift = 0;
	caps = 0;
	selectable = true;
	focus = 0; // Allow focus
	alignmentHor = ALIGN_CENTER;
	alignmentVert = ALIGN_MIDDLE;

	strncpy(kbTextStr, t, max);

	kbTextStr[max] = 0;
	kbTextMaxLen = max;

	Key thekeys[4][11] =
	{
		{{'1','!'}, {'2','@'}, {'3','#'}, {'4','$'}, {'5','%'}, {'6','^'}, {'7','&'}, {'8','*'}, {'9','('}, {'0',')'}, {'\0','\0'}},
		{{'q','Q'}, {'w','W'}, {'e','E'}, {'r','R'}, {'t','T'}, {'y','Y'}, {'u','U'}, {'i','I'}, {'o','O'}, {'p','P'}, {'-','_'}},
		{{'a','A'}, {'s','S'}, {'d','D'}, {'f','F'}, {'g','G'}, {'h','H'}, {'j','J'}, {'k','K'}, {'l','L'}, {';',':'}, {'\'','"'}},
		{{'z','Z'}, {'x','X'}, {'c','C'}, {'v','V'}, {'b','B'}, {'n','N'}, {'m','M'}, {',','<'}, {'.','>'}, {'/','?'}, {'\0','\0'}}
	};

	memcpy(keys, thekeys, sizeof(thekeys));
	
	// Sounds
	keySoundOver = new GuiSound(ButtonOver_pcm, ButtonOver_pcm_size, SOUND_PCM, cfg.SFXVolume);
	keySoundClick = new GuiSound(ButtonClick_pcm, ButtonClick_pcm_size, SOUND_PCM, cfg.SFXVolume);

	// Triggers
	triggerA = new GuiTrigger;
	triggerA->SetSimpleTrigger(-1, WPAD_BUTTON_A | WPAD_CLASSIC_BUTTON_A, PAD_BUTTON_A);

	triggerB = new GuiTrigger;
	triggerB->SetButtonOnlyTrigger(-1, WPAD_BUTTON_B | WPAD_CLASSIC_BUTTON_B, PAD_BUTTON_B);

	// Objects	
	keyTextbox = new GuiImageData(KeyboardTextbox_png);
	keyTextboxImg = new GuiImage(keyTextbox);
	
	keyMedium = new GuiImageData(KeyboardMediumKey_png);
	keyBackImg = new GuiImage(keyMedium);

	keyMediumOver = new GuiImageData(KeyboardMediumKeyOver_png);
	keyBackOverImg = new GuiImage(keyMediumOver);

	keyLarge = new GuiImageData(KeyboardLargeKey_png);
	keySpaceImg = new GuiImage(keyLarge);

	keyLargeOver = new GuiImageData(KeyboardLargeKeyOver_png);
	keySpaceOverImg = new GuiImage(keyLargeOver);

	keyTextboxImg->SetAlignment(ALIGN_CENTER, ALIGN_TOP);
	keyTextboxImg->SetPosition(0, 36);

	kbText = new GuiText(GetDisplayText(kbTextStr), 20, (GXColor){0, 0, 0, 0xff});
	kbText->SetAlignment(ALIGN_CENTER, ALIGN_TOP);
	kbText->SetPosition(0, 49);

	keyBackText = new GuiText(GetText("Back"), 20, (GXColor){0, 0, 0, 0xff});
	keyBack = new GuiButton(keyMedium->GetWidth(), keyMedium->GetHeight());
	keyBack->SetImage(keyBackImg);
	keyBack->SetImageOver(keyBackOverImg);
	keyBack->SetLabel(keyBackText);
	keyBack->SetSoundOver(keySoundOver);
	keyBack->SetSoundClick(keySoundClick);
	keyBack->SetTrigger(triggerA);
	keyBack->SetPosition(10 * 42 + 40, 0 * 42 + 80);
	keyBack->SetEffectGrow();
	
	keyCapsImg = new GuiImage(keyMedium);
	keyCapsOverImg = new GuiImage(keyMediumOver);
	keyCapsText = new GuiText(GetText("Caps"), 20, (GXColor){0, 0, 0, 0xff});
	keyCaps = new GuiButton(keyMedium->GetWidth(), keyMedium->GetHeight());
	keyCaps->SetImage(keyCapsImg);
	keyCaps->SetImageOver(keyCapsOverImg);
	keyCaps->SetLabel(keyCapsText);
	keyCaps->SetSoundOver(keySoundOver);
	keyCaps->SetSoundClick(keySoundClick);
	keyCaps->SetTrigger(triggerA);
	keyCaps->SetPosition(0, 2 * 42 + 80);
	keyCaps->SetEffectGrow();

	keyShiftImg = new GuiImage(keyMedium);
	keyShiftOverImg = new GuiImage(keyMediumOver);
	keyShiftText = new GuiText(GetText("Shift"), 20, (GXColor){0, 0, 0, 0xff});
	keyShift = new GuiButton(keyMedium->GetWidth(), keyMedium->GetHeight());
	keyShift->SetImage(keyShiftImg);
	keyShift->SetImageOver(keyShiftOverImg);
	keyShift->SetLabel(keyShiftText);
	keyShift->SetSoundOver(keySoundOver);
	keyShift->SetSoundClick(keySoundClick);
	keyShift->SetTrigger(triggerA);
	keyShift->SetPosition(21, 3 * 42 + 80);
	keyShift->SetEffectGrow();
	
	keySpace = new GuiButton(keyLarge->GetWidth(), keyLarge->GetHeight());
	keySpace->SetImage(keySpaceImg);
	keySpace->SetImageOver(keySpaceOverImg);
	keySpace->SetSoundOver(keySoundOver);
	keySpace->SetSoundClick(keySoundClick);
	keySpace->SetTrigger(triggerA);
	keySpace->SetPosition(0, 4 * 42 + 80);
	keySpace->SetAlignment(ALIGN_CENTER, ALIGN_TOP);
	keySpace->SetEffectGrow();
	
	// Set Widescreen
	if (cfg.Widescreen == Widescreen_On)
	{
		keyTextboxImg->SetWidescreen(true);
		kbText->SetWidescreen(true);
		keyBackImg->SetWidescreen(true);
		keyBackOverImg->SetWidescreen(true);
		keyBackText->SetWidescreen(true);
		keyCapsImg->SetWidescreen(true);
		keyCapsOverImg->SetWidescreen(true);
		keyCapsText->SetWidescreen(true);
		keyShiftImg->SetWidescreen(true);
		keyShiftOverImg->SetWidescreen(true);
		keyShiftText->SetWidescreen(true);
		keySpaceImg->SetWidescreen(true);
		keySpaceOverImg->SetWidescreen(true);
	}

	// Append objects
	this->Append(keyTextboxImg);
	this->Append(kbText);
	this->Append(keyBack);
	this->Append(keyCaps);
	this->Append(keyShift);
	this->Append(keySpace);

	key = new GuiImageData(KeyboardKey_png);
	keyOver = new GuiImageData(KeyboardKeyOver_png);

	char txt[2] = {0, 0};

	for (int i = 0; i < 4; i++)
	{
		for (int j = 0; j < 11; j++)
		{
			if (keys[i][j].ch != '\0')
			{
				txt[0] = keys[i][j].ch;
				keyImg[i][j] = new GuiImage(key);
				keyImgOver[i][j] = new GuiImage(keyOver);
				keyTxt[i][j] = new GuiText(txt, 20, (GXColor){0, 0, 0, 0xff});
				keyTxt[i][j]->SetAlignment(ALIGN_CENTER, ALIGN_BOTTOM);
				keyTxt[i][j]->SetPosition(0, -10);
				keyBtn[i][j] = new GuiButton(key->GetWidth(), key->GetHeight());
				keyBtn[i][j]->SetImage(keyImg[i][j]);
				keyBtn[i][j]->SetImageOver(keyImgOver[i][j]);
				keyBtn[i][j]->SetSoundOver(keySoundOver);
				keyBtn[i][j]->SetSoundClick(keySoundClick);
				keyBtn[i][j]->SetTrigger(triggerA);
				keyBtn[i][j]->SetLabel(keyTxt[i][j]);
				keyBtn[i][j]->SetPosition(j * 42 + 21 * i + 40, i * 42 + 80);
				keyBtn[i][j]->SetEffectGrow();
				
				// Set Widescreen
				if (cfg.Widescreen == Widescreen_On)
				{
					keyImg[i][j]->SetWidescreen(true);
					keyImgOver[i][j]->SetWidescreen(true);
				}

				// Append objects
				this->Append(keyBtn[i][j]);
			}
		}
	}
}

// Class destructor
GuiKeyboard::~GuiKeyboard()
{
	delete kbText;
	delete keyTextbox;
	delete keyTextboxImg;
	delete keyCapsText;
	delete keyCapsImg;
	delete keyCapsOverImg;
	delete keyCaps;
	delete keyShiftText;
	delete keyShiftImg;
	delete keyShiftOverImg;
	delete keyShift;
	delete keyBackText;
	delete keyBackImg;
	delete keyBackOverImg;
	delete keyBack;
	delete keySpaceImg;
	delete keySpaceOverImg;
	delete keySpace;
	delete key;
	delete keyOver;
	delete keyMedium;
	delete keyMediumOver;
	delete keyLarge;
	delete keyLargeOver;
	delete keySoundOver;
	delete keySoundClick;
	delete triggerA;
	delete triggerB;

	for (int i = 0; i < 4; i++)
	{
		for (int j = 0; j < 11; j++)
		{
			if (keys[i][j].ch != '\0')
			{
				delete keyImg[i][j];
				delete keyImgOver[i][j];
				delete keyTxt[i][j];
				delete keyBtn[i][j];
			}
		}
	}
}

// Constantly called to allow the GuiKeyboard to respond to updated input data
void GuiKeyboard::Update(GuiTrigger *t)
{
	if (AllElements.size() == 0 || (state == STATE_DISABLED && parentElement))
		return;

	for (u8 i = 0; i < AllElements.size(); i++)
	{
		try
		{
			AllElements.at(i)->Update(t);
		}
		catch (const std::exception& e)
		{
		}
	}

	bool update = false;

	if (keySpace->GetState() == STATE_CLICKED)
	{
		if (strlen(kbTextStr) < kbTextMaxLen)
		{
			kbTextStr[strlen(kbTextStr)] = ' ';
			kbText->SetText(kbTextStr);
		}

		keySpace->SetState(STATE_SELECTED, t->chan);
	}
	else if (keyBack->GetState() == STATE_CLICKED)
	{
		kbTextStr[strlen(kbTextStr) - 1] = 0;
		kbText->SetText(GetDisplayText(kbTextStr));
		keyBack->SetState(STATE_SELECTED, t->chan);
	}
	else if (keyShift->GetState() == STATE_CLICKED)
	{
		shift ^= 1;
		keyShift->SetState(STATE_SELECTED, t->chan);
		update = true;
	}
	else if (keyCaps->GetState() == STATE_CLICKED)
	{
		caps ^= 1;
		keyCaps->SetState(STATE_SELECTED, t->chan);
		update = true;
	}

	char txt[2] = {0, 0};

startLoop:
	for (int i = 0; i < 4; i++)
	{
		for (int j = 0; j < 11; j++)
		{
			if (keys[i][j].ch != '\0')
			{
				if (update)
				{
					if (shift || caps)
						txt[0] = keys[i][j].chShift;
					else
						txt[0] = keys[i][j].ch;

					keyTxt[i][j]->SetText(txt);
				}

				if (keyBtn[i][j]->GetState() == STATE_CLICKED)
				{
					if (strlen(kbTextStr) < kbTextMaxLen)
					{
						if (shift || caps)
							kbTextStr[strlen(kbTextStr)] = keys[i][j].chShift;
						else
							kbTextStr[strlen(kbTextStr)] = keys[i][j].ch;
					}
					
					kbText->SetText(GetDisplayText(kbTextStr));
					keyBtn[i][j]->SetState(STATE_SELECTED, t->chan);

					if (shift)
					{
						shift ^= 1;
						update = true;
						
						goto startLoop;
					}
				}
			}
		}
	}

	this->ToggleFocus(t);

	if (focus) // Only send actions to this window if it's in focus
	{
		// PAD/Joystick navigation
		if (t->Right())
			this->MoveSelectionHor(1);
		else if (t->Left())
			this->MoveSelectionHor(-1);
		else if (t->Down())
			this->MoveSelectionVert(1);
		else if (t->Up())
			this->MoveSelectionVert(-1);
	}
}
