///////////////////////////////////////////////////////
// LibWiiGUI Copyright 2009
// by Tantric
///////////////////////////////////////////////////////

///////////////////////////////////////////////////////
// GUI image class definitions
// Display, manage and manipulate images in the GUI
///////////////////////////////////////////////////////

// Includes
#include "GUI.h"

// Class constructor
GuiImage::GuiImage()
{
	image = NULL;
	width = 0;
	height = 0;
	imageAngle = 0;
	tile = -1;
	stripe = 0;
	widescreen = 0;
	imgType = IMAGE_DATA;
}

// Sets up a new image
GuiImage::GuiImage(GuiImageData *img)
{
	image = NULL;
	width = 0;
	height = 0;
	
	if (img)
	{
		image = img->GetImage();
		width = img->GetWidth();
		height = img->GetHeight();
	}

	imageAngle = 0;
	tile = -1;
	stripe = 0;
	widescreen = 0;
	parentAngle = true;
	imgType = IMAGE_DATA;
}

// Sets up a new image from the image data specified
GuiImage::GuiImage(u8 *img, int w, int h)
{
	image = img;
	width = w;
	height = h;
	imageAngle = 0;
	tile = -1;
	stripe = 0;
	widescreen = 0;
	parentAngle = true;
	imgType = IMAGE_TEXTURE;
}

// Creates an image filled with the specified color
GuiImage::GuiImage(int w, int h, GXColor c)
{
	image = (u8 *)memalign(32, w * h * 4);
	width = w;
	height = h;
	imageAngle = 0;
	tile = -1;
	stripe = 0;
	widescreen = 0;
	parentAngle = true;
	imgType = IMAGE_COLOR;
	
	if (!image)
		return;

	int x, y;

	for (y = 0; y < h; y++)
	{
		for (x = 0; x < w; x++)
		{
			this->SetPixel(x, y, c);
		}
	}

	int len = w * h * 4;

	if (len % 32)
		len += (32 - len % 32);

	DCFlushRange(image, len);
}

// Class destructor
GuiImage::~GuiImage()
{
	if (imgType == IMAGE_COLOR && image)
		free(image);
}

// Gets the image data
u8 *GuiImage::GetImage()
{
	return image;
}

// Sets up a new image using the GuiImageData object specified
void GuiImage::SetImage(GuiImageData *img)
{
	image = NULL;
	width = 0;
	height = 0;
	
	if (img)
	{
		image = img->GetImage();
		width = img->GetWidth();
		height = img->GetHeight();
	}

	imgType = IMAGE_DATA;
}

// Sets up a new image using the GuiImageData object specified
void GuiImage::SetImage(u8 *img, int w, int h)
{
	image = img;
	width = w;
	height = h;
	imgType = IMAGE_TEXTURE;
}

// Sets up a new image using the GuiImageData object specified
void GuiImage::SetImage(int w, int h, GXColor c)
{
	image = (u8 *)memalign(32, w * h * 4);
	width = w;
	height = h;
	imgType = IMAGE_COLOR;
	
	if (!image)
		return;

	int x, y;

	for (y = 0; y < h; y++)
	{
		for (x = 0; x < w; x++)
			this->SetPixel(x, y, c);
	}

	int len = w * h * 4;
	
	if (len % 32)
		len += (32 - len % 32);
	
	DCFlushRange(image, len);
}

// Sets the image rotation angle for drawing
void GuiImage::SetAngle(float a)
{
	imageAngle = a;
}

// Sets the number of times to draw the image horizontally
void GuiImage::SetTile(int t)
{
	tile = t;
}

// Set horizontal scale to 0.8
void GuiImage::SetWidescreen(bool w)
{
	widescreen = w;
}

// Set or disable the use of parentelement angle (default true)
void GuiImage::SetParentAngle(bool a)
{
	parentAngle = a;
}

// Gets the pixel color at the specified coordinates of the image
GXColor GuiImage::GetPixel(int x, int y)
{
	if (!image || this->GetWidth() <= 0 || x < 0 || y < 0)
		return (GXColor){0, 0, 0, 0};

	u32 offset = (((y >> 2) << 4) * this->GetWidth()) + ((x >> 2) << 6) + (((y % 4 << 2) + x % 4) << 1);

	GXColor color;

	color.a = *(image + offset);
	color.r = *(image + offset + 1);
	color.g = *(image + offset + 32);
	color.b = *(image + offset + 33);

	return color;
}

// Sets the pixel color at the specified coordinates of the image
void GuiImage::SetPixel(int x, int y, GXColor color)
{
	if (!image || this->GetWidth() <= 0 || x < 0 || y < 0)
		return;

	u32 offset = (((y >> 2) << 4) * this->GetWidth()) + ((x >> 2) << 6) + (((y % 4 << 2) + x % 4) << 1);
	
	*(image + offset) = color.a;
	*(image + offset + 1) = color.r;
	*(image + offset + 32) = color.g;
	*(image + offset + 33) = color.b;
}

// Sets a stripe effect on the image, overlaying alpha blended rectangles
// Does not alter the image data
void GuiImage::SetStripe(int s)
{
	stripe = s;
}

// Directly modifies the image data to create a color-striped effect
// Alters the RGB values by the specified amount
void GuiImage::ColorStripe(int shift)
{
	int 	x, y;
	GXColor color;
	int 	alt = 0;

	for (y = 0; y < this->GetHeight(); y++)
	{
		if (y % 3 == 0)
			alt ^= 1;

		for (x=0; x < this->GetWidth(); x++)
		{
			color = GetPixel(x, y);

			if (alt)
			{
				if (color.r < 255 - shift)
					color.r += shift;
				else
					color.r = 255;

				if (color.g < 255 - shift)
					color.g += shift;
				else
					color.g = 255;

				if (color.b < 255 - shift)
					color.b += shift;
				else
					color.b = 255;

				color.a = 255;
			}
			else
			{
				if (color.r > shift)
					color.r -= shift;
				else
					color.r = 0;

				if (color.g > shift)
					color.g -= shift;
				else
					color.g = 0;

				if (color.b > shift)
					color.b -= shift;
				else
					color.b = 0;

				color.a = 255;
			}
			
			SetPixel(x, y, color);
		}
	}
	
	int len = width*height*4;
	
	if (len % 32)
		len += (32 - len % 32);
	
	DCFlushRange(image, len);
}

// Directly modifies the image data to change the image to grayscale
void GuiImage::Grayscale()
{
	GXColor color;
	u32 offset, gray;

	for (int x = 0; x < width; x++)
	{
		for (int y = 0; y < height; y++)
		{
			offset = (((y >> 2) << 4) * this->GetWidth()) + ((x >> 2) << 6) + (((y % 4 << 2) + x % 4) << 1);

			color.r = *(image + offset + 1);
			color.g = *(image + offset + 32);
			color.b = *(image + offset + 33);

			gray = (77 * color.r + 150 * color.g + 28 * color.b) / 255;

			*(image + offset + 1) = gray;
			*(image + offset + 32) = gray;
			*(image + offset + 33) = gray;
		}
	}

	int len = width * height * 4;

	if (len % 32)
		len += (32 - len % 32);

	DCFlushRange(image, len);
}

// Draw the image on screen
void GuiImage::Draw()
{
	if (!image || !this->IsVisible() || tile == 0)
		return;

	float currScale = this->GetScale();
	int	  currLeft = this->GetLeft();

	if (tile > 0)
	{
		for (int i = 0; i < tile; i++)
			Menu_DrawImg(currLeft + width * i, this->GetTop(), width, height, image, imageAngle, widescreen ? currScale * 0.8 : currScale, currScale, this->GetAlpha());
	}
	else
	{
		// Temporary used to correct offset for scaled images
		if (scale != 1)
			currLeft = currLeft - width / 2 + (width * scale) / 2;

		Menu_DrawImg(currLeft, this->GetTop(), width, height, image, imageAngle, widescreen ? currScale * 0.8 : currScale, currScale, this->GetAlpha());
	}

	if (stripe > 0)
		for (int y = 0; y < this->GetHeight(); y += 6)
			Menu_DrawRectangle(currLeft, this->GetTop() + y, this->GetWidth(), 3, (GXColor){0, 0, 0, stripe}, 1);

	this->UpdateEffects();
}
