///////////////////////////////////////////////////////
// LibWiiGUI Copyright 2009
// by Tantric
///////////////////////////////////////////////////////

///////////////////////////////////////////////////////
// GUI element class definitions
// Primary GUI class, most other classes inherit from this class
///////////////////////////////////////////////////////

// Includes
#include "GUI.h"

// Class constructor
GuiElement::GuiElement()
{
	xOffset = 0;
	yOffset = 0;
	xMin = 0;
	xMax = 0;
	yMin = 0;
	yMax = 0;
	width = 0;
	height = 0;
	alpha = 255;
	scale = 1;
	state = STATE_DEFAULT;
	stateChan = -1;
	trigger[0] = NULL;
	trigger[1] = NULL;
	trigger[2] = NULL;
	trigger[3] = NULL;
	trigger[4] = NULL;
	trigger[5] = NULL;
	parentElement = NULL;
	rumble = true;
	selectable = false;
	clickable = false;
	holdable = false;
	visible = true;
	focus = -1; // Cannot be focused
	updateCB = NULL;
	yOffsetDyn = 0;
	xOffsetDyn = 0;
	alphaDyn = -1;
	scaleDyn = 1;
	effects = 0;
	effectAmount = 0;
	effectTarget = 0;
	effectLoop = -1;
	effectsOver = 0;
	effectAmountOver = 0;
	effectTargetOver = 0;

	// Default alignment (align to top left)
	alignmentVert = ALIGN_TOP;
	alignmentHor = ALIGN_LEFT;
}

// Class destructor
GuiElement::~GuiElement()
{
}

// Set the element's parent
void GuiElement::SetParent(GuiElement *e)
{
	parentElement = e;
}

// Gets the element's parent
GuiElement *GuiElement::GetParent()
{
	return parentElement;
}

// Gets the current leftmost coordinate of the element
// Considers horizontal alignment, x offset, width and parent element's GetLeft() / GetWidth() values
int GuiElement::GetLeft()
{
	int x = 0;
	int pWidth = 0;
	int pLeft = 0;

	if (parentElement)
	{
		pWidth = parentElement->GetWidth();
		pLeft = parentElement->GetLeft();
	}

	if (effects & (EFFECT_SLIDE_IN | EFFECT_SLIDE_OUT))
		pLeft += xOffsetDyn;

	switch (alignmentHor)
	{
		case ALIGN_LEFT:
			x = pLeft;

			break;

		case ALIGN_CENTER:
			x = pLeft + (pWidth / 2) - (width / 2);

			break;

		case ALIGN_RIGHT:
			x = pLeft + pWidth - width;

			break;
	}

	return x + xOffset;
}

// Gets the current topmost coordinate of the element
// Considers vertical alignment, y offset, height and parent element's GetTop() / GetHeight() values
int GuiElement::GetTop()
{
	int y = 0;
	int pHeight = 0;
	int pTop = 0;

	if (parentElement)
	{
		pHeight = parentElement->GetHeight();
		pTop = parentElement->GetTop();
	}

	if (effects & (EFFECT_SLIDE_IN | EFFECT_SLIDE_OUT))
		pTop += yOffsetDyn;

	switch (alignmentVert)
	{
		case ALIGN_TOP:
			y = pTop;

			break;

		case ALIGN_MIDDLE:
			y = pTop + (pHeight / 2) - (height / 2);

			break;

		case ALIGN_BOTTOM:
			y = pTop + pHeight - height;

			break;
	}

	return y + yOffset;
}

// Sets the minimum x offset of the element
void GuiElement::SetMinX(int x)
{
	xMin = x;
}

// Gets the minimum x offset of the element
int GuiElement::GetMinX()
{
	return xMin;
}

// Sets the maximum x offset of the element
void GuiElement::SetMaxX(int x)
{
	xMax = x;
}

// Gets the maximum x offset of the element
int GuiElement::GetMaxX()
{
	return xMax;
}

// Sets the minimum y offset of the element
void GuiElement::SetMinY(int y)
{
	yMin = y;
}

// Gets the minimum y offset of the element
int GuiElement::GetMinY()
{
	return yMin;
}

// Sets the maximum y offset of the element
void GuiElement::SetMaxY(int y)
{
	yMax = y;
}

// Gets the maximum y offset of the element
int GuiElement::GetMaxY()
{
	return yMax;
}

// Gets the current width of the element. Does not currently consider the scale
int GuiElement::GetWidth()
{
	return width;
}

// Gets the height of the element. Does not currently consider the scale
int GuiElement::GetHeight()
{
	return height;
}

// Sets the size (width/height) of the element
void GuiElement::SetSize(int w, int h)
{

	width = w;
	height = h;
}

// Checks whether or not the element is visible
bool GuiElement::IsVisible()
{
	return visible;
}

// Checks whether or not the element is selectable
bool GuiElement::IsSelectable()
{
	if (state == STATE_DISABLED || state == STATE_CLICKED)
		return false;
	else
		return selectable;
}

// Checks whether or not the element is clickable
bool GuiElement::IsClickable()
{
	if (state == STATE_DISABLED || state == STATE_CLICKED || state == STATE_HELD)
		return false;
	else
		return clickable;
}

// Checks whether or not the element is holdable
bool GuiElement::IsHoldable()
{
	if (state == STATE_DISABLED)
		return false;
	else
		return holdable;
}

// Sets the element's visibility
void GuiElement::SetVisible(bool v)
{
	visible = v;
}

// Sets whether or not the element is selectable
void GuiElement::SetSelectable(bool s)
{
	selectable = s;
}

// Sets whether or not the element is clickable
void GuiElement::SetClickable(bool c)
{
	clickable = c;
}

// Sets whether or not the element is holdable
void GuiElement::SetHoldable(bool d)
{
	holdable = d;
}

// Sets the element's alpha value
void GuiElement::SetAlpha(int a)
{
	alpha = a;
}

// Gets the element's alpha value
// Considers alpha, alphaDyn and the parent element's GetAlpha() value
int GuiElement::GetAlpha()
{
	int a;

	if (alphaDyn >= 0)
		a = alphaDyn;
	else
		a = alpha;

	if (parentElement)
		a *= parentElement->GetAlpha() / 255.0;

	return a;
}

// Sets the element's scale
void GuiElement::SetScale(float s)
{
	scale = s;
}

// Gets the element's current scale
// Considers scale, scaleDyn and the parent element's GetScale() value
float GuiElement::GetScale()
{
	float s = scale * scaleDyn;

	if (parentElement)
		s *= parentElement->GetScale();

	return s;
}

// Gets the element's current state
int GuiElement::GetState()
{
	return state;
}

// Gets the controller channel that last changed the element's state
int GuiElement::GetStateChan()
{
	return stateChan;
}

// Sets the element's state
void GuiElement::SetState(int s, int c)
{
	state = s;
	stateChan = c;
}

// Resets the element's state to STATE_DEFAULT
void GuiElement::ResetState()
{
	if (state != STATE_DISABLED)
	{
		state = STATE_DEFAULT;
		stateChan = -1;
	}
}

// Sets the element's focus
void GuiElement::SetFocus(int f)
{
	focus = f;
}

// Checks whether the element is in focus
int GuiElement::IsFocused()
{
	return focus;
}

// Set a new GuiTrigger for the element
void GuiElement::SetTrigger(GuiTrigger *t)
{
	if (!trigger[0])
		trigger[0] = t;
	else if (!trigger[1])
		trigger[1] = t;
	else if (!trigger[2])
		trigger[2] = t;
	else if (!trigger[3])
		trigger[3] = t;
	else if (!trigger[4])
		trigger[4] = t;
	else if (!trigger[5])
		trigger[5] = t;
	else // All were assigned, so we'll just overwrite the first one
		trigger[0] = t;
}

// Overload
void GuiElement::SetTrigger(u8 i, GuiTrigger *t)
{
	trigger[i] = t;
}

// Checks whether rumble was requested by the element
bool GuiElement::Rumble()
{
	return rumble;
}

// Sets whether or not the element is requesting a rumble event
void GuiElement::SetRumble(bool r)
{
	rumble = r;
}

// Gets the current element effects
int GuiElement::GetEffect()
{
	return effects;
}

// Set an effect for the element
void GuiElement::SetEffect(int eff, int amount, int target, int loop)
{
	if (eff & EFFECT_SLIDE_IN)
	{
		// These calculations overcompensate a little
		if (eff & EFFECT_SLIDE_TOP)
			yOffsetDyn = -ScreenHeight;
		else if (eff & EFFECT_SLIDE_LEFT)
			xOffsetDyn = -ScreenWidth;
		else if (eff & EFFECT_SLIDE_BOTTOM)
			yOffsetDyn = ScreenHeight;
		else if (eff & EFFECT_SLIDE_RIGHT)
			xOffsetDyn = ScreenWidth;
	}
	
	if (eff & EFFECT_FADE && amount > 0)
		alphaDyn = 0;
	else if (eff & EFFECT_FADE && amount < 0)
		alphaDyn = alpha;
	
	if (eff & EFFECT_WARNING && amount > 0)
		alphaDyn = 0;
	else if (eff & EFFECT_WARNING && amount < 0)
		alphaDyn = alpha;

	effects |= eff;
	effectAmount = amount;
	effectTarget = target;
	effectLoop = loop;
}

// Sets an effect to be enabled on wiimote cursor over
void GuiElement::SetEffectOnOver(int eff, int amount, int target)
{
	effectsOver |= eff;
	effectAmountOver = amount;
	effectTargetOver = target;
}

// Shortcut to SetEffectOnOver(EFFECT_SCALE, 4, 110)
void GuiElement::SetEffectGrow()
{
	SetEffectOnOver(EFFECT_SCALE, 4, 110);
}

// Stops the current element effect
void GuiElement::StopEffect()
{
	xOffsetDyn = 0;
	yOffsetDyn = 0;
	effects = 0;
	effectsOver = 0;
	effectAmount = 0;
	effectAmountOver = 0;
	effectTarget = 0;
	effectTargetOver = 0;
	effectLoop = -1;
	scaleDyn = 1;
}

// Updates the element's effects (dynamic values)
// Called by Draw(), used for animation purposes
void GuiElement::UpdateEffects()
{
	if (effects & (EFFECT_SLIDE_IN | EFFECT_SLIDE_OUT))
	{
		if (effects & EFFECT_SLIDE_IN)
		{
			if (effects & EFFECT_SLIDE_LEFT)
			{
				xOffsetDyn += effectAmount;

				if (xOffsetDyn >= 0)
				{
					xOffsetDyn = 0;
					effects = 0;
				}
			}
			else if (effects & EFFECT_SLIDE_RIGHT)
			{
				xOffsetDyn -= effectAmount;

				if (xOffsetDyn <= 0)
				{
					xOffsetDyn = 0;
					effects = 0;
				}
			}
			else if (effects & EFFECT_SLIDE_TOP)
			{
				yOffsetDyn += effectAmount;

				if (yOffsetDyn >= 0)
				{
					yOffsetDyn = 0;
					effects = 0;
				}
			}
			else if (effects & EFFECT_SLIDE_BOTTOM)
			{
				yOffsetDyn -= effectAmount;

				if (yOffsetDyn <= 0)
				{
					yOffsetDyn = 0;
					effects = 0;
				}
			}
		}
		else
		{
			if (effects & EFFECT_SLIDE_LEFT)
			{
				xOffsetDyn -= effectAmount;

				if (xOffsetDyn <= -ScreenWidth)
					effects = 0; // Shut off effect
			}
			else if (effects & EFFECT_SLIDE_RIGHT)
			{
				xOffsetDyn += effectAmount;

				if (xOffsetDyn >= ScreenWidth)
					effects = 0; // Shut off effect
			}
			else if (effects & EFFECT_SLIDE_TOP)
			{
				yOffsetDyn -= effectAmount;

				if (yOffsetDyn <= -ScreenHeight)
					effects = 0; // Shut off effect
			}
			else if (effects & EFFECT_SLIDE_BOTTOM)
			{
				yOffsetDyn += effectAmount;

				if (yOffsetDyn >= ScreenHeight)
					effects = 0; // Shut off effect
			}
		}
	}

	if (effects & EFFECT_FADE)
	{
		alphaDyn += effectAmount;

		if (effectAmount < 0 && alphaDyn <= 0)
		{
			alphaDyn = 0;
			effects = 0; // Shut off effect
		}
		else if (effectAmount > 0 && alphaDyn >= alpha)
		{
			alphaDyn = alpha;
			effects = 0; // Shut off effect
		}
	}

	if (effects & EFFECT_WARNING)
	{
		if (effectAmount < 0)
		{
			if (alphaDyn >= effectTarget)
				alphaDyn += effectAmount;
			else if (alphaDyn < effectTarget)
				effectAmount = -effectAmount;
		}
		else if (effectAmount > 0)
		{
			if (alphaDyn < alpha)
				alphaDyn += effectAmount;
			else if (alphaDyn >= alpha)
			{
				effectAmount = -effectAmount;

				if (effectLoop > 0)
					effectLoop--;
			}
		}
		
		if (effectLoop == 0)
			effects = 0; // Shut off effect
	}

	if (effects & EFFECT_SCALE)
	{
		scaleDyn += effectAmount/100.0;

		if ((effectAmount < 0 && scaleDyn <= effectTarget / 100.0) || (effectAmount > 0 && scaleDyn >= effectTarget / 100.0))
		{
			scaleDyn = effectTarget / 100.0;
			effects = 0; // Shut off effect
		}
	}
}

// Called constantly to allow the element to respond to the current input data
void GuiElement::Update(GuiTrigger *t)
{
	if (updateCB)
		updateCB(this);
}

// Sets a function to called after after Update()
// Callback function can be used to response to changes in the state of the element and/or update the element's attributes
void GuiElement::SetUpdateCallback(UpdateCallback u)
{
	updateCB = u;
}

// Sets the element's position
void GuiElement::SetPosition(int xoff, int yoff)
{
	xOffset = xoff;
	yOffset = yoff;
}

// Sets the element's alignment respective to its parent element
void GuiElement::SetAlignment(int hor, int vert)
{
	alignmentHor = hor;
	alignmentVert = vert;
}

// Gets whether or not the element is in STATE_SELECTED
int GuiElement::GetSelected()
{
	return -1;
}

// Draw an element on screen
void GuiElement::Draw()
{
}

// Draw Tooltips on screen
void GuiElement::DrawTooltip()
{
}

// Checks whether the specified coordinates are within the element's boundaries
bool GuiElement::IsInside(int x, int y)
{
	return (x > this->GetLeft() && x < (this->GetLeft() + width) && y > this->GetTop() && y < (this->GetTop() + height));
}
