///////////////////////////////////////////////////////
// LibWiiGUI is a GUI library for the Wii, created to help structure the
// design of a complicated GUI interface and to enable an author to create
// a sophisticated, feature-rich GUI. It was originally conceived and written
// after I started to design a GUI for Snes9x GX and found libwiisprite and
// GRRLIB inadequate for the purpose. It uses GX for drawing and makes use
// of PNGU for displaying images and FreeTypeGX for text. It was designed to
// be flexible and is easy to modify - don't be afraid to change the way it
// works or expand it to suit your GUI's purposes! If you do and you think
// your changes might benefit others, please share them so they might be
// added to the project!
//
// Quickstart
// Start from the supplied template example. For more advanced uses, see the
// source code for Snes9x GX, FCE Ultra GX and Visual Boy Advance GX.
//
// Contact
// If you have any suggestions for the library or documentation, or want to
// contribute, please visit the LibWiiGUI website:
// http://code.google.com/p/libwiigui/
//
// Credits
// This library was wholly designed and written by Tantric. Thanks to the
// authors of PNGU and FreeTypeGX, of which this library makes use. Thanks
// also to the authors of GRRLIB and LibWiiSprite for laying the foundations.
///////////////////////////////////////////////////////
#ifndef LIBWIIGUI_H
	#define LIBWIIGUI_H

	// Includes
	#include <gccore.h>
	#include <unistd.h>
	#include <malloc.h>
	#include <stdlib.h>
	#include <string.h>
	#include <vector>
	#include <exception>
	#include <wchar.h>
	#include <math.h>
	#include <asndlib.h>
	#include <wiiuse/wpad.h>

	#include "Libs/PNGU/PNGU.h"
	#include "System/Sound/OGGPlayer.h"
	#include "System/Input.h"
	#include "System/FreeTypeGX.h"
	#include "System/Video.h"
	#include "FileList.h"

	// Extern variables
	extern FreeTypeGX *fontSystem;

	// Constants
	#define SCROLL_INITIAL_DELAY 	20
	#define SCROLL_LOOP_DELAY		3
	#define SAVE_PAGESIZE 			4
	#define MII_PAGESIZE 			9
	#define DEVICE_PAGESIZE 		9
	#define CUSTOM_PAGESIZE 		9
	#define PAGESIZE 				8
	#define MAX_OPTIONS 			30
	#define MAX_SAVES 				200
	#define MAX_KEYBOARD_DISPLAY	32
	#define MAX_LINES_TO_DRAW	    10

	typedef void (*UpdateCallback)(void *e);

	// Enumerations
	enum
	{
		ALIGN_LEFT,
		ALIGN_RIGHT,
		ALIGN_CENTER,
		ALIGN_TOP,
		ALIGN_BOTTOM,
		ALIGN_MIDDLE
	};

	enum
	{
		STATE_DEFAULT,
		STATE_SELECTED,
		STATE_CLICKED,
		STATE_HELD,
		STATE_DISABLED
	};

	enum
	{
		SOUND_PCM,
		SOUND_OGG
	};

	enum
	{
		IMAGE_TEXTURE,
		IMAGE_COLOR,
		IMAGE_DATA
	};

	enum
	{
		TRIGGER_SIMPLE,
		TRIGGER_HELD,
		TRIGGER_BUTTON_ONLY,
		TRIGGER_BUTTON_ONLY_IN_FOCUS
	};

	// Structures
	typedef struct _paddata
	{
		u16	btns_d;
		u16	btns_u;
		u16	btns_h;
		s8	stickX;
		s8	stickY;
		s8	substickX;
		s8	substickY;
		u8	triggerL;
		u8	triggerR;
	} PADData;

	// Constants
	#define EFFECT_SLIDE_TOP			1
	#define EFFECT_SLIDE_BOTTOM			2
	#define EFFECT_SLIDE_RIGHT			4
	#define EFFECT_SLIDE_LEFT			8
	#define EFFECT_SLIDE_IN				16
	#define EFFECT_SLIDE_OUT			32
	#define EFFECT_FADE					64
	#define EFFECT_SCALE				128
	#define EFFECT_COLOR_TRANSITION		256
	#define EFFECT_WARNING				512

	// Classes
	class GuiSound
	{
		public:
			GuiSound(const u8 *, s32, int, int);
			~GuiSound(void);

			void Play(void);
			void Stop(void);
			void Pause(void);
			void Resume(void);
			bool IsPlaying(void);
			void SetVolume(int);
			void SetLoop(bool);

		protected:
			const u8 *sound;	// Pointer to the sound data
			int 	 type; 		// Sound format type (SOUND_PCM or SOUND_OGG)
			s32 	 length; 	// Length of sound data
			s32 	 voice; 	// Currently assigned ASND voice channel
			s32 	 volume; 	// Sound volume (0-100)
			bool 	 loop; 		// Loop sound playback
	};

	class GuiTrigger
	{
		// Public members
		public:
			GuiTrigger(void);
			~GuiTrigger(void);
			
			void SetSimpleTrigger(s32, u32, u16);
			void SetHeldTrigger(s32, u32, u16);
			void SetButtonOnlyTrigger(s32, u32, u16);
			void SetButtonOnlyInFocusTrigger(s32, u32, u16);
			s8   WPAD_Stick(u8, int);
			bool Left(void);
			bool Right(void);
			bool Up(void);
			bool Down(void);

			u8 		 type; 		// Trigger type (TRIGGER_SIMPLE, TRIGGER_HELD, TRIGGER_BUTTON_ONLY, TRIGGER_BUTTON_ONLY_IN_FOCUS)
			s32 	 chan; 		// Trigger controller channel (0-3, -1 for all)
			WPADData *wPad; 	// Wii controller trigger
			WPADData wPadData; 	// Wii controller trigger data
			PADData  pad; 		// GameCube controller trigger data
	};

	// Extern variables
	extern GuiTrigger userInput[4];

	class GuiElement
	{
		public:
			GuiElement(void);
			~GuiElement(void);
			
			void 		SetParent(GuiElement *);
			GuiElement *GetParent(void);
			int			GetLeft(void);
			int			GetTop(void);
			void		SetMinY(int);
			int			GetMinY(void);
			void		SetMaxY(int);
			int			GetMaxY(void);
			void 		SetMinX(int);
			int			GetMinX(void);
			void		SetMaxX(int);
			int			GetMaxX(void);
			int			GetWidth(void);
			int			GetHeight(void);
			void		SetSize(int, int);
			bool		IsVisible(void);
			bool		IsSelectable(void);
			bool		IsClickable(void);
			bool		IsHoldable(void);
			void		SetSelectable(bool);
			void		SetClickable(bool);
			void		SetHoldable(bool);
			int			GetState(void);
			int			GetStateChan(void);
			void		SetAlpha(int);
			int 		GetAlpha(void);
			void 		SetScale(float);
			float 		GetScale(void);
			void  		SetTrigger(GuiTrigger *);
			void  		SetTrigger(u8, GuiTrigger *);
			bool  		Rumble(void);
			void  		SetRumble(bool);
			void  		SetEffect(int, int, int t = 0, int l = -1);
			void  		SetEffectOnOver(int, int, int t = 0);
			void  		SetEffectGrow(void);
			void  		StopEffect(void);
			int   		GetEffect(void);
			bool  		IsInside(int, int);
			void  		SetPosition(int, int);
			void  		UpdateEffects(void);
			void  		SetUpdateCallback(UpdateCallback);
			int   		IsFocused(void);

			virtual void SetVisible(bool);
			virtual void SetFocus(int);
			virtual void SetState(int, int c = -1);
			virtual void ResetState(void);
			virtual int  GetSelected(void);
			virtual void SetAlignment(int, int);
			virtual void Update(GuiTrigger *);
			virtual void Draw(void);
			virtual void DrawTooltip(void);
			
		protected:
			bool 			visible; 			// Visibility of the element. If false, Draw() is skipped
			int 			focus; 				// Element focus (-1 = focus disabled, 0 = not focused, 1 = focused)
			int 			width; 				// Element width
			int 			height; 			// Element height
			int 			xOffset; 			// Element X offset
			int 			yOffset; 			// Element Y offset
			int 			yMin; 				// Element's min Y offset allowed
			int 			yMax; 				// Element's max Y offset allowed
			int 			xMin; 				// Element's min X offset allowed
			int 			xMax; 				// Element's max X offset allowed
			int 			xOffsetDyn; 		// Element X offset, dynamic (added to xoffset value for animation effects)
			int 			yOffsetDyn; 		// Element Y offset, dynamic (added to yoffset value for animation effects)
			int 			alpha; 				// Element alpha value (0-255)
			f32 			scale; 				// Element scale (1 = 100%)
			int 			alphaDyn; 			// Element alpha, dynamic (multiplied by alpha value for blending/fading effects)
			f32 			scaleDyn; 			// Element scale, dynamic (multiplied by alpha value for blending/fading effects)
			bool 			rumble; 			// Wiimote rumble (on/off) - set to on when this element requests a rumble event
			int  			effects; 			// Currently enabled effect(s). 0 when no effects are enabled
			int  			effectAmount; 		// Effect amount. Used by different effects for different purposes
			int  			effectTarget; 		// Effect target amount. Used by different effects for different purposes
			int  			effectLoop; 		// Effect number Loop. Used by WARNING effects for number of alphaMax/alphaMin loop
			int  			effectsOver; 		// Effects to enable when wiimote cursor is over this element. Copied to effects variable on over event
			int  			effectAmountOver;	// EffectAmount to set when wiimote cursor is over this element
			int  			effectTargetOver; 	// EffectTarget to set when wiimote cursor is over this element
			int  			alignmentHor; 		// Horizontal element alignment, respective to parent element (LEFT, RIGHT, CENTER)
			int  			alignmentVert; 		// Horizontal element alignment, respective to parent element (TOP, BOTTOM, MIDDLE)
			int  			state; 				// Element state (DEFAULT, SELECTED, CLICKED, DISABLED)
			int  			stateChan; 			// Which controller channel is responsible for the last change in state
			bool 			selectable; 		// Whether or not this element selectable (can change to SELECTED state)
			bool 			clickable; 			// Whether or not this element is clickable (can change to CLICKED state)
			bool 			holdable; 			// Whether or not this element is holdable (can change to HELD state)
			GuiTrigger 		*trigger[6]; 		// GuiTriggers (input actions) that this element responds to
			GuiElement 		*parentElement;		// Parent element
			UpdateCallback 	updateCB; 			// Callback function to call when this element is updated
			int 			slide;
	};

	class GuiWindow : public GuiElement
	{
		public:
			GuiWindow(void);
			GuiWindow(int, int);
			~GuiWindow(void);

			void 		Append(GuiElement *);
			void 		Insert(GuiElement *, u32);
			void 		Remove(GuiElement *);
			void 		RemoveAll(void);
			GuiElement	*GetGuiElementAt(u32) const;
			u32 		GetSize(void);
			void 		SetVisible(bool);
			void 		ResetState(void);
			void 		SetState(int);
			int  		GetSelected(void);
			void 		SetFocus(int);
			void 		ChangeFocus(GuiElement *);
			void 		ToggleFocus(GuiTrigger *);
			void 		MoveSelectionHor(int);
			void 		MoveSelectionVert(int);
			void 		Draw(void);
			void 		DrawTooltip(void);
			void 		Update(GuiTrigger *);

		protected:
			std::vector<GuiElement*> AllElements; // Contains all elements within the GuiWindow
	};

	class GuiImageData
	{
		public:
			GuiImageData(const u8 *i);
			~GuiImageData(void);

			u8  *GetImage(void);
			int GetWidth(void);
			int GetHeight(void);

		protected:
			u8  *data;	// Image data
			int height;	// Image height
			int width;	// Image width
	};

	class GuiImage : public GuiElement
	{
		public:
			GuiImage(void);
			GuiImage(GuiImageData *);
			GuiImage(u8 *, int, int);
			GuiImage(int, int, GXColor);
			~GuiImage(void);

			void 	SetAngle(float);
			void 	SetTile(int);
			void 	SetWidescreen(bool);
			void 	Draw(void);
			u8 		*GetImage(void);
			void 	SetImage(GuiImageData *);
			void 	SetImage(u8 *, int, int);
			void 	SetImage(int, int, GXColor);
			GXColor GetPixel(int, int y);
			void 	SetPixel(int, int, GXColor);
			void 	ColorStripe(int);
			void 	SetParentAngle(bool);
			void 	Grayscale(void);
			void 	SetStripe(int);

		protected:
			int 	imgType; 		// Type of image data (IMAGE_TEXTURE, IMAGE_COLOR, IMAGE_DATA)
			u8 		*image; 		// Poiner to image data. May be shared with GuiImageData data
			f32 	imageAngle; 	// Angle to draw the image
			int 	tile; 			// Number of times to draw (tile) the image horizontally
			int 	stripe; 		// Alpha value (0-255) to apply a stripe effect to the texture
			short 	widescreen;
			bool 	parentAngle;
	};

	class GuiText : public GuiElement
	{
		public:
			enum
			{
				WRAP,
				DOTTED,
				SCROLL,
				SCROLLFULL,
				LONGTEXT
			};

			GuiText(const char *, int, GXColor);
			GuiText(const char *);
			~GuiText(void);

			void 		SetText(const char *);
			void 		SetTextf(const char *, ...) __attribute__((format(printf, 2, 3)));
			void 		SetText(const wchar_t *);
			static void SetPresets(int, GXColor, int, int, u16, int, int);
			void 		SetFontSize(int);
			void 		SetMaxWidth(int, short m = GuiText::WRAP);
			void 		SetColor(GXColor);
			void 		SetStyle(u16, u16 m = 0xffff);
			void 		SetAlignment(int, int);
			void 		SetFont(FreeTypeGX *);
			void 		SetPassChar(wchar_t);
			int  		GetTextWidth(void);
			void 		SetWidescreen(bool);
			void 		SetNumLines(int);
			void 		SetFirstLine(int);
			int  		GetNumLines(void);
			int  		GetFirstLine(void);
			int  		GetLineHeight(int);
			void 		SetLinesToDraw(int);
			int  		GetTotalLines(void);
			void 		Draw(void);

		protected:
			wchar_t 	*text; 							// Unicode text value
			wchar_t 	*tmpText[MAX_LINES_TO_DRAW];	// Wrapped lines text values
			u32 		*lineBreak;
			int 		size; 							// Font size
			int 		maxWidth; 						// Maximum width of the generated text object (for text wrapping)
			short 		wrapMode;
			short 		scrollPos1;
			short 		scrollPos2;
			short 		scrollOffset;
			u32 		scrollDelay;
			u16 		style; 							// FreeTypeGX style attributes
			GXColor 	color; 							// Font color
			FreeTypeGX	*font;
			short 		widescreen;

			// These are default until the text is drawn
			int 		firstLine; 						// First line drawn when the text is wrapped
			int			numLines;						// Number of lines drawn (Default is -1 and it means that all lines are drawn)
			int			linesToDraw;
			int			totalLines; 					// Total number of lines when in wrap mode
			wchar_t		passChar; 						// Password character
			bool		longListChanged;
	};

	class GuiTooltip : public GuiElement
	{
		public:
			GuiTooltip(const char *, int Alpha = 255);
			~ GuiTooltip(void);

			float	GetScale(void);
			void	SetText(const char *);
			void	SetWidescreen(bool);
			void	Draw(void);

		protected:
			GuiImage	leftImage;	// Tooltip left image
			GuiImage	tileImage;	// Tooltip tile image
			GuiImage	rightImage;	// Tooltip right image
			GuiText		*text;
	};

	class GuiButton : public GuiElement
	{
		public:
			GuiButton(int, int);
			~GuiButton(void);

			void SetImage(GuiImage *);
			void SetImageOver(GuiImage *);
			void SetImageHold(GuiImage *);
			void SetImageClick(GuiImage *);
			void SetIcon(GuiImage *);
			void SetIcon2(GuiImage *);
			void SetIconOver(GuiImage *);
			void SetIconHold(GuiImage *);
			void SetIconClick(GuiImage *);
			void SetLabel(GuiText *, int n = 0);
			void SetLabelOver(GuiText *, int n = 0);
			void SetLabelHold(GuiText *, int n = 0);
			void SetLabelClick(GuiText *, int n = 0);
			void SetSoundOver(GuiSound *);
			void SetSoundHold(GuiSound *);
			void SetSoundClick(GuiSound *);
			void SetToolTip(GuiTooltip *, int, int, int h = ALIGN_RIGHT, int v = ALIGN_TOP);
			void RemoveToolTip(void);
			void Draw(void);
			void DrawTooltip(void);
			void Update(GuiTrigger *);

		protected:
			GuiImage 	*image; 		// Button image (default)
			GuiImage 	*imageOver; 	// Button image for STATE_SELECTED
			GuiImage 	*imageHold; 	// Button image for STATE_HELD
			GuiImage 	*imageClick;    // Button image for STATE_CLICKED
			GuiImage 	*icon; 		  	// Button icon (drawn after button image)
			GuiImage 	*icon2; 		// Button icon 2 (drawn after button image)
			GuiImage 	*iconOver; 	  	// Button icon for STATE_SELECTED
			GuiImage 	*iconHold; 	  	// Button icon for STATE_HELD
			GuiImage 	*iconClick; 	// Button icon for STATE_CLICKED
			GuiTooltip	*toolTip;
			time_t 		time1, time2;	// Tooltip time constants
			GuiText 	*label[3];		// Label(s) to display (default)
			GuiText 	*labelOver[3];	// Label(s) to display for STATE_SELECTED
			GuiText 	*labelHold[3];	// Label(s) to display for STATE_HELD
			GuiText 	*labelClick[3];	// Label(s) to display for STATE_CLICKED
			GuiSound 	*soundOver; 	// Sound to play for STATE_SELECTED
			GuiSound 	*soundHold;  	// Sound to play for STATE_HELD
			GuiSound 	*soundClick; 	// Sound to play for STATE_CLICKED
	};

	typedef struct _keytype
	{
		char ch;
		char chShift;
	} Key;

	class GuiKeyboard : public GuiWindow
	{
		public:
			GuiKeyboard(char *, u32);
			~GuiKeyboard(void);

			void Update(GuiTrigger *);
			char kbTextStr[256];
			
		protected:
			u32				kbTextMaxLen;
			Key				keys[4][11];
			int				shift;
			int				caps;
			GuiText			*kbText;
			GuiImage		*keyTextboxImg;
			GuiText			*keyCapsText;
			GuiImage		*keyCapsImg;
			GuiImage		*keyCapsOverImg;
			GuiButton		*keyCaps;
			GuiText			*keyShiftText;
			GuiImage		*keyShiftImg;
			GuiImage		*keyShiftOverImg;
			GuiButton		*keyShift;
			GuiText			*keyBackText;
			GuiImage		*keyBackImg;
			GuiImage		*keyBackOverImg;
			GuiButton		*keyBack;
			GuiImage		*keySpaceImg;
			GuiImage		*keySpaceOverImg;
			GuiButton		*keySpace;
			GuiButton		*keyBtn[4][11];
			GuiImage		*keyImg[4][11];
			GuiImage		*keyImgOver[4][11];
			GuiText			*keyTxt[4][11];
			GuiImageData	*keyTextbox;
			GuiImageData	*key;
			GuiImageData	*keyOver;
			GuiImageData	*keyMedium;
			GuiImageData	*keyMediumOver;
			GuiImageData	*keyLarge;
			GuiImageData	*keyLargeOver;
			GuiSound		*keySoundOver;
			GuiSound		*keySoundClick;
			GuiTrigger		*triggerA;
			GuiTrigger		*triggerB;
	};

	typedef struct _optionlist
	{
		int  length;
		char name[MAX_OPTIONS][50];
		char value[MAX_OPTIONS][MAXPATHLEN];
	} OptionList;

	class GuiOptionBrowser : public GuiElement
	{
		public:
			GuiOptionBrowser(int, int, OptionList *);
			~GuiOptionBrowser(void);

			int  FindMenuItem(int, int);
			int  GetClickedOption(void);
			int  GetSelectedOption(void);
			void ResetState(void);
			void SetFocus(int);
			void Draw(void);
			void TriggerUpdate(void);
			void Update(GuiTrigger *);
			
			GuiText 		*optionVal[PAGESIZE];
			
		protected:
			int  			selectedItem;
			int  			listOffset;
			bool 			listChanged;

			OptionList 		*options;
			int        		optionIndex[PAGESIZE];
			GuiButton  		*optionBtn[PAGESIZE];
			GuiText    		*optionTxt[PAGESIZE];
			GuiImage   		*optionBg[PAGESIZE];

			GuiButton 		*arrowUpBtn;
			GuiButton 		*arrowDownBtn;
			GuiButton 		*scrollbarBoxBtn;
			GuiImage 		*bgOptionsImg;
			GuiImage 		*scrollbarImg;
			GuiImage 		*arrowDownImg;
			GuiImage 		*arrowDownOverImg;
			GuiImage 		*arrowUpImg;
			GuiImage 		*arrowUpOverImg;
			GuiImage 		*scrollbarBoxImg;
			GuiImage 		*scrollbarBoxOverImg;
			GuiImageData 	*bgOptions;
			GuiImageData 	*bgOptionsEntry;
			GuiImageData 	*scrollbar;
			GuiImageData 	*arrowDown;
			GuiImageData 	*arrowDownOver;
			GuiImageData 	*arrowUp;
			GuiImageData 	*arrowUpOver;
			GuiImageData 	*scrollbarBox;
			GuiImageData 	*scrollbarBoxOver;
			GuiSound   		*btnSoundOver;
			GuiSound   		*btnSoundClick;
			GuiTrigger 		*triggerA;
			GuiTrigger 		*triggerHeldA;
	};
#endif
