///////////////////////////////////////////////////////
// Text translation
///////////////////////////////////////////////////////

// Includes
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <gctypes.h>

#include "Language/GetText.h"
#include "Settings/Settings.h"
#include "sysCheckGX/Tools.h"

// Structures
typedef struct _MSG
{
	u32			id;
	char		*msgStr;
	struct _MSG *next;
} MSG;

// Variables
static MSG *baseMSG = 0;

// Constants
#define HASH_WORD_BITS 32

// Defines the so called `HashPJW' function by P.J. Weinberger
// [see Aho/Sethi/Ullman, COMPILERS: Principles, Techniques and Tools, 1986, 1987 Bell Telephone Laboratories, Inc.]
static inline u32

HashString(const char *stringToHash)
{
	u32		   hval, g;
	const char *str = stringToHash;

	// Compute the hash value for the given string
	hval = 0;

	while (*str != '\0')
	{
		hval <<= 4;
		hval += (u8) *str++;
		g = hval & ((u32) 0xf << (HASH_WORD_BITS - 4));

		if (g != 0)
		{
			hval ^= g >> (HASH_WORD_BITS - 8);
			hval ^= g;
		}
	}

	return hval;
}

static MSG *FindMsg(u32 id)
{
	MSG *msg;

	for (msg = baseMSG; msg; msg = msg->next)
	{
		if (msg->id == id)
			return msg;
	}

	return NULL;
}

static MSG *SetMsg(const char *msgId, const char *msgStr)
{
	u32 id = HashString(msgId);
	MSG *msg = FindMsg(id);

	if (!msg)
	{
		msg = (MSG *)malloc(sizeof(MSG));
		msg->id	= id;
		msg->msgStr = NULL;
		msg->next = baseMSG;
		baseMSG = msg;
	}

	if (msg)
	{
		if (msgStr)
		{
			if (msg->msgStr)
				free(msg->msgStr);

			msg->msgStr = strdup(msgStr);
		}

		return msg;
	}

	return NULL;
}

void GetTextCleanUp(void)
{
	while (baseMSG)
	{
		MSG *nextMsg = baseMSG->next;
		
		free(baseMSG->msgStr);
		free(baseMSG);
		
		baseMSG = nextMsg;
	}
}

bool GetTextLoadLanguage(const char *languageFile)
{
	FILE *f;
	char line[256];
	char *lastID = NULL;

	GetTextCleanUp();
	
	f = fopen(languageFile, "r");

	if (f)
	{
		while (fgets(line, sizeof(line), f))
		{
			// Lines starting with # are comments
			if (line[0] == '#')
				continue;
			else if (strncmp(line, "msgId \"", 7) == 0)
			{
				char *msgId, *end;
				
				if (lastID)
				{
					free(lastID);

					lastID = NULL;
				}
				
				msgId = &line[7];
				end = strrchr(msgId, '"');

				if (end && end-msgId > 1)
				{
					*end = 0;

					lastID = strdup(msgId);
				}
			}
			else if (strncmp(line, "msgStr \"", 8) == 0)
			{
				char *msgStr, *end;

				if (lastID == NULL)
					continue;

				msgStr = &line[8];
				end = strrchr(msgStr, '"');

				if (end && end-msgStr > 1)
				{
					*end = 0;

					SetMsg(lastID, msgStr);
				}

				free(lastID);
				
				lastID = NULL;
			}
		}

		fclose(f);
		
		return true;
	}
	else
		return false;
}

// Translate text
const char *GetText(const char *msgId)
{
	MSG *msg = FindMsg(HashString(msgId));

	if (msg)
		return msg->msgStr;
	else
		return msgId;
}

const char *GetReportText(const char *msgId, bool detailedReport)
{
	MSG *msg;

	char stringToHash[256];

	if (detailedReport)
		snprintf(stringToHash, sizeof(stringToHash), "Report::%s", msgId);
	else
		snprintf(stringToHash, sizeof(stringToHash), "%s", msgId);

	msg = FindMsg(HashString(stringToHash));

	if (msg)
		return msg->msgStr;
	else
		return msgId;
}
