/*
 * isepic.c - ISEPIC emulation.
 *
 * Written by
 *  Marco van den Heuvel <blackystardust68@yahoo.com>
 * 
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

/*
 * ISEPIC is a RAM based freeze cart.
 *
 * It has 2KB of ram which is banked into a 256 byte page in the $DF00-$DFFF area.
 *
 * The page is selected by any kind of access to the I/O-1 area, as follows:
 *
 * PAGE   ACCESS ADDRESS
 * ----   --------------
 *  0         $DE00
 *  1         $DE04
 *  2         $DE02
 *  3         $DE06
 *  4         $DE01
 *  5         $DE05
 *  6         $DE03
 *  7         $DE07
 *
 * Because of the incomplete decoding this 8 byte area is mirrored throughout $DE08-$DEFF.
 *
 * The isepic cart has a switch which controls if the registers and ram is mapped in.
 *
 * When the switch is switched away from the computer the cart is put in 'hidden' mode,
 * where the registers, window and ram is not accessable.
 *
 * When the switch is switched towards the computer the cart is put in ultimax mode,
 * with the registers mapped, and the current page being mapped into any unmapped ultimax
 * memory space, it will also generate an NMI. Which activates the freezer.
 *
 */

#include "vice.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "archdep.h"
#include "c64cart.h"
#include "c64cartmem.h"
#include "c64cartsystem.h"
#include "c64export.h"
#include "c64io.h"
#include "c64mem.h"
#include "cartridge.h"
#include "cmdline.h"
#include "crt.h"
#include "isepic.h"
#include "lib.h"
#include "log.h"
#include "machine.h"
#include "mem.h"
#include "resources.h"
#include "snapshot.h"
#include "translate.h"
#include "types.h"
#include "util.h"

/* #define DEBUGISEPIC */

#ifdef DEBUGISEPIC
#define DBG(x) printf x
#else
#define DBG(x)
#endif

/* ------------------------------------------------------------------------- */

/* Flag: Do we enable the ISEPIC?  */
static int isepic_enabled;

/* Flag: what direction is the switch at, 0 = away, 1 = towards computer */
int isepic_switch = 0;

/* 2 KB RAM */
static BYTE *isepic_ram;

/* current page */
static unsigned int isepic_page = 0;

static char *isepic_filename = NULL;
static int isepic_filetype = 0;

static const char STRING_ISEPIC[] = "Isepic Cartridge";

#define ISEPIC_RAM_SIZE 2048

/* ------------------------------------------------------------------------- */

/* some prototypes are needed */
static BYTE REGPARM1 isepic_io1_read(WORD addr);
static void REGPARM2 isepic_io1_store(WORD addr, BYTE byte);
static BYTE REGPARM1 isepic_io2_read(WORD addr);
static void REGPARM2 isepic_io2_store(WORD addr, BYTE byte);

static io_source_t isepic_io1_device = {
    "ISEPIC",
    IO_DETACH_RESOURCE,
    "IsepicCartridgeEnabled",
    0xde00, 0xdeff, 0x07,
    0, /* read is never valid */
    isepic_io1_store,
    isepic_io1_read,
    NULL,
    NULL,
    CARTRIDGE_ISEPIC
};

static io_source_t isepic_io2_device = {
    "ISEPIC",
    IO_DETACH_RESOURCE,
    "IsepicCartridgeEnabled",
    0xdf00, 0xdfff, 0xff,
    0,
    isepic_io2_store,
    isepic_io2_read,
    NULL,
    NULL,
    CARTRIDGE_ISEPIC
};

static const c64export_resource_t export_res = {
    "ISEPIC", 1, 1, &isepic_io1_device, &isepic_io2_device, CARTRIDGE_ISEPIC
};

static io_source_list_t *isepic_io1_list_item = NULL;
static io_source_list_t *isepic_io2_list_item = NULL;

/* ------------------------------------------------------------------------- */

int isepic_cart_enabled(void)
{
    if (isepic_enabled && isepic_switch) {
        return 1;
    }
    return 0;
}

int isepic_freeze_allowed(void)
{
    return isepic_cart_enabled();
}

void isepic_freeze(void)
{
    /* FIXME: do nothing ? */
}

static int set_isepic_enabled(int val, void *param)
{
    DBG(("set enabled: %d\n", val));
    if (isepic_enabled && !val) {
        cart_power_off();
        lib_free(isepic_ram);
        isepic_ram = NULL;
        if (isepic_filename) {
            lib_free(isepic_filename);
            isepic_filename = NULL;
        }
        c64io_unregister(isepic_io1_list_item);
        c64io_unregister(isepic_io2_list_item);
        isepic_io1_list_item = NULL;
        isepic_io2_list_item = NULL;
        c64export_remove(&export_res);
        isepic_enabled = 0;
        if (isepic_switch) {
            cartridge_config_changed(2, 2, CMODE_READ | CMODE_RELEASE_FREEZE);
        }
    } else if (!isepic_enabled && val) {
        cart_power_off();
        isepic_ram = lib_malloc(ISEPIC_RAM_SIZE);
        isepic_io1_list_item = c64io_register(&isepic_io1_device);
        isepic_io2_list_item = c64io_register(&isepic_io2_device);
        if (c64export_add(&export_res) < 0) {
            lib_free(isepic_ram);
            isepic_ram = NULL;
            c64io_unregister(isepic_io1_list_item);
            c64io_unregister(isepic_io2_list_item);
            isepic_io1_list_item = NULL;
            isepic_io2_list_item = NULL;
            return -1;
        }
        isepic_enabled = 1;
        if (isepic_switch) {
            cartridge_config_changed(2, 3, CMODE_READ | CMODE_RELEASE_FREEZE);
        }
    }
    return 0;
}

int isepic_enable(void)
{
    DBG(("ISEPIC: enable\n"));
    if (resources_set_int("IsepicCartridgeEnabled", 1) < 0) {
        return -1;
    }
    return 0;
}

static int set_isepic_switch(int val, void *param)
{
    DBG(("set switch: %d\n", val));
    if (isepic_switch && !val) {
        isepic_switch = 0;
        if (isepic_enabled) {
            cartridge_config_changed(2, 2, CMODE_READ | CMODE_RELEASE_FREEZE);
        }
    }

    if (!isepic_switch && val) {
        isepic_switch = 1;
        if (isepic_enabled) {
            cartridge_trigger_freeze();
            cartridge_config_changed(2, 3, CMODE_READ | CMODE_RELEASE_FREEZE);
        }
    }
    return 0;
}

/* ------------------------------------------------------------------------- */

static const resource_int_t resources_int[] = {
    { "IsepicCartridgeEnabled", 0, RES_EVENT_STRICT, (resource_value_t)0,
      &isepic_enabled, set_isepic_enabled, NULL },
    { "IsepicSwitch", 0, RES_EVENT_STRICT, (resource_value_t)1,
      &isepic_switch, set_isepic_switch, NULL },
    { NULL }
};

int isepic_resources_init(void)
{
    return resources_register_int(resources_int);
}
void isepic_resources_shutdown(void)
{
}

/* ------------------------------------------------------------------------- */

static const cmdline_option_t cmdline_options[] =
{
    { "-isepic", SET_RESOURCE, 0,
      NULL, NULL, "IsepicCartridgeEnabled", (resource_value_t)1,
      USE_PARAM_STRING, USE_DESCRIPTION_ID,
      IDCLS_UNUSED, IDCLS_ENABLE_ISEPIC,
      NULL, NULL },
    { "+isepic", SET_RESOURCE, 0,
      NULL, NULL, "IsepicCartridgeEnabled", (resource_value_t)0,
      USE_PARAM_STRING, USE_DESCRIPTION_ID,
      IDCLS_UNUSED, IDCLS_DISABLE_ISEPIC,
      NULL, NULL },
    { NULL }
};

int isepic_cmdline_options_init(void)
{
    return cmdline_register_options(cmdline_options);
}

/* ------------------------------------------------------------------------- */

BYTE REGPARM1 isepic_io1_read(WORD addr)
{
    DBG(("io1 r %04x (sw:%d)\n", addr, isepic_switch));

    if (isepic_switch) {
        isepic_page = ((addr & 4) >> 2) | (addr & 2) | ((addr & 1) << 2);
    }
    return 0;
}

void REGPARM2 isepic_io1_store(WORD addr, BYTE byte)
{
    DBG(("io1 w %04x %02x (sw:%d)\n", addr, byte, isepic_switch));

    if (isepic_switch) {
        isepic_page = ((addr & 4) >> 2) | (addr & 2) | ((addr & 1) << 2);
    }
}

BYTE REGPARM1 isepic_io2_read(WORD addr)
{
    BYTE retval = 0;

    DBG(("io2 r %04x (sw:%d) (p:%d)\n", addr, isepic_switch, isepic_page));

    isepic_io2_device.io_source_valid = 0;

    if (isepic_switch) {
        isepic_io2_device.io_source_valid = 1;
        retval = isepic_ram[(isepic_page * 256) + (addr & 0xff)];
    }

    return retval;
}

void REGPARM2 isepic_io2_store(WORD addr, BYTE byte)
{
    DBG(("io2 w %04x %02x (sw:%d)\n", addr, byte, isepic_switch));

    if (isepic_switch) {
        isepic_ram[(isepic_page * 256) + (addr & 0xff)] = byte;
    }
}

/* ------------------------------------------------------------------------- */

BYTE REGPARM1 isepic_romh_read(WORD addr)
{
    switch (addr) {
        case 0xfffa:
        case 0xfffb:
            return isepic_ram[(isepic_page * 256) + (addr & 0xff)];
            break;
        default:
            return mem_read_without_ultimax(addr);
            break;
    }
}

void REGPARM2 isepic_romh_store(WORD addr, BYTE byte)
{
    switch (addr) {
        case 0xfffa:
        case 0xfffb:
            isepic_ram[(isepic_page * 256) + (addr & 0xff)] = byte;
            break;
        default:
            mem_store_without_ultimax(addr, byte);
            break;
    }
}

BYTE REGPARM1 isepic_page_read(WORD addr)
{
    if (isepic_switch) {
        return isepic_ram[(isepic_page * 256) + (addr & 0xff)];
    } else {
        return mem_read_without_ultimax(addr);
    }
}

void REGPARM2 isepic_page_store(WORD addr, BYTE value)
{
    if (isepic_switch) {
        isepic_ram[(isepic_page * 256) + (addr & 0xff)] = value;
    } else {
        mem_store_without_ultimax(addr, value);
    }
}

/* ---------------------------------------------------------------------*/

const char *isepic_get_file_name(void)
{
    return isepic_filename;
}

static void isepic_set_filename(const char *filename)
{
    if (isepic_filename) {
        lib_free(isepic_filename);
    }
    isepic_filename = strdup(filename);
}

void isepic_config_init(void)
{
    /* FIXME: do nothing ? */
}

void isepic_reset(void)
{
    /* FIXME: do nothing ? */
}

void isepic_config_setup(BYTE *rawcart)
{
    memcpy(isepic_ram, rawcart, ISEPIC_RAM_SIZE);
}

static int isepic_common_attach(BYTE *rawcart)
{
    if (resources_set_int("IsepicCartridgeEnabled", 1) < 0) {
        return -1;
    }
    if (isepic_enabled) {
        memcpy(isepic_ram, rawcart, ISEPIC_RAM_SIZE);
        return 0;
    }
    return -1;
}

int isepic_bin_attach(const char *filename, BYTE *rawcart)
{
    if (util_file_load(filename, rawcart, ISEPIC_RAM_SIZE, UTIL_FILE_LOAD_SKIP_ADDRESS) < 0) {
        return -1;
    }

    isepic_set_filename(filename);
    isepic_filetype = CARTRIDGE_FILETYPE_BIN;

    return isepic_common_attach(rawcart);
}

int isepic_bin_save(const char *filename)
{
    FILE *fd;

    if (filename == NULL) {
        return -1;
    }

    fd = fopen(filename, MODE_WRITE);

    if (fd == NULL) {
        return -1;
    }

    if (fwrite(isepic_ram, 1, ISEPIC_RAM_SIZE, fd) != ISEPIC_RAM_SIZE) {
        fclose(fd);
        return -1;
    }

    fclose(fd);
    return 0;
}

int isepic_crt_attach(FILE *fd, BYTE *rawcart, const char *filename)
{
    BYTE chipheader[0x10];

    if (fread(chipheader, 0x10, 1, fd) < 1) {
        return -1;
    }

    if (fread(rawcart, ISEPIC_RAM_SIZE, 1, fd) < 1) {
        return -1;
    }

    isepic_set_filename(filename);
    isepic_filetype = CARTRIDGE_FILETYPE_CRT;

    resources_set_int("IsepicSwitch", 0);
    return isepic_common_attach(rawcart);
}

int isepic_crt_save(const char *filename)
{
    FILE *fd;
    BYTE header[0x40], chipheader[0x10];

    fd = fopen(filename, MODE_WRITE);

    if (fd == NULL) {
        return -1;
    }

    /*
     * Initialize headers to zero.
     */
    memset(header, 0x0, 0x40);
    memset(chipheader, 0x0, 0x10);

    /*
     * Construct CRT header.
     */
    strcpy((char *)header, CRT_HEADER);

    /*
     * fileheader-length (= 0x0040)
     */
    header[0x10] = 0x00;
    header[0x11] = 0x00;
    header[0x12] = 0x00;
    header[0x13] = 0x40;

    /*
     * Version (= 0x0100)
     */
    header[0x14] = 0x01;
    header[0x15] = 0x00;

    /*
     * Hardware type (= CARTRIDGE_ISEPIC)
     */
    header[0x16] = 0x00;
    header[0x17] = CARTRIDGE_ISEPIC;

    /*
     * Exrom line
     */
    header[0x18] = 0x01;            /* ? */

    /*
     * Game line
     */
    header[0x19] = 0x01;            /* ? */

    /*
     * Set name.
     */
    strcpy((char *)&header[0x20], STRING_ISEPIC);

    /*
     * Write CRT header.
     */
    if (fwrite(header, sizeof(BYTE), 0x40, fd) != 0x40) {
        fclose(fd);
        return -1;
    }

    /*
     * Construct chip packet.
     */
    strcpy((char *)chipheader, CHIP_HEADER);

    /*
     * Packet length. (= 0x0810; 0x10 + 0x0800)
     */
    chipheader[0x04] = 0x00;
    chipheader[0x05] = 0x00;
    chipheader[0x06] = 0x08;
    chipheader[0x07] = 0x10;

    /*
     * Chip type. (= FlashROM?)
     */
    chipheader[0x08] = 0x00;
    chipheader[0x09] = 0x02;

    /*
     * Bank nr. (= 0)
     */
    chipheader[0x0a] = 0x00;
    chipheader[0x0b] = 0x00;

    /*
     * Address. (= 0x8000)
     */
    chipheader[0x0c] = 0x80;
    chipheader[0x0d] = 0x00;

    /*
     * Length. (= 0x0800)
     */
    chipheader[0x0e] = 0x08;
    chipheader[0x0f] = 0x00;

    /*
     * Write CHIP header.
     */
    if (fwrite(chipheader, sizeof(BYTE), 0x10, fd) != 0x10) {
        fclose(fd);
        return -1;
    }

    /*
     * Write CHIP packet data.
     */
    if (fwrite(isepic_ram, sizeof(char), ISEPIC_RAM_SIZE, fd) != ISEPIC_RAM_SIZE) {
        fclose(fd);
        return -1;
    }

    fclose(fd);

    return 0;
}

int isepic_flush_image(void)
{
    if (isepic_filetype == CARTRIDGE_FILETYPE_BIN) {
        return isepic_bin_save(isepic_filename);
    } else if (isepic_filetype == CARTRIDGE_FILETYPE_CRT) {
        return isepic_crt_save(isepic_filename);
    }
    return -1;
}

void isepic_detach(void)
{
    resources_set_int("IsepicCartridgeEnabled", 0);
}

