/******************************************************************************* 
 *  -- menu.h - Menu interface for Vice Playstation 3
 *
 *     VICE PS3 -   Commodore 64 emulator for the Playstation 3
 *                  ported from the original VICE distribution
 *                  located at http://sourceforge.net/projects/vice-emu/
 *
 *
 *  Copyright (C) 2010
 *  Created on: Oct 10, 2010
 *      Original author:      halsafar
 *      Adapted for Vice by:  TimRex
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 ********************************************************************************/


#include <string.h>
#include <stack>
#include <vector>
#include <cell/pad.h>
#include <cell/audio.h>
#include <cell/sysmodule.h>
#include <cell/cell_fs.h>
#include <cell/dbgfont.h>
#include <sysutil/sysutil_sysparam.h>

#include "cellframework/fileio/FileBrowser.h"
#include "cellframework/input/cellInput.h"

#include "PS3Graphics.h"
#include "emulator.h"
#include "joy.h"

extern "C" {
#include "sid.h"
#include "siddefs.h"
#include "common.h"
#include "util.h"
#include "log.h"
#include "machine.h"
#include "videoarch.h"
#include "resources.h"
#include "mouse.h"
#include "attach.h"
#include "tape.h"
#include "ui.h"
#include "zfile.h"
}

#include "menu.h"

#include "common.h"

#define MIN(x,y) ((x) < (y) ? (x) : (y))
#define MAX(x,y) ((x) > (y) ? (x) : (y))

#define NUM_ENTRY_PER_PAGE 23

// is the menu running
bool menuRunning = false;

// menu to render
typedef void (*curMenuPtr)();

//
std::stack<curMenuPtr> menuStack;

// main file browser->for rom browser
FileBrowser* browser = NULL;

// tmp file browser->for everything else
FileBrowser* tmpBrowser = NULL;

int16_t currently_selected_setting = 0;
int16_t currently_selected_vice_setting = 0;
int16_t currently_selected_path_setting = 0;




void MenuStop()
{
    menuRunning = false;

    /*
    while (!menuStack.empty())
    {
        menuStack.pop();
    }
    */
}


bool MenuIsRunning()
{
    return menuRunning;
}


static bool selection_changed=false;

void UpdateBrowser(FileBrowser* b)
{
    if (CellInput->WasButtonPressed(0,CTRL_DOWN) | CellInput->IsAnalogPressedDown(0,CTRL_LSTICK))
    {
        b->IncrementEntry();
        selection_changed=true;
    }
    if (CellInput->WasButtonPressed(0,CTRL_UP) | CellInput->IsAnalogPressedUp(0,CTRL_LSTICK))
    {
        b->DecrementEntry();
        selection_changed=true;
    }
    if (CellInput->WasButtonPressed(0,CTRL_RIGHT) | CellInput->IsAnalogPressedRight(0,CTRL_LSTICK))
    {
        b->GotoEntry(MIN(b->GetCurrentEntryIndex()+5, b->GetCurrentDirectoryInfo().numEntries-1));
        selection_changed=true;
    }
    if (CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->IsAnalogPressedLeft(0,CTRL_LSTICK))
    {
        if (b->GetCurrentEntryIndex() <= 5)
        {
            b->GotoEntry(0);
            selection_changed=true;
        }
        else
        {
            b->GotoEntry(b->GetCurrentEntryIndex()-5);
            selection_changed=true;
        }
    }
    if (CellInput->WasButtonPressed(0,CTRL_R1) && !(CellInput->IsButtonPressed(0,CTRL_TRIANGLE)) )
    {
        b->GotoEntry(MIN(b->GetCurrentEntryIndex()+NUM_ENTRY_PER_PAGE, b->GetCurrentDirectoryInfo().numEntries-1));
        selection_changed=true;
    }
    if (CellInput->WasButtonPressed(0,CTRL_L1) && !(CellInput->IsButtonPressed(0,CTRL_TRIANGLE)) )
    {
        if (b->GetCurrentEntryIndex() <= NUM_ENTRY_PER_PAGE)
        {
            b->GotoEntry(0);
            selection_changed=true;
        }
        else
        {
            b->GotoEntry(b->GetCurrentEntryIndex()-NUM_ENTRY_PER_PAGE);
            selection_changed=true;
        }
    }

    if (CellInput->WasButtonPressed(0, CTRL_CIRCLE))
    {
        // don't let people back out past root
        if (b->DirectoryStackCount() > 1)
        {
            b->PopDirectory();
            selection_changed=true;
        }
    }
}

void RenderBrowser(FileBrowser* b)
{
    uint32_t file_count = b->GetCurrentDirectoryInfo().numEntries;
    int current_index = b->GetCurrentEntryIndex();

    if (file_count <= 0)
    {
    }
    else if (current_index > file_count)
    {
    }
    else
    {
        int page_number = current_index / NUM_ENTRY_PER_PAGE;
        int page_base = page_number * NUM_ENTRY_PER_PAGE;
        float currentX = 0.05f;
        float currentY = 0.00f;
        float ySpacing = 0.035f;
        for (int i = page_base; i < file_count && i < page_base + NUM_ENTRY_PER_PAGE; ++i)
        {
            currentY = currentY + ySpacing;
            cellDbgFontPuts(currentX, currentY, Emulator_GetFontSize(),
            i == current_index ? RED : (*b)[i]->d_type == CELL_FS_TYPE_DIRECTORY ? GREEN : WHITE,
            (*b)[i]->d_name);
            Graphics->FlushDbgFont();
        }
    }

/*
    // Locate any screenshots for the currently selected file
    if (selection_changed) {
        selection_changed=false;
                    
        if ((*b)[current_index]->d_type == CELL_FS_TYPE_REGULAR) {
            // TODO check file extension
        
            string ext = FileBrowser::GetExtension((*b)[current_index]->d_name);
            string rom_path = b->GetCurrentDirectoryInfo().dir + "/" + (*b)[current_index]->d_name;

            if ( ((ext == "zip") || (ext == "ZIP")) && (zipfile_entries(rom_path.c_str()) != 1) ) {
                // Do nothing
                // If we have a zipfile with multiple entries, we can use zipfile browsing support.
                // If it only has a single entry, we can use this file as it is
                
                // Clear screenshot
            }
            else {
                long size;
                unsigned char * screenshot;

                string path = b->GetCurrentDirectoryInfo().dir + "/" + (*b)[current_index]->d_name;
                screenshot = load_screenshot(path.c_str(), &size);
    
                if (screenshot != NULL) {
                    // Display screenshot
    
                    // proof of concept. Really should get the dimensions fromthe screenshot file.
                    // Also, this draws full-screen.. should draw as a seperate Quad as an overlay
                    Graphics->Draw(Graphics->ContextWidth(), Graphics->ContextHeight(), (std::uint16_t*) screenshot, (std::uint16_t*) NULL);
                    Graphics->Swap();
                    sys_timer_usleep (1000 * 1000 * 1/2);
                } else {
                    // Clear screenshot
                }
            }
        } else {
            // Clear screenshot
        }
    }
    
*/

    
    Graphics->FlushDbgFont();
}

void do_shaderChoice()
{
    string path;

    if (tmpBrowser == NULL)
        tmpBrowser = new FileBrowser(util_concat(VICE_USRDIR, "shaders", NULL));

    if (CellInput->UpdateDevice(0) == CELL_PAD_OK)
    {
        UpdateBrowser(tmpBrowser);

        if (CellInput->WasButtonPressed(0,CTRL_CROSS))
        {
            if(tmpBrowser->IsCurrentADirectory())
            {
                tmpBrowser->PushDirectory(    tmpBrowser->GetCurrentDirectoryInfo().dir + "/" + tmpBrowser->GetCurrentEntry()->d_name,
                        CELL_FS_TYPE_REGULAR | CELL_FS_TYPE_DIRECTORY, "cg");
            }
            else if (tmpBrowser->IsCurrentAFile())
            {
                path = tmpBrowser->GetCurrentDirectoryInfo().dir + "/" + tmpBrowser->GetCurrentEntry()->d_name;

                //load shader
                Graphics->LoadFragmentShader(path);
                Graphics->SetSmooth(false);
                resources_set_int ("PS3HardwareFilter", false);
                menuStack.pop();



                // Display the screen render for a moment, so we know we want this shader
                // render was previously saved before the menu was launched.

                // TODO  On initial launch, the screen dump will be blank. We need a default image here.
                Graphics->DumpScreen();
                Graphics->Swap();
                sys_timer_usleep (1000 * 1000 * 2);
            }
        }

        if (CellInput->WasButtonHeld(0, CTRL_TRIANGLE)) {
            menuStack.pop();
        }
    }

    cellDbgFontPuts(0.09f, 0.88f, Emulator_GetFontSize(), YELLOW, "CROSS    - Select shader");
    cellDbgFontPuts(0.09f, 0.92f, Emulator_GetFontSize(), PURPLE, "TRIANGLE - return to settings");
    Graphics->FlushDbgFont();

    RenderBrowser(tmpBrowser);
}

void do_pathChoice()
{
    string path;

    if (tmpBrowser == NULL)
        tmpBrowser = new FileBrowser("/\0");

    if (CellInput->UpdateDevice(0) == CELL_PAD_OK)
    {
        UpdateBrowser(tmpBrowser);
        if (CellInput->WasButtonPressed(0,CTRL_SQUARE))
        {
            if(tmpBrowser->IsCurrentADirectory())
            {
                path = tmpBrowser->GetCurrentDirectoryInfo().dir + "/" + tmpBrowser->GetCurrentEntry()->d_name + "/";
                switch(currently_selected_path_setting)
                {
                    case SETTING_PATH_DEFAULT_ROM_DIRECTORY:
                                    resources_set_string("PS3PathRomDir", path.c_str());
                                    break;
                }
                menuStack.pop();
            }
        }
        if (CellInput->WasButtonPressed(0, CTRL_TRIANGLE))
        {
            path = VICE_USRDIR;
            switch(currently_selected_path_setting)
            {
                case SETTING_PATH_DEFAULT_ROM_DIRECTORY:
                                        resources_set_string("PS3PathRomDir", path.c_str());
                                        break;
            }
            menuStack.pop();
        }
        if (CellInput->WasButtonPressed(0,CTRL_CROSS))
        {
            if(tmpBrowser->IsCurrentADirectory())
            {
                tmpBrowser->PushDirectory(tmpBrowser->GetCurrentDirectoryInfo().dir + "/" + tmpBrowser->GetCurrentEntry()->d_name, CELL_FS_TYPE_REGULAR | CELL_FS_TYPE_DIRECTORY, "empty");
            }
        }
    }
            
    cellDbgFontPuts(0.05f, 0.88f, Emulator_GetFontSize(), YELLOW, "CROSS    - enter directory");
    cellDbgFontPuts(0.05f, 0.92f, Emulator_GetFontSize(), BLUE,   "SQUARE   - select directory as path");
    cellDbgFontPuts(0.55f, 0.92f, Emulator_GetFontSize(), PURPLE, "TRIANGLE - return to settings");
    Graphics->FlushDbgFont();
        
    RenderBrowser(tmpBrowser);
}



// void do_path_settings()
// // called from ROM menu by pressing the SELECT button
// // return to ROM menu by pressing the CIRCLE button
void do_path_settings()
{
    const char *res_string;

    if(CellInput->UpdateDevice(0) == CELL_OK)
    {
        // back to ROM menu if CIRCLE is pressed
        if (CellInput->WasButtonPressed(0, CTRL_L1) | CellInput->WasButtonPressed(0, CTRL_CIRCLE))
        {
            menuStack.pop();
            return;
        }

        if (CellInput->WasButtonPressed(0, CTRL_DOWN) | CellInput->WasAnalogPressedDown(0, CTRL_LSTICK))    // down to next setting
        {
            currently_selected_path_setting++;
            if (currently_selected_path_setting >= MAX_NO_OF_PATH_SETTINGS)
            {
                currently_selected_path_setting = 0;
            }
        }

        if (CellInput->WasButtonPressed(0,CTRL_START))
        {
            MenuStop();
            Emulator_StartROMRunning();
            return;
        }

        if (CellInput->WasButtonPressed(0, CTRL_UP) | CellInput->WasAnalogPressedUp(0, CTRL_LSTICK))    // up to previous setting
        {
            currently_selected_path_setting--;
            if (currently_selected_path_setting < 0)
            {
                currently_selected_path_setting = MAX_NO_OF_PATH_SETTINGS-1;
            }
        }
        switch(currently_selected_path_setting)
        {
            case SETTING_PATH_DEFAULT_ROM_DIRECTORY:
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_CROSS))
                {
                    menuStack.push(do_pathChoice);
                    tmpBrowser = NULL;
                }
                if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
                {
                    resources_set_string("PS3PathRomDir", "/");
                }
                break;
            case SETTING_PATH_SAVE_SETTINGS:
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_CROSS))
                {
                    resources_save(NULL);
                }
                if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
                {
                    resources_save(NULL);
                }
                break;
            case SETTING_PATH_DEFAULT_ALL:
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_CROSS))
                {
                    resources_set_string("PS3PathRomDir", "/");
                }
                break;
            default:
                break;
        } // end of switch
    }

    float yPos = 0.13;
    float ySpacing = 0.04;

    cellDbgFontPuts (0.09f, 0.05f, Emulator_GetFontSize(), BLUE,   "GENERAL");
    cellDbgFontPuts (0.3f,  0.05f, Emulator_GetFontSize(), BLUE,   "VICE");
    cellDbgFontPuts (0.51f, 0.05f, Emulator_GetFontSize(), PURPLE, "PATHS");
    Graphics->FlushDbgFont();

    resources_get_string ("PS3PathRomDir", &res_string);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_path_setting == SETTING_PATH_DEFAULT_ROM_DIRECTORY ? YELLOW : WHITE, "Startup ROM Directory");
    cellDbgFontPuts (0.5f,  yPos, Emulator_GetFontSize(), res_string == "/" ? YELLOW : GREEN, res_string);


    yPos += ySpacing;
    yPos += ySpacing;
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_path_setting == SETTING_PATH_SAVE_SETTINGS ? YELLOW : GREEN, "SAVE SETTINGS");
    Graphics->FlushDbgFont();


    yPos += ySpacing;
    cellDbgFontPrintf(0.09f, yPos, Emulator_GetFontSize(), currently_selected_path_setting == SETTING_PATH_DEFAULT_ALL ? YELLOW : GREEN, "DEFAULT");

    cellDbgFontPuts(0.05f, 0.88f, Emulator_GetFontSize(), YELLOW, "UP/DOWN   - select    X/LEFT/RIGHT - change          TRIANGLE - default");
    cellDbgFontPuts(0.05f, 0.92f, Emulator_GetFontSize(), YELLOW, util_concat("L1/CIRCLE - back      START        - return to ", machine_name, NULL));
    Graphics->FlushDbgFont();
}


void inline process_drive_change (const char *res, int default_type)
{
    int drive_type_index=0;
    int res_value;                                        
    resources_get_int(res, &res_value);

    for (int i = 0; i <= MAX_DRIVE_TYPES; i++) {
        // find the resource value in the list of values
        if (drive_type[i].id == res_value) {
            drive_type_index=i;
            break;
        }
    }
    if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK))
    {
        if (drive_type_index > 0) {
            drive_type_index--;
            res_value = drive_type[drive_type_index].id;
            resources_set_int(res, res_value);
        }
    }
    if(CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0,CTRL_CROSS))
    {
        if (drive_type_index < MAX_DRIVE_TYPES) {
            drive_type_index++;
            res_value = drive_type[drive_type_index].id;
            resources_set_int(res, res_value);
        }
    }
    if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
    {
        res_value = default_type;
        resources_set_int(res, res_value);
    }
}


void do_vice_settings()
{
    int res_value;
    int model_index=0;

    int sid_engine;
    int sid_model;
    int SidEngineModel;

    bool SidOptionPassbandDisabled = false;
    bool MouseDisabled = false;

                                            
    resources_get_int("SidEngine", &res_value);
    if (res_value == SID_ENGINE_FASTSID) {
        SidOptionPassbandDisabled = true;
    } else if (res_value == SID_ENGINE_RESID) {
        int sampling_mode;
        if (resources_get_int("SidResidSampling", &sampling_mode) == 0) {
            if (sampling_mode == SAMPLE_FAST) {
                SidOptionPassbandDisabled = true;
            }
            if (sampling_mode == SAMPLE_INTERPOLATE) {
                SidOptionPassbandDisabled = true;
            }
            if (sampling_mode == SAMPLE_RESAMPLE_INTERPOLATE) {
                SidOptionPassbandDisabled = false;
            }
        }
    } else if (res_value == SID_ENGINE_RESID_FP) {
        int sampling_mode;
        if (resources_get_int("SidResidSampling", &sampling_mode) == 0) {
            if (sampling_mode == SAMPLE_FAST) {
                SidOptionPassbandDisabled = true;
            }
            if (sampling_mode == SAMPLE_INTERPOLATE) {
                SidOptionPassbandDisabled = true;
            }
            if (sampling_mode == SAMPLE_RESAMPLE_INTERPOLATE) {
                SidOptionPassbandDisabled = false;
            }
        }
    }

    if (resources_get_int("Mouse", &res_value) == 0)
        MouseDisabled = !res_value;


    if(CellInput->UpdateDevice(0) == CELL_PAD_OK)
    {
        // back to ROM menu if CIRCLE is pressed
        if (CellInput->WasButtonPressed(0, CTRL_L1) | CellInput->WasButtonPressed(0, CTRL_CIRCLE))
        {
            menuStack.pop();
            return;
        }

        if (CellInput->WasButtonPressed(0, CTRL_R1))
        {
            menuStack.push(do_path_settings);
            return;
        }

        if (CellInput->WasButtonPressed(0, CTRL_DOWN) | CellInput->WasAnalogPressedDown(0, CTRL_LSTICK))    // down to next setting
        {
            currently_selected_vice_setting++;
            if (currently_selected_vice_setting >= MAX_NO_OF_VICE_SETTINGS)
            {
                currently_selected_vice_setting = 0;
            }
        }
        if (CellInput->WasButtonPressed(0, CTRL_UP) | CellInput->WasAnalogPressedUp(0, CTRL_LSTICK))    // up to previous setting
        {
            currently_selected_vice_setting--;
            if (currently_selected_vice_setting < 0)
            {
                currently_selected_vice_setting = MAX_NO_OF_VICE_SETTINGS-1;
            }
    }
            
    if (CellInput->WasButtonPressed(0,CTRL_START))
    {
        MenuStop();
        Emulator_StartROMRunning();
        return;
    }

    bool dirty=false;
    switch(currently_selected_vice_setting)
    {
        // display framerate on/off
        case SETTING_VICE_DISPLAY_FRAMERATE:
            resources_get_int("DisplayFrameRate", &res_value);
            if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_CROSS))
            {
                res_value = !res_value;
                resources_set_int("DisplayFrameRate", res_value);
            }
            if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
            {
                res_value = false;
                resources_set_int("DisplayFrameRate", res_value);
            }
            break;
        case SETTING_VICE_DISPLAY_DRIVE_INDICATORS:
            resources_get_int("DisplayDriveIndicators", &res_value);
            if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_CROSS))
            {
                res_value = !res_value;
                resources_set_int("DisplayDriveIndicators", res_value);
            }
            if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
            {
                res_value = true;
                resources_set_int("DisplayDriveIndicators", res_value);
            }
            break;
        case SETTING_VICE_SID_FILTERS:
            resources_get_int("SidFilters", &res_value);
            if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_CROSS))
            {
                res_value = !res_value;
                resources_set_int("SidFilters", res_value);
            }
            if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
            {
                res_value = true;
                resources_set_int("SidFilters", res_value);
            }
            break;
        case SETTING_VICE_SID_MODEL:
            dirty=false;

            resources_get_int("SidEngine", &sid_engine);
            resources_get_int("SidModel", &sid_model);
            SidEngineModel = (sid_engine << 8) | sid_model;

            for (int i=MIN_ALL_SID_MODELS; i <= MAX_ALL_SID_MODELS; i++)
            {
                if (ui_sid_engine_model_id[i] == SidEngineModel)
                {
                    model_index=i;
                    break;
                }
            }

            if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK))
            {
                if (model_index > MIN_ALL_SID_MODELS) {
                    model_index--;
                    SidEngineModel = ui_sid_engine_model_id[model_index];
                    dirty=true;
                }
            }
            if(CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0,CTRL_CROSS))
            {
                if (model_index < MAX_ALL_SID_MODELS) {
                    model_index++;
                    SidEngineModel = ui_sid_engine_model_id[model_index];
                    dirty=true;
                }
            }
            if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
            {
                SidEngineModel = SID_RESIDFP_6581R4AR_3789;
                dirty=true;
            }

            if (dirty==false)
                break;


            sid_engine = SidEngineModel >> 8;
            sid_model  = SidEngineModel & 0xff;

            for (int i=MIN_ALL_SID_MODELS; i <= MAX_ALL_SID_MODELS; i++)
            {
                if (ui_sid_engine_model_id[i] == SidEngineModel)
                {
                    model_index=i;
                    break;
                }
            }

            resources_set_int("SidEngine", sid_engine);
            if (model_index < MIN_RESID_MODELS)
                resources_set_int("SidResidSampling", SAMPLE_FAST);
            else if (model_index < MIN_RESID_FP_MODELS)
                resources_set_int("SidResidSampling", SAMPLE_INTERPOLATE);
            else
                resources_set_int("SidResidSampling", SAMPLE_RESAMPLE_INTERPOLATE);

            resources_set_int("SidModel", sid_model);

            //sid_set_engine_model(sid_engine, sid_model);

            break;
        case SETTING_VICE_SID_RESID_PASSBAND:
            if (SidOptionPassbandDisabled)
                break;

            resources_get_int("SidResidPassband", &res_value);
            if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK))
            {
                if (res_value >= 30)
                {
                    res_value -= 10;
                    resources_set_int("SidResidPassband", res_value);
                }
            }
            if(CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0,CTRL_CROSS))
            {
                if (res_value <= 80)
                {
                    res_value += 10;
                    resources_set_int("SidResidPassband", res_value);
                }
            }
            if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
            {
                res_value = 90;
                resources_set_int("SidResidPassband", res_value);
            }
            break;
        case SETTING_VICE_MOUSE_SUPPORT:
            resources_get_int("Mouse", &res_value);
            if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_CROSS))
            {
                res_value = !res_value;
                resources_set_int("Mouse", res_value);
            }
            if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
            {
                res_value = false;
                resources_set_int("Mouse", res_value);
            }
            break;
        case SETTING_VICE_MOUSE_TYPE:
            if (MouseDisabled)
                break;

            resources_get_int("Mousetype", &res_value);
            if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_CROSS))
            {
                res_value = (res_value==MOUSE_TYPE_1351) ? MOUSE_TYPE_NEOS : MOUSE_TYPE_1351;
                resources_set_int("Mousetype", res_value);
            }
            if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
            {
                res_value = MOUSE_TYPE_1351;
                resources_set_int("Mousetype", res_value);
            }
            break;
        case SETTING_VICE_KEYMAP:
            resources_get_int("KeymapIndex", &res_value);
            if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK))
            {
                if (res_value > 0) {
                    res_value--;
                    resources_set_int("KeymapIndex", res_value);
                }
            }
            if(CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK))
            {
                if (res_value < 1) {
                    res_value++;
                    resources_set_int("KeymapIndex", res_value);
                }
            }
            if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
            {
                res_value = 0;
                resources_set_int("KeymapIndex", res_value);
            }
            break;
        case SETTING_VICE_DRIVE8_TYPE:
            process_drive_change("Drive8Type", DRIVE_TYPE_1541);
            break;
        case SETTING_VICE_DRIVE9_TYPE:
            process_drive_change("Drive9Type", DRIVE_TYPE_NONE);
            break;
        case SETTING_VICE_DRIVE10_TYPE:
            process_drive_change("Drive10Type", DRIVE_TYPE_NONE);
            break;
        case SETTING_VICE_DRIVE11_TYPE:
            process_drive_change("Drive11Type", DRIVE_TYPE_NONE);
            break;
        case SETTING_VICE_HARD_RESET:
            if (CellInput->WasButtonPressed(0, CTRL_CROSS))
            {
                machine_trigger_reset(MACHINE_RESET_MODE_HARD);
            }
            break;
        case SETTING_VICE_SAVE_SETTINGS:
            if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_CROSS))
            {
                resources_save(NULL);
            }
            if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
            {
                resources_save(NULL);
            }
            break;
        case SETTING_VICE_DEFAULT_ALL:
            if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK) | CellInput->IsButtonPressed(0, CTRL_TRIANGLE) | CellInput->WasButtonPressed(0, CTRL_CROSS))
            {
                resources_set_int ("DisplayFrameRate", false);
                resources_set_int ("DisplayDriveIndicators", true);
                resources_set_int ("SidFilters", true);
                resources_set_int ("SidEngine", SID_ENGINE_RESID_FP);
                resources_set_int ("SidResidSampling", SAMPLE_RESAMPLE_INTERPOLATE);
                resources_set_int ("SidEngine", (SID_RESIDFP_6581R4AR_3789 >> 8) );
                resources_set_int ("SidModel",  (SID_RESIDFP_6581R4AR_3789 & 0xff) );
                resources_set_int ("SidResidPassband", 90);
                resources_set_int ("Mouse", false);
                resources_set_int ("Mousetype", MOUSE_TYPE_1351);
                resources_set_int ("Drive8Type", DRIVE_TYPE_1541);
                resources_set_int ("Drive9Type", DRIVE_TYPE_NONE);
                resources_set_int ("Drive10Type", DRIVE_TYPE_NONE);
                resources_set_int ("Drive11Type", DRIVE_TYPE_NONE);
            }
            break;

        default:
            break;
        } // end of switch
    }

    float yPos=0.05f;
    float ybrk=0.04f;

    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), BLUE,   "GENERAL");
    cellDbgFontPuts (0.3f,  yPos, Emulator_GetFontSize(), PURPLE, "VICE");
    cellDbgFontPuts (0.51f, yPos, Emulator_GetFontSize(), BLUE,   "PATHS");
    Graphics->FlushDbgFont();

    yPos+=ybrk;
    yPos+=ybrk;
    resources_get_int("DisplayFrameRate", &res_value);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_DISPLAY_FRAMERATE ? YELLOW : WHITE, "Display framerate");
    cellDbgFontPuts (0.5f,  yPos, Emulator_GetFontSize(), res_value == false ? GREEN : RED, res_value == true ? "ON" : "OFF");
    Graphics->FlushDbgFont();

    yPos+=ybrk;
    resources_get_int("DisplayDriveIndicators", &res_value);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_DISPLAY_DRIVE_INDICATORS ? YELLOW : WHITE, "Show Disk Activity Indicators");
    cellDbgFontPuts (0.5f,  yPos, Emulator_GetFontSize(), res_value == false ? GREEN : RED, res_value == true ? "ON" : "OFF");
    Graphics->FlushDbgFont();

    yPos+=ybrk;
    resources_get_int("SidFilters", &res_value);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_SID_FILTERS ? YELLOW : WHITE, "SID Filtering Enabled");
    cellDbgFontPuts (0.5f,  yPos, Emulator_GetFontSize(), res_value == true ? GREEN : RED, res_value == true ? "ON" : "OFF");
    Graphics->FlushDbgFont();


    yPos+=ybrk;
    resources_get_int("SidEngine", &res_value);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_SID_ENGINE ? YELLOW : WHITE, "SID Engine");

    switch (res_value) {
        case SID_ENGINE_FASTSID:
            cellDbgFontPuts        (0.5f, yPos, Emulator_GetFontSize(), GRAY, "FastSID   (low quality)");
            break;
        case SID_ENGINE_RESID:
            cellDbgFontPuts        (0.5f, yPos, Emulator_GetFontSize(), GRAY, "ReSID     (medium quality)");
            break;
        case SID_ENGINE_RESID_FP:
            cellDbgFontPuts        (0.5f, yPos, Emulator_GetFontSize(), GRAY, "ReSID-FP  (highest quality)");
            break;
    }
    Graphics->FlushDbgFont();




    yPos+=ybrk;
    resources_get_int("SidResidSampling", &res_value);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_SID_RESID_SAMPLING ? YELLOW : WHITE, "ReSID Sampling Mode");
    Graphics->FlushDbgFont();

    switch (res_value) {
        case SAMPLE_FAST:
             cellDbgFontPuts (0.5f, yPos, Emulator_GetFontSize(), GRAY, "Fast (lowest quality)");
             break;
        case SAMPLE_INTERPOLATE:
             cellDbgFontPuts (0.5f, yPos, Emulator_GetFontSize(), GRAY, "Interpolate (medium quality)");
             break;
        case SAMPLE_RESAMPLE_INTERPOLATE:
             cellDbgFontPuts (0.5f, yPos, Emulator_GetFontSize(), GRAY, "Resampling (highest quality)");
             break;
    }



    yPos+=ybrk;
    resources_get_int("SidEngine", &sid_engine);
    resources_get_int("SidModel", &sid_model);
    SidEngineModel = (sid_engine << 8) | sid_model;
    for (int i=MIN_ALL_SID_MODELS; i <= MAX_ALL_SID_MODELS; i++)
    {
        if (ui_sid_engine_model_id[i] == SidEngineModel)
        {
            model_index=i;
            break;
        }
    }
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_SID_MODEL ? YELLOW : WHITE, "SID Model");
    cellDbgFontPuts (0.5f,  yPos, Emulator_GetFontSize(), ui_sid_engine_model_id[model_index] == SID_RESIDFP_6581R4AR_3789 ? GREEN : RED, ui_sid_engine_model[model_index]);
    Graphics->FlushDbgFont();



    yPos+=ybrk;
    resources_get_int("SidResidPassband", &res_value);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_SID_RESID_PASSBAND ? YELLOW : WHITE, "ReSID Passband Filter (20-90)");
    cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), SidOptionPassbandDisabled==true ? GRAY : (res_value == 90 ? GREEN : RED), "%d", res_value);
    Graphics->FlushDbgFont();


    yPos+=ybrk;
    resources_get_int("Mouse", &res_value);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_MOUSE_SUPPORT ? YELLOW : WHITE, "Mouse Enabled");
    cellDbgFontPuts (0.5f,  yPos, Emulator_GetFontSize(), res_value == false ? GREEN : RED, res_value == true ? "ON":"OFF");
    Graphics->FlushDbgFont();


    yPos+=ybrk;
    resources_get_int("Mousetype", &res_value);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_MOUSE_TYPE ? YELLOW : WHITE, "Mouse Type");
    cellDbgFontPuts (0.5f,  yPos, Emulator_GetFontSize(), MouseDisabled==true ? GRAY : (res_value == MOUSE_TYPE_1351 ? GREEN : RED), res_value == MOUSE_TYPE_1351 ? "CBM 1351" : "NEOS");
    Graphics->FlushDbgFont();


    yPos+=ybrk;
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_KEYMAP ? YELLOW : WHITE, "Keyboard Mapping");
    resources_get_int("KeymapIndex", &res_value);
    cellDbgFontPuts (0.5f,  yPos, Emulator_GetFontSize(), res_value == 0 ? GREEN : RED, res_value == 0 ? "Symbolic" : "Positional");
    Graphics->FlushDbgFont();


    yPos+=ybrk;
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_DRIVE8_TYPE ? YELLOW : WHITE, "Drive 8");
    resources_get_int("Drive8Type", &res_value);
    // find the resource value in the list of values
    for (int i = 0; i <= MAX_DRIVE_TYPES; i++) {
        if (drive_type[i].id == res_value) {
            cellDbgFontPuts (0.5f, yPos, Emulator_GetFontSize(), res_value == DRIVE_TYPE_1541 ? GREEN : RED, drive_type[i].name);
            break;
        }
    }
    Graphics->FlushDbgFont();

    yPos+=ybrk;
    resources_get_int("Drive9Type", &res_value);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_DRIVE9_TYPE ? YELLOW : WHITE, "Drive 9");
    // find the resource value in the list of values
    for (int i = 0; i <= MAX_DRIVE_TYPES; i++) {
        if (drive_type[i].id == res_value) {
            cellDbgFontPuts (0.5f, yPos, Emulator_GetFontSize(), res_value == DRIVE_TYPE_NONE ? GREEN : RED, drive_type[i].name);
            break;
        }
    }
    Graphics->FlushDbgFont();

    yPos+=ybrk;
    resources_get_int("Drive10Type", &res_value);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_DRIVE10_TYPE ? YELLOW : WHITE, "Drive 10");
    // find the resource value in the list of values
    for (int i = 0; i <= MAX_DRIVE_TYPES; i++) {
        if (drive_type[i].id == res_value) {
            cellDbgFontPuts (0.5f, yPos, Emulator_GetFontSize(), res_value == DRIVE_TYPE_NONE ? GREEN : RED, drive_type[i].name);
            break;
        }
    }
    Graphics->FlushDbgFont();

    yPos+=ybrk;
    resources_get_int("Drive11Type", &res_value);
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_DRIVE11_TYPE ? YELLOW : WHITE, "Drive 11");
    // find the resource value in the list of values
    for (int i = 0; i <= MAX_DRIVE_TYPES; i++) {
        if (drive_type[i].id == res_value) {
            cellDbgFontPuts (0.5f, yPos, Emulator_GetFontSize(), res_value == DRIVE_TYPE_NONE ? GREEN : RED, drive_type[i].name);
            break;
        }
    }
    Graphics->FlushDbgFont();


    yPos+=ybrk;
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_HARD_RESET ? YELLOW : GREEN, "ISSUE HARD RESET");
    Graphics->FlushDbgFont();


    yPos+=ybrk;
    yPos+=ybrk;
    cellDbgFontPuts (0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_SAVE_SETTINGS ? YELLOW : GREEN, "SAVE SETTINGS");
    Graphics->FlushDbgFont();


    yPos+=ybrk;
    cellDbgFontPrintf(0.09f, yPos, Emulator_GetFontSize(), currently_selected_vice_setting == SETTING_VICE_DEFAULT_ALL ? YELLOW : GREEN, "DEFAULT");
    Graphics->FlushDbgFont();


    cellDbgFontPuts(0.05f, 0.88f, Emulator_GetFontSize(), YELLOW,             "UP/DOWN   - select    X/LEFT/RIGHT - change          TRIANGLE - default");
    cellDbgFontPuts(0.05f, 0.92f, Emulator_GetFontSize(), YELLOW, util_concat("L1/CIRCLE - back      START        - return to ", machine_name, "   R1       - forward", NULL));


    Graphics->FlushDbgFont();
}

void do_general_settings()
{
    int res_int;
    int overscan_enabled;
    int overscan_amount;

    if(CellInput->UpdateDevice(0) == CELL_PAD_OK)
    {
        // back to ROM menu if CIRCLE is pressed
        if (CellInput->WasButtonPressed(0, CTRL_CIRCLE))
        {
            menuStack.pop();
            return;
        }

        if (CellInput->WasButtonPressed(0, CTRL_R1))
        {
            menuStack.push(do_vice_settings);
            return;
        }

        if (CellInput->WasButtonPressed(0, CTRL_DOWN) | CellInput->WasAnalogPressedDown(0, CTRL_LSTICK))    // down to next setting
        {
            currently_selected_setting++;
            if (currently_selected_setting >= MAX_NO_OF_SETTINGS)
            {
                currently_selected_setting = 0;
            }
        }

        if (CellInput->WasButtonPressed(0, CTRL_UP) | CellInput->WasAnalogPressedUp(0, CTRL_LSTICK))    // up to previous setting
        {
            currently_selected_setting--;
            if (currently_selected_setting < 0)
            {
                currently_selected_setting = MAX_NO_OF_SETTINGS-1;
            }
        }

        if (CellInput->WasButtonPressed(0,CTRL_START))
        {
            MenuStop();
            Emulator_StartROMRunning();
            return;
        }

        switch(currently_selected_setting)
        {
            // display framerate on/off
            case SETTING_CHANGE_RESOLUTION:
                resources_get_int ("PS3Pal60", &res_int);
                if(CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK))
                {
                    Graphics->NextResolution();
                }
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK))
                {
                    Graphics->PreviousResolution();
                }
                if(CellInput->WasButtonPressed(0, CTRL_CROSS))
                {
                    if (Graphics->GetCurrentResolution() == CELL_VIDEO_OUT_RESOLUTION_576)
                    {
                        if(Graphics->CheckResolution(CELL_VIDEO_OUT_RESOLUTION_576))
                        {
                            Graphics->SetPAL60Hz(res_int);
                            Graphics->SwitchResolution(Graphics->GetCurrentResolution(), res_int);
                        }
                    }
                    else
                    {
                        Graphics->SetPAL60Hz(false);
                        Graphics->SwitchResolution(Graphics->GetCurrentResolution(), false);
                    }
                }
                if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
                {
                    Graphics->SwitchResolution(Graphics->GetInitialResolution(), res_int);
                }
                break;
            case SETTING_PAL60_MODE:
                resources_get_int ("PS3Pal60", &res_int);
                if(CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0,CTRL_CROSS) | CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK))
                {
                    if (Graphics->GetCurrentResolution() == CELL_VIDEO_OUT_RESOLUTION_576)
                    {
                        if(Graphics->CheckResolution(CELL_VIDEO_OUT_RESOLUTION_576))
                        {
                            res_int = ! res_int;
                            resources_set_int ("PS3Pal60", res_int);
                            Graphics->SetPAL60Hz(res_int);
                            Graphics->SwitchResolution(Graphics->GetCurrentResolution(), res_int);
                        }
                    }
 
                }
                break;
            case SETTING_SHADER:
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_CROSS))
                {
                    menuStack.push(do_shaderChoice);
                    tmpBrowser = NULL;
                }
                if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
                {
                    Graphics->LoadFragmentShader(DEFAULT_SHADER_FILE);
                    Graphics->SetSmooth(false);
                    resources_set_int ("PS3HardwareFilter", false);
                }
                break;
            case SETTING_FONT_SIZE:
                resources_get_int ("PS3FontSize", &res_int);
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0,CTRL_CROSS))
                {
                    if(res_int > -100)
                        res_int--;
                }
                if(CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0,CTRL_CROSS))
                {
                    if((res_int < 100))
                        res_int++;
                }
                if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
                {
                    res_int = 100;
                }
                resources_set_int ("PS3FontSize", res_int);
                break;
            case SETTING_KEEP_ASPECT_RATIO:
                resources_get_int ("PS3KeepAspect", &res_int);
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0,CTRL_CROSS))
                {
                    res_int = !res_int;
                    Graphics->SetAspectRatio(res_int);
                }
                if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
                {
                    res_int = false;
                    Graphics->SetAspectRatio(res_int);
                }
                resources_set_int ("PS3KeepAspect", res_int);
                break;
            case SETTING_HW_TEXTURE_FILTER:
                resources_get_int ("PS3HardwareFilter", &res_int);
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0,CTRL_CROSS))
                {
                    res_int = !res_int;
                    Graphics->SetSmooth(res_int);
                }
                if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
                {
                    res_int = false;
                    Graphics->SetSmooth(res_int);
                }
                resources_set_int ("PS3HardwareFilter", res_int);
                break;
            case SETTING_HW_OVERSCAN_AMOUNT:
                resources_get_int ("PS3OverscanAmount", &overscan_amount);
                resources_get_int ("PS3Overscan", &overscan_enabled);
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0,CTRL_CROSS))
                {
                    if(overscan_amount > -40)
                    {
                        overscan_amount--;
                        overscan_enabled = true;
                        Graphics->SetOverscan(overscan_enabled, (float)overscan_amount/100);
                    }
                    if(overscan_amount == 0)
                    {
                        overscan_enabled = false;
                        Graphics->SetOverscan(overscan_enabled, (float)overscan_amount/100);
                    }
                }
                if(CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0,CTRL_CROSS))
                {
                    if((overscan_amount < 40))
                    {
                        overscan_amount++;
                        overscan_enabled = true;
                        Graphics->SetOverscan(overscan_enabled, (float)overscan_amount/100);
                    }
                    if(overscan_amount == 0)
                    {
                        overscan_enabled = false;
                        Graphics->SetOverscan(overscan_enabled, (float)overscan_amount/100);
                    }
                }
                if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
                {
                    overscan_amount = 0;
                    overscan_enabled = false;
                    Graphics->SetOverscan(overscan_enabled, (float)overscan_amount/100);
                }
                resources_set_int ("PS3Overscan", overscan_enabled);
                resources_set_int ("PS3OverscanAmount", overscan_amount);
                break;

/*
            case SETTING_RSOUND_ENABLED:
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0,CTRL_CROSS))
                {
                    Settings.RSoundEnabled = !Settings.RSoundEnabled;
                    Emulator_Implementation_ToggleSound();
                }
                if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
                {
                    Settings.RSoundEnabled = false;
                    Emulator_Implementation_ToggleSound();
                }
                break;
            case SETTING_RSOUND_SERVER_IP_ADDRESS:
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasButtonPressed(0, CTRL_CROSS) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK))
                {
                    Emulator_OSKStart(L"Enter the IP address for the RSound Server. Example (below):", L"192.168.1.1");
                    Settings.RSoundServerIPAddress = Emulator_OSKOutputString();
                }
                if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
                {
                    Settings.RSoundServerIPAddress = "0.0.0.0";
                }
                break;
*/

            case SETTING_SAVE_SETTINGS:
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0, CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_CROSS))
                {
                    resources_save(NULL);
                }
                if(CellInput->IsButtonPressed(0, CTRL_TRIANGLE))
                {
                    resources_save(NULL);
                }
                break;
            case SETTING_DEFAULT_ALL:
                if(CellInput->WasButtonPressed(0, CTRL_LEFT) | CellInput->WasAnalogPressedLeft(0,CTRL_LSTICK) | CellInput->WasButtonPressed(0, CTRL_RIGHT) | CellInput->WasAnalogPressedRight(0,CTRL_LSTICK) | CellInput->IsButtonPressed(0, CTRL_TRIANGLE) | CellInput->WasButtonPressed(0, CTRL_CROSS))
                {
                    resources_set_int ("PS3KeepAspect", false);
                    resources_set_int ("PS3HardwareFilter", false);
                    resources_set_int ("PS3Overscan", false);
                    resources_set_int ("PS3OverscanAmount", 0);
                    resources_set_int ("PS3KeepAspect", false);
                    resources_set_int ("PS3Pal60", false);
                    resources_set_int ("PS3FontSize", 100);

                    Graphics->SetAspectRatio(false);
                    Graphics->SetSmooth(false);
                    Graphics->SetOverscan(false, (float)0);
                    Graphics->SetPAL60Hz(false);
                    //Settings.RSoundEnabled = false;
                    //Settings.RSoundServerIPAddress = "0.0.0.0";
                }
                break;
            default:
                break;
        } // end of switch
    }

    cellDbgFontPuts (0.09f, 0.05f, Emulator_GetFontSize(), PURPLE, "GENERAL");
    cellDbgFontPuts (0.3f,  0.05f, Emulator_GetFontSize(), BLUE,   "VICE");
    cellDbgFontPuts (0.51f, 0.05f, Emulator_GetFontSize(), BLUE,   "PATHS");

    float yPos = 0.13;
    float ySpacing = 0.04;

    cellDbgFontPuts(0.09f, yPos, Emulator_GetFontSize(), currently_selected_setting == SETTING_CHANGE_RESOLUTION ? YELLOW : WHITE, "Resolution");

    switch(Graphics->GetCurrentResolution())
    {
        case CELL_VIDEO_OUT_RESOLUTION_480:
            cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), Graphics->GetInitialResolution() == CELL_VIDEO_OUT_RESOLUTION_480 ? GREEN : RED, "720x480 (480p)");
            break;
        case CELL_VIDEO_OUT_RESOLUTION_720:
            cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), Graphics->GetInitialResolution() == CELL_VIDEO_OUT_RESOLUTION_720 ? GREEN : RED, "1280x720 (720p)");
            break;
        case CELL_VIDEO_OUT_RESOLUTION_1080:
            cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), Graphics->GetInitialResolution() == CELL_VIDEO_OUT_RESOLUTION_1080 ? GREEN : RED, "1920x1080 (1080p)");
            break;
        case CELL_VIDEO_OUT_RESOLUTION_576:
            cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), Graphics->GetInitialResolution() == CELL_VIDEO_OUT_RESOLUTION_576 ? GREEN : RED, "720x576 (576p)");
            break;
        case CELL_VIDEO_OUT_RESOLUTION_1600x1080:
            cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), Graphics->GetInitialResolution() == CELL_VIDEO_OUT_RESOLUTION_1600x1080 ? GREEN : RED, "1600x1080");
            break;
        case CELL_VIDEO_OUT_RESOLUTION_1440x1080:
            cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), Graphics->GetInitialResolution() == CELL_VIDEO_OUT_RESOLUTION_1440x1080 ? GREEN : RED, "1440x1080");
            break;
        case CELL_VIDEO_OUT_RESOLUTION_1280x1080:
            cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), Graphics->GetInitialResolution() == CELL_VIDEO_OUT_RESOLUTION_1280x1080 ? GREEN : RED, "1280x1080");
            break;
        case CELL_VIDEO_OUT_RESOLUTION_960x1080:
            cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), Graphics->GetInitialResolution() == CELL_VIDEO_OUT_RESOLUTION_960x1080 ? GREEN : RED, "960x1080");
            break;
    }
    Graphics->FlushDbgFont();

    yPos += ySpacing;
                           
    resources_get_int ("PS3Pal60", &res_int);
    cellDbgFontPuts   (0.09f, yPos, Emulator_GetFontSize(), currently_selected_setting == SETTING_PAL60_MODE ? YELLOW : WHITE,    "PAL60 Mode (576p only)");
    cellDbgFontPrintf (0.5f,  yPos, Emulator_GetFontSize(), res_int == true ? RED : GREEN, res_int == true ? "ON" : "OFF");

    yPos += ySpacing;
    cellDbgFontPuts(0.09f, yPos, Emulator_GetFontSize(), currently_selected_setting == SETTING_SHADER ? YELLOW : WHITE, "Selected shader");
    cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), GREEN, "%s", Graphics->GetFragmentShaderPath().substr(Graphics->GetFragmentShaderPath().find_last_of('/')).c_str());

    yPos += ySpacing;
    resources_get_int ("PS3FontSize", &res_int);
    cellDbgFontPuts(0.09f,  yPos, Emulator_GetFontSize(), currently_selected_setting == SETTING_FONT_SIZE ? YELLOW : WHITE, "Font size");
    cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), res_int == 100 ? GREEN : RED, "%f", Emulator_GetFontSize());

    yPos += ySpacing;
    resources_get_int ("PS3KeepAspect", &res_int);
    cellDbgFontPuts(0.09f, yPos, Emulator_GetFontSize(), currently_selected_setting == SETTING_KEEP_ASPECT_RATIO ? YELLOW : WHITE, "Aspect Ratio");
    cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), (res_int == false) ? GREEN : RED, "%s", res_int == true ? "Scaled" : "Stretched");
    Graphics->FlushDbgFont();

    yPos += ySpacing;
    resources_get_int ("PS3HardwareFilter", &res_int);
    cellDbgFontPuts(0.09f, yPos, Emulator_GetFontSize(), currently_selected_setting == SETTING_HW_TEXTURE_FILTER ? YELLOW : WHITE, "Hardware Filtering");
    cellDbgFontPrintf(0.5f, yPos, Emulator_GetFontSize(), res_int == false ? GREEN : RED, "%s", res_int == true ? "Linear interpolation" : "Point filtering");

    yPos += ySpacing;
    resources_get_int ("PS3OverscanAmount", &res_int);
    cellDbgFontPuts   (0.09f, yPos, Emulator_GetFontSize(), currently_selected_setting == SETTING_HW_OVERSCAN_AMOUNT ? YELLOW : WHITE, "Overscan");
    cellDbgFontPrintf (0.5f,  yPos, Emulator_GetFontSize(), res_int == 0 ? GREEN : RED, "%f", (float)res_int/100);


/*
    yPos += ySpacing;
    cellDbgFontPuts(0.09f, yPos, Emulator_GetFontSize(), currently_selected_setting == SETTING_RSOUND_ENABLED ? YELLOW : WHITE, "Sound");
    cellDbgFontPuts(0.5f, yPos, Emulator_GetFontSize(), Settings.RSoundEnabled == false ? GREEN : RED, Settings.RSoundEnabled == true ? "RSound" : "Normal");

    yPos += ySpacing;
    cellDbgFontPuts(0.09f, yPos, Emulator_GetFontSize(), currently_selected_setting == SETTING_RSOUND_SERVER_IP_ADDRESS ? YELLOW : WHITE, "RSound Server IP Address");
    cellDbgFontPuts(0.5f, yPos, Emulator_GetFontSize(), strcmp(Settings.RSoundServerIPAddress,"0.0.0.0") ? RED : GREEN, Settings.RSoundServerIPAddress);
*/

    yPos += ySpacing;
    yPos += ySpacing;
    cellDbgFontPuts  (0.09f, yPos, Emulator_GetFontSize(), currently_selected_setting == SETTING_SAVE_SETTINGS ? YELLOW : GREEN, "SAVE SETTINGS");
    Graphics->FlushDbgFont();

    yPos += ySpacing;
    cellDbgFontPrintf(0.09f, yPos, Emulator_GetFontSize(), currently_selected_setting == SETTING_DEFAULT_ALL ? YELLOW : GREEN, "DEFAULT");
    Graphics->FlushDbgFont();


    cellDbgFontPuts(0.05f, 0.88f, Emulator_GetFontSize(), YELLOW,              "UP/DOWN   - select    X/LEFT/RIGHT - change          TRIANGLE - default");
    cellDbgFontPuts(0.05f, 0.92f, Emulator_GetFontSize(), YELLOW, util_concat ("CIRCLE    - back      START        - return to ", machine_name, "   R1       - forward", NULL));

    Graphics->FlushDbgFont();
}

void do_ROMMenu()
{
    string rom_path;
    int res_int;

    if (CellInput->UpdateDevice(0) == CELL_PAD_OK)
    {
        UpdateBrowser(browser);

        if (CellInput->WasButtonPressed(0,CTRL_SELECT))
        {
            menuStack.push(do_general_settings);
        }

        if (CellInput->WasButtonPressed(0,CTRL_CROSS))
        {
            if(browser->IsCurrentADirectory())
            {
                browser->PushDirectory( browser->GetCurrentDirectoryInfo().dir + "/" + browser->GetCurrentEntry()->d_name,
                        CELL_FS_TYPE_REGULAR | CELL_FS_TYPE_DIRECTORY,
                        "d64|d71|d80|d81|d82|g64||g41|x64|t64|tap|prg|p00|crt|bin|D64|D71|D81|D82|G64|G41|X64|T64|TAP|PRG|P00|CRT|BIN|zip|ZIP|gz|GZ|d6z|D6Z|d7z|D7Z|d8z|D8Z|g6z|G6Z|g4z|G4Z|x6z|X6Z");
            }
            else if (browser->IsCurrentAFile())
            {
                // load game (standard controls), go back to main loop
                rom_path = browser->GetCurrentDirectoryInfo().dir + "/" + browser->GetCurrentEntry()->d_name;

                MenuStop();

                // switch emulator to emulate mode
                Emulator_StartROMRunning();


                //FIXME: 1x dirty const char* to char* casts... menu sucks.
                Emulator_RequestLoadROM((char*)rom_path.c_str(), true, false);

                return;
            }
        }
        if (CellInput->WasButtonPressed(0,CTRL_SQUARE))
        {
            if (browser->IsCurrentAFile())
            {
                // Load the disk image.
                // Reset the emulator with FastLoad disabled

                rom_path = browser->GetCurrentDirectoryInfo().dir + "/" + browser->GetCurrentEntry()->d_name;
                MenuStop();

                // switch emulator to emulate mode
                Emulator_StartROMRunning();
                Emulator_RequestLoadROM((char*)rom_path.c_str(), true, true);
                return;
            }
        }
/*
        if (CellInput->WasButtonPressed(0,CTRL_TRIANGLE) )
                {
            if (browser->IsCurrentAFile())
            {
                                // Load without any disk image. Do "not" reset the machine
                rom_path = browser->GetCurrentDirectoryInfo().dir + "/" + browser->GetCurrentEntry()->d_name;
                                MenuStop();

                                // switch emulator to emulate mode
                                Emulator_StartROMRunning();
                Emulator_RequestLoadROM((char*)rom_path.c_str(), false, true);
                                return;
                        }
                }
*/
        if (CellInput->IsButtonPressed(0,CTRL_TRIANGLE) && (CellInput->WasButtonPressed(0,CTRL_L1)))
        {
            if (browser->IsCurrentAFile())
            {
                // Insert the disk into Drive8
                rom_path = browser->GetCurrentDirectoryInfo().dir + "/" + browser->GetCurrentEntry()->d_name;
                if (file_system_attach_disk(8, rom_path.c_str()) < 0) {
                    log_message (LOG_DEFAULT, "could not attach image to device 8 : %s\n", rom_path.c_str());
                }
                log_message (LOG_DEFAULT, "Attached disk image to device %d\n", 8);
            }
        }
        if (CellInput->IsButtonPressed(0,CTRL_TRIANGLE) && (CellInput->WasButtonPressed(0,CTRL_L2)))
        {
            if (browser->IsCurrentAFile())
            {
                // Insert the disk into Drive9
                rom_path = browser->GetCurrentDirectoryInfo().dir + "/" + browser->GetCurrentEntry()->d_name;
                if (file_system_attach_disk(9, rom_path.c_str()) < 0) {
                    log_message (LOG_DEFAULT, "could not attach image to device 9 : %s\n", rom_path.c_str());
                 }
                log_message (LOG_DEFAULT, "Attached disk image to device %d\n", 9);
            }
        }
        if (CellInput->IsButtonPressed(0,CTRL_TRIANGLE) && (CellInput->WasButtonPressed(0,CTRL_R1)))
        {
            if (browser->IsCurrentAFile())
            {
                // Insert the disk into Drive10
                rom_path = browser->GetCurrentDirectoryInfo().dir + "/" + browser->GetCurrentEntry()->d_name;
                if (file_system_attach_disk(10, rom_path.c_str()) < 0) {
                    log_message (LOG_DEFAULT, "could not attach image to device 10 : %s\n", rom_path.c_str());
                }
                log_message (LOG_DEFAULT, "Attached disk image to device %d\n", 10);
            }
        }
        if (CellInput->IsButtonPressed(0,CTRL_TRIANGLE) && (CellInput->WasButtonPressed(0,CTRL_R2)))
        {
            if (browser->IsCurrentAFile())
            {
                // Insert the disk into Drive11
                rom_path = browser->GetCurrentDirectoryInfo().dir + "/" + browser->GetCurrentEntry()->d_name;
                if (file_system_attach_disk(11, rom_path.c_str()) < 0) {
                    log_message (LOG_DEFAULT, "could not attach image to device 11 : %s\n", rom_path.c_str());
                }
                log_message (LOG_DEFAULT, "Attached disk image to device %d\n", 11);
            }
        }
        if (CellInput->WasButtonPressed(0,CTRL_START))
        {
            // Don't load any new rom. Go back to the emulator as is
            MenuStop();
            Emulator_StartROMRunning();
            return;
        }
    }

                               
    const char *filenameptr=NULL;
    const char *filepathptr=NULL;

    float yPos=0.000f;
    float ybrk=0.035f;

    resources_get_int("Drive8Type", &res_int);
    if (res_int != DRIVE_TYPE_NONE) {
        yPos+=ybrk;
        cellDbgFontPuts(0.65f, yPos, Emulator_GetFontSize(), PURPLE, "Drive 8");
        filepathptr = file_system_get_disk_name(8);
        if (filepathptr != NULL) {
            filenameptr = strrchr(filepathptr, '/');
            if (filenameptr != NULL)
                cellDbgFontPrintf(0.755f, yPos, Emulator_GetFontSize(), CYAN, "%s", ++filenameptr);
             else 
                cellDbgFontPrintf(0.755f, yPos, Emulator_GetFontSize(), CYAN, "%s", filepathptr);
        }
    }
    resources_get_int("Drive9Type", &res_int);
    if (res_int != DRIVE_TYPE_NONE) {
        yPos+=ybrk;
        cellDbgFontPuts(0.65f, yPos, Emulator_GetFontSize(), PURPLE,  "Drive 9");
        filepathptr = file_system_get_disk_name(9);
        if (filepathptr != NULL) {
            filenameptr = strrchr(filepathptr, '/');
            if (filenameptr != NULL)
                cellDbgFontPrintf(0.755f, yPos, Emulator_GetFontSize(),  CYAN, "%s", ++filenameptr);
            else
                cellDbgFontPrintf(0.755f, yPos, Emulator_GetFontSize(),  CYAN, "%s", filepathptr);
        }
    }
    resources_get_int("Drive10Type", &res_int);
    if (res_int != DRIVE_TYPE_NONE) {
        yPos+=ybrk;
        cellDbgFontPuts(0.65f, yPos, Emulator_GetFontSize(), PURPLE, "Drive 10");
        filepathptr = file_system_get_disk_name(10);
        if (filepathptr != NULL) {
            filenameptr = strrchr(filepathptr, '/');
            if (filenameptr != NULL)
                cellDbgFontPrintf(0.755f, yPos, Emulator_GetFontSize(),  CYAN, "%s", ++filenameptr);
            else
                cellDbgFontPrintf(0.755f, yPos, Emulator_GetFontSize(),  CYAN, "%s", filepathptr);
        }
    }
    resources_get_int("Drive11Type", &res_int);
    if (res_int != DRIVE_TYPE_NONE) {
        yPos+=ybrk;
        cellDbgFontPuts(0.65f, yPos, Emulator_GetFontSize(), PURPLE,  "Drive 11");
        filepathptr = file_system_get_disk_name(11);
        if (filepathptr != NULL) {
            filenameptr = strrchr(filepathptr, '/');
            if (filenameptr != NULL)
                cellDbgFontPrintf(0.755f, yPos, Emulator_GetFontSize(),  CYAN, "%s", ++filenameptr);
            else
                cellDbgFontPrintf(0.755f, yPos, Emulator_GetFontSize(),  CYAN, "%s", filepathptr);
        }
    }
    yPos+=ybrk;
    cellDbgFontPuts(0.65f, yPos, Emulator_GetFontSize(), PURPLE,  "Tape");
    filepathptr = tape_get_file_name();
    if (filepathptr != NULL) {
        filenameptr = strrchr(filepathptr, '/');
        if (filenameptr != NULL)
            cellDbgFontPrintf(0.755f, yPos, Emulator_GetFontSize(),  CYAN, "%s", ++filenameptr);
        else
            cellDbgFontPrintf(0.755f, yPos, Emulator_GetFontSize(),  CYAN, "%s", filepathptr);
    }
    Graphics->FlushDbgFont();

    cellDbgFontPuts(0.09f, 0.88f, Emulator_GetFontSize(), BLUE,                "SELECT - settings screen");
    cellDbgFontPuts(0.09f, 0.92f, Emulator_GetFontSize(), PURPLE, util_concat ("START  - Return to ", machine_name, NULL));
    Graphics->FlushDbgFont();

    cellDbgFontPuts(0.44f, 0.84f, Emulator_GetFontSize(), GREEN,  " X - Reset with image (FastLoad)");
    cellDbgFontPuts(0.44f, 0.88f, Emulator_GetFontSize(), YELLOW, "[] - Reset with image (SlowLoad + TDE)");
    cellDbgFontPuts(0.44f, 0.92f, Emulator_GetFontSize(), CYAN ,  "/\\ + L1/L2/R1/R2 - Attach to D8/9/10/11");
    Graphics->FlushDbgFont();

    RenderBrowser(browser);
}

void MenuMainLoop()
{
    const char *res_string;

    // create file browser->if null
    if (browser == NULL)
    {
        resources_get_string ("PS3PathRomDir", &res_string);
        browser = new FileBrowser(res_string);
    }


    // FIXME: could always just return to last menu item... don't pop on resume kinda thing
    if (menuStack.empty())
    {
        menuStack.push(do_ROMMenu);
    }

    // menu loop
    menuRunning = true;
    while (!menuStack.empty() && menuRunning)
    {
        Graphics->Clear();

        menuStack.top()();

        Graphics->Swap();

        cellSysutilCheckCallback();

        if (Emulator_GetMode() == MODE_EXIT) {
            // Emulator_Shutdown will be called by our atexit handler
            exit(0);
        }
    }
}
