/******************************************************************************* 
 *  -- joy.cpp      Pad driver for Playstation 3
 *
 *     VICE PS3 -   Commodore 64 emulator for the Playstation 3
 *                  ported from the original VICE distribution
 *                  located at http://sourceforge.net/projects/vice-emu/
 *
 *  Copyright (C) 2010
 *
 *      Original Author :   Bernhard Kuhn <kuhn@eikon.e-technik.tu-muenchen.de>
 *                          Ulmer Lionel <ulmer@poly.polytechnique.fr>
 *
 *      Updated by          TimRex
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 ********************************************************************************/


#include "vice.h"

#include "cellframework/input/cellInput.h"

#include "joy.h"
#include "kbd.h"
#include "keyboard.h"
#include "ps3debug.h"
#include "emulator.h"

extern "C" {
#include "machine.h"
#include "resources.h"
#include "videoarch.h"
#include "vsync.h"
#include "ui.h"
#include "main.h"
}

/* ------------------------------------------------------------------------- */

/* Joystick devices.  */


/* Initialize joystick support.  */
int joy_arch_init(void)
{
    // TODO : Currently init'ed from menu.cpp
    // Temporary untill we move menu.cpp to be called as part of ui.c
    //CellInput = new CellInputFacade;
    //CellInput->Init();

    return 0;
}

int joystick_init_cmdline_options(void)
{
    return 0;
}

void joystick_close(void)
{
    return;
}

int joystick_arch_init_resources(void)
{
    return 0;
}


int joystick(void)
{
    // pad buttons that map to the keyboard
    static bool tri_state=false;
    static bool squ_state=false;
    static bool cir_state=false;
    static bool l1_state=false;
    static bool l2_state=false;
    static bool l3_state=false;
    static bool r1_state=false;
    static bool r2_state=false;
    static bool r3_state=false;

    static bool key_cursorup    = false;
    static bool key_cursordown  = false;
    static bool key_cursorleft  = false;
    static bool key_cursorright = false;


    static bool joyswap=false;

    uint8_t pads_connected = CellInput->NumberPadsConnected();
    for (uint8_t i = 0; i < pads_connected; ++i)
    {
        int value = 0;

        CellInput->UpdateDevice(i);


        if (CellInput->IsButtonPressed(i,CTRL_LEFT) | CellInput->IsAnalogPressedLeft(i,CTRL_LSTICK)) {
            value |= 4;
        }
        if (CellInput->IsButtonPressed(i,CTRL_RIGHT) | CellInput->IsAnalogPressedRight(i,CTRL_LSTICK)) {
            value |= 8;
        }
        if (CellInput->IsButtonPressed(i,CTRL_UP) | CellInput->IsAnalogPressedUp(i,CTRL_LSTICK)) {
            value |= 1;
        }
        if (CellInput->IsButtonPressed(i,CTRL_DOWN) | CellInput->IsAnalogPressedDown(i,CTRL_LSTICK)) {
            value |= 2;
        }
        if (CellInput->IsButtonPressed(i,CTRL_CROSS)) {
            value |= 16;
        }


        if (CellInput->WasButtonPressed(i,CTRL_SQUARE))
            if (!squ_state) {
                keyboard_key_pressed((signed long)  ' ');  // <space>
		squ_state = true;
            }
        if (CellInput->WasButtonReleased(i,CTRL_SQUARE))
            if (squ_state) {
                keyboard_key_released((signed long) ' ');  // <space>
		squ_state = false;
            }

        if (CellInput->WasButtonPressed(i,CTRL_TRIANGLE))
            if (!tri_state) {
                keyboard_key_pressed((signed long)  'y');  // Y
		tri_state = true;
            }
        if (CellInput->WasButtonReleased(i,CTRL_TRIANGLE))
            if (tri_state) {
                keyboard_key_released((signed long) 'y');  // Y
		tri_state = false;
            }



        if (CellInput->WasButtonPressed(i,CTRL_CIRCLE))
        {
            if (!cir_state) {
                keyboard_key_pressed((signed long)  'n');  // N
		cir_state = true;
            }
            // Dirty hack, to allow second controller button as 'UP'
            // Helpful for games that use 'UP' as the JUMP action
            value |= 1;
        }
        if (CellInput->WasButtonReleased(i,CTRL_CIRCLE))
            if (cir_state) {
                keyboard_key_released((signed long) 'n');  // N
		cir_state = false;
            }


        if (CellInput->WasButtonPressed(i,CTRL_L1))
            if (!l1_state) {
                keyboard_key_pressed((signed long)  32826);  // F1
		l1_state = true;
            }
        if (CellInput->WasButtonReleased(i,CTRL_L1))
            if (l1_state) {
                keyboard_key_released((signed long) 32826);  // F1
		l1_state = false;
            }


        if (CellInput->WasButtonPressed(i,CTRL_L2))
            if (!l2_state) {
                keyboard_key_pressed((signed long)  32828);  // F3
		l2_state = true;
            }
        if (CellInput->WasButtonReleased(i,CTRL_L2))
            if (l2_state) {
                keyboard_key_released((signed long) 32828);  // F3
		l2_state = false;
            }


        if (CellInput->WasButtonPressed(i,CTRL_R1))
            if (!r1_state) {
                keyboard_key_pressed((signed long)  32830);  // F5
		l2_state = true;
            }
        if (CellInput->WasButtonReleased(i,CTRL_R1))
            if (r1_state) {
                keyboard_key_released((signed long) 32830);  // F5
		l2_state = false;
            }


        if (CellInput->WasButtonPressed(i,CTRL_R2))
            if (!r2_state) {
                keyboard_key_pressed((signed long)  32832);  // F7
		r2_state = true;
            }
        if (CellInput->WasButtonReleased(i,CTRL_R2))
            if (r2_state) {
                keyboard_key_released((signed long) 32832);  // F7
		r2_state = false;
            }

        if (CellInput->WasButtonPressed(i,CTRL_L3))
            if (!l3_state) {
                keyboard_key_pressed((signed long)  32809);  // RUN/STOP
                debug_printf_quick ("Triggered RUN/STOP key\n");
		l3_state = true;
            }
                
        if (CellInput->WasButtonReleased(i,CTRL_L3))
            if (l3_state) {
                keyboard_key_released((signed long)  32809);  // RUN/STOP
                debug_printf_quick ("Released RUN/STOP key\n");
		l3_state = false;
            }


        // Emulator cursor keys
        if (CellInput->IsAnalogPressedUp(i,CTRL_RSTICK)) {
            if (!key_cursorup) {
                keyboard_key_pressed((signed long)  32850);  // Cursor Up key pressed
                key_cursorup = true;
            }
        } else {
            if (key_cursorup) {
                keyboard_key_released((signed long) 32850);  // Cursor Up key released
                key_cursorup = false;
            }
        }

        if (CellInput->IsAnalogPressedDown(i,CTRL_RSTICK)) {
            if (!key_cursordown) {
                keyboard_key_pressed((signed long)  32849);  // Cursor Down key pressed
                key_cursordown = true;
            }
        } else {
            if (key_cursordown) {
                keyboard_key_released((signed long) 32849);  // Cursor Down key released
                key_cursordown = false;
            }
        }

        if (CellInput->IsAnalogPressedLeft(i,CTRL_RSTICK)) {
            if (!key_cursorleft) {
                keyboard_key_pressed((signed long)  32848);  // Cursor Left key pressed
                key_cursorleft = true;
            }
        } else {
            if (key_cursorleft) {
                keyboard_key_released((signed long) 32848);  // Cursor Left key released
                key_cursorleft = false;
            }
        }

        if (CellInput->IsAnalogPressedRight(i,CTRL_RSTICK)) {
            if (!key_cursorright) {
                keyboard_key_pressed((signed long)  32847);  // Cursor Right key pressed
                key_cursorright = true;
            }
        } else {
            if (key_cursorright) {
                keyboard_key_released((signed long) 32847);  // Cursor Right key released
                key_cursorright = false;
            }
        }


        // TODO Re-enable screenshots, with compression
/*
        if(CellInput->WasButtonPressed(i,CTRL_R1) & CellInput->IsButtonPressed(i,CTRL_L1))
        {
            Graphics->ScreenDump();
                save_screenshot(current_rom);
        }
*/

        static bool warp_mode=false;
        if(CellInput->IsButtonPressed(i,CTRL_R2) && CellInput->IsButtonPressed(i,CTRL_L2))
        {
            if (!warp_mode)
            {
                // Disable sound (makes warp mode infinitely faster)
                resources_set_int("Sound", 0);

                // Enable Warp Mode
                resources_set_int("WarpMode", 1);
                warp_mode=true;
            }
        }
        else
        {
            if (warp_mode)
            {
                resources_set_int("WarpMode", 0);
                resources_set_int("Sound", 1);
                warp_mode=false;
            }
        }


        if (CellInput->IsButtonPressed(i,CTRL_L3) && CellInput->IsButtonPressed(i,CTRL_R3) )
            machine_trigger_reset(MACHINE_RESET_MODE_HARD);


        if(CellInput->WasButtonPressed(i,CTRL_R3) & !CellInput->IsButtonPressed(i,CTRL_L3))
        {
            debug_printf_quick ("OSK starting\n");
            if (!osk->Start(L"Characters entered here will be relayed to the emulator ", L""))
                debug_printf ("WARNING: OSK could not start\n");

            debug_printf_quick ("OSK started\n");
            // Just in case. This ensures we check to see if the screen has updated, and if not.. force one
            // The OSK fails to draw if the screen doesn't update.
            debug_printf_quick ("OSK callback check\n");
        }
            

        // Swap joysticks
        if (CellInput->WasButtonPressed(i,CTRL_SELECT)) {
            // Do nothing
        }
        // Required because keyloop kills us
        if (CellInput->WasButtonReleased(i,CTRL_SELECT)) {
            // TODO : Test this has the desired effect
            if (pads_connected == 1)
                // Only allow us to swap joystick ports if there is only one controller
                // Otherwise, multiplayer madness and much punching will ensue.
                joyswap=!joyswap;
        }


        if(CellInput->WasButtonPressed(i,CTRL_START))
        {
            // TODO : Launch this as a panel over the top of the current display
            // Possibly allow the emulator to continue running underneath?
            // Would allow us to see changes as we make them.

            menu();
            // Stop the clock (effectively, tells the emulator we were paused)
            vsync_suspend_speed_eval();
            // Apply any changes as necessary.
        }


        // Check this here, since the 'ui_callback_alarm' gets suspended
        // after a warp_mode event

        // TODO : If this works, do we even need a ui_callback_alarm' anymore?
        // TODO : This might even fix any buffer underrun with the sound, since this will be called more regularly.
        cellSysutilCheckCallback();
        if (is_sysutil_drawing())
            sysutil_callback_redraw();

        // TODO 
        // Check that this works when the screen no longer refreshes... otherwise we need to 
        // renable the alarm handler
        ui_callback();


        // Set the joystick values

        if (joyswap)
            joystick_set_value_absolute( ((i+1) % 2 ) + 1, value);  
            // pad 0 becomes port 2
            // pad 1 becomes port 1
        else
            joystick_set_value_absolute(i+1, value); 
            // pad 0 becomes port 1
            // pad 1 becomes port 2
    }

    return 0;
}


