/******************************************************************************* 
 *  -- emulator.cpp -Integration with menu.cpp to provide a frontend for Playstation 3
 *
 *     VICE PS3 -   Commodore 64 emulator for the Playstation 3
 *                  ported from the original VICE distribution
 *                  located at http://sourceforge.net/projects/vice-emu/
 *
 *
 *  Copyright (C) 2010
 *  Created on: Oct 15, 2010
 *      Author:     Squarepusher2 
 *      Updated by  TimRex
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 ********************************************************************************/


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/timer.h>
#include <sys/return_code.h>
#include <sys/process.h>
#include <cell/sysmodule.h>
#include <cell/cell_fs.h>
#include <cell/pad.h>
#include <cell/dbgfont.h>
#include <cell/audio.h>
#include <stddef.h>
#include <math.h>

#include <sysutil/sysutil_sysparam.h>
#include <sys/spu_initialize.h>
#include <cell/hash/libmd5.h>

#include "kbd.h"

#include "common.h"
//#include "settings.h"
#include "emulator.h"

#include "PS3Graphics.h"
#include "audio/rsound.hpp"
 
#include "joy.h"
#include "mousedrv.h"
#include "machine.h"
#include "ps3debug.h"

//extern "C" int main_program(int argc, char **argv);
extern "C" {
#include "archdep.h"
#include "main.h"
#include "sid.h"
#include "util.h"
#include "lib.h"
#include "log.h"
#include "autostart.h"
#include "attach.h"
#include "tape.h"
#include "siddefs-fp.h"
#include "resources.h"
#include "mouse.h"
}

#include "menu.h"


extern int ps3_audio_suspend(void);
extern int ps3_audio_resume(void);

CellInputFacade* CellInput;
PS3Graphics* Graphics;
SYS_PROCESS_PARAM(1001, 0x10000);


// is emulator loaded?
bool emulator_loaded = false;

// current rom being emulated
char* current_rom = NULL;

// mode the main loop is in
static Emulator_Modes mode_switch = MODE_MENU;


void Emulator_SwitchMode(Emulator_Modes m)
{
    mode_switch = m;
}

Emulator_Modes Emulator_GetMode(void)
{
    return mode_switch;
}


void Emulator_Shutdown()
{
    debug_printf ("Emulator_Shutdown()\n");

    cellSysutilUnregisterCallback(0);

    mousedrv_destroy();

    if (osk) {
        osk->Close();
        delete osk;
    }

    if (Graphics)
        delete Graphics;

//   delete oskutil;
    debug_close();
    cellSysmoduleUnloadModule(CELL_SYSMODULE_AUDIO);
    cellSysmoduleUnloadModule(CELL_SYSMODULE_NET);
    cellSysmoduleUnloadModule(CELL_SYSMODULE_FS);
    cellSysmoduleUnloadModule(CELL_SYSMODULE_IO);
}




void Emulator_StartROMRunning()
{
    Emulator_SwitchMode(MODE_EMULATION);
}

void Emulator_RequestLoadROM(char* rom, bool forceReboot, bool compatibility_mode) 
{
    if (current_rom == NULL || strcmp(rom, current_rom) != 0)
    {
        if (current_rom != NULL)
        {
            free(current_rom);
        }

        current_rom = strdup(rom);
    }


    if (forceReboot) {

        // Tell the emulator to install 'current_rom' for autoload

        // Turn off warp for compatibility mode
        set_autostart_warp(!compatibility_mode, NULL);

        // Set True Drive Emulation (TDE) for compatibility mode
        resources_set_int("DriveTrueEmulation", (int) compatibility_mode);

        // name, program_name, program_number, run/load
        if (autostart_autodetect(current_rom, NULL, 0, AUTOSTART_MODE_RUN) < 0) {
            log_warning (LOG_DEFAULT, "autostart_autodetect failed for image : '%s'", current_rom);
        }
    }
    else {
        // Insert the disk image, don't force a reboot.
        // TODO : Allow device selection (drive 8, 9, 10, 11 or tape)
        // This hack assumes we're attaching a disk image to drive 8

        if (file_system_attach_disk(8, current_rom) < 0 && tape_image_attach(1, current_rom) < 0 ) {
            log_warning (LOG_DEFAULT, "could not attach image : %s to any disk/tape device", current_rom);
        }

        log_message (LOG_DEFAULT, "Attached disk image (%s) to device %d\n", current_rom, 8);
    }
}


/*
 * Callback for PS3 System operations
 */

extern "C" void sysutil_drawing(int);
extern "C" int  is_sysutil_drawing(void);
extern "C" void force_redraw (void);

void sysutil_exit_callback (uint64_t status, uint64_t param, void *userdata)
{
    (void) param;
    (void) userdata;

    switch (status)
    {
        case CELL_SYSUTIL_REQUEST_EXITGAME:
            Emulator_SwitchMode(MODE_EXIT);
            break;
        case CELL_SYSUTIL_DRAWING_BEGIN:
            sysutil_drawing (1);
            debug_printf_quick ("SYSUTIL_DRAWING_BEGIN\n");
            break;
        case CELL_SYSUTIL_DRAWING_END:
            sysutil_drawing (0);
            debug_printf_quick ("SYSUTIL_DRAWING_END\n");
            break;
        case CELL_SYSUTIL_OSKDIALOG_FINISHED:
            osk->Stop();
            //debug_printf ("OSK Returned string : %s\n", osk->OutputString());
            osk_kbd_append_buffer ((char *)osk->OutputString());
            //osk->Close();
            break;

        case CELL_SYSUTIL_OSKDIALOG_INPUT_ENTERED:
            //debug_printf ("OSKDIALOG_INPUT_ENTERED\n");
            break;

        case CELL_SYSUTIL_OSKDIALOG_INPUT_CANCELED:
            //debug_printf ("OSKDIALOG_INPUT_CANCELLED\n");
            break;

        case CELL_OSKDIALOG_INPUT_DEVICE_KEYBOARD:
            debug_printf_quick ("OSK_INPUT_DEVICE_KEYBOARD\n");
            break;

        case CELL_OSKDIALOG_INPUT_DEVICE_PAD:
            debug_printf_quick ("OSK_INPUT_DEVICE_PAD\n");
            break;

        case CELL_SYSUTIL_OSKDIALOG_DISPLAY_CHANGED:
            debug_printf_quick ("OSK_DISPLAY_CHANGED\n");
            break;
    }

}

void sysutil_callback_redraw(void)
{
    if (Graphics->TimeSinceLastDraw() >= 20000)
    {
        // Refresh the display. 
        Graphics->Refresh();
    }
}


int cellInit(void)
{
   debug_init();
   sys_spu_initialize(6, 1);

    cellSysmoduleLoadModule(CELL_SYSMODULE_FS);
    cellSysmoduleLoadModule(CELL_SYSMODULE_IO);

    CellInput = new CellInputFacade();
    CellInput->Init();

        osk = new OSKUtil();
        if (!osk->Init()) {
        debug_printf ("WARNING: OSK could not be initialised\n");
        // TODO: handle this
    }

    Graphics = new PS3Graphics();
    Graphics->Init();  // width, height, depth

    //Graphics->SetOverscan(Settings.PS3OverscanEnabled,(float)Settings.PS3OverscanAmount/100);
        // TODO : We used to use saved overscan values.. but we haven't loaded them for VICE yet.
    Graphics->SetOverscan(false,(float)0);

    Graphics->InitDbgFont();

    return 0;
}


int main (void) {

    cellSysutilRegisterCallback(0, sysutil_exit_callback, NULL);
    cellInit();


    // Start running Vice.
    // When it init's the UI, if willcall back here for the 'menu' function below
    char  arg0[] = "vice";
    char* argv[] = { &arg0[0], NULL };
    int   argc   = 1;

    emulator_loaded = true;

    main_program (argc, &argv[0]);

    debug_printf ("Vice Terminated...  (How?!)\n");
    emulator_loaded = false;

    ps3_audio_suspend();
    machine_shutdown();
    Emulator_Shutdown();
    exit(0);
}


// TODO  No need for a 16 MB buffer. 
#define FILEBUF_SIZE            16*1024*1024
static unsigned char g_buf[FILEBUF_SIZE];
static char digest_string[CELL_MD5_DIGEST_SIZE*2+1];

char * md5_sum_file(const char *filepath)
{
    int fd;
    uint64_t  nread;
    int res;
    CellMd5WorkArea ctx;
    unsigned char digest[CELL_MD5_DIGEST_SIZE];

    res=cellFsOpen(filepath, CELL_FS_O_RDONLY, &fd, NULL, 0);

    if (fd < 0) {
        debug_printf("cellFsOpen('%s') error: 0x%08X\n", filepath, res);
        return NULL;
    }
    
    cellMd5BlockInit(&ctx);
    do {
        res = cellFsRead(fd, g_buf, sizeof(g_buf), &nread);
        if (res < 0) {
            debug_printf("cellFsRead('%s') error: 0x%08X\n", filepath, res);
            cellFsClose(fd);    
            return NULL;
        }
        debug_printf("read in %lld bytes from file\n", nread);
        debug_printf("Calculating MD5 chunk...\n");
        cellMd5BlockUpdate(&ctx, g_buf, (unsigned int)nread);
    } while (res == sizeof(g_buf));
    cellMd5BlockResult(&ctx, digest);
    cellFsClose(fd);

    {
        int i;

        debug_printf("\nMD5 Hash calculated from %s:\n", filepath);
        char *digest_ptr = digest_string;
        for(i = 0; i < CELL_MD5_DIGEST_SIZE; i++) {
            snprintf (digest_ptr, 2+1, "%02x", digest[i]);
            digest_ptr += 2;
        }
    }

    return digest_string;
}


int save_screenshot(char *filename)
{
    if (filename == NULL)
            return 1;

    unsigned char *pPixels = Graphics->RetrieveDump();

    static unsigned int len;
    static unsigned int isdir;

    if (archdep_stat(VICE_SCREENSHOT_DIR, &len, &isdir) != 0) {
        debug_printf ("archdep_mkstemp_fd creating VICE_TMPDIR : %s\n", VICE_SCREENSHOT_DIR);
        archdep_mkdir(VICE_SCREENSHOT_DIR, 0755);
    }

    // Write the screendump to a hash lookup file of the currently running image file
    // Get MD5 of 'current_rom'
        
    char *digest;
    digest = md5_sum_file(filename);

    if (digest) {
        // save pPixels to VICE_SCREENSHOT_DIR
        debug_printf ("Writing screenshot to %s\n", util_concat (VICE_SCREENSHOT_DIR, digest, NULL) );

/*
        if (screenshot_save("PNG", util_concat (VICE_SCREENSHOT_DIR, digest, NULL), last_canvas) < 0) {
            debug_printf ("Screenshot failed\n");
        }
*/

        FILE *fd = fopen (util_concat (VICE_SCREENSHOT_DIR, digest, NULL), "w");

        if (fd == NULL) {
            log_error (LOG_DEFAULT, "Failed to open file for writing %s\n", util_concat (VICE_SCREENSHOT_DIR, digest, NULL) );
            return 1;
        }
        int bytes_wrote = fwrite (pPixels, 1, Graphics->RetrieveDumpSize(), fd);

        if (bytes_wrote < Graphics->RetrieveDumpSize()) {
            log_error (LOG_DEFAULT, "Failed to write screenshot, wrote %d bytes of total %d.\n", bytes_wrote, Graphics->RetrieveDumpSize());
            return 1;
        }

        if (fclose(fd) != 0) {
            log_warning (LOG_DEFAULT, "fclose failed");
        }

        return 0;
    }
    return 1;
}
 
unsigned char * load_screenshot(const char *filename, long *bytes)
{
    char *digest;
    static unsigned char *pPixels;

    // Get MD5 of 'current_rom'
    digest = md5_sum_file(filename);

    if (digest) {
        
        debug_printf ("Loading screenshot from %s\n", util_concat (VICE_SCREENSHOT_DIR, digest, NULL) );
        // read pPixels from VICE_SCREENSHOT_DIR
        FILE *fd = fopen (util_concat (VICE_SCREENSHOT_DIR, digest, NULL), "r");

        if (fd == NULL) {
            debug_printf ("Failed to open file %s\n", util_concat (VICE_SCREENSHOT_DIR, digest, NULL));
            return NULL;
        }

        if (fseek(fd, 0L, SEEK_END) == -1) {
            debug_printf ("fseek failed at file %s\n", util_concat (VICE_SCREENSHOT_DIR, digest, NULL));
            return NULL;
        }

        long filesize = ftell (fd);
        if (filesize == -1) {
            debug_printf ("ftell failed at file %s\n", util_concat (VICE_SCREENSHOT_DIR, digest, NULL));
            return NULL;
        }

        if (fseek(fd, 0L, SEEK_SET) == -1) {
            debug_printf ("fseek failed at file %s\n", util_concat (VICE_SCREENSHOT_DIR, digest, NULL));
            return NULL;
        }

        // TODO optimize this
        if (pPixels) {
            free (pPixels);
            pPixels=NULL;
        }

        pPixels = (unsigned char *) lib_malloc (filesize);

        *bytes = fread (pPixels, 1, filesize, fd);
        if (*bytes < filesize) {
            debug_printf ("fread returned %d bytes of %d filesize\n", *bytes, filesize);
            return NULL;
        } 

        if (fclose(fd) != 0) {
            debug_printf ("WARNING: fclose failed\n");
        }
    }
    return pPixels;
}

extern "C" int menu(void)
{
    // TODO : Maybe we should pause the emulator?

    ps3_audio_suspend();
    Graphics->ScreenDump();

    Emulator_SwitchMode(MODE_MENU);

    while(1)
    {
        switch(mode_switch)
        {
            case MODE_MENU:
                MenuMainLoop();
                break;
            case MODE_EMULATION:
                // Break out and return to Vice.
                Graphics->DestroyDump();
                ps3_audio_resume();

                // The C64 only redraws if it needs to, so we force one here to clena up after the menu
                force_redraw();
                return 0;
                break;
            case MODE_EXIT:
                Graphics->DestroyDump();
                // letting this be handled by atexit
                ps3_audio_suspend();
                machine_shutdown();
                Emulator_Shutdown();
                exit(0);
        }
    }

    // We should never get here
    return 0;
}

float Emulator_GetFontSize()
{
    int res_int;
    resources_get_int ("PS3FontSize", &res_int);
    return res_int/100.0;
}
