implement genjisx0212;

include "sys.m";
include "draw.m";

genjisx0212 : module {
	init: fn(ctxt: ref Draw->Context, args: list of string);
};

DATAFILE : con "/lib/convcs/jisx0212";

PAGESZ : con 94;
NPAGES : con 77;
PAGE0 : con 16rA1;
CHAR0 : con 16rA1;

BADCHAR : con 16rFFFD;

init(nil: ref Draw->Context, nil: list of string)
{
	sys := load Sys Sys->PATH;
	fd := sys->create(DATAFILE, Sys->OWRITE, 8r644);
	if (fd == nil) {
		sys->print("cannot create %s: %r\n", DATAFILE);
		return;
	}

	pages := array [NPAGES * PAGESZ] of { * => BADCHAR };

	for (i := 0; i < len data; i++) {
		(nil, toks) := sys->tokenize(data[i], "\t");
		(bytes, ucode) := (hd toks, hd tl toks);
		u := hex2int(ucode);
		(nil, blist) := sys->tokenize(bytes, " ");
		b1 := hex2int(hd blist);
		b2 := hex2int(hd tl blist);

		page := b1 - PAGE0;
		char := b2 - CHAR0;

		pages[(page * PAGESZ)+char] = u;
	}

	s := "";
	for (i = 0; i < len pages; i++)
		s[i] = pages[i];
	pages = nil;
	buf := array of byte s;
	sys->write(fd, buf, len buf);
}


hex2int(s: string): int
{
	n := 0;
	for (i := 0; i < len s; i++) {
		case s[i] {
		'0' to '9' =>
			n = 16*n + s[i] - '0';
		'A' to 'F' =>
			n = 16*n + s[i] + 10 - 'A';
		'a' to 'f' =>
			n = 16*n + s[i] + 10 - 'a';
		* =>
			return n;
		}
	}
	return n;
}


# Data is derived from Unicode Consortium "CharmapML" mapping data
# for EUC-JP.
# JISX0212 appears as charset G3 of the ISO-2022 based EUC-JP encoding
# Although this format is not convenient for building our mapping data file
# it is easily extracted from the Unicode Consortium data.

data := array [] of {
		"A2 C2	00A1",
		"A2 F0	00A4",
		"A2 C3	00A6",
		"A2 ED	00A9",
		"A2 EC	00AA",
		"A2 EE	00AE",
		"A2 B4	00AF",
		"A2 B1	00B8",
		"A2 EB	00BA",
		"A2 C4	00BF",
		"AA A2	00C0",
		"AA A1	00C1",
		"AA A4	00C2",
		"AA AA	00C3",
		"AA A3	00C4",
		"AA A9	00C5",
		"A9 A1	00C6",
		"AA AE	00C7",
		"AA B2	00C8",
		"AA B1	00C9",
		"AA B4	00CA",
		"AA B3	00CB",
		"AA C0	00CC",
		"AA BF	00CD",
		"AA C2	00CE",
		"AA C1	00CF",
		"AA D0	00D1",
		"AA D2	00D2",
		"AA D1	00D3",
		"AA D4	00D4",
		"AA D8	00D5",
		"AA D3	00D6",
		"A9 AC	00D8",
		"AA E3	00D9",
		"AA E2	00DA",
		"AA E5	00DB",
		"AA E4	00DC",
		"AA F2	00DD",
		"A9 B0	00DE",
		"A9 CE	00DF",
		"AB A2	00E0",
		"AB A1	00E1",
		"AB A4	00E2",
		"AB AA	00E3",
		"AB A3	00E4",
		"AB A9	00E5",
		"A9 C1	00E6",
		"AB AE	00E7",
		"AB B2	00E8",
		"AB B1	00E9",
		"AB B4	00EA",
		"AB B3	00EB",
		"AB C0	00EC",
		"AB BF	00ED",
		"AB C2	00EE",
		"AB C1	00EF",
		"A9 C3	00F0",
		"AB D0	00F1",
		"AB D2	00F2",
		"AB D1	00F3",
		"AB D4	00F4",
		"AB D8	00F5",
		"AB D3	00F6",
		"A9 CC	00F8",
		"AB E3	00F9",
		"AB E2	00FA",
		"AB E5	00FB",
		"AB E4	00FC",
		"AB F2	00FD",
		"A9 D0	00FE",
		"AB F3	00FF",
		"AA A7	0100",
		"AB A7	0101",
		"AA A5	0102",
		"AB A5	0103",
		"AA A8	0104",
		"AB A8	0105",
		"AA AB	0106",
		"AB AB	0107",
		"AA AC	0108",
		"AB AC	0109",
		"AA AF	010A",
		"AB AF	010B",
		"AA AD	010C",
		"AB AD	010D",
		"AA B0	010E",
		"AB B0	010F",
		"A9 A2	0110",
		"A9 C2	0111",
		"AA B7	0112",
		"AB B7	0113",
		"AA B6	0116",
		"AB B6	0117",
		"AA B8	0118",
		"AB B8	0119",
		"AA B5	011A",
		"AB B5	011B",
		"AA BA	011C",
		"AB BA	011D",
		"AA BB	011E",
		"AB BB	011F",
		"AA BD	0120",
		"AB BD	0121",
		"AA BC	0122",
		"AA BE	0124",
		"AB BE	0125",
		"A9 A4	0126",
		"A9 C4	0127",
		"AA C7	0128",
		"AB C7	0129",
		"AA C5	012A",
		"AB C5	012B",
		"AA C6	012E",
		"AB C6	012F",
		"AA C4	0130",
		"A9 C5	0131",
		"A9 A6	0132",
		"A9 C6	0133",
		"AA C8	0134",
		"AB C8	0135",
		"AA C9	0136",
		"AB C9	0137",
		"A9 C7	0138",
		"AA CA	0139",
		"AB CA	013A",
		"AA CC	013B",
		"AB CC	013C",
		"AA CB	013D",
		"AB CB	013E",
		"A9 A9	013F",
		"A9 C9	0140",
		"A9 A8	0141",
		"A9 C8	0142",
		"AA CD	0143",
		"AB CD	0144",
		"AA CF	0145",
		"AB CF	0146",
		"AA CE	0147",
		"AB CE	0148",
		"A9 CA	0149",
		"A9 AB	014A",
		"A9 CB	014B",
		"AA D7	014C",
		"AB D7	014D",
		"AA D6	0150",
		"AB D6	0151",
		"A9 AD	0152",
		"A9 CD	0153",
		"AA D9	0154",
		"AB D9	0155",
		"AA DB	0156",
		"AB DB	0157",
		"AA DA	0158",
		"AB DA	0159",
		"AA DC	015A",
		"AB DC	015B",
		"AA DD	015C",
		"AB DD	015D",
		"AA DF	015E",
		"AB DF	015F",
		"AA DE	0160",
		"AB DE	0161",
		"AA E1	0162",
		"AB E1	0163",
		"AA E0	0164",
		"AB E0	0165",
		"A9 AF	0166",
		"A9 CF	0167",
		"AA EC	0168",
		"AB EC	0169",
		"AA E9	016A",
		"AB E9	016B",
		"AA E6	016C",
		"AB E6	016D",
		"AA EB	016E",
		"AB EB	016F",
		"AA E8	0170",
		"AB E8	0171",
		"AA EA	0172",
		"AB EA	0173",
		"AA F1	0174",
		"AB F1	0175",
		"AA F4	0176",
		"AB F4	0177",
		"AA F3	0178",
		"AA F5	0179",
		"AB F5	017A",
		"AA F7	017B",
		"AB F7	017C",
		"AA F6	017D",
		"AB F6	017E",
		"AA A6	01CD",
		"AB A6	01CE",
		"AA C3	01CF",
		"AB C3	01D0",
		"AA D5	01D1",
		"AB D5	01D2",
		"AA E7	01D3",
		"AB E7	01D4",
		"AA F0	01D5",
		"AB F0	01D6",
		"AA ED	01D7",
		"AB ED	01D8",
		"AA EF	01D9",
		"AB EF	01DA",
		"AA EE	01DB",
		"AB EE	01DC",
		"AB B9	01F5",
		"A2 B0	02C7",
		"A2 AF	02D8",
		"A2 B2	02D9",
		"A2 B6	02DA",
		"A2 B5	02DB",
		"A2 B3	02DD",
		"A2 B8	0384",
		"A2 B9	0385",
		"A6 E1	0386",
		"A6 E2	0388",
		"A6 E3	0389",
		"A6 E4	038A",
		"A6 E7	038C",
		"A6 E9	038E",
		"A6 EC	038F",
		"A6 F6	0390",
		"A6 E5	03AA",
		"A6 EA	03AB",
		"A6 F1	03AC",
		"A6 F2	03AD",
		"A6 F3	03AE",
		"A6 F4	03AF",
		"A6 FB	03B0",
		"A6 F8	03C2",
		"A6 F5	03CA",
		"A6 FA	03CB",
		"A6 F7	03CC",
		"A6 F9	03CD",
		"A6 FC	03CE",
		"A7 C2	0402",
		"A7 C3	0403",
		"A7 C4	0404",
		"A7 C5	0405",
		"A7 C6	0406",
		"A7 C7	0407",
		"A7 C8	0408",
		"A7 C9	0409",
		"A7 CA	040A",
		"A7 CB	040B",
		"A7 CC	040C",
		"A7 CD	040E",
		"A7 CE	040F",
		"A7 F2	0452",
		"A7 F3	0453",
		"A7 F4	0454",
		"A7 F5	0455",
		"A7 F6	0456",
		"A7 F7	0457",
		"A7 F8	0458",
		"A7 F9	0459",
		"A7 FA	045A",
		"A7 FB	045B",
		"A7 FC	045C",
		"A7 FD	045E",
		"A7 FE	045F",
		"A2 F1	2116",
		"A2 EF	2122",
		"B0 A1	4E02",
		"B0 A2	4E04",
		"B0 A3	4E05",
		"B0 A4	4E0C",
		"B0 A5	4E12",
		"B0 A6	4E1F",
		"B0 A7	4E23",
		"B0 A8	4E24",
		"B0 A9	4E28",
		"B0 AA	4E2B",
		"B0 AB	4E2E",
		"B0 AC	4E2F",
		"B0 AD	4E30",
		"B0 AE	4E35",
		"B0 AF	4E40",
		"B0 B0	4E41",
		"B0 B1	4E44",
		"B0 B2	4E47",
		"B0 B3	4E51",
		"B0 B4	4E5A",
		"B0 B5	4E5C",
		"B0 B6	4E63",
		"B0 B7	4E68",
		"B0 B8	4E69",
		"B0 B9	4E74",
		"B0 BA	4E75",
		"B0 BB	4E79",
		"B0 BC	4E7F",
		"B0 BD	4E8D",
		"B0 BE	4E96",
		"B0 BF	4E97",
		"B0 C0	4E9D",
		"B0 C1	4EAF",
		"B0 C2	4EB9",
		"B0 C3	4EC3",
		"B0 C4	4ED0",
		"B0 C5	4EDA",
		"B0 C6	4EDB",
		"B0 C7	4EE0",
		"B0 C8	4EE1",
		"B0 C9	4EE2",
		"B0 CA	4EE8",
		"B0 CB	4EEF",
		"B0 CC	4EF1",
		"B0 CD	4EF3",
		"B0 CE	4EF5",
		"B0 CF	4EFD",
		"B0 D0	4EFE",
		"B0 D1	4EFF",
		"B0 D2	4F00",
		"B0 D3	4F02",
		"B0 D4	4F03",
		"B0 D5	4F08",
		"B0 D6	4F0B",
		"B0 D7	4F0C",
		"B0 D8	4F12",
		"B0 D9	4F15",
		"B0 DA	4F16",
		"B0 DB	4F17",
		"B0 DC	4F19",
		"B0 DD	4F2E",
		"B0 DE	4F31",
		"B0 E0	4F33",
		"B0 E1	4F35",
		"B0 E2	4F37",
		"B0 E3	4F39",
		"B0 E4	4F3B",
		"B0 E5	4F3E",
		"B0 E6	4F40",
		"B0 E7	4F42",
		"B0 E8	4F48",
		"B0 E9	4F49",
		"B0 EA	4F4B",
		"B0 EB	4F4C",
		"B0 EC	4F52",
		"B0 ED	4F54",
		"B0 EE	4F56",
		"B0 EF	4F58",
		"B0 F0	4F5F",
		"B0 DF	4F60",
		"B0 F1	4F63",
		"B0 F2	4F6A",
		"B0 F3	4F6C",
		"B0 F4	4F6E",
		"B0 F5	4F71",
		"B0 F6	4F77",
		"B0 F7	4F78",
		"B0 F8	4F79",
		"B0 F9	4F7A",
		"B0 FA	4F7D",
		"B0 FB	4F7E",
		"B0 FC	4F81",
		"B0 FD	4F82",
		"B0 FE	4F84",
		"B1 A1	4F85",
		"B1 A2	4F89",
		"B1 A3	4F8A",
		"B1 A4	4F8C",
		"B1 A5	4F8E",
		"B1 A6	4F90",
		"B1 A7	4F92",
		"B1 A8	4F93",
		"B1 A9	4F94",
		"B1 AA	4F97",
		"B1 AB	4F99",
		"B1 AC	4F9A",
		"B1 AD	4F9E",
		"B1 AE	4F9F",
		"B1 AF	4FB2",
		"B1 B0	4FB7",
		"B1 B1	4FB9",
		"B1 B2	4FBB",
		"B1 B3	4FBC",
		"B1 B4	4FBD",
		"B1 B5	4FBE",
		"B1 B6	4FC0",
		"B1 B7	4FC1",
		"B1 B8	4FC5",
		"B1 B9	4FC6",
		"B1 BA	4FC8",
		"B1 BB	4FC9",
		"B1 BC	4FCB",
		"B1 BD	4FCC",
		"B1 BE	4FCD",
		"B1 BF	4FCF",
		"B1 C0	4FD2",
		"B1 C1	4FDC",
		"B1 C2	4FE0",
		"B1 C3	4FE2",
		"B1 C4	4FF0",
		"B1 C5	4FF2",
		"B1 C6	4FFC",
		"B1 C7	4FFD",
		"B1 C8	4FFF",
		"B1 C9	5000",
		"B1 CA	5001",
		"B1 CB	5004",
		"B1 CC	5007",
		"B1 CD	500A",
		"B1 CE	500C",
		"B1 CF	500E",
		"B1 D0	5010",
		"B1 D1	5013",
		"B1 D2	5017",
		"B1 D3	5018",
		"B1 D4	501B",
		"B1 D5	501C",
		"B1 D6	501D",
		"B1 D7	501E",
		"B1 D8	5022",
		"B1 D9	5027",
		"B1 DA	502E",
		"B1 DB	5030",
		"B1 DC	5032",
		"B1 DD	5033",
		"B1 DE	5035",
		"B1 F6	503B",
		"B1 DF	5040",
		"B1 E0	5041",
		"B1 E1	5042",
		"B1 E2	5045",
		"B1 E3	5046",
		"B1 E4	504A",
		"B1 E5	504C",
		"B1 E6	504E",
		"B1 E7	5051",
		"B1 E8	5052",
		"B1 E9	5053",
		"B1 EA	5057",
		"B1 EB	5059",
		"B1 EC	505F",
		"B1 ED	5060",
		"B1 EE	5062",
		"B1 EF	5063",
		"B1 F0	5066",
		"B1 F1	5067",
		"B1 F2	506A",
		"B1 F3	506D",
		"B1 F4	5070",
		"B1 F5	5071",
		"B1 F7	5081",
		"B1 F8	5083",
		"B1 F9	5084",
		"B1 FA	5086",
		"B1 FB	508A",
		"B1 FC	508E",
		"B1 FD	508F",
		"B1 FE	5090",
		"B2 A1	5092",
		"B2 A2	5093",
		"B2 A3	5094",
		"B2 A4	5096",
		"B2 A5	509B",
		"B2 A6	509C",
		"B2 A7	509E",
		"B2 A8	509F",
		"B2 A9	50A0",
		"B2 AA	50A1",
		"B2 AB	50A2",
		"B2 AC	50AA",
		"B2 AD	50AF",
		"B2 AE	50B0",
		"B2 AF	50B9",
		"B2 B0	50BA",
		"B2 B1	50BD",
		"B2 B2	50C0",
		"B2 B3	50C3",
		"B2 B4	50C4",
		"B2 B5	50C7",
		"B2 B6	50CC",
		"B2 B7	50CE",
		"B2 B8	50D0",
		"B2 B9	50D3",
		"B2 BA	50D4",
		"B2 BB	50D8",
		"B2 BC	50DC",
		"B2 BD	50DD",
		"B2 BE	50DF",
		"B2 BF	50E2",
		"B2 C0	50E4",
		"B2 C1	50E6",
		"B2 C2	50E8",
		"B2 C3	50E9",
		"B2 C4	50EF",
		"B2 C5	50F1",
		"B2 D1	50F2",
		"B2 C6	50F6",
		"B2 C7	50FA",
		"B2 C8	50FE",
		"B2 C9	5103",
		"B2 CA	5106",
		"B2 CB	5107",
		"B2 CC	5108",
		"B2 CD	510B",
		"B2 CE	510C",
		"B2 CF	510D",
		"B2 D0	510E",
		"B2 D2	5110",
		"B2 D3	5117",
		"B2 D4	5119",
		"B2 D5	511B",
		"B2 D6	511C",
		"B2 D7	511D",
		"B2 D8	511E",
		"B2 D9	5123",
		"B2 DA	5127",
		"B2 DB	5128",
		"B2 DC	512C",
		"B2 DD	512D",
		"B2 DE	512F",
		"B2 DF	5131",
		"B2 E0	5133",
		"B2 E1	5134",
		"B2 E2	5135",
		"B2 E3	5138",
		"B2 E4	5139",
		"B2 E5	5142",
		"B2 E6	514A",
		"B2 E7	514F",
		"B2 E8	5153",
		"B2 E9	5155",
		"B2 EA	5157",
		"B2 EB	5158",
		"B2 EC	515F",
		"B2 ED	5164",
		"B2 EE	5166",
		"B2 EF	517E",
		"B2 F0	5183",
		"B2 F1	5184",
		"B2 F2	518B",
		"B2 F3	518E",
		"B2 F4	5198",
		"B2 F5	519D",
		"B2 F6	51A1",
		"B2 F7	51A3",
		"B2 F8	51AD",
		"B2 F9	51B8",
		"B2 FA	51BA",
		"B2 FB	51BC",
		"B2 FC	51BE",
		"B2 FD	51BF",
		"B2 FE	51C2",
		"B3 A1	51C8",
		"B3 A2	51CF",
		"B3 A3	51D1",
		"B3 A4	51D2",
		"B3 A5	51D3",
		"B3 A6	51D5",
		"B3 A7	51D8",
		"B3 A8	51DE",
		"B3 A9	51E2",
		"B3 AA	51E5",
		"B3 AB	51EE",
		"B3 AC	51F2",
		"B3 AD	51F3",
		"B3 AE	51F4",
		"B3 AF	51F7",
		"B3 B0	5201",
		"B3 B1	5202",
		"B3 B2	5205",
		"B3 B3	5212",
		"B3 B4	5213",
		"B3 B5	5215",
		"B3 B6	5216",
		"B3 B7	5218",
		"B3 B8	5222",
		"B3 B9	5228",
		"B3 BA	5231",
		"B3 BB	5232",
		"B3 BC	5235",
		"B3 BD	523C",
		"B3 BE	5245",
		"B3 BF	5249",
		"B3 C0	5255",
		"B3 C1	5257",
		"B3 C2	5258",
		"B3 C3	525A",
		"B3 C4	525C",
		"B3 C5	525F",
		"B3 C6	5260",
		"B3 C7	5261",
		"B3 C8	5266",
		"B3 C9	526E",
		"B3 CA	5277",
		"B3 CB	5278",
		"B3 CC	5279",
		"B3 CD	5280",
		"B3 CE	5282",
		"B3 CF	5285",
		"B3 D0	528A",
		"B3 D1	528C",
		"B3 D2	5293",
		"B3 D3	5295",
		"B3 D4	5296",
		"B3 D5	5297",
		"B3 D6	5298",
		"B3 D7	529A",
		"B3 D8	529C",
		"B3 D9	52A4",
		"B3 DA	52A5",
		"B3 DB	52A6",
		"B3 DC	52A7",
		"B3 DD	52AF",
		"B3 DE	52B0",
		"B3 DF	52B6",
		"B3 E0	52B7",
		"B3 E1	52B8",
		"B3 E2	52BA",
		"B3 E3	52BB",
		"B3 E4	52BD",
		"B3 E5	52C0",
		"B3 E6	52C4",
		"B3 E7	52C6",
		"B3 E8	52C8",
		"B3 E9	52CC",
		"B3 EA	52CF",
		"B3 EB	52D1",
		"B3 EC	52D4",
		"B3 ED	52D6",
		"B3 EE	52DB",
		"B3 EF	52DC",
		"B3 F0	52E1",
		"B3 F1	52E5",
		"B3 F2	52E8",
		"B3 F3	52E9",
		"B3 F4	52EA",
		"B3 F5	52EC",
		"B3 F6	52F0",
		"B3 F7	52F1",
		"B3 F8	52F4",
		"B3 F9	52F6",
		"B3 FA	52F7",
		"B3 FB	5300",
		"B3 FC	5303",
		"B3 FD	530A",
		"B3 FE	530B",
		"B4 A1	530C",
		"B4 A2	5311",
		"B4 A3	5313",
		"B4 A4	5318",
		"B4 A5	531B",
		"B4 A6	531C",
		"B4 A7	531E",
		"B4 A8	531F",
		"B4 A9	5325",
		"B4 AA	5327",
		"B4 AB	5328",
		"B4 AC	5329",
		"B4 AD	532B",
		"B4 AE	532C",
		"B4 AF	532D",
		"B4 B0	5330",
		"B4 B1	5332",
		"B4 B2	5335",
		"B4 B3	533C",
		"B4 B4	533D",
		"B4 B5	533E",
		"B4 B6	5342",
		"B4 B8	534B",
		"B4 B7	534C",
		"B4 B9	5359",
		"B4 BA	535B",
		"B4 BB	5361",
		"B4 BC	5363",
		"B4 BD	5365",
		"B4 BE	536C",
		"B4 BF	536D",
		"B4 C0	5372",
		"B4 C1	5379",
		"B4 C2	537E",
		"B4 C3	5383",
		"B4 C4	5387",
		"B4 C5	5388",
		"B4 C6	538E",
		"B4 C7	5393",
		"B4 C8	5394",
		"B4 C9	5399",
		"B4 CA	539D",
		"B4 CB	53A1",
		"B4 CC	53A4",
		"B4 CD	53AA",
		"B4 CE	53AB",
		"B4 CF	53AF",
		"B4 D0	53B2",
		"B4 D1	53B4",
		"B4 D2	53B5",
		"B4 D3	53B7",
		"B4 D4	53B8",
		"B4 D5	53BA",
		"B4 D6	53BD",
		"B4 D7	53C0",
		"B4 D8	53C5",
		"B4 D9	53CF",
		"B4 DA	53D2",
		"B4 DB	53D3",
		"B4 DC	53D5",
		"B4 DD	53DA",
		"B4 DE	53DD",
		"B4 DF	53DE",
		"B4 E0	53E0",
		"B4 E1	53E6",
		"B4 E2	53E7",
		"B4 E3	53F5",
		"B4 E4	5402",
		"B4 E5	5413",
		"B4 E6	541A",
		"B4 E7	5421",
		"B4 E8	5427",
		"B4 E9	5428",
		"B4 EA	542A",
		"B4 EB	542F",
		"B4 EC	5431",
		"B4 ED	5434",
		"B4 EE	5435",
		"B4 EF	5443",
		"B4 F0	5444",
		"B4 F1	5447",
		"B4 F2	544D",
		"B4 F3	544F",
		"B4 F4	545E",
		"B4 F5	5462",
		"B4 F6	5464",
		"B4 F7	5466",
		"B4 F8	5467",
		"B4 F9	5469",
		"B4 FA	546B",
		"B4 FB	546D",
		"B4 FC	546E",
		"B4 FD	5474",
		"B4 FE	547F",
		"B5 A1	5481",
		"B5 A2	5483",
		"B5 A3	5485",
		"B5 A4	5488",
		"B5 A5	5489",
		"B5 A6	548D",
		"B5 A7	5491",
		"B5 A8	5495",
		"B5 A9	5496",
		"B5 AA	549C",
		"B5 AB	549F",
		"B5 AC	54A1",
		"B5 AD	54A6",
		"B5 AE	54A7",
		"B5 AF	54A9",
		"B5 B0	54AA",
		"B5 B1	54AD",
		"B5 B2	54AE",
		"B5 B3	54B1",
		"B5 B4	54B7",
		"B5 B5	54B9",
		"B5 B6	54BA",
		"B5 B7	54BB",
		"B5 B8	54BF",
		"B5 B9	54C6",
		"B5 BA	54CA",
		"B5 BB	54CD",
		"B5 BC	54CE",
		"B5 BD	54E0",
		"B5 BE	54EA",
		"B5 BF	54EC",
		"B5 C0	54EF",
		"B5 C1	54F6",
		"B5 C2	54FC",
		"B5 C3	54FE",
		"B5 C4	54FF",
		"B5 C5	5500",
		"B5 C6	5501",
		"B5 C7	5505",
		"B5 C8	5508",
		"B5 C9	5509",
		"B5 CA	550C",
		"B5 CB	550D",
		"B5 CC	550E",
		"B5 CD	5515",
		"B5 CE	552A",
		"B5 CF	552B",
		"B5 D0	5532",
		"B5 D1	5535",
		"B5 D2	5536",
		"B5 D3	553B",
		"B5 D4	553C",
		"B5 D5	553D",
		"B5 D6	5541",
		"B5 D7	5547",
		"B5 D8	5549",
		"B5 D9	554A",
		"B5 DA	554D",
		"B5 DB	5550",
		"B5 DC	5551",
		"B5 DD	5558",
		"B5 DE	555A",
		"B5 DF	555B",
		"B5 E0	555E",
		"B5 E1	5560",
		"B5 E2	5561",
		"B5 E3	5564",
		"B5 E4	5566",
		"B5 E5	557F",
		"B5 E6	5581",
		"B5 E7	5582",
		"B5 E8	5586",
		"B5 E9	5588",
		"B5 EA	558E",
		"B5 EB	558F",
		"B5 EC	5591",
		"B5 ED	5592",
		"B5 EE	5593",
		"B5 EF	5594",
		"B5 F0	5597",
		"B5 F1	55A3",
		"B5 F2	55A4",
		"B5 F3	55AD",
		"B5 F4	55B2",
		"B5 F5	55BF",
		"B5 F6	55C1",
		"B5 F7	55C3",
		"B5 F8	55C6",
		"B5 F9	55C9",
		"B5 FA	55CB",
		"B5 FB	55CC",
		"B5 FC	55CE",
		"B5 FD	55D1",
		"B5 FE	55D2",
		"B6 A1	55D3",
		"B6 A2	55D7",
		"B6 A3	55D8",
		"B6 A4	55DB",
		"B6 A5	55DE",
		"B6 A6	55E2",
		"B6 A7	55E9",
		"B6 A8	55F6",
		"B6 A9	55FF",
		"B6 AA	5605",
		"B6 AB	5608",
		"B6 AC	560A",
		"B6 AD	560D",
		"B6 AE	560E",
		"B6 AF	560F",
		"B6 B0	5610",
		"B6 B1	5611",
		"B6 B2	5612",
		"B6 B3	5619",
		"B6 B4	562C",
		"B6 B5	5630",
		"B6 B6	5633",
		"B6 B7	5635",
		"B6 B8	5637",
		"B6 B9	5639",
		"B6 BA	563B",
		"B6 BB	563C",
		"B6 BC	563D",
		"B6 BD	563F",
		"B6 BE	5640",
		"B6 BF	5641",
		"B6 C0	5643",
		"B6 C1	5644",
		"B6 C2	5646",
		"B6 C3	5649",
		"B6 C4	564B",
		"B6 C5	564D",
		"B6 C6	564F",
		"B6 C7	5654",
		"B6 C8	565E",
		"B6 C9	5660",
		"B6 CA	5661",
		"B6 CB	5662",
		"B6 CC	5663",
		"B6 CD	5666",
		"B6 CE	5669",
		"B6 CF	566D",
		"B6 D0	566F",
		"B6 D1	5671",
		"B6 D2	5672",
		"B6 D3	5675",
		"B6 D4	5684",
		"B6 D5	5685",
		"B6 D6	5688",
		"B6 D7	568B",
		"B6 D8	568C",
		"B6 D9	5695",
		"B6 DA	5699",
		"B6 DB	569A",
		"B6 DC	569D",
		"B6 DD	569E",
		"B6 DE	569F",
		"B6 DF	56A6",
		"B6 E0	56A7",
		"B6 E1	56A8",
		"B6 E2	56A9",
		"B6 E3	56AB",
		"B6 E4	56AC",
		"B6 E5	56AD",
		"B6 E6	56B1",
		"B6 E7	56B3",
		"B6 E8	56B7",
		"B6 E9	56BE",
		"B6 EA	56C5",
		"B6 EB	56C9",
		"B6 EC	56CA",
		"B6 ED	56CB",
		"B6 F0	56CC",
		"B6 F1	56CD",
		"B6 EE	56CF",
		"B6 EF	56D0",
		"B6 F2	56D9",
		"B6 F3	56DC",
		"B6 F4	56DD",
		"B6 F5	56DF",
		"B6 F6	56E1",
		"B6 F7	56E4",
		"B6 F8	56E5",
		"B6 F9	56E6",
		"B6 FA	56E7",
		"B6 FB	56E8",
		"B6 FD	56EB",
		"B6 FE	56ED",
		"B6 FC	56F1",
		"B7 A1	56F6",
		"B7 A2	56F7",
		"B7 A3	5701",
		"B7 A4	5702",
		"B7 A5	5707",
		"B7 A6	570A",
		"B7 A7	570C",
		"B7 A8	5711",
		"B7 A9	5715",
		"B7 AA	571A",
		"B7 AB	571B",
		"B7 AC	571D",
		"B7 AD	5720",
		"B7 AE	5722",
		"B7 AF	5723",
		"B7 B0	5724",
		"B7 B1	5725",
		"B7 B2	5729",
		"B7 B3	572A",
		"B7 B4	572C",
		"B7 B5	572E",
		"B7 B6	572F",
		"B7 B7	5733",
		"B7 B8	5734",
		"B7 B9	573D",
		"B7 BA	573E",
		"B7 BB	573F",
		"B7 BC	5745",
		"B7 BD	5746",
		"B7 BE	574C",
		"B7 BF	574D",
		"B7 C0	5752",
		"B7 C1	5762",
		"B7 C2	5765",
		"B7 C3	5767",
		"B7 C4	5768",
		"B7 C5	576B",
		"B7 C6	576D",
		"B7 C7	576E",
		"B7 C8	576F",
		"B7 C9	5770",
		"B7 CA	5771",
		"B7 CB	5773",
		"B7 CC	5774",
		"B7 CD	5775",
		"B7 CE	5777",
		"B7 CF	5779",
		"B7 D0	577A",
		"B7 D1	577B",
		"B7 D2	577C",
		"B7 D3	577E",
		"B7 D4	5781",
		"B7 D5	5783",
		"B7 D6	578C",
		"B7 D7	5794",
		"B7 E0	5795",
		"B7 D8	5797",
		"B7 D9	5799",
		"B7 DA	579A",
		"B7 DB	579C",
		"B7 DC	579D",
		"B7 DD	579E",
		"B7 DE	579F",
		"B7 DF	57A1",
		"B7 E1	57A7",
		"B7 E2	57A8",
		"B7 E3	57A9",
		"B7 E4	57AC",
		"B7 E5	57B8",
		"B7 E6	57BD",
		"B7 E7	57C7",
		"B7 E8	57C8",
		"B7 E9	57CC",
		"B7 EA	57CF",
		"B7 EB	57D5",
		"B7 EC	57DD",
		"B7 ED	57DE",
		"B7 FE	57E1",
		"B7 EE	57E4",
		"B7 EF	57E6",
		"B7 F0	57E7",
		"B7 F1	57E9",
		"B7 F2	57ED",
		"B7 F3	57F0",
		"B7 F4	57F5",
		"B7 F5	57F6",
		"B7 F6	57F8",
		"B7 F7	57FD",
		"B7 F8	57FE",
		"B7 F9	57FF",
		"B7 FA	5803",
		"B7 FB	5804",
		"B7 FC	5808",
		"B7 FD	5809",
		"B8 A1	580C",
		"B8 A2	580D",
		"B8 A3	581B",
		"B8 A4	581E",
		"B8 A5	581F",
		"B8 A6	5820",
		"B8 A7	5826",
		"B8 A8	5827",
		"B8 A9	582D",
		"B8 AA	5832",
		"B8 AB	5839",
		"B8 AC	583F",
		"B8 AD	5849",
		"B8 AE	584C",
		"B8 AF	584D",
		"B8 B0	584F",
		"B8 B1	5850",
		"B8 B2	5855",
		"B8 B3	585F",
		"B8 B4	5861",
		"B8 B5	5864",
		"B8 B6	5867",
		"B8 B7	5868",
		"B8 B8	5878",
		"B8 B9	587C",
		"B8 BA	587F",
		"B8 BB	5880",
		"B8 BC	5881",
		"B8 BD	5887",
		"B8 BE	5888",
		"B8 BF	5889",
		"B8 C0	588A",
		"B8 C1	588C",
		"B8 C2	588D",
		"B8 C3	588F",
		"B8 C4	5890",
		"B8 C5	5894",
		"B8 C6	5896",
		"B8 C7	589D",
		"B8 C8	58A0",
		"B8 C9	58A1",
		"B8 CA	58A2",
		"B8 CB	58A6",
		"B8 CC	58A9",
		"B8 CD	58B1",
		"B8 CE	58B2",
		"B8 D0	58BC",
		"B8 D1	58C2",
		"B8 CF	58C4",
		"B8 D2	58C8",
		"B8 D3	58CD",
		"B8 D4	58CE",
		"B8 D5	58D0",
		"B8 D6	58D2",
		"B8 D7	58D4",
		"B8 D8	58D6",
		"B8 D9	58DA",
		"B8 DA	58DD",
		"B8 DB	58E1",
		"B8 DC	58E2",
		"B8 DD	58E9",
		"B8 DE	58F3",
		"B8 DF	5905",
		"B8 E0	5906",
		"B8 E1	590B",
		"B8 E2	590C",
		"B8 E3	5912",
		"B8 E4	5913",
		"B8 E5	5914",
		"B8 E7	591D",
		"B8 E8	5921",
		"B8 E9	5923",
		"B8 EA	5924",
		"B8 EB	5928",
		"B8 EC	592F",
		"B8 ED	5930",
		"B8 EE	5933",
		"B8 EF	5935",
		"B8 F0	5936",
		"B8 F1	593F",
		"B8 F2	5943",
		"B8 F3	5946",
		"B8 F4	5952",
		"B8 F5	5953",
		"B8 F6	5959",
		"B8 F7	595B",
		"B8 F8	595D",
		"B8 F9	595E",
		"B8 FA	595F",
		"B8 FB	5961",
		"B8 FC	5963",
		"B8 FD	596B",
		"B8 FE	596D",
		"B9 A1	596F",
		"B9 A2	5972",
		"B9 A3	5975",
		"B9 A4	5976",
		"B9 A5	5979",
		"B9 A6	597B",
		"B9 A7	597C",
		"B9 A8	598B",
		"B9 A9	598C",
		"B9 AA	598E",
		"B9 AB	5992",
		"B9 AC	5995",
		"B9 AD	5997",
		"B9 AE	599F",
		"B9 AF	59A4",
		"B9 B0	59A7",
		"B9 B1	59AD",
		"B9 B2	59AE",
		"B9 B3	59AF",
		"B9 B4	59B0",
		"B9 B5	59B3",
		"B9 B6	59B7",
		"B9 B7	59BA",
		"B9 B8	59BC",
		"B9 B9	59C1",
		"B9 BA	59C3",
		"B9 BB	59C4",
		"B9 BC	59C8",
		"B9 BD	59CA",
		"B9 BE	59CD",
		"B9 BF	59D2",
		"B9 C0	59DD",
		"B9 C1	59DE",
		"B9 C2	59DF",
		"B9 C3	59E3",
		"B9 C4	59E4",
		"B9 C5	59E7",
		"B9 C6	59EE",
		"B9 C7	59EF",
		"B9 C8	59F1",
		"B9 C9	59F2",
		"B9 CA	59F4",
		"B9 CB	59F7",
		"B9 CC	5A00",
		"B9 CD	5A04",
		"B9 CE	5A0C",
		"B9 CF	5A0D",
		"B9 D0	5A0E",
		"B9 D1	5A12",
		"B9 D2	5A13",
		"B9 D3	5A1E",
		"B9 D4	5A23",
		"B9 D5	5A24",
		"B9 D6	5A27",
		"B9 D7	5A28",
		"B9 D8	5A2A",
		"B9 D9	5A2D",
		"B9 DA	5A30",
		"B9 DB	5A44",
		"B9 DC	5A45",
		"B9 DD	5A47",
		"B9 DE	5A48",
		"B9 DF	5A4C",
		"B9 E0	5A50",
		"B9 E1	5A55",
		"B9 E2	5A5E",
		"B9 E3	5A63",
		"B9 E4	5A65",
		"B9 E5	5A67",
		"B9 E6	5A6D",
		"B9 E7	5A77",
		"B9 E8	5A7A",
		"B9 E9	5A7B",
		"B9 EA	5A7E",
		"B9 EB	5A8B",
		"B9 EC	5A90",
		"B9 ED	5A93",
		"B9 EE	5A96",
		"B9 EF	5A99",
		"B9 F0	5A9C",
		"B9 F1	5A9E",
		"B9 F2	5A9F",
		"B9 F3	5AA0",
		"B9 F4	5AA2",
		"B9 F5	5AA7",
		"B9 F6	5AAC",
		"B9 F7	5AB1",
		"B9 F8	5AB2",
		"B9 F9	5AB3",
		"B9 FA	5AB5",
		"B9 FB	5AB8",
		"B9 FC	5ABA",
		"B9 FD	5ABB",
		"B9 FE	5ABF",
		"BA A1	5AC4",
		"BA A2	5AC6",
		"BA A3	5AC8",
		"BA A4	5ACF",
		"BA A5	5ADA",
		"BA A6	5ADC",
		"BA A7	5AE0",
		"BA A8	5AE5",
		"BA A9	5AEA",
		"BA AA	5AEE",
		"BA AB	5AF5",
		"BA AC	5AF6",
		"BA AD	5AFD",
		"BA AE	5B00",
		"BA AF	5B01",
		"BA B0	5B08",
		"BA B1	5B17",
		"BA B3	5B19",
		"BA B4	5B1B",
		"BA B5	5B1D",
		"BA B6	5B21",
		"BA B7	5B25",
		"BA B8	5B2D",
		"BA B2	5B34",
		"BA B9	5B38",
		"BA BA	5B41",
		"BA BB	5B4B",
		"BA BC	5B4C",
		"BA BD	5B52",
		"BA BE	5B56",
		"BA BF	5B5E",
		"BA C0	5B68",
		"BA C1	5B6E",
		"BA C2	5B6F",
		"BA C3	5B7C",
		"BA C4	5B7D",
		"BA C5	5B7E",
		"BA C6	5B7F",
		"BA C7	5B81",
		"BA C8	5B84",
		"BA C9	5B86",
		"BA CA	5B8A",
		"BA CB	5B8E",
		"BA CC	5B90",
		"BA CD	5B91",
		"BA CE	5B93",
		"BA CF	5B94",
		"BA D0	5B96",
		"BA D1	5BA8",
		"BA D2	5BA9",
		"BA D3	5BAC",
		"BA D4	5BAD",
		"BA D5	5BAF",
		"BA D6	5BB1",
		"BA D7	5BB2",
		"BA D8	5BB7",
		"BA D9	5BBA",
		"BA DA	5BBC",
		"BA DB	5BC0",
		"BA DC	5BC1",
		"BA DD	5BCD",
		"BA DE	5BCF",
		"BA DF	5BD6",
		"BA E0	5BD7",
		"BA E1	5BD8",
		"BA E2	5BD9",
		"BA E3	5BDA",
		"BA E4	5BE0",
		"BA E5	5BEF",
		"BA E6	5BF1",
		"BA E7	5BF4",
		"BA E8	5BFD",
		"BA E9	5C0C",
		"BA EA	5C17",
		"BA EB	5C1E",
		"BA EC	5C1F",
		"BA ED	5C23",
		"BA EE	5C26",
		"BA EF	5C29",
		"BA F0	5C2B",
		"BA F1	5C2C",
		"BA F2	5C2E",
		"BA F3	5C30",
		"BA F4	5C32",
		"BA F5	5C35",
		"BA F6	5C36",
		"BA F7	5C59",
		"BA F8	5C5A",
		"BA F9	5C5C",
		"BA FA	5C62",
		"BA FB	5C63",
		"BA FC	5C67",
		"BA FD	5C68",
		"BA FE	5C69",
		"BB A1	5C6D",
		"BB A2	5C70",
		"BB A3	5C74",
		"BB A4	5C75",
		"BB A5	5C7A",
		"BB A6	5C7B",
		"BB A7	5C7C",
		"BB A8	5C7D",
		"BB A9	5C87",
		"BB AA	5C88",
		"BB AB	5C8A",
		"BB AC	5C8F",
		"BB AD	5C92",
		"BB AE	5C9D",
		"BB AF	5C9F",
		"BB B0	5CA0",
		"BB B1	5CA2",
		"BB B2	5CA3",
		"BB B3	5CA6",
		"BB B4	5CAA",
		"BB B5	5CB2",
		"BB B6	5CB4",
		"BB B7	5CB5",
		"BB B8	5CBA",
		"BB B9	5CC9",
		"BB BA	5CCB",
		"BB BB	5CD2",
		"BB BD	5CD7",
		"BB BC	5CDD",
		"BB BE	5CEE",
		"BB BF	5CF1",
		"BB C0	5CF2",
		"BB C1	5CF4",
		"BB C2	5D01",
		"BB C3	5D06",
		"BB C4	5D0D",
		"BB C5	5D12",
		"BB C7	5D23",
		"BB C8	5D24",
		"BB C9	5D26",
		"BB CA	5D27",
		"BB C6	5D2B",
		"BB CB	5D31",
		"BB CC	5D34",
		"BB CD	5D39",
		"BB CE	5D3D",
		"BB CF	5D3F",
		"BB D0	5D42",
		"BB D1	5D43",
		"BB D2	5D46",
		"BB D3	5D48",
		"BB D7	5D4A",
		"BB D5	5D51",
		"BB D4	5D55",
		"BB D6	5D59",
		"BB D8	5D5F",
		"BB D9	5D60",
		"BB DA	5D61",
		"BB DB	5D62",
		"BB DC	5D64",
		"BB DD	5D6A",
		"BB DE	5D6D",
		"BB DF	5D70",
		"BB E0	5D79",
		"BB E1	5D7A",
		"BB E2	5D7E",
		"BB E3	5D7F",
		"BB E4	5D81",
		"BB E5	5D83",
		"BB E6	5D88",
		"BB E7	5D8A",
		"BB E8	5D92",
		"BB E9	5D93",
		"BB EA	5D94",
		"BB EB	5D95",
		"BB EC	5D99",
		"BB ED	5D9B",
		"BB EE	5D9F",
		"BB EF	5DA0",
		"BB F0	5DA7",
		"BB F1	5DAB",
		"BB F2	5DB0",
		"E6 F4	5DB2",
		"BB F3	5DB4",
		"BB F4	5DB8",
		"BB F5	5DB9",
		"BB F6	5DC3",
		"BB F7	5DC7",
		"BB F8	5DCB",
		"BB FA	5DCE",
		"BB F9	5DD0",
		"BB FB	5DD8",
		"BB FC	5DD9",
		"BB FD	5DE0",
		"BB FE	5DE4",
		"BC A1	5DE9",
		"BC A2	5DF8",
		"BC A3	5DF9",
		"BC A4	5E00",
		"BC A5	5E07",
		"BC A6	5E0D",
		"BC A7	5E12",
		"BC A8	5E14",
		"BC A9	5E15",
		"BC AA	5E18",
		"BC AB	5E1F",
		"BC AC	5E20",
		"BC AE	5E28",
		"BC AD	5E2E",
		"BC AF	5E32",
		"BC B0	5E35",
		"BC B1	5E3E",
		"BC B4	5E49",
		"BC B2	5E4B",
		"BC B3	5E50",
		"BC B5	5E51",
		"BC B6	5E56",
		"BC B7	5E58",
		"BC B8	5E5B",
		"BC B9	5E5C",
		"BC BA	5E5E",
		"BC BB	5E68",
		"BC BC	5E6A",
		"BC BD	5E6B",
		"BC BE	5E6C",
		"BC BF	5E6D",
		"BC C0	5E6E",
		"BC C1	5E70",
		"BC C2	5E80",
		"BC C3	5E8B",
		"BC C4	5E8E",
		"BC C5	5EA2",
		"BC C6	5EA4",
		"BC C7	5EA5",
		"BC C8	5EA8",
		"BC C9	5EAA",
		"BC CA	5EAC",
		"BC CB	5EB1",
		"BC CC	5EB3",
		"BC CD	5EBD",
		"BC CE	5EBE",
		"BC CF	5EBF",
		"BC D0	5EC6",
		"BC D2	5ECB",
		"BC D1	5ECC",
		"BC D3	5ECE",
		"BC D4	5ED1",
		"BC D5	5ED2",
		"BC D6	5ED4",
		"BC D7	5ED5",
		"BC D8	5EDC",
		"BC D9	5EDE",
		"BC DA	5EE5",
		"BC DB	5EEB",
		"BC DC	5F02",
		"BC DD	5F06",
		"BC DE	5F07",
		"BC DF	5F08",
		"BC E0	5F0E",
		"BC E1	5F19",
		"BC E2	5F1C",
		"BC E3	5F1D",
		"BC E4	5F21",
		"BC E5	5F22",
		"BC E6	5F23",
		"BC E7	5F24",
		"BC E8	5F28",
		"BC E9	5F2B",
		"BC EA	5F2C",
		"BC EB	5F2E",
		"BC EC	5F30",
		"BC ED	5F34",
		"BC EE	5F36",
		"BC EF	5F3B",
		"BC F0	5F3D",
		"BC F1	5F3F",
		"BC F2	5F40",
		"BC F3	5F44",
		"BC F4	5F45",
		"BC F5	5F47",
		"BC F6	5F4D",
		"BC F7	5F50",
		"BC F8	5F54",
		"BC F9	5F58",
		"BC FA	5F5B",
		"BC FB	5F60",
		"BC FC	5F63",
		"BC FD	5F64",
		"BC FE	5F67",
		"BD A1	5F6F",
		"BD A2	5F72",
		"BD A3	5F74",
		"BD A4	5F75",
		"BD A5	5F78",
		"BD A6	5F7A",
		"BD A7	5F7D",
		"BD A8	5F7E",
		"BD A9	5F89",
		"BD AA	5F8D",
		"BD AB	5F8F",
		"BD AC	5F96",
		"BD AD	5F9C",
		"BD AE	5F9D",
		"BD AF	5FA2",
		"BD B2	5FA4",
		"BD B0	5FA7",
		"BD B1	5FAB",
		"BD B3	5FAC",
		"BD B4	5FAF",
		"BD B5	5FB0",
		"BD B6	5FB1",
		"BD B7	5FB8",
		"BD B8	5FC4",
		"BD B9	5FC7",
		"BD BA	5FC8",
		"BD BB	5FC9",
		"BD BC	5FCB",
		"BD BD	5FD0",
		"BD BE	5FD1",
		"BD BF	5FD2",
		"BD C0	5FD3",
		"BD C1	5FD4",
		"BD C2	5FDE",
		"BD C3	5FE1",
		"BD C4	5FE2",
		"BD C5	5FE8",
		"BD C6	5FE9",
		"BD C7	5FEA",
		"BD C8	5FEC",
		"BD C9	5FED",
		"BD CA	5FEE",
		"BD CB	5FEF",
		"BD CC	5FF2",
		"BD CD	5FF3",
		"BD CE	5FF6",
		"BD CF	5FFA",
		"BD D0	5FFC",
		"BD D1	6007",
		"BD D2	600A",
		"BD D3	600D",
		"BD D4	6013",
		"BD D5	6014",
		"BD D6	6017",
		"BD D7	6018",
		"BD D8	601A",
		"BD D9	601F",
		"BD DA	6024",
		"BD DB	602D",
		"BD DC	6033",
		"BD DD	6035",
		"BD DE	6040",
		"BD DF	6047",
		"BD E0	6048",
		"BD E1	6049",
		"BD E2	604C",
		"BD E3	6051",
		"BD E4	6054",
		"BD E5	6056",
		"BD E6	6057",
		"BD E7	605D",
		"BD E8	6061",
		"BD E9	6067",
		"BD EA	6071",
		"BD EB	607E",
		"BD EC	607F",
		"BD ED	6082",
		"BD EE	6086",
		"BD EF	6088",
		"BD F0	608A",
		"BD F1	608E",
		"BD F2	6091",
		"BD F3	6093",
		"BD F4	6095",
		"BD F5	6098",
		"BD F6	609D",
		"BD F7	609E",
		"BD F8	60A2",
		"BD F9	60A4",
		"BD FA	60A5",
		"BD FB	60A8",
		"BD FC	60B0",
		"BD FD	60B1",
		"BD FE	60B7",
		"BE A1	60BB",
		"BE A2	60BE",
		"BE A3	60C2",
		"BE A4	60C4",
		"BE A5	60C8",
		"BE A6	60C9",
		"BE A7	60CA",
		"BE A8	60CB",
		"BE A9	60CE",
		"BE AA	60CF",
		"BE AB	60D4",
		"BE AC	60D5",
		"BE AD	60D9",
		"BE AE	60DB",
		"BE AF	60DD",
		"BE B0	60DE",
		"BE B1	60E2",
		"BE B2	60E5",
		"BE B3	60F2",
		"BE B4	60F5",
		"BE B5	60F8",
		"BE B6	60FC",
		"BE B7	60FD",
		"BE B8	6102",
		"BE B9	6107",
		"BE BA	610A",
		"BE BB	610C",
		"BE BC	6110",
		"BE BD	6111",
		"BE BE	6112",
		"BE BF	6113",
		"BE C0	6114",
		"BE C1	6116",
		"BE C2	6117",
		"BE C3	6119",
		"BE C4	611C",
		"BE C5	611E",
		"BE C6	6122",
		"BE C7	612A",
		"BE C8	612B",
		"BE C9	6130",
		"BE CA	6131",
		"BE CB	6135",
		"BE CC	6136",
		"BE CD	6137",
		"BE CE	6139",
		"BE CF	6141",
		"BE D0	6145",
		"BE D1	6146",
		"BE D2	6149",
		"BE D3	615E",
		"BE D4	6160",
		"BE D5	616C",
		"BE D6	6172",
		"BE D7	6178",
		"BE D8	617B",
		"BE D9	617C",
		"BE DA	617F",
		"BE DB	6180",
		"BE DC	6181",
		"BE DD	6183",
		"BE DE	6184",
		"BE DF	618B",
		"BE E0	618D",
		"BE E1	6192",
		"BE E2	6193",
		"BE E3	6197",
		"BE E4	6198",
		"BE E5	619C",
		"BE E6	619D",
		"BE E7	619F",
		"BE E8	61A0",
		"BE E9	61A5",
		"BE EA	61A8",
		"BE EB	61AA",
		"BE EC	61AD",
		"BE ED	61B8",
		"BE EE	61B9",
		"BE EF	61BC",
		"BE F0	61C0",
		"BE F1	61C1",
		"BE F2	61C2",
		"BE F3	61CE",
		"BE F4	61CF",
		"BE F5	61D5",
		"BE F6	61DC",
		"BE F7	61DD",
		"BE F8	61DE",
		"BE F9	61DF",
		"BE FA	61E1",
		"BE FB	61E2",
		"BE FE	61E5",
		"BE FC	61E7",
		"BE FD	61E9",
		"BF A1	61EC",
		"BF A2	61ED",
		"BF A3	61EF",
		"BF A4	6201",
		"BF A5	6203",
		"BF A6	6204",
		"BF A7	6207",
		"BF A8	6213",
		"BF A9	6215",
		"BF AA	621C",
		"BF AB	6220",
		"BF AC	6222",
		"BF AD	6223",
		"BF AE	6227",
		"BF AF	6229",
		"BF B0	622B",
		"BF B1	6239",
		"BF B2	623D",
		"BF B3	6242",
		"BF B4	6243",
		"BF B5	6244",
		"BF B6	6246",
		"BF B7	624C",
		"BF B8	6250",
		"BF B9	6251",
		"BF BA	6252",
		"BF BB	6254",
		"BF BC	6256",
		"BF BD	625A",
		"BF BE	625C",
		"BF BF	6264",
		"BF C0	626D",
		"BF C1	626F",
		"BF C2	6273",
		"BF C3	627A",
		"BF C4	627D",
		"BF C5	628D",
		"BF C6	628E",
		"BF C7	628F",
		"BF C8	6290",
		"BF C9	62A6",
		"BF CA	62A8",
		"BF CB	62B3",
		"BF CC	62B6",
		"BF CD	62B7",
		"BF CE	62BA",
		"BF CF	62BE",
		"BF D0	62BF",
		"BF D1	62C4",
		"BF D2	62CE",
		"BF D3	62D5",
		"BF D4	62D6",
		"BF D5	62DA",
		"BF D6	62EA",
		"BF D7	62F2",
		"BF D8	62F4",
		"BF D9	62FC",
		"BF DA	62FD",
		"BF DB	6303",
		"BF DC	6304",
		"BF DD	630A",
		"BF DE	630B",
		"BF DF	630D",
		"BF E0	6310",
		"BF E1	6313",
		"BF E2	6316",
		"BF E3	6318",
		"BF E4	6329",
		"BF E5	632A",
		"BF E6	632D",
		"BF E7	6335",
		"BF E8	6336",
		"BF E9	6339",
		"BF EA	633C",
		"BF EB	6341",
		"BF EC	6342",
		"BF ED	6343",
		"BF EE	6344",
		"BF EF	6346",
		"BF F0	634A",
		"BF F1	634B",
		"BF F2	634E",
		"BF F3	6352",
		"BF F4	6353",
		"BF F5	6354",
		"BF F6	6358",
		"BF F7	635B",
		"BF F8	6365",
		"BF F9	6366",
		"BF FA	636C",
		"BF FB	636D",
		"BF FC	6371",
		"BF FD	6374",
		"BF FE	6375",
		"C0 A1	6378",
		"C0 A2	637C",
		"C0 A3	637D",
		"C0 A4	637F",
		"C0 A5	6382",
		"C0 A6	6384",
		"C0 A7	6387",
		"C0 A8	638A",
		"C0 A9	6390",
		"C0 AA	6394",
		"C0 AB	6395",
		"C0 AC	6399",
		"C0 AD	639A",
		"C0 AE	639E",
		"C0 AF	63A4",
		"C0 B0	63A6",
		"C0 B1	63AD",
		"C0 B2	63AE",
		"C0 B3	63AF",
		"C0 B4	63BD",
		"C0 B5	63C1",
		"C0 B6	63C5",
		"C0 B7	63C8",
		"C0 B8	63CE",
		"C0 B9	63D1",
		"C0 BA	63D3",
		"C0 BB	63D4",
		"C0 BC	63D5",
		"C0 BD	63DC",
		"C0 BE	63E0",
		"C0 BF	63E5",
		"C0 C0	63EA",
		"C0 C1	63EC",
		"C0 C2	63F2",
		"C0 C3	63F3",
		"C0 C4	63F5",
		"C0 C5	63F8",
		"C0 C6	63F9",
		"C0 C7	6409",
		"C0 C8	640A",
		"C0 C9	6410",
		"C0 CA	6412",
		"C0 CB	6414",
		"C0 CC	6418",
		"C0 CD	641E",
		"C0 CE	6420",
		"C0 CF	6422",
		"C0 D0	6424",
		"C0 D1	6425",
		"C0 D2	6429",
		"C0 D3	642A",
		"C0 D4	642F",
		"C0 D5	6430",
		"C0 D6	6435",
		"C0 D7	643D",
		"C0 D8	643F",
		"C0 D9	644B",
		"C0 DA	644F",
		"C0 DB	6451",
		"C0 DC	6452",
		"C0 DD	6453",
		"C0 DE	6454",
		"C0 DF	645A",
		"C0 E0	645B",
		"C0 E1	645C",
		"C0 E2	645D",
		"C0 E3	645F",
		"C0 E4	6460",
		"C0 E5	6461",
		"C0 E6	6463",
		"C0 E7	646D",
		"C0 E8	6473",
		"C0 E9	6474",
		"C0 EA	647B",
		"C0 EB	647D",
		"C0 EC	6485",
		"C0 ED	6487",
		"C0 EE	648F",
		"C0 EF	6490",
		"C0 F0	6491",
		"C0 F1	6498",
		"C0 F2	6499",
		"C0 F3	649B",
		"C0 F4	649D",
		"C0 F5	649F",
		"C0 F6	64A1",
		"C0 F7	64A3",
		"C0 F8	64A6",
		"C0 F9	64A8",
		"C0 FA	64AC",
		"C0 FB	64B3",
		"C0 FC	64BD",
		"C0 FD	64BE",
		"C0 FE	64BF",
		"C1 A1	64C4",
		"C1 A2	64C9",
		"C1 A3	64CA",
		"C1 A4	64CB",
		"C1 A5	64CC",
		"C1 A6	64CE",
		"C1 A7	64D0",
		"C1 A8	64D1",
		"C1 A9	64D5",
		"C1 AA	64D7",
		"C1 AB	64E4",
		"C1 AC	64E5",
		"C1 AD	64E9",
		"C1 AE	64EA",
		"C1 AF	64ED",
		"C1 B0	64F0",
		"C1 B1	64F5",
		"C1 B2	64F7",
		"C1 B3	64FB",
		"C1 B4	64FF",
		"C1 B5	6501",
		"C1 B6	6504",
		"C1 B7	6508",
		"C1 B8	6509",
		"C1 B9	650A",
		"C1 BA	650F",
		"C1 BB	6513",
		"C1 BC	6514",
		"C1 BD	6516",
		"C1 BE	6519",
		"C1 BF	651B",
		"C1 C0	651E",
		"C1 C1	651F",
		"C1 C2	6522",
		"C1 C3	6526",
		"C1 C4	6529",
		"C1 C5	652E",
		"C1 C6	6531",
		"C1 C7	653A",
		"C1 C8	653C",
		"C1 C9	653D",
		"C1 CA	6543",
		"C1 CB	6547",
		"C1 CC	6549",
		"C1 CD	6550",
		"C1 CE	6552",
		"C1 CF	6554",
		"C1 D0	655F",
		"C1 D1	6560",
		"C1 D2	6567",
		"C1 D3	656B",
		"C1 D4	657A",
		"C1 D5	657D",
		"C1 D6	6581",
		"C1 D7	6585",
		"C1 D8	658A",
		"C1 D9	6592",
		"C1 DA	6595",
		"C1 DB	6598",
		"C1 DC	659D",
		"C1 DD	65A0",
		"C1 DE	65A3",
		"C1 DF	65A6",
		"C1 E0	65AE",
		"C1 E1	65B2",
		"C1 E2	65B3",
		"C1 E3	65B4",
		"C1 E4	65BF",
		"C1 E5	65C2",
		"C1 E6	65C8",
		"C1 E7	65C9",
		"C1 E8	65CE",
		"C1 E9	65D0",
		"C1 EA	65D4",
		"C1 EB	65D6",
		"C1 EC	65D8",
		"C1 ED	65DF",
		"C1 EE	65F0",
		"C1 EF	65F2",
		"C1 F0	65F4",
		"C1 F1	65F5",
		"C1 F2	65F9",
		"C1 F3	65FE",
		"C1 F4	65FF",
		"C1 F5	6600",
		"C1 F6	6604",
		"C1 F7	6608",
		"C1 F8	6609",
		"C1 F9	660D",
		"C1 FA	6611",
		"C1 FB	6612",
		"C1 FC	6615",
		"C1 FD	6616",
		"C1 FE	661D",
		"C2 A1	661E",
		"C2 A2	6621",
		"C2 A3	6622",
		"C2 A4	6623",
		"C2 A5	6624",
		"C2 A6	6626",
		"C2 A7	6629",
		"C2 A8	662A",
		"C2 A9	662B",
		"C2 AA	662C",
		"C2 AB	662E",
		"C2 AC	6630",
		"C2 AD	6631",
		"C2 AE	6633",
		"C2 B0	6637",
		"C2 AF	6639",
		"C2 B1	6640",
		"C2 B2	6645",
		"C2 B3	6646",
		"C2 B4	664A",
		"C2 B5	664C",
		"C2 B7	664E",
		"C2 B6	6651",
		"C2 B8	6657",
		"C2 B9	6658",
		"C2 BA	6659",
		"C2 BB	665B",
		"C2 BC	665C",
		"C2 BD	6660",
		"C2 BE	6661",
		"C2 C0	666A",
		"C2 C1	666B",
		"C2 C2	666C",
		"C2 C4	6673",
		"C2 C5	6675",
		"C2 C7	6677",
		"C2 C8	6678",
		"C2 C9	6679",
		"C2 CA	667B",
		"C2 CC	667C",
		"C2 C3	667E",
		"C2 C6	667F",
		"C2 CB	6680",
		"C2 CD	668B",
		"C2 CE	668C",
		"C2 CF	668D",
		"C2 D0	6690",
		"C2 D1	6692",
		"C2 D2	6699",
		"C2 D3	669A",
		"C2 D4	669B",
		"C2 D5	669C",
		"C2 D6	669F",
		"C2 D7	66A0",
		"C2 D8	66A4",
		"C2 D9	66AD",
		"C2 DA	66B1",
		"C2 DB	66B2",
		"C2 DC	66B5",
		"C2 DD	66BB",
		"C2 DE	66BF",
		"C2 DF	66C0",
		"C2 E0	66C2",
		"C2 E1	66C3",
		"C2 E2	66C8",
		"C2 E3	66CC",
		"C2 E4	66CE",
		"C2 E5	66CF",
		"C2 E6	66D4",
		"C2 E7	66DB",
		"C2 E8	66DF",
		"C2 E9	66E8",
		"C2 EA	66EB",
		"C2 EB	66EC",
		"C2 EC	66EE",
		"C2 ED	66FA",
		"C2 BF	66FB",
		"C2 EE	6705",
		"C2 EF	6707",
		"C2 F0	670E",
		"C2 F1	6713",
		"C2 F2	6719",
		"C2 F3	671C",
		"C2 F4	6720",
		"C2 F5	6722",
		"C2 F6	6733",
		"C2 F7	673E",
		"C2 F8	6745",
		"C2 F9	6747",
		"C2 FA	6748",
		"C2 FB	674C",
		"C2 FC	6754",
		"C2 FD	6755",
		"C2 FE	675D",
		"C3 A1	6766",
		"C3 A2	676C",
		"C3 A3	676E",
		"C3 A4	6774",
		"C3 A5	6776",
		"C3 A6	677B",
		"C3 A7	6781",
		"C3 A8	6784",
		"C3 A9	678E",
		"C3 AA	678F",
		"C3 AB	6791",
		"C3 AC	6793",
		"C3 AD	6796",
		"C3 AE	6798",
		"C3 AF	6799",
		"C3 B0	679B",
		"C3 B1	67B0",
		"C3 B2	67B1",
		"C3 B3	67B2",
		"C3 B4	67B5",
		"C3 B5	67BB",
		"C3 B6	67BC",
		"C3 B7	67BD",
		"C3 B9	67C0",
		"C3 BA	67C2",
		"C3 BB	67C3",
		"C3 BC	67C5",
		"C3 BD	67C8",
		"C3 BE	67C9",
		"C3 BF	67D2",
		"C3 C0	67D7",
		"C3 C1	67D9",
		"C3 C2	67DC",
		"C3 C3	67E1",
		"C3 C4	67E6",
		"C3 C5	67F0",
		"C3 C6	67F2",
		"C3 C7	67F6",
		"C3 C8	67F7",
		"C3 B8	67F9",
		"C3 CA	6814",
		"C3 CB	6819",
		"C3 CC	681D",
		"C3 CD	681F",
		"C3 CF	6827",
		"C3 CE	6828",
		"C3 D0	682C",
		"C3 D1	682D",
		"C3 D2	682F",
		"C3 D3	6830",
		"C3 D4	6831",
		"C3 D5	6833",
		"C3 D6	683B",
		"C3 D7	683F",
		"C3 D8	6844",
		"C3 D9	6845",
		"C3 DA	684A",
		"C3 DB	684C",
		"C3 C9	6852",
		"C3 DC	6855",
		"C3 DD	6857",
		"C3 DE	6858",
		"C3 DF	685B",
		"C3 E0	686B",
		"C3 E1	686E",
		"C3 E2	686F",
		"C3 E3	6870",
		"C3 E4	6871",
		"C3 E5	6872",
		"C3 E6	6875",
		"C3 E7	6879",
		"C3 E8	687A",
		"C3 E9	687B",
		"C3 EA	687C",
		"C3 EB	6882",
		"C3 EC	6884",
		"C3 ED	6886",
		"C3 EE	6888",
		"C3 EF	6896",
		"C3 F0	6898",
		"C3 F1	689A",
		"C3 F2	689C",
		"C3 F3	68A1",
		"C3 F4	68A3",
		"C3 F5	68A5",
		"C3 F6	68A9",
		"C3 F7	68AA",
		"C3 F8	68AE",
		"C3 F9	68B2",
		"C3 FA	68BB",
		"C3 FB	68C5",
		"C3 FC	68C8",
		"C3 FD	68CC",
		"C3 FE	68CF",
		"C4 A1	68D0",
		"C4 A2	68D1",
		"C4 A3	68D3",
		"C4 A4	68D6",
		"C4 A5	68D9",
		"C4 A6	68DC",
		"C4 A7	68DD",
		"C4 A8	68E5",
		"C4 A9	68E8",
		"C4 AA	68EA",
		"C4 AB	68EB",
		"C4 AC	68EC",
		"C4 AD	68ED",
		"C4 AE	68F0",
		"C4 AF	68F1",
		"C4 B0	68F5",
		"C4 B1	68F6",
		"C4 B2	68FB",
		"C4 B3	68FC",
		"C4 B4	68FD",
		"C4 B5	6906",
		"C4 B6	6909",
		"C4 B7	690A",
		"C4 B8	6910",
		"C4 B9	6911",
		"C4 BA	6913",
		"C4 BB	6916",
		"C4 BC	6917",
		"C4 BD	6931",
		"C4 BE	6933",
		"C4 BF	6935",
		"C4 C0	6938",
		"C4 C1	693B",
		"C4 C2	6942",
		"C4 C3	6945",
		"C4 C4	6949",
		"C4 C5	694E",
		"C4 C6	6957",
		"C4 C7	695B",
		"C4 C8	6963",
		"C4 C9	6964",
		"C4 CA	6965",
		"C4 CB	6966",
		"C4 CC	6968",
		"C4 CD	6969",
		"C4 CE	696C",
		"C4 CF	6970",
		"C4 D0	6971",
		"C4 D1	6972",
		"C4 D2	697A",
		"C4 D3	697B",
		"C4 D4	697F",
		"C4 D5	6980",
		"C4 D6	698D",
		"C4 D7	6992",
		"C4 D8	6996",
		"C4 D9	6998",
		"C4 DA	69A1",
		"C4 DB	69A5",
		"C4 DC	69A6",
		"C4 DD	69A8",
		"C4 DE	69AB",
		"C4 DF	69AD",
		"C4 E0	69AF",
		"C4 E1	69B7",
		"C4 E2	69B8",
		"C4 E3	69BA",
		"C4 E4	69BC",
		"C4 E5	69C5",
		"C4 E6	69C8",
		"C4 E7	69D1",
		"C4 E8	69D6",
		"C4 E9	69D7",
		"C4 EA	69E2",
		"C4 EB	69E5",
		"C4 EC	69EE",
		"C4 ED	69EF",
		"C4 EE	69F1",
		"C4 EF	69F3",
		"C4 F0	69F5",
		"C4 F1	69FE",
		"C4 F2	6A00",
		"C4 F3	6A01",
		"C4 F4	6A03",
		"C4 F5	6A0F",
		"C4 F6	6A11",
		"C4 F7	6A15",
		"C4 F8	6A1A",
		"C4 F9	6A1D",
		"C4 FA	6A20",
		"C4 FB	6A24",
		"C4 FC	6A28",
		"C4 FD	6A30",
		"C4 FE	6A32",
		"C5 A1	6A34",
		"C5 A2	6A37",
		"C5 A3	6A3B",
		"C5 A4	6A3E",
		"C5 A5	6A3F",
		"C5 A6	6A45",
		"C5 A7	6A46",
		"C5 A8	6A49",
		"C5 A9	6A4A",
		"C5 AA	6A4E",
		"C5 AB	6A50",
		"C5 AC	6A51",
		"C5 AD	6A52",
		"C5 AE	6A55",
		"C5 AF	6A56",
		"C5 B0	6A5B",
		"C5 B1	6A64",
		"C5 B2	6A67",
		"C5 B3	6A6A",
		"C5 B4	6A71",
		"C5 B5	6A73",
		"C5 B6	6A7E",
		"C5 B7	6A81",
		"C5 B8	6A83",
		"C5 B9	6A86",
		"C5 BA	6A87",
		"C5 BB	6A89",
		"C5 BC	6A8B",
		"C5 BD	6A91",
		"C5 BE	6A9B",
		"C5 BF	6A9D",
		"C5 C0	6A9E",
		"C5 C1	6A9F",
		"C5 C2	6AA5",
		"C5 C3	6AAB",
		"C5 C4	6AAF",
		"C5 C5	6AB0",
		"C5 C6	6AB1",
		"C5 C7	6AB4",
		"C5 C8	6ABD",
		"C5 C9	6ABE",
		"C5 CA	6ABF",
		"C5 CB	6AC6",
		"C5 CD	6AC8",
		"C5 CC	6AC9",
		"C5 CE	6ACC",
		"C5 CF	6AD0",
		"C5 D0	6AD4",
		"C5 D1	6AD5",
		"C5 D2	6AD6",
		"C5 D3	6ADC",
		"C5 D4	6ADD",
		"C5 D5	6AE4",
		"C5 D6	6AE7",
		"C5 D7	6AEC",
		"C5 D8	6AF0",
		"C5 D9	6AF1",
		"C5 DA	6AF2",
		"C5 DB	6AFC",
		"C5 DC	6AFD",
		"C5 DD	6B02",
		"C5 DE	6B03",
		"C5 DF	6B06",
		"C5 E0	6B07",
		"C5 E1	6B09",
		"C5 E2	6B0F",
		"C5 E3	6B10",
		"C5 E4	6B11",
		"C5 E5	6B17",
		"C5 E6	6B1B",
		"C5 E7	6B1E",
		"C5 E8	6B24",
		"C5 E9	6B28",
		"C5 EA	6B2B",
		"C5 EB	6B2C",
		"C5 EC	6B2F",
		"C5 ED	6B35",
		"C5 EE	6B36",
		"C5 EF	6B3B",
		"C5 F0	6B3F",
		"C5 F1	6B46",
		"C5 F2	6B4A",
		"C5 F3	6B4D",
		"C5 F4	6B52",
		"C5 F5	6B56",
		"C5 F6	6B58",
		"C5 F7	6B5D",
		"C5 F8	6B60",
		"C5 F9	6B67",
		"C5 FA	6B6B",
		"C5 FB	6B6E",
		"C5 FC	6B70",
		"C5 FD	6B75",
		"C5 FE	6B7D",
		"C6 A1	6B7E",
		"C6 A2	6B82",
		"C6 A3	6B85",
		"C6 A4	6B97",
		"C6 A5	6B9B",
		"C6 A6	6B9F",
		"C6 A7	6BA0",
		"C6 A8	6BA2",
		"C6 A9	6BA3",
		"C6 AA	6BA8",
		"C6 AB	6BA9",
		"C6 AC	6BAC",
		"C6 AD	6BAD",
		"C6 AE	6BAE",
		"C6 AF	6BB0",
		"C6 B0	6BB8",
		"C6 B1	6BB9",
		"C6 B2	6BBD",
		"C6 B3	6BBE",
		"C6 B4	6BC3",
		"C6 B5	6BC4",
		"C6 B6	6BC9",
		"C6 B7	6BCC",
		"C6 B8	6BD6",
		"C6 B9	6BDA",
		"C6 BA	6BE1",
		"C6 BB	6BE3",
		"C6 BC	6BE6",
		"C6 BD	6BE7",
		"C6 BE	6BEE",
		"C6 BF	6BF1",
		"C6 C0	6BF7",
		"C6 C1	6BF9",
		"C6 C2	6BFF",
		"C6 C3	6C02",
		"C6 C4	6C04",
		"C6 C5	6C05",
		"C6 C6	6C09",
		"C6 C7	6C0D",
		"C6 C8	6C0E",
		"C6 C9	6C10",
		"C6 CA	6C12",
		"C6 CB	6C19",
		"C6 CC	6C1F",
		"C6 CD	6C26",
		"C6 CE	6C27",
		"C6 CF	6C28",
		"C6 D0	6C2C",
		"C6 D1	6C2E",
		"C6 D2	6C33",
		"C6 D3	6C35",
		"C6 D4	6C36",
		"C6 D5	6C3A",
		"C6 D6	6C3B",
		"C6 D7	6C3F",
		"C6 D8	6C4A",
		"C6 D9	6C4B",
		"C6 DA	6C4D",
		"C6 DB	6C4F",
		"C6 DC	6C52",
		"C6 DD	6C54",
		"C6 DE	6C59",
		"C6 DF	6C5B",
		"C6 E0	6C5C",
		"C7 B6	6C67",
		"C6 E1	6C6B",
		"C6 E2	6C6D",
		"C6 E3	6C6F",
		"C6 E4	6C74",
		"C6 E5	6C76",
		"C6 E6	6C78",
		"C6 E7	6C79",
		"C6 E8	6C7B",
		"C6 E9	6C85",
		"C6 EA	6C86",
		"C6 EB	6C87",
		"C6 EC	6C89",
		"C6 ED	6C94",
		"C6 EE	6C95",
		"C6 EF	6C97",
		"C6 F0	6C98",
		"C6 F1	6C9C",
		"C6 F2	6C9F",
		"C6 F3	6CB0",
		"C6 F4	6CB2",
		"C6 F5	6CB4",
		"C6 F6	6CC2",
		"C6 F7	6CC6",
		"C6 F8	6CCD",
		"C6 F9	6CCF",
		"C6 FA	6CD0",
		"C6 FB	6CD1",
		"C6 FC	6CD2",
		"C6 FD	6CD4",
		"C6 FE	6CD6",
		"C7 A1	6CDA",
		"C7 A2	6CDC",
		"C7 A3	6CE0",
		"C7 A4	6CE7",
		"C7 A5	6CE9",
		"C7 A6	6CEB",
		"C7 A7	6CEC",
		"C7 A8	6CEE",
		"C7 A9	6CF2",
		"C7 AA	6CF4",
		"C7 AB	6D04",
		"C7 AC	6D07",
		"C7 AD	6D0A",
		"C7 AE	6D0E",
		"C7 AF	6D0F",
		"C7 B0	6D11",
		"C7 B1	6D13",
		"C7 B2	6D1A",
		"C7 B3	6D26",
		"C7 B4	6D27",
		"C7 B5	6D28",
		"C7 B7	6D2E",
		"C7 B8	6D2F",
		"C7 B9	6D31",
		"C7 BA	6D39",
		"C7 BB	6D3C",
		"C7 BC	6D3F",
		"C7 BD	6D57",
		"C7 BE	6D5E",
		"C7 BF	6D5F",
		"C7 C0	6D61",
		"C7 C1	6D65",
		"C7 C2	6D67",
		"C7 C3	6D6F",
		"C7 C4	6D70",
		"C7 C5	6D7C",
		"C7 C6	6D82",
		"C7 C7	6D87",
		"C7 C8	6D91",
		"C7 C9	6D92",
		"C7 CA	6D94",
		"C7 CB	6D96",
		"C7 CC	6D97",
		"C7 CD	6D98",
		"C7 CE	6DAA",
		"C7 CF	6DAC",
		"C7 D0	6DB4",
		"C7 D1	6DB7",
		"C7 D2	6DB9",
		"C7 D3	6DBD",
		"C7 D4	6DBF",
		"C7 D5	6DC4",
		"C7 D6	6DC8",
		"C7 D7	6DCA",
		"C7 D8	6DCE",
		"C7 D9	6DCF",
		"C7 DA	6DD6",
		"C7 DB	6DDB",
		"C7 DC	6DDD",
		"C7 DD	6DDF",
		"C7 DE	6DE0",
		"C7 DF	6DE2",
		"C7 E0	6DE5",
		"C7 E1	6DE9",
		"C7 E2	6DEF",
		"C7 E3	6DF0",
		"C7 E4	6DF4",
		"C7 E5	6DF6",
		"C7 E6	6DFC",
		"C7 E7	6E00",
		"C7 E8	6E04",
		"C7 E9	6E1E",
		"C7 EA	6E22",
		"C7 EB	6E27",
		"C7 EC	6E32",
		"C7 ED	6E36",
		"C7 EE	6E39",
		"C7 EF	6E3B",
		"C7 F0	6E3C",
		"C7 F1	6E44",
		"C7 F2	6E45",
		"C7 F3	6E48",
		"C7 F4	6E49",
		"C7 F5	6E4B",
		"C7 F6	6E4F",
		"C7 F7	6E51",
		"C7 F8	6E52",
		"C7 F9	6E53",
		"C7 FA	6E54",
		"C7 FB	6E57",
		"C7 FC	6E5C",
		"C7 FD	6E5D",
		"C7 FE	6E5E",
		"C8 A1	6E62",
		"C8 A2	6E63",
		"C8 A3	6E68",
		"C8 A4	6E73",
		"C8 A5	6E7B",
		"C8 A6	6E7D",
		"C8 A7	6E8D",
		"C8 A8	6E93",
		"C8 A9	6E99",
		"C8 AA	6EA0",
		"C8 AB	6EA7",
		"C8 AC	6EAD",
		"C8 AD	6EAE",
		"C8 AE	6EB1",
		"C8 AF	6EB3",
		"C8 B0	6EBB",
		"C8 B1	6EBF",
		"C8 B2	6EC0",
		"C8 B3	6EC1",
		"C8 B4	6EC3",
		"C8 B5	6EC7",
		"C8 B6	6EC8",
		"C8 B7	6ECA",
		"C8 B8	6ECD",
		"C8 B9	6ECE",
		"C8 BA	6ECF",
		"C8 BB	6EEB",
		"C8 BC	6EED",
		"C8 BD	6EEE",
		"C8 BE	6EF9",
		"C8 BF	6EFB",
		"C8 C0	6EFD",
		"C8 C1	6F04",
		"C8 C2	6F08",
		"C8 C3	6F0A",
		"C8 C4	6F0C",
		"C8 C5	6F0D",
		"C8 C6	6F16",
		"C8 C7	6F18",
		"C8 C8	6F1A",
		"C8 C9	6F1B",
		"C8 CA	6F26",
		"C8 CB	6F29",
		"C8 CC	6F2A",
		"C8 D3	6F2D",
		"C8 CD	6F2F",
		"C8 CE	6F30",
		"C8 CF	6F33",
		"C8 D0	6F36",
		"C8 D1	6F3B",
		"C8 D2	6F3C",
		"C8 D4	6F4F",
		"C8 D5	6F51",
		"C8 D6	6F52",
		"C8 D7	6F53",
		"C8 D8	6F57",
		"C8 D9	6F59",
		"C8 DA	6F5A",
		"C8 DB	6F5D",
		"C8 DC	6F5E",
		"C8 DD	6F61",
		"C8 DE	6F62",
		"C8 DF	6F68",
		"C8 E0	6F6C",
		"C8 E1	6F7D",
		"C8 E2	6F7E",
		"C8 E3	6F83",
		"C8 E4	6F87",
		"C8 E5	6F88",
		"C8 E6	6F8B",
		"C8 E7	6F8C",
		"C8 E8	6F8D",
		"C8 E9	6F90",
		"C8 EA	6F92",
		"C8 EB	6F93",
		"C8 EC	6F94",
		"C8 ED	6F96",
		"C8 EE	6F9A",
		"C8 EF	6F9F",
		"C8 F0	6FA0",
		"C8 F1	6FA5",
		"C8 F2	6FA6",
		"C8 F3	6FA7",
		"C8 F4	6FA8",
		"C8 F5	6FAE",
		"C8 F6	6FAF",
		"C8 F7	6FB0",
		"C8 F8	6FB5",
		"C8 F9	6FB6",
		"C8 FA	6FBC",
		"C8 FB	6FC5",
		"C8 FC	6FC7",
		"C8 FD	6FC8",
		"C8 FE	6FCA",
		"C9 A1	6FDA",
		"C9 A2	6FDE",
		"C9 A3	6FE8",
		"C9 A4	6FE9",
		"C9 A5	6FF0",
		"C9 A6	6FF5",
		"C9 A7	6FF9",
		"C9 A8	6FFC",
		"C9 A9	6FFD",
		"C9 AA	7000",
		"C9 AB	7005",
		"C9 AC	7006",
		"C9 AD	7007",
		"C9 AE	700D",
		"C9 AF	7017",
		"C9 B0	7020",
		"C9 B1	7023",
		"C9 B2	702F",
		"C9 B3	7034",
		"C9 B4	7037",
		"C9 B5	7039",
		"C9 B6	703C",
		"C9 B7	7043",
		"C9 B8	7044",
		"C9 B9	7048",
		"C9 BA	7049",
		"C9 BB	704A",
		"C9 BC	704B",
		"C9 C1	704E",
		"C9 BD	7054",
		"C9 BE	7055",
		"C9 BF	705D",
		"C9 C0	705E",
		"C9 C2	7064",
		"C9 C3	7065",
		"C9 C4	706C",
		"C9 C5	706E",
		"C9 C6	7075",
		"C9 C7	7076",
		"C9 C8	707E",
		"C9 C9	7081",
		"C9 CA	7085",
		"C9 CB	7086",
		"C9 CC	7094",
		"C9 CD	7095",
		"C9 CE	7096",
		"C9 CF	7097",
		"C9 D0	7098",
		"C9 D1	709B",
		"C9 D2	70A4",
		"C9 D3	70AB",
		"C9 D4	70B0",
		"C9 D5	70B1",
		"C9 D6	70B4",
		"C9 D7	70B7",
		"C9 D8	70CA",
		"C9 D9	70D1",
		"C9 DA	70D3",
		"C9 DB	70D4",
		"C9 DC	70D5",
		"C9 DD	70D6",
		"C9 DE	70D8",
		"C9 DF	70DC",
		"C9 E0	70E4",
		"C9 E1	70FA",
		"C9 E2	7103",
		"C9 E3	7104",
		"C9 E4	7105",
		"C9 E5	7106",
		"C9 E6	7107",
		"C9 E7	710B",
		"C9 E8	710C",
		"C9 E9	710F",
		"C9 EA	711E",
		"C9 EB	7120",
		"C9 EC	712B",
		"C9 ED	712D",
		"C9 EE	712F",
		"C9 EF	7130",
		"C9 F0	7131",
		"C9 F1	7138",
		"C9 F2	7141",
		"C9 F3	7145",
		"C9 F4	7146",
		"C9 F5	7147",
		"C9 F6	714A",
		"C9 F7	714B",
		"C9 F8	7150",
		"C9 F9	7152",
		"C9 FA	7157",
		"C9 FB	715A",
		"C9 FC	715C",
		"C9 FD	715E",
		"C9 FE	7160",
		"CA A1	7168",
		"CA A2	7179",
		"CA A3	7180",
		"CA A4	7185",
		"CA A5	7187",
		"CA A6	718C",
		"CA A7	7192",
		"CA A8	719A",
		"CA A9	719B",
		"CA AA	71A0",
		"CA AB	71A2",
		"CA AC	71AF",
		"CA AD	71B0",
		"CA AE	71B2",
		"CA AF	71B3",
		"CA B0	71BA",
		"CA B1	71BF",
		"CA B2	71C0",
		"CA B3	71C1",
		"CA B4	71C4",
		"CA B5	71CB",
		"CA B6	71CC",
		"CA B7	71D3",
		"CA B8	71D6",
		"CA B9	71D9",
		"CA BA	71DA",
		"CA BB	71DC",
		"CA BC	71F8",
		"CA BD	71FE",
		"CA BE	7200",
		"CA BF	7207",
		"CA C0	7208",
		"CA C1	7209",
		"CA C2	7213",
		"CA C3	7217",
		"CA C4	721A",
		"CA C5	721D",
		"CA C6	721F",
		"CA C7	7224",
		"CA C8	722B",
		"CA C9	722F",
		"CA CA	7234",
		"CA CB	7238",
		"CA CC	7239",
		"CA CD	7241",
		"CA CE	7242",
		"CA CF	7243",
		"CA D0	7245",
		"CA D1	724E",
		"CA D2	724F",
		"CA D3	7250",
		"CA D4	7253",
		"CA D5	7255",
		"CA D6	7256",
		"CA D7	725A",
		"CA D8	725C",
		"CA D9	725E",
		"CA DA	7260",
		"CA DB	7263",
		"CA DC	7268",
		"CA DD	726B",
		"CA DE	726E",
		"CA DF	726F",
		"CA E0	7271",
		"CA E1	7277",
		"CA E2	7278",
		"CA E3	727B",
		"CA E4	727C",
		"CA E5	727F",
		"CA E6	7284",
		"CA E7	7289",
		"CA E8	728D",
		"CA E9	728E",
		"CA EA	7293",
		"CA EB	729B",
		"CA EC	72A8",
		"CA ED	72AD",
		"CA EE	72AE",
		"CA EF	72B1",
		"CA F0	72B4",
		"CA F1	72BE",
		"CA F2	72C1",
		"CA F3	72C7",
		"CA F4	72C9",
		"CA F5	72CC",
		"CA F6	72D5",
		"CA F7	72D6",
		"CA F8	72D8",
		"CA F9	72DF",
		"CA FA	72E5",
		"CA FB	72F3",
		"CA FC	72F4",
		"CA FD	72FA",
		"CA FE	72FB",
		"CB A1	72FE",
		"CB A2	7302",
		"CB A3	7304",
		"CB A4	7305",
		"CB A5	7307",
		"CB A6	730B",
		"CB A7	730D",
		"CB A8	7312",
		"CB A9	7313",
		"CB AA	7318",
		"CB AB	7319",
		"CB AC	731E",
		"CB AD	7322",
		"CB AE	7324",
		"CB AF	7327",
		"CB B0	7328",
		"CB B1	732C",
		"CB B2	7331",
		"CB B3	7332",
		"CB B4	7335",
		"CB B5	733A",
		"CB B6	733B",
		"CB B7	733D",
		"CB B8	7343",
		"CB B9	734D",
		"CB BA	7350",
		"CB BB	7352",
		"CB BC	7356",
		"CB BD	7358",
		"CB BE	735D",
		"CB BF	735E",
		"CB C0	735F",
		"CB C1	7360",
		"CB C2	7366",
		"CB C3	7367",
		"CB C4	7369",
		"CB C5	736B",
		"CB C6	736C",
		"CB C7	736E",
		"CB C8	736F",
		"CB C9	7371",
		"CB CA	7377",
		"CB CB	7379",
		"CB CC	737C",
		"CB CD	7380",
		"CB CE	7381",
		"CB CF	7383",
		"CB D0	7385",
		"CB D1	7386",
		"CB D2	738E",
		"CB D3	7390",
		"CB D4	7393",
		"CB D5	7395",
		"CB D6	7397",
		"CB D7	7398",
		"CB D8	739C",
		"CB D9	739E",
		"CB DA	739F",
		"CB DB	73A0",
		"CB DC	73A2",
		"CB DD	73A5",
		"CB DE	73A6",
		"CB DF	73AA",
		"CB E0	73AB",
		"CB E1	73AD",
		"CB E2	73B5",
		"CB E3	73B7",
		"CB E4	73B9",
		"CB E5	73BC",
		"CB E6	73BD",
		"CB E7	73BF",
		"CB E8	73C5",
		"CB E9	73C6",
		"CB EA	73C9",
		"CB EB	73CB",
		"CB EC	73CC",
		"CB ED	73CF",
		"CB EE	73D2",
		"CB EF	73D3",
		"CB F0	73D6",
		"CB F1	73D9",
		"CB F2	73DD",
		"CB F3	73E1",
		"CB F4	73E3",
		"CB F5	73E6",
		"CB F6	73E7",
		"CB F7	73E9",
		"CB F8	73F4",
		"CB F9	73F5",
		"CB FA	73F7",
		"CB FB	73F9",
		"CB FC	73FA",
		"CB FD	73FB",
		"CB FE	73FD",
		"CC A1	73FF",
		"CC A2	7400",
		"CC A3	7401",
		"CC A4	7404",
		"CC A5	7407",
		"CC A6	740A",
		"CC A7	7411",
		"CC A8	741A",
		"CC A9	741B",
		"CC AA	7424",
		"CC AB	7426",
		"CC AC	7428",
		"CC AD	7429",
		"CC AE	742A",
		"CC AF	742B",
		"CC B0	742C",
		"CC B1	742D",
		"CC B2	742E",
		"CC B3	742F",
		"CC B4	7430",
		"CC B5	7431",
		"CC B6	7439",
		"CC B7	7440",
		"CC B8	7443",
		"CC B9	7444",
		"CC BA	7446",
		"CC BB	7447",
		"CC BC	744B",
		"CC BD	744D",
		"CC BE	7451",
		"CC BF	7452",
		"CC C0	7457",
		"CC C1	745D",
		"CC C2	7462",
		"CC C3	7466",
		"CC C4	7467",
		"CC C5	7468",
		"CC C6	746B",
		"CC C7	746D",
		"CC C8	746E",
		"CC C9	7471",
		"CC CA	7472",
		"CC CB	7480",
		"CC CC	7481",
		"CC CD	7485",
		"CC CE	7486",
		"CC CF	7487",
		"CC D0	7489",
		"CC D1	748F",
		"CC D2	7490",
		"CC D3	7491",
		"CC D4	7492",
		"CC D5	7498",
		"CC D6	7499",
		"CC D7	749A",
		"CC D8	749C",
		"CC D9	749F",
		"CC DA	74A0",
		"CC DB	74A1",
		"CC DC	74A3",
		"CC DD	74A6",
		"CC DE	74A8",
		"CC DF	74A9",
		"CC E0	74AA",
		"CC E1	74AB",
		"CC E2	74AE",
		"CC E3	74AF",
		"CC E4	74B1",
		"CC E5	74B2",
		"CC E6	74B5",
		"CC E7	74B9",
		"CC E8	74BB",
		"CC E9	74BF",
		"CC EA	74C8",
		"CC EB	74C9",
		"CC EC	74CC",
		"CC ED	74D0",
		"CC EE	74D3",
		"CC EF	74D8",
		"CC F0	74DA",
		"CC F1	74DB",
		"CC F2	74DE",
		"CC F3	74DF",
		"CC F4	74E4",
		"CC F5	74E8",
		"CC F6	74EA",
		"CC F7	74EB",
		"CC F8	74EF",
		"CC F9	74F4",
		"CC FA	74FA",
		"CC FB	74FB",
		"CC FC	74FC",
		"CC FD	74FF",
		"CC FE	7506",
		"CD A1	7512",
		"CD A2	7516",
		"CD A3	7517",
		"CD A4	7520",
		"CD A5	7521",
		"CD A6	7524",
		"CD A7	7527",
		"CD A8	7529",
		"CD A9	752A",
		"CD AA	752F",
		"CD AB	7536",
		"CD AC	7539",
		"CD AD	753D",
		"CD AE	753E",
		"CD AF	753F",
		"CD B0	7540",
		"CD B1	7543",
		"CD B2	7547",
		"CD B3	7548",
		"CD B4	754E",
		"CD B5	7550",
		"CD B6	7552",
		"CD B7	7557",
		"CD B8	755E",
		"CD B9	755F",
		"CD BA	7561",
		"CD BB	756F",
		"CD BC	7571",
		"CD BD	7579",
		"CD BE	757A",
		"CD BF	757B",
		"CD C0	757C",
		"CD C1	757D",
		"CD C2	757E",
		"CD C3	7581",
		"CD C4	7585",
		"CD C5	7590",
		"CD C6	7592",
		"CD C7	7593",
		"CD C8	7595",
		"CD C9	7599",
		"CD CA	759C",
		"CD CB	75A2",
		"CD CC	75A4",
		"CD CD	75B4",
		"CD CE	75BA",
		"CD CF	75BF",
		"CD D0	75C0",
		"CD D1	75C1",
		"CD D2	75C4",
		"CD D3	75C6",
		"CD D4	75CC",
		"CD D5	75CE",
		"CD D6	75CF",
		"CD D7	75D7",
		"CD D8	75DC",
		"CD D9	75DF",
		"CD DA	75E0",
		"CD DB	75E1",
		"CD DC	75E4",
		"CD DD	75E7",
		"CD DE	75EC",
		"CD DF	75EE",
		"CD E0	75EF",
		"CD E1	75F1",
		"CD E2	75F9",
		"CD E3	7600",
		"CD E4	7602",
		"CD E5	7603",
		"CD E6	7604",
		"CD E7	7607",
		"CD E8	7608",
		"CD E9	760A",
		"CD EA	760C",
		"CD EB	760F",
		"CD EC	7612",
		"CD ED	7613",
		"CD EE	7615",
		"CD EF	7616",
		"CD F0	7619",
		"CD F1	761B",
		"CD F2	761C",
		"CD F3	761D",
		"CD F4	761E",
		"CD F5	7623",
		"CD F6	7625",
		"CD F7	7626",
		"CD F8	7629",
		"CD F9	762D",
		"CD FA	7632",
		"CD FB	7633",
		"CD FC	7635",
		"CD FD	7638",
		"CD FE	7639",
		"CE A1	763A",
		"CE A2	763C",
		"CE A4	7640",
		"CE A5	7641",
		"CE A6	7643",
		"CE A7	7644",
		"CE A8	7645",
		"CE A9	7649",
		"CE A3	764A",
		"CE AA	764B",
		"CE AB	7655",
		"CE AC	7659",
		"CE AD	765F",
		"CE AE	7664",
		"CE AF	7665",
		"CE B0	766D",
		"CE B1	766E",
		"CE B2	766F",
		"CE B3	7671",
		"CE B4	7674",
		"CE B5	7681",
		"CE B6	7685",
		"CE B7	768C",
		"CE B8	768D",
		"CE B9	7695",
		"CE BA	769B",
		"CE BB	769C",
		"CE BC	769D",
		"CE BD	769F",
		"CE BE	76A0",
		"CE BF	76A2",
		"CE C0	76A3",
		"CE C1	76A4",
		"CE C2	76A5",
		"CE C3	76A6",
		"CE C4	76A7",
		"CE C5	76A8",
		"CE C6	76AA",
		"CE C7	76AD",
		"CE C8	76BD",
		"CE C9	76C1",
		"CE CA	76C5",
		"CE CB	76C9",
		"CE CC	76CB",
		"CE CD	76CC",
		"CE CE	76CE",
		"CE CF	76D4",
		"CE D0	76D9",
		"CE D1	76E0",
		"CE D2	76E6",
		"CE D3	76E8",
		"CE D4	76EC",
		"CE D5	76F0",
		"CE D6	76F1",
		"CE D7	76F6",
		"CE D8	76F9",
		"CE D9	76FC",
		"CE DA	7700",
		"CE DB	7706",
		"CE DC	770A",
		"CE DD	770E",
		"CE DE	7712",
		"CE DF	7714",
		"CE E0	7715",
		"CE E1	7717",
		"CE E2	7719",
		"CE E3	771A",
		"CE E4	771C",
		"CE E5	7722",
		"CE E6	7728",
		"CE E7	772D",
		"CE E8	772E",
		"CE E9	772F",
		"CE EA	7734",
		"CE EB	7735",
		"CE EC	7736",
		"CE ED	7739",
		"CE EE	773D",
		"CE EF	773E",
		"CE F0	7742",
		"CE F1	7745",
		"CE F2	7746",
		"CE F3	774A",
		"CE F4	774D",
		"CE F5	774E",
		"CE F6	774F",
		"CE F7	7752",
		"CE F8	7756",
		"CE F9	7757",
		"CE FA	775C",
		"CE FB	775E",
		"CE FC	775F",
		"CE FD	7760",
		"CE FE	7762",
		"CF A1	7764",
		"CF A2	7767",
		"CF A3	776A",
		"CF A4	776C",
		"CF A5	7770",
		"CF A6	7772",
		"CF A7	7773",
		"CF A8	7774",
		"CF A9	777A",
		"CF AA	777D",
		"CF AB	7780",
		"CF AC	7784",
		"CF AD	778C",
		"CF AE	778D",
		"CF AF	7794",
		"CF B0	7795",
		"CF B1	7796",
		"CF B2	779A",
		"CF B3	779F",
		"CF B4	77A2",
		"CF B5	77A7",
		"CF B6	77AA",
		"CF B7	77AE",
		"CF B8	77AF",
		"CF B9	77B1",
		"CF BA	77B5",
		"CF BB	77BE",
		"CF BC	77C3",
		"CF BD	77C9",
		"CF BE	77D1",
		"CF BF	77D2",
		"CF C0	77D5",
		"CF C1	77D9",
		"CF C2	77DE",
		"CF C3	77DF",
		"CF C4	77E0",
		"CF C5	77E4",
		"CF C6	77E6",
		"CF C7	77EA",
		"CF C8	77EC",
		"CF C9	77F0",
		"CF CA	77F1",
		"CF CB	77F4",
		"CF CC	77F8",
		"CF CD	77FB",
		"CF CE	7805",
		"CF CF	7806",
		"CF D0	7809",
		"CF D1	780D",
		"CF D2	780E",
		"CF D3	7811",
		"CF D4	781D",
		"CF D5	7821",
		"CF D6	7822",
		"CF D7	7823",
		"CF D8	782D",
		"CF D9	782E",
		"CF DA	7830",
		"CF DB	7835",
		"CF DC	7837",
		"CF DD	7843",
		"CF DE	7844",
		"CF DF	7847",
		"CF E0	7848",
		"CF E1	784C",
		"CF E2	784E",
		"CF E3	7852",
		"CF E4	785C",
		"CF E5	785E",
		"CF E6	7860",
		"CF E7	7861",
		"CF E8	7863",
		"CF E9	7864",
		"CF EA	7868",
		"CF EB	786A",
		"CF EC	786E",
		"CF ED	787A",
		"CF EE	787E",
		"CF EF	788A",
		"CF F0	788F",
		"CF F1	7894",
		"CF F2	7898",
		"CF F4	789D",
		"CF F5	789E",
		"CF F6	789F",
		"CF F3	78A1",
		"CF F7	78A4",
		"CF F8	78A8",
		"CF F9	78AC",
		"CF FA	78AD",
		"CF FB	78B0",
		"CF FC	78B1",
		"CF FD	78B2",
		"CF FE	78B3",
		"D0 A1	78BB",
		"D0 A2	78BD",
		"D0 A3	78BF",
		"D0 A4	78C7",
		"D0 A5	78C8",
		"D0 A6	78C9",
		"D0 A7	78CC",
		"D0 A8	78CE",
		"D0 A9	78D2",
		"D0 AA	78D3",
		"D0 AB	78D5",
		"D0 AC	78D6",
		"D0 AE	78DB",
		"D0 AF	78DF",
		"D0 B0	78E0",
		"D0 B1	78E1",
		"D0 AD	78E4",
		"D0 B2	78E6",
		"D0 B3	78EA",
		"D0 B4	78F2",
		"D0 B5	78F3",
		"D0 B7	78F6",
		"D0 B8	78F7",
		"D0 B9	78FA",
		"D0 BA	78FB",
		"D0 BB	78FF",
		"D0 B6	7900",
		"D0 BC	7906",
		"D0 BD	790C",
		"D0 BE	7910",
		"D0 BF	791A",
		"D0 C0	791C",
		"D0 C1	791E",
		"D0 C2	791F",
		"D0 C3	7920",
		"D0 C4	7925",
		"D0 C5	7927",
		"D0 C6	7929",
		"D0 C7	792D",
		"D0 C8	7931",
		"D0 C9	7934",
		"D0 CA	7935",
		"D0 CB	793B",
		"D0 CC	793D",
		"D0 CD	793F",
		"D0 CE	7944",
		"D0 CF	7945",
		"D0 D0	7946",
		"D0 D1	794A",
		"D0 D2	794B",
		"D0 D3	794F",
		"D0 D4	7951",
		"D0 D5	7954",
		"D0 D6	7958",
		"D0 D7	795B",
		"D0 D8	795C",
		"D0 D9	7967",
		"D0 DA	7969",
		"D0 DB	796B",
		"D0 DC	7972",
		"D0 DD	7979",
		"D0 DE	797B",
		"D0 DF	797C",
		"D0 E0	797E",
		"D0 E1	798B",
		"D0 E2	798C",
		"D0 E3	7991",
		"D0 E4	7993",
		"D0 E5	7994",
		"D0 E6	7995",
		"D0 E7	7996",
		"D0 E8	7998",
		"D0 E9	799B",
		"D0 EA	799C",
		"D0 EB	79A1",
		"D0 EC	79A8",
		"D0 ED	79A9",
		"D0 EE	79AB",
		"D0 EF	79AF",
		"D0 F0	79B1",
		"D0 F1	79B4",
		"D0 F2	79B8",
		"D0 F3	79BB",
		"D0 F4	79C2",
		"D0 F5	79C4",
		"D0 F6	79C7",
		"D0 F7	79C8",
		"D0 F8	79CA",
		"D0 F9	79CF",
		"D0 FA	79D4",
		"D0 FB	79D6",
		"D0 FC	79DA",
		"D0 FD	79DD",
		"D0 FE	79DE",
		"D1 A1	79E0",
		"D1 A2	79E2",
		"D1 A3	79E5",
		"D1 A4	79EA",
		"D1 A5	79EB",
		"D1 A6	79ED",
		"D1 A7	79F1",
		"D1 A8	79F8",
		"D1 A9	79FC",
		"D1 AA	7A02",
		"D1 AB	7A03",
		"D1 AC	7A07",
		"D1 AD	7A09",
		"D1 AE	7A0A",
		"D1 AF	7A0C",
		"D1 B0	7A11",
		"D1 B1	7A15",
		"D1 B2	7A1B",
		"D1 B3	7A1E",
		"D1 B4	7A21",
		"D1 B5	7A27",
		"D1 B6	7A2B",
		"D1 B7	7A2D",
		"D1 B8	7A2F",
		"D1 B9	7A30",
		"D1 BA	7A34",
		"D1 BB	7A35",
		"D1 BC	7A38",
		"D1 BD	7A39",
		"D1 BE	7A3A",
		"D1 BF	7A44",
		"D1 C0	7A45",
		"D1 C1	7A47",
		"D1 C2	7A48",
		"D1 C3	7A4C",
		"D1 C4	7A55",
		"D1 C5	7A56",
		"D1 C6	7A59",
		"D1 C7	7A5C",
		"D1 C8	7A5D",
		"D1 C9	7A5F",
		"D1 CA	7A60",
		"D1 CB	7A65",
		"D1 CC	7A67",
		"D1 CD	7A6A",
		"D1 CE	7A6D",
		"D1 CF	7A75",
		"D1 D0	7A78",
		"D1 D1	7A7E",
		"D1 D2	7A80",
		"D1 D3	7A82",
		"D1 D4	7A85",
		"D1 D5	7A86",
		"D1 D6	7A8A",
		"D1 D7	7A8B",
		"D1 D8	7A90",
		"D1 D9	7A91",
		"D1 DA	7A94",
		"D1 DB	7A9E",
		"D1 DC	7AA0",
		"D1 DD	7AA3",
		"D1 DE	7AAC",
		"D1 DF	7AB3",
		"D1 E0	7AB5",
		"D1 E1	7AB9",
		"D1 E2	7ABB",
		"D1 E3	7ABC",
		"D1 E4	7AC6",
		"D1 E5	7AC9",
		"D1 E6	7ACC",
		"D1 E7	7ACE",
		"D1 E8	7AD1",
		"D1 E9	7ADB",
		"D1 EA	7AE8",
		"D1 EB	7AE9",
		"D1 EC	7AEB",
		"D1 ED	7AEC",
		"D1 EE	7AF1",
		"D1 EF	7AF4",
		"D1 F0	7AFB",
		"D1 F1	7AFD",
		"D1 F2	7AFE",
		"D1 F3	7B07",
		"D1 F4	7B14",
		"D1 F5	7B1F",
		"D1 F6	7B23",
		"D1 F7	7B27",
		"D1 F8	7B29",
		"D1 F9	7B2A",
		"D1 FA	7B2B",
		"D1 FB	7B2D",
		"D1 FC	7B2E",
		"D1 FD	7B2F",
		"D1 FE	7B30",
		"D2 A1	7B31",
		"D2 A2	7B34",
		"D2 A3	7B3D",
		"D2 A4	7B3F",
		"D2 A5	7B40",
		"D2 A6	7B41",
		"D2 A7	7B47",
		"D2 A8	7B4E",
		"D2 A9	7B55",
		"D2 AA	7B60",
		"D2 AB	7B64",
		"D2 AC	7B66",
		"D2 AD	7B69",
		"D2 AE	7B6A",
		"D2 AF	7B6D",
		"D2 B0	7B6F",
		"D2 B1	7B72",
		"D2 B2	7B73",
		"D2 B3	7B77",
		"D2 B4	7B84",
		"D2 B5	7B89",
		"D2 B6	7B8E",
		"D2 B7	7B90",
		"D2 B8	7B91",
		"D2 B9	7B96",
		"D2 BA	7B9B",
		"D2 BB	7B9E",
		"D2 BC	7BA0",
		"D2 BD	7BA5",
		"D2 BE	7BAC",
		"D2 BF	7BAF",
		"D2 C0	7BB0",
		"D2 C1	7BB2",
		"D2 C2	7BB5",
		"D2 C3	7BB6",
		"D2 C4	7BBA",
		"D2 C5	7BBB",
		"D2 C6	7BBC",
		"D2 C7	7BBD",
		"D2 C8	7BC2",
		"D2 C9	7BC5",
		"D2 CA	7BC8",
		"D2 CB	7BCA",
		"D2 CC	7BD4",
		"D2 CD	7BD6",
		"D2 CE	7BD7",
		"D2 CF	7BD9",
		"D2 D0	7BDA",
		"D2 D1	7BDB",
		"D2 D2	7BE8",
		"D2 D3	7BEA",
		"D2 D4	7BF2",
		"D2 D5	7BF4",
		"D2 D6	7BF5",
		"D2 D7	7BF8",
		"D2 D8	7BF9",
		"D2 D9	7BFA",
		"D2 DA	7BFC",
		"D2 DB	7BFE",
		"D2 DC	7C01",
		"D2 DD	7C02",
		"D2 DE	7C03",
		"D2 DF	7C04",
		"D2 E0	7C06",
		"D2 E1	7C09",
		"D2 E2	7C0B",
		"D2 E3	7C0C",
		"D2 E4	7C0E",
		"D2 E5	7C0F",
		"D2 E6	7C19",
		"D2 E7	7C1B",
		"D2 E8	7C20",
		"D2 E9	7C25",
		"D2 EA	7C26",
		"D2 EB	7C28",
		"D2 EC	7C2C",
		"D2 ED	7C31",
		"D2 EE	7C33",
		"D2 EF	7C34",
		"D2 F0	7C36",
		"D2 F1	7C39",
		"D2 F2	7C3A",
		"D2 F3	7C46",
		"D2 F4	7C4A",
		"D2 F6	7C51",
		"D2 F7	7C52",
		"D2 F8	7C53",
		"D2 F5	7C55",
		"D2 F9	7C59",
		"D2 FA	7C5A",
		"D2 FB	7C5B",
		"D2 FC	7C5C",
		"D2 FD	7C5D",
		"D2 FE	7C5E",
		"D3 A1	7C61",
		"D3 A2	7C63",
		"D3 A3	7C67",
		"D3 A4	7C69",
		"D3 A5	7C6D",
		"D3 A6	7C6E",
		"D3 A7	7C70",
		"D3 A8	7C72",
		"D3 A9	7C79",
		"D3 AA	7C7C",
		"D3 AB	7C7D",
		"D3 AC	7C86",
		"D3 AD	7C87",
		"D3 AE	7C8F",
		"D3 AF	7C94",
		"D3 B0	7C9E",
		"D3 B1	7CA0",
		"D3 B2	7CA6",
		"D3 B3	7CB0",
		"D3 B4	7CB6",
		"D3 B5	7CB7",
		"D3 B6	7CBA",
		"D3 B7	7CBB",
		"D3 B8	7CBC",
		"D3 B9	7CBF",
		"D3 BA	7CC4",
		"D3 BB	7CC7",
		"D3 BC	7CC8",
		"D3 BD	7CC9",
		"D3 BE	7CCD",
		"D3 BF	7CCF",
		"D3 C0	7CD3",
		"D3 C1	7CD4",
		"D3 C2	7CD5",
		"D3 C3	7CD7",
		"D3 C4	7CD9",
		"D3 C5	7CDA",
		"D3 C6	7CDD",
		"D3 C7	7CE6",
		"D3 C8	7CE9",
		"D3 C9	7CEB",
		"D3 CA	7CF5",
		"D3 CB	7D03",
		"D3 CC	7D07",
		"D3 CD	7D08",
		"D3 CE	7D09",
		"D3 CF	7D0F",
		"D3 D0	7D11",
		"D3 D1	7D12",
		"D3 D2	7D13",
		"D3 D3	7D16",
		"D3 D4	7D1D",
		"D3 D5	7D1E",
		"D3 D6	7D23",
		"D3 D7	7D26",
		"D3 D8	7D2A",
		"D3 D9	7D2D",
		"D3 DA	7D31",
		"D3 DB	7D3C",
		"D3 DC	7D3D",
		"D3 DD	7D3E",
		"D3 DE	7D40",
		"D3 DF	7D41",
		"D3 E0	7D47",
		"D3 E1	7D48",
		"D3 E2	7D4D",
		"D3 E3	7D51",
		"D3 E4	7D53",
		"D3 E5	7D57",
		"D3 E6	7D59",
		"D3 E7	7D5A",
		"D3 E8	7D5C",
		"D3 E9	7D5D",
		"D3 EA	7D65",
		"D3 EB	7D67",
		"D3 EC	7D6A",
		"D3 ED	7D70",
		"D3 EE	7D78",
		"D3 EF	7D7A",
		"D3 F0	7D7B",
		"D3 F1	7D7F",
		"D3 F2	7D81",
		"D3 F3	7D82",
		"D3 F4	7D83",
		"D3 F5	7D85",
		"D3 F6	7D86",
		"D3 F7	7D88",
		"D3 F8	7D8B",
		"D3 F9	7D8C",
		"D3 FA	7D8D",
		"D3 FB	7D91",
		"D3 FC	7D96",
		"D3 FD	7D97",
		"D3 FE	7D9D",
		"D4 A1	7D9E",
		"D4 A2	7DA6",
		"D4 A3	7DA7",
		"D4 A4	7DAA",
		"D4 A5	7DB3",
		"D4 A6	7DB6",
		"D4 A7	7DB7",
		"D4 A8	7DB9",
		"D4 A9	7DC2",
		"D4 AA	7DC3",
		"D4 AB	7DC4",
		"D4 AC	7DC5",
		"D4 AD	7DC6",
		"D4 AE	7DCC",
		"D4 AF	7DCD",
		"D4 B0	7DCE",
		"D4 B1	7DD7",
		"D4 B2	7DD9",
		"D4 B4	7DE2",
		"D4 B5	7DE5",
		"D4 B6	7DE6",
		"D4 B7	7DEA",
		"D4 B8	7DEB",
		"D4 B9	7DED",
		"D4 BA	7DF1",
		"D4 BB	7DF5",
		"D4 BC	7DF6",
		"D4 BD	7DF9",
		"D4 BE	7DFA",
		"D4 B3	7E00",
		"D4 BF	7E08",
		"D4 C0	7E10",
		"D4 C1	7E11",
		"D4 C2	7E15",
		"D4 C3	7E17",
		"D4 C4	7E1C",
		"D4 C5	7E1D",
		"D4 C6	7E20",
		"D4 C7	7E27",
		"D4 C8	7E28",
		"D4 C9	7E2C",
		"D4 CA	7E2D",
		"D4 CB	7E2F",
		"D4 CC	7E33",
		"D4 CD	7E36",
		"D4 CE	7E3F",
		"D4 CF	7E44",
		"D4 D0	7E45",
		"D4 D1	7E47",
		"D4 D2	7E4E",
		"D4 D3	7E50",
		"D4 D4	7E52",
		"D4 D5	7E58",
		"D4 D6	7E5F",
		"D4 D7	7E61",
		"D4 D8	7E62",
		"D4 D9	7E65",
		"D4 DA	7E6B",
		"D4 DB	7E6E",
		"D4 DC	7E6F",
		"D4 DD	7E73",
		"D4 DE	7E78",
		"D4 DF	7E7E",
		"D4 E0	7E81",
		"D4 E1	7E86",
		"D4 E2	7E87",
		"D4 E3	7E8A",
		"D4 E4	7E8D",
		"D4 E5	7E91",
		"D4 E6	7E95",
		"D4 E7	7E98",
		"D4 E8	7E9A",
		"D4 E9	7E9D",
		"D4 EA	7E9E",
		"D4 EC	7F3B",
		"D4 EB	7F3C",
		"D4 ED	7F3D",
		"D4 EE	7F3E",
		"D4 EF	7F3F",
		"D4 F0	7F43",
		"D4 F1	7F44",
		"D4 F2	7F47",
		"D4 F3	7F4F",
		"D4 F4	7F52",
		"D4 F5	7F53",
		"D4 F6	7F5B",
		"D4 F7	7F5C",
		"D4 F8	7F5D",
		"D4 F9	7F61",
		"D4 FA	7F63",
		"D4 FB	7F64",
		"D4 FC	7F65",
		"D4 FD	7F66",
		"D4 FE	7F6D",
		"D5 A1	7F71",
		"D5 A2	7F7D",
		"D5 A3	7F7E",
		"D5 A4	7F7F",
		"D5 A5	7F80",
		"D5 A6	7F8B",
		"D5 A7	7F8D",
		"D5 A8	7F8F",
		"D5 A9	7F90",
		"D5 AA	7F91",
		"D5 AB	7F96",
		"D5 AC	7F97",
		"D5 AD	7F9C",
		"D5 AE	7FA1",
		"D5 AF	7FA2",
		"D5 B0	7FA6",
		"D5 B1	7FAA",
		"D5 B2	7FAD",
		"D5 B3	7FB4",
		"D5 B4	7FBC",
		"D5 B5	7FBF",
		"D5 B6	7FC0",
		"D5 B7	7FC3",
		"D5 B8	7FC8",
		"D5 B9	7FCE",
		"D5 BA	7FCF",
		"D5 BB	7FDB",
		"D5 BC	7FDF",
		"D5 BD	7FE3",
		"D5 BE	7FE5",
		"D5 BF	7FE8",
		"D5 C0	7FEC",
		"D5 C1	7FEE",
		"D5 C2	7FEF",
		"D5 C3	7FF2",
		"D5 C4	7FFA",
		"D5 C5	7FFD",
		"D5 C6	7FFE",
		"D5 C7	7FFF",
		"D5 C8	8007",
		"D5 C9	8008",
		"D5 CA	800A",
		"D5 CB	800D",
		"D5 CC	800E",
		"D5 CD	800F",
		"D5 CE	8011",
		"D5 CF	8013",
		"D5 D0	8014",
		"D5 D1	8016",
		"D5 D2	801D",
		"D5 D3	801E",
		"D5 D4	801F",
		"D5 D5	8020",
		"D5 D6	8024",
		"D5 D7	8026",
		"D5 D8	802C",
		"D5 D9	802E",
		"D5 DA	8030",
		"D5 DB	8034",
		"D5 DC	8035",
		"D5 DD	8037",
		"D5 DE	8039",
		"D5 DF	803A",
		"D5 E0	803C",
		"D5 E1	803E",
		"D5 E2	8040",
		"D5 E3	8044",
		"D5 E4	8060",
		"D5 E5	8064",
		"D5 E6	8066",
		"D5 E7	806D",
		"D5 E8	8071",
		"D5 E9	8075",
		"D5 EA	8081",
		"D5 EB	8088",
		"D5 EC	808E",
		"D5 ED	809C",
		"D5 EE	809E",
		"D5 EF	80A6",
		"D5 F0	80A7",
		"D5 F1	80AB",
		"D5 F2	80B8",
		"D5 F3	80B9",
		"D5 F4	80C8",
		"D5 F5	80CD",
		"D5 F6	80CF",
		"D5 F7	80D2",
		"D5 F8	80D4",
		"D5 F9	80D5",
		"D5 FA	80D7",
		"D5 FB	80D8",
		"D5 FC	80E0",
		"D5 FD	80ED",
		"D5 FE	80EE",
		"D6 A1	80F0",
		"D6 A2	80F2",
		"D6 A3	80F3",
		"D6 A4	80F6",
		"D6 A5	80F9",
		"D6 A6	80FA",
		"D6 A7	80FE",
		"D6 A8	8103",
		"D6 A9	810B",
		"D6 AA	8116",
		"D6 AB	8117",
		"D6 AC	8118",
		"D6 AD	811C",
		"D6 AE	811E",
		"D6 AF	8120",
		"D6 B0	8124",
		"D6 B1	8127",
		"D6 B2	812C",
		"D6 B3	8130",
		"D6 B4	8135",
		"D6 B5	813A",
		"D6 B6	813C",
		"D6 B7	8145",
		"D6 B8	8147",
		"D6 B9	814A",
		"D6 BA	814C",
		"D6 BB	8152",
		"D6 BC	8157",
		"D6 BD	8160",
		"D6 BE	8161",
		"D6 BF	8167",
		"D6 C0	8168",
		"D6 C1	8169",
		"D6 C2	816D",
		"D6 C3	816F",
		"D6 C4	8177",
		"D6 C5	8181",
		"D6 C7	8184",
		"D6 C8	8185",
		"D6 C9	8186",
		"D6 CA	818B",
		"D6 CB	818E",
		"D6 C6	8190",
		"D6 CC	8196",
		"D6 CD	8198",
		"D6 CE	819B",
		"D6 CF	819E",
		"D6 D0	81A2",
		"D6 D1	81AE",
		"D6 D2	81B2",
		"D6 D3	81B4",
		"D6 D4	81BB",
		"D6 D6	81C3",
		"D6 D7	81C5",
		"D6 D8	81CA",
		"D6 D5	81CB",
		"D6 D9	81CE",
		"D6 DA	81CF",
		"D6 DB	81D5",
		"D6 DC	81D7",
		"D6 DD	81DB",
		"D6 DE	81DD",
		"D6 DF	81DE",
		"D6 E0	81E1",
		"D6 E1	81E4",
		"D6 E2	81EB",
		"D6 E3	81EC",
		"D6 E4	81F0",
		"D6 E5	81F1",
		"D6 E6	81F2",
		"D6 E7	81F5",
		"D6 E8	81F6",
		"D6 E9	81F8",
		"D6 EA	81F9",
		"D6 EB	81FD",
		"D6 EC	81FF",
		"D6 ED	8200",
		"D6 EE	8203",
		"D6 EF	820F",
		"D6 F0	8213",
		"D6 F1	8214",
		"D6 F2	8219",
		"D6 F3	821A",
		"D6 F4	821D",
		"D6 F5	8221",
		"D6 F6	8222",
		"D6 F7	8228",
		"D6 F8	8232",
		"D6 F9	8234",
		"D6 FA	823A",
		"D6 FB	8243",
		"D6 FC	8244",
		"D6 FD	8245",
		"D6 FE	8246",
		"D7 A1	824B",
		"D7 A2	824E",
		"D7 A3	824F",
		"D7 A4	8251",
		"D7 A5	8256",
		"D7 A6	825C",
		"D7 A7	8260",
		"D7 A8	8263",
		"D7 A9	8267",
		"D7 AA	826D",
		"D7 AB	8274",
		"D7 AC	827B",
		"D7 AD	827D",
		"D7 AE	827F",
		"D7 AF	8280",
		"D7 B0	8281",
		"D7 B1	8283",
		"D7 B2	8284",
		"D7 B3	8287",
		"D7 B4	8289",
		"D7 B5	828A",
		"D7 B6	828E",
		"D7 B7	8291",
		"D7 B8	8294",
		"D7 B9	8296",
		"D7 BA	8298",
		"D7 BB	829A",
		"D7 BC	829B",
		"D7 BD	82A0",
		"D7 BE	82A1",
		"D7 BF	82A3",
		"D7 C0	82A4",
		"D7 C1	82A7",
		"D7 C2	82A8",
		"D7 C3	82A9",
		"D7 C4	82AA",
		"D7 C5	82AE",
		"D7 C6	82B0",
		"D7 C7	82B2",
		"D7 C8	82B4",
		"D7 C9	82B7",
		"D7 CA	82BA",
		"D7 CB	82BC",
		"D7 CC	82BE",
		"D7 CD	82BF",
		"D7 CE	82C6",
		"D7 CF	82D0",
		"D7 D0	82D5",
		"D7 D1	82DA",
		"D7 D2	82E0",
		"D7 D3	82E2",
		"D7 D4	82E4",
		"D7 D5	82E8",
		"D7 D6	82EA",
		"D7 D7	82ED",
		"D7 D8	82EF",
		"D7 D9	82F6",
		"D7 DA	82F7",
		"D7 DB	82FD",
		"D7 DC	82FE",
		"D7 DD	8300",
		"D7 DE	8301",
		"D7 DF	8307",
		"D7 E0	8308",
		"D7 E1	830A",
		"D7 E2	830B",
		"D7 E4	831B",
		"D7 E5	831D",
		"D7 E6	831E",
		"D7 E7	831F",
		"D7 E8	8321",
		"D7 E9	8322",
		"D7 EA	832C",
		"D7 EB	832D",
		"D7 EC	832E",
		"D7 ED	8330",
		"D7 EE	8333",
		"D7 EF	8337",
		"D7 F0	833A",
		"D7 F1	833C",
		"D7 F2	833D",
		"D7 F3	8342",
		"D7 F4	8343",
		"D7 F5	8344",
		"D7 F6	8347",
		"D7 F7	834D",
		"D7 F8	834E",
		"D7 F9	8351",
		"D8 BE	8353",
		"D7 E3	8354",
		"D7 FA	8355",
		"D7 FB	8356",
		"D7 FC	8357",
		"D7 FD	8370",
		"D7 FE	8378",
		"D8 A1	837D",
		"D8 A2	837F",
		"D8 A3	8380",
		"D8 A4	8382",
		"D8 A5	8384",
		"D8 A6	8386",
		"D8 A7	838D",
		"D8 A8	8392",
		"D8 A9	8394",
		"D8 AA	8395",
		"D8 AB	8398",
		"D8 AC	8399",
		"D8 AD	839B",
		"D8 AE	839C",
		"D8 AF	839D",
		"D8 B0	83A6",
		"D8 B1	83A7",
		"D8 B2	83A9",
		"D8 B3	83AC",
		"D8 CC	83AD",
		"D8 B4	83BE",
		"D8 B5	83BF",
		"D8 B6	83C0",
		"D8 B7	83C7",
		"D8 B8	83C9",
		"D8 B9	83CF",
		"D8 BA	83D0",
		"D8 BB	83D1",
		"D8 BC	83D4",
		"D8 BD	83DD",
		"D8 BF	83E8",
		"D8 C0	83EA",
		"D8 C1	83F6",
		"D8 C2	83F8",
		"D8 C3	83F9",
		"D8 C4	83FC",
		"D8 C5	8401",
		"D8 C6	8406",
		"D8 C7	840A",
		"D8 C8	840F",
		"D8 C9	8411",
		"D8 CA	8415",
		"D8 CB	8419",
		"D8 CD	842F",
		"D8 CE	8439",
		"D8 CF	8445",
		"D8 D0	8447",
		"D8 D1	8448",
		"D8 D2	844A",
		"D8 D3	844D",
		"D8 D4	844F",
		"D8 D5	8451",
		"D8 D6	8452",
		"D8 D7	8456",
		"D8 D8	8458",
		"D8 D9	8459",
		"D8 DA	845A",
		"D8 DB	845C",
		"D8 DC	8460",
		"D8 DD	8464",
		"D8 DE	8465",
		"D8 DF	8467",
		"D8 E0	846A",
		"D8 E1	8470",
		"D8 E2	8473",
		"D8 E3	8474",
		"D8 E4	8476",
		"D8 E5	8478",
		"D8 E6	847C",
		"D8 E7	847D",
		"D8 E8	8481",
		"D8 E9	8485",
		"D8 EA	8492",
		"D8 EB	8493",
		"D8 EC	8495",
		"D8 ED	849E",
		"D8 EE	84A6",
		"D8 EF	84A8",
		"D8 F0	84A9",
		"D8 F1	84AA",
		"D8 F2	84AF",
		"D8 F3	84B1",
		"D8 F4	84B4",
		"D8 F5	84BA",
		"D8 F6	84BD",
		"D8 F7	84BE",
		"D8 F8	84C0",
		"D8 F9	84C2",
		"D8 FA	84C7",
		"D8 FB	84C8",
		"D8 FC	84CC",
		"D8 FD	84CF",
		"D8 FE	84D3",
		"D9 A1	84DC",
		"D9 A2	84E7",
		"D9 A3	84EA",
		"D9 A4	84EF",
		"D9 A5	84F0",
		"D9 A6	84F1",
		"D9 A7	84F2",
		"D9 A8	84F7",
		"D9 AA	84FA",
		"D9 AB	84FB",
		"D9 AC	84FD",
		"D9 AD	8502",
		"D9 AE	8503",
		"D9 AF	8507",
		"D9 B0	850C",
		"D9 B1	850E",
		"D9 B2	8510",
		"D9 B3	851C",
		"D9 B4	851E",
		"D9 B5	8522",
		"D9 B6	8523",
		"D9 B7	8524",
		"D9 B8	8525",
		"D9 B9	8527",
		"D9 BA	852A",
		"D9 BB	852B",
		"D9 BC	852F",
		"D9 A9	8532",
		"D9 BD	8533",
		"D9 BE	8534",
		"D9 BF	8536",
		"D9 C0	853F",
		"D9 C1	8546",
		"D9 C2	854F",
		"D9 C3	8550",
		"D9 C4	8551",
		"D9 C5	8552",
		"D9 C6	8553",
		"D9 C7	8556",
		"D9 C8	8559",
		"D9 C9	855C",
		"D9 CA	855D",
		"D9 CB	855E",
		"D9 CC	855F",
		"D9 CD	8560",
		"D9 CE	8561",
		"D9 CF	8562",
		"D9 D0	8564",
		"D9 D1	856B",
		"D9 D2	856F",
		"D9 D3	8579",
		"D9 D4	857A",
		"D9 D5	857B",
		"D9 D6	857D",
		"D9 D7	857F",
		"D9 D8	8581",
		"D9 D9	8585",
		"D9 DA	8586",
		"D9 DB	8589",
		"D9 DC	858B",
		"D9 DD	858C",
		"D9 DE	858F",
		"D9 DF	8593",
		"D9 E0	8598",
		"D9 E1	859D",
		"D9 E2	859F",
		"D9 E3	85A0",
		"D9 E4	85A2",
		"D9 E5	85A5",
		"D9 E6	85A7",
		"D9 F4	85AD",
		"D9 E7	85B4",
		"D9 E8	85B6",
		"D9 E9	85B7",
		"D9 EA	85B8",
		"D9 EB	85BC",
		"D9 EC	85BD",
		"D9 ED	85BE",
		"D9 EE	85BF",
		"D9 EF	85C2",
		"D9 F0	85C7",
		"D9 F1	85CA",
		"D9 F2	85CB",
		"D9 F3	85CE",
		"D9 F5	85D8",
		"D9 F6	85DA",
		"D9 F7	85DF",
		"D9 F8	85E0",
		"D9 F9	85E6",
		"D9 FA	85E8",
		"D9 FB	85ED",
		"D9 FC	85F3",
		"D9 FD	85F6",
		"D9 FE	85FC",
		"DA A1	85FF",
		"DA A2	8600",
		"DA A3	8604",
		"DA A4	8605",
		"DA A5	860D",
		"DA A6	860E",
		"DA A7	8610",
		"DA A8	8611",
		"DA A9	8612",
		"DA AA	8618",
		"DA AB	8619",
		"DA AC	861B",
		"DA AD	861E",
		"DA AE	8621",
		"DA AF	8627",
		"DA B0	8629",
		"DA B1	8636",
		"DA B2	8638",
		"DA B3	863A",
		"DA B4	863C",
		"DA B5	863D",
		"DA B6	8640",
		"B8 E6	8641",
		"DA B7	8642",
		"DA B8	8646",
		"DA B9	8652",
		"DA BA	8653",
		"DA BB	8656",
		"DA BC	8657",
		"DA BD	8658",
		"DA BE	8659",
		"DA BF	865D",
		"DA C0	8660",
		"DA C1	8661",
		"DA C2	8662",
		"DA C3	8663",
		"DA C4	8664",
		"DA C5	8669",
		"DA C6	866C",
		"DA C7	866F",
		"DA C8	8675",
		"DA C9	8676",
		"DA CA	8677",
		"DA CB	867A",
		"DA ED	8688",
		"DA CC	868D",
		"DA CD	8691",
		"DA CE	8696",
		"DA CF	8698",
		"DA D0	869A",
		"DA D1	869C",
		"DA D2	86A1",
		"DA D3	86A6",
		"DA D4	86A7",
		"DA D5	86A8",
		"DA D6	86AD",
		"DA D7	86B1",
		"DA D8	86B3",
		"DA D9	86B4",
		"DA DA	86B5",
		"DA DB	86B7",
		"DA DC	86B8",
		"DA DD	86B9",
		"DA DE	86BF",
		"DA DF	86C0",
		"DA E0	86C1",
		"DA E1	86C3",
		"DA E2	86C5",
		"DA E3	86D1",
		"DA E4	86D2",
		"DA E5	86D5",
		"DA E6	86D7",
		"DA E7	86DA",
		"DA E8	86DC",
		"DA E9	86E0",
		"DA EA	86E3",
		"DA EB	86E5",
		"DA EC	86E7",
		"DA EE	86FA",
		"DA EF	86FC",
		"DA F0	86FD",
		"DA F1	8704",
		"DA F2	8705",
		"DA F3	8707",
		"DA F4	870B",
		"DA F5	870E",
		"DA F6	870F",
		"DA F7	8710",
		"DA F8	8713",
		"DA F9	8714",
		"DA FA	8719",
		"DA FB	871E",
		"DA FC	871F",
		"DA FD	8721",
		"DA FE	8723",
		"DB A1	8728",
		"DB A2	872E",
		"DB A3	872F",
		"DB A4	8731",
		"DB A5	8732",
		"DB A6	8739",
		"DB A7	873A",
		"DB A8	873C",
		"DB A9	873D",
		"DB AA	873E",
		"DB AB	8740",
		"DB AC	8743",
		"DB AD	8745",
		"DB AE	874D",
		"DB AF	8758",
		"DB B0	875D",
		"DB B1	8761",
		"DB B2	8764",
		"DB B3	8765",
		"DB B4	876F",
		"DB B5	8771",
		"DB B6	8772",
		"DB B7	877B",
		"DB B8	8783",
		"DB B9	8784",
		"DB BA	8785",
		"DB BB	8786",
		"DB BC	8787",
		"DB BD	8788",
		"DB BE	8789",
		"DB BF	878B",
		"DB C0	878C",
		"DB C1	8790",
		"DB C2	8793",
		"DB C3	8795",
		"DB C4	8797",
		"DB C5	8798",
		"DB C6	8799",
		"DB C7	879E",
		"DB C8	87A0",
		"DB C9	87A3",
		"DB CA	87A7",
		"DB CB	87AC",
		"DB CC	87AD",
		"DB CD	87AE",
		"DB CE	87B1",
		"DB CF	87B5",
		"DB D0	87BE",
		"DB D1	87BF",
		"DB D2	87C1",
		"DB D3	87C8",
		"DB D4	87C9",
		"DB D5	87CA",
		"DB D6	87CE",
		"DB D7	87D5",
		"DB D8	87D6",
		"DB D9	87D9",
		"DB DA	87DA",
		"DB DB	87DC",
		"DB DC	87DF",
		"DB DD	87E2",
		"DB DE	87E3",
		"DB DF	87E4",
		"DB E0	87EA",
		"DB E1	87EB",
		"DB E2	87ED",
		"DB E3	87F1",
		"DB E4	87F3",
		"DB E5	87F8",
		"DB E6	87FA",
		"DB E7	87FF",
		"DB E8	8801",
		"DB E9	8803",
		"DB EA	8806",
		"DB EB	8809",
		"DB EC	880A",
		"DB ED	880B",
		"DB EE	8810",
		"DB F0	8812",
		"DB F1	8813",
		"DB F2	8814",
		"DB F3	8818",
		"DB EF	8819",
		"DB F4	881A",
		"DB F5	881B",
		"DB F6	881C",
		"DB F7	881E",
		"DB F8	881F",
		"DB F9	8828",
		"DB FA	882D",
		"DB FB	882E",
		"DB FC	8830",
		"DB FD	8832",
		"DB FE	8835",
		"DC A1	883A",
		"DC A2	883C",
		"DC A3	8841",
		"DC A4	8843",
		"DC A5	8845",
		"DC A6	8848",
		"DC A7	8849",
		"DC A8	884A",
		"DC A9	884B",
		"DC AA	884E",
		"DC AB	8851",
		"DC AC	8855",
		"DC AD	8856",
		"DC AE	8858",
		"DC AF	885A",
		"DC B0	885C",
		"DC B1	885F",
		"DC B2	8860",
		"DC B3	8864",
		"DC B4	8869",
		"DC B5	8871",
		"DC B6	8879",
		"DC B7	887B",
		"DC B8	8880",
		"DC B9	8898",
		"DC BA	889A",
		"DC BB	889B",
		"DC BC	889C",
		"DC BD	889F",
		"DC BE	88A0",
		"DC BF	88A8",
		"DC C0	88AA",
		"DC C1	88BA",
		"DC C2	88BD",
		"DC C3	88BE",
		"DC C4	88C0",
		"DC C5	88CA",
		"DC C6	88CB",
		"DC C7	88CC",
		"DC C8	88CD",
		"DC C9	88CE",
		"DC CA	88D1",
		"DC CB	88D2",
		"DC CC	88D3",
		"DC CD	88DB",
		"DC CE	88DE",
		"DC CF	88E7",
		"DC D0	88EF",
		"DC D1	88F0",
		"DC D2	88F1",
		"DC D3	88F5",
		"DC D4	88F7",
		"DC D5	8901",
		"DC D6	8906",
		"DC D7	890D",
		"DC D8	890E",
		"DC D9	890F",
		"DC DA	8915",
		"DC DB	8916",
		"DC DC	8918",
		"DC DD	8919",
		"DC DE	891A",
		"DC DF	891C",
		"DC E0	8920",
		"DC E1	8926",
		"DC E2	8927",
		"DC E3	8928",
		"DC E4	8930",
		"DC E5	8931",
		"DC E6	8932",
		"DC E7	8935",
		"DC E8	8939",
		"DC E9	893A",
		"DC EA	893E",
		"DC EB	8940",
		"DC EC	8942",
		"DC ED	8945",
		"DC EE	8946",
		"DC EF	8949",
		"DC F0	894F",
		"DC F1	8952",
		"DC F2	8957",
		"DC F3	895A",
		"DC F4	895B",
		"DC F5	895C",
		"DC F6	8961",
		"DC F7	8962",
		"DC F8	8963",
		"DC F9	896B",
		"DC FA	896E",
		"DC FB	8970",
		"DC FC	8973",
		"DC FD	8975",
		"DC FE	897A",
		"DD A1	897B",
		"DD A2	897C",
		"DD A3	897D",
		"DD A4	8989",
		"DD A5	898D",
		"DD A6	8990",
		"DD A7	8994",
		"DD A8	8995",
		"DD A9	899B",
		"DD AA	899C",
		"DD AB	899F",
		"DD AC	89A0",
		"DD AD	89A5",
		"DD AE	89B0",
		"DD AF	89B4",
		"DD B0	89B5",
		"DD B1	89B6",
		"DD B2	89B7",
		"DD B3	89BC",
		"DD B4	89D4",
		"DD B5	89D5",
		"DD B6	89D6",
		"DD B7	89D7",
		"DD B8	89D8",
		"DD B9	89E5",
		"DD BA	89E9",
		"DD BB	89EB",
		"DD BC	89ED",
		"DD BD	89F1",
		"DD BE	89F3",
		"DD BF	89F6",
		"DD C0	89F9",
		"DD C1	89FD",
		"DD C2	89FF",
		"DD C3	8A04",
		"DD C4	8A05",
		"DD C5	8A07",
		"DD C6	8A0F",
		"DD C7	8A11",
		"DD C8	8A12",
		"DD C9	8A14",
		"DD CA	8A15",
		"DD CB	8A1E",
		"DD CC	8A20",
		"DD CD	8A22",
		"DD CE	8A24",
		"DD CF	8A26",
		"DD D0	8A2B",
		"DD D1	8A2C",
		"DD D2	8A2F",
		"DD D3	8A35",
		"DD D4	8A37",
		"DD D5	8A3D",
		"DD D6	8A3E",
		"DD D7	8A40",
		"DD D8	8A43",
		"DD D9	8A45",
		"DD DA	8A47",
		"DD DB	8A49",
		"DD DC	8A4D",
		"DD DD	8A4E",
		"DD DE	8A53",
		"DD DF	8A56",
		"DD E0	8A57",
		"DD E1	8A58",
		"DD E2	8A5C",
		"DD E3	8A5D",
		"DD E4	8A61",
		"DD E5	8A65",
		"DD E6	8A67",
		"DD E7	8A75",
		"DD E8	8A76",
		"DD E9	8A77",
		"DD EA	8A79",
		"DD EB	8A7A",
		"DD EC	8A7B",
		"DD ED	8A7E",
		"DD EE	8A7F",
		"DD EF	8A80",
		"DD F0	8A83",
		"DD F1	8A86",
		"DD F2	8A8B",
		"DD F3	8A8F",
		"DD F4	8A90",
		"DD F5	8A92",
		"DD F6	8A96",
		"DD F7	8A97",
		"DD F8	8A99",
		"DD F9	8A9F",
		"DD FA	8AA7",
		"DD FB	8AA9",
		"DD FC	8AAE",
		"DD FD	8AAF",
		"DD FE	8AB3",
		"DE A1	8AB6",
		"DE A2	8AB7",
		"DE A3	8ABB",
		"DE A4	8ABE",
		"DE A5	8AC3",
		"DE A6	8AC6",
		"DE A7	8AC8",
		"DE A8	8AC9",
		"DE A9	8ACA",
		"DE AA	8AD1",
		"DE AB	8AD3",
		"DE AC	8AD4",
		"DE AD	8AD5",
		"DE AE	8AD7",
		"DE AF	8ADD",
		"DE B0	8ADF",
		"DE B1	8AEC",
		"DE B2	8AF0",
		"DE B3	8AF4",
		"DE B4	8AF5",
		"DE B5	8AF6",
		"DE B6	8AFC",
		"DE B7	8AFF",
		"DE B8	8B05",
		"DE B9	8B06",
		"DE BF	8B0A",
		"DE BA	8B0B",
		"DE BB	8B11",
		"DE BC	8B1C",
		"DE BD	8B1E",
		"DE BE	8B1F",
		"DE C0	8B2D",
		"DE C1	8B30",
		"DE C2	8B37",
		"DE C3	8B3C",
		"DE C4	8B42",
		"DE C5	8B43",
		"DE C6	8B44",
		"DE C7	8B45",
		"DE C8	8B46",
		"DE C9	8B48",
		"DE CE	8B4D",
		"DE CA	8B52",
		"DE CB	8B53",
		"DE CC	8B54",
		"DE CD	8B59",
		"DE CF	8B5E",
		"DE D0	8B63",
		"DE D1	8B6D",
		"DE D2	8B76",
		"DE D3	8B78",
		"DE D4	8B79",
		"DE D5	8B7C",
		"DE D6	8B7E",
		"DE D7	8B81",
		"DE D8	8B84",
		"DE D9	8B85",
		"DE DA	8B8B",
		"DE DB	8B8D",
		"DE DC	8B8F",
		"DE DD	8B94",
		"DE DE	8B95",
		"DE DF	8B9C",
		"DE E0	8B9E",
		"DE E1	8B9F",
		"DE E2	8C38",
		"DE E3	8C39",
		"DE E4	8C3D",
		"DE E5	8C3E",
		"DE E6	8C45",
		"DE E7	8C47",
		"DE E8	8C49",
		"DE E9	8C4B",
		"DE EA	8C4F",
		"DE EB	8C51",
		"DE EC	8C53",
		"DE ED	8C54",
		"DE EE	8C57",
		"DE EF	8C58",
		"DE F2	8C59",
		"DE F0	8C5B",
		"DE F1	8C5D",
		"DE F3	8C63",
		"DE F4	8C64",
		"DE F5	8C66",
		"DE F6	8C68",
		"DE F7	8C69",
		"DE F8	8C6D",
		"DE F9	8C73",
		"DE FA	8C75",
		"DE FB	8C76",
		"DE FC	8C7B",
		"DE FD	8C7E",
		"DE FE	8C86",
		"DF A1	8C87",
		"DF A2	8C8B",
		"DF A3	8C90",
		"DF A4	8C92",
		"DF A5	8C93",
		"DF A6	8C99",
		"DF A7	8C9B",
		"DF A8	8C9C",
		"DF A9	8CA4",
		"DF AA	8CB9",
		"DF AB	8CBA",
		"DF AC	8CC5",
		"DF AD	8CC6",
		"DF AE	8CC9",
		"DF AF	8CCB",
		"DF B0	8CCF",
		"DF B2	8CD5",
		"DF B1	8CD6",
		"DF B3	8CD9",
		"DF B4	8CDD",
		"DF B5	8CE1",
		"DF B6	8CE8",
		"DF B7	8CEC",
		"DF B8	8CEF",
		"DF B9	8CF0",
		"DF BA	8CF2",
		"DF BB	8CF5",
		"DF BC	8CF7",
		"DF BD	8CF8",
		"DF BE	8CFE",
		"DF BF	8CFF",
		"DF C0	8D01",
		"DF C1	8D03",
		"DF C2	8D09",
		"DF C3	8D12",
		"DF C4	8D17",
		"DF C5	8D1B",
		"DF C6	8D65",
		"DF C7	8D69",
		"DF C8	8D6C",
		"DF C9	8D6E",
		"DF CA	8D7F",
		"DF CB	8D82",
		"DF CC	8D84",
		"DF CD	8D88",
		"DF CE	8D8D",
		"DF CF	8D90",
		"DF D0	8D91",
		"DF D1	8D95",
		"DF D2	8D9E",
		"DF D3	8D9F",
		"DF D4	8DA0",
		"DF D5	8DA6",
		"DF D6	8DAB",
		"DF D7	8DAC",
		"DF D8	8DAF",
		"DF D9	8DB2",
		"DF DA	8DB5",
		"DF DB	8DB7",
		"DF DC	8DB9",
		"DF DD	8DBB",
		"DF EF	8DBC",
		"DF DE	8DC0",
		"DF DF	8DC5",
		"DF E0	8DC6",
		"DF E1	8DC7",
		"DF E2	8DC8",
		"DF E3	8DCA",
		"DF E4	8DCE",
		"DF E5	8DD1",
		"DF E6	8DD4",
		"DF E7	8DD5",
		"DF E8	8DD7",
		"DF E9	8DD9",
		"DF EA	8DE4",
		"DF EB	8DE5",
		"DF EC	8DE7",
		"DF ED	8DEC",
		"DF EE	8DF0",
		"DF F0	8DF1",
		"DF F1	8DF2",
		"DF F2	8DF4",
		"DF F3	8DFD",
		"DF F4	8E01",
		"DF F5	8E04",
		"DF F6	8E05",
		"DF F7	8E06",
		"DF F8	8E0B",
		"DF F9	8E11",
		"DF FA	8E14",
		"DF FB	8E16",
		"DF FC	8E20",
		"DF FD	8E21",
		"DF FE	8E22",
		"E0 A1	8E23",
		"E0 A2	8E26",
		"E0 A3	8E27",
		"E0 A4	8E31",
		"E0 A5	8E33",
		"E0 A6	8E36",
		"E0 A7	8E37",
		"E0 A8	8E38",
		"E0 A9	8E39",
		"E0 AA	8E3D",
		"E0 AB	8E40",
		"E0 AC	8E41",
		"E0 AD	8E4B",
		"E0 AE	8E4D",
		"E0 AF	8E4E",
		"E0 B0	8E4F",
		"E0 B1	8E54",
		"E0 B2	8E5B",
		"E0 B3	8E5C",
		"E0 B4	8E5D",
		"E0 B5	8E5E",
		"E0 B6	8E61",
		"E0 B7	8E62",
		"E0 B8	8E69",
		"E0 B9	8E6C",
		"E0 BA	8E6D",
		"E0 BB	8E6F",
		"E0 BC	8E70",
		"E0 BD	8E71",
		"E0 BE	8E79",
		"E0 BF	8E7A",
		"E0 C0	8E7B",
		"E0 C1	8E82",
		"E0 C2	8E83",
		"E0 C3	8E89",
		"E0 C4	8E90",
		"E0 C5	8E92",
		"E0 C6	8E95",
		"E0 C7	8E9A",
		"E0 C8	8E9B",
		"E0 C9	8E9D",
		"E0 CA	8E9E",
		"E0 CB	8EA2",
		"E0 CC	8EA7",
		"E0 CD	8EA9",
		"E0 CE	8EAD",
		"E0 CF	8EAE",
		"E0 D0	8EB3",
		"E0 D1	8EB5",
		"E0 D2	8EBA",
		"E0 D3	8EBB",
		"E0 D4	8EC0",
		"E0 D5	8EC1",
		"E0 D6	8EC3",
		"E0 D7	8EC4",
		"E0 D8	8EC7",
		"E0 D9	8ECF",
		"E0 DA	8ED1",
		"E0 DB	8ED4",
		"E0 DC	8EDC",
		"E0 DD	8EE8",
		"E0 E4	8EED",
		"E0 DE	8EEE",
		"E0 DF	8EF0",
		"E0 E0	8EF1",
		"E0 E1	8EF7",
		"E0 E2	8EF9",
		"E0 E3	8EFA",
		"E0 E5	8F00",
		"E0 E6	8F02",
		"E0 E7	8F07",
		"E0 E8	8F08",
		"E0 E9	8F0F",
		"E0 EA	8F10",
		"E0 EB	8F16",
		"E0 EC	8F17",
		"E0 ED	8F18",
		"E0 EE	8F1E",
		"E0 EF	8F20",
		"E0 F0	8F21",
		"E0 F1	8F23",
		"E0 F2	8F25",
		"E0 F3	8F27",
		"E0 F4	8F28",
		"E0 F5	8F2C",
		"E0 F6	8F2D",
		"E0 F7	8F2E",
		"E0 F8	8F34",
		"E0 F9	8F35",
		"E0 FA	8F36",
		"E0 FB	8F37",
		"E0 FC	8F3A",
		"E0 FD	8F40",
		"E0 FE	8F41",
		"E1 A1	8F43",
		"E1 A2	8F47",
		"E1 A3	8F4F",
		"E1 A4	8F51",
		"E1 A5	8F52",
		"E1 A6	8F53",
		"E1 A7	8F54",
		"E1 A8	8F55",
		"E1 A9	8F58",
		"E1 AA	8F5D",
		"E1 AB	8F5E",
		"E1 AC	8F65",
		"E1 AD	8F9D",
		"E1 AE	8FA0",
		"E1 AF	8FA1",
		"E1 B0	8FA4",
		"E1 B1	8FA5",
		"E1 B2	8FA6",
		"E1 B3	8FB5",
		"E1 B4	8FB6",
		"E1 B5	8FB8",
		"E1 B6	8FBE",
		"E1 B7	8FC0",
		"E1 B8	8FC1",
		"E1 B9	8FC6",
		"E1 BA	8FCA",
		"E1 BB	8FCB",
		"E1 BC	8FCD",
		"E1 BD	8FD0",
		"E1 BE	8FD2",
		"E1 BF	8FD3",
		"E1 C0	8FD5",
		"E1 C1	8FE0",
		"E1 C2	8FE3",
		"E1 C3	8FE4",
		"E1 C4	8FE8",
		"E1 C5	8FEE",
		"E1 C6	8FF1",
		"E1 C7	8FF5",
		"E1 C8	8FF6",
		"E1 C9	8FFB",
		"E1 CA	8FFE",
		"E1 CB	9002",
		"E1 CC	9004",
		"E1 CD	9008",
		"E1 CE	900C",
		"E1 CF	9018",
		"E1 D0	901B",
		"E1 D1	9028",
		"E1 D2	9029",
		"E1 D4	902A",
		"E1 D5	902C",
		"E1 D6	902D",
		"E1 D3	902F",
		"E1 D7	9033",
		"E1 D8	9034",
		"E1 D9	9037",
		"E1 DA	903F",
		"E1 DB	9043",
		"E1 DC	9044",
		"E1 DD	904C",
		"E1 DE	905B",
		"E1 DF	905D",
		"E1 E0	9062",
		"E1 E1	9066",
		"E1 E2	9067",
		"E1 E3	906C",
		"E1 E4	9070",
		"E1 E5	9074",
		"E1 E6	9079",
		"E1 E7	9085",
		"E1 E8	9088",
		"E1 E9	908B",
		"E1 EA	908C",
		"E1 EB	908E",
		"E1 EC	9090",
		"E1 ED	9095",
		"E1 EE	9097",
		"E1 EF	9098",
		"E1 F0	9099",
		"E1 F1	909B",
		"E1 F2	90A0",
		"E1 F3	90A1",
		"E1 F4	90A2",
		"E1 F5	90A5",
		"E1 F6	90B0",
		"E1 F7	90B2",
		"E1 F8	90B3",
		"E1 F9	90B4",
		"E1 FA	90B6",
		"E1 FB	90BD",
		"E1 FD	90BE",
		"E1 FE	90C3",
		"E2 A1	90C4",
		"E2 A2	90C5",
		"E2 A3	90C7",
		"E2 A4	90C8",
		"E1 FC	90CC",
		"E2 AD	90D2",
		"E2 A5	90D5",
		"E2 A6	90D7",
		"E2 A7	90D8",
		"E2 A8	90D9",
		"E2 A9	90DC",
		"E2 AA	90DD",
		"E2 AB	90DF",
		"E2 AC	90E5",
		"E2 AF	90EB",
		"E2 B0	90EF",
		"E2 B1	90F0",
		"E2 B2	90F4",
		"E2 AE	90F6",
		"E2 B3	90FE",
		"E2 B4	90FF",
		"E2 B5	9100",
		"E2 B6	9104",
		"E2 B7	9105",
		"E2 B8	9106",
		"E2 B9	9108",
		"E2 BA	910D",
		"E2 BB	9110",
		"E2 BC	9114",
		"E2 BD	9116",
		"E2 BE	9117",
		"E2 BF	9118",
		"E2 C0	911A",
		"E2 C1	911C",
		"E2 C2	911E",
		"E2 C3	9120",
		"E2 C5	9122",
		"E2 C6	9123",
		"E2 C4	9125",
		"E2 C7	9127",
		"E2 C8	9129",
		"E2 C9	912E",
		"E2 CA	912F",
		"E2 CB	9131",
		"E2 CC	9134",
		"E2 CD	9136",
		"E2 CE	9137",
		"E2 CF	9139",
		"E2 D0	913A",
		"E2 D1	913C",
		"E2 D2	913D",
		"E2 D3	9143",
		"E2 D4	9147",
		"E2 D5	9148",
		"E2 D6	914F",
		"E2 D7	9153",
		"E2 D8	9157",
		"E2 D9	9159",
		"E2 DA	915A",
		"E2 DB	915B",
		"E2 DC	9161",
		"E2 DD	9164",
		"E2 DE	9167",
		"E2 DF	916D",
		"E2 E0	9174",
		"E2 E1	9179",
		"E2 E2	917A",
		"E2 E3	917B",
		"E2 E4	9181",
		"E2 E5	9183",
		"E2 E6	9185",
		"E2 E7	9186",
		"E2 E8	918A",
		"E2 E9	918E",
		"E2 EA	9191",
		"E2 EB	9193",
		"E2 EC	9194",
		"E2 ED	9195",
		"E2 EE	9198",
		"E2 EF	919E",
		"E2 F0	91A1",
		"E2 F1	91A6",
		"E2 F2	91A8",
		"E2 F3	91AC",
		"E2 F4	91AD",
		"E2 F5	91AE",
		"E2 F6	91B0",
		"E2 F7	91B1",
		"E2 F8	91B2",
		"E2 F9	91B3",
		"E2 FA	91B6",
		"E2 FB	91BB",
		"E2 FC	91BC",
		"E2 FD	91BD",
		"E2 FE	91BF",
		"E3 A1	91C2",
		"E3 A2	91C3",
		"E3 A3	91C5",
		"E3 A4	91D3",
		"E3 A5	91D4",
		"E3 A6	91D7",
		"E3 A7	91D9",
		"E3 A8	91DA",
		"E3 A9	91DE",
		"E3 AA	91E4",
		"E3 AB	91E5",
		"E3 AC	91E9",
		"E3 AD	91EA",
		"E3 AE	91EC",
		"E3 AF	91ED",
		"E3 B0	91EE",
		"E3 B1	91EF",
		"E3 B2	91F0",
		"E3 B3	91F1",
		"E3 B4	91F7",
		"E3 B5	91F9",
		"E3 B6	91FB",
		"E3 B7	91FD",
		"E3 B8	9200",
		"E3 B9	9201",
		"E3 BA	9204",
		"E3 BB	9205",
		"E3 BC	9206",
		"E3 BD	9207",
		"E3 BE	9209",
		"E3 BF	920A",
		"E3 C0	920C",
		"E3 C1	9210",
		"E3 C2	9212",
		"E3 C3	9213",
		"E3 C4	9216",
		"E3 C5	9218",
		"E3 C6	921C",
		"E3 C7	921D",
		"E3 C8	9223",
		"E3 C9	9224",
		"E3 CA	9225",
		"E3 CB	9226",
		"E3 CC	9228",
		"E3 CD	922E",
		"E3 CE	922F",
		"E3 CF	9230",
		"E3 D0	9233",
		"E3 D1	9235",
		"E3 D2	9236",
		"E3 D3	9238",
		"E3 D4	9239",
		"E3 D5	923A",
		"E3 D6	923C",
		"E3 D7	923E",
		"E3 D8	9240",
		"E3 D9	9242",
		"E3 DA	9243",
		"E3 DB	9246",
		"E3 DC	9247",
		"E3 DD	924A",
		"E3 DE	924D",
		"E3 DF	924E",
		"E3 E0	924F",
		"E3 E1	9251",
		"E3 E2	9258",
		"E3 E3	9259",
		"E3 E4	925C",
		"E3 E5	925D",
		"E3 E6	9260",
		"E3 E7	9261",
		"E3 E8	9265",
		"E3 E9	9267",
		"E3 EA	9268",
		"E3 EB	9269",
		"E3 EC	926E",
		"E3 ED	926F",
		"E3 EE	9270",
		"E3 EF	9275",
		"E3 F0	9276",
		"E3 F1	9277",
		"E3 F2	9278",
		"E3 F3	9279",
		"E3 F4	927B",
		"E3 F5	927C",
		"E3 F6	927D",
		"E3 F7	927F",
		"E3 F8	9288",
		"E3 F9	9289",
		"E3 FA	928A",
		"E3 FB	928D",
		"E3 FC	928E",
		"E3 FD	9292",
		"E3 FE	9297",
		"E4 A1	9299",
		"E4 A2	929F",
		"E4 A3	92A0",
		"E4 A4	92A4",
		"E4 A5	92A5",
		"E4 A6	92A7",
		"E4 A7	92A8",
		"E4 A8	92AB",
		"E4 A9	92AF",
		"E4 AA	92B2",
		"E4 AB	92B6",
		"E4 AC	92B8",
		"E4 AD	92BA",
		"E4 AE	92BB",
		"E4 AF	92BC",
		"E4 B0	92BD",
		"E4 B1	92BF",
		"E4 B2	92C0",
		"E4 B3	92C1",
		"E4 B4	92C2",
		"E4 B5	92C3",
		"E4 B6	92C5",
		"E4 B7	92C6",
		"E4 B8	92C7",
		"E4 B9	92C8",
		"E4 BA	92CB",
		"E4 BB	92CC",
		"E4 BC	92CD",
		"E4 BD	92CE",
		"E4 BE	92D0",
		"E4 BF	92D3",
		"E4 C0	92D5",
		"E4 C1	92D7",
		"E4 C2	92D8",
		"E4 C3	92D9",
		"E4 C4	92DC",
		"E4 C5	92DD",
		"E4 C6	92DF",
		"E4 C7	92E0",
		"E4 C8	92E1",
		"E4 C9	92E3",
		"E4 CA	92E5",
		"E4 CB	92E7",
		"E4 CC	92E8",
		"E4 CD	92EC",
		"E4 CE	92EE",
		"E4 CF	92F0",
		"E4 D0	92F9",
		"E4 D1	92FB",
		"E4 D2	92FF",
		"E4 D3	9300",
		"E4 D4	9302",
		"E4 D5	9308",
		"E4 D6	930D",
		"E4 D7	9311",
		"E4 D8	9314",
		"E4 D9	9315",
		"E4 DA	931C",
		"E4 DB	931D",
		"E4 DC	931E",
		"E4 DD	931F",
		"E4 DE	9321",
		"E4 DF	9324",
		"E4 E0	9325",
		"E4 E1	9327",
		"E4 E2	9329",
		"E4 E3	932A",
		"E4 E4	9333",
		"E4 E5	9334",
		"E4 E6	9336",
		"E4 E7	9337",
		"E4 E8	9347",
		"E4 E9	9348",
		"E4 EA	9349",
		"E4 EB	9350",
		"E4 EC	9351",
		"E4 ED	9352",
		"E4 EE	9355",
		"E4 EF	9357",
		"E4 F0	9358",
		"E4 F1	935A",
		"E4 F2	935E",
		"E4 F3	9364",
		"E4 F4	9365",
		"E4 F5	9367",
		"E4 F6	9369",
		"E4 F7	936A",
		"E4 F8	936D",
		"E4 F9	936F",
		"E4 FA	9370",
		"E4 FB	9371",
		"E4 FC	9373",
		"E4 FD	9374",
		"E4 FE	9376",
		"E5 A1	937A",
		"E5 A2	937D",
		"E5 A3	937F",
		"E5 A4	9380",
		"E5 A5	9381",
		"E5 A6	9382",
		"E5 A7	9388",
		"E5 A8	938A",
		"E5 A9	938B",
		"E5 AA	938D",
		"E5 AB	938F",
		"E5 AC	9392",
		"E5 AD	9395",
		"E5 AE	9398",
		"E5 AF	939B",
		"E5 B0	939E",
		"E5 B1	93A1",
		"E5 B2	93A3",
		"E5 B3	93A4",
		"E5 B4	93A6",
		"E5 B5	93A8",
		"E5 BB	93A9",
		"E5 B6	93AB",
		"E5 B7	93B4",
		"E5 B8	93B5",
		"E5 B9	93B6",
		"E5 BA	93BA",
		"E5 BC	93C1",
		"E5 BD	93C4",
		"E5 BE	93C5",
		"E5 BF	93C6",
		"E5 C0	93C7",
		"E5 C1	93C9",
		"E5 C2	93CA",
		"E5 C3	93CB",
		"E5 C4	93CC",
		"E5 C5	93CD",
		"E5 C6	93D3",
		"E5 C7	93D9",
		"E5 C8	93DC",
		"E5 C9	93DE",
		"E5 CA	93DF",
		"E5 CB	93E2",
		"E5 CC	93E6",
		"E5 CD	93E7",
		"E5 CF	93F7",
		"E5 D0	93F8",
		"E5 CE	93F9",
		"E5 D1	93FA",
		"E5 D2	93FB",
		"E5 D3	93FD",
		"E5 D4	9401",
		"E5 D5	9402",
		"E5 D6	9404",
		"E5 D7	9408",
		"E5 D8	9409",
		"E5 D9	940D",
		"E5 DA	940E",
		"E5 DB	940F",
		"E5 DC	9415",
		"E5 DD	9416",
		"E5 DE	9417",
		"E5 DF	941F",
		"E5 E0	942E",
		"E5 E1	942F",
		"E5 E2	9431",
		"E5 E3	9432",
		"E5 E4	9433",
		"E5 E5	9434",
		"E5 E6	943B",
		"E5 E8	943D",
		"E5 E7	943F",
		"E5 E9	9443",
		"E5 EA	9445",
		"E5 EB	9448",
		"E5 EC	944A",
		"E5 ED	944C",
		"E5 EE	9455",
		"E5 EF	9459",
		"E5 F0	945C",
		"E5 F1	945F",
		"E5 F2	9461",
		"E5 F3	9463",
		"E5 F4	9468",
		"E5 F5	946B",
		"E5 F6	946D",
		"E5 F7	946E",
		"E5 F8	946F",
		"E5 F9	9471",
		"E5 FA	9472",
		"E5 FC	9483",
		"E5 FB	9484",
		"E5 FD	9578",
		"E5 FE	9579",
		"E6 A1	957E",
		"E6 A2	9584",
		"E6 A3	9588",
		"E6 A4	958C",
		"E6 A5	958D",
		"E6 A6	958E",
		"E6 A7	959D",
		"E6 A8	959E",
		"E6 A9	959F",
		"E6 AA	95A1",
		"E6 AB	95A6",
		"E6 AC	95A9",
		"E6 AD	95AB",
		"E6 AE	95AC",
		"E6 AF	95B4",
		"E6 B0	95B6",
		"E6 B1	95BA",
		"E6 B2	95BD",
		"E6 B3	95BF",
		"E6 B4	95C6",
		"E6 B5	95C8",
		"E6 B6	95C9",
		"E6 B7	95CB",
		"E6 B8	95D0",
		"E6 B9	95D1",
		"E6 BA	95D2",
		"E6 BB	95D3",
		"E6 BC	95D9",
		"E6 BD	95DA",
		"E6 BE	95DD",
		"E6 BF	95DE",
		"E6 C0	95DF",
		"E6 C1	95E0",
		"E6 C2	95E4",
		"E6 C3	95E6",
		"E6 C4	961D",
		"E6 C5	961E",
		"E6 C6	9622",
		"E6 C7	9624",
		"E6 C8	9625",
		"E6 C9	9626",
		"E6 CA	962C",
		"E6 CB	9631",
		"E6 CC	9633",
		"E6 CD	9637",
		"E6 CE	9638",
		"E6 CF	9639",
		"E6 D0	963A",
		"E6 D1	963C",
		"E6 D2	963D",
		"E6 D3	9641",
		"E6 D4	9652",
		"E6 D5	9654",
		"E6 D6	9656",
		"E6 D7	9657",
		"E6 D8	9658",
		"E6 D9	9661",
		"E6 DA	966E",
		"E6 DB	9674",
		"E6 DC	967B",
		"E6 DD	967C",
		"E6 DE	967E",
		"E6 DF	967F",
		"E6 E0	9681",
		"E6 E1	9682",
		"E6 E2	9683",
		"E6 E3	9684",
		"E6 E4	9689",
		"E6 E5	9691",
		"E6 E6	9696",
		"E6 E7	969A",
		"E6 E8	969D",
		"E6 E9	969F",
		"E6 EA	96A4",
		"E6 EB	96A5",
		"E6 EC	96A6",
		"E6 ED	96A9",
		"E6 EE	96AE",
		"E6 EF	96AF",
		"E6 F0	96B3",
		"E6 F1	96BA",
		"E6 F2	96CA",
		"E6 F3	96D2",
		"E6 F5	96D8",
		"E6 F6	96DA",
		"E6 F7	96DD",
		"E6 F8	96DE",
		"E6 F9	96DF",
		"E6 FA	96E9",
		"E6 FB	96EF",
		"E6 FC	96F1",
		"E6 FD	96FA",
		"E6 FE	9702",
		"E7 A1	9703",
		"E7 A2	9705",
		"E7 A3	9709",
		"E7 A4	971A",
		"E7 A5	971B",
		"E7 A6	971D",
		"E7 A7	9721",
		"E7 A8	9722",
		"E7 A9	9723",
		"E7 AA	9728",
		"E7 AB	9731",
		"E7 AC	9733",
		"E7 AD	9741",
		"E7 AE	9743",
		"E7 AF	974A",
		"E7 B0	974E",
		"E7 B1	974F",
		"E7 B2	9755",
		"E7 B3	9757",
		"E7 B4	9758",
		"E7 B5	975A",
		"E7 B6	975B",
		"E7 B7	9763",
		"E7 B8	9767",
		"E7 B9	976A",
		"E7 BA	976E",
		"E7 BB	9773",
		"E7 BC	9776",
		"E7 BD	9777",
		"E7 BE	9778",
		"E7 BF	977B",
		"E7 C0	977D",
		"E7 C1	977F",
		"E7 C2	9780",
		"E7 C3	9789",
		"E7 C4	9795",
		"E7 C5	9796",
		"E7 C6	9797",
		"E7 C7	9799",
		"E7 C8	979A",
		"E7 C9	979E",
		"E7 CA	979F",
		"E7 CB	97A2",
		"E7 CC	97AC",
		"E7 CD	97AE",
		"E7 CE	97B1",
		"E7 CF	97B2",
		"E7 D0	97B5",
		"E7 D1	97B6",
		"E7 D2	97B8",
		"E7 D3	97B9",
		"E7 D4	97BA",
		"E7 D5	97BC",
		"E7 D6	97BE",
		"E7 D7	97BF",
		"E7 D8	97C1",
		"E7 D9	97C4",
		"E7 DA	97C5",
		"E7 DB	97C7",
		"E7 DC	97C9",
		"E7 DD	97CA",
		"E7 DE	97CC",
		"E7 DF	97CD",
		"E7 E0	97CE",
		"E7 E1	97D0",
		"E7 E2	97D1",
		"E7 E3	97D4",
		"E7 E4	97D7",
		"E7 E5	97D8",
		"E7 E6	97D9",
		"E7 EA	97DB",
		"E7 E7	97DD",
		"E7 E8	97DE",
		"E7 E9	97E0",
		"E7 EB	97E1",
		"E7 EC	97E4",
		"E7 ED	97EF",
		"E7 EE	97F1",
		"E7 EF	97F4",
		"E7 F0	97F7",
		"E7 F1	97F8",
		"E7 F2	97FA",
		"E7 F3	9807",
		"E7 F4	980A",
		"E7 F6	980D",
		"E7 F7	980E",
		"E7 F8	9814",
		"E7 F9	9816",
		"E7 F5	9819",
		"E7 FA	981C",
		"E7 FB	981E",
		"E7 FC	9820",
		"E7 FD	9823",
		"E8 A8	9825",
		"E7 FE	9826",
		"E8 A1	982B",
		"E8 A2	982E",
		"E8 A3	982F",
		"E8 A4	9830",
		"E8 A5	9832",
		"E8 A6	9833",
		"E8 A7	9835",
		"E8 A9	983E",
		"E8 AA	9844",
		"E8 AB	9847",
		"E8 AC	984A",
		"E8 AD	9851",
		"E8 AE	9852",
		"E8 AF	9853",
		"E8 B0	9856",
		"E8 B1	9857",
		"E8 B2	9859",
		"E8 B3	985A",
		"E8 B4	9862",
		"E8 B5	9863",
		"E8 B6	9865",
		"E8 B7	9866",
		"E8 B8	986A",
		"E8 B9	986C",
		"E8 BA	98AB",
		"E8 BB	98AD",
		"E8 BC	98AE",
		"E8 BD	98B0",
		"E8 BE	98B4",
		"E8 BF	98B7",
		"E8 C0	98B8",
		"E8 C1	98BA",
		"E8 C2	98BB",
		"E8 C3	98BF",
		"E8 C4	98C2",
		"E8 C5	98C5",
		"E8 C6	98C8",
		"E8 C7	98CC",
		"E8 C8	98E1",
		"E8 C9	98E3",
		"E8 CA	98E5",
		"E8 CB	98E6",
		"E8 CC	98E7",
		"E8 CD	98EA",
		"E8 CE	98F3",
		"E8 CF	98F6",
		"E8 D0	9902",
		"E8 D1	9907",
		"E8 D2	9908",
		"E8 D3	9911",
		"E8 D4	9915",
		"E8 D5	9916",
		"E8 D6	9917",
		"E8 D7	991A",
		"E8 D8	991B",
		"E8 D9	991C",
		"E8 DA	991F",
		"E8 DB	9922",
		"E8 DC	9926",
		"E8 DD	9927",
		"E8 DE	992B",
		"E8 DF	9931",
		"E8 E0	9932",
		"E8 E1	9933",
		"E8 E2	9934",
		"E8 E3	9935",
		"E8 E4	9939",
		"E8 E5	993A",
		"E8 E6	993B",
		"E8 E7	993C",
		"E8 E8	9940",
		"E8 E9	9941",
		"E8 EA	9946",
		"E8 EB	9947",
		"E8 EC	9948",
		"E8 ED	994D",
		"E8 EE	994E",
		"E8 EF	9954",
		"E8 F0	9958",
		"E8 F1	9959",
		"E8 F2	995B",
		"E8 F3	995C",
		"E8 F4	995E",
		"E8 F5	995F",
		"E8 F6	9960",
		"E8 F7	999B",
		"E8 F8	999D",
		"E8 F9	999F",
		"E8 FA	99A6",
		"E8 FB	99B0",
		"E8 FC	99B1",
		"E8 FD	99B2",
		"E8 FE	99B5",
		"E9 A1	99B9",
		"E9 A2	99BA",
		"E9 A3	99BD",
		"E9 A4	99BF",
		"E9 A5	99C3",
		"E9 A6	99C9",
		"E9 A7	99D3",
		"E9 A8	99D4",
		"E9 A9	99D9",
		"E9 AA	99DA",
		"E9 AB	99DC",
		"E9 AC	99DE",
		"E9 AD	99E7",
		"E9 AE	99EA",
		"E9 AF	99EB",
		"E9 B0	99EC",
		"E9 B1	99F0",
		"E9 B2	99F4",
		"E9 B3	99F5",
		"E9 B4	99F9",
		"E9 B5	99FD",
		"E9 B6	99FE",
		"E9 B7	9A02",
		"E9 B8	9A03",
		"E9 B9	9A04",
		"E9 BA	9A0B",
		"E9 BB	9A0C",
		"E9 BC	9A10",
		"E9 BD	9A11",
		"E9 BE	9A16",
		"E9 BF	9A1E",
		"E9 C0	9A20",
		"E9 C1	9A22",
		"E9 C2	9A23",
		"E9 C3	9A24",
		"E9 C4	9A27",
		"E9 C5	9A2D",
		"E9 C6	9A2E",
		"E9 C7	9A33",
		"E9 C8	9A35",
		"E9 C9	9A36",
		"E9 CA	9A38",
		"E9 CC	9A41",
		"E9 CD	9A44",
		"E9 CB	9A47",
		"E9 CE	9A4A",
		"E9 CF	9A4B",
		"E9 D0	9A4C",
		"E9 D1	9A4E",
		"E9 D2	9A51",
		"E9 D3	9A54",
		"E9 D4	9A56",
		"E9 D5	9A5D",
		"E9 D6	9AAA",
		"E9 D7	9AAC",
		"E9 D8	9AAE",
		"E9 D9	9AAF",
		"E9 DA	9AB2",
		"E9 DB	9AB4",
		"E9 DC	9AB5",
		"E9 DD	9AB6",
		"E9 DE	9AB9",
		"E9 DF	9ABB",
		"E9 E0	9ABE",
		"E9 E1	9ABF",
		"E9 E2	9AC1",
		"E9 E3	9AC3",
		"E9 E4	9AC6",
		"E9 E5	9AC8",
		"E9 E6	9ACE",
		"E9 E7	9AD0",
		"E9 E8	9AD2",
		"E9 E9	9AD5",
		"E9 EA	9AD6",
		"E9 EB	9AD7",
		"E9 EC	9ADB",
		"E9 ED	9ADC",
		"E9 EE	9AE0",
		"E9 EF	9AE4",
		"E9 F0	9AE5",
		"E9 F1	9AE7",
		"E9 F2	9AE9",
		"E9 F3	9AEC",
		"E9 F4	9AF2",
		"E9 F5	9AF3",
		"E9 F6	9AF5",
		"E9 F7	9AF9",
		"E9 F8	9AFA",
		"E9 F9	9AFD",
		"E9 FA	9AFF",
		"E9 FB	9B00",
		"E9 FC	9B01",
		"E9 FD	9B02",
		"E9 FE	9B03",
		"EA A1	9B04",
		"EA A2	9B05",
		"EA A3	9B08",
		"EA A4	9B09",
		"EA A5	9B0B",
		"EA A6	9B0C",
		"EA A7	9B0D",
		"EA A8	9B0E",
		"EA A9	9B10",
		"EA AA	9B12",
		"EA AB	9B16",
		"EA AC	9B19",
		"EA AD	9B1B",
		"EA AE	9B1C",
		"EA AF	9B20",
		"EA B0	9B26",
		"EA B1	9B2B",
		"EA B2	9B2D",
		"EA B3	9B33",
		"EA B4	9B34",
		"EA B5	9B35",
		"EA B6	9B37",
		"EA B7	9B39",
		"EA B8	9B3A",
		"EA B9	9B3D",
		"EA BA	9B48",
		"EA BB	9B4B",
		"EA BC	9B4C",
		"EA BD	9B55",
		"EA BE	9B56",
		"EA BF	9B57",
		"EA C0	9B5B",
		"EA C1	9B5E",
		"EA C2	9B61",
		"EA C3	9B63",
		"EA C4	9B65",
		"EA C5	9B66",
		"EA C6	9B68",
		"EA C7	9B6A",
		"EA C8	9B6B",
		"EA C9	9B6C",
		"EA CA	9B6D",
		"EA CB	9B6E",
		"EA CC	9B73",
		"EA CD	9B75",
		"EA CE	9B77",
		"EA CF	9B78",
		"EA D0	9B79",
		"EA D1	9B7F",
		"EA D2	9B80",
		"EA D3	9B84",
		"EA D4	9B85",
		"EA D5	9B86",
		"EA D6	9B87",
		"EA D7	9B89",
		"EA D8	9B8A",
		"EA D9	9B8B",
		"EA DA	9B8D",
		"EA DB	9B8F",
		"EA DC	9B90",
		"EA DD	9B94",
		"EA DE	9B9A",
		"EA DF	9B9D",
		"EA E0	9B9E",
		"EA E1	9BA6",
		"EA E2	9BA7",
		"EA E3	9BA9",
		"EA E4	9BAC",
		"EA E5	9BB0",
		"EA E6	9BB1",
		"EA E7	9BB2",
		"EA E8	9BB7",
		"EA E9	9BB8",
		"EA EA	9BBB",
		"EA EB	9BBC",
		"EA EC	9BBE",
		"EA ED	9BBF",
		"EA EE	9BC1",
		"EA EF	9BC7",
		"EA F0	9BC8",
		"EA F1	9BCE",
		"EA F2	9BD0",
		"EA F3	9BD7",
		"EA F4	9BD8",
		"EA F5	9BDD",
		"EA F6	9BDF",
		"EA F7	9BE5",
		"EA F8	9BE7",
		"EA F9	9BEA",
		"EA FA	9BEB",
		"EA FB	9BEF",
		"EA FC	9BF3",
		"EA FD	9BF7",
		"EA FE	9BF8",
		"EB A1	9BF9",
		"EB A2	9BFA",
		"EB A3	9BFD",
		"EB A4	9BFF",
		"EB A5	9C00",
		"EB A6	9C02",
		"EB A7	9C0B",
		"EB A8	9C0F",
		"EB A9	9C11",
		"EB AA	9C16",
		"EB AB	9C18",
		"EB AC	9C19",
		"EB AD	9C1A",
		"EB AE	9C1C",
		"EB AF	9C1E",
		"EB B0	9C22",
		"EB B1	9C23",
		"EB B2	9C26",
		"EB B3	9C27",
		"EB B4	9C28",
		"EB B5	9C29",
		"EB B6	9C2A",
		"EB B7	9C31",
		"EB B8	9C35",
		"EB B9	9C36",
		"EB BA	9C37",
		"EB BB	9C3D",
		"EB BC	9C41",
		"EB BD	9C43",
		"EB BE	9C44",
		"EB BF	9C45",
		"EB C0	9C49",
		"EB C1	9C4A",
		"EB C2	9C4E",
		"EB C3	9C4F",
		"EB C4	9C50",
		"EB C5	9C53",
		"EB C6	9C54",
		"EB C7	9C56",
		"EB C8	9C58",
		"EB C9	9C5B",
		"EB D0	9C5C",
		"EB CA	9C5D",
		"EB CB	9C5E",
		"EB CC	9C5F",
		"EB CD	9C63",
		"EB D2	9C68",
		"EB CE	9C69",
		"EB CF	9C6A",
		"EB D1	9C6B",
		"EB D3	9C6E",
		"EB D4	9C70",
		"EB D5	9C72",
		"EB D6	9C75",
		"EB D7	9C77",
		"EB D8	9C7B",
		"EB D9	9CE6",
		"EB DA	9CF2",
		"EB DB	9CF7",
		"EB DC	9CF9",
		"EB DE	9D02",
		"EB DD	9D0B",
		"EB DF	9D11",
		"EB E0	9D17",
		"EB E1	9D18",
		"EB E2	9D1C",
		"EB E3	9D1D",
		"EB E4	9D1E",
		"EB E5	9D2F",
		"EB E6	9D30",
		"EB E7	9D32",
		"EB E8	9D33",
		"EB E9	9D34",
		"EB EA	9D3A",
		"EB EB	9D3C",
		"EB ED	9D3D",
		"EB EE	9D42",
		"EB EF	9D43",
		"EB EC	9D45",
		"EB F0	9D47",
		"EB F1	9D4A",
		"EB F2	9D53",
		"EB F3	9D54",
		"EB F4	9D5F",
		"EB F6	9D62",
		"EB F5	9D63",
		"EB F7	9D65",
		"EB F8	9D69",
		"EB F9	9D6A",
		"EB FA	9D6B",
		"EB FB	9D70",
		"EB FC	9D76",
		"EB FD	9D77",
		"EB FE	9D7B",
		"EC A1	9D7C",
		"EC A2	9D7E",
		"EC A3	9D83",
		"EC A4	9D84",
		"EC A5	9D86",
		"EC A6	9D8A",
		"EC A7	9D8D",
		"EC A8	9D8E",
		"EC A9	9D92",
		"EC AA	9D93",
		"EC AB	9D95",
		"EC AC	9D96",
		"EC AD	9D97",
		"EC AE	9D98",
		"EC AF	9DA1",
		"EC B0	9DAA",
		"EC B1	9DAC",
		"EC B2	9DAE",
		"EC B3	9DB1",
		"EC B4	9DB5",
		"EC B5	9DB9",
		"EC B6	9DBC",
		"EC B7	9DBF",
		"EC B8	9DC3",
		"EC B9	9DC7",
		"EC BA	9DC9",
		"EC BB	9DCA",
		"EC BC	9DD4",
		"EC BD	9DD5",
		"EC BE	9DD6",
		"EC BF	9DD7",
		"EC C0	9DDA",
		"EC C1	9DDE",
		"EC C2	9DDF",
		"EC C3	9DE0",
		"EC C4	9DE5",
		"EC C5	9DE7",
		"EC C6	9DE9",
		"EC C7	9DEB",
		"EC C8	9DEE",
		"EC C9	9DF0",
		"EC CA	9DF3",
		"EC CB	9DF4",
		"EC CC	9DFE",
		"EC CE	9E02",
		"EC CF	9E07",
		"EC CD	9E0A",
		"EC D0	9E0E",
		"EC D1	9E10",
		"EC D2	9E11",
		"EC D3	9E12",
		"EC D4	9E15",
		"EC D5	9E16",
		"EC D6	9E19",
		"EC D7	9E1C",
		"EC D8	9E1D",
		"EC D9	9E7A",
		"EC DA	9E7B",
		"EC DB	9E7C",
		"EC DC	9E80",
		"EC DD	9E82",
		"EC DE	9E83",
		"EC DF	9E84",
		"EC E0	9E85",
		"EC E1	9E87",
		"EC E2	9E8E",
		"EC E3	9E8F",
		"EC E4	9E96",
		"EC E5	9E98",
		"EC E6	9E9B",
		"EC E7	9E9E",
		"EC E8	9EA4",
		"EC E9	9EA8",
		"EC EA	9EAC",
		"EC EB	9EAE",
		"EC EC	9EAF",
		"EC ED	9EB0",
		"EC EE	9EB3",
		"EC EF	9EB4",
		"EC F0	9EB5",
		"EC F1	9EC6",
		"EC F2	9EC8",
		"EC F3	9ECB",
		"EC F4	9ED5",
		"EC F5	9EDF",
		"EC F6	9EE4",
		"EC F7	9EE7",
		"EC F8	9EEC",
		"EC F9	9EED",
		"EC FA	9EEE",
		"EC FB	9EF0",
		"EC FC	9EF1",
		"EC FD	9EF2",
		"EC FE	9EF5",
		"ED A1	9EF8",
		"ED A2	9EFF",
		"ED A3	9F02",
		"ED A4	9F03",
		"ED A5	9F09",
		"ED A6	9F0F",
		"ED A7	9F10",
		"ED A8	9F11",
		"ED A9	9F12",
		"ED AA	9F14",
		"ED AB	9F16",
		"ED AC	9F17",
		"ED AD	9F19",
		"ED AE	9F1A",
		"ED AF	9F1B",
		"ED B0	9F1F",
		"ED B1	9F22",
		"ED B2	9F26",
		"ED B3	9F2A",
		"ED B4	9F2B",
		"ED B5	9F2F",
		"ED B6	9F31",
		"ED B7	9F32",
		"ED B8	9F34",
		"ED B9	9F37",
		"ED BA	9F39",
		"ED BB	9F3A",
		"ED BC	9F3C",
		"ED BD	9F3D",
		"ED BE	9F3F",
		"ED BF	9F41",
		"ED C0	9F43",
		"ED C1	9F44",
		"ED C2	9F45",
		"ED C3	9F46",
		"ED C4	9F47",
		"ED C5	9F53",
		"ED C6	9F55",
		"ED C7	9F56",
		"ED C8	9F57",
		"ED C9	9F58",
		"ED CA	9F5A",
		"ED CB	9F5D",
		"ED CC	9F5E",
		"ED CD	9F68",
		"ED CE	9F69",
		"ED CF	9F6D",
		"ED D0	9F6E",
		"ED D1	9F6F",
		"ED D2	9F70",
		"ED D3	9F71",
		"ED D4	9F73",
		"ED D5	9F75",
		"ED D6	9F7A",
		"ED D7	9F7D",
		"ED D8	9F8F",
		"ED D9	9F90",
		"ED DA	9F91",
		"ED DB	9F92",
		"ED DC	9F94",
		"ED DD	9F96",
		"ED DE	9F97",
		"ED DF	9F9E",
		"ED E0	9FA1",
		"ED E1	9FA2",
		"ED E2	9FA3",
		"ED E3	9FA5",
		"A2 B7	FF5E",
};