implement genjisx0208;

include "sys.m";
include "draw.m";

genjisx0208 : module {
	init: fn(ctxt: ref Draw->Context, args: list of string);
};

DATAFILE : con "/lib/convcs/jisx0208-1997";

PAGESZ : con 94;
NPAGES : con 84;
PAGE0 : con 16rA1;
CHAR0 : con 16rA1;

BADCHAR : con 16rFFFD;

init(nil: ref Draw->Context, nil: list of string)
{
	sys := load Sys Sys->PATH;
	fd := sys->create(DATAFILE, Sys->OWRITE, 8r644);
	if (fd == nil) {
		sys->print("cannot create %s: %r\n", DATAFILE);
		return;
	}

	pages := array [NPAGES * PAGESZ] of { * => BADCHAR };

	for (i := 0; i < len data; i++) {
		(nil, toks) := sys->tokenize(data[i], "\t");
		(bytes, ucode) := (hd toks, hd tl toks);
		u := hex2int(ucode);
		(nil, blist) := sys->tokenize(bytes, " ");
		b1 := hex2int(hd blist);
		b2 := hex2int(hd tl blist);

		page := b1 - PAGE0;
		char := b2 - CHAR0;

		pages[(page * PAGESZ)+char] = u;
	}

	s := "";
	for (i = 0; i < len pages; i++)
		s[i] = pages[i];
	pages = nil;
	buf := array of byte s;
	sys->write(fd, buf, len buf);
}


hex2int(s: string): int
{
	n := 0;
	for (i := 0; i < len s; i++) {
		case s[i] {
		'0' to '9' =>
			n = 16*n + s[i] - '0';
		'A' to 'F' =>
			n = 16*n + s[i] + 10 - 'A';
		'a' to 'f' =>
			n = 16*n + s[i] + 10 - 'a';
		* =>
			return n;
		}
	}
	return n;
}


# Data is derived from Unicode Consortium "CharmapML" mapping data
# for EUC-JP.
# JISX0208 appears as charset G1 of the ISO-2022 based EUC-JP encoding
# Although this format is not convenient for building our mapping data file
# it is easily extracted from the Unicode Consortium data.

data := array [] of {
		"A1 F1	00A2",
		"A1 F2	00A3",
		"A1 F8	00A7",
		"A1 AF	00A8",
		"A2 CC	00AC",
		"A1 EB	00B0",
		"A1 DE	00B1",
		"A1 AD	00B4",
		"A2 F9	00B6",
		"A1 DF	00D7",
		"A1 E0	00F7",
		"A6 A1	0391",
		"A6 A2	0392",
		"A6 A3	0393",
		"A6 A4	0394",
		"A6 A5	0395",
		"A6 A6	0396",
		"A6 A7	0397",
		"A6 A8	0398",
		"A6 A9	0399",
		"A6 AA	039A",
		"A6 AB	039B",
		"A6 AC	039C",
		"A6 AD	039D",
		"A6 AE	039E",
		"A6 AF	039F",
		"A6 B0	03A0",
		"A6 B1	03A1",
		"A6 B2	03A3",
		"A6 B3	03A4",
		"A6 B4	03A5",
		"A6 B5	03A6",
		"A6 B6	03A7",
		"A6 B7	03A8",
		"A6 B8	03A9",
		"A6 C1	03B1",
		"A6 C2	03B2",
		"A6 C3	03B3",
		"A6 C4	03B4",
		"A6 C5	03B5",
		"A6 C6	03B6",
		"A6 C7	03B7",
		"A6 C8	03B8",
		"A6 C9	03B9",
		"A6 CA	03BA",
		"A6 CB	03BB",
		"A6 CC	03BC",
		"A6 CD	03BD",
		"A6 CE	03BE",
		"A6 CF	03BF",
		"A6 D0	03C0",
		"A6 D1	03C1",
		"A6 D2	03C3",
		"A6 D3	03C4",
		"A6 D4	03C5",
		"A6 D5	03C6",
		"A6 D6	03C7",
		"A6 D7	03C8",
		"A6 D8	03C9",
		"A7 A7	0401",
		"A7 A1	0410",
		"A7 A2	0411",
		"A7 A3	0412",
		"A7 A4	0413",
		"A7 A5	0414",
		"A7 A6	0415",
		"A7 A8	0416",
		"A7 A9	0417",
		"A7 AA	0418",
		"A7 AB	0419",
		"A7 AC	041A",
		"A7 AD	041B",
		"A7 AE	041C",
		"A7 AF	041D",
		"A7 B0	041E",
		"A7 B1	041F",
		"A7 B2	0420",
		"A7 B3	0421",
		"A7 B4	0422",
		"A7 B5	0423",
		"A7 B6	0424",
		"A7 B7	0425",
		"A7 B8	0426",
		"A7 B9	0427",
		"A7 BA	0428",
		"A7 BB	0429",
		"A7 BC	042A",
		"A7 BD	042B",
		"A7 BE	042C",
		"A7 BF	042D",
		"A7 C0	042E",
		"A7 C1	042F",
		"A7 D1	0430",
		"A7 D2	0431",
		"A7 D3	0432",
		"A7 D4	0433",
		"A7 D5	0434",
		"A7 D6	0435",
		"A7 D8	0436",
		"A7 D9	0437",
		"A7 DA	0438",
		"A7 DB	0439",
		"A7 DC	043A",
		"A7 DD	043B",
		"A7 DE	043C",
		"A7 DF	043D",
		"A7 E0	043E",
		"A7 E1	043F",
		"A7 E2	0440",
		"A7 E3	0441",
		"A7 E4	0442",
		"A7 E5	0443",
		"A7 E6	0444",
		"A7 E7	0445",
		"A7 E8	0446",
		"A7 E9	0447",
		"A7 EA	0448",
		"A7 EB	0449",
		"A7 EC	044A",
		"A7 ED	044B",
		"A7 EE	044C",
		"A7 EF	044D",
		"A7 F0	044E",
		"A7 F1	044F",
		"A7 D7	0451",
		"A1 BE	2010",
		"A1 BD	2015",
		"A1 C2	2016",
		"A1 C6	2018",
		"A1 C7	2019",
		"A1 C8	201C",
		"A1 C9	201D",
		"A2 F7	2020",
		"A2 F8	2021",
		"A1 C5	2025",
		"A1 C4	2026",
		"A2 F3	2030",
		"A1 EC	2032",
		"A1 ED	2033",
		"A2 A8	203B",
		"A1 EE	2103",
		"A2 F2	212B",
		"A2 AB	2190",
		"A2 AC	2191",
		"A2 AA	2192",
		"A2 AD	2193",
		"A2 CD	21D2",
		"A2 CE	21D4",
		"A2 CF	2200",
		"A2 DF	2202",
		"A2 D0	2203",
		"A2 E0	2207",
		"A2 BA	2208",
		"A2 BB	220B",
		"A1 DD	2212",
		"A2 E5	221A",
		"A2 E7	221D",
		"A1 E7	221E",
		"A2 DC	2220",
		"A2 CA	2227",
		"A2 CB	2228",
		"A2 C1	2229",
		"A2 C0	222A",
		"A2 E9	222B",
		"A2 EA	222C",
		"A1 E8	2234",
		"A2 E8	2235",
		"A2 E6	223D",
		"A2 E2	2252",
		"A1 E2	2260",
		"A2 E1	2261",
		"A1 E5	2266",
		"A1 E6	2267",
		"A2 E3	226A",
		"A2 E4	226B",
		"A2 BE	2282",
		"A2 BF	2283",
		"A2 BC	2286",
		"A2 BD	2287",
		"A2 DD	22A5",
		"A2 DE	2312",
		"A8 A1	2500",
		"A8 AC	2501",
		"A8 A2	2502",
		"A8 AD	2503",
		"A8 A3	250C",
		"A8 AE	250F",
		"A8 A4	2510",
		"A8 AF	2513",
		"A8 A6	2514",
		"A8 B1	2517",
		"A8 A5	2518",
		"A8 B0	251B",
		"A8 A7	251C",
		"A8 BC	251D",
		"A8 B7	2520",
		"A8 B2	2523",
		"A8 A9	2524",
		"A8 BE	2525",
		"A8 B9	2528",
		"A8 B4	252B",
		"A8 A8	252C",
		"A8 B8	252F",
		"A8 BD	2530",
		"A8 B3	2533",
		"A8 AA	2534",
		"A8 BA	2537",
		"A8 BF	2538",
		"A8 B5	253B",
		"A8 AB	253C",
		"A8 BB	253F",
		"A8 C0	2542",
		"A8 B6	254B",
		"A2 A3	25A0",
		"A2 A2	25A1",
		"A2 A5	25B2",
		"A2 A4	25B3",
		"A2 A7	25BC",
		"A2 A6	25BD",
		"A2 A1	25C6",
		"A1 FE	25C7",
		"A1 FB	25CB",
		"A1 FD	25CE",
		"A1 FC	25CF",
		"A2 FE	25EF",
		"A1 FA	2605",
		"A1 F9	2606",
		"A1 EA	2640",
		"A1 E9	2642",
		"A2 F6	266A",
		"A2 F5	266D",
		"A2 F4	266F",
		"A1 A1	3000",
		"A1 A2	3001",
		"A1 A3	3002",
		"A1 B7	3003",
		"A1 B9	3005",
		"A1 BA	3006",
		"A1 BB	3007",
		"A1 D2	3008",
		"A1 D3	3009",
		"A1 D4	300A",
		"A1 D5	300B",
		"A1 D6	300C",
		"A1 D7	300D",
		"A1 D8	300E",
		"A1 D9	300F",
		"A1 DA	3010",
		"A1 DB	3011",
		"A2 A9	3012",
		"A2 AE	3013",
		"A1 CC	3014",
		"A1 CD	3015",
		"A1 C1	301C",
		"A4 A1	3041",
		"A4 A2	3042",
		"A4 A3	3043",
		"A4 A4	3044",
		"A4 A5	3045",
		"A4 A6	3046",
		"A4 A7	3047",
		"A4 A8	3048",
		"A4 A9	3049",
		"A4 AA	304A",
		"A4 AB	304B",
		"A4 AC	304C",
		"A4 AD	304D",
		"A4 AE	304E",
		"A4 AF	304F",
		"A4 B0	3050",
		"A4 B1	3051",
		"A4 B2	3052",
		"A4 B3	3053",
		"A4 B4	3054",
		"A4 B5	3055",
		"A4 B6	3056",
		"A4 B7	3057",
		"A4 B8	3058",
		"A4 B9	3059",
		"A4 BA	305A",
		"A4 BB	305B",
		"A4 BC	305C",
		"A4 BD	305D",
		"A4 BE	305E",
		"A4 BF	305F",
		"A4 C0	3060",
		"A4 C1	3061",
		"A4 C2	3062",
		"A4 C3	3063",
		"A4 C4	3064",
		"A4 C5	3065",
		"A4 C6	3066",
		"A4 C7	3067",
		"A4 C8	3068",
		"A4 C9	3069",
		"A4 CA	306A",
		"A4 CB	306B",
		"A4 CC	306C",
		"A4 CD	306D",
		"A4 CE	306E",
		"A4 CF	306F",
		"A4 D0	3070",
		"A4 D1	3071",
		"A4 D2	3072",
		"A4 D3	3073",
		"A4 D4	3074",
		"A4 D5	3075",
		"A4 D6	3076",
		"A4 D7	3077",
		"A4 D8	3078",
		"A4 D9	3079",
		"A4 DA	307A",
		"A4 DB	307B",
		"A4 DC	307C",
		"A4 DD	307D",
		"A4 DE	307E",
		"A4 DF	307F",
		"A4 E0	3080",
		"A4 E1	3081",
		"A4 E2	3082",
		"A4 E3	3083",
		"A4 E4	3084",
		"A4 E5	3085",
		"A4 E6	3086",
		"A4 E7	3087",
		"A4 E8	3088",
		"A4 E9	3089",
		"A4 EA	308A",
		"A4 EB	308B",
		"A4 EC	308C",
		"A4 ED	308D",
		"A4 EE	308E",
		"A4 EF	308F",
		"A4 F0	3090",
		"A4 F1	3091",
		"A4 F2	3092",
		"A4 F3	3093",
		"A1 AB	309B",
		"A1 AC	309C",
		"A1 B5	309D",
		"A1 B6	309E",
		"A5 A1	30A1",
		"A5 A2	30A2",
		"A5 A3	30A3",
		"A5 A4	30A4",
		"A5 A5	30A5",
		"A5 A6	30A6",
		"A5 A7	30A7",
		"A5 A8	30A8",
		"A5 A9	30A9",
		"A5 AA	30AA",
		"A5 AB	30AB",
		"A5 AC	30AC",
		"A5 AD	30AD",
		"A5 AE	30AE",
		"A5 AF	30AF",
		"A5 B0	30B0",
		"A5 B1	30B1",
		"A5 B2	30B2",
		"A5 B3	30B3",
		"A5 B4	30B4",
		"A5 B5	30B5",
		"A5 B6	30B6",
		"A5 B7	30B7",
		"A5 B8	30B8",
		"A5 B9	30B9",
		"A5 BA	30BA",
		"A5 BB	30BB",
		"A5 BC	30BC",
		"A5 BD	30BD",
		"A5 BE	30BE",
		"A5 BF	30BF",
		"A5 C0	30C0",
		"A5 C1	30C1",
		"A5 C2	30C2",
		"A5 C3	30C3",
		"A5 C4	30C4",
		"A5 C5	30C5",
		"A5 C6	30C6",
		"A5 C7	30C7",
		"A5 C8	30C8",
		"A5 C9	30C9",
		"A5 CA	30CA",
		"A5 CB	30CB",
		"A5 CC	30CC",
		"A5 CD	30CD",
		"A5 CE	30CE",
		"A5 CF	30CF",
		"A5 D0	30D0",
		"A5 D1	30D1",
		"A5 D2	30D2",
		"A5 D3	30D3",
		"A5 D4	30D4",
		"A5 D5	30D5",
		"A5 D6	30D6",
		"A5 D7	30D7",
		"A5 D8	30D8",
		"A5 D9	30D9",
		"A5 DA	30DA",
		"A5 DB	30DB",
		"A5 DC	30DC",
		"A5 DD	30DD",
		"A5 DE	30DE",
		"A5 DF	30DF",
		"A5 E0	30E0",
		"A5 E1	30E1",
		"A5 E2	30E2",
		"A5 E3	30E3",
		"A5 E4	30E4",
		"A5 E5	30E5",
		"A5 E6	30E6",
		"A5 E7	30E7",
		"A5 E8	30E8",
		"A5 E9	30E9",
		"A5 EA	30EA",
		"A5 EB	30EB",
		"A5 EC	30EC",
		"A5 ED	30ED",
		"A5 EE	30EE",
		"A5 EF	30EF",
		"A5 F0	30F0",
		"A5 F1	30F1",
		"A5 F2	30F2",
		"A5 F3	30F3",
		"A5 F4	30F4",
		"A5 F5	30F5",
		"A5 F6	30F6",
		"A1 A6	30FB",
		"A1 BC	30FC",
		"A1 B3	30FD",
		"A1 B4	30FE",
		"B0 EC	4E00",
		"C3 FA	4E01",
		"BC B7	4E03",
		"CB FC	4E07",
		"BE E6	4E08",
		"BB B0	4E09",
		"BE E5	4E0A",
		"B2 BC	4E0B",
		"C9 D4	4E0D",
		"CD BF	4E0E",
		"D0 A2	4E10",
		"B1 AF	4E11",
		"B3 EE	4E14",
		"D0 A3	4E15",
		"C0 A4	4E16",
		"D2 C2	4E17",
		"B5 D6	4E18",
		"CA BA	4E19",
		"BE E7	4E1E",
		"CE BE	4E21",
		"CA C2	4E26",
		"D0 A4	4E2A",
		"C3 E6	4E2D",
		"D0 A5	4E31",
		"B6 FA	4E32",
		"D0 A6	4E36",
		"B4 DD	4E38",
		"C3 B0	4E39",
		"BC E7	4E3B",
		"D0 A7	4E3C",
		"D0 A8	4E3F",
		"D0 A9	4E42",
		"C7 B5	4E43",
		"B5 D7	4E45",
		"C7 B7	4E4B",
		"C6 E3	4E4D",
		"B8 C3	4E4E",
		"CB B3	4E4F",
		"E9 C9	4E55",
		"D0 AA	4E56",
		"BE E8	4E57",
		"D0 AB	4E58",
		"B2 B5	4E59",
		"B6 E5	4E5D",
		"B8 F0	4E5E",
		"CC E9	4E5F",
		"D6 A6	4E62",
		"CD F0	4E71",
		"C6 FD	4E73",
		"B4 A5	4E7E",
		"B5 B5	4E80",
		"D0 AC	4E82",
		"D0 AD	4E85",
		"CE BB	4E86",
		"CD BD	4E88",
		"C1 E8	4E89",
		"D0 AF	4E8A",
		"BB F6	4E8B",
		"C6 F3	4E8C",
		"D0 B2	4E8E",
		"B1 BE	4E91",
		"B8 DF	4E92",
		"B8 DE	4E94",
		"B0 E6	4E95",
		"CF CB	4E98",
		"CF CA	4E99",
		"BA B3	4E9B",
		"B0 A1	4E9C",
		"D0 B3	4E9E",
		"D0 B4	4E9F",
		"D0 B5	4EA0",
		"CB B4	4EA1",
		"D0 B6	4EA2",
		"B8 F2	4EA4",
		"B0 E7	4EA5",
		"CB F2	4EA6",
		"B5 FC	4EA8",
		"B5 FD	4EAB",
		"B5 FE	4EAC",
		"C4 E2	4EAD",
		"CE BC	4EAE",
		"D0 B7	4EB0",
		"D0 B8	4EB3",
		"D0 B9	4EB6",
		"BF CD	4EBA",
		"BD BA	4EC0",
		"BF CE	4EC1",
		"D0 BE	4EC2",
		"D0 BC	4EC4",
		"D0 BD	4EC6",
		"B5 D8	4EC7",
		"BA A3	4ECA",
		"B2 F0	4ECB",
		"D0 BB	4ECD",
		"D0 BA	4ECE",
		"CA A9	4ECF",
		"BB C6	4ED4",
		"BB C5	4ED5",
		"C2 BE	4ED6",
		"D0 BF	4ED7",
		"C9 D5	4ED8",
		"C0 E7	4ED9",
		"A1 B8	4EDD",
		"D0 C0	4EDE",
		"D0 C2	4EDF",
		"C2 E5	4EE3",
		"CE E1	4EE4",
		"B0 CA	4EE5",
		"D0 C1	4EED",
		"B2 BE	4EEE",
		"B6 C4	4EF0",
		"C3 E7	4EF2",
		"B7 EF	4EF6",
		"D0 C3	4EF7",
		"C7 A4	4EFB",
		"B4 EB	4F01",
		"D0 C4	4F09",
		"B0 CB	4F0A",
		"B8 E0	4F0D",
		"B4 EC	4F0E",
		"C9 FA	4F0F",
		"C8 B2	4F10",
		"B5 D9	4F11",
		"B2 F1	4F1A",
		"D0 E7	4F1C",
		"C5 C1	4F1D",
		"C7 EC	4F2F",
		"D0 C6	4F30",
		"C8 BC	4F34",
		"CE E2	4F36",
		"BF AD	4F38",
		"BB C7	4F3A",
		"BB F7	4F3C",
		"B2 C0	4F3D",
		"C4 D1	4F43",
		"C3 A2	4F46",
		"D0 CA	4F47",
		"B0 CC	4F4D",
		"C4 E3	4F4E",
		"BD BB	4F4F",
		"BA B4	4F50",
		"CD A4	4F51",
		"C2 CE	4F53",
		"B2 BF	4F55",
		"D0 C9	4F57",
		"CD BE	4F59",
		"D0 C5	4F5A",
		"D0 C7	4F5B",
		"BA EE	4F5C",
		"D0 C8	4F5D",
		"D5 A4	4F5E",
		"D0 D0	4F69",
		"D0 D3	4F6F",
		"D0 D1	4F70",
		"B2 C2	4F73",
		"CA BB	4F75",
		"D0 CB	4F76",
		"D0 CF	4F7B",
		"B8 F3	4F7C",
		"BB C8	4F7F",
		"B4 A6	4F83",
		"D0 D4	4F86",
		"D0 CC	4F88",
		"CE E3	4F8B",
		"BB F8	4F8D",
		"D0 CD	4F8F",
		"D0 D2	4F91",
		"D0 D5	4F96",
		"D0 CE	4F98",
		"B6 A1	4F9B",
		"B0 CD	4F9D",
		"B6 A2	4FA0",
		"B2 C1	4FA1",
		"D5 A5	4FAB",
		"CB F9	4FAD",
		"C9 EE	4FAE",
		"B8 F4	4FAF",
		"BF AF	4FB5",
		"CE B7	4FB6",
		"CA D8	4FBF",
		"B7 B8	4FC2",
		"C2 A5	4FC3",
		"B2 E4	4FC4",
		"BD D3	4FCA",
		"D0 D9	4FCE",
		"D0 DE	4FD0",
		"D0 DC	4FD1",
		"D0 D7	4FD4",
		"C2 AF	4FD7",
		"D0 DA	4FD8",
		"D0 DD	4FDA",
		"D0 DB	4FDB",
		"CA DD	4FDD",
		"D0 D8	4FDF",
		"BF AE	4FE1",
		"CB F3	4FE3",
		"D0 DF	4FE4",
		"D0 E0	4FE5",
		"BD A4	4FEE",
		"D0 ED	4FEF",
		"C7 D0	4FF3",
		"C9 B6	4FF5",
		"D0 E8	4FF6",
		"CA F0	4FF8",
		"B2 B6	4FFA",
		"D0 EC	4FFE",
		"D0 E6	5005",
		"D0 EF	5006",
		"C1 D2	5009",
		"B8 C4	500B",
		"C7 DC	500D",
		"E0 C7	500F",
		"D0 EE	5011",
		"C5 DD	5012",
		"D0 E3	5014",
		"B8 F6	5016",
		"B8 F5	5019",
		"D0 E1	501A",
		"BC DA	501F",
		"D0 E9	5021",
		"CA EF	5023",
		"C3 CD	5024",
		"D0 E5	5025",
		"B7 F1	5026",
		"D0 E2	5028",
		"D0 EA	5029",
		"D0 E4	502A",
		"CE D1	502B",
		"D0 EB	502C",
		"CF C1	502D",
		"B6 E6	5036",
		"B7 F0	5039",
		"D0 F0	5043",
		"D0 F1	5047",
		"D0 F5	5048",
		"B0 CE	5049",
		"CA D0	504F",
		"D0 F4	5050",
		"D0 F3	5055",
		"D0 F7	5056",
		"D0 F6	505A",
		"C4 E4	505C",
		"B7 F2	5065",
		"D0 F8	506C",
		"BC C5	5072",
		"C2 A6	5074",
		"C4 E5	5075",
		"B6 F6	5076",
		"D0 F9	5078",
		"B5 B6	507D",
		"D0 FA	5080",
		"D0 FC	5085",
		"CB B5	508D",
		"B7 E6	5091",
		"BB B1	5098",
		"C8 F7	5099",
		"D0 FB	509A",
		"BA C5	50AC",
		"CD C3	50AD",
		"D0 FE	50B2",
		"D1 A3	50B3",
		"D0 FD	50B4",
		"BA C4	50B5",
		"BD FD	50B7",
		"B7 B9	50BE",
		"D1 A4	50C2",
		"B6 CF	50C5",
		"D1 A1	50C9",
		"D1 A2	50CA",
		"C6 AF	50CD",
		"C1 FC	50CF",
		"B6 A3	50D1",
		"CB CD	50D5",
		"D1 A5	50D6",
		"CE BD	50DA",
		"D1 A6	50DE",
		"D1 A9	50E3",
		"D1 A7	50E5",
		"C1 CE	50E7",
		"D1 A8	50ED",
		"D1 AA	50EE",
		"D1 AC	50F5",
		"D1 AB	50F9",
		"CA C8	50FB",
		"B5 B7	5100",
		"D1 AE	5101",
		"D1 AF	5102",
		"B2 AF	5104",
		"D1 AD	5109",
		"BC F4	5112",
		"D1 B2	5114",
		"D1 B1	5115",
		"D1 B0	5116",
		"D0 D6	5118",
		"D1 B3	511A",
		"BD FE	511F",
		"D1 B4	5121",
		"CD A5	512A",
		"CC D9	5132",
		"D1 B6	5137",
		"D1 B5	513A",
		"D1 B8	513B",
		"D1 B7	513C",
		"D1 B9	513F",
		"D1 BA	5140",
		"B0 F4	5141",
		"B8 B5	5143",
		"B7 BB	5144",
		"BD BC	5145",
		"C3 FB	5146",
		"B6 A4	5147",
		"C0 E8	5148",
		"B8 F7	5149",
		"B9 EE	514B",
		"D1 BC	514C",
		"CC C8	514D",
		"C5 C6	514E",
		"BB F9	5150",
		"D1 BB	5152",
		"D1 BD	5154",
		"C5 DE	515A",
		"B3 F5	515C",
		"D1 BE	5162",
		"C6 FE	5165",
		"C1 B4	5168",
		"D1 C0	5169",
		"D1 C1	516A",
		"C8 AC	516B",
		"B8 F8	516C",
		"CF BB	516D",
		"D1 C2	516E",
		"B6 A6	5171",
		"CA BC	5175",
		"C2 B6	5176",
		"B6 F1	5177",
		"C5 B5	5178",
		"B7 F3	517C",
		"D1 C3	5180",
		"D1 C4	5182",
		"C6 E2	5185",
		"B1 DF	5186",
		"D1 C7	5189",
		"BA FD	518A",
		"D1 C6	518C",
		"BA C6	518D",
		"D1 C8	518F",
		"E6 EE	5190",
		"D1 C9	5191",
		"CB C1	5192",
		"D1 CA	5193",
		"D1 CB	5195",
		"D1 CC	5196",
		"BE E9	5197",
		"BC CC	5199",
		"B4 A7	51A0",
		"D1 CF	51A2",
		"D1 CD	51A4",
		"CC BD	51A5",
		"D1 CE	51A6",
		"C9 DA	51A8",
		"D1 D0	51A9",
		"D1 D1	51AA",
		"D1 D2	51AB",
		"C5 DF	51AC",
		"D1 D6	51B0",
		"D1 D4	51B1",
		"D1 D5	51B2",
		"D1 D3	51B3",
		"BA E3	51B4",
		"D1 D7	51B5",
		"CC EA	51B6",
		"CE E4	51B7",
		"D1 D8	51BD",
		"C0 A8	51C4",
		"D1 D9	51C5",
		"BD DA	51C6",
		"D1 DA	51C9",
		"C3 FC	51CB",
		"CE BF	51CC",
		"C5 E0	51CD",
		"D2 C5	51D6",
		"D1 DB	51DB",
		"F4 A5	51DC",
		"B6 C5	51DD",
		"D1 DC	51E0",
		"CB DE	51E1",
		"BD E8	51E6",
		"C2 FC	51E7",
		"D1 DE	51E9",
		"C6 E4	51EA",
		"D1 DF	51ED",
		"D1 E0	51F0",
		"B3 AE	51F1",
		"D1 E1	51F5",
		"B6 A7	51F6",
		"C6 CC	51F8",
		"B1 FA	51F9",
		"BD D0	51FA",
		"C8 A1	51FD",
		"D1 E2	51FE",
		"C5 E1	5200",
		"BF CF	5203",
		"D1 E3	5204",
		"CA AC	5206",
		"C0 DA	5207",
		"B4 A2	5208",
		"B4 A9	520A",
		"D1 E4	520B",
		"D1 E6	520E",
		"B7 BA	5211",
		"D1 E5	5214",
		"CE F3	5217",
		"BD E9	521D",
		"C8 BD	5224",
		"CA CC	5225",
		"D1 E7	5227",
		"CD F8	5229",
		"D1 E8	522A",
		"D1 E9	522E",
		"C5 FE	5230",
		"D1 EA	5233",
		"C0 A9	5236",
		"BA FE	5237",
		"B7 F4	5238",
		"D1 EB	5239",
		"BB C9	523A",
		"B9 EF	523B",
		"C4 E6	5243",
		"D1 ED	5244",
		"C2 A7	5247",
		"BA EF	524A",
		"D1 EE	524B",
		"D1 EF	524C",
		"C1 B0	524D",
		"D1 EC	524F",
		"D1 F1	5254",
		"CB B6	5256",
		"B9 E4	525B",
		"D1 F0	525E",
		"B7 F5	5263",
		"BA DE	5264",
		"C7 ED	5265",
		"D1 F4	5269",
		"D1 F2	526A",
		"C9 FB	526F",
		"BE EA	5270",
		"D1 FB	5271",
		"B3 E4	5272",
		"D1 F5	5273",
		"D1 F3	5274",
		"C1 CF	5275",
		"D1 F7	527D",
		"D1 F6	527F",
		"B3 C4	5283",
		"B7 E0	5287",
		"D1 FC	5288",
		"CE AD	5289",
		"D1 F8	528D",
		"D1 FD	5291",
		"D1 FA	5292",
		"D1 F9	5294",
		"CE CF	529B",
		"B8 F9	529F",
		"B2 C3	52A0",
		"CE F4	52A3",
		"BD F5	52A9",
		"C5 D8	52AA",
		"B9 E5	52AB",
		"D2 A2	52AC",
		"D2 A3	52AD",
		"CE E5	52B1",
		"CF AB	52B4",
		"D2 A5	52B5",
		"B8 FA	52B9",
		"D2 A4	52BC",
		"B3 AF	52BE",
		"D2 A6	52C1",
		"CB D6	52C3",
		"C4 BC	52C5",
		"CD A6	52C7",
		"CA D9	52C9",
		"D2 A7	52CD",
		"F0 D5	52D2",
		"C6 B0	52D5",
		"D2 A8	52D7",
		"B4 AA	52D8",
		"CC B3	52D9",
		"BE A1	52DD",
		"D2 A9	52DE",
		"CA E7	52DF",
		"D2 AD	52E0",
		"C0 AA	52E2",
		"D2 AA	52E3",
		"B6 D0	52E4",
		"D2 AB	52E6",
		"B4 AB	52E7",
		"B7 AE	52F2",
		"D2 AE	52F3",
		"D2 AF	52F5",
		"D2 B0	52F8",
		"D2 B1	52F9",
		"BC DB	52FA",
		"B8 FB	52FE",
		"CC DE	52FF",
		"CC E8	5301",
		"C6 F7	5302",
		"CA F1	5305",
		"D2 B2	5306",
		"D2 B3	5308",
		"D2 B5	530D",
		"D2 B7	530F",
		"D2 B6	5310",
		"D2 B8	5315",
		"B2 BD	5316",
		"CB CC	5317",
		"BA FC	5319",
		"D2 B9	531A",
		"C1 D9	531D",
		"BE A2	5320",
		"B6 A9	5321",
		"D2 BA	5323",
		"C8 DB	532A",
		"D2 BB	532F",
		"D2 BC	5331",
		"D2 BD	5333",
		"D2 BE	5338",
		"C9 A4	5339",
		"B6 E8	533A",
		"B0 E5	533B",
		"C6 BF	533F",
		"D2 BF	5340",
		"BD BD	5341",
		"C0 E9	5343",
		"D2 C1	5345",
		"D2 C0	5346",
		"BE A3	5347",
		"B8 E1	5348",
		"D2 C3	5349",
		"C8 BE	534A",
		"D2 C4	534D",
		"C8 DC	5351",
		"C2 B4	5352",
		"C2 EE	5353",
		"B6 A8	5354",
		"C6 EE	5357",
		"C3 B1	5358",
		"C7 EE	535A",
		"CB CE	535C",
		"D2 C6	535E",
		"C0 EA	5360",
		"B7 B5	5366",
		"D2 C7	5369",
		"D2 C8	536E",
		"B1 AC	536F",
		"B0 F5	5370",
		"B4 ED	5371",
		"C2 A8	5373",
		"B5 D1	5374",
		"CD F1	5375",
		"D2 CB	5377",
		"B2 B7	5378",
		"D2 CA	537B",
		"B6 AA	537F",
		"D2 CC	5382",
		"CC F1	5384",
		"D2 CD	5396",
		"CE D2	5398",
		"B8 FC	539A",
		"B8 B6	539F",
		"D2 CE	53A0",
		"D2 D0	53A5",
		"D2 CF	53A6",
		"BF DF	53A8",
		"B1 B9	53A9",
		"B1 DE	53AD",
		"D2 D1	53AE",
		"D2 D2	53B0",
		"B8 B7	53B3",
		"D2 D3	53B6",
		"B5 EE	53BB",
		"BB B2	53C2",
		"D2 D4	53C3",
		"CB F4	53C8",
		"BA B5	53C9",
		"B5 DA	53CA",
		"CD A7	53CB",
		"C1 D0	53CC",
		"C8 BF	53CD",
		"BC FD	53CE",
		"BD C7	53D4",
		"BC E8	53D6",
		"BC F5	53D7",
		"BD F6	53D9",
		"C8 C0	53DB",
		"D2 D7	53DF",
		"B1 C3	53E1",
		"C1 D1	53E2",
		"B8 FD	53E3",
		"B8 C5	53E4",
		"B6 E7	53E5",
		"D2 DB	53E8",
		"C3 A1	53E9",
		"C2 FE	53EA",
		"B6 AB	53EB",
		"BE A4	53EC",
		"D2 DC	53ED",
		"D2 DA	53EE",
		"B2 C4	53EF",
		"C2 E6	53F0",
		"BC B8	53F1",
		"BB CB	53F2",
		"B1 A6	53F3",
		"B3 F0	53F6",
		"B9 E6	53F7",
		"BB CA	53F8",
		"D2 DD	53FA",
		"D2 DE	5401",
		"B5 C9	5403",
		"B3 C6	5404",
		"B9 E7	5408",
		"B5 C8	5409",
		"C4 DF	540A",
		"B1 A5	540B",
		"C6 B1	540C",
		"CC BE	540D",
		"B9 A1	540E",
		"CD F9	540F",
		"C5 C7	5410",
		"B8 FE	5411",
		"B7 AF	541B",
		"D2 E7	541D",
		"B6 E3	541F",
		"CB CA	5420",
		"C8 DD	5426",
		"D2 E6	5429",
		"B4 DE	542B",
		"D2 E1	542C",
		"D2 E2	542D",
		"D2 E4	542E",
		"D2 E5	5436",
		"B5 DB	5438",
		"BF E1	5439",
		"CA AD	543B",
		"D2 E3	543C",
		"D2 DF	543D",
		"B8 E3	543E",
		"D2 E0	5440",
		"CF A4	5442",
		"CA F2	5446",
		"C4 E8	5448",
		"B8 E2	5449",
		"B9 F0	544A",
		"D2 E8	544E",
		"C6 DD	5451",
		"D2 EC	545F",
		"BC FE	5468",
		"BC F6	546A",
		"D2 EF	5470",
		"D2 ED	5471",
		"CC A3	5473",
		"D2 EA	5475",
		"D2 F3	5476",
		"D2 EE	5477",
		"D2 F1	547B",
		"B8 C6	547C",
		"CC BF	547D",
		"D2 F2	5480",
		"D2 F4	5484",
		"D2 F6	5486",
		"BA F0	548B",
		"CF C2	548C",
		"D2 EB	548E",
		"D2 E9	548F",
		"D2 F5	5490",
		"D2 F0	5492",
		"D2 F8	54A2",
		"D3 A3	54A4",
		"D2 FA	54A5",
		"D2 FE	54A8",
		"D3 A1	54AB",
		"D2 FB	54AC",
		"D3 BE	54AF",
		"BA E9	54B2",
		"B3 B1	54B3",
		"D2 F9	54B8",
		"D3 A5	54BC",
		"B0 F6	54BD",
		"D3 A4	54BE",
		"B0 A5	54C0",
		"C9 CA	54C1",
		"D3 A2	54C2",
		"D2 FC	54C4",
		"D2 F7	54C7",
		"D2 FD	54C8",
		"BA C8	54C9",
		"D3 A6	54D8",
		"B0 F7	54E1",
		"D3 AF	54E2",
		"D3 A7	54E5",
		"D3 A8	54E6",
		"BE A5	54E8",
		"CB E9	54E9",
		"D3 AD	54ED",
		"D3 AC	54EE",
		"C5 AF	54F2",
		"D3 AE	54FA",
		"D3 AB	54FD",
		"B1 B4	5504",
		"BA B6	5506",
		"BF B0	5507",
		"D3 A9	550F",
		"C5 E2	5510",
		"D3 AA	5514",
		"B0 A2	5516",
		"D3 B4	552E",
		"CD A3	552F",
		"BE A7	5531",
		"D3 BA	5533",
		"D3 B9	5538",
		"D3 B0	5539",
		"C2 C3	553E",
		"D3 B1	5540",
		"C2 EF	5544",
		"D3 B6	5545",
		"BE A6	5546",
		"D3 B3	554C",
		"CC E4	554F",
		"B7 BC	5553",
		"D3 B7	5556",
		"D3 B8	5557",
		"D3 B5	555C",
		"D3 BB	555D",
		"D3 B2	5563",
		"D3 C1	557B",
		"D3 C6	557C",
		"D3 C2	557E",
		"D3 BD	5580",
		"D3 C7	5583",
		"C1 B1	5584",
		"D3 C9	5587",
		"B9 A2	5589",
		"D3 BF	558A",
		"C3 FD	558B",
		"D3 C3	5598",
		"D3 BC	5599",
		"B4 AD	559A",
		"B4 EE	559C",
		"B3 E5	559D",
		"D3 C4	559E",
		"D3 C0	559F",
		"B7 F6	55A7",
		"D3 CA	55A8",
		"D3 C8	55A9",
		"C1 D3	55AA",
		"B5 CA	55AB",
		"B6 AC	55AC",
		"D3 C5	55AE",
		"B6 F4	55B0",
		"B1 C4	55B6",
		"D3 CE	55C4",
		"D3 CC	55C5",
		"D4 A7	55C7",
		"D3 D1	55D4",
		"D3 CB	55DA",
		"D3 CF	55DC",
		"D3 CD	55DF",
		"BB CC	55E3",
		"D3 D0	55E4",
		"D3 D3	55F7",
		"D3 D8	55F9",
		"D3 D6	55FD",
		"D3 D5	55FE",
		"C3 B2	5606",
		"B2 C5	5609",
		"D3 D2	5614",
		"D3 D4	5616",
		"BE A8	5617",
		"B1 B3	5618",
		"D3 D7	561B",
		"B2 DE	5629",
		"D3 E2	562F",
		"BE FC	5631",
		"D3 DE	5632",
		"D3 DC	5634",
		"D3 DD	5636",
		"D3 DF	5638",
		"B1 BD	5642",
		"C1 B9	564C",
		"D3 D9	564E",
		"D3 DA	5650",
		"B3 FA	565B",
		"D3 E1	5664",
		"B4 EF	5668",
		"D3 E4	566A",
		"D3 E0	566B",
		"D3 E3	566C",
		"CA AE	5674",
		"C6 D5	5678",
		"C8 B8	567A",
		"D3 E6	5680",
		"D3 E5	5686",
		"B3 C5	5687",
		"D3 E7	568A",
		"D3 EA	568F",
		"D3 E9	5694",
		"D3 E8	56A0",
		"C7 B9	56A2",
		"D3 EB	56A5",
		"D3 EC	56AE",
		"D3 EE	56B4",
		"D3 ED	56B6",
		"D3 F0	56BC",
		"D3 F3	56C0",
		"D3 F1	56C1",
		"D3 EF	56C2",
		"D3 F2	56C3",
		"D3 F4	56C8",
		"D3 F5	56CE",
		"D3 F6	56D1",
		"D3 F7	56D3",
		"D3 F8	56D7",
		"D1 C5	56D8",
		"BC FC	56DA",
		"BB CD	56DB",
		"B2 F3	56DE",
		"B0 F8	56E0",
		"C3 C4	56E3",
		"D3 F9	56EE",
		"BA A4	56F0",
		"B0 CF	56F2",
		"BF DE	56F3",
		"D3 FA	56F9",
		"B8 C7	56FA",
		"B9 F1	56FD",
		"D3 FC	56FF",
		"D3 FB	5700",
		"CA E0	5703",
		"D3 FD	5704",
		"D4 A1	5708",
		"D3 FE	5709",
		"D4 A2	570B",
		"D4 A3	570D",
		"B7 F7	570F",
		"B1 E0	5712",
		"D4 A4	5713",
		"D4 A6	5716",
		"D4 A5	5718",
		"D4 A8	571C",
		"C5 DA	571F",
		"D4 A9	5726",
		"B0 B5	5727",
		"BA DF	5728",
		"B7 BD	572D",
		"C3 CF	5730",
		"D4 AA	5737",
		"D4 AB	5738",
		"D4 AD	573B",
		"D4 AE	5740",
		"BA E4	5742",
		"B6 D1	5747",
		"CB B7	574A",
		"D4 AC	574E",
		"D4 AF	574F",
		"BA C1	5750",
		"B9 A3	5751",
		"D4 B3	5761",
		"BA A5	5764",
		"C3 B3	5766",
		"D4 B0	5769",
		"C4 DA	576A",
		"D4 B4	577F",
		"BF E2	5782",
		"D4 B2	5788",
		"D4 B5	5789",
		"B7 BF	578B",
		"D4 B6	5793",
		"D4 B7	57A0",
		"B9 A4	57A2",
		"B3 C0	57A3",
		"D4 B9	57A4",
		"D4 BA	57AA",
		"D4 BB	57B0",
		"D4 B8	57B3",
		"D4 B1	57C0",
		"D4 BC	57C3",
		"D4 BD	57C6",
		"CB E4	57CB",
		"BE EB	57CE",
		"D4 BF	57D2",
		"D4 C0	57D3",
		"D4 BE	57D4",
		"D4 C2	57D6",
		"C7 B8	57DC",
		"B0 E8	57DF",
		"C9 D6	57E0",
		"D4 C3	57E3",
		"BE FD	57F4",
		"BC B9	57F7",
		"C7 DD	57F9",
		"B4 F0	57FA",
		"BA EB	57FC",
		"CB D9	5800",
		"C6 B2	5802",
		"B7 F8	5805",
		"C2 CF	5806",
		"D4 C1	580A",
		"D4 C4	580B",
		"C2 C4	5815",
		"D4 C5	5819",
		"D4 C6	581D",
		"D4 C8	5821",
		"C4 E9	5824",
		"B4 AE	582A",
		"F4 A1	582F",
		"B1 E1	5830",
		"CA F3	5831",
		"BE EC	5834",
		"C5 C8	5835",
		"BA E6	583A",
		"D4 CE	583D",
		"CA BD	5840",
		"CE DD	5841",
		"B2 F4	584A",
		"D4 CA	584B",
		"C1 BA	5851",
		"D4 CD	5852",
		"C5 E3	5854",
		"C5 C9	5857",
		"C5 E4	5858",
		"C8 B9	5859",
		"C4 CD	585A",
		"BA C9	585E",
		"D4 C9	5862",
		"B1 F6	5869",
		"C5 B6	586B",
		"D4 CB	5870",
		"D4 C7	5872",
		"BF D0	5875",
		"D4 CF	5879",
		"BD CE	587E",
		"B6 AD	5883",
		"D4 D0	5885",
		"CA E8	5893",
		"C1 FD	5897",
		"C4 C6	589C",
		"D4 D2	589F",
		"CB CF	58A8",
		"D4 D3	58AB",
		"D4 D8	58AE",
		"CA AF	58B3",
		"D4 D7	58B8",
		"D4 D1	58B9",
		"D4 D4	58BA",
		"D4 D6	58BB",
		"BA A6	58BE",
		"CA C9	58C1",
		"D4 D9	58C5",
		"C3 C5	58C7",
		"B2 F5	58CA",
		"BE ED	58CC",
		"D4 DB	58D1",
		"D4 DA	58D3",
		"B9 E8	58D5",
		"D4 DC	58D7",
		"D4 DE	58D8",
		"D4 DD	58D9",
		"D4 E0	58DC",
		"D4 D5	58DE",
		"D4 E2	58DF",
		"D4 E1	58E4",
		"D4 DF	58E5",
		"BB CE	58EB",
		"BF D1	58EC",
		"C1 D4	58EE",
		"D4 E3	58EF",
		"C0 BC	58F0",
		"B0 ED	58F1",
		"C7 E4	58F2",
		"C4 DB	58F7",
		"D4 E5	58F9",
		"D4 E4	58FA",
		"D4 E6	58FB",
		"D4 E7	58FC",
		"D4 E8	58FD",
		"D4 E9	5902",
		"CA D1	5909",
		"D4 EA	590A",
		"B2 C6	590F",
		"D4 EB	5910",
		"CD BC	5915",
		"B3 B0	5916",
		"D2 C9	5918",
		"BD C8	5919",
		"C2 BF	591A",
		"D4 EC	591B",
		"CC EB	591C",
		"CC B4	5922",
		"D4 EE	5925",
		"C2 E7	5927",
		"C5 B7	5929",
		"C2 C0	592A",
		"C9 D7	592B",
		"D4 EF	592C",
		"D4 F0	592D",
		"B1 FB	592E",
		"BC BA	5931",
		"D4 F1	5932",
		"B0 D0	5937",
		"D4 F2	5938",
		"D4 F3	593E",
		"B1 E2	5944",
		"B4 F1	5947",
		"C6 E0	5948",
		"CA F4	5949",
		"D4 F7	594E",
		"C1 D5	594F",
		"D4 F6	5950",
		"B7 C0	5951",
		"CB DB	5954",
		"D4 F5	5955",
		"C5 E5	5957",
		"D4 F9	5958",
		"D4 F8	595A",
		"D4 FB	5960",
		"D4 FA	5962",
		"B1 FC	5965",
		"D4 FC	5967",
		"BE A9	5968",
		"D4 FE	5969",
		"C3 A5	596A",
		"D4 FD	596C",
		"CA B3	596E",
		"BD F7	5973",
		"C5 DB	5974",
		"D5 A1	5978",
		"B9 A5	597D",
		"D5 A2	5981",
		"C7 A1	5982",
		"C8 DE	5983",
		"CC D1	5984",
		"C7 A5	598A",
		"D5 AB	598D",
		"B5 B8	5993",
		"CD C5	5996",
		"CC AF	5999",
		"D6 AC	599B",
		"D5 A3	599D",
		"D5 A6	59A3",
		"C2 C5	59A5",
		"CB B8	59A8",
		"C5 CA	59AC",
		"D5 A7	59B2",
		"CB E5	59B9",
		"BA CA	59BB",
		"BE AA	59BE",
		"D5 A8	59C6",
		"BB D0	59C9",
		"BB CF	59CB",
		"B0 B9	59D0",
		"B8 C8	59D1",
		"C0 AB	59D3",
		"B0 D1	59D4",
		"D5 AC	59D9",
		"D5 AD	59DA",
		"D5 AA	59DC",
		"B1 B8	59E5",
		"B4 AF	59E6",
		"D5 A9	59E8",
		"CC C5	59EA",
		"C9 B1	59EB",
		"B0 A8	59F6",
		"B0 F9	59FB",
		"BB D1	59FF",
		"B0 D2	5A01",
		"B0 A3	5A03",
		"D5 B2	5A09",
		"D5 B0	5A11",
		"CC BC	5A18",
		"D5 B3	5A1A",
		"D5 B1	5A1C",
		"D5 AF	5A1F",
		"BF B1	5A20",
		"D5 AE	5A25",
		"CA DA	5A29",
		"B8 E4	5A2F",
		"D5 B7	5A35",
		"D5 B8	5A36",
		"BE AB	5A3C",
		"D5 B4	5A40",
		"CF AC	5A41",
		"C7 CC	5A46",
		"D5 B6	5A49",
		"BA A7	5A5A",
		"D5 B9	5A62",
		"C9 D8	5A66",
		"D5 BA	5A6A",
		"D5 B5	5A6C",
		"CC BB	5A7F",
		"C7 DE	5A92",
		"D5 BB	5A9A",
		"C9 B2	5A9B",
		"D5 BC	5ABC",
		"D5 C0	5ABD",
		"D5 BD	5ABE",
		"B2 C7	5AC1",
		"D5 BF	5AC2",
		"BC BB	5AC9",
		"D5 BE	5ACB",
		"B7 F9	5ACC",
		"D5 CC	5AD0",
		"D5 C5	5AD6",
		"D5 C2	5AD7",
		"C3 E4	5AE1",
		"D5 C1	5AE3",
		"D5 C3	5AE6",
		"D5 C4	5AE9",
		"D5 C6	5AFA",
		"D5 C7	5AFB",
		"B4 F2	5B09",
		"D5 C9	5B0B",
		"D5 C8	5B0C",
		"D5 CA	5B16",
		"BE EE	5B22",
		"D5 CD	5B2A",
		"C4 DC	5B2C",
		"B1 C5	5B30",
		"D5 CB	5B32",
		"D5 CE	5B36",
		"D5 CF	5B3E",
		"D5 D2	5B40",
		"D5 D0	5B43",
		"D5 D1	5B45",
		"BB D2	5B50",
		"D5 D3	5B51",
		"B9 A6	5B54",
		"D5 D4	5B55",
		"BB FA	5B57",
		"C2 B8	5B58",
		"D5 D5	5B5A",
		"D5 D6	5B5B",
		"BB DA	5B5C",
		"B9 A7	5B5D",
		"CC D2	5B5F",
		"B5 A8	5B63",
		"B8 C9	5B64",
		"D5 D7	5B65",
		"B3 D8	5B66",
		"D5 D8	5B69",
		"C2 B9	5B6B",
		"D5 D9	5B70",
		"D6 A3	5B71",
		"D5 DA	5B73",
		"D5 DB	5B75",
		"D5 DC	5B78",
		"D5 DE	5B7A",
		"D5 DF	5B80",
		"D5 E0	5B83",
		"C2 F0	5B85",
		"B1 A7	5B87",
		"BC E9	5B88",
		"B0 C2	5B89",
		"C1 D7	5B8B",
		"B4 B0	5B8C",
		"BC B5	5B8D",
		"B9 A8	5B8F",
		"C5 E6	5B95",
		"BD A1	5B97",
		"B4 B1	5B98",
		"C3 E8	5B99",
		"C4 EA	5B9A",
		"B0 B8	5B9B",
		"B5 B9	5B9C",
		"CA F5	5B9D",
		"BC C2	5B9F",
		"B5 D2	5BA2",
		"C0 EB	5BA3",
		"BC BC	5BA4",
		"CD A8	5BA5",
		"D5 E1	5BA6",
		"B5 DC	5BAE",
		"BA CB	5BB0",
		"B3 B2	5BB3",
		"B1 E3	5BB4",
		"BE AC	5BB5",
		"B2 C8	5BB6",
		"D5 E2	5BB8",
		"CD C6	5BB9",
		"BD C9	5BBF",
		"BC E4	5BC2",
		"D5 E3	5BC3",
		"B4 F3	5BC4",
		"C6 D2	5BC5",
		"CC A9	5BC6",
		"D5 E4	5BC7",
		"D5 E5	5BC9",
		"C9 D9	5BCC",
		"D5 E7	5BD0",
		"B4 A8	5BD2",
		"B6 F7	5BD3",
		"D5 E6	5BD4",
		"B4 B2	5BDB",
		"BF B2	5BDD",
		"D5 EB	5BDE",
		"BB A1	5BDF",
		"B2 C9	5BE1",
		"D5 EA	5BE2",
		"D5 E8	5BE4",
		"D5 EC	5BE5",
		"D5 E9	5BE6",
		"C7 AB	5BE7",
		"DC CD	5BE8",
		"BF B3	5BE9",
		"D5 ED	5BEB",
		"CE C0	5BEE",
		"D5 EE	5BF0",
		"D5 F0	5BF3",
		"C3 FE	5BF5",
		"D5 EF	5BF6",
		"C0 A3	5BF8",
		"BB FB	5BFA",
		"C2 D0	5BFE",
		"BC F7	5BFF",
		"C9 F5	5C01",
		"C0 EC	5C02",
		"BC CD	5C04",
		"D5 F1	5C05",
		"BE AD	5C06",
		"D5 F2	5C07",
		"D5 F3	5C08",
		"B0 D3	5C09",
		"C2 BA	5C0A",
		"BF D2	5C0B",
		"D5 F4	5C0D",
		"C6 B3	5C0E",
		"BE AE	5C0F",
		"BE AF	5C11",
		"D5 F5	5C13",
		"C0 ED	5C16",
		"BE B0	5C1A",
		"D5 F6	5C20",
		"D5 F7	5C22",
		"CC E0	5C24",
		"D5 F8	5C28",
		"B6 C6	5C2D",
		"BD A2	5C31",
		"D5 F9	5C38",
		"D5 FA	5C39",
		"BC DC	5C3A",
		"BF AC	5C3B",
		"C6 F4	5C3C",
		"BF D4	5C3D",
		"C8 F8	5C3E",
		"C7 A2	5C3F",
		"B6 C9	5C40",
		"D5 FB	5C41",
		"B5 EF	5C45",
		"D5 FC	5C46",
		"B6 FE	5C48",
		"C6 CF	5C4A",
		"B2 B0	5C4B",
		"BB D3	5C4D",
		"D5 FD	5C4E",
		"D6 A2	5C4F",
		"D6 A1	5C50",
		"B6 FD	5C51",
		"D5 FE	5C53",
		"C5 B8	5C55",
		"C2 B0	5C5E",
		"C5 CB	5C60",
		"BC C8	5C61",
		"C1 D8	5C64",
		"CD FA	5C65",
		"D6 A4	5C6C",
		"D6 A5	5C6E",
		"C6 D6	5C6F",
		"BB B3	5C71",
		"D6 A7	5C76",
		"D6 A8	5C79",
		"D6 A9	5C8C",
		"B4 F4	5C90",
		"D6 AA	5C91",
		"D6 AB	5C94",
		"B2 AC	5CA1",
		"C1 BB	5CA8",
		"B4 E4	5CA9",
		"D6 AD	5CAB",
		"CC A8	5CAC",
		"C2 D2	5CB1",
		"B3 D9	5CB3",
		"D6 AF	5CB6",
		"D6 B1	5CB7",
		"B4 DF	5CB8",
		"D6 AE	5CBB",
		"D6 B0	5CBC",
		"D6 B3	5CBE",
		"D6 B2	5CC5",
		"D6 B4	5CC7",
		"D6 B5	5CD9",
		"C6 BD	5CE0",
		"B6 AE	5CE1",
		"B2 E5	5CE8",
		"D6 B6	5CE9",
		"D6 BB	5CEA",
		"D6 B9	5CED",
		"CA F7	5CEF",
		"CA F6	5CF0",
		"C5 E7	5CF6",
		"D6 B8	5CFA",
		"BD D4	5CFB",
		"D6 B7	5CFD",
		"BF F2	5D07",
		"D6 BC	5D0B",
		"BA EA	5D0E",
		"D6 C2	5D11",
		"D6 C3	5D14",
		"D6 BD	5D15",
		"B3 B3	5D16",
		"D6 BE	5D17",
		"D6 C7	5D18",
		"D6 C6	5D19",
		"D6 C5	5D1A",
		"D6 C1	5D1B",
		"D6 C0	5D1F",
		"D6 C4	5D22",
		"CA F8	5D29",
		"D6 CB	5D4B",
		"D6 C8	5D4C",
		"D6 CA	5D4E",
		"CD F2	5D50",
		"D6 C9	5D52",
		"D6 BF	5D5C",
		"BF F3	5D69",
		"D6 CC	5D6C",
		"BA B7	5D6F",
		"D6 CD	5D73",
		"D6 CE	5D76",
		"D6 D1	5D82",
		"D6 D0	5D84",
		"D6 CF	5D87",
		"C5 E8	5D8B",
		"D6 BA	5D8C",
		"D6 D7	5D90",
		"D6 D3	5D9D",
		"D6 D2	5DA2",
		"D6 D4	5DAC",
		"D6 D5	5DAE",
		"D6 D8	5DB7",
		"CE E6	5DBA",
		"D6 D9	5DBC",
		"D6 D6	5DBD",
		"D6 DA	5DC9",
		"B4 E0	5DCC",
		"D6 DB	5DCD",
		"D6 DD	5DD2",
		"D6 DC	5DD3",
		"D6 DE	5DD6",
		"D6 DF	5DDB",
		"C0 EE	5DDD",
		"BD A3	5DDE",
		"BD E4	5DE1",
		"C1 E3	5DE3",
		"B9 A9	5DE5",
		"BA B8	5DE6",
		"B9 AA	5DE7",
		"B5 F0	5DE8",
		"D6 E0	5DEB",
		"BA B9	5DEE",
		"B8 CA	5DF1",
		"D6 E1	5DF2",
		"CC A6	5DF3",
		"C7 C3	5DF4",
		"D6 E2	5DF5",
		"B9 AB	5DF7",
		"B4 AC	5DFB",
		"C3 A7	5DFD",
		"B6 D2	5DFE",
		"BB D4	5E02",
		"C9 DB	5E03",
		"C8 C1	5E06",
		"D6 E3	5E0B",
		"B4 F5	5E0C",
		"D6 E6	5E11",
		"C4 A1	5E16",
		"D6 E5	5E19",
		"D6 E4	5E1A",
		"D6 E7	5E1B",
		"C4 EB	5E1D",
		"BF E3	5E25",
		"BB D5	5E2B",
		"C0 CA	5E2D",
		"C2 D3	5E2F",
		"B5 A2	5E30",
		"C4 A2	5E33",
		"D6 E8	5E36",
		"D6 E9	5E37",
		"BE EF	5E38",
		"CB B9	5E3D",
		"D6 EC	5E40",
		"D6 EB	5E43",
		"D6 EA	5E44",
		"C9 FD	5E45",
		"D6 F3	5E47",
		"CB DA	5E4C",
		"D6 ED	5E4E",
		"D6 EF	5E54",
		"CB EB	5E55",
		"D6 EE	5E57",
		"D6 F0	5E5F",
		"C8 A8	5E61",
		"D6 F1	5E62",
		"CA BE	5E63",
		"D6 F2	5E64",
		"B4 B3	5E72",
		"CA BF	5E73",
		"C7 AF	5E74",
		"D6 F4	5E75",
		"D6 F5	5E76",
		"B9 AC	5E78",
		"B4 B4	5E79",
		"D6 F6	5E7A",
		"B8 B8	5E7B",
		"CD C4	5E7C",
		"CD A9	5E7D",
		"B4 F6	5E7E",
		"D6 F8	5E7F",
		"C4 A3	5E81",
		"B9 AD	5E83",
		"BE B1	5E84",
		"C8 DF	5E87",
		"BE B2	5E8A",
		"BD F8	5E8F",
		"C4 EC	5E95",
		"CA F9	5E96",
		"C5 B9	5E97",
		"B9 AE	5E9A",
		"C9 DC	5E9C",
		"D6 F9	5EA0",
		"C5 D9	5EA6",
		"BA C2	5EA7",
		"B8 CB	5EAB",
		"C4 ED	5EAD",
		"B0 C3	5EB5",
		"BD EE	5EB6",
		"B9 AF	5EB7",
		"CD C7	5EB8",
		"D6 FA	5EC1",
		"D6 FB	5EC2",
		"C7 D1	5EC3",
		"D6 FC	5EC8",
		"CE F7	5EC9",
		"CF AD	5ECA",
		"D6 FE	5ECF",
		"D6 FD	5ED0",
		"B3 C7	5ED3",
		"D7 A1	5ED6",
		"D7 A4	5EDA",
		"D7 A5	5EDB",
		"D7 A3	5EDD",
		"C9 C0	5EDF",
		"BE B3	5EE0",
		"D7 A7	5EE1",
		"D7 A6	5EE2",
		"D7 A2	5EE3",
		"D7 A8	5EE8",
		"D7 A9	5EE9",
		"D7 AA	5EEC",
		"D7 AD	5EF0",
		"D7 AB	5EF1",
		"D7 AC	5EF3",
		"D7 AE	5EF4",
		"B1 E4	5EF6",
		"C4 EE	5EF7",
		"D7 AF	5EF8",
		"B7 FA	5EFA",
		"B2 F6	5EFB",
		"C7 B6	5EFC",
		"D7 B0	5EFE",
		"C6 FB	5EFF",
		"CA DB	5F01",
		"D7 B1	5F03",
		"CF AE	5F04",
		"D7 B2	5F09",
		"CA C0	5F0A",
		"D7 B5	5F0B",
		"D0 A1	5F0C",
		"D0 B1	5F0D",
		"BC B0	5F0F",
		"C6 F5	5F10",
		"D7 B6	5F11",
		"B5 DD	5F13",
		"C4 A4	5F14",
		"B0 FA	5F15",
		"D7 B7	5F16",
		"CA A6	5F17",
		"B9 B0	5F18",
		"C3 D0	5F1B",
		"C4 EF	5F1F",
		"CC EF	5F25",
		"B8 B9	5F26",
		"B8 CC	5F27",
		"D7 B8	5F29",
		"D7 B9	5F2D",
		"D7 BF	5F2F",
		"BC E5	5F31",
		"C4 A5	5F35",
		"B6 AF	5F37",
		"D7 BA	5F38",
		"C9 AB	5F3C",
		"C3 C6	5F3E",
		"D7 BB	5F41",
		"D7 BC	5F48",
		"B6 B0	5F4A",
		"D7 BD	5F4C",
		"D7 BE	5F4E",
		"D7 C0	5F51",
		"C5 F6	5F53",
		"D7 C1	5F56",
		"D7 C2	5F57",
		"D7 C3	5F59",
		"D7 B4	5F5C",
		"D7 B3	5F5D",
		"D7 C4	5F61",
		"B7 C1	5F62",
		"C9 A7	5F66",
		"BA CC	5F69",
		"C9 B7	5F6A",
		"C4 A6	5F6B",
		"C9 CB	5F6C",
		"D7 C5	5F6D",
		"BE B4	5F70",
		"B1 C6	5F71",
		"D7 C6	5F73",
		"D7 C7	5F77",
		"CC F2	5F79",
		"C8 E0	5F7C",
		"D7 CA	5F7F",
		"B1 FD	5F80",
		"C0 AC	5F81",
		"D7 C9	5F82",
		"D7 C8	5F83",
		"B7 C2	5F84",
		"C2 D4	5F85",
		"D7 CE	5F87",
		"D7 CC	5F88",
		"D7 CB	5F8A",
		"CE A7	5F8B",
		"B8 E5	5F8C",
		"BD F9	5F90",
		"D7 CD	5F91",
		"C5 CC	5F92",
		"BD BE	5F93",
		"C6 C0	5F97",
		"D7 D1	5F98",
		"D7 D0	5F99",
		"D7 CF	5F9E",
		"D7 D2	5FA0",
		"B8 E6	5FA1",
		"D7 D3	5FA8",
		"C9 FC	5FA9",
		"BD DB	5FAA",
		"D7 D4	5FAD",
		"C8 F9	5FAE",
		"C6 C1	5FB3",
		"C4 A7	5FB4",
		"C5 B0	5FB9",
		"D7 D5	5FBC",
		"B5 AB	5FBD",
		"BF B4	5FC3",
		"C9 AC	5FC5",
		"B4 F7	5FCC",
		"C7 A6	5FCD",
		"D7 D6	5FD6",
		"BB D6	5FD7",
		"CB BA	5FD8",
		"CB BB	5FD9",
		"B1 FE	5FDC",
		"D7 DB	5FDD",
		"C3 E9	5FE0",
		"D7 D8	5FE4",
		"B2 F7	5FEB",
		"D8 AD	5FF0",
		"D7 DA	5FF1",
		"C7 B0	5FF5",
		"D7 D9	5FF8",
		"D7 D7	5FFB",
		"B9 FA	5FFD",
		"D7 DD	5FFF",
		"D7 E3	600E",
		"D7 E9	600F",
		"D7 E1	6010",
		"C5 DC	6012",
		"D7 E6	6015",
		"C9 DD	6016",
		"D7 E0	6019",
		"D7 E5	601B",
		"CE E7	601C",
		"BB D7	601D",
		"C2 D5	6020",
		"D7 DE	6021",
		"B5 DE	6025",
		"D7 E8	6026",
		"C0 AD	6027",
		"B1 E5	6028",
		"D7 E2	6029",
		"B2 F8	602A",
		"D7 E7	602B",
		"B6 B1	602F",
		"D7 E4	6031",
		"D7 EA	603A",
		"D7 EC	6041",
		"D7 F6	6042",
		"D7 F4	6043",
		"D7 F1	6046",
		"D7 F0	604A",
		"CE F8	604B",
		"D7 F2	604D",
		"B6 B2	6050",
		"B9 B1	6052",
		"BD FA	6055",
		"D7 F9	6059",
		"D7 EB	605A",
		"D7 EF	605F",
		"D7 DF	6060",
		"B2 FA	6062",
		"D7 F3	6063",
		"D7 F5	6064",
		"C3 D1	6065",
		"BA A8	6068",
		"B2 B8	6069",
		"D7 ED	606A",
		"D7 F8	606B",
		"D7 F7	606C",
		"B6 B3	606D",
		"C2 A9	606F",
		"B3 E6	6070",
		"B7 C3	6075",
		"D7 EE	6077",
		"D7 FA	6081",
		"D7 FD	6083",
		"D8 A1	6084",
		"BC BD	6089",
		"D8 A7	608B",
		"C4 F0	608C",
		"D7 FB	608D",
		"D8 A5	6092",
		"B2 F9	6094",
		"D8 A3	6096",
		"D8 A4	6097",
		"D7 FE	609A",
		"D8 A2	609B",
		"B8 E7	609F",
		"CD AA	60A0",
		"B4 B5	60A3",
		"B1 D9	60A6",
		"D8 A6	60A7",
		"C7 BA	60A9",
		"B0 AD	60AA",
		"C8 E1	60B2",
		"D7 DC	60B3",
		"D8 AC	60B4",
		"D8 B0	60B5",
		"CC E5	60B6",
		"D8 A9	60B8",
		"C5 E9	60BC",
		"D8 AE	60BD",
		"BE F0	60C5",
		"D8 AF	60C6",
		"C6 D7	60C7",
		"CF C7	60D1",
		"D8 AB	60D3",
		"D8 B1	60D8",
		"B9 FB	60DA",
		"C0 CB	60DC",
		"B0 D4	60DF",
		"D8 AA	60E0",
		"D8 A8	60E1",
		"C1 DA	60E3",
		"D7 FC	60E7",
		"BB B4	60E8",
		"C2 C6	60F0",
		"D8 BD	60F1",
		"C1 DB	60F3",
		"D8 B8	60F4",
		"D8 B5	60F6",
		"D8 B6	60F7",
		"BC E6	60F9",
		"D8 B9	60FA",
		"D8 BC	60FB",
		"D8 B7	6100",
		"BD A5	6101",
		"D8 BA	6103",
		"D8 B4	6106",
		"CC FC	6108",
		"CC FB	6109",
		"D8 BE	610D",
		"D8 BF	610E",
		"B0 D5	610F",
		"D8 B3	6115",
		"B6 F2	611A",
		"B0 A6	611B",
		"B4 B6	611F",
		"D8 BB	6121",
		"D8 C3	6127",
		"D8 C2	6128",
		"D8 C7	612C",
		"D8 C8	6134",
		"D8 C6	613C",
		"D8 C9	613D",
		"D8 C1	613E",
		"D8 C5	613F",
		"D8 CA	6142",
		"D8 CB	6144",
		"D8 C0	6147",
		"BB FC	6148",
		"D8 C4	614A",
		"C2 D6	614B",
		"B9 B2	614C",
		"D8 B2	614D",
		"BF B5	614E",
		"D8 D8	6153",
		"CA E9	6155",
		"D8 CE	6158",
		"D8 CF	6159",
		"D8 D0	615A",
		"D8 D7	615D",
		"D8 D6	615F",
		"CB FD	6162",
		"B4 B7	6163",
		"D8 D4	6165",
		"B7 C5	6167",
		"B3 B4	6168",
		"D8 D1	616B",
		"CE B8	616E",
		"D8 D3	616F",
		"B0 D6	6170",
		"D8 D5	6171",
		"D8 CC	6173",
		"D8 D2	6174",
		"D8 D9	6175",
		"B7 C4	6176",
		"D8 CD	6177",
		"CD DD	617E",
		"CD AB	6182",
		"D8 DC	6187",
		"D8 E0	618A",
		"C1 FE	618E",
		"CE F9	6190",
		"D8 E1	6191",
		"D8 DE	6194",
		"D8 DB	6196",
		"D8 DA	6199",
		"D8 DF	619A",
		"CA B0	61A4",
		"C6 B4	61A7",
		"B7 C6	61A9",
		"D8 E2	61AB",
		"D8 DD	61AC",
		"D8 E3	61AE",
		"B7 FB	61B2",
		"B2 B1	61B6",
		"D8 EB	61BA",
		"B4 B8	61BE",
		"D8 E9	61C3",
		"D8 EA	61C6",
		"BA A9	61C7",
		"D8 E8	61C8",
		"D8 E6	61C9",
		"D8 E5	61CA",
		"D8 EC	61CB",
		"D8 E4	61CC",
		"D8 EE	61CD",
		"B2 FB	61D0",
		"D8 F0	61E3",
		"D8 EF	61E6",
		"C4 A8	61F2",
		"D8 F3	61F4",
		"D8 F1	61F6",
		"D8 E7	61F7",
		"B7 FC	61F8",
		"D8 F2	61FA",
		"D8 F6	61FC",
		"D8 F5	61FD",
		"D8 F7	61FE",
		"D8 F4	61FF",
		"D8 F8	6200",
		"D8 F9	6208",
		"D8 FA	6209",
		"CA EA	620A",
		"D8 FC	620C",
		"D8 FB	620D",
		"BD BF	620E",
		"C0 AE	6210",
		"B2 E6	6211",
		"B2 FC	6212",
		"D8 FD	6214",
		"B0 BF	6216",
		"C0 CC	621A",
		"D8 FE	621B",
		"EC C3	621D",
		"D9 A1	621E",
		"B7 E1	621F",
		"D9 A2	6221",
		"C0 EF	6226",
		"D9 A3	622A",
		"D9 A4	622E",
		"B5 BA	622F",
		"D9 A5	6230",
		"D9 A6	6232",
		"D9 A7	6233",
		"C2 D7	6234",
		"B8 CD	6238",
		"CC E1	623B",
		"CB BC	623F",
		"BD EA	6240",
		"D9 A8	6241",
		"C0 F0	6247",
		"EE BD	6248",
		"C8 E2	6249",
		"BC EA	624B",
		"BA CD	624D",
		"D9 A9	624E",
		"C2 C7	6253",
		"CA A7	6255",
		"C2 F1	6258",
		"D9 AC	625B",
		"D9 AA	625E",
		"D9 AD	6260",
		"D9 AB	6263",
		"D9 AE	6268",
		"CA B1	626E",
		"B0 B7	6271",
		"C9 DE	6276",
		"C8 E3	6279",
		"D9 AF	627C",
		"D9 B2	627E",
		"BE B5	627F",
		"B5 BB	6280",
		"D9 B0	6282",
		"D9 B7	6283",
		"BE B6	6284",
		"D9 B1	6289",
		"C7 C4	628A",
		"CD DE	6291",
		"D9 B3	6292",
		"D9 B4	6293",
		"D9 B8	6294",
		"C5 EA	6295",
		"D9 B5	6296",
		"B9 B3	6297",
		"C0 DE	6298",
		"D9 C6	629B",
		"C8 B4	629C",
		"C2 F2	629E",
		"C8 E4	62AB",
		"DA AD	62AC",
		"CA FA	62B1",
		"C4 F1	62B5",
		"CB F5	62B9",
		"D9 BB	62BB",
		"B2 A1	62BC",
		"C3 EA	62BD",
		"D9 C4	62C2",
		"C3 B4	62C5",
		"D9 BE	62C6",
		"D9 C5	62C7",
		"D9 C0	62C8",
		"D9 C7	62C9",
		"D9 C3	62CA",
		"D9 C2	62CC",
		"C7 EF	62CD",
		"D9 BC	62CF",
		"B2 FD	62D0",
		"D9 BA	62D1",
		"B5 F1	62D2",
		"C2 F3	62D3",
		"D9 B6	62D4",
		"D9 B9	62D7",
		"B9 B4	62D8",
		"C0 DB	62D9",
		"BE B7	62DB",
		"D9 C1	62DC",
		"C7 D2	62DD",
		"B5 F2	62E0",
		"B3 C8	62E1",
		"B3 E7	62EC",
		"BF A1	62ED",
		"D9 C9	62EE",
		"D9 CE	62EF",
		"D9 CA	62F1",
		"B7 FD	62F3",
		"D9 CF	62F5",
		"BB A2	62F6",
		"B9 E9	62F7",
		"BD A6	62FE",
		"D9 BD	62FF",
		"BB FD	6301",
		"D9 CC	6302",
		"BB D8	6307",
		"D9 CD	6308",
		"B0 C4	6309",
		"D9 C8	630C",
		"C4 A9	6311",
		"B5 F3	6319",
		"B6 B4	631F",
		"D9 CB	6327",
		"B0 A7	6328",
		"BA C3	632B",
		"BF B6	632F",
		"C4 F2	633A",
		"C8 D4	633D",
		"D9 D1	633E",
		"C1 DE	633F",
		"C2 AA	6349",
		"BB AB	634C",
		"D9 D2	634D",
		"D9 D4	634F",
		"D9 D0	6350",
		"CA E1	6355",
		"C4 BD	6357",
		"C1 DC	635C",
		"CA FB	6367",
		"BC CE	6368",
		"D9 E0	6369",
		"D9 DF	636B",
		"BF F8	636E",
		"B7 FE	6372",
		"D9 D9	6376",
		"BE B9	6377",
		"C6 E8	637A",
		"C7 B1	637B",
		"D9 D7	6380",
		"C1 DD	6383",
		"BC F8	6388",
		"D9 DC	6389",
		"BE B8	638C",
		"D9 D6	638E",
		"D9 DB	638F",
		"C7 D3	6392",
		"D9 D5	6396",
		"B7 A1	6398",
		"B3 DD	639B",
		"D9 DD	639F",
		"CE AB	63A0",
		"BA CE	63A1",
		"C3 B5	63A2",
		"D9 DA	63A3",
		"C0 DC	63A5",
		"B9 B5	63A7",
		"BF E4	63A8",
		"B1 E6	63A9",
		"C1 BC	63AA",
		"D9 D8	63AB",
		"B5 C5	63AC",
		"B7 C7	63B2",
		"C4 CF	63B4",
		"D9 DE	63B5",
		"C1 DF	63BB",
		"D9 E1	63BE",
		"D9 E3	63C0",
		"C2 B7	63C3",
		"D9 E9	63C4",
		"D9 E4	63C6",
		"D9 E6	63C9",
		"C9 C1	63CF",
		"C4 F3	63D0",
		"D9 E7	63D2",
		"CD AC	63D6",
		"CD C8	63DA",
		"B4 B9	63DB",
		"B0 AE	63E1",
		"D9 E5	63E3",
		"D9 E2	63E9",
		"B4 F8	63EE",
		"B1 E7	63F4",
		"D9 E8	63F6",
		"CD C9	63FA",
		"D9 EC	6406",
		"C2 BB	640D",
		"D9 F3	640F",
		"D9 ED	6413",
		"D9 EA	6416",
		"D9 F1	6417",
		"D9 D3	641C",
		"D9 EE	6426",
		"D9 F2	6428",
		"C8 C2	642C",
		"C5 EB	642D",
		"D9 EB	6434",
		"D9 EF	6436",
		"B7 C8	643A",
		"BA F1	643E",
		"C0 DD	6442",
		"D9 F7	644E",
		"C5 A6	6458",
		"D9 F4	6467",
		"CB E0	6469",
		"D9 F5	646F",
		"D9 F6	6476",
		"CC CE	6478",
		"C0 A2	647A",
		"B7 E2	6483",
		"D9 FD	6488",
		"BB B5	6492",
		"D9 FA	6493",
		"D9 F9	6495",
		"C7 B2	649A",
		"C6 B5	649E",
		"C5 B1	64A4",
		"D9 FB	64A5",
		"D9 FC	64A9",
		"C9 EF	64AB",
		"C7 C5	64AD",
		"BB A3	64AE",
		"C0 F1	64B0",
		"CB D0	64B2",
		"B3 C9	64B9",
		"DA A5	64BB",
		"D9 FE	64BC",
		"CD CA	64C1",
		"DA A7	64C2",
		"DA A3	64C5",
		"DA A4	64C7",
		"C1 E0	64CD",
		"DA A2	64D2",
		"D9 BF	64D4",
		"DA A6	64D8",
		"DA A1	64DA",
		"DA AB	64E0",
		"DA AC	64E1",
		"C5 A7	64E2",
		"DA AE	64E3",
		"BB A4	64E6",
		"DA A9	64E7",
		"B5 BC	64EC",
		"DA AF	64EF",
		"DA A8	64F1",
		"DA B3	64F2",
		"DA B2	64F4",
		"DA B1	64F6",
		"DA B4	64FA",
		"DA B6	64FD",
		"BE F1	64FE",
		"DA B5	6500",
		"DA B9	6505",
		"DA B7	6518",
		"DA B8	651C",
		"D9 F0	651D",
		"DA BB	6523",
		"DA BA	6524",
		"D9 F8	652A",
		"DA BC	652B",
		"DA B0	652C",
		"BB D9	652F",
		"DA BD	6534",
		"DA BE	6535",
		"DA C0	6536",
		"DA BF	6537",
		"DA C1	6538",
		"B2 FE	6539",
		"B9 B6	653B",
		"CA FC	653E",
		"C0 AF	653F",
		"B8 CE	6545",
		"DA C3	6548",
		"DA C6	654D",
		"C9 D2	654F",
		"B5 DF	6551",
		"DA C5	6555",
		"DA C4	6556",
		"C7 D4	6557",
		"DA C7	6558",
		"B6 B5	6559",
		"DA C9	655D",
		"DA C8	655E",
		"B4 BA	6562",
		"BB B6	6563",
		"C6 D8	6566",
		"B7 C9	656C",
		"BF F4	6570",
		"DA CA	6572",
		"C0 B0	6574",
		"C5 A8	6575",
		"C9 DF	6577",
		"DA CB	6578",
		"DA CC	6582",
		"DA CD	6583",
		"CA B8	6587",
		"D5 DD	6588",
		"C0 C6	6589",
		"C9 CC	658C",
		"BA D8	658E",
		"C8 E5	6590",
		"C8 C3	6591",
		"C5 CD	6597",
		"CE C1	6599",
		"DA CF	659B",
		"BC D0	659C",
		"DA D0	659F",
		"B0 B6	65A1",
		"B6 D4	65A4",
		"C0 CD	65A5",
		"C9 E0	65A7",
		"DA D1	65AB",
		"BB C2	65AC",
		"C3 C7	65AD",
		"BB DB	65AF",
		"BF B7	65B0",
		"DA D2	65B7",
		"CA FD	65B9",
		"B1 F7	65BC",
		"BB DC	65BD",
		"DA D5	65C1",
		"DA D3	65C3",
		"DA D6	65C4",
		"CE B9	65C5",
		"DA D4	65C6",
		"C0 FB	65CB",
		"DA D7	65CC",
		"C2 B2	65CF",
		"DA D8	65D2",
		"B4 FA	65D7",
		"DA DA	65D9",
		"DA D9	65DB",
		"DA DB	65E0",
		"DA DC	65E1",
		"B4 FB	65E2",
		"C6 FC	65E5",
		"C3 B6	65E6",
		"B5 EC	65E7",
		"BB DD	65E8",
		"C1 E1	65E9",
		"BD DC	65EC",
		"B0 B0	65ED",
		"DA DD	65F1",
		"B2 A2	65FA",
		"DA E1	65FB",
		"B9 B7	6602",
		"DA E0	6603",
		"BA AB	6606",
		"BE BA	6607",
		"DA DF	660A",
		"BE BB	660C",
		"CC C0	660E",
		"BA AA	660F",
		"B0 D7	6613",
		"C0 CE	6614",
		"DA E6	661C",
		"C0 B1	661F",
		"B1 C7	6620",
		"BD D5	6625",
		"CB E6	6627",
		"BA F2	6628",
		"BE BC	662D",
		"C0 A7	662F",
		"DA E5	6634",
		"DA E3	6635",
		"DA E4	6636",
		"C3 EB	663C",
		"DB A6	663F",
		"DA EA	6641",
		"BB FE	6642",
		"B9 B8	6643",
		"DA E8	6644",
		"DA E9	6649",
		"BF B8	664B",
		"DA E7	664F",
		"BB AF	6652",
		"DA EC	665D",
		"DA EB	665E",
		"DA F0	665F",
		"DA F1	6662",
		"DA ED	6664",
		"B3 A2	6666",
		"DA EE	6667",
		"DA EF	6668",
		"C8 D5	6669",
		"C9 E1	666E",
		"B7 CA	666F",
		"DA F2	6670",
		"C0 B2	6674",
		"BE BD	6676",
		"C3 D2	667A",
		"B6 C7	6681",
		"DA F3	6683",
		"DA F7	6684",
		"B2 CB	6687",
		"DA F4	6688",
		"DA F6	6689",
		"DA F5	668E",
		"BD EB	6691",
		"C3 C8	6696",
		"B0 C5	6697",
		"DA F8	6698",
		"DA F9	669D",
		"C4 AA	66A2",
		"CE F1	66A6",
		"BB C3	66AB",
		"CA EB	66AE",
		"CB BD	66B4",
		"DB A2	66B8",
		"DA FB	66B9",
		"DA FE	66BC",
		"DA FD	66BE",
		"DA FA	66C1",
		"DB A1	66C4",
		"C6 DE	66C7",
		"DA FC	66C9",
		"DB A3	66D6",
		"BD EC	66D9",
		"DB A4	66DA",
		"CD CB	66DC",
		"C7 F8	66DD",
		"DB A5	66E0",
		"DB A7	66E6",
		"DB A8	66E9",
		"DB A9	66F0",
		"B6 CA	66F2",
		"B1 C8	66F3",
		"B9 B9	66F4",
		"DB AA	66F5",
		"DB AB	66F7",
		"BD F1	66F8",
		"C1 E2	66F9",
		"D2 D8	66FC",
		"C1 BE	66FD",
		"C1 BD	66FE",
		"C2 D8	66FF",
		"BA C7	6700",
		"D0 F2	6703",
		"B7 EE	6708",
		"CD AD	6709",
		"CA FE	670B",
		"C9 FE	670D",
		"DB AC	670F",
		"BA F3	6714",
		"C4 BF	6715",
		"DB AD	6716",
		"CF AF	6717",
		"CB BE	671B",
		"C4 AB	671D",
		"DB AE	671E",
		"B4 FC	671F",
		"DB AF	6726",
		"DB B0	6727",
		"CC DA	6728",
		"CC A4	672A",
		"CB F6	672B",
		"CB DC	672C",
		"BB A5	672D",
		"DB B2	672E",
		"BC EB	6731",
		"CB D1	6734",
		"DB B4	6736",
		"DB B7	6737",
		"DB B6	6738",
		"B4 F9	673A",
		"B5 E0	673D",
		"DB B3	673F",
		"DB B5	6741",
		"DB B8	6746",
		"BF F9	6749",
		"CD FB	674E",
		"B0 C9	674F",
		"BA E0	6750",
		"C2 BC	6751",
		"BC DD	6753",
		"BE F3	6756",
		"DB BB	6759",
		"C5 CE	675C",
		"DB B9	675E",
		"C2 AB	675F",
		"DB BA	6760",
		"BE F2	6761",
		"CC DD	6762",
		"DB BC	6763",
		"DB BD	6764",
		"CD E8	6765",
		"DB C2	676A",
		"B9 BA	676D",
		"C7 D5	676F",
		"DB BF	6770",
		"C5 EC	6771",
		"DA DE	6772",
		"DA E2	6773",
		"B5 CF	6775",
		"C7 C7	6777",
		"DB C1	677C",
		"BE BE	677E",
		"C8 C4	677F",
		"DB C7	6785",
		"C8 FA	6787",
		"DB BE	6789",
		"DB C4	678B",
		"DB C3	678C",
		"C0 CF	6790",
		"CB ED	6795",
		"CE D3	6797",
		"CB E7	679A",
		"B2 CC	679C",
		"BB DE	679D",
		"CF C8	67A0",
		"DB C6	67A1",
		"BF F5	67A2",
		"DB C5	67A6",
		"DB C0	67A9",
		"B8 CF	67AF",
		"DB CC	67B3",
		"DB CA	67B4",
		"B2 CD	67B6",
		"DB C8	67B7",
		"DB CE	67B8",
		"DB D4	67B9",
		"C2 C8	67C1",
		"CA C1	67C4",
		"DB D6	67C6",
		"C9 A2	67CA",
		"DB D5	67CE",
		"C7 F0	67CF",
		"CB BF	67D0",
		"B4 BB	67D1",
		"C0 F7	67D3",
		"BD C0	67D4",
		"C4 D3	67D8",
		"CD AE	67DA",
		"DB D1	67DD",
		"DB D0	67DE",
		"DB D2	67E2",
		"DB CF	67E4",
		"DB D7	67E7",
		"DB CD	67E9",
		"DB CB	67EC",
		"DB D3	67EE",
		"DB C9	67EF",
		"C3 EC	67F1",
		"CC F8	67F3",
		"BC C6	67F4",
		"BA F4	67F5",
		"BA BA	67FB",
		"CB EF	67FE",
		"B3 C1	67FF",
		"C4 CE	6802",
		"C6 CA	6803",
		"B1 C9	6804",
		"C0 F2	6813",
		"C0 B4	6816",
		"B7 AA	6817",
		"DB D9	681E",
		"B9 BB	6821",
		"B3 FC	6822",
		"DB DB	6829",
		"B3 F4	682A",
		"DB E1	682B",
		"DB DE	6832",
		"C0 F3	6834",
		"B3 CB	6838",
		"BA AC	6839",
		"B3 CA	683C",
		"BA CF	683D",
		"DB DC	6840",
		"B7 E5	6841",
		"B7 CB	6842",
		"C5 ED	6843",
		"DB DA	6846",
		"B0 C6	6848",
		"DB DD	684D",
		"DB DF	684E",
		"B6 CD	6850",
		"B7 AC	6851",
		"B4 BC	6853",
		"B5 CB	6854",
		"DB E2	6859",
		"BA F9	685C",
		"CB F1	685D",
		"BB B7	685F",
		"DB E3	6863",
		"C9 B0	6867",
		"DB EF	6874",
		"B2 B3	6876",
		"DB E4	6877",
		"DB F5	687E",
		"DB E5	687F",
		"CE C2	6881",
		"DB EC	6883",
		"C7 DF	6885",
		"DB F4	688D",
		"DB E7	688F",
		"B0 B4	6893",
		"DB E9	6894",
		"B9 BC	6897",
		"DB EB	689B",
		"DB EA	689D",
		"DB E6	689F",
		"DB F1	68A0",
		"BE BF	68A2",
		"D4 ED	68A6",
		"B8 E8	68A7",
		"CD FC	68A8",
		"DB E8	68AD",
		"C4 F4	68AF",
		"B3 A3	68B0",
		"BA AD	68B1",
		"DB E0	68B3",
		"DB F0	68B5",
		"B3 E1	68B6",
		"DB EE	68B9",
		"DB F2	68BA",
		"C5 EE	68BC",
		"B4 FE	68C4",
		"DC B2	68C6",
		"CC C9	68C9",
		"DB F7	68CA",
		"B4 FD	68CB",
		"DB FE	68CD",
		"CB C0	68D2",
		"DC A1	68D4",
		"DC A3	68D5",
		"DC A7	68D7",
		"DB F9	68D8",
		"C3 AA	68DA",
		"C5 EF	68DF",
		"DC AB	68E0",
		"DB FC	68E1",
		"DC A8	68E3",
		"DC A2	68E7",
		"BF B9	68EE",
		"DC AC	68EF",
		"C0 B3	68F2",
		"DC AA	68F9",
		"B4 BD	68FA",
		"CF D0	6900",
		"DB F6	6901",
		"DC A6	6904",
		"B0 D8	6905",
		"DB F8	6908",
		"CC BA	690B",
		"DB FD	690C",
		"BF A2	690D",
		"C4 C7	690E",
		"DB F3	690F",
		"DC A5	6912",
		"BF FA	6919",
		"DC AF	691A",
		"B3 F1	691B",
		"B8 A1	691C",
		"DC B1	6921",
		"DB FA	6922",
		"DC B0	6923",
		"DC A9	6925",
		"DB FB	6926",
		"DC AD	6928",
		"DC AE	692A",
		"DC BF	6930",
		"C6 CE	6934",
		"DC A4	6936",
		"DC BB	6939",
		"DC BD	693D",
		"C4 D8	693F",
		"CD CC	694A",
		"C9 F6	6953",
		"DC B8	6954",
		"C2 CA	6955",
		"DC BE	6959",
		"C1 BF	695A",
		"DC B5	695C",
		"DC C2	695D",
		"DC C1	695E",
		"C6 EF	6960",
		"DC C0	6961",
		"C6 EA	6962",
		"DC C4	696A",
		"DC B7	696B",
		"B6 C8	696D",
		"DC BA	696E",
		"BD DD	696F",
		"C7 E0	6973",
		"DC BC	6974",
		"B6 CB	6975",
		"DC B4	6977",
		"DC B6	6978",
		"DC B3	6979",
		"CF B0	697C",
		"B3 DA	697D",
		"DC B9	697E",
		"DC C3	6981",
		"B3 B5	6982",
		"BA E7	698A",
		"B1 DD	698E",
		"DC D4	6991",
		"CF B1	6994",
		"DC D7	6995",
		"BF BA	699B",
		"DC D6	699C",
		"DC D5	69A0",
		"DC D2	69A7",
		"DC C6	69AE",
		"DC E3	69B1",
		"DC C5	69B2",
		"DC D8	69B4",
		"DC D0	69BB",
		"DC CB	69BE",
		"DC C8	69BF",
		"DC C9	69C1",
		"DC D1	69C3",
		"F4 A2	69C7",
		"DC CE	69CA",
		"B9 BD	69CB",
		"C4 C8	69CC",
		"C1 E4	69CD",
		"DC CC	69CE",
		"DC C7	69D0",
		"DC CA	69D3",
		"CD CD	69D8",
		"CB EA	69D9",
		"DC CF	69DD",
		"DC D9	69DE",
		"DC E1	69E7",
		"DC DA	69E8",
		"DC E7	69EB",
		"DC E5	69ED",
		"DC E0	69F2",
		"DC DF	69F9",
		"C4 D0	69FB",
		"C1 E5	69FD",
		"DC DD	69FF",
		"DC DB	6A02",
		"DC E2	6A05",
		"DC E8	6A0A",
		"C8 F5	6A0B",
		"DC EE	6A0C",
		"DC E9	6A12",
		"DC EC	6A13",
		"DC E6	6A14",
		"C3 F4	6A17",
		"C9 B8	6A19",
		"DC DC	6A1B",
		"DC E4	6A1E",
		"BE C0	6A1F",
		"CC CF	6A21",
		"DC F8	6A22",
		"DC EB	6A23",
		"B8 A2	6A29",
		"B2 A3	6A2A",
		"B3 DF	6A2B",
		"DC D3	6A2E",
		"BE C1	6A35",
		"DC F0	6A36",
		"DC F7	6A38",
		"BC F9	6A39",
		"B3 F2	6A3A",
		"C3 AE	6A3D",
		"DC ED	6A44",
		"DC F2	6A47",
		"DC F6	6A48",
		"B6 B6	6A4B",
		"B5 CC	6A58",
		"DC F4	6A59",
		"B5 A1	6A5F",
		"C6 CB	6A61",
		"DC F3	6A62",
		"DC F5	6A66",
		"DC EF	6A72",
		"DC F1	6A78",
		"B3 E0	6A7F",
		"C3 C9	6A80",
		"DC FC	6A84",
		"DC FA	6A8D",
		"B8 E9	6A8E",
		"DC F9	6A90",
		"DD A1	6A97",
		"DB D8	6A9C",
		"DC FB	6AA0",
		"DC FD	6AA2",
		"DC FE	6AA3",
		"DD AC	6AAA",
		"DD A8	6AAC",
		"DB ED	6AAE",
		"DD A7	6AB3",
		"DD A6	6AB8",
		"DD A3	6ABB",
		"DC EA	6AC1",
		"DD A5	6AC2",
		"DD A4	6AC3",
		"DD AA	6AD1",
		"CF A6	6AD3",
		"DD AD	6ADA",
		"B6 FB	6ADB",
		"DD A9	6ADE",
		"DD AB	6ADF",
		"C8 A7	6AE8",
		"DD AE	6AEA",
		"DD B2	6AFA",
		"DD AF	6AFB",
		"CD F3	6B04",
		"DD B0	6B05",
		"DC DE	6B0A",
		"DD B3	6B12",
		"DD B4	6B16",
		"B1 B5	6B1D",
		"DD B6	6B1F",
		"B7 E7	6B20",
		"BC A1	6B21",
		"B6 D5	6B23",
		"B2 A4	6B27",
		"CD DF	6B32",
		"DD B8	6B37",
		"DD B7	6B38",
		"DD BA	6B39",
		"B5 BD	6B3A",
		"B6 D6	6B3D",
		"B4 BE	6B3E",
		"DD BD	6B43",
		"DD BC	6B47",
		"DD BE	6B49",
		"B2 CE	6B4C",
		"C3 B7	6B4E",
		"DD BF	6B50",
		"B4 BF	6B53",
		"DD C1	6B54",
		"DD C0	6B59",
		"DD C2	6B5B",
		"DD C3	6B5F",
		"DD C4	6B61",
		"BB DF	6B62",
		"C0 B5	6B63",
		"BA A1	6B64",
		"C9 F0	6B66",
		"CA E2	6B69",
		"CF C4	6B6A",
		"BB F5	6B6F",
		"BA D0	6B73",
		"CE F2	6B74",
		"DD C5	6B78",
		"DD C6	6B79",
		"BB E0	6B7B",
		"DD C7	6B7F",
		"DD C8	6B80",
		"DD CA	6B83",
		"DD C9	6B84",
		"CB D8	6B86",
		"BD DE	6B89",
		"BC EC	6B8A",
		"BB C4	6B8B",
		"DD CB	6B8D",
		"DD CD	6B95",
		"BF A3	6B96",
		"DD CC	6B98",
		"DD CE	6B9E",
		"DD CF	6BA4",
		"DD D0	6BAA",
		"DD D1	6BAB",
		"DD D2	6BAF",
		"DD D4	6BB1",
		"DD D3	6BB2",
		"DD D5	6BB3",
		"B2 A5	6BB4",
		"C3 CA	6BB5",
		"DD D6	6BB7",
		"BB A6	6BBA",
		"B3 CC	6BBB",
		"DD D7	6BBC",
		"C5 C2	6BBF",
		"D4 CC	6BC0",
		"B5 A3	6BC5",
		"DD D8	6BC6",
		"DD D9	6BCB",
		"CA EC	6BCD",
		"CB E8	6BCE",
		"C6 C7	6BD2",
		"DD DA	6BD3",
		"C8 E6	6BD4",
		"C8 FB	6BD8",
		"CC D3	6BDB",
		"DD DB	6BDF",
		"DD DD	6BEB",
		"DD DC	6BEC",
		"DD DF	6BEF",
		"DD DE	6BF3",
		"DD E1	6C08",
		"BB E1	6C0F",
		"CC B1	6C11",
		"DD E2	6C13",
		"DD E3	6C14",
		"B5 A4	6C17",
		"DD E4	6C1B",
		"DD E6	6C23",
		"DD E5	6C24",
		"BF E5	6C34",
		"C9 B9	6C37",
		"B1 CA	6C38",
		"C8 C5	6C3E",
		"C4 F5	6C40",
		"BD C1	6C41",
		"B5 E1	6C42",
		"C8 C6	6C4E",
		"BC AE	6C50",
		"DD E8	6C55",
		"B4 C0	6C57",
		"B1 F8	6C5A",
		"C6 F2	6C5D",
		"DD E7	6C5E",
		"B9 BE	6C5F",
		"C3 D3	6C60",
		"DD E9	6C62",
		"DD F1	6C68",
		"DD EA	6C6A",
		"C2 C1	6C70",
		"B5 E2	6C72",
		"DD F2	6C73",
		"B7 E8	6C7A",
		"B5 A5	6C7D",
		"DD F0	6C7E",
		"DD EE	6C81",
		"DD EB	6C82",
		"CD E0	6C83",
		"C4 C0	6C88",
		"C6 D9	6C8C",
		"DD EC	6C8D",
		"DD F4	6C90",
		"DD F3	6C92",
		"B7 A3	6C93",
		"B2 AD	6C96",
		"BA BB	6C99",
		"DD ED	6C9A",
		"DD EF	6C9B",
		"CB D7	6CA1",
		"C2 F4	6CA2",
		"CB F7	6CAB",
		"DD FC	6CAE",
		"DD FD	6CB1",
		"B2 CF	6CB3",
		"CA A8	6CB8",
		"CC FD	6CB9",
		"DE A1	6CBA",
		"BC A3	6CBB",
		"BE C2	6CBC",
		"DD F8	6CBD",
		"DD FE	6CBE",
		"B1 E8	6CBF",
		"B6 B7	6CC1",
		"DD F5	6CC4",
		"DD FA	6CC5",
		"C0 F4	6CC9",
		"C7 F1	6CCA",
		"C8 E7	6CCC",
		"DD F7	6CD3",
		"CB A1	6CD5",
		"DD F9	6CD7",
		"DE A4	6CD9",
		"DE A2	6CDB",
		"DD FB	6CDD",
		"CB A2	6CE1",
		"C7 C8	6CE2",
		"B5 E3	6CE3",
		"C5 A5	6CE5",
		"C3 ED	6CE8",
		"DE A5	6CEA",
		"DE A3	6CEF",
		"C2 D9	6CF0",
		"DD F6	6CF1",
		"B1 CB	6CF3",
		"CD CE	6D0B",
		"DE B0	6D0C",
		"DE AF	6D12",
		"C0 F6	6D17",
		"DE AC	6D19",
		"CD EC	6D1B",
		"C6 B6	6D1E",
		"DE A6	6D1F",
		"C4 C5	6D25",
		"B1 CC	6D29",
		"B9 BF	6D2A",
		"DE A9	6D2B",
		"BD A7	6D32",
		"DE AE	6D33",
		"DE AD	6D35",
		"DE A8	6D36",
		"DE AB	6D38",
		"B3 E8	6D3B",
		"DE AA	6D3D",
		"C7 C9	6D3E",
		"CE AE	6D41",
		"BE F4	6D44",
		"C0 F5	6D45",
		"DE B6	6D59",
		"DE B4	6D5A",
		"C9 CD	6D5C",
		"DE B1	6D63",
		"DE B3	6D64",
		"B1 BA	6D66",
		"B9 C0	6D69",
		"CF B2	6D6A",
		"B3 BD	6D6C",
		"C9 E2	6D6E",
		"CD E1	6D74",
		"B3 A4	6D77",
		"BF BB	6D78",
		"DE B5	6D79",
		"DE BA	6D85",
		"BE C3	6D88",
		"CD B0	6D8C",
		"DE B7	6D8E",
		"DE B2	6D93",
		"DE B8	6D95",
		"CE DE	6D99",
		"C5 F3	6D9B",
		"C6 C2	6D9C",
		"B3 B6	6DAF",
		"B1 D5	6DB2",
		"DE BE	6DB5",
		"DE C1	6DB8",
		"CE C3	6DBC",
		"CD E4	6DC0",
		"DE C8	6DC5",
		"DE C2	6DC6",
		"DE BF	6DC7",
		"CE D4	6DCB",
		"DE C5	6DCC",
		"BD CA	6DD1",
		"DE C7	6DD2",
		"DE CC	6DD5",
		"C5 F1	6DD8",
		"DE CA	6DD9",
		"DE C4	6DDE",
		"C3 B8	6DE1",
		"DE CB	6DE4",
		"DE C0	6DE6",
		"DE C6	6DE8",
		"DE CD	6DEA",
		"B0 FC	6DEB",
		"DE C3	6DEC",
		"DE CE	6DEE",
		"BF BC	6DF1",
		"BD DF	6DF3",
		"CA A5	6DF5",
		"BA AE	6DF7",
		"DE BB	6DF9",
		"DE C9	6DFA",
		"C5 BA	6DFB",
		"C0 B6	6E05",
		"B3 E9	6E07",
		"BA D1	6E08",
		"BE C4	6E09",
		"DE BD	6E0A",
		"BD C2	6E0B",
		"B7 CC	6E13",
		"DE BC	6E15",
		"DE D2	6E19",
		"BD ED	6E1A",
		"B8 BA	6E1B",
		"DE E1	6E1D",
		"DE DB	6E1F",
		"B5 F4	6E20",
		"C5 CF	6E21",
		"DE D6	6E23",
		"DE DF	6E24",
		"B0 AF	6E25",
		"B1 B2	6E26",
		"B2 B9	6E29",
		"DE D8	6E2B",
		"C2 AC	6E2C",
		"DE CF	6E2D",
		"DE D1	6E2E",
		"B9 C1	6E2F",
		"DE E2	6E38",
		"DE DD	6E3A",
		"DE D5	6E3E",
		"DE DC	6E43",
		"CC AB	6E4A",
		"DE DA	6E4D",
		"DE DE	6E4E",
		"B8 D0	6E56",
		"BE C5	6E58",
		"C3 B9	6E5B",
		"DE D4	6E5F",
		"CD AF	6E67",
		"DE D7	6E6B",
		"DE D0	6E6E",
		"C5 F2	6E6F",
		"DE D3	6E72",
		"DE D9	6E76",
		"CF D1	6E7E",
		"BC BE	6E7F",
		"CB FE	6E80",
		"DE E3	6E82",
		"C8 AE	6E8C",
		"DE EF	6E8F",
		"B8 BB	6E90",
		"BD E0	6E96",
		"DE E5	6E98",
		"CE AF	6E9C",
		"B9 C2	6E9D",
		"DE F2	6E9F",
		"B0 EE	6EA2",
		"DE F0	6EA5",
		"DE E4	6EAA",
		"DE EA	6EAF",
		"DE EC	6EB2",
		"CD CF	6EB6",
		"DE E7	6EB7",
		"C5 AE	6EBA",
		"DE E9	6EBD",
		"DE F1	6EC2",
		"DE EB	6EC4",
		"CC C7	6EC5",
		"DE E6	6EC9",
		"BC A2	6ECB",
		"DE FE	6ECC",
		"B3 EA	6ED1",
		"DE E8	6ED3",
		"DE ED	6ED4",
		"DE EE	6ED5",
		"C2 EC	6EDD",
		"C2 DA	6EDE",
		"DE F6	6EEC",
		"DE FC	6EEF",
		"DE FA	6EF2",
		"C5 A9	6EF4",
		"DF A3	6EF7",
		"DE F7	6EF8",
		"DE F8	6EFE",
		"DE E0	6EFF",
		"B5 F9	6F01",
		"C9 BA	6F02",
		"BC BF	6F06",
		"B9 F7	6F09",
		"CF B3	6F0F",
		"DE F4	6F11",
		"DF A2	6F13",
		"B1 E9	6F14",
		"C1 E6	6F15",
		"C7 F9	6F20",
		"B4 C1	6F22",
		"CE FA	6F23",
		"CC A1	6F2B",
		"C4 D2	6F2C",
		"DE FB	6F31",
		"DE FD	6F32",
		"C1 B2	6F38",
		"DF A1	6F3E",
		"DE F9	6F3F",
		"DE F3	6F41",
		"B4 C3	6F45",
		"B7 E9	6F54",
		"DF AF	6F58",
		"DF AA	6F5B",
		"C0 F8	6F5C",
		"B3 E3	6F5F",
		"BD E1	6F64",
		"DF B3	6F66",
		"DF AC	6F6D",
		"C4 AC	6F6E",
		"DF A9	6F6F",
		"C4 D9	6F70",
		"DF CC	6F74",
		"DF A6	6F78",
		"DF A5	6F7A",
		"DF AE	6F7C",
		"DF A8	6F80",
		"DF A7	6F81",
		"DF AD	6F82",
		"C0 A1	6F84",
		"DF A4	6F86",
		"DF B0	6F8E",
		"DF B1	6F91",
		"B4 C2	6F97",
		"DF B6	6FA1",
		"DF B5	6FA3",
		"DF B7	6FA4",
		"DF BA	6FAA",
		"C5 C3	6FB1",
		"DF B4	6FB3",
		"DF B8	6FB9",
		"B7 E3	6FC0",
		"C2 F9	6FC1",
		"DF B2	6FC2",
		"C7 BB	6FC3",
		"DF B9	6FC6",
		"DF BE	6FD4",
		"DF BC	6FD5",
		"DF BF	6FD8",
		"DF C2	6FDB",
		"DF BB	6FDF",
		"B9 EA	6FE0",
		"C7 A8	6FE1",
		"DE B9	6FE4",
		"CD F4	6FEB",
		"DF BD	6FEC",
		"DF C1	6FEE",
		"C2 F5	6FEF",
		"DF C0	6FF1",
		"DF AB	6FF3",
		"EF E9	6FF6",
		"DF C5	6FFA",
		"DF C9	6FFE",
		"DF C7	7001",
		"DF C3	7009",
		"DF C4	700B",
		"DF C8	700F",
		"DF C6	7011",
		"C9 CE	7015",
		"DF CE	7018",
		"DF CB	701A",
		"DF CA	701B",
		"DF CD	701D",
		"C6 D4	701E",
		"DF CF	701F",
		"C3 F5	7026",
		"C2 ED	7027",
		"C0 A5	702C",
		"DF D0	7030",
		"DF D2	7032",
		"DF D1	703E",
		"DE F5	704C",
		"DF D3	7051",
		"C6 E7	7058",
		"DF D4	7063",
		"B2 D0	706B",
		"C5 F4	706F",
		"B3 A5	7070",
		"B5 E4	7078",
		"BC DE	707C",
		"BA D2	707D",
		"CF A7	7089",
		"BF E6	708A",
		"B1 EA	708E",
		"DF D6	7092",
		"DF D5	7099",
		"DF D9	70AC",
		"C3 BA	70AD",
		"DF DC	70AE",
		"DF D7	70AF",
		"DF DB	70B3",
		"DF DA	70B8",
		"C5 C0	70B9",
		"B0 D9	70BA",
		"CE F5	70C8",
		"DF DE	70CB",
		"B1 A8	70CF",
		"DF E0	70D9",
		"DF DF	70DD",
		"DF DD	70DF",
		"DF D8	70F1",
		"CB A3	70F9",
		"DF E2	70FD",
		"DF E1	7109",
		"B1 EB	7114",
		"DF E4	7119",
		"CA B2	711A",
		"DF E3	711C",
		"CC B5	7121",
		"BE C7	7126",
		"C1 B3	7136",
		"BE C6	713C",
		"CE FB	7149",
		"DF EA	714C",
		"C0 F9	714E",
		"DF E6	7155",
		"DF EB	7156",
		"B1 EC	7159",
		"DF E9	7162",
		"C7 E1	7164",
		"DF E5	7165",
		"DF E8	7166",
		"BE C8	7167",
		"C8 D1	7169",
		"DF EC	716C",
		"BC D1	716E",
		"C0 FA	717D",
		"DF EF	7184",
		"DF E7	7188",
		"B7 A7	718A",
		"DF ED	718F",
		"CD D0	7194",
		"DF F0	7195",
		"F4 A6	7199",
		"BD CF	719F",
		"DF F1	71A8",
		"DF F2	71AC",
		"C7 AE	71B1",
		"DF F4	71B9",
		"DF F5	71BE",
		"C7 B3	71C3",
		"C5 F5	71C8",
		"DF F7	71C9",
		"DF F9	71CE",
		"CE D5	71D0",
		"DF F6	71D2",
		"DF F8	71D4",
		"B1 ED	71D5",
		"DF F3	71D7",
		"D3 DB	71DF",
		"DF FA	71E0",
		"C1 E7	71E5",
		"BB B8	71E6",
		"DF FC	71E7",
		"DF FB	71EC",
		"BF A4	71ED",
		"D2 D9	71EE",
		"DF FD	71F5",
		"E0 A1	71F9",
		"DF EE	71FB",
		"DF FE	71FC",
		"E0 A2	71FF",
		"C7 FA	7206",
		"E0 A3	720D",
		"E0 A4	7210",
		"E0 A5	721B",
		"E0 A6	7228",
		"C4 DE	722A",
		"E0 A8	722C",
		"E0 A7	722D",
		"E0 A9	7230",
		"E0 AA	7232",
		"BC DF	7235",
		"C9 E3	7236",
		"CC EC	723A",
		"E0 AB	723B",
		"E0 AC	723C",
		"C1 D6	723D",
		"BC A4	723E",
		"E0 AD	723F",
		"E0 AE	7240",
		"E0 AF	7246",
		"CA D2	7247",
		"C8 C7	7248",
		"E0 B0	724B",
		"C7 D7	724C",
		"C4 AD	7252",
		"E0 B1	7258",
		"B2 E7	7259",
		"B5 ED	725B",
		"CC C6	725D",
		"CC B6	725F",
		"B2 B4	7261",
		"CF B4	7262",
		"CB D2	7267",
		"CA AA	7269",
		"C0 B7	7272",
		"E0 B2	7274",
		"C6 C3	7279",
		"B8 A3	727D",
		"E0 B3	727E",
		"BA D4	7280",
		"E0 B5	7281",
		"E0 B4	7282",
		"E0 B6	7287",
		"E0 B7	7292",
		"E0 B8	7296",
		"B5 BE	72A0",
		"E0 B9	72A2",
		"E0 BA	72A7",
		"B8 A4	72AC",
		"C8 C8	72AF",
		"E0 BC	72B2",
		"BE F5	72B6",
		"E0 BB	72B9",
		"B6 B8	72C2",
		"E0 BD	72C3",
		"E0 BF	72C4",
		"E0 BE	72C6",
		"E0 C0	72CE",
		"B8 D1	72D0",
		"E0 C1	72D2",
		"B6 E9	72D7",
		"C1 C0	72D9",
		"B9 FD	72DB",
		"E0 C3	72E0",
		"E0 C4	72E1",
		"E0 C2	72E2",
		"BC ED	72E9",
		"C6 C8	72EC",
		"B6 B9	72ED",
		"E0 C6	72F7",
		"C3 AC	72F8",
		"E0 C5	72F9",
		"CF B5	72FC",
		"C7 E2	72FD",
		"E0 C9	730A",
		"E0 CB	7316",
		"E0 C8	7317",
		"CC D4	731B",
		"E0 CA	731C",
		"E0 CC	731D",
		"CE C4	731F",
		"E0 D0	7325",
		"E0 CF	7329",
		"C3 F6	732A",
		"C7 AD	732B",
		"B8 A5	732E",
		"E0 CE	732F",
		"E0 CD	7334",
		"CD B1	7336",
		"CD B2	7337",
		"E0 D1	733E",
		"B1 EE	733F",
		"B9 F6	7344",
		"BB E2	7345",
		"E0 D2	734E",
		"E0 D3	734F",
		"E0 D5	7357",
		"BD C3	7363",
		"E0 D7	7368",
		"E0 D6	736A",
		"E0 D8	7370",
		"B3 CD	7372",
		"E0 DA	7375",
		"E0 D9	7378",
		"E0 DC	737A",
		"E0 DB	737B",
		"B8 BC	7384",
		"CE A8	7387",
		"B6 CC	7389",
		"B2 A6	738B",
		"B6 EA	7396",
		"B4 E1	73A9",
		"CE E8	73B2",
		"E0 DE	73B3",
		"E0 E0	73BB",
		"E0 E1	73C0",
		"B2 D1	73C2",
		"E0 DD	73C8",
		"BB B9	73CA",
		"C4 C1	73CD",
		"E0 DF	73CE",
		"E0 E4	73DE",
		"BC EE	73E0",
		"E0 E2	73E5",
		"B7 BE	73EA",
		"C8 C9	73ED",
		"E0 E3	73EE",
		"E0 FE	73F1",
		"E0 E9	73F8",
		"B8 BD	73FE",
		"B5 E5	7403",
		"E0 E6	7405",
		"CD FD	7406",
		"CE B0	7409",
		"C2 F6	7422",
		"E0 E8	7425",
		"E0 EA	7432",
		"CE D6	7433",
		"B6 D7	7434",
		"C8 FC	7435",
		"C7 CA	7436",
		"E0 EB	743A",
		"E0 ED	743F",
		"E0 F0	7441",
		"E0 EC	7455",
		"E0 EF	7459",
		"B8 EA	745A",
		"B1 CD	745B",
		"E0 F1	745C",
		"BF F0	745E",
		"E0 EE	745F",
		"CE DC	7460",
		"E0 F4	7463",
		"F4 A4	7464",
		"E0 F2	7469",
		"E0 F5	746A",
		"E0 E7	746F",
		"E0 F3	7470",
		"BA BC	7473",
		"E0 F6	7476",
		"E0 F7	747E",
		"CD FE	7483",
		"E0 F8	748B",
		"E0 F9	749E",
		"E0 E5	74A2",
		"E0 FA	74A7",
		"B4 C4	74B0",
		"BC A5	74BD",
		"E0 FB	74CA",
		"E0 FC	74CF",
		"E0 FD	74D4",
		"B1 BB	74DC",
		"E1 A1	74E0",
		"C9 BB	74E2",
		"E1 A2	74E3",
		"B4 A4	74E6",
		"E1 A3	74E7",
		"E1 A4	74E9",
		"E1 A5	74EE",
		"E1 A7	74F0",
		"E1 A8	74F1",
		"E1 A6	74F2",
		"C9 D3	74F6",
		"E1 AA	74F7",
		"E1 A9	74F8",
		"E1 AC	7503",
		"E1 AB	7504",
		"E1 AD	7505",
		"E1 AE	750C",
		"E1 B0	750D",
		"E1 AF	750E",
		"B9 F9	7511",
		"E1 B2	7513",
		"E1 B1	7515",
		"B4 C5	7518",
		"BF D3	751A",
		"C5 BC	751C",
		"E1 B3	751E",
		"C0 B8	751F",
		"BB BA	7523",
		"B1 F9	7525",
		"E1 B4	7526",
		"CD D1	7528",
		"CA E3	752B",
		"E1 B5	752C",
		"C5 C4	7530",
		"CD B3	7531",
		"B9 C3	7532",
		"BF BD	7533",
		"C3 CB	7537",
		"D2 B4	7538",
		"C4 AE	753A",
		"B2 E8	753B",
		"E1 B6	753C",
		"E1 B7	7544",
		"E1 BC	7546",
		"E1 BA	7549",
		"E1 B9	754A",
		"DA C2	754B",
		"B3 A6	754C",
		"E1 B8	754D",
		"B0 DA	754F",
		"C8 AA	7551",
		"C8 CA	7554",
		"CE B1	7559",
		"E1 BD	755A",
		"E1 BB	755B",
		"C3 DC	755C",
		"C0 A6	755D",
		"C8 AB	7560",
		"C9 AD	7562",
		"E1 BF	7564",
		"CE AC	7565",
		"B7 CD	7566",
		"E1 C0	7567",
		"E1 BE	7569",
		"C8 D6	756A",
		"E1 C1	756B",
		"E1 C2	756D",
		"B0 DB	7570",
		"BE F6	7573",
		"E1 C7	7574",
		"E1 C4	7576",
		"C6 ED	7577",
		"E1 C3	7578",
		"B5 A6	757F",
		"E1 CA	7582",
		"E1 C5	7586",
		"E1 C6	7587",
		"E1 C9	7589",
		"E1 C8	758A",
		"C9 A5	758B",
		"C1 C2	758E",
		"C1 C1	758F",
		"B5 BF	7591",
		"E1 CB	7594",
		"E1 CC	759A",
		"E1 CD	759D",
		"E1 CF	75A3",
		"E1 CE	75A5",
		"B1 D6	75AB",
		"E1 D7	75B1",
		"C8 E8	75B2",
		"E1 D1	75B3",
		"E1 D3	75B5",
		"E1 D5	75B8",
		"BF BE	75B9",
		"E1 D6	75BC",
		"E1 D4	75BD",
		"BC C0	75BE",
		"E1 D0	75C2",
		"E1 D2	75C3",
		"C9 C2	75C5",
		"BE C9	75C7",
		"E1 D9	75CA",
		"E1 D8	75CD",
		"E1 DA	75D2",
		"BC A6	75D4",
		"BA AF	75D5",
		"C5 F7	75D8",
		"E1 DB	75D9",
		"C4 CB	75DB",
		"E1 DD	75DE",
		"CE A1	75E2",
		"E1 DC	75E3",
		"C1 E9	75E9",
		"E1 E2	75F0",
		"E1 E4	75F2",
		"E1 E5	75F3",
		"C3 D4	75F4",
		"E1 E3	75FA",
		"E1 E0	75FC",
		"E1 DE	75FE",
		"E1 DF	75FF",
		"E1 E1	7601",
		"E1 E8	7609",
		"E1 E6	760B",
		"E1 E7	760D",
		"E1 E9	761F",
		"E1 EB	7620",
		"E1 EC	7621",
		"E1 ED	7622",
		"E1 EE	7624",
		"E1 EA	7627",
		"E1 F0	7630",
		"E1 EF	7634",
		"E1 F1	763B",
		"CE C5	7642",
		"E1 F4	7646",
		"E1 F2	7647",
		"E1 F3	7648",
		"B4 E2	764C",
		"CC FE	7652",
		"CA CA	7656",
		"E1 F6	7658",
		"E1 F5	765C",
		"E1 F7	7661",
		"E1 F8	7662",
		"E1 FC	7667",
		"E1 F9	7668",
		"E1 FA	7669",
		"E1 FB	766A",
		"E1 FD	766C",
		"E1 FE	7670",
		"E2 A1	7672",
		"E2 A2	7676",
		"E2 A3	7678",
		"C8 AF	767A",
		"C5 D0	767B",
		"E2 A4	767C",
		"C7 F2	767D",
		"C9 B4	767E",
		"E2 A5	7680",
		"E2 A6	7683",
		"C5 AA	7684",
		"B3 A7	7686",
		"B9 C4	7687",
		"E2 A7	7688",
		"E2 A8	768B",
		"E2 A9	768E",
		"BB A9	7690",
		"E2 AB	7693",
		"E2 AA	7696",
		"E2 AC	7699",
		"E2 AD	769A",
		"C8 E9	76AE",
		"E2 AE	76B0",
		"E2 AF	76B4",
		"F3 E9	76B7",
		"E2 B0	76B8",
		"E2 B1	76B9",
		"E2 B2	76BA",
		"BB AE	76BF",
		"E2 B3	76C2",
		"C7 D6	76C3",
		"CB DF	76C6",
		"B1 CE	76C8",
		"B1 D7	76CA",
		"E2 B4	76CD",
		"E2 B6	76D2",
		"E2 B5	76D6",
		"C5 F0	76D7",
		"C0 B9	76DB",
		"DD B9	76DC",
		"E2 B7	76DE",
		"CC C1	76DF",
		"E2 B8	76E1",
		"B4 C6	76E3",
		"C8 D7	76E4",
		"E2 B9	76E5",
		"E2 BA	76E7",
		"E2 BB	76EA",
		"CC DC	76EE",
		"CC D5	76F2",
		"C4 BE	76F4",
		"C1 EA	76F8",
		"E2 BD	76FB",
		"BD E2	76FE",
		"BE CA	7701",
		"E2 C0	7704",
		"E2 BF	7707",
		"E2 BE	7708",
		"C8 FD	7709",
		"B4 C7	770B",
		"B8 A9	770C",
		"E2 C6	771B",
		"E2 C3	771E",
		"BF BF	771F",
		"CC B2	7720",
		"E2 C2	7724",
		"E2 C4	7725",
		"E2 C5	7726",
		"E2 C1	7729",
		"E2 C7	7737",
		"E2 C8	7738",
		"C4 AF	773A",
		"B4 E3	773C",
		"C3 E5	7740",
		"E2 C9	7747",
		"E2 CA	775A",
		"E2 CD	775B",
		"BF E7	7761",
		"C6 C4	7763",
		"E2 CE	7765",
		"CB D3	7766",
		"E2 CB	7768",
		"E2 CC	776B",
		"E2 D1	7779",
		"E2 D0	777E",
		"E2 CF	777F",
		"E2 D3	778B",
		"E2 D2	778E",
		"E2 D4	7791",
		"E2 D6	779E",
		"E2 D5	77A0",
		"CA CD	77A5",
		"BD D6	77AC",
		"CE C6	77AD",
		"E2 D7	77B0",
		"C6 B7	77B3",
		"E2 D8	77B6",
		"E2 D9	77B9",
		"E2 DD	77BB",
		"E2 DB	77BC",
		"E2 DC	77BD",
		"E2 DA	77BF",
		"E2 DE	77C7",
		"E2 DF	77CD",
		"E2 E0	77D7",
		"E2 E1	77DA",
		"CC B7	77DB",
		"E2 E2	77DC",
		"CC F0	77E2",
		"E2 E3	77E3",
		"C3 CE	77E5",
		"C7 EA	77E7",
		"B6 EB	77E9",
		"C3 BB	77ED",
		"E2 E4	77EE",
		"B6 BA	77EF",
		"C0 D0	77F3",
		"E2 E5	77FC",
		"BA BD	7802",
		"E2 E6	780C",
		"E2 E7	7812",
		"B8 A6	7814",
		"BA D5	7815",
		"E2 E9	7820",
		"C5 D6	7825",
		"BA D6	7826",
		"B5 CE	7827",
		"CB A4	7832",
		"C7 CB	7834",
		"C5 D7	783A",
		"B9 DC	783F",
		"E2 EB	7845",
		"BE CB	785D",
		"CE B2	786B",
		"B9 C5	786C",
		"B8 A7	786F",
		"C8 A3	7872",
		"E2 ED	7874",
		"E2 EF	787C",
		"B8 EB	7881",
		"E2 EE	7886",
		"C4 F6	7887",
		"E2 F1	788C",
		"B3 B7	788D",
		"E2 EC	788E",
		"C8 EA	7891",
		"B1 B0	7893",
		"BA EC	7895",
		"CF D2	7897",
		"E2 F0	789A",
		"E2 F2	78A3",
		"CA CB	78A7",
		"C0 D9	78A9",
		"E2 F4	78AA",
		"E2 F5	78AF",
		"E2 F3	78B5",
		"B3 CE	78BA",
		"E2 FB	78BC",
		"E2 FA	78BE",
		"BC A7	78C1",
		"E2 FC	78C5",
		"E2 F7	78C6",
		"E2 FD	78CA",
		"E2 F8	78CB",
		"C8 D8	78D0",
		"E2 F6	78D1",
		"E2 F9	78D4",
		"E3 A2	78DA",
		"E3 A1	78E7",
		"CB E1	78E8",
		"E2 FE	78EC",
		"B0 EB	78EF",
		"E3 A4	78F4",
		"E3 A3	78FD",
		"BE CC	7901",
		"E3 A5	7907",
		"C1 C3	790E",
		"E3 A7	7911",
		"E3 A6	7912",
		"E3 A8	7919",
		"E2 E8	7926",
		"E2 EA	792A",
		"E3 AA	792B",
		"E3 A9	792C",
		"BC A8	793A",
		"CE E9	793C",
		"BC D2	793E",
		"E3 AB	7940",
		"B7 B7	7941",
		"B5 C0	7947",
		"B5 A7	7948",
		"BB E3	7949",
		"CD B4	7950",
		"E3 B1	7953",
		"E3 B0	7955",
		"C1 C4	7956",
		"E3 AD	7957",
		"E3 AF	795A",
		"BD CB	795D",
		"BF C0	795E",
		"E3 AE	795F",
		"E3 AC	7960",
		"C7 AA	7962",
		"BE CD	7965",
		"C9 BC	7968",
		"BA D7	796D",
		"C5 F8	7977",
		"E3 B2	797A",
		"E3 B3	797F",
		"E3 C9	7980",
		"B6 D8	7981",
		"CF BD	7984",
		"C1 B5	7985",
		"E3 B4	798A",
		"B2 D2	798D",
		"C4 F7	798E",
		"CA A1	798F",
		"E3 B5	799D",
		"B5 FA	79A6",
		"E3 B6	79A7",
		"E3 B8	79AA",
		"E3 B9	79AE",
		"C7 A9	79B0",
		"E3 BA	79B3",
		"E3 BB	79B9",
		"E3 BC	79BA",
		"B6 D9	79BD",
		"B2 D3	79BE",
		"C6 C5	79BF",
		"BD A8	79C0",
		"BB E4	79C1",
		"E3 BD	79C9",
		"BD A9	79CB",
		"B2 CA	79D1",
		"C9 C3	79D2",
		"E3 BE	79D5",
		"C8 EB	79D8",
		"C1 C5	79DF",
		"E3 C1	79E1",
		"E3 C2	79E3",
		"C7 E9	79E4",
		"BF C1	79E6",
		"E3 BF	79E7",
		"C3 E1	79E9",
		"E3 C0	79EC",
		"BE CE	79F0",
		"B0 DC	79FB",
		"B5 A9	7A00",
		"E3 C3	7A08",
		"C4 F8	7A0B",
		"E3 C4	7A0D",
		"C0 C7	7A0E",
		"CC AD	7A14",
		"C9 A3	7A17",
		"E3 C5	7A18",
		"E3 C6	7A19",
		"C3 D5	7A1A",
		"CE C7	7A1C",
		"E3 C8	7A1F",
		"E3 C7	7A20",
		"BC EF	7A2E",
		"E3 CA	7A31",
		"B0 F0	7A32",
		"E3 CD	7A37",
		"E3 CB	7A3B",
		"B2 D4	7A3C",
		"B7 CE	7A3D",
		"E3 CC	7A3E",
		"B9 C6	7A3F",
		"B9 F2	7A40",
		"CA E6	7A42",
		"E3 CE	7A43",
		"CB D4	7A46",
		"E3 D0	7A49",
		"C0 D1	7A4D",
		"B1 CF	7A4E",
		"B2 BA	7A4F",
		"B0 AC	7A50",
		"E3 CF	7A57",
		"E3 D1	7A61",
		"E3 D2	7A62",
		"BE F7	7A63",
		"E3 D3	7A69",
		"B3 CF	7A6B",
		"E3 D5	7A70",
		"B7 EA	7A74",
		"B5 E6	7A76",
		"E3 D6	7A79",
		"B6 F5	7A7A",
		"E3 D7	7A7D",
		"C0 FC	7A7F",
		"C6 CD	7A81",
		"C0 E0	7A83",
		"BA F5	7A84",
		"E3 D8	7A88",
		"C3 E2	7A92",
		"C1 EB	7A93",
		"E3 DA	7A95",
		"E3 DC	7A96",
		"E3 D9	7A97",
		"E3 DB	7A98",
		"B7 A2	7A9F",
		"E3 DD	7AA9",
		"B7 A6	7AAA",
		"B5 E7	7AAE",
		"CD D2	7AAF",
		"E3 DF	7AB0",
		"E3 E0	7AB6",
		"B1 AE	7ABA",
		"E3 E3	7ABF",
		"B3 F6	7AC3",
		"E3 E2	7AC4",
		"E3 E1	7AC5",
		"E3 E5	7AC7",
		"E3 DE	7AC8",
		"E3 E6	7ACA",
		"CE A9	7ACB",
		"E3 E7	7ACD",
		"E3 E8	7ACF",
		"D4 F4	7AD2",
		"E3 EA	7AD3",
		"E3 E9	7AD5",
		"E3 EB	7AD9",
		"E3 EC	7ADA",
		"CE B5	7ADC",
		"E3 ED	7ADD",
		"F0 EF	7ADF",
		"BE CF	7AE0",
		"E3 EE	7AE1",
		"E3 EF	7AE2",
		"BD D7	7AE3",
		"C6 B8	7AE5",
		"E3 F0	7AE6",
		"C3 A8	7AEA",
		"E3 F1	7AED",
		"C3 BC	7AEF",
		"E3 F2	7AF0",
		"B6 A5	7AF6",
		"D1 BF	7AF8",
		"C3 DD	7AF9",
		"BC B3	7AFA",
		"B4 C8	7AFF",
		"E3 F3	7B02",
		"E4 A2	7B04",
		"E3 F6	7B06",
		"B5 E8	7B08",
		"E3 F5	7B0A",
		"E4 A4	7B0B",
		"E3 F4	7B0F",
		"BE D0	7B11",
		"E3 F8	7B18",
		"E3 F9	7B19",
		"C5 AB	7B1B",
		"E3 FA	7B1E",
		"B3 DE	7B20",
		"BF DA	7B25",
		"C9 E4	7B26",
		"E3 FC	7B28",
		"C2 E8	7B2C",
		"E3 F7	7B33",
		"E3 FB	7B35",
		"E3 FD	7B36",
		"BA FB	7B39",
		"E4 A6	7B45",
		"C9 AE	7B46",
		"C8 A6	7B48",
		"C5 F9	7B49",
		"B6 DA	7B4B",
		"E4 A5	7B4C",
		"E4 A3	7B4D",
		"C8 B5	7B4F",
		"E3 FE	7B50",
		"C3 DE	7B51",
		"C5 FB	7B52",
		"C5 FA	7B54",
		"BA F6	7B56",
		"E4 B8	7B5D",
		"E4 A8	7B65",
		"E4 AA	7B67",
		"E4 AD	7B6C",
		"E4 AE	7B6E",
		"E4 AB	7B70",
		"E4 AC	7B71",
		"E4 A9	7B74",
		"E4 A7	7B75",
		"E4 A1	7B7A",
		"CA CF	7B86",
		"B2 D5	7B87",
		"E4 B5	7B8B",
		"E4 B2	7B8D",
		"E4 B7	7B8F",
		"E4 B6	7B92",
		"C7 F3	7B94",
		"CC A7	7B95",
		"BB BB	7B97",
		"E4 B0	7B98",
		"E4 B9	7B99",
		"E4 B4	7B9A",
		"E4 B3	7B9C",
		"E4 AF	7B9D",
		"E4 B1	7B9F",
		"B4 C9	7BA1",
		"C3 BD	7BAA",
		"C0 FD	7BAD",
		"C8 A2	7BB1",
		"E4 BE	7BB4",
		"C8 A4	7BB8",
		"C0 E1	7BC0",
		"E4 BB	7BC1",
		"C8 CF	7BC4",
		"E4 BF	7BC6",
		"CA D3	7BC7",
		"C3 DB	7BC9",
		"E4 BA	7BCB",
		"E4 BC	7BCC",
		"E4 BD	7BCF",
		"E4 C0	7BDD",
		"BC C4	7BE0",
		"C6 C6	7BE4",
		"E4 C5	7BE5",
		"E4 C4	7BE6",
		"E4 C1	7BE9",
		"CF B6	7BED",
		"E4 CA	7BF3",
		"E4 CE	7BF6",
		"E4 CB	7BF7",
		"E4 C7	7C00",
		"E4 C8	7C07",
		"E4 CD	7C0D",
		"E4 C2	7C11",
		"D2 D5	7C12",
		"E4 C9	7C13",
		"E4 C3	7C14",
		"E4 CC	7C17",
		"E4 D2	7C1F",
		"B4 CA	7C21",
		"E4 CF	7C23",
		"E4 D0	7C27",
		"E4 D1	7C2A",
		"E4 D4	7C2B",
		"E4 D3	7C37",
		"C8 F6	7C38",
		"E4 D5	7C3D",
		"CE FC	7C3E",
		"CA ED	7C3F",
		"E4 DA	7C40",
		"E4 D7	7C43",
		"E4 D6	7C4C",
		"C0 D2	7C4D",
		"E4 D9	7C4F",
		"E4 DB	7C50",
		"E4 D8	7C54",
		"E4 DF	7C56",
		"E4 DC	7C58",
		"E4 DD	7C5F",
		"E4 C6	7C60",
		"E4 DE	7C64",
		"E4 E0	7C65",
		"E4 E1	7C6C",
		"CA C6	7C73",
		"E4 E2	7C75",
		"CC E2	7C7E",
		"B6 CE	7C81",
		"B7 A9	7C82",
		"E4 E3	7C83",
		"CA B4	7C89",
		"BF E8	7C8B",
		"CC B0	7C8D",
		"E4 E4	7C90",
		"CE B3	7C92",
		"C7 F4	7C95",
		"C1 C6	7C97",
		"C7 B4	7C98",
		"BD CD	7C9B",
		"B0 C0	7C9F",
		"E4 E9	7CA1",
		"E4 E7	7CA2",
		"E4 E5	7CA4",
		"B4 A1	7CA5",
		"BE D1	7CA7",
		"E4 EA	7CA8",
		"E4 E8	7CAB",
		"E4 E6	7CAD",
		"E4 EE	7CAE",
		"E4 ED	7CB1",
		"E4 EC	7CB2",
		"E4 EB	7CB3",
		"E4 EF	7CB9",
		"E4 F0	7CBD",
		"C0 BA	7CBE",
		"E4 F1	7CC0",
		"E4 F3	7CC2",
		"E4 F2	7CC5",
		"B8 D2	7CCA",
		"C1 B8	7CCE",
		"E4 F5	7CD2",
		"C5 FC	7CD6",
		"E4 F4	7CD8",
		"E4 F6	7CDC",
		"CA B5	7CDE",
		"C1 EC	7CDF",
		"B9 C7	7CE0",
		"E4 F7	7CE2",
		"CE C8	7CE7",
		"E4 F9	7CEF",
		"E4 FA	7CF2",
		"E4 FB	7CF4",
		"E4 FC	7CF6",
		"BB E5	7CF8",
		"E4 FD	7CFA",
		"B7 CF	7CFB",
		"B5 EA	7CFE",
		"B5 AA	7D00",
		"E5 A1	7D02",
		"CC F3	7D04",
		"B9 C8	7D05",
		"E4 FE	7D06",
		"E5 A4	7D0A",
		"CC E6	7D0B",
		"C7 BC	7D0D",
		"C9 B3	7D10",
		"BD E3	7D14",
		"E5 A3	7D15",
		"BC D3	7D17",
		"B9 C9	7D18",
		"BB E6	7D19",
		"B5 E9	7D1A",
		"CA B6	7D1B",
		"E5 A2	7D1C",
		"C1 C7	7D20",
		"CB C2	7D21",
		"BA F7	7D22",
		"BB E7	7D2B",
		"C4 DD	7D2C",
		"E5 A7	7D2E",
		"CE DF	7D2F",
		"BA D9	7D30",
		"E5 A8	7D32",
		"BF C2	7D33",
		"E5 AA	7D35",
		"BE D2	7D39",
		"BA B0	7D3A",
		"E5 A9	7D3F",
		"BD AA	7D42",
		"B8 BE	7D43",
		"C1 C8	7D44",
		"E5 A5	7D45",
		"E5 AB	7D46",
		"E5 A6	7D4B",
		"B7 D0	7D4C",
		"E5 AE	7D4E",
		"E5 B2	7D4F",
		"B7 EB	7D50",
		"E5 AD	7D56",
		"E5 B6	7D5B",
		"B9 CA	7D5E",
		"CD ED	7D61",
		"B0 BC	7D62",
		"E5 B3	7D63",
		"B5 EB	7D66",
		"E5 B0	7D68",
		"E5 B1	7D6E",
		"C5 FD	7D71",
		"E5 AF	7D72",
		"E5 AC	7D73",
		"B3 A8	7D75",
		"C0 E4	7D76",
		"B8 A8	7D79",
		"E5 B8	7D7D",
		"E5 B5	7D89",
		"E5 B7	7D8F",
		"E5 B4	7D93",
		"B7 D1	7D99",
		"C2 B3	7D9A",
		"E5 B9	7D9B",
		"C1 EE	7D9C",
		"E5 C6	7D9F",
		"E5 C2	7DA2",
		"E5 BC	7DA3",
		"E5 C0	7DAB",
		"BC FA	7DAC",
		"B0 DD	7DAD",
		"E5 BB	7DAE",
		"E5 C3	7DAF",
		"E5 C7	7DB0",
		"B9 CB	7DB1",
		"CC D6	7DB2",
		"C4 D6	7DB4",
		"E5 BD	7DB5",
		"E5 C5	7DB8",
		"E5 BA	7DBA",
		"C3 BE	7DBB",
		"E5 BF	7DBD",
		"B0 BD	7DBE",
		"CC CA	7DBF",
		"E5 BE	7DC7",
		"B6 DB	7DCA",
		"C8 EC	7DCB",
		"C1 ED	7DCF",
		"CE D0	7DD1",
		"BD EF	7DD2",
		"E5 EE	7DD5",
		"E5 C8	7DD8",
		"C0 FE	7DDA",
		"E5 C4	7DDC",
		"E5 C9	7DDD",
		"E5 CB	7DDE",
		"C4 F9	7DE0",
		"E5 CE	7DE1",
		"E5 CA	7DE4",
		"CA D4	7DE8",
		"B4 CB	7DE9",
		"CC CB	7DEC",
		"B0 DE	7DEF",
		"E5 CD	7DF2",
		"CE FD	7DF4",
		"E5 CC	7DFB",
		"B1 EF	7E01",
		"C6 EC	7E04",
		"E5 CF	7E05",
		"E5 D6	7E09",
		"E5 D0	7E0A",
		"E5 D7	7E0B",
		"E5 D3	7E12",
		"C7 FB	7E1B",
		"BC CA	7E1E",
		"E5 D5	7E1F",
		"E5 D2	7E21",
		"E5 D8	7E22",
		"E5 D1	7E23",
		"BD C4	7E26",
		"CB A5	7E2B",
		"BD CC	7E2E",
		"E5 D4	7E31",
		"E5 E0	7E32",
		"E5 DC	7E35",
		"E5 DF	7E37",
		"E5 DD	7E39",
		"E5 E1	7E3A",
		"E5 DB	7E3B",
		"E5 C1	7E3D",
		"C0 D3	7E3E",
		"C8 CB	7E41",
		"E5 DE	7E43",
		"E5 D9	7E46",
		"C1 A1	7E4A",
		"B7 D2	7E4B",
		"BD AB	7E4D",
		"BF A5	7E54",
		"C1 B6	7E55",
		"E5 E4	7E56",
		"E5 E6	7E59",
		"E5 E7	7E5A",
		"E5 E3	7E5D",
		"E5 E5	7E5E",
		"E5 DA	7E66",
		"E5 E2	7E67",
		"E5 EA	7E69",
		"E5 E9	7E6A",
		"CB FA	7E6D",
		"B7 AB	7E70",
		"E5 E8	7E79",
		"E5 EC	7E7B",
		"E5 EB	7E7C",
		"E5 EF	7E7D",
		"E5 F1	7E7F",
		"BB BC	7E82",
		"E5 ED	7E83",
		"E5 F2	7E88",
		"E5 F3	7E89",
		"E5 F4	7E8C",
		"E5 FA	7E8E",
		"C5 BB	7E8F",
		"E5 F6	7E90",
		"E5 F5	7E92",
		"E5 F7	7E93",
		"E5 F8	7E94",
		"E5 F9	7E96",
		"E5 FB	7E9B",
		"E5 FC	7E9C",
		"B4 CC	7F36",
		"E5 FD	7F38",
		"E5 FE	7F3A",
		"E6 A1	7F45",
		"E6 A2	7F4C",
		"E6 A3	7F4D",
		"E6 A4	7F4E",
		"E6 A5	7F50",
		"E6 A6	7F51",
		"E6 A8	7F54",
		"E6 A7	7F55",
		"E6 A9	7F58",
		"E6 AA	7F5F",
		"E6 AB	7F60",
		"E6 AE	7F67",
		"E6 AC	7F68",
		"E6 AD	7F69",
		"BA E1	7F6A",
		"B7 D3	7F6B",
		"C3 D6	7F6E",
		"C8 B3	7F70",
		"BD F0	7F72",
		"C7 CD	7F75",
		"C8 ED	7F77",
		"E6 AF	7F78",
		"D8 ED	7F79",
		"E6 B0	7F82",
		"E6 B2	7F83",
		"CD E5	7F85",
		"E6 B1	7F86",
		"E6 B4	7F87",
		"E6 B3	7F88",
		"CD D3	7F8A",
		"E6 B5	7F8C",
		"C8 FE	7F8E",
		"E6 B6	7F94",
		"E6 B9	7F9A",
		"E6 B8	7F9D",
		"E6 B7	7F9E",
		"E6 BA	7FA3",
		"B7 B2	7FA4",
		"C1 A2	7FA8",
		"B5 C1	7FA9",
		"E6 BE	7FAE",
		"E6 BB	7FAF",
		"E6 BC	7FB2",
		"E6 BF	7FB6",
		"E6 C0	7FB8",
		"E6 BD	7FB9",
		"B1 A9	7FBD",
		"B2 A7	7FC1",
		"E6 C2	7FC5",
		"E6 C3	7FC6",
		"E6 C4	7FCA",
		"CD E2	7FCC",
		"BD AC	7FD2",
		"E6 C6	7FD4",
		"E6 C5	7FD5",
		"BF E9	7FE0",
		"E6 C7	7FE1",
		"E6 C8	7FE6",
		"E6 C9	7FE9",
		"B4 E5	7FEB",
		"B4 CD	7FF0",
		"E6 CA	7FF3",
		"E6 CB	7FF9",
		"CB DD	7FFB",
		"CD E3	7FFC",
		"CD D4	8000",
		"CF B7	8001",
		"B9 CD	8003",
		"E6 CE	8004",
		"BC D4	8005",
		"E6 CD	8006",
		"E6 CF	800B",
		"BC A9	800C",
		"C2 D1	8010",
		"E6 D0	8012",
		"B9 CC	8015",
		"CC D7	8017",
		"E6 D1	8018",
		"E6 D2	8019",
		"E6 D3	801C",
		"E6 D4	8021",
		"E6 D5	8028",
		"BC AA	8033",
		"CC ED	8036",
		"E6 D7	803B",
		"C3 BF	803D",
		"E6 D6	803F",
		"E6 D9	8046",
		"E6 D8	804A",
		"E6 DA	8052",
		"C0 BB	8056",
		"E6 DB	8058",
		"E6 DC	805A",
		"CA B9	805E",
		"E6 DD	805F",
		"C1 EF	8061",
		"E6 DE	8062",
		"E6 DF	8068",
		"CE FE	806F",
		"E6 E2	8070",
		"E6 E1	8072",
		"E6 E0	8073",
		"C4 B0	8074",
		"E6 E3	8076",
		"BF A6	8077",
		"E6 E4	8079",
		"E6 E5	807D",
		"CF B8	807E",
		"E6 E6	807F",
		"E6 E7	8084",
		"E6 E9	8085",
		"E6 E8	8086",
		"C8 A5	8087",
		"C6 F9	8089",
		"CF BE	808B",
		"C8 A9	808C",
		"E6 EB	8093",
		"BE D3	8096",
		"C9 AA	8098",
		"E6 EC	809A",
		"E6 EA	809B",
		"B4 CE	809D",
		"B8 D4	80A1",
		"BB E8	80A2",
		"C8 EE	80A5",
		"B8 AA	80A9",
		"CB C3	80AA",
		"E6 EF	80AC",
		"E6 ED	80AD",
		"B9 CE	80AF",
		"B9 CF	80B1",
		"B0 E9	80B2",
		"BA E8	80B4",
		"C7 D9	80BA",
		"B0 DF	80C3",
		"E6 F4	80C4",
		"C3 C0	80C6",
		"C7 D8	80CC",
		"C2 DB	80CE",
		"E6 F6	80D6",
		"E6 F2	80D9",
		"E6 F5	80DA",
		"E6 F0	80DB",
		"E6 F3	80DD",
		"CB A6	80DE",
		"B8 D5	80E1",
		"B0 FD	80E4",
		"E6 F1	80E5",
		"E6 F8	80EF",
		"E6 F9	80F1",
		"C6 B9	80F4",
		"B6 BB	80F8",
		"E7 A6	80FC",
		"C7 BD	80FD",
		"BB E9	8102",
		"B6 BC	8105",
		"C0 C8	8106",
		"CF C6	8107",
		"CC AE	8108",
		"E6 F7	8109",
		"C0 D4	810A",
		"B5 D3	811A",
		"E6 FA	811B",
		"E6 FC	8123",
		"E6 FB	8129",
		"E6 FD	812F",
		"C3 A6	8131",
		"C7 BE	8133",
		"C4 B1	8139",
		"E7 A3	813E",
		"E7 A2	8146",
		"E6 FE	814B",
		"BF D5	814E",
		"C9 E5	8150",
		"E7 A5	8151",
		"E7 A4	8153",
		"B9 D0	8154",
		"CF D3	8155",
		"E7 B5	815F",
		"E7 A9	8165",
		"E7 AA	8166",
		"BC F0	816B",
		"E7 A8	816E",
		"B9 F8	8170",
		"E7 A7	8171",
		"E7 AB	8174",
		"C4 B2	8178",
		"CA A2	8179",
		"C1 A3	817A",
		"C2 DC	817F",
		"E7 AF	8180",
		"E7 B0	8182",
		"E7 AC	8183",
		"E7 AD	8188",
		"E7 AE	818A",
		"B9 D1	818F",
		"E7 B6	8193",
		"E7 B2	8195",
		"C9 E6	819A",
		"CB EC	819C",
		"C9 A8	819D",
		"E7 B1	81A0",
		"E7 B4	81A3",
		"E7 B3	81A4",
		"CB C4	81A8",
		"E7 B7	81A9",
		"E7 B8	81B0",
		"C1 B7	81B3",
		"E7 B9	81B5",
		"E7 BB	81B8",
		"E7 BF	81BA",
		"E7 BC	81BD",
		"E7 BA	81BE",
		"C7 BF	81BF",
		"E7 BD	81C0",
		"E7 BE	81C2",
		"B2 B2	81C6",
		"E7 C5	81C8",
		"E7 C0	81C9",
		"E7 C1	81CD",
		"E7 C2	81D1",
		"C2 A1	81D3",
		"E7 C4	81D8",
		"E7 C3	81D9",
		"E7 C6	81DA",
		"E7 C7	81DF",
		"E7 C8	81E0",
		"BF C3	81E3",
		"B2 E9	81E5",
		"E7 C9	81E7",
		"CE D7	81E8",
		"BC AB	81EA",
		"BD AD	81ED",
		"BB EA	81F3",
		"C3 D7	81F4",
		"E7 CA	81FA",
		"E7 CB	81FB",
		"B1 B1	81FC",
		"E7 CC	81FE",
		"E7 CD	8201",
		"E7 CE	8202",
		"E7 CF	8205",
		"E7 D0	8207",
		"B6 BD	8208",
		"DA AA	8209",
		"E7 D1	820A",
		"C0 E5	820C",
		"E7 D2	820D",
		"BC CB	820E",
		"E7 D3	8210",
		"D0 B0	8212",
		"E7 D4	8216",
		"CA DE	8217",
		"B4 DC	8218",
		"C1 A4	821B",
		"BD D8	821C",
		"C9 F1	821E",
		"BD AE	821F",
		"E7 D5	8229",
		"B9 D2	822A",
		"E7 D6	822B",
		"C8 CC	822C",
		"E7 E4	822E",
		"E7 D8	8233",
		"C2 C9	8235",
		"C7 F5	8236",
		"B8 BF	8237",
		"E7 D7	8238",
		"C1 A5	8239",
		"E7 D9	8240",
		"C4 FA	8247",
		"E7 DB	8258",
		"E7 DA	8259",
		"E7 DD	825A",
		"E7 DC	825D",
		"E7 DE	825F",
		"E7 E0	8262",
		"E7 DF	8264",
		"B4 CF	8266",
		"E7 E1	8268",
		"E7 E2	826A",
		"E7 E3	826B",
		"BA B1	826E",
		"CE C9	826F",
		"E7 E5	8271",
		"BF A7	8272",
		"B1 F0	8276",
		"E7 E6	8277",
		"E7 E7	8278",
		"E7 E8	827E",
		"B0 F2	828B",
		"E7 E9	828D",
		"E7 EA	8292",
		"C9 E7	8299",
		"BC C7	829D",
		"E7 EC	829F",
		"B3 A9	82A5",
		"B0 B2	82A6",
		"E7 EB	82AB",
		"E7 EE	82AC",
		"C7 CE	82AD",
		"BF C4	82AF",
		"B2 D6	82B1",
		"CB A7	82B3",
		"B7 DD	82B8",
		"B6 DC	82B9",
		"E7 ED	82BB",
		"B2 EA	82BD",
		"B4 A3	82C5",
		"B1 F1	82D1",
		"E7 F2	82D2",
		"CE EA	82D3",
		"C2 DD	82D4",
		"C9 C4	82D7",
		"E7 FE	82D9",
		"B2 D7	82DB",
		"E7 FC	82DC",
		"E7 FA	82DE",
		"E7 F1	82DF",
		"E7 EF	82E1",
		"E7 F0	82E3",
		"BC E3	82E5",
		"B6 EC	82E6",
		"C3 F7	82E7",
		"C6 D1	82EB",
		"B1 D1	82F1",
		"E7 F4	82F3",
		"E7 F3	82F4",
		"E7 F9	82F9",
		"E7 F5	82FA",
		"E7 F8	82FB",
		"CC D0	8302",
		"E7 F7	8303",
		"B2 D8	8304",
		"B3 FD	8305",
		"E7 FB	8306",
		"E7 FD	8309",
		"B7 D4	830E",
		"E8 A3	8316",
		"E8 AC	8317",
		"E8 AD	8318",
		"B0 AB	831C",
		"E8 B4	8323",
		"B0 F1	8328",
		"E8 AB	832B",
		"E8 AA	832F",
		"E8 A5	8331",
		"E8 A4	8332",
		"E8 A2	8334",
		"E8 A1	8335",
		"C3 E3	8336",
		"C2 FB	8338",
		"E8 A7	8339",
		"E8 A6	8340",
		"E8 A9	8345",
		"C1 F0	8349",
		"B7 D5	834A",
		"B1 C1	834F",
		"E8 A8	8350",
		"B9 D3	8352",
		"C1 F1	8358",
		"E8 BA	8373",
		"E8 BB	8375",
		"B2 D9	8377",
		"B2 AE	837B",
		"E8 B8	837C",
		"E8 AE	8385",
		"E8 B6	8387",
		"E8 BD	8389",
		"E8 B7	838A",
		"E8 B5	838E",
		"E7 F6	8393",
		"E8 B3	8396",
		"E8 AF	839A",
		"B4 D0	839E",
		"E8 B1	839F",
		"E8 BC	83A0",
		"E8 B2	83A2",
		"E8 BE	83A8",
		"E8 B0	83AA",
		"C7 FC	83AB",
		"CD E9	83B1",
		"E8 B9	83B5",
		"E8 CF	83BD",
		"E8 C7	83C1",
		"BF FB	83C5",
		"B5 C6	83CA",
		"B6 DD	83CC",
		"E8 C2	83CE",
		"B2 DB	83D3",
		"BE D4	83D6",
		"E8 C5	83D8",
		"BA DA	83DC",
		"C5 D1	83DF",
		"E8 CA	83E0",
		"CA EE	83E9",
		"E8 C1	83EB",
		"B2 DA	83EF",
		"B8 D6	83F0",
		"C9 A9	83F1",
		"E8 CB	83F2",
		"E8 BF	83F4",
		"E8 C8	83F7",
		"E8 D2	83FB",
		"E8 C3	83FD",
		"E8 C4	8403",
		"C6 BA	8404",
		"E8 C9	8407",
		"E8 C6	840B",
		"CB A8	840C",
		"E8 CC	840D",
		"B0 E0	840E",
		"E8 C0	8413",
		"E8 CE	8420",
		"E8 CD	8422",
		"C7 EB	8429",
		"E8 D4	842A",
		"E8 DF	842C",
		"B3 FE	8431",
		"E8 E2	8435",
		"E8 D0	8438",
		"E8 D5	843C",
		"CD EE	843D",
		"E8 DE	8446",
		"CD D5	8449",
		"CE AA	844E",
		"C3 F8	8457",
		"B3 EB	845B",
		"C9 F2	8461",
		"E8 E4	8462",
		"C6 A1	8463",
		"B0 B1	8466",
		"E8 DD	8469",
		"E8 D9	846B",
		"C1 F2	846C",
		"E8 D3	846D",
		"E8 DB	846E",
		"E8 E0	846F",
		"C7 AC	8471",
		"B0 AA	8475",
		"E8 D8	8477",
		"E8 E1	8479",
		"C9 F8	847A",
		"E8 DC	8482",
		"E8 D7	8484",
		"BE D5	848B",
		"BD AF	8490",
		"BC AC	8494",
		"CC D8	8499",
		"C9 C7	849C",
		"E8 E7	849F",
		"E8 F0	84A1",
		"E8 DA	84AD",
		"B3 F7	84B2",
		"BE F8	84B8",
		"E8 E5	84B9",
		"E8 EA	84BB",
		"C1 F3	84BC",
		"E8 E6	84BF",
		"E8 ED	84C1",
		"C3 DF	84C4",
		"E8 EE	84C6",
		"CD D6	84C9",
		"E8 E3	84CA",
		"B3 B8	84CB",
		"E8 E9	84CD",
		"E8 EC	84D0",
		"CC AC	84D1",
		"E8 EF	84D6",
		"E8 E8	84D9",
		"E8 EB	84DA",
		"CB A9	84EC",
		"CF A1	84EE",
		"E8 F3	84F4",
		"E8 FA	84FC",
		"E8 F2	84FF",
		"BC C3	8500",
		"E8 D1	8506",
		"CA CE	8511",
		"CC A2	8513",
		"E8 F9	8514",
		"E8 F8	8515",
		"E8 F4	8517",
		"E8 F5	8518",
		"B1 B6	851A",
		"E8 F7	851F",
		"E8 F1	8521",
		"C4 D5	8526",
		"E8 F6	852C",
		"B0 FE	852D",
		"C2 A2	8535",
		"CA C3	853D",
		"E8 FB	8540",
		"E9 A1	8541",
		"C8 D9	8543",
		"E8 FE	8548",
		"BE D6	8549",
		"BC C9	854A",
		"E9 A3	854B",
		"B6 BE	854E",
		"E9 A4	8555",
		"C9 F9	8557",
		"E8 FD	8558",
		"E8 D6	855A",
		"E8 FC	8563",
		"CF CF	8568",
		"C6 A2	8569",
		"C9 F3	856A",
		"E9 AB	856D",
		"E9 B1	8577",
		"E9 B2	857E",
		"E9 A5	8580",
		"C7 F6	8584",
		"E9 AF	8587",
		"E9 A7	8588",
		"E9 A9	858A",
		"E9 B3	8590",
		"E9 A8	8591",
		"E9 AC	8594",
		"B1 F2	8597",
		"C6 E5	8599",
		"E9 AD	859B",
		"E9 B0	859C",
		"E9 A6	85A4",
		"C1 A6	85A6",
		"E9 AA	85A8",
		"BB A7	85A9",
		"BF C5	85AA",
		"B7 B0	85AB",
		"CC F4	85AC",
		"CC F9	85AE",
		"BD F2	85AF",
		"E9 B7	85B9",
		"E9 B5	85BA",
		"CF CE	85C1",
		"E9 B4	85C9",
		"CD F5	85CD",
		"E9 B6	85CF",
		"E9 B8	85D0",
		"E9 B9	85D5",
		"E9 BC	85DC",
		"E9 BA	85DD",
		"C6 A3	85E4",
		"E9 BB	85E5",
		"C8 CD	85E9",
		"E9 AE	85EA",
		"BD F3	85F7",
		"E9 BD	85F9",
		"E9 C2	85FA",
		"C1 F4	85FB",
		"E9 C1	85FE",
		"E9 A2	8602",
		"E9 C3	8606",
		"C1 C9	8607",
		"E9 BE	860A",
		"E9 C0	860B",
		"E9 BF	8613",
		"DD B1	8616",
		"DD A2	8617",
		"E9 C5	861A",
		"E9 C4	8622",
		"CD F6	862D",
		"E2 BC	862F",
		"E9 C6	8630",
		"E9 C7	863F",
		"E9 C8	864D",
		"B8 D7	864E",
		"B5 D4	8650",
		"E9 CA	8654",
		"D1 DD	8655",
		"B5 F5	865A",
		"CE BA	865C",
		"B6 F3	865E",
		"E9 CB	865F",
		"E9 CC	8667",
		"C3 EE	866B",
		"E9 CD	8671",
		"C6 FA	8679",
		"B0 BA	867B",
		"B2 E3	868A",
		"E9 D2	868B",
		"E9 D3	868C",
		"E9 CE	8693",
		"BB BD	8695",
		"E9 CF	86A3",
		"C7 C2	86A4",
		"E9 D0	86A9",
		"E9 D1	86AA",
		"E9 DB	86AB",
		"E9 D5	86AF",
		"E9 D8	86B0",
		"E9 D4	86B6",
		"E9 D6	86C4",
		"E9 D7	86C6",
		"BC D8	86C7",
		"E9 D9	86C9",
		"C3 C1	86CB",
		"B7 D6	86CD",
		"B3 C2	86CE",
		"E9 DC	86D4",
		"B3 BF	86D9",
		"E9 E1	86DB",
		"E9 DD	86DE",
		"E9 E0	86DF",
		"C8 BA	86E4",
		"E9 DE	86E9",
		"E9 DF	86EC",
		"C9 C8	86ED",
		"C8 DA	86EE",
		"E9 E2	86EF",
		"C2 FD	86F8",
		"E9 EC	86F9",
		"E9 E8	86FB",
		"B2 EB	86FE",
		"E9 E6	8700",
		"CB AA	8702",
		"E9 E7	8703",
		"E9 E4	8706",
		"E9 E5	8708",
		"E9 EA	8709",
		"E9 ED	870A",
		"E9 EB	870D",
		"E9 E9	8711",
		"E9 E3	8712",
		"C3 D8	8718",
		"E9 F4	871A",
		"CC AA	871C",
		"E9 F2	8725",
		"E9 F3	8729",
		"E9 EE	8734",
		"E9 F0	8737",
		"E9 F1	873B",
		"E9 EF	873F",
		"C0 E6	8749",
		"CF B9	874B",
		"E9 F8	874C",
		"E9 F9	874E",
		"EA A1	8753",
		"BF AA	8755",
		"E9 FB	8757",
		"E9 FE	8759",
		"E9 F6	875F",
		"E9 F5	8760",
		"EA A2	8763",
		"B2 DC	8766",
		"E9 FC	8768",
		"EA A3	876A",
		"E9 FD	876E",
		"E9 FA	8774",
		"C4 B3	8776",
		"E9 F7	8778",
		"C7 E8	877F",
		"EA A7	8782",
		"CD BB	878D",
		"EA A6	879F",
		"EA A5	87A2",
		"EA AE	87AB",
		"EA A8	87AF",
		"EA B0	87B3",
		"CD E6	87BA",
		"EA B3	87BB",
		"EA AA	87BD",
		"EA AB	87C0",
		"EA AF	87C4",
		"EA B2	87C6",
		"EA B1	87C7",
		"EA A9	87CB",
		"EA AC	87D0",
		"EA BD	87D2",
		"EA B6	87E0",
		"EA B4	87EF",
		"EA B5	87F2",
		"EA BA	87F6",
		"EA BB	87F7",
		"B3 AA	87F9",
		"B5 C2	87FB",
		"EA B9	87FE",
		"EA A4	8805",
		"EA B8	880D",
		"EA BC	880E",
		"EA B7	880F",
		"EA BE	8811",
		"EA C0	8815",
		"EA BF	8816",
		"EA C2	8821",
		"EA C1	8822",
		"E9 DA	8823",
		"EA C6	8827",
		"EA C3	8831",
		"EA C4	8836",
		"EA C5	8839",
		"EA C7	883B",
		"B7 EC	8840",
		"EA C9	8842",
		"EA C8	8844",
		"BD B0	8846",
		"B9 D4	884C",
		"DE A7	884D",
		"EA CA	8852",
		"BD D1	8853",
		"B3 B9	8857",
		"EA CB	8859",
		"B1 D2	885B",
		"BE D7	885D",
		"EA CC	885E",
		"B9 D5	8861",
		"EA CD	8862",
		"B0 E1	8863",
		"C9 BD	8868",
		"EA CE	886B",
		"BF EA	8870",
		"EA D5	8872",
		"EA D2	8875",
		"C3 EF	8877",
		"EA D3	887D",
		"EA D0	887E",
		"B6 DE	887F",
		"EA CF	8881",
		"EA D6	8882",
		"B7 B6	8888",
		"C2 DE	888B",
		"EA DC	888D",
		"EA D8	8892",
		"C2 B5	8896",
		"EA D7	8897",
		"EA DA	8899",
		"EA D1	889E",
		"EA DB	88A2",
		"EA DD	88A4",
		"C8 EF	88AB",
		"EA D9	88AE",
		"EA DE	88B0",
		"EA E0	88B1",
		"B8 D3	88B4",
		"EA D4	88B5",
		"B0 C1	88B7",
		"EA DF	88BF",
		"BA DB	88C1",
		"CE F6	88C2",
		"EA E1	88C3",
		"EA E2	88C4",
		"C1 F5	88C5",
		"CE A2	88CF",
		"EA E3	88D4",
		"CD B5	88D5",
		"EA E4	88D8",
		"EA E5	88D9",
		"CA E4	88DC",
		"EA E6	88DD",
		"BA C0	88DF",
		"CE A3	88E1",
		"EA EB	88E8",
		"EA EC	88F2",
		"BE D8	88F3",
		"EA EA	88F4",
		"CD E7	88F8",
		"EA E7	88F9",
		"EA E9	88FC",
		"C0 BD	88FD",
		"BF FE	88FE",
		"EA E8	8902",
		"EA ED	8904",
		"CA A3	8907",
		"EA EF	890A",
		"EA EE	890C",
		"B3 EC	8910",
		"CB AB	8912",
		"EA F0	8913",
		"EA FC	891D",
		"EA F2	891E",
		"EA F3	8925",
		"EA F4	892A",
		"EA F5	892B",
		"EA F9	8936",
		"EA FA	8938",
		"EA F8	893B",
		"EA F6	8941",
		"EA F1	8943",
		"EA F7	8944",
		"EA FB	894C",
		"F0 B7	894D",
		"B2 A8	8956",
		"EA FE	895E",
		"B6 DF	895F",
		"EA FD	8960",
		"EB A2	8964",
		"EB A1	8966",
		"EB A4	896A",
		"EB A3	896D",
		"EB A5	896F",
		"BD B1	8972",
		"EB A6	8974",
		"EB A7	8977",
		"EB A8	897E",
		"C0 BE	897F",
		"CD D7	8981",
		"EB A9	8983",
		"CA A4	8986",
		"C7 C6	8987",
		"EB AA	8988",
		"EB AB	898A",
		"B8 AB	898B",
		"B5 AC	898F",
		"EB AC	8993",
		"BB EB	8996",
		"C7 C1	8997",
		"EB AD	8998",
		"B3 D0	899A",
		"EB AE	89A1",
		"EB B0	89A6",
		"CD F7	89A7",
		"EB AF	89A9",
		"BF C6	89AA",
		"EB B1	89AC",
		"EB B2	89AF",
		"EB B3	89B2",
		"B4 D1	89B3",
		"EB B4	89BA",
		"EB B5	89BD",
		"EB B6	89BF",
		"EB B7	89C0",
		"B3 D1	89D2",
		"EB B8	89DA",
		"EB B9	89DC",
		"EB BA	89DD",
		"B2 F2	89E3",
		"BF A8	89E6",
		"EB BB	89E7",
		"EB BC	89F4",
		"EB BD	89F8",
		"B8 C0	8A00",
		"C4 FB	8A02",
		"EB BE	8A03",
		"B7 D7	8A08",
		"BF D6	8A0A",
		"EB C1	8A0C",
		"C6 A4	8A0E",
		"EB C0	8A10",
		"B7 B1	8A13",
		"EB BF	8A16",
		"C2 F7	8A17",
		"B5 AD	8A18",
		"EB C2	8A1B",
		"EB C3	8A1D",
		"BE D9	8A1F",
		"B7 ED	8A23",
		"EB C4	8A25",
		"CB AC	8A2A",
		"C0 DF	8A2D",
		"B5 F6	8A31",
		"CC F5	8A33",
		"C1 CA	8A34",
		"EB C5	8A36",
		"BF C7	8A3A",
		"C3 F0	8A3B",
		"BE DA	8A3C",
		"EB C6	8A41",
		"EB C9	8A46",
		"EB CA	8A48",
		"BA BE	8A50",
		"C2 C2	8A51",
		"EB C8	8A52",
		"BE DB	8A54",
		"C9 BE	8A55",
		"EB C7	8A5B",
		"BB EC	8A5E",
		"B1 D3	8A60",
		"EB CE	8A62",
		"B7 D8	8A63",
		"BB EE	8A66",
		"BB ED	8A69",
		"CF CD	8A6B",
		"EB CD	8A6C",
		"EB CC	8A6D",
		"C1 A7	8A6E",
		"B5 CD	8A70",
		"CF C3	8A71",
		"B3 BA	8A72",
		"BE DC	8A73",
		"EB CB	8A7C",
		"EB D0	8A82",
		"EB D1	8A84",
		"EB CF	8A85",
		"B8 D8	8A87",
		"CD C0	8A89",
		"BB EF	8A8C",
		"C7 A7	8A8D",
		"EB D4	8A91",
		"C0 C0	8A93",
		"C3 C2	8A95",
		"CD B6	8A98",
		"EB D7	8A9A",
		"B8 EC	8A9E",
		"C0 BF	8AA0",
		"EB D3	8AA1",
		"EB D8	8AA3",
		"B8 ED	8AA4",
		"EB D5	8AA5",
		"EB D6	8AA6",
		"EB D2	8AA8",
		"C0 E2	8AAC",
		"C6 C9	8AAD",
		"C3 AF	8AB0",
		"B2 DD	8AB2",
		"C8 F0	8AB9",
		"B5 C3	8ABC",
		"C4 B4	8ABF",
		"EB DB	8AC2",
		"EB D9	8AC4",
		"C3 CC	8AC7",
		"C0 C1	8ACB",
		"B4 D2	8ACC",
		"EB DA	8ACD",
		"BF DB	8ACF",
		"CE CA	8AD2",
		"CF C0	8AD6",
		"EB DC	8ADA",
		"EB E7	8ADB",
		"C4 B5	8ADC",
		"EB E6	8ADE",
		"EB E3	8AE0",
		"EB EB	8AE1",
		"EB E4	8AE2",
		"EB E0	8AE4",
		"C4 FC	8AE6",
		"EB DF	8AE7",
		"EB DD	8AEB",
		"CD A1	8AED",
		"BB F0	8AEE",
		"EB E1	8AF1",
		"EB DE	8AF3",
		"EB E5	8AF7",
		"BD F4	8AF8",
		"B8 C1	8AFA",
		"C2 FA	8AFE",
		"CB C5	8B00",
		"B1 DA	8B01",
		"B0 E2	8B02",
		"C6 A5	8B04",
		"EB E9	8B07",
		"EB E8	8B0C",
		"C6 E6	8B0E",
		"EB ED	8B10",
		"EB E2	8B14",
		"EB EC	8B16",
		"EB EE	8B17",
		"B8 AC	8B19",
		"EB EA	8B1A",
		"B9 D6	8B1B",
		"BC D5	8B1D",
		"EB EF	8B20",
		"CD D8	8B21",
		"EB F2	8B26",
		"EB F5	8B28",
		"EB F3	8B2B",
		"C9 B5	8B2C",
		"EB F0	8B33",
		"B6 E0	8B39",
		"EB F4	8B3E",
		"EB F6	8B41",
		"EB FA	8B49",
		"EB F7	8B4C",
		"EB F9	8B4E",
		"EB F8	8B4F",
		"EB FB	8B56",
		"BC B1	8B58",
		"EB FD	8B5A",
		"EB FC	8B5B",
		"C9 E8	8B5C",
		"EC A1	8B5F",
		"B7 D9	8B66",
		"EB FE	8B6B",
		"EC A2	8B6C",
		"EC A3	8B6F",
		"B5 C4	8B70",
		"E6 C1	8B71",
		"BE F9	8B72",
		"EC A4	8B74",
		"B8 EE	8B77",
		"EC A5	8B7D",
		"EC A6	8B80",
		"BB BE	8B83",
		"DA CE	8B8A",
		"EC A7	8B8C",
		"EC A8	8B8E",
		"BD B2	8B90",
		"EC A9	8B92",
		"EC AA	8B93",
		"EC AB	8B96",
		"EC AC	8B99",
		"EC AD	8B9A",
		"C3 AB	8C37",
		"EC AE	8C3A",
		"EC B0	8C3F",
		"EC AF	8C41",
		"C6 A6	8C46",
		"EC B1	8C48",
		"CB AD	8C4A",
		"EC B2	8C4C",
		"EC B3	8C4E",
		"EC B4	8C50",
		"EC B5	8C55",
		"C6 DA	8C5A",
		"BE DD	8C61",
		"EC B6	8C62",
		"B9 EB	8C6A",
		"D0 AE	8C6B",
		"EC B7	8C6C",
		"EC B8	8C78",
		"C9 BF	8C79",
		"EC B9	8C7A",
		"EC C1	8C7C",
		"EC BA	8C82",
		"EC BC	8C85",
		"EC BB	8C89",
		"EC BD	8C8A",
		"CB C6	8C8C",
		"EC BE	8C8D",
		"EC BF	8C8E",
		"EC C0	8C94",
		"EC C2	8C98",
		"B3 AD	8C9D",
		"C4 E7	8C9E",
		"C9 E9	8CA0",
		"BA E2	8CA1",
		"B9 D7	8CA2",
		"C9 CF	8CA7",
		"B2 DF	8CA8",
		"C8 CE	8CA9",
		"EC C5	8CAA",
		"B4 D3	8CAB",
		"C0 D5	8CAC",
		"EC C4	8CAD",
		"EC C9	8CAE",
		"C3 F9	8CAF",
		"CC E3	8CB0",
		"EC C7	8CB2",
		"EC C8	8CB3",
		"B5 AE	8CB4",
		"EC CA	8CB6",
		"C7 E3	8CB7",
		"C2 DF	8CB8",
		"C8 F1	8CBB",
		"C5 BD	8CBC",
		"EC C6	8CBD",
		"CB C7	8CBF",
		"B2 EC	8CC0",
		"EC CC	8CC1",
		"CF A8	8CC2",
		"C4 C2	8CC3",
		"CF C5	8CC4",
		"BB F1	8CC7",
		"EC CB	8CC8",
		"C2 B1	8CCA",
		"EC DC	8CCD",
		"C1 A8	8CCE",
		"C6 F8	8CD1",
		"C9 D0	8CD3",
		"EC CF	8CDA",
		"BB BF	8CDB",
		"BB F2	8CDC",
		"BE DE	8CDE",
		"C7 E5	8CE0",
		"B8 AD	8CE2",
		"EC CE	8CE3",
		"EC CD	8CE4",
		"C9 EA	8CE6",
		"BC C1	8CEA",
		"C5 D2	8CED",
		"EC D1	8CFA",
		"EC D2	8CFB",
		"B9 D8	8CFC",
		"EC D0	8CFD",
		"EC D3	8D04",
		"EC D4	8D05",
		"EC D6	8D07",
		"C2 A3	8D08",
		"EC D5	8D0A",
		"B4 E6	8D0B",
		"EC D8	8D0D",
		"EC D7	8D0F",
		"EC D9	8D10",
		"EC DB	8D13",
		"EC DD	8D14",
		"EC DE	8D16",
		"C0 D6	8D64",
		"BC CF	8D66",
		"EC DF	8D67",
		"B3 D2	8D6B",
		"EC E0	8D6D",
		"C1 F6	8D70",
		"EC E1	8D71",
		"EC E2	8D73",
		"C9 EB	8D74",
		"B5 AF	8D77",
		"EC E3	8D81",
		"C4 B6	8D85",
		"B1 DB	8D8A",
		"EC E4	8D99",
		"BC F1	8DA3",
		"BF F6	8DA8",
		"C2 AD	8DB3",
		"EC E7	8DBA",
		"EC E6	8DBE",
		"EC E5	8DC2",
		"EC ED	8DCB",
		"EC EB	8DCC",
		"EC E8	8DCF",
		"EC EA	8DD6",
		"EC E9	8DDA",
		"EC EC	8DDB",
		"B5 F7	8DDD",
		"EC F0	8DDF",
		"C0 D7	8DE1",
		"EC F1	8DE3",
		"B8 D9	8DE8",
		"EC EE	8DEA",
		"EC EF	8DEB",
		"CF A9	8DEF",
		"C4 B7	8DF3",
		"C1 A9	8DF5",
		"EC F2	8DFC",
		"EC F5	8DFF",
		"EC F3	8E08",
		"EC F4	8E09",
		"CD D9	8E0A",
		"C6 A7	8E0F",
		"EC F8	8E10",
		"EC F6	8E1D",
		"EC F7	8E1E",
		"EC F9	8E1F",
		"ED A9	8E2A",
		"EC FC	8E30",
		"EC FD	8E34",
		"EC FB	8E35",
		"EC FA	8E42",
		"C4 FD	8E44",
		"ED A1	8E47",
		"ED A5	8E48",
		"ED A2	8E49",
		"EC FE	8E4A",
		"ED A3	8E4C",
		"ED A4	8E50",
		"ED AB	8E55",
		"ED A6	8E59",
		"C0 D8	8E5F",
		"ED A8	8E60",
		"ED AA	8E63",
		"ED A7	8E64",
		"ED AD	8E72",
		"BD B3	8E74",
		"ED AC	8E76",
		"ED AE	8E7C",
		"ED AF	8E81",
		"ED B2	8E84",
		"ED B1	8E85",
		"ED B0	8E87",
		"ED B4	8E8A",
		"ED B3	8E8B",
		"CC F6	8E8D",
		"ED B6	8E91",
		"ED B5	8E93",
		"ED B7	8E94",
		"ED B8	8E99",
		"ED BA	8EA1",
		"ED B9	8EAA",
		"BF C8	8EAB",
		"ED BB	8EAC",
		"B6 ED	8EAF",
		"ED BC	8EB0",
		"ED BE	8EB1",
		"ED BF	8EBE",
		"ED C0	8EC5",
		"ED BD	8EC6",
		"ED C1	8EC8",
		"BC D6	8ECA",
		"ED C2	8ECB",
		"B5 B0	8ECC",
		"B7 B3	8ECD",
		"B8 AE	8ED2",
		"ED C3	8EDB",
		"C6 F0	8EDF",
		"C5 BE	8EE2",
		"ED C4	8EE3",
		"ED C7	8EEB",
		"BC B4	8EF8",
		"ED C6	8EFB",
		"ED C5	8EFC",
		"B7 DA	8EFD",
		"ED C8	8EFE",
		"B3 D3	8F03",
		"ED CA	8F05",
		"BA DC	8F09",
		"ED C9	8F0A",
		"ED D2	8F0C",
		"ED CC	8F12",
		"ED CE	8F13",
		"CA E5	8F14",
		"ED CB	8F15",
		"ED CD	8F19",
		"ED D1	8F1B",
		"ED CF	8F1C",
		"B5 B1	8F1D",
		"ED D0	8F1F",
		"ED D3	8F26",
		"C7 DA	8F29",
		"CE D8	8F2A",
		"BD B4	8F2F",
		"ED D4	8F33",
		"CD A2	8F38",
		"ED D6	8F39",
		"ED D5	8F3B",
		"ED D9	8F3E",
		"CD C1	8F3F",
		"ED D8	8F42",
		"B3 ED	8F44",
		"ED D7	8F45",
		"ED DC	8F46",
		"ED DB	8F49",
		"ED DA	8F4C",
		"C5 B2	8F4D",
		"ED DD	8F4E",
		"ED DE	8F57",
		"ED DF	8F5C",
		"B9 EC	8F5F",
		"B7 A5	8F61",
		"ED E0	8F62",
		"ED E1	8F63",
		"ED E2	8F64",
		"BF C9	8F9B",
		"ED E3	8F9C",
		"BC AD	8F9E",
		"ED E4	8F9F",
		"ED E5	8FA3",
		"D2 A1	8FA7",
		"D1 FE	8FA8",
		"ED E6	8FAD",
		"E5 F0	8FAE",
		"ED E7	8FAF",
		"C3 A4	8FB0",
		"BF AB	8FB1",
		"C7 C0	8FB2",
		"ED E8	8FB7",
		"CA D5	8FBA",
		"C4 D4	8FBB",
		"B9 FE	8FBC",
		"C3 A9	8FBF",
		"B1 AA	8FC2",
		"CB F8	8FC4",
		"BF D7	8FC5",
		"B7 DE	8FCE",
		"B6 E1	8FD1",
		"CA D6	8FD4",
		"ED E9	8FDA",
		"ED EB	8FE2",
		"ED EA	8FE5",
		"B2 E0	8FE6",
		"C6 F6	8FE9",
		"ED EC	8FEA",
		"C7 F7	8FEB",
		"C5 B3	8FED",
		"ED ED	8FEF",
		"BD D2	8FF0",
		"ED EF	8FF4",
		"CC C2	8FF7",
		"ED FE	8FF8",
		"ED F1	8FF9",
		"ED F2	8FFA",
		"C4 C9	8FFD",
		"C2 E0	9000",
		"C1 F7	9001",
		"C6 A8	9003",
		"ED F0	9005",
		"B5 D5	9006",
		"ED F9	900B",
		"ED F6	900D",
		"EE A5	900E",
		"C6 A9	900F",
		"C3 E0	9010",
		"ED F3	9011",
		"C4 FE	9013",
		"C5 D3	9014",
		"ED F4	9015",
		"ED F8	9016",
		"BF E0	9017",
		"C7 E7	9019",
		"C4 CC	901A",
		"C0 C2	901D",
		"ED F7	901E",
		"C2 AE	901F",
		"C2 A4	9020",
		"ED F5	9021",
		"B0 A9	9022",
		"CF A2	9023",
		"ED FA	9027",
		"C2 E1	902E",
		"BD B5	9031",
		"BF CA	9032",
		"ED FC	9035",
		"ED FB	9036",
		"B0 EF	9038",
		"ED FD	9039",
		"C9 AF	903C",
		"EE A7	903E",
		"C6 DB	9041",
		"BF EB	9042",
		"C3 D9	9045",
		"B6 F8	9047",
		"EE A6	9049",
		"CD B7	904A",
		"B1 BF	904B",
		"CA D7	904D",
		"B2 E1	904E",
		"EE A1	904F",
		"EE A2	9050",
		"EE A3	9051",
		"EE A4	9052",
		"C6 BB	9053",
		"C3 A3	9054",
		"B0 E3	9055",
		"EE A8	9056",
		"EE A9	9058",
		"F4 A3	9059",
		"C2 BD	905C",
		"EE AA	905E",
		"B1 F3	9060",
		"C1 CC	9061",
		"B8 AF	9063",
		"CD DA	9065",
		"EE AB	9068",
		"C5 AC	9069",
		"C1 F8	906D",
		"BC D7	906E",
		"EE AC	906F",
		"EE AF	9072",
		"BD E5	9075",
		"EE AD	9076",
		"C1 AB	9077",
		"C1 AA	9078",
		"B0 E4	907A",
		"CE CB	907C",
		"EE B1	907D",
		"C8 F2	907F",
		"EE B3	9080",
		"EE B2	9081",
		"EE B0	9082",
		"E3 E4	9083",
		"B4 D4	9084",
		"ED EE	9087",
		"EE B5	9089",
		"EE B4	908A",
		"EE B6	908F",
		"CD B8	9091",
		"C6 E1	90A3",
		"CB AE	90A6",
		"EE B7	90A8",
		"BC D9	90AA",
		"EE B8	90AF",
		"EE B9	90B1",
		"EE BA	90B5",
		"C5 A1	90B8",
		"B0 EA	90C1",
		"B9 D9	90CA",
		"CF BA	90CE",
		"EE BE	90DB",
		"B7 B4	90E1",
		"EE BB	90E2",
		"EE BC	90E4",
		"C9 F4	90E8",
		"B3 D4	90ED",
		"CD B9	90F5",
		"B6 BF	90F7",
		"C5 D4	90FD",
		"EE BF	9102",
		"EE C0	9112",
		"EE C1	9119",
		"C5 A2	912D",
		"EE C3	9130",
		"EE C2	9132",
		"C6 D3	9149",
		"EE C4	914A",
		"BD B6	914B",
		"BC E0	914C",
		"C7 DB	914D",
		"C3 F1	914E",
		"BC F2	9152",
		"BF EC	9154",
		"EE C5	9156",
		"EE C6	9158",
		"BF DD	9162",
		"EE C7	9163",
		"EE C8	9165",
		"EE C9	9169",
		"CD EF	916A",
		"BD B7	916C",
		"EE CB	9172",
		"EE CA	9173",
		"B9 DA	9175",
		"B9 F3	9177",
		"BB C0	9178",
		"EE CE	9182",
		"BD E6	9187",
		"EE CD	9189",
		"EE CC	918B",
		"C2 E9	918D",
		"B8 EF	9190",
		"C0 C3	9192",
		"C8 B0	9197",
		"BD B9	919C",
		"EE CF	91A2",
		"BE DF	91A4",
		"EE D2	91AA",
		"EE D0	91AB",
		"EE D1	91AF",
		"EE D4	91B4",
		"EE D3	91B5",
		"BE FA	91B8",
		"EE D5	91BA",
		"EE D6	91C0",
		"EE D7	91C1",
		"C8 D0	91C6",
		"BA D3	91C7",
		"BC E1	91C8",
		"EE D8	91C9",
		"EE D9	91CB",
		"CE A4	91CC",
		"BD C5	91CD",
		"CC EE	91CE",
		"CE CC	91CF",
		"EE DA	91D0",
		"B6 E2	91D1",
		"EE DB	91D6",
		"C5 A3	91D8",
		"EE DE	91DB",
		"B3 F8	91DC",
		"BF CB	91DD",
		"EE DC	91DF",
		"EE DD	91E1",
		"C4 E0	91E3",
		"CB D5	91E6",
		"B6 FC	91E7",
		"EE E0	91F5",
		"EE E1	91F6",
		"EE DF	91FC",
		"EE E3	91FF",
		"C6 DF	920D",
		"B3 C3	920E",
		"EE E7	9211",
		"EE E4	9214",
		"EE E6	9215",
		"EE E2	921E",
		"EF CF	9229",
		"EE E5	922C",
		"CE EB	9234",
		"B8 DA	9237",
		"EE EF	923F",
		"C5 B4	9244",
		"EE EA	9245",
		"EE ED	9248",
		"EE EB	9249",
		"EE F0	924B",
		"EE F1	9250",
		"EE E9	9257",
		"EE F6	925A",
		"B1 F4	925B",
		"EE E8	925E",
		"C8 AD	9262",
		"EE EC	9264",
		"BE E0	9266",
		"B9 DB	9271",
		"CB C8	927E",
		"B6 E4	9280",
		"BD C6	9283",
		"C6 BC	9285",
		"C1 AD	9291",
		"EE F4	9293",
		"EE EE	9295",
		"EE F3	9296",
		"CC C3	9298",
		"C4 B8	929A",
		"EE F5	929B",
		"EE F2	929C",
		"C1 AC	92AD",
		"EE F9	92B7",
		"EE F8	92B9",
		"EE F7	92CF",
		"CB AF	92D2",
		"BD FB	92E4",
		"EE FA	92E9",
		"CA DF	92EA",
		"B1 D4	92ED",
		"C9 C6	92F2",
		"C3 F2	92F3",
		"B5 F8	92F8",
		"EE FC	92FA",
		"B9 DD	92FC",
		"BB AC	9306",
		"EE FB	930F",
		"BF ED	9310",
		"BF EE	9318",
		"EF A1	9319",
		"EF A3	931A",
		"BE FB	9320",
		"EF A2	9322",
		"EF A4	9323",
		"B6 D3	9326",
		"C9 C5	9328",
		"BC E2	932B",
		"CF A3	932C",
		"EE FE	932E",
		"BA F8	932F",
		"CF BF	9332",
		"EF A6	9335",
		"EF A5	933A",
		"EF A7	933B",
		"EE FD	9344",
		"C6 E9	934B",
		"C5 D5	934D",
		"C4 D7	9354",
		"EF AC	9356",
		"C3 C3	935B",
		"EF A8	935C",
		"EF A9	9360",
		"B7 AD	936C",
		"EF AB	936E",
		"B8 B0	9375",
		"EF AA	937C",
		"BE E1	937E",
		"B3 F9	938C",
		"EF B0	9394",
		"BA BF	9396",
		"C1 F9	9397",
		"C4 CA	939A",
		"B3 BB	93A7",
		"EF AE	93AC",
		"EF AF	93AD",
		"C4 C3	93AE",
		"EF AD	93B0",
		"EF B1	93B9",
		"EF B7	93C3",
		"EF BA	93C8",
		"EF B9	93D0",
		"C5 AD	93D1",
		"EF B2	93D6",
		"EF B3	93D7",
		"EF B6	93D8",
		"EF B8	93DD",
		"B6 C0	93E1",
		"EF BB	93E4",
		"EF B5	93E5",
		"EF B4	93E8",
		"EF BF	9403",
		"EF C0	9407",
		"EF C1	9410",
		"EF BE	9413",
		"EF BD	9414",
		"BE E2	9418",
		"C6 AA	9419",
		"EF BC	941A",
		"EF C5	9421",
		"EF C3	942B",
		"EF C4	9435",
		"EF C2	9436",
		"C2 F8	9438",
		"EF C6	943A",
		"EF C7	9441",
		"EF C9	9444",
		"B4 D5	9451",
		"EF C8	9452",
		"CC FA	9453",
		"EF D4	945A",
		"EF CA	945B",
		"EF CD	945E",
		"EF CB	9460",
		"EF CC	9462",
		"EF CE	946A",
		"EF D0	9470",
		"EF D1	9475",
		"EF D2	9477",
		"EF D5	947C",
		"EF D3	947D",
		"EF D6	947E",
		"EF D8	947F",
		"EF D7	9481",
		"C4 B9	9577",
		"CC E7	9580",
		"EF D9	9582",
		"C1 AE	9583",
		"EF DA	9587",
		"CA C4	9589",
		"EF DB	958A",
		"B3 AB	958B",
		"B1 BC	958F",
		"B4 D7	9591",
		"B4 D6	9593",
		"EF DC	9594",
		"EF DD	9596",
		"EF DE	9598",
		"EF DF	9599",
		"EF E0	95A0",
		"B4 D8	95A2",
		"B3 D5	95A3",
		"B9 DE	95A4",
		"C8 B6	95A5",
		"EF E2	95A7",
		"EF E1	95A8",
		"EF E3	95AD",
		"B1 DC	95B2",
		"EF E6	95B9",
		"EF E5	95BB",
		"EF E4	95BC",
		"EF E7	95BE",
		"EF EA	95C3",
		"B0 C7	95C7",
		"EF E8	95CA",
		"EF EC	95CC",
		"EF EB	95CD",
		"EF EE	95D4",
		"EF ED	95D5",
		"EF EF	95D6",
		"C6 AE	95D8",
		"EF F0	95DC",
		"EF F1	95E1",
		"EF F3	95E2",
		"EF F2	95E5",
		"C9 EC	961C",
		"EF F4	9621",
		"EF F5	9628",
		"BA E5	962A",
		"EF F6	962E",
		"EF F7	962F",
		"CB C9	9632",
		"C1 CB	963B",
		"B0 A4	963F",
		"C2 CB	9640",
		"EF F8	9642",
		"C9 ED	9644",
		"EF FB	964B",
		"EF F9	964C",
		"B9 DF	964D",
		"EF FA	964F",
		"B8 C2	9650",
		"CA C5	965B",
		"EF FD	965C",
		"F0 A1	965D",
		"EF FE	965E",
		"F0 A2	965F",
		"B1 A1	9662",
		"BF D8	9663",
		"BD FC	9664",
		"B4 D9	9665",
		"F0 A3	9666",
		"C7 E6	966A",
		"F0 A5	966C",
		"B1 A2	9670",
		"F0 A4	9672",
		"C4 C4	9673",
		"CE CD	9675",
		"C6 AB	9676",
		"EF FC	9677",
		"CE A6	9678",
		"B8 B1	967A",
		"CD DB	967D",
		"B6 F9	9685",
		"CE B4	9686",
		"B7 A8	9688",
		"C2 E2	968A",
		"E7 A1	968B",
		"F0 A6	968D",
		"B3 AC	968E",
		"BF EF	968F",
		"B3 D6	9694",
		"F0 A8	9695",
		"F0 A9	9697",
		"F0 A7	9698",
		"B7 E4	9699",
		"BA DD	969B",
		"BE E3	969C",
		"B1 A3	96A0",
		"CE D9	96A3",
		"F0 AB	96A7",
		"EE AE	96A8",
		"F0 AA	96AA",
		"F0 AE	96B0",
		"F0 AC	96B1",
		"F0 AD	96B2",
		"F0 AF	96B4",
		"F0 B0	96B6",
		"CE EC	96B7",
		"F0 B1	96B8",
		"F0 B2	96B9",
		"C0 C9	96BB",
		"C8 BB	96BC",
		"BF FD	96C0",
		"B4 E7	96C1",
		"CD BA	96C4",
		"B2 ED	96C5",
		"BD B8	96C6",
		"B8 DB	96C7",
		"F0 B5	96C9",
		"F0 B4	96CB",
		"BB F3	96CC",
		"F0 B6	96CD",
		"F0 B3	96CE",
		"BB A8	96D1",
		"F0 BA	96D5",
		"EA AD	96D6",
		"D2 D6	96D9",
		"BF F7	96DB",
		"F0 B8	96DC",
		"CE A5	96E2",
		"C6 F1	96E3",
		"B1 AB	96E8",
		"C0 E3	96EA",
		"BC B6	96EB",
		"CA B7	96F0",
		"B1 C0	96F2",
		"CE ED	96F6",
		"CD EB	96F7",
		"F0 BB	96F9",
		"C5 C5	96FB",
		"BC FB	9700",
		"F0 BC	9704",
		"F0 BD	9706",
		"BF CC	9707",
		"F0 BE	9708",
		"CE EE	970A",
		"F0 B9	970D",
		"F0 C0	970E",
		"F0 C2	970F",
		"F0 C1	9711",
		"F0 BF	9713",
		"F0 C3	9716",
		"F0 C4	9719",
		"C1 FA	971C",
		"B2 E2	971E",
		"F0 C5	9724",
		"CC B8	9727",
		"F0 C6	972A",
		"F0 C7	9730",
		"CF AA	9732",
		"DB B1	9738",
		"F0 C8	9739",
		"F0 C9	973D",
		"F0 CA	973E",
		"F0 CE	9742",
		"F0 CB	9744",
		"F0 CC	9746",
		"F0 CD	9748",
		"F0 CF	9749",
		"C0 C4	9752",
		"CC F7	9756",
		"C0 C5	9759",
		"F0 D0	975C",
		"C8 F3	975E",
		"F0 D1	9760",
		"F3 D3	9761",
		"CC CC	9762",
		"F0 D2	9764",
		"F0 D3	9766",
		"F0 D4	9768",
		"B3 D7	9769",
		"F0 D6	976B",
		"BF D9	976D",
		"F0 D7	9771",
		"B7 A4	9774",
		"F0 D8	9779",
		"F0 DC	977A",
		"F0 DA	977C",
		"F0 DB	9781",
		"B3 F3	9784",
		"F0 D9	9785",
		"F0 DD	9786",
		"F0 DE	978B",
		"B0 C8	978D",
		"F0 DF	978F",
		"F0 E0	9790",
		"BE E4	9798",
		"F0 E1	979C",
		"B5 C7	97A0",
		"F0 E4	97A3",
		"F0 E3	97A6",
		"F0 E2	97A8",
		"EB F1	97AB",
		"CA DC	97AD",
		"F0 E5	97B3",
		"F0 E6	97B4",
		"F0 E7	97C3",
		"F0 E8	97C6",
		"F0 E9	97C8",
		"F0 EA	97CB",
		"B4 DA	97D3",
		"F0 EB	97DC",
		"F0 EC	97ED",
		"C7 A3	97EE",
		"F0 EE	97F2",
		"B2 BB	97F3",
		"F0 F1	97F5",
		"F0 F0	97F6",
		"B1 A4	97FB",
		"B6 C1	97FF",
		"CA C7	9801",
		"C4 BA	9802",
		"BA A2	9803",
		"B9 E0	9805",
		"BD E7	9806",
		"BF DC	9808",
		"F0 F3	980C",
		"F0 F2	980F",
		"CD C2	9810",
		"B4 E8	9811",
		"C8 D2	9812",
		"C6 DC	9813",
		"BF FC	9817",
		"CE CE	9818",
		"B7 DB	981A",
		"F0 F6	9821",
		"F0 F5	9824",
		"CB CB	982C",
		"C6 AC	982D",
		"B1 D0	9834",
		"F0 F7	9837",
		"F0 F4	9838",
		"C9 D1	983B",
		"CD EA	983C",
		"F0 F8	983D",
		"F0 F9	9846",
		"F0 FB	984B",
		"C2 EA	984C",
		"B3 DB	984D",
		"B3 DC	984E",
		"F0 FA	984F",
		"B4 E9	9854",
		"B8 B2	9855",
		"B4 EA	9858",
		"C5 BF	985B",
		"CE E0	985E",
		"B8 DC	9867",
		"F0 FC	986B",
		"F0 FD	986F",
		"F0 FE	9870",
		"F1 A1	9871",
		"F1 A3	9873",
		"F1 A2	9874",
		"C9 F7	98A8",
		"F1 A4	98AA",
		"F1 A5	98AF",
		"F1 A6	98B1",
		"F1 A7	98B6",
		"F1 A9	98C3",
		"F1 A8	98C4",
		"F1 AA	98C6",
		"C8 F4	98DB",
		"E6 CC	98DC",
		"BF A9	98DF",
		"B5 B2	98E2",
		"F1 AB	98E9",
		"F1 AC	98EB",
		"D2 AC	98ED",
		"DD BB	98EE",
		"C8 D3	98EF",
		"B0 FB	98F2",
		"B0 BB	98F4",
		"BB F4	98FC",
		"CB B0	98FD",
		"BE FE	98FE",
		"F1 AD	9903",
		"CC DF	9905",
		"F1 AE	9909",
		"CD DC	990A",
		"B1 C2	990C",
		"BB C1	9910",
		"F1 AF	9912",
		"B2 EE	9913",
		"F1 B0	9914",
		"F1 B1	9918",
		"F1 B3	991D",
		"F1 B4	991E",
		"F1 B6	9920",
		"F1 B2	9921",
		"F1 B5	9924",
		"B4 DB	9928",
		"F1 B7	992C",
		"F1 B8	992E",
		"F1 B9	993D",
		"F1 BA	993E",
		"F1 BB	9942",
		"F1 BD	9945",
		"F1 BC	9949",
		"F1 BF	994B",
		"F1 C2	994C",
		"F1 BE	9950",
		"F1 C0	9951",
		"F1 C1	9952",
		"F1 C3	9955",
		"B6 C2	9957",
		"BC F3	9996",
		"F1 C4	9997",
		"F1 C5	9998",
		"B9 E1	9999",
		"F1 C6	99A5",
		"B3 BE	99A8",
		"C7 CF	99AC",
		"F1 C7	99AD",
		"F1 C8	99AE",
		"C3 DA	99B3",
		"C6 EB	99B4",
		"F1 C9	99BC",
		"C7 FD	99C1",
		"C2 CC	99C4",
		"B1 D8	99C5",
		"B6 EE	99C6",
		"B6 EF	99C8",
		"C3 F3	99D0",
		"F1 CE	99D1",
		"B6 F0	99D2",
		"B2 EF	99D5",
		"F1 CD	99D8",
		"F1 CB	99DB",
		"F1 CC	99DD",
		"F1 CA	99DF",
		"F1 D8	99E2",
		"F1 CF	99ED",
		"F1 D0	99EE",
		"F1 D1	99F1",
		"F1 D2	99F2",
		"F1 D4	99F8",
		"F1 D3	99FB",
		"BD D9	99FF",
		"F1 D5	9A01",
		"F1 D7	9A05",
		"B5 B3	9A0E",
		"F1 D6	9A0F",
		"C1 FB	9A12",
		"B8 B3	9A13",
		"F1 D9	9A19",
		"C2 CD	9A28",
		"F1 DA	9A2B",
		"C6 AD	9A30",
		"F1 DB	9A37",
		"F1 E0	9A3E",
		"F1 DE	9A40",
		"F1 DD	9A42",
		"F1 DF	9A43",
		"F1 DC	9A45",
		"F1 E2	9A4D",
		"F1 E1	9A55",
		"F1 E4	9A57",
		"B6 C3	9A5A",
		"F1 E3	9A5B",
		"F1 E5	9A5F",
		"F1 E6	9A62",
		"F1 E8	9A64",
		"F1 E7	9A65",
		"F1 E9	9A69",
		"F1 EB	9A6A",
		"F1 EA	9A6B",
		"B9 FC	9AA8",
		"F1 EC	9AAD",
		"F1 ED	9AB0",
		"B3 BC	9AB8",
		"F1 EE	9ABC",
		"F1 EF	9AC0",
		"BF F1	9AC4",
		"F1 F0	9ACF",
		"F1 F1	9AD1",
		"F1 F2	9AD3",
		"F1 F3	9AD4",
		"B9 E2	9AD8",
		"F1 F4	9ADE",
		"F1 F5	9ADF",
		"F1 F6	9AE2",
		"F1 F7	9AE3",
		"F1 F8	9AE6",
		"C8 B1	9AEA",
		"F1 FA	9AEB",
		"C9 A6	9AED",
		"F1 FB	9AEE",
		"F1 F9	9AEF",
		"F1 FD	9AF1",
		"F1 FC	9AF4",
		"F1 FE	9AF7",
		"F2 A1	9AFB",
		"F2 A2	9B06",
		"F2 A3	9B18",
		"F2 A4	9B1A",
		"F2 A5	9B1F",
		"F2 A6	9B22",
		"F2 A7	9B23",
		"F2 A8	9B25",
		"F2 A9	9B27",
		"F2 AA	9B28",
		"F2 AB	9B29",
		"F2 AC	9B2A",
		"F2 AD	9B2E",
		"F2 AE	9B2F",
		"DD B5	9B31",
		"F2 AF	9B32",
		"E4 F8	9B3B",
		"B5 B4	9B3C",
		"B3 A1	9B41",
		"BA B2	9B42",
		"F2 B1	9B43",
		"F2 B0	9B44",
		"CC A5	9B45",
		"F2 B3	9B4D",
		"F2 B4	9B4E",
		"F2 B2	9B4F",
		"F2 B5	9B51",
		"CB E2	9B54",
		"F2 B6	9B58",
		"B5 FB	9B5A",
		"CF A5	9B6F",
		"F2 B7	9B74",
		"F2 B9	9B83",
		"B0 BE	9B8E",
		"F2 BA	9B91",
		"CA AB	9B92",
		"F2 B8	9B93",
		"F2 BB	9B96",
		"F2 BC	9B97",
		"F2 BD	9B9F",
		"F2 BE	9BA0",
		"F2 BF	9BA8",
		"CB EE	9BAA",
		"BB AD	9BAB",
		"BA FA	9BAD",
		"C1 AF	9BAE",
		"F2 C0	9BB4",
		"F2 C3	9BB9",
		"F2 C1	9BC0",
		"F2 C4	9BC6",
		"B8 F1	9BC9",
		"F2 C2	9BCA",
		"F2 C5	9BCF",
		"F2 C6	9BD1",
		"F2 C7	9BD2",
		"F2 CB	9BD4",
		"BB AA	9BD6",
		"C2 E4	9BDB",
		"F2 CC	9BE1",
		"F2 C9	9BE2",
		"F2 C8	9BE3",
		"F2 CA	9BE4",
		"B7 DF	9BE8",
		"F2 D0	9BF0",
		"F2 CF	9BF1",
		"F2 CE	9BF2",
		"B0 B3	9BF5",
		"F2 DA	9C04",
		"F2 D6	9C06",
		"F2 D7	9C08",
		"F2 D3	9C09",
		"F2 D9	9C0A",
		"F2 D5	9C0C",
		"B3 E2	9C0D",
		"CF CC	9C10",
		"F2 D8	9C12",
		"F2 D4	9C13",
		"F2 D2	9C14",
		"F2 D1	9C15",
		"F2 DC	9C1B",
		"F2 DF	9C21",
		"F2 DE	9C24",
		"F2 DD	9C25",
		"C9 C9	9C2D",
		"F2 DB	9C2E",
		"B0 F3	9C2F",
		"F2 E0	9C30",
		"F2 E2	9C32",
		"B3 EF	9C39",
		"F2 CD	9C3A",
		"B1 B7	9C3B",
		"F2 E4	9C3E",
		"F2 E3	9C46",
		"F2 E1	9C47",
		"C3 AD	9C48",
		"CB F0	9C52",
		"CE DA	9C57",
		"F2 E5	9C5A",
		"F2 E6	9C60",
		"F2 E7	9C67",
		"F2 E8	9C76",
		"F2 E9	9C78",
		"C4 BB	9CE5",
		"F2 EA	9CE7",
		"C8 B7	9CE9",
		"F2 EF	9CEB",
		"F2 EB	9CEC",
		"F2 EC	9CF0",
		"CB B1	9CF3",
		"CC C4	9CF4",
		"C6 D0	9CF6",
		"F2 F0	9D03",
		"F2 F1	9D06",
		"C6 BE	9D07",
		"F2 EE	9D08",
		"F2 ED	9D09",
		"B2 AA	9D0E",
		"F2 F9	9D12",
		"F2 F8	9D15",
		"B1 F5	9D1B",
		"F2 F6	9D1F",
		"F2 F5	9D23",
		"F2 F3	9D26",
		"B3 FB	9D28",
		"F2 F2	9D2A",
		"BC B2	9D2B",
		"B2 A9	9D2C",
		"B9 E3	9D3B",
		"F2 FC	9D3E",
		"F2 FB	9D3F",
		"F2 FA	9D41",
		"F2 F7	9D44",
		"F2 FD	9D46",
		"F2 FE	9D48",
		"F3 A5	9D50",
		"F3 A4	9D51",
		"F3 A6	9D59",
		"B1 AD	9D5C",
		"F3 A1	9D5D",
		"F3 A2	9D5E",
		"B9 F4	9D60",
		"CC B9	9D61",
		"F3 A3	9D64",
		"CB B2	9D6C",
		"F3 AB	9D6F",
		"F3 A7	9D72",
		"F3 AC	9D7A",
		"F3 A9	9D87",
		"F3 A8	9D89",
		"B7 DC	9D8F",
		"F3 AD	9D9A",
		"F3 AE	9DA4",
		"F3 AF	9DA9",
		"F3 AA	9DAB",
		"F2 F4	9DAF",
		"F3 B0	9DB2",
		"C4 E1	9DB4",
		"F3 B4	9DB8",
		"F3 B5	9DBA",
		"F3 B3	9DBB",
		"F3 B2	9DC1",
		"F3 B8	9DC2",
		"F3 B1	9DC4",
		"F3 B6	9DC6",
		"F3 B7	9DCF",
		"F3 BA	9DD3",
		"F3 B9	9DD9",
		"F3 BC	9DE6",
		"F3 BD	9DED",
		"F3 BE	9DEF",
		"CF C9	9DF2",
		"F3 BB	9DF8",
		"C2 EB	9DF9",
		"BA ED	9DFA",
		"F3 BF	9DFD",
		"F3 C0	9E1A",
		"F3 C1	9E1B",
		"F3 C2	9E1E",
		"F3 C3	9E75",
		"B8 B4	9E78",
		"F3 C4	9E79",
		"F3 C5	9E7D",
		"BC AF	9E7F",
		"F3 C6	9E81",
		"F3 C7	9E88",
		"F3 C8	9E8B",
		"F3 C9	9E8C",
		"F3 CC	9E91",
		"F3 CA	9E92",
		"CF BC	9E93",
		"F3 CB	9E95",
		"CE EF	9E97",
		"F3 CD	9E9D",
		"CE DB	9E9F",
		"F3 CE	9EA5",
		"C7 FE	9EA6",
		"F3 CF	9EA9",
		"F3 D1	9EAA",
		"F3 D2	9EAD",
		"F3 D0	9EB8",
		"B9 ED	9EB9",
		"CC CD	9EBA",
		"CB E3	9EBB",
		"D6 F7	9EBC",
		"DD E0	9EBE",
		"CB FB	9EBF",
		"B2 AB	9EC4",
		"F3 D4	9ECC",
		"B5 D0	9ECD",
		"F3 D5	9ECE",
		"F3 D6	9ECF",
		"F3 D7	9ED0",
		"B9 F5	9ED2",
		"F3 D8	9ED4",
		"E0 D4	9ED8",
		"CC DB	9ED9",
		"C2 E3	9EDB",
		"F3 D9	9EDC",
		"F3 DB	9EDD",
		"F3 DA	9EDE",
		"F3 DC	9EE0",
		"F3 DD	9EE5",
		"F3 DE	9EE8",
		"F3 DF	9EEF",
		"F3 E0	9EF4",
		"F3 E1	9EF6",
		"F3 E2	9EF7",
		"F3 E3	9EF9",
		"F3 E4	9EFB",
		"F3 E5	9EFC",
		"F3 E6	9EFD",
		"F3 E7	9F07",
		"F3 E8	9F08",
		"C5 A4	9F0E",
		"B8 DD	9F13",
		"F3 EA	9F15",
		"C1 CD	9F20",
		"F3 EB	9F21",
		"F3 EC	9F2C",
		"C9 A1	9F3B",
		"F3 ED	9F3E",
		"F3 EE	9F4A",
		"E3 B7	9F4B",
		"EC DA	9F4E",
		"F0 ED	9F4F",
		"F3 EF	9F52",
		"F3 F0	9F54",
		"F3 F2	9F5F",
		"F3 F3	9F60",
		"F3 F4	9F61",
		"CE F0	9F62",
		"F3 F1	9F63",
		"F3 F5	9F66",
		"F3 F6	9F67",
		"F3 F8	9F6A",
		"F3 F7	9F6C",
		"F3 FA	9F72",
		"F3 FB	9F76",
		"F3 F9	9F77",
		"CE B6	9F8D",
		"F3 FC	9F95",
		"F3 FD	9F9C",
		"E3 D4	9F9D",
		"F3 FE	9FA0",
		"A1 AA	FF01",
		"A1 F4	FF03",
		"A1 F0	FF04",
		"A1 F3	FF05",
		"A1 F5	FF06",
		"A1 CA	FF08",
		"A1 CB	FF09",
		"A1 F6	FF0A",
		"A1 DC	FF0B",
		"A1 A4	FF0C",
		"A1 A5	FF0E",
		"A1 BF	FF0F",
		"A3 B0	FF10",
		"A3 B1	FF11",
		"A3 B2	FF12",
		"A3 B3	FF13",
		"A3 B4	FF14",
		"A3 B5	FF15",
		"A3 B6	FF16",
		"A3 B7	FF17",
		"A3 B8	FF18",
		"A3 B9	FF19",
		"A1 A7	FF1A",
		"A1 A8	FF1B",
		"A1 E3	FF1C",
		"A1 E1	FF1D",
		"A1 E4	FF1E",
		"A1 A9	FF1F",
		"A1 F7	FF20",
		"A3 C1	FF21",
		"A3 C2	FF22",
		"A3 C3	FF23",
		"A3 C4	FF24",
		"A3 C5	FF25",
		"A3 C6	FF26",
		"A3 C7	FF27",
		"A3 C8	FF28",
		"A3 C9	FF29",
		"A3 CA	FF2A",
		"A3 CB	FF2B",
		"A3 CC	FF2C",
		"A3 CD	FF2D",
		"A3 CE	FF2E",
		"A3 CF	FF2F",
		"A3 D0	FF30",
		"A3 D1	FF31",
		"A3 D2	FF32",
		"A3 D3	FF33",
		"A3 D4	FF34",
		"A3 D5	FF35",
		"A3 D6	FF36",
		"A3 D7	FF37",
		"A3 D8	FF38",
		"A3 D9	FF39",
		"A3 DA	FF3A",
		"A1 CE	FF3B",
		"A1 C0	FF3C",
		"A1 CF	FF3D",
		"A1 B0	FF3E",
		"A1 B2	FF3F",
		"A1 AE	FF40",
		"A3 E1	FF41",
		"A3 E2	FF42",
		"A3 E3	FF43",
		"A3 E4	FF44",
		"A3 E5	FF45",
		"A3 E6	FF46",
		"A3 E7	FF47",
		"A3 E8	FF48",
		"A3 E9	FF49",
		"A3 EA	FF4A",
		"A3 EB	FF4B",
		"A3 EC	FF4C",
		"A3 ED	FF4D",
		"A3 EE	FF4E",
		"A3 EF	FF4F",
		"A3 F0	FF50",
		"A3 F1	FF51",
		"A3 F2	FF52",
		"A3 F3	FF53",
		"A3 F4	FF54",
		"A3 F5	FF55",
		"A3 F6	FF56",
		"A3 F7	FF57",
		"A3 F8	FF58",
		"A3 F9	FF59",
		"A3 FA	FF5A",
		"A1 D0	FF5B",
		"A1 C3	FF5C",
		"A1 D1	FF5D",
		"A1 B1	FFE3",
		"A1 EF	FFE5",
};