implement gencp932;

include "sys.m";
include "draw.m";

gencp932 : module {
	init: fn(ctxt: ref Draw->Context, args: list of string);
};

DATAFILE : con "/lib/convcs/cp932";

CP932PAGES : con 45;		# 81..84, 87..9f, e0..ea, ed..ee, fa..fc
CP932PAGESZ : con 189;		# 40..fc (including 7f)
CP932CHAR0 : con 16r40;

BADCHAR : con 16rFFFD;

init(nil: ref Draw->Context, nil: list of string)
{
	sys := load Sys Sys->PATH;
	fd := sys->create(DATAFILE, Sys->OWRITE, 8r644);
	if (fd == nil) {
		sys->print("cannot create %s: %r\n", DATAFILE);
		return;
	}

	# pre-calculate DBCS page numbers to mapping file page numbers
	dbcsoff := array [256] of { * => -1 };
	p := 0;
	for (i := 16r81; i <= 16r84; i++)
		dbcsoff[i] = p++;
	for (i = 16r87; i <= 16r9f; i++)
		dbcsoff[i] = p++;
	for (i = 16re0; i <= 16rea; i++)
		dbcsoff[i] = p++;
	for (i = 16red; i <= 16ree; i++)
		dbcsoff[i] = p++;
	for (i = 16rfa; i <= 16rfc; i++)
		dbcsoff[i] = p++;

	pages := array [CP932PAGES * CP932PAGESZ] of { * => BADCHAR };

	for (i = 0; i < len cp932; i++) {
		(nil, toks) := sys->tokenize(cp932[i], "\t");
		(bytes, ucode) := (hd toks, hd tl toks);
		u := hex2int(ucode);
		(nil, blist) := sys->tokenize(bytes, " ");
		b1 := hex2int(hd blist);
		b2 := hex2int(hd tl blist);

		page := dbcsoff[b1];
		if (page == -1) {
			sys->print("conversion error\n");
			raise "fail:bad data";
		}
		char := b2 - CP932CHAR0;
		pages[(page * CP932PAGESZ) + char] = u;
	}

	s := "";
	for (i = 0; i < len pages; i++)
		s[i] = pages[i];
	pages = nil;
	buf := array of byte s;
	sys->write(fd, buf, len buf);
}

hex2int(s: string): int
{
	n := 0;
	for (i := 0; i < len s; i++) {
		case s[i] {
		'0' to '9' =>
			n = 16*n + s[i] - '0';
		'A' to 'F' =>
			n = 16*n + s[i] + 10 - 'A';
		'a' to 'f' =>
			n = 16*n + s[i] + 10 - 'a';
		* =>
			return n;
		}
	}
	return n;
}

# This data derived directly from the unicode mapping file shiftjis.txt
# available from http://www.unicode.org/Public//MAPPINGS

cp932 := array [] of {
	"81 40	3000",	#IDEOGRAPHIC SPACE
	"81 41	3001",	#IDEOGRAPHIC COMMA
	"81 42	3002",	#IDEOGRAPHIC FULL STOP
	"81 43	FF0C",	#FULLWIDTH COMMA
	"81 44	FF0E",	#FULLWIDTH FULL STOP
	"81 45	30FB",	#KATAKANA MIDDLE DOT
	"81 46	FF1A",	#FULLWIDTH COLON
	"81 47	FF1B",	#FULLWIDTH SEMICOLON
	"81 48	FF1F",	#FULLWIDTH QUESTION MARK
	"81 49	FF01",	#FULLWIDTH EXCLAMATION MARK
	"81 4A	309B",	#KATAKANA-HIRAGANA VOICED SOUND MARK
	"81 4B	309C",	#KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
	"81 4C	00B4",	#ACUTE ACCENT
	"81 4D	FF40",	#FULLWIDTH GRAVE ACCENT
	"81 4E	00A8",	#DIAERESIS
	"81 4F	FF3E",	#FULLWIDTH CIRCUMFLEX ACCENT
	"81 50	FFE3",	#FULLWIDTH MACRON
	"81 51	FF3F",	#FULLWIDTH LOW LINE
	"81 52	30FD",	#KATAKANA ITERATION MARK
	"81 53	30FE",	#KATAKANA VOICED ITERATION MARK
	"81 54	309D",	#HIRAGANA ITERATION MARK
	"81 55	309E",	#HIRAGANA VOICED ITERATION MARK
	"81 56	3003",	#DITTO MARK
	"81 57	4EDD",	#CJK UNIFIED IDEOGRAPH
	"81 58	3005",	#IDEOGRAPHIC ITERATION MARK
	"81 59	3006",	#IDEOGRAPHIC CLOSING MARK
	"81 5A	3007",	#IDEOGRAPHIC NUMBER ZERO
	"81 5B	30FC",	#KATAKANA-HIRAGANA PROLONGED SOUND MARK
	"81 5C	2015",	#HORIZONTAL BAR
	"81 5D	2010",	#HYPHEN
	"81 5E	FF0F",	#FULLWIDTH SOLIDUS
	"81 5F	FF3C",	#FULLWIDTH REVERSE SOLIDUS
	"81 60	FF5E",	#FULLWIDTH TILDE
	"81 61	2225",	#PARALLEL TO
	"81 62	FF5C",	#FULLWIDTH VERTICAL LINE
	"81 63	2026",	#HORIZONTAL ELLIPSIS
	"81 64	2025",	#TWO DOT LEADER
	"81 65	2018",	#LEFT SINGLE QUOTATION MARK
	"81 66	2019",	#RIGHT SINGLE QUOTATION MARK
	"81 67	201C",	#LEFT DOUBLE QUOTATION MARK
	"81 68	201D",	#RIGHT DOUBLE QUOTATION MARK
	"81 69	FF08",	#FULLWIDTH LEFT PARENTHESIS
	"81 6A	FF09",	#FULLWIDTH RIGHT PARENTHESIS
	"81 6B	3014",	#LEFT TORTOISE SHELL BRACKET
	"81 6C	3015",	#RIGHT TORTOISE SHELL BRACKET
	"81 6D	FF3B",	#FULLWIDTH LEFT SQUARE BRACKET
	"81 6E	FF3D",	#FULLWIDTH RIGHT SQUARE BRACKET
	"81 6F	FF5B",	#FULLWIDTH LEFT CURLY BRACKET
	"81 70	FF5D",	#FULLWIDTH RIGHT CURLY BRACKET
	"81 71	3008",	#LEFT ANGLE BRACKET
	"81 72	3009",	#RIGHT ANGLE BRACKET
	"81 73	300A",	#LEFT DOUBLE ANGLE BRACKET
	"81 74	300B",	#RIGHT DOUBLE ANGLE BRACKET
	"81 75	300C",	#LEFT CORNER BRACKET
	"81 76	300D",	#RIGHT CORNER BRACKET
	"81 77	300E",	#LEFT WHITE CORNER BRACKET
	"81 78	300F",	#RIGHT WHITE CORNER BRACKET
	"81 79	3010",	#LEFT BLACK LENTICULAR BRACKET
	"81 7A	3011",	#RIGHT BLACK LENTICULAR BRACKET
	"81 7B	FF0B",	#FULLWIDTH PLUS SIGN
	"81 7C	FF0D",	#FULLWIDTH HYPHEN-MINUS
	"81 7D	00B1",	#PLUS-MINUS SIGN
	"81 7E	00D7",	#MULTIPLICATION SIGN
	"81 80	00F7",	#DIVISION SIGN
	"81 81	FF1D",	#FULLWIDTH EQUALS SIGN
	"81 82	2260",	#NOT EQUAL TO
	"81 83	FF1C",	#FULLWIDTH LESS-THAN SIGN
	"81 84	FF1E",	#FULLWIDTH GREATER-THAN SIGN
	"81 85	2266",	#LESS-THAN OVER EQUAL TO
	"81 86	2267",	#GREATER-THAN OVER EQUAL TO
	"81 87	221E",	#INFINITY
	"81 88	2234",	#THEREFORE
	"81 89	2642",	#MALE SIGN
	"81 8A	2640",	#FEMALE SIGN
	"81 8B	00B0",	#DEGREE SIGN
	"81 8C	2032",	#PRIME
	"81 8D	2033",	#DOUBLE PRIME
	"81 8E	2103",	#DEGREE CELSIUS
	"81 8F	FFE5",	#FULLWIDTH YEN SIGN
	"81 90	FF04",	#FULLWIDTH DOLLAR SIGN
	"81 91	FFE0",	#FULLWIDTH CENT SIGN
	"81 92	FFE1",	#FULLWIDTH POUND SIGN
	"81 93	FF05",	#FULLWIDTH PERCENT SIGN
	"81 94	FF03",	#FULLWIDTH NUMBER SIGN
	"81 95	FF06",	#FULLWIDTH AMPERSAND
	"81 96	FF0A",	#FULLWIDTH ASTERISK
	"81 97	FF20",	#FULLWIDTH COMMERCIAL AT
	"81 98	00A7",	#SECTION SIGN
	"81 99	2606",	#WHITE STAR
	"81 9A	2605",	#BLACK STAR
	"81 9B	25CB",	#WHITE CIRCLE
	"81 9C	25CF",	#BLACK CIRCLE
	"81 9D	25CE",	#BULLSEYE
	"81 9E	25C7",	#WHITE DIAMOND
	"81 9F	25C6",	#BLACK DIAMOND
	"81 A0	25A1",	#WHITE SQUARE
	"81 A1	25A0",	#BLACK SQUARE
	"81 A2	25B3",	#WHITE UP-POINTING TRIANGLE
	"81 A3	25B2",	#BLACK UP-POINTING TRIANGLE
	"81 A4	25BD",	#WHITE DOWN-POINTING TRIANGLE
	"81 A5	25BC",	#BLACK DOWN-POINTING TRIANGLE
	"81 A6	203B",	#REFERENCE MARK
	"81 A7	3012",	#POSTAL MARK
	"81 A8	2192",	#RIGHTWARDS ARROW
	"81 A9	2190",	#LEFTWARDS ARROW
	"81 AA	2191",	#UPWARDS ARROW
	"81 AB	2193",	#DOWNWARDS ARROW
	"81 AC	3013",	#GETA MARK
	"81 B8	2208",	#ELEMENT OF
	"81 B9	220B",	#CONTAINS AS MEMBER
	"81 BA	2286",	#SUBSET OF OR EQUAL TO
	"81 BB	2287",	#SUPERSET OF OR EQUAL TO
	"81 BC	2282",	#SUBSET OF
	"81 BD	2283",	#SUPERSET OF
	"81 BE	222A",	#UNION
	"81 BF	2229",	#INTERSECTION
	"81 C8	2227",	#LOGICAL AND
	"81 C9	2228",	#LOGICAL OR
	"81 CA	FFE2",	#FULLWIDTH NOT SIGN
	"81 CB	21D2",	#RIGHTWARDS DOUBLE ARROW
	"81 CC	21D4",	#LEFT RIGHT DOUBLE ARROW
	"81 CD	2200",	#FOR ALL
	"81 CE	2203",	#THERE EXISTS
	"81 DA	2220",	#ANGLE
	"81 DB	22A5",	#UP TACK
	"81 DC	2312",	#ARC
	"81 DD	2202",	#PARTIAL DIFFERENTIAL
	"81 DE	2207",	#NABLA
	"81 DF	2261",	#IDENTICAL TO
	"81 E0	2252",	#APPROXIMATELY EQUAL TO OR THE IMAGE OF
	"81 E1	226A",	#MUCH LESS-THAN
	"81 E2	226B",	#MUCH GREATER-THAN
	"81 E3	221A",	#SQUARE ROOT
	"81 E4	223D",	#REVERSED TILDE
	"81 E5	221D",	#PROPORTIONAL TO
	"81 E6	2235",	#BECAUSE
	"81 E7	222B",	#INTEGRAL
	"81 E8	222C",	#DOUBLE INTEGRAL
	"81 F0	212B",	#ANGSTROM SIGN
	"81 F1	2030",	#PER MILLE SIGN
	"81 F2	266F",	#MUSIC SHARP SIGN
	"81 F3	266D",	#MUSIC FLAT SIGN
	"81 F4	266A",	#EIGHTH NOTE
	"81 F5	2020",	#DAGGER
	"81 F6	2021",	#DOUBLE DAGGER
	"81 F7	00B6",	#PILCROW SIGN
	"81 FC	25EF",	#LARGE CIRCLE
	"82 4F	FF10",	#FULLWIDTH DIGIT ZERO
	"82 50	FF11",	#FULLWIDTH DIGIT ONE
	"82 51	FF12",	#FULLWIDTH DIGIT TWO
	"82 52	FF13",	#FULLWIDTH DIGIT THREE
	"82 53	FF14",	#FULLWIDTH DIGIT FOUR
	"82 54	FF15",	#FULLWIDTH DIGIT FIVE
	"82 55	FF16",	#FULLWIDTH DIGIT SIX
	"82 56	FF17",	#FULLWIDTH DIGIT SEVEN
	"82 57	FF18",	#FULLWIDTH DIGIT EIGHT
	"82 58	FF19",	#FULLWIDTH DIGIT NINE
	"82 60	FF21",	#FULLWIDTH LATIN CAPITAL LETTER A
	"82 61	FF22",	#FULLWIDTH LATIN CAPITAL LETTER B
	"82 62	FF23",	#FULLWIDTH LATIN CAPITAL LETTER C
	"82 63	FF24",	#FULLWIDTH LATIN CAPITAL LETTER D
	"82 64	FF25",	#FULLWIDTH LATIN CAPITAL LETTER E
	"82 65	FF26",	#FULLWIDTH LATIN CAPITAL LETTER F
	"82 66	FF27",	#FULLWIDTH LATIN CAPITAL LETTER G
	"82 67	FF28",	#FULLWIDTH LATIN CAPITAL LETTER H
	"82 68	FF29",	#FULLWIDTH LATIN CAPITAL LETTER I
	"82 69	FF2A",	#FULLWIDTH LATIN CAPITAL LETTER J
	"82 6A	FF2B",	#FULLWIDTH LATIN CAPITAL LETTER K
	"82 6B	FF2C",	#FULLWIDTH LATIN CAPITAL LETTER L
	"82 6C	FF2D",	#FULLWIDTH LATIN CAPITAL LETTER M
	"82 6D	FF2E",	#FULLWIDTH LATIN CAPITAL LETTER N
	"82 6E	FF2F",	#FULLWIDTH LATIN CAPITAL LETTER O
	"82 6F	FF30",	#FULLWIDTH LATIN CAPITAL LETTER P
	"82 70	FF31",	#FULLWIDTH LATIN CAPITAL LETTER Q
	"82 71	FF32",	#FULLWIDTH LATIN CAPITAL LETTER R
	"82 72	FF33",	#FULLWIDTH LATIN CAPITAL LETTER S
	"82 73	FF34",	#FULLWIDTH LATIN CAPITAL LETTER T
	"82 74	FF35",	#FULLWIDTH LATIN CAPITAL LETTER U
	"82 75	FF36",	#FULLWIDTH LATIN CAPITAL LETTER V
	"82 76	FF37",	#FULLWIDTH LATIN CAPITAL LETTER W
	"82 77	FF38",	#FULLWIDTH LATIN CAPITAL LETTER X
	"82 78	FF39",	#FULLWIDTH LATIN CAPITAL LETTER Y
	"82 79	FF3A",	#FULLWIDTH LATIN CAPITAL LETTER Z
	"82 81	FF41",	#FULLWIDTH LATIN SMALL LETTER A
	"82 82	FF42",	#FULLWIDTH LATIN SMALL LETTER B
	"82 83	FF43",	#FULLWIDTH LATIN SMALL LETTER C
	"82 84	FF44",	#FULLWIDTH LATIN SMALL LETTER D
	"82 85	FF45",	#FULLWIDTH LATIN SMALL LETTER E
	"82 86	FF46",	#FULLWIDTH LATIN SMALL LETTER F
	"82 87	FF47",	#FULLWIDTH LATIN SMALL LETTER G
	"82 88	FF48",	#FULLWIDTH LATIN SMALL LETTER H
	"82 89	FF49",	#FULLWIDTH LATIN SMALL LETTER I
	"82 8A	FF4A",	#FULLWIDTH LATIN SMALL LETTER J
	"82 8B	FF4B",	#FULLWIDTH LATIN SMALL LETTER K
	"82 8C	FF4C",	#FULLWIDTH LATIN SMALL LETTER L
	"82 8D	FF4D",	#FULLWIDTH LATIN SMALL LETTER M
	"82 8E	FF4E",	#FULLWIDTH LATIN SMALL LETTER N
	"82 8F	FF4F",	#FULLWIDTH LATIN SMALL LETTER O
	"82 90	FF50",	#FULLWIDTH LATIN SMALL LETTER P
	"82 91	FF51",	#FULLWIDTH LATIN SMALL LETTER Q
	"82 92	FF52",	#FULLWIDTH LATIN SMALL LETTER R
	"82 93	FF53",	#FULLWIDTH LATIN SMALL LETTER S
	"82 94	FF54",	#FULLWIDTH LATIN SMALL LETTER T
	"82 95	FF55",	#FULLWIDTH LATIN SMALL LETTER U
	"82 96	FF56",	#FULLWIDTH LATIN SMALL LETTER V
	"82 97	FF57",	#FULLWIDTH LATIN SMALL LETTER W
	"82 98	FF58",	#FULLWIDTH LATIN SMALL LETTER X
	"82 99	FF59",	#FULLWIDTH LATIN SMALL LETTER Y
	"82 9A	FF5A",	#FULLWIDTH LATIN SMALL LETTER Z
	"82 9F	3041",	#HIRAGANA LETTER SMALL A
	"82 A0	3042",	#HIRAGANA LETTER A
	"82 A1	3043",	#HIRAGANA LETTER SMALL I
	"82 A2	3044",	#HIRAGANA LETTER I
	"82 A3	3045",	#HIRAGANA LETTER SMALL U
	"82 A4	3046",	#HIRAGANA LETTER U
	"82 A5	3047",	#HIRAGANA LETTER SMALL E
	"82 A6	3048",	#HIRAGANA LETTER E
	"82 A7	3049",	#HIRAGANA LETTER SMALL O
	"82 A8	304A",	#HIRAGANA LETTER O
	"82 A9	304B",	#HIRAGANA LETTER KA
	"82 AA	304C",	#HIRAGANA LETTER GA
	"82 AB	304D",	#HIRAGANA LETTER KI
	"82 AC	304E",	#HIRAGANA LETTER GI
	"82 AD	304F",	#HIRAGANA LETTER KU
	"82 AE	3050",	#HIRAGANA LETTER GU
	"82 AF	3051",	#HIRAGANA LETTER KE
	"82 B0	3052",	#HIRAGANA LETTER GE
	"82 B1	3053",	#HIRAGANA LETTER KO
	"82 B2	3054",	#HIRAGANA LETTER GO
	"82 B3	3055",	#HIRAGANA LETTER SA
	"82 B4	3056",	#HIRAGANA LETTER ZA
	"82 B5	3057",	#HIRAGANA LETTER SI
	"82 B6	3058",	#HIRAGANA LETTER ZI
	"82 B7	3059",	#HIRAGANA LETTER SU
	"82 B8	305A",	#HIRAGANA LETTER ZU
	"82 B9	305B",	#HIRAGANA LETTER SE
	"82 BA	305C",	#HIRAGANA LETTER ZE
	"82 BB	305D",	#HIRAGANA LETTER SO
	"82 BC	305E",	#HIRAGANA LETTER ZO
	"82 BD	305F",	#HIRAGANA LETTER TA
	"82 BE	3060",	#HIRAGANA LETTER DA
	"82 BF	3061",	#HIRAGANA LETTER TI
	"82 C0	3062",	#HIRAGANA LETTER DI
	"82 C1	3063",	#HIRAGANA LETTER SMALL TU
	"82 C2	3064",	#HIRAGANA LETTER TU
	"82 C3	3065",	#HIRAGANA LETTER DU
	"82 C4	3066",	#HIRAGANA LETTER TE
	"82 C5	3067",	#HIRAGANA LETTER DE
	"82 C6	3068",	#HIRAGANA LETTER TO
	"82 C7	3069",	#HIRAGANA LETTER DO
	"82 C8	306A",	#HIRAGANA LETTER NA
	"82 C9	306B",	#HIRAGANA LETTER NI
	"82 CA	306C",	#HIRAGANA LETTER NU
	"82 CB	306D",	#HIRAGANA LETTER NE
	"82 CC	306E",	#HIRAGANA LETTER NO
	"82 CD	306F",	#HIRAGANA LETTER HA
	"82 CE	3070",	#HIRAGANA LETTER BA
	"82 CF	3071",	#HIRAGANA LETTER PA
	"82 D0	3072",	#HIRAGANA LETTER HI
	"82 D1	3073",	#HIRAGANA LETTER BI
	"82 D2	3074",	#HIRAGANA LETTER PI
	"82 D3	3075",	#HIRAGANA LETTER HU
	"82 D4	3076",	#HIRAGANA LETTER BU
	"82 D5	3077",	#HIRAGANA LETTER PU
	"82 D6	3078",	#HIRAGANA LETTER HE
	"82 D7	3079",	#HIRAGANA LETTER BE
	"82 D8	307A",	#HIRAGANA LETTER PE
	"82 D9	307B",	#HIRAGANA LETTER HO
	"82 DA	307C",	#HIRAGANA LETTER BO
	"82 DB	307D",	#HIRAGANA LETTER PO
	"82 DC	307E",	#HIRAGANA LETTER MA
	"82 DD	307F",	#HIRAGANA LETTER MI
	"82 DE	3080",	#HIRAGANA LETTER MU
	"82 DF	3081",	#HIRAGANA LETTER ME
	"82 E0	3082",	#HIRAGANA LETTER MO
	"82 E1	3083",	#HIRAGANA LETTER SMALL YA
	"82 E2	3084",	#HIRAGANA LETTER YA
	"82 E3	3085",	#HIRAGANA LETTER SMALL YU
	"82 E4	3086",	#HIRAGANA LETTER YU
	"82 E5	3087",	#HIRAGANA LETTER SMALL YO
	"82 E6	3088",	#HIRAGANA LETTER YO
	"82 E7	3089",	#HIRAGANA LETTER RA
	"82 E8	308A",	#HIRAGANA LETTER RI
	"82 E9	308B",	#HIRAGANA LETTER RU
	"82 EA	308C",	#HIRAGANA LETTER RE
	"82 EB	308D",	#HIRAGANA LETTER RO
	"82 EC	308E",	#HIRAGANA LETTER SMALL WA
	"82 ED	308F",	#HIRAGANA LETTER WA
	"82 EE	3090",	#HIRAGANA LETTER WI
	"82 EF	3091",	#HIRAGANA LETTER WE
	"82 F0	3092",	#HIRAGANA LETTER WO
	"82 F1	3093",	#HIRAGANA LETTER N
	"83 40	30A1",	#KATAKANA LETTER SMALL A
	"83 41	30A2",	#KATAKANA LETTER A
	"83 42	30A3",	#KATAKANA LETTER SMALL I
	"83 43	30A4",	#KATAKANA LETTER I
	"83 44	30A5",	#KATAKANA LETTER SMALL U
	"83 45	30A6",	#KATAKANA LETTER U
	"83 46	30A7",	#KATAKANA LETTER SMALL E
	"83 47	30A8",	#KATAKANA LETTER E
	"83 48	30A9",	#KATAKANA LETTER SMALL O
	"83 49	30AA",	#KATAKANA LETTER O
	"83 4A	30AB",	#KATAKANA LETTER KA
	"83 4B	30AC",	#KATAKANA LETTER GA
	"83 4C	30AD",	#KATAKANA LETTER KI
	"83 4D	30AE",	#KATAKANA LETTER GI
	"83 4E	30AF",	#KATAKANA LETTER KU
	"83 4F	30B0",	#KATAKANA LETTER GU
	"83 50	30B1",	#KATAKANA LETTER KE
	"83 51	30B2",	#KATAKANA LETTER GE
	"83 52	30B3",	#KATAKANA LETTER KO
	"83 53	30B4",	#KATAKANA LETTER GO
	"83 54	30B5",	#KATAKANA LETTER SA
	"83 55	30B6",	#KATAKANA LETTER ZA
	"83 56	30B7",	#KATAKANA LETTER SI
	"83 57	30B8",	#KATAKANA LETTER ZI
	"83 58	30B9",	#KATAKANA LETTER SU
	"83 59	30BA",	#KATAKANA LETTER ZU
	"83 5A	30BB",	#KATAKANA LETTER SE
	"83 5B	30BC",	#KATAKANA LETTER ZE
	"83 5C	30BD",	#KATAKANA LETTER SO
	"83 5D	30BE",	#KATAKANA LETTER ZO
	"83 5E	30BF",	#KATAKANA LETTER TA
	"83 5F	30C0",	#KATAKANA LETTER DA
	"83 60	30C1",	#KATAKANA LETTER TI
	"83 61	30C2",	#KATAKANA LETTER DI
	"83 62	30C3",	#KATAKANA LETTER SMALL TU
	"83 63	30C4",	#KATAKANA LETTER TU
	"83 64	30C5",	#KATAKANA LETTER DU
	"83 65	30C6",	#KATAKANA LETTER TE
	"83 66	30C7",	#KATAKANA LETTER DE
	"83 67	30C8",	#KATAKANA LETTER TO
	"83 68	30C9",	#KATAKANA LETTER DO
	"83 69	30CA",	#KATAKANA LETTER NA
	"83 6A	30CB",	#KATAKANA LETTER NI
	"83 6B	30CC",	#KATAKANA LETTER NU
	"83 6C	30CD",	#KATAKANA LETTER NE
	"83 6D	30CE",	#KATAKANA LETTER NO
	"83 6E	30CF",	#KATAKANA LETTER HA
	"83 6F	30D0",	#KATAKANA LETTER BA
	"83 70	30D1",	#KATAKANA LETTER PA
	"83 71	30D2",	#KATAKANA LETTER HI
	"83 72	30D3",	#KATAKANA LETTER BI
	"83 73	30D4",	#KATAKANA LETTER PI
	"83 74	30D5",	#KATAKANA LETTER HU
	"83 75	30D6",	#KATAKANA LETTER BU
	"83 76	30D7",	#KATAKANA LETTER PU
	"83 77	30D8",	#KATAKANA LETTER HE
	"83 78	30D9",	#KATAKANA LETTER BE
	"83 79	30DA",	#KATAKANA LETTER PE
	"83 7A	30DB",	#KATAKANA LETTER HO
	"83 7B	30DC",	#KATAKANA LETTER BO
	"83 7C	30DD",	#KATAKANA LETTER PO
	"83 7D	30DE",	#KATAKANA LETTER MA
	"83 7E	30DF",	#KATAKANA LETTER MI
	"83 80	30E0",	#KATAKANA LETTER MU
	"83 81	30E1",	#KATAKANA LETTER ME
	"83 82	30E2",	#KATAKANA LETTER MO
	"83 83	30E3",	#KATAKANA LETTER SMALL YA
	"83 84	30E4",	#KATAKANA LETTER YA
	"83 85	30E5",	#KATAKANA LETTER SMALL YU
	"83 86	30E6",	#KATAKANA LETTER YU
	"83 87	30E7",	#KATAKANA LETTER SMALL YO
	"83 88	30E8",	#KATAKANA LETTER YO
	"83 89	30E9",	#KATAKANA LETTER RA
	"83 8A	30EA",	#KATAKANA LETTER RI
	"83 8B	30EB",	#KATAKANA LETTER RU
	"83 8C	30EC",	#KATAKANA LETTER RE
	"83 8D	30ED",	#KATAKANA LETTER RO
	"83 8E	30EE",	#KATAKANA LETTER SMALL WA
	"83 8F	30EF",	#KATAKANA LETTER WA
	"83 90	30F0",	#KATAKANA LETTER WI
	"83 91	30F1",	#KATAKANA LETTER WE
	"83 92	30F2",	#KATAKANA LETTER WO
	"83 93	30F3",	#KATAKANA LETTER N
	"83 94	30F4",	#KATAKANA LETTER VU
	"83 95	30F5",	#KATAKANA LETTER SMALL KA
	"83 96	30F6",	#KATAKANA LETTER SMALL KE
	"83 9F	0391",	#GREEK CAPITAL LETTER ALPHA
	"83 A0	0392",	#GREEK CAPITAL LETTER BETA
	"83 A1	0393",	#GREEK CAPITAL LETTER GAMMA
	"83 A2	0394",	#GREEK CAPITAL LETTER DELTA
	"83 A3	0395",	#GREEK CAPITAL LETTER EPSILON
	"83 A4	0396",	#GREEK CAPITAL LETTER ZETA
	"83 A5	0397",	#GREEK CAPITAL LETTER ETA
	"83 A6	0398",	#GREEK CAPITAL LETTER THETA
	"83 A7	0399",	#GREEK CAPITAL LETTER IOTA
	"83 A8	039A",	#GREEK CAPITAL LETTER KAPPA
	"83 A9	039B",	#GREEK CAPITAL LETTER LAMDA
	"83 AA	039C",	#GREEK CAPITAL LETTER MU
	"83 AB	039D",	#GREEK CAPITAL LETTER NU
	"83 AC	039E",	#GREEK CAPITAL LETTER XI
	"83 AD	039F",	#GREEK CAPITAL LETTER OMICRON
	"83 AE	03A0",	#GREEK CAPITAL LETTER PI
	"83 AF	03A1",	#GREEK CAPITAL LETTER RHO
	"83 B0	03A3",	#GREEK CAPITAL LETTER SIGMA
	"83 B1	03A4",	#GREEK CAPITAL LETTER TAU
	"83 B2	03A5",	#GREEK CAPITAL LETTER UPSILON
	"83 B3	03A6",	#GREEK CAPITAL LETTER PHI
	"83 B4	03A7",	#GREEK CAPITAL LETTER CHI
	"83 B5	03A8",	#GREEK CAPITAL LETTER PSI
	"83 B6	03A9",	#GREEK CAPITAL LETTER OMEGA
	"83 BF	03B1",	#GREEK SMALL LETTER ALPHA
	"83 C0	03B2",	#GREEK SMALL LETTER BETA
	"83 C1	03B3",	#GREEK SMALL LETTER GAMMA
	"83 C2	03B4",	#GREEK SMALL LETTER DELTA
	"83 C3	03B5",	#GREEK SMALL LETTER EPSILON
	"83 C4	03B6",	#GREEK SMALL LETTER ZETA
	"83 C5	03B7",	#GREEK SMALL LETTER ETA
	"83 C6	03B8",	#GREEK SMALL LETTER THETA
	"83 C7	03B9",	#GREEK SMALL LETTER IOTA
	"83 C8	03BA",	#GREEK SMALL LETTER KAPPA
	"83 C9	03BB",	#GREEK SMALL LETTER LAMDA
	"83 CA	03BC",	#GREEK SMALL LETTER MU
	"83 CB	03BD",	#GREEK SMALL LETTER NU
	"83 CC	03BE",	#GREEK SMALL LETTER XI
	"83 CD	03BF",	#GREEK SMALL LETTER OMICRON
	"83 CE	03C0",	#GREEK SMALL LETTER PI
	"83 CF	03C1",	#GREEK SMALL LETTER RHO
	"83 D0	03C3",	#GREEK SMALL LETTER SIGMA
	"83 D1	03C4",	#GREEK SMALL LETTER TAU
	"83 D2	03C5",	#GREEK SMALL LETTER UPSILON
	"83 D3	03C6",	#GREEK SMALL LETTER PHI
	"83 D4	03C7",	#GREEK SMALL LETTER CHI
	"83 D5	03C8",	#GREEK SMALL LETTER PSI
	"83 D6	03C9",	#GREEK SMALL LETTER OMEGA
	"84 40	0410",	#CYRILLIC CAPITAL LETTER A
	"84 41	0411",	#CYRILLIC CAPITAL LETTER BE
	"84 42	0412",	#CYRILLIC CAPITAL LETTER VE
	"84 43	0413",	#CYRILLIC CAPITAL LETTER GHE
	"84 44	0414",	#CYRILLIC CAPITAL LETTER DE
	"84 45	0415",	#CYRILLIC CAPITAL LETTER IE
	"84 46	0401",	#CYRILLIC CAPITAL LETTER IO
	"84 47	0416",	#CYRILLIC CAPITAL LETTER ZHE
	"84 48	0417",	#CYRILLIC CAPITAL LETTER ZE
	"84 49	0418",	#CYRILLIC CAPITAL LETTER I
	"84 4A	0419",	#CYRILLIC CAPITAL LETTER SHORT I
	"84 4B	041A",	#CYRILLIC CAPITAL LETTER KA
	"84 4C	041B",	#CYRILLIC CAPITAL LETTER EL
	"84 4D	041C",	#CYRILLIC CAPITAL LETTER EM
	"84 4E	041D",	#CYRILLIC CAPITAL LETTER EN
	"84 4F	041E",	#CYRILLIC CAPITAL LETTER O
	"84 50	041F",	#CYRILLIC CAPITAL LETTER PE
	"84 51	0420",	#CYRILLIC CAPITAL LETTER ER
	"84 52	0421",	#CYRILLIC CAPITAL LETTER ES
	"84 53	0422",	#CYRILLIC CAPITAL LETTER TE
	"84 54	0423",	#CYRILLIC CAPITAL LETTER U
	"84 55	0424",	#CYRILLIC CAPITAL LETTER EF
	"84 56	0425",	#CYRILLIC CAPITAL LETTER HA
	"84 57	0426",	#CYRILLIC CAPITAL LETTER TSE
	"84 58	0427",	#CYRILLIC CAPITAL LETTER CHE
	"84 59	0428",	#CYRILLIC CAPITAL LETTER SHA
	"84 5A	0429",	#CYRILLIC CAPITAL LETTER SHCHA
	"84 5B	042A",	#CYRILLIC CAPITAL LETTER HARD SIGN
	"84 5C	042B",	#CYRILLIC CAPITAL LETTER YERU
	"84 5D	042C",	#CYRILLIC CAPITAL LETTER SOFT SIGN
	"84 5E	042D",	#CYRILLIC CAPITAL LETTER E
	"84 5F	042E",	#CYRILLIC CAPITAL LETTER YU
	"84 60	042F",	#CYRILLIC CAPITAL LETTER YA
	"84 70	0430",	#CYRILLIC SMALL LETTER A
	"84 71	0431",	#CYRILLIC SMALL LETTER BE
	"84 72	0432",	#CYRILLIC SMALL LETTER VE
	"84 73	0433",	#CYRILLIC SMALL LETTER GHE
	"84 74	0434",	#CYRILLIC SMALL LETTER DE
	"84 75	0435",	#CYRILLIC SMALL LETTER IE
	"84 76	0451",	#CYRILLIC SMALL LETTER IO
	"84 77	0436",	#CYRILLIC SMALL LETTER ZHE
	"84 78	0437",	#CYRILLIC SMALL LETTER ZE
	"84 79	0438",	#CYRILLIC SMALL LETTER I
	"84 7A	0439",	#CYRILLIC SMALL LETTER SHORT I
	"84 7B	043A",	#CYRILLIC SMALL LETTER KA
	"84 7C	043B",	#CYRILLIC SMALL LETTER EL
	"84 7D	043C",	#CYRILLIC SMALL LETTER EM
	"84 7E	043D",	#CYRILLIC SMALL LETTER EN
	"84 80	043E",	#CYRILLIC SMALL LETTER O
	"84 81	043F",	#CYRILLIC SMALL LETTER PE
	"84 82	0440",	#CYRILLIC SMALL LETTER ER
	"84 83	0441",	#CYRILLIC SMALL LETTER ES
	"84 84	0442",	#CYRILLIC SMALL LETTER TE
	"84 85	0443",	#CYRILLIC SMALL LETTER U
	"84 86	0444",	#CYRILLIC SMALL LETTER EF
	"84 87	0445",	#CYRILLIC SMALL LETTER HA
	"84 88	0446",	#CYRILLIC SMALL LETTER TSE
	"84 89	0447",	#CYRILLIC SMALL LETTER CHE
	"84 8A	0448",	#CYRILLIC SMALL LETTER SHA
	"84 8B	0449",	#CYRILLIC SMALL LETTER SHCHA
	"84 8C	044A",	#CYRILLIC SMALL LETTER HARD SIGN
	"84 8D	044B",	#CYRILLIC SMALL LETTER YERU
	"84 8E	044C",	#CYRILLIC SMALL LETTER SOFT SIGN
	"84 8F	044D",	#CYRILLIC SMALL LETTER E
	"84 90	044E",	#CYRILLIC SMALL LETTER YU
	"84 91	044F",	#CYRILLIC SMALL LETTER YA
	"84 9F	2500",	#BOX DRAWINGS LIGHT HORIZONTAL
	"84 A0	2502",	#BOX DRAWINGS LIGHT VERTICAL
	"84 A1	250C",	#BOX DRAWINGS LIGHT DOWN AND RIGHT
	"84 A2	2510",	#BOX DRAWINGS LIGHT DOWN AND LEFT
	"84 A3	2518",	#BOX DRAWINGS LIGHT UP AND LEFT
	"84 A4	2514",	#BOX DRAWINGS LIGHT UP AND RIGHT
	"84 A5	251C",	#BOX DRAWINGS LIGHT VERTICAL AND RIGHT
	"84 A6	252C",	#BOX DRAWINGS LIGHT DOWN AND HORIZONTAL
	"84 A7	2524",	#BOX DRAWINGS LIGHT VERTICAL AND LEFT
	"84 A8	2534",	#BOX DRAWINGS LIGHT UP AND HORIZONTAL
	"84 A9	253C",	#BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL
	"84 AA	2501",	#BOX DRAWINGS HEAVY HORIZONTAL
	"84 AB	2503",	#BOX DRAWINGS HEAVY VERTICAL
	"84 AC	250F",	#BOX DRAWINGS HEAVY DOWN AND RIGHT
	"84 AD	2513",	#BOX DRAWINGS HEAVY DOWN AND LEFT
	"84 AE	251B",	#BOX DRAWINGS HEAVY UP AND LEFT
	"84 AF	2517",	#BOX DRAWINGS HEAVY UP AND RIGHT
	"84 B0	2523",	#BOX DRAWINGS HEAVY VERTICAL AND RIGHT
	"84 B1	2533",	#BOX DRAWINGS HEAVY DOWN AND HORIZONTAL
	"84 B2	252B",	#BOX DRAWINGS HEAVY VERTICAL AND LEFT
	"84 B3	253B",	#BOX DRAWINGS HEAVY UP AND HORIZONTAL
	"84 B4	254B",	#BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL
	"84 B5	2520",	#BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT
	"84 B6	252F",	#BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY
	"84 B7	2528",	#BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT
	"84 B8	2537",	#BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY
	"84 B9	253F",	#BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY
	"84 BA	251D",	#BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY
	"84 BB	2530",	#BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT
	"84 BC	2525",	#BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY
	"84 BD	2538",	#BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT
	"84 BE	2542",	#BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT
	"87 40	2460",	#CIRCLED DIGIT ONE
	"87 41	2461",	#CIRCLED DIGIT TWO
	"87 42	2462",	#CIRCLED DIGIT THREE
	"87 43	2463",	#CIRCLED DIGIT FOUR
	"87 44	2464",	#CIRCLED DIGIT FIVE
	"87 45	2465",	#CIRCLED DIGIT SIX
	"87 46	2466",	#CIRCLED DIGIT SEVEN
	"87 47	2467",	#CIRCLED DIGIT EIGHT
	"87 48	2468",	#CIRCLED DIGIT NINE
	"87 49	2469",	#CIRCLED NUMBER TEN
	"87 4A	246A",	#CIRCLED NUMBER ELEVEN
	"87 4B	246B",	#CIRCLED NUMBER TWELVE
	"87 4C	246C",	#CIRCLED NUMBER THIRTEEN
	"87 4D	246D",	#CIRCLED NUMBER FOURTEEN
	"87 4E	246E",	#CIRCLED NUMBER FIFTEEN
	"87 4F	246F",	#CIRCLED NUMBER SIXTEEN
	"87 50	2470",	#CIRCLED NUMBER SEVENTEEN
	"87 51	2471",	#CIRCLED NUMBER EIGHTEEN
	"87 52	2472",	#CIRCLED NUMBER NINETEEN
	"87 53	2473",	#CIRCLED NUMBER TWENTY
	"87 54	2160",	#ROMAN NUMERAL ONE
	"87 55	2161",	#ROMAN NUMERAL TWO
	"87 56	2162",	#ROMAN NUMERAL THREE
	"87 57	2163",	#ROMAN NUMERAL FOUR
	"87 58	2164",	#ROMAN NUMERAL FIVE
	"87 59	2165",	#ROMAN NUMERAL SIX
	"87 5A	2166",	#ROMAN NUMERAL SEVEN
	"87 5B	2167",	#ROMAN NUMERAL EIGHT
	"87 5C	2168",	#ROMAN NUMERAL NINE
	"87 5D	2169",	#ROMAN NUMERAL TEN
	"87 5F	3349",	#SQUARE MIRI
	"87 60	3314",	#SQUARE KIRO
	"87 61	3322",	#SQUARE SENTI
	"87 62	334D",	#SQUARE MEETORU
	"87 63	3318",	#SQUARE GURAMU
	"87 64	3327",	#SQUARE TON
	"87 65	3303",	#SQUARE AARU
	"87 66	3336",	#SQUARE HEKUTAARU
	"87 67	3351",	#SQUARE RITTORU
	"87 68	3357",	#SQUARE WATTO
	"87 69	330D",	#SQUARE KARORII
	"87 6A	3326",	#SQUARE DORU
	"87 6B	3323",	#SQUARE SENTO
	"87 6C	332B",	#SQUARE PAASENTO
	"87 6D	334A",	#SQUARE MIRIBAARU
	"87 6E	333B",	#SQUARE PEEZI
	"87 6F	339C",	#SQUARE MM
	"87 70	339D",	#SQUARE CM
	"87 71	339E",	#SQUARE KM
	"87 72	338E",	#SQUARE MG
	"87 73	338F",	#SQUARE KG
	"87 74	33C4",	#SQUARE CC
	"87 75	33A1",	#SQUARE M SQUARED
	"87 7E	337B",	#SQUARE ERA NAME HEISEI
	"87 80	301D",	#REVERSED DOUBLE PRIME QUOTATION MARK
	"87 81	301F",	#LOW DOUBLE PRIME QUOTATION MARK
	"87 82	2116",	#NUMERO SIGN
	"87 83	33CD",	#SQUARE KK
	"87 84	2121",	#TELEPHONE SIGN
	"87 85	32A4",	#CIRCLED IDEOGRAPH HIGH
	"87 86	32A5",	#CIRCLED IDEOGRAPH CENTRE
	"87 87	32A6",	#CIRCLED IDEOGRAPH LOW
	"87 88	32A7",	#CIRCLED IDEOGRAPH LEFT
	"87 89	32A8",	#CIRCLED IDEOGRAPH RIGHT
	"87 8A	3231",	#PARENTHESIZED IDEOGRAPH STOCK
	"87 8B	3232",	#PARENTHESIZED IDEOGRAPH HAVE
	"87 8C	3239",	#PARENTHESIZED IDEOGRAPH REPRESENT
	"87 8D	337E",	#SQUARE ERA NAME MEIZI
	"87 8E	337D",	#SQUARE ERA NAME TAISYOU
	"87 8F	337C",	#SQUARE ERA NAME SYOUWA
	"87 90	2252",	#APPROXIMATELY EQUAL TO OR THE IMAGE OF
	"87 91	2261",	#IDENTICAL TO
	"87 92	222B",	#INTEGRAL
	"87 93	222E",	#CONTOUR INTEGRAL
	"87 94	2211",	#N-ARY SUMMATION
	"87 95	221A",	#SQUARE ROOT
	"87 96	22A5",	#UP TACK
	"87 97	2220",	#ANGLE
	"87 98	221F",	#RIGHT ANGLE
	"87 99	22BF",	#RIGHT TRIANGLE
	"87 9A	2235",	#BECAUSE
	"87 9B	2229",	#INTERSECTION
	"87 9C	222A",	#UNION
	"88 9F	4E9C",	#CJK UNIFIED IDEOGRAPH
	"88 A0	5516",	#CJK UNIFIED IDEOGRAPH
	"88 A1	5A03",	#CJK UNIFIED IDEOGRAPH
	"88 A2	963F",	#CJK UNIFIED IDEOGRAPH
	"88 A3	54C0",	#CJK UNIFIED IDEOGRAPH
	"88 A4	611B",	#CJK UNIFIED IDEOGRAPH
	"88 A5	6328",	#CJK UNIFIED IDEOGRAPH
	"88 A6	59F6",	#CJK UNIFIED IDEOGRAPH
	"88 A7	9022",	#CJK UNIFIED IDEOGRAPH
	"88 A8	8475",	#CJK UNIFIED IDEOGRAPH
	"88 A9	831C",	#CJK UNIFIED IDEOGRAPH
	"88 AA	7A50",	#CJK UNIFIED IDEOGRAPH
	"88 AB	60AA",	#CJK UNIFIED IDEOGRAPH
	"88 AC	63E1",	#CJK UNIFIED IDEOGRAPH
	"88 AD	6E25",	#CJK UNIFIED IDEOGRAPH
	"88 AE	65ED",	#CJK UNIFIED IDEOGRAPH
	"88 AF	8466",	#CJK UNIFIED IDEOGRAPH
	"88 B0	82A6",	#CJK UNIFIED IDEOGRAPH
	"88 B1	9BF5",	#CJK UNIFIED IDEOGRAPH
	"88 B2	6893",	#CJK UNIFIED IDEOGRAPH
	"88 B3	5727",	#CJK UNIFIED IDEOGRAPH
	"88 B4	65A1",	#CJK UNIFIED IDEOGRAPH
	"88 B5	6271",	#CJK UNIFIED IDEOGRAPH
	"88 B6	5B9B",	#CJK UNIFIED IDEOGRAPH
	"88 B7	59D0",	#CJK UNIFIED IDEOGRAPH
	"88 B8	867B",	#CJK UNIFIED IDEOGRAPH
	"88 B9	98F4",	#CJK UNIFIED IDEOGRAPH
	"88 BA	7D62",	#CJK UNIFIED IDEOGRAPH
	"88 BB	7DBE",	#CJK UNIFIED IDEOGRAPH
	"88 BC	9B8E",	#CJK UNIFIED IDEOGRAPH
	"88 BD	6216",	#CJK UNIFIED IDEOGRAPH
	"88 BE	7C9F",	#CJK UNIFIED IDEOGRAPH
	"88 BF	88B7",	#CJK UNIFIED IDEOGRAPH
	"88 C0	5B89",	#CJK UNIFIED IDEOGRAPH
	"88 C1	5EB5",	#CJK UNIFIED IDEOGRAPH
	"88 C2	6309",	#CJK UNIFIED IDEOGRAPH
	"88 C3	6697",	#CJK UNIFIED IDEOGRAPH
	"88 C4	6848",	#CJK UNIFIED IDEOGRAPH
	"88 C5	95C7",	#CJK UNIFIED IDEOGRAPH
	"88 C6	978D",	#CJK UNIFIED IDEOGRAPH
	"88 C7	674F",	#CJK UNIFIED IDEOGRAPH
	"88 C8	4EE5",	#CJK UNIFIED IDEOGRAPH
	"88 C9	4F0A",	#CJK UNIFIED IDEOGRAPH
	"88 CA	4F4D",	#CJK UNIFIED IDEOGRAPH
	"88 CB	4F9D",	#CJK UNIFIED IDEOGRAPH
	"88 CC	5049",	#CJK UNIFIED IDEOGRAPH
	"88 CD	56F2",	#CJK UNIFIED IDEOGRAPH
	"88 CE	5937",	#CJK UNIFIED IDEOGRAPH
	"88 CF	59D4",	#CJK UNIFIED IDEOGRAPH
	"88 D0	5A01",	#CJK UNIFIED IDEOGRAPH
	"88 D1	5C09",	#CJK UNIFIED IDEOGRAPH
	"88 D2	60DF",	#CJK UNIFIED IDEOGRAPH
	"88 D3	610F",	#CJK UNIFIED IDEOGRAPH
	"88 D4	6170",	#CJK UNIFIED IDEOGRAPH
	"88 D5	6613",	#CJK UNIFIED IDEOGRAPH
	"88 D6	6905",	#CJK UNIFIED IDEOGRAPH
	"88 D7	70BA",	#CJK UNIFIED IDEOGRAPH
	"88 D8	754F",	#CJK UNIFIED IDEOGRAPH
	"88 D9	7570",	#CJK UNIFIED IDEOGRAPH
	"88 DA	79FB",	#CJK UNIFIED IDEOGRAPH
	"88 DB	7DAD",	#CJK UNIFIED IDEOGRAPH
	"88 DC	7DEF",	#CJK UNIFIED IDEOGRAPH
	"88 DD	80C3",	#CJK UNIFIED IDEOGRAPH
	"88 DE	840E",	#CJK UNIFIED IDEOGRAPH
	"88 DF	8863",	#CJK UNIFIED IDEOGRAPH
	"88 E0	8B02",	#CJK UNIFIED IDEOGRAPH
	"88 E1	9055",	#CJK UNIFIED IDEOGRAPH
	"88 E2	907A",	#CJK UNIFIED IDEOGRAPH
	"88 E3	533B",	#CJK UNIFIED IDEOGRAPH
	"88 E4	4E95",	#CJK UNIFIED IDEOGRAPH
	"88 E5	4EA5",	#CJK UNIFIED IDEOGRAPH
	"88 E6	57DF",	#CJK UNIFIED IDEOGRAPH
	"88 E7	80B2",	#CJK UNIFIED IDEOGRAPH
	"88 E8	90C1",	#CJK UNIFIED IDEOGRAPH
	"88 E9	78EF",	#CJK UNIFIED IDEOGRAPH
	"88 EA	4E00",	#CJK UNIFIED IDEOGRAPH
	"88 EB	58F1",	#CJK UNIFIED IDEOGRAPH
	"88 EC	6EA2",	#CJK UNIFIED IDEOGRAPH
	"88 ED	9038",	#CJK UNIFIED IDEOGRAPH
	"88 EE	7A32",	#CJK UNIFIED IDEOGRAPH
	"88 EF	8328",	#CJK UNIFIED IDEOGRAPH
	"88 F0	828B",	#CJK UNIFIED IDEOGRAPH
	"88 F1	9C2F",	#CJK UNIFIED IDEOGRAPH
	"88 F2	5141",	#CJK UNIFIED IDEOGRAPH
	"88 F3	5370",	#CJK UNIFIED IDEOGRAPH
	"88 F4	54BD",	#CJK UNIFIED IDEOGRAPH
	"88 F5	54E1",	#CJK UNIFIED IDEOGRAPH
	"88 F6	56E0",	#CJK UNIFIED IDEOGRAPH
	"88 F7	59FB",	#CJK UNIFIED IDEOGRAPH
	"88 F8	5F15",	#CJK UNIFIED IDEOGRAPH
	"88 F9	98F2",	#CJK UNIFIED IDEOGRAPH
	"88 FA	6DEB",	#CJK UNIFIED IDEOGRAPH
	"88 FB	80E4",	#CJK UNIFIED IDEOGRAPH
	"88 FC	852D",	#CJK UNIFIED IDEOGRAPH
	"89 40	9662",	#CJK UNIFIED IDEOGRAPH
	"89 41	9670",	#CJK UNIFIED IDEOGRAPH
	"89 42	96A0",	#CJK UNIFIED IDEOGRAPH
	"89 43	97FB",	#CJK UNIFIED IDEOGRAPH
	"89 44	540B",	#CJK UNIFIED IDEOGRAPH
	"89 45	53F3",	#CJK UNIFIED IDEOGRAPH
	"89 46	5B87",	#CJK UNIFIED IDEOGRAPH
	"89 47	70CF",	#CJK UNIFIED IDEOGRAPH
	"89 48	7FBD",	#CJK UNIFIED IDEOGRAPH
	"89 49	8FC2",	#CJK UNIFIED IDEOGRAPH
	"89 4A	96E8",	#CJK UNIFIED IDEOGRAPH
	"89 4B	536F",	#CJK UNIFIED IDEOGRAPH
	"89 4C	9D5C",	#CJK UNIFIED IDEOGRAPH
	"89 4D	7ABA",	#CJK UNIFIED IDEOGRAPH
	"89 4E	4E11",	#CJK UNIFIED IDEOGRAPH
	"89 4F	7893",	#CJK UNIFIED IDEOGRAPH
	"89 50	81FC",	#CJK UNIFIED IDEOGRAPH
	"89 51	6E26",	#CJK UNIFIED IDEOGRAPH
	"89 52	5618",	#CJK UNIFIED IDEOGRAPH
	"89 53	5504",	#CJK UNIFIED IDEOGRAPH
	"89 54	6B1D",	#CJK UNIFIED IDEOGRAPH
	"89 55	851A",	#CJK UNIFIED IDEOGRAPH
	"89 56	9C3B",	#CJK UNIFIED IDEOGRAPH
	"89 57	59E5",	#CJK UNIFIED IDEOGRAPH
	"89 58	53A9",	#CJK UNIFIED IDEOGRAPH
	"89 59	6D66",	#CJK UNIFIED IDEOGRAPH
	"89 5A	74DC",	#CJK UNIFIED IDEOGRAPH
	"89 5B	958F",	#CJK UNIFIED IDEOGRAPH
	"89 5C	5642",	#CJK UNIFIED IDEOGRAPH
	"89 5D	4E91",	#CJK UNIFIED IDEOGRAPH
	"89 5E	904B",	#CJK UNIFIED IDEOGRAPH
	"89 5F	96F2",	#CJK UNIFIED IDEOGRAPH
	"89 60	834F",	#CJK UNIFIED IDEOGRAPH
	"89 61	990C",	#CJK UNIFIED IDEOGRAPH
	"89 62	53E1",	#CJK UNIFIED IDEOGRAPH
	"89 63	55B6",	#CJK UNIFIED IDEOGRAPH
	"89 64	5B30",	#CJK UNIFIED IDEOGRAPH
	"89 65	5F71",	#CJK UNIFIED IDEOGRAPH
	"89 66	6620",	#CJK UNIFIED IDEOGRAPH
	"89 67	66F3",	#CJK UNIFIED IDEOGRAPH
	"89 68	6804",	#CJK UNIFIED IDEOGRAPH
	"89 69	6C38",	#CJK UNIFIED IDEOGRAPH
	"89 6A	6CF3",	#CJK UNIFIED IDEOGRAPH
	"89 6B	6D29",	#CJK UNIFIED IDEOGRAPH
	"89 6C	745B",	#CJK UNIFIED IDEOGRAPH
	"89 6D	76C8",	#CJK UNIFIED IDEOGRAPH
	"89 6E	7A4E",	#CJK UNIFIED IDEOGRAPH
	"89 6F	9834",	#CJK UNIFIED IDEOGRAPH
	"89 70	82F1",	#CJK UNIFIED IDEOGRAPH
	"89 71	885B",	#CJK UNIFIED IDEOGRAPH
	"89 72	8A60",	#CJK UNIFIED IDEOGRAPH
	"89 73	92ED",	#CJK UNIFIED IDEOGRAPH
	"89 74	6DB2",	#CJK UNIFIED IDEOGRAPH
	"89 75	75AB",	#CJK UNIFIED IDEOGRAPH
	"89 76	76CA",	#CJK UNIFIED IDEOGRAPH
	"89 77	99C5",	#CJK UNIFIED IDEOGRAPH
	"89 78	60A6",	#CJK UNIFIED IDEOGRAPH
	"89 79	8B01",	#CJK UNIFIED IDEOGRAPH
	"89 7A	8D8A",	#CJK UNIFIED IDEOGRAPH
	"89 7B	95B2",	#CJK UNIFIED IDEOGRAPH
	"89 7C	698E",	#CJK UNIFIED IDEOGRAPH
	"89 7D	53AD",	#CJK UNIFIED IDEOGRAPH
	"89 7E	5186",	#CJK UNIFIED IDEOGRAPH
	"89 80	5712",	#CJK UNIFIED IDEOGRAPH
	"89 81	5830",	#CJK UNIFIED IDEOGRAPH
	"89 82	5944",	#CJK UNIFIED IDEOGRAPH
	"89 83	5BB4",	#CJK UNIFIED IDEOGRAPH
	"89 84	5EF6",	#CJK UNIFIED IDEOGRAPH
	"89 85	6028",	#CJK UNIFIED IDEOGRAPH
	"89 86	63A9",	#CJK UNIFIED IDEOGRAPH
	"89 87	63F4",	#CJK UNIFIED IDEOGRAPH
	"89 88	6CBF",	#CJK UNIFIED IDEOGRAPH
	"89 89	6F14",	#CJK UNIFIED IDEOGRAPH
	"89 8A	708E",	#CJK UNIFIED IDEOGRAPH
	"89 8B	7114",	#CJK UNIFIED IDEOGRAPH
	"89 8C	7159",	#CJK UNIFIED IDEOGRAPH
	"89 8D	71D5",	#CJK UNIFIED IDEOGRAPH
	"89 8E	733F",	#CJK UNIFIED IDEOGRAPH
	"89 8F	7E01",	#CJK UNIFIED IDEOGRAPH
	"89 90	8276",	#CJK UNIFIED IDEOGRAPH
	"89 91	82D1",	#CJK UNIFIED IDEOGRAPH
	"89 92	8597",	#CJK UNIFIED IDEOGRAPH
	"89 93	9060",	#CJK UNIFIED IDEOGRAPH
	"89 94	925B",	#CJK UNIFIED IDEOGRAPH
	"89 95	9D1B",	#CJK UNIFIED IDEOGRAPH
	"89 96	5869",	#CJK UNIFIED IDEOGRAPH
	"89 97	65BC",	#CJK UNIFIED IDEOGRAPH
	"89 98	6C5A",	#CJK UNIFIED IDEOGRAPH
	"89 99	7525",	#CJK UNIFIED IDEOGRAPH
	"89 9A	51F9",	#CJK UNIFIED IDEOGRAPH
	"89 9B	592E",	#CJK UNIFIED IDEOGRAPH
	"89 9C	5965",	#CJK UNIFIED IDEOGRAPH
	"89 9D	5F80",	#CJK UNIFIED IDEOGRAPH
	"89 9E	5FDC",	#CJK UNIFIED IDEOGRAPH
	"89 9F	62BC",	#CJK UNIFIED IDEOGRAPH
	"89 A0	65FA",	#CJK UNIFIED IDEOGRAPH
	"89 A1	6A2A",	#CJK UNIFIED IDEOGRAPH
	"89 A2	6B27",	#CJK UNIFIED IDEOGRAPH
	"89 A3	6BB4",	#CJK UNIFIED IDEOGRAPH
	"89 A4	738B",	#CJK UNIFIED IDEOGRAPH
	"89 A5	7FC1",	#CJK UNIFIED IDEOGRAPH
	"89 A6	8956",	#CJK UNIFIED IDEOGRAPH
	"89 A7	9D2C",	#CJK UNIFIED IDEOGRAPH
	"89 A8	9D0E",	#CJK UNIFIED IDEOGRAPH
	"89 A9	9EC4",	#CJK UNIFIED IDEOGRAPH
	"89 AA	5CA1",	#CJK UNIFIED IDEOGRAPH
	"89 AB	6C96",	#CJK UNIFIED IDEOGRAPH
	"89 AC	837B",	#CJK UNIFIED IDEOGRAPH
	"89 AD	5104",	#CJK UNIFIED IDEOGRAPH
	"89 AE	5C4B",	#CJK UNIFIED IDEOGRAPH
	"89 AF	61B6",	#CJK UNIFIED IDEOGRAPH
	"89 B0	81C6",	#CJK UNIFIED IDEOGRAPH
	"89 B1	6876",	#CJK UNIFIED IDEOGRAPH
	"89 B2	7261",	#CJK UNIFIED IDEOGRAPH
	"89 B3	4E59",	#CJK UNIFIED IDEOGRAPH
	"89 B4	4FFA",	#CJK UNIFIED IDEOGRAPH
	"89 B5	5378",	#CJK UNIFIED IDEOGRAPH
	"89 B6	6069",	#CJK UNIFIED IDEOGRAPH
	"89 B7	6E29",	#CJK UNIFIED IDEOGRAPH
	"89 B8	7A4F",	#CJK UNIFIED IDEOGRAPH
	"89 B9	97F3",	#CJK UNIFIED IDEOGRAPH
	"89 BA	4E0B",	#CJK UNIFIED IDEOGRAPH
	"89 BB	5316",	#CJK UNIFIED IDEOGRAPH
	"89 BC	4EEE",	#CJK UNIFIED IDEOGRAPH
	"89 BD	4F55",	#CJK UNIFIED IDEOGRAPH
	"89 BE	4F3D",	#CJK UNIFIED IDEOGRAPH
	"89 BF	4FA1",	#CJK UNIFIED IDEOGRAPH
	"89 C0	4F73",	#CJK UNIFIED IDEOGRAPH
	"89 C1	52A0",	#CJK UNIFIED IDEOGRAPH
	"89 C2	53EF",	#CJK UNIFIED IDEOGRAPH
	"89 C3	5609",	#CJK UNIFIED IDEOGRAPH
	"89 C4	590F",	#CJK UNIFIED IDEOGRAPH
	"89 C5	5AC1",	#CJK UNIFIED IDEOGRAPH
	"89 C6	5BB6",	#CJK UNIFIED IDEOGRAPH
	"89 C7	5BE1",	#CJK UNIFIED IDEOGRAPH
	"89 C8	79D1",	#CJK UNIFIED IDEOGRAPH
	"89 C9	6687",	#CJK UNIFIED IDEOGRAPH
	"89 CA	679C",	#CJK UNIFIED IDEOGRAPH
	"89 CB	67B6",	#CJK UNIFIED IDEOGRAPH
	"89 CC	6B4C",	#CJK UNIFIED IDEOGRAPH
	"89 CD	6CB3",	#CJK UNIFIED IDEOGRAPH
	"89 CE	706B",	#CJK UNIFIED IDEOGRAPH
	"89 CF	73C2",	#CJK UNIFIED IDEOGRAPH
	"89 D0	798D",	#CJK UNIFIED IDEOGRAPH
	"89 D1	79BE",	#CJK UNIFIED IDEOGRAPH
	"89 D2	7A3C",	#CJK UNIFIED IDEOGRAPH
	"89 D3	7B87",	#CJK UNIFIED IDEOGRAPH
	"89 D4	82B1",	#CJK UNIFIED IDEOGRAPH
	"89 D5	82DB",	#CJK UNIFIED IDEOGRAPH
	"89 D6	8304",	#CJK UNIFIED IDEOGRAPH
	"89 D7	8377",	#CJK UNIFIED IDEOGRAPH
	"89 D8	83EF",	#CJK UNIFIED IDEOGRAPH
	"89 D9	83D3",	#CJK UNIFIED IDEOGRAPH
	"89 DA	8766",	#CJK UNIFIED IDEOGRAPH
	"89 DB	8AB2",	#CJK UNIFIED IDEOGRAPH
	"89 DC	5629",	#CJK UNIFIED IDEOGRAPH
	"89 DD	8CA8",	#CJK UNIFIED IDEOGRAPH
	"89 DE	8FE6",	#CJK UNIFIED IDEOGRAPH
	"89 DF	904E",	#CJK UNIFIED IDEOGRAPH
	"89 E0	971E",	#CJK UNIFIED IDEOGRAPH
	"89 E1	868A",	#CJK UNIFIED IDEOGRAPH
	"89 E2	4FC4",	#CJK UNIFIED IDEOGRAPH
	"89 E3	5CE8",	#CJK UNIFIED IDEOGRAPH
	"89 E4	6211",	#CJK UNIFIED IDEOGRAPH
	"89 E5	7259",	#CJK UNIFIED IDEOGRAPH
	"89 E6	753B",	#CJK UNIFIED IDEOGRAPH
	"89 E7	81E5",	#CJK UNIFIED IDEOGRAPH
	"89 E8	82BD",	#CJK UNIFIED IDEOGRAPH
	"89 E9	86FE",	#CJK UNIFIED IDEOGRAPH
	"89 EA	8CC0",	#CJK UNIFIED IDEOGRAPH
	"89 EB	96C5",	#CJK UNIFIED IDEOGRAPH
	"89 EC	9913",	#CJK UNIFIED IDEOGRAPH
	"89 ED	99D5",	#CJK UNIFIED IDEOGRAPH
	"89 EE	4ECB",	#CJK UNIFIED IDEOGRAPH
	"89 EF	4F1A",	#CJK UNIFIED IDEOGRAPH
	"89 F0	89E3",	#CJK UNIFIED IDEOGRAPH
	"89 F1	56DE",	#CJK UNIFIED IDEOGRAPH
	"89 F2	584A",	#CJK UNIFIED IDEOGRAPH
	"89 F3	58CA",	#CJK UNIFIED IDEOGRAPH
	"89 F4	5EFB",	#CJK UNIFIED IDEOGRAPH
	"89 F5	5FEB",	#CJK UNIFIED IDEOGRAPH
	"89 F6	602A",	#CJK UNIFIED IDEOGRAPH
	"89 F7	6094",	#CJK UNIFIED IDEOGRAPH
	"89 F8	6062",	#CJK UNIFIED IDEOGRAPH
	"89 F9	61D0",	#CJK UNIFIED IDEOGRAPH
	"89 FA	6212",	#CJK UNIFIED IDEOGRAPH
	"89 FB	62D0",	#CJK UNIFIED IDEOGRAPH
	"89 FC	6539",	#CJK UNIFIED IDEOGRAPH
	"8A 40	9B41",	#CJK UNIFIED IDEOGRAPH
	"8A 41	6666",	#CJK UNIFIED IDEOGRAPH
	"8A 42	68B0",	#CJK UNIFIED IDEOGRAPH
	"8A 43	6D77",	#CJK UNIFIED IDEOGRAPH
	"8A 44	7070",	#CJK UNIFIED IDEOGRAPH
	"8A 45	754C",	#CJK UNIFIED IDEOGRAPH
	"8A 46	7686",	#CJK UNIFIED IDEOGRAPH
	"8A 47	7D75",	#CJK UNIFIED IDEOGRAPH
	"8A 48	82A5",	#CJK UNIFIED IDEOGRAPH
	"8A 49	87F9",	#CJK UNIFIED IDEOGRAPH
	"8A 4A	958B",	#CJK UNIFIED IDEOGRAPH
	"8A 4B	968E",	#CJK UNIFIED IDEOGRAPH
	"8A 4C	8C9D",	#CJK UNIFIED IDEOGRAPH
	"8A 4D	51F1",	#CJK UNIFIED IDEOGRAPH
	"8A 4E	52BE",	#CJK UNIFIED IDEOGRAPH
	"8A 4F	5916",	#CJK UNIFIED IDEOGRAPH
	"8A 50	54B3",	#CJK UNIFIED IDEOGRAPH
	"8A 51	5BB3",	#CJK UNIFIED IDEOGRAPH
	"8A 52	5D16",	#CJK UNIFIED IDEOGRAPH
	"8A 53	6168",	#CJK UNIFIED IDEOGRAPH
	"8A 54	6982",	#CJK UNIFIED IDEOGRAPH
	"8A 55	6DAF",	#CJK UNIFIED IDEOGRAPH
	"8A 56	788D",	#CJK UNIFIED IDEOGRAPH
	"8A 57	84CB",	#CJK UNIFIED IDEOGRAPH
	"8A 58	8857",	#CJK UNIFIED IDEOGRAPH
	"8A 59	8A72",	#CJK UNIFIED IDEOGRAPH
	"8A 5A	93A7",	#CJK UNIFIED IDEOGRAPH
	"8A 5B	9AB8",	#CJK UNIFIED IDEOGRAPH
	"8A 5C	6D6C",	#CJK UNIFIED IDEOGRAPH
	"8A 5D	99A8",	#CJK UNIFIED IDEOGRAPH
	"8A 5E	86D9",	#CJK UNIFIED IDEOGRAPH
	"8A 5F	57A3",	#CJK UNIFIED IDEOGRAPH
	"8A 60	67FF",	#CJK UNIFIED IDEOGRAPH
	"8A 61	86CE",	#CJK UNIFIED IDEOGRAPH
	"8A 62	920E",	#CJK UNIFIED IDEOGRAPH
	"8A 63	5283",	#CJK UNIFIED IDEOGRAPH
	"8A 64	5687",	#CJK UNIFIED IDEOGRAPH
	"8A 65	5404",	#CJK UNIFIED IDEOGRAPH
	"8A 66	5ED3",	#CJK UNIFIED IDEOGRAPH
	"8A 67	62E1",	#CJK UNIFIED IDEOGRAPH
	"8A 68	64B9",	#CJK UNIFIED IDEOGRAPH
	"8A 69	683C",	#CJK UNIFIED IDEOGRAPH
	"8A 6A	6838",	#CJK UNIFIED IDEOGRAPH
	"8A 6B	6BBB",	#CJK UNIFIED IDEOGRAPH
	"8A 6C	7372",	#CJK UNIFIED IDEOGRAPH
	"8A 6D	78BA",	#CJK UNIFIED IDEOGRAPH
	"8A 6E	7A6B",	#CJK UNIFIED IDEOGRAPH
	"8A 6F	899A",	#CJK UNIFIED IDEOGRAPH
	"8A 70	89D2",	#CJK UNIFIED IDEOGRAPH
	"8A 71	8D6B",	#CJK UNIFIED IDEOGRAPH
	"8A 72	8F03",	#CJK UNIFIED IDEOGRAPH
	"8A 73	90ED",	#CJK UNIFIED IDEOGRAPH
	"8A 74	95A3",	#CJK UNIFIED IDEOGRAPH
	"8A 75	9694",	#CJK UNIFIED IDEOGRAPH
	"8A 76	9769",	#CJK UNIFIED IDEOGRAPH
	"8A 77	5B66",	#CJK UNIFIED IDEOGRAPH
	"8A 78	5CB3",	#CJK UNIFIED IDEOGRAPH
	"8A 79	697D",	#CJK UNIFIED IDEOGRAPH
	"8A 7A	984D",	#CJK UNIFIED IDEOGRAPH
	"8A 7B	984E",	#CJK UNIFIED IDEOGRAPH
	"8A 7C	639B",	#CJK UNIFIED IDEOGRAPH
	"8A 7D	7B20",	#CJK UNIFIED IDEOGRAPH
	"8A 7E	6A2B",	#CJK UNIFIED IDEOGRAPH
	"8A 80	6A7F",	#CJK UNIFIED IDEOGRAPH
	"8A 81	68B6",	#CJK UNIFIED IDEOGRAPH
	"8A 82	9C0D",	#CJK UNIFIED IDEOGRAPH
	"8A 83	6F5F",	#CJK UNIFIED IDEOGRAPH
	"8A 84	5272",	#CJK UNIFIED IDEOGRAPH
	"8A 85	559D",	#CJK UNIFIED IDEOGRAPH
	"8A 86	6070",	#CJK UNIFIED IDEOGRAPH
	"8A 87	62EC",	#CJK UNIFIED IDEOGRAPH
	"8A 88	6D3B",	#CJK UNIFIED IDEOGRAPH
	"8A 89	6E07",	#CJK UNIFIED IDEOGRAPH
	"8A 8A	6ED1",	#CJK UNIFIED IDEOGRAPH
	"8A 8B	845B",	#CJK UNIFIED IDEOGRAPH
	"8A 8C	8910",	#CJK UNIFIED IDEOGRAPH
	"8A 8D	8F44",	#CJK UNIFIED IDEOGRAPH
	"8A 8E	4E14",	#CJK UNIFIED IDEOGRAPH
	"8A 8F	9C39",	#CJK UNIFIED IDEOGRAPH
	"8A 90	53F6",	#CJK UNIFIED IDEOGRAPH
	"8A 91	691B",	#CJK UNIFIED IDEOGRAPH
	"8A 92	6A3A",	#CJK UNIFIED IDEOGRAPH
	"8A 93	9784",	#CJK UNIFIED IDEOGRAPH
	"8A 94	682A",	#CJK UNIFIED IDEOGRAPH
	"8A 95	515C",	#CJK UNIFIED IDEOGRAPH
	"8A 96	7AC3",	#CJK UNIFIED IDEOGRAPH
	"8A 97	84B2",	#CJK UNIFIED IDEOGRAPH
	"8A 98	91DC",	#CJK UNIFIED IDEOGRAPH
	"8A 99	938C",	#CJK UNIFIED IDEOGRAPH
	"8A 9A	565B",	#CJK UNIFIED IDEOGRAPH
	"8A 9B	9D28",	#CJK UNIFIED IDEOGRAPH
	"8A 9C	6822",	#CJK UNIFIED IDEOGRAPH
	"8A 9D	8305",	#CJK UNIFIED IDEOGRAPH
	"8A 9E	8431",	#CJK UNIFIED IDEOGRAPH
	"8A 9F	7CA5",	#CJK UNIFIED IDEOGRAPH
	"8A A0	5208",	#CJK UNIFIED IDEOGRAPH
	"8A A1	82C5",	#CJK UNIFIED IDEOGRAPH
	"8A A2	74E6",	#CJK UNIFIED IDEOGRAPH
	"8A A3	4E7E",	#CJK UNIFIED IDEOGRAPH
	"8A A4	4F83",	#CJK UNIFIED IDEOGRAPH
	"8A A5	51A0",	#CJK UNIFIED IDEOGRAPH
	"8A A6	5BD2",	#CJK UNIFIED IDEOGRAPH
	"8A A7	520A",	#CJK UNIFIED IDEOGRAPH
	"8A A8	52D8",	#CJK UNIFIED IDEOGRAPH
	"8A A9	52E7",	#CJK UNIFIED IDEOGRAPH
	"8A AA	5DFB",	#CJK UNIFIED IDEOGRAPH
	"8A AB	559A",	#CJK UNIFIED IDEOGRAPH
	"8A AC	582A",	#CJK UNIFIED IDEOGRAPH
	"8A AD	59E6",	#CJK UNIFIED IDEOGRAPH
	"8A AE	5B8C",	#CJK UNIFIED IDEOGRAPH
	"8A AF	5B98",	#CJK UNIFIED IDEOGRAPH
	"8A B0	5BDB",	#CJK UNIFIED IDEOGRAPH
	"8A B1	5E72",	#CJK UNIFIED IDEOGRAPH
	"8A B2	5E79",	#CJK UNIFIED IDEOGRAPH
	"8A B3	60A3",	#CJK UNIFIED IDEOGRAPH
	"8A B4	611F",	#CJK UNIFIED IDEOGRAPH
	"8A B5	6163",	#CJK UNIFIED IDEOGRAPH
	"8A B6	61BE",	#CJK UNIFIED IDEOGRAPH
	"8A B7	63DB",	#CJK UNIFIED IDEOGRAPH
	"8A B8	6562",	#CJK UNIFIED IDEOGRAPH
	"8A B9	67D1",	#CJK UNIFIED IDEOGRAPH
	"8A BA	6853",	#CJK UNIFIED IDEOGRAPH
	"8A BB	68FA",	#CJK UNIFIED IDEOGRAPH
	"8A BC	6B3E",	#CJK UNIFIED IDEOGRAPH
	"8A BD	6B53",	#CJK UNIFIED IDEOGRAPH
	"8A BE	6C57",	#CJK UNIFIED IDEOGRAPH
	"8A BF	6F22",	#CJK UNIFIED IDEOGRAPH
	"8A C0	6F97",	#CJK UNIFIED IDEOGRAPH
	"8A C1	6F45",	#CJK UNIFIED IDEOGRAPH
	"8A C2	74B0",	#CJK UNIFIED IDEOGRAPH
	"8A C3	7518",	#CJK UNIFIED IDEOGRAPH
	"8A C4	76E3",	#CJK UNIFIED IDEOGRAPH
	"8A C5	770B",	#CJK UNIFIED IDEOGRAPH
	"8A C6	7AFF",	#CJK UNIFIED IDEOGRAPH
	"8A C7	7BA1",	#CJK UNIFIED IDEOGRAPH
	"8A C8	7C21",	#CJK UNIFIED IDEOGRAPH
	"8A C9	7DE9",	#CJK UNIFIED IDEOGRAPH
	"8A CA	7F36",	#CJK UNIFIED IDEOGRAPH
	"8A CB	7FF0",	#CJK UNIFIED IDEOGRAPH
	"8A CC	809D",	#CJK UNIFIED IDEOGRAPH
	"8A CD	8266",	#CJK UNIFIED IDEOGRAPH
	"8A CE	839E",	#CJK UNIFIED IDEOGRAPH
	"8A CF	89B3",	#CJK UNIFIED IDEOGRAPH
	"8A D0	8ACC",	#CJK UNIFIED IDEOGRAPH
	"8A D1	8CAB",	#CJK UNIFIED IDEOGRAPH
	"8A D2	9084",	#CJK UNIFIED IDEOGRAPH
	"8A D3	9451",	#CJK UNIFIED IDEOGRAPH
	"8A D4	9593",	#CJK UNIFIED IDEOGRAPH
	"8A D5	9591",	#CJK UNIFIED IDEOGRAPH
	"8A D6	95A2",	#CJK UNIFIED IDEOGRAPH
	"8A D7	9665",	#CJK UNIFIED IDEOGRAPH
	"8A D8	97D3",	#CJK UNIFIED IDEOGRAPH
	"8A D9	9928",	#CJK UNIFIED IDEOGRAPH
	"8A DA	8218",	#CJK UNIFIED IDEOGRAPH
	"8A DB	4E38",	#CJK UNIFIED IDEOGRAPH
	"8A DC	542B",	#CJK UNIFIED IDEOGRAPH
	"8A DD	5CB8",	#CJK UNIFIED IDEOGRAPH
	"8A DE	5DCC",	#CJK UNIFIED IDEOGRAPH
	"8A DF	73A9",	#CJK UNIFIED IDEOGRAPH
	"8A E0	764C",	#CJK UNIFIED IDEOGRAPH
	"8A E1	773C",	#CJK UNIFIED IDEOGRAPH
	"8A E2	5CA9",	#CJK UNIFIED IDEOGRAPH
	"8A E3	7FEB",	#CJK UNIFIED IDEOGRAPH
	"8A E4	8D0B",	#CJK UNIFIED IDEOGRAPH
	"8A E5	96C1",	#CJK UNIFIED IDEOGRAPH
	"8A E6	9811",	#CJK UNIFIED IDEOGRAPH
	"8A E7	9854",	#CJK UNIFIED IDEOGRAPH
	"8A E8	9858",	#CJK UNIFIED IDEOGRAPH
	"8A E9	4F01",	#CJK UNIFIED IDEOGRAPH
	"8A EA	4F0E",	#CJK UNIFIED IDEOGRAPH
	"8A EB	5371",	#CJK UNIFIED IDEOGRAPH
	"8A EC	559C",	#CJK UNIFIED IDEOGRAPH
	"8A ED	5668",	#CJK UNIFIED IDEOGRAPH
	"8A EE	57FA",	#CJK UNIFIED IDEOGRAPH
	"8A EF	5947",	#CJK UNIFIED IDEOGRAPH
	"8A F0	5B09",	#CJK UNIFIED IDEOGRAPH
	"8A F1	5BC4",	#CJK UNIFIED IDEOGRAPH
	"8A F2	5C90",	#CJK UNIFIED IDEOGRAPH
	"8A F3	5E0C",	#CJK UNIFIED IDEOGRAPH
	"8A F4	5E7E",	#CJK UNIFIED IDEOGRAPH
	"8A F5	5FCC",	#CJK UNIFIED IDEOGRAPH
	"8A F6	63EE",	#CJK UNIFIED IDEOGRAPH
	"8A F7	673A",	#CJK UNIFIED IDEOGRAPH
	"8A F8	65D7",	#CJK UNIFIED IDEOGRAPH
	"8A F9	65E2",	#CJK UNIFIED IDEOGRAPH
	"8A FA	671F",	#CJK UNIFIED IDEOGRAPH
	"8A FB	68CB",	#CJK UNIFIED IDEOGRAPH
	"8A FC	68C4",	#CJK UNIFIED IDEOGRAPH
	"8B 40	6A5F",	#CJK UNIFIED IDEOGRAPH
	"8B 41	5E30",	#CJK UNIFIED IDEOGRAPH
	"8B 42	6BC5",	#CJK UNIFIED IDEOGRAPH
	"8B 43	6C17",	#CJK UNIFIED IDEOGRAPH
	"8B 44	6C7D",	#CJK UNIFIED IDEOGRAPH
	"8B 45	757F",	#CJK UNIFIED IDEOGRAPH
	"8B 46	7948",	#CJK UNIFIED IDEOGRAPH
	"8B 47	5B63",	#CJK UNIFIED IDEOGRAPH
	"8B 48	7A00",	#CJK UNIFIED IDEOGRAPH
	"8B 49	7D00",	#CJK UNIFIED IDEOGRAPH
	"8B 4A	5FBD",	#CJK UNIFIED IDEOGRAPH
	"8B 4B	898F",	#CJK UNIFIED IDEOGRAPH
	"8B 4C	8A18",	#CJK UNIFIED IDEOGRAPH
	"8B 4D	8CB4",	#CJK UNIFIED IDEOGRAPH
	"8B 4E	8D77",	#CJK UNIFIED IDEOGRAPH
	"8B 4F	8ECC",	#CJK UNIFIED IDEOGRAPH
	"8B 50	8F1D",	#CJK UNIFIED IDEOGRAPH
	"8B 51	98E2",	#CJK UNIFIED IDEOGRAPH
	"8B 52	9A0E",	#CJK UNIFIED IDEOGRAPH
	"8B 53	9B3C",	#CJK UNIFIED IDEOGRAPH
	"8B 54	4E80",	#CJK UNIFIED IDEOGRAPH
	"8B 55	507D",	#CJK UNIFIED IDEOGRAPH
	"8B 56	5100",	#CJK UNIFIED IDEOGRAPH
	"8B 57	5993",	#CJK UNIFIED IDEOGRAPH
	"8B 58	5B9C",	#CJK UNIFIED IDEOGRAPH
	"8B 59	622F",	#CJK UNIFIED IDEOGRAPH
	"8B 5A	6280",	#CJK UNIFIED IDEOGRAPH
	"8B 5B	64EC",	#CJK UNIFIED IDEOGRAPH
	"8B 5C	6B3A",	#CJK UNIFIED IDEOGRAPH
	"8B 5D	72A0",	#CJK UNIFIED IDEOGRAPH
	"8B 5E	7591",	#CJK UNIFIED IDEOGRAPH
	"8B 5F	7947",	#CJK UNIFIED IDEOGRAPH
	"8B 60	7FA9",	#CJK UNIFIED IDEOGRAPH
	"8B 61	87FB",	#CJK UNIFIED IDEOGRAPH
	"8B 62	8ABC",	#CJK UNIFIED IDEOGRAPH
	"8B 63	8B70",	#CJK UNIFIED IDEOGRAPH
	"8B 64	63AC",	#CJK UNIFIED IDEOGRAPH
	"8B 65	83CA",	#CJK UNIFIED IDEOGRAPH
	"8B 66	97A0",	#CJK UNIFIED IDEOGRAPH
	"8B 67	5409",	#CJK UNIFIED IDEOGRAPH
	"8B 68	5403",	#CJK UNIFIED IDEOGRAPH
	"8B 69	55AB",	#CJK UNIFIED IDEOGRAPH
	"8B 6A	6854",	#CJK UNIFIED IDEOGRAPH
	"8B 6B	6A58",	#CJK UNIFIED IDEOGRAPH
	"8B 6C	8A70",	#CJK UNIFIED IDEOGRAPH
	"8B 6D	7827",	#CJK UNIFIED IDEOGRAPH
	"8B 6E	6775",	#CJK UNIFIED IDEOGRAPH
	"8B 6F	9ECD",	#CJK UNIFIED IDEOGRAPH
	"8B 70	5374",	#CJK UNIFIED IDEOGRAPH
	"8B 71	5BA2",	#CJK UNIFIED IDEOGRAPH
	"8B 72	811A",	#CJK UNIFIED IDEOGRAPH
	"8B 73	8650",	#CJK UNIFIED IDEOGRAPH
	"8B 74	9006",	#CJK UNIFIED IDEOGRAPH
	"8B 75	4E18",	#CJK UNIFIED IDEOGRAPH
	"8B 76	4E45",	#CJK UNIFIED IDEOGRAPH
	"8B 77	4EC7",	#CJK UNIFIED IDEOGRAPH
	"8B 78	4F11",	#CJK UNIFIED IDEOGRAPH
	"8B 79	53CA",	#CJK UNIFIED IDEOGRAPH
	"8B 7A	5438",	#CJK UNIFIED IDEOGRAPH
	"8B 7B	5BAE",	#CJK UNIFIED IDEOGRAPH
	"8B 7C	5F13",	#CJK UNIFIED IDEOGRAPH
	"8B 7D	6025",	#CJK UNIFIED IDEOGRAPH
	"8B 7E	6551",	#CJK UNIFIED IDEOGRAPH
	"8B 80	673D",	#CJK UNIFIED IDEOGRAPH
	"8B 81	6C42",	#CJK UNIFIED IDEOGRAPH
	"8B 82	6C72",	#CJK UNIFIED IDEOGRAPH
	"8B 83	6CE3",	#CJK UNIFIED IDEOGRAPH
	"8B 84	7078",	#CJK UNIFIED IDEOGRAPH
	"8B 85	7403",	#CJK UNIFIED IDEOGRAPH
	"8B 86	7A76",	#CJK UNIFIED IDEOGRAPH
	"8B 87	7AAE",	#CJK UNIFIED IDEOGRAPH
	"8B 88	7B08",	#CJK UNIFIED IDEOGRAPH
	"8B 89	7D1A",	#CJK UNIFIED IDEOGRAPH
	"8B 8A	7CFE",	#CJK UNIFIED IDEOGRAPH
	"8B 8B	7D66",	#CJK UNIFIED IDEOGRAPH
	"8B 8C	65E7",	#CJK UNIFIED IDEOGRAPH
	"8B 8D	725B",	#CJK UNIFIED IDEOGRAPH
	"8B 8E	53BB",	#CJK UNIFIED IDEOGRAPH
	"8B 8F	5C45",	#CJK UNIFIED IDEOGRAPH
	"8B 90	5DE8",	#CJK UNIFIED IDEOGRAPH
	"8B 91	62D2",	#CJK UNIFIED IDEOGRAPH
	"8B 92	62E0",	#CJK UNIFIED IDEOGRAPH
	"8B 93	6319",	#CJK UNIFIED IDEOGRAPH
	"8B 94	6E20",	#CJK UNIFIED IDEOGRAPH
	"8B 95	865A",	#CJK UNIFIED IDEOGRAPH
	"8B 96	8A31",	#CJK UNIFIED IDEOGRAPH
	"8B 97	8DDD",	#CJK UNIFIED IDEOGRAPH
	"8B 98	92F8",	#CJK UNIFIED IDEOGRAPH
	"8B 99	6F01",	#CJK UNIFIED IDEOGRAPH
	"8B 9A	79A6",	#CJK UNIFIED IDEOGRAPH
	"8B 9B	9B5A",	#CJK UNIFIED IDEOGRAPH
	"8B 9C	4EA8",	#CJK UNIFIED IDEOGRAPH
	"8B 9D	4EAB",	#CJK UNIFIED IDEOGRAPH
	"8B 9E	4EAC",	#CJK UNIFIED IDEOGRAPH
	"8B 9F	4F9B",	#CJK UNIFIED IDEOGRAPH
	"8B A0	4FA0",	#CJK UNIFIED IDEOGRAPH
	"8B A1	50D1",	#CJK UNIFIED IDEOGRAPH
	"8B A2	5147",	#CJK UNIFIED IDEOGRAPH
	"8B A3	7AF6",	#CJK UNIFIED IDEOGRAPH
	"8B A4	5171",	#CJK UNIFIED IDEOGRAPH
	"8B A5	51F6",	#CJK UNIFIED IDEOGRAPH
	"8B A6	5354",	#CJK UNIFIED IDEOGRAPH
	"8B A7	5321",	#CJK UNIFIED IDEOGRAPH
	"8B A8	537F",	#CJK UNIFIED IDEOGRAPH
	"8B A9	53EB",	#CJK UNIFIED IDEOGRAPH
	"8B AA	55AC",	#CJK UNIFIED IDEOGRAPH
	"8B AB	5883",	#CJK UNIFIED IDEOGRAPH
	"8B AC	5CE1",	#CJK UNIFIED IDEOGRAPH
	"8B AD	5F37",	#CJK UNIFIED IDEOGRAPH
	"8B AE	5F4A",	#CJK UNIFIED IDEOGRAPH
	"8B AF	602F",	#CJK UNIFIED IDEOGRAPH
	"8B B0	6050",	#CJK UNIFIED IDEOGRAPH
	"8B B1	606D",	#CJK UNIFIED IDEOGRAPH
	"8B B2	631F",	#CJK UNIFIED IDEOGRAPH
	"8B B3	6559",	#CJK UNIFIED IDEOGRAPH
	"8B B4	6A4B",	#CJK UNIFIED IDEOGRAPH
	"8B B5	6CC1",	#CJK UNIFIED IDEOGRAPH
	"8B B6	72C2",	#CJK UNIFIED IDEOGRAPH
	"8B B7	72ED",	#CJK UNIFIED IDEOGRAPH
	"8B B8	77EF",	#CJK UNIFIED IDEOGRAPH
	"8B B9	80F8",	#CJK UNIFIED IDEOGRAPH
	"8B BA	8105",	#CJK UNIFIED IDEOGRAPH
	"8B BB	8208",	#CJK UNIFIED IDEOGRAPH
	"8B BC	854E",	#CJK UNIFIED IDEOGRAPH
	"8B BD	90F7",	#CJK UNIFIED IDEOGRAPH
	"8B BE	93E1",	#CJK UNIFIED IDEOGRAPH
	"8B BF	97FF",	#CJK UNIFIED IDEOGRAPH
	"8B C0	9957",	#CJK UNIFIED IDEOGRAPH
	"8B C1	9A5A",	#CJK UNIFIED IDEOGRAPH
	"8B C2	4EF0",	#CJK UNIFIED IDEOGRAPH
	"8B C3	51DD",	#CJK UNIFIED IDEOGRAPH
	"8B C4	5C2D",	#CJK UNIFIED IDEOGRAPH
	"8B C5	6681",	#CJK UNIFIED IDEOGRAPH
	"8B C6	696D",	#CJK UNIFIED IDEOGRAPH
	"8B C7	5C40",	#CJK UNIFIED IDEOGRAPH
	"8B C8	66F2",	#CJK UNIFIED IDEOGRAPH
	"8B C9	6975",	#CJK UNIFIED IDEOGRAPH
	"8B CA	7389",	#CJK UNIFIED IDEOGRAPH
	"8B CB	6850",	#CJK UNIFIED IDEOGRAPH
	"8B CC	7C81",	#CJK UNIFIED IDEOGRAPH
	"8B CD	50C5",	#CJK UNIFIED IDEOGRAPH
	"8B CE	52E4",	#CJK UNIFIED IDEOGRAPH
	"8B CF	5747",	#CJK UNIFIED IDEOGRAPH
	"8B D0	5DFE",	#CJK UNIFIED IDEOGRAPH
	"8B D1	9326",	#CJK UNIFIED IDEOGRAPH
	"8B D2	65A4",	#CJK UNIFIED IDEOGRAPH
	"8B D3	6B23",	#CJK UNIFIED IDEOGRAPH
	"8B D4	6B3D",	#CJK UNIFIED IDEOGRAPH
	"8B D5	7434",	#CJK UNIFIED IDEOGRAPH
	"8B D6	7981",	#CJK UNIFIED IDEOGRAPH
	"8B D7	79BD",	#CJK UNIFIED IDEOGRAPH
	"8B D8	7B4B",	#CJK UNIFIED IDEOGRAPH
	"8B D9	7DCA",	#CJK UNIFIED IDEOGRAPH
	"8B DA	82B9",	#CJK UNIFIED IDEOGRAPH
	"8B DB	83CC",	#CJK UNIFIED IDEOGRAPH
	"8B DC	887F",	#CJK UNIFIED IDEOGRAPH
	"8B DD	895F",	#CJK UNIFIED IDEOGRAPH
	"8B DE	8B39",	#CJK UNIFIED IDEOGRAPH
	"8B DF	8FD1",	#CJK UNIFIED IDEOGRAPH
	"8B E0	91D1",	#CJK UNIFIED IDEOGRAPH
	"8B E1	541F",	#CJK UNIFIED IDEOGRAPH
	"8B E2	9280",	#CJK UNIFIED IDEOGRAPH
	"8B E3	4E5D",	#CJK UNIFIED IDEOGRAPH
	"8B E4	5036",	#CJK UNIFIED IDEOGRAPH
	"8B E5	53E5",	#CJK UNIFIED IDEOGRAPH
	"8B E6	533A",	#CJK UNIFIED IDEOGRAPH
	"8B E7	72D7",	#CJK UNIFIED IDEOGRAPH
	"8B E8	7396",	#CJK UNIFIED IDEOGRAPH
	"8B E9	77E9",	#CJK UNIFIED IDEOGRAPH
	"8B EA	82E6",	#CJK UNIFIED IDEOGRAPH
	"8B EB	8EAF",	#CJK UNIFIED IDEOGRAPH
	"8B EC	99C6",	#CJK UNIFIED IDEOGRAPH
	"8B ED	99C8",	#CJK UNIFIED IDEOGRAPH
	"8B EE	99D2",	#CJK UNIFIED IDEOGRAPH
	"8B EF	5177",	#CJK UNIFIED IDEOGRAPH
	"8B F0	611A",	#CJK UNIFIED IDEOGRAPH
	"8B F1	865E",	#CJK UNIFIED IDEOGRAPH
	"8B F2	55B0",	#CJK UNIFIED IDEOGRAPH
	"8B F3	7A7A",	#CJK UNIFIED IDEOGRAPH
	"8B F4	5076",	#CJK UNIFIED IDEOGRAPH
	"8B F5	5BD3",	#CJK UNIFIED IDEOGRAPH
	"8B F6	9047",	#CJK UNIFIED IDEOGRAPH
	"8B F7	9685",	#CJK UNIFIED IDEOGRAPH
	"8B F8	4E32",	#CJK UNIFIED IDEOGRAPH
	"8B F9	6ADB",	#CJK UNIFIED IDEOGRAPH
	"8B FA	91E7",	#CJK UNIFIED IDEOGRAPH
	"8B FB	5C51",	#CJK UNIFIED IDEOGRAPH
	"8B FC	5C48",	#CJK UNIFIED IDEOGRAPH
	"8C 40	6398",	#CJK UNIFIED IDEOGRAPH
	"8C 41	7A9F",	#CJK UNIFIED IDEOGRAPH
	"8C 42	6C93",	#CJK UNIFIED IDEOGRAPH
	"8C 43	9774",	#CJK UNIFIED IDEOGRAPH
	"8C 44	8F61",	#CJK UNIFIED IDEOGRAPH
	"8C 45	7AAA",	#CJK UNIFIED IDEOGRAPH
	"8C 46	718A",	#CJK UNIFIED IDEOGRAPH
	"8C 47	9688",	#CJK UNIFIED IDEOGRAPH
	"8C 48	7C82",	#CJK UNIFIED IDEOGRAPH
	"8C 49	6817",	#CJK UNIFIED IDEOGRAPH
	"8C 4A	7E70",	#CJK UNIFIED IDEOGRAPH
	"8C 4B	6851",	#CJK UNIFIED IDEOGRAPH
	"8C 4C	936C",	#CJK UNIFIED IDEOGRAPH
	"8C 4D	52F2",	#CJK UNIFIED IDEOGRAPH
	"8C 4E	541B",	#CJK UNIFIED IDEOGRAPH
	"8C 4F	85AB",	#CJK UNIFIED IDEOGRAPH
	"8C 50	8A13",	#CJK UNIFIED IDEOGRAPH
	"8C 51	7FA4",	#CJK UNIFIED IDEOGRAPH
	"8C 52	8ECD",	#CJK UNIFIED IDEOGRAPH
	"8C 53	90E1",	#CJK UNIFIED IDEOGRAPH
	"8C 54	5366",	#CJK UNIFIED IDEOGRAPH
	"8C 55	8888",	#CJK UNIFIED IDEOGRAPH
	"8C 56	7941",	#CJK UNIFIED IDEOGRAPH
	"8C 57	4FC2",	#CJK UNIFIED IDEOGRAPH
	"8C 58	50BE",	#CJK UNIFIED IDEOGRAPH
	"8C 59	5211",	#CJK UNIFIED IDEOGRAPH
	"8C 5A	5144",	#CJK UNIFIED IDEOGRAPH
	"8C 5B	5553",	#CJK UNIFIED IDEOGRAPH
	"8C 5C	572D",	#CJK UNIFIED IDEOGRAPH
	"8C 5D	73EA",	#CJK UNIFIED IDEOGRAPH
	"8C 5E	578B",	#CJK UNIFIED IDEOGRAPH
	"8C 5F	5951",	#CJK UNIFIED IDEOGRAPH
	"8C 60	5F62",	#CJK UNIFIED IDEOGRAPH
	"8C 61	5F84",	#CJK UNIFIED IDEOGRAPH
	"8C 62	6075",	#CJK UNIFIED IDEOGRAPH
	"8C 63	6176",	#CJK UNIFIED IDEOGRAPH
	"8C 64	6167",	#CJK UNIFIED IDEOGRAPH
	"8C 65	61A9",	#CJK UNIFIED IDEOGRAPH
	"8C 66	63B2",	#CJK UNIFIED IDEOGRAPH
	"8C 67	643A",	#CJK UNIFIED IDEOGRAPH
	"8C 68	656C",	#CJK UNIFIED IDEOGRAPH
	"8C 69	666F",	#CJK UNIFIED IDEOGRAPH
	"8C 6A	6842",	#CJK UNIFIED IDEOGRAPH
	"8C 6B	6E13",	#CJK UNIFIED IDEOGRAPH
	"8C 6C	7566",	#CJK UNIFIED IDEOGRAPH
	"8C 6D	7A3D",	#CJK UNIFIED IDEOGRAPH
	"8C 6E	7CFB",	#CJK UNIFIED IDEOGRAPH
	"8C 6F	7D4C",	#CJK UNIFIED IDEOGRAPH
	"8C 70	7D99",	#CJK UNIFIED IDEOGRAPH
	"8C 71	7E4B",	#CJK UNIFIED IDEOGRAPH
	"8C 72	7F6B",	#CJK UNIFIED IDEOGRAPH
	"8C 73	830E",	#CJK UNIFIED IDEOGRAPH
	"8C 74	834A",	#CJK UNIFIED IDEOGRAPH
	"8C 75	86CD",	#CJK UNIFIED IDEOGRAPH
	"8C 76	8A08",	#CJK UNIFIED IDEOGRAPH
	"8C 77	8A63",	#CJK UNIFIED IDEOGRAPH
	"8C 78	8B66",	#CJK UNIFIED IDEOGRAPH
	"8C 79	8EFD",	#CJK UNIFIED IDEOGRAPH
	"8C 7A	981A",	#CJK UNIFIED IDEOGRAPH
	"8C 7B	9D8F",	#CJK UNIFIED IDEOGRAPH
	"8C 7C	82B8",	#CJK UNIFIED IDEOGRAPH
	"8C 7D	8FCE",	#CJK UNIFIED IDEOGRAPH
	"8C 7E	9BE8",	#CJK UNIFIED IDEOGRAPH
	"8C 80	5287",	#CJK UNIFIED IDEOGRAPH
	"8C 81	621F",	#CJK UNIFIED IDEOGRAPH
	"8C 82	6483",	#CJK UNIFIED IDEOGRAPH
	"8C 83	6FC0",	#CJK UNIFIED IDEOGRAPH
	"8C 84	9699",	#CJK UNIFIED IDEOGRAPH
	"8C 85	6841",	#CJK UNIFIED IDEOGRAPH
	"8C 86	5091",	#CJK UNIFIED IDEOGRAPH
	"8C 87	6B20",	#CJK UNIFIED IDEOGRAPH
	"8C 88	6C7A",	#CJK UNIFIED IDEOGRAPH
	"8C 89	6F54",	#CJK UNIFIED IDEOGRAPH
	"8C 8A	7A74",	#CJK UNIFIED IDEOGRAPH
	"8C 8B	7D50",	#CJK UNIFIED IDEOGRAPH
	"8C 8C	8840",	#CJK UNIFIED IDEOGRAPH
	"8C 8D	8A23",	#CJK UNIFIED IDEOGRAPH
	"8C 8E	6708",	#CJK UNIFIED IDEOGRAPH
	"8C 8F	4EF6",	#CJK UNIFIED IDEOGRAPH
	"8C 90	5039",	#CJK UNIFIED IDEOGRAPH
	"8C 91	5026",	#CJK UNIFIED IDEOGRAPH
	"8C 92	5065",	#CJK UNIFIED IDEOGRAPH
	"8C 93	517C",	#CJK UNIFIED IDEOGRAPH
	"8C 94	5238",	#CJK UNIFIED IDEOGRAPH
	"8C 95	5263",	#CJK UNIFIED IDEOGRAPH
	"8C 96	55A7",	#CJK UNIFIED IDEOGRAPH
	"8C 97	570F",	#CJK UNIFIED IDEOGRAPH
	"8C 98	5805",	#CJK UNIFIED IDEOGRAPH
	"8C 99	5ACC",	#CJK UNIFIED IDEOGRAPH
	"8C 9A	5EFA",	#CJK UNIFIED IDEOGRAPH
	"8C 9B	61B2",	#CJK UNIFIED IDEOGRAPH
	"8C 9C	61F8",	#CJK UNIFIED IDEOGRAPH
	"8C 9D	62F3",	#CJK UNIFIED IDEOGRAPH
	"8C 9E	6372",	#CJK UNIFIED IDEOGRAPH
	"8C 9F	691C",	#CJK UNIFIED IDEOGRAPH
	"8C A0	6A29",	#CJK UNIFIED IDEOGRAPH
	"8C A1	727D",	#CJK UNIFIED IDEOGRAPH
	"8C A2	72AC",	#CJK UNIFIED IDEOGRAPH
	"8C A3	732E",	#CJK UNIFIED IDEOGRAPH
	"8C A4	7814",	#CJK UNIFIED IDEOGRAPH
	"8C A5	786F",	#CJK UNIFIED IDEOGRAPH
	"8C A6	7D79",	#CJK UNIFIED IDEOGRAPH
	"8C A7	770C",	#CJK UNIFIED IDEOGRAPH
	"8C A8	80A9",	#CJK UNIFIED IDEOGRAPH
	"8C A9	898B",	#CJK UNIFIED IDEOGRAPH
	"8C AA	8B19",	#CJK UNIFIED IDEOGRAPH
	"8C AB	8CE2",	#CJK UNIFIED IDEOGRAPH
	"8C AC	8ED2",	#CJK UNIFIED IDEOGRAPH
	"8C AD	9063",	#CJK UNIFIED IDEOGRAPH
	"8C AE	9375",	#CJK UNIFIED IDEOGRAPH
	"8C AF	967A",	#CJK UNIFIED IDEOGRAPH
	"8C B0	9855",	#CJK UNIFIED IDEOGRAPH
	"8C B1	9A13",	#CJK UNIFIED IDEOGRAPH
	"8C B2	9E78",	#CJK UNIFIED IDEOGRAPH
	"8C B3	5143",	#CJK UNIFIED IDEOGRAPH
	"8C B4	539F",	#CJK UNIFIED IDEOGRAPH
	"8C B5	53B3",	#CJK UNIFIED IDEOGRAPH
	"8C B6	5E7B",	#CJK UNIFIED IDEOGRAPH
	"8C B7	5F26",	#CJK UNIFIED IDEOGRAPH
	"8C B8	6E1B",	#CJK UNIFIED IDEOGRAPH
	"8C B9	6E90",	#CJK UNIFIED IDEOGRAPH
	"8C BA	7384",	#CJK UNIFIED IDEOGRAPH
	"8C BB	73FE",	#CJK UNIFIED IDEOGRAPH
	"8C BC	7D43",	#CJK UNIFIED IDEOGRAPH
	"8C BD	8237",	#CJK UNIFIED IDEOGRAPH
	"8C BE	8A00",	#CJK UNIFIED IDEOGRAPH
	"8C BF	8AFA",	#CJK UNIFIED IDEOGRAPH
	"8C C0	9650",	#CJK UNIFIED IDEOGRAPH
	"8C C1	4E4E",	#CJK UNIFIED IDEOGRAPH
	"8C C2	500B",	#CJK UNIFIED IDEOGRAPH
	"8C C3	53E4",	#CJK UNIFIED IDEOGRAPH
	"8C C4	547C",	#CJK UNIFIED IDEOGRAPH
	"8C C5	56FA",	#CJK UNIFIED IDEOGRAPH
	"8C C6	59D1",	#CJK UNIFIED IDEOGRAPH
	"8C C7	5B64",	#CJK UNIFIED IDEOGRAPH
	"8C C8	5DF1",	#CJK UNIFIED IDEOGRAPH
	"8C C9	5EAB",	#CJK UNIFIED IDEOGRAPH
	"8C CA	5F27",	#CJK UNIFIED IDEOGRAPH
	"8C CB	6238",	#CJK UNIFIED IDEOGRAPH
	"8C CC	6545",	#CJK UNIFIED IDEOGRAPH
	"8C CD	67AF",	#CJK UNIFIED IDEOGRAPH
	"8C CE	6E56",	#CJK UNIFIED IDEOGRAPH
	"8C CF	72D0",	#CJK UNIFIED IDEOGRAPH
	"8C D0	7CCA",	#CJK UNIFIED IDEOGRAPH
	"8C D1	88B4",	#CJK UNIFIED IDEOGRAPH
	"8C D2	80A1",	#CJK UNIFIED IDEOGRAPH
	"8C D3	80E1",	#CJK UNIFIED IDEOGRAPH
	"8C D4	83F0",	#CJK UNIFIED IDEOGRAPH
	"8C D5	864E",	#CJK UNIFIED IDEOGRAPH
	"8C D6	8A87",	#CJK UNIFIED IDEOGRAPH
	"8C D7	8DE8",	#CJK UNIFIED IDEOGRAPH
	"8C D8	9237",	#CJK UNIFIED IDEOGRAPH
	"8C D9	96C7",	#CJK UNIFIED IDEOGRAPH
	"8C DA	9867",	#CJK UNIFIED IDEOGRAPH
	"8C DB	9F13",	#CJK UNIFIED IDEOGRAPH
	"8C DC	4E94",	#CJK UNIFIED IDEOGRAPH
	"8C DD	4E92",	#CJK UNIFIED IDEOGRAPH
	"8C DE	4F0D",	#CJK UNIFIED IDEOGRAPH
	"8C DF	5348",	#CJK UNIFIED IDEOGRAPH
	"8C E0	5449",	#CJK UNIFIED IDEOGRAPH
	"8C E1	543E",	#CJK UNIFIED IDEOGRAPH
	"8C E2	5A2F",	#CJK UNIFIED IDEOGRAPH
	"8C E3	5F8C",	#CJK UNIFIED IDEOGRAPH
	"8C E4	5FA1",	#CJK UNIFIED IDEOGRAPH
	"8C E5	609F",	#CJK UNIFIED IDEOGRAPH
	"8C E6	68A7",	#CJK UNIFIED IDEOGRAPH
	"8C E7	6A8E",	#CJK UNIFIED IDEOGRAPH
	"8C E8	745A",	#CJK UNIFIED IDEOGRAPH
	"8C E9	7881",	#CJK UNIFIED IDEOGRAPH
	"8C EA	8A9E",	#CJK UNIFIED IDEOGRAPH
	"8C EB	8AA4",	#CJK UNIFIED IDEOGRAPH
	"8C EC	8B77",	#CJK UNIFIED IDEOGRAPH
	"8C ED	9190",	#CJK UNIFIED IDEOGRAPH
	"8C EE	4E5E",	#CJK UNIFIED IDEOGRAPH
	"8C EF	9BC9",	#CJK UNIFIED IDEOGRAPH
	"8C F0	4EA4",	#CJK UNIFIED IDEOGRAPH
	"8C F1	4F7C",	#CJK UNIFIED IDEOGRAPH
	"8C F2	4FAF",	#CJK UNIFIED IDEOGRAPH
	"8C F3	5019",	#CJK UNIFIED IDEOGRAPH
	"8C F4	5016",	#CJK UNIFIED IDEOGRAPH
	"8C F5	5149",	#CJK UNIFIED IDEOGRAPH
	"8C F6	516C",	#CJK UNIFIED IDEOGRAPH
	"8C F7	529F",	#CJK UNIFIED IDEOGRAPH
	"8C F8	52B9",	#CJK UNIFIED IDEOGRAPH
	"8C F9	52FE",	#CJK UNIFIED IDEOGRAPH
	"8C FA	539A",	#CJK UNIFIED IDEOGRAPH
	"8C FB	53E3",	#CJK UNIFIED IDEOGRAPH
	"8C FC	5411",	#CJK UNIFIED IDEOGRAPH
	"8D 40	540E",	#CJK UNIFIED IDEOGRAPH
	"8D 41	5589",	#CJK UNIFIED IDEOGRAPH
	"8D 42	5751",	#CJK UNIFIED IDEOGRAPH
	"8D 43	57A2",	#CJK UNIFIED IDEOGRAPH
	"8D 44	597D",	#CJK UNIFIED IDEOGRAPH
	"8D 45	5B54",	#CJK UNIFIED IDEOGRAPH
	"8D 46	5B5D",	#CJK UNIFIED IDEOGRAPH
	"8D 47	5B8F",	#CJK UNIFIED IDEOGRAPH
	"8D 48	5DE5",	#CJK UNIFIED IDEOGRAPH
	"8D 49	5DE7",	#CJK UNIFIED IDEOGRAPH
	"8D 4A	5DF7",	#CJK UNIFIED IDEOGRAPH
	"8D 4B	5E78",	#CJK UNIFIED IDEOGRAPH
	"8D 4C	5E83",	#CJK UNIFIED IDEOGRAPH
	"8D 4D	5E9A",	#CJK UNIFIED IDEOGRAPH
	"8D 4E	5EB7",	#CJK UNIFIED IDEOGRAPH
	"8D 4F	5F18",	#CJK UNIFIED IDEOGRAPH
	"8D 50	6052",	#CJK UNIFIED IDEOGRAPH
	"8D 51	614C",	#CJK UNIFIED IDEOGRAPH
	"8D 52	6297",	#CJK UNIFIED IDEOGRAPH
	"8D 53	62D8",	#CJK UNIFIED IDEOGRAPH
	"8D 54	63A7",	#CJK UNIFIED IDEOGRAPH
	"8D 55	653B",	#CJK UNIFIED IDEOGRAPH
	"8D 56	6602",	#CJK UNIFIED IDEOGRAPH
	"8D 57	6643",	#CJK UNIFIED IDEOGRAPH
	"8D 58	66F4",	#CJK UNIFIED IDEOGRAPH
	"8D 59	676D",	#CJK UNIFIED IDEOGRAPH
	"8D 5A	6821",	#CJK UNIFIED IDEOGRAPH
	"8D 5B	6897",	#CJK UNIFIED IDEOGRAPH
	"8D 5C	69CB",	#CJK UNIFIED IDEOGRAPH
	"8D 5D	6C5F",	#CJK UNIFIED IDEOGRAPH
	"8D 5E	6D2A",	#CJK UNIFIED IDEOGRAPH
	"8D 5F	6D69",	#CJK UNIFIED IDEOGRAPH
	"8D 60	6E2F",	#CJK UNIFIED IDEOGRAPH
	"8D 61	6E9D",	#CJK UNIFIED IDEOGRAPH
	"8D 62	7532",	#CJK UNIFIED IDEOGRAPH
	"8D 63	7687",	#CJK UNIFIED IDEOGRAPH
	"8D 64	786C",	#CJK UNIFIED IDEOGRAPH
	"8D 65	7A3F",	#CJK UNIFIED IDEOGRAPH
	"8D 66	7CE0",	#CJK UNIFIED IDEOGRAPH
	"8D 67	7D05",	#CJK UNIFIED IDEOGRAPH
	"8D 68	7D18",	#CJK UNIFIED IDEOGRAPH
	"8D 69	7D5E",	#CJK UNIFIED IDEOGRAPH
	"8D 6A	7DB1",	#CJK UNIFIED IDEOGRAPH
	"8D 6B	8015",	#CJK UNIFIED IDEOGRAPH
	"8D 6C	8003",	#CJK UNIFIED IDEOGRAPH
	"8D 6D	80AF",	#CJK UNIFIED IDEOGRAPH
	"8D 6E	80B1",	#CJK UNIFIED IDEOGRAPH
	"8D 6F	8154",	#CJK UNIFIED IDEOGRAPH
	"8D 70	818F",	#CJK UNIFIED IDEOGRAPH
	"8D 71	822A",	#CJK UNIFIED IDEOGRAPH
	"8D 72	8352",	#CJK UNIFIED IDEOGRAPH
	"8D 73	884C",	#CJK UNIFIED IDEOGRAPH
	"8D 74	8861",	#CJK UNIFIED IDEOGRAPH
	"8D 75	8B1B",	#CJK UNIFIED IDEOGRAPH
	"8D 76	8CA2",	#CJK UNIFIED IDEOGRAPH
	"8D 77	8CFC",	#CJK UNIFIED IDEOGRAPH
	"8D 78	90CA",	#CJK UNIFIED IDEOGRAPH
	"8D 79	9175",	#CJK UNIFIED IDEOGRAPH
	"8D 7A	9271",	#CJK UNIFIED IDEOGRAPH
	"8D 7B	783F",	#CJK UNIFIED IDEOGRAPH
	"8D 7C	92FC",	#CJK UNIFIED IDEOGRAPH
	"8D 7D	95A4",	#CJK UNIFIED IDEOGRAPH
	"8D 7E	964D",	#CJK UNIFIED IDEOGRAPH
	"8D 80	9805",	#CJK UNIFIED IDEOGRAPH
	"8D 81	9999",	#CJK UNIFIED IDEOGRAPH
	"8D 82	9AD8",	#CJK UNIFIED IDEOGRAPH
	"8D 83	9D3B",	#CJK UNIFIED IDEOGRAPH
	"8D 84	525B",	#CJK UNIFIED IDEOGRAPH
	"8D 85	52AB",	#CJK UNIFIED IDEOGRAPH
	"8D 86	53F7",	#CJK UNIFIED IDEOGRAPH
	"8D 87	5408",	#CJK UNIFIED IDEOGRAPH
	"8D 88	58D5",	#CJK UNIFIED IDEOGRAPH
	"8D 89	62F7",	#CJK UNIFIED IDEOGRAPH
	"8D 8A	6FE0",	#CJK UNIFIED IDEOGRAPH
	"8D 8B	8C6A",	#CJK UNIFIED IDEOGRAPH
	"8D 8C	8F5F",	#CJK UNIFIED IDEOGRAPH
	"8D 8D	9EB9",	#CJK UNIFIED IDEOGRAPH
	"8D 8E	514B",	#CJK UNIFIED IDEOGRAPH
	"8D 8F	523B",	#CJK UNIFIED IDEOGRAPH
	"8D 90	544A",	#CJK UNIFIED IDEOGRAPH
	"8D 91	56FD",	#CJK UNIFIED IDEOGRAPH
	"8D 92	7A40",	#CJK UNIFIED IDEOGRAPH
	"8D 93	9177",	#CJK UNIFIED IDEOGRAPH
	"8D 94	9D60",	#CJK UNIFIED IDEOGRAPH
	"8D 95	9ED2",	#CJK UNIFIED IDEOGRAPH
	"8D 96	7344",	#CJK UNIFIED IDEOGRAPH
	"8D 97	6F09",	#CJK UNIFIED IDEOGRAPH
	"8D 98	8170",	#CJK UNIFIED IDEOGRAPH
	"8D 99	7511",	#CJK UNIFIED IDEOGRAPH
	"8D 9A	5FFD",	#CJK UNIFIED IDEOGRAPH
	"8D 9B	60DA",	#CJK UNIFIED IDEOGRAPH
	"8D 9C	9AA8",	#CJK UNIFIED IDEOGRAPH
	"8D 9D	72DB",	#CJK UNIFIED IDEOGRAPH
	"8D 9E	8FBC",	#CJK UNIFIED IDEOGRAPH
	"8D 9F	6B64",	#CJK UNIFIED IDEOGRAPH
	"8D A0	9803",	#CJK UNIFIED IDEOGRAPH
	"8D A1	4ECA",	#CJK UNIFIED IDEOGRAPH
	"8D A2	56F0",	#CJK UNIFIED IDEOGRAPH
	"8D A3	5764",	#CJK UNIFIED IDEOGRAPH
	"8D A4	58BE",	#CJK UNIFIED IDEOGRAPH
	"8D A5	5A5A",	#CJK UNIFIED IDEOGRAPH
	"8D A6	6068",	#CJK UNIFIED IDEOGRAPH
	"8D A7	61C7",	#CJK UNIFIED IDEOGRAPH
	"8D A8	660F",	#CJK UNIFIED IDEOGRAPH
	"8D A9	6606",	#CJK UNIFIED IDEOGRAPH
	"8D AA	6839",	#CJK UNIFIED IDEOGRAPH
	"8D AB	68B1",	#CJK UNIFIED IDEOGRAPH
	"8D AC	6DF7",	#CJK UNIFIED IDEOGRAPH
	"8D AD	75D5",	#CJK UNIFIED IDEOGRAPH
	"8D AE	7D3A",	#CJK UNIFIED IDEOGRAPH
	"8D AF	826E",	#CJK UNIFIED IDEOGRAPH
	"8D B0	9B42",	#CJK UNIFIED IDEOGRAPH
	"8D B1	4E9B",	#CJK UNIFIED IDEOGRAPH
	"8D B2	4F50",	#CJK UNIFIED IDEOGRAPH
	"8D B3	53C9",	#CJK UNIFIED IDEOGRAPH
	"8D B4	5506",	#CJK UNIFIED IDEOGRAPH
	"8D B5	5D6F",	#CJK UNIFIED IDEOGRAPH
	"8D B6	5DE6",	#CJK UNIFIED IDEOGRAPH
	"8D B7	5DEE",	#CJK UNIFIED IDEOGRAPH
	"8D B8	67FB",	#CJK UNIFIED IDEOGRAPH
	"8D B9	6C99",	#CJK UNIFIED IDEOGRAPH
	"8D BA	7473",	#CJK UNIFIED IDEOGRAPH
	"8D BB	7802",	#CJK UNIFIED IDEOGRAPH
	"8D BC	8A50",	#CJK UNIFIED IDEOGRAPH
	"8D BD	9396",	#CJK UNIFIED IDEOGRAPH
	"8D BE	88DF",	#CJK UNIFIED IDEOGRAPH
	"8D BF	5750",	#CJK UNIFIED IDEOGRAPH
	"8D C0	5EA7",	#CJK UNIFIED IDEOGRAPH
	"8D C1	632B",	#CJK UNIFIED IDEOGRAPH
	"8D C2	50B5",	#CJK UNIFIED IDEOGRAPH
	"8D C3	50AC",	#CJK UNIFIED IDEOGRAPH
	"8D C4	518D",	#CJK UNIFIED IDEOGRAPH
	"8D C5	6700",	#CJK UNIFIED IDEOGRAPH
	"8D C6	54C9",	#CJK UNIFIED IDEOGRAPH
	"8D C7	585E",	#CJK UNIFIED IDEOGRAPH
	"8D C8	59BB",	#CJK UNIFIED IDEOGRAPH
	"8D C9	5BB0",	#CJK UNIFIED IDEOGRAPH
	"8D CA	5F69",	#CJK UNIFIED IDEOGRAPH
	"8D CB	624D",	#CJK UNIFIED IDEOGRAPH
	"8D CC	63A1",	#CJK UNIFIED IDEOGRAPH
	"8D CD	683D",	#CJK UNIFIED IDEOGRAPH
	"8D CE	6B73",	#CJK UNIFIED IDEOGRAPH
	"8D CF	6E08",	#CJK UNIFIED IDEOGRAPH
	"8D D0	707D",	#CJK UNIFIED IDEOGRAPH
	"8D D1	91C7",	#CJK UNIFIED IDEOGRAPH
	"8D D2	7280",	#CJK UNIFIED IDEOGRAPH
	"8D D3	7815",	#CJK UNIFIED IDEOGRAPH
	"8D D4	7826",	#CJK UNIFIED IDEOGRAPH
	"8D D5	796D",	#CJK UNIFIED IDEOGRAPH
	"8D D6	658E",	#CJK UNIFIED IDEOGRAPH
	"8D D7	7D30",	#CJK UNIFIED IDEOGRAPH
	"8D D8	83DC",	#CJK UNIFIED IDEOGRAPH
	"8D D9	88C1",	#CJK UNIFIED IDEOGRAPH
	"8D DA	8F09",	#CJK UNIFIED IDEOGRAPH
	"8D DB	969B",	#CJK UNIFIED IDEOGRAPH
	"8D DC	5264",	#CJK UNIFIED IDEOGRAPH
	"8D DD	5728",	#CJK UNIFIED IDEOGRAPH
	"8D DE	6750",	#CJK UNIFIED IDEOGRAPH
	"8D DF	7F6A",	#CJK UNIFIED IDEOGRAPH
	"8D E0	8CA1",	#CJK UNIFIED IDEOGRAPH
	"8D E1	51B4",	#CJK UNIFIED IDEOGRAPH
	"8D E2	5742",	#CJK UNIFIED IDEOGRAPH
	"8D E3	962A",	#CJK UNIFIED IDEOGRAPH
	"8D E4	583A",	#CJK UNIFIED IDEOGRAPH
	"8D E5	698A",	#CJK UNIFIED IDEOGRAPH
	"8D E6	80B4",	#CJK UNIFIED IDEOGRAPH
	"8D E7	54B2",	#CJK UNIFIED IDEOGRAPH
	"8D E8	5D0E",	#CJK UNIFIED IDEOGRAPH
	"8D E9	57FC",	#CJK UNIFIED IDEOGRAPH
	"8D EA	7895",	#CJK UNIFIED IDEOGRAPH
	"8D EB	9DFA",	#CJK UNIFIED IDEOGRAPH
	"8D EC	4F5C",	#CJK UNIFIED IDEOGRAPH
	"8D ED	524A",	#CJK UNIFIED IDEOGRAPH
	"8D EE	548B",	#CJK UNIFIED IDEOGRAPH
	"8D EF	643E",	#CJK UNIFIED IDEOGRAPH
	"8D F0	6628",	#CJK UNIFIED IDEOGRAPH
	"8D F1	6714",	#CJK UNIFIED IDEOGRAPH
	"8D F2	67F5",	#CJK UNIFIED IDEOGRAPH
	"8D F3	7A84",	#CJK UNIFIED IDEOGRAPH
	"8D F4	7B56",	#CJK UNIFIED IDEOGRAPH
	"8D F5	7D22",	#CJK UNIFIED IDEOGRAPH
	"8D F6	932F",	#CJK UNIFIED IDEOGRAPH
	"8D F7	685C",	#CJK UNIFIED IDEOGRAPH
	"8D F8	9BAD",	#CJK UNIFIED IDEOGRAPH
	"8D F9	7B39",	#CJK UNIFIED IDEOGRAPH
	"8D FA	5319",	#CJK UNIFIED IDEOGRAPH
	"8D FB	518A",	#CJK UNIFIED IDEOGRAPH
	"8D FC	5237",	#CJK UNIFIED IDEOGRAPH
	"8E 40	5BDF",	#CJK UNIFIED IDEOGRAPH
	"8E 41	62F6",	#CJK UNIFIED IDEOGRAPH
	"8E 42	64AE",	#CJK UNIFIED IDEOGRAPH
	"8E 43	64E6",	#CJK UNIFIED IDEOGRAPH
	"8E 44	672D",	#CJK UNIFIED IDEOGRAPH
	"8E 45	6BBA",	#CJK UNIFIED IDEOGRAPH
	"8E 46	85A9",	#CJK UNIFIED IDEOGRAPH
	"8E 47	96D1",	#CJK UNIFIED IDEOGRAPH
	"8E 48	7690",	#CJK UNIFIED IDEOGRAPH
	"8E 49	9BD6",	#CJK UNIFIED IDEOGRAPH
	"8E 4A	634C",	#CJK UNIFIED IDEOGRAPH
	"8E 4B	9306",	#CJK UNIFIED IDEOGRAPH
	"8E 4C	9BAB",	#CJK UNIFIED IDEOGRAPH
	"8E 4D	76BF",	#CJK UNIFIED IDEOGRAPH
	"8E 4E	6652",	#CJK UNIFIED IDEOGRAPH
	"8E 4F	4E09",	#CJK UNIFIED IDEOGRAPH
	"8E 50	5098",	#CJK UNIFIED IDEOGRAPH
	"8E 51	53C2",	#CJK UNIFIED IDEOGRAPH
	"8E 52	5C71",	#CJK UNIFIED IDEOGRAPH
	"8E 53	60E8",	#CJK UNIFIED IDEOGRAPH
	"8E 54	6492",	#CJK UNIFIED IDEOGRAPH
	"8E 55	6563",	#CJK UNIFIED IDEOGRAPH
	"8E 56	685F",	#CJK UNIFIED IDEOGRAPH
	"8E 57	71E6",	#CJK UNIFIED IDEOGRAPH
	"8E 58	73CA",	#CJK UNIFIED IDEOGRAPH
	"8E 59	7523",	#CJK UNIFIED IDEOGRAPH
	"8E 5A	7B97",	#CJK UNIFIED IDEOGRAPH
	"8E 5B	7E82",	#CJK UNIFIED IDEOGRAPH
	"8E 5C	8695",	#CJK UNIFIED IDEOGRAPH
	"8E 5D	8B83",	#CJK UNIFIED IDEOGRAPH
	"8E 5E	8CDB",	#CJK UNIFIED IDEOGRAPH
	"8E 5F	9178",	#CJK UNIFIED IDEOGRAPH
	"8E 60	9910",	#CJK UNIFIED IDEOGRAPH
	"8E 61	65AC",	#CJK UNIFIED IDEOGRAPH
	"8E 62	66AB",	#CJK UNIFIED IDEOGRAPH
	"8E 63	6B8B",	#CJK UNIFIED IDEOGRAPH
	"8E 64	4ED5",	#CJK UNIFIED IDEOGRAPH
	"8E 65	4ED4",	#CJK UNIFIED IDEOGRAPH
	"8E 66	4F3A",	#CJK UNIFIED IDEOGRAPH
	"8E 67	4F7F",	#CJK UNIFIED IDEOGRAPH
	"8E 68	523A",	#CJK UNIFIED IDEOGRAPH
	"8E 69	53F8",	#CJK UNIFIED IDEOGRAPH
	"8E 6A	53F2",	#CJK UNIFIED IDEOGRAPH
	"8E 6B	55E3",	#CJK UNIFIED IDEOGRAPH
	"8E 6C	56DB",	#CJK UNIFIED IDEOGRAPH
	"8E 6D	58EB",	#CJK UNIFIED IDEOGRAPH
	"8E 6E	59CB",	#CJK UNIFIED IDEOGRAPH
	"8E 6F	59C9",	#CJK UNIFIED IDEOGRAPH
	"8E 70	59FF",	#CJK UNIFIED IDEOGRAPH
	"8E 71	5B50",	#CJK UNIFIED IDEOGRAPH
	"8E 72	5C4D",	#CJK UNIFIED IDEOGRAPH
	"8E 73	5E02",	#CJK UNIFIED IDEOGRAPH
	"8E 74	5E2B",	#CJK UNIFIED IDEOGRAPH
	"8E 75	5FD7",	#CJK UNIFIED IDEOGRAPH
	"8E 76	601D",	#CJK UNIFIED IDEOGRAPH
	"8E 77	6307",	#CJK UNIFIED IDEOGRAPH
	"8E 78	652F",	#CJK UNIFIED IDEOGRAPH
	"8E 79	5B5C",	#CJK UNIFIED IDEOGRAPH
	"8E 7A	65AF",	#CJK UNIFIED IDEOGRAPH
	"8E 7B	65BD",	#CJK UNIFIED IDEOGRAPH
	"8E 7C	65E8",	#CJK UNIFIED IDEOGRAPH
	"8E 7D	679D",	#CJK UNIFIED IDEOGRAPH
	"8E 7E	6B62",	#CJK UNIFIED IDEOGRAPH
	"8E 80	6B7B",	#CJK UNIFIED IDEOGRAPH
	"8E 81	6C0F",	#CJK UNIFIED IDEOGRAPH
	"8E 82	7345",	#CJK UNIFIED IDEOGRAPH
	"8E 83	7949",	#CJK UNIFIED IDEOGRAPH
	"8E 84	79C1",	#CJK UNIFIED IDEOGRAPH
	"8E 85	7CF8",	#CJK UNIFIED IDEOGRAPH
	"8E 86	7D19",	#CJK UNIFIED IDEOGRAPH
	"8E 87	7D2B",	#CJK UNIFIED IDEOGRAPH
	"8E 88	80A2",	#CJK UNIFIED IDEOGRAPH
	"8E 89	8102",	#CJK UNIFIED IDEOGRAPH
	"8E 8A	81F3",	#CJK UNIFIED IDEOGRAPH
	"8E 8B	8996",	#CJK UNIFIED IDEOGRAPH
	"8E 8C	8A5E",	#CJK UNIFIED IDEOGRAPH
	"8E 8D	8A69",	#CJK UNIFIED IDEOGRAPH
	"8E 8E	8A66",	#CJK UNIFIED IDEOGRAPH
	"8E 8F	8A8C",	#CJK UNIFIED IDEOGRAPH
	"8E 90	8AEE",	#CJK UNIFIED IDEOGRAPH
	"8E 91	8CC7",	#CJK UNIFIED IDEOGRAPH
	"8E 92	8CDC",	#CJK UNIFIED IDEOGRAPH
	"8E 93	96CC",	#CJK UNIFIED IDEOGRAPH
	"8E 94	98FC",	#CJK UNIFIED IDEOGRAPH
	"8E 95	6B6F",	#CJK UNIFIED IDEOGRAPH
	"8E 96	4E8B",	#CJK UNIFIED IDEOGRAPH
	"8E 97	4F3C",	#CJK UNIFIED IDEOGRAPH
	"8E 98	4F8D",	#CJK UNIFIED IDEOGRAPH
	"8E 99	5150",	#CJK UNIFIED IDEOGRAPH
	"8E 9A	5B57",	#CJK UNIFIED IDEOGRAPH
	"8E 9B	5BFA",	#CJK UNIFIED IDEOGRAPH
	"8E 9C	6148",	#CJK UNIFIED IDEOGRAPH
	"8E 9D	6301",	#CJK UNIFIED IDEOGRAPH
	"8E 9E	6642",	#CJK UNIFIED IDEOGRAPH
	"8E 9F	6B21",	#CJK UNIFIED IDEOGRAPH
	"8E A0	6ECB",	#CJK UNIFIED IDEOGRAPH
	"8E A1	6CBB",	#CJK UNIFIED IDEOGRAPH
	"8E A2	723E",	#CJK UNIFIED IDEOGRAPH
	"8E A3	74BD",	#CJK UNIFIED IDEOGRAPH
	"8E A4	75D4",	#CJK UNIFIED IDEOGRAPH
	"8E A5	78C1",	#CJK UNIFIED IDEOGRAPH
	"8E A6	793A",	#CJK UNIFIED IDEOGRAPH
	"8E A7	800C",	#CJK UNIFIED IDEOGRAPH
	"8E A8	8033",	#CJK UNIFIED IDEOGRAPH
	"8E A9	81EA",	#CJK UNIFIED IDEOGRAPH
	"8E AA	8494",	#CJK UNIFIED IDEOGRAPH
	"8E AB	8F9E",	#CJK UNIFIED IDEOGRAPH
	"8E AC	6C50",	#CJK UNIFIED IDEOGRAPH
	"8E AD	9E7F",	#CJK UNIFIED IDEOGRAPH
	"8E AE	5F0F",	#CJK UNIFIED IDEOGRAPH
	"8E AF	8B58",	#CJK UNIFIED IDEOGRAPH
	"8E B0	9D2B",	#CJK UNIFIED IDEOGRAPH
	"8E B1	7AFA",	#CJK UNIFIED IDEOGRAPH
	"8E B2	8EF8",	#CJK UNIFIED IDEOGRAPH
	"8E B3	5B8D",	#CJK UNIFIED IDEOGRAPH
	"8E B4	96EB",	#CJK UNIFIED IDEOGRAPH
	"8E B5	4E03",	#CJK UNIFIED IDEOGRAPH
	"8E B6	53F1",	#CJK UNIFIED IDEOGRAPH
	"8E B7	57F7",	#CJK UNIFIED IDEOGRAPH
	"8E B8	5931",	#CJK UNIFIED IDEOGRAPH
	"8E B9	5AC9",	#CJK UNIFIED IDEOGRAPH
	"8E BA	5BA4",	#CJK UNIFIED IDEOGRAPH
	"8E BB	6089",	#CJK UNIFIED IDEOGRAPH
	"8E BC	6E7F",	#CJK UNIFIED IDEOGRAPH
	"8E BD	6F06",	#CJK UNIFIED IDEOGRAPH
	"8E BE	75BE",	#CJK UNIFIED IDEOGRAPH
	"8E BF	8CEA",	#CJK UNIFIED IDEOGRAPH
	"8E C0	5B9F",	#CJK UNIFIED IDEOGRAPH
	"8E C1	8500",	#CJK UNIFIED IDEOGRAPH
	"8E C2	7BE0",	#CJK UNIFIED IDEOGRAPH
	"8E C3	5072",	#CJK UNIFIED IDEOGRAPH
	"8E C4	67F4",	#CJK UNIFIED IDEOGRAPH
	"8E C5	829D",	#CJK UNIFIED IDEOGRAPH
	"8E C6	5C61",	#CJK UNIFIED IDEOGRAPH
	"8E C7	854A",	#CJK UNIFIED IDEOGRAPH
	"8E C8	7E1E",	#CJK UNIFIED IDEOGRAPH
	"8E C9	820E",	#CJK UNIFIED IDEOGRAPH
	"8E CA	5199",	#CJK UNIFIED IDEOGRAPH
	"8E CB	5C04",	#CJK UNIFIED IDEOGRAPH
	"8E CC	6368",	#CJK UNIFIED IDEOGRAPH
	"8E CD	8D66",	#CJK UNIFIED IDEOGRAPH
	"8E CE	659C",	#CJK UNIFIED IDEOGRAPH
	"8E CF	716E",	#CJK UNIFIED IDEOGRAPH
	"8E D0	793E",	#CJK UNIFIED IDEOGRAPH
	"8E D1	7D17",	#CJK UNIFIED IDEOGRAPH
	"8E D2	8005",	#CJK UNIFIED IDEOGRAPH
	"8E D3	8B1D",	#CJK UNIFIED IDEOGRAPH
	"8E D4	8ECA",	#CJK UNIFIED IDEOGRAPH
	"8E D5	906E",	#CJK UNIFIED IDEOGRAPH
	"8E D6	86C7",	#CJK UNIFIED IDEOGRAPH
	"8E D7	90AA",	#CJK UNIFIED IDEOGRAPH
	"8E D8	501F",	#CJK UNIFIED IDEOGRAPH
	"8E D9	52FA",	#CJK UNIFIED IDEOGRAPH
	"8E DA	5C3A",	#CJK UNIFIED IDEOGRAPH
	"8E DB	6753",	#CJK UNIFIED IDEOGRAPH
	"8E DC	707C",	#CJK UNIFIED IDEOGRAPH
	"8E DD	7235",	#CJK UNIFIED IDEOGRAPH
	"8E DE	914C",	#CJK UNIFIED IDEOGRAPH
	"8E DF	91C8",	#CJK UNIFIED IDEOGRAPH
	"8E E0	932B",	#CJK UNIFIED IDEOGRAPH
	"8E E1	82E5",	#CJK UNIFIED IDEOGRAPH
	"8E E2	5BC2",	#CJK UNIFIED IDEOGRAPH
	"8E E3	5F31",	#CJK UNIFIED IDEOGRAPH
	"8E E4	60F9",	#CJK UNIFIED IDEOGRAPH
	"8E E5	4E3B",	#CJK UNIFIED IDEOGRAPH
	"8E E6	53D6",	#CJK UNIFIED IDEOGRAPH
	"8E E7	5B88",	#CJK UNIFIED IDEOGRAPH
	"8E E8	624B",	#CJK UNIFIED IDEOGRAPH
	"8E E9	6731",	#CJK UNIFIED IDEOGRAPH
	"8E EA	6B8A",	#CJK UNIFIED IDEOGRAPH
	"8E EB	72E9",	#CJK UNIFIED IDEOGRAPH
	"8E EC	73E0",	#CJK UNIFIED IDEOGRAPH
	"8E ED	7A2E",	#CJK UNIFIED IDEOGRAPH
	"8E EE	816B",	#CJK UNIFIED IDEOGRAPH
	"8E EF	8DA3",	#CJK UNIFIED IDEOGRAPH
	"8E F0	9152",	#CJK UNIFIED IDEOGRAPH
	"8E F1	9996",	#CJK UNIFIED IDEOGRAPH
	"8E F2	5112",	#CJK UNIFIED IDEOGRAPH
	"8E F3	53D7",	#CJK UNIFIED IDEOGRAPH
	"8E F4	546A",	#CJK UNIFIED IDEOGRAPH
	"8E F5	5BFF",	#CJK UNIFIED IDEOGRAPH
	"8E F6	6388",	#CJK UNIFIED IDEOGRAPH
	"8E F7	6A39",	#CJK UNIFIED IDEOGRAPH
	"8E F8	7DAC",	#CJK UNIFIED IDEOGRAPH
	"8E F9	9700",	#CJK UNIFIED IDEOGRAPH
	"8E FA	56DA",	#CJK UNIFIED IDEOGRAPH
	"8E FB	53CE",	#CJK UNIFIED IDEOGRAPH
	"8E FC	5468",	#CJK UNIFIED IDEOGRAPH
	"8F 40	5B97",	#CJK UNIFIED IDEOGRAPH
	"8F 41	5C31",	#CJK UNIFIED IDEOGRAPH
	"8F 42	5DDE",	#CJK UNIFIED IDEOGRAPH
	"8F 43	4FEE",	#CJK UNIFIED IDEOGRAPH
	"8F 44	6101",	#CJK UNIFIED IDEOGRAPH
	"8F 45	62FE",	#CJK UNIFIED IDEOGRAPH
	"8F 46	6D32",	#CJK UNIFIED IDEOGRAPH
	"8F 47	79C0",	#CJK UNIFIED IDEOGRAPH
	"8F 48	79CB",	#CJK UNIFIED IDEOGRAPH
	"8F 49	7D42",	#CJK UNIFIED IDEOGRAPH
	"8F 4A	7E4D",	#CJK UNIFIED IDEOGRAPH
	"8F 4B	7FD2",	#CJK UNIFIED IDEOGRAPH
	"8F 4C	81ED",	#CJK UNIFIED IDEOGRAPH
	"8F 4D	821F",	#CJK UNIFIED IDEOGRAPH
	"8F 4E	8490",	#CJK UNIFIED IDEOGRAPH
	"8F 4F	8846",	#CJK UNIFIED IDEOGRAPH
	"8F 50	8972",	#CJK UNIFIED IDEOGRAPH
	"8F 51	8B90",	#CJK UNIFIED IDEOGRAPH
	"8F 52	8E74",	#CJK UNIFIED IDEOGRAPH
	"8F 53	8F2F",	#CJK UNIFIED IDEOGRAPH
	"8F 54	9031",	#CJK UNIFIED IDEOGRAPH
	"8F 55	914B",	#CJK UNIFIED IDEOGRAPH
	"8F 56	916C",	#CJK UNIFIED IDEOGRAPH
	"8F 57	96C6",	#CJK UNIFIED IDEOGRAPH
	"8F 58	919C",	#CJK UNIFIED IDEOGRAPH
	"8F 59	4EC0",	#CJK UNIFIED IDEOGRAPH
	"8F 5A	4F4F",	#CJK UNIFIED IDEOGRAPH
	"8F 5B	5145",	#CJK UNIFIED IDEOGRAPH
	"8F 5C	5341",	#CJK UNIFIED IDEOGRAPH
	"8F 5D	5F93",	#CJK UNIFIED IDEOGRAPH
	"8F 5E	620E",	#CJK UNIFIED IDEOGRAPH
	"8F 5F	67D4",	#CJK UNIFIED IDEOGRAPH
	"8F 60	6C41",	#CJK UNIFIED IDEOGRAPH
	"8F 61	6E0B",	#CJK UNIFIED IDEOGRAPH
	"8F 62	7363",	#CJK UNIFIED IDEOGRAPH
	"8F 63	7E26",	#CJK UNIFIED IDEOGRAPH
	"8F 64	91CD",	#CJK UNIFIED IDEOGRAPH
	"8F 65	9283",	#CJK UNIFIED IDEOGRAPH
	"8F 66	53D4",	#CJK UNIFIED IDEOGRAPH
	"8F 67	5919",	#CJK UNIFIED IDEOGRAPH
	"8F 68	5BBF",	#CJK UNIFIED IDEOGRAPH
	"8F 69	6DD1",	#CJK UNIFIED IDEOGRAPH
	"8F 6A	795D",	#CJK UNIFIED IDEOGRAPH
	"8F 6B	7E2E",	#CJK UNIFIED IDEOGRAPH
	"8F 6C	7C9B",	#CJK UNIFIED IDEOGRAPH
	"8F 6D	587E",	#CJK UNIFIED IDEOGRAPH
	"8F 6E	719F",	#CJK UNIFIED IDEOGRAPH
	"8F 6F	51FA",	#CJK UNIFIED IDEOGRAPH
	"8F 70	8853",	#CJK UNIFIED IDEOGRAPH
	"8F 71	8FF0",	#CJK UNIFIED IDEOGRAPH
	"8F 72	4FCA",	#CJK UNIFIED IDEOGRAPH
	"8F 73	5CFB",	#CJK UNIFIED IDEOGRAPH
	"8F 74	6625",	#CJK UNIFIED IDEOGRAPH
	"8F 75	77AC",	#CJK UNIFIED IDEOGRAPH
	"8F 76	7AE3",	#CJK UNIFIED IDEOGRAPH
	"8F 77	821C",	#CJK UNIFIED IDEOGRAPH
	"8F 78	99FF",	#CJK UNIFIED IDEOGRAPH
	"8F 79	51C6",	#CJK UNIFIED IDEOGRAPH
	"8F 7A	5FAA",	#CJK UNIFIED IDEOGRAPH
	"8F 7B	65EC",	#CJK UNIFIED IDEOGRAPH
	"8F 7C	696F",	#CJK UNIFIED IDEOGRAPH
	"8F 7D	6B89",	#CJK UNIFIED IDEOGRAPH
	"8F 7E	6DF3",	#CJK UNIFIED IDEOGRAPH
	"8F 80	6E96",	#CJK UNIFIED IDEOGRAPH
	"8F 81	6F64",	#CJK UNIFIED IDEOGRAPH
	"8F 82	76FE",	#CJK UNIFIED IDEOGRAPH
	"8F 83	7D14",	#CJK UNIFIED IDEOGRAPH
	"8F 84	5DE1",	#CJK UNIFIED IDEOGRAPH
	"8F 85	9075",	#CJK UNIFIED IDEOGRAPH
	"8F 86	9187",	#CJK UNIFIED IDEOGRAPH
	"8F 87	9806",	#CJK UNIFIED IDEOGRAPH
	"8F 88	51E6",	#CJK UNIFIED IDEOGRAPH
	"8F 89	521D",	#CJK UNIFIED IDEOGRAPH
	"8F 8A	6240",	#CJK UNIFIED IDEOGRAPH
	"8F 8B	6691",	#CJK UNIFIED IDEOGRAPH
	"8F 8C	66D9",	#CJK UNIFIED IDEOGRAPH
	"8F 8D	6E1A",	#CJK UNIFIED IDEOGRAPH
	"8F 8E	5EB6",	#CJK UNIFIED IDEOGRAPH
	"8F 8F	7DD2",	#CJK UNIFIED IDEOGRAPH
	"8F 90	7F72",	#CJK UNIFIED IDEOGRAPH
	"8F 91	66F8",	#CJK UNIFIED IDEOGRAPH
	"8F 92	85AF",	#CJK UNIFIED IDEOGRAPH
	"8F 93	85F7",	#CJK UNIFIED IDEOGRAPH
	"8F 94	8AF8",	#CJK UNIFIED IDEOGRAPH
	"8F 95	52A9",	#CJK UNIFIED IDEOGRAPH
	"8F 96	53D9",	#CJK UNIFIED IDEOGRAPH
	"8F 97	5973",	#CJK UNIFIED IDEOGRAPH
	"8F 98	5E8F",	#CJK UNIFIED IDEOGRAPH
	"8F 99	5F90",	#CJK UNIFIED IDEOGRAPH
	"8F 9A	6055",	#CJK UNIFIED IDEOGRAPH
	"8F 9B	92E4",	#CJK UNIFIED IDEOGRAPH
	"8F 9C	9664",	#CJK UNIFIED IDEOGRAPH
	"8F 9D	50B7",	#CJK UNIFIED IDEOGRAPH
	"8F 9E	511F",	#CJK UNIFIED IDEOGRAPH
	"8F 9F	52DD",	#CJK UNIFIED IDEOGRAPH
	"8F A0	5320",	#CJK UNIFIED IDEOGRAPH
	"8F A1	5347",	#CJK UNIFIED IDEOGRAPH
	"8F A2	53EC",	#CJK UNIFIED IDEOGRAPH
	"8F A3	54E8",	#CJK UNIFIED IDEOGRAPH
	"8F A4	5546",	#CJK UNIFIED IDEOGRAPH
	"8F A5	5531",	#CJK UNIFIED IDEOGRAPH
	"8F A6	5617",	#CJK UNIFIED IDEOGRAPH
	"8F A7	5968",	#CJK UNIFIED IDEOGRAPH
	"8F A8	59BE",	#CJK UNIFIED IDEOGRAPH
	"8F A9	5A3C",	#CJK UNIFIED IDEOGRAPH
	"8F AA	5BB5",	#CJK UNIFIED IDEOGRAPH
	"8F AB	5C06",	#CJK UNIFIED IDEOGRAPH
	"8F AC	5C0F",	#CJK UNIFIED IDEOGRAPH
	"8F AD	5C11",	#CJK UNIFIED IDEOGRAPH
	"8F AE	5C1A",	#CJK UNIFIED IDEOGRAPH
	"8F AF	5E84",	#CJK UNIFIED IDEOGRAPH
	"8F B0	5E8A",	#CJK UNIFIED IDEOGRAPH
	"8F B1	5EE0",	#CJK UNIFIED IDEOGRAPH
	"8F B2	5F70",	#CJK UNIFIED IDEOGRAPH
	"8F B3	627F",	#CJK UNIFIED IDEOGRAPH
	"8F B4	6284",	#CJK UNIFIED IDEOGRAPH
	"8F B5	62DB",	#CJK UNIFIED IDEOGRAPH
	"8F B6	638C",	#CJK UNIFIED IDEOGRAPH
	"8F B7	6377",	#CJK UNIFIED IDEOGRAPH
	"8F B8	6607",	#CJK UNIFIED IDEOGRAPH
	"8F B9	660C",	#CJK UNIFIED IDEOGRAPH
	"8F BA	662D",	#CJK UNIFIED IDEOGRAPH
	"8F BB	6676",	#CJK UNIFIED IDEOGRAPH
	"8F BC	677E",	#CJK UNIFIED IDEOGRAPH
	"8F BD	68A2",	#CJK UNIFIED IDEOGRAPH
	"8F BE	6A1F",	#CJK UNIFIED IDEOGRAPH
	"8F BF	6A35",	#CJK UNIFIED IDEOGRAPH
	"8F C0	6CBC",	#CJK UNIFIED IDEOGRAPH
	"8F C1	6D88",	#CJK UNIFIED IDEOGRAPH
	"8F C2	6E09",	#CJK UNIFIED IDEOGRAPH
	"8F C3	6E58",	#CJK UNIFIED IDEOGRAPH
	"8F C4	713C",	#CJK UNIFIED IDEOGRAPH
	"8F C5	7126",	#CJK UNIFIED IDEOGRAPH
	"8F C6	7167",	#CJK UNIFIED IDEOGRAPH
	"8F C7	75C7",	#CJK UNIFIED IDEOGRAPH
	"8F C8	7701",	#CJK UNIFIED IDEOGRAPH
	"8F C9	785D",	#CJK UNIFIED IDEOGRAPH
	"8F CA	7901",	#CJK UNIFIED IDEOGRAPH
	"8F CB	7965",	#CJK UNIFIED IDEOGRAPH
	"8F CC	79F0",	#CJK UNIFIED IDEOGRAPH
	"8F CD	7AE0",	#CJK UNIFIED IDEOGRAPH
	"8F CE	7B11",	#CJK UNIFIED IDEOGRAPH
	"8F CF	7CA7",	#CJK UNIFIED IDEOGRAPH
	"8F D0	7D39",	#CJK UNIFIED IDEOGRAPH
	"8F D1	8096",	#CJK UNIFIED IDEOGRAPH
	"8F D2	83D6",	#CJK UNIFIED IDEOGRAPH
	"8F D3	848B",	#CJK UNIFIED IDEOGRAPH
	"8F D4	8549",	#CJK UNIFIED IDEOGRAPH
	"8F D5	885D",	#CJK UNIFIED IDEOGRAPH
	"8F D6	88F3",	#CJK UNIFIED IDEOGRAPH
	"8F D7	8A1F",	#CJK UNIFIED IDEOGRAPH
	"8F D8	8A3C",	#CJK UNIFIED IDEOGRAPH
	"8F D9	8A54",	#CJK UNIFIED IDEOGRAPH
	"8F DA	8A73",	#CJK UNIFIED IDEOGRAPH
	"8F DB	8C61",	#CJK UNIFIED IDEOGRAPH
	"8F DC	8CDE",	#CJK UNIFIED IDEOGRAPH
	"8F DD	91A4",	#CJK UNIFIED IDEOGRAPH
	"8F DE	9266",	#CJK UNIFIED IDEOGRAPH
	"8F DF	937E",	#CJK UNIFIED IDEOGRAPH
	"8F E0	9418",	#CJK UNIFIED IDEOGRAPH
	"8F E1	969C",	#CJK UNIFIED IDEOGRAPH
	"8F E2	9798",	#CJK UNIFIED IDEOGRAPH
	"8F E3	4E0A",	#CJK UNIFIED IDEOGRAPH
	"8F E4	4E08",	#CJK UNIFIED IDEOGRAPH
	"8F E5	4E1E",	#CJK UNIFIED IDEOGRAPH
	"8F E6	4E57",	#CJK UNIFIED IDEOGRAPH
	"8F E7	5197",	#CJK UNIFIED IDEOGRAPH
	"8F E8	5270",	#CJK UNIFIED IDEOGRAPH
	"8F E9	57CE",	#CJK UNIFIED IDEOGRAPH
	"8F EA	5834",	#CJK UNIFIED IDEOGRAPH
	"8F EB	58CC",	#CJK UNIFIED IDEOGRAPH
	"8F EC	5B22",	#CJK UNIFIED IDEOGRAPH
	"8F ED	5E38",	#CJK UNIFIED IDEOGRAPH
	"8F EE	60C5",	#CJK UNIFIED IDEOGRAPH
	"8F EF	64FE",	#CJK UNIFIED IDEOGRAPH
	"8F F0	6761",	#CJK UNIFIED IDEOGRAPH
	"8F F1	6756",	#CJK UNIFIED IDEOGRAPH
	"8F F2	6D44",	#CJK UNIFIED IDEOGRAPH
	"8F F3	72B6",	#CJK UNIFIED IDEOGRAPH
	"8F F4	7573",	#CJK UNIFIED IDEOGRAPH
	"8F F5	7A63",	#CJK UNIFIED IDEOGRAPH
	"8F F6	84B8",	#CJK UNIFIED IDEOGRAPH
	"8F F7	8B72",	#CJK UNIFIED IDEOGRAPH
	"8F F8	91B8",	#CJK UNIFIED IDEOGRAPH
	"8F F9	9320",	#CJK UNIFIED IDEOGRAPH
	"8F FA	5631",	#CJK UNIFIED IDEOGRAPH
	"8F FB	57F4",	#CJK UNIFIED IDEOGRAPH
	"8F FC	98FE",	#CJK UNIFIED IDEOGRAPH
	"90 40	62ED",	#CJK UNIFIED IDEOGRAPH
	"90 41	690D",	#CJK UNIFIED IDEOGRAPH
	"90 42	6B96",	#CJK UNIFIED IDEOGRAPH
	"90 43	71ED",	#CJK UNIFIED IDEOGRAPH
	"90 44	7E54",	#CJK UNIFIED IDEOGRAPH
	"90 45	8077",	#CJK UNIFIED IDEOGRAPH
	"90 46	8272",	#CJK UNIFIED IDEOGRAPH
	"90 47	89E6",	#CJK UNIFIED IDEOGRAPH
	"90 48	98DF",	#CJK UNIFIED IDEOGRAPH
	"90 49	8755",	#CJK UNIFIED IDEOGRAPH
	"90 4A	8FB1",	#CJK UNIFIED IDEOGRAPH
	"90 4B	5C3B",	#CJK UNIFIED IDEOGRAPH
	"90 4C	4F38",	#CJK UNIFIED IDEOGRAPH
	"90 4D	4FE1",	#CJK UNIFIED IDEOGRAPH
	"90 4E	4FB5",	#CJK UNIFIED IDEOGRAPH
	"90 4F	5507",	#CJK UNIFIED IDEOGRAPH
	"90 50	5A20",	#CJK UNIFIED IDEOGRAPH
	"90 51	5BDD",	#CJK UNIFIED IDEOGRAPH
	"90 52	5BE9",	#CJK UNIFIED IDEOGRAPH
	"90 53	5FC3",	#CJK UNIFIED IDEOGRAPH
	"90 54	614E",	#CJK UNIFIED IDEOGRAPH
	"90 55	632F",	#CJK UNIFIED IDEOGRAPH
	"90 56	65B0",	#CJK UNIFIED IDEOGRAPH
	"90 57	664B",	#CJK UNIFIED IDEOGRAPH
	"90 58	68EE",	#CJK UNIFIED IDEOGRAPH
	"90 59	699B",	#CJK UNIFIED IDEOGRAPH
	"90 5A	6D78",	#CJK UNIFIED IDEOGRAPH
	"90 5B	6DF1",	#CJK UNIFIED IDEOGRAPH
	"90 5C	7533",	#CJK UNIFIED IDEOGRAPH
	"90 5D	75B9",	#CJK UNIFIED IDEOGRAPH
	"90 5E	771F",	#CJK UNIFIED IDEOGRAPH
	"90 5F	795E",	#CJK UNIFIED IDEOGRAPH
	"90 60	79E6",	#CJK UNIFIED IDEOGRAPH
	"90 61	7D33",	#CJK UNIFIED IDEOGRAPH
	"90 62	81E3",	#CJK UNIFIED IDEOGRAPH
	"90 63	82AF",	#CJK UNIFIED IDEOGRAPH
	"90 64	85AA",	#CJK UNIFIED IDEOGRAPH
	"90 65	89AA",	#CJK UNIFIED IDEOGRAPH
	"90 66	8A3A",	#CJK UNIFIED IDEOGRAPH
	"90 67	8EAB",	#CJK UNIFIED IDEOGRAPH
	"90 68	8F9B",	#CJK UNIFIED IDEOGRAPH
	"90 69	9032",	#CJK UNIFIED IDEOGRAPH
	"90 6A	91DD",	#CJK UNIFIED IDEOGRAPH
	"90 6B	9707",	#CJK UNIFIED IDEOGRAPH
	"90 6C	4EBA",	#CJK UNIFIED IDEOGRAPH
	"90 6D	4EC1",	#CJK UNIFIED IDEOGRAPH
	"90 6E	5203",	#CJK UNIFIED IDEOGRAPH
	"90 6F	5875",	#CJK UNIFIED IDEOGRAPH
	"90 70	58EC",	#CJK UNIFIED IDEOGRAPH
	"90 71	5C0B",	#CJK UNIFIED IDEOGRAPH
	"90 72	751A",	#CJK UNIFIED IDEOGRAPH
	"90 73	5C3D",	#CJK UNIFIED IDEOGRAPH
	"90 74	814E",	#CJK UNIFIED IDEOGRAPH
	"90 75	8A0A",	#CJK UNIFIED IDEOGRAPH
	"90 76	8FC5",	#CJK UNIFIED IDEOGRAPH
	"90 77	9663",	#CJK UNIFIED IDEOGRAPH
	"90 78	976D",	#CJK UNIFIED IDEOGRAPH
	"90 79	7B25",	#CJK UNIFIED IDEOGRAPH
	"90 7A	8ACF",	#CJK UNIFIED IDEOGRAPH
	"90 7B	9808",	#CJK UNIFIED IDEOGRAPH
	"90 7C	9162",	#CJK UNIFIED IDEOGRAPH
	"90 7D	56F3",	#CJK UNIFIED IDEOGRAPH
	"90 7E	53A8",	#CJK UNIFIED IDEOGRAPH
	"90 80	9017",	#CJK UNIFIED IDEOGRAPH
	"90 81	5439",	#CJK UNIFIED IDEOGRAPH
	"90 82	5782",	#CJK UNIFIED IDEOGRAPH
	"90 83	5E25",	#CJK UNIFIED IDEOGRAPH
	"90 84	63A8",	#CJK UNIFIED IDEOGRAPH
	"90 85	6C34",	#CJK UNIFIED IDEOGRAPH
	"90 86	708A",	#CJK UNIFIED IDEOGRAPH
	"90 87	7761",	#CJK UNIFIED IDEOGRAPH
	"90 88	7C8B",	#CJK UNIFIED IDEOGRAPH
	"90 89	7FE0",	#CJK UNIFIED IDEOGRAPH
	"90 8A	8870",	#CJK UNIFIED IDEOGRAPH
	"90 8B	9042",	#CJK UNIFIED IDEOGRAPH
	"90 8C	9154",	#CJK UNIFIED IDEOGRAPH
	"90 8D	9310",	#CJK UNIFIED IDEOGRAPH
	"90 8E	9318",	#CJK UNIFIED IDEOGRAPH
	"90 8F	968F",	#CJK UNIFIED IDEOGRAPH
	"90 90	745E",	#CJK UNIFIED IDEOGRAPH
	"90 91	9AC4",	#CJK UNIFIED IDEOGRAPH
	"90 92	5D07",	#CJK UNIFIED IDEOGRAPH
	"90 93	5D69",	#CJK UNIFIED IDEOGRAPH
	"90 94	6570",	#CJK UNIFIED IDEOGRAPH
	"90 95	67A2",	#CJK UNIFIED IDEOGRAPH
	"90 96	8DA8",	#CJK UNIFIED IDEOGRAPH
	"90 97	96DB",	#CJK UNIFIED IDEOGRAPH
	"90 98	636E",	#CJK UNIFIED IDEOGRAPH
	"90 99	6749",	#CJK UNIFIED IDEOGRAPH
	"90 9A	6919",	#CJK UNIFIED IDEOGRAPH
	"90 9B	83C5",	#CJK UNIFIED IDEOGRAPH
	"90 9C	9817",	#CJK UNIFIED IDEOGRAPH
	"90 9D	96C0",	#CJK UNIFIED IDEOGRAPH
	"90 9E	88FE",	#CJK UNIFIED IDEOGRAPH
	"90 9F	6F84",	#CJK UNIFIED IDEOGRAPH
	"90 A0	647A",	#CJK UNIFIED IDEOGRAPH
	"90 A1	5BF8",	#CJK UNIFIED IDEOGRAPH
	"90 A2	4E16",	#CJK UNIFIED IDEOGRAPH
	"90 A3	702C",	#CJK UNIFIED IDEOGRAPH
	"90 A4	755D",	#CJK UNIFIED IDEOGRAPH
	"90 A5	662F",	#CJK UNIFIED IDEOGRAPH
	"90 A6	51C4",	#CJK UNIFIED IDEOGRAPH
	"90 A7	5236",	#CJK UNIFIED IDEOGRAPH
	"90 A8	52E2",	#CJK UNIFIED IDEOGRAPH
	"90 A9	59D3",	#CJK UNIFIED IDEOGRAPH
	"90 AA	5F81",	#CJK UNIFIED IDEOGRAPH
	"90 AB	6027",	#CJK UNIFIED IDEOGRAPH
	"90 AC	6210",	#CJK UNIFIED IDEOGRAPH
	"90 AD	653F",	#CJK UNIFIED IDEOGRAPH
	"90 AE	6574",	#CJK UNIFIED IDEOGRAPH
	"90 AF	661F",	#CJK UNIFIED IDEOGRAPH
	"90 B0	6674",	#CJK UNIFIED IDEOGRAPH
	"90 B1	68F2",	#CJK UNIFIED IDEOGRAPH
	"90 B2	6816",	#CJK UNIFIED IDEOGRAPH
	"90 B3	6B63",	#CJK UNIFIED IDEOGRAPH
	"90 B4	6E05",	#CJK UNIFIED IDEOGRAPH
	"90 B5	7272",	#CJK UNIFIED IDEOGRAPH
	"90 B6	751F",	#CJK UNIFIED IDEOGRAPH
	"90 B7	76DB",	#CJK UNIFIED IDEOGRAPH
	"90 B8	7CBE",	#CJK UNIFIED IDEOGRAPH
	"90 B9	8056",	#CJK UNIFIED IDEOGRAPH
	"90 BA	58F0",	#CJK UNIFIED IDEOGRAPH
	"90 BB	88FD",	#CJK UNIFIED IDEOGRAPH
	"90 BC	897F",	#CJK UNIFIED IDEOGRAPH
	"90 BD	8AA0",	#CJK UNIFIED IDEOGRAPH
	"90 BE	8A93",	#CJK UNIFIED IDEOGRAPH
	"90 BF	8ACB",	#CJK UNIFIED IDEOGRAPH
	"90 C0	901D",	#CJK UNIFIED IDEOGRAPH
	"90 C1	9192",	#CJK UNIFIED IDEOGRAPH
	"90 C2	9752",	#CJK UNIFIED IDEOGRAPH
	"90 C3	9759",	#CJK UNIFIED IDEOGRAPH
	"90 C4	6589",	#CJK UNIFIED IDEOGRAPH
	"90 C5	7A0E",	#CJK UNIFIED IDEOGRAPH
	"90 C6	8106",	#CJK UNIFIED IDEOGRAPH
	"90 C7	96BB",	#CJK UNIFIED IDEOGRAPH
	"90 C8	5E2D",	#CJK UNIFIED IDEOGRAPH
	"90 C9	60DC",	#CJK UNIFIED IDEOGRAPH
	"90 CA	621A",	#CJK UNIFIED IDEOGRAPH
	"90 CB	65A5",	#CJK UNIFIED IDEOGRAPH
	"90 CC	6614",	#CJK UNIFIED IDEOGRAPH
	"90 CD	6790",	#CJK UNIFIED IDEOGRAPH
	"90 CE	77F3",	#CJK UNIFIED IDEOGRAPH
	"90 CF	7A4D",	#CJK UNIFIED IDEOGRAPH
	"90 D0	7C4D",	#CJK UNIFIED IDEOGRAPH
	"90 D1	7E3E",	#CJK UNIFIED IDEOGRAPH
	"90 D2	810A",	#CJK UNIFIED IDEOGRAPH
	"90 D3	8CAC",	#CJK UNIFIED IDEOGRAPH
	"90 D4	8D64",	#CJK UNIFIED IDEOGRAPH
	"90 D5	8DE1",	#CJK UNIFIED IDEOGRAPH
	"90 D6	8E5F",	#CJK UNIFIED IDEOGRAPH
	"90 D7	78A9",	#CJK UNIFIED IDEOGRAPH
	"90 D8	5207",	#CJK UNIFIED IDEOGRAPH
	"90 D9	62D9",	#CJK UNIFIED IDEOGRAPH
	"90 DA	63A5",	#CJK UNIFIED IDEOGRAPH
	"90 DB	6442",	#CJK UNIFIED IDEOGRAPH
	"90 DC	6298",	#CJK UNIFIED IDEOGRAPH
	"90 DD	8A2D",	#CJK UNIFIED IDEOGRAPH
	"90 DE	7A83",	#CJK UNIFIED IDEOGRAPH
	"90 DF	7BC0",	#CJK UNIFIED IDEOGRAPH
	"90 E0	8AAC",	#CJK UNIFIED IDEOGRAPH
	"90 E1	96EA",	#CJK UNIFIED IDEOGRAPH
	"90 E2	7D76",	#CJK UNIFIED IDEOGRAPH
	"90 E3	820C",	#CJK UNIFIED IDEOGRAPH
	"90 E4	8749",	#CJK UNIFIED IDEOGRAPH
	"90 E5	4ED9",	#CJK UNIFIED IDEOGRAPH
	"90 E6	5148",	#CJK UNIFIED IDEOGRAPH
	"90 E7	5343",	#CJK UNIFIED IDEOGRAPH
	"90 E8	5360",	#CJK UNIFIED IDEOGRAPH
	"90 E9	5BA3",	#CJK UNIFIED IDEOGRAPH
	"90 EA	5C02",	#CJK UNIFIED IDEOGRAPH
	"90 EB	5C16",	#CJK UNIFIED IDEOGRAPH
	"90 EC	5DDD",	#CJK UNIFIED IDEOGRAPH
	"90 ED	6226",	#CJK UNIFIED IDEOGRAPH
	"90 EE	6247",	#CJK UNIFIED IDEOGRAPH
	"90 EF	64B0",	#CJK UNIFIED IDEOGRAPH
	"90 F0	6813",	#CJK UNIFIED IDEOGRAPH
	"90 F1	6834",	#CJK UNIFIED IDEOGRAPH
	"90 F2	6CC9",	#CJK UNIFIED IDEOGRAPH
	"90 F3	6D45",	#CJK UNIFIED IDEOGRAPH
	"90 F4	6D17",	#CJK UNIFIED IDEOGRAPH
	"90 F5	67D3",	#CJK UNIFIED IDEOGRAPH
	"90 F6	6F5C",	#CJK UNIFIED IDEOGRAPH
	"90 F7	714E",	#CJK UNIFIED IDEOGRAPH
	"90 F8	717D",	#CJK UNIFIED IDEOGRAPH
	"90 F9	65CB",	#CJK UNIFIED IDEOGRAPH
	"90 FA	7A7F",	#CJK UNIFIED IDEOGRAPH
	"90 FB	7BAD",	#CJK UNIFIED IDEOGRAPH
	"90 FC	7DDA",	#CJK UNIFIED IDEOGRAPH
	"91 40	7E4A",	#CJK UNIFIED IDEOGRAPH
	"91 41	7FA8",	#CJK UNIFIED IDEOGRAPH
	"91 42	817A",	#CJK UNIFIED IDEOGRAPH
	"91 43	821B",	#CJK UNIFIED IDEOGRAPH
	"91 44	8239",	#CJK UNIFIED IDEOGRAPH
	"91 45	85A6",	#CJK UNIFIED IDEOGRAPH
	"91 46	8A6E",	#CJK UNIFIED IDEOGRAPH
	"91 47	8CCE",	#CJK UNIFIED IDEOGRAPH
	"91 48	8DF5",	#CJK UNIFIED IDEOGRAPH
	"91 49	9078",	#CJK UNIFIED IDEOGRAPH
	"91 4A	9077",	#CJK UNIFIED IDEOGRAPH
	"91 4B	92AD",	#CJK UNIFIED IDEOGRAPH
	"91 4C	9291",	#CJK UNIFIED IDEOGRAPH
	"91 4D	9583",	#CJK UNIFIED IDEOGRAPH
	"91 4E	9BAE",	#CJK UNIFIED IDEOGRAPH
	"91 4F	524D",	#CJK UNIFIED IDEOGRAPH
	"91 50	5584",	#CJK UNIFIED IDEOGRAPH
	"91 51	6F38",	#CJK UNIFIED IDEOGRAPH
	"91 52	7136",	#CJK UNIFIED IDEOGRAPH
	"91 53	5168",	#CJK UNIFIED IDEOGRAPH
	"91 54	7985",	#CJK UNIFIED IDEOGRAPH
	"91 55	7E55",	#CJK UNIFIED IDEOGRAPH
	"91 56	81B3",	#CJK UNIFIED IDEOGRAPH
	"91 57	7CCE",	#CJK UNIFIED IDEOGRAPH
	"91 58	564C",	#CJK UNIFIED IDEOGRAPH
	"91 59	5851",	#CJK UNIFIED IDEOGRAPH
	"91 5A	5CA8",	#CJK UNIFIED IDEOGRAPH
	"91 5B	63AA",	#CJK UNIFIED IDEOGRAPH
	"91 5C	66FE",	#CJK UNIFIED IDEOGRAPH
	"91 5D	66FD",	#CJK UNIFIED IDEOGRAPH
	"91 5E	695A",	#CJK UNIFIED IDEOGRAPH
	"91 5F	72D9",	#CJK UNIFIED IDEOGRAPH
	"91 60	758F",	#CJK UNIFIED IDEOGRAPH
	"91 61	758E",	#CJK UNIFIED IDEOGRAPH
	"91 62	790E",	#CJK UNIFIED IDEOGRAPH
	"91 63	7956",	#CJK UNIFIED IDEOGRAPH
	"91 64	79DF",	#CJK UNIFIED IDEOGRAPH
	"91 65	7C97",	#CJK UNIFIED IDEOGRAPH
	"91 66	7D20",	#CJK UNIFIED IDEOGRAPH
	"91 67	7D44",	#CJK UNIFIED IDEOGRAPH
	"91 68	8607",	#CJK UNIFIED IDEOGRAPH
	"91 69	8A34",	#CJK UNIFIED IDEOGRAPH
	"91 6A	963B",	#CJK UNIFIED IDEOGRAPH
	"91 6B	9061",	#CJK UNIFIED IDEOGRAPH
	"91 6C	9F20",	#CJK UNIFIED IDEOGRAPH
	"91 6D	50E7",	#CJK UNIFIED IDEOGRAPH
	"91 6E	5275",	#CJK UNIFIED IDEOGRAPH
	"91 6F	53CC",	#CJK UNIFIED IDEOGRAPH
	"91 70	53E2",	#CJK UNIFIED IDEOGRAPH
	"91 71	5009",	#CJK UNIFIED IDEOGRAPH
	"91 72	55AA",	#CJK UNIFIED IDEOGRAPH
	"91 73	58EE",	#CJK UNIFIED IDEOGRAPH
	"91 74	594F",	#CJK UNIFIED IDEOGRAPH
	"91 75	723D",	#CJK UNIFIED IDEOGRAPH
	"91 76	5B8B",	#CJK UNIFIED IDEOGRAPH
	"91 77	5C64",	#CJK UNIFIED IDEOGRAPH
	"91 78	531D",	#CJK UNIFIED IDEOGRAPH
	"91 79	60E3",	#CJK UNIFIED IDEOGRAPH
	"91 7A	60F3",	#CJK UNIFIED IDEOGRAPH
	"91 7B	635C",	#CJK UNIFIED IDEOGRAPH
	"91 7C	6383",	#CJK UNIFIED IDEOGRAPH
	"91 7D	633F",	#CJK UNIFIED IDEOGRAPH
	"91 7E	63BB",	#CJK UNIFIED IDEOGRAPH
	"91 80	64CD",	#CJK UNIFIED IDEOGRAPH
	"91 81	65E9",	#CJK UNIFIED IDEOGRAPH
	"91 82	66F9",	#CJK UNIFIED IDEOGRAPH
	"91 83	5DE3",	#CJK UNIFIED IDEOGRAPH
	"91 84	69CD",	#CJK UNIFIED IDEOGRAPH
	"91 85	69FD",	#CJK UNIFIED IDEOGRAPH
	"91 86	6F15",	#CJK UNIFIED IDEOGRAPH
	"91 87	71E5",	#CJK UNIFIED IDEOGRAPH
	"91 88	4E89",	#CJK UNIFIED IDEOGRAPH
	"91 89	75E9",	#CJK UNIFIED IDEOGRAPH
	"91 8A	76F8",	#CJK UNIFIED IDEOGRAPH
	"91 8B	7A93",	#CJK UNIFIED IDEOGRAPH
	"91 8C	7CDF",	#CJK UNIFIED IDEOGRAPH
	"91 8D	7DCF",	#CJK UNIFIED IDEOGRAPH
	"91 8E	7D9C",	#CJK UNIFIED IDEOGRAPH
	"91 8F	8061",	#CJK UNIFIED IDEOGRAPH
	"91 90	8349",	#CJK UNIFIED IDEOGRAPH
	"91 91	8358",	#CJK UNIFIED IDEOGRAPH
	"91 92	846C",	#CJK UNIFIED IDEOGRAPH
	"91 93	84BC",	#CJK UNIFIED IDEOGRAPH
	"91 94	85FB",	#CJK UNIFIED IDEOGRAPH
	"91 95	88C5",	#CJK UNIFIED IDEOGRAPH
	"91 96	8D70",	#CJK UNIFIED IDEOGRAPH
	"91 97	9001",	#CJK UNIFIED IDEOGRAPH
	"91 98	906D",	#CJK UNIFIED IDEOGRAPH
	"91 99	9397",	#CJK UNIFIED IDEOGRAPH
	"91 9A	971C",	#CJK UNIFIED IDEOGRAPH
	"91 9B	9A12",	#CJK UNIFIED IDEOGRAPH
	"91 9C	50CF",	#CJK UNIFIED IDEOGRAPH
	"91 9D	5897",	#CJK UNIFIED IDEOGRAPH
	"91 9E	618E",	#CJK UNIFIED IDEOGRAPH
	"91 9F	81D3",	#CJK UNIFIED IDEOGRAPH
	"91 A0	8535",	#CJK UNIFIED IDEOGRAPH
	"91 A1	8D08",	#CJK UNIFIED IDEOGRAPH
	"91 A2	9020",	#CJK UNIFIED IDEOGRAPH
	"91 A3	4FC3",	#CJK UNIFIED IDEOGRAPH
	"91 A4	5074",	#CJK UNIFIED IDEOGRAPH
	"91 A5	5247",	#CJK UNIFIED IDEOGRAPH
	"91 A6	5373",	#CJK UNIFIED IDEOGRAPH
	"91 A7	606F",	#CJK UNIFIED IDEOGRAPH
	"91 A8	6349",	#CJK UNIFIED IDEOGRAPH
	"91 A9	675F",	#CJK UNIFIED IDEOGRAPH
	"91 AA	6E2C",	#CJK UNIFIED IDEOGRAPH
	"91 AB	8DB3",	#CJK UNIFIED IDEOGRAPH
	"91 AC	901F",	#CJK UNIFIED IDEOGRAPH
	"91 AD	4FD7",	#CJK UNIFIED IDEOGRAPH
	"91 AE	5C5E",	#CJK UNIFIED IDEOGRAPH
	"91 AF	8CCA",	#CJK UNIFIED IDEOGRAPH
	"91 B0	65CF",	#CJK UNIFIED IDEOGRAPH
	"91 B1	7D9A",	#CJK UNIFIED IDEOGRAPH
	"91 B2	5352",	#CJK UNIFIED IDEOGRAPH
	"91 B3	8896",	#CJK UNIFIED IDEOGRAPH
	"91 B4	5176",	#CJK UNIFIED IDEOGRAPH
	"91 B5	63C3",	#CJK UNIFIED IDEOGRAPH
	"91 B6	5B58",	#CJK UNIFIED IDEOGRAPH
	"91 B7	5B6B",	#CJK UNIFIED IDEOGRAPH
	"91 B8	5C0A",	#CJK UNIFIED IDEOGRAPH
	"91 B9	640D",	#CJK UNIFIED IDEOGRAPH
	"91 BA	6751",	#CJK UNIFIED IDEOGRAPH
	"91 BB	905C",	#CJK UNIFIED IDEOGRAPH
	"91 BC	4ED6",	#CJK UNIFIED IDEOGRAPH
	"91 BD	591A",	#CJK UNIFIED IDEOGRAPH
	"91 BE	592A",	#CJK UNIFIED IDEOGRAPH
	"91 BF	6C70",	#CJK UNIFIED IDEOGRAPH
	"91 C0	8A51",	#CJK UNIFIED IDEOGRAPH
	"91 C1	553E",	#CJK UNIFIED IDEOGRAPH
	"91 C2	5815",	#CJK UNIFIED IDEOGRAPH
	"91 C3	59A5",	#CJK UNIFIED IDEOGRAPH
	"91 C4	60F0",	#CJK UNIFIED IDEOGRAPH
	"91 C5	6253",	#CJK UNIFIED IDEOGRAPH
	"91 C6	67C1",	#CJK UNIFIED IDEOGRAPH
	"91 C7	8235",	#CJK UNIFIED IDEOGRAPH
	"91 C8	6955",	#CJK UNIFIED IDEOGRAPH
	"91 C9	9640",	#CJK UNIFIED IDEOGRAPH
	"91 CA	99C4",	#CJK UNIFIED IDEOGRAPH
	"91 CB	9A28",	#CJK UNIFIED IDEOGRAPH
	"91 CC	4F53",	#CJK UNIFIED IDEOGRAPH
	"91 CD	5806",	#CJK UNIFIED IDEOGRAPH
	"91 CE	5BFE",	#CJK UNIFIED IDEOGRAPH
	"91 CF	8010",	#CJK UNIFIED IDEOGRAPH
	"91 D0	5CB1",	#CJK UNIFIED IDEOGRAPH
	"91 D1	5E2F",	#CJK UNIFIED IDEOGRAPH
	"91 D2	5F85",	#CJK UNIFIED IDEOGRAPH
	"91 D3	6020",	#CJK UNIFIED IDEOGRAPH
	"91 D4	614B",	#CJK UNIFIED IDEOGRAPH
	"91 D5	6234",	#CJK UNIFIED IDEOGRAPH
	"91 D6	66FF",	#CJK UNIFIED IDEOGRAPH
	"91 D7	6CF0",	#CJK UNIFIED IDEOGRAPH
	"91 D8	6EDE",	#CJK UNIFIED IDEOGRAPH
	"91 D9	80CE",	#CJK UNIFIED IDEOGRAPH
	"91 DA	817F",	#CJK UNIFIED IDEOGRAPH
	"91 DB	82D4",	#CJK UNIFIED IDEOGRAPH
	"91 DC	888B",	#CJK UNIFIED IDEOGRAPH
	"91 DD	8CB8",	#CJK UNIFIED IDEOGRAPH
	"91 DE	9000",	#CJK UNIFIED IDEOGRAPH
	"91 DF	902E",	#CJK UNIFIED IDEOGRAPH
	"91 E0	968A",	#CJK UNIFIED IDEOGRAPH
	"91 E1	9EDB",	#CJK UNIFIED IDEOGRAPH
	"91 E2	9BDB",	#CJK UNIFIED IDEOGRAPH
	"91 E3	4EE3",	#CJK UNIFIED IDEOGRAPH
	"91 E4	53F0",	#CJK UNIFIED IDEOGRAPH
	"91 E5	5927",	#CJK UNIFIED IDEOGRAPH
	"91 E6	7B2C",	#CJK UNIFIED IDEOGRAPH
	"91 E7	918D",	#CJK UNIFIED IDEOGRAPH
	"91 E8	984C",	#CJK UNIFIED IDEOGRAPH
	"91 E9	9DF9",	#CJK UNIFIED IDEOGRAPH
	"91 EA	6EDD",	#CJK UNIFIED IDEOGRAPH
	"91 EB	7027",	#CJK UNIFIED IDEOGRAPH
	"91 EC	5353",	#CJK UNIFIED IDEOGRAPH
	"91 ED	5544",	#CJK UNIFIED IDEOGRAPH
	"91 EE	5B85",	#CJK UNIFIED IDEOGRAPH
	"91 EF	6258",	#CJK UNIFIED IDEOGRAPH
	"91 F0	629E",	#CJK UNIFIED IDEOGRAPH
	"91 F1	62D3",	#CJK UNIFIED IDEOGRAPH
	"91 F2	6CA2",	#CJK UNIFIED IDEOGRAPH
	"91 F3	6FEF",	#CJK UNIFIED IDEOGRAPH
	"91 F4	7422",	#CJK UNIFIED IDEOGRAPH
	"91 F5	8A17",	#CJK UNIFIED IDEOGRAPH
	"91 F6	9438",	#CJK UNIFIED IDEOGRAPH
	"91 F7	6FC1",	#CJK UNIFIED IDEOGRAPH
	"91 F8	8AFE",	#CJK UNIFIED IDEOGRAPH
	"91 F9	8338",	#CJK UNIFIED IDEOGRAPH
	"91 FA	51E7",	#CJK UNIFIED IDEOGRAPH
	"91 FB	86F8",	#CJK UNIFIED IDEOGRAPH
	"91 FC	53EA",	#CJK UNIFIED IDEOGRAPH
	"92 40	53E9",	#CJK UNIFIED IDEOGRAPH
	"92 41	4F46",	#CJK UNIFIED IDEOGRAPH
	"92 42	9054",	#CJK UNIFIED IDEOGRAPH
	"92 43	8FB0",	#CJK UNIFIED IDEOGRAPH
	"92 44	596A",	#CJK UNIFIED IDEOGRAPH
	"92 45	8131",	#CJK UNIFIED IDEOGRAPH
	"92 46	5DFD",	#CJK UNIFIED IDEOGRAPH
	"92 47	7AEA",	#CJK UNIFIED IDEOGRAPH
	"92 48	8FBF",	#CJK UNIFIED IDEOGRAPH
	"92 49	68DA",	#CJK UNIFIED IDEOGRAPH
	"92 4A	8C37",	#CJK UNIFIED IDEOGRAPH
	"92 4B	72F8",	#CJK UNIFIED IDEOGRAPH
	"92 4C	9C48",	#CJK UNIFIED IDEOGRAPH
	"92 4D	6A3D",	#CJK UNIFIED IDEOGRAPH
	"92 4E	8AB0",	#CJK UNIFIED IDEOGRAPH
	"92 4F	4E39",	#CJK UNIFIED IDEOGRAPH
	"92 50	5358",	#CJK UNIFIED IDEOGRAPH
	"92 51	5606",	#CJK UNIFIED IDEOGRAPH
	"92 52	5766",	#CJK UNIFIED IDEOGRAPH
	"92 53	62C5",	#CJK UNIFIED IDEOGRAPH
	"92 54	63A2",	#CJK UNIFIED IDEOGRAPH
	"92 55	65E6",	#CJK UNIFIED IDEOGRAPH
	"92 56	6B4E",	#CJK UNIFIED IDEOGRAPH
	"92 57	6DE1",	#CJK UNIFIED IDEOGRAPH
	"92 58	6E5B",	#CJK UNIFIED IDEOGRAPH
	"92 59	70AD",	#CJK UNIFIED IDEOGRAPH
	"92 5A	77ED",	#CJK UNIFIED IDEOGRAPH
	"92 5B	7AEF",	#CJK UNIFIED IDEOGRAPH
	"92 5C	7BAA",	#CJK UNIFIED IDEOGRAPH
	"92 5D	7DBB",	#CJK UNIFIED IDEOGRAPH
	"92 5E	803D",	#CJK UNIFIED IDEOGRAPH
	"92 5F	80C6",	#CJK UNIFIED IDEOGRAPH
	"92 60	86CB",	#CJK UNIFIED IDEOGRAPH
	"92 61	8A95",	#CJK UNIFIED IDEOGRAPH
	"92 62	935B",	#CJK UNIFIED IDEOGRAPH
	"92 63	56E3",	#CJK UNIFIED IDEOGRAPH
	"92 64	58C7",	#CJK UNIFIED IDEOGRAPH
	"92 65	5F3E",	#CJK UNIFIED IDEOGRAPH
	"92 66	65AD",	#CJK UNIFIED IDEOGRAPH
	"92 67	6696",	#CJK UNIFIED IDEOGRAPH
	"92 68	6A80",	#CJK UNIFIED IDEOGRAPH
	"92 69	6BB5",	#CJK UNIFIED IDEOGRAPH
	"92 6A	7537",	#CJK UNIFIED IDEOGRAPH
	"92 6B	8AC7",	#CJK UNIFIED IDEOGRAPH
	"92 6C	5024",	#CJK UNIFIED IDEOGRAPH
	"92 6D	77E5",	#CJK UNIFIED IDEOGRAPH
	"92 6E	5730",	#CJK UNIFIED IDEOGRAPH
	"92 6F	5F1B",	#CJK UNIFIED IDEOGRAPH
	"92 70	6065",	#CJK UNIFIED IDEOGRAPH
	"92 71	667A",	#CJK UNIFIED IDEOGRAPH
	"92 72	6C60",	#CJK UNIFIED IDEOGRAPH
	"92 73	75F4",	#CJK UNIFIED IDEOGRAPH
	"92 74	7A1A",	#CJK UNIFIED IDEOGRAPH
	"92 75	7F6E",	#CJK UNIFIED IDEOGRAPH
	"92 76	81F4",	#CJK UNIFIED IDEOGRAPH
	"92 77	8718",	#CJK UNIFIED IDEOGRAPH
	"92 78	9045",	#CJK UNIFIED IDEOGRAPH
	"92 79	99B3",	#CJK UNIFIED IDEOGRAPH
	"92 7A	7BC9",	#CJK UNIFIED IDEOGRAPH
	"92 7B	755C",	#CJK UNIFIED IDEOGRAPH
	"92 7C	7AF9",	#CJK UNIFIED IDEOGRAPH
	"92 7D	7B51",	#CJK UNIFIED IDEOGRAPH
	"92 7E	84C4",	#CJK UNIFIED IDEOGRAPH
	"92 80	9010",	#CJK UNIFIED IDEOGRAPH
	"92 81	79E9",	#CJK UNIFIED IDEOGRAPH
	"92 82	7A92",	#CJK UNIFIED IDEOGRAPH
	"92 83	8336",	#CJK UNIFIED IDEOGRAPH
	"92 84	5AE1",	#CJK UNIFIED IDEOGRAPH
	"92 85	7740",	#CJK UNIFIED IDEOGRAPH
	"92 86	4E2D",	#CJK UNIFIED IDEOGRAPH
	"92 87	4EF2",	#CJK UNIFIED IDEOGRAPH
	"92 88	5B99",	#CJK UNIFIED IDEOGRAPH
	"92 89	5FE0",	#CJK UNIFIED IDEOGRAPH
	"92 8A	62BD",	#CJK UNIFIED IDEOGRAPH
	"92 8B	663C",	#CJK UNIFIED IDEOGRAPH
	"92 8C	67F1",	#CJK UNIFIED IDEOGRAPH
	"92 8D	6CE8",	#CJK UNIFIED IDEOGRAPH
	"92 8E	866B",	#CJK UNIFIED IDEOGRAPH
	"92 8F	8877",	#CJK UNIFIED IDEOGRAPH
	"92 90	8A3B",	#CJK UNIFIED IDEOGRAPH
	"92 91	914E",	#CJK UNIFIED IDEOGRAPH
	"92 92	92F3",	#CJK UNIFIED IDEOGRAPH
	"92 93	99D0",	#CJK UNIFIED IDEOGRAPH
	"92 94	6A17",	#CJK UNIFIED IDEOGRAPH
	"92 95	7026",	#CJK UNIFIED IDEOGRAPH
	"92 96	732A",	#CJK UNIFIED IDEOGRAPH
	"92 97	82E7",	#CJK UNIFIED IDEOGRAPH
	"92 98	8457",	#CJK UNIFIED IDEOGRAPH
	"92 99	8CAF",	#CJK UNIFIED IDEOGRAPH
	"92 9A	4E01",	#CJK UNIFIED IDEOGRAPH
	"92 9B	5146",	#CJK UNIFIED IDEOGRAPH
	"92 9C	51CB",	#CJK UNIFIED IDEOGRAPH
	"92 9D	558B",	#CJK UNIFIED IDEOGRAPH
	"92 9E	5BF5",	#CJK UNIFIED IDEOGRAPH
	"92 9F	5E16",	#CJK UNIFIED IDEOGRAPH
	"92 A0	5E33",	#CJK UNIFIED IDEOGRAPH
	"92 A1	5E81",	#CJK UNIFIED IDEOGRAPH
	"92 A2	5F14",	#CJK UNIFIED IDEOGRAPH
	"92 A3	5F35",	#CJK UNIFIED IDEOGRAPH
	"92 A4	5F6B",	#CJK UNIFIED IDEOGRAPH
	"92 A5	5FB4",	#CJK UNIFIED IDEOGRAPH
	"92 A6	61F2",	#CJK UNIFIED IDEOGRAPH
	"92 A7	6311",	#CJK UNIFIED IDEOGRAPH
	"92 A8	66A2",	#CJK UNIFIED IDEOGRAPH
	"92 A9	671D",	#CJK UNIFIED IDEOGRAPH
	"92 AA	6F6E",	#CJK UNIFIED IDEOGRAPH
	"92 AB	7252",	#CJK UNIFIED IDEOGRAPH
	"92 AC	753A",	#CJK UNIFIED IDEOGRAPH
	"92 AD	773A",	#CJK UNIFIED IDEOGRAPH
	"92 AE	8074",	#CJK UNIFIED IDEOGRAPH
	"92 AF	8139",	#CJK UNIFIED IDEOGRAPH
	"92 B0	8178",	#CJK UNIFIED IDEOGRAPH
	"92 B1	8776",	#CJK UNIFIED IDEOGRAPH
	"92 B2	8ABF",	#CJK UNIFIED IDEOGRAPH
	"92 B3	8ADC",	#CJK UNIFIED IDEOGRAPH
	"92 B4	8D85",	#CJK UNIFIED IDEOGRAPH
	"92 B5	8DF3",	#CJK UNIFIED IDEOGRAPH
	"92 B6	929A",	#CJK UNIFIED IDEOGRAPH
	"92 B7	9577",	#CJK UNIFIED IDEOGRAPH
	"92 B8	9802",	#CJK UNIFIED IDEOGRAPH
	"92 B9	9CE5",	#CJK UNIFIED IDEOGRAPH
	"92 BA	52C5",	#CJK UNIFIED IDEOGRAPH
	"92 BB	6357",	#CJK UNIFIED IDEOGRAPH
	"92 BC	76F4",	#CJK UNIFIED IDEOGRAPH
	"92 BD	6715",	#CJK UNIFIED IDEOGRAPH
	"92 BE	6C88",	#CJK UNIFIED IDEOGRAPH
	"92 BF	73CD",	#CJK UNIFIED IDEOGRAPH
	"92 C0	8CC3",	#CJK UNIFIED IDEOGRAPH
	"92 C1	93AE",	#CJK UNIFIED IDEOGRAPH
	"92 C2	9673",	#CJK UNIFIED IDEOGRAPH
	"92 C3	6D25",	#CJK UNIFIED IDEOGRAPH
	"92 C4	589C",	#CJK UNIFIED IDEOGRAPH
	"92 C5	690E",	#CJK UNIFIED IDEOGRAPH
	"92 C6	69CC",	#CJK UNIFIED IDEOGRAPH
	"92 C7	8FFD",	#CJK UNIFIED IDEOGRAPH
	"92 C8	939A",	#CJK UNIFIED IDEOGRAPH
	"92 C9	75DB",	#CJK UNIFIED IDEOGRAPH
	"92 CA	901A",	#CJK UNIFIED IDEOGRAPH
	"92 CB	585A",	#CJK UNIFIED IDEOGRAPH
	"92 CC	6802",	#CJK UNIFIED IDEOGRAPH
	"92 CD	63B4",	#CJK UNIFIED IDEOGRAPH
	"92 CE	69FB",	#CJK UNIFIED IDEOGRAPH
	"92 CF	4F43",	#CJK UNIFIED IDEOGRAPH
	"92 D0	6F2C",	#CJK UNIFIED IDEOGRAPH
	"92 D1	67D8",	#CJK UNIFIED IDEOGRAPH
	"92 D2	8FBB",	#CJK UNIFIED IDEOGRAPH
	"92 D3	8526",	#CJK UNIFIED IDEOGRAPH
	"92 D4	7DB4",	#CJK UNIFIED IDEOGRAPH
	"92 D5	9354",	#CJK UNIFIED IDEOGRAPH
	"92 D6	693F",	#CJK UNIFIED IDEOGRAPH
	"92 D7	6F70",	#CJK UNIFIED IDEOGRAPH
	"92 D8	576A",	#CJK UNIFIED IDEOGRAPH
	"92 D9	58F7",	#CJK UNIFIED IDEOGRAPH
	"92 DA	5B2C",	#CJK UNIFIED IDEOGRAPH
	"92 DB	7D2C",	#CJK UNIFIED IDEOGRAPH
	"92 DC	722A",	#CJK UNIFIED IDEOGRAPH
	"92 DD	540A",	#CJK UNIFIED IDEOGRAPH
	"92 DE	91E3",	#CJK UNIFIED IDEOGRAPH
	"92 DF	9DB4",	#CJK UNIFIED IDEOGRAPH
	"92 E0	4EAD",	#CJK UNIFIED IDEOGRAPH
	"92 E1	4F4E",	#CJK UNIFIED IDEOGRAPH
	"92 E2	505C",	#CJK UNIFIED IDEOGRAPH
	"92 E3	5075",	#CJK UNIFIED IDEOGRAPH
	"92 E4	5243",	#CJK UNIFIED IDEOGRAPH
	"92 E5	8C9E",	#CJK UNIFIED IDEOGRAPH
	"92 E6	5448",	#CJK UNIFIED IDEOGRAPH
	"92 E7	5824",	#CJK UNIFIED IDEOGRAPH
	"92 E8	5B9A",	#CJK UNIFIED IDEOGRAPH
	"92 E9	5E1D",	#CJK UNIFIED IDEOGRAPH
	"92 EA	5E95",	#CJK UNIFIED IDEOGRAPH
	"92 EB	5EAD",	#CJK UNIFIED IDEOGRAPH
	"92 EC	5EF7",	#CJK UNIFIED IDEOGRAPH
	"92 ED	5F1F",	#CJK UNIFIED IDEOGRAPH
	"92 EE	608C",	#CJK UNIFIED IDEOGRAPH
	"92 EF	62B5",	#CJK UNIFIED IDEOGRAPH
	"92 F0	633A",	#CJK UNIFIED IDEOGRAPH
	"92 F1	63D0",	#CJK UNIFIED IDEOGRAPH
	"92 F2	68AF",	#CJK UNIFIED IDEOGRAPH
	"92 F3	6C40",	#CJK UNIFIED IDEOGRAPH
	"92 F4	7887",	#CJK UNIFIED IDEOGRAPH
	"92 F5	798E",	#CJK UNIFIED IDEOGRAPH
	"92 F6	7A0B",	#CJK UNIFIED IDEOGRAPH
	"92 F7	7DE0",	#CJK UNIFIED IDEOGRAPH
	"92 F8	8247",	#CJK UNIFIED IDEOGRAPH
	"92 F9	8A02",	#CJK UNIFIED IDEOGRAPH
	"92 FA	8AE6",	#CJK UNIFIED IDEOGRAPH
	"92 FB	8E44",	#CJK UNIFIED IDEOGRAPH
	"92 FC	9013",	#CJK UNIFIED IDEOGRAPH
	"93 40	90B8",	#CJK UNIFIED IDEOGRAPH
	"93 41	912D",	#CJK UNIFIED IDEOGRAPH
	"93 42	91D8",	#CJK UNIFIED IDEOGRAPH
	"93 43	9F0E",	#CJK UNIFIED IDEOGRAPH
	"93 44	6CE5",	#CJK UNIFIED IDEOGRAPH
	"93 45	6458",	#CJK UNIFIED IDEOGRAPH
	"93 46	64E2",	#CJK UNIFIED IDEOGRAPH
	"93 47	6575",	#CJK UNIFIED IDEOGRAPH
	"93 48	6EF4",	#CJK UNIFIED IDEOGRAPH
	"93 49	7684",	#CJK UNIFIED IDEOGRAPH
	"93 4A	7B1B",	#CJK UNIFIED IDEOGRAPH
	"93 4B	9069",	#CJK UNIFIED IDEOGRAPH
	"93 4C	93D1",	#CJK UNIFIED IDEOGRAPH
	"93 4D	6EBA",	#CJK UNIFIED IDEOGRAPH
	"93 4E	54F2",	#CJK UNIFIED IDEOGRAPH
	"93 4F	5FB9",	#CJK UNIFIED IDEOGRAPH
	"93 50	64A4",	#CJK UNIFIED IDEOGRAPH
	"93 51	8F4D",	#CJK UNIFIED IDEOGRAPH
	"93 52	8FED",	#CJK UNIFIED IDEOGRAPH
	"93 53	9244",	#CJK UNIFIED IDEOGRAPH
	"93 54	5178",	#CJK UNIFIED IDEOGRAPH
	"93 55	586B",	#CJK UNIFIED IDEOGRAPH
	"93 56	5929",	#CJK UNIFIED IDEOGRAPH
	"93 57	5C55",	#CJK UNIFIED IDEOGRAPH
	"93 58	5E97",	#CJK UNIFIED IDEOGRAPH
	"93 59	6DFB",	#CJK UNIFIED IDEOGRAPH
	"93 5A	7E8F",	#CJK UNIFIED IDEOGRAPH
	"93 5B	751C",	#CJK UNIFIED IDEOGRAPH
	"93 5C	8CBC",	#CJK UNIFIED IDEOGRAPH
	"93 5D	8EE2",	#CJK UNIFIED IDEOGRAPH
	"93 5E	985B",	#CJK UNIFIED IDEOGRAPH
	"93 5F	70B9",	#CJK UNIFIED IDEOGRAPH
	"93 60	4F1D",	#CJK UNIFIED IDEOGRAPH
	"93 61	6BBF",	#CJK UNIFIED IDEOGRAPH
	"93 62	6FB1",	#CJK UNIFIED IDEOGRAPH
	"93 63	7530",	#CJK UNIFIED IDEOGRAPH
	"93 64	96FB",	#CJK UNIFIED IDEOGRAPH
	"93 65	514E",	#CJK UNIFIED IDEOGRAPH
	"93 66	5410",	#CJK UNIFIED IDEOGRAPH
	"93 67	5835",	#CJK UNIFIED IDEOGRAPH
	"93 68	5857",	#CJK UNIFIED IDEOGRAPH
	"93 69	59AC",	#CJK UNIFIED IDEOGRAPH
	"93 6A	5C60",	#CJK UNIFIED IDEOGRAPH
	"93 6B	5F92",	#CJK UNIFIED IDEOGRAPH
	"93 6C	6597",	#CJK UNIFIED IDEOGRAPH
	"93 6D	675C",	#CJK UNIFIED IDEOGRAPH
	"93 6E	6E21",	#CJK UNIFIED IDEOGRAPH
	"93 6F	767B",	#CJK UNIFIED IDEOGRAPH
	"93 70	83DF",	#CJK UNIFIED IDEOGRAPH
	"93 71	8CED",	#CJK UNIFIED IDEOGRAPH
	"93 72	9014",	#CJK UNIFIED IDEOGRAPH
	"93 73	90FD",	#CJK UNIFIED IDEOGRAPH
	"93 74	934D",	#CJK UNIFIED IDEOGRAPH
	"93 75	7825",	#CJK UNIFIED IDEOGRAPH
	"93 76	783A",	#CJK UNIFIED IDEOGRAPH
	"93 77	52AA",	#CJK UNIFIED IDEOGRAPH
	"93 78	5EA6",	#CJK UNIFIED IDEOGRAPH
	"93 79	571F",	#CJK UNIFIED IDEOGRAPH
	"93 7A	5974",	#CJK UNIFIED IDEOGRAPH
	"93 7B	6012",	#CJK UNIFIED IDEOGRAPH
	"93 7C	5012",	#CJK UNIFIED IDEOGRAPH
	"93 7D	515A",	#CJK UNIFIED IDEOGRAPH
	"93 7E	51AC",	#CJK UNIFIED IDEOGRAPH
	"93 80	51CD",	#CJK UNIFIED IDEOGRAPH
	"93 81	5200",	#CJK UNIFIED IDEOGRAPH
	"93 82	5510",	#CJK UNIFIED IDEOGRAPH
	"93 83	5854",	#CJK UNIFIED IDEOGRAPH
	"93 84	5858",	#CJK UNIFIED IDEOGRAPH
	"93 85	5957",	#CJK UNIFIED IDEOGRAPH
	"93 86	5B95",	#CJK UNIFIED IDEOGRAPH
	"93 87	5CF6",	#CJK UNIFIED IDEOGRAPH
	"93 88	5D8B",	#CJK UNIFIED IDEOGRAPH
	"93 89	60BC",	#CJK UNIFIED IDEOGRAPH
	"93 8A	6295",	#CJK UNIFIED IDEOGRAPH
	"93 8B	642D",	#CJK UNIFIED IDEOGRAPH
	"93 8C	6771",	#CJK UNIFIED IDEOGRAPH
	"93 8D	6843",	#CJK UNIFIED IDEOGRAPH
	"93 8E	68BC",	#CJK UNIFIED IDEOGRAPH
	"93 8F	68DF",	#CJK UNIFIED IDEOGRAPH
	"93 90	76D7",	#CJK UNIFIED IDEOGRAPH
	"93 91	6DD8",	#CJK UNIFIED IDEOGRAPH
	"93 92	6E6F",	#CJK UNIFIED IDEOGRAPH
	"93 93	6D9B",	#CJK UNIFIED IDEOGRAPH
	"93 94	706F",	#CJK UNIFIED IDEOGRAPH
	"93 95	71C8",	#CJK UNIFIED IDEOGRAPH
	"93 96	5F53",	#CJK UNIFIED IDEOGRAPH
	"93 97	75D8",	#CJK UNIFIED IDEOGRAPH
	"93 98	7977",	#CJK UNIFIED IDEOGRAPH
	"93 99	7B49",	#CJK UNIFIED IDEOGRAPH
	"93 9A	7B54",	#CJK UNIFIED IDEOGRAPH
	"93 9B	7B52",	#CJK UNIFIED IDEOGRAPH
	"93 9C	7CD6",	#CJK UNIFIED IDEOGRAPH
	"93 9D	7D71",	#CJK UNIFIED IDEOGRAPH
	"93 9E	5230",	#CJK UNIFIED IDEOGRAPH
	"93 9F	8463",	#CJK UNIFIED IDEOGRAPH
	"93 A0	8569",	#CJK UNIFIED IDEOGRAPH
	"93 A1	85E4",	#CJK UNIFIED IDEOGRAPH
	"93 A2	8A0E",	#CJK UNIFIED IDEOGRAPH
	"93 A3	8B04",	#CJK UNIFIED IDEOGRAPH
	"93 A4	8C46",	#CJK UNIFIED IDEOGRAPH
	"93 A5	8E0F",	#CJK UNIFIED IDEOGRAPH
	"93 A6	9003",	#CJK UNIFIED IDEOGRAPH
	"93 A7	900F",	#CJK UNIFIED IDEOGRAPH
	"93 A8	9419",	#CJK UNIFIED IDEOGRAPH
	"93 A9	9676",	#CJK UNIFIED IDEOGRAPH
	"93 AA	982D",	#CJK UNIFIED IDEOGRAPH
	"93 AB	9A30",	#CJK UNIFIED IDEOGRAPH
	"93 AC	95D8",	#CJK UNIFIED IDEOGRAPH
	"93 AD	50CD",	#CJK UNIFIED IDEOGRAPH
	"93 AE	52D5",	#CJK UNIFIED IDEOGRAPH
	"93 AF	540C",	#CJK UNIFIED IDEOGRAPH
	"93 B0	5802",	#CJK UNIFIED IDEOGRAPH
	"93 B1	5C0E",	#CJK UNIFIED IDEOGRAPH
	"93 B2	61A7",	#CJK UNIFIED IDEOGRAPH
	"93 B3	649E",	#CJK UNIFIED IDEOGRAPH
	"93 B4	6D1E",	#CJK UNIFIED IDEOGRAPH
	"93 B5	77B3",	#CJK UNIFIED IDEOGRAPH
	"93 B6	7AE5",	#CJK UNIFIED IDEOGRAPH
	"93 B7	80F4",	#CJK UNIFIED IDEOGRAPH
	"93 B8	8404",	#CJK UNIFIED IDEOGRAPH
	"93 B9	9053",	#CJK UNIFIED IDEOGRAPH
	"93 BA	9285",	#CJK UNIFIED IDEOGRAPH
	"93 BB	5CE0",	#CJK UNIFIED IDEOGRAPH
	"93 BC	9D07",	#CJK UNIFIED IDEOGRAPH
	"93 BD	533F",	#CJK UNIFIED IDEOGRAPH
	"93 BE	5F97",	#CJK UNIFIED IDEOGRAPH
	"93 BF	5FB3",	#CJK UNIFIED IDEOGRAPH
	"93 C0	6D9C",	#CJK UNIFIED IDEOGRAPH
	"93 C1	7279",	#CJK UNIFIED IDEOGRAPH
	"93 C2	7763",	#CJK UNIFIED IDEOGRAPH
	"93 C3	79BF",	#CJK UNIFIED IDEOGRAPH
	"93 C4	7BE4",	#CJK UNIFIED IDEOGRAPH
	"93 C5	6BD2",	#CJK UNIFIED IDEOGRAPH
	"93 C6	72EC",	#CJK UNIFIED IDEOGRAPH
	"93 C7	8AAD",	#CJK UNIFIED IDEOGRAPH
	"93 C8	6803",	#CJK UNIFIED IDEOGRAPH
	"93 C9	6A61",	#CJK UNIFIED IDEOGRAPH
	"93 CA	51F8",	#CJK UNIFIED IDEOGRAPH
	"93 CB	7A81",	#CJK UNIFIED IDEOGRAPH
	"93 CC	6934",	#CJK UNIFIED IDEOGRAPH
	"93 CD	5C4A",	#CJK UNIFIED IDEOGRAPH
	"93 CE	9CF6",	#CJK UNIFIED IDEOGRAPH
	"93 CF	82EB",	#CJK UNIFIED IDEOGRAPH
	"93 D0	5BC5",	#CJK UNIFIED IDEOGRAPH
	"93 D1	9149",	#CJK UNIFIED IDEOGRAPH
	"93 D2	701E",	#CJK UNIFIED IDEOGRAPH
	"93 D3	5678",	#CJK UNIFIED IDEOGRAPH
	"93 D4	5C6F",	#CJK UNIFIED IDEOGRAPH
	"93 D5	60C7",	#CJK UNIFIED IDEOGRAPH
	"93 D6	6566",	#CJK UNIFIED IDEOGRAPH
	"93 D7	6C8C",	#CJK UNIFIED IDEOGRAPH
	"93 D8	8C5A",	#CJK UNIFIED IDEOGRAPH
	"93 D9	9041",	#CJK UNIFIED IDEOGRAPH
	"93 DA	9813",	#CJK UNIFIED IDEOGRAPH
	"93 DB	5451",	#CJK UNIFIED IDEOGRAPH
	"93 DC	66C7",	#CJK UNIFIED IDEOGRAPH
	"93 DD	920D",	#CJK UNIFIED IDEOGRAPH
	"93 DE	5948",	#CJK UNIFIED IDEOGRAPH
	"93 DF	90A3",	#CJK UNIFIED IDEOGRAPH
	"93 E0	5185",	#CJK UNIFIED IDEOGRAPH
	"93 E1	4E4D",	#CJK UNIFIED IDEOGRAPH
	"93 E2	51EA",	#CJK UNIFIED IDEOGRAPH
	"93 E3	8599",	#CJK UNIFIED IDEOGRAPH
	"93 E4	8B0E",	#CJK UNIFIED IDEOGRAPH
	"93 E5	7058",	#CJK UNIFIED IDEOGRAPH
	"93 E6	637A",	#CJK UNIFIED IDEOGRAPH
	"93 E7	934B",	#CJK UNIFIED IDEOGRAPH
	"93 E8	6962",	#CJK UNIFIED IDEOGRAPH
	"93 E9	99B4",	#CJK UNIFIED IDEOGRAPH
	"93 EA	7E04",	#CJK UNIFIED IDEOGRAPH
	"93 EB	7577",	#CJK UNIFIED IDEOGRAPH
	"93 EC	5357",	#CJK UNIFIED IDEOGRAPH
	"93 ED	6960",	#CJK UNIFIED IDEOGRAPH
	"93 EE	8EDF",	#CJK UNIFIED IDEOGRAPH
	"93 EF	96E3",	#CJK UNIFIED IDEOGRAPH
	"93 F0	6C5D",	#CJK UNIFIED IDEOGRAPH
	"93 F1	4E8C",	#CJK UNIFIED IDEOGRAPH
	"93 F2	5C3C",	#CJK UNIFIED IDEOGRAPH
	"93 F3	5F10",	#CJK UNIFIED IDEOGRAPH
	"93 F4	8FE9",	#CJK UNIFIED IDEOGRAPH
	"93 F5	5302",	#CJK UNIFIED IDEOGRAPH
	"93 F6	8CD1",	#CJK UNIFIED IDEOGRAPH
	"93 F7	8089",	#CJK UNIFIED IDEOGRAPH
	"93 F8	8679",	#CJK UNIFIED IDEOGRAPH
	"93 F9	5EFF",	#CJK UNIFIED IDEOGRAPH
	"93 FA	65E5",	#CJK UNIFIED IDEOGRAPH
	"93 FB	4E73",	#CJK UNIFIED IDEOGRAPH
	"93 FC	5165",	#CJK UNIFIED IDEOGRAPH
	"94 40	5982",	#CJK UNIFIED IDEOGRAPH
	"94 41	5C3F",	#CJK UNIFIED IDEOGRAPH
	"94 42	97EE",	#CJK UNIFIED IDEOGRAPH
	"94 43	4EFB",	#CJK UNIFIED IDEOGRAPH
	"94 44	598A",	#CJK UNIFIED IDEOGRAPH
	"94 45	5FCD",	#CJK UNIFIED IDEOGRAPH
	"94 46	8A8D",	#CJK UNIFIED IDEOGRAPH
	"94 47	6FE1",	#CJK UNIFIED IDEOGRAPH
	"94 48	79B0",	#CJK UNIFIED IDEOGRAPH
	"94 49	7962",	#CJK UNIFIED IDEOGRAPH
	"94 4A	5BE7",	#CJK UNIFIED IDEOGRAPH
	"94 4B	8471",	#CJK UNIFIED IDEOGRAPH
	"94 4C	732B",	#CJK UNIFIED IDEOGRAPH
	"94 4D	71B1",	#CJK UNIFIED IDEOGRAPH
	"94 4E	5E74",	#CJK UNIFIED IDEOGRAPH
	"94 4F	5FF5",	#CJK UNIFIED IDEOGRAPH
	"94 50	637B",	#CJK UNIFIED IDEOGRAPH
	"94 51	649A",	#CJK UNIFIED IDEOGRAPH
	"94 52	71C3",	#CJK UNIFIED IDEOGRAPH
	"94 53	7C98",	#CJK UNIFIED IDEOGRAPH
	"94 54	4E43",	#CJK UNIFIED IDEOGRAPH
	"94 55	5EFC",	#CJK UNIFIED IDEOGRAPH
	"94 56	4E4B",	#CJK UNIFIED IDEOGRAPH
	"94 57	57DC",	#CJK UNIFIED IDEOGRAPH
	"94 58	56A2",	#CJK UNIFIED IDEOGRAPH
	"94 59	60A9",	#CJK UNIFIED IDEOGRAPH
	"94 5A	6FC3",	#CJK UNIFIED IDEOGRAPH
	"94 5B	7D0D",	#CJK UNIFIED IDEOGRAPH
	"94 5C	80FD",	#CJK UNIFIED IDEOGRAPH
	"94 5D	8133",	#CJK UNIFIED IDEOGRAPH
	"94 5E	81BF",	#CJK UNIFIED IDEOGRAPH
	"94 5F	8FB2",	#CJK UNIFIED IDEOGRAPH
	"94 60	8997",	#CJK UNIFIED IDEOGRAPH
	"94 61	86A4",	#CJK UNIFIED IDEOGRAPH
	"94 62	5DF4",	#CJK UNIFIED IDEOGRAPH
	"94 63	628A",	#CJK UNIFIED IDEOGRAPH
	"94 64	64AD",	#CJK UNIFIED IDEOGRAPH
	"94 65	8987",	#CJK UNIFIED IDEOGRAPH
	"94 66	6777",	#CJK UNIFIED IDEOGRAPH
	"94 67	6CE2",	#CJK UNIFIED IDEOGRAPH
	"94 68	6D3E",	#CJK UNIFIED IDEOGRAPH
	"94 69	7436",	#CJK UNIFIED IDEOGRAPH
	"94 6A	7834",	#CJK UNIFIED IDEOGRAPH
	"94 6B	5A46",	#CJK UNIFIED IDEOGRAPH
	"94 6C	7F75",	#CJK UNIFIED IDEOGRAPH
	"94 6D	82AD",	#CJK UNIFIED IDEOGRAPH
	"94 6E	99AC",	#CJK UNIFIED IDEOGRAPH
	"94 6F	4FF3",	#CJK UNIFIED IDEOGRAPH
	"94 70	5EC3",	#CJK UNIFIED IDEOGRAPH
	"94 71	62DD",	#CJK UNIFIED IDEOGRAPH
	"94 72	6392",	#CJK UNIFIED IDEOGRAPH
	"94 73	6557",	#CJK UNIFIED IDEOGRAPH
	"94 74	676F",	#CJK UNIFIED IDEOGRAPH
	"94 75	76C3",	#CJK UNIFIED IDEOGRAPH
	"94 76	724C",	#CJK UNIFIED IDEOGRAPH
	"94 77	80CC",	#CJK UNIFIED IDEOGRAPH
	"94 78	80BA",	#CJK UNIFIED IDEOGRAPH
	"94 79	8F29",	#CJK UNIFIED IDEOGRAPH
	"94 7A	914D",	#CJK UNIFIED IDEOGRAPH
	"94 7B	500D",	#CJK UNIFIED IDEOGRAPH
	"94 7C	57F9",	#CJK UNIFIED IDEOGRAPH
	"94 7D	5A92",	#CJK UNIFIED IDEOGRAPH
	"94 7E	6885",	#CJK UNIFIED IDEOGRAPH
	"94 80	6973",	#CJK UNIFIED IDEOGRAPH
	"94 81	7164",	#CJK UNIFIED IDEOGRAPH
	"94 82	72FD",	#CJK UNIFIED IDEOGRAPH
	"94 83	8CB7",	#CJK UNIFIED IDEOGRAPH
	"94 84	58F2",	#CJK UNIFIED IDEOGRAPH
	"94 85	8CE0",	#CJK UNIFIED IDEOGRAPH
	"94 86	966A",	#CJK UNIFIED IDEOGRAPH
	"94 87	9019",	#CJK UNIFIED IDEOGRAPH
	"94 88	877F",	#CJK UNIFIED IDEOGRAPH
	"94 89	79E4",	#CJK UNIFIED IDEOGRAPH
	"94 8A	77E7",	#CJK UNIFIED IDEOGRAPH
	"94 8B	8429",	#CJK UNIFIED IDEOGRAPH
	"94 8C	4F2F",	#CJK UNIFIED IDEOGRAPH
	"94 8D	5265",	#CJK UNIFIED IDEOGRAPH
	"94 8E	535A",	#CJK UNIFIED IDEOGRAPH
	"94 8F	62CD",	#CJK UNIFIED IDEOGRAPH
	"94 90	67CF",	#CJK UNIFIED IDEOGRAPH
	"94 91	6CCA",	#CJK UNIFIED IDEOGRAPH
	"94 92	767D",	#CJK UNIFIED IDEOGRAPH
	"94 93	7B94",	#CJK UNIFIED IDEOGRAPH
	"94 94	7C95",	#CJK UNIFIED IDEOGRAPH
	"94 95	8236",	#CJK UNIFIED IDEOGRAPH
	"94 96	8584",	#CJK UNIFIED IDEOGRAPH
	"94 97	8FEB",	#CJK UNIFIED IDEOGRAPH
	"94 98	66DD",	#CJK UNIFIED IDEOGRAPH
	"94 99	6F20",	#CJK UNIFIED IDEOGRAPH
	"94 9A	7206",	#CJK UNIFIED IDEOGRAPH
	"94 9B	7E1B",	#CJK UNIFIED IDEOGRAPH
	"94 9C	83AB",	#CJK UNIFIED IDEOGRAPH
	"94 9D	99C1",	#CJK UNIFIED IDEOGRAPH
	"94 9E	9EA6",	#CJK UNIFIED IDEOGRAPH
	"94 9F	51FD",	#CJK UNIFIED IDEOGRAPH
	"94 A0	7BB1",	#CJK UNIFIED IDEOGRAPH
	"94 A1	7872",	#CJK UNIFIED IDEOGRAPH
	"94 A2	7BB8",	#CJK UNIFIED IDEOGRAPH
	"94 A3	8087",	#CJK UNIFIED IDEOGRAPH
	"94 A4	7B48",	#CJK UNIFIED IDEOGRAPH
	"94 A5	6AE8",	#CJK UNIFIED IDEOGRAPH
	"94 A6	5E61",	#CJK UNIFIED IDEOGRAPH
	"94 A7	808C",	#CJK UNIFIED IDEOGRAPH
	"94 A8	7551",	#CJK UNIFIED IDEOGRAPH
	"94 A9	7560",	#CJK UNIFIED IDEOGRAPH
	"94 AA	516B",	#CJK UNIFIED IDEOGRAPH
	"94 AB	9262",	#CJK UNIFIED IDEOGRAPH
	"94 AC	6E8C",	#CJK UNIFIED IDEOGRAPH
	"94 AD	767A",	#CJK UNIFIED IDEOGRAPH
	"94 AE	9197",	#CJK UNIFIED IDEOGRAPH
	"94 AF	9AEA",	#CJK UNIFIED IDEOGRAPH
	"94 B0	4F10",	#CJK UNIFIED IDEOGRAPH
	"94 B1	7F70",	#CJK UNIFIED IDEOGRAPH
	"94 B2	629C",	#CJK UNIFIED IDEOGRAPH
	"94 B3	7B4F",	#CJK UNIFIED IDEOGRAPH
	"94 B4	95A5",	#CJK UNIFIED IDEOGRAPH
	"94 B5	9CE9",	#CJK UNIFIED IDEOGRAPH
	"94 B6	567A",	#CJK UNIFIED IDEOGRAPH
	"94 B7	5859",	#CJK UNIFIED IDEOGRAPH
	"94 B8	86E4",	#CJK UNIFIED IDEOGRAPH
	"94 B9	96BC",	#CJK UNIFIED IDEOGRAPH
	"94 BA	4F34",	#CJK UNIFIED IDEOGRAPH
	"94 BB	5224",	#CJK UNIFIED IDEOGRAPH
	"94 BC	534A",	#CJK UNIFIED IDEOGRAPH
	"94 BD	53CD",	#CJK UNIFIED IDEOGRAPH
	"94 BE	53DB",	#CJK UNIFIED IDEOGRAPH
	"94 BF	5E06",	#CJK UNIFIED IDEOGRAPH
	"94 C0	642C",	#CJK UNIFIED IDEOGRAPH
	"94 C1	6591",	#CJK UNIFIED IDEOGRAPH
	"94 C2	677F",	#CJK UNIFIED IDEOGRAPH
	"94 C3	6C3E",	#CJK UNIFIED IDEOGRAPH
	"94 C4	6C4E",	#CJK UNIFIED IDEOGRAPH
	"94 C5	7248",	#CJK UNIFIED IDEOGRAPH
	"94 C6	72AF",	#CJK UNIFIED IDEOGRAPH
	"94 C7	73ED",	#CJK UNIFIED IDEOGRAPH
	"94 C8	7554",	#CJK UNIFIED IDEOGRAPH
	"94 C9	7E41",	#CJK UNIFIED IDEOGRAPH
	"94 CA	822C",	#CJK UNIFIED IDEOGRAPH
	"94 CB	85E9",	#CJK UNIFIED IDEOGRAPH
	"94 CC	8CA9",	#CJK UNIFIED IDEOGRAPH
	"94 CD	7BC4",	#CJK UNIFIED IDEOGRAPH
	"94 CE	91C6",	#CJK UNIFIED IDEOGRAPH
	"94 CF	7169",	#CJK UNIFIED IDEOGRAPH
	"94 D0	9812",	#CJK UNIFIED IDEOGRAPH
	"94 D1	98EF",	#CJK UNIFIED IDEOGRAPH
	"94 D2	633D",	#CJK UNIFIED IDEOGRAPH
	"94 D3	6669",	#CJK UNIFIED IDEOGRAPH
	"94 D4	756A",	#CJK UNIFIED IDEOGRAPH
	"94 D5	76E4",	#CJK UNIFIED IDEOGRAPH
	"94 D6	78D0",	#CJK UNIFIED IDEOGRAPH
	"94 D7	8543",	#CJK UNIFIED IDEOGRAPH
	"94 D8	86EE",	#CJK UNIFIED IDEOGRAPH
	"94 D9	532A",	#CJK UNIFIED IDEOGRAPH
	"94 DA	5351",	#CJK UNIFIED IDEOGRAPH
	"94 DB	5426",	#CJK UNIFIED IDEOGRAPH
	"94 DC	5983",	#CJK UNIFIED IDEOGRAPH
	"94 DD	5E87",	#CJK UNIFIED IDEOGRAPH
	"94 DE	5F7C",	#CJK UNIFIED IDEOGRAPH
	"94 DF	60B2",	#CJK UNIFIED IDEOGRAPH
	"94 E0	6249",	#CJK UNIFIED IDEOGRAPH
	"94 E1	6279",	#CJK UNIFIED IDEOGRAPH
	"94 E2	62AB",	#CJK UNIFIED IDEOGRAPH
	"94 E3	6590",	#CJK UNIFIED IDEOGRAPH
	"94 E4	6BD4",	#CJK UNIFIED IDEOGRAPH
	"94 E5	6CCC",	#CJK UNIFIED IDEOGRAPH
	"94 E6	75B2",	#CJK UNIFIED IDEOGRAPH
	"94 E7	76AE",	#CJK UNIFIED IDEOGRAPH
	"94 E8	7891",	#CJK UNIFIED IDEOGRAPH
	"94 E9	79D8",	#CJK UNIFIED IDEOGRAPH
	"94 EA	7DCB",	#CJK UNIFIED IDEOGRAPH
	"94 EB	7F77",	#CJK UNIFIED IDEOGRAPH
	"94 EC	80A5",	#CJK UNIFIED IDEOGRAPH
	"94 ED	88AB",	#CJK UNIFIED IDEOGRAPH
	"94 EE	8AB9",	#CJK UNIFIED IDEOGRAPH
	"94 EF	8CBB",	#CJK UNIFIED IDEOGRAPH
	"94 F0	907F",	#CJK UNIFIED IDEOGRAPH
	"94 F1	975E",	#CJK UNIFIED IDEOGRAPH
	"94 F2	98DB",	#CJK UNIFIED IDEOGRAPH
	"94 F3	6A0B",	#CJK UNIFIED IDEOGRAPH
	"94 F4	7C38",	#CJK UNIFIED IDEOGRAPH
	"94 F5	5099",	#CJK UNIFIED IDEOGRAPH
	"94 F6	5C3E",	#CJK UNIFIED IDEOGRAPH
	"94 F7	5FAE",	#CJK UNIFIED IDEOGRAPH
	"94 F8	6787",	#CJK UNIFIED IDEOGRAPH
	"94 F9	6BD8",	#CJK UNIFIED IDEOGRAPH
	"94 FA	7435",	#CJK UNIFIED IDEOGRAPH
	"94 FB	7709",	#CJK UNIFIED IDEOGRAPH
	"94 FC	7F8E",	#CJK UNIFIED IDEOGRAPH
	"95 40	9F3B",	#CJK UNIFIED IDEOGRAPH
	"95 41	67CA",	#CJK UNIFIED IDEOGRAPH
	"95 42	7A17",	#CJK UNIFIED IDEOGRAPH
	"95 43	5339",	#CJK UNIFIED IDEOGRAPH
	"95 44	758B",	#CJK UNIFIED IDEOGRAPH
	"95 45	9AED",	#CJK UNIFIED IDEOGRAPH
	"95 46	5F66",	#CJK UNIFIED IDEOGRAPH
	"95 47	819D",	#CJK UNIFIED IDEOGRAPH
	"95 48	83F1",	#CJK UNIFIED IDEOGRAPH
	"95 49	8098",	#CJK UNIFIED IDEOGRAPH
	"95 4A	5F3C",	#CJK UNIFIED IDEOGRAPH
	"95 4B	5FC5",	#CJK UNIFIED IDEOGRAPH
	"95 4C	7562",	#CJK UNIFIED IDEOGRAPH
	"95 4D	7B46",	#CJK UNIFIED IDEOGRAPH
	"95 4E	903C",	#CJK UNIFIED IDEOGRAPH
	"95 4F	6867",	#CJK UNIFIED IDEOGRAPH
	"95 50	59EB",	#CJK UNIFIED IDEOGRAPH
	"95 51	5A9B",	#CJK UNIFIED IDEOGRAPH
	"95 52	7D10",	#CJK UNIFIED IDEOGRAPH
	"95 53	767E",	#CJK UNIFIED IDEOGRAPH
	"95 54	8B2C",	#CJK UNIFIED IDEOGRAPH
	"95 55	4FF5",	#CJK UNIFIED IDEOGRAPH
	"95 56	5F6A",	#CJK UNIFIED IDEOGRAPH
	"95 57	6A19",	#CJK UNIFIED IDEOGRAPH
	"95 58	6C37",	#CJK UNIFIED IDEOGRAPH
	"95 59	6F02",	#CJK UNIFIED IDEOGRAPH
	"95 5A	74E2",	#CJK UNIFIED IDEOGRAPH
	"95 5B	7968",	#CJK UNIFIED IDEOGRAPH
	"95 5C	8868",	#CJK UNIFIED IDEOGRAPH
	"95 5D	8A55",	#CJK UNIFIED IDEOGRAPH
	"95 5E	8C79",	#CJK UNIFIED IDEOGRAPH
	"95 5F	5EDF",	#CJK UNIFIED IDEOGRAPH
	"95 60	63CF",	#CJK UNIFIED IDEOGRAPH
	"95 61	75C5",	#CJK UNIFIED IDEOGRAPH
	"95 62	79D2",	#CJK UNIFIED IDEOGRAPH
	"95 63	82D7",	#CJK UNIFIED IDEOGRAPH
	"95 64	9328",	#CJK UNIFIED IDEOGRAPH
	"95 65	92F2",	#CJK UNIFIED IDEOGRAPH
	"95 66	849C",	#CJK UNIFIED IDEOGRAPH
	"95 67	86ED",	#CJK UNIFIED IDEOGRAPH
	"95 68	9C2D",	#CJK UNIFIED IDEOGRAPH
	"95 69	54C1",	#CJK UNIFIED IDEOGRAPH
	"95 6A	5F6C",	#CJK UNIFIED IDEOGRAPH
	"95 6B	658C",	#CJK UNIFIED IDEOGRAPH
	"95 6C	6D5C",	#CJK UNIFIED IDEOGRAPH
	"95 6D	7015",	#CJK UNIFIED IDEOGRAPH
	"95 6E	8CA7",	#CJK UNIFIED IDEOGRAPH
	"95 6F	8CD3",	#CJK UNIFIED IDEOGRAPH
	"95 70	983B",	#CJK UNIFIED IDEOGRAPH
	"95 71	654F",	#CJK UNIFIED IDEOGRAPH
	"95 72	74F6",	#CJK UNIFIED IDEOGRAPH
	"95 73	4E0D",	#CJK UNIFIED IDEOGRAPH
	"95 74	4ED8",	#CJK UNIFIED IDEOGRAPH
	"95 75	57E0",	#CJK UNIFIED IDEOGRAPH
	"95 76	592B",	#CJK UNIFIED IDEOGRAPH
	"95 77	5A66",	#CJK UNIFIED IDEOGRAPH
	"95 78	5BCC",	#CJK UNIFIED IDEOGRAPH
	"95 79	51A8",	#CJK UNIFIED IDEOGRAPH
	"95 7A	5E03",	#CJK UNIFIED IDEOGRAPH
	"95 7B	5E9C",	#CJK UNIFIED IDEOGRAPH
	"95 7C	6016",	#CJK UNIFIED IDEOGRAPH
	"95 7D	6276",	#CJK UNIFIED IDEOGRAPH
	"95 7E	6577",	#CJK UNIFIED IDEOGRAPH
	"95 80	65A7",	#CJK UNIFIED IDEOGRAPH
	"95 81	666E",	#CJK UNIFIED IDEOGRAPH
	"95 82	6D6E",	#CJK UNIFIED IDEOGRAPH
	"95 83	7236",	#CJK UNIFIED IDEOGRAPH
	"95 84	7B26",	#CJK UNIFIED IDEOGRAPH
	"95 85	8150",	#CJK UNIFIED IDEOGRAPH
	"95 86	819A",	#CJK UNIFIED IDEOGRAPH
	"95 87	8299",	#CJK UNIFIED IDEOGRAPH
	"95 88	8B5C",	#CJK UNIFIED IDEOGRAPH
	"95 89	8CA0",	#CJK UNIFIED IDEOGRAPH
	"95 8A	8CE6",	#CJK UNIFIED IDEOGRAPH
	"95 8B	8D74",	#CJK UNIFIED IDEOGRAPH
	"95 8C	961C",	#CJK UNIFIED IDEOGRAPH
	"95 8D	9644",	#CJK UNIFIED IDEOGRAPH
	"95 8E	4FAE",	#CJK UNIFIED IDEOGRAPH
	"95 8F	64AB",	#CJK UNIFIED IDEOGRAPH
	"95 90	6B66",	#CJK UNIFIED IDEOGRAPH
	"95 91	821E",	#CJK UNIFIED IDEOGRAPH
	"95 92	8461",	#CJK UNIFIED IDEOGRAPH
	"95 93	856A",	#CJK UNIFIED IDEOGRAPH
	"95 94	90E8",	#CJK UNIFIED IDEOGRAPH
	"95 95	5C01",	#CJK UNIFIED IDEOGRAPH
	"95 96	6953",	#CJK UNIFIED IDEOGRAPH
	"95 97	98A8",	#CJK UNIFIED IDEOGRAPH
	"95 98	847A",	#CJK UNIFIED IDEOGRAPH
	"95 99	8557",	#CJK UNIFIED IDEOGRAPH
	"95 9A	4F0F",	#CJK UNIFIED IDEOGRAPH
	"95 9B	526F",	#CJK UNIFIED IDEOGRAPH
	"95 9C	5FA9",	#CJK UNIFIED IDEOGRAPH
	"95 9D	5E45",	#CJK UNIFIED IDEOGRAPH
	"95 9E	670D",	#CJK UNIFIED IDEOGRAPH
	"95 9F	798F",	#CJK UNIFIED IDEOGRAPH
	"95 A0	8179",	#CJK UNIFIED IDEOGRAPH
	"95 A1	8907",	#CJK UNIFIED IDEOGRAPH
	"95 A2	8986",	#CJK UNIFIED IDEOGRAPH
	"95 A3	6DF5",	#CJK UNIFIED IDEOGRAPH
	"95 A4	5F17",	#CJK UNIFIED IDEOGRAPH
	"95 A5	6255",	#CJK UNIFIED IDEOGRAPH
	"95 A6	6CB8",	#CJK UNIFIED IDEOGRAPH
	"95 A7	4ECF",	#CJK UNIFIED IDEOGRAPH
	"95 A8	7269",	#CJK UNIFIED IDEOGRAPH
	"95 A9	9B92",	#CJK UNIFIED IDEOGRAPH
	"95 AA	5206",	#CJK UNIFIED IDEOGRAPH
	"95 AB	543B",	#CJK UNIFIED IDEOGRAPH
	"95 AC	5674",	#CJK UNIFIED IDEOGRAPH
	"95 AD	58B3",	#CJK UNIFIED IDEOGRAPH
	"95 AE	61A4",	#CJK UNIFIED IDEOGRAPH
	"95 AF	626E",	#CJK UNIFIED IDEOGRAPH
	"95 B0	711A",	#CJK UNIFIED IDEOGRAPH
	"95 B1	596E",	#CJK UNIFIED IDEOGRAPH
	"95 B2	7C89",	#CJK UNIFIED IDEOGRAPH
	"95 B3	7CDE",	#CJK UNIFIED IDEOGRAPH
	"95 B4	7D1B",	#CJK UNIFIED IDEOGRAPH
	"95 B5	96F0",	#CJK UNIFIED IDEOGRAPH
	"95 B6	6587",	#CJK UNIFIED IDEOGRAPH
	"95 B7	805E",	#CJK UNIFIED IDEOGRAPH
	"95 B8	4E19",	#CJK UNIFIED IDEOGRAPH
	"95 B9	4F75",	#CJK UNIFIED IDEOGRAPH
	"95 BA	5175",	#CJK UNIFIED IDEOGRAPH
	"95 BB	5840",	#CJK UNIFIED IDEOGRAPH
	"95 BC	5E63",	#CJK UNIFIED IDEOGRAPH
	"95 BD	5E73",	#CJK UNIFIED IDEOGRAPH
	"95 BE	5F0A",	#CJK UNIFIED IDEOGRAPH
	"95 BF	67C4",	#CJK UNIFIED IDEOGRAPH
	"95 C0	4E26",	#CJK UNIFIED IDEOGRAPH
	"95 C1	853D",	#CJK UNIFIED IDEOGRAPH
	"95 C2	9589",	#CJK UNIFIED IDEOGRAPH
	"95 C3	965B",	#CJK UNIFIED IDEOGRAPH
	"95 C4	7C73",	#CJK UNIFIED IDEOGRAPH
	"95 C5	9801",	#CJK UNIFIED IDEOGRAPH
	"95 C6	50FB",	#CJK UNIFIED IDEOGRAPH
	"95 C7	58C1",	#CJK UNIFIED IDEOGRAPH
	"95 C8	7656",	#CJK UNIFIED IDEOGRAPH
	"95 C9	78A7",	#CJK UNIFIED IDEOGRAPH
	"95 CA	5225",	#CJK UNIFIED IDEOGRAPH
	"95 CB	77A5",	#CJK UNIFIED IDEOGRAPH
	"95 CC	8511",	#CJK UNIFIED IDEOGRAPH
	"95 CD	7B86",	#CJK UNIFIED IDEOGRAPH
	"95 CE	504F",	#CJK UNIFIED IDEOGRAPH
	"95 CF	5909",	#CJK UNIFIED IDEOGRAPH
	"95 D0	7247",	#CJK UNIFIED IDEOGRAPH
	"95 D1	7BC7",	#CJK UNIFIED IDEOGRAPH
	"95 D2	7DE8",	#CJK UNIFIED IDEOGRAPH
	"95 D3	8FBA",	#CJK UNIFIED IDEOGRAPH
	"95 D4	8FD4",	#CJK UNIFIED IDEOGRAPH
	"95 D5	904D",	#CJK UNIFIED IDEOGRAPH
	"95 D6	4FBF",	#CJK UNIFIED IDEOGRAPH
	"95 D7	52C9",	#CJK UNIFIED IDEOGRAPH
	"95 D8	5A29",	#CJK UNIFIED IDEOGRAPH
	"95 D9	5F01",	#CJK UNIFIED IDEOGRAPH
	"95 DA	97AD",	#CJK UNIFIED IDEOGRAPH
	"95 DB	4FDD",	#CJK UNIFIED IDEOGRAPH
	"95 DC	8217",	#CJK UNIFIED IDEOGRAPH
	"95 DD	92EA",	#CJK UNIFIED IDEOGRAPH
	"95 DE	5703",	#CJK UNIFIED IDEOGRAPH
	"95 DF	6355",	#CJK UNIFIED IDEOGRAPH
	"95 E0	6B69",	#CJK UNIFIED IDEOGRAPH
	"95 E1	752B",	#CJK UNIFIED IDEOGRAPH
	"95 E2	88DC",	#CJK UNIFIED IDEOGRAPH
	"95 E3	8F14",	#CJK UNIFIED IDEOGRAPH
	"95 E4	7A42",	#CJK UNIFIED IDEOGRAPH
	"95 E5	52DF",	#CJK UNIFIED IDEOGRAPH
	"95 E6	5893",	#CJK UNIFIED IDEOGRAPH
	"95 E7	6155",	#CJK UNIFIED IDEOGRAPH
	"95 E8	620A",	#CJK UNIFIED IDEOGRAPH
	"95 E9	66AE",	#CJK UNIFIED IDEOGRAPH
	"95 EA	6BCD",	#CJK UNIFIED IDEOGRAPH
	"95 EB	7C3F",	#CJK UNIFIED IDEOGRAPH
	"95 EC	83E9",	#CJK UNIFIED IDEOGRAPH
	"95 ED	5023",	#CJK UNIFIED IDEOGRAPH
	"95 EE	4FF8",	#CJK UNIFIED IDEOGRAPH
	"95 EF	5305",	#CJK UNIFIED IDEOGRAPH
	"95 F0	5446",	#CJK UNIFIED IDEOGRAPH
	"95 F1	5831",	#CJK UNIFIED IDEOGRAPH
	"95 F2	5949",	#CJK UNIFIED IDEOGRAPH
	"95 F3	5B9D",	#CJK UNIFIED IDEOGRAPH
	"95 F4	5CF0",	#CJK UNIFIED IDEOGRAPH
	"95 F5	5CEF",	#CJK UNIFIED IDEOGRAPH
	"95 F6	5D29",	#CJK UNIFIED IDEOGRAPH
	"95 F7	5E96",	#CJK UNIFIED IDEOGRAPH
	"95 F8	62B1",	#CJK UNIFIED IDEOGRAPH
	"95 F9	6367",	#CJK UNIFIED IDEOGRAPH
	"95 FA	653E",	#CJK UNIFIED IDEOGRAPH
	"95 FB	65B9",	#CJK UNIFIED IDEOGRAPH
	"95 FC	670B",	#CJK UNIFIED IDEOGRAPH
	"96 40	6CD5",	#CJK UNIFIED IDEOGRAPH
	"96 41	6CE1",	#CJK UNIFIED IDEOGRAPH
	"96 42	70F9",	#CJK UNIFIED IDEOGRAPH
	"96 43	7832",	#CJK UNIFIED IDEOGRAPH
	"96 44	7E2B",	#CJK UNIFIED IDEOGRAPH
	"96 45	80DE",	#CJK UNIFIED IDEOGRAPH
	"96 46	82B3",	#CJK UNIFIED IDEOGRAPH
	"96 47	840C",	#CJK UNIFIED IDEOGRAPH
	"96 48	84EC",	#CJK UNIFIED IDEOGRAPH
	"96 49	8702",	#CJK UNIFIED IDEOGRAPH
	"96 4A	8912",	#CJK UNIFIED IDEOGRAPH
	"96 4B	8A2A",	#CJK UNIFIED IDEOGRAPH
	"96 4C	8C4A",	#CJK UNIFIED IDEOGRAPH
	"96 4D	90A6",	#CJK UNIFIED IDEOGRAPH
	"96 4E	92D2",	#CJK UNIFIED IDEOGRAPH
	"96 4F	98FD",	#CJK UNIFIED IDEOGRAPH
	"96 50	9CF3",	#CJK UNIFIED IDEOGRAPH
	"96 51	9D6C",	#CJK UNIFIED IDEOGRAPH
	"96 52	4E4F",	#CJK UNIFIED IDEOGRAPH
	"96 53	4EA1",	#CJK UNIFIED IDEOGRAPH
	"96 54	508D",	#CJK UNIFIED IDEOGRAPH
	"96 55	5256",	#CJK UNIFIED IDEOGRAPH
	"96 56	574A",	#CJK UNIFIED IDEOGRAPH
	"96 57	59A8",	#CJK UNIFIED IDEOGRAPH
	"96 58	5E3D",	#CJK UNIFIED IDEOGRAPH
	"96 59	5FD8",	#CJK UNIFIED IDEOGRAPH
	"96 5A	5FD9",	#CJK UNIFIED IDEOGRAPH
	"96 5B	623F",	#CJK UNIFIED IDEOGRAPH
	"96 5C	66B4",	#CJK UNIFIED IDEOGRAPH
	"96 5D	671B",	#CJK UNIFIED IDEOGRAPH
	"96 5E	67D0",	#CJK UNIFIED IDEOGRAPH
	"96 5F	68D2",	#CJK UNIFIED IDEOGRAPH
	"96 60	5192",	#CJK UNIFIED IDEOGRAPH
	"96 61	7D21",	#CJK UNIFIED IDEOGRAPH
	"96 62	80AA",	#CJK UNIFIED IDEOGRAPH
	"96 63	81A8",	#CJK UNIFIED IDEOGRAPH
	"96 64	8B00",	#CJK UNIFIED IDEOGRAPH
	"96 65	8C8C",	#CJK UNIFIED IDEOGRAPH
	"96 66	8CBF",	#CJK UNIFIED IDEOGRAPH
	"96 67	927E",	#CJK UNIFIED IDEOGRAPH
	"96 68	9632",	#CJK UNIFIED IDEOGRAPH
	"96 69	5420",	#CJK UNIFIED IDEOGRAPH
	"96 6A	982C",	#CJK UNIFIED IDEOGRAPH
	"96 6B	5317",	#CJK UNIFIED IDEOGRAPH
	"96 6C	50D5",	#CJK UNIFIED IDEOGRAPH
	"96 6D	535C",	#CJK UNIFIED IDEOGRAPH
	"96 6E	58A8",	#CJK UNIFIED IDEOGRAPH
	"96 6F	64B2",	#CJK UNIFIED IDEOGRAPH
	"96 70	6734",	#CJK UNIFIED IDEOGRAPH
	"96 71	7267",	#CJK UNIFIED IDEOGRAPH
	"96 72	7766",	#CJK UNIFIED IDEOGRAPH
	"96 73	7A46",	#CJK UNIFIED IDEOGRAPH
	"96 74	91E6",	#CJK UNIFIED IDEOGRAPH
	"96 75	52C3",	#CJK UNIFIED IDEOGRAPH
	"96 76	6CA1",	#CJK UNIFIED IDEOGRAPH
	"96 77	6B86",	#CJK UNIFIED IDEOGRAPH
	"96 78	5800",	#CJK UNIFIED IDEOGRAPH
	"96 79	5E4C",	#CJK UNIFIED IDEOGRAPH
	"96 7A	5954",	#CJK UNIFIED IDEOGRAPH
	"96 7B	672C",	#CJK UNIFIED IDEOGRAPH
	"96 7C	7FFB",	#CJK UNIFIED IDEOGRAPH
	"96 7D	51E1",	#CJK UNIFIED IDEOGRAPH
	"96 7E	76C6",	#CJK UNIFIED IDEOGRAPH
	"96 80	6469",	#CJK UNIFIED IDEOGRAPH
	"96 81	78E8",	#CJK UNIFIED IDEOGRAPH
	"96 82	9B54",	#CJK UNIFIED IDEOGRAPH
	"96 83	9EBB",	#CJK UNIFIED IDEOGRAPH
	"96 84	57CB",	#CJK UNIFIED IDEOGRAPH
	"96 85	59B9",	#CJK UNIFIED IDEOGRAPH
	"96 86	6627",	#CJK UNIFIED IDEOGRAPH
	"96 87	679A",	#CJK UNIFIED IDEOGRAPH
	"96 88	6BCE",	#CJK UNIFIED IDEOGRAPH
	"96 89	54E9",	#CJK UNIFIED IDEOGRAPH
	"96 8A	69D9",	#CJK UNIFIED IDEOGRAPH
	"96 8B	5E55",	#CJK UNIFIED IDEOGRAPH
	"96 8C	819C",	#CJK UNIFIED IDEOGRAPH
	"96 8D	6795",	#CJK UNIFIED IDEOGRAPH
	"96 8E	9BAA",	#CJK UNIFIED IDEOGRAPH
	"96 8F	67FE",	#CJK UNIFIED IDEOGRAPH
	"96 90	9C52",	#CJK UNIFIED IDEOGRAPH
	"96 91	685D",	#CJK UNIFIED IDEOGRAPH
	"96 92	4EA6",	#CJK UNIFIED IDEOGRAPH
	"96 93	4FE3",	#CJK UNIFIED IDEOGRAPH
	"96 94	53C8",	#CJK UNIFIED IDEOGRAPH
	"96 95	62B9",	#CJK UNIFIED IDEOGRAPH
	"96 96	672B",	#CJK UNIFIED IDEOGRAPH
	"96 97	6CAB",	#CJK UNIFIED IDEOGRAPH
	"96 98	8FC4",	#CJK UNIFIED IDEOGRAPH
	"96 99	4FAD",	#CJK UNIFIED IDEOGRAPH
	"96 9A	7E6D",	#CJK UNIFIED IDEOGRAPH
	"96 9B	9EBF",	#CJK UNIFIED IDEOGRAPH
	"96 9C	4E07",	#CJK UNIFIED IDEOGRAPH
	"96 9D	6162",	#CJK UNIFIED IDEOGRAPH
	"96 9E	6E80",	#CJK UNIFIED IDEOGRAPH
	"96 9F	6F2B",	#CJK UNIFIED IDEOGRAPH
	"96 A0	8513",	#CJK UNIFIED IDEOGRAPH
	"96 A1	5473",	#CJK UNIFIED IDEOGRAPH
	"96 A2	672A",	#CJK UNIFIED IDEOGRAPH
	"96 A3	9B45",	#CJK UNIFIED IDEOGRAPH
	"96 A4	5DF3",	#CJK UNIFIED IDEOGRAPH
	"96 A5	7B95",	#CJK UNIFIED IDEOGRAPH
	"96 A6	5CAC",	#CJK UNIFIED IDEOGRAPH
	"96 A7	5BC6",	#CJK UNIFIED IDEOGRAPH
	"96 A8	871C",	#CJK UNIFIED IDEOGRAPH
	"96 A9	6E4A",	#CJK UNIFIED IDEOGRAPH
	"96 AA	84D1",	#CJK UNIFIED IDEOGRAPH
	"96 AB	7A14",	#CJK UNIFIED IDEOGRAPH
	"96 AC	8108",	#CJK UNIFIED IDEOGRAPH
	"96 AD	5999",	#CJK UNIFIED IDEOGRAPH
	"96 AE	7C8D",	#CJK UNIFIED IDEOGRAPH
	"96 AF	6C11",	#CJK UNIFIED IDEOGRAPH
	"96 B0	7720",	#CJK UNIFIED IDEOGRAPH
	"96 B1	52D9",	#CJK UNIFIED IDEOGRAPH
	"96 B2	5922",	#CJK UNIFIED IDEOGRAPH
	"96 B3	7121",	#CJK UNIFIED IDEOGRAPH
	"96 B4	725F",	#CJK UNIFIED IDEOGRAPH
	"96 B5	77DB",	#CJK UNIFIED IDEOGRAPH
	"96 B6	9727",	#CJK UNIFIED IDEOGRAPH
	"96 B7	9D61",	#CJK UNIFIED IDEOGRAPH
	"96 B8	690B",	#CJK UNIFIED IDEOGRAPH
	"96 B9	5A7F",	#CJK UNIFIED IDEOGRAPH
	"96 BA	5A18",	#CJK UNIFIED IDEOGRAPH
	"96 BB	51A5",	#CJK UNIFIED IDEOGRAPH
	"96 BC	540D",	#CJK UNIFIED IDEOGRAPH
	"96 BD	547D",	#CJK UNIFIED IDEOGRAPH
	"96 BE	660E",	#CJK UNIFIED IDEOGRAPH
	"96 BF	76DF",	#CJK UNIFIED IDEOGRAPH
	"96 C0	8FF7",	#CJK UNIFIED IDEOGRAPH
	"96 C1	9298",	#CJK UNIFIED IDEOGRAPH
	"96 C2	9CF4",	#CJK UNIFIED IDEOGRAPH
	"96 C3	59EA",	#CJK UNIFIED IDEOGRAPH
	"96 C4	725D",	#CJK UNIFIED IDEOGRAPH
	"96 C5	6EC5",	#CJK UNIFIED IDEOGRAPH
	"96 C6	514D",	#CJK UNIFIED IDEOGRAPH
	"96 C7	68C9",	#CJK UNIFIED IDEOGRAPH
	"96 C8	7DBF",	#CJK UNIFIED IDEOGRAPH
	"96 C9	7DEC",	#CJK UNIFIED IDEOGRAPH
	"96 CA	9762",	#CJK UNIFIED IDEOGRAPH
	"96 CB	9EBA",	#CJK UNIFIED IDEOGRAPH
	"96 CC	6478",	#CJK UNIFIED IDEOGRAPH
	"96 CD	6A21",	#CJK UNIFIED IDEOGRAPH
	"96 CE	8302",	#CJK UNIFIED IDEOGRAPH
	"96 CF	5984",	#CJK UNIFIED IDEOGRAPH
	"96 D0	5B5F",	#CJK UNIFIED IDEOGRAPH
	"96 D1	6BDB",	#CJK UNIFIED IDEOGRAPH
	"96 D2	731B",	#CJK UNIFIED IDEOGRAPH
	"96 D3	76F2",	#CJK UNIFIED IDEOGRAPH
	"96 D4	7DB2",	#CJK UNIFIED IDEOGRAPH
	"96 D5	8017",	#CJK UNIFIED IDEOGRAPH
	"96 D6	8499",	#CJK UNIFIED IDEOGRAPH
	"96 D7	5132",	#CJK UNIFIED IDEOGRAPH
	"96 D8	6728",	#CJK UNIFIED IDEOGRAPH
	"96 D9	9ED9",	#CJK UNIFIED IDEOGRAPH
	"96 DA	76EE",	#CJK UNIFIED IDEOGRAPH
	"96 DB	6762",	#CJK UNIFIED IDEOGRAPH
	"96 DC	52FF",	#CJK UNIFIED IDEOGRAPH
	"96 DD	9905",	#CJK UNIFIED IDEOGRAPH
	"96 DE	5C24",	#CJK UNIFIED IDEOGRAPH
	"96 DF	623B",	#CJK UNIFIED IDEOGRAPH
	"96 E0	7C7E",	#CJK UNIFIED IDEOGRAPH
	"96 E1	8CB0",	#CJK UNIFIED IDEOGRAPH
	"96 E2	554F",	#CJK UNIFIED IDEOGRAPH
	"96 E3	60B6",	#CJK UNIFIED IDEOGRAPH
	"96 E4	7D0B",	#CJK UNIFIED IDEOGRAPH
	"96 E5	9580",	#CJK UNIFIED IDEOGRAPH
	"96 E6	5301",	#CJK UNIFIED IDEOGRAPH
	"96 E7	4E5F",	#CJK UNIFIED IDEOGRAPH
	"96 E8	51B6",	#CJK UNIFIED IDEOGRAPH
	"96 E9	591C",	#CJK UNIFIED IDEOGRAPH
	"96 EA	723A",	#CJK UNIFIED IDEOGRAPH
	"96 EB	8036",	#CJK UNIFIED IDEOGRAPH
	"96 EC	91CE",	#CJK UNIFIED IDEOGRAPH
	"96 ED	5F25",	#CJK UNIFIED IDEOGRAPH
	"96 EE	77E2",	#CJK UNIFIED IDEOGRAPH
	"96 EF	5384",	#CJK UNIFIED IDEOGRAPH
	"96 F0	5F79",	#CJK UNIFIED IDEOGRAPH
	"96 F1	7D04",	#CJK UNIFIED IDEOGRAPH
	"96 F2	85AC",	#CJK UNIFIED IDEOGRAPH
	"96 F3	8A33",	#CJK UNIFIED IDEOGRAPH
	"96 F4	8E8D",	#CJK UNIFIED IDEOGRAPH
	"96 F5	9756",	#CJK UNIFIED IDEOGRAPH
	"96 F6	67F3",	#CJK UNIFIED IDEOGRAPH
	"96 F7	85AE",	#CJK UNIFIED IDEOGRAPH
	"96 F8	9453",	#CJK UNIFIED IDEOGRAPH
	"96 F9	6109",	#CJK UNIFIED IDEOGRAPH
	"96 FA	6108",	#CJK UNIFIED IDEOGRAPH
	"96 FB	6CB9",	#CJK UNIFIED IDEOGRAPH
	"96 FC	7652",	#CJK UNIFIED IDEOGRAPH
	"97 40	8AED",	#CJK UNIFIED IDEOGRAPH
	"97 41	8F38",	#CJK UNIFIED IDEOGRAPH
	"97 42	552F",	#CJK UNIFIED IDEOGRAPH
	"97 43	4F51",	#CJK UNIFIED IDEOGRAPH
	"97 44	512A",	#CJK UNIFIED IDEOGRAPH
	"97 45	52C7",	#CJK UNIFIED IDEOGRAPH
	"97 46	53CB",	#CJK UNIFIED IDEOGRAPH
	"97 47	5BA5",	#CJK UNIFIED IDEOGRAPH
	"97 48	5E7D",	#CJK UNIFIED IDEOGRAPH
	"97 49	60A0",	#CJK UNIFIED IDEOGRAPH
	"97 4A	6182",	#CJK UNIFIED IDEOGRAPH
	"97 4B	63D6",	#CJK UNIFIED IDEOGRAPH
	"97 4C	6709",	#CJK UNIFIED IDEOGRAPH
	"97 4D	67DA",	#CJK UNIFIED IDEOGRAPH
	"97 4E	6E67",	#CJK UNIFIED IDEOGRAPH
	"97 4F	6D8C",	#CJK UNIFIED IDEOGRAPH
	"97 50	7336",	#CJK UNIFIED IDEOGRAPH
	"97 51	7337",	#CJK UNIFIED IDEOGRAPH
	"97 52	7531",	#CJK UNIFIED IDEOGRAPH
	"97 53	7950",	#CJK UNIFIED IDEOGRAPH
	"97 54	88D5",	#CJK UNIFIED IDEOGRAPH
	"97 55	8A98",	#CJK UNIFIED IDEOGRAPH
	"97 56	904A",	#CJK UNIFIED IDEOGRAPH
	"97 57	9091",	#CJK UNIFIED IDEOGRAPH
	"97 58	90F5",	#CJK UNIFIED IDEOGRAPH
	"97 59	96C4",	#CJK UNIFIED IDEOGRAPH
	"97 5A	878D",	#CJK UNIFIED IDEOGRAPH
	"97 5B	5915",	#CJK UNIFIED IDEOGRAPH
	"97 5C	4E88",	#CJK UNIFIED IDEOGRAPH
	"97 5D	4F59",	#CJK UNIFIED IDEOGRAPH
	"97 5E	4E0E",	#CJK UNIFIED IDEOGRAPH
	"97 5F	8A89",	#CJK UNIFIED IDEOGRAPH
	"97 60	8F3F",	#CJK UNIFIED IDEOGRAPH
	"97 61	9810",	#CJK UNIFIED IDEOGRAPH
	"97 62	50AD",	#CJK UNIFIED IDEOGRAPH
	"97 63	5E7C",	#CJK UNIFIED IDEOGRAPH
	"97 64	5996",	#CJK UNIFIED IDEOGRAPH
	"97 65	5BB9",	#CJK UNIFIED IDEOGRAPH
	"97 66	5EB8",	#CJK UNIFIED IDEOGRAPH
	"97 67	63DA",	#CJK UNIFIED IDEOGRAPH
	"97 68	63FA",	#CJK UNIFIED IDEOGRAPH
	"97 69	64C1",	#CJK UNIFIED IDEOGRAPH
	"97 6A	66DC",	#CJK UNIFIED IDEOGRAPH
	"97 6B	694A",	#CJK UNIFIED IDEOGRAPH
	"97 6C	69D8",	#CJK UNIFIED IDEOGRAPH
	"97 6D	6D0B",	#CJK UNIFIED IDEOGRAPH
	"97 6E	6EB6",	#CJK UNIFIED IDEOGRAPH
	"97 6F	7194",	#CJK UNIFIED IDEOGRAPH
	"97 70	7528",	#CJK UNIFIED IDEOGRAPH
	"97 71	7AAF",	#CJK UNIFIED IDEOGRAPH
	"97 72	7F8A",	#CJK UNIFIED IDEOGRAPH
	"97 73	8000",	#CJK UNIFIED IDEOGRAPH
	"97 74	8449",	#CJK UNIFIED IDEOGRAPH
	"97 75	84C9",	#CJK UNIFIED IDEOGRAPH
	"97 76	8981",	#CJK UNIFIED IDEOGRAPH
	"97 77	8B21",	#CJK UNIFIED IDEOGRAPH
	"97 78	8E0A",	#CJK UNIFIED IDEOGRAPH
	"97 79	9065",	#CJK UNIFIED IDEOGRAPH
	"97 7A	967D",	#CJK UNIFIED IDEOGRAPH
	"97 7B	990A",	#CJK UNIFIED IDEOGRAPH
	"97 7C	617E",	#CJK UNIFIED IDEOGRAPH
	"97 7D	6291",	#CJK UNIFIED IDEOGRAPH
	"97 7E	6B32",	#CJK UNIFIED IDEOGRAPH
	"97 80	6C83",	#CJK UNIFIED IDEOGRAPH
	"97 81	6D74",	#CJK UNIFIED IDEOGRAPH
	"97 82	7FCC",	#CJK UNIFIED IDEOGRAPH
	"97 83	7FFC",	#CJK UNIFIED IDEOGRAPH
	"97 84	6DC0",	#CJK UNIFIED IDEOGRAPH
	"97 85	7F85",	#CJK UNIFIED IDEOGRAPH
	"97 86	87BA",	#CJK UNIFIED IDEOGRAPH
	"97 87	88F8",	#CJK UNIFIED IDEOGRAPH
	"97 88	6765",	#CJK UNIFIED IDEOGRAPH
	"97 89	83B1",	#CJK UNIFIED IDEOGRAPH
	"97 8A	983C",	#CJK UNIFIED IDEOGRAPH
	"97 8B	96F7",	#CJK UNIFIED IDEOGRAPH
	"97 8C	6D1B",	#CJK UNIFIED IDEOGRAPH
	"97 8D	7D61",	#CJK UNIFIED IDEOGRAPH
	"97 8E	843D",	#CJK UNIFIED IDEOGRAPH
	"97 8F	916A",	#CJK UNIFIED IDEOGRAPH
	"97 90	4E71",	#CJK UNIFIED IDEOGRAPH
	"97 91	5375",	#CJK UNIFIED IDEOGRAPH
	"97 92	5D50",	#CJK UNIFIED IDEOGRAPH
	"97 93	6B04",	#CJK UNIFIED IDEOGRAPH
	"97 94	6FEB",	#CJK UNIFIED IDEOGRAPH
	"97 95	85CD",	#CJK UNIFIED IDEOGRAPH
	"97 96	862D",	#CJK UNIFIED IDEOGRAPH
	"97 97	89A7",	#CJK UNIFIED IDEOGRAPH
	"97 98	5229",	#CJK UNIFIED IDEOGRAPH
	"97 99	540F",	#CJK UNIFIED IDEOGRAPH
	"97 9A	5C65",	#CJK UNIFIED IDEOGRAPH
	"97 9B	674E",	#CJK UNIFIED IDEOGRAPH
	"97 9C	68A8",	#CJK UNIFIED IDEOGRAPH
	"97 9D	7406",	#CJK UNIFIED IDEOGRAPH
	"97 9E	7483",	#CJK UNIFIED IDEOGRAPH
	"97 9F	75E2",	#CJK UNIFIED IDEOGRAPH
	"97 A0	88CF",	#CJK UNIFIED IDEOGRAPH
	"97 A1	88E1",	#CJK UNIFIED IDEOGRAPH
	"97 A2	91CC",	#CJK UNIFIED IDEOGRAPH
	"97 A3	96E2",	#CJK UNIFIED IDEOGRAPH
	"97 A4	9678",	#CJK UNIFIED IDEOGRAPH
	"97 A5	5F8B",	#CJK UNIFIED IDEOGRAPH
	"97 A6	7387",	#CJK UNIFIED IDEOGRAPH
	"97 A7	7ACB",	#CJK UNIFIED IDEOGRAPH
	"97 A8	844E",	#CJK UNIFIED IDEOGRAPH
	"97 A9	63A0",	#CJK UNIFIED IDEOGRAPH
	"97 AA	7565",	#CJK UNIFIED IDEOGRAPH
	"97 AB	5289",	#CJK UNIFIED IDEOGRAPH
	"97 AC	6D41",	#CJK UNIFIED IDEOGRAPH
	"97 AD	6E9C",	#CJK UNIFIED IDEOGRAPH
	"97 AE	7409",	#CJK UNIFIED IDEOGRAPH
	"97 AF	7559",	#CJK UNIFIED IDEOGRAPH
	"97 B0	786B",	#CJK UNIFIED IDEOGRAPH
	"97 B1	7C92",	#CJK UNIFIED IDEOGRAPH
	"97 B2	9686",	#CJK UNIFIED IDEOGRAPH
	"97 B3	7ADC",	#CJK UNIFIED IDEOGRAPH
	"97 B4	9F8D",	#CJK UNIFIED IDEOGRAPH
	"97 B5	4FB6",	#CJK UNIFIED IDEOGRAPH
	"97 B6	616E",	#CJK UNIFIED IDEOGRAPH
	"97 B7	65C5",	#CJK UNIFIED IDEOGRAPH
	"97 B8	865C",	#CJK UNIFIED IDEOGRAPH
	"97 B9	4E86",	#CJK UNIFIED IDEOGRAPH
	"97 BA	4EAE",	#CJK UNIFIED IDEOGRAPH
	"97 BB	50DA",	#CJK UNIFIED IDEOGRAPH
	"97 BC	4E21",	#CJK UNIFIED IDEOGRAPH
	"97 BD	51CC",	#CJK UNIFIED IDEOGRAPH
	"97 BE	5BEE",	#CJK UNIFIED IDEOGRAPH
	"97 BF	6599",	#CJK UNIFIED IDEOGRAPH
	"97 C0	6881",	#CJK UNIFIED IDEOGRAPH
	"97 C1	6DBC",	#CJK UNIFIED IDEOGRAPH
	"97 C2	731F",	#CJK UNIFIED IDEOGRAPH
	"97 C3	7642",	#CJK UNIFIED IDEOGRAPH
	"97 C4	77AD",	#CJK UNIFIED IDEOGRAPH
	"97 C5	7A1C",	#CJK UNIFIED IDEOGRAPH
	"97 C6	7CE7",	#CJK UNIFIED IDEOGRAPH
	"97 C7	826F",	#CJK UNIFIED IDEOGRAPH
	"97 C8	8AD2",	#CJK UNIFIED IDEOGRAPH
	"97 C9	907C",	#CJK UNIFIED IDEOGRAPH
	"97 CA	91CF",	#CJK UNIFIED IDEOGRAPH
	"97 CB	9675",	#CJK UNIFIED IDEOGRAPH
	"97 CC	9818",	#CJK UNIFIED IDEOGRAPH
	"97 CD	529B",	#CJK UNIFIED IDEOGRAPH
	"97 CE	7DD1",	#CJK UNIFIED IDEOGRAPH
	"97 CF	502B",	#CJK UNIFIED IDEOGRAPH
	"97 D0	5398",	#CJK UNIFIED IDEOGRAPH
	"97 D1	6797",	#CJK UNIFIED IDEOGRAPH
	"97 D2	6DCB",	#CJK UNIFIED IDEOGRAPH
	"97 D3	71D0",	#CJK UNIFIED IDEOGRAPH
	"97 D4	7433",	#CJK UNIFIED IDEOGRAPH
	"97 D5	81E8",	#CJK UNIFIED IDEOGRAPH
	"97 D6	8F2A",	#CJK UNIFIED IDEOGRAPH
	"97 D7	96A3",	#CJK UNIFIED IDEOGRAPH
	"97 D8	9C57",	#CJK UNIFIED IDEOGRAPH
	"97 D9	9E9F",	#CJK UNIFIED IDEOGRAPH
	"97 DA	7460",	#CJK UNIFIED IDEOGRAPH
	"97 DB	5841",	#CJK UNIFIED IDEOGRAPH
	"97 DC	6D99",	#CJK UNIFIED IDEOGRAPH
	"97 DD	7D2F",	#CJK UNIFIED IDEOGRAPH
	"97 DE	985E",	#CJK UNIFIED IDEOGRAPH
	"97 DF	4EE4",	#CJK UNIFIED IDEOGRAPH
	"97 E0	4F36",	#CJK UNIFIED IDEOGRAPH
	"97 E1	4F8B",	#CJK UNIFIED IDEOGRAPH
	"97 E2	51B7",	#CJK UNIFIED IDEOGRAPH
	"97 E3	52B1",	#CJK UNIFIED IDEOGRAPH
	"97 E4	5DBA",	#CJK UNIFIED IDEOGRAPH
	"97 E5	601C",	#CJK UNIFIED IDEOGRAPH
	"97 E6	73B2",	#CJK UNIFIED IDEOGRAPH
	"97 E7	793C",	#CJK UNIFIED IDEOGRAPH
	"97 E8	82D3",	#CJK UNIFIED IDEOGRAPH
	"97 E9	9234",	#CJK UNIFIED IDEOGRAPH
	"97 EA	96B7",	#CJK UNIFIED IDEOGRAPH
	"97 EB	96F6",	#CJK UNIFIED IDEOGRAPH
	"97 EC	970A",	#CJK UNIFIED IDEOGRAPH
	"97 ED	9E97",	#CJK UNIFIED IDEOGRAPH
	"97 EE	9F62",	#CJK UNIFIED IDEOGRAPH
	"97 EF	66A6",	#CJK UNIFIED IDEOGRAPH
	"97 F0	6B74",	#CJK UNIFIED IDEOGRAPH
	"97 F1	5217",	#CJK UNIFIED IDEOGRAPH
	"97 F2	52A3",	#CJK UNIFIED IDEOGRAPH
	"97 F3	70C8",	#CJK UNIFIED IDEOGRAPH
	"97 F4	88C2",	#CJK UNIFIED IDEOGRAPH
	"97 F5	5EC9",	#CJK UNIFIED IDEOGRAPH
	"97 F6	604B",	#CJK UNIFIED IDEOGRAPH
	"97 F7	6190",	#CJK UNIFIED IDEOGRAPH
	"97 F8	6F23",	#CJK UNIFIED IDEOGRAPH
	"97 F9	7149",	#CJK UNIFIED IDEOGRAPH
	"97 FA	7C3E",	#CJK UNIFIED IDEOGRAPH
	"97 FB	7DF4",	#CJK UNIFIED IDEOGRAPH
	"97 FC	806F",	#CJK UNIFIED IDEOGRAPH
	"98 40	84EE",	#CJK UNIFIED IDEOGRAPH
	"98 41	9023",	#CJK UNIFIED IDEOGRAPH
	"98 42	932C",	#CJK UNIFIED IDEOGRAPH
	"98 43	5442",	#CJK UNIFIED IDEOGRAPH
	"98 44	9B6F",	#CJK UNIFIED IDEOGRAPH
	"98 45	6AD3",	#CJK UNIFIED IDEOGRAPH
	"98 46	7089",	#CJK UNIFIED IDEOGRAPH
	"98 47	8CC2",	#CJK UNIFIED IDEOGRAPH
	"98 48	8DEF",	#CJK UNIFIED IDEOGRAPH
	"98 49	9732",	#CJK UNIFIED IDEOGRAPH
	"98 4A	52B4",	#CJK UNIFIED IDEOGRAPH
	"98 4B	5A41",	#CJK UNIFIED IDEOGRAPH
	"98 4C	5ECA",	#CJK UNIFIED IDEOGRAPH
	"98 4D	5F04",	#CJK UNIFIED IDEOGRAPH
	"98 4E	6717",	#CJK UNIFIED IDEOGRAPH
	"98 4F	697C",	#CJK UNIFIED IDEOGRAPH
	"98 50	6994",	#CJK UNIFIED IDEOGRAPH
	"98 51	6D6A",	#CJK UNIFIED IDEOGRAPH
	"98 52	6F0F",	#CJK UNIFIED IDEOGRAPH
	"98 53	7262",	#CJK UNIFIED IDEOGRAPH
	"98 54	72FC",	#CJK UNIFIED IDEOGRAPH
	"98 55	7BED",	#CJK UNIFIED IDEOGRAPH
	"98 56	8001",	#CJK UNIFIED IDEOGRAPH
	"98 57	807E",	#CJK UNIFIED IDEOGRAPH
	"98 58	874B",	#CJK UNIFIED IDEOGRAPH
	"98 59	90CE",	#CJK UNIFIED IDEOGRAPH
	"98 5A	516D",	#CJK UNIFIED IDEOGRAPH
	"98 5B	9E93",	#CJK UNIFIED IDEOGRAPH
	"98 5C	7984",	#CJK UNIFIED IDEOGRAPH
	"98 5D	808B",	#CJK UNIFIED IDEOGRAPH
	"98 5E	9332",	#CJK UNIFIED IDEOGRAPH
	"98 5F	8AD6",	#CJK UNIFIED IDEOGRAPH
	"98 60	502D",	#CJK UNIFIED IDEOGRAPH
	"98 61	548C",	#CJK UNIFIED IDEOGRAPH
	"98 62	8A71",	#CJK UNIFIED IDEOGRAPH
	"98 63	6B6A",	#CJK UNIFIED IDEOGRAPH
	"98 64	8CC4",	#CJK UNIFIED IDEOGRAPH
	"98 65	8107",	#CJK UNIFIED IDEOGRAPH
	"98 66	60D1",	#CJK UNIFIED IDEOGRAPH
	"98 67	67A0",	#CJK UNIFIED IDEOGRAPH
	"98 68	9DF2",	#CJK UNIFIED IDEOGRAPH
	"98 69	4E99",	#CJK UNIFIED IDEOGRAPH
	"98 6A	4E98",	#CJK UNIFIED IDEOGRAPH
	"98 6B	9C10",	#CJK UNIFIED IDEOGRAPH
	"98 6C	8A6B",	#CJK UNIFIED IDEOGRAPH
	"98 6D	85C1",	#CJK UNIFIED IDEOGRAPH
	"98 6E	8568",	#CJK UNIFIED IDEOGRAPH
	"98 6F	6900",	#CJK UNIFIED IDEOGRAPH
	"98 70	6E7E",	#CJK UNIFIED IDEOGRAPH
	"98 71	7897",	#CJK UNIFIED IDEOGRAPH
	"98 72	8155",	#CJK UNIFIED IDEOGRAPH
	"98 9F	5F0C",	#CJK UNIFIED IDEOGRAPH
	"98 A0	4E10",	#CJK UNIFIED IDEOGRAPH
	"98 A1	4E15",	#CJK UNIFIED IDEOGRAPH
	"98 A2	4E2A",	#CJK UNIFIED IDEOGRAPH
	"98 A3	4E31",	#CJK UNIFIED IDEOGRAPH
	"98 A4	4E36",	#CJK UNIFIED IDEOGRAPH
	"98 A5	4E3C",	#CJK UNIFIED IDEOGRAPH
	"98 A6	4E3F",	#CJK UNIFIED IDEOGRAPH
	"98 A7	4E42",	#CJK UNIFIED IDEOGRAPH
	"98 A8	4E56",	#CJK UNIFIED IDEOGRAPH
	"98 A9	4E58",	#CJK UNIFIED IDEOGRAPH
	"98 AA	4E82",	#CJK UNIFIED IDEOGRAPH
	"98 AB	4E85",	#CJK UNIFIED IDEOGRAPH
	"98 AC	8C6B",	#CJK UNIFIED IDEOGRAPH
	"98 AD	4E8A",	#CJK UNIFIED IDEOGRAPH
	"98 AE	8212",	#CJK UNIFIED IDEOGRAPH
	"98 AF	5F0D",	#CJK UNIFIED IDEOGRAPH
	"98 B0	4E8E",	#CJK UNIFIED IDEOGRAPH
	"98 B1	4E9E",	#CJK UNIFIED IDEOGRAPH
	"98 B2	4E9F",	#CJK UNIFIED IDEOGRAPH
	"98 B3	4EA0",	#CJK UNIFIED IDEOGRAPH
	"98 B4	4EA2",	#CJK UNIFIED IDEOGRAPH
	"98 B5	4EB0",	#CJK UNIFIED IDEOGRAPH
	"98 B6	4EB3",	#CJK UNIFIED IDEOGRAPH
	"98 B7	4EB6",	#CJK UNIFIED IDEOGRAPH
	"98 B8	4ECE",	#CJK UNIFIED IDEOGRAPH
	"98 B9	4ECD",	#CJK UNIFIED IDEOGRAPH
	"98 BA	4EC4",	#CJK UNIFIED IDEOGRAPH
	"98 BB	4EC6",	#CJK UNIFIED IDEOGRAPH
	"98 BC	4EC2",	#CJK UNIFIED IDEOGRAPH
	"98 BD	4ED7",	#CJK UNIFIED IDEOGRAPH
	"98 BE	4EDE",	#CJK UNIFIED IDEOGRAPH
	"98 BF	4EED",	#CJK UNIFIED IDEOGRAPH
	"98 C0	4EDF",	#CJK UNIFIED IDEOGRAPH
	"98 C1	4EF7",	#CJK UNIFIED IDEOGRAPH
	"98 C2	4F09",	#CJK UNIFIED IDEOGRAPH
	"98 C3	4F5A",	#CJK UNIFIED IDEOGRAPH
	"98 C4	4F30",	#CJK UNIFIED IDEOGRAPH
	"98 C5	4F5B",	#CJK UNIFIED IDEOGRAPH
	"98 C6	4F5D",	#CJK UNIFIED IDEOGRAPH
	"98 C7	4F57",	#CJK UNIFIED IDEOGRAPH
	"98 C8	4F47",	#CJK UNIFIED IDEOGRAPH
	"98 C9	4F76",	#CJK UNIFIED IDEOGRAPH
	"98 CA	4F88",	#CJK UNIFIED IDEOGRAPH
	"98 CB	4F8F",	#CJK UNIFIED IDEOGRAPH
	"98 CC	4F98",	#CJK UNIFIED IDEOGRAPH
	"98 CD	4F7B",	#CJK UNIFIED IDEOGRAPH
	"98 CE	4F69",	#CJK UNIFIED IDEOGRAPH
	"98 CF	4F70",	#CJK UNIFIED IDEOGRAPH
	"98 D0	4F91",	#CJK UNIFIED IDEOGRAPH
	"98 D1	4F6F",	#CJK UNIFIED IDEOGRAPH
	"98 D2	4F86",	#CJK UNIFIED IDEOGRAPH
	"98 D3	4F96",	#CJK UNIFIED IDEOGRAPH
	"98 D4	5118",	#CJK UNIFIED IDEOGRAPH
	"98 D5	4FD4",	#CJK UNIFIED IDEOGRAPH
	"98 D6	4FDF",	#CJK UNIFIED IDEOGRAPH
	"98 D7	4FCE",	#CJK UNIFIED IDEOGRAPH
	"98 D8	4FD8",	#CJK UNIFIED IDEOGRAPH
	"98 D9	4FDB",	#CJK UNIFIED IDEOGRAPH
	"98 DA	4FD1",	#CJK UNIFIED IDEOGRAPH
	"98 DB	4FDA",	#CJK UNIFIED IDEOGRAPH
	"98 DC	4FD0",	#CJK UNIFIED IDEOGRAPH
	"98 DD	4FE4",	#CJK UNIFIED IDEOGRAPH
	"98 DE	4FE5",	#CJK UNIFIED IDEOGRAPH
	"98 DF	501A",	#CJK UNIFIED IDEOGRAPH
	"98 E0	5028",	#CJK UNIFIED IDEOGRAPH
	"98 E1	5014",	#CJK UNIFIED IDEOGRAPH
	"98 E2	502A",	#CJK UNIFIED IDEOGRAPH
	"98 E3	5025",	#CJK UNIFIED IDEOGRAPH
	"98 E4	5005",	#CJK UNIFIED IDEOGRAPH
	"98 E5	4F1C",	#CJK UNIFIED IDEOGRAPH
	"98 E6	4FF6",	#CJK UNIFIED IDEOGRAPH
	"98 E7	5021",	#CJK UNIFIED IDEOGRAPH
	"98 E8	5029",	#CJK UNIFIED IDEOGRAPH
	"98 E9	502C",	#CJK UNIFIED IDEOGRAPH
	"98 EA	4FFE",	#CJK UNIFIED IDEOGRAPH
	"98 EB	4FEF",	#CJK UNIFIED IDEOGRAPH
	"98 EC	5011",	#CJK UNIFIED IDEOGRAPH
	"98 ED	5006",	#CJK UNIFIED IDEOGRAPH
	"98 EE	5043",	#CJK UNIFIED IDEOGRAPH
	"98 EF	5047",	#CJK UNIFIED IDEOGRAPH
	"98 F0	6703",	#CJK UNIFIED IDEOGRAPH
	"98 F1	5055",	#CJK UNIFIED IDEOGRAPH
	"98 F2	5050",	#CJK UNIFIED IDEOGRAPH
	"98 F3	5048",	#CJK UNIFIED IDEOGRAPH
	"98 F4	505A",	#CJK UNIFIED IDEOGRAPH
	"98 F5	5056",	#CJK UNIFIED IDEOGRAPH
	"98 F6	506C",	#CJK UNIFIED IDEOGRAPH
	"98 F7	5078",	#CJK UNIFIED IDEOGRAPH
	"98 F8	5080",	#CJK UNIFIED IDEOGRAPH
	"98 F9	509A",	#CJK UNIFIED IDEOGRAPH
	"98 FA	5085",	#CJK UNIFIED IDEOGRAPH
	"98 FB	50B4",	#CJK UNIFIED IDEOGRAPH
	"98 FC	50B2",	#CJK UNIFIED IDEOGRAPH
	"99 40	50C9",	#CJK UNIFIED IDEOGRAPH
	"99 41	50CA",	#CJK UNIFIED IDEOGRAPH
	"99 42	50B3",	#CJK UNIFIED IDEOGRAPH
	"99 43	50C2",	#CJK UNIFIED IDEOGRAPH
	"99 44	50D6",	#CJK UNIFIED IDEOGRAPH
	"99 45	50DE",	#CJK UNIFIED IDEOGRAPH
	"99 46	50E5",	#CJK UNIFIED IDEOGRAPH
	"99 47	50ED",	#CJK UNIFIED IDEOGRAPH
	"99 48	50E3",	#CJK UNIFIED IDEOGRAPH
	"99 49	50EE",	#CJK UNIFIED IDEOGRAPH
	"99 4A	50F9",	#CJK UNIFIED IDEOGRAPH
	"99 4B	50F5",	#CJK UNIFIED IDEOGRAPH
	"99 4C	5109",	#CJK UNIFIED IDEOGRAPH
	"99 4D	5101",	#CJK UNIFIED IDEOGRAPH
	"99 4E	5102",	#CJK UNIFIED IDEOGRAPH
	"99 4F	5116",	#CJK UNIFIED IDEOGRAPH
	"99 50	5115",	#CJK UNIFIED IDEOGRAPH
	"99 51	5114",	#CJK UNIFIED IDEOGRAPH
	"99 52	511A",	#CJK UNIFIED IDEOGRAPH
	"99 53	5121",	#CJK UNIFIED IDEOGRAPH
	"99 54	513A",	#CJK UNIFIED IDEOGRAPH
	"99 55	5137",	#CJK UNIFIED IDEOGRAPH
	"99 56	513C",	#CJK UNIFIED IDEOGRAPH
	"99 57	513B",	#CJK UNIFIED IDEOGRAPH
	"99 58	513F",	#CJK UNIFIED IDEOGRAPH
	"99 59	5140",	#CJK UNIFIED IDEOGRAPH
	"99 5A	5152",	#CJK UNIFIED IDEOGRAPH
	"99 5B	514C",	#CJK UNIFIED IDEOGRAPH
	"99 5C	5154",	#CJK UNIFIED IDEOGRAPH
	"99 5D	5162",	#CJK UNIFIED IDEOGRAPH
	"99 5E	7AF8",	#CJK UNIFIED IDEOGRAPH
	"99 5F	5169",	#CJK UNIFIED IDEOGRAPH
	"99 60	516A",	#CJK UNIFIED IDEOGRAPH
	"99 61	516E",	#CJK UNIFIED IDEOGRAPH
	"99 62	5180",	#CJK UNIFIED IDEOGRAPH
	"99 63	5182",	#CJK UNIFIED IDEOGRAPH
	"99 64	56D8",	#CJK UNIFIED IDEOGRAPH
	"99 65	518C",	#CJK UNIFIED IDEOGRAPH
	"99 66	5189",	#CJK UNIFIED IDEOGRAPH
	"99 67	518F",	#CJK UNIFIED IDEOGRAPH
	"99 68	5191",	#CJK UNIFIED IDEOGRAPH
	"99 69	5193",	#CJK UNIFIED IDEOGRAPH
	"99 6A	5195",	#CJK UNIFIED IDEOGRAPH
	"99 6B	5196",	#CJK UNIFIED IDEOGRAPH
	"99 6C	51A4",	#CJK UNIFIED IDEOGRAPH
	"99 6D	51A6",	#CJK UNIFIED IDEOGRAPH
	"99 6E	51A2",	#CJK UNIFIED IDEOGRAPH
	"99 6F	51A9",	#CJK UNIFIED IDEOGRAPH
	"99 70	51AA",	#CJK UNIFIED IDEOGRAPH
	"99 71	51AB",	#CJK UNIFIED IDEOGRAPH
	"99 72	51B3",	#CJK UNIFIED IDEOGRAPH
	"99 73	51B1",	#CJK UNIFIED IDEOGRAPH
	"99 74	51B2",	#CJK UNIFIED IDEOGRAPH
	"99 75	51B0",	#CJK UNIFIED IDEOGRAPH
	"99 76	51B5",	#CJK UNIFIED IDEOGRAPH
	"99 77	51BD",	#CJK UNIFIED IDEOGRAPH
	"99 78	51C5",	#CJK UNIFIED IDEOGRAPH
	"99 79	51C9",	#CJK UNIFIED IDEOGRAPH
	"99 7A	51DB",	#CJK UNIFIED IDEOGRAPH
	"99 7B	51E0",	#CJK UNIFIED IDEOGRAPH
	"99 7C	8655",	#CJK UNIFIED IDEOGRAPH
	"99 7D	51E9",	#CJK UNIFIED IDEOGRAPH
	"99 7E	51ED",	#CJK UNIFIED IDEOGRAPH
	"99 80	51F0",	#CJK UNIFIED IDEOGRAPH
	"99 81	51F5",	#CJK UNIFIED IDEOGRAPH
	"99 82	51FE",	#CJK UNIFIED IDEOGRAPH
	"99 83	5204",	#CJK UNIFIED IDEOGRAPH
	"99 84	520B",	#CJK UNIFIED IDEOGRAPH
	"99 85	5214",	#CJK UNIFIED IDEOGRAPH
	"99 86	520E",	#CJK UNIFIED IDEOGRAPH
	"99 87	5227",	#CJK UNIFIED IDEOGRAPH
	"99 88	522A",	#CJK UNIFIED IDEOGRAPH
	"99 89	522E",	#CJK UNIFIED IDEOGRAPH
	"99 8A	5233",	#CJK UNIFIED IDEOGRAPH
	"99 8B	5239",	#CJK UNIFIED IDEOGRAPH
	"99 8C	524F",	#CJK UNIFIED IDEOGRAPH
	"99 8D	5244",	#CJK UNIFIED IDEOGRAPH
	"99 8E	524B",	#CJK UNIFIED IDEOGRAPH
	"99 8F	524C",	#CJK UNIFIED IDEOGRAPH
	"99 90	525E",	#CJK UNIFIED IDEOGRAPH
	"99 91	5254",	#CJK UNIFIED IDEOGRAPH
	"99 92	526A",	#CJK UNIFIED IDEOGRAPH
	"99 93	5274",	#CJK UNIFIED IDEOGRAPH
	"99 94	5269",	#CJK UNIFIED IDEOGRAPH
	"99 95	5273",	#CJK UNIFIED IDEOGRAPH
	"99 96	527F",	#CJK UNIFIED IDEOGRAPH
	"99 97	527D",	#CJK UNIFIED IDEOGRAPH
	"99 98	528D",	#CJK UNIFIED IDEOGRAPH
	"99 99	5294",	#CJK UNIFIED IDEOGRAPH
	"99 9A	5292",	#CJK UNIFIED IDEOGRAPH
	"99 9B	5271",	#CJK UNIFIED IDEOGRAPH
	"99 9C	5288",	#CJK UNIFIED IDEOGRAPH
	"99 9D	5291",	#CJK UNIFIED IDEOGRAPH
	"99 9E	8FA8",	#CJK UNIFIED IDEOGRAPH
	"99 9F	8FA7",	#CJK UNIFIED IDEOGRAPH
	"99 A0	52AC",	#CJK UNIFIED IDEOGRAPH
	"99 A1	52AD",	#CJK UNIFIED IDEOGRAPH
	"99 A2	52BC",	#CJK UNIFIED IDEOGRAPH
	"99 A3	52B5",	#CJK UNIFIED IDEOGRAPH
	"99 A4	52C1",	#CJK UNIFIED IDEOGRAPH
	"99 A5	52CD",	#CJK UNIFIED IDEOGRAPH
	"99 A6	52D7",	#CJK UNIFIED IDEOGRAPH
	"99 A7	52DE",	#CJK UNIFIED IDEOGRAPH
	"99 A8	52E3",	#CJK UNIFIED IDEOGRAPH
	"99 A9	52E6",	#CJK UNIFIED IDEOGRAPH
	"99 AA	98ED",	#CJK UNIFIED IDEOGRAPH
	"99 AB	52E0",	#CJK UNIFIED IDEOGRAPH
	"99 AC	52F3",	#CJK UNIFIED IDEOGRAPH
	"99 AD	52F5",	#CJK UNIFIED IDEOGRAPH
	"99 AE	52F8",	#CJK UNIFIED IDEOGRAPH
	"99 AF	52F9",	#CJK UNIFIED IDEOGRAPH
	"99 B0	5306",	#CJK UNIFIED IDEOGRAPH
	"99 B1	5308",	#CJK UNIFIED IDEOGRAPH
	"99 B2	7538",	#CJK UNIFIED IDEOGRAPH
	"99 B3	530D",	#CJK UNIFIED IDEOGRAPH
	"99 B4	5310",	#CJK UNIFIED IDEOGRAPH
	"99 B5	530F",	#CJK UNIFIED IDEOGRAPH
	"99 B6	5315",	#CJK UNIFIED IDEOGRAPH
	"99 B7	531A",	#CJK UNIFIED IDEOGRAPH
	"99 B8	5323",	#CJK UNIFIED IDEOGRAPH
	"99 B9	532F",	#CJK UNIFIED IDEOGRAPH
	"99 BA	5331",	#CJK UNIFIED IDEOGRAPH
	"99 BB	5333",	#CJK UNIFIED IDEOGRAPH
	"99 BC	5338",	#CJK UNIFIED IDEOGRAPH
	"99 BD	5340",	#CJK UNIFIED IDEOGRAPH
	"99 BE	5346",	#CJK UNIFIED IDEOGRAPH
	"99 BF	5345",	#CJK UNIFIED IDEOGRAPH
	"99 C0	4E17",	#CJK UNIFIED IDEOGRAPH
	"99 C1	5349",	#CJK UNIFIED IDEOGRAPH
	"99 C2	534D",	#CJK UNIFIED IDEOGRAPH
	"99 C3	51D6",	#CJK UNIFIED IDEOGRAPH
	"99 C4	535E",	#CJK UNIFIED IDEOGRAPH
	"99 C5	5369",	#CJK UNIFIED IDEOGRAPH
	"99 C6	536E",	#CJK UNIFIED IDEOGRAPH
	"99 C7	5918",	#CJK UNIFIED IDEOGRAPH
	"99 C8	537B",	#CJK UNIFIED IDEOGRAPH
	"99 C9	5377",	#CJK UNIFIED IDEOGRAPH
	"99 CA	5382",	#CJK UNIFIED IDEOGRAPH
	"99 CB	5396",	#CJK UNIFIED IDEOGRAPH
	"99 CC	53A0",	#CJK UNIFIED IDEOGRAPH
	"99 CD	53A6",	#CJK UNIFIED IDEOGRAPH
	"99 CE	53A5",	#CJK UNIFIED IDEOGRAPH
	"99 CF	53AE",	#CJK UNIFIED IDEOGRAPH
	"99 D0	53B0",	#CJK UNIFIED IDEOGRAPH
	"99 D1	53B6",	#CJK UNIFIED IDEOGRAPH
	"99 D2	53C3",	#CJK UNIFIED IDEOGRAPH
	"99 D3	7C12",	#CJK UNIFIED IDEOGRAPH
	"99 D4	96D9",	#CJK UNIFIED IDEOGRAPH
	"99 D5	53DF",	#CJK UNIFIED IDEOGRAPH
	"99 D6	66FC",	#CJK UNIFIED IDEOGRAPH
	"99 D7	71EE",	#CJK UNIFIED IDEOGRAPH
	"99 D8	53EE",	#CJK UNIFIED IDEOGRAPH
	"99 D9	53E8",	#CJK UNIFIED IDEOGRAPH
	"99 DA	53ED",	#CJK UNIFIED IDEOGRAPH
	"99 DB	53FA",	#CJK UNIFIED IDEOGRAPH
	"99 DC	5401",	#CJK UNIFIED IDEOGRAPH
	"99 DD	543D",	#CJK UNIFIED IDEOGRAPH
	"99 DE	5440",	#CJK UNIFIED IDEOGRAPH
	"99 DF	542C",	#CJK UNIFIED IDEOGRAPH
	"99 E0	542D",	#CJK UNIFIED IDEOGRAPH
	"99 E1	543C",	#CJK UNIFIED IDEOGRAPH
	"99 E2	542E",	#CJK UNIFIED IDEOGRAPH
	"99 E3	5436",	#CJK UNIFIED IDEOGRAPH
	"99 E4	5429",	#CJK UNIFIED IDEOGRAPH
	"99 E5	541D",	#CJK UNIFIED IDEOGRAPH
	"99 E6	544E",	#CJK UNIFIED IDEOGRAPH
	"99 E7	548F",	#CJK UNIFIED IDEOGRAPH
	"99 E8	5475",	#CJK UNIFIED IDEOGRAPH
	"99 E9	548E",	#CJK UNIFIED IDEOGRAPH
	"99 EA	545F",	#CJK UNIFIED IDEOGRAPH
	"99 EB	5471",	#CJK UNIFIED IDEOGRAPH
	"99 EC	5477",	#CJK UNIFIED IDEOGRAPH
	"99 ED	5470",	#CJK UNIFIED IDEOGRAPH
	"99 EE	5492",	#CJK UNIFIED IDEOGRAPH
	"99 EF	547B",	#CJK UNIFIED IDEOGRAPH
	"99 F0	5480",	#CJK UNIFIED IDEOGRAPH
	"99 F1	5476",	#CJK UNIFIED IDEOGRAPH
	"99 F2	5484",	#CJK UNIFIED IDEOGRAPH
	"99 F3	5490",	#CJK UNIFIED IDEOGRAPH
	"99 F4	5486",	#CJK UNIFIED IDEOGRAPH
	"99 F5	54C7",	#CJK UNIFIED IDEOGRAPH
	"99 F6	54A2",	#CJK UNIFIED IDEOGRAPH
	"99 F7	54B8",	#CJK UNIFIED IDEOGRAPH
	"99 F8	54A5",	#CJK UNIFIED IDEOGRAPH
	"99 F9	54AC",	#CJK UNIFIED IDEOGRAPH
	"99 FA	54C4",	#CJK UNIFIED IDEOGRAPH
	"99 FB	54C8",	#CJK UNIFIED IDEOGRAPH
	"99 FC	54A8",	#CJK UNIFIED IDEOGRAPH
	"9A 40	54AB",	#CJK UNIFIED IDEOGRAPH
	"9A 41	54C2",	#CJK UNIFIED IDEOGRAPH
	"9A 42	54A4",	#CJK UNIFIED IDEOGRAPH
	"9A 43	54BE",	#CJK UNIFIED IDEOGRAPH
	"9A 44	54BC",	#CJK UNIFIED IDEOGRAPH
	"9A 45	54D8",	#CJK UNIFIED IDEOGRAPH
	"9A 46	54E5",	#CJK UNIFIED IDEOGRAPH
	"9A 47	54E6",	#CJK UNIFIED IDEOGRAPH
	"9A 48	550F",	#CJK UNIFIED IDEOGRAPH
	"9A 49	5514",	#CJK UNIFIED IDEOGRAPH
	"9A 4A	54FD",	#CJK UNIFIED IDEOGRAPH
	"9A 4B	54EE",	#CJK UNIFIED IDEOGRAPH
	"9A 4C	54ED",	#CJK UNIFIED IDEOGRAPH
	"9A 4D	54FA",	#CJK UNIFIED IDEOGRAPH
	"9A 4E	54E2",	#CJK UNIFIED IDEOGRAPH
	"9A 4F	5539",	#CJK UNIFIED IDEOGRAPH
	"9A 50	5540",	#CJK UNIFIED IDEOGRAPH
	"9A 51	5563",	#CJK UNIFIED IDEOGRAPH
	"9A 52	554C",	#CJK UNIFIED IDEOGRAPH
	"9A 53	552E",	#CJK UNIFIED IDEOGRAPH
	"9A 54	555C",	#CJK UNIFIED IDEOGRAPH
	"9A 55	5545",	#CJK UNIFIED IDEOGRAPH
	"9A 56	5556",	#CJK UNIFIED IDEOGRAPH
	"9A 57	5557",	#CJK UNIFIED IDEOGRAPH
	"9A 58	5538",	#CJK UNIFIED IDEOGRAPH
	"9A 59	5533",	#CJK UNIFIED IDEOGRAPH
	"9A 5A	555D",	#CJK UNIFIED IDEOGRAPH
	"9A 5B	5599",	#CJK UNIFIED IDEOGRAPH
	"9A 5C	5580",	#CJK UNIFIED IDEOGRAPH
	"9A 5D	54AF",	#CJK UNIFIED IDEOGRAPH
	"9A 5E	558A",	#CJK UNIFIED IDEOGRAPH
	"9A 5F	559F",	#CJK UNIFIED IDEOGRAPH
	"9A 60	557B",	#CJK UNIFIED IDEOGRAPH
	"9A 61	557E",	#CJK UNIFIED IDEOGRAPH
	"9A 62	5598",	#CJK UNIFIED IDEOGRAPH
	"9A 63	559E",	#CJK UNIFIED IDEOGRAPH
	"9A 64	55AE",	#CJK UNIFIED IDEOGRAPH
	"9A 65	557C",	#CJK UNIFIED IDEOGRAPH
	"9A 66	5583",	#CJK UNIFIED IDEOGRAPH
	"9A 67	55A9",	#CJK UNIFIED IDEOGRAPH
	"9A 68	5587",	#CJK UNIFIED IDEOGRAPH
	"9A 69	55A8",	#CJK UNIFIED IDEOGRAPH
	"9A 6A	55DA",	#CJK UNIFIED IDEOGRAPH
	"9A 6B	55C5",	#CJK UNIFIED IDEOGRAPH
	"9A 6C	55DF",	#CJK UNIFIED IDEOGRAPH
	"9A 6D	55C4",	#CJK UNIFIED IDEOGRAPH
	"9A 6E	55DC",	#CJK UNIFIED IDEOGRAPH
	"9A 6F	55E4",	#CJK UNIFIED IDEOGRAPH
	"9A 70	55D4",	#CJK UNIFIED IDEOGRAPH
	"9A 71	5614",	#CJK UNIFIED IDEOGRAPH
	"9A 72	55F7",	#CJK UNIFIED IDEOGRAPH
	"9A 73	5616",	#CJK UNIFIED IDEOGRAPH
	"9A 74	55FE",	#CJK UNIFIED IDEOGRAPH
	"9A 75	55FD",	#CJK UNIFIED IDEOGRAPH
	"9A 76	561B",	#CJK UNIFIED IDEOGRAPH
	"9A 77	55F9",	#CJK UNIFIED IDEOGRAPH
	"9A 78	564E",	#CJK UNIFIED IDEOGRAPH
	"9A 79	5650",	#CJK UNIFIED IDEOGRAPH
	"9A 7A	71DF",	#CJK UNIFIED IDEOGRAPH
	"9A 7B	5634",	#CJK UNIFIED IDEOGRAPH
	"9A 7C	5636",	#CJK UNIFIED IDEOGRAPH
	"9A 7D	5632",	#CJK UNIFIED IDEOGRAPH
	"9A 7E	5638",	#CJK UNIFIED IDEOGRAPH
	"9A 80	566B",	#CJK UNIFIED IDEOGRAPH
	"9A 81	5664",	#CJK UNIFIED IDEOGRAPH
	"9A 82	562F",	#CJK UNIFIED IDEOGRAPH
	"9A 83	566C",	#CJK UNIFIED IDEOGRAPH
	"9A 84	566A",	#CJK UNIFIED IDEOGRAPH
	"9A 85	5686",	#CJK UNIFIED IDEOGRAPH
	"9A 86	5680",	#CJK UNIFIED IDEOGRAPH
	"9A 87	568A",	#CJK UNIFIED IDEOGRAPH
	"9A 88	56A0",	#CJK UNIFIED IDEOGRAPH
	"9A 89	5694",	#CJK UNIFIED IDEOGRAPH
	"9A 8A	568F",	#CJK UNIFIED IDEOGRAPH
	"9A 8B	56A5",	#CJK UNIFIED IDEOGRAPH
	"9A 8C	56AE",	#CJK UNIFIED IDEOGRAPH
	"9A 8D	56B6",	#CJK UNIFIED IDEOGRAPH
	"9A 8E	56B4",	#CJK UNIFIED IDEOGRAPH
	"9A 8F	56C2",	#CJK UNIFIED IDEOGRAPH
	"9A 90	56BC",	#CJK UNIFIED IDEOGRAPH
	"9A 91	56C1",	#CJK UNIFIED IDEOGRAPH
	"9A 92	56C3",	#CJK UNIFIED IDEOGRAPH
	"9A 93	56C0",	#CJK UNIFIED IDEOGRAPH
	"9A 94	56C8",	#CJK UNIFIED IDEOGRAPH
	"9A 95	56CE",	#CJK UNIFIED IDEOGRAPH
	"9A 96	56D1",	#CJK UNIFIED IDEOGRAPH
	"9A 97	56D3",	#CJK UNIFIED IDEOGRAPH
	"9A 98	56D7",	#CJK UNIFIED IDEOGRAPH
	"9A 99	56EE",	#CJK UNIFIED IDEOGRAPH
	"9A 9A	56F9",	#CJK UNIFIED IDEOGRAPH
	"9A 9B	5700",	#CJK UNIFIED IDEOGRAPH
	"9A 9C	56FF",	#CJK UNIFIED IDEOGRAPH
	"9A 9D	5704",	#CJK UNIFIED IDEOGRAPH
	"9A 9E	5709",	#CJK UNIFIED IDEOGRAPH
	"9A 9F	5708",	#CJK UNIFIED IDEOGRAPH
	"9A A0	570B",	#CJK UNIFIED IDEOGRAPH
	"9A A1	570D",	#CJK UNIFIED IDEOGRAPH
	"9A A2	5713",	#CJK UNIFIED IDEOGRAPH
	"9A A3	5718",	#CJK UNIFIED IDEOGRAPH
	"9A A4	5716",	#CJK UNIFIED IDEOGRAPH
	"9A A5	55C7",	#CJK UNIFIED IDEOGRAPH
	"9A A6	571C",	#CJK UNIFIED IDEOGRAPH
	"9A A7	5726",	#CJK UNIFIED IDEOGRAPH
	"9A A8	5737",	#CJK UNIFIED IDEOGRAPH
	"9A A9	5738",	#CJK UNIFIED IDEOGRAPH
	"9A AA	574E",	#CJK UNIFIED IDEOGRAPH
	"9A AB	573B",	#CJK UNIFIED IDEOGRAPH
	"9A AC	5740",	#CJK UNIFIED IDEOGRAPH
	"9A AD	574F",	#CJK UNIFIED IDEOGRAPH
	"9A AE	5769",	#CJK UNIFIED IDEOGRAPH
	"9A AF	57C0",	#CJK UNIFIED IDEOGRAPH
	"9A B0	5788",	#CJK UNIFIED IDEOGRAPH
	"9A B1	5761",	#CJK UNIFIED IDEOGRAPH
	"9A B2	577F",	#CJK UNIFIED IDEOGRAPH
	"9A B3	5789",	#CJK UNIFIED IDEOGRAPH
	"9A B4	5793",	#CJK UNIFIED IDEOGRAPH
	"9A B5	57A0",	#CJK UNIFIED IDEOGRAPH
	"9A B6	57B3",	#CJK UNIFIED IDEOGRAPH
	"9A B7	57A4",	#CJK UNIFIED IDEOGRAPH
	"9A B8	57AA",	#CJK UNIFIED IDEOGRAPH
	"9A B9	57B0",	#CJK UNIFIED IDEOGRAPH
	"9A BA	57C3",	#CJK UNIFIED IDEOGRAPH
	"9A BB	57C6",	#CJK UNIFIED IDEOGRAPH
	"9A BC	57D4",	#CJK UNIFIED IDEOGRAPH
	"9A BD	57D2",	#CJK UNIFIED IDEOGRAPH
	"9A BE	57D3",	#CJK UNIFIED IDEOGRAPH
	"9A BF	580A",	#CJK UNIFIED IDEOGRAPH
	"9A C0	57D6",	#CJK UNIFIED IDEOGRAPH
	"9A C1	57E3",	#CJK UNIFIED IDEOGRAPH
	"9A C2	580B",	#CJK UNIFIED IDEOGRAPH
	"9A C3	5819",	#CJK UNIFIED IDEOGRAPH
	"9A C4	581D",	#CJK UNIFIED IDEOGRAPH
	"9A C5	5872",	#CJK UNIFIED IDEOGRAPH
	"9A C6	5821",	#CJK UNIFIED IDEOGRAPH
	"9A C7	5862",	#CJK UNIFIED IDEOGRAPH
	"9A C8	584B",	#CJK UNIFIED IDEOGRAPH
	"9A C9	5870",	#CJK UNIFIED IDEOGRAPH
	"9A CA	6BC0",	#CJK UNIFIED IDEOGRAPH
	"9A CB	5852",	#CJK UNIFIED IDEOGRAPH
	"9A CC	583D",	#CJK UNIFIED IDEOGRAPH
	"9A CD	5879",	#CJK UNIFIED IDEOGRAPH
	"9A CE	5885",	#CJK UNIFIED IDEOGRAPH
	"9A CF	58B9",	#CJK UNIFIED IDEOGRAPH
	"9A D0	589F",	#CJK UNIFIED IDEOGRAPH
	"9A D1	58AB",	#CJK UNIFIED IDEOGRAPH
	"9A D2	58BA",	#CJK UNIFIED IDEOGRAPH
	"9A D3	58DE",	#CJK UNIFIED IDEOGRAPH
	"9A D4	58BB",	#CJK UNIFIED IDEOGRAPH
	"9A D5	58B8",	#CJK UNIFIED IDEOGRAPH
	"9A D6	58AE",	#CJK UNIFIED IDEOGRAPH
	"9A D7	58C5",	#CJK UNIFIED IDEOGRAPH
	"9A D8	58D3",	#CJK UNIFIED IDEOGRAPH
	"9A D9	58D1",	#CJK UNIFIED IDEOGRAPH
	"9A DA	58D7",	#CJK UNIFIED IDEOGRAPH
	"9A DB	58D9",	#CJK UNIFIED IDEOGRAPH
	"9A DC	58D8",	#CJK UNIFIED IDEOGRAPH
	"9A DD	58E5",	#CJK UNIFIED IDEOGRAPH
	"9A DE	58DC",	#CJK UNIFIED IDEOGRAPH
	"9A DF	58E4",	#CJK UNIFIED IDEOGRAPH
	"9A E0	58DF",	#CJK UNIFIED IDEOGRAPH
	"9A E1	58EF",	#CJK UNIFIED IDEOGRAPH
	"9A E2	58FA",	#CJK UNIFIED IDEOGRAPH
	"9A E3	58F9",	#CJK UNIFIED IDEOGRAPH
	"9A E4	58FB",	#CJK UNIFIED IDEOGRAPH
	"9A E5	58FC",	#CJK UNIFIED IDEOGRAPH
	"9A E6	58FD",	#CJK UNIFIED IDEOGRAPH
	"9A E7	5902",	#CJK UNIFIED IDEOGRAPH
	"9A E8	590A",	#CJK UNIFIED IDEOGRAPH
	"9A E9	5910",	#CJK UNIFIED IDEOGRAPH
	"9A EA	591B",	#CJK UNIFIED IDEOGRAPH
	"9A EB	68A6",	#CJK UNIFIED IDEOGRAPH
	"9A EC	5925",	#CJK UNIFIED IDEOGRAPH
	"9A ED	592C",	#CJK UNIFIED IDEOGRAPH
	"9A EE	592D",	#CJK UNIFIED IDEOGRAPH
	"9A EF	5932",	#CJK UNIFIED IDEOGRAPH
	"9A F0	5938",	#CJK UNIFIED IDEOGRAPH
	"9A F1	593E",	#CJK UNIFIED IDEOGRAPH
	"9A F2	7AD2",	#CJK UNIFIED IDEOGRAPH
	"9A F3	5955",	#CJK UNIFIED IDEOGRAPH
	"9A F4	5950",	#CJK UNIFIED IDEOGRAPH
	"9A F5	594E",	#CJK UNIFIED IDEOGRAPH
	"9A F6	595A",	#CJK UNIFIED IDEOGRAPH
	"9A F7	5958",	#CJK UNIFIED IDEOGRAPH
	"9A F8	5962",	#CJK UNIFIED IDEOGRAPH
	"9A F9	5960",	#CJK UNIFIED IDEOGRAPH
	"9A FA	5967",	#CJK UNIFIED IDEOGRAPH
	"9A FB	596C",	#CJK UNIFIED IDEOGRAPH
	"9A FC	5969",	#CJK UNIFIED IDEOGRAPH
	"9B 40	5978",	#CJK UNIFIED IDEOGRAPH
	"9B 41	5981",	#CJK UNIFIED IDEOGRAPH
	"9B 42	599D",	#CJK UNIFIED IDEOGRAPH
	"9B 43	4F5E",	#CJK UNIFIED IDEOGRAPH
	"9B 44	4FAB",	#CJK UNIFIED IDEOGRAPH
	"9B 45	59A3",	#CJK UNIFIED IDEOGRAPH
	"9B 46	59B2",	#CJK UNIFIED IDEOGRAPH
	"9B 47	59C6",	#CJK UNIFIED IDEOGRAPH
	"9B 48	59E8",	#CJK UNIFIED IDEOGRAPH
	"9B 49	59DC",	#CJK UNIFIED IDEOGRAPH
	"9B 4A	598D",	#CJK UNIFIED IDEOGRAPH
	"9B 4B	59D9",	#CJK UNIFIED IDEOGRAPH
	"9B 4C	59DA",	#CJK UNIFIED IDEOGRAPH
	"9B 4D	5A25",	#CJK UNIFIED IDEOGRAPH
	"9B 4E	5A1F",	#CJK UNIFIED IDEOGRAPH
	"9B 4F	5A11",	#CJK UNIFIED IDEOGRAPH
	"9B 50	5A1C",	#CJK UNIFIED IDEOGRAPH
	"9B 51	5A09",	#CJK UNIFIED IDEOGRAPH
	"9B 52	5A1A",	#CJK UNIFIED IDEOGRAPH
	"9B 53	5A40",	#CJK UNIFIED IDEOGRAPH
	"9B 54	5A6C",	#CJK UNIFIED IDEOGRAPH
	"9B 55	5A49",	#CJK UNIFIED IDEOGRAPH
	"9B 56	5A35",	#CJK UNIFIED IDEOGRAPH
	"9B 57	5A36",	#CJK UNIFIED IDEOGRAPH
	"9B 58	5A62",	#CJK UNIFIED IDEOGRAPH
	"9B 59	5A6A",	#CJK UNIFIED IDEOGRAPH
	"9B 5A	5A9A",	#CJK UNIFIED IDEOGRAPH
	"9B 5B	5ABC",	#CJK UNIFIED IDEOGRAPH
	"9B 5C	5ABE",	#CJK UNIFIED IDEOGRAPH
	"9B 5D	5ACB",	#CJK UNIFIED IDEOGRAPH
	"9B 5E	5AC2",	#CJK UNIFIED IDEOGRAPH
	"9B 5F	5ABD",	#CJK UNIFIED IDEOGRAPH
	"9B 60	5AE3",	#CJK UNIFIED IDEOGRAPH
	"9B 61	5AD7",	#CJK UNIFIED IDEOGRAPH
	"9B 62	5AE6",	#CJK UNIFIED IDEOGRAPH
	"9B 63	5AE9",	#CJK UNIFIED IDEOGRAPH
	"9B 64	5AD6",	#CJK UNIFIED IDEOGRAPH
	"9B 65	5AFA",	#CJK UNIFIED IDEOGRAPH
	"9B 66	5AFB",	#CJK UNIFIED IDEOGRAPH
	"9B 67	5B0C",	#CJK UNIFIED IDEOGRAPH
	"9B 68	5B0B",	#CJK UNIFIED IDEOGRAPH
	"9B 69	5B16",	#CJK UNIFIED IDEOGRAPH
	"9B 6A	5B32",	#CJK UNIFIED IDEOGRAPH
	"9B 6B	5AD0",	#CJK UNIFIED IDEOGRAPH
	"9B 6C	5B2A",	#CJK UNIFIED IDEOGRAPH
	"9B 6D	5B36",	#CJK UNIFIED IDEOGRAPH
	"9B 6E	5B3E",	#CJK UNIFIED IDEOGRAPH
	"9B 6F	5B43",	#CJK UNIFIED IDEOGRAPH
	"9B 70	5B45",	#CJK UNIFIED IDEOGRAPH
	"9B 71	5B40",	#CJK UNIFIED IDEOGRAPH
	"9B 72	5B51",	#CJK UNIFIED IDEOGRAPH
	"9B 73	5B55",	#CJK UNIFIED IDEOGRAPH
	"9B 74	5B5A",	#CJK UNIFIED IDEOGRAPH
	"9B 75	5B5B",	#CJK UNIFIED IDEOGRAPH
	"9B 76	5B65",	#CJK UNIFIED IDEOGRAPH
	"9B 77	5B69",	#CJK UNIFIED IDEOGRAPH
	"9B 78	5B70",	#CJK UNIFIED IDEOGRAPH
	"9B 79	5B73",	#CJK UNIFIED IDEOGRAPH
	"9B 7A	5B75",	#CJK UNIFIED IDEOGRAPH
	"9B 7B	5B78",	#CJK UNIFIED IDEOGRAPH
	"9B 7C	6588",	#CJK UNIFIED IDEOGRAPH
	"9B 7D	5B7A",	#CJK UNIFIED IDEOGRAPH
	"9B 7E	5B80",	#CJK UNIFIED IDEOGRAPH
	"9B 80	5B83",	#CJK UNIFIED IDEOGRAPH
	"9B 81	5BA6",	#CJK UNIFIED IDEOGRAPH
	"9B 82	5BB8",	#CJK UNIFIED IDEOGRAPH
	"9B 83	5BC3",	#CJK UNIFIED IDEOGRAPH
	"9B 84	5BC7",	#CJK UNIFIED IDEOGRAPH
	"9B 85	5BC9",	#CJK UNIFIED IDEOGRAPH
	"9B 86	5BD4",	#CJK UNIFIED IDEOGRAPH
	"9B 87	5BD0",	#CJK UNIFIED IDEOGRAPH
	"9B 88	5BE4",	#CJK UNIFIED IDEOGRAPH
	"9B 89	5BE6",	#CJK UNIFIED IDEOGRAPH
	"9B 8A	5BE2",	#CJK UNIFIED IDEOGRAPH
	"9B 8B	5BDE",	#CJK UNIFIED IDEOGRAPH
	"9B 8C	5BE5",	#CJK UNIFIED IDEOGRAPH
	"9B 8D	5BEB",	#CJK UNIFIED IDEOGRAPH
	"9B 8E	5BF0",	#CJK UNIFIED IDEOGRAPH
	"9B 8F	5BF6",	#CJK UNIFIED IDEOGRAPH
	"9B 90	5BF3",	#CJK UNIFIED IDEOGRAPH
	"9B 91	5C05",	#CJK UNIFIED IDEOGRAPH
	"9B 92	5C07",	#CJK UNIFIED IDEOGRAPH
	"9B 93	5C08",	#CJK UNIFIED IDEOGRAPH
	"9B 94	5C0D",	#CJK UNIFIED IDEOGRAPH
	"9B 95	5C13",	#CJK UNIFIED IDEOGRAPH
	"9B 96	5C20",	#CJK UNIFIED IDEOGRAPH
	"9B 97	5C22",	#CJK UNIFIED IDEOGRAPH
	"9B 98	5C28",	#CJK UNIFIED IDEOGRAPH
	"9B 99	5C38",	#CJK UNIFIED IDEOGRAPH
	"9B 9A	5C39",	#CJK UNIFIED IDEOGRAPH
	"9B 9B	5C41",	#CJK UNIFIED IDEOGRAPH
	"9B 9C	5C46",	#CJK UNIFIED IDEOGRAPH
	"9B 9D	5C4E",	#CJK UNIFIED IDEOGRAPH
	"9B 9E	5C53",	#CJK UNIFIED IDEOGRAPH
	"9B 9F	5C50",	#CJK UNIFIED IDEOGRAPH
	"9B A0	5C4F",	#CJK UNIFIED IDEOGRAPH
	"9B A1	5B71",	#CJK UNIFIED IDEOGRAPH
	"9B A2	5C6C",	#CJK UNIFIED IDEOGRAPH
	"9B A3	5C6E",	#CJK UNIFIED IDEOGRAPH
	"9B A4	4E62",	#CJK UNIFIED IDEOGRAPH
	"9B A5	5C76",	#CJK UNIFIED IDEOGRAPH
	"9B A6	5C79",	#CJK UNIFIED IDEOGRAPH
	"9B A7	5C8C",	#CJK UNIFIED IDEOGRAPH
	"9B A8	5C91",	#CJK UNIFIED IDEOGRAPH
	"9B A9	5C94",	#CJK UNIFIED IDEOGRAPH
	"9B AA	599B",	#CJK UNIFIED IDEOGRAPH
	"9B AB	5CAB",	#CJK UNIFIED IDEOGRAPH
	"9B AC	5CBB",	#CJK UNIFIED IDEOGRAPH
	"9B AD	5CB6",	#CJK UNIFIED IDEOGRAPH
	"9B AE	5CBC",	#CJK UNIFIED IDEOGRAPH
	"9B AF	5CB7",	#CJK UNIFIED IDEOGRAPH
	"9B B0	5CC5",	#CJK UNIFIED IDEOGRAPH
	"9B B1	5CBE",	#CJK UNIFIED IDEOGRAPH
	"9B B2	5CC7",	#CJK UNIFIED IDEOGRAPH
	"9B B3	5CD9",	#CJK UNIFIED IDEOGRAPH
	"9B B4	5CE9",	#CJK UNIFIED IDEOGRAPH
	"9B B5	5CFD",	#CJK UNIFIED IDEOGRAPH
	"9B B6	5CFA",	#CJK UNIFIED IDEOGRAPH
	"9B B7	5CED",	#CJK UNIFIED IDEOGRAPH
	"9B B8	5D8C",	#CJK UNIFIED IDEOGRAPH
	"9B B9	5CEA",	#CJK UNIFIED IDEOGRAPH
	"9B BA	5D0B",	#CJK UNIFIED IDEOGRAPH
	"9B BB	5D15",	#CJK UNIFIED IDEOGRAPH
	"9B BC	5D17",	#CJK UNIFIED IDEOGRAPH
	"9B BD	5D5C",	#CJK UNIFIED IDEOGRAPH
	"9B BE	5D1F",	#CJK UNIFIED IDEOGRAPH
	"9B BF	5D1B",	#CJK UNIFIED IDEOGRAPH
	"9B C0	5D11",	#CJK UNIFIED IDEOGRAPH
	"9B C1	5D14",	#CJK UNIFIED IDEOGRAPH
	"9B C2	5D22",	#CJK UNIFIED IDEOGRAPH
	"9B C3	5D1A",	#CJK UNIFIED IDEOGRAPH
	"9B C4	5D19",	#CJK UNIFIED IDEOGRAPH
	"9B C5	5D18",	#CJK UNIFIED IDEOGRAPH
	"9B C6	5D4C",	#CJK UNIFIED IDEOGRAPH
	"9B C7	5D52",	#CJK UNIFIED IDEOGRAPH
	"9B C8	5D4E",	#CJK UNIFIED IDEOGRAPH
	"9B C9	5D4B",	#CJK UNIFIED IDEOGRAPH
	"9B CA	5D6C",	#CJK UNIFIED IDEOGRAPH
	"9B CB	5D73",	#CJK UNIFIED IDEOGRAPH
	"9B CC	5D76",	#CJK UNIFIED IDEOGRAPH
	"9B CD	5D87",	#CJK UNIFIED IDEOGRAPH
	"9B CE	5D84",	#CJK UNIFIED IDEOGRAPH
	"9B CF	5D82",	#CJK UNIFIED IDEOGRAPH
	"9B D0	5DA2",	#CJK UNIFIED IDEOGRAPH
	"9B D1	5D9D",	#CJK UNIFIED IDEOGRAPH
	"9B D2	5DAC",	#CJK UNIFIED IDEOGRAPH
	"9B D3	5DAE",	#CJK UNIFIED IDEOGRAPH
	"9B D4	5DBD",	#CJK UNIFIED IDEOGRAPH
	"9B D5	5D90",	#CJK UNIFIED IDEOGRAPH
	"9B D6	5DB7",	#CJK UNIFIED IDEOGRAPH
	"9B D7	5DBC",	#CJK UNIFIED IDEOGRAPH
	"9B D8	5DC9",	#CJK UNIFIED IDEOGRAPH
	"9B D9	5DCD",	#CJK UNIFIED IDEOGRAPH
	"9B DA	5DD3",	#CJK UNIFIED IDEOGRAPH
	"9B DB	5DD2",	#CJK UNIFIED IDEOGRAPH
	"9B DC	5DD6",	#CJK UNIFIED IDEOGRAPH
	"9B DD	5DDB",	#CJK UNIFIED IDEOGRAPH
	"9B DE	5DEB",	#CJK UNIFIED IDEOGRAPH
	"9B DF	5DF2",	#CJK UNIFIED IDEOGRAPH
	"9B E0	5DF5",	#CJK UNIFIED IDEOGRAPH
	"9B E1	5E0B",	#CJK UNIFIED IDEOGRAPH
	"9B E2	5E1A",	#CJK UNIFIED IDEOGRAPH
	"9B E3	5E19",	#CJK UNIFIED IDEOGRAPH
	"9B E4	5E11",	#CJK UNIFIED IDEOGRAPH
	"9B E5	5E1B",	#CJK UNIFIED IDEOGRAPH
	"9B E6	5E36",	#CJK UNIFIED IDEOGRAPH
	"9B E7	5E37",	#CJK UNIFIED IDEOGRAPH
	"9B E8	5E44",	#CJK UNIFIED IDEOGRAPH
	"9B E9	5E43",	#CJK UNIFIED IDEOGRAPH
	"9B EA	5E40",	#CJK UNIFIED IDEOGRAPH
	"9B EB	5E4E",	#CJK UNIFIED IDEOGRAPH
	"9B EC	5E57",	#CJK UNIFIED IDEOGRAPH
	"9B ED	5E54",	#CJK UNIFIED IDEOGRAPH
	"9B EE	5E5F",	#CJK UNIFIED IDEOGRAPH
	"9B EF	5E62",	#CJK UNIFIED IDEOGRAPH
	"9B F0	5E64",	#CJK UNIFIED IDEOGRAPH
	"9B F1	5E47",	#CJK UNIFIED IDEOGRAPH
	"9B F2	5E75",	#CJK UNIFIED IDEOGRAPH
	"9B F3	5E76",	#CJK UNIFIED IDEOGRAPH
	"9B F4	5E7A",	#CJK UNIFIED IDEOGRAPH
	"9B F5	9EBC",	#CJK UNIFIED IDEOGRAPH
	"9B F6	5E7F",	#CJK UNIFIED IDEOGRAPH
	"9B F7	5EA0",	#CJK UNIFIED IDEOGRAPH
	"9B F8	5EC1",	#CJK UNIFIED IDEOGRAPH
	"9B F9	5EC2",	#CJK UNIFIED IDEOGRAPH
	"9B FA	5EC8",	#CJK UNIFIED IDEOGRAPH
	"9B FB	5ED0",	#CJK UNIFIED IDEOGRAPH
	"9B FC	5ECF",	#CJK UNIFIED IDEOGRAPH
	"9C 40	5ED6",	#CJK UNIFIED IDEOGRAPH
	"9C 41	5EE3",	#CJK UNIFIED IDEOGRAPH
	"9C 42	5EDD",	#CJK UNIFIED IDEOGRAPH
	"9C 43	5EDA",	#CJK UNIFIED IDEOGRAPH
	"9C 44	5EDB",	#CJK UNIFIED IDEOGRAPH
	"9C 45	5EE2",	#CJK UNIFIED IDEOGRAPH
	"9C 46	5EE1",	#CJK UNIFIED IDEOGRAPH
	"9C 47	5EE8",	#CJK UNIFIED IDEOGRAPH
	"9C 48	5EE9",	#CJK UNIFIED IDEOGRAPH
	"9C 49	5EEC",	#CJK UNIFIED IDEOGRAPH
	"9C 4A	5EF1",	#CJK UNIFIED IDEOGRAPH
	"9C 4B	5EF3",	#CJK UNIFIED IDEOGRAPH
	"9C 4C	5EF0",	#CJK UNIFIED IDEOGRAPH
	"9C 4D	5EF4",	#CJK UNIFIED IDEOGRAPH
	"9C 4E	5EF8",	#CJK UNIFIED IDEOGRAPH
	"9C 4F	5EFE",	#CJK UNIFIED IDEOGRAPH
	"9C 50	5F03",	#CJK UNIFIED IDEOGRAPH
	"9C 51	5F09",	#CJK UNIFIED IDEOGRAPH
	"9C 52	5F5D",	#CJK UNIFIED IDEOGRAPH
	"9C 53	5F5C",	#CJK UNIFIED IDEOGRAPH
	"9C 54	5F0B",	#CJK UNIFIED IDEOGRAPH
	"9C 55	5F11",	#CJK UNIFIED IDEOGRAPH
	"9C 56	5F16",	#CJK UNIFIED IDEOGRAPH
	"9C 57	5F29",	#CJK UNIFIED IDEOGRAPH
	"9C 58	5F2D",	#CJK UNIFIED IDEOGRAPH
	"9C 59	5F38",	#CJK UNIFIED IDEOGRAPH
	"9C 5A	5F41",	#CJK UNIFIED IDEOGRAPH
	"9C 5B	5F48",	#CJK UNIFIED IDEOGRAPH
	"9C 5C	5F4C",	#CJK UNIFIED IDEOGRAPH
	"9C 5D	5F4E",	#CJK UNIFIED IDEOGRAPH
	"9C 5E	5F2F",	#CJK UNIFIED IDEOGRAPH
	"9C 5F	5F51",	#CJK UNIFIED IDEOGRAPH
	"9C 60	5F56",	#CJK UNIFIED IDEOGRAPH
	"9C 61	5F57",	#CJK UNIFIED IDEOGRAPH
	"9C 62	5F59",	#CJK UNIFIED IDEOGRAPH
	"9C 63	5F61",	#CJK UNIFIED IDEOGRAPH
	"9C 64	5F6D",	#CJK UNIFIED IDEOGRAPH
	"9C 65	5F73",	#CJK UNIFIED IDEOGRAPH
	"9C 66	5F77",	#CJK UNIFIED IDEOGRAPH
	"9C 67	5F83",	#CJK UNIFIED IDEOGRAPH
	"9C 68	5F82",	#CJK UNIFIED IDEOGRAPH
	"9C 69	5F7F",	#CJK UNIFIED IDEOGRAPH
	"9C 6A	5F8A",	#CJK UNIFIED IDEOGRAPH
	"9C 6B	5F88",	#CJK UNIFIED IDEOGRAPH
	"9C 6C	5F91",	#CJK UNIFIED IDEOGRAPH
	"9C 6D	5F87",	#CJK UNIFIED IDEOGRAPH
	"9C 6E	5F9E",	#CJK UNIFIED IDEOGRAPH
	"9C 6F	5F99",	#CJK UNIFIED IDEOGRAPH
	"9C 70	5F98",	#CJK UNIFIED IDEOGRAPH
	"9C 71	5FA0",	#CJK UNIFIED IDEOGRAPH
	"9C 72	5FA8",	#CJK UNIFIED IDEOGRAPH
	"9C 73	5FAD",	#CJK UNIFIED IDEOGRAPH
	"9C 74	5FBC",	#CJK UNIFIED IDEOGRAPH
	"9C 75	5FD6",	#CJK UNIFIED IDEOGRAPH
	"9C 76	5FFB",	#CJK UNIFIED IDEOGRAPH
	"9C 77	5FE4",	#CJK UNIFIED IDEOGRAPH
	"9C 78	5FF8",	#CJK UNIFIED IDEOGRAPH
	"9C 79	5FF1",	#CJK UNIFIED IDEOGRAPH
	"9C 7A	5FDD",	#CJK UNIFIED IDEOGRAPH
	"9C 7B	60B3",	#CJK UNIFIED IDEOGRAPH
	"9C 7C	5FFF",	#CJK UNIFIED IDEOGRAPH
	"9C 7D	6021",	#CJK UNIFIED IDEOGRAPH
	"9C 7E	6060",	#CJK UNIFIED IDEOGRAPH
	"9C 80	6019",	#CJK UNIFIED IDEOGRAPH
	"9C 81	6010",	#CJK UNIFIED IDEOGRAPH
	"9C 82	6029",	#CJK UNIFIED IDEOGRAPH
	"9C 83	600E",	#CJK UNIFIED IDEOGRAPH
	"9C 84	6031",	#CJK UNIFIED IDEOGRAPH
	"9C 85	601B",	#CJK UNIFIED IDEOGRAPH
	"9C 86	6015",	#CJK UNIFIED IDEOGRAPH
	"9C 87	602B",	#CJK UNIFIED IDEOGRAPH
	"9C 88	6026",	#CJK UNIFIED IDEOGRAPH
	"9C 89	600F",	#CJK UNIFIED IDEOGRAPH
	"9C 8A	603A",	#CJK UNIFIED IDEOGRAPH
	"9C 8B	605A",	#CJK UNIFIED IDEOGRAPH
	"9C 8C	6041",	#CJK UNIFIED IDEOGRAPH
	"9C 8D	606A",	#CJK UNIFIED IDEOGRAPH
	"9C 8E	6077",	#CJK UNIFIED IDEOGRAPH
	"9C 8F	605F",	#CJK UNIFIED IDEOGRAPH
	"9C 90	604A",	#CJK UNIFIED IDEOGRAPH
	"9C 91	6046",	#CJK UNIFIED IDEOGRAPH
	"9C 92	604D",	#CJK UNIFIED IDEOGRAPH
	"9C 93	6063",	#CJK UNIFIED IDEOGRAPH
	"9C 94	6043",	#CJK UNIFIED IDEOGRAPH
	"9C 95	6064",	#CJK UNIFIED IDEOGRAPH
	"9C 96	6042",	#CJK UNIFIED IDEOGRAPH
	"9C 97	606C",	#CJK UNIFIED IDEOGRAPH
	"9C 98	606B",	#CJK UNIFIED IDEOGRAPH
	"9C 99	6059",	#CJK UNIFIED IDEOGRAPH
	"9C 9A	6081",	#CJK UNIFIED IDEOGRAPH
	"9C 9B	608D",	#CJK UNIFIED IDEOGRAPH
	"9C 9C	60E7",	#CJK UNIFIED IDEOGRAPH
	"9C 9D	6083",	#CJK UNIFIED IDEOGRAPH
	"9C 9E	609A",	#CJK UNIFIED IDEOGRAPH
	"9C 9F	6084",	#CJK UNIFIED IDEOGRAPH
	"9C A0	609B",	#CJK UNIFIED IDEOGRAPH
	"9C A1	6096",	#CJK UNIFIED IDEOGRAPH
	"9C A2	6097",	#CJK UNIFIED IDEOGRAPH
	"9C A3	6092",	#CJK UNIFIED IDEOGRAPH
	"9C A4	60A7",	#CJK UNIFIED IDEOGRAPH
	"9C A5	608B",	#CJK UNIFIED IDEOGRAPH
	"9C A6	60E1",	#CJK UNIFIED IDEOGRAPH
	"9C A7	60B8",	#CJK UNIFIED IDEOGRAPH
	"9C A8	60E0",	#CJK UNIFIED IDEOGRAPH
	"9C A9	60D3",	#CJK UNIFIED IDEOGRAPH
	"9C AA	60B4",	#CJK UNIFIED IDEOGRAPH
	"9C AB	5FF0",	#CJK UNIFIED IDEOGRAPH
	"9C AC	60BD",	#CJK UNIFIED IDEOGRAPH
	"9C AD	60C6",	#CJK UNIFIED IDEOGRAPH
	"9C AE	60B5",	#CJK UNIFIED IDEOGRAPH
	"9C AF	60D8",	#CJK UNIFIED IDEOGRAPH
	"9C B0	614D",	#CJK UNIFIED IDEOGRAPH
	"9C B1	6115",	#CJK UNIFIED IDEOGRAPH
	"9C B2	6106",	#CJK UNIFIED IDEOGRAPH
	"9C B3	60F6",	#CJK UNIFIED IDEOGRAPH
	"9C B4	60F7",	#CJK UNIFIED IDEOGRAPH
	"9C B5	6100",	#CJK UNIFIED IDEOGRAPH
	"9C B6	60F4",	#CJK UNIFIED IDEOGRAPH
	"9C B7	60FA",	#CJK UNIFIED IDEOGRAPH
	"9C B8	6103",	#CJK UNIFIED IDEOGRAPH
	"9C B9	6121",	#CJK UNIFIED IDEOGRAPH
	"9C BA	60FB",	#CJK UNIFIED IDEOGRAPH
	"9C BB	60F1",	#CJK UNIFIED IDEOGRAPH
	"9C BC	610D",	#CJK UNIFIED IDEOGRAPH
	"9C BD	610E",	#CJK UNIFIED IDEOGRAPH
	"9C BE	6147",	#CJK UNIFIED IDEOGRAPH
	"9C BF	613E",	#CJK UNIFIED IDEOGRAPH
	"9C C0	6128",	#CJK UNIFIED IDEOGRAPH
	"9C C1	6127",	#CJK UNIFIED IDEOGRAPH
	"9C C2	614A",	#CJK UNIFIED IDEOGRAPH
	"9C C3	613F",	#CJK UNIFIED IDEOGRAPH
	"9C C4	613C",	#CJK UNIFIED IDEOGRAPH
	"9C C5	612C",	#CJK UNIFIED IDEOGRAPH
	"9C C6	6134",	#CJK UNIFIED IDEOGRAPH
	"9C C7	613D",	#CJK UNIFIED IDEOGRAPH
	"9C C8	6142",	#CJK UNIFIED IDEOGRAPH
	"9C C9	6144",	#CJK UNIFIED IDEOGRAPH
	"9C CA	6173",	#CJK UNIFIED IDEOGRAPH
	"9C CB	6177",	#CJK UNIFIED IDEOGRAPH
	"9C CC	6158",	#CJK UNIFIED IDEOGRAPH
	"9C CD	6159",	#CJK UNIFIED IDEOGRAPH
	"9C CE	615A",	#CJK UNIFIED IDEOGRAPH
	"9C CF	616B",	#CJK UNIFIED IDEOGRAPH
	"9C D0	6174",	#CJK UNIFIED IDEOGRAPH
	"9C D1	616F",	#CJK UNIFIED IDEOGRAPH
	"9C D2	6165",	#CJK UNIFIED IDEOGRAPH
	"9C D3	6171",	#CJK UNIFIED IDEOGRAPH
	"9C D4	615F",	#CJK UNIFIED IDEOGRAPH
	"9C D5	615D",	#CJK UNIFIED IDEOGRAPH
	"9C D6	6153",	#CJK UNIFIED IDEOGRAPH
	"9C D7	6175",	#CJK UNIFIED IDEOGRAPH
	"9C D8	6199",	#CJK UNIFIED IDEOGRAPH
	"9C D9	6196",	#CJK UNIFIED IDEOGRAPH
	"9C DA	6187",	#CJK UNIFIED IDEOGRAPH
	"9C DB	61AC",	#CJK UNIFIED IDEOGRAPH
	"9C DC	6194",	#CJK UNIFIED IDEOGRAPH
	"9C DD	619A",	#CJK UNIFIED IDEOGRAPH
	"9C DE	618A",	#CJK UNIFIED IDEOGRAPH
	"9C DF	6191",	#CJK UNIFIED IDEOGRAPH
	"9C E0	61AB",	#CJK UNIFIED IDEOGRAPH
	"9C E1	61AE",	#CJK UNIFIED IDEOGRAPH
	"9C E2	61CC",	#CJK UNIFIED IDEOGRAPH
	"9C E3	61CA",	#CJK UNIFIED IDEOGRAPH
	"9C E4	61C9",	#CJK UNIFIED IDEOGRAPH
	"9C E5	61F7",	#CJK UNIFIED IDEOGRAPH
	"9C E6	61C8",	#CJK UNIFIED IDEOGRAPH
	"9C E7	61C3",	#CJK UNIFIED IDEOGRAPH
	"9C E8	61C6",	#CJK UNIFIED IDEOGRAPH
	"9C E9	61BA",	#CJK UNIFIED IDEOGRAPH
	"9C EA	61CB",	#CJK UNIFIED IDEOGRAPH
	"9C EB	7F79",	#CJK UNIFIED IDEOGRAPH
	"9C EC	61CD",	#CJK UNIFIED IDEOGRAPH
	"9C ED	61E6",	#CJK UNIFIED IDEOGRAPH
	"9C EE	61E3",	#CJK UNIFIED IDEOGRAPH
	"9C EF	61F6",	#CJK UNIFIED IDEOGRAPH
	"9C F0	61FA",	#CJK UNIFIED IDEOGRAPH
	"9C F1	61F4",	#CJK UNIFIED IDEOGRAPH
	"9C F2	61FF",	#CJK UNIFIED IDEOGRAPH
	"9C F3	61FD",	#CJK UNIFIED IDEOGRAPH
	"9C F4	61FC",	#CJK UNIFIED IDEOGRAPH
	"9C F5	61FE",	#CJK UNIFIED IDEOGRAPH
	"9C F6	6200",	#CJK UNIFIED IDEOGRAPH
	"9C F7	6208",	#CJK UNIFIED IDEOGRAPH
	"9C F8	6209",	#CJK UNIFIED IDEOGRAPH
	"9C F9	620D",	#CJK UNIFIED IDEOGRAPH
	"9C FA	620C",	#CJK UNIFIED IDEOGRAPH
	"9C FB	6214",	#CJK UNIFIED IDEOGRAPH
	"9C FC	621B",	#CJK UNIFIED IDEOGRAPH
	"9D 40	621E",	#CJK UNIFIED IDEOGRAPH
	"9D 41	6221",	#CJK UNIFIED IDEOGRAPH
	"9D 42	622A",	#CJK UNIFIED IDEOGRAPH
	"9D 43	622E",	#CJK UNIFIED IDEOGRAPH
	"9D 44	6230",	#CJK UNIFIED IDEOGRAPH
	"9D 45	6232",	#CJK UNIFIED IDEOGRAPH
	"9D 46	6233",	#CJK UNIFIED IDEOGRAPH
	"9D 47	6241",	#CJK UNIFIED IDEOGRAPH
	"9D 48	624E",	#CJK UNIFIED IDEOGRAPH
	"9D 49	625E",	#CJK UNIFIED IDEOGRAPH
	"9D 4A	6263",	#CJK UNIFIED IDEOGRAPH
	"9D 4B	625B",	#CJK UNIFIED IDEOGRAPH
	"9D 4C	6260",	#CJK UNIFIED IDEOGRAPH
	"9D 4D	6268",	#CJK UNIFIED IDEOGRAPH
	"9D 4E	627C",	#CJK UNIFIED IDEOGRAPH
	"9D 4F	6282",	#CJK UNIFIED IDEOGRAPH
	"9D 50	6289",	#CJK UNIFIED IDEOGRAPH
	"9D 51	627E",	#CJK UNIFIED IDEOGRAPH
	"9D 52	6292",	#CJK UNIFIED IDEOGRAPH
	"9D 53	6293",	#CJK UNIFIED IDEOGRAPH
	"9D 54	6296",	#CJK UNIFIED IDEOGRAPH
	"9D 55	62D4",	#CJK UNIFIED IDEOGRAPH
	"9D 56	6283",	#CJK UNIFIED IDEOGRAPH
	"9D 57	6294",	#CJK UNIFIED IDEOGRAPH
	"9D 58	62D7",	#CJK UNIFIED IDEOGRAPH
	"9D 59	62D1",	#CJK UNIFIED IDEOGRAPH
	"9D 5A	62BB",	#CJK UNIFIED IDEOGRAPH
	"9D 5B	62CF",	#CJK UNIFIED IDEOGRAPH
	"9D 5C	62FF",	#CJK UNIFIED IDEOGRAPH
	"9D 5D	62C6",	#CJK UNIFIED IDEOGRAPH
	"9D 5E	64D4",	#CJK UNIFIED IDEOGRAPH
	"9D 5F	62C8",	#CJK UNIFIED IDEOGRAPH
	"9D 60	62DC",	#CJK UNIFIED IDEOGRAPH
	"9D 61	62CC",	#CJK UNIFIED IDEOGRAPH
	"9D 62	62CA",	#CJK UNIFIED IDEOGRAPH
	"9D 63	62C2",	#CJK UNIFIED IDEOGRAPH
	"9D 64	62C7",	#CJK UNIFIED IDEOGRAPH
	"9D 65	629B",	#CJK UNIFIED IDEOGRAPH
	"9D 66	62C9",	#CJK UNIFIED IDEOGRAPH
	"9D 67	630C",	#CJK UNIFIED IDEOGRAPH
	"9D 68	62EE",	#CJK UNIFIED IDEOGRAPH
	"9D 69	62F1",	#CJK UNIFIED IDEOGRAPH
	"9D 6A	6327",	#CJK UNIFIED IDEOGRAPH
	"9D 6B	6302",	#CJK UNIFIED IDEOGRAPH
	"9D 6C	6308",	#CJK UNIFIED IDEOGRAPH
	"9D 6D	62EF",	#CJK UNIFIED IDEOGRAPH
	"9D 6E	62F5",	#CJK UNIFIED IDEOGRAPH
	"9D 6F	6350",	#CJK UNIFIED IDEOGRAPH
	"9D 70	633E",	#CJK UNIFIED IDEOGRAPH
	"9D 71	634D",	#CJK UNIFIED IDEOGRAPH
	"9D 72	641C",	#CJK UNIFIED IDEOGRAPH
	"9D 73	634F",	#CJK UNIFIED IDEOGRAPH
	"9D 74	6396",	#CJK UNIFIED IDEOGRAPH
	"9D 75	638E",	#CJK UNIFIED IDEOGRAPH
	"9D 76	6380",	#CJK UNIFIED IDEOGRAPH
	"9D 77	63AB",	#CJK UNIFIED IDEOGRAPH
	"9D 78	6376",	#CJK UNIFIED IDEOGRAPH
	"9D 79	63A3",	#CJK UNIFIED IDEOGRAPH
	"9D 7A	638F",	#CJK UNIFIED IDEOGRAPH
	"9D 7B	6389",	#CJK UNIFIED IDEOGRAPH
	"9D 7C	639F",	#CJK UNIFIED IDEOGRAPH
	"9D 7D	63B5",	#CJK UNIFIED IDEOGRAPH
	"9D 7E	636B",	#CJK UNIFIED IDEOGRAPH
	"9D 80	6369",	#CJK UNIFIED IDEOGRAPH
	"9D 81	63BE",	#CJK UNIFIED IDEOGRAPH
	"9D 82	63E9",	#CJK UNIFIED IDEOGRAPH
	"9D 83	63C0",	#CJK UNIFIED IDEOGRAPH
	"9D 84	63C6",	#CJK UNIFIED IDEOGRAPH
	"9D 85	63E3",	#CJK UNIFIED IDEOGRAPH
	"9D 86	63C9",	#CJK UNIFIED IDEOGRAPH
	"9D 87	63D2",	#CJK UNIFIED IDEOGRAPH
	"9D 88	63F6",	#CJK UNIFIED IDEOGRAPH
	"9D 89	63C4",	#CJK UNIFIED IDEOGRAPH
	"9D 8A	6416",	#CJK UNIFIED IDEOGRAPH
	"9D 8B	6434",	#CJK UNIFIED IDEOGRAPH
	"9D 8C	6406",	#CJK UNIFIED IDEOGRAPH
	"9D 8D	6413",	#CJK UNIFIED IDEOGRAPH
	"9D 8E	6426",	#CJK UNIFIED IDEOGRAPH
	"9D 8F	6436",	#CJK UNIFIED IDEOGRAPH
	"9D 90	651D",	#CJK UNIFIED IDEOGRAPH
	"9D 91	6417",	#CJK UNIFIED IDEOGRAPH
	"9D 92	6428",	#CJK UNIFIED IDEOGRAPH
	"9D 93	640F",	#CJK UNIFIED IDEOGRAPH
	"9D 94	6467",	#CJK UNIFIED IDEOGRAPH
	"9D 95	646F",	#CJK UNIFIED IDEOGRAPH
	"9D 96	6476",	#CJK UNIFIED IDEOGRAPH
	"9D 97	644E",	#CJK UNIFIED IDEOGRAPH
	"9D 98	652A",	#CJK UNIFIED IDEOGRAPH
	"9D 99	6495",	#CJK UNIFIED IDEOGRAPH
	"9D 9A	6493",	#CJK UNIFIED IDEOGRAPH
	"9D 9B	64A5",	#CJK UNIFIED IDEOGRAPH
	"9D 9C	64A9",	#CJK UNIFIED IDEOGRAPH
	"9D 9D	6488",	#CJK UNIFIED IDEOGRAPH
	"9D 9E	64BC",	#CJK UNIFIED IDEOGRAPH
	"9D 9F	64DA",	#CJK UNIFIED IDEOGRAPH
	"9D A0	64D2",	#CJK UNIFIED IDEOGRAPH
	"9D A1	64C5",	#CJK UNIFIED IDEOGRAPH
	"9D A2	64C7",	#CJK UNIFIED IDEOGRAPH
	"9D A3	64BB",	#CJK UNIFIED IDEOGRAPH
	"9D A4	64D8",	#CJK UNIFIED IDEOGRAPH
	"9D A5	64C2",	#CJK UNIFIED IDEOGRAPH
	"9D A6	64F1",	#CJK UNIFIED IDEOGRAPH
	"9D A7	64E7",	#CJK UNIFIED IDEOGRAPH
	"9D A8	8209",	#CJK UNIFIED IDEOGRAPH
	"9D A9	64E0",	#CJK UNIFIED IDEOGRAPH
	"9D AA	64E1",	#CJK UNIFIED IDEOGRAPH
	"9D AB	62AC",	#CJK UNIFIED IDEOGRAPH
	"9D AC	64E3",	#CJK UNIFIED IDEOGRAPH
	"9D AD	64EF",	#CJK UNIFIED IDEOGRAPH
	"9D AE	652C",	#CJK UNIFIED IDEOGRAPH
	"9D AF	64F6",	#CJK UNIFIED IDEOGRAPH
	"9D B0	64F4",	#CJK UNIFIED IDEOGRAPH
	"9D B1	64F2",	#CJK UNIFIED IDEOGRAPH
	"9D B2	64FA",	#CJK UNIFIED IDEOGRAPH
	"9D B3	6500",	#CJK UNIFIED IDEOGRAPH
	"9D B4	64FD",	#CJK UNIFIED IDEOGRAPH
	"9D B5	6518",	#CJK UNIFIED IDEOGRAPH
	"9D B6	651C",	#CJK UNIFIED IDEOGRAPH
	"9D B7	6505",	#CJK UNIFIED IDEOGRAPH
	"9D B8	6524",	#CJK UNIFIED IDEOGRAPH
	"9D B9	6523",	#CJK UNIFIED IDEOGRAPH
	"9D BA	652B",	#CJK UNIFIED IDEOGRAPH
	"9D BB	6534",	#CJK UNIFIED IDEOGRAPH
	"9D BC	6535",	#CJK UNIFIED IDEOGRAPH
	"9D BD	6537",	#CJK UNIFIED IDEOGRAPH
	"9D BE	6536",	#CJK UNIFIED IDEOGRAPH
	"9D BF	6538",	#CJK UNIFIED IDEOGRAPH
	"9D C0	754B",	#CJK UNIFIED IDEOGRAPH
	"9D C1	6548",	#CJK UNIFIED IDEOGRAPH
	"9D C2	6556",	#CJK UNIFIED IDEOGRAPH
	"9D C3	6555",	#CJK UNIFIED IDEOGRAPH
	"9D C4	654D",	#CJK UNIFIED IDEOGRAPH
	"9D C5	6558",	#CJK UNIFIED IDEOGRAPH
	"9D C6	655E",	#CJK UNIFIED IDEOGRAPH
	"9D C7	655D",	#CJK UNIFIED IDEOGRAPH
	"9D C8	6572",	#CJK UNIFIED IDEOGRAPH
	"9D C9	6578",	#CJK UNIFIED IDEOGRAPH
	"9D CA	6582",	#CJK UNIFIED IDEOGRAPH
	"9D CB	6583",	#CJK UNIFIED IDEOGRAPH
	"9D CC	8B8A",	#CJK UNIFIED IDEOGRAPH
	"9D CD	659B",	#CJK UNIFIED IDEOGRAPH
	"9D CE	659F",	#CJK UNIFIED IDEOGRAPH
	"9D CF	65AB",	#CJK UNIFIED IDEOGRAPH
	"9D D0	65B7",	#CJK UNIFIED IDEOGRAPH
	"9D D1	65C3",	#CJK UNIFIED IDEOGRAPH
	"9D D2	65C6",	#CJK UNIFIED IDEOGRAPH
	"9D D3	65C1",	#CJK UNIFIED IDEOGRAPH
	"9D D4	65C4",	#CJK UNIFIED IDEOGRAPH
	"9D D5	65CC",	#CJK UNIFIED IDEOGRAPH
	"9D D6	65D2",	#CJK UNIFIED IDEOGRAPH
	"9D D7	65DB",	#CJK UNIFIED IDEOGRAPH
	"9D D8	65D9",	#CJK UNIFIED IDEOGRAPH
	"9D D9	65E0",	#CJK UNIFIED IDEOGRAPH
	"9D DA	65E1",	#CJK UNIFIED IDEOGRAPH
	"9D DB	65F1",	#CJK UNIFIED IDEOGRAPH
	"9D DC	6772",	#CJK UNIFIED IDEOGRAPH
	"9D DD	660A",	#CJK UNIFIED IDEOGRAPH
	"9D DE	6603",	#CJK UNIFIED IDEOGRAPH
	"9D DF	65FB",	#CJK UNIFIED IDEOGRAPH
	"9D E0	6773",	#CJK UNIFIED IDEOGRAPH
	"9D E1	6635",	#CJK UNIFIED IDEOGRAPH
	"9D E2	6636",	#CJK UNIFIED IDEOGRAPH
	"9D E3	6634",	#CJK UNIFIED IDEOGRAPH
	"9D E4	661C",	#CJK UNIFIED IDEOGRAPH
	"9D E5	664F",	#CJK UNIFIED IDEOGRAPH
	"9D E6	6644",	#CJK UNIFIED IDEOGRAPH
	"9D E7	6649",	#CJK UNIFIED IDEOGRAPH
	"9D E8	6641",	#CJK UNIFIED IDEOGRAPH
	"9D E9	665E",	#CJK UNIFIED IDEOGRAPH
	"9D EA	665D",	#CJK UNIFIED IDEOGRAPH
	"9D EB	6664",	#CJK UNIFIED IDEOGRAPH
	"9D EC	6667",	#CJK UNIFIED IDEOGRAPH
	"9D ED	6668",	#CJK UNIFIED IDEOGRAPH
	"9D EE	665F",	#CJK UNIFIED IDEOGRAPH
	"9D EF	6662",	#CJK UNIFIED IDEOGRAPH
	"9D F0	6670",	#CJK UNIFIED IDEOGRAPH
	"9D F1	6683",	#CJK UNIFIED IDEOGRAPH
	"9D F2	6688",	#CJK UNIFIED IDEOGRAPH
	"9D F3	668E",	#CJK UNIFIED IDEOGRAPH
	"9D F4	6689",	#CJK UNIFIED IDEOGRAPH
	"9D F5	6684",	#CJK UNIFIED IDEOGRAPH
	"9D F6	6698",	#CJK UNIFIED IDEOGRAPH
	"9D F7	669D",	#CJK UNIFIED IDEOGRAPH
	"9D F8	66C1",	#CJK UNIFIED IDEOGRAPH
	"9D F9	66B9",	#CJK UNIFIED IDEOGRAPH
	"9D FA	66C9",	#CJK UNIFIED IDEOGRAPH
	"9D FB	66BE",	#CJK UNIFIED IDEOGRAPH
	"9D FC	66BC",	#CJK UNIFIED IDEOGRAPH
	"9E 40	66C4",	#CJK UNIFIED IDEOGRAPH
	"9E 41	66B8",	#CJK UNIFIED IDEOGRAPH
	"9E 42	66D6",	#CJK UNIFIED IDEOGRAPH
	"9E 43	66DA",	#CJK UNIFIED IDEOGRAPH
	"9E 44	66E0",	#CJK UNIFIED IDEOGRAPH
	"9E 45	663F",	#CJK UNIFIED IDEOGRAPH
	"9E 46	66E6",	#CJK UNIFIED IDEOGRAPH
	"9E 47	66E9",	#CJK UNIFIED IDEOGRAPH
	"9E 48	66F0",	#CJK UNIFIED IDEOGRAPH
	"9E 49	66F5",	#CJK UNIFIED IDEOGRAPH
	"9E 4A	66F7",	#CJK UNIFIED IDEOGRAPH
	"9E 4B	670F",	#CJK UNIFIED IDEOGRAPH
	"9E 4C	6716",	#CJK UNIFIED IDEOGRAPH
	"9E 4D	671E",	#CJK UNIFIED IDEOGRAPH
	"9E 4E	6726",	#CJK UNIFIED IDEOGRAPH
	"9E 4F	6727",	#CJK UNIFIED IDEOGRAPH
	"9E 50	9738",	#CJK UNIFIED IDEOGRAPH
	"9E 51	672E",	#CJK UNIFIED IDEOGRAPH
	"9E 52	673F",	#CJK UNIFIED IDEOGRAPH
	"9E 53	6736",	#CJK UNIFIED IDEOGRAPH
	"9E 54	6741",	#CJK UNIFIED IDEOGRAPH
	"9E 55	6738",	#CJK UNIFIED IDEOGRAPH
	"9E 56	6737",	#CJK UNIFIED IDEOGRAPH
	"9E 57	6746",	#CJK UNIFIED IDEOGRAPH
	"9E 58	675E",	#CJK UNIFIED IDEOGRAPH
	"9E 59	6760",	#CJK UNIFIED IDEOGRAPH
	"9E 5A	6759",	#CJK UNIFIED IDEOGRAPH
	"9E 5B	6763",	#CJK UNIFIED IDEOGRAPH
	"9E 5C	6764",	#CJK UNIFIED IDEOGRAPH
	"9E 5D	6789",	#CJK UNIFIED IDEOGRAPH
	"9E 5E	6770",	#CJK UNIFIED IDEOGRAPH
	"9E 5F	67A9",	#CJK UNIFIED IDEOGRAPH
	"9E 60	677C",	#CJK UNIFIED IDEOGRAPH
	"9E 61	676A",	#CJK UNIFIED IDEOGRAPH
	"9E 62	678C",	#CJK UNIFIED IDEOGRAPH
	"9E 63	678B",	#CJK UNIFIED IDEOGRAPH
	"9E 64	67A6",	#CJK UNIFIED IDEOGRAPH
	"9E 65	67A1",	#CJK UNIFIED IDEOGRAPH
	"9E 66	6785",	#CJK UNIFIED IDEOGRAPH
	"9E 67	67B7",	#CJK UNIFIED IDEOGRAPH
	"9E 68	67EF",	#CJK UNIFIED IDEOGRAPH
	"9E 69	67B4",	#CJK UNIFIED IDEOGRAPH
	"9E 6A	67EC",	#CJK UNIFIED IDEOGRAPH
	"9E 6B	67B3",	#CJK UNIFIED IDEOGRAPH
	"9E 6C	67E9",	#CJK UNIFIED IDEOGRAPH
	"9E 6D	67B8",	#CJK UNIFIED IDEOGRAPH
	"9E 6E	67E4",	#CJK UNIFIED IDEOGRAPH
	"9E 6F	67DE",	#CJK UNIFIED IDEOGRAPH
	"9E 70	67DD",	#CJK UNIFIED IDEOGRAPH
	"9E 71	67E2",	#CJK UNIFIED IDEOGRAPH
	"9E 72	67EE",	#CJK UNIFIED IDEOGRAPH
	"9E 73	67B9",	#CJK UNIFIED IDEOGRAPH
	"9E 74	67CE",	#CJK UNIFIED IDEOGRAPH
	"9E 75	67C6",	#CJK UNIFIED IDEOGRAPH
	"9E 76	67E7",	#CJK UNIFIED IDEOGRAPH
	"9E 77	6A9C",	#CJK UNIFIED IDEOGRAPH
	"9E 78	681E",	#CJK UNIFIED IDEOGRAPH
	"9E 79	6846",	#CJK UNIFIED IDEOGRAPH
	"9E 7A	6829",	#CJK UNIFIED IDEOGRAPH
	"9E 7B	6840",	#CJK UNIFIED IDEOGRAPH
	"9E 7C	684D",	#CJK UNIFIED IDEOGRAPH
	"9E 7D	6832",	#CJK UNIFIED IDEOGRAPH
	"9E 7E	684E",	#CJK UNIFIED IDEOGRAPH
	"9E 80	68B3",	#CJK UNIFIED IDEOGRAPH
	"9E 81	682B",	#CJK UNIFIED IDEOGRAPH
	"9E 82	6859",	#CJK UNIFIED IDEOGRAPH
	"9E 83	6863",	#CJK UNIFIED IDEOGRAPH
	"9E 84	6877",	#CJK UNIFIED IDEOGRAPH
	"9E 85	687F",	#CJK UNIFIED IDEOGRAPH
	"9E 86	689F",	#CJK UNIFIED IDEOGRAPH
	"9E 87	688F",	#CJK UNIFIED IDEOGRAPH
	"9E 88	68AD",	#CJK UNIFIED IDEOGRAPH
	"9E 89	6894",	#CJK UNIFIED IDEOGRAPH
	"9E 8A	689D",	#CJK UNIFIED IDEOGRAPH
	"9E 8B	689B",	#CJK UNIFIED IDEOGRAPH
	"9E 8C	6883",	#CJK UNIFIED IDEOGRAPH
	"9E 8D	6AAE",	#CJK UNIFIED IDEOGRAPH
	"9E 8E	68B9",	#CJK UNIFIED IDEOGRAPH
	"9E 8F	6874",	#CJK UNIFIED IDEOGRAPH
	"9E 90	68B5",	#CJK UNIFIED IDEOGRAPH
	"9E 91	68A0",	#CJK UNIFIED IDEOGRAPH
	"9E 92	68BA",	#CJK UNIFIED IDEOGRAPH
	"9E 93	690F",	#CJK UNIFIED IDEOGRAPH
	"9E 94	688D",	#CJK UNIFIED IDEOGRAPH
	"9E 95	687E",	#CJK UNIFIED IDEOGRAPH
	"9E 96	6901",	#CJK UNIFIED IDEOGRAPH
	"9E 97	68CA",	#CJK UNIFIED IDEOGRAPH
	"9E 98	6908",	#CJK UNIFIED IDEOGRAPH
	"9E 99	68D8",	#CJK UNIFIED IDEOGRAPH
	"9E 9A	6922",	#CJK UNIFIED IDEOGRAPH
	"9E 9B	6926",	#CJK UNIFIED IDEOGRAPH
	"9E 9C	68E1",	#CJK UNIFIED IDEOGRAPH
	"9E 9D	690C",	#CJK UNIFIED IDEOGRAPH
	"9E 9E	68CD",	#CJK UNIFIED IDEOGRAPH
	"9E 9F	68D4",	#CJK UNIFIED IDEOGRAPH
	"9E A0	68E7",	#CJK UNIFIED IDEOGRAPH
	"9E A1	68D5",	#CJK UNIFIED IDEOGRAPH
	"9E A2	6936",	#CJK UNIFIED IDEOGRAPH
	"9E A3	6912",	#CJK UNIFIED IDEOGRAPH
	"9E A4	6904",	#CJK UNIFIED IDEOGRAPH
	"9E A5	68D7",	#CJK UNIFIED IDEOGRAPH
	"9E A6	68E3",	#CJK UNIFIED IDEOGRAPH
	"9E A7	6925",	#CJK UNIFIED IDEOGRAPH
	"9E A8	68F9",	#CJK UNIFIED IDEOGRAPH
	"9E A9	68E0",	#CJK UNIFIED IDEOGRAPH
	"9E AA	68EF",	#CJK UNIFIED IDEOGRAPH
	"9E AB	6928",	#CJK UNIFIED IDEOGRAPH
	"9E AC	692A",	#CJK UNIFIED IDEOGRAPH
	"9E AD	691A",	#CJK UNIFIED IDEOGRAPH
	"9E AE	6923",	#CJK UNIFIED IDEOGRAPH
	"9E AF	6921",	#CJK UNIFIED IDEOGRAPH
	"9E B0	68C6",	#CJK UNIFIED IDEOGRAPH
	"9E B1	6979",	#CJK UNIFIED IDEOGRAPH
	"9E B2	6977",	#CJK UNIFIED IDEOGRAPH
	"9E B3	695C",	#CJK UNIFIED IDEOGRAPH
	"9E B4	6978",	#CJK UNIFIED IDEOGRAPH
	"9E B5	696B",	#CJK UNIFIED IDEOGRAPH
	"9E B6	6954",	#CJK UNIFIED IDEOGRAPH
	"9E B7	697E",	#CJK UNIFIED IDEOGRAPH
	"9E B8	696E",	#CJK UNIFIED IDEOGRAPH
	"9E B9	6939",	#CJK UNIFIED IDEOGRAPH
	"9E BA	6974",	#CJK UNIFIED IDEOGRAPH
	"9E BB	693D",	#CJK UNIFIED IDEOGRAPH
	"9E BC	6959",	#CJK UNIFIED IDEOGRAPH
	"9E BD	6930",	#CJK UNIFIED IDEOGRAPH
	"9E BE	6961",	#CJK UNIFIED IDEOGRAPH
	"9E BF	695E",	#CJK UNIFIED IDEOGRAPH
	"9E C0	695D",	#CJK UNIFIED IDEOGRAPH
	"9E C1	6981",	#CJK UNIFIED IDEOGRAPH
	"9E C2	696A",	#CJK UNIFIED IDEOGRAPH
	"9E C3	69B2",	#CJK UNIFIED IDEOGRAPH
	"9E C4	69AE",	#CJK UNIFIED IDEOGRAPH
	"9E C5	69D0",	#CJK UNIFIED IDEOGRAPH
	"9E C6	69BF",	#CJK UNIFIED IDEOGRAPH
	"9E C7	69C1",	#CJK UNIFIED IDEOGRAPH
	"9E C8	69D3",	#CJK UNIFIED IDEOGRAPH
	"9E C9	69BE",	#CJK UNIFIED IDEOGRAPH
	"9E CA	69CE",	#CJK UNIFIED IDEOGRAPH
	"9E CB	5BE8",	#CJK UNIFIED IDEOGRAPH
	"9E CC	69CA",	#CJK UNIFIED IDEOGRAPH
	"9E CD	69DD",	#CJK UNIFIED IDEOGRAPH
	"9E CE	69BB",	#CJK UNIFIED IDEOGRAPH
	"9E CF	69C3",	#CJK UNIFIED IDEOGRAPH
	"9E D0	69A7",	#CJK UNIFIED IDEOGRAPH
	"9E D1	6A2E",	#CJK UNIFIED IDEOGRAPH
	"9E D2	6991",	#CJK UNIFIED IDEOGRAPH
	"9E D3	69A0",	#CJK UNIFIED IDEOGRAPH
	"9E D4	699C",	#CJK UNIFIED IDEOGRAPH
	"9E D5	6995",	#CJK UNIFIED IDEOGRAPH
	"9E D6	69B4",	#CJK UNIFIED IDEOGRAPH
	"9E D7	69DE",	#CJK UNIFIED IDEOGRAPH
	"9E D8	69E8",	#CJK UNIFIED IDEOGRAPH
	"9E D9	6A02",	#CJK UNIFIED IDEOGRAPH
	"9E DA	6A1B",	#CJK UNIFIED IDEOGRAPH
	"9E DB	69FF",	#CJK UNIFIED IDEOGRAPH
	"9E DC	6B0A",	#CJK UNIFIED IDEOGRAPH
	"9E DD	69F9",	#CJK UNIFIED IDEOGRAPH
	"9E DE	69F2",	#CJK UNIFIED IDEOGRAPH
	"9E DF	69E7",	#CJK UNIFIED IDEOGRAPH
	"9E E0	6A05",	#CJK UNIFIED IDEOGRAPH
	"9E E1	69B1",	#CJK UNIFIED IDEOGRAPH
	"9E E2	6A1E",	#CJK UNIFIED IDEOGRAPH
	"9E E3	69ED",	#CJK UNIFIED IDEOGRAPH
	"9E E4	6A14",	#CJK UNIFIED IDEOGRAPH
	"9E E5	69EB",	#CJK UNIFIED IDEOGRAPH
	"9E E6	6A0A",	#CJK UNIFIED IDEOGRAPH
	"9E E7	6A12",	#CJK UNIFIED IDEOGRAPH
	"9E E8	6AC1",	#CJK UNIFIED IDEOGRAPH
	"9E E9	6A23",	#CJK UNIFIED IDEOGRAPH
	"9E EA	6A13",	#CJK UNIFIED IDEOGRAPH
	"9E EB	6A44",	#CJK UNIFIED IDEOGRAPH
	"9E EC	6A0C",	#CJK UNIFIED IDEOGRAPH
	"9E ED	6A72",	#CJK UNIFIED IDEOGRAPH
	"9E EE	6A36",	#CJK UNIFIED IDEOGRAPH
	"9E EF	6A78",	#CJK UNIFIED IDEOGRAPH
	"9E F0	6A47",	#CJK UNIFIED IDEOGRAPH
	"9E F1	6A62",	#CJK UNIFIED IDEOGRAPH
	"9E F2	6A59",	#CJK UNIFIED IDEOGRAPH
	"9E F3	6A66",	#CJK UNIFIED IDEOGRAPH
	"9E F4	6A48",	#CJK UNIFIED IDEOGRAPH
	"9E F5	6A38",	#CJK UNIFIED IDEOGRAPH
	"9E F6	6A22",	#CJK UNIFIED IDEOGRAPH
	"9E F7	6A90",	#CJK UNIFIED IDEOGRAPH
	"9E F8	6A8D",	#CJK UNIFIED IDEOGRAPH
	"9E F9	6AA0",	#CJK UNIFIED IDEOGRAPH
	"9E FA	6A84",	#CJK UNIFIED IDEOGRAPH
	"9E FB	6AA2",	#CJK UNIFIED IDEOGRAPH
	"9E FC	6AA3",	#CJK UNIFIED IDEOGRAPH
	"9F 40	6A97",	#CJK UNIFIED IDEOGRAPH
	"9F 41	8617",	#CJK UNIFIED IDEOGRAPH
	"9F 42	6ABB",	#CJK UNIFIED IDEOGRAPH
	"9F 43	6AC3",	#CJK UNIFIED IDEOGRAPH
	"9F 44	6AC2",	#CJK UNIFIED IDEOGRAPH
	"9F 45	6AB8",	#CJK UNIFIED IDEOGRAPH
	"9F 46	6AB3",	#CJK UNIFIED IDEOGRAPH
	"9F 47	6AAC",	#CJK UNIFIED IDEOGRAPH
	"9F 48	6ADE",	#CJK UNIFIED IDEOGRAPH
	"9F 49	6AD1",	#CJK UNIFIED IDEOGRAPH
	"9F 4A	6ADF",	#CJK UNIFIED IDEOGRAPH
	"9F 4B	6AAA",	#CJK UNIFIED IDEOGRAPH
	"9F 4C	6ADA",	#CJK UNIFIED IDEOGRAPH
	"9F 4D	6AEA",	#CJK UNIFIED IDEOGRAPH
	"9F 4E	6AFB",	#CJK UNIFIED IDEOGRAPH
	"9F 4F	6B05",	#CJK UNIFIED IDEOGRAPH
	"9F 50	8616",	#CJK UNIFIED IDEOGRAPH
	"9F 51	6AFA",	#CJK UNIFIED IDEOGRAPH
	"9F 52	6B12",	#CJK UNIFIED IDEOGRAPH
	"9F 53	6B16",	#CJK UNIFIED IDEOGRAPH
	"9F 54	9B31",	#CJK UNIFIED IDEOGRAPH
	"9F 55	6B1F",	#CJK UNIFIED IDEOGRAPH
	"9F 56	6B38",	#CJK UNIFIED IDEOGRAPH
	"9F 57	6B37",	#CJK UNIFIED IDEOGRAPH
	"9F 58	76DC",	#CJK UNIFIED IDEOGRAPH
	"9F 59	6B39",	#CJK UNIFIED IDEOGRAPH
	"9F 5A	98EE",	#CJK UNIFIED IDEOGRAPH
	"9F 5B	6B47",	#CJK UNIFIED IDEOGRAPH
	"9F 5C	6B43",	#CJK UNIFIED IDEOGRAPH
	"9F 5D	6B49",	#CJK UNIFIED IDEOGRAPH
	"9F 5E	6B50",	#CJK UNIFIED IDEOGRAPH
	"9F 5F	6B59",	#CJK UNIFIED IDEOGRAPH
	"9F 60	6B54",	#CJK UNIFIED IDEOGRAPH
	"9F 61	6B5B",	#CJK UNIFIED IDEOGRAPH
	"9F 62	6B5F",	#CJK UNIFIED IDEOGRAPH
	"9F 63	6B61",	#CJK UNIFIED IDEOGRAPH
	"9F 64	6B78",	#CJK UNIFIED IDEOGRAPH
	"9F 65	6B79",	#CJK UNIFIED IDEOGRAPH
	"9F 66	6B7F",	#CJK UNIFIED IDEOGRAPH
	"9F 67	6B80",	#CJK UNIFIED IDEOGRAPH
	"9F 68	6B84",	#CJK UNIFIED IDEOGRAPH
	"9F 69	6B83",	#CJK UNIFIED IDEOGRAPH
	"9F 6A	6B8D",	#CJK UNIFIED IDEOGRAPH
	"9F 6B	6B98",	#CJK UNIFIED IDEOGRAPH
	"9F 6C	6B95",	#CJK UNIFIED IDEOGRAPH
	"9F 6D	6B9E",	#CJK UNIFIED IDEOGRAPH
	"9F 6E	6BA4",	#CJK UNIFIED IDEOGRAPH
	"9F 6F	6BAA",	#CJK UNIFIED IDEOGRAPH
	"9F 70	6BAB",	#CJK UNIFIED IDEOGRAPH
	"9F 71	6BAF",	#CJK UNIFIED IDEOGRAPH
	"9F 72	6BB2",	#CJK UNIFIED IDEOGRAPH
	"9F 73	6BB1",	#CJK UNIFIED IDEOGRAPH
	"9F 74	6BB3",	#CJK UNIFIED IDEOGRAPH
	"9F 75	6BB7",	#CJK UNIFIED IDEOGRAPH
	"9F 76	6BBC",	#CJK UNIFIED IDEOGRAPH
	"9F 77	6BC6",	#CJK UNIFIED IDEOGRAPH
	"9F 78	6BCB",	#CJK UNIFIED IDEOGRAPH
	"9F 79	6BD3",	#CJK UNIFIED IDEOGRAPH
	"9F 7A	6BDF",	#CJK UNIFIED IDEOGRAPH
	"9F 7B	6BEC",	#CJK UNIFIED IDEOGRAPH
	"9F 7C	6BEB",	#CJK UNIFIED IDEOGRAPH
	"9F 7D	6BF3",	#CJK UNIFIED IDEOGRAPH
	"9F 7E	6BEF",	#CJK UNIFIED IDEOGRAPH
	"9F 80	9EBE",	#CJK UNIFIED IDEOGRAPH
	"9F 81	6C08",	#CJK UNIFIED IDEOGRAPH
	"9F 82	6C13",	#CJK UNIFIED IDEOGRAPH
	"9F 83	6C14",	#CJK UNIFIED IDEOGRAPH
	"9F 84	6C1B",	#CJK UNIFIED IDEOGRAPH
	"9F 85	6C24",	#CJK UNIFIED IDEOGRAPH
	"9F 86	6C23",	#CJK UNIFIED IDEOGRAPH
	"9F 87	6C5E",	#CJK UNIFIED IDEOGRAPH
	"9F 88	6C55",	#CJK UNIFIED IDEOGRAPH
	"9F 89	6C62",	#CJK UNIFIED IDEOGRAPH
	"9F 8A	6C6A",	#CJK UNIFIED IDEOGRAPH
	"9F 8B	6C82",	#CJK UNIFIED IDEOGRAPH
	"9F 8C	6C8D",	#CJK UNIFIED IDEOGRAPH
	"9F 8D	6C9A",	#CJK UNIFIED IDEOGRAPH
	"9F 8E	6C81",	#CJK UNIFIED IDEOGRAPH
	"9F 8F	6C9B",	#CJK UNIFIED IDEOGRAPH
	"9F 90	6C7E",	#CJK UNIFIED IDEOGRAPH
	"9F 91	6C68",	#CJK UNIFIED IDEOGRAPH
	"9F 92	6C73",	#CJK UNIFIED IDEOGRAPH
	"9F 93	6C92",	#CJK UNIFIED IDEOGRAPH
	"9F 94	6C90",	#CJK UNIFIED IDEOGRAPH
	"9F 95	6CC4",	#CJK UNIFIED IDEOGRAPH
	"9F 96	6CF1",	#CJK UNIFIED IDEOGRAPH
	"9F 97	6CD3",	#CJK UNIFIED IDEOGRAPH
	"9F 98	6CBD",	#CJK UNIFIED IDEOGRAPH
	"9F 99	6CD7",	#CJK UNIFIED IDEOGRAPH
	"9F 9A	6CC5",	#CJK UNIFIED IDEOGRAPH
	"9F 9B	6CDD",	#CJK UNIFIED IDEOGRAPH
	"9F 9C	6CAE",	#CJK UNIFIED IDEOGRAPH
	"9F 9D	6CB1",	#CJK UNIFIED IDEOGRAPH
	"9F 9E	6CBE",	#CJK UNIFIED IDEOGRAPH
	"9F 9F	6CBA",	#CJK UNIFIED IDEOGRAPH
	"9F A0	6CDB",	#CJK UNIFIED IDEOGRAPH
	"9F A1	6CEF",	#CJK UNIFIED IDEOGRAPH
	"9F A2	6CD9",	#CJK UNIFIED IDEOGRAPH
	"9F A3	6CEA",	#CJK UNIFIED IDEOGRAPH
	"9F A4	6D1F",	#CJK UNIFIED IDEOGRAPH
	"9F A5	884D",	#CJK UNIFIED IDEOGRAPH
	"9F A6	6D36",	#CJK UNIFIED IDEOGRAPH
	"9F A7	6D2B",	#CJK UNIFIED IDEOGRAPH
	"9F A8	6D3D",	#CJK UNIFIED IDEOGRAPH
	"9F A9	6D38",	#CJK UNIFIED IDEOGRAPH
	"9F AA	6D19",	#CJK UNIFIED IDEOGRAPH
	"9F AB	6D35",	#CJK UNIFIED IDEOGRAPH
	"9F AC	6D33",	#CJK UNIFIED IDEOGRAPH
	"9F AD	6D12",	#CJK UNIFIED IDEOGRAPH
	"9F AE	6D0C",	#CJK UNIFIED IDEOGRAPH
	"9F AF	6D63",	#CJK UNIFIED IDEOGRAPH
	"9F B0	6D93",	#CJK UNIFIED IDEOGRAPH
	"9F B1	6D64",	#CJK UNIFIED IDEOGRAPH
	"9F B2	6D5A",	#CJK UNIFIED IDEOGRAPH
	"9F B3	6D79",	#CJK UNIFIED IDEOGRAPH
	"9F B4	6D59",	#CJK UNIFIED IDEOGRAPH
	"9F B5	6D8E",	#CJK UNIFIED IDEOGRAPH
	"9F B6	6D95",	#CJK UNIFIED IDEOGRAPH
	"9F B7	6FE4",	#CJK UNIFIED IDEOGRAPH
	"9F B8	6D85",	#CJK UNIFIED IDEOGRAPH
	"9F B9	6DF9",	#CJK UNIFIED IDEOGRAPH
	"9F BA	6E15",	#CJK UNIFIED IDEOGRAPH
	"9F BB	6E0A",	#CJK UNIFIED IDEOGRAPH
	"9F BC	6DB5",	#CJK UNIFIED IDEOGRAPH
	"9F BD	6DC7",	#CJK UNIFIED IDEOGRAPH
	"9F BE	6DE6",	#CJK UNIFIED IDEOGRAPH
	"9F BF	6DB8",	#CJK UNIFIED IDEOGRAPH
	"9F C0	6DC6",	#CJK UNIFIED IDEOGRAPH
	"9F C1	6DEC",	#CJK UNIFIED IDEOGRAPH
	"9F C2	6DDE",	#CJK UNIFIED IDEOGRAPH
	"9F C3	6DCC",	#CJK UNIFIED IDEOGRAPH
	"9F C4	6DE8",	#CJK UNIFIED IDEOGRAPH
	"9F C5	6DD2",	#CJK UNIFIED IDEOGRAPH
	"9F C6	6DC5",	#CJK UNIFIED IDEOGRAPH
	"9F C7	6DFA",	#CJK UNIFIED IDEOGRAPH
	"9F C8	6DD9",	#CJK UNIFIED IDEOGRAPH
	"9F C9	6DE4",	#CJK UNIFIED IDEOGRAPH
	"9F CA	6DD5",	#CJK UNIFIED IDEOGRAPH
	"9F CB	6DEA",	#CJK UNIFIED IDEOGRAPH
	"9F CC	6DEE",	#CJK UNIFIED IDEOGRAPH
	"9F CD	6E2D",	#CJK UNIFIED IDEOGRAPH
	"9F CE	6E6E",	#CJK UNIFIED IDEOGRAPH
	"9F CF	6E2E",	#CJK UNIFIED IDEOGRAPH
	"9F D0	6E19",	#CJK UNIFIED IDEOGRAPH
	"9F D1	6E72",	#CJK UNIFIED IDEOGRAPH
	"9F D2	6E5F",	#CJK UNIFIED IDEOGRAPH
	"9F D3	6E3E",	#CJK UNIFIED IDEOGRAPH
	"9F D4	6E23",	#CJK UNIFIED IDEOGRAPH
	"9F D5	6E6B",	#CJK UNIFIED IDEOGRAPH
	"9F D6	6E2B",	#CJK UNIFIED IDEOGRAPH
	"9F D7	6E76",	#CJK UNIFIED IDEOGRAPH
	"9F D8	6E4D",	#CJK UNIFIED IDEOGRAPH
	"9F D9	6E1F",	#CJK UNIFIED IDEOGRAPH
	"9F DA	6E43",	#CJK UNIFIED IDEOGRAPH
	"9F DB	6E3A",	#CJK UNIFIED IDEOGRAPH
	"9F DC	6E4E",	#CJK UNIFIED IDEOGRAPH
	"9F DD	6E24",	#CJK UNIFIED IDEOGRAPH
	"9F DE	6EFF",	#CJK UNIFIED IDEOGRAPH
	"9F DF	6E1D",	#CJK UNIFIED IDEOGRAPH
	"9F E0	6E38",	#CJK UNIFIED IDEOGRAPH
	"9F E1	6E82",	#CJK UNIFIED IDEOGRAPH
	"9F E2	6EAA",	#CJK UNIFIED IDEOGRAPH
	"9F E3	6E98",	#CJK UNIFIED IDEOGRAPH
	"9F E4	6EC9",	#CJK UNIFIED IDEOGRAPH
	"9F E5	6EB7",	#CJK UNIFIED IDEOGRAPH
	"9F E6	6ED3",	#CJK UNIFIED IDEOGRAPH
	"9F E7	6EBD",	#CJK UNIFIED IDEOGRAPH
	"9F E8	6EAF",	#CJK UNIFIED IDEOGRAPH
	"9F E9	6EC4",	#CJK UNIFIED IDEOGRAPH
	"9F EA	6EB2",	#CJK UNIFIED IDEOGRAPH
	"9F EB	6ED4",	#CJK UNIFIED IDEOGRAPH
	"9F EC	6ED5",	#CJK UNIFIED IDEOGRAPH
	"9F ED	6E8F",	#CJK UNIFIED IDEOGRAPH
	"9F EE	6EA5",	#CJK UNIFIED IDEOGRAPH
	"9F EF	6EC2",	#CJK UNIFIED IDEOGRAPH
	"9F F0	6E9F",	#CJK UNIFIED IDEOGRAPH
	"9F F1	6F41",	#CJK UNIFIED IDEOGRAPH
	"9F F2	6F11",	#CJK UNIFIED IDEOGRAPH
	"9F F3	704C",	#CJK UNIFIED IDEOGRAPH
	"9F F4	6EEC",	#CJK UNIFIED IDEOGRAPH
	"9F F5	6EF8",	#CJK UNIFIED IDEOGRAPH
	"9F F6	6EFE",	#CJK UNIFIED IDEOGRAPH
	"9F F7	6F3F",	#CJK UNIFIED IDEOGRAPH
	"9F F8	6EF2",	#CJK UNIFIED IDEOGRAPH
	"9F F9	6F31",	#CJK UNIFIED IDEOGRAPH
	"9F FA	6EEF",	#CJK UNIFIED IDEOGRAPH
	"9F FB	6F32",	#CJK UNIFIED IDEOGRAPH
	"9F FC	6ECC",	#CJK UNIFIED IDEOGRAPH
	"E0 40	6F3E",	#CJK UNIFIED IDEOGRAPH
	"E0 41	6F13",	#CJK UNIFIED IDEOGRAPH
	"E0 42	6EF7",	#CJK UNIFIED IDEOGRAPH
	"E0 43	6F86",	#CJK UNIFIED IDEOGRAPH
	"E0 44	6F7A",	#CJK UNIFIED IDEOGRAPH
	"E0 45	6F78",	#CJK UNIFIED IDEOGRAPH
	"E0 46	6F81",	#CJK UNIFIED IDEOGRAPH
	"E0 47	6F80",	#CJK UNIFIED IDEOGRAPH
	"E0 48	6F6F",	#CJK UNIFIED IDEOGRAPH
	"E0 49	6F5B",	#CJK UNIFIED IDEOGRAPH
	"E0 4A	6FF3",	#CJK UNIFIED IDEOGRAPH
	"E0 4B	6F6D",	#CJK UNIFIED IDEOGRAPH
	"E0 4C	6F82",	#CJK UNIFIED IDEOGRAPH
	"E0 4D	6F7C",	#CJK UNIFIED IDEOGRAPH
	"E0 4E	6F58",	#CJK UNIFIED IDEOGRAPH
	"E0 4F	6F8E",	#CJK UNIFIED IDEOGRAPH
	"E0 50	6F91",	#CJK UNIFIED IDEOGRAPH
	"E0 51	6FC2",	#CJK UNIFIED IDEOGRAPH
	"E0 52	6F66",	#CJK UNIFIED IDEOGRAPH
	"E0 53	6FB3",	#CJK UNIFIED IDEOGRAPH
	"E0 54	6FA3",	#CJK UNIFIED IDEOGRAPH
	"E0 55	6FA1",	#CJK UNIFIED IDEOGRAPH
	"E0 56	6FA4",	#CJK UNIFIED IDEOGRAPH
	"E0 57	6FB9",	#CJK UNIFIED IDEOGRAPH
	"E0 58	6FC6",	#CJK UNIFIED IDEOGRAPH
	"E0 59	6FAA",	#CJK UNIFIED IDEOGRAPH
	"E0 5A	6FDF",	#CJK UNIFIED IDEOGRAPH
	"E0 5B	6FD5",	#CJK UNIFIED IDEOGRAPH
	"E0 5C	6FEC",	#CJK UNIFIED IDEOGRAPH
	"E0 5D	6FD4",	#CJK UNIFIED IDEOGRAPH
	"E0 5E	6FD8",	#CJK UNIFIED IDEOGRAPH
	"E0 5F	6FF1",	#CJK UNIFIED IDEOGRAPH
	"E0 60	6FEE",	#CJK UNIFIED IDEOGRAPH
	"E0 61	6FDB",	#CJK UNIFIED IDEOGRAPH
	"E0 62	7009",	#CJK UNIFIED IDEOGRAPH
	"E0 63	700B",	#CJK UNIFIED IDEOGRAPH
	"E0 64	6FFA",	#CJK UNIFIED IDEOGRAPH
	"E0 65	7011",	#CJK UNIFIED IDEOGRAPH
	"E0 66	7001",	#CJK UNIFIED IDEOGRAPH
	"E0 67	700F",	#CJK UNIFIED IDEOGRAPH
	"E0 68	6FFE",	#CJK UNIFIED IDEOGRAPH
	"E0 69	701B",	#CJK UNIFIED IDEOGRAPH
	"E0 6A	701A",	#CJK UNIFIED IDEOGRAPH
	"E0 6B	6F74",	#CJK UNIFIED IDEOGRAPH
	"E0 6C	701D",	#CJK UNIFIED IDEOGRAPH
	"E0 6D	7018",	#CJK UNIFIED IDEOGRAPH
	"E0 6E	701F",	#CJK UNIFIED IDEOGRAPH
	"E0 6F	7030",	#CJK UNIFIED IDEOGRAPH
	"E0 70	703E",	#CJK UNIFIED IDEOGRAPH
	"E0 71	7032",	#CJK UNIFIED IDEOGRAPH
	"E0 72	7051",	#CJK UNIFIED IDEOGRAPH
	"E0 73	7063",	#CJK UNIFIED IDEOGRAPH
	"E0 74	7099",	#CJK UNIFIED IDEOGRAPH
	"E0 75	7092",	#CJK UNIFIED IDEOGRAPH
	"E0 76	70AF",	#CJK UNIFIED IDEOGRAPH
	"E0 77	70F1",	#CJK UNIFIED IDEOGRAPH
	"E0 78	70AC",	#CJK UNIFIED IDEOGRAPH
	"E0 79	70B8",	#CJK UNIFIED IDEOGRAPH
	"E0 7A	70B3",	#CJK UNIFIED IDEOGRAPH
	"E0 7B	70AE",	#CJK UNIFIED IDEOGRAPH
	"E0 7C	70DF",	#CJK UNIFIED IDEOGRAPH
	"E0 7D	70CB",	#CJK UNIFIED IDEOGRAPH
	"E0 7E	70DD",	#CJK UNIFIED IDEOGRAPH
	"E0 80	70D9",	#CJK UNIFIED IDEOGRAPH
	"E0 81	7109",	#CJK UNIFIED IDEOGRAPH
	"E0 82	70FD",	#CJK UNIFIED IDEOGRAPH
	"E0 83	711C",	#CJK UNIFIED IDEOGRAPH
	"E0 84	7119",	#CJK UNIFIED IDEOGRAPH
	"E0 85	7165",	#CJK UNIFIED IDEOGRAPH
	"E0 86	7155",	#CJK UNIFIED IDEOGRAPH
	"E0 87	7188",	#CJK UNIFIED IDEOGRAPH
	"E0 88	7166",	#CJK UNIFIED IDEOGRAPH
	"E0 89	7162",	#CJK UNIFIED IDEOGRAPH
	"E0 8A	714C",	#CJK UNIFIED IDEOGRAPH
	"E0 8B	7156",	#CJK UNIFIED IDEOGRAPH
	"E0 8C	716C",	#CJK UNIFIED IDEOGRAPH
	"E0 8D	718F",	#CJK UNIFIED IDEOGRAPH
	"E0 8E	71FB",	#CJK UNIFIED IDEOGRAPH
	"E0 8F	7184",	#CJK UNIFIED IDEOGRAPH
	"E0 90	7195",	#CJK UNIFIED IDEOGRAPH
	"E0 91	71A8",	#CJK UNIFIED IDEOGRAPH
	"E0 92	71AC",	#CJK UNIFIED IDEOGRAPH
	"E0 93	71D7",	#CJK UNIFIED IDEOGRAPH
	"E0 94	71B9",	#CJK UNIFIED IDEOGRAPH
	"E0 95	71BE",	#CJK UNIFIED IDEOGRAPH
	"E0 96	71D2",	#CJK UNIFIED IDEOGRAPH
	"E0 97	71C9",	#CJK UNIFIED IDEOGRAPH
	"E0 98	71D4",	#CJK UNIFIED IDEOGRAPH
	"E0 99	71CE",	#CJK UNIFIED IDEOGRAPH
	"E0 9A	71E0",	#CJK UNIFIED IDEOGRAPH
	"E0 9B	71EC",	#CJK UNIFIED IDEOGRAPH
	"E0 9C	71E7",	#CJK UNIFIED IDEOGRAPH
	"E0 9D	71F5",	#CJK UNIFIED IDEOGRAPH
	"E0 9E	71FC",	#CJK UNIFIED IDEOGRAPH
	"E0 9F	71F9",	#CJK UNIFIED IDEOGRAPH
	"E0 A0	71FF",	#CJK UNIFIED IDEOGRAPH
	"E0 A1	720D",	#CJK UNIFIED IDEOGRAPH
	"E0 A2	7210",	#CJK UNIFIED IDEOGRAPH
	"E0 A3	721B",	#CJK UNIFIED IDEOGRAPH
	"E0 A4	7228",	#CJK UNIFIED IDEOGRAPH
	"E0 A5	722D",	#CJK UNIFIED IDEOGRAPH
	"E0 A6	722C",	#CJK UNIFIED IDEOGRAPH
	"E0 A7	7230",	#CJK UNIFIED IDEOGRAPH
	"E0 A8	7232",	#CJK UNIFIED IDEOGRAPH
	"E0 A9	723B",	#CJK UNIFIED IDEOGRAPH
	"E0 AA	723C",	#CJK UNIFIED IDEOGRAPH
	"E0 AB	723F",	#CJK UNIFIED IDEOGRAPH
	"E0 AC	7240",	#CJK UNIFIED IDEOGRAPH
	"E0 AD	7246",	#CJK UNIFIED IDEOGRAPH
	"E0 AE	724B",	#CJK UNIFIED IDEOGRAPH
	"E0 AF	7258",	#CJK UNIFIED IDEOGRAPH
	"E0 B0	7274",	#CJK UNIFIED IDEOGRAPH
	"E0 B1	727E",	#CJK UNIFIED IDEOGRAPH
	"E0 B2	7282",	#CJK UNIFIED IDEOGRAPH
	"E0 B3	7281",	#CJK UNIFIED IDEOGRAPH
	"E0 B4	7287",	#CJK UNIFIED IDEOGRAPH
	"E0 B5	7292",	#CJK UNIFIED IDEOGRAPH
	"E0 B6	7296",	#CJK UNIFIED IDEOGRAPH
	"E0 B7	72A2",	#CJK UNIFIED IDEOGRAPH
	"E0 B8	72A7",	#CJK UNIFIED IDEOGRAPH
	"E0 B9	72B9",	#CJK UNIFIED IDEOGRAPH
	"E0 BA	72B2",	#CJK UNIFIED IDEOGRAPH
	"E0 BB	72C3",	#CJK UNIFIED IDEOGRAPH
	"E0 BC	72C6",	#CJK UNIFIED IDEOGRAPH
	"E0 BD	72C4",	#CJK UNIFIED IDEOGRAPH
	"E0 BE	72CE",	#CJK UNIFIED IDEOGRAPH
	"E0 BF	72D2",	#CJK UNIFIED IDEOGRAPH
	"E0 C0	72E2",	#CJK UNIFIED IDEOGRAPH
	"E0 C1	72E0",	#CJK UNIFIED IDEOGRAPH
	"E0 C2	72E1",	#CJK UNIFIED IDEOGRAPH
	"E0 C3	72F9",	#CJK UNIFIED IDEOGRAPH
	"E0 C4	72F7",	#CJK UNIFIED IDEOGRAPH
	"E0 C5	500F",	#CJK UNIFIED IDEOGRAPH
	"E0 C6	7317",	#CJK UNIFIED IDEOGRAPH
	"E0 C7	730A",	#CJK UNIFIED IDEOGRAPH
	"E0 C8	731C",	#CJK UNIFIED IDEOGRAPH
	"E0 C9	7316",	#CJK UNIFIED IDEOGRAPH
	"E0 CA	731D",	#CJK UNIFIED IDEOGRAPH
	"E0 CB	7334",	#CJK UNIFIED IDEOGRAPH
	"E0 CC	732F",	#CJK UNIFIED IDEOGRAPH
	"E0 CD	7329",	#CJK UNIFIED IDEOGRAPH
	"E0 CE	7325",	#CJK UNIFIED IDEOGRAPH
	"E0 CF	733E",	#CJK UNIFIED IDEOGRAPH
	"E0 D0	734E",	#CJK UNIFIED IDEOGRAPH
	"E0 D1	734F",	#CJK UNIFIED IDEOGRAPH
	"E0 D2	9ED8",	#CJK UNIFIED IDEOGRAPH
	"E0 D3	7357",	#CJK UNIFIED IDEOGRAPH
	"E0 D4	736A",	#CJK UNIFIED IDEOGRAPH
	"E0 D5	7368",	#CJK UNIFIED IDEOGRAPH
	"E0 D6	7370",	#CJK UNIFIED IDEOGRAPH
	"E0 D7	7378",	#CJK UNIFIED IDEOGRAPH
	"E0 D8	7375",	#CJK UNIFIED IDEOGRAPH
	"E0 D9	737B",	#CJK UNIFIED IDEOGRAPH
	"E0 DA	737A",	#CJK UNIFIED IDEOGRAPH
	"E0 DB	73C8",	#CJK UNIFIED IDEOGRAPH
	"E0 DC	73B3",	#CJK UNIFIED IDEOGRAPH
	"E0 DD	73CE",	#CJK UNIFIED IDEOGRAPH
	"E0 DE	73BB",	#CJK UNIFIED IDEOGRAPH
	"E0 DF	73C0",	#CJK UNIFIED IDEOGRAPH
	"E0 E0	73E5",	#CJK UNIFIED IDEOGRAPH
	"E0 E1	73EE",	#CJK UNIFIED IDEOGRAPH
	"E0 E2	73DE",	#CJK UNIFIED IDEOGRAPH
	"E0 E3	74A2",	#CJK UNIFIED IDEOGRAPH
	"E0 E4	7405",	#CJK UNIFIED IDEOGRAPH
	"E0 E5	746F",	#CJK UNIFIED IDEOGRAPH
	"E0 E6	7425",	#CJK UNIFIED IDEOGRAPH
	"E0 E7	73F8",	#CJK UNIFIED IDEOGRAPH
	"E0 E8	7432",	#CJK UNIFIED IDEOGRAPH
	"E0 E9	743A",	#CJK UNIFIED IDEOGRAPH
	"E0 EA	7455",	#CJK UNIFIED IDEOGRAPH
	"E0 EB	743F",	#CJK UNIFIED IDEOGRAPH
	"E0 EC	745F",	#CJK UNIFIED IDEOGRAPH
	"E0 ED	7459",	#CJK UNIFIED IDEOGRAPH
	"E0 EE	7441",	#CJK UNIFIED IDEOGRAPH
	"E0 EF	745C",	#CJK UNIFIED IDEOGRAPH
	"E0 F0	7469",	#CJK UNIFIED IDEOGRAPH
	"E0 F1	7470",	#CJK UNIFIED IDEOGRAPH
	"E0 F2	7463",	#CJK UNIFIED IDEOGRAPH
	"E0 F3	746A",	#CJK UNIFIED IDEOGRAPH
	"E0 F4	7476",	#CJK UNIFIED IDEOGRAPH
	"E0 F5	747E",	#CJK UNIFIED IDEOGRAPH
	"E0 F6	748B",	#CJK UNIFIED IDEOGRAPH
	"E0 F7	749E",	#CJK UNIFIED IDEOGRAPH
	"E0 F8	74A7",	#CJK UNIFIED IDEOGRAPH
	"E0 F9	74CA",	#CJK UNIFIED IDEOGRAPH
	"E0 FA	74CF",	#CJK UNIFIED IDEOGRAPH
	"E0 FB	74D4",	#CJK UNIFIED IDEOGRAPH
	"E0 FC	73F1",	#CJK UNIFIED IDEOGRAPH
	"E1 40	74E0",	#CJK UNIFIED IDEOGRAPH
	"E1 41	74E3",	#CJK UNIFIED IDEOGRAPH
	"E1 42	74E7",	#CJK UNIFIED IDEOGRAPH
	"E1 43	74E9",	#CJK UNIFIED IDEOGRAPH
	"E1 44	74EE",	#CJK UNIFIED IDEOGRAPH
	"E1 45	74F2",	#CJK UNIFIED IDEOGRAPH
	"E1 46	74F0",	#CJK UNIFIED IDEOGRAPH
	"E1 47	74F1",	#CJK UNIFIED IDEOGRAPH
	"E1 48	74F8",	#CJK UNIFIED IDEOGRAPH
	"E1 49	74F7",	#CJK UNIFIED IDEOGRAPH
	"E1 4A	7504",	#CJK UNIFIED IDEOGRAPH
	"E1 4B	7503",	#CJK UNIFIED IDEOGRAPH
	"E1 4C	7505",	#CJK UNIFIED IDEOGRAPH
	"E1 4D	750C",	#CJK UNIFIED IDEOGRAPH
	"E1 4E	750E",	#CJK UNIFIED IDEOGRAPH
	"E1 4F	750D",	#CJK UNIFIED IDEOGRAPH
	"E1 50	7515",	#CJK UNIFIED IDEOGRAPH
	"E1 51	7513",	#CJK UNIFIED IDEOGRAPH
	"E1 52	751E",	#CJK UNIFIED IDEOGRAPH
	"E1 53	7526",	#CJK UNIFIED IDEOGRAPH
	"E1 54	752C",	#CJK UNIFIED IDEOGRAPH
	"E1 55	753C",	#CJK UNIFIED IDEOGRAPH
	"E1 56	7544",	#CJK UNIFIED IDEOGRAPH
	"E1 57	754D",	#CJK UNIFIED IDEOGRAPH
	"E1 58	754A",	#CJK UNIFIED IDEOGRAPH
	"E1 59	7549",	#CJK UNIFIED IDEOGRAPH
	"E1 5A	755B",	#CJK UNIFIED IDEOGRAPH
	"E1 5B	7546",	#CJK UNIFIED IDEOGRAPH
	"E1 5C	755A",	#CJK UNIFIED IDEOGRAPH
	"E1 5D	7569",	#CJK UNIFIED IDEOGRAPH
	"E1 5E	7564",	#CJK UNIFIED IDEOGRAPH
	"E1 5F	7567",	#CJK UNIFIED IDEOGRAPH
	"E1 60	756B",	#CJK UNIFIED IDEOGRAPH
	"E1 61	756D",	#CJK UNIFIED IDEOGRAPH
	"E1 62	7578",	#CJK UNIFIED IDEOGRAPH
	"E1 63	7576",	#CJK UNIFIED IDEOGRAPH
	"E1 64	7586",	#CJK UNIFIED IDEOGRAPH
	"E1 65	7587",	#CJK UNIFIED IDEOGRAPH
	"E1 66	7574",	#CJK UNIFIED IDEOGRAPH
	"E1 67	758A",	#CJK UNIFIED IDEOGRAPH
	"E1 68	7589",	#CJK UNIFIED IDEOGRAPH
	"E1 69	7582",	#CJK UNIFIED IDEOGRAPH
	"E1 6A	7594",	#CJK UNIFIED IDEOGRAPH
	"E1 6B	759A",	#CJK UNIFIED IDEOGRAPH
	"E1 6C	759D",	#CJK UNIFIED IDEOGRAPH
	"E1 6D	75A5",	#CJK UNIFIED IDEOGRAPH
	"E1 6E	75A3",	#CJK UNIFIED IDEOGRAPH
	"E1 6F	75C2",	#CJK UNIFIED IDEOGRAPH
	"E1 70	75B3",	#CJK UNIFIED IDEOGRAPH
	"E1 71	75C3",	#CJK UNIFIED IDEOGRAPH
	"E1 72	75B5",	#CJK UNIFIED IDEOGRAPH
	"E1 73	75BD",	#CJK UNIFIED IDEOGRAPH
	"E1 74	75B8",	#CJK UNIFIED IDEOGRAPH
	"E1 75	75BC",	#CJK UNIFIED IDEOGRAPH
	"E1 76	75B1",	#CJK UNIFIED IDEOGRAPH
	"E1 77	75CD",	#CJK UNIFIED IDEOGRAPH
	"E1 78	75CA",	#CJK UNIFIED IDEOGRAPH
	"E1 79	75D2",	#CJK UNIFIED IDEOGRAPH
	"E1 7A	75D9",	#CJK UNIFIED IDEOGRAPH
	"E1 7B	75E3",	#CJK UNIFIED IDEOGRAPH
	"E1 7C	75DE",	#CJK UNIFIED IDEOGRAPH
	"E1 7D	75FE",	#CJK UNIFIED IDEOGRAPH
	"E1 7E	75FF",	#CJK UNIFIED IDEOGRAPH
	"E1 80	75FC",	#CJK UNIFIED IDEOGRAPH
	"E1 81	7601",	#CJK UNIFIED IDEOGRAPH
	"E1 82	75F0",	#CJK UNIFIED IDEOGRAPH
	"E1 83	75FA",	#CJK UNIFIED IDEOGRAPH
	"E1 84	75F2",	#CJK UNIFIED IDEOGRAPH
	"E1 85	75F3",	#CJK UNIFIED IDEOGRAPH
	"E1 86	760B",	#CJK UNIFIED IDEOGRAPH
	"E1 87	760D",	#CJK UNIFIED IDEOGRAPH
	"E1 88	7609",	#CJK UNIFIED IDEOGRAPH
	"E1 89	761F",	#CJK UNIFIED IDEOGRAPH
	"E1 8A	7627",	#CJK UNIFIED IDEOGRAPH
	"E1 8B	7620",	#CJK UNIFIED IDEOGRAPH
	"E1 8C	7621",	#CJK UNIFIED IDEOGRAPH
	"E1 8D	7622",	#CJK UNIFIED IDEOGRAPH
	"E1 8E	7624",	#CJK UNIFIED IDEOGRAPH
	"E1 8F	7634",	#CJK UNIFIED IDEOGRAPH
	"E1 90	7630",	#CJK UNIFIED IDEOGRAPH
	"E1 91	763B",	#CJK UNIFIED IDEOGRAPH
	"E1 92	7647",	#CJK UNIFIED IDEOGRAPH
	"E1 93	7648",	#CJK UNIFIED IDEOGRAPH
	"E1 94	7646",	#CJK UNIFIED IDEOGRAPH
	"E1 95	765C",	#CJK UNIFIED IDEOGRAPH
	"E1 96	7658",	#CJK UNIFIED IDEOGRAPH
	"E1 97	7661",	#CJK UNIFIED IDEOGRAPH
	"E1 98	7662",	#CJK UNIFIED IDEOGRAPH
	"E1 99	7668",	#CJK UNIFIED IDEOGRAPH
	"E1 9A	7669",	#CJK UNIFIED IDEOGRAPH
	"E1 9B	766A",	#CJK UNIFIED IDEOGRAPH
	"E1 9C	7667",	#CJK UNIFIED IDEOGRAPH
	"E1 9D	766C",	#CJK UNIFIED IDEOGRAPH
	"E1 9E	7670",	#CJK UNIFIED IDEOGRAPH
	"E1 9F	7672",	#CJK UNIFIED IDEOGRAPH
	"E1 A0	7676",	#CJK UNIFIED IDEOGRAPH
	"E1 A1	7678",	#CJK UNIFIED IDEOGRAPH
	"E1 A2	767C",	#CJK UNIFIED IDEOGRAPH
	"E1 A3	7680",	#CJK UNIFIED IDEOGRAPH
	"E1 A4	7683",	#CJK UNIFIED IDEOGRAPH
	"E1 A5	7688",	#CJK UNIFIED IDEOGRAPH
	"E1 A6	768B",	#CJK UNIFIED IDEOGRAPH
	"E1 A7	768E",	#CJK UNIFIED IDEOGRAPH
	"E1 A8	7696",	#CJK UNIFIED IDEOGRAPH
	"E1 A9	7693",	#CJK UNIFIED IDEOGRAPH
	"E1 AA	7699",	#CJK UNIFIED IDEOGRAPH
	"E1 AB	769A",	#CJK UNIFIED IDEOGRAPH
	"E1 AC	76B0",	#CJK UNIFIED IDEOGRAPH
	"E1 AD	76B4",	#CJK UNIFIED IDEOGRAPH
	"E1 AE	76B8",	#CJK UNIFIED IDEOGRAPH
	"E1 AF	76B9",	#CJK UNIFIED IDEOGRAPH
	"E1 B0	76BA",	#CJK UNIFIED IDEOGRAPH
	"E1 B1	76C2",	#CJK UNIFIED IDEOGRAPH
	"E1 B2	76CD",	#CJK UNIFIED IDEOGRAPH
	"E1 B3	76D6",	#CJK UNIFIED IDEOGRAPH
	"E1 B4	76D2",	#CJK UNIFIED IDEOGRAPH
	"E1 B5	76DE",	#CJK UNIFIED IDEOGRAPH
	"E1 B6	76E1",	#CJK UNIFIED IDEOGRAPH
	"E1 B7	76E5",	#CJK UNIFIED IDEOGRAPH
	"E1 B8	76E7",	#CJK UNIFIED IDEOGRAPH
	"E1 B9	76EA",	#CJK UNIFIED IDEOGRAPH
	"E1 BA	862F",	#CJK UNIFIED IDEOGRAPH
	"E1 BB	76FB",	#CJK UNIFIED IDEOGRAPH
	"E1 BC	7708",	#CJK UNIFIED IDEOGRAPH
	"E1 BD	7707",	#CJK UNIFIED IDEOGRAPH
	"E1 BE	7704",	#CJK UNIFIED IDEOGRAPH
	"E1 BF	7729",	#CJK UNIFIED IDEOGRAPH
	"E1 C0	7724",	#CJK UNIFIED IDEOGRAPH
	"E1 C1	771E",	#CJK UNIFIED IDEOGRAPH
	"E1 C2	7725",	#CJK UNIFIED IDEOGRAPH
	"E1 C3	7726",	#CJK UNIFIED IDEOGRAPH
	"E1 C4	771B",	#CJK UNIFIED IDEOGRAPH
	"E1 C5	7737",	#CJK UNIFIED IDEOGRAPH
	"E1 C6	7738",	#CJK UNIFIED IDEOGRAPH
	"E1 C7	7747",	#CJK UNIFIED IDEOGRAPH
	"E1 C8	775A",	#CJK UNIFIED IDEOGRAPH
	"E1 C9	7768",	#CJK UNIFIED IDEOGRAPH
	"E1 CA	776B",	#CJK UNIFIED IDEOGRAPH
	"E1 CB	775B",	#CJK UNIFIED IDEOGRAPH
	"E1 CC	7765",	#CJK UNIFIED IDEOGRAPH
	"E1 CD	777F",	#CJK UNIFIED IDEOGRAPH
	"E1 CE	777E",	#CJK UNIFIED IDEOGRAPH
	"E1 CF	7779",	#CJK UNIFIED IDEOGRAPH
	"E1 D0	778E",	#CJK UNIFIED IDEOGRAPH
	"E1 D1	778B",	#CJK UNIFIED IDEOGRAPH
	"E1 D2	7791",	#CJK UNIFIED IDEOGRAPH
	"E1 D3	77A0",	#CJK UNIFIED IDEOGRAPH
	"E1 D4	779E",	#CJK UNIFIED IDEOGRAPH
	"E1 D5	77B0",	#CJK UNIFIED IDEOGRAPH
	"E1 D6	77B6",	#CJK UNIFIED IDEOGRAPH
	"E1 D7	77B9",	#CJK UNIFIED IDEOGRAPH
	"E1 D8	77BF",	#CJK UNIFIED IDEOGRAPH
	"E1 D9	77BC",	#CJK UNIFIED IDEOGRAPH
	"E1 DA	77BD",	#CJK UNIFIED IDEOGRAPH
	"E1 DB	77BB",	#CJK UNIFIED IDEOGRAPH
	"E1 DC	77C7",	#CJK UNIFIED IDEOGRAPH
	"E1 DD	77CD",	#CJK UNIFIED IDEOGRAPH
	"E1 DE	77D7",	#CJK UNIFIED IDEOGRAPH
	"E1 DF	77DA",	#CJK UNIFIED IDEOGRAPH
	"E1 E0	77DC",	#CJK UNIFIED IDEOGRAPH
	"E1 E1	77E3",	#CJK UNIFIED IDEOGRAPH
	"E1 E2	77EE",	#CJK UNIFIED IDEOGRAPH
	"E1 E3	77FC",	#CJK UNIFIED IDEOGRAPH
	"E1 E4	780C",	#CJK UNIFIED IDEOGRAPH
	"E1 E5	7812",	#CJK UNIFIED IDEOGRAPH
	"E1 E6	7926",	#CJK UNIFIED IDEOGRAPH
	"E1 E7	7820",	#CJK UNIFIED IDEOGRAPH
	"E1 E8	792A",	#CJK UNIFIED IDEOGRAPH
	"E1 E9	7845",	#CJK UNIFIED IDEOGRAPH
	"E1 EA	788E",	#CJK UNIFIED IDEOGRAPH
	"E1 EB	7874",	#CJK UNIFIED IDEOGRAPH
	"E1 EC	7886",	#CJK UNIFIED IDEOGRAPH
	"E1 ED	787C",	#CJK UNIFIED IDEOGRAPH
	"E1 EE	789A",	#CJK UNIFIED IDEOGRAPH
	"E1 EF	788C",	#CJK UNIFIED IDEOGRAPH
	"E1 F0	78A3",	#CJK UNIFIED IDEOGRAPH
	"E1 F1	78B5",	#CJK UNIFIED IDEOGRAPH
	"E1 F2	78AA",	#CJK UNIFIED IDEOGRAPH
	"E1 F3	78AF",	#CJK UNIFIED IDEOGRAPH
	"E1 F4	78D1",	#CJK UNIFIED IDEOGRAPH
	"E1 F5	78C6",	#CJK UNIFIED IDEOGRAPH
	"E1 F6	78CB",	#CJK UNIFIED IDEOGRAPH
	"E1 F7	78D4",	#CJK UNIFIED IDEOGRAPH
	"E1 F8	78BE",	#CJK UNIFIED IDEOGRAPH
	"E1 F9	78BC",	#CJK UNIFIED IDEOGRAPH
	"E1 FA	78C5",	#CJK UNIFIED IDEOGRAPH
	"E1 FB	78CA",	#CJK UNIFIED IDEOGRAPH
	"E1 FC	78EC",	#CJK UNIFIED IDEOGRAPH
	"E2 40	78E7",	#CJK UNIFIED IDEOGRAPH
	"E2 41	78DA",	#CJK UNIFIED IDEOGRAPH
	"E2 42	78FD",	#CJK UNIFIED IDEOGRAPH
	"E2 43	78F4",	#CJK UNIFIED IDEOGRAPH
	"E2 44	7907",	#CJK UNIFIED IDEOGRAPH
	"E2 45	7912",	#CJK UNIFIED IDEOGRAPH
	"E2 46	7911",	#CJK UNIFIED IDEOGRAPH
	"E2 47	7919",	#CJK UNIFIED IDEOGRAPH
	"E2 48	792C",	#CJK UNIFIED IDEOGRAPH
	"E2 49	792B",	#CJK UNIFIED IDEOGRAPH
	"E2 4A	7940",	#CJK UNIFIED IDEOGRAPH
	"E2 4B	7960",	#CJK UNIFIED IDEOGRAPH
	"E2 4C	7957",	#CJK UNIFIED IDEOGRAPH
	"E2 4D	795F",	#CJK UNIFIED IDEOGRAPH
	"E2 4E	795A",	#CJK UNIFIED IDEOGRAPH
	"E2 4F	7955",	#CJK UNIFIED IDEOGRAPH
	"E2 50	7953",	#CJK UNIFIED IDEOGRAPH
	"E2 51	797A",	#CJK UNIFIED IDEOGRAPH
	"E2 52	797F",	#CJK UNIFIED IDEOGRAPH
	"E2 53	798A",	#CJK UNIFIED IDEOGRAPH
	"E2 54	799D",	#CJK UNIFIED IDEOGRAPH
	"E2 55	79A7",	#CJK UNIFIED IDEOGRAPH
	"E2 56	9F4B",	#CJK UNIFIED IDEOGRAPH
	"E2 57	79AA",	#CJK UNIFIED IDEOGRAPH
	"E2 58	79AE",	#CJK UNIFIED IDEOGRAPH
	"E2 59	79B3",	#CJK UNIFIED IDEOGRAPH
	"E2 5A	79B9",	#CJK UNIFIED IDEOGRAPH
	"E2 5B	79BA",	#CJK UNIFIED IDEOGRAPH
	"E2 5C	79C9",	#CJK UNIFIED IDEOGRAPH
	"E2 5D	79D5",	#CJK UNIFIED IDEOGRAPH
	"E2 5E	79E7",	#CJK UNIFIED IDEOGRAPH
	"E2 5F	79EC",	#CJK UNIFIED IDEOGRAPH
	"E2 60	79E1",	#CJK UNIFIED IDEOGRAPH
	"E2 61	79E3",	#CJK UNIFIED IDEOGRAPH
	"E2 62	7A08",	#CJK UNIFIED IDEOGRAPH
	"E2 63	7A0D",	#CJK UNIFIED IDEOGRAPH
	"E2 64	7A18",	#CJK UNIFIED IDEOGRAPH
	"E2 65	7A19",	#CJK UNIFIED IDEOGRAPH
	"E2 66	7A20",	#CJK UNIFIED IDEOGRAPH
	"E2 67	7A1F",	#CJK UNIFIED IDEOGRAPH
	"E2 68	7980",	#CJK UNIFIED IDEOGRAPH
	"E2 69	7A31",	#CJK UNIFIED IDEOGRAPH
	"E2 6A	7A3B",	#CJK UNIFIED IDEOGRAPH
	"E2 6B	7A3E",	#CJK UNIFIED IDEOGRAPH
	"E2 6C	7A37",	#CJK UNIFIED IDEOGRAPH
	"E2 6D	7A43",	#CJK UNIFIED IDEOGRAPH
	"E2 6E	7A57",	#CJK UNIFIED IDEOGRAPH
	"E2 6F	7A49",	#CJK UNIFIED IDEOGRAPH
	"E2 70	7A61",	#CJK UNIFIED IDEOGRAPH
	"E2 71	7A62",	#CJK UNIFIED IDEOGRAPH
	"E2 72	7A69",	#CJK UNIFIED IDEOGRAPH
	"E2 73	9F9D",	#CJK UNIFIED IDEOGRAPH
	"E2 74	7A70",	#CJK UNIFIED IDEOGRAPH
	"E2 75	7A79",	#CJK UNIFIED IDEOGRAPH
	"E2 76	7A7D",	#CJK UNIFIED IDEOGRAPH
	"E2 77	7A88",	#CJK UNIFIED IDEOGRAPH
	"E2 78	7A97",	#CJK UNIFIED IDEOGRAPH
	"E2 79	7A95",	#CJK UNIFIED IDEOGRAPH
	"E2 7A	7A98",	#CJK UNIFIED IDEOGRAPH
	"E2 7B	7A96",	#CJK UNIFIED IDEOGRAPH
	"E2 7C	7AA9",	#CJK UNIFIED IDEOGRAPH
	"E2 7D	7AC8",	#CJK UNIFIED IDEOGRAPH
	"E2 7E	7AB0",	#CJK UNIFIED IDEOGRAPH
	"E2 80	7AB6",	#CJK UNIFIED IDEOGRAPH
	"E2 81	7AC5",	#CJK UNIFIED IDEOGRAPH
	"E2 82	7AC4",	#CJK UNIFIED IDEOGRAPH
	"E2 83	7ABF",	#CJK UNIFIED IDEOGRAPH
	"E2 84	9083",	#CJK UNIFIED IDEOGRAPH
	"E2 85	7AC7",	#CJK UNIFIED IDEOGRAPH
	"E2 86	7ACA",	#CJK UNIFIED IDEOGRAPH
	"E2 87	7ACD",	#CJK UNIFIED IDEOGRAPH
	"E2 88	7ACF",	#CJK UNIFIED IDEOGRAPH
	"E2 89	7AD5",	#CJK UNIFIED IDEOGRAPH
	"E2 8A	7AD3",	#CJK UNIFIED IDEOGRAPH
	"E2 8B	7AD9",	#CJK UNIFIED IDEOGRAPH
	"E2 8C	7ADA",	#CJK UNIFIED IDEOGRAPH
	"E2 8D	7ADD",	#CJK UNIFIED IDEOGRAPH
	"E2 8E	7AE1",	#CJK UNIFIED IDEOGRAPH
	"E2 8F	7AE2",	#CJK UNIFIED IDEOGRAPH
	"E2 90	7AE6",	#CJK UNIFIED IDEOGRAPH
	"E2 91	7AED",	#CJK UNIFIED IDEOGRAPH
	"E2 92	7AF0",	#CJK UNIFIED IDEOGRAPH
	"E2 93	7B02",	#CJK UNIFIED IDEOGRAPH
	"E2 94	7B0F",	#CJK UNIFIED IDEOGRAPH
	"E2 95	7B0A",	#CJK UNIFIED IDEOGRAPH
	"E2 96	7B06",	#CJK UNIFIED IDEOGRAPH
	"E2 97	7B33",	#CJK UNIFIED IDEOGRAPH
	"E2 98	7B18",	#CJK UNIFIED IDEOGRAPH
	"E2 99	7B19",	#CJK UNIFIED IDEOGRAPH
	"E2 9A	7B1E",	#CJK UNIFIED IDEOGRAPH
	"E2 9B	7B35",	#CJK UNIFIED IDEOGRAPH
	"E2 9C	7B28",	#CJK UNIFIED IDEOGRAPH
	"E2 9D	7B36",	#CJK UNIFIED IDEOGRAPH
	"E2 9E	7B50",	#CJK UNIFIED IDEOGRAPH
	"E2 9F	7B7A",	#CJK UNIFIED IDEOGRAPH
	"E2 A0	7B04",	#CJK UNIFIED IDEOGRAPH
	"E2 A1	7B4D",	#CJK UNIFIED IDEOGRAPH
	"E2 A2	7B0B",	#CJK UNIFIED IDEOGRAPH
	"E2 A3	7B4C",	#CJK UNIFIED IDEOGRAPH
	"E2 A4	7B45",	#CJK UNIFIED IDEOGRAPH
	"E2 A5	7B75",	#CJK UNIFIED IDEOGRAPH
	"E2 A6	7B65",	#CJK UNIFIED IDEOGRAPH
	"E2 A7	7B74",	#CJK UNIFIED IDEOGRAPH
	"E2 A8	7B67",	#CJK UNIFIED IDEOGRAPH
	"E2 A9	7B70",	#CJK UNIFIED IDEOGRAPH
	"E2 AA	7B71",	#CJK UNIFIED IDEOGRAPH
	"E2 AB	7B6C",	#CJK UNIFIED IDEOGRAPH
	"E2 AC	7B6E",	#CJK UNIFIED IDEOGRAPH
	"E2 AD	7B9D",	#CJK UNIFIED IDEOGRAPH
	"E2 AE	7B98",	#CJK UNIFIED IDEOGRAPH
	"E2 AF	7B9F",	#CJK UNIFIED IDEOGRAPH
	"E2 B0	7B8D",	#CJK UNIFIED IDEOGRAPH
	"E2 B1	7B9C",	#CJK UNIFIED IDEOGRAPH
	"E2 B2	7B9A",	#CJK UNIFIED IDEOGRAPH
	"E2 B3	7B8B",	#CJK UNIFIED IDEOGRAPH
	"E2 B4	7B92",	#CJK UNIFIED IDEOGRAPH
	"E2 B5	7B8F",	#CJK UNIFIED IDEOGRAPH
	"E2 B6	7B5D",	#CJK UNIFIED IDEOGRAPH
	"E2 B7	7B99",	#CJK UNIFIED IDEOGRAPH
	"E2 B8	7BCB",	#CJK UNIFIED IDEOGRAPH
	"E2 B9	7BC1",	#CJK UNIFIED IDEOGRAPH
	"E2 BA	7BCC",	#CJK UNIFIED IDEOGRAPH
	"E2 BB	7BCF",	#CJK UNIFIED IDEOGRAPH
	"E2 BC	7BB4",	#CJK UNIFIED IDEOGRAPH
	"E2 BD	7BC6",	#CJK UNIFIED IDEOGRAPH
	"E2 BE	7BDD",	#CJK UNIFIED IDEOGRAPH
	"E2 BF	7BE9",	#CJK UNIFIED IDEOGRAPH
	"E2 C0	7C11",	#CJK UNIFIED IDEOGRAPH
	"E2 C1	7C14",	#CJK UNIFIED IDEOGRAPH
	"E2 C2	7BE6",	#CJK UNIFIED IDEOGRAPH
	"E2 C3	7BE5",	#CJK UNIFIED IDEOGRAPH
	"E2 C4	7C60",	#CJK UNIFIED IDEOGRAPH
	"E2 C5	7C00",	#CJK UNIFIED IDEOGRAPH
	"E2 C6	7C07",	#CJK UNIFIED IDEOGRAPH
	"E2 C7	7C13",	#CJK UNIFIED IDEOGRAPH
	"E2 C8	7BF3",	#CJK UNIFIED IDEOGRAPH
	"E2 C9	7BF7",	#CJK UNIFIED IDEOGRAPH
	"E2 CA	7C17",	#CJK UNIFIED IDEOGRAPH
	"E2 CB	7C0D",	#CJK UNIFIED IDEOGRAPH
	"E2 CC	7BF6",	#CJK UNIFIED IDEOGRAPH
	"E2 CD	7C23",	#CJK UNIFIED IDEOGRAPH
	"E2 CE	7C27",	#CJK UNIFIED IDEOGRAPH
	"E2 CF	7C2A",	#CJK UNIFIED IDEOGRAPH
	"E2 D0	7C1F",	#CJK UNIFIED IDEOGRAPH
	"E2 D1	7C37",	#CJK UNIFIED IDEOGRAPH
	"E2 D2	7C2B",	#CJK UNIFIED IDEOGRAPH
	"E2 D3	7C3D",	#CJK UNIFIED IDEOGRAPH
	"E2 D4	7C4C",	#CJK UNIFIED IDEOGRAPH
	"E2 D5	7C43",	#CJK UNIFIED IDEOGRAPH
	"E2 D6	7C54",	#CJK UNIFIED IDEOGRAPH
	"E2 D7	7C4F",	#CJK UNIFIED IDEOGRAPH
	"E2 D8	7C40",	#CJK UNIFIED IDEOGRAPH
	"E2 D9	7C50",	#CJK UNIFIED IDEOGRAPH
	"E2 DA	7C58",	#CJK UNIFIED IDEOGRAPH
	"E2 DB	7C5F",	#CJK UNIFIED IDEOGRAPH
	"E2 DC	7C64",	#CJK UNIFIED IDEOGRAPH
	"E2 DD	7C56",	#CJK UNIFIED IDEOGRAPH
	"E2 DE	7C65",	#CJK UNIFIED IDEOGRAPH
	"E2 DF	7C6C",	#CJK UNIFIED IDEOGRAPH
	"E2 E0	7C75",	#CJK UNIFIED IDEOGRAPH
	"E2 E1	7C83",	#CJK UNIFIED IDEOGRAPH
	"E2 E2	7C90",	#CJK UNIFIED IDEOGRAPH
	"E2 E3	7CA4",	#CJK UNIFIED IDEOGRAPH
	"E2 E4	7CAD",	#CJK UNIFIED IDEOGRAPH
	"E2 E5	7CA2",	#CJK UNIFIED IDEOGRAPH
	"E2 E6	7CAB",	#CJK UNIFIED IDEOGRAPH
	"E2 E7	7CA1",	#CJK UNIFIED IDEOGRAPH
	"E2 E8	7CA8",	#CJK UNIFIED IDEOGRAPH
	"E2 E9	7CB3",	#CJK UNIFIED IDEOGRAPH
	"E2 EA	7CB2",	#CJK UNIFIED IDEOGRAPH
	"E2 EB	7CB1",	#CJK UNIFIED IDEOGRAPH
	"E2 EC	7CAE",	#CJK UNIFIED IDEOGRAPH
	"E2 ED	7CB9",	#CJK UNIFIED IDEOGRAPH
	"E2 EE	7CBD",	#CJK UNIFIED IDEOGRAPH
	"E2 EF	7CC0",	#CJK UNIFIED IDEOGRAPH
	"E2 F0	7CC5",	#CJK UNIFIED IDEOGRAPH
	"E2 F1	7CC2",	#CJK UNIFIED IDEOGRAPH
	"E2 F2	7CD8",	#CJK UNIFIED IDEOGRAPH
	"E2 F3	7CD2",	#CJK UNIFIED IDEOGRAPH
	"E2 F4	7CDC",	#CJK UNIFIED IDEOGRAPH
	"E2 F5	7CE2",	#CJK UNIFIED IDEOGRAPH
	"E2 F6	9B3B",	#CJK UNIFIED IDEOGRAPH
	"E2 F7	7CEF",	#CJK UNIFIED IDEOGRAPH
	"E2 F8	7CF2",	#CJK UNIFIED IDEOGRAPH
	"E2 F9	7CF4",	#CJK UNIFIED IDEOGRAPH
	"E2 FA	7CF6",	#CJK UNIFIED IDEOGRAPH
	"E2 FB	7CFA",	#CJK UNIFIED IDEOGRAPH
	"E2 FC	7D06",	#CJK UNIFIED IDEOGRAPH
	"E3 40	7D02",	#CJK UNIFIED IDEOGRAPH
	"E3 41	7D1C",	#CJK UNIFIED IDEOGRAPH
	"E3 42	7D15",	#CJK UNIFIED IDEOGRAPH
	"E3 43	7D0A",	#CJK UNIFIED IDEOGRAPH
	"E3 44	7D45",	#CJK UNIFIED IDEOGRAPH
	"E3 45	7D4B",	#CJK UNIFIED IDEOGRAPH
	"E3 46	7D2E",	#CJK UNIFIED IDEOGRAPH
	"E3 47	7D32",	#CJK UNIFIED IDEOGRAPH
	"E3 48	7D3F",	#CJK UNIFIED IDEOGRAPH
	"E3 49	7D35",	#CJK UNIFIED IDEOGRAPH
	"E3 4A	7D46",	#CJK UNIFIED IDEOGRAPH
	"E3 4B	7D73",	#CJK UNIFIED IDEOGRAPH
	"E3 4C	7D56",	#CJK UNIFIED IDEOGRAPH
	"E3 4D	7D4E",	#CJK UNIFIED IDEOGRAPH
	"E3 4E	7D72",	#CJK UNIFIED IDEOGRAPH
	"E3 4F	7D68",	#CJK UNIFIED IDEOGRAPH
	"E3 50	7D6E",	#CJK UNIFIED IDEOGRAPH
	"E3 51	7D4F",	#CJK UNIFIED IDEOGRAPH
	"E3 52	7D63",	#CJK UNIFIED IDEOGRAPH
	"E3 53	7D93",	#CJK UNIFIED IDEOGRAPH
	"E3 54	7D89",	#CJK UNIFIED IDEOGRAPH
	"E3 55	7D5B",	#CJK UNIFIED IDEOGRAPH
	"E3 56	7D8F",	#CJK UNIFIED IDEOGRAPH
	"E3 57	7D7D",	#CJK UNIFIED IDEOGRAPH
	"E3 58	7D9B",	#CJK UNIFIED IDEOGRAPH
	"E3 59	7DBA",	#CJK UNIFIED IDEOGRAPH
	"E3 5A	7DAE",	#CJK UNIFIED IDEOGRAPH
	"E3 5B	7DA3",	#CJK UNIFIED IDEOGRAPH
	"E3 5C	7DB5",	#CJK UNIFIED IDEOGRAPH
	"E3 5D	7DC7",	#CJK UNIFIED IDEOGRAPH
	"E3 5E	7DBD",	#CJK UNIFIED IDEOGRAPH
	"E3 5F	7DAB",	#CJK UNIFIED IDEOGRAPH
	"E3 60	7E3D",	#CJK UNIFIED IDEOGRAPH
	"E3 61	7DA2",	#CJK UNIFIED IDEOGRAPH
	"E3 62	7DAF",	#CJK UNIFIED IDEOGRAPH
	"E3 63	7DDC",	#CJK UNIFIED IDEOGRAPH
	"E3 64	7DB8",	#CJK UNIFIED IDEOGRAPH
	"E3 65	7D9F",	#CJK UNIFIED IDEOGRAPH
	"E3 66	7DB0",	#CJK UNIFIED IDEOGRAPH
	"E3 67	7DD8",	#CJK UNIFIED IDEOGRAPH
	"E3 68	7DDD",	#CJK UNIFIED IDEOGRAPH
	"E3 69	7DE4",	#CJK UNIFIED IDEOGRAPH
	"E3 6A	7DDE",	#CJK UNIFIED IDEOGRAPH
	"E3 6B	7DFB",	#CJK UNIFIED IDEOGRAPH
	"E3 6C	7DF2",	#CJK UNIFIED IDEOGRAPH
	"E3 6D	7DE1",	#CJK UNIFIED IDEOGRAPH
	"E3 6E	7E05",	#CJK UNIFIED IDEOGRAPH
	"E3 6F	7E0A",	#CJK UNIFIED IDEOGRAPH
	"E3 70	7E23",	#CJK UNIFIED IDEOGRAPH
	"E3 71	7E21",	#CJK UNIFIED IDEOGRAPH
	"E3 72	7E12",	#CJK UNIFIED IDEOGRAPH
	"E3 73	7E31",	#CJK UNIFIED IDEOGRAPH
	"E3 74	7E1F",	#CJK UNIFIED IDEOGRAPH
	"E3 75	7E09",	#CJK UNIFIED IDEOGRAPH
	"E3 76	7E0B",	#CJK UNIFIED IDEOGRAPH
	"E3 77	7E22",	#CJK UNIFIED IDEOGRAPH
	"E3 78	7E46",	#CJK UNIFIED IDEOGRAPH
	"E3 79	7E66",	#CJK UNIFIED IDEOGRAPH
	"E3 7A	7E3B",	#CJK UNIFIED IDEOGRAPH
	"E3 7B	7E35",	#CJK UNIFIED IDEOGRAPH
	"E3 7C	7E39",	#CJK UNIFIED IDEOGRAPH
	"E3 7D	7E43",	#CJK UNIFIED IDEOGRAPH
	"E3 7E	7E37",	#CJK UNIFIED IDEOGRAPH
	"E3 80	7E32",	#CJK UNIFIED IDEOGRAPH
	"E3 81	7E3A",	#CJK UNIFIED IDEOGRAPH
	"E3 82	7E67",	#CJK UNIFIED IDEOGRAPH
	"E3 83	7E5D",	#CJK UNIFIED IDEOGRAPH
	"E3 84	7E56",	#CJK UNIFIED IDEOGRAPH
	"E3 85	7E5E",	#CJK UNIFIED IDEOGRAPH
	"E3 86	7E59",	#CJK UNIFIED IDEOGRAPH
	"E3 87	7E5A",	#CJK UNIFIED IDEOGRAPH
	"E3 88	7E79",	#CJK UNIFIED IDEOGRAPH
	"E3 89	7E6A",	#CJK UNIFIED IDEOGRAPH
	"E3 8A	7E69",	#CJK UNIFIED IDEOGRAPH
	"E3 8B	7E7C",	#CJK UNIFIED IDEOGRAPH
	"E3 8C	7E7B",	#CJK UNIFIED IDEOGRAPH
	"E3 8D	7E83",	#CJK UNIFIED IDEOGRAPH
	"E3 8E	7DD5",	#CJK UNIFIED IDEOGRAPH
	"E3 8F	7E7D",	#CJK UNIFIED IDEOGRAPH
	"E3 90	8FAE",	#CJK UNIFIED IDEOGRAPH
	"E3 91	7E7F",	#CJK UNIFIED IDEOGRAPH
	"E3 92	7E88",	#CJK UNIFIED IDEOGRAPH
	"E3 93	7E89",	#CJK UNIFIED IDEOGRAPH
	"E3 94	7E8C",	#CJK UNIFIED IDEOGRAPH
	"E3 95	7E92",	#CJK UNIFIED IDEOGRAPH
	"E3 96	7E90",	#CJK UNIFIED IDEOGRAPH
	"E3 97	7E93",	#CJK UNIFIED IDEOGRAPH
	"E3 98	7E94",	#CJK UNIFIED IDEOGRAPH
	"E3 99	7E96",	#CJK UNIFIED IDEOGRAPH
	"E3 9A	7E8E",	#CJK UNIFIED IDEOGRAPH
	"E3 9B	7E9B",	#CJK UNIFIED IDEOGRAPH
	"E3 9C	7E9C",	#CJK UNIFIED IDEOGRAPH
	"E3 9D	7F38",	#CJK UNIFIED IDEOGRAPH
	"E3 9E	7F3A",	#CJK UNIFIED IDEOGRAPH
	"E3 9F	7F45",	#CJK UNIFIED IDEOGRAPH
	"E3 A0	7F4C",	#CJK UNIFIED IDEOGRAPH
	"E3 A1	7F4D",	#CJK UNIFIED IDEOGRAPH
	"E3 A2	7F4E",	#CJK UNIFIED IDEOGRAPH
	"E3 A3	7F50",	#CJK UNIFIED IDEOGRAPH
	"E3 A4	7F51",	#CJK UNIFIED IDEOGRAPH
	"E3 A5	7F55",	#CJK UNIFIED IDEOGRAPH
	"E3 A6	7F54",	#CJK UNIFIED IDEOGRAPH
	"E3 A7	7F58",	#CJK UNIFIED IDEOGRAPH
	"E3 A8	7F5F",	#CJK UNIFIED IDEOGRAPH
	"E3 A9	7F60",	#CJK UNIFIED IDEOGRAPH
	"E3 AA	7F68",	#CJK UNIFIED IDEOGRAPH
	"E3 AB	7F69",	#CJK UNIFIED IDEOGRAPH
	"E3 AC	7F67",	#CJK UNIFIED IDEOGRAPH
	"E3 AD	7F78",	#CJK UNIFIED IDEOGRAPH
	"E3 AE	7F82",	#CJK UNIFIED IDEOGRAPH
	"E3 AF	7F86",	#CJK UNIFIED IDEOGRAPH
	"E3 B0	7F83",	#CJK UNIFIED IDEOGRAPH
	"E3 B1	7F88",	#CJK UNIFIED IDEOGRAPH
	"E3 B2	7F87",	#CJK UNIFIED IDEOGRAPH
	"E3 B3	7F8C",	#CJK UNIFIED IDEOGRAPH
	"E3 B4	7F94",	#CJK UNIFIED IDEOGRAPH
	"E3 B5	7F9E",	#CJK UNIFIED IDEOGRAPH
	"E3 B6	7F9D",	#CJK UNIFIED IDEOGRAPH
	"E3 B7	7F9A",	#CJK UNIFIED IDEOGRAPH
	"E3 B8	7FA3",	#CJK UNIFIED IDEOGRAPH
	"E3 B9	7FAF",	#CJK UNIFIED IDEOGRAPH
	"E3 BA	7FB2",	#CJK UNIFIED IDEOGRAPH
	"E3 BB	7FB9",	#CJK UNIFIED IDEOGRAPH
	"E3 BC	7FAE",	#CJK UNIFIED IDEOGRAPH
	"E3 BD	7FB6",	#CJK UNIFIED IDEOGRAPH
	"E3 BE	7FB8",	#CJK UNIFIED IDEOGRAPH
	"E3 BF	8B71",	#CJK UNIFIED IDEOGRAPH
	"E3 C0	7FC5",	#CJK UNIFIED IDEOGRAPH
	"E3 C1	7FC6",	#CJK UNIFIED IDEOGRAPH
	"E3 C2	7FCA",	#CJK UNIFIED IDEOGRAPH
	"E3 C3	7FD5",	#CJK UNIFIED IDEOGRAPH
	"E3 C4	7FD4",	#CJK UNIFIED IDEOGRAPH
	"E3 C5	7FE1",	#CJK UNIFIED IDEOGRAPH
	"E3 C6	7FE6",	#CJK UNIFIED IDEOGRAPH
	"E3 C7	7FE9",	#CJK UNIFIED IDEOGRAPH
	"E3 C8	7FF3",	#CJK UNIFIED IDEOGRAPH
	"E3 C9	7FF9",	#CJK UNIFIED IDEOGRAPH
	"E3 CA	98DC",	#CJK UNIFIED IDEOGRAPH
	"E3 CB	8006",	#CJK UNIFIED IDEOGRAPH
	"E3 CC	8004",	#CJK UNIFIED IDEOGRAPH
	"E3 CD	800B",	#CJK UNIFIED IDEOGRAPH
	"E3 CE	8012",	#CJK UNIFIED IDEOGRAPH
	"E3 CF	8018",	#CJK UNIFIED IDEOGRAPH
	"E3 D0	8019",	#CJK UNIFIED IDEOGRAPH
	"E3 D1	801C",	#CJK UNIFIED IDEOGRAPH
	"E3 D2	8021",	#CJK UNIFIED IDEOGRAPH
	"E3 D3	8028",	#CJK UNIFIED IDEOGRAPH
	"E3 D4	803F",	#CJK UNIFIED IDEOGRAPH
	"E3 D5	803B",	#CJK UNIFIED IDEOGRAPH
	"E3 D6	804A",	#CJK UNIFIED IDEOGRAPH
	"E3 D7	8046",	#CJK UNIFIED IDEOGRAPH
	"E3 D8	8052",	#CJK UNIFIED IDEOGRAPH
	"E3 D9	8058",	#CJK UNIFIED IDEOGRAPH
	"E3 DA	805A",	#CJK UNIFIED IDEOGRAPH
	"E3 DB	805F",	#CJK UNIFIED IDEOGRAPH
	"E3 DC	8062",	#CJK UNIFIED IDEOGRAPH
	"E3 DD	8068",	#CJK UNIFIED IDEOGRAPH
	"E3 DE	8073",	#CJK UNIFIED IDEOGRAPH
	"E3 DF	8072",	#CJK UNIFIED IDEOGRAPH
	"E3 E0	8070",	#CJK UNIFIED IDEOGRAPH
	"E3 E1	8076",	#CJK UNIFIED IDEOGRAPH
	"E3 E2	8079",	#CJK UNIFIED IDEOGRAPH
	"E3 E3	807D",	#CJK UNIFIED IDEOGRAPH
	"E3 E4	807F",	#CJK UNIFIED IDEOGRAPH
	"E3 E5	8084",	#CJK UNIFIED IDEOGRAPH
	"E3 E6	8086",	#CJK UNIFIED IDEOGRAPH
	"E3 E7	8085",	#CJK UNIFIED IDEOGRAPH
	"E3 E8	809B",	#CJK UNIFIED IDEOGRAPH
	"E3 E9	8093",	#CJK UNIFIED IDEOGRAPH
	"E3 EA	809A",	#CJK UNIFIED IDEOGRAPH
	"E3 EB	80AD",	#CJK UNIFIED IDEOGRAPH
	"E3 EC	5190",	#CJK UNIFIED IDEOGRAPH
	"E3 ED	80AC",	#CJK UNIFIED IDEOGRAPH
	"E3 EE	80DB",	#CJK UNIFIED IDEOGRAPH
	"E3 EF	80E5",	#CJK UNIFIED IDEOGRAPH
	"E3 F0	80D9",	#CJK UNIFIED IDEOGRAPH
	"E3 F1	80DD",	#CJK UNIFIED IDEOGRAPH
	"E3 F2	80C4",	#CJK UNIFIED IDEOGRAPH
	"E3 F3	80DA",	#CJK UNIFIED IDEOGRAPH
	"E3 F4	80D6",	#CJK UNIFIED IDEOGRAPH
	"E3 F5	8109",	#CJK UNIFIED IDEOGRAPH
	"E3 F6	80EF",	#CJK UNIFIED IDEOGRAPH
	"E3 F7	80F1",	#CJK UNIFIED IDEOGRAPH
	"E3 F8	811B",	#CJK UNIFIED IDEOGRAPH
	"E3 F9	8129",	#CJK UNIFIED IDEOGRAPH
	"E3 FA	8123",	#CJK UNIFIED IDEOGRAPH
	"E3 FB	812F",	#CJK UNIFIED IDEOGRAPH
	"E3 FC	814B",	#CJK UNIFIED IDEOGRAPH
	"E4 40	968B",	#CJK UNIFIED IDEOGRAPH
	"E4 41	8146",	#CJK UNIFIED IDEOGRAPH
	"E4 42	813E",	#CJK UNIFIED IDEOGRAPH
	"E4 43	8153",	#CJK UNIFIED IDEOGRAPH
	"E4 44	8151",	#CJK UNIFIED IDEOGRAPH
	"E4 45	80FC",	#CJK UNIFIED IDEOGRAPH
	"E4 46	8171",	#CJK UNIFIED IDEOGRAPH
	"E4 47	816E",	#CJK UNIFIED IDEOGRAPH
	"E4 48	8165",	#CJK UNIFIED IDEOGRAPH
	"E4 49	8166",	#CJK UNIFIED IDEOGRAPH
	"E4 4A	8174",	#CJK UNIFIED IDEOGRAPH
	"E4 4B	8183",	#CJK UNIFIED IDEOGRAPH
	"E4 4C	8188",	#CJK UNIFIED IDEOGRAPH
	"E4 4D	818A",	#CJK UNIFIED IDEOGRAPH
	"E4 4E	8180",	#CJK UNIFIED IDEOGRAPH
	"E4 4F	8182",	#CJK UNIFIED IDEOGRAPH
	"E4 50	81A0",	#CJK UNIFIED IDEOGRAPH
	"E4 51	8195",	#CJK UNIFIED IDEOGRAPH
	"E4 52	81A4",	#CJK UNIFIED IDEOGRAPH
	"E4 53	81A3",	#CJK UNIFIED IDEOGRAPH
	"E4 54	815F",	#CJK UNIFIED IDEOGRAPH
	"E4 55	8193",	#CJK UNIFIED IDEOGRAPH
	"E4 56	81A9",	#CJK UNIFIED IDEOGRAPH
	"E4 57	81B0",	#CJK UNIFIED IDEOGRAPH
	"E4 58	81B5",	#CJK UNIFIED IDEOGRAPH
	"E4 59	81BE",	#CJK UNIFIED IDEOGRAPH
	"E4 5A	81B8",	#CJK UNIFIED IDEOGRAPH
	"E4 5B	81BD",	#CJK UNIFIED IDEOGRAPH
	"E4 5C	81C0",	#CJK UNIFIED IDEOGRAPH
	"E4 5D	81C2",	#CJK UNIFIED IDEOGRAPH
	"E4 5E	81BA",	#CJK UNIFIED IDEOGRAPH
	"E4 5F	81C9",	#CJK UNIFIED IDEOGRAPH
	"E4 60	81CD",	#CJK UNIFIED IDEOGRAPH
	"E4 61	81D1",	#CJK UNIFIED IDEOGRAPH
	"E4 62	81D9",	#CJK UNIFIED IDEOGRAPH
	"E4 63	81D8",	#CJK UNIFIED IDEOGRAPH
	"E4 64	81C8",	#CJK UNIFIED IDEOGRAPH
	"E4 65	81DA",	#CJK UNIFIED IDEOGRAPH
	"E4 66	81DF",	#CJK UNIFIED IDEOGRAPH
	"E4 67	81E0",	#CJK UNIFIED IDEOGRAPH
	"E4 68	81E7",	#CJK UNIFIED IDEOGRAPH
	"E4 69	81FA",	#CJK UNIFIED IDEOGRAPH
	"E4 6A	81FB",	#CJK UNIFIED IDEOGRAPH
	"E4 6B	81FE",	#CJK UNIFIED IDEOGRAPH
	"E4 6C	8201",	#CJK UNIFIED IDEOGRAPH
	"E4 6D	8202",	#CJK UNIFIED IDEOGRAPH
	"E4 6E	8205",	#CJK UNIFIED IDEOGRAPH
	"E4 6F	8207",	#CJK UNIFIED IDEOGRAPH
	"E4 70	820A",	#CJK UNIFIED IDEOGRAPH
	"E4 71	820D",	#CJK UNIFIED IDEOGRAPH
	"E4 72	8210",	#CJK UNIFIED IDEOGRAPH
	"E4 73	8216",	#CJK UNIFIED IDEOGRAPH
	"E4 74	8229",	#CJK UNIFIED IDEOGRAPH
	"E4 75	822B",	#CJK UNIFIED IDEOGRAPH
	"E4 76	8238",	#CJK UNIFIED IDEOGRAPH
	"E4 77	8233",	#CJK UNIFIED IDEOGRAPH
	"E4 78	8240",	#CJK UNIFIED IDEOGRAPH
	"E4 79	8259",	#CJK UNIFIED IDEOGRAPH
	"E4 7A	8258",	#CJK UNIFIED IDEOGRAPH
	"E4 7B	825D",	#CJK UNIFIED IDEOGRAPH
	"E4 7C	825A",	#CJK UNIFIED IDEOGRAPH
	"E4 7D	825F",	#CJK UNIFIED IDEOGRAPH
	"E4 7E	8264",	#CJK UNIFIED IDEOGRAPH
	"E4 80	8262",	#CJK UNIFIED IDEOGRAPH
	"E4 81	8268",	#CJK UNIFIED IDEOGRAPH
	"E4 82	826A",	#CJK UNIFIED IDEOGRAPH
	"E4 83	826B",	#CJK UNIFIED IDEOGRAPH
	"E4 84	822E",	#CJK UNIFIED IDEOGRAPH
	"E4 85	8271",	#CJK UNIFIED IDEOGRAPH
	"E4 86	8277",	#CJK UNIFIED IDEOGRAPH
	"E4 87	8278",	#CJK UNIFIED IDEOGRAPH
	"E4 88	827E",	#CJK UNIFIED IDEOGRAPH
	"E4 89	828D",	#CJK UNIFIED IDEOGRAPH
	"E4 8A	8292",	#CJK UNIFIED IDEOGRAPH
	"E4 8B	82AB",	#CJK UNIFIED IDEOGRAPH
	"E4 8C	829F",	#CJK UNIFIED IDEOGRAPH
	"E4 8D	82BB",	#CJK UNIFIED IDEOGRAPH
	"E4 8E	82AC",	#CJK UNIFIED IDEOGRAPH
	"E4 8F	82E1",	#CJK UNIFIED IDEOGRAPH
	"E4 90	82E3",	#CJK UNIFIED IDEOGRAPH
	"E4 91	82DF",	#CJK UNIFIED IDEOGRAPH
	"E4 92	82D2",	#CJK UNIFIED IDEOGRAPH
	"E4 93	82F4",	#CJK UNIFIED IDEOGRAPH
	"E4 94	82F3",	#CJK UNIFIED IDEOGRAPH
	"E4 95	82FA",	#CJK UNIFIED IDEOGRAPH
	"E4 96	8393",	#CJK UNIFIED IDEOGRAPH
	"E4 97	8303",	#CJK UNIFIED IDEOGRAPH
	"E4 98	82FB",	#CJK UNIFIED IDEOGRAPH
	"E4 99	82F9",	#CJK UNIFIED IDEOGRAPH
	"E4 9A	82DE",	#CJK UNIFIED IDEOGRAPH
	"E4 9B	8306",	#CJK UNIFIED IDEOGRAPH
	"E4 9C	82DC",	#CJK UNIFIED IDEOGRAPH
	"E4 9D	8309",	#CJK UNIFIED IDEOGRAPH
	"E4 9E	82D9",	#CJK UNIFIED IDEOGRAPH
	"E4 9F	8335",	#CJK UNIFIED IDEOGRAPH
	"E4 A0	8334",	#CJK UNIFIED IDEOGRAPH
	"E4 A1	8316",	#CJK UNIFIED IDEOGRAPH
	"E4 A2	8332",	#CJK UNIFIED IDEOGRAPH
	"E4 A3	8331",	#CJK UNIFIED IDEOGRAPH
	"E4 A4	8340",	#CJK UNIFIED IDEOGRAPH
	"E4 A5	8339",	#CJK UNIFIED IDEOGRAPH
	"E4 A6	8350",	#CJK UNIFIED IDEOGRAPH
	"E4 A7	8345",	#CJK UNIFIED IDEOGRAPH
	"E4 A8	832F",	#CJK UNIFIED IDEOGRAPH
	"E4 A9	832B",	#CJK UNIFIED IDEOGRAPH
	"E4 AA	8317",	#CJK UNIFIED IDEOGRAPH
	"E4 AB	8318",	#CJK UNIFIED IDEOGRAPH
	"E4 AC	8385",	#CJK UNIFIED IDEOGRAPH
	"E4 AD	839A",	#CJK UNIFIED IDEOGRAPH
	"E4 AE	83AA",	#CJK UNIFIED IDEOGRAPH
	"E4 AF	839F",	#CJK UNIFIED IDEOGRAPH
	"E4 B0	83A2",	#CJK UNIFIED IDEOGRAPH
	"E4 B1	8396",	#CJK UNIFIED IDEOGRAPH
	"E4 B2	8323",	#CJK UNIFIED IDEOGRAPH
	"E4 B3	838E",	#CJK UNIFIED IDEOGRAPH
	"E4 B4	8387",	#CJK UNIFIED IDEOGRAPH
	"E4 B5	838A",	#CJK UNIFIED IDEOGRAPH
	"E4 B6	837C",	#CJK UNIFIED IDEOGRAPH
	"E4 B7	83B5",	#CJK UNIFIED IDEOGRAPH
	"E4 B8	8373",	#CJK UNIFIED IDEOGRAPH
	"E4 B9	8375",	#CJK UNIFIED IDEOGRAPH
	"E4 BA	83A0",	#CJK UNIFIED IDEOGRAPH
	"E4 BB	8389",	#CJK UNIFIED IDEOGRAPH
	"E4 BC	83A8",	#CJK UNIFIED IDEOGRAPH
	"E4 BD	83F4",	#CJK UNIFIED IDEOGRAPH
	"E4 BE	8413",	#CJK UNIFIED IDEOGRAPH
	"E4 BF	83EB",	#CJK UNIFIED IDEOGRAPH
	"E4 C0	83CE",	#CJK UNIFIED IDEOGRAPH
	"E4 C1	83FD",	#CJK UNIFIED IDEOGRAPH
	"E4 C2	8403",	#CJK UNIFIED IDEOGRAPH
	"E4 C3	83D8",	#CJK UNIFIED IDEOGRAPH
	"E4 C4	840B",	#CJK UNIFIED IDEOGRAPH
	"E4 C5	83C1",	#CJK UNIFIED IDEOGRAPH
	"E4 C6	83F7",	#CJK UNIFIED IDEOGRAPH
	"E4 C7	8407",	#CJK UNIFIED IDEOGRAPH
	"E4 C8	83E0",	#CJK UNIFIED IDEOGRAPH
	"E4 C9	83F2",	#CJK UNIFIED IDEOGRAPH
	"E4 CA	840D",	#CJK UNIFIED IDEOGRAPH
	"E4 CB	8422",	#CJK UNIFIED IDEOGRAPH
	"E4 CC	8420",	#CJK UNIFIED IDEOGRAPH
	"E4 CD	83BD",	#CJK UNIFIED IDEOGRAPH
	"E4 CE	8438",	#CJK UNIFIED IDEOGRAPH
	"E4 CF	8506",	#CJK UNIFIED IDEOGRAPH
	"E4 D0	83FB",	#CJK UNIFIED IDEOGRAPH
	"E4 D1	846D",	#CJK UNIFIED IDEOGRAPH
	"E4 D2	842A",	#CJK UNIFIED IDEOGRAPH
	"E4 D3	843C",	#CJK UNIFIED IDEOGRAPH
	"E4 D4	855A",	#CJK UNIFIED IDEOGRAPH
	"E4 D5	8484",	#CJK UNIFIED IDEOGRAPH
	"E4 D6	8477",	#CJK UNIFIED IDEOGRAPH
	"E4 D7	846B",	#CJK UNIFIED IDEOGRAPH
	"E4 D8	84AD",	#CJK UNIFIED IDEOGRAPH
	"E4 D9	846E",	#CJK UNIFIED IDEOGRAPH
	"E4 DA	8482",	#CJK UNIFIED IDEOGRAPH
	"E4 DB	8469",	#CJK UNIFIED IDEOGRAPH
	"E4 DC	8446",	#CJK UNIFIED IDEOGRAPH
	"E4 DD	842C",	#CJK UNIFIED IDEOGRAPH
	"E4 DE	846F",	#CJK UNIFIED IDEOGRAPH
	"E4 DF	8479",	#CJK UNIFIED IDEOGRAPH
	"E4 E0	8435",	#CJK UNIFIED IDEOGRAPH
	"E4 E1	84CA",	#CJK UNIFIED IDEOGRAPH
	"E4 E2	8462",	#CJK UNIFIED IDEOGRAPH
	"E4 E3	84B9",	#CJK UNIFIED IDEOGRAPH
	"E4 E4	84BF",	#CJK UNIFIED IDEOGRAPH
	"E4 E5	849F",	#CJK UNIFIED IDEOGRAPH
	"E4 E6	84D9",	#CJK UNIFIED IDEOGRAPH
	"E4 E7	84CD",	#CJK UNIFIED IDEOGRAPH
	"E4 E8	84BB",	#CJK UNIFIED IDEOGRAPH
	"E4 E9	84DA",	#CJK UNIFIED IDEOGRAPH
	"E4 EA	84D0",	#CJK UNIFIED IDEOGRAPH
	"E4 EB	84C1",	#CJK UNIFIED IDEOGRAPH
	"E4 EC	84C6",	#CJK UNIFIED IDEOGRAPH
	"E4 ED	84D6",	#CJK UNIFIED IDEOGRAPH
	"E4 EE	84A1",	#CJK UNIFIED IDEOGRAPH
	"E4 EF	8521",	#CJK UNIFIED IDEOGRAPH
	"E4 F0	84FF",	#CJK UNIFIED IDEOGRAPH
	"E4 F1	84F4",	#CJK UNIFIED IDEOGRAPH
	"E4 F2	8517",	#CJK UNIFIED IDEOGRAPH
	"E4 F3	8518",	#CJK UNIFIED IDEOGRAPH
	"E4 F4	852C",	#CJK UNIFIED IDEOGRAPH
	"E4 F5	851F",	#CJK UNIFIED IDEOGRAPH
	"E4 F6	8515",	#CJK UNIFIED IDEOGRAPH
	"E4 F7	8514",	#CJK UNIFIED IDEOGRAPH
	"E4 F8	84FC",	#CJK UNIFIED IDEOGRAPH
	"E4 F9	8540",	#CJK UNIFIED IDEOGRAPH
	"E4 FA	8563",	#CJK UNIFIED IDEOGRAPH
	"E4 FB	8558",	#CJK UNIFIED IDEOGRAPH
	"E4 FC	8548",	#CJK UNIFIED IDEOGRAPH
	"E5 40	8541",	#CJK UNIFIED IDEOGRAPH
	"E5 41	8602",	#CJK UNIFIED IDEOGRAPH
	"E5 42	854B",	#CJK UNIFIED IDEOGRAPH
	"E5 43	8555",	#CJK UNIFIED IDEOGRAPH
	"E5 44	8580",	#CJK UNIFIED IDEOGRAPH
	"E5 45	85A4",	#CJK UNIFIED IDEOGRAPH
	"E5 46	8588",	#CJK UNIFIED IDEOGRAPH
	"E5 47	8591",	#CJK UNIFIED IDEOGRAPH
	"E5 48	858A",	#CJK UNIFIED IDEOGRAPH
	"E5 49	85A8",	#CJK UNIFIED IDEOGRAPH
	"E5 4A	856D",	#CJK UNIFIED IDEOGRAPH
	"E5 4B	8594",	#CJK UNIFIED IDEOGRAPH
	"E5 4C	859B",	#CJK UNIFIED IDEOGRAPH
	"E5 4D	85EA",	#CJK UNIFIED IDEOGRAPH
	"E5 4E	8587",	#CJK UNIFIED IDEOGRAPH
	"E5 4F	859C",	#CJK UNIFIED IDEOGRAPH
	"E5 50	8577",	#CJK UNIFIED IDEOGRAPH
	"E5 51	857E",	#CJK UNIFIED IDEOGRAPH
	"E5 52	8590",	#CJK UNIFIED IDEOGRAPH
	"E5 53	85C9",	#CJK UNIFIED IDEOGRAPH
	"E5 54	85BA",	#CJK UNIFIED IDEOGRAPH
	"E5 55	85CF",	#CJK UNIFIED IDEOGRAPH
	"E5 56	85B9",	#CJK UNIFIED IDEOGRAPH
	"E5 57	85D0",	#CJK UNIFIED IDEOGRAPH
	"E5 58	85D5",	#CJK UNIFIED IDEOGRAPH
	"E5 59	85DD",	#CJK UNIFIED IDEOGRAPH
	"E5 5A	85E5",	#CJK UNIFIED IDEOGRAPH
	"E5 5B	85DC",	#CJK UNIFIED IDEOGRAPH
	"E5 5C	85F9",	#CJK UNIFIED IDEOGRAPH
	"E5 5D	860A",	#CJK UNIFIED IDEOGRAPH
	"E5 5E	8613",	#CJK UNIFIED IDEOGRAPH
	"E5 5F	860B",	#CJK UNIFIED IDEOGRAPH
	"E5 60	85FE",	#CJK UNIFIED IDEOGRAPH
	"E5 61	85FA",	#CJK UNIFIED IDEOGRAPH
	"E5 62	8606",	#CJK UNIFIED IDEOGRAPH
	"E5 63	8622",	#CJK UNIFIED IDEOGRAPH
	"E5 64	861A",	#CJK UNIFIED IDEOGRAPH
	"E5 65	8630",	#CJK UNIFIED IDEOGRAPH
	"E5 66	863F",	#CJK UNIFIED IDEOGRAPH
	"E5 67	864D",	#CJK UNIFIED IDEOGRAPH
	"E5 68	4E55",	#CJK UNIFIED IDEOGRAPH
	"E5 69	8654",	#CJK UNIFIED IDEOGRAPH
	"E5 6A	865F",	#CJK UNIFIED IDEOGRAPH
	"E5 6B	8667",	#CJK UNIFIED IDEOGRAPH
	"E5 6C	8671",	#CJK UNIFIED IDEOGRAPH
	"E5 6D	8693",	#CJK UNIFIED IDEOGRAPH
	"E5 6E	86A3",	#CJK UNIFIED IDEOGRAPH
	"E5 6F	86A9",	#CJK UNIFIED IDEOGRAPH
	"E5 70	86AA",	#CJK UNIFIED IDEOGRAPH
	"E5 71	868B",	#CJK UNIFIED IDEOGRAPH
	"E5 72	868C",	#CJK UNIFIED IDEOGRAPH
	"E5 73	86B6",	#CJK UNIFIED IDEOGRAPH
	"E5 74	86AF",	#CJK UNIFIED IDEOGRAPH
	"E5 75	86C4",	#CJK UNIFIED IDEOGRAPH
	"E5 76	86C6",	#CJK UNIFIED IDEOGRAPH
	"E5 77	86B0",	#CJK UNIFIED IDEOGRAPH
	"E5 78	86C9",	#CJK UNIFIED IDEOGRAPH
	"E5 79	8823",	#CJK UNIFIED IDEOGRAPH
	"E5 7A	86AB",	#CJK UNIFIED IDEOGRAPH
	"E5 7B	86D4",	#CJK UNIFIED IDEOGRAPH
	"E5 7C	86DE",	#CJK UNIFIED IDEOGRAPH
	"E5 7D	86E9",	#CJK UNIFIED IDEOGRAPH
	"E5 7E	86EC",	#CJK UNIFIED IDEOGRAPH
	"E5 80	86DF",	#CJK UNIFIED IDEOGRAPH
	"E5 81	86DB",	#CJK UNIFIED IDEOGRAPH
	"E5 82	86EF",	#CJK UNIFIED IDEOGRAPH
	"E5 83	8712",	#CJK UNIFIED IDEOGRAPH
	"E5 84	8706",	#CJK UNIFIED IDEOGRAPH
	"E5 85	8708",	#CJK UNIFIED IDEOGRAPH
	"E5 86	8700",	#CJK UNIFIED IDEOGRAPH
	"E5 87	8703",	#CJK UNIFIED IDEOGRAPH
	"E5 88	86FB",	#CJK UNIFIED IDEOGRAPH
	"E5 89	8711",	#CJK UNIFIED IDEOGRAPH
	"E5 8A	8709",	#CJK UNIFIED IDEOGRAPH
	"E5 8B	870D",	#CJK UNIFIED IDEOGRAPH
	"E5 8C	86F9",	#CJK UNIFIED IDEOGRAPH
	"E5 8D	870A",	#CJK UNIFIED IDEOGRAPH
	"E5 8E	8734",	#CJK UNIFIED IDEOGRAPH
	"E5 8F	873F",	#CJK UNIFIED IDEOGRAPH
	"E5 90	8737",	#CJK UNIFIED IDEOGRAPH
	"E5 91	873B",	#CJK UNIFIED IDEOGRAPH
	"E5 92	8725",	#CJK UNIFIED IDEOGRAPH
	"E5 93	8729",	#CJK UNIFIED IDEOGRAPH
	"E5 94	871A",	#CJK UNIFIED IDEOGRAPH
	"E5 95	8760",	#CJK UNIFIED IDEOGRAPH
	"E5 96	875F",	#CJK UNIFIED IDEOGRAPH
	"E5 97	8778",	#CJK UNIFIED IDEOGRAPH
	"E5 98	874C",	#CJK UNIFIED IDEOGRAPH
	"E5 99	874E",	#CJK UNIFIED IDEOGRAPH
	"E5 9A	8774",	#CJK UNIFIED IDEOGRAPH
	"E5 9B	8757",	#CJK UNIFIED IDEOGRAPH
	"E5 9C	8768",	#CJK UNIFIED IDEOGRAPH
	"E5 9D	876E",	#CJK UNIFIED IDEOGRAPH
	"E5 9E	8759",	#CJK UNIFIED IDEOGRAPH
	"E5 9F	8753",	#CJK UNIFIED IDEOGRAPH
	"E5 A0	8763",	#CJK UNIFIED IDEOGRAPH
	"E5 A1	876A",	#CJK UNIFIED IDEOGRAPH
	"E5 A2	8805",	#CJK UNIFIED IDEOGRAPH
	"E5 A3	87A2",	#CJK UNIFIED IDEOGRAPH
	"E5 A4	879F",	#CJK UNIFIED IDEOGRAPH
	"E5 A5	8782",	#CJK UNIFIED IDEOGRAPH
	"E5 A6	87AF",	#CJK UNIFIED IDEOGRAPH
	"E5 A7	87CB",	#CJK UNIFIED IDEOGRAPH
	"E5 A8	87BD",	#CJK UNIFIED IDEOGRAPH
	"E5 A9	87C0",	#CJK UNIFIED IDEOGRAPH
	"E5 AA	87D0",	#CJK UNIFIED IDEOGRAPH
	"E5 AB	96D6",	#CJK UNIFIED IDEOGRAPH
	"E5 AC	87AB",	#CJK UNIFIED IDEOGRAPH
	"E5 AD	87C4",	#CJK UNIFIED IDEOGRAPH
	"E5 AE	87B3",	#CJK UNIFIED IDEOGRAPH
	"E5 AF	87C7",	#CJK UNIFIED IDEOGRAPH
	"E5 B0	87C6",	#CJK UNIFIED IDEOGRAPH
	"E5 B1	87BB",	#CJK UNIFIED IDEOGRAPH
	"E5 B2	87EF",	#CJK UNIFIED IDEOGRAPH
	"E5 B3	87F2",	#CJK UNIFIED IDEOGRAPH
	"E5 B4	87E0",	#CJK UNIFIED IDEOGRAPH
	"E5 B5	880F",	#CJK UNIFIED IDEOGRAPH
	"E5 B6	880D",	#CJK UNIFIED IDEOGRAPH
	"E5 B7	87FE",	#CJK UNIFIED IDEOGRAPH
	"E5 B8	87F6",	#CJK UNIFIED IDEOGRAPH
	"E5 B9	87F7",	#CJK UNIFIED IDEOGRAPH
	"E5 BA	880E",	#CJK UNIFIED IDEOGRAPH
	"E5 BB	87D2",	#CJK UNIFIED IDEOGRAPH
	"E5 BC	8811",	#CJK UNIFIED IDEOGRAPH
	"E5 BD	8816",	#CJK UNIFIED IDEOGRAPH
	"E5 BE	8815",	#CJK UNIFIED IDEOGRAPH
	"E5 BF	8822",	#CJK UNIFIED IDEOGRAPH
	"E5 C0	8821",	#CJK UNIFIED IDEOGRAPH
	"E5 C1	8831",	#CJK UNIFIED IDEOGRAPH
	"E5 C2	8836",	#CJK UNIFIED IDEOGRAPH
	"E5 C3	8839",	#CJK UNIFIED IDEOGRAPH
	"E5 C4	8827",	#CJK UNIFIED IDEOGRAPH
	"E5 C5	883B",	#CJK UNIFIED IDEOGRAPH
	"E5 C6	8844",	#CJK UNIFIED IDEOGRAPH
	"E5 C7	8842",	#CJK UNIFIED IDEOGRAPH
	"E5 C8	8852",	#CJK UNIFIED IDEOGRAPH
	"E5 C9	8859",	#CJK UNIFIED IDEOGRAPH
	"E5 CA	885E",	#CJK UNIFIED IDEOGRAPH
	"E5 CB	8862",	#CJK UNIFIED IDEOGRAPH
	"E5 CC	886B",	#CJK UNIFIED IDEOGRAPH
	"E5 CD	8881",	#CJK UNIFIED IDEOGRAPH
	"E5 CE	887E",	#CJK UNIFIED IDEOGRAPH
	"E5 CF	889E",	#CJK UNIFIED IDEOGRAPH
	"E5 D0	8875",	#CJK UNIFIED IDEOGRAPH
	"E5 D1	887D",	#CJK UNIFIED IDEOGRAPH
	"E5 D2	88B5",	#CJK UNIFIED IDEOGRAPH
	"E5 D3	8872",	#CJK UNIFIED IDEOGRAPH
	"E5 D4	8882",	#CJK UNIFIED IDEOGRAPH
	"E5 D5	8897",	#CJK UNIFIED IDEOGRAPH
	"E5 D6	8892",	#CJK UNIFIED IDEOGRAPH
	"E5 D7	88AE",	#CJK UNIFIED IDEOGRAPH
	"E5 D8	8899",	#CJK UNIFIED IDEOGRAPH
	"E5 D9	88A2",	#CJK UNIFIED IDEOGRAPH
	"E5 DA	888D",	#CJK UNIFIED IDEOGRAPH
	"E5 DB	88A4",	#CJK UNIFIED IDEOGRAPH
	"E5 DC	88B0",	#CJK UNIFIED IDEOGRAPH
	"E5 DD	88BF",	#CJK UNIFIED IDEOGRAPH
	"E5 DE	88B1",	#CJK UNIFIED IDEOGRAPH
	"E5 DF	88C3",	#CJK UNIFIED IDEOGRAPH
	"E5 E0	88C4",	#CJK UNIFIED IDEOGRAPH
	"E5 E1	88D4",	#CJK UNIFIED IDEOGRAPH
	"E5 E2	88D8",	#CJK UNIFIED IDEOGRAPH
	"E5 E3	88D9",	#CJK UNIFIED IDEOGRAPH
	"E5 E4	88DD",	#CJK UNIFIED IDEOGRAPH
	"E5 E5	88F9",	#CJK UNIFIED IDEOGRAPH
	"E5 E6	8902",	#CJK UNIFIED IDEOGRAPH
	"E5 E7	88FC",	#CJK UNIFIED IDEOGRAPH
	"E5 E8	88F4",	#CJK UNIFIED IDEOGRAPH
	"E5 E9	88E8",	#CJK UNIFIED IDEOGRAPH
	"E5 EA	88F2",	#CJK UNIFIED IDEOGRAPH
	"E5 EB	8904",	#CJK UNIFIED IDEOGRAPH
	"E5 EC	890C",	#CJK UNIFIED IDEOGRAPH
	"E5 ED	890A",	#CJK UNIFIED IDEOGRAPH
	"E5 EE	8913",	#CJK UNIFIED IDEOGRAPH
	"E5 EF	8943",	#CJK UNIFIED IDEOGRAPH
	"E5 F0	891E",	#CJK UNIFIED IDEOGRAPH
	"E5 F1	8925",	#CJK UNIFIED IDEOGRAPH
	"E5 F2	892A",	#CJK UNIFIED IDEOGRAPH
	"E5 F3	892B",	#CJK UNIFIED IDEOGRAPH
	"E5 F4	8941",	#CJK UNIFIED IDEOGRAPH
	"E5 F5	8944",	#CJK UNIFIED IDEOGRAPH
	"E5 F6	893B",	#CJK UNIFIED IDEOGRAPH
	"E5 F7	8936",	#CJK UNIFIED IDEOGRAPH
	"E5 F8	8938",	#CJK UNIFIED IDEOGRAPH
	"E5 F9	894C",	#CJK UNIFIED IDEOGRAPH
	"E5 FA	891D",	#CJK UNIFIED IDEOGRAPH
	"E5 FB	8960",	#CJK UNIFIED IDEOGRAPH
	"E5 FC	895E",	#CJK UNIFIED IDEOGRAPH
	"E6 40	8966",	#CJK UNIFIED IDEOGRAPH
	"E6 41	8964",	#CJK UNIFIED IDEOGRAPH
	"E6 42	896D",	#CJK UNIFIED IDEOGRAPH
	"E6 43	896A",	#CJK UNIFIED IDEOGRAPH
	"E6 44	896F",	#CJK UNIFIED IDEOGRAPH
	"E6 45	8974",	#CJK UNIFIED IDEOGRAPH
	"E6 46	8977",	#CJK UNIFIED IDEOGRAPH
	"E6 47	897E",	#CJK UNIFIED IDEOGRAPH
	"E6 48	8983",	#CJK UNIFIED IDEOGRAPH
	"E6 49	8988",	#CJK UNIFIED IDEOGRAPH
	"E6 4A	898A",	#CJK UNIFIED IDEOGRAPH
	"E6 4B	8993",	#CJK UNIFIED IDEOGRAPH
	"E6 4C	8998",	#CJK UNIFIED IDEOGRAPH
	"E6 4D	89A1",	#CJK UNIFIED IDEOGRAPH
	"E6 4E	89A9",	#CJK UNIFIED IDEOGRAPH
	"E6 4F	89A6",	#CJK UNIFIED IDEOGRAPH
	"E6 50	89AC",	#CJK UNIFIED IDEOGRAPH
	"E6 51	89AF",	#CJK UNIFIED IDEOGRAPH
	"E6 52	89B2",	#CJK UNIFIED IDEOGRAPH
	"E6 53	89BA",	#CJK UNIFIED IDEOGRAPH
	"E6 54	89BD",	#CJK UNIFIED IDEOGRAPH
	"E6 55	89BF",	#CJK UNIFIED IDEOGRAPH
	"E6 56	89C0",	#CJK UNIFIED IDEOGRAPH
	"E6 57	89DA",	#CJK UNIFIED IDEOGRAPH
	"E6 58	89DC",	#CJK UNIFIED IDEOGRAPH
	"E6 59	89DD",	#CJK UNIFIED IDEOGRAPH
	"E6 5A	89E7",	#CJK UNIFIED IDEOGRAPH
	"E6 5B	89F4",	#CJK UNIFIED IDEOGRAPH
	"E6 5C	89F8",	#CJK UNIFIED IDEOGRAPH
	"E6 5D	8A03",	#CJK UNIFIED IDEOGRAPH
	"E6 5E	8A16",	#CJK UNIFIED IDEOGRAPH
	"E6 5F	8A10",	#CJK UNIFIED IDEOGRAPH
	"E6 60	8A0C",	#CJK UNIFIED IDEOGRAPH
	"E6 61	8A1B",	#CJK UNIFIED IDEOGRAPH
	"E6 62	8A1D",	#CJK UNIFIED IDEOGRAPH
	"E6 63	8A25",	#CJK UNIFIED IDEOGRAPH
	"E6 64	8A36",	#CJK UNIFIED IDEOGRAPH
	"E6 65	8A41",	#CJK UNIFIED IDEOGRAPH
	"E6 66	8A5B",	#CJK UNIFIED IDEOGRAPH
	"E6 67	8A52",	#CJK UNIFIED IDEOGRAPH
	"E6 68	8A46",	#CJK UNIFIED IDEOGRAPH
	"E6 69	8A48",	#CJK UNIFIED IDEOGRAPH
	"E6 6A	8A7C",	#CJK UNIFIED IDEOGRAPH
	"E6 6B	8A6D",	#CJK UNIFIED IDEOGRAPH
	"E6 6C	8A6C",	#CJK UNIFIED IDEOGRAPH
	"E6 6D	8A62",	#CJK UNIFIED IDEOGRAPH
	"E6 6E	8A85",	#CJK UNIFIED IDEOGRAPH
	"E6 6F	8A82",	#CJK UNIFIED IDEOGRAPH
	"E6 70	8A84",	#CJK UNIFIED IDEOGRAPH
	"E6 71	8AA8",	#CJK UNIFIED IDEOGRAPH
	"E6 72	8AA1",	#CJK UNIFIED IDEOGRAPH
	"E6 73	8A91",	#CJK UNIFIED IDEOGRAPH
	"E6 74	8AA5",	#CJK UNIFIED IDEOGRAPH
	"E6 75	8AA6",	#CJK UNIFIED IDEOGRAPH
	"E6 76	8A9A",	#CJK UNIFIED IDEOGRAPH
	"E6 77	8AA3",	#CJK UNIFIED IDEOGRAPH
	"E6 78	8AC4",	#CJK UNIFIED IDEOGRAPH
	"E6 79	8ACD",	#CJK UNIFIED IDEOGRAPH
	"E6 7A	8AC2",	#CJK UNIFIED IDEOGRAPH
	"E6 7B	8ADA",	#CJK UNIFIED IDEOGRAPH
	"E6 7C	8AEB",	#CJK UNIFIED IDEOGRAPH
	"E6 7D	8AF3",	#CJK UNIFIED IDEOGRAPH
	"E6 7E	8AE7",	#CJK UNIFIED IDEOGRAPH
	"E6 80	8AE4",	#CJK UNIFIED IDEOGRAPH
	"E6 81	8AF1",	#CJK UNIFIED IDEOGRAPH
	"E6 82	8B14",	#CJK UNIFIED IDEOGRAPH
	"E6 83	8AE0",	#CJK UNIFIED IDEOGRAPH
	"E6 84	8AE2",	#CJK UNIFIED IDEOGRAPH
	"E6 85	8AF7",	#CJK UNIFIED IDEOGRAPH
	"E6 86	8ADE",	#CJK UNIFIED IDEOGRAPH
	"E6 87	8ADB",	#CJK UNIFIED IDEOGRAPH
	"E6 88	8B0C",	#CJK UNIFIED IDEOGRAPH
	"E6 89	8B07",	#CJK UNIFIED IDEOGRAPH
	"E6 8A	8B1A",	#CJK UNIFIED IDEOGRAPH
	"E6 8B	8AE1",	#CJK UNIFIED IDEOGRAPH
	"E6 8C	8B16",	#CJK UNIFIED IDEOGRAPH
	"E6 8D	8B10",	#CJK UNIFIED IDEOGRAPH
	"E6 8E	8B17",	#CJK UNIFIED IDEOGRAPH
	"E6 8F	8B20",	#CJK UNIFIED IDEOGRAPH
	"E6 90	8B33",	#CJK UNIFIED IDEOGRAPH
	"E6 91	97AB",	#CJK UNIFIED IDEOGRAPH
	"E6 92	8B26",	#CJK UNIFIED IDEOGRAPH
	"E6 93	8B2B",	#CJK UNIFIED IDEOGRAPH
	"E6 94	8B3E",	#CJK UNIFIED IDEOGRAPH
	"E6 95	8B28",	#CJK UNIFIED IDEOGRAPH
	"E6 96	8B41",	#CJK UNIFIED IDEOGRAPH
	"E6 97	8B4C",	#CJK UNIFIED IDEOGRAPH
	"E6 98	8B4F",	#CJK UNIFIED IDEOGRAPH
	"E6 99	8B4E",	#CJK UNIFIED IDEOGRAPH
	"E6 9A	8B49",	#CJK UNIFIED IDEOGRAPH
	"E6 9B	8B56",	#CJK UNIFIED IDEOGRAPH
	"E6 9C	8B5B",	#CJK UNIFIED IDEOGRAPH
	"E6 9D	8B5A",	#CJK UNIFIED IDEOGRAPH
	"E6 9E	8B6B",	#CJK UNIFIED IDEOGRAPH
	"E6 9F	8B5F",	#CJK UNIFIED IDEOGRAPH
	"E6 A0	8B6C",	#CJK UNIFIED IDEOGRAPH
	"E6 A1	8B6F",	#CJK UNIFIED IDEOGRAPH
	"E6 A2	8B74",	#CJK UNIFIED IDEOGRAPH
	"E6 A3	8B7D",	#CJK UNIFIED IDEOGRAPH
	"E6 A4	8B80",	#CJK UNIFIED IDEOGRAPH
	"E6 A5	8B8C",	#CJK UNIFIED IDEOGRAPH
	"E6 A6	8B8E",	#CJK UNIFIED IDEOGRAPH
	"E6 A7	8B92",	#CJK UNIFIED IDEOGRAPH
	"E6 A8	8B93",	#CJK UNIFIED IDEOGRAPH
	"E6 A9	8B96",	#CJK UNIFIED IDEOGRAPH
	"E6 AA	8B99",	#CJK UNIFIED IDEOGRAPH
	"E6 AB	8B9A",	#CJK UNIFIED IDEOGRAPH
	"E6 AC	8C3A",	#CJK UNIFIED IDEOGRAPH
	"E6 AD	8C41",	#CJK UNIFIED IDEOGRAPH
	"E6 AE	8C3F",	#CJK UNIFIED IDEOGRAPH
	"E6 AF	8C48",	#CJK UNIFIED IDEOGRAPH
	"E6 B0	8C4C",	#CJK UNIFIED IDEOGRAPH
	"E6 B1	8C4E",	#CJK UNIFIED IDEOGRAPH
	"E6 B2	8C50",	#CJK UNIFIED IDEOGRAPH
	"E6 B3	8C55",	#CJK UNIFIED IDEOGRAPH
	"E6 B4	8C62",	#CJK UNIFIED IDEOGRAPH
	"E6 B5	8C6C",	#CJK UNIFIED IDEOGRAPH
	"E6 B6	8C78",	#CJK UNIFIED IDEOGRAPH
	"E6 B7	8C7A",	#CJK UNIFIED IDEOGRAPH
	"E6 B8	8C82",	#CJK UNIFIED IDEOGRAPH
	"E6 B9	8C89",	#CJK UNIFIED IDEOGRAPH
	"E6 BA	8C85",	#CJK UNIFIED IDEOGRAPH
	"E6 BB	8C8A",	#CJK UNIFIED IDEOGRAPH
	"E6 BC	8C8D",	#CJK UNIFIED IDEOGRAPH
	"E6 BD	8C8E",	#CJK UNIFIED IDEOGRAPH
	"E6 BE	8C94",	#CJK UNIFIED IDEOGRAPH
	"E6 BF	8C7C",	#CJK UNIFIED IDEOGRAPH
	"E6 C0	8C98",	#CJK UNIFIED IDEOGRAPH
	"E6 C1	621D",	#CJK UNIFIED IDEOGRAPH
	"E6 C2	8CAD",	#CJK UNIFIED IDEOGRAPH
	"E6 C3	8CAA",	#CJK UNIFIED IDEOGRAPH
	"E6 C4	8CBD",	#CJK UNIFIED IDEOGRAPH
	"E6 C5	8CB2",	#CJK UNIFIED IDEOGRAPH
	"E6 C6	8CB3",	#CJK UNIFIED IDEOGRAPH
	"E6 C7	8CAE",	#CJK UNIFIED IDEOGRAPH
	"E6 C8	8CB6",	#CJK UNIFIED IDEOGRAPH
	"E6 C9	8CC8",	#CJK UNIFIED IDEOGRAPH
	"E6 CA	8CC1",	#CJK UNIFIED IDEOGRAPH
	"E6 CB	8CE4",	#CJK UNIFIED IDEOGRAPH
	"E6 CC	8CE3",	#CJK UNIFIED IDEOGRAPH
	"E6 CD	8CDA",	#CJK UNIFIED IDEOGRAPH
	"E6 CE	8CFD",	#CJK UNIFIED IDEOGRAPH
	"E6 CF	8CFA",	#CJK UNIFIED IDEOGRAPH
	"E6 D0	8CFB",	#CJK UNIFIED IDEOGRAPH
	"E6 D1	8D04",	#CJK UNIFIED IDEOGRAPH
	"E6 D2	8D05",	#CJK UNIFIED IDEOGRAPH
	"E6 D3	8D0A",	#CJK UNIFIED IDEOGRAPH
	"E6 D4	8D07",	#CJK UNIFIED IDEOGRAPH
	"E6 D5	8D0F",	#CJK UNIFIED IDEOGRAPH
	"E6 D6	8D0D",	#CJK UNIFIED IDEOGRAPH
	"E6 D7	8D10",	#CJK UNIFIED IDEOGRAPH
	"E6 D8	9F4E",	#CJK UNIFIED IDEOGRAPH
	"E6 D9	8D13",	#CJK UNIFIED IDEOGRAPH
	"E6 DA	8CCD",	#CJK UNIFIED IDEOGRAPH
	"E6 DB	8D14",	#CJK UNIFIED IDEOGRAPH
	"E6 DC	8D16",	#CJK UNIFIED IDEOGRAPH
	"E6 DD	8D67",	#CJK UNIFIED IDEOGRAPH
	"E6 DE	8D6D",	#CJK UNIFIED IDEOGRAPH
	"E6 DF	8D71",	#CJK UNIFIED IDEOGRAPH
	"E6 E0	8D73",	#CJK UNIFIED IDEOGRAPH
	"E6 E1	8D81",	#CJK UNIFIED IDEOGRAPH
	"E6 E2	8D99",	#CJK UNIFIED IDEOGRAPH
	"E6 E3	8DC2",	#CJK UNIFIED IDEOGRAPH
	"E6 E4	8DBE",	#CJK UNIFIED IDEOGRAPH
	"E6 E5	8DBA",	#CJK UNIFIED IDEOGRAPH
	"E6 E6	8DCF",	#CJK UNIFIED IDEOGRAPH
	"E6 E7	8DDA",	#CJK UNIFIED IDEOGRAPH
	"E6 E8	8DD6",	#CJK UNIFIED IDEOGRAPH
	"E6 E9	8DCC",	#CJK UNIFIED IDEOGRAPH
	"E6 EA	8DDB",	#CJK UNIFIED IDEOGRAPH
	"E6 EB	8DCB",	#CJK UNIFIED IDEOGRAPH
	"E6 EC	8DEA",	#CJK UNIFIED IDEOGRAPH
	"E6 ED	8DEB",	#CJK UNIFIED IDEOGRAPH
	"E6 EE	8DDF",	#CJK UNIFIED IDEOGRAPH
	"E6 EF	8DE3",	#CJK UNIFIED IDEOGRAPH
	"E6 F0	8DFC",	#CJK UNIFIED IDEOGRAPH
	"E6 F1	8E08",	#CJK UNIFIED IDEOGRAPH
	"E6 F2	8E09",	#CJK UNIFIED IDEOGRAPH
	"E6 F3	8DFF",	#CJK UNIFIED IDEOGRAPH
	"E6 F4	8E1D",	#CJK UNIFIED IDEOGRAPH
	"E6 F5	8E1E",	#CJK UNIFIED IDEOGRAPH
	"E6 F6	8E10",	#CJK UNIFIED IDEOGRAPH
	"E6 F7	8E1F",	#CJK UNIFIED IDEOGRAPH
	"E6 F8	8E42",	#CJK UNIFIED IDEOGRAPH
	"E6 F9	8E35",	#CJK UNIFIED IDEOGRAPH
	"E6 FA	8E30",	#CJK UNIFIED IDEOGRAPH
	"E6 FB	8E34",	#CJK UNIFIED IDEOGRAPH
	"E6 FC	8E4A",	#CJK UNIFIED IDEOGRAPH
	"E7 40	8E47",	#CJK UNIFIED IDEOGRAPH
	"E7 41	8E49",	#CJK UNIFIED IDEOGRAPH
	"E7 42	8E4C",	#CJK UNIFIED IDEOGRAPH
	"E7 43	8E50",	#CJK UNIFIED IDEOGRAPH
	"E7 44	8E48",	#CJK UNIFIED IDEOGRAPH
	"E7 45	8E59",	#CJK UNIFIED IDEOGRAPH
	"E7 46	8E64",	#CJK UNIFIED IDEOGRAPH
	"E7 47	8E60",	#CJK UNIFIED IDEOGRAPH
	"E7 48	8E2A",	#CJK UNIFIED IDEOGRAPH
	"E7 49	8E63",	#CJK UNIFIED IDEOGRAPH
	"E7 4A	8E55",	#CJK UNIFIED IDEOGRAPH
	"E7 4B	8E76",	#CJK UNIFIED IDEOGRAPH
	"E7 4C	8E72",	#CJK UNIFIED IDEOGRAPH
	"E7 4D	8E7C",	#CJK UNIFIED IDEOGRAPH
	"E7 4E	8E81",	#CJK UNIFIED IDEOGRAPH
	"E7 4F	8E87",	#CJK UNIFIED IDEOGRAPH
	"E7 50	8E85",	#CJK UNIFIED IDEOGRAPH
	"E7 51	8E84",	#CJK UNIFIED IDEOGRAPH
	"E7 52	8E8B",	#CJK UNIFIED IDEOGRAPH
	"E7 53	8E8A",	#CJK UNIFIED IDEOGRAPH
	"E7 54	8E93",	#CJK UNIFIED IDEOGRAPH
	"E7 55	8E91",	#CJK UNIFIED IDEOGRAPH
	"E7 56	8E94",	#CJK UNIFIED IDEOGRAPH
	"E7 57	8E99",	#CJK UNIFIED IDEOGRAPH
	"E7 58	8EAA",	#CJK UNIFIED IDEOGRAPH
	"E7 59	8EA1",	#CJK UNIFIED IDEOGRAPH
	"E7 5A	8EAC",	#CJK UNIFIED IDEOGRAPH
	"E7 5B	8EB0",	#CJK UNIFIED IDEOGRAPH
	"E7 5C	8EC6",	#CJK UNIFIED IDEOGRAPH
	"E7 5D	8EB1",	#CJK UNIFIED IDEOGRAPH
	"E7 5E	8EBE",	#CJK UNIFIED IDEOGRAPH
	"E7 5F	8EC5",	#CJK UNIFIED IDEOGRAPH
	"E7 60	8EC8",	#CJK UNIFIED IDEOGRAPH
	"E7 61	8ECB",	#CJK UNIFIED IDEOGRAPH
	"E7 62	8EDB",	#CJK UNIFIED IDEOGRAPH
	"E7 63	8EE3",	#CJK UNIFIED IDEOGRAPH
	"E7 64	8EFC",	#CJK UNIFIED IDEOGRAPH
	"E7 65	8EFB",	#CJK UNIFIED IDEOGRAPH
	"E7 66	8EEB",	#CJK UNIFIED IDEOGRAPH
	"E7 67	8EFE",	#CJK UNIFIED IDEOGRAPH
	"E7 68	8F0A",	#CJK UNIFIED IDEOGRAPH
	"E7 69	8F05",	#CJK UNIFIED IDEOGRAPH
	"E7 6A	8F15",	#CJK UNIFIED IDEOGRAPH
	"E7 6B	8F12",	#CJK UNIFIED IDEOGRAPH
	"E7 6C	8F19",	#CJK UNIFIED IDEOGRAPH
	"E7 6D	8F13",	#CJK UNIFIED IDEOGRAPH
	"E7 6E	8F1C",	#CJK UNIFIED IDEOGRAPH
	"E7 6F	8F1F",	#CJK UNIFIED IDEOGRAPH
	"E7 70	8F1B",	#CJK UNIFIED IDEOGRAPH
	"E7 71	8F0C",	#CJK UNIFIED IDEOGRAPH
	"E7 72	8F26",	#CJK UNIFIED IDEOGRAPH
	"E7 73	8F33",	#CJK UNIFIED IDEOGRAPH
	"E7 74	8F3B",	#CJK UNIFIED IDEOGRAPH
	"E7 75	8F39",	#CJK UNIFIED IDEOGRAPH
	"E7 76	8F45",	#CJK UNIFIED IDEOGRAPH
	"E7 77	8F42",	#CJK UNIFIED IDEOGRAPH
	"E7 78	8F3E",	#CJK UNIFIED IDEOGRAPH
	"E7 79	8F4C",	#CJK UNIFIED IDEOGRAPH
	"E7 7A	8F49",	#CJK UNIFIED IDEOGRAPH
	"E7 7B	8F46",	#CJK UNIFIED IDEOGRAPH
	"E7 7C	8F4E",	#CJK UNIFIED IDEOGRAPH
	"E7 7D	8F57",	#CJK UNIFIED IDEOGRAPH
	"E7 7E	8F5C",	#CJK UNIFIED IDEOGRAPH
	"E7 80	8F62",	#CJK UNIFIED IDEOGRAPH
	"E7 81	8F63",	#CJK UNIFIED IDEOGRAPH
	"E7 82	8F64",	#CJK UNIFIED IDEOGRAPH
	"E7 83	8F9C",	#CJK UNIFIED IDEOGRAPH
	"E7 84	8F9F",	#CJK UNIFIED IDEOGRAPH
	"E7 85	8FA3",	#CJK UNIFIED IDEOGRAPH
	"E7 86	8FAD",	#CJK UNIFIED IDEOGRAPH
	"E7 87	8FAF",	#CJK UNIFIED IDEOGRAPH
	"E7 88	8FB7",	#CJK UNIFIED IDEOGRAPH
	"E7 89	8FDA",	#CJK UNIFIED IDEOGRAPH
	"E7 8A	8FE5",	#CJK UNIFIED IDEOGRAPH
	"E7 8B	8FE2",	#CJK UNIFIED IDEOGRAPH
	"E7 8C	8FEA",	#CJK UNIFIED IDEOGRAPH
	"E7 8D	8FEF",	#CJK UNIFIED IDEOGRAPH
	"E7 8E	9087",	#CJK UNIFIED IDEOGRAPH
	"E7 8F	8FF4",	#CJK UNIFIED IDEOGRAPH
	"E7 90	9005",	#CJK UNIFIED IDEOGRAPH
	"E7 91	8FF9",	#CJK UNIFIED IDEOGRAPH
	"E7 92	8FFA",	#CJK UNIFIED IDEOGRAPH
	"E7 93	9011",	#CJK UNIFIED IDEOGRAPH
	"E7 94	9015",	#CJK UNIFIED IDEOGRAPH
	"E7 95	9021",	#CJK UNIFIED IDEOGRAPH
	"E7 96	900D",	#CJK UNIFIED IDEOGRAPH
	"E7 97	901E",	#CJK UNIFIED IDEOGRAPH
	"E7 98	9016",	#CJK UNIFIED IDEOGRAPH
	"E7 99	900B",	#CJK UNIFIED IDEOGRAPH
	"E7 9A	9027",	#CJK UNIFIED IDEOGRAPH
	"E7 9B	9036",	#CJK UNIFIED IDEOGRAPH
	"E7 9C	9035",	#CJK UNIFIED IDEOGRAPH
	"E7 9D	9039",	#CJK UNIFIED IDEOGRAPH
	"E7 9E	8FF8",	#CJK UNIFIED IDEOGRAPH
	"E7 9F	904F",	#CJK UNIFIED IDEOGRAPH
	"E7 A0	9050",	#CJK UNIFIED IDEOGRAPH
	"E7 A1	9051",	#CJK UNIFIED IDEOGRAPH
	"E7 A2	9052",	#CJK UNIFIED IDEOGRAPH
	"E7 A3	900E",	#CJK UNIFIED IDEOGRAPH
	"E7 A4	9049",	#CJK UNIFIED IDEOGRAPH
	"E7 A5	903E",	#CJK UNIFIED IDEOGRAPH
	"E7 A6	9056",	#CJK UNIFIED IDEOGRAPH
	"E7 A7	9058",	#CJK UNIFIED IDEOGRAPH
	"E7 A8	905E",	#CJK UNIFIED IDEOGRAPH
	"E7 A9	9068",	#CJK UNIFIED IDEOGRAPH
	"E7 AA	906F",	#CJK UNIFIED IDEOGRAPH
	"E7 AB	9076",	#CJK UNIFIED IDEOGRAPH
	"E7 AC	96A8",	#CJK UNIFIED IDEOGRAPH
	"E7 AD	9072",	#CJK UNIFIED IDEOGRAPH
	"E7 AE	9082",	#CJK UNIFIED IDEOGRAPH
	"E7 AF	907D",	#CJK UNIFIED IDEOGRAPH
	"E7 B0	9081",	#CJK UNIFIED IDEOGRAPH
	"E7 B1	9080",	#CJK UNIFIED IDEOGRAPH
	"E7 B2	908A",	#CJK UNIFIED IDEOGRAPH
	"E7 B3	9089",	#CJK UNIFIED IDEOGRAPH
	"E7 B4	908F",	#CJK UNIFIED IDEOGRAPH
	"E7 B5	90A8",	#CJK UNIFIED IDEOGRAPH
	"E7 B6	90AF",	#CJK UNIFIED IDEOGRAPH
	"E7 B7	90B1",	#CJK UNIFIED IDEOGRAPH
	"E7 B8	90B5",	#CJK UNIFIED IDEOGRAPH
	"E7 B9	90E2",	#CJK UNIFIED IDEOGRAPH
	"E7 BA	90E4",	#CJK UNIFIED IDEOGRAPH
	"E7 BB	6248",	#CJK UNIFIED IDEOGRAPH
	"E7 BC	90DB",	#CJK UNIFIED IDEOGRAPH
	"E7 BD	9102",	#CJK UNIFIED IDEOGRAPH
	"E7 BE	9112",	#CJK UNIFIED IDEOGRAPH
	"E7 BF	9119",	#CJK UNIFIED IDEOGRAPH
	"E7 C0	9132",	#CJK UNIFIED IDEOGRAPH
	"E7 C1	9130",	#CJK UNIFIED IDEOGRAPH
	"E7 C2	914A",	#CJK UNIFIED IDEOGRAPH
	"E7 C3	9156",	#CJK UNIFIED IDEOGRAPH
	"E7 C4	9158",	#CJK UNIFIED IDEOGRAPH
	"E7 C5	9163",	#CJK UNIFIED IDEOGRAPH
	"E7 C6	9165",	#CJK UNIFIED IDEOGRAPH
	"E7 C7	9169",	#CJK UNIFIED IDEOGRAPH
	"E7 C8	9173",	#CJK UNIFIED IDEOGRAPH
	"E7 C9	9172",	#CJK UNIFIED IDEOGRAPH
	"E7 CA	918B",	#CJK UNIFIED IDEOGRAPH
	"E7 CB	9189",	#CJK UNIFIED IDEOGRAPH
	"E7 CC	9182",	#CJK UNIFIED IDEOGRAPH
	"E7 CD	91A2",	#CJK UNIFIED IDEOGRAPH
	"E7 CE	91AB",	#CJK UNIFIED IDEOGRAPH
	"E7 CF	91AF",	#CJK UNIFIED IDEOGRAPH
	"E7 D0	91AA",	#CJK UNIFIED IDEOGRAPH
	"E7 D1	91B5",	#CJK UNIFIED IDEOGRAPH
	"E7 D2	91B4",	#CJK UNIFIED IDEOGRAPH
	"E7 D3	91BA",	#CJK UNIFIED IDEOGRAPH
	"E7 D4	91C0",	#CJK UNIFIED IDEOGRAPH
	"E7 D5	91C1",	#CJK UNIFIED IDEOGRAPH
	"E7 D6	91C9",	#CJK UNIFIED IDEOGRAPH
	"E7 D7	91CB",	#CJK UNIFIED IDEOGRAPH
	"E7 D8	91D0",	#CJK UNIFIED IDEOGRAPH
	"E7 D9	91D6",	#CJK UNIFIED IDEOGRAPH
	"E7 DA	91DF",	#CJK UNIFIED IDEOGRAPH
	"E7 DB	91E1",	#CJK UNIFIED IDEOGRAPH
	"E7 DC	91DB",	#CJK UNIFIED IDEOGRAPH
	"E7 DD	91FC",	#CJK UNIFIED IDEOGRAPH
	"E7 DE	91F5",	#CJK UNIFIED IDEOGRAPH
	"E7 DF	91F6",	#CJK UNIFIED IDEOGRAPH
	"E7 E0	921E",	#CJK UNIFIED IDEOGRAPH
	"E7 E1	91FF",	#CJK UNIFIED IDEOGRAPH
	"E7 E2	9214",	#CJK UNIFIED IDEOGRAPH
	"E7 E3	922C",	#CJK UNIFIED IDEOGRAPH
	"E7 E4	9215",	#CJK UNIFIED IDEOGRAPH
	"E7 E5	9211",	#CJK UNIFIED IDEOGRAPH
	"E7 E6	925E",	#CJK UNIFIED IDEOGRAPH
	"E7 E7	9257",	#CJK UNIFIED IDEOGRAPH
	"E7 E8	9245",	#CJK UNIFIED IDEOGRAPH
	"E7 E9	9249",	#CJK UNIFIED IDEOGRAPH
	"E7 EA	9264",	#CJK UNIFIED IDEOGRAPH
	"E7 EB	9248",	#CJK UNIFIED IDEOGRAPH
	"E7 EC	9295",	#CJK UNIFIED IDEOGRAPH
	"E7 ED	923F",	#CJK UNIFIED IDEOGRAPH
	"E7 EE	924B",	#CJK UNIFIED IDEOGRAPH
	"E7 EF	9250",	#CJK UNIFIED IDEOGRAPH
	"E7 F0	929C",	#CJK UNIFIED IDEOGRAPH
	"E7 F1	9296",	#CJK UNIFIED IDEOGRAPH
	"E7 F2	9293",	#CJK UNIFIED IDEOGRAPH
	"E7 F3	929B",	#CJK UNIFIED IDEOGRAPH
	"E7 F4	925A",	#CJK UNIFIED IDEOGRAPH
	"E7 F5	92CF",	#CJK UNIFIED IDEOGRAPH
	"E7 F6	92B9",	#CJK UNIFIED IDEOGRAPH
	"E7 F7	92B7",	#CJK UNIFIED IDEOGRAPH
	"E7 F8	92E9",	#CJK UNIFIED IDEOGRAPH
	"E7 F9	930F",	#CJK UNIFIED IDEOGRAPH
	"E7 FA	92FA",	#CJK UNIFIED IDEOGRAPH
	"E7 FB	9344",	#CJK UNIFIED IDEOGRAPH
	"E7 FC	932E",	#CJK UNIFIED IDEOGRAPH
	"E8 40	9319",	#CJK UNIFIED IDEOGRAPH
	"E8 41	9322",	#CJK UNIFIED IDEOGRAPH
	"E8 42	931A",	#CJK UNIFIED IDEOGRAPH
	"E8 43	9323",	#CJK UNIFIED IDEOGRAPH
	"E8 44	933A",	#CJK UNIFIED IDEOGRAPH
	"E8 45	9335",	#CJK UNIFIED IDEOGRAPH
	"E8 46	933B",	#CJK UNIFIED IDEOGRAPH
	"E8 47	935C",	#CJK UNIFIED IDEOGRAPH
	"E8 48	9360",	#CJK UNIFIED IDEOGRAPH
	"E8 49	937C",	#CJK UNIFIED IDEOGRAPH
	"E8 4A	936E",	#CJK UNIFIED IDEOGRAPH
	"E8 4B	9356",	#CJK UNIFIED IDEOGRAPH
	"E8 4C	93B0",	#CJK UNIFIED IDEOGRAPH
	"E8 4D	93AC",	#CJK UNIFIED IDEOGRAPH
	"E8 4E	93AD",	#CJK UNIFIED IDEOGRAPH
	"E8 4F	9394",	#CJK UNIFIED IDEOGRAPH
	"E8 50	93B9",	#CJK UNIFIED IDEOGRAPH
	"E8 51	93D6",	#CJK UNIFIED IDEOGRAPH
	"E8 52	93D7",	#CJK UNIFIED IDEOGRAPH
	"E8 53	93E8",	#CJK UNIFIED IDEOGRAPH
	"E8 54	93E5",	#CJK UNIFIED IDEOGRAPH
	"E8 55	93D8",	#CJK UNIFIED IDEOGRAPH
	"E8 56	93C3",	#CJK UNIFIED IDEOGRAPH
	"E8 57	93DD",	#CJK UNIFIED IDEOGRAPH
	"E8 58	93D0",	#CJK UNIFIED IDEOGRAPH
	"E8 59	93C8",	#CJK UNIFIED IDEOGRAPH
	"E8 5A	93E4",	#CJK UNIFIED IDEOGRAPH
	"E8 5B	941A",	#CJK UNIFIED IDEOGRAPH
	"E8 5C	9414",	#CJK UNIFIED IDEOGRAPH
	"E8 5D	9413",	#CJK UNIFIED IDEOGRAPH
	"E8 5E	9403",	#CJK UNIFIED IDEOGRAPH
	"E8 5F	9407",	#CJK UNIFIED IDEOGRAPH
	"E8 60	9410",	#CJK UNIFIED IDEOGRAPH
	"E8 61	9436",	#CJK UNIFIED IDEOGRAPH
	"E8 62	942B",	#CJK UNIFIED IDEOGRAPH
	"E8 63	9435",	#CJK UNIFIED IDEOGRAPH
	"E8 64	9421",	#CJK UNIFIED IDEOGRAPH
	"E8 65	943A",	#CJK UNIFIED IDEOGRAPH
	"E8 66	9441",	#CJK UNIFIED IDEOGRAPH
	"E8 67	9452",	#CJK UNIFIED IDEOGRAPH
	"E8 68	9444",	#CJK UNIFIED IDEOGRAPH
	"E8 69	945B",	#CJK UNIFIED IDEOGRAPH
	"E8 6A	9460",	#CJK UNIFIED IDEOGRAPH
	"E8 6B	9462",	#CJK UNIFIED IDEOGRAPH
	"E8 6C	945E",	#CJK UNIFIED IDEOGRAPH
	"E8 6D	946A",	#CJK UNIFIED IDEOGRAPH
	"E8 6E	9229",	#CJK UNIFIED IDEOGRAPH
	"E8 6F	9470",	#CJK UNIFIED IDEOGRAPH
	"E8 70	9475",	#CJK UNIFIED IDEOGRAPH
	"E8 71	9477",	#CJK UNIFIED IDEOGRAPH
	"E8 72	947D",	#CJK UNIFIED IDEOGRAPH
	"E8 73	945A",	#CJK UNIFIED IDEOGRAPH
	"E8 74	947C",	#CJK UNIFIED IDEOGRAPH
	"E8 75	947E",	#CJK UNIFIED IDEOGRAPH
	"E8 76	9481",	#CJK UNIFIED IDEOGRAPH
	"E8 77	947F",	#CJK UNIFIED IDEOGRAPH
	"E8 78	9582",	#CJK UNIFIED IDEOGRAPH
	"E8 79	9587",	#CJK UNIFIED IDEOGRAPH
	"E8 7A	958A",	#CJK UNIFIED IDEOGRAPH
	"E8 7B	9594",	#CJK UNIFIED IDEOGRAPH
	"E8 7C	9596",	#CJK UNIFIED IDEOGRAPH
	"E8 7D	9598",	#CJK UNIFIED IDEOGRAPH
	"E8 7E	9599",	#CJK UNIFIED IDEOGRAPH
	"E8 80	95A0",	#CJK UNIFIED IDEOGRAPH
	"E8 81	95A8",	#CJK UNIFIED IDEOGRAPH
	"E8 82	95A7",	#CJK UNIFIED IDEOGRAPH
	"E8 83	95AD",	#CJK UNIFIED IDEOGRAPH
	"E8 84	95BC",	#CJK UNIFIED IDEOGRAPH
	"E8 85	95BB",	#CJK UNIFIED IDEOGRAPH
	"E8 86	95B9",	#CJK UNIFIED IDEOGRAPH
	"E8 87	95BE",	#CJK UNIFIED IDEOGRAPH
	"E8 88	95CA",	#CJK UNIFIED IDEOGRAPH
	"E8 89	6FF6",	#CJK UNIFIED IDEOGRAPH
	"E8 8A	95C3",	#CJK UNIFIED IDEOGRAPH
	"E8 8B	95CD",	#CJK UNIFIED IDEOGRAPH
	"E8 8C	95CC",	#CJK UNIFIED IDEOGRAPH
	"E8 8D	95D5",	#CJK UNIFIED IDEOGRAPH
	"E8 8E	95D4",	#CJK UNIFIED IDEOGRAPH
	"E8 8F	95D6",	#CJK UNIFIED IDEOGRAPH
	"E8 90	95DC",	#CJK UNIFIED IDEOGRAPH
	"E8 91	95E1",	#CJK UNIFIED IDEOGRAPH
	"E8 92	95E5",	#CJK UNIFIED IDEOGRAPH
	"E8 93	95E2",	#CJK UNIFIED IDEOGRAPH
	"E8 94	9621",	#CJK UNIFIED IDEOGRAPH
	"E8 95	9628",	#CJK UNIFIED IDEOGRAPH
	"E8 96	962E",	#CJK UNIFIED IDEOGRAPH
	"E8 97	962F",	#CJK UNIFIED IDEOGRAPH
	"E8 98	9642",	#CJK UNIFIED IDEOGRAPH
	"E8 99	964C",	#CJK UNIFIED IDEOGRAPH
	"E8 9A	964F",	#CJK UNIFIED IDEOGRAPH
	"E8 9B	964B",	#CJK UNIFIED IDEOGRAPH
	"E8 9C	9677",	#CJK UNIFIED IDEOGRAPH
	"E8 9D	965C",	#CJK UNIFIED IDEOGRAPH
	"E8 9E	965E",	#CJK UNIFIED IDEOGRAPH
	"E8 9F	965D",	#CJK UNIFIED IDEOGRAPH
	"E8 A0	965F",	#CJK UNIFIED IDEOGRAPH
	"E8 A1	9666",	#CJK UNIFIED IDEOGRAPH
	"E8 A2	9672",	#CJK UNIFIED IDEOGRAPH
	"E8 A3	966C",	#CJK UNIFIED IDEOGRAPH
	"E8 A4	968D",	#CJK UNIFIED IDEOGRAPH
	"E8 A5	9698",	#CJK UNIFIED IDEOGRAPH
	"E8 A6	9695",	#CJK UNIFIED IDEOGRAPH
	"E8 A7	9697",	#CJK UNIFIED IDEOGRAPH
	"E8 A8	96AA",	#CJK UNIFIED IDEOGRAPH
	"E8 A9	96A7",	#CJK UNIFIED IDEOGRAPH
	"E8 AA	96B1",	#CJK UNIFIED IDEOGRAPH
	"E8 AB	96B2",	#CJK UNIFIED IDEOGRAPH
	"E8 AC	96B0",	#CJK UNIFIED IDEOGRAPH
	"E8 AD	96B4",	#CJK UNIFIED IDEOGRAPH
	"E8 AE	96B6",	#CJK UNIFIED IDEOGRAPH
	"E8 AF	96B8",	#CJK UNIFIED IDEOGRAPH
	"E8 B0	96B9",	#CJK UNIFIED IDEOGRAPH
	"E8 B1	96CE",	#CJK UNIFIED IDEOGRAPH
	"E8 B2	96CB",	#CJK UNIFIED IDEOGRAPH
	"E8 B3	96C9",	#CJK UNIFIED IDEOGRAPH
	"E8 B4	96CD",	#CJK UNIFIED IDEOGRAPH
	"E8 B5	894D",	#CJK UNIFIED IDEOGRAPH
	"E8 B6	96DC",	#CJK UNIFIED IDEOGRAPH
	"E8 B7	970D",	#CJK UNIFIED IDEOGRAPH
	"E8 B8	96D5",	#CJK UNIFIED IDEOGRAPH
	"E8 B9	96F9",	#CJK UNIFIED IDEOGRAPH
	"E8 BA	9704",	#CJK UNIFIED IDEOGRAPH
	"E8 BB	9706",	#CJK UNIFIED IDEOGRAPH
	"E8 BC	9708",	#CJK UNIFIED IDEOGRAPH
	"E8 BD	9713",	#CJK UNIFIED IDEOGRAPH
	"E8 BE	970E",	#CJK UNIFIED IDEOGRAPH
	"E8 BF	9711",	#CJK UNIFIED IDEOGRAPH
	"E8 C0	970F",	#CJK UNIFIED IDEOGRAPH
	"E8 C1	9716",	#CJK UNIFIED IDEOGRAPH
	"E8 C2	9719",	#CJK UNIFIED IDEOGRAPH
	"E8 C3	9724",	#CJK UNIFIED IDEOGRAPH
	"E8 C4	972A",	#CJK UNIFIED IDEOGRAPH
	"E8 C5	9730",	#CJK UNIFIED IDEOGRAPH
	"E8 C6	9739",	#CJK UNIFIED IDEOGRAPH
	"E8 C7	973D",	#CJK UNIFIED IDEOGRAPH
	"E8 C8	973E",	#CJK UNIFIED IDEOGRAPH
	"E8 C9	9744",	#CJK UNIFIED IDEOGRAPH
	"E8 CA	9746",	#CJK UNIFIED IDEOGRAPH
	"E8 CB	9748",	#CJK UNIFIED IDEOGRAPH
	"E8 CC	9742",	#CJK UNIFIED IDEOGRAPH
	"E8 CD	9749",	#CJK UNIFIED IDEOGRAPH
	"E8 CE	975C",	#CJK UNIFIED IDEOGRAPH
	"E8 CF	9760",	#CJK UNIFIED IDEOGRAPH
	"E8 D0	9764",	#CJK UNIFIED IDEOGRAPH
	"E8 D1	9766",	#CJK UNIFIED IDEOGRAPH
	"E8 D2	9768",	#CJK UNIFIED IDEOGRAPH
	"E8 D3	52D2",	#CJK UNIFIED IDEOGRAPH
	"E8 D4	976B",	#CJK UNIFIED IDEOGRAPH
	"E8 D5	9771",	#CJK UNIFIED IDEOGRAPH
	"E8 D6	9779",	#CJK UNIFIED IDEOGRAPH
	"E8 D7	9785",	#CJK UNIFIED IDEOGRAPH
	"E8 D8	977C",	#CJK UNIFIED IDEOGRAPH
	"E8 D9	9781",	#CJK UNIFIED IDEOGRAPH
	"E8 DA	977A",	#CJK UNIFIED IDEOGRAPH
	"E8 DB	9786",	#CJK UNIFIED IDEOGRAPH
	"E8 DC	978B",	#CJK UNIFIED IDEOGRAPH
	"E8 DD	978F",	#CJK UNIFIED IDEOGRAPH
	"E8 DE	9790",	#CJK UNIFIED IDEOGRAPH
	"E8 DF	979C",	#CJK UNIFIED IDEOGRAPH
	"E8 E0	97A8",	#CJK UNIFIED IDEOGRAPH
	"E8 E1	97A6",	#CJK UNIFIED IDEOGRAPH
	"E8 E2	97A3",	#CJK UNIFIED IDEOGRAPH
	"E8 E3	97B3",	#CJK UNIFIED IDEOGRAPH
	"E8 E4	97B4",	#CJK UNIFIED IDEOGRAPH
	"E8 E5	97C3",	#CJK UNIFIED IDEOGRAPH
	"E8 E6	97C6",	#CJK UNIFIED IDEOGRAPH
	"E8 E7	97C8",	#CJK UNIFIED IDEOGRAPH
	"E8 E8	97CB",	#CJK UNIFIED IDEOGRAPH
	"E8 E9	97DC",	#CJK UNIFIED IDEOGRAPH
	"E8 EA	97ED",	#CJK UNIFIED IDEOGRAPH
	"E8 EB	9F4F",	#CJK UNIFIED IDEOGRAPH
	"E8 EC	97F2",	#CJK UNIFIED IDEOGRAPH
	"E8 ED	7ADF",	#CJK UNIFIED IDEOGRAPH
	"E8 EE	97F6",	#CJK UNIFIED IDEOGRAPH
	"E8 EF	97F5",	#CJK UNIFIED IDEOGRAPH
	"E8 F0	980F",	#CJK UNIFIED IDEOGRAPH
	"E8 F1	980C",	#CJK UNIFIED IDEOGRAPH
	"E8 F2	9838",	#CJK UNIFIED IDEOGRAPH
	"E8 F3	9824",	#CJK UNIFIED IDEOGRAPH
	"E8 F4	9821",	#CJK UNIFIED IDEOGRAPH
	"E8 F5	9837",	#CJK UNIFIED IDEOGRAPH
	"E8 F6	983D",	#CJK UNIFIED IDEOGRAPH
	"E8 F7	9846",	#CJK UNIFIED IDEOGRAPH
	"E8 F8	984F",	#CJK UNIFIED IDEOGRAPH
	"E8 F9	984B",	#CJK UNIFIED IDEOGRAPH
	"E8 FA	986B",	#CJK UNIFIED IDEOGRAPH
	"E8 FB	986F",	#CJK UNIFIED IDEOGRAPH
	"E8 FC	9870",	#CJK UNIFIED IDEOGRAPH
	"E9 40	9871",	#CJK UNIFIED IDEOGRAPH
	"E9 41	9874",	#CJK UNIFIED IDEOGRAPH
	"E9 42	9873",	#CJK UNIFIED IDEOGRAPH
	"E9 43	98AA",	#CJK UNIFIED IDEOGRAPH
	"E9 44	98AF",	#CJK UNIFIED IDEOGRAPH
	"E9 45	98B1",	#CJK UNIFIED IDEOGRAPH
	"E9 46	98B6",	#CJK UNIFIED IDEOGRAPH
	"E9 47	98C4",	#CJK UNIFIED IDEOGRAPH
	"E9 48	98C3",	#CJK UNIFIED IDEOGRAPH
	"E9 49	98C6",	#CJK UNIFIED IDEOGRAPH
	"E9 4A	98E9",	#CJK UNIFIED IDEOGRAPH
	"E9 4B	98EB",	#CJK UNIFIED IDEOGRAPH
	"E9 4C	9903",	#CJK UNIFIED IDEOGRAPH
	"E9 4D	9909",	#CJK UNIFIED IDEOGRAPH
	"E9 4E	9912",	#CJK UNIFIED IDEOGRAPH
	"E9 4F	9914",	#CJK UNIFIED IDEOGRAPH
	"E9 50	9918",	#CJK UNIFIED IDEOGRAPH
	"E9 51	9921",	#CJK UNIFIED IDEOGRAPH
	"E9 52	991D",	#CJK UNIFIED IDEOGRAPH
	"E9 53	991E",	#CJK UNIFIED IDEOGRAPH
	"E9 54	9924",	#CJK UNIFIED IDEOGRAPH
	"E9 55	9920",	#CJK UNIFIED IDEOGRAPH
	"E9 56	992C",	#CJK UNIFIED IDEOGRAPH
	"E9 57	992E",	#CJK UNIFIED IDEOGRAPH
	"E9 58	993D",	#CJK UNIFIED IDEOGRAPH
	"E9 59	993E",	#CJK UNIFIED IDEOGRAPH
	"E9 5A	9942",	#CJK UNIFIED IDEOGRAPH
	"E9 5B	9949",	#CJK UNIFIED IDEOGRAPH
	"E9 5C	9945",	#CJK UNIFIED IDEOGRAPH
	"E9 5D	9950",	#CJK UNIFIED IDEOGRAPH
	"E9 5E	994B",	#CJK UNIFIED IDEOGRAPH
	"E9 5F	9951",	#CJK UNIFIED IDEOGRAPH
	"E9 60	9952",	#CJK UNIFIED IDEOGRAPH
	"E9 61	994C",	#CJK UNIFIED IDEOGRAPH
	"E9 62	9955",	#CJK UNIFIED IDEOGRAPH
	"E9 63	9997",	#CJK UNIFIED IDEOGRAPH
	"E9 64	9998",	#CJK UNIFIED IDEOGRAPH
	"E9 65	99A5",	#CJK UNIFIED IDEOGRAPH
	"E9 66	99AD",	#CJK UNIFIED IDEOGRAPH
	"E9 67	99AE",	#CJK UNIFIED IDEOGRAPH
	"E9 68	99BC",	#CJK UNIFIED IDEOGRAPH
	"E9 69	99DF",	#CJK UNIFIED IDEOGRAPH
	"E9 6A	99DB",	#CJK UNIFIED IDEOGRAPH
	"E9 6B	99DD",	#CJK UNIFIED IDEOGRAPH
	"E9 6C	99D8",	#CJK UNIFIED IDEOGRAPH
	"E9 6D	99D1",	#CJK UNIFIED IDEOGRAPH
	"E9 6E	99ED",	#CJK UNIFIED IDEOGRAPH
	"E9 6F	99EE",	#CJK UNIFIED IDEOGRAPH
	"E9 70	99F1",	#CJK UNIFIED IDEOGRAPH
	"E9 71	99F2",	#CJK UNIFIED IDEOGRAPH
	"E9 72	99FB",	#CJK UNIFIED IDEOGRAPH
	"E9 73	99F8",	#CJK UNIFIED IDEOGRAPH
	"E9 74	9A01",	#CJK UNIFIED IDEOGRAPH
	"E9 75	9A0F",	#CJK UNIFIED IDEOGRAPH
	"E9 76	9A05",	#CJK UNIFIED IDEOGRAPH
	"E9 77	99E2",	#CJK UNIFIED IDEOGRAPH
	"E9 78	9A19",	#CJK UNIFIED IDEOGRAPH
	"E9 79	9A2B",	#CJK UNIFIED IDEOGRAPH
	"E9 7A	9A37",	#CJK UNIFIED IDEOGRAPH
	"E9 7B	9A45",	#CJK UNIFIED IDEOGRAPH
	"E9 7C	9A42",	#CJK UNIFIED IDEOGRAPH
	"E9 7D	9A40",	#CJK UNIFIED IDEOGRAPH
	"E9 7E	9A43",	#CJK UNIFIED IDEOGRAPH
	"E9 80	9A3E",	#CJK UNIFIED IDEOGRAPH
	"E9 81	9A55",	#CJK UNIFIED IDEOGRAPH
	"E9 82	9A4D",	#CJK UNIFIED IDEOGRAPH
	"E9 83	9A5B",	#CJK UNIFIED IDEOGRAPH
	"E9 84	9A57",	#CJK UNIFIED IDEOGRAPH
	"E9 85	9A5F",	#CJK UNIFIED IDEOGRAPH
	"E9 86	9A62",	#CJK UNIFIED IDEOGRAPH
	"E9 87	9A65",	#CJK UNIFIED IDEOGRAPH
	"E9 88	9A64",	#CJK UNIFIED IDEOGRAPH
	"E9 89	9A69",	#CJK UNIFIED IDEOGRAPH
	"E9 8A	9A6B",	#CJK UNIFIED IDEOGRAPH
	"E9 8B	9A6A",	#CJK UNIFIED IDEOGRAPH
	"E9 8C	9AAD",	#CJK UNIFIED IDEOGRAPH
	"E9 8D	9AB0",	#CJK UNIFIED IDEOGRAPH
	"E9 8E	9ABC",	#CJK UNIFIED IDEOGRAPH
	"E9 8F	9AC0",	#CJK UNIFIED IDEOGRAPH
	"E9 90	9ACF",	#CJK UNIFIED IDEOGRAPH
	"E9 91	9AD1",	#CJK UNIFIED IDEOGRAPH
	"E9 92	9AD3",	#CJK UNIFIED IDEOGRAPH
	"E9 93	9AD4",	#CJK UNIFIED IDEOGRAPH
	"E9 94	9ADE",	#CJK UNIFIED IDEOGRAPH
	"E9 95	9ADF",	#CJK UNIFIED IDEOGRAPH
	"E9 96	9AE2",	#CJK UNIFIED IDEOGRAPH
	"E9 97	9AE3",	#CJK UNIFIED IDEOGRAPH
	"E9 98	9AE6",	#CJK UNIFIED IDEOGRAPH
	"E9 99	9AEF",	#CJK UNIFIED IDEOGRAPH
	"E9 9A	9AEB",	#CJK UNIFIED IDEOGRAPH
	"E9 9B	9AEE",	#CJK UNIFIED IDEOGRAPH
	"E9 9C	9AF4",	#CJK UNIFIED IDEOGRAPH
	"E9 9D	9AF1",	#CJK UNIFIED IDEOGRAPH
	"E9 9E	9AF7",	#CJK UNIFIED IDEOGRAPH
	"E9 9F	9AFB",	#CJK UNIFIED IDEOGRAPH
	"E9 A0	9B06",	#CJK UNIFIED IDEOGRAPH
	"E9 A1	9B18",	#CJK UNIFIED IDEOGRAPH
	"E9 A2	9B1A",	#CJK UNIFIED IDEOGRAPH
	"E9 A3	9B1F",	#CJK UNIFIED IDEOGRAPH
	"E9 A4	9B22",	#CJK UNIFIED IDEOGRAPH
	"E9 A5	9B23",	#CJK UNIFIED IDEOGRAPH
	"E9 A6	9B25",	#CJK UNIFIED IDEOGRAPH
	"E9 A7	9B27",	#CJK UNIFIED IDEOGRAPH
	"E9 A8	9B28",	#CJK UNIFIED IDEOGRAPH
	"E9 A9	9B29",	#CJK UNIFIED IDEOGRAPH
	"E9 AA	9B2A",	#CJK UNIFIED IDEOGRAPH
	"E9 AB	9B2E",	#CJK UNIFIED IDEOGRAPH
	"E9 AC	9B2F",	#CJK UNIFIED IDEOGRAPH
	"E9 AD	9B32",	#CJK UNIFIED IDEOGRAPH
	"E9 AE	9B44",	#CJK UNIFIED IDEOGRAPH
	"E9 AF	9B43",	#CJK UNIFIED IDEOGRAPH
	"E9 B0	9B4F",	#CJK UNIFIED IDEOGRAPH
	"E9 B1	9B4D",	#CJK UNIFIED IDEOGRAPH
	"E9 B2	9B4E",	#CJK UNIFIED IDEOGRAPH
	"E9 B3	9B51",	#CJK UNIFIED IDEOGRAPH
	"E9 B4	9B58",	#CJK UNIFIED IDEOGRAPH
	"E9 B5	9B74",	#CJK UNIFIED IDEOGRAPH
	"E9 B6	9B93",	#CJK UNIFIED IDEOGRAPH
	"E9 B7	9B83",	#CJK UNIFIED IDEOGRAPH
	"E9 B8	9B91",	#CJK UNIFIED IDEOGRAPH
	"E9 B9	9B96",	#CJK UNIFIED IDEOGRAPH
	"E9 BA	9B97",	#CJK UNIFIED IDEOGRAPH
	"E9 BB	9B9F",	#CJK UNIFIED IDEOGRAPH
	"E9 BC	9BA0",	#CJK UNIFIED IDEOGRAPH
	"E9 BD	9BA8",	#CJK UNIFIED IDEOGRAPH
	"E9 BE	9BB4",	#CJK UNIFIED IDEOGRAPH
	"E9 BF	9BC0",	#CJK UNIFIED IDEOGRAPH
	"E9 C0	9BCA",	#CJK UNIFIED IDEOGRAPH
	"E9 C1	9BB9",	#CJK UNIFIED IDEOGRAPH
	"E9 C2	9BC6",	#CJK UNIFIED IDEOGRAPH
	"E9 C3	9BCF",	#CJK UNIFIED IDEOGRAPH
	"E9 C4	9BD1",	#CJK UNIFIED IDEOGRAPH
	"E9 C5	9BD2",	#CJK UNIFIED IDEOGRAPH
	"E9 C6	9BE3",	#CJK UNIFIED IDEOGRAPH
	"E9 C7	9BE2",	#CJK UNIFIED IDEOGRAPH
	"E9 C8	9BE4",	#CJK UNIFIED IDEOGRAPH
	"E9 C9	9BD4",	#CJK UNIFIED IDEOGRAPH
	"E9 CA	9BE1",	#CJK UNIFIED IDEOGRAPH
	"E9 CB	9C3A",	#CJK UNIFIED IDEOGRAPH
	"E9 CC	9BF2",	#CJK UNIFIED IDEOGRAPH
	"E9 CD	9BF1",	#CJK UNIFIED IDEOGRAPH
	"E9 CE	9BF0",	#CJK UNIFIED IDEOGRAPH
	"E9 CF	9C15",	#CJK UNIFIED IDEOGRAPH
	"E9 D0	9C14",	#CJK UNIFIED IDEOGRAPH
	"E9 D1	9C09",	#CJK UNIFIED IDEOGRAPH
	"E9 D2	9C13",	#CJK UNIFIED IDEOGRAPH
	"E9 D3	9C0C",	#CJK UNIFIED IDEOGRAPH
	"E9 D4	9C06",	#CJK UNIFIED IDEOGRAPH
	"E9 D5	9C08",	#CJK UNIFIED IDEOGRAPH
	"E9 D6	9C12",	#CJK UNIFIED IDEOGRAPH
	"E9 D7	9C0A",	#CJK UNIFIED IDEOGRAPH
	"E9 D8	9C04",	#CJK UNIFIED IDEOGRAPH
	"E9 D9	9C2E",	#CJK UNIFIED IDEOGRAPH
	"E9 DA	9C1B",	#CJK UNIFIED IDEOGRAPH
	"E9 DB	9C25",	#CJK UNIFIED IDEOGRAPH
	"E9 DC	9C24",	#CJK UNIFIED IDEOGRAPH
	"E9 DD	9C21",	#CJK UNIFIED IDEOGRAPH
	"E9 DE	9C30",	#CJK UNIFIED IDEOGRAPH
	"E9 DF	9C47",	#CJK UNIFIED IDEOGRAPH
	"E9 E0	9C32",	#CJK UNIFIED IDEOGRAPH
	"E9 E1	9C46",	#CJK UNIFIED IDEOGRAPH
	"E9 E2	9C3E",	#CJK UNIFIED IDEOGRAPH
	"E9 E3	9C5A",	#CJK UNIFIED IDEOGRAPH
	"E9 E4	9C60",	#CJK UNIFIED IDEOGRAPH
	"E9 E5	9C67",	#CJK UNIFIED IDEOGRAPH
	"E9 E6	9C76",	#CJK UNIFIED IDEOGRAPH
	"E9 E7	9C78",	#CJK UNIFIED IDEOGRAPH
	"E9 E8	9CE7",	#CJK UNIFIED IDEOGRAPH
	"E9 E9	9CEC",	#CJK UNIFIED IDEOGRAPH
	"E9 EA	9CF0",	#CJK UNIFIED IDEOGRAPH
	"E9 EB	9D09",	#CJK UNIFIED IDEOGRAPH
	"E9 EC	9D08",	#CJK UNIFIED IDEOGRAPH
	"E9 ED	9CEB",	#CJK UNIFIED IDEOGRAPH
	"E9 EE	9D03",	#CJK UNIFIED IDEOGRAPH
	"E9 EF	9D06",	#CJK UNIFIED IDEOGRAPH
	"E9 F0	9D2A",	#CJK UNIFIED IDEOGRAPH
	"E9 F1	9D26",	#CJK UNIFIED IDEOGRAPH
	"E9 F2	9DAF",	#CJK UNIFIED IDEOGRAPH
	"E9 F3	9D23",	#CJK UNIFIED IDEOGRAPH
	"E9 F4	9D1F",	#CJK UNIFIED IDEOGRAPH
	"E9 F5	9D44",	#CJK UNIFIED IDEOGRAPH
	"E9 F6	9D15",	#CJK UNIFIED IDEOGRAPH
	"E9 F7	9D12",	#CJK UNIFIED IDEOGRAPH
	"E9 F8	9D41",	#CJK UNIFIED IDEOGRAPH
	"E9 F9	9D3F",	#CJK UNIFIED IDEOGRAPH
	"E9 FA	9D3E",	#CJK UNIFIED IDEOGRAPH
	"E9 FB	9D46",	#CJK UNIFIED IDEOGRAPH
	"E9 FC	9D48",	#CJK UNIFIED IDEOGRAPH
	"EA 40	9D5D",	#CJK UNIFIED IDEOGRAPH
	"EA 41	9D5E",	#CJK UNIFIED IDEOGRAPH
	"EA 42	9D64",	#CJK UNIFIED IDEOGRAPH
	"EA 43	9D51",	#CJK UNIFIED IDEOGRAPH
	"EA 44	9D50",	#CJK UNIFIED IDEOGRAPH
	"EA 45	9D59",	#CJK UNIFIED IDEOGRAPH
	"EA 46	9D72",	#CJK UNIFIED IDEOGRAPH
	"EA 47	9D89",	#CJK UNIFIED IDEOGRAPH
	"EA 48	9D87",	#CJK UNIFIED IDEOGRAPH
	"EA 49	9DAB",	#CJK UNIFIED IDEOGRAPH
	"EA 4A	9D6F",	#CJK UNIFIED IDEOGRAPH
	"EA 4B	9D7A",	#CJK UNIFIED IDEOGRAPH
	"EA 4C	9D9A",	#CJK UNIFIED IDEOGRAPH
	"EA 4D	9DA4",	#CJK UNIFIED IDEOGRAPH
	"EA 4E	9DA9",	#CJK UNIFIED IDEOGRAPH
	"EA 4F	9DB2",	#CJK UNIFIED IDEOGRAPH
	"EA 50	9DC4",	#CJK UNIFIED IDEOGRAPH
	"EA 51	9DC1",	#CJK UNIFIED IDEOGRAPH
	"EA 52	9DBB",	#CJK UNIFIED IDEOGRAPH
	"EA 53	9DB8",	#CJK UNIFIED IDEOGRAPH
	"EA 54	9DBA",	#CJK UNIFIED IDEOGRAPH
	"EA 55	9DC6",	#CJK UNIFIED IDEOGRAPH
	"EA 56	9DCF",	#CJK UNIFIED IDEOGRAPH
	"EA 57	9DC2",	#CJK UNIFIED IDEOGRAPH
	"EA 58	9DD9",	#CJK UNIFIED IDEOGRAPH
	"EA 59	9DD3",	#CJK UNIFIED IDEOGRAPH
	"EA 5A	9DF8",	#CJK UNIFIED IDEOGRAPH
	"EA 5B	9DE6",	#CJK UNIFIED IDEOGRAPH
	"EA 5C	9DED",	#CJK UNIFIED IDEOGRAPH
	"EA 5D	9DEF",	#CJK UNIFIED IDEOGRAPH
	"EA 5E	9DFD",	#CJK UNIFIED IDEOGRAPH
	"EA 5F	9E1A",	#CJK UNIFIED IDEOGRAPH
	"EA 60	9E1B",	#CJK UNIFIED IDEOGRAPH
	"EA 61	9E1E",	#CJK UNIFIED IDEOGRAPH
	"EA 62	9E75",	#CJK UNIFIED IDEOGRAPH
	"EA 63	9E79",	#CJK UNIFIED IDEOGRAPH
	"EA 64	9E7D",	#CJK UNIFIED IDEOGRAPH
	"EA 65	9E81",	#CJK UNIFIED IDEOGRAPH
	"EA 66	9E88",	#CJK UNIFIED IDEOGRAPH
	"EA 67	9E8B",	#CJK UNIFIED IDEOGRAPH
	"EA 68	9E8C",	#CJK UNIFIED IDEOGRAPH
	"EA 69	9E92",	#CJK UNIFIED IDEOGRAPH
	"EA 6A	9E95",	#CJK UNIFIED IDEOGRAPH
	"EA 6B	9E91",	#CJK UNIFIED IDEOGRAPH
	"EA 6C	9E9D",	#CJK UNIFIED IDEOGRAPH
	"EA 6D	9EA5",	#CJK UNIFIED IDEOGRAPH
	"EA 6E	9EA9",	#CJK UNIFIED IDEOGRAPH
	"EA 6F	9EB8",	#CJK UNIFIED IDEOGRAPH
	"EA 70	9EAA",	#CJK UNIFIED IDEOGRAPH
	"EA 71	9EAD",	#CJK UNIFIED IDEOGRAPH
	"EA 72	9761",	#CJK UNIFIED IDEOGRAPH
	"EA 73	9ECC",	#CJK UNIFIED IDEOGRAPH
	"EA 74	9ECE",	#CJK UNIFIED IDEOGRAPH
	"EA 75	9ECF",	#CJK UNIFIED IDEOGRAPH
	"EA 76	9ED0",	#CJK UNIFIED IDEOGRAPH
	"EA 77	9ED4",	#CJK UNIFIED IDEOGRAPH
	"EA 78	9EDC",	#CJK UNIFIED IDEOGRAPH
	"EA 79	9EDE",	#CJK UNIFIED IDEOGRAPH
	"EA 7A	9EDD",	#CJK UNIFIED IDEOGRAPH
	"EA 7B	9EE0",	#CJK UNIFIED IDEOGRAPH
	"EA 7C	9EE5",	#CJK UNIFIED IDEOGRAPH
	"EA 7D	9EE8",	#CJK UNIFIED IDEOGRAPH
	"EA 7E	9EEF",	#CJK UNIFIED IDEOGRAPH
	"EA 80	9EF4",	#CJK UNIFIED IDEOGRAPH
	"EA 81	9EF6",	#CJK UNIFIED IDEOGRAPH
	"EA 82	9EF7",	#CJK UNIFIED IDEOGRAPH
	"EA 83	9EF9",	#CJK UNIFIED IDEOGRAPH
	"EA 84	9EFB",	#CJK UNIFIED IDEOGRAPH
	"EA 85	9EFC",	#CJK UNIFIED IDEOGRAPH
	"EA 86	9EFD",	#CJK UNIFIED IDEOGRAPH
	"EA 87	9F07",	#CJK UNIFIED IDEOGRAPH
	"EA 88	9F08",	#CJK UNIFIED IDEOGRAPH
	"EA 89	76B7",	#CJK UNIFIED IDEOGRAPH
	"EA 8A	9F15",	#CJK UNIFIED IDEOGRAPH
	"EA 8B	9F21",	#CJK UNIFIED IDEOGRAPH
	"EA 8C	9F2C",	#CJK UNIFIED IDEOGRAPH
	"EA 8D	9F3E",	#CJK UNIFIED IDEOGRAPH
	"EA 8E	9F4A",	#CJK UNIFIED IDEOGRAPH
	"EA 8F	9F52",	#CJK UNIFIED IDEOGRAPH
	"EA 90	9F54",	#CJK UNIFIED IDEOGRAPH
	"EA 91	9F63",	#CJK UNIFIED IDEOGRAPH
	"EA 92	9F5F",	#CJK UNIFIED IDEOGRAPH
	"EA 93	9F60",	#CJK UNIFIED IDEOGRAPH
	"EA 94	9F61",	#CJK UNIFIED IDEOGRAPH
	"EA 95	9F66",	#CJK UNIFIED IDEOGRAPH
	"EA 96	9F67",	#CJK UNIFIED IDEOGRAPH
	"EA 97	9F6C",	#CJK UNIFIED IDEOGRAPH
	"EA 98	9F6A",	#CJK UNIFIED IDEOGRAPH
	"EA 99	9F77",	#CJK UNIFIED IDEOGRAPH
	"EA 9A	9F72",	#CJK UNIFIED IDEOGRAPH
	"EA 9B	9F76",	#CJK UNIFIED IDEOGRAPH
	"EA 9C	9F95",	#CJK UNIFIED IDEOGRAPH
	"EA 9D	9F9C",	#CJK UNIFIED IDEOGRAPH
	"EA 9E	9FA0",	#CJK UNIFIED IDEOGRAPH
	"EA 9F	582F",	#CJK UNIFIED IDEOGRAPH
	"EA A0	69C7",	#CJK UNIFIED IDEOGRAPH
	"EA A1	9059",	#CJK UNIFIED IDEOGRAPH
	"EA A2	7464",	#CJK UNIFIED IDEOGRAPH
	"EA A3	51DC",	#CJK UNIFIED IDEOGRAPH
	"EA A4	7199",	#CJK UNIFIED IDEOGRAPH
	"ED 40	7E8A",	#CJK UNIFIED IDEOGRAPH
	"ED 41	891C",	#CJK UNIFIED IDEOGRAPH
	"ED 42	9348",	#CJK UNIFIED IDEOGRAPH
	"ED 43	9288",	#CJK UNIFIED IDEOGRAPH
	"ED 44	84DC",	#CJK UNIFIED IDEOGRAPH
	"ED 45	4FC9",	#CJK UNIFIED IDEOGRAPH
	"ED 46	70BB",	#CJK UNIFIED IDEOGRAPH
	"ED 47	6631",	#CJK UNIFIED IDEOGRAPH
	"ED 48	68C8",	#CJK UNIFIED IDEOGRAPH
	"ED 49	92F9",	#CJK UNIFIED IDEOGRAPH
	"ED 4A	66FB",	#CJK UNIFIED IDEOGRAPH
	"ED 4B	5F45",	#CJK UNIFIED IDEOGRAPH
	"ED 4C	4E28",	#CJK UNIFIED IDEOGRAPH
	"ED 4D	4EE1",	#CJK UNIFIED IDEOGRAPH
	"ED 4E	4EFC",	#CJK UNIFIED IDEOGRAPH
	"ED 4F	4F00",	#CJK UNIFIED IDEOGRAPH
	"ED 50	4F03",	#CJK UNIFIED IDEOGRAPH
	"ED 51	4F39",	#CJK UNIFIED IDEOGRAPH
	"ED 52	4F56",	#CJK UNIFIED IDEOGRAPH
	"ED 53	4F92",	#CJK UNIFIED IDEOGRAPH
	"ED 54	4F8A",	#CJK UNIFIED IDEOGRAPH
	"ED 55	4F9A",	#CJK UNIFIED IDEOGRAPH
	"ED 56	4F94",	#CJK UNIFIED IDEOGRAPH
	"ED 57	4FCD",	#CJK UNIFIED IDEOGRAPH
	"ED 58	5040",	#CJK UNIFIED IDEOGRAPH
	"ED 59	5022",	#CJK UNIFIED IDEOGRAPH
	"ED 5A	4FFF",	#CJK UNIFIED IDEOGRAPH
	"ED 5B	501E",	#CJK UNIFIED IDEOGRAPH
	"ED 5C	5046",	#CJK UNIFIED IDEOGRAPH
	"ED 5D	5070",	#CJK UNIFIED IDEOGRAPH
	"ED 5E	5042",	#CJK UNIFIED IDEOGRAPH
	"ED 5F	5094",	#CJK UNIFIED IDEOGRAPH
	"ED 60	50F4",	#CJK UNIFIED IDEOGRAPH
	"ED 61	50D8",	#CJK UNIFIED IDEOGRAPH
	"ED 62	514A",	#CJK UNIFIED IDEOGRAPH
	"ED 63	5164",	#CJK UNIFIED IDEOGRAPH
	"ED 64	519D",	#CJK UNIFIED IDEOGRAPH
	"ED 65	51BE",	#CJK UNIFIED IDEOGRAPH
	"ED 66	51EC",	#CJK UNIFIED IDEOGRAPH
	"ED 67	5215",	#CJK UNIFIED IDEOGRAPH
	"ED 68	529C",	#CJK UNIFIED IDEOGRAPH
	"ED 69	52A6",	#CJK UNIFIED IDEOGRAPH
	"ED 6A	52C0",	#CJK UNIFIED IDEOGRAPH
	"ED 6B	52DB",	#CJK UNIFIED IDEOGRAPH
	"ED 6C	5300",	#CJK UNIFIED IDEOGRAPH
	"ED 6D	5307",	#CJK UNIFIED IDEOGRAPH
	"ED 6E	5324",	#CJK UNIFIED IDEOGRAPH
	"ED 6F	5372",	#CJK UNIFIED IDEOGRAPH
	"ED 70	5393",	#CJK UNIFIED IDEOGRAPH
	"ED 71	53B2",	#CJK UNIFIED IDEOGRAPH
	"ED 72	53DD",	#CJK UNIFIED IDEOGRAPH
	"ED 73	FA0E",	#CJK COMPATIBILITY IDEOGRAPH
	"ED 74	549C",	#CJK UNIFIED IDEOGRAPH
	"ED 75	548A",	#CJK UNIFIED IDEOGRAPH
	"ED 76	54A9",	#CJK UNIFIED IDEOGRAPH
	"ED 77	54FF",	#CJK UNIFIED IDEOGRAPH
	"ED 78	5586",	#CJK UNIFIED IDEOGRAPH
	"ED 79	5759",	#CJK UNIFIED IDEOGRAPH
	"ED 7A	5765",	#CJK UNIFIED IDEOGRAPH
	"ED 7B	57AC",	#CJK UNIFIED IDEOGRAPH
	"ED 7C	57C8",	#CJK UNIFIED IDEOGRAPH
	"ED 7D	57C7",	#CJK UNIFIED IDEOGRAPH
	"ED 7E	FA0F",	#CJK COMPATIBILITY IDEOGRAPH
	"ED 80	FA10",	#CJK COMPATIBILITY IDEOGRAPH
	"ED 81	589E",	#CJK UNIFIED IDEOGRAPH
	"ED 82	58B2",	#CJK UNIFIED IDEOGRAPH
	"ED 83	590B",	#CJK UNIFIED IDEOGRAPH
	"ED 84	5953",	#CJK UNIFIED IDEOGRAPH
	"ED 85	595B",	#CJK UNIFIED IDEOGRAPH
	"ED 86	595D",	#CJK UNIFIED IDEOGRAPH
	"ED 87	5963",	#CJK UNIFIED IDEOGRAPH
	"ED 88	59A4",	#CJK UNIFIED IDEOGRAPH
	"ED 89	59BA",	#CJK UNIFIED IDEOGRAPH
	"ED 8A	5B56",	#CJK UNIFIED IDEOGRAPH
	"ED 8B	5BC0",	#CJK UNIFIED IDEOGRAPH
	"ED 8C	752F",	#CJK UNIFIED IDEOGRAPH
	"ED 8D	5BD8",	#CJK UNIFIED IDEOGRAPH
	"ED 8E	5BEC",	#CJK UNIFIED IDEOGRAPH
	"ED 8F	5C1E",	#CJK UNIFIED IDEOGRAPH
	"ED 90	5CA6",	#CJK UNIFIED IDEOGRAPH
	"ED 91	5CBA",	#CJK UNIFIED IDEOGRAPH
	"ED 92	5CF5",	#CJK UNIFIED IDEOGRAPH
	"ED 93	5D27",	#CJK UNIFIED IDEOGRAPH
	"ED 94	5D53",	#CJK UNIFIED IDEOGRAPH
	"ED 95	FA11",	#CJK COMPATIBILITY IDEOGRAPH
	"ED 96	5D42",	#CJK UNIFIED IDEOGRAPH
	"ED 97	5D6D",	#CJK UNIFIED IDEOGRAPH
	"ED 98	5DB8",	#CJK UNIFIED IDEOGRAPH
	"ED 99	5DB9",	#CJK UNIFIED IDEOGRAPH
	"ED 9A	5DD0",	#CJK UNIFIED IDEOGRAPH
	"ED 9B	5F21",	#CJK UNIFIED IDEOGRAPH
	"ED 9C	5F34",	#CJK UNIFIED IDEOGRAPH
	"ED 9D	5F67",	#CJK UNIFIED IDEOGRAPH
	"ED 9E	5FB7",	#CJK UNIFIED IDEOGRAPH
	"ED 9F	5FDE",	#CJK UNIFIED IDEOGRAPH
	"ED A0	605D",	#CJK UNIFIED IDEOGRAPH
	"ED A1	6085",	#CJK UNIFIED IDEOGRAPH
	"ED A2	608A",	#CJK UNIFIED IDEOGRAPH
	"ED A3	60DE",	#CJK UNIFIED IDEOGRAPH
	"ED A4	60D5",	#CJK UNIFIED IDEOGRAPH
	"ED A5	6120",	#CJK UNIFIED IDEOGRAPH
	"ED A6	60F2",	#CJK UNIFIED IDEOGRAPH
	"ED A7	6111",	#CJK UNIFIED IDEOGRAPH
	"ED A8	6137",	#CJK UNIFIED IDEOGRAPH
	"ED A9	6130",	#CJK UNIFIED IDEOGRAPH
	"ED AA	6198",	#CJK UNIFIED IDEOGRAPH
	"ED AB	6213",	#CJK UNIFIED IDEOGRAPH
	"ED AC	62A6",	#CJK UNIFIED IDEOGRAPH
	"ED AD	63F5",	#CJK UNIFIED IDEOGRAPH
	"ED AE	6460",	#CJK UNIFIED IDEOGRAPH
	"ED AF	649D",	#CJK UNIFIED IDEOGRAPH
	"ED B0	64CE",	#CJK UNIFIED IDEOGRAPH
	"ED B1	654E",	#CJK UNIFIED IDEOGRAPH
	"ED B2	6600",	#CJK UNIFIED IDEOGRAPH
	"ED B3	6615",	#CJK UNIFIED IDEOGRAPH
	"ED B4	663B",	#CJK UNIFIED IDEOGRAPH
	"ED B5	6609",	#CJK UNIFIED IDEOGRAPH
	"ED B6	662E",	#CJK UNIFIED IDEOGRAPH
	"ED B7	661E",	#CJK UNIFIED IDEOGRAPH
	"ED B8	6624",	#CJK UNIFIED IDEOGRAPH
	"ED B9	6665",	#CJK UNIFIED IDEOGRAPH
	"ED BA	6657",	#CJK UNIFIED IDEOGRAPH
	"ED BB	6659",	#CJK UNIFIED IDEOGRAPH
	"ED BC	FA12",	#CJK COMPATIBILITY IDEOGRAPH
	"ED BD	6673",	#CJK UNIFIED IDEOGRAPH
	"ED BE	6699",	#CJK UNIFIED IDEOGRAPH
	"ED BF	66A0",	#CJK UNIFIED IDEOGRAPH
	"ED C0	66B2",	#CJK UNIFIED IDEOGRAPH
	"ED C1	66BF",	#CJK UNIFIED IDEOGRAPH
	"ED C2	66FA",	#CJK UNIFIED IDEOGRAPH
	"ED C3	670E",	#CJK UNIFIED IDEOGRAPH
	"ED C4	F929",	#CJK COMPATIBILITY IDEOGRAPH
	"ED C5	6766",	#CJK UNIFIED IDEOGRAPH
	"ED C6	67BB",	#CJK UNIFIED IDEOGRAPH
	"ED C7	6852",	#CJK UNIFIED IDEOGRAPH
	"ED C8	67C0",	#CJK UNIFIED IDEOGRAPH
	"ED C9	6801",	#CJK UNIFIED IDEOGRAPH
	"ED CA	6844",	#CJK UNIFIED IDEOGRAPH
	"ED CB	68CF",	#CJK UNIFIED IDEOGRAPH
	"ED CC	FA13",	#CJK COMPATIBILITY IDEOGRAPH
	"ED CD	6968",	#CJK UNIFIED IDEOGRAPH
	"ED CE	FA14",	#CJK COMPATIBILITY IDEOGRAPH
	"ED CF	6998",	#CJK UNIFIED IDEOGRAPH
	"ED D0	69E2",	#CJK UNIFIED IDEOGRAPH
	"ED D1	6A30",	#CJK UNIFIED IDEOGRAPH
	"ED D2	6A6B",	#CJK UNIFIED IDEOGRAPH
	"ED D3	6A46",	#CJK UNIFIED IDEOGRAPH
	"ED D4	6A73",	#CJK UNIFIED IDEOGRAPH
	"ED D5	6A7E",	#CJK UNIFIED IDEOGRAPH
	"ED D6	6AE2",	#CJK UNIFIED IDEOGRAPH
	"ED D7	6AE4",	#CJK UNIFIED IDEOGRAPH
	"ED D8	6BD6",	#CJK UNIFIED IDEOGRAPH
	"ED D9	6C3F",	#CJK UNIFIED IDEOGRAPH
	"ED DA	6C5C",	#CJK UNIFIED IDEOGRAPH
	"ED DB	6C86",	#CJK UNIFIED IDEOGRAPH
	"ED DC	6C6F",	#CJK UNIFIED IDEOGRAPH
	"ED DD	6CDA",	#CJK UNIFIED IDEOGRAPH
	"ED DE	6D04",	#CJK UNIFIED IDEOGRAPH
	"ED DF	6D87",	#CJK UNIFIED IDEOGRAPH
	"ED E0	6D6F",	#CJK UNIFIED IDEOGRAPH
	"ED E1	6D96",	#CJK UNIFIED IDEOGRAPH
	"ED E2	6DAC",	#CJK UNIFIED IDEOGRAPH
	"ED E3	6DCF",	#CJK UNIFIED IDEOGRAPH
	"ED E4	6DF8",	#CJK UNIFIED IDEOGRAPH
	"ED E5	6DF2",	#CJK UNIFIED IDEOGRAPH
	"ED E6	6DFC",	#CJK UNIFIED IDEOGRAPH
	"ED E7	6E39",	#CJK UNIFIED IDEOGRAPH
	"ED E8	6E5C",	#CJK UNIFIED IDEOGRAPH
	"ED E9	6E27",	#CJK UNIFIED IDEOGRAPH
	"ED EA	6E3C",	#CJK UNIFIED IDEOGRAPH
	"ED EB	6EBF",	#CJK UNIFIED IDEOGRAPH
	"ED EC	6F88",	#CJK UNIFIED IDEOGRAPH
	"ED ED	6FB5",	#CJK UNIFIED IDEOGRAPH
	"ED EE	6FF5",	#CJK UNIFIED IDEOGRAPH
	"ED EF	7005",	#CJK UNIFIED IDEOGRAPH
	"ED F0	7007",	#CJK UNIFIED IDEOGRAPH
	"ED F1	7028",	#CJK UNIFIED IDEOGRAPH
	"ED F2	7085",	#CJK UNIFIED IDEOGRAPH
	"ED F3	70AB",	#CJK UNIFIED IDEOGRAPH
	"ED F4	710F",	#CJK UNIFIED IDEOGRAPH
	"ED F5	7104",	#CJK UNIFIED IDEOGRAPH
	"ED F6	715C",	#CJK UNIFIED IDEOGRAPH
	"ED F7	7146",	#CJK UNIFIED IDEOGRAPH
	"ED F8	7147",	#CJK UNIFIED IDEOGRAPH
	"ED F9	FA15",	#CJK COMPATIBILITY IDEOGRAPH
	"ED FA	71C1",	#CJK UNIFIED IDEOGRAPH
	"ED FB	71FE",	#CJK UNIFIED IDEOGRAPH
	"ED FC	72B1",	#CJK UNIFIED IDEOGRAPH
	"EE 40	72BE",	#CJK UNIFIED IDEOGRAPH
	"EE 41	7324",	#CJK UNIFIED IDEOGRAPH
	"EE 42	FA16",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 43	7377",	#CJK UNIFIED IDEOGRAPH
	"EE 44	73BD",	#CJK UNIFIED IDEOGRAPH
	"EE 45	73C9",	#CJK UNIFIED IDEOGRAPH
	"EE 46	73D6",	#CJK UNIFIED IDEOGRAPH
	"EE 47	73E3",	#CJK UNIFIED IDEOGRAPH
	"EE 48	73D2",	#CJK UNIFIED IDEOGRAPH
	"EE 49	7407",	#CJK UNIFIED IDEOGRAPH
	"EE 4A	73F5",	#CJK UNIFIED IDEOGRAPH
	"EE 4B	7426",	#CJK UNIFIED IDEOGRAPH
	"EE 4C	742A",	#CJK UNIFIED IDEOGRAPH
	"EE 4D	7429",	#CJK UNIFIED IDEOGRAPH
	"EE 4E	742E",	#CJK UNIFIED IDEOGRAPH
	"EE 4F	7462",	#CJK UNIFIED IDEOGRAPH
	"EE 50	7489",	#CJK UNIFIED IDEOGRAPH
	"EE 51	749F",	#CJK UNIFIED IDEOGRAPH
	"EE 52	7501",	#CJK UNIFIED IDEOGRAPH
	"EE 53	756F",	#CJK UNIFIED IDEOGRAPH
	"EE 54	7682",	#CJK UNIFIED IDEOGRAPH
	"EE 55	769C",	#CJK UNIFIED IDEOGRAPH
	"EE 56	769E",	#CJK UNIFIED IDEOGRAPH
	"EE 57	769B",	#CJK UNIFIED IDEOGRAPH
	"EE 58	76A6",	#CJK UNIFIED IDEOGRAPH
	"EE 59	FA17",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 5A	7746",	#CJK UNIFIED IDEOGRAPH
	"EE 5B	52AF",	#CJK UNIFIED IDEOGRAPH
	"EE 5C	7821",	#CJK UNIFIED IDEOGRAPH
	"EE 5D	784E",	#CJK UNIFIED IDEOGRAPH
	"EE 5E	7864",	#CJK UNIFIED IDEOGRAPH
	"EE 5F	787A",	#CJK UNIFIED IDEOGRAPH
	"EE 60	7930",	#CJK UNIFIED IDEOGRAPH
	"EE 61	FA18",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 62	FA19",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 63	FA1A",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 64	7994",	#CJK UNIFIED IDEOGRAPH
	"EE 65	FA1B",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 66	799B",	#CJK UNIFIED IDEOGRAPH
	"EE 67	7AD1",	#CJK UNIFIED IDEOGRAPH
	"EE 68	7AE7",	#CJK UNIFIED IDEOGRAPH
	"EE 69	FA1C",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 6A	7AEB",	#CJK UNIFIED IDEOGRAPH
	"EE 6B	7B9E",	#CJK UNIFIED IDEOGRAPH
	"EE 6C	FA1D",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 6D	7D48",	#CJK UNIFIED IDEOGRAPH
	"EE 6E	7D5C",	#CJK UNIFIED IDEOGRAPH
	"EE 6F	7DB7",	#CJK UNIFIED IDEOGRAPH
	"EE 70	7DA0",	#CJK UNIFIED IDEOGRAPH
	"EE 71	7DD6",	#CJK UNIFIED IDEOGRAPH
	"EE 72	7E52",	#CJK UNIFIED IDEOGRAPH
	"EE 73	7F47",	#CJK UNIFIED IDEOGRAPH
	"EE 74	7FA1",	#CJK UNIFIED IDEOGRAPH
	"EE 75	FA1E",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 76	8301",	#CJK UNIFIED IDEOGRAPH
	"EE 77	8362",	#CJK UNIFIED IDEOGRAPH
	"EE 78	837F",	#CJK UNIFIED IDEOGRAPH
	"EE 79	83C7",	#CJK UNIFIED IDEOGRAPH
	"EE 7A	83F6",	#CJK UNIFIED IDEOGRAPH
	"EE 7B	8448",	#CJK UNIFIED IDEOGRAPH
	"EE 7C	84B4",	#CJK UNIFIED IDEOGRAPH
	"EE 7D	8553",	#CJK UNIFIED IDEOGRAPH
	"EE 7E	8559",	#CJK UNIFIED IDEOGRAPH
	"EE 80	856B",	#CJK UNIFIED IDEOGRAPH
	"EE 81	FA1F",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 82	85B0",	#CJK UNIFIED IDEOGRAPH
	"EE 83	FA20",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 84	FA21",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 85	8807",	#CJK UNIFIED IDEOGRAPH
	"EE 86	88F5",	#CJK UNIFIED IDEOGRAPH
	"EE 87	8A12",	#CJK UNIFIED IDEOGRAPH
	"EE 88	8A37",	#CJK UNIFIED IDEOGRAPH
	"EE 89	8A79",	#CJK UNIFIED IDEOGRAPH
	"EE 8A	8AA7",	#CJK UNIFIED IDEOGRAPH
	"EE 8B	8ABE",	#CJK UNIFIED IDEOGRAPH
	"EE 8C	8ADF",	#CJK UNIFIED IDEOGRAPH
	"EE 8D	FA22",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 8E	8AF6",	#CJK UNIFIED IDEOGRAPH
	"EE 8F	8B53",	#CJK UNIFIED IDEOGRAPH
	"EE 90	8B7F",	#CJK UNIFIED IDEOGRAPH
	"EE 91	8CF0",	#CJK UNIFIED IDEOGRAPH
	"EE 92	8CF4",	#CJK UNIFIED IDEOGRAPH
	"EE 93	8D12",	#CJK UNIFIED IDEOGRAPH
	"EE 94	8D76",	#CJK UNIFIED IDEOGRAPH
	"EE 95	FA23",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 96	8ECF",	#CJK UNIFIED IDEOGRAPH
	"EE 97	FA24",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 98	FA25",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 99	9067",	#CJK UNIFIED IDEOGRAPH
	"EE 9A	90DE",	#CJK UNIFIED IDEOGRAPH
	"EE 9B	FA26",	#CJK COMPATIBILITY IDEOGRAPH
	"EE 9C	9115",	#CJK UNIFIED IDEOGRAPH
	"EE 9D	9127",	#CJK UNIFIED IDEOGRAPH
	"EE 9E	91DA",	#CJK UNIFIED IDEOGRAPH
	"EE 9F	91D7",	#CJK UNIFIED IDEOGRAPH
	"EE A0	91DE",	#CJK UNIFIED IDEOGRAPH
	"EE A1	91ED",	#CJK UNIFIED IDEOGRAPH
	"EE A2	91EE",	#CJK UNIFIED IDEOGRAPH
	"EE A3	91E4",	#CJK UNIFIED IDEOGRAPH
	"EE A4	91E5",	#CJK UNIFIED IDEOGRAPH
	"EE A5	9206",	#CJK UNIFIED IDEOGRAPH
	"EE A6	9210",	#CJK UNIFIED IDEOGRAPH
	"EE A7	920A",	#CJK UNIFIED IDEOGRAPH
	"EE A8	923A",	#CJK UNIFIED IDEOGRAPH
	"EE A9	9240",	#CJK UNIFIED IDEOGRAPH
	"EE AA	923C",	#CJK UNIFIED IDEOGRAPH
	"EE AB	924E",	#CJK UNIFIED IDEOGRAPH
	"EE AC	9259",	#CJK UNIFIED IDEOGRAPH
	"EE AD	9251",	#CJK UNIFIED IDEOGRAPH
	"EE AE	9239",	#CJK UNIFIED IDEOGRAPH
	"EE AF	9267",	#CJK UNIFIED IDEOGRAPH
	"EE B0	92A7",	#CJK UNIFIED IDEOGRAPH
	"EE B1	9277",	#CJK UNIFIED IDEOGRAPH
	"EE B2	9278",	#CJK UNIFIED IDEOGRAPH
	"EE B3	92E7",	#CJK UNIFIED IDEOGRAPH
	"EE B4	92D7",	#CJK UNIFIED IDEOGRAPH
	"EE B5	92D9",	#CJK UNIFIED IDEOGRAPH
	"EE B6	92D0",	#CJK UNIFIED IDEOGRAPH
	"EE B7	FA27",	#CJK COMPATIBILITY IDEOGRAPH
	"EE B8	92D5",	#CJK UNIFIED IDEOGRAPH
	"EE B9	92E0",	#CJK UNIFIED IDEOGRAPH
	"EE BA	92D3",	#CJK UNIFIED IDEOGRAPH
	"EE BB	9325",	#CJK UNIFIED IDEOGRAPH
	"EE BC	9321",	#CJK UNIFIED IDEOGRAPH
	"EE BD	92FB",	#CJK UNIFIED IDEOGRAPH
	"EE BE	FA28",	#CJK COMPATIBILITY IDEOGRAPH
	"EE BF	931E",	#CJK UNIFIED IDEOGRAPH
	"EE C0	92FF",	#CJK UNIFIED IDEOGRAPH
	"EE C1	931D",	#CJK UNIFIED IDEOGRAPH
	"EE C2	9302",	#CJK UNIFIED IDEOGRAPH
	"EE C3	9370",	#CJK UNIFIED IDEOGRAPH
	"EE C4	9357",	#CJK UNIFIED IDEOGRAPH
	"EE C5	93A4",	#CJK UNIFIED IDEOGRAPH
	"EE C6	93C6",	#CJK UNIFIED IDEOGRAPH
	"EE C7	93DE",	#CJK UNIFIED IDEOGRAPH
	"EE C8	93F8",	#CJK UNIFIED IDEOGRAPH
	"EE C9	9431",	#CJK UNIFIED IDEOGRAPH
	"EE CA	9445",	#CJK UNIFIED IDEOGRAPH
	"EE CB	9448",	#CJK UNIFIED IDEOGRAPH
	"EE CC	9592",	#CJK UNIFIED IDEOGRAPH
	"EE CD	F9DC",	#CJK COMPATIBILITY IDEOGRAPH
	"EE CE	FA29",	#CJK COMPATIBILITY IDEOGRAPH
	"EE CF	969D",	#CJK UNIFIED IDEOGRAPH
	"EE D0	96AF",	#CJK UNIFIED IDEOGRAPH
	"EE D1	9733",	#CJK UNIFIED IDEOGRAPH
	"EE D2	973B",	#CJK UNIFIED IDEOGRAPH
	"EE D3	9743",	#CJK UNIFIED IDEOGRAPH
	"EE D4	974D",	#CJK UNIFIED IDEOGRAPH
	"EE D5	974F",	#CJK UNIFIED IDEOGRAPH
	"EE D6	9751",	#CJK UNIFIED IDEOGRAPH
	"EE D7	9755",	#CJK UNIFIED IDEOGRAPH
	"EE D8	9857",	#CJK UNIFIED IDEOGRAPH
	"EE D9	9865",	#CJK UNIFIED IDEOGRAPH
	"EE DA	FA2A",	#CJK COMPATIBILITY IDEOGRAPH
	"EE DB	FA2B",	#CJK COMPATIBILITY IDEOGRAPH
	"EE DC	9927",	#CJK UNIFIED IDEOGRAPH
	"EE DD	FA2C",	#CJK COMPATIBILITY IDEOGRAPH
	"EE DE	999E",	#CJK UNIFIED IDEOGRAPH
	"EE DF	9A4E",	#CJK UNIFIED IDEOGRAPH
	"EE E0	9AD9",	#CJK UNIFIED IDEOGRAPH
	"EE E1	9ADC",	#CJK UNIFIED IDEOGRAPH
	"EE E2	9B75",	#CJK UNIFIED IDEOGRAPH
	"EE E3	9B72",	#CJK UNIFIED IDEOGRAPH
	"EE E4	9B8F",	#CJK UNIFIED IDEOGRAPH
	"EE E5	9BB1",	#CJK UNIFIED IDEOGRAPH
	"EE E6	9BBB",	#CJK UNIFIED IDEOGRAPH
	"EE E7	9C00",	#CJK UNIFIED IDEOGRAPH
	"EE E8	9D70",	#CJK UNIFIED IDEOGRAPH
	"EE E9	9D6B",	#CJK UNIFIED IDEOGRAPH
	"EE EA	FA2D",	#CJK COMPATIBILITY IDEOGRAPH
	"EE EB	9E19",	#CJK UNIFIED IDEOGRAPH
	"EE EC	9ED1",	#CJK UNIFIED IDEOGRAPH
	"EE EF	2170",	#SMALL ROMAN NUMERAL ONE
	"EE F0	2171",	#SMALL ROMAN NUMERAL TWO
	"EE F1	2172",	#SMALL ROMAN NUMERAL THREE
	"EE F2	2173",	#SMALL ROMAN NUMERAL FOUR
	"EE F3	2174",	#SMALL ROMAN NUMERAL FIVE
	"EE F4	2175",	#SMALL ROMAN NUMERAL SIX
	"EE F5	2176",	#SMALL ROMAN NUMERAL SEVEN
	"EE F6	2177",	#SMALL ROMAN NUMERAL EIGHT
	"EE F7	2178",	#SMALL ROMAN NUMERAL NINE
	"EE F8	2179",	#SMALL ROMAN NUMERAL TEN
	"EE F9	FFE2",	#FULLWIDTH NOT SIGN
	"EE FA	FFE4",	#FULLWIDTH BROKEN BAR
	"EE FB	FF07",	#FULLWIDTH APOSTROPHE
	"EE FC	FF02",	#FULLWIDTH QUOTATION MARK
	"FA 40	2170",	#SMALL ROMAN NUMERAL ONE
	"FA 41	2171",	#SMALL ROMAN NUMERAL TWO
	"FA 42	2172",	#SMALL ROMAN NUMERAL THREE
	"FA 43	2173",	#SMALL ROMAN NUMERAL FOUR
	"FA 44	2174",	#SMALL ROMAN NUMERAL FIVE
	"FA 45	2175",	#SMALL ROMAN NUMERAL SIX
	"FA 46	2176",	#SMALL ROMAN NUMERAL SEVEN
	"FA 47	2177",	#SMALL ROMAN NUMERAL EIGHT
	"FA 48	2178",	#SMALL ROMAN NUMERAL NINE
	"FA 49	2179",	#SMALL ROMAN NUMERAL TEN
	"FA 4A	2160",	#ROMAN NUMERAL ONE
	"FA 4B	2161",	#ROMAN NUMERAL TWO
	"FA 4C	2162",	#ROMAN NUMERAL THREE
	"FA 4D	2163",	#ROMAN NUMERAL FOUR
	"FA 4E	2164",	#ROMAN NUMERAL FIVE
	"FA 4F	2165",	#ROMAN NUMERAL SIX
	"FA 50	2166",	#ROMAN NUMERAL SEVEN
	"FA 51	2167",	#ROMAN NUMERAL EIGHT
	"FA 52	2168",	#ROMAN NUMERAL NINE
	"FA 53	2169",	#ROMAN NUMERAL TEN
	"FA 54	FFE2",	#FULLWIDTH NOT SIGN
	"FA 55	FFE4",	#FULLWIDTH BROKEN BAR
	"FA 56	FF07",	#FULLWIDTH APOSTROPHE
	"FA 57	FF02",	#FULLWIDTH QUOTATION MARK
	"FA 58	3231",	#PARENTHESIZED IDEOGRAPH STOCK
	"FA 59	2116",	#NUMERO SIGN
	"FA 5A	2121",	#TELEPHONE SIGN
	"FA 5B	2235",	#BECAUSE
	"FA 5C	7E8A",	#CJK UNIFIED IDEOGRAPH
	"FA 5D	891C",	#CJK UNIFIED IDEOGRAPH
	"FA 5E	9348",	#CJK UNIFIED IDEOGRAPH
	"FA 5F	9288",	#CJK UNIFIED IDEOGRAPH
	"FA 60	84DC",	#CJK UNIFIED IDEOGRAPH
	"FA 61	4FC9",	#CJK UNIFIED IDEOGRAPH
	"FA 62	70BB",	#CJK UNIFIED IDEOGRAPH
	"FA 63	6631",	#CJK UNIFIED IDEOGRAPH
	"FA 64	68C8",	#CJK UNIFIED IDEOGRAPH
	"FA 65	92F9",	#CJK UNIFIED IDEOGRAPH
	"FA 66	66FB",	#CJK UNIFIED IDEOGRAPH
	"FA 67	5F45",	#CJK UNIFIED IDEOGRAPH
	"FA 68	4E28",	#CJK UNIFIED IDEOGRAPH
	"FA 69	4EE1",	#CJK UNIFIED IDEOGRAPH
	"FA 6A	4EFC",	#CJK UNIFIED IDEOGRAPH
	"FA 6B	4F00",	#CJK UNIFIED IDEOGRAPH
	"FA 6C	4F03",	#CJK UNIFIED IDEOGRAPH
	"FA 6D	4F39",	#CJK UNIFIED IDEOGRAPH
	"FA 6E	4F56",	#CJK UNIFIED IDEOGRAPH
	"FA 6F	4F92",	#CJK UNIFIED IDEOGRAPH
	"FA 70	4F8A",	#CJK UNIFIED IDEOGRAPH
	"FA 71	4F9A",	#CJK UNIFIED IDEOGRAPH
	"FA 72	4F94",	#CJK UNIFIED IDEOGRAPH
	"FA 73	4FCD",	#CJK UNIFIED IDEOGRAPH
	"FA 74	5040",	#CJK UNIFIED IDEOGRAPH
	"FA 75	5022",	#CJK UNIFIED IDEOGRAPH
	"FA 76	4FFF",	#CJK UNIFIED IDEOGRAPH
	"FA 77	501E",	#CJK UNIFIED IDEOGRAPH
	"FA 78	5046",	#CJK UNIFIED IDEOGRAPH
	"FA 79	5070",	#CJK UNIFIED IDEOGRAPH
	"FA 7A	5042",	#CJK UNIFIED IDEOGRAPH
	"FA 7B	5094",	#CJK UNIFIED IDEOGRAPH
	"FA 7C	50F4",	#CJK UNIFIED IDEOGRAPH
	"FA 7D	50D8",	#CJK UNIFIED IDEOGRAPH
	"FA 7E	514A",	#CJK UNIFIED IDEOGRAPH
	"FA 80	5164",	#CJK UNIFIED IDEOGRAPH
	"FA 81	519D",	#CJK UNIFIED IDEOGRAPH
	"FA 82	51BE",	#CJK UNIFIED IDEOGRAPH
	"FA 83	51EC",	#CJK UNIFIED IDEOGRAPH
	"FA 84	5215",	#CJK UNIFIED IDEOGRAPH
	"FA 85	529C",	#CJK UNIFIED IDEOGRAPH
	"FA 86	52A6",	#CJK UNIFIED IDEOGRAPH
	"FA 87	52C0",	#CJK UNIFIED IDEOGRAPH
	"FA 88	52DB",	#CJK UNIFIED IDEOGRAPH
	"FA 89	5300",	#CJK UNIFIED IDEOGRAPH
	"FA 8A	5307",	#CJK UNIFIED IDEOGRAPH
	"FA 8B	5324",	#CJK UNIFIED IDEOGRAPH
	"FA 8C	5372",	#CJK UNIFIED IDEOGRAPH
	"FA 8D	5393",	#CJK UNIFIED IDEOGRAPH
	"FA 8E	53B2",	#CJK UNIFIED IDEOGRAPH
	"FA 8F	53DD",	#CJK UNIFIED IDEOGRAPH
	"FA 90	FA0E",	#CJK COMPATIBILITY IDEOGRAPH
	"FA 91	549C",	#CJK UNIFIED IDEOGRAPH
	"FA 92	548A",	#CJK UNIFIED IDEOGRAPH
	"FA 93	54A9",	#CJK UNIFIED IDEOGRAPH
	"FA 94	54FF",	#CJK UNIFIED IDEOGRAPH
	"FA 95	5586",	#CJK UNIFIED IDEOGRAPH
	"FA 96	5759",	#CJK UNIFIED IDEOGRAPH
	"FA 97	5765",	#CJK UNIFIED IDEOGRAPH
	"FA 98	57AC",	#CJK UNIFIED IDEOGRAPH
	"FA 99	57C8",	#CJK UNIFIED IDEOGRAPH
	"FA 9A	57C7",	#CJK UNIFIED IDEOGRAPH
	"FA 9B	FA0F",	#CJK COMPATIBILITY IDEOGRAPH
	"FA 9C	FA10",	#CJK COMPATIBILITY IDEOGRAPH
	"FA 9D	589E",	#CJK UNIFIED IDEOGRAPH
	"FA 9E	58B2",	#CJK UNIFIED IDEOGRAPH
	"FA 9F	590B",	#CJK UNIFIED IDEOGRAPH
	"FA A0	5953",	#CJK UNIFIED IDEOGRAPH
	"FA A1	595B",	#CJK UNIFIED IDEOGRAPH
	"FA A2	595D",	#CJK UNIFIED IDEOGRAPH
	"FA A3	5963",	#CJK UNIFIED IDEOGRAPH
	"FA A4	59A4",	#CJK UNIFIED IDEOGRAPH
	"FA A5	59BA",	#CJK UNIFIED IDEOGRAPH
	"FA A6	5B56",	#CJK UNIFIED IDEOGRAPH
	"FA A7	5BC0",	#CJK UNIFIED IDEOGRAPH
	"FA A8	752F",	#CJK UNIFIED IDEOGRAPH
	"FA A9	5BD8",	#CJK UNIFIED IDEOGRAPH
	"FA AA	5BEC",	#CJK UNIFIED IDEOGRAPH
	"FA AB	5C1E",	#CJK UNIFIED IDEOGRAPH
	"FA AC	5CA6",	#CJK UNIFIED IDEOGRAPH
	"FA AD	5CBA",	#CJK UNIFIED IDEOGRAPH
	"FA AE	5CF5",	#CJK UNIFIED IDEOGRAPH
	"FA AF	5D27",	#CJK UNIFIED IDEOGRAPH
	"FA B0	5D53",	#CJK UNIFIED IDEOGRAPH
	"FA B1	FA11",	#CJK COMPATIBILITY IDEOGRAPH
	"FA B2	5D42",	#CJK UNIFIED IDEOGRAPH
	"FA B3	5D6D",	#CJK UNIFIED IDEOGRAPH
	"FA B4	5DB8",	#CJK UNIFIED IDEOGRAPH
	"FA B5	5DB9",	#CJK UNIFIED IDEOGRAPH
	"FA B6	5DD0",	#CJK UNIFIED IDEOGRAPH
	"FA B7	5F21",	#CJK UNIFIED IDEOGRAPH
	"FA B8	5F34",	#CJK UNIFIED IDEOGRAPH
	"FA B9	5F67",	#CJK UNIFIED IDEOGRAPH
	"FA BA	5FB7",	#CJK UNIFIED IDEOGRAPH
	"FA BB	5FDE",	#CJK UNIFIED IDEOGRAPH
	"FA BC	605D",	#CJK UNIFIED IDEOGRAPH
	"FA BD	6085",	#CJK UNIFIED IDEOGRAPH
	"FA BE	608A",	#CJK UNIFIED IDEOGRAPH
	"FA BF	60DE",	#CJK UNIFIED IDEOGRAPH
	"FA C0	60D5",	#CJK UNIFIED IDEOGRAPH
	"FA C1	6120",	#CJK UNIFIED IDEOGRAPH
	"FA C2	60F2",	#CJK UNIFIED IDEOGRAPH
	"FA C3	6111",	#CJK UNIFIED IDEOGRAPH
	"FA C4	6137",	#CJK UNIFIED IDEOGRAPH
	"FA C5	6130",	#CJK UNIFIED IDEOGRAPH
	"FA C6	6198",	#CJK UNIFIED IDEOGRAPH
	"FA C7	6213",	#CJK UNIFIED IDEOGRAPH
	"FA C8	62A6",	#CJK UNIFIED IDEOGRAPH
	"FA C9	63F5",	#CJK UNIFIED IDEOGRAPH
	"FA CA	6460",	#CJK UNIFIED IDEOGRAPH
	"FA CB	649D",	#CJK UNIFIED IDEOGRAPH
	"FA CC	64CE",	#CJK UNIFIED IDEOGRAPH
	"FA CD	654E",	#CJK UNIFIED IDEOGRAPH
	"FA CE	6600",	#CJK UNIFIED IDEOGRAPH
	"FA CF	6615",	#CJK UNIFIED IDEOGRAPH
	"FA D0	663B",	#CJK UNIFIED IDEOGRAPH
	"FA D1	6609",	#CJK UNIFIED IDEOGRAPH
	"FA D2	662E",	#CJK UNIFIED IDEOGRAPH
	"FA D3	661E",	#CJK UNIFIED IDEOGRAPH
	"FA D4	6624",	#CJK UNIFIED IDEOGRAPH
	"FA D5	6665",	#CJK UNIFIED IDEOGRAPH
	"FA D6	6657",	#CJK UNIFIED IDEOGRAPH
	"FA D7	6659",	#CJK UNIFIED IDEOGRAPH
	"FA D8	FA12",	#CJK COMPATIBILITY IDEOGRAPH
	"FA D9	6673",	#CJK UNIFIED IDEOGRAPH
	"FA DA	6699",	#CJK UNIFIED IDEOGRAPH
	"FA DB	66A0",	#CJK UNIFIED IDEOGRAPH
	"FA DC	66B2",	#CJK UNIFIED IDEOGRAPH
	"FA DD	66BF",	#CJK UNIFIED IDEOGRAPH
	"FA DE	66FA",	#CJK UNIFIED IDEOGRAPH
	"FA DF	670E",	#CJK UNIFIED IDEOGRAPH
	"FA E0	F929",	#CJK COMPATIBILITY IDEOGRAPH
	"FA E1	6766",	#CJK UNIFIED IDEOGRAPH
	"FA E2	67BB",	#CJK UNIFIED IDEOGRAPH
	"FA E3	6852",	#CJK UNIFIED IDEOGRAPH
	"FA E4	67C0",	#CJK UNIFIED IDEOGRAPH
	"FA E5	6801",	#CJK UNIFIED IDEOGRAPH
	"FA E6	6844",	#CJK UNIFIED IDEOGRAPH
	"FA E7	68CF",	#CJK UNIFIED IDEOGRAPH
	"FA E8	FA13",	#CJK COMPATIBILITY IDEOGRAPH
	"FA E9	6968",	#CJK UNIFIED IDEOGRAPH
	"FA EA	FA14",	#CJK COMPATIBILITY IDEOGRAPH
	"FA EB	6998",	#CJK UNIFIED IDEOGRAPH
	"FA EC	69E2",	#CJK UNIFIED IDEOGRAPH
	"FA ED	6A30",	#CJK UNIFIED IDEOGRAPH
	"FA EE	6A6B",	#CJK UNIFIED IDEOGRAPH
	"FA EF	6A46",	#CJK UNIFIED IDEOGRAPH
	"FA F0	6A73",	#CJK UNIFIED IDEOGRAPH
	"FA F1	6A7E",	#CJK UNIFIED IDEOGRAPH
	"FA F2	6AE2",	#CJK UNIFIED IDEOGRAPH
	"FA F3	6AE4",	#CJK UNIFIED IDEOGRAPH
	"FA F4	6BD6",	#CJK UNIFIED IDEOGRAPH
	"FA F5	6C3F",	#CJK UNIFIED IDEOGRAPH
	"FA F6	6C5C",	#CJK UNIFIED IDEOGRAPH
	"FA F7	6C86",	#CJK UNIFIED IDEOGRAPH
	"FA F8	6C6F",	#CJK UNIFIED IDEOGRAPH
	"FA F9	6CDA",	#CJK UNIFIED IDEOGRAPH
	"FA FA	6D04",	#CJK UNIFIED IDEOGRAPH
	"FA FB	6D87",	#CJK UNIFIED IDEOGRAPH
	"FA FC	6D6F",	#CJK UNIFIED IDEOGRAPH
	"FB 40	6D96",	#CJK UNIFIED IDEOGRAPH
	"FB 41	6DAC",	#CJK UNIFIED IDEOGRAPH
	"FB 42	6DCF",	#CJK UNIFIED IDEOGRAPH
	"FB 43	6DF8",	#CJK UNIFIED IDEOGRAPH
	"FB 44	6DF2",	#CJK UNIFIED IDEOGRAPH
	"FB 45	6DFC",	#CJK UNIFIED IDEOGRAPH
	"FB 46	6E39",	#CJK UNIFIED IDEOGRAPH
	"FB 47	6E5C",	#CJK UNIFIED IDEOGRAPH
	"FB 48	6E27",	#CJK UNIFIED IDEOGRAPH
	"FB 49	6E3C",	#CJK UNIFIED IDEOGRAPH
	"FB 4A	6EBF",	#CJK UNIFIED IDEOGRAPH
	"FB 4B	6F88",	#CJK UNIFIED IDEOGRAPH
	"FB 4C	6FB5",	#CJK UNIFIED IDEOGRAPH
	"FB 4D	6FF5",	#CJK UNIFIED IDEOGRAPH
	"FB 4E	7005",	#CJK UNIFIED IDEOGRAPH
	"FB 4F	7007",	#CJK UNIFIED IDEOGRAPH
	"FB 50	7028",	#CJK UNIFIED IDEOGRAPH
	"FB 51	7085",	#CJK UNIFIED IDEOGRAPH
	"FB 52	70AB",	#CJK UNIFIED IDEOGRAPH
	"FB 53	710F",	#CJK UNIFIED IDEOGRAPH
	"FB 54	7104",	#CJK UNIFIED IDEOGRAPH
	"FB 55	715C",	#CJK UNIFIED IDEOGRAPH
	"FB 56	7146",	#CJK UNIFIED IDEOGRAPH
	"FB 57	7147",	#CJK UNIFIED IDEOGRAPH
	"FB 58	FA15",	#CJK COMPATIBILITY IDEOGRAPH
	"FB 59	71C1",	#CJK UNIFIED IDEOGRAPH
	"FB 5A	71FE",	#CJK UNIFIED IDEOGRAPH
	"FB 5B	72B1",	#CJK UNIFIED IDEOGRAPH
	"FB 5C	72BE",	#CJK UNIFIED IDEOGRAPH
	"FB 5D	7324",	#CJK UNIFIED IDEOGRAPH
	"FB 5E	FA16",	#CJK COMPATIBILITY IDEOGRAPH
	"FB 5F	7377",	#CJK UNIFIED IDEOGRAPH
	"FB 60	73BD",	#CJK UNIFIED IDEOGRAPH
	"FB 61	73C9",	#CJK UNIFIED IDEOGRAPH
	"FB 62	73D6",	#CJK UNIFIED IDEOGRAPH
	"FB 63	73E3",	#CJK UNIFIED IDEOGRAPH
	"FB 64	73D2",	#CJK UNIFIED IDEOGRAPH
	"FB 65	7407",	#CJK UNIFIED IDEOGRAPH
	"FB 66	73F5",	#CJK UNIFIED IDEOGRAPH
	"FB 67	7426",	#CJK UNIFIED IDEOGRAPH
	"FB 68	742A",	#CJK UNIFIED IDEOGRAPH
	"FB 69	7429",	#CJK UNIFIED IDEOGRAPH
	"FB 6A	742E",	#CJK UNIFIED IDEOGRAPH
	"FB 6B	7462",	#CJK UNIFIED IDEOGRAPH
	"FB 6C	7489",	#CJK UNIFIED IDEOGRAPH
	"FB 6D	749F",	#CJK UNIFIED IDEOGRAPH
	"FB 6E	7501",	#CJK UNIFIED IDEOGRAPH
	"FB 6F	756F",	#CJK UNIFIED IDEOGRAPH
	"FB 70	7682",	#CJK UNIFIED IDEOGRAPH
	"FB 71	769C",	#CJK UNIFIED IDEOGRAPH
	"FB 72	769E",	#CJK UNIFIED IDEOGRAPH
	"FB 73	769B",	#CJK UNIFIED IDEOGRAPH
	"FB 74	76A6",	#CJK UNIFIED IDEOGRAPH
	"FB 75	FA17",	#CJK COMPATIBILITY IDEOGRAPH
	"FB 76	7746",	#CJK UNIFIED IDEOGRAPH
	"FB 77	52AF",	#CJK UNIFIED IDEOGRAPH
	"FB 78	7821",	#CJK UNIFIED IDEOGRAPH
	"FB 79	784E",	#CJK UNIFIED IDEOGRAPH
	"FB 7A	7864",	#CJK UNIFIED IDEOGRAPH
	"FB 7B	787A",	#CJK UNIFIED IDEOGRAPH
	"FB 7C	7930",	#CJK UNIFIED IDEOGRAPH
	"FB 7D	FA18",	#CJK COMPATIBILITY IDEOGRAPH
	"FB 7E	FA19",	#CJK COMPATIBILITY IDEOGRAPH
	"FB 80	FA1A",	#CJK COMPATIBILITY IDEOGRAPH
	"FB 81	7994",	#CJK UNIFIED IDEOGRAPH
	"FB 82	FA1B",	#CJK COMPATIBILITY IDEOGRAPH
	"FB 83	799B",	#CJK UNIFIED IDEOGRAPH
	"FB 84	7AD1",	#CJK UNIFIED IDEOGRAPH
	"FB 85	7AE7",	#CJK UNIFIED IDEOGRAPH
	"FB 86	FA1C",	#CJK COMPATIBILITY IDEOGRAPH
	"FB 87	7AEB",	#CJK UNIFIED IDEOGRAPH
	"FB 88	7B9E",	#CJK UNIFIED IDEOGRAPH
	"FB 89	FA1D",	#CJK COMPATIBILITY IDEOGRAPH
	"FB 8A	7D48",	#CJK UNIFIED IDEOGRAPH
	"FB 8B	7D5C",	#CJK UNIFIED IDEOGRAPH
	"FB 8C	7DB7",	#CJK UNIFIED IDEOGRAPH
	"FB 8D	7DA0",	#CJK UNIFIED IDEOGRAPH
	"FB 8E	7DD6",	#CJK UNIFIED IDEOGRAPH
	"FB 8F	7E52",	#CJK UNIFIED IDEOGRAPH
	"FB 90	7F47",	#CJK UNIFIED IDEOGRAPH
	"FB 91	7FA1",	#CJK UNIFIED IDEOGRAPH
	"FB 92	FA1E",	#CJK COMPATIBILITY IDEOGRAPH
	"FB 93	8301",	#CJK UNIFIED IDEOGRAPH
	"FB 94	8362",	#CJK UNIFIED IDEOGRAPH
	"FB 95	837F",	#CJK UNIFIED IDEOGRAPH
	"FB 96	83C7",	#CJK UNIFIED IDEOGRAPH
	"FB 97	83F6",	#CJK UNIFIED IDEOGRAPH
	"FB 98	8448",	#CJK UNIFIED IDEOGRAPH
	"FB 99	84B4",	#CJK UNIFIED IDEOGRAPH
	"FB 9A	8553",	#CJK UNIFIED IDEOGRAPH
	"FB 9B	8559",	#CJK UNIFIED IDEOGRAPH
	"FB 9C	856B",	#CJK UNIFIED IDEOGRAPH
	"FB 9D	FA1F",	#CJK COMPATIBILITY IDEOGRAPH
	"FB 9E	85B0",	#CJK UNIFIED IDEOGRAPH
	"FB 9F	FA20",	#CJK COMPATIBILITY IDEOGRAPH
	"FB A0	FA21",	#CJK COMPATIBILITY IDEOGRAPH
	"FB A1	8807",	#CJK UNIFIED IDEOGRAPH
	"FB A2	88F5",	#CJK UNIFIED IDEOGRAPH
	"FB A3	8A12",	#CJK UNIFIED IDEOGRAPH
	"FB A4	8A37",	#CJK UNIFIED IDEOGRAPH
	"FB A5	8A79",	#CJK UNIFIED IDEOGRAPH
	"FB A6	8AA7",	#CJK UNIFIED IDEOGRAPH
	"FB A7	8ABE",	#CJK UNIFIED IDEOGRAPH
	"FB A8	8ADF",	#CJK UNIFIED IDEOGRAPH
	"FB A9	FA22",	#CJK COMPATIBILITY IDEOGRAPH
	"FB AA	8AF6",	#CJK UNIFIED IDEOGRAPH
	"FB AB	8B53",	#CJK UNIFIED IDEOGRAPH
	"FB AC	8B7F",	#CJK UNIFIED IDEOGRAPH
	"FB AD	8CF0",	#CJK UNIFIED IDEOGRAPH
	"FB AE	8CF4",	#CJK UNIFIED IDEOGRAPH
	"FB AF	8D12",	#CJK UNIFIED IDEOGRAPH
	"FB B0	8D76",	#CJK UNIFIED IDEOGRAPH
	"FB B1	FA23",	#CJK COMPATIBILITY IDEOGRAPH
	"FB B2	8ECF",	#CJK UNIFIED IDEOGRAPH
	"FB B3	FA24",	#CJK COMPATIBILITY IDEOGRAPH
	"FB B4	FA25",	#CJK COMPATIBILITY IDEOGRAPH
	"FB B5	9067",	#CJK UNIFIED IDEOGRAPH
	"FB B6	90DE",	#CJK UNIFIED IDEOGRAPH
	"FB B7	FA26",	#CJK COMPATIBILITY IDEOGRAPH
	"FB B8	9115",	#CJK UNIFIED IDEOGRAPH
	"FB B9	9127",	#CJK UNIFIED IDEOGRAPH
	"FB BA	91DA",	#CJK UNIFIED IDEOGRAPH
	"FB BB	91D7",	#CJK UNIFIED IDEOGRAPH
	"FB BC	91DE",	#CJK UNIFIED IDEOGRAPH
	"FB BD	91ED",	#CJK UNIFIED IDEOGRAPH
	"FB BE	91EE",	#CJK UNIFIED IDEOGRAPH
	"FB BF	91E4",	#CJK UNIFIED IDEOGRAPH
	"FB C0	91E5",	#CJK UNIFIED IDEOGRAPH
	"FB C1	9206",	#CJK UNIFIED IDEOGRAPH
	"FB C2	9210",	#CJK UNIFIED IDEOGRAPH
	"FB C3	920A",	#CJK UNIFIED IDEOGRAPH
	"FB C4	923A",	#CJK UNIFIED IDEOGRAPH
	"FB C5	9240",	#CJK UNIFIED IDEOGRAPH
	"FB C6	923C",	#CJK UNIFIED IDEOGRAPH
	"FB C7	924E",	#CJK UNIFIED IDEOGRAPH
	"FB C8	9259",	#CJK UNIFIED IDEOGRAPH
	"FB C9	9251",	#CJK UNIFIED IDEOGRAPH
	"FB CA	9239",	#CJK UNIFIED IDEOGRAPH
	"FB CB	9267",	#CJK UNIFIED IDEOGRAPH
	"FB CC	92A7",	#CJK UNIFIED IDEOGRAPH
	"FB CD	9277",	#CJK UNIFIED IDEOGRAPH
	"FB CE	9278",	#CJK UNIFIED IDEOGRAPH
	"FB CF	92E7",	#CJK UNIFIED IDEOGRAPH
	"FB D0	92D7",	#CJK UNIFIED IDEOGRAPH
	"FB D1	92D9",	#CJK UNIFIED IDEOGRAPH
	"FB D2	92D0",	#CJK UNIFIED IDEOGRAPH
	"FB D3	FA27",	#CJK COMPATIBILITY IDEOGRAPH
	"FB D4	92D5",	#CJK UNIFIED IDEOGRAPH
	"FB D5	92E0",	#CJK UNIFIED IDEOGRAPH
	"FB D6	92D3",	#CJK UNIFIED IDEOGRAPH
	"FB D7	9325",	#CJK UNIFIED IDEOGRAPH
	"FB D8	9321",	#CJK UNIFIED IDEOGRAPH
	"FB D9	92FB",	#CJK UNIFIED IDEOGRAPH
	"FB DA	FA28",	#CJK COMPATIBILITY IDEOGRAPH
	"FB DB	931E",	#CJK UNIFIED IDEOGRAPH
	"FB DC	92FF",	#CJK UNIFIED IDEOGRAPH
	"FB DD	931D",	#CJK UNIFIED IDEOGRAPH
	"FB DE	9302",	#CJK UNIFIED IDEOGRAPH
	"FB DF	9370",	#CJK UNIFIED IDEOGRAPH
	"FB E0	9357",	#CJK UNIFIED IDEOGRAPH
	"FB E1	93A4",	#CJK UNIFIED IDEOGRAPH
	"FB E2	93C6",	#CJK UNIFIED IDEOGRAPH
	"FB E3	93DE",	#CJK UNIFIED IDEOGRAPH
	"FB E4	93F8",	#CJK UNIFIED IDEOGRAPH
	"FB E5	9431",	#CJK UNIFIED IDEOGRAPH
	"FB E6	9445",	#CJK UNIFIED IDEOGRAPH
	"FB E7	9448",	#CJK UNIFIED IDEOGRAPH
	"FB E8	9592",	#CJK UNIFIED IDEOGRAPH
	"FB E9	F9DC",	#CJK COMPATIBILITY IDEOGRAPH
	"FB EA	FA29",	#CJK COMPATIBILITY IDEOGRAPH
	"FB EB	969D",	#CJK UNIFIED IDEOGRAPH
	"FB EC	96AF",	#CJK UNIFIED IDEOGRAPH
	"FB ED	9733",	#CJK UNIFIED IDEOGRAPH
	"FB EE	973B",	#CJK UNIFIED IDEOGRAPH
	"FB EF	9743",	#CJK UNIFIED IDEOGRAPH
	"FB F0	974D",	#CJK UNIFIED IDEOGRAPH
	"FB F1	974F",	#CJK UNIFIED IDEOGRAPH
	"FB F2	9751",	#CJK UNIFIED IDEOGRAPH
	"FB F3	9755",	#CJK UNIFIED IDEOGRAPH
	"FB F4	9857",	#CJK UNIFIED IDEOGRAPH
	"FB F5	9865",	#CJK UNIFIED IDEOGRAPH
	"FB F6	FA2A",	#CJK COMPATIBILITY IDEOGRAPH
	"FB F7	FA2B",	#CJK COMPATIBILITY IDEOGRAPH
	"FB F8	9927",	#CJK UNIFIED IDEOGRAPH
	"FB F9	FA2C",	#CJK COMPATIBILITY IDEOGRAPH
	"FB FA	999E",	#CJK UNIFIED IDEOGRAPH
	"FB FB	9A4E",	#CJK UNIFIED IDEOGRAPH
	"FB FC	9AD9",	#CJK UNIFIED IDEOGRAPH
	"FC 40	9ADC",	#CJK UNIFIED IDEOGRAPH
	"FC 41	9B75",	#CJK UNIFIED IDEOGRAPH
	"FC 42	9B72",	#CJK UNIFIED IDEOGRAPH
	"FC 43	9B8F",	#CJK UNIFIED IDEOGRAPH
	"FC 44	9BB1",	#CJK UNIFIED IDEOGRAPH
	"FC 45	9BBB",	#CJK UNIFIED IDEOGRAPH
	"FC 46	9C00",	#CJK UNIFIED IDEOGRAPH
	"FC 47	9D70",	#CJK UNIFIED IDEOGRAPH
	"FC 48	9D6B",	#CJK UNIFIED IDEOGRAPH
	"FC 49	FA2D",	#CJK COMPATIBILITY IDEOGRAPH
	"FC 4A	9E19",	#CJK UNIFIED IDEOGRAPH
	"FC 4B	9ED1",	#CJK UNIFIED IDEOGRAPH
};
