// Hyperbolic Rogue
// Copyright (C) 2011-2012 Zeno Rogue, see 'hyper.cpp' for details

// basic graphics:

SDL_Surface *s;
TTF_Font *font[256];

// x resolutions

#define NUMMODES 7

int xres[NUMMODES] = { 0, 640, 800, 1024, 1280, 1600, 640 };
int yres[NUMMODES] = { 0, 480, 600,  768, 1024, 1200, 480 };

struct videopar {
  int modeid;
  ld scale, eye, alpha, aspeed;
  bool full;
  bool goteyes;
  bool quick;
  
  int xres, yres;
  
  // paramaters calculated from the above
  int xcenter, ycenter;
  int radius;
  ld alphax, beta;
  
  int fsize;
  };

videopar vid;

cell *mouseover; ld modist;

int mousex, mousey, mousedist, mousedest;
hyperpoint mouseh;

int getcstat; ld getcshift;

int ZZ;

string help;

enum emtype {emNormal, emHelp, emVisual, emQuit} cmode;

int darken = 0;

int& qpixel(SDL_Surface *surf, int x, int y) {
  if(x<0 || y<0 || x >= surf->w || y >= surf->h) return ZZ;
  char *p = (char*) surf->pixels;
  p += y * surf->pitch;
  int *pi = (int*) (p);
  return pi[x];
  }

int textwidth(int siz, const string &str) {
  if(size(str) == 0) return 0;
  
  if(!font[siz])
    font[siz] = TTF_OpenFont("VeraBd.ttf", siz);
  int w, h;
  TTF_SizeText(font[siz], str.c_str(), &w, &h);
  // printf("width = %d [%d]\n", w, size(str));
  return w;
  }

int darkened(int c) {
  for(int i=0; i<darken; i++) c &= 0xFEFEFE, c >>= 1;
  return c;
  }

bool displaystr(int x, int y, int shift, int size, const char *str, int color, int align) {

  if(strlen(str) == 0) return false;

  if(size <= 0 || size > 255) {
    // printf("size = %d\n", size);
    return false;
    }
  SDL_Color col;
  col.r = (color >> 16) & 255;
  col.g = (color >> 8 ) & 255;
  col.b = (color >> 0 ) & 255;
  
  col.r >>= darken; col.g >>= darken; col.b >>= darken;

  if(!font[size])
    font[size] = TTF_OpenFont("VeraBd.ttf", size);

  SDL_Surface *txt = TTF_RenderText_Solid(font[size], str, col);
  
  if(txt == NULL) return false;

  SDL_Rect rect;

  rect.w = txt->w;
  rect.h = txt->h;

  rect.x = x - rect.w * align / 16;
  rect.y = y - rect.h/2;
  
  bool clicked = (mousex >= rect.x && mousey >= rect.y && mousex <= rect.x+rect.w && mousey <= rect.y+rect.h);
  
  if(shift) {
    SDL_Surface* txt2 = SDL_DisplayFormat(txt);
    SDL_LockSurface(txt2);
    int c0 = qpixel(txt2, 0, 0);
    for(int yy=0; yy<rect.h; yy++)
    for(int xx=0; xx<rect.w; xx++) if(qpixel(txt2, xx, yy) != c0)
      qpixel(s, rect.x+xx-shift, rect.y+yy) |= color & 0xFF0000,
      qpixel(s, rect.x+xx+shift, rect.y+yy) |= color & 0x00FFFF;
    SDL_UnlockSurface(txt2);
    SDL_FreeSurface(txt2);
    }
  else {
    SDL_UnlockSurface(s);
    SDL_BlitSurface(txt, NULL, s,&rect); 
    SDL_LockSurface(s);
    }
  SDL_FreeSurface(txt);
  
  return clicked;
  }
                  
bool displaystr(int x, int y, int shift, int size, const string &s, int color, int align) {
  return displaystr(x, y, shift, size, s.c_str(), color, align);
  }

bool displayfr(int x, int y, int b, int size, const string &s, int color, int align) {
  displaystr(x-b, y, 0, size, s, 0, align);
  displaystr(x+b, y, 0, size, s, 0, align);
  displaystr(x, y-b, 0, size, s, 0, align);
  displaystr(x, y+b, 0, size, s, 0, align);
  return displaystr(x, y, 0, size, s, color, align);
  }

bool displaychr(int x, int y, int shift, int size, char chr, int col) {

  char buf[2];
  buf[0] = chr; buf[1] = 0;
  return displaystr(x, y, shift, size, buf, col, 8);
  }

void displaynum(int x, int y, int shift, int size, int col, int val, string title) {
  char buf[64];
  sprintf(buf, "%d", val);
  displaystr(x-8, y, shift, size, buf, col, 16);
  displaystr(x, y, shift, size, title, col, 0);
  }

vector<pair<int, string> > msgs;

void addMessage(string s) {
  msgs.push_back(make_pair(SDL_GetTicks(), s));
  // printf("%s\n", s.c_str());
  }

void drawmessages() {
  int i = 0;
  int t = SDL_GetTicks();
  for(int j=0; j<size(msgs); j++) {
    int age = t - msgs[j].first;
    if(age < 256*8) {
      int x = vid.xres / 2;
      int y = vid.yres - vid.fsize * (size(msgs) - j);
      displayfr(x, y, 1, vid.fsize, msgs[j].second, 0x10101 * (255 - age/8), 8);
      msgs[i++] = msgs[j];
      }
    }
  msgs.resize(i);
  }

hyperpoint gethyper(ld x, ld y) {
  ld hx = (x - vid.xcenter) / vid.radius;
  ld hy = (y - vid.ycenter) / vid.radius;
  ld hr = hx*hx+hy*hy;
  
  if(hr > 1) return Hypc;
  
  // hz*hz-(hx/(hz+alpha))^2 - (hy/(hz+alpha))^2 =
  
  // hz*hz-hr*(hz+alpha)^2 == 1
  // hz*hz - hr*hr*hz*Hz
  
  ld A = 1-hr;
  ld B = 2*hr*vid.alphax;
  ld C = 1 + hr*vid.alphax*vid.alphax;
  
  // Az^2 - Bz = C
  B /= A; C /= A;
  
  // z^2 - Bz = C
  // z^2 - Bz + (B^2/4) = C + (B^2/4)
  // z = (B/2) + sqrt(C + B^2/4)
  
  ld hz = B / 2 + sqrt(C + B*B/4);
  hyperpoint H;
  H[0] = hx * (hz+vid.alphax);
  H[1] = hy * (hz+vid.alphax);
  H[2] = hz;
  
  return H;
  }

void getcoord(const hyperpoint& H, int& x, int& y, int &shift) {
  
  if(H[2] < 0.999) {
    printf("error: %s\n", display(H));
    // exit(1);
    }
  ld tz = vid.alphax+H[2];
  if(tz < 1e-3 && tz > -1e-3) tz = 1000;
  x = vid.xcenter + int(vid.radius * H[0] / tz);
  y = vid.ycenter + int(vid.radius * H[1] / tz);
  shift = vid.goteyes ? int(vid.eye * vid.radius * (1 - vid.beta / tz)) : 0;
  }

int dlit;

void drawline(const hyperpoint& H1, int x1, int y1, const hyperpoint& H2, int x2, int y2, int col) {
  dlit++; if(dlit>500) return;
  if((x1-x2)*(x1-x2)+(y1-y2)*(y1-y2) <= 2) return;
  
  hyperpoint H3 = mid(H1, H2);
  int x3, y3, s3; getcoord(H3, x3, y3, s3);
  qpixel(s, x3-s3, y3) |= col & 0xFF0000;
  qpixel(s, x3+s3, y3) |= col & 0x00FFFF;

  drawline(H1, x1, y1, H3, x3, y3, col);
  drawline(H3, x3, y3, H2, x2, y2, col);
  
  }

void drawline(const hyperpoint& H1, const hyperpoint& H2, int col) {
  // printf("line\n");
  dlit = 0;
  int x1, y1, s1; getcoord(H1, x1, y1, s1);
  int x2, y2, s2; getcoord(H2, x2, y2, s2);
  drawline(H1, x1, y1, H2, x2, y2, col);
  // printf("ok\n");
  }

// game-related graphics

transmatrix View; // current rotation, relative to viewctr
transmatrix cwtV; // player-relative view
heptspin viewctr; // heptagon and rotation where the view is centered at
bool playerfound; // has player been found in the last drawing?

#include "geometry.cpp"

ld spina(cell *c, int dir) {
  return 2 * M_PI * dir / c->type;
  }

int gradient(int c0, int c1, ld v0, ld v, ld v1) {
  int vv = int(256 * ((v-v0) / (v1-v0)));
  int c = 0;
  for(int a=0; a<3; a++) {
    int p0 = (c0 >> (8*a)) & 255;
    int p1 = (c1 >> (8*a)) & 255;
    int p = (p0*(256-vv) + p1*vv + 127) >> 8;
    c += p << (8*a);
    }
  return c;
  }

int flashat, lightat;

void drawFlash() { flashat = SDL_GetTicks(); }
void drawLightning() { lightat = SDL_GetTicks(); }

void drawcell(cell *c, transmatrix V, int spinv) {
  
  // draw a web-like map
  if(0) {
    if(c->type == 6) {
      for(int a=0; a<3; a++)
      drawline(V*Crad[a*7], V*Crad[a*7+21], 0xd0d0 >> darken);
      }
    else {
      for(int a=0; a<7; a++)
      drawline(V*C0, V*Crad[(21+a*6)%42], 0xd0d0 >> darken);
      }
    }
  
  // save the player's view center
  if(c == cwt.c) {
    playerfound = true;
    cwtV = V * spin(-cwt.spin * 2*M_PI/c->type) * spin(M_PI);
    }

  if(1) {
  
    if(intval(mouseh, V*C0) < modist) {
      modist = intval(mouseh, V*C0);
      mouseover = c;
      }

    int xc, yc, sc, xs, ys, ss;
    getcoord(V*C0, xc, yc, sc);
    getcoord(V*xpush(.5)*C0, xs, ys, ss);
    // int col = 0xFFFFFF - 0x20 * c->maxdist - 0x2000 * c->cpdist;

    if(c->mpdist > 8) return; // not yet generated

    char ch = winf[c->wall].glyph;
    int col = winf[c->wall].color;
    
    if(c->land == laCrossroads && c->wall == waNone) col = 0xFF0000;
    if(c->land == laDesert && c->wall == waNone) col = 0xEDC9AF;
    if(c->land == laJungle) col = 0x8000;
    if(c->land == laMirror && c->wall == waNone) col = 0x404040;

    if(c->land == laIce && isIcy(c)) {
      if(c->heat < 0)
        col = 0x4040FF;
      else if(c->heat < 0.2)
        col = gradient(0x8080FF, 0xFFFFFF, 0, c->heat, 0.2);
      // else if(c->heat < 0.4)
      //  col = gradient(0xFFFFFF, 0xFFFF00, 0.2, c->heat, 0.4);
      else if(c->heat < 0.6)
        col = gradient(0xFFFFFF, 0xFF0000, 0.2, c->heat, 0.6);
      else col = 0xFF0000;
      if(c->wall == waNone) 
        col = (col & 0xFEFEFE) >> 1;
      }
    
    if(c->item) 
      ch = iinf[c->item].glyph, col = iinf[c->item].color;

    if(c->monst)
      ch = minf[c->monst].glyph, col = minf[c->monst].color;
    
    if(c->wall == waBonfire && c->tmp == 0)
      col = 0x404040;
      
    if(c->wall == waBonfire && c->tmp > 0)
      col = gradient(0xFF0000, 0xFFFF00, -1, sin(SDL_GetTicks()/100.0), 1);
      
    if(c->wall == waThumper && c->tmp == 0)
      col = 0xEDC9AF;
      
    if(c->wall == waThumper && c->tmp > 0) {
      int ds = SDL_GetTicks();;
      for(int u=0; u<5; u++) {
        ld rad = hexf * (.3 * u + (ds%1000) * .0003);
        int col = gradient(0xFFFFFF, 0, 0, rad, 1.5 * hexf);
        for(int a=0; a<84; a++)
          drawline(V*ddi(a, rad)*C0, V*ddi(a+1, rad)*C0, col);
        }
      }
    
    if(c->cpdist == 0) { ch = '@'; col = 0xD0D0D0; }
    
    if(c->monst == moSlime) {
      col = winf[c->wall].color;
      col |= (col>>1);
      }
    
    if(c->ligon) {
      int tim = SDL_GetTicks() - lightat;
      if(tim > 1000) tim = 800;
      for(int t=0; t<7; t++) if(c->mov[t] && c->mov[t]->ligon) {
        int hdir = 42 - t * 84 / c->type;
        int col = gradient(iinf[itOrbLightning].color, 0, 0, tim, 1100);
        drawline(V*ddi(SDL_GetTicks(), hexf/2)*C0, V*ddi(hdir, crossf)*C0, col);
        }
      }

    /*
    int ct = c->type;
    
    if(ch == '.') {
      col = darkened(col);
      for(int t=0; t<ct; t++)
        drawline(V*ddi(t*84/ct, hexf/3)*C0, V*ddi((t+1)*84/ct, hexf/3)*C0, col);
      }
      
    else if(ch == '#') {
      col = darkened(col);
      for(int u=1; u<6; u++)
      for(int t=0; t<ct; t++)
        drawline(V*ddi(0 + t*84/ct, u*hexf/6)*C0, V*ddi(0 + (t+1)*84/ct, u*hexf/6)*C0, col);
      }
      
    else*/
    displaychr(xc, yc, sc, int(sqrt(squar(xc-xs)+squar(yc-ys))), ch, col);
    
    // process mouse
    
    for(int i=-1; i<cwt.c->type; i++) if(i == -1 ? cwt.c == c : cwt.c->mov[i % cwt.c->type] == c) {
      int ndist = (xc-mousex) * (xc-mousex) + (yc-mousey) * (yc-mousey);
      if(ndist < mousedist) mousedist = ndist, mousedest = i;
      }
    
    // drawline(V*C0, V*Crad[0], 0xC00000);
    if(c->bardir != NODIR) {
      drawline(V*C0, V*heptmove[c->bardir]*C0, 0x505050 >> darken);
      drawline(V*C0, V*hexmove[c->bardir]*C0, 0x505050 >> darken);
      }

    if(isWorm(c) && c->mondir != NODIR) {
      int hdir = 42 - c->mondir * 84 / c->type;
      for(int u=-1; u<=1; u++)
        drawline(V*ddi(hdir+21, u*crossf/5)*C0, V*ddi(hdir, crossf)*ddi(hdir+21, u*crossf/5)*C0, 0x606020 >> darken);
      }

    if(isIvy(c) && c->mondir != NODIR) {
      int hdir = 42 - c->mondir * 84 / c->type;
      for(int u=-1; u<=1; u++)
        drawline(V*ddi(hdir+21, u*crossf/5)*C0, V*ddi(hdir, crossf)*ddi(hdir+21, u*crossf/5)*C0, 0x606020 >> darken);
      }
    
    if(isMimic(c)) {
    
      if(mouseh[2] < .5) return;
    
      int hdir = 42 - c->mondir * 84 / c->type;

      /*
      int mi = c->monst == moMimic ? 1 : -1;

      static int alph = 0;
      alph += 1;
      
      drawline(V*C0, V*ddi(hdir, 0.5)*C0, 0xFFFFFF);
      drawline(cwtV*C0, cwtV*ddi(0, 0.5)*C0, 0xFFFFFF);
      drawline(V*C0, V*ddi(hdir+alph, 0.3)*C0, 0xFFFFFF);
      drawline(cwtV*C0, cwtV*ddi(mi*alph, 0.3)*C0, 0xFFFFFF);
      */
      
      transmatrix mirrortrans = Id;
      
      if(c->monst == moMirror) mirrortrans[1][1] = -1;

      hyperpoint M = mirrortrans * inverse(cwtV) * mouseh;

      // ld ang = spina(c, c->mondir);
      
      hyperpoint V2 = V * spin(M_PI*hdir/42) * M;
      
      int xc, yc, sc;
      getcoord(V2, xc, yc, sc);
      displaychr(xc, yc, sc, 10, 'x', 0xFF00);
      }

    if(c->cpdist == 0) {
      if(items[itOrbShield] > 1) {
        float ds = SDL_GetTicks() / 300.;
        int col = darkened(iinf[itOrbShield].color);
        for(int a=0; a<84*5; a++)
          drawline(V*ddi(a, hexf + sin(ds + M_PI*2*a/20)*.1)*C0, V*ddi((a+1), hexf + sin(ds + M_PI*2*(a+1)/20)*.1)*C0, col);
        }

      if(items[itOrbSpeed]) {
        ld ds = SDL_GetTicks() / 10.;
        int col = darkened(iinf[itOrbSpeed].color);
        for(int b=0; b<84; b+=14)
        for(int a=0; a<84; a++)
          drawline(V*ddi(ds+b+a, hexf*a/84)*C0, V*ddi(ds+b+(a+1), hexf*(a+1)/84)*C0, col);
        }

      int ct = c->type;
      
      if(items[itOrbLightning]) {
        ld ds = SDL_GetTicks() / 50.;
        int col = darkened(iinf[itOrbLightning].color);
        for(int a=0; a<ct; a++)
          drawline(V*ddi(ds+a*84/ct, 2*hexf)*C0, V*ddi(ds+(a+(ct-1)/2)*84/ct, 2*hexf)*C0, col);
        }

      if(items[itOrbFlash]) {
        float ds = SDL_GetTicks() / 300.;
        int col = darkened(iinf[itOrbFlash].color);
        for(int u=0; u<5; u++) {
          ld rad = hexf * (2.5 + .5 * sin(ds+u*.3));
          for(int a=0; a<84; a++)
            drawline(V*ddi(a, rad)*C0, V*ddi(a+1, rad)*C0, col);
          }
        }

      if(items[itOrbWinter]) {
        int hdir = 42 - cwt.spin * 84 / c->type;
        float ds = SDL_GetTicks() / 300.;
        int col = darkened(iinf[itOrbWinter].color);
        for(int u=0; u<20; u++) {
          ld rad = 6 * sin(ds+u * 2 * M_PI / 20);
          drawline(V*ddi(hdir+rad, hexf*.5)*C0, V*ddi(hdir+rad, hexf*3)*C0, col);
          }
        }
      
      if(flashat > 0) {
        int tim = SDL_GetTicks() - flashat;
        if(tim > 1000) flashat = 0;
        for(int u=0; u<=tim; u++) {
          if((u-tim)%50) continue;
          if(u < tim-150) continue;
          ld rad = u * 3 / 1000.;
          rad = rad * (5-rad) / 2;
          rad *= hexf;
          int col = iinf[itOrbFlash].color;
          if(u > 500) col = gradient(col, 0, 500, u, 1100);
          for(int a=0; a<84; a++)
            drawline(V*ddi(a, rad)*C0, V*ddi(a+1, rad)*C0, col);
          }
        }
      }
    }
  }

const char *helptext =
  "Welcome to HyperRogue II! (version 2.1)\n"
  "\n"
  "You have been trapped in a strange place. Collect as much treasure as possible "
  "before being caught by monsters. The more treasure you collect, the more "
  "monsters come to hunt you, as long as you are in the same land type.\n"
  "You can fight most monsters by moving into its location. "
  "The monster could also kill you by moving into your location, but the game "
  "automatically cancels all moves which result in that.\n\n"
  "Move with mouse, num pad, wersdzxc, or hjklyubn. Wait by pressing a or '.'. Spin the world with arrows, PageUp/Down, and Home/Space.\n\n"
  "See more on the website: http//roguetemple.com/z/hyper.php\n\n"
  "HyperRogue II by Zeno Rogue <zeno@attnam.com> "
  "released under GNU General Public License version 2 and thus "
  "comes with absolutely no warranty; see COPYING for details";

void describeMouseover() {
  cell *c = mouseover;
  string out;
  if(!c) { out = "Press F1 or right click for help"; help = helptext; }
  else if(cmode == emNormal) {
    out = linf[c->land].name;
    help = linf[c->land].help;
    
    // if(c->land == laIce) out = "Icy Lands (" + fts(60 * (c->heat - .4)) + " C)";
    if(c->land == laIce) out += " (" + fts(60 * (c->heat-.4)) + " C)";
  
    if(c->wall) { out += ", "; out += winf[c->wall].name; help = winf[c->wall].help;}
    
    if(isActiv(c)) {
      if(c->tmp < 0) out += " (touch to activate)";
      if(c->tmp == 0) out += " (expired)";
      if(c->tmp > 0) out += " [" + its(c->tmp) + " turns]";
      }
  
    if(c->monst) {out += ", "; out += minf[c->monst].name; help = minf[c->monst].help;}
  
    if(c->item) {out += ", "; out += iinf[c->item].name; help = iinf[c->item].help;}
  
    if(!c->cpdist) out += ", you";
    }
  else if(cmode == emVisual) {
    if(getcstat == 'p') {
      out = "0 = Klein model, 1 = Poincare model";
      if(vid.alpha < -0.5)
        out = "you are looking through it!";
      }
    if(getcstat == 'a' && vid.aspeed > -4.99)
      out = "+5 = center instantly, -5 = do not center the map";
    else if(getcstat == 'a')
      out = "press Space or Home to center on the PC";
    else if(getcstat == 'e')
      out = "You need special glasses to view the game in 3D";
    }
    
  displayfr(vid.xres/2, vid.fsize,   2, vid.fsize, out, linf[cwt.c->land].color, 8);
  if(mousey < vid.fsize * 3/2) getcstat = SDLK_F1;
  }

void drawrec(const heptspin& hs, int lev, hstate s, transmatrix V) {
  
  drawcell(hs.h->c7, V * spin(hs.spin*2*M_PI/7), hs.spin);
  
  if(lev == 0) return;
  
  for(int d=0; d<7; d++) {
    int ds = fixrot(hs.spin + d);
    createMov(hs.h->c7, ds);
    if(hs.h->c7->spn[ds] == 0)
      drawcell(hs.h->c7->mov[ds], V * hexmove[d], 0);
    }

  for(int d=0; d<7; d++) {
    hstate s2 = transition(s, d);
    if(s2 == hsError) continue;
    heptspin hs2 = hsstep(hsspin(hs, d), 0);
    drawrec(hs2, lev-1, s2, V * heptmove[d]);
    }
  
  }

void drawthemap() {
  mousedist = INF;
  mousedest = -1;
  modist = 1e20; mouseover = NULL;
  if(mouseh[2] < .5) 
    modist = -5;
  playerfound = false;
  drawrec(viewctr, 4, hsOrigin, View);
  }

void centerpc(ld aspd) { 
  hyperpoint H = cwtV * C0;
  ld R = sqrt(H[0] * H[0] + H[1] * H[1]);
  if(R < 1e-9) return;
  aspd *= (1+R);
  if(R < aspd) {
    hyperpoint H2 = spintox(H) * H;
    View = rspintox(H) * pushxto0(H2) * spintox(H) * View;
    }
  else 
    View = rspintox(H) * xpush(-aspd) * spintox(H) * View;
  }

void drawmovestar() {

  if(!playerfound) return;

  hyperpoint H = cwtV * C0;
  ld R = sqrt(H[0] * H[0] + H[1] * H[1]);
  transmatrix Centered = Id;
  if(R > 1e-9) {
    hyperpoint H2 = spintox(H) * H;
    Centered = rspintox(H) * rpushxto0(H2) * spintox(H) * Centered;
    }
  
  for(int d=0; d<8; d++) 
    drawline(Centered * C0, Centered * spin(M_PI*d/4)* xpush(.5) * C0, (vid.goteyes? 0xE08060 : 0xC00000) >> darken);
  }

void optimizeview() {
  
  int turn;
  ld best = INF;
  
  transmatrix TB;
  
  for(int i=-1; i<7; i++) {

    ld trot = -i * M_PI * 2 / 7.0;
    transmatrix T = i < 0 ? Id : spin(trot) * xpush(tessf) * spin(M_PI);
    hyperpoint H = View * T * C0;
    if(H[2] < best) best = H[2], turn = i, TB = T;
    }
  
  if(turn >= 0) {
    View = View * TB;
    fixmatrix(View);
    viewctr = hsspin(viewctr, turn);
    viewctr = hsstep(viewctr, 0);
    }
  }

void movepckeydir(int d) {
  hyperpoint H = cwtV * C0;
  ld R = sqrt(H[0] * H[0] + H[1] * H[1]);
  transmatrix Centered = cwtV;
  if(R > 1e-9) {
    hyperpoint H2 = spintox(H) * H;
    Centered = rspintox(H) * pushxto0(H2) * spintox(H) * Centered;
    }
  int bdir = -1;
  ld binv = 99;
  hyperpoint MT = spin(-d * M_PI/4) * xpush(1) * C0;
  for(int i=0; i<cwt.c->type; i++) {
    ld inv = intval(Centered * spin(-i * 2 * M_PI /cwt.c->type) * xpush(1) * C0, MT);
    if(inv < binv) binv = inv, bdir = i;
    }
  movepcto(bdir);
  }

void calcparam() {
  vid.xcenter = vid.xres / 2;
  vid.ycenter = vid.yres / 2;
  vid.radius = int(vid.scale * vid.ycenter) - 40;
  vid.beta = 1 + vid.alpha + vid.eye;
  vid.alphax = vid.alpha + vid.eye;
  vid.goteyes = vid.eye > 0.001 || vid.eye < -0.001;
  }

void displayStat(int y, const char *name, const string& val, char mkey) {
  
  int dy = vid.fsize * y + vid.yres/4;
  int dx = vid.xres/2 - 100;
  
  bool xthis = (mousey >= dy-vid.fsize/2 && mousey <= dy + vid.fsize/2);
  int xcol = 0x808080;
  
  if(xthis) {
    getcstat = mkey; getcshift = 0;
    int mx = mousex - dx;
    if(mx >= 0 && mx <= 100) {
      if(mx < 20) getcshift = -1   , xcol = 0xFF0000;
      else if(mx < 40) getcshift = -0.1 , xcol = 0x0000FF;
      else if(mx < 50) getcshift = -0.01, xcol = 0x00FF00;
      if(mx > 80) getcshift = +1   , xcol = 0xFF0000;
      else if(mx > 60) getcshift = +0.1 , xcol = 0x0000FF;
      else if(mx > 50) getcshift = +0.01, xcol = 0x00FF00;
      }
    }
  
  if(val != "") {
    displaystr(dx,    dy, 0, vid.fsize, val, xthis ? 0xFFFF00 : 0x808080, 16);
    displaystr(dx+25, dy, 0, vid.fsize, "-", xthis && getcshift < 0 ? xcol : 0x808080, 8);
    displaystr(dx+75, dy, 0, vid.fsize, "+", xthis && getcshift > 0 ? xcol : 0x808080, 8);
    }

  displaystr(dx+100, dy, 0, vid.fsize, s0 + mkey, xthis ? 0xFFFF00 : 0xC0F0C0, 0);

  displaystr(dx+125, dy, 0, vid.fsize, name, xthis ? 0xFFFF00 : 0x808080, 0);
  }

void displayStatHelp(int y, const char *name) {
  
  int dy = vid.fsize * y + vid.yres/4;
  int dx = vid.xres/2 - 100;
  
  displaystr(dx+100, dy, 0, vid.fsize, name, 0xC0C0C0, 0);
  }

void displayButton(int x, int y, const string& name, int key, int align, int rad = 0) {
  if(displayfr(x, y, rad, vid.fsize, name, 0x808080, align)) {
    displayfr(x, y, rad, vid.fsize, name, 0xFFFF00, align);
    getcstat = key;
    }
  }

void quitOrAgain() {
  int y = vid.yres * (618) / 1000;
  displayButton(vid.xres/2, y + vid.fsize*1/2, "Press 'q' to quit", 'q', 8, 2);
  displayButton(vid.xres/2, y + vid.fsize*2, "or 'a' to restart", 'a', 8, 2);
  if(canmove) displayButton(vid.xres/2, y + vid.fsize*7/2, "or another key to continue", ' ', 8, 2);
  }

void showGameover() {
  int y = vid.yres * (1000-618) / 1000 - vid.fsize * 7/2;
  displayfr(vid.xres/2, y, 4, vid.fsize*2, 
    canmove ? "Are you sure you want to quit?" : "GAME OVER", 0xC00000, 8
    );
  displayfr(vid.xres/2, y + vid.fsize*2, 2, vid.fsize,   "Your score: " + its(gold()), 0xD0D0D0, 8);
  displayfr(vid.xres/2, y + vid.fsize*3, 2, vid.fsize,   "Enemies killed: " + its(tkills()), 0xD0D0D0, 8);
  quitOrAgain();
  }

void drawscreen() {
  calcparam();
  SDL_LockSurface(s);
  // unsigned char *b = (unsigned char*) s->pixels;
  // int n = vid.xres * vid.yres * 4;
  // while(n) *b >>= 1, b++, n--;
  memset(s->pixels, 0, vid.xres * vid.yres * 4);
  
  if(!canmove) darken = 1;
  if(cmode != emNormal) darken = 2;
  
  if(!vid.goteyes) for(int r=0; r<1500; r++)
    qpixel(s, vid.xcenter + int(vid.radius * sin(r)), vid.ycenter + int(vid.radius * cos(r))) = 0xFF;
    
  if(1) for(int t=0; t<int(lines.size()); t++) drawline(View * lines[t].P1, View * lines[t].P2, lines[t].col >> darken);

    DEB("dmap");
  drawthemap();
    DEB("mstar");
  drawmovestar();
  
    DEB("stats");
  int vx = vid.xres - vid.fsize * 3;
  
  int cy = vid.fsize;
  displaynum(vx, cy, 0, vid.fsize, 0xFFFFFF, gold(), "$$$"); cy += vid.fsize*2;

  for(int i=0; i<ittypes; i++) {
    if(cy == firstorb) cy += vid.fsize / 2;
    if(items[i])  
      displaynum(vx, cy, 0, vid.fsize, iinf[i].color, items[i], s0 + iinf[i].glyph), cy+=vid.fsize;
    }

  cy += vid.fsize/2;
  
  for(int i=1; i<motypes; i++) if(kills[i])  
    displaynum(vx, cy, 0, vid.fsize, minf[i].color, kills[i], s0 + minf[i].glyph), cy+=vid.fsize;
 
  cy += vid.fsize/2;
  
  // displaynum(vx,100, 0, 24, 0xc0c0c0, celldist(cwt.c), ":");
  if(kills[0])
    displaynum(vx, cy, 0, vid.fsize, 0xFFFFFF, kills[0], "#"), cy += vid.fsize;

    DEB("msgs");
  drawmessages();
  
  darken = 0;
  
  getcstat = 0;
  
    DEB("msgs1");
  if(cmode == emNormal) {
    if(!canmove) showGameover();
    displayButton(vid.xres-8, vid.yres-vid.fsize*2, "ESC to quit", SDLK_ESCAPE, 16);
    }
  
  if(cmode == emVisual) {
    displayStatHelp(1, "Configuration:");
    displayStat(3, "video resolution", its(vid.xres) + "x"+its(vid.yres), 'x');
    displayStat(4, "fullscreen mode", vid.full ? "ON" : "OFF", 'f');
    displayStat(6, "animation speed", fts(vid.aspeed), 'a');
    displayStat(7, "dist from hyperboloid ctr", fts(vid.alpha), 'p');
    displayStat(8, "scale factor", fts(vid.scale), 'z');
    displayStat(9, "distance between eyes", fts(vid.eye * 10), 'e');
    displayStatHelp(11, "use Shift to decrease");
    displayStatHelp(12, "and Ctrl to fine tune");
    displayStatHelp(13, "(e.g. Shift+Ctrl+Z)");
    displayStat(15, "exit configuration", "", 'v');
    displayStat(16, "see the help screen", "", 'h');
    displayStat(17, "save the current config", "", 's');
    }
  
    DEB("msgs2");
  describeMouseover();

  if(cmode == emNormal || cmode == emVisual)
    displayButton(vid.xres-8, vid.yres-vid.fsize, "(v) config", 'v', 16);
  
  if(cmode == emQuit) {
    showGameover();
    }

  if(cmode == emHelp) {
    int last = 0;
    int lastspace = 0;
    int cy = vid.fsize * 4;
    
    int xs = vid.xres * 618/1000;
    int xo = vid.xres * 186/1000;
    
    for(int i=0; i<=size(help); i++) {
      int ls = 0;
      int prev = last;
      if(help[i] == ' ') lastspace = i;
      if(textwidth(vid.fsize, help.substr(last, i-last)) > xs) {
        if(lastspace == last) ls = i-1, last = i-1;
        else ls = lastspace, last = ls+1;
        }
      if(help[i] == 10 || i == size(help)) ls = i, last = i+1;
      if(ls) {
        displayfr(xo, cy, 2, vid.fsize, help.substr(prev, ls-prev), 0xC0C0C0, 0);
        if(ls == prev) cy += vid.fsize/2;
        else cy += vid.fsize;
        lastspace = last;
        }
      }
    }
  
    DEB("msgs3");
  SDL_UnlockSurface(s);
  
  if(playermoved && vid.aspeed > 4.99) { 
    centerpc(1000);
    playermoved = false; 
    return; 
    }
  SDL_UpdateRect(s, 0, 0, vid.xres, vid.yres);
  }

void setvideomode() {
  if(vid.modeid < 0) vid.modeid = 0;
  if(vid.modeid >= NUMMODES) vid.modeid = NUMMODES - 1;  
  vid.xres = xres[vid.modeid];
  vid.yres = yres[vid.modeid];
  if(vid.modeid == 0 && !vid.full) 
    vid.xres -= vid.xres/10, vid.yres -= vid.yres/10;

  s= SDL_SetVideoMode(vid.xres, vid.yres, 32, vid.full ? SDL_FULLSCREEN : 0);
  if(!s) {
    addMessage("Failed to set the graphical mode: "+its(vid.xres)+"x"+its(vid.yres)+(vid.full ? " fullscreen" : " windowed"));
    vid.xres = 640;
    vid.yres = 480;
    s= SDL_SetVideoMode(vid.xres, vid.yres, 32, 0);
    }
  if(vid.modeid != 6)
    vid.fsize = vid.yres / 32;
  }

void restartGraph() {
  viewctr.h = &origin;
  viewctr.spin = 0;
  View = Id;
  }

void saveConfig() {
  FILE *f = fopen(conffile, "wt");
  if(!f) {
    addMessage(s0 + "Could not open the config file: " + conffile);
    return;
    }
  fprintf(f, "%d %d %d %d\n", vid.xres, vid.yres, vid.full, vid.fsize);
  fprintf(f, "%f %f %f %f\n", float(vid.scale), float(vid.eye), float(vid.alpha), float(vid.aspeed));
  fprintf(f, "\n\nThe numbers are:\n");
  fprintf(f, "screen width & height, fullscreen mode (0=windowed, 1=fullscreen), font size\n");
  fprintf(f, "scale, eye distance, parameter, animation speed\n");
  
  fclose(f);
  addMessage(s0 + "Configuration saved to: " + conffile);
  }

void loadConfig() {
  FILE *f = fopen(conffile, "rt");
  if(!f) return;
  int fs;
  fscanf(f, "%d%d%d%d", &xres[6], &yres[6], &fs, &vid.fsize);
  vid.full = fs;
  float a, b, c, d;
  fscanf(f, "%f%f%f%f\n", &a, &b, &c, &d);
  vid.scale = a; vid.eye = b; vid.alpha = c; vid.aspeed = d;
  fclose(f);
  printf("Loaded configuration: %s\n", conffile);
  vid.modeid = 6;
  }

void initgraph() {

  vid.modeid = 0;
  vid.scale = 1;
  vid.alpha = 1;
  vid.aspeed = 0;
  vid.eye = 0;
  vid.full = false;
  vid.quick = true;
  
  SDL_Init(SDL_INIT_VIDEO);
  const SDL_VideoInfo *inf = SDL_GetVideoInfo();
  xres[0] = inf->current_w;
  yres[0] = inf->current_h;
  // seems to be buggy on Windows
  if(xres[0] < 320 || xres[0] > 2000) xres[0] = 640, vid.modeid = 3;
  if(yres[0] < 200 || yres[0] > 2000) yres[0] = 480, vid.modeid = 3;
  
  loadConfig();

  restartGraph();
  
  initgeo();

  setvideomode();
  if(!s) {
    printf("Failed to initialize graphics.\n");
    exit(2);
    }
    
  SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
  
  if(TTF_Init() != 0) {
    printf("Failed to initialize TTF.\n");
    exit(2);
    }

  }

int frames;

void mainloop() {
  int curt, lastt;
  curt = SDL_GetTicks();
  while(true) {
    DEB("screen");
    frames++;
    optimizeview();
    lastt = curt; curt = SDL_GetTicks();
    if(playermoved && vid.aspeed > -4.99)
      centerpc((curt - lastt) / 1000.0 * exp(vid.aspeed));
    drawscreen();
    SDL_Event ev;
    DEB("react");
    while(SDL_PollEvent(&ev)) {
      int sym = 0;
      ld shift = 1;
      
      if(ev.type == SDL_KEYDOWN) {
        sym = ev.key.keysym.sym;        
        if(ev.key.keysym.mod & (KMOD_LSHIFT | KMOD_RSHIFT)) shift = -1;
        if(ev.key.keysym.mod & (KMOD_LCTRL | KMOD_RCTRL)) shift /= 10;
        }

      if(ev.type == SDL_MOUSEBUTTONDOWN) {
        sym = getcstat, shift = getcshift;
        if(ev.button.button==SDL_BUTTON_RIGHT) sym = SDLK_F1;
        }

      if(sym == SDLK_RIGHT) View = xpush(-0.2*shift) * View, playermoved = false;
      if(sym == SDLK_LEFT) View = xpush(+0.2*shift) * View, playermoved = false;
      if(sym == SDLK_UP) View = ypush(+0.2*shift) * View, playermoved = false;
      if(sym == SDLK_DOWN) View = ypush(-0.2*shift) * View, playermoved = false;
      if(sym == SDLK_PAGEUP) View = spin(0.2*shift) * View;
      if(sym == SDLK_PAGEDOWN) View = spin(-0.2*shift) * View;
      
      if(ev.type == SDL_MOUSEMOTION) {
        mousex = ev.motion.x;
        mousey = ev.motion.y;
        mouseh = gethyper(mousex, mousey);
        }

      DEB("r1");
      if(sym == SDLK_F7) {

        time_t timer;
        timer = time(NULL);
        char buf[128]; strftime(buf, 128, "shot-%y%m%d-%H%M%S.bmp", localtime(&timer));

        SDL_SaveBMP(s, buf);
        addMessage(s0 + "Screenshot saved to " + buf);
        }

      DEB("r2");
      if(cmode == emNormal) {
      
        if(sym == 'l' || sym == 'd' || sym == SDLK_KP6) movepckeydir(0);
        if(sym == 'n' || sym == 'c' || sym == SDLK_KP3) movepckeydir(1);
        if(sym == 'j' || sym == 'x' || sym == SDLK_KP2) movepckeydir(2);
        if(sym == 'b' || sym == 'z' || sym == SDLK_KP1) movepckeydir(3);
        if(sym == 'h' || sym == 's' || sym == SDLK_KP4) movepckeydir(4);
        if(sym == 'y' || sym == 'w' || sym == SDLK_KP7) movepckeydir(5);
        if(sym == 'k' || sym == 'e' || sym == SDLK_KP8) movepckeydir(6);
        if(sym == 'u' || sym == 'r' || sym == SDLK_KP9) movepckeydir(7);

        if(sym == '.' || sym == 'a') movepcto(-1);
        if(sym == SDLK_KP5) movepcto(-1);

        if(sym == SDLK_F5)  restartGame();
        if(sym == SDLK_ESCAPE) cmode = emQuit;
        if(sym == SDLK_F10) return;
        
        if(sym == 'q' && !canmove) return;
        if(sym == 'a' && !canmove) restartGame();
        
        if(sym == SDLK_HOME || sym == SDLK_F3 || sym == ' ') {
          if(playerfound) centerpc(INF);
          else {
            View = Id;
            viewctr.h = cwt.c->master;
            SDL_LockSurface(s);
            drawthemap(); 
            SDL_UnlockSurface(s);
            centerpc(INF);
            }
          }
        
        if(sym == 'v' || sym == SDLK_F2) {
          cmode = emVisual;
          }

        if(ev.type == SDL_MOUSEBUTTONDOWN && sym == 0) {
          if(mousedest == -1)
            movepcto(mousedest);
          else
            movepcto((mousedest + 42 - cwt.spin)%42);
          }
        
        if(sym == SDLK_F1) {
          cmode = emHelp;
          }
        }
      
      else if(cmode == emVisual) {
      
        if(sym == 'p') vid.alpha += shift * 0.1;        
        if(sym == 'e') vid.eye += shift * 0.01;        
        if(sym == 'z') vid.scale += shift * 0.1;
        if(sym == 'a') vid.aspeed += shift;
        if(sym == 'x') vid.modeid += (shift > 0 ? 1 : -1), setvideomode();        
        if(sym == 'f') vid.full = !vid.full, setvideomode();
        
        if(sym == 'v') cmode = emNormal;
        if(sym == 's') saveConfig();
        
        if(sym == SDLK_F4) cheat();
        }
        
      else if(cmode == emHelp) {
        if(sym != 0 || ev.type == SDL_MOUSEBUTTONDOWN) cmode = emNormal;
        }
        
      else if(cmode == emQuit) {
        if(sym == 'q') return;
        else if(sym == 'a' || sym == SDLK_F5) restartGame(), cmode = emNormal;
        else if(sym != 0 || ev.type == SDL_MOUSEBUTTONDOWN) cmode = emNormal;
        }
      if(ev.type == SDL_QUIT)
        return;

      DEB("r3");
      }
    
    if(playerdead) break;
    }
  
  }

void cleargraph() {
  for(int i=0; i<256; i++) if(font[i]) TTF_CloseFont(font[i]);
  }
