/*
 * usb_mass.c - USB Mass storage driver for PS2
 *
 * (C) 2001, Gustavo Scotti (gustavo@scotti.com)
 * (C) 2002, David Ryan ( oobles@hotmail.com )
 * (C) 2004, Marek Olejnik (ole00@post.cz)
 *
 * VFAT.IRX
 * (C) 2005, Marcansoft
 * (C) 2005, MrX
 * (C) 2005, Hermes/PS2REALITY 
 *
 * IOP file io driver and RPC server
 *
 * See the file LICENSE included with this distribution for licensing terms.
 */

#include <tamtypes.h>
#include <thbase.h>
#include <thsemap.h>
#include <sifrpc.h>
#include <sysmem.h>
#include <ioman.h>
#include <io_common.h>
#include <loadcore.h>
#include <stdio.h>
#include <sysclib.h>
#include <stat.h>
#include "mass_stor.h"

#include "fat/fat.h"

#define DEBUG 1
#include "mass_debug.h"

tVolume *V;

// para compatibilidad con usb_mass 
#define FIO_S_IFREG2		0x10		// Regular file
#define FIO_S_IFDIR2		0x20		// Directory

// FileIO structure.
static iop_device_t driver;

// Function array for fileio structure.
static iop_device_ops_t functions;

extern PartType PartitionTypes[];

#define RPC_FUNC_NONE		0
#define RPC_FUNC_INIT		1
#define RPC_FUNC_OPEN		2
#define RPC_FUNC_CLOSE		3
#define RPC_FUNC_READ		4
#define RPC_FUNC_WRITE		5
#define RPC_FUNC_LSEEK		6
#define RPC_FUNC_REMOVE		7
#define RPC_FUNC_MKDIR		8
#define RPC_FUNC_RMDIR		9
#define RPC_FUNC_DOPEN		10
#define RPC_FUNC_DCLOSE		11
#define RPC_FUNC_DREAD		12
#define RPC_FUNC_GETSTAT	13
#define RPC_FUNC_CHSTAT		14
#define RPC_FUNC_RENAME		15
#define RPC_FUNC_FSINFO		16

#define RPC_FUNC_GETFIRST 128 // compatibilidad con usb_mass (RPC)
#define RPC_FUNC_GETNEXT  129

typedef int (*f_io_format)(iop_file_t *, ...);

int rpc_func=RPC_FUNC_NONE;
iop_file_t *rf;
iop_device_t *rdriver;
const char *rname;
int rmode;
unsigned int rsmask;
unsigned long roffset;
int rorigin;
void *rbuf;
int rsize;

int rret;

int rpc_tid;

int sema_call;
int sema_ret;
int sema_mutex;

char status;

typedef struct {
	unsigned char reserved;
	unsigned char sec;
	unsigned char min;
	unsigned char hour;
	unsigned char day;
	unsigned char month;
	unsigned short year;
} io_date __attribute__((packed));
	

#define HI_PRIO 41 //8 //1
#define LO_PRIO 48//40 //123

void _offset0(void)
{
return;
}

void offset0(void)
{
return;
}
int voidprintf(char *a,...)
{
}

unsigned char * malloc(int x)
{
int olds;
CpuSuspendIntr(&olds);
x=AllocSysMemory(0,x,0);
CpuResumeIntr(olds);
return ((unsigned char *) x);
}

int free(int x)
{
int olds;
CpuSuspendIntr(&olds);
x=FreeSysMemory(x);
CpuResumeIntr(olds);
return x;
}

int dummy()
{
        //printf("vfat: dummy function called.\n");
	return -38;
}

void connect(void) {
	//printf("vfat: device connected\n");
	status=1;
}

void disconnect(void) {
	//printf("vfat: device disconnected\n");
	if(V) {
		fat_unmount();
		V=NULL;
		//printf("vfat: filesystem unmounted\n");
	}
	status=2;
}

int write_sectors(void *buffer, unsigned int start, unsigned int num) {
	return massWrite(buffer,start,num);
}
int read_sectors(void *buffer, unsigned int start, unsigned int num) {
	return massRead(buffer,start,num);
}
unsigned int get_blocksize(void) {
	return massBlockSize();
}
unsigned int get_totalblocks(void) {
	return massTotalBlocks();
}

char *fixname(const char *name) {
	char *buf;
	char *p;
	buf=malloc(strlen(name)+1);
	strcpy(buf,name);
	p=buf;
	while(*p) {
		if(*p=='/') *p='\\';
		p++;
	}
	return buf;
}
/* TODO: They are really ints aren't they? need to check on that */
#if 0
#define TOBCD(i) ((i%10)+((i/10)<<4))
#define FROMBCD(i) ((i&7)+(i>>4)*10)
#else
#define TOBCD(i) (i)
#define FROMBCD(i) (i)
#endif
io_date *dos2io(unsigned long long d) {
	static io_date ret;
	unsigned int year;
	ret.reserved=0;
	ret.sec   = TOBCD(((d>>0)&0x1F)*2);
	ret.min   = TOBCD(((d>>5)&0x3F));
	ret.hour  = TOBCD(((d>>11)&0x1F));
	ret.day   = TOBCD(((d>>16)&0x1F));
	ret.month = TOBCD(((d>>21)&0x0F));
	year      = (((d>>25)&0x7F))+1980;
	#if 0
	ret.year1 = TOBCD(year/100);
	ret.year2 = TOBCD(year%100);
	#else
	//ret.year1=year&0xFF;
	//ret.year2=year>>8;
	ret.year=year;
	#endif
	return &ret;
}

unsigned long long io2dos(io_date d) {
	unsigned long long ret=0;
	unsigned int year;
	ret += (FROMBCD(d.sec)/2)<<0;
	ret += (FROMBCD(d.min))<<5;
	ret += (FROMBCD(d.hour))<<11;
	ret += (FROMBCD(d.day))<<16;
	ret += (FROMBCD(d.month))<<21;
	#if 0
	year  = FROMBCD(d.year1)*100;
	year += FROMBCD(d.year2);
	#else
	//year=d.year1+(d.year2<<8);
	year=d.year;
	#endif
	ret += (FROMBCD(year))<<25;
	return ret;
}

int fat_checkmount(void) {
	int ret;
	////printf("vfat: Checking mount...\n");
	if(!V) {
		//printf("vfat: filesystem not mounted, trying to mount now...\n");
		if(massGetStatus()) {
			//printf("vfat: device connected and warmed up, mounting FAT filesystem... ");
			ret=fat_mount(0);
			//printf("done. %d\n",ret);
			return 1;
		}
		//printf("vfat: device not connected, mount failed\n");
		return 0;
	}
	////printf("vfat: filesystem is mounted\n");
	return 1;
}



tFile *Flist=0xFFFFDEAD;
#define FAT_MAX_NAME 128

#define DIR_CHAIN_SIZE 10

typedef struct _fat_dir_record { // 140 bytes
	unsigned char attr;		//attributes (bits:5-Archive 4-Directory 3-Volume Label 2-System 1-Hidden 0-Read Only)
	unsigned char name[FAT_MAX_NAME];
	unsigned char date[4];	//D:M:Yl:Yh
	unsigned char time[3];  //H:M:S
	unsigned int  size;		//file size, 0 for directory
} fat_dir_record;

fd32_fs_lfnfind_t fde;


int rpc_getNext(void* buf) {
int ret;
fat_dir_record fatDir;	
	
	// read    
				if((unsigned ) Flist==0xFFFFDEAD) {
					return 0;
				}
				
				ret=fat_readdir(Flist,&fde);
				if(ret==FD32_ENMFILE) {
                fat_close(Flist);
				Flist=0xFFFFDEAD;
				return 0;
				}
				if(ret<0) {
					fat_close(Flist);Flist=0xFFFFDEAD;return 0;
				}
				
				
				if(fde.Attr & FD32_ADIR) {
					fatDir.attr=FIO_S_IFDIR2;
				} else {
					fatDir.attr=FIO_S_IFREG2;
				}
				fatDir.attr |= FIO_S_IRUSR|FIO_S_IWUSR|FIO_S_IXUSR;
//				de.stat.mode |= FIO_S_IRGRP|FIO_S_IWGRP|FIO_S_IXGRP; compatibilidad con usb_mass
				fatDir.attr |= FIO_S_IROTH|FIO_S_IWOTH|FIO_S_IXOTH;
				fatDir.size=fde.SizeLo;
				
			

	
	
	
	fatDir.time[2]   = TOBCD(((fde.MTime>>0)&0x1F)*2);
	fatDir.time[1]   = TOBCD(((fde.MTime>>5)&0x3F));
	fatDir.time[0]  = TOBCD(((fde.MTime>>11)&0x1F));
	fatDir.date[0]  = TOBCD(((fde.MTime>>16)&0x1F));
	fatDir.date[1]  = TOBCD(((fde.MTime>>21)&0x0F));
	fatDir.date[2]     = ((((((((unsigned)(fde.MTime>>25))&0x7F))+80) % 100)>>4) & 16)+48;
	fatDir.date[3]     = (((((((unsigned)(fde.MTime>>25))&0x7F))+80) % 100) & 16)+48;
	
	
	strncpy(fatDir.name,fde.LongName,FAT_MAX_NAME);
    fatDir.name[FAT_MAX_NAME-1]=0;
	//printf("RPC name:%s\n",fatDir.name);			
    memcpy(buf, &fatDir, sizeof(fat_dir_record)); //copy only important things
	return 1;
}

int rpc_getFirst(void* buf) {
	
	int ret;
	char *nname;
	nname=fixname(buf);
		
	ret=fat_open(nname, FD32_OREAD | FD32_ODIR | FD32_OEXIST,FD32_ANONE, 0, &Flist);
	free(nname);
				if(ret<0) {Flist=0xFFFFDEAD;
					return 0;
				}
				
//printf("RPC Prime\n");
return rpc_getNext(buf);

}


void vfatRpcThread(void* param) {
	tFile *F;
	int ret;
	long long int off;
	int nmode;
	io_dirent_t de;
	io_stat_t attr;
	char *nname;
	char *ndest;
	fd32_fs_attr_t fattr;
	fd32_fs_lfnfind_t fde;
	fd32_getfsfree_t Fr;
	
	//printf("vfat: thread started\n");
	
	while(1) {
		WaitSema(sema_call);
		ChangeThreadPriority(rpc_tid,LO_PRIO);
		switch(rpc_func) {
			case RPC_FUNC_GETFIRST:
				rret=rpc_getFirst(rname);
				break;
			case RPC_FUNC_GETNEXT:
				rret=rpc_getNext(rname);
				break;
			case RPC_FUNC_NONE:
				//printf("vfat: thread woken up, but no RPC function chosen!\n");
				rret=-1;
				break;
			case RPC_FUNC_INIT:
				//printf("vfat: init called\n");
				rret=0;
				break;
			case RPC_FUNC_OPEN:
				//printf("vfat: open '%s' mode 0x%X\n",rname,rmode);
				if(rname[0]==0) {
					if(!fat_checkmount()) {
						//printf("vfat: Mounting failed, aborting.\n");
						rret=FD32_ENODEV;
						//printf("vfat: open device %d\n",rret);
						break;
					}
					rf->privdata=(void*)0xFFFFDEAD;
					//printf("vfat: open device %d\n",(int)rf);
					rret=(int)rf;
					break;
				}
				nname=fixname(rname);
				//printf("vfat: name: '%s'\n",nname);
				switch(rmode&3) {
					case 0:
						//printf("vfat: no mode bits selected, assuming O_RDONLY\n");
						nmode=FD32_OREAD;
						break;
					case O_RDONLY:
						nmode=FD32_OREAD;
						break;
					case O_WRONLY:
						nmode=FD32_OWRITE;
						break;
					case O_RDWR:
						nmode=FD32_ORDWR;
						break;
					default:
						//printf("vfat: WTF?\n");
						nmode=FD32_OREAD;
						break;
				}
				if(rmode&O_CREAT)
				{nmode |= FD32_OCREAT  |  FD32_OTRUNC ;}
				else
				/*if(rmode&O_TRUNC)
					nmode |= FD32_OTRUNC;
				if(!(rmode&O_TRUNC) && !(rmode&O_CREAT))*/
					nmode |= FD32_OEXIST;
				
				ret=fat_open(nname,nmode,FD32_AARCHIV,0,&F);
				//printf("vfat: open %d\n",ret);
				free(nname);
				if(ret<0) {
					//printf("vfat: open %d\n",ret);
					rret=ret;
					break;
				}
				if(rmode&O_APPEND) {
					off=0;
					ret=fat_lseek(F,&off,FD32_SEEKEND);
					if(ret<0) {
						fat_close(F);
						rret=ret;
						break;
					}
				}
				rf->privdata=F;
				//printf("vfat: open fd %d\n",(int)rf);
				rret=(int)rf;
				break;
			case RPC_FUNC_CLOSE:
			case RPC_FUNC_DCLOSE:
				//printf("vfat: close fd %d\n",(int)rf);
				if(((unsigned int)rf->privdata)==0xFFFFDEAD) {
					rret=1;
					//printf("vfat: close device %d\n",rret);
					break;
				}
				F=(tFile*)rf->privdata;
				ret=fat_close(F);
				//printf("vfat: close %d\n",ret);
				rret=ret;
				break;
			case RPC_FUNC_READ:
				//printf("vfat: read fd %d buf 0x%X size %d\n",(int)rf,(unsigned)rbuf,rsize);
				if(((unsigned int)rf->privdata)==0xFFFFDEAD) {
					rret=FD32_EINVAL;
					//printf("vfat: read device %d\n",rret);
					break;
				}
				F=(tFile*)rf->privdata;
				ret=fat_read(F,rbuf,rsize);
				//printf("vfat: read %d\n",ret);
				rret=ret;
				break;
			case RPC_FUNC_WRITE:
				//printf("vfat: write fd %d buf 0x%X size %d\n",(int)rf,(unsigned)rbuf,rsize);
				if(((unsigned int)rf->privdata)==0xFFFFDEAD) {
					rret=FD32_EINVAL;
					//printf("vfat: write device %d\n",rret);
					break;
				}
				F=(tFile*)rf->privdata;
				#ifdef FATWRITE
				ret=fat_write(F,rbuf,rsize);
				#else
				ret=-5;
				#endif
				//printf("vfat: write %d\n",ret);
				rret=ret;
				break;
			case RPC_FUNC_LSEEK:
				//printf("vfat: lseek fd %d offset 0x%lX origin %d\n",(int)rf,roffset,rorigin);
				if(((unsigned int)rf->privdata)==0xFFFFDEAD) {
					rret=FD32_EINVAL;
					//printf("vfat: lseek device %d\n",rret);
					break;
				}
				F=(tFile*)rf->privdata;
				off=roffset;
				ret=fat_lseek(F,&off,rorigin);
				if(ret==0) ret=(int)off;
				//printf("vfat: lseek %d\n",ret);
				rret=ret;
				break;
			case RPC_FUNC_REMOVE:
				//printf("vfat: remove '%s'\n",rname);
				nname=fixname(rname);
				//printf("vfat: name: '%s'\n",nname);
				#ifdef FATWRITE
				ret=fat_unlink(nname,FD32_FAALL&(~FD32_FADIR));
				#else
				ret=-5;
				#endif
				free(nname);
				//printf("vfat: remove %d\n",ret);
				rret=ret;
				break;
			case RPC_FUNC_MKDIR:
				//printf("vfat: mkdir '%s'\n",rname);
				nname=fixname(rname);
				//printf("vfat: name: '%s'\n",nname);
				#ifdef FATWRITE
				ret=fat_mkdir(nname);
				#else
				ret=-5;
				#endif
				free(nname);
				//printf("vfat: mkdir %d\n",ret);
				rret=ret;
				break;
			case RPC_FUNC_RMDIR:
				//printf("vfat: rmdir '%s'\n",rname);
				nname=fixname(rname);
				//printf("vfat: name: '%s'\n",nname);
				#ifdef FATWRITE
				ret=fat_rmdir(nname);
				#else
				ret=-5;
				#endif
				free(nname);
				//printf("vfat: rmdir %d\n",ret);
				rret=ret;
				break;
			case RPC_FUNC_DOPEN:
				//printf("vfat: dopen '%s'\n",rname);
				nname=fixname(rname);
				//printf("vfat: name: '%s'\n",nname);
				ret=fat_open(nname, FD32_OREAD | FD32_ODIR | FD32_OEXIST,FD32_ANONE, 0, &F);
				free(nname);
				if(ret<0) {
					rret=ret;
					//printf("vfat: dopen %d\n",ret);
					break;
				}
				rf->privdata=F;
				//printf("vfat: dopen fd %d\n",(int)rf);
				rret=(int)rf;
				break;
			case RPC_FUNC_DREAD:
				//printf("vfat: dread fd %d buf 0x%X\n",(int)rf,(unsigned)rbuf);
				if(((unsigned int)rf->privdata)==0xFFFFDEAD) {
					rret=FD32_EINVAL;
					//printf("vfat: dread device %d\n",rret);
					break;
				}
				F=(tFile*)rf->privdata;
				ret=fat_readdir(F,&fde);
				if(ret==FD32_ENMFILE) {
					rret=0;
					break;
				}
				if(ret<0) {
					//printf("vfat: dread %d\n",ret);
					rret=ret;
					break;
				}
				de.unknown=0;
				strncpy(de.name,fde.LongName,255);
				de.name[255]=0;
				if(fde.Attr & FD32_ADIR) {
					de.stat.mode=FIO_S_IFDIR2;
				} else {
					de.stat.mode=FIO_S_IFREG2;
				}
				de.stat.mode |= FIO_S_IRUSR|FIO_S_IWUSR|FIO_S_IXUSR;
//				de.stat.mode |= FIO_S_IRGRP|FIO_S_IWGRP|FIO_S_IXGRP; compatibilidad con usb_mass
				de.stat.mode |= FIO_S_IROTH|FIO_S_IWOTH|FIO_S_IXOTH;
				de.stat.attr=0;
				de.stat.size=fde.SizeLo;
				de.stat.hisize=fde.SizeHi;
				memcpy(de.stat.ctime,dos2io(fde.CTime),8);
				memcpy(de.stat.atime,dos2io(fde.ATime),8);
				memcpy(de.stat.mtime,dos2io(fde.MTime),8);
				memcpy(rbuf,&de,sizeof(de));
				//printf("vfat: dread %d\n",ret);
				rret=strlen(fde.LongName);
				break;
			case RPC_FUNC_RENAME:
				//printf("vfat: rename '%s'\n",rname);
				nname=fixname(rname);
				ndest=nname;
				while(*ndest && (*ndest != '#'))
					ndest++;
				if(*ndest != '#') {
					free(nname);
					//printf("vfat: rename %d\n",-1);
					rret=-1;
					break;
				}
				*ndest=0;
				ndest++;
				//printf("vfat: name: '%s' -> '%s'\n",nname,ndest);
				#ifdef FATWRITE
				ret=fat_rename(nname,ndest);
				#else
				ret=-5;
				#endif
				free(nname);
				//printf("vfat: rename %d\n",ret);
				rret=ret;
				break;
			case RPC_FUNC_GETSTAT:
				//printf("vfat: getstat '%s'\n",rname);
				nname=fixname(rname);
				//printf("vfat: name: '%s'\n",nname);
				ret=fat_open(nname,FD32_OREAD|FD32_OEXIST|FD32_OCOMMIT,FD32_AARCHIV,0,&F);
				if(ret==FD32_EACCES) {
					ret=fat_open(nname,FD32_OREAD|FD32_OEXIST|FD32_OCOMMIT|FD32_ODIR,FD32_AARCHIV,0,&F);
					if(ret<0) ret=FD32_EACCES;
				}
				free(nname);
				if(ret<0) {
					//printf("vfat: getstat %d\n",ret);
					rret=ret;
					break;
				}
				ret=fat_get_attr(F, &fattr);
				if(ret<0) {
					//printf("vfat: getstat %d\n",ret);
					fat_close(F);
					rret=ret;
					break;
				}
				attr.size=F->DirEntry.FileSize;
				attr.hisize=0;
				ret=fat_close(F);
				if(ret<0) {
					//printf("vfat: getstat %d\n",ret);
					fat_close(F);
					rret=ret;
					break;
				}

				if(fattr.Attr & FD32_ADIR) {
					attr.mode=FIO_S_IFDIR2;
				} else {
					attr.mode=FIO_S_IFREG2;
				}
				attr.mode |= FIO_S_IRUSR|FIO_S_IWUSR|FIO_S_IXUSR;
//				attr.mode |= FIO_S_IRGRP|FIO_S_IWGRP|FIO_S_IXGRP; compatibilidad con usb_mass
				attr.mode |= FIO_S_IROTH|FIO_S_IWOTH|FIO_S_IXOTH;
				attr.attr=0;
				memcpy(attr.ctime,dos2io(fde.CTime),8);
				memcpy(attr.atime,dos2io(fde.ATime),8);
				memcpy(attr.mtime,dos2io(fde.MTime),8);
				memcpy(rbuf,&attr,sizeof(attr));
				//printf("vfat: getstat %d\n",rret);
				rret=ret;
				break;
			case RPC_FUNC_CHSTAT:
				//printf("vfat: chstat '%s'\nvfat: chstat not implemented yet\n",rname);
				rret=0; //Pretend it went OK
				break;
			case RPC_FUNC_FSINFO:
				//printf("vfat: fsinfo\n");
				Fr.Size=sizeof(fd32_getfsfree_t);
				ret=fat_get_fsfree(&Fr);
				if(ret<0) {
					//printf("vfat: fsinfo %d\n",ret);
					rret=ret;
					break;
				}
				attr.size=Fr.AvailClus;
				attr.hisize=Fr.TotalClus;
				attr.mode=Fr.SecPerClus;
				attr.attr=Fr.BytesPerSec;
				attr.atime[0]=status;
				if(status==1) status=0;
				memcpy(rbuf,&attr,sizeof(attr));
				//printf("vfat: fsinfo %d [%d] [%X]\n",ret,attr.atime[0],(int)rbuf);
				rret=ret;
				break;
			default:
				//printf("vfat: unknown RPC function %d",rpc_func);
				rret=-1;
				break;
		}
		rpc_func=RPC_FUNC_NONE;
		ChangeThreadPriority(rpc_tid,HI_PRIO);
		SignalSema(sema_ret);
	}
}



int fd_initialize( iop_device_t *driver) {
	WaitSema(sema_mutex);
	rpc_func=RPC_FUNC_INIT;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}

int fd_open( iop_file_t *f, const char *name, int mode, ...) {
	WaitSema(sema_mutex);
	if((name[0]!=0) && (name[1]!=0) && (name[0]=='r') && (name[1]=='#')) {
		name+=2;
		rname=name;
		rf=f;
		rpc_func=RPC_FUNC_RENAME;
		SignalSema(sema_call);
		WaitSema(sema_ret);
		SignalSema(sema_mutex);
		return rret;
	}
	rname=name;
	rmode=mode;
	rf=f;
	rpc_func=RPC_FUNC_OPEN;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}

int fd_close(iop_file_t *f) {
	WaitSema(sema_mutex);
	rf=f;
	rpc_func=RPC_FUNC_CLOSE;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}


int fd_read(iop_file_t *f, void *buf, int size) {
	int ret;
	WaitSema(sema_mutex);
	rf=f;

	ret=0;
//	while(size>0)
//	{
	rbuf=buf;
	rsize=size;
//	if(rsize>16384) rsize=16384;
	rpc_func=RPC_FUNC_READ;
	SignalSema(sema_call);
	WaitSema(sema_ret);
  //  if(rret<=0) break;
	ret+=rret;
	//buf=(void *) (((unsigned) buf)+rret);
	//size-=rret;
	//}
	SignalSema(sema_mutex);
	return ret;
}
int fd_write(iop_file_t *f, void *buf, int size) {
	WaitSema(sema_mutex);
	rf=f;
	rbuf=buf;
	
	
	rsize=size;
	rpc_func=RPC_FUNC_WRITE;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}
int cap_read(iop_file_t * kernel_fd,char * buffer,int size  );
int fd_lseek(iop_file_t *f, unsigned long offset, int origin) {


	if(origin==0)
	{
	if(((unsigned) offset)==0xffffffea) return 0xc0c0; //identificacion

	if(((unsigned) offset)==0xffffffec) return f;
	if(((unsigned) offset)==0xffffffed) return cap_read;
	if(((unsigned) offset)==0xffffffee) return 0;
	}

	//WaitSema(sema_mutex);
	rf=f;
	roffset=offset;
	rorigin=origin;
	rpc_func=RPC_FUNC_LSEEK;
	/////////////////
	if(((unsigned int)rf->privdata)==0xFFFFDEAD) {
					rret=FD32_EINVAL;
					//printf("vfat: lseek device %d\n",rret);
					
				}
	else
	{
	tFile *F;
	int ret;
	long long int off;
				F=(tFile*)rf->privdata;
				off=roffset;
				ret=fat_lseek(F,&off,rorigin);
				if(ret==0) ret=(int)off;
				//printf("vfat: lseek %d\n",ret);
				rret=ret;
	}
	/////////////////

	//SignalSema(sema_call);
	//WaitSema(sema_ret);
	//SignalSema(sema_mutex);
	return rret;
}

int fd_remove(iop_file_t *f, const char *name) {
	WaitSema(sema_mutex);
	rf=f;
	rname=name;
	rpc_func=RPC_FUNC_REMOVE;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}

int fd_mkdir(iop_file_t *f, const char *name) {
	WaitSema(sema_mutex);
	rf=f;
	rname=name;
	rpc_func=RPC_FUNC_MKDIR;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}

int fd_rmdir(iop_file_t *f, const char *name) {
	WaitSema(sema_mutex);
	rf=f;
	rname=name;
	rpc_func=RPC_FUNC_RMDIR;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}

int fd_dopen(iop_file_t *f, const char *name) {
	WaitSema(sema_mutex);
	rf=f;
	rname=name;
	rpc_func=RPC_FUNC_DOPEN;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}

int fd_dclose(iop_file_t *f) {
	WaitSema(sema_mutex);
	rf=f;
	rpc_func=RPC_FUNC_DCLOSE;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}

int fd_dread(iop_file_t *f, void *buf) {
	WaitSema(sema_mutex);
	rf=f;
	rbuf=buf;
	rpc_func=RPC_FUNC_DREAD;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}

int fd_getstat(iop_file_t *f, const char *name, void *buf) {
	WaitSema(sema_mutex);
	rf=f;
	rname=name;
	rbuf=buf;
	if((name[0]!=0) && (name[1]!=0) && (name[0]=='f') && (name[1]=='#')) {
		rpc_func=RPC_FUNC_FSINFO;
		SignalSema(sema_call);
		WaitSema(sema_ret);
		SignalSema(sema_mutex);
		return rret;
	}
	rpc_func=RPC_FUNC_GETSTAT;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}

int fd_chstat(iop_file_t *f, const char *name, void *buf, unsigned int statmask) {
	WaitSema(sema_mutex);
	rf=f;
	rname=name;
	rbuf=buf;
	rsmask=statmask;
	rpc_func=RPC_FUNC_CHSTAT;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}

void _local()
{
__asm __volatile__ ("nop
					nop
					nop
					nop
					");
}

int cap_init( iop_device_t *driver)
{
unsigned gpsav;
int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);


ret=fd_initialize( driver);

__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;
}
int cap_open(iop_file_t * kernel_fd,char *name, int mode)
{
unsigned gpsav;
int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);
ret=fd_open(kernel_fd, name, mode);
__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);


return ret;

}
 
int cap_read(iop_file_t * kernel_fd,char * buffer,int size  )
{
unsigned gpsav;
register int ret;

__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);


ret=fd_read(kernel_fd, buffer, size );
__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;

}
int cap_write(iop_file_t * kernel_fd,char * buffer,int size  )
{
unsigned gpsav;
register int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);
ret=fd_write(kernel_fd, buffer, size );

__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;

}
int cap_lseek(iop_file_t * kernel_fd,int  offset,const int whence)
{
unsigned gpsav;
register int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);
ret=fd_lseek(kernel_fd,offset,whence);
__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;

}


int cap_close(iop_file_t *  kernel_fd)
{
unsigned gpsav;
register int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);
ret=fd_close(kernel_fd);
__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;
}


int cap_remove(iop_file_t *f,  char *name)
{
unsigned gpsav;
int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);

ret=fd_remove(f, name);


__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;
}

int cap_mkdir(iop_file_t *f, char *name)
{
unsigned gpsav;
int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);

ret=fd_mkdir(f, name);


__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;
}

int cap_rmdir(iop_file_t *f,  char *name)
{
unsigned gpsav;
int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);

ret=fd_rmdir(f, name);


__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;
}

int cap_dopen(iop_file_t *f,  char *name)
{
unsigned gpsav;
int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);

ret=fd_dopen(f, name);


__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;
}

int cap_dclose(iop_file_t *f)
{
unsigned gpsav;
int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);

ret=fd_dclose(f);


__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;
}

int cap_dread(iop_file_t *f, void *buf)
{
unsigned gpsav;
int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);

ret=fd_dread(f, buf);


__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;
}

int cap_getstat(iop_file_t *f, char *name, void *buf)
{
unsigned gpsav;
int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);

ret=fd_getstat(f, name, buf);


__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;
}

int cap_chstat(iop_file_t *f, char *name, void *buf, unsigned int statmask) 
{
unsigned gpsav;
int ret;
__asm __volatile__ ("
					sw $gp,0(%1)
					lw $gp,0(%0)
					"	
					:: "r"(_local), "r"(&gpsav)
					);

ret=fd_chstat(f, name, buf, statmask);


__asm __volatile__ ("
					lw $gp,0(%0)
					"
					::  "r"(&gpsav)
					);

return ret;
}


#define BIND_RPC_ID 0x500C0F1
static SifRpcDataQueue_t rpc_queue __attribute__((aligned(64)));
static SifRpcServerData_t rpc_server __attribute((aligned(64)));
static int _rpc_buffer[512] __attribute((aligned(64)));

int getFirst( void *buff) {
	WaitSema(sema_mutex);
	
	rname=buff;
	rpc_func=RPC_FUNC_GETFIRST;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}
int getNext( void *buff) {
	WaitSema(sema_mutex);
	
	rname=buff;
	rpc_func=RPC_FUNC_GETNEXT;
	SignalSema(sema_call);
	WaitSema(sema_ret);
	SignalSema(sema_mutex);
	return rret;
}
void *rpcCommandHandler(u32 command, void *buffer, int size)

{
	int* buf = (int*) buffer;
	int ret = 0;
//printf("RPC command %i\n",command);
	switch (command) {
		case 1: //getFirstDirentry
			ret  = getFirst(((char*) buffer) + 4); //reserve 4 bytes for returncode
			break;
		case 2: //getNextDirentry
			ret = getNext(((char*) buffer) + 4);
		 	break;
		/*
		case 3: //dumpContent
			printf("rpc called - dump disk content\n");
			dumpDiskContentFile(buf[0], buf[1], ((char*) buffer) + 8);
			break;
		case 4: //dump system info
			ret = fat_dumpSystemInfo(buf[0], buf[1]);
			break;*/
	}
	buf[0] = ret; //store return code
	return buffer;
}
void rpcMainThread(void* param)
{
	int ret=-1;
	int tid;

	SifInitRpc(0);


	

	tid = GetThreadId();

	SifSetRpcQueue(&rpc_queue, tid);
	SifRegisterRpc(&rpc_server, BIND_RPC_ID, (void *) rpcCommandHandler, (u8 *) &_rpc_buffer, 0, 0, &rpc_queue);

	SifRpcLoop(&rpc_queue);
}
int _start( int argc, char **argv)
{
	iop_sema_t scall;
	iop_sema_t sret;
	iop_sema_t smutex;
	iop_thread_t param;
	int fd,ret;
	int th;

	__asm __volatile__ ("sw $gp,0(%0) 
					   
					"	
					:: "r"(_local)
					);

	FlushDcache();

	//printf("vfat: FAT driver v0.1 initialising...\n");
	
	status=2;
	
	massInit(PartitionTypes,connect,disconnect);
	//printf("vfat: USB mass storage initialised\n");
	//printf("vfat: USB mass storage initialised\n");
	
	driver.name = "mass";
	driver.type = 16;
	driver.version = 1;
	driver.desc = "Read/Write FAT filesystem layer for USB Mass Storage";
	driver.ops = &functions;


	functions.init = cap_init;
	functions.deinit = dummy;
	functions.format = (f_io_format)dummy;
	functions.open = cap_open;
	functions.close = cap_close;
	functions.read = cap_read;
	functions.write = cap_write;
	functions.lseek = cap_lseek;
	functions.ioctl = dummy;
	functions.remove = cap_remove;
	functions.mkdir = cap_mkdir;
	functions.rmdir = cap_rmdir;
	functions.dopen =  cap_dopen;
	functions.dclose = cap_dclose;
	functions.dread = cap_dread;
	functions.getstat = cap_getstat;
	functions.chstat = cap_chstat;

	scall.initial = 0;
	scall.max = 1;
	scall.option = 0;
	scall.attr = 0;
	sema_call = CreateSema(&scall);
	
	sret.initial = 0;
	sret.max = 1;
	sret.option = 0;
	sret.attr = 0;
	sema_ret = CreateSema(&sret);
	
	smutex.initial = 1;
	smutex.max = 1;
	smutex.option = 0;
	smutex.attr = 0;
	sema_mutex = CreateSema(&smutex);

	param.attr         = TH_C;
	param.thread     = vfatRpcThread;
	param.priority = HI_PRIO;
	param.stacksize    = 0x2000; //0x8000;
	param.option      = 0;

	
	rpc_tid = CreateThread(&param);
	if (rpc_tid > 0) {
		StartThread(rpc_tid,0);
		WakeupThread(rpc_tid);
	} else  {
		//printf("vfat: Init error\n");
		return 0;
	}

	

	

        // now install the fileio driver
	DelDrv( "mass" );
	AddDrv( &driver );

	//printf("vfat: FAT driver init OK\n");
	/*create thread*/
	param.attr         = TH_C;
	param.thread     = rpcMainThread;
	param.priority = 40;
	param.stacksize    = 0x800;
	param.option      = 0;
	th = CreateThread(&param);
	if (th > 0) {
		StartThread(th,0);
		return 0;
	} else  {
		return 0;
	}

	return 0;

}


