/**************************************************************************
 * FreeDOS 32 FAT Driver                                                  *
 * by Salvo Isaja                                                         *
 *                                                                        *
 * Copyright (C) 2001-2003, Salvatore Isaja                               *
 *                                                                        *
 * This is "support.c" - Small generic support functions                  *
 *                                                                        *
 *                                                                        *
 * This file is part of the FreeDOS 32 FAT Driver.                        *
 *                                                                        *
 * The FreeDOS 32 FAT Driver is free software; you can redistribute it    *
 * and/or modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2 of the  *
 * License, or (at your option) any later version.                        *
 *                                                                        *
 * The FreeDOS 32 FAT Driver is distributed in the hope that it will be   *
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 * GNU General Public License for more details.                           *
 *                                                                        *
 * You should have received a copy of the GNU General Public License      *
 * along with the FreeDOS 32 FAT Driver; see the file COPYING;            *
 * if not, write to the Free Software Foundation, Inc.,                   *
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA                *
 **************************************************************************/

//ADDED lots of junk stuff from FD32 that the FAT driver depends on, mostly here.
 
#include "fat.h"
#include "xunicode.h"
#include "cdvd.h"

#ifdef DEBUG
 #define LOG_PRINTF(s) fd32_log_printf s
#else
 #define LOG_PRINTF(s)
#endif

int oemcp_to_utf8(char *Source, UTF8 *Dest);
int utf8_to_oemcp(UTF8 *Source, int SourceSize, char *Dest,   int DestSize);
int oemcp_skipchar(char *Dest);

int fd32_expand_fcb_name(char *Dest, BYTE *Source);

/* The sector number of the first sector of a valid data cluster N, where: */
/* - the first valid data cluster is 2 (cluster 0 and 1 are reserved)      */
/* - the last valid data cluster is Vol.DataClusters + 1                   */
/* - the number of clusters (including reserved) is Vol.DataClusters + 2   */
/* Returns 0 if the N is out of range of valid clusters.                   */
/* Called by open_existing (open.c), fat_creat (creat.c),                  */
/* fat_read (readwrit.c) and fat_write (readwrit.c).                       */
DWORD fat_first_sector_of_cluster(DWORD N)
{
  if ((N < 2) || (N > V->DataClusters + 1)) return 0;
  return ((N - 2) * V->Bpb.BPB_SecPerClus) + V->FirstDataSector;
}

#define FROMBCD(i) ((i&0xf)+(i>>4)*10)

/* Gets the current date and time in standard DOS format:          */
/*  Time bits: 15-11 hours (0-23), 10-5 minutes, 4-0 seconds/2     */
/*  Date bits: 15-9 year-1980, 8-5 month, 4-0 day                  */
/*  Hund: hundredths of second past the time in Time (0-199).      */
/* Pointers can be NULL.                                           */
/* Called by fat_creat (creat.c), fat_read, truncate_or_extend     */
/* and fat_write (readwrit.c).                                     */
void fat_timestamps(WORD *Time, WORD *Date, BYTE *Hund)
{
  CdCLOCK rtc;
  WORD time=0;
  WORD date=0;
  unsigned int year;
  
  CdReadClock(&rtc);
  
  time += (FROMBCD(rtc.second)/2)<<0;
  time += (FROMBCD(rtc.minute))<<5;
  time += (FROMBCD(rtc.hour))<<11;
  date += (FROMBCD(rtc.day))<<0;
  date += (FROMBCD(rtc.month))<<5;

  year  = FROMBCD(rtc.year)+20;
  
  date += year<<9;

  if(Time) *Time=time;
  if(Hund) *Hund=0;
  if(Date) *Date=date;
}


/* Moves the file pointer (the target position for read and write  */
/* operations) for a file, according to the origin:                */
/*  0 - the file pointer is moved to the offset specified          */
/*  1 - the file pointer is moved relative to its current position */
/*  2 - the file pointer is moved from the end of the file (note   */
/*      that this Origin is not valid for directories, since the   */
/*      FileSize for a directory is always zero).                  */
/* On success, returns 0 and, if Result is not NULL, fills Result  */
/* with the new absolute target position.                          */
/* This is a public driver function.                               */
int fat_lseek(tFile *F, long long int *Offset, int Origin)
{
  if(!fat_checkmount()) {
    LOG_PRINTF(("FAT: Mounting failed, aborting.\n"));
    return FD32_ENODEV;
  }
  
  switch (Origin)
  {
    case FD32_SEEKSET : F->TargetPos = *Offset; break;
    case FD32_SEEKCUR : F->TargetPos += *Offset; break;
    case FD32_SEEKEND : if (F->DirEntry.Attr & FD32_ADIR) return FD32_EINVAL;
                        F->TargetPos = F->DirEntry.FileSize + *Offset;
                        break;
    default           : return FD32_EINVAL;
  }
  *Offset = F->TargetPos;
  return 0;
}


/* Gets file system informations.                             */
/* Returns 0 on success, or a negative error code on failure. */
int fat_get_fsinfo(fd32_fs_info_t *Fsi)
{
  if (Fsi->Size < sizeof(fd32_fs_info_t)) return FD32_EFORMAT;
  #ifdef FATLFN
  Fsi->Flags   = FD32_FSICASEPRES | FD32_FSIUNICODE | FD32_FSILFN;
  Fsi->NameMax = FD32_LFNMAX;
  Fsi->PathMax = FD32_LFNPMAX;
  #else
  Fsi->Flags   = 0;
  Fsi->NameMax = FD32_SFNMAX;
  Fsi->PathMax = FD32_SFNPMAX;
  #endif
  strncpy(Fsi->FSName, "FAT", Fsi->FSNameSize - 1);
  return 0;
}


/* Gets allocation informations on a FAT volume.              */
/* Returns 0 on success, or a negative error code on failure. */
int fat_get_fsfree(fd32_getfsfree_t *F)
{
  #ifdef FATREMOVABLE
  int      Res;
  #endif
  
  if(!fat_checkmount()) {
    LOG_PRINTF(("FAT: Mounting failed, aborting.\n"));
    return FD32_ENODEV;
  }
  if (F->Size < sizeof(fd32_getfsfree_t)) return FD32_EFORMAT;
  if (V->VolSig != FAT_VOLSIG) return FD32_ENODEV;
  #ifdef FATREMOVABLE
  if ((Res = fat_mediachange(V)) < 0) return Res;
  #endif
  F->SecPerClus  = V->Bpb.BPB_SecPerClus;
  F->BytesPerSec = V->Bpb.BPB_BytsPerSec;
  F->AvailClus   = V->FSI_Free_Count;
  F->TotalClus   = V->DataClusters;
  return 0;
}


#if 0
/* The bytes actually allocated for the file F (usually > FileSize) */
static inline DWORD file_occupation_in_bytes(tFile *F)
{
  return clusters_amount(F->FileSize, &F->Volume->Bpb)
         * F->Volume->Bpb.BPB_BytsPerSec
         * F->Volume->Bpb.BPB_SecPerClus;
}
#endif


/* Given a file name Source in UTF-8, checks if it's valid */
/* and returns in Dest the file name in FCB format.        */
/* On success, returns 0 if no wildcards are present, or a */
/* positive number if '?' wildcards are present in Dest.   */
/* On failure, returns a negative error code.              */
static char  SourceU[FD32_LFNPMAX];
int fd32_build_fcb_name(BYTE *Dest, char *Source)
{
  int   WildCards = 0;
//  char  SourceU[FD32_LFNPMAX];
  int   Res;
  int   k;

  /* Name and extension have to be padded with spaces */
  memset(Dest, 0x20, 11);
  
  /* Build ".          " and "..         " if Source is "." or ".." */
  if ((strcmp(Source, ".") == 0) || (strcmp(Source, "..") == 0))
  {
    for (; *Source; Source++, Dest++) *Dest = *Source;
    return 0;
  }

  if ((Res = utf8_strupr(SourceU, Source)) < 0) return FD32_EUTF8;
  for (k = 0; (SourceU[k] != '.') && SourceU[k]; k++);
  utf8_to_oemcp(SourceU, SourceU[k] ? k : -1, Dest, 8);
  if (SourceU[k]) utf8_to_oemcp(&SourceU[k + 1], -1, &Dest[8], 3);

  if (Dest[0] == ' ') return FD32_EFORMAT;
  if (Dest[0] == 0xE5) Dest[0] = 0x05;
  for (k = 0; k < 11;)
  {
    if (Dest[k] < 0x20) return FD32_EFORMAT;
    switch (Dest[k])
    {
      case '"': case '+': case ',': case '.': case '/': case ':': case ';':
      case '<': case '=': case '>': case '[': case '\\': case ']':  case '|':
        return FD32_EFORMAT;
      case '?': WildCards = 1;
                k++;
                break;
      case '*': WildCards = 1;
                if (k < 8) for (; k < 8; k++) Dest[k] = '?';
                else for (; k < 11; k++) Dest[k] = '?';
                break;
      default : k += oemcp_skipchar(&Dest[k]);
    }
  }
  return WildCards;
}

/* Generates a valid 8.3 file name from a valid long file name. */
/* The generated short name has not a numeric tail.             */
/* Returns 0 on success, or a negative error code on failure.   */
static char  Purified[FD32_LFNPMAX]; /* A long name without invalid 8.3 characters */
int fd32_gen_short_fname(char *Dest, char *Source, DWORD Flags)
{
  BYTE  ShortName[11] = "           ";
//  char  Purified[FD32_LFNPMAX]; /* A long name without invalid 8.3 characters */
  char *DotPos = NULL; /* Position of the last embedded dot in Source */
  char *s;
  int   Res = 0;

  /* Find the last embedded dot, if present */
  if (!(*Source)) return FD32_EFORMAT;
  for (s = Source + 1; *s; s++)
    if ((*s == '.') && (*(s-1) != '.') && (*(s+1) != '.') && *(s+1))
      DotPos = s;

  /* Convert all characters of Source that are invalid for short names */
  for (s = Purified; *Source;)
    if (!(*Source & 0x80))
    {
      if ((*Source >= 'a') && (*Source <= 'z'))
      {
        *s++ = *Source + 'A' - 'a';
        Res |= FD32_GENSFN_CASE_CHANGED;
      }
      else if (*Source < 0x20) return FD32_EFORMAT;
      else switch (*Source)
      {
        /* Spaces and periods must be removed */
        case ' ': break;
        case '.': if (Source == DotPos) DotPos = s, *s++ = '.';
                                   else Res |= FD32_GENSFN_WAS_INVALID;
                  break;
        /* + , ; = [ ] are valid for LFN but not for short names */
        case '+': case ',': case ';': case '=': case '[': case ']':
          *s++ = '_'; Res |= FD32_GENSFN_WAS_INVALID; break;
        /* Check for invalid LFN characters */
        case '"': case '*' : case '/': case ':': case '<': case '>':
        case '?': case '\\': case '|':
          return FD32_EFORMAT;
        /* Return any other single-byte character unchanged */
        default : *s++ = *Source;
      }
      Source++;
    }
    else /* Process extended characters */
    {
      UTF32 Ch, upCh;
      int   Skip;

      if ((Skip = fd32_utf8to32(Source, &Ch)) < 0) return FD32_EUTF8;
      Source += Skip;
      upCh = unicode_toupper(Ch);
      if (upCh != Ch) Res |= FD32_GENSFN_CASE_CHANGED;
      if (upCh < 0x80) Res |= FD32_GENSFN_WAS_INVALID;
      s += fd32_utf32to8(upCh, s);
    }
  *s = 0;

  /* Convert the Purified name to the OEM code page in FCB format */
  /* TODO: Must report WAS_INVALID if an extended char maps to ASCII! */
  if (utf8_to_oemcp(Purified, DotPos ? DotPos - Purified : -1, ShortName, 8))
    Res |= FD32_GENSFN_WAS_INVALID;
  if (DotPos) if (utf8_to_oemcp(DotPos + 1, -1, &ShortName[8], 3))
                Res |= FD32_GENSFN_WAS_INVALID;
  if (ShortName[0] == ' ') return FD32_EFORMAT;
  if (ShortName[0] == 0xE5) Dest[0] = (char) 0x05;

  /* Return the generated short name in the specified format */
  switch (Flags & FD32_GENSFN_FORMAT_MASK)
  {
    case FD32_GENSFN_FORMAT_FCB    : memcpy(Dest, ShortName, 11);
                                     return Res;
    case FD32_GENSFN_FORMAT_NORMAL : fd32_expand_fcb_name(Dest, ShortName);
                                     return Res;
    default                        : return FD32_EINVAL;
  }
}


// This doesn't belong here
/* Gets a UTF-8 short file name from an FCB file name. */
int fd32_expand_fcb_name(char *Dest, BYTE *Source)
{
  BYTE *NameEnd;
  BYTE *ExtEnd;
  char  Aux[13];
  BYTE *s = Source;
  char *a = Aux;

  /* Count padding spaces at the end of the name and the extension */
  for (NameEnd = Source + 7;  *NameEnd == ' '; NameEnd--);
  for (ExtEnd  = Source + 10; *ExtEnd  == ' '; ExtEnd--);

  /* Put the name, a dot and the extension in Aux */
  if (*s == 0x05) *a++ = (char) 0xE5, s++;
  for (; s <= NameEnd; *a++ = (char) *s++);
  if (Source + 8 <= ExtEnd) *a++ = '.';
  for (s = Source + 8; s <= ExtEnd; *a++ = (char) *s++);
  *a = 0;

  /* And finally convert Aux from the OEM code page to UTF-8 */
  return oemcp_to_utf8(Aux, Dest);
}

/* Compares two file names in FCB format. Name2 may contain '?' wildcards. */
/* Returns zero if the names match, or nonzero if they don't match.        */
/* TODO: Works only on single byte character sets!                       */
int fd32_compare_fcb_names(BYTE *Name1, BYTE *Name2)
{
  int k;
  for (k = 0; k < 11; k++)
  {
    if (Name2[k] == '?') continue;
    if (Name1[k] != Name2[k]) return -1;
  }
  return 0;
}

#define SPACE_PAD     4  /* Padding possibility       */
#define ZERO_PAD      8
#define LEFT_PAD      16
#define ADD_PLUS      2  /* Add + for positive/floats */


char todigit(int c)
{
    if (c >= 0 && c <= 9) return(c+'0');
    else if (c >= 10 && c <= 15) return(c + 'A' - 10);
    else return(c);    
}

unsigned ucvt(unsigned long v,char *buffer,int base,int width,int flag)
{
    register int i = 0,j;
    unsigned ret = 0,abs_base;
    unsigned long abs_v;
    char tmp[12];
    /* Extract the less significant digit */
    /* Put it into temporary buffer       */
    /* It has to be local to have 	  */
    /* reentrant functions		  */
    /*
    MG: fix to handle zero correctly
    if (v == 0) {
	*buffer++ = '0';
	*buffer = 0;
	return(1);
    }
    */
    
    /* MG: is this required? */
    /* the vsprintf() function seems to call this procedure with */
    /* this flag inverted */
    flag ^= LEFT_PAD;
    
    abs_base = (base >= 0) ? base : -base;
    if (base < 0) abs_v = ((long)(v) < 0) ? -v : v;
    else abs_v = v;
    /* Process left space-padding flags */
    if (flag & ADD_PLUS || ((base < 0) && ((long)(v) < 0))) {
	ret++;
    }
    /* MG: fix to handle zero correctly */
    if (v == 0)
        tmp[i++]='0';
    else
        while (abs_v > 0) {
	    tmp[i++] = todigit(abs_v % abs_base);
	    abs_v = abs_v / abs_base;
        }
    ret += i;
    if ((flag & LEFT_PAD) && (flag & SPACE_PAD)) {
	j = ret;
	while (j < width) {
	    *buffer++ = ' ';
	    ret++;
	    j++;
	}
    }
    /* Add plus if wanted */
    if (base < 0) {
	if (((long)(v)) < 0) *buffer++ = '-';
        else if (flag & ADD_PLUS) *buffer++ = '+';
    } else if (flag & ADD_PLUS) *buffer++ = '+';
    /* Process left zero-padding flags */
    if ((flag & LEFT_PAD) && (flag & ZERO_PAD)) {
	j = ret;
	while (j < width) {
	    *buffer++ = '0';
	    ret++;
	    j++;
	}
    }
    /* Reverse temporary buffer into output buffer */
    /* If wanted left pad with zero/space; anyway only one at once is ok */
    for (j = i-1; j >= 0; j--) *buffer++ = tmp[j];
    if ((flag & (SPACE_PAD)) && !(flag & LEFT_PAD)) {
	/* If wanted, pad with space to specified width */
	j = ret;
	while (j < width) {
	    *buffer++ = ' ';
	    ret++;
	    j++;
	}
    }
    /* Null terminate the output buffer */
    *buffer = 0;
    return(ret);
}

unsigned dcvt(long v,char *buffer,int base,int width,int flag)
{
    return(ucvt((unsigned long)(v),buffer,-base,width,flag));
}
