﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Windows.Forms;

namespace PokémonConquestPasswordGenerator
{
    public partial class Form1 : Form
    {
        public string Version = "v1.1.1";

        public enum PasswordMode
        {
            Pokémon,
            Event,
            Invalid
        }
        public enum Pokémon
        {
            Eevee,
            Vaporeon,
            Jolteon,
            Flareon,
            Espeon,
            Umbreon,
            Leafeon,
            Glaceon,
            Ralts,
            Kirlia,
            Gardevoir,
            Gallade,
            Magikarp,
            Gyarados,
            Pichu,
            Pikachu,
            Raichu,
            Wooper,
            Quagsire,
            Igglybuff,
            Jigglypuff,
            Wigglytuff,
            Zubat,
            Golbat,
            Crobat,
            Starly,
            Staravia,
            Staraptor,
            Bidoof,
            Bibarel,
            Venipede,
            Whirlipede,
            Scolipede,
            Shinx,
            Luxio,
            Luxray,
            Litwick,
            Lampent,
            Chandelure,
            Roggenrola,
            Boldore,
            Gigalith,
            Petilil,
            Lilligant,
            Mareep,
            Flaaffy,
            Ampharos,
            Cottonee,
            Whimsicott,
            Riolu,
            Lucario,
            Chingling,
            Chimecho,
            Ekans,
            Arbok,
            Pineco,
            Forretress,
            Meowth,
            Persian,
            Spheal,
            Sealeo,
            Walrein,
            Gothita,
            Gothorita,
            Gothitelle,
            Sandile,
            Krokorok,
            Krookodile,
            Duskull,
            Dusclops,
            Dusknoir,
            Munna,
            Musharna,
            Blitzle,
            Zebstrika,
            Dratini,
            Dragonair,
            Dragonite,
            Larvitar,
            Pupitar,
            Tyranitar,
            Beldum,
            Metang,
            Metagross,
            Gible,
            Gabite,
            Garchomp,
            Croagunk,
            Toxicroak,
            Deino,
            Zweilous,
            Hydreigon,
            Snorunt,
            Glalie,
            Froslass,
            Minccino,
            Cinccino,
            Machop,
            Machoke,
            Machamp,
            Timburr,
            Gurdurr,
            Conkeldurr,
            Cubchoo,
            Beartic,
            Oshawott,
            Dewott,
            Samurott,
            Charmander,
            Charmeleon,
            Charizard,
            Gastly,
            Haunter,
            Gengar,
            Chimchar,
            Monferno,
            Infernape,
            Snivy,
            Servine,
            Serperior,
            Tepig,
            Pignite,
            Emboar,
            Sewaddle,
            Swadloon,
            Leavanny,
            Abra,
            Kadabra,
            Alakazam,
            Treecko,
            Grovyle,
            Sceptile,
            Piplup,
            Prinplup,
            Empoleon,
            Pansage,
            Simisage,
            Pansear,
            Simisear,
            Panpour,
            Simipour,
            Darumaka,
            Darmanitan,
            Axew,
            Fraxure,
            Haxorus,
            Joltik,
            Galvantula,
            Aron,
            Lairon,
            Aggron,
            Drilbur,
            Excadrill,
            Zorua,
            Zoroark,
            Skorupi,
            Drapion,
            Pawniard,
            Bisharp,
            Rhyhorn,
            Rhydon,
            Rhyperior,
            Shieldon,
            Bastiodon,
            Scraggy,
            Scrafty,
            Drifloon,
            Drifblim,
            Rufflet,
            Braviary,
            Anorith,
            Armaldo,
            Larvesta,
            Volcarona,
            Onix,
            Steelix,
            Beedrill,
            Munchlax,
            Snorlax,
            Emolga,
            Sneasel,
            Weavile,
            Misdreavus,
            Mismagius,
            Audino,
            Carnivine,
            Spiritomb,
            Scyther,
            Scizor,
            Lapras,
            Terrakion,
            Articuno,
            Registeel,
            Groudon,
            Dialga,
            Mewtwo,
            Reshiram,
            Zekrom,
            Arceus,
            Rayquaza
        }
        public enum PasswordRegion
        {
            NorthAmerica,
            Japan
        }

        static char[] NAPasswordChars = "RJ8XB2cCmkZnDVqExFgGHraK3LwMN1ApW5iPUQzuS4TtvYb6de9fhjsy7".ToCharArray();
        static char[] JPPasswordChars = "カイドタホ０トハミロオ２キセテナフヘマルメリレビエゼボクネガズサウヒノ６ニチ４ダケゴグバデコモヨ７ギ１ゲ９ヲ３ザ５ヤラブゾム８アヂベ".ToCharArray();
        static string[] NumberChars = new string[] { "0０Oo", "1１Il|", "2２", "3３", "4４", "5５", "6６", "7７", "8８", "9９" };  // Correct chars in same order as region, followed by incorrect chars
        static string[] JPSimilarChars = new string[] { "アァｧｱ", "イィｨｲ", "ウゥｩｳ", "エェｪｴ", "オォｫｵ", "ヤャｬﾔ", "ヨョｮﾖ", "カヵｶ", "キｷ", "クｸ", "ケヶｹ", "コｺ", "サｻ", "セｾ", "タﾀ", "チﾁ", "テﾃ", "トﾄ", "ナﾅ", "ニﾆ", "ネﾈ", "ノﾉ", "ハﾊ", "ヒﾋと", "フﾌ", "ヘﾍへ", "ホﾎ", "マﾏ", "ミﾐ", "ムﾑ", "メﾒ", "モﾓ", "ラﾗ", "リﾘり", "ルﾙ", "レﾚ", "ロﾛ", "ヲｦ" };
        static int[] NAScramblePatterns = { 0x012453, 0x302541, 0x352041, 0x342501, 0x542301, 0x132540, 0x512403, 0x142530, 0x032451, 0x102435, 0x142350, 0x402531, 0x402351, 0x542103, 0x052143, 0x312054, 0x342015, 0x402315, 0x432150, 0x412530, 0x312405, 0x312540, 0x142503, 0x352041, 0x432051, 0x352014, 0x152043, 0x412503, 0x102543, 0x302415, 0x032451, 0x012354 };
        static int[] JPScramblePatterns = { 0x302154, 0x142053, 0x342501, 0x452310, 0x542310, 0x452301, 0x132045, 0x452301, 0x512403, 0x142305, 0x142503, 0x302415, 0x312504, 0x142503, 0x432105, 0x532041, 0x352410, 0x012543, 0x012345, 0x042135, 0x542013, 0x302514, 0x142503, 0x052431, 0x142305, 0x412035, 0x452310, 0x512304, 0x042513, 0x152340, 0x132540, 0x302415 };
        static List<byte> NotAllowedPokémon = new List<byte>() { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x0A, 0x0B, 0x10, 0x15, 0x26, 0x29, 0x40, 0x46, 0x4C, 0x4D, 0x53, 0x56, 0x5B, 0x5E, 0x63, 0x66, 0x6A, 0x6B, 0x6D, 0x6E, 0x71, 0x73, 0x74, 0x76, 0x77, 0x79, 0x7A, 0x80, 0x82, 0x83, 0x85, 0x86, 0x88, 0x8A, 0x8C, 0x91, 0x96, 0xA1, 0xAB, 0xAD, 0xAF, 0xB2, 0xB7, 0xBA, 0xBC, 0xBE, 0xBF, 0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7 };

        RadioButtonGroup RegionRadioButtons = new RadioButtonGroup();
        Random Randomizer = new Random();
        PasswordRegion CurrentRegion = PasswordRegion.NorthAmerica;

        public struct Password
        {
            private byte _mode;
            private byte _flag;
            private byte _pokémon;
            private byte _scramblepattern;

            public PasswordMode Mode
            {
                get
                {
                    if (_mode >= 2)
                        return PasswordMode.Invalid;
                    else
                        return (PasswordMode)this._mode;
                }
                set
                {
                    if (value == PasswordMode.Invalid)
                        throw new ArgumentOutOfRangeException("Password cannot be invalid.");
                    else
                        this._mode = (byte)value;
                }
            }
            public int Flag
            {
                get
                {
                    return this._flag;
                }
                set
                {
                    switch (this.Mode)
                    {
                        case (PasswordMode.Event):
                            if (value > 5)
                                throw new ArgumentOutOfRangeException("Only 5 flags are available for Event Passwords.");
                            else
                                this._flag = (byte)value;
                            break;
                        case (PasswordMode.Pokémon):
                            if (value > 0x3F)
                                throw new ArgumentOutOfRangeException("Only 64 flags are available for Pokémon Passwords.");
                            else
                                this._flag = (byte)value;
                            break;
                        default:
                            if (value > 0x3F)
                                throw new ArgumentOutOfRangeException("Only 64 flags are available.");
                            else
                                this._flag = (byte)value;
                            break;
                    }
                }
            }
            public Pokémon Pokémon
            {
                get
                {
                    return (Pokémon)this._pokémon;
                }
                set
                {
                    if (NotAllowedPokémon.Contains((byte)value))
                        throw new InvalidEnumArgumentException("Pokémon " + value + " is restricted.");
                    this._pokémon = (byte)value;
                }
            }
            public int ScramblePattern
            {
                get
                {
                    return this._scramblepattern;
                }
                set
                {
                    if (value >= 32)
                        throw new ArgumentOutOfRangeException("Scramble pattern does not exist.");
                    this._scramblepattern = (byte)ScramblePattern;
                }
            }

            public Password(ref string PassString, PasswordRegion Region)
            {
                // Load correct region-specific password characters and scramble patterns
                char[] PasswordChars = new char[0];
                int[] ScramblePatterns = new int[0];
                int PassLength = 0;
                switch (Region)
                {
                    case PasswordRegion.Japan:
                        PasswordChars = JPPasswordChars;
                        ScramblePatterns = JPScramblePatterns;
                        PassLength = 8;
                        break;
                    case PasswordRegion.NorthAmerica:
                        PasswordChars = NAPasswordChars;
                        ScramblePatterns = NAScramblePatterns;
                        PassLength = 10;
                        break;
                    default:
                        throw new NotSupportedException("Region not supported.");
                }

                // Check password length
                if (PassString.Length != PassLength)
                    throw new InvalidPasswordException("Password is not " + PassLength + " characters.");

                // Replace similar characters
                char[] PassChars = PassString.ToCharArray();
                for (int n = 0; n < PassLength; n++)
                    foreach (string NumberCharString in NumberChars)
                        if (NumberCharString.Contains(PassChars[n]))
                            PassChars[n] = NumberCharString[(int)Region];
                if (Region == PasswordRegion.Japan)
                    for (int n = 0; n < PassLength; n++)
                        foreach (string SimilarCharString in JPSimilarChars)
                            if (SimilarCharString.Contains(PassChars[n]))
                                PassChars[n] = SimilarCharString[0];
                PassString = new string(PassChars);

                // Check password character usage
                foreach (char Character in PassString)
                    if (!PasswordChars.Contains(Character))
                        throw new InvalidPasswordException("Password contains illegal character '" + Character + "'.");

                // Convert password string to bytes
                byte[] PassBytes = new byte[PassLength];
                for (int n = 0; n < PassLength; n++)
                    PassBytes[n] = (byte)Array.IndexOf(PasswordChars, PassString[n]);

                // Decode password string bytes to get scrambled password bytes
                // Also, check primary checksum
                byte[] ScrambleBytes = new byte[6];
                switch (Region)
                {
                    case PasswordRegion.Japan:
                        for (int n = 0; n < 2; n++)
                        {
                            ScrambleBytes[n * 3] = (byte)((PassBytes[n * 4 + 3] & 3) * 0x42 + PassBytes[n * 4]);
                            ScrambleBytes[n * 3 + 1] = (byte)(((PassBytes[n * 4 + 3] >> 2) & 3) * 0x42 + PassBytes[n * 4 + 1]);
                            ScrambleBytes[n * 3 + 2] = (byte)(((PassBytes[n * 4 + 3] >> 4) & 3) * 0x42 + PassBytes[n * 4 + 2]);

                            // Japanese passwords don't have a primary checksum!
                        }
                        break;
                    case PasswordRegion.NorthAmerica:
                        bool ChecksumMatch = true;

                        for (int n = 0; n < 2; n++)
                        {
                            ScrambleBytes[n * 3] = (byte)((PassBytes[n * 5 + 3] & 7) * 0x39 + PassBytes[n * 5]);
                            ScrambleBytes[n * 3 + 1] = (byte)((PassBytes[n * 5 + 3] >> 3) * 0x39 + PassBytes[n * 5 + 1]);
                            ScrambleBytes[n * 3 + 2] = (byte)((PassBytes[n * 5 + 4] & 7) * 0x39 + PassBytes[n * 5 + 2]);

                            ulong PassChecksum = (ulong)(PassBytes[n * 5 + 4] >> 3);
                            ulong Mesh = (byte)((ScrambleBytes[n * 3] & 0xF) | ((ScrambleBytes[n * 3 + 1] & 0xF) << 4));
                            ulong ValidChecksum = (((Mesh * 0x1F7047DD) >> 0x20) + ((Mesh - ((Mesh * 0x1F7047DD) >> 0x20)) / 2)) >> 5;
                            if (PassChecksum != ValidChecksum)
                                ChecksumMatch = false;
                        }

                        if (!ChecksumMatch)
                            throw new InvalidPasswordException("Password primary checksum does not match.");
                        break;
                }

                // Unscramble password bytes
                if (((ScrambleBytes[2] & 0x3F) ^ 2) >= ScramblePatterns.Length)
                    throw new InvalidPasswordException("Password uses an invalid byte scramble pattern.");
                byte[] RawBytes = new byte[6];
                {
                    int ScramblePattern = ScramblePatterns[(ScrambleBytes[2] & 0x3F) ^ 2];
                    for (int n = 0; n < 6; n++)
                        RawBytes[n] = (byte)(ScrambleBytes[(ScramblePattern >> (4 * (5 - n))) & 0xF] ^ n);
                }

                // Extract password data
                this._mode = (byte)(RawBytes[0] & 0x3F);
                this._flag = (byte)(RawBytes[1] & 0x3F);
                this._pokémon = RawBytes[5];
                this._scramblepattern = (byte)(RawBytes[2] & 0x3F);

                // Check secondary checksum
                {
                    int PassChecksum2 = 0;
                    for (int n = 0; n < 5; n++)
                        PassChecksum2 |= (RawBytes[n] >> 6) << (n * 2);

                    if (PassChecksum2 != CalcChecksum2(RawBytes))
                        throw new InvalidPasswordException("Password secondary checksum does not match.");
                }

                // Check password integrity
                if (((RawBytes[2] & 0x3F) + 1 != (RawBytes[3] & 0x3F)) || ((RawBytes[2] & 0x3F) + 2 != (RawBytes[4] & 0x3F)))
                    throw new InvalidPasswordException("Password integrity not valid.");

                // Check if password mode-specific arguments are in range
                if ((RawBytes[0] & 0x3F) == 0)
                {
                    if (RawBytes[5] >= 0xC8)
                        throw new InvalidPasswordException("Password unlocks nonexistent Pokémon '0x" + RawBytes[5].ToString("X2") + "'.");

                    if (NotAllowedPokémon.Contains(RawBytes[5]))
                        throw new InvalidPasswordException("Password unlocks restricted Pokémon " + (Pokémon)RawBytes[5] + ".");
                }
                else if ((RawBytes[0] & 0x3F) == 1)
                {
                    if ((RawBytes[1] & 0x3F) >= 5)
                        throw new InvalidPasswordException("Password enables nonexistent event '0x" + (RawBytes[1] & 0x3F).ToString("X1") + "'.");
                }
                else
                    throw new InvalidPasswordException("Password mode not valid.");
            }
            public string GetPassword(PasswordRegion Region)
            {
                // Load correct region-specific password characters and scramble patterns
                char[] PasswordChars = new char[0];
                int[] ScramblePatterns = new int[0];
                int PassLength = 0;
                switch (Region)
                {
                    case PasswordRegion.Japan:
                        PasswordChars = JPPasswordChars;
                        ScramblePatterns = JPScramblePatterns;
                        PassLength = 8;
                        break;
                    case PasswordRegion.NorthAmerica:
                        PasswordChars = NAPasswordChars;
                        ScramblePatterns = NAScramblePatterns;
                        PassLength = 10;
                        break;
                    default:
                        throw new NotSupportedException("Region not supported.");
                }

                byte[] RawBytes = new byte[6];

                // Fill in data bytes
                RawBytes[0] = this._mode;
                RawBytes[1] = this._flag;
                RawBytes[2] = this._scramblepattern;
                RawBytes[3] = (byte)(this._scramblepattern + 1);
                RawBytes[4] = (byte)(this._scramblepattern + 2);
                RawBytes[5] = this._pokémon;

                // Set secondary checksum
                {
                    int ValidChecksum2 = CalcChecksum2(RawBytes);
                    for (int n = 0; n < 5; n++)
                        RawBytes[n] |= (byte)(((ValidChecksum2 >> (n * 2)) & 0x3) << 6);
                }

                // Shuffle password bytes
                byte[] ShuffleBytes = new byte[6];
                {
                    int ShufflePattern = ScramblePatterns[RawBytes[2] & 0x3F];
                    for (int n = 0; n < 6; n++)
                        ShuffleBytes[(ShufflePattern >> (4 * (5 - n))) & 0xF] = (byte)(RawBytes[n] ^ n);
                }

                // Encode password bytes and set primary checksum
                byte[] PassBytes = new byte[PassLength];
                switch (Region)
                {
                    case PasswordRegion.Japan:
                        for (int n = 0, m; n < 2; n++)
                            for (m = 0; m < 3; m++)
                            {
                                PassBytes[n * 4 + m] = (byte)(ShuffleBytes[n * 3 + m] % 0x42);
                                PassBytes[n * 4 + 3] |= (byte)((ShuffleBytes[n * 3 + m] / 0x42) << (2 * m));

                                // Japanese passwords don't have a primary checksum!
                            }
                        break;
                    case PasswordRegion.NorthAmerica:
                        for (int n = 0; n < 2; n++)
                        {
                            PassBytes[n * 5] = (byte)(ShuffleBytes[n * 3] % 0x39);
                            PassBytes[n * 5 + 3] |= (byte)(ShuffleBytes[n * 3] / 0x39);
                            PassBytes[n * 5 + 1] = (byte)(ShuffleBytes[n * 3 + 1] % 0x39);
                            PassBytes[n * 5 + 3] |= (byte)((ShuffleBytes[n * 3 + 1] / 0x39) << 3);
                            PassBytes[n * 5 + 2] = (byte)(ShuffleBytes[n * 3 + 2] % 0x39);
                            PassBytes[n * 5 + 4] |= (byte)(ShuffleBytes[n * 3 + 2] / 0x39);

                            ulong Mesh = (byte)((ShuffleBytes[n * 3] & 0xF) | ((ShuffleBytes[n * 3 + 1] & 0xF) << 4));
                            ulong ValidChecksum = (((Mesh * 0x1F7047DD) >> 0x20) + ((Mesh - ((Mesh * 0x1F7047DD) >> 0x20)) / 2)) >> 5;
                            PassBytes[n * 5 + 4] |= (byte)((ValidChecksum & 0x7) << 3);
                        }
                        break;
                }

                // Convert password bytes to string
                string PassString = "";
                for (int n = 0; n < PassLength; n++)
                    PassString += PasswordChars[PassBytes[n]];

                return PassString;
            }

            private int CalcChecksum2(byte[] RawBytes)
            {
                int ValidChecksum2 = RawBytes[5];
                for (int n = 0; n < 5; n++)
                    ValidChecksum2 += RawBytes[n] & 0x3F;
                return ValidChecksum2 &= 0x3FF;
            }
        }
        public struct RadioButtonGroup
        {
            public RadioButton[] RadioButtons;

            public int SelectedIndex
            {
                get
                {
                    for (int n = 0; n < this.RadioButtons.Length; n++)
                        if (RadioButtons[n].Checked)
                            return n;
                    return -1;
                }
                set
                {
                    RadioButtons[value].Checked = true;
                }
            }

            public RadioButtonGroup(RadioButton[] RadioButtons)
            {
                this.RadioButtons = RadioButtons;
            }
        }

        public Form1()
        {
            NotAllowedPokémon.Sort();
            List<string> PokémonNames = Enum.GetNames(typeof(Pokémon)).ToList();
            for (int n = NotAllowedPokémon.Count - 1; n >= 0; n--)
                PokémonNames.RemoveAt(NotAllowedPokémon[n]);
            PokémonNames.Sort();

            InitializeComponent();
            RegionRadioButtons = new RadioButtonGroup(new RadioButton[] { radioButton1, radioButton2 });

            comboBox2.Items.AddRange(PokémonNames.ToArray());
        }

        private void textBox1_TextChanged(object sender, EventArgs e)
        {
            button1.Enabled = textBox1.TextLength >= 8;
        }
        private void button1_Click(object sender, EventArgs e)
        {
            Password Password = new Password();
            PasswordRegion PasswordRegion = (PasswordRegion)RegionRadioButtons.SelectedIndex;
            string PassString = textBox1.Text;

            // Try currently selected region
            InvalidPasswordException PassException = new InvalidPasswordException();
            try
            {
                Password = new Password(ref PassString, PasswordRegion);
            }
            catch (InvalidPasswordException Exception)
            {
                // Try autodetecting
                if (TryAutoDetect(ref PassString, out PasswordRegion, out Password))
                    RegionRadioButtons.SelectedIndex = (int)PasswordRegion;
                else
                {
                    MessageBox.Show(Exception.Message, "Invalid password");
                    return;
                }
            }
            catch (Exception Exception)
            {
                MessageBox.Show(Exception.Message, "Unknown error");
                return;
            }

            switch (Password.Mode)
            {
                case (PasswordMode.Pokémon):
                    comboBox1.SelectedIndex = (int)Password.Mode;

                    comboBox3.Visible = false;
                    comboBox3.Enabled = false;

                    label3.Enabled = true;
                    numericUpDown1.Enabled = true;
                    label3.Visible = true;
                    numericUpDown1.Value = Password.Flag;
                    numericUpDown1.Visible = true;

                    label2.Enabled = true;
                    comboBox2.Enabled = true;
                    label2.Visible = true;
                    comboBox2.Visible = true;
                    comboBox2.SelectedIndex = comboBox2.Items.IndexOf(Password.Pokémon.ToString());
                    break;
                case (PasswordMode.Event):
                    comboBox1.SelectedIndex = (int)Password.Mode;

                    label3.Visible = false;
                    numericUpDown1.Visible = false;
                    label3.Enabled = false;
                    numericUpDown1.Enabled = false;

                    label2.Visible = false;
                    comboBox2.Visible = false;
                    label2.Enabled = false;
                    comboBox2.Enabled = false;

                    comboBox3.Enabled = true;
                    comboBox3.SelectedIndex = Password.Flag;
                    comboBox3.Visible = true;
                    break;
                default:
                    comboBox1.SelectedIndex = -1;

                    label3.Visible = false;
                    numericUpDown1.Visible = false;
                    label3.Enabled = false;
                    numericUpDown1.Enabled = false;

                    label2.Visible = false;
                    comboBox2.Visible = false;
                    label2.Enabled = false;
                    comboBox2.Enabled = false;

                    comboBox3.Visible = false;
                    comboBox3.Enabled = false;
                    break;
            }
            textBox1.Text = PassString;
            MessageBox.Show(textBox1.Text + " is a valid password!", "Password recognized");
        }
        private bool TryAutoDetect(ref string PassString, out PasswordRegion PasswordRegion, out Password Password)
        {
            string[] RegionNames = Enum.GetNames(typeof(PasswordRegion));
            foreach (string RegionName in RegionNames)
            {
                PasswordRegion = (PasswordRegion)Enum.Parse(typeof(PasswordRegion), RegionName);

                try
                {
                    Password = new Password(ref PassString, PasswordRegion);
                    return true;
                }
                catch
                {
                }
            }

            PasswordRegion = new PasswordRegion();
            Password = new Password();
            return false;
        }

        private void comboBox1_SelectedIndexChanged(object sender, EventArgs e)
        {
            switch ((PasswordMode)comboBox1.SelectedIndex)
            {
                case PasswordMode.Pokémon:
                    comboBox3.Visible = false;
                    comboBox3.Enabled = false;

                    label3.Enabled = true;
                    numericUpDown1.Enabled = true;
                    label3.Visible = true;
                    numericUpDown1.Visible = true;

                    label2.Enabled = true;
                    comboBox2.Enabled = true;
                    label2.Visible = true;
                    comboBox2.Visible = true;
                    break;
                case PasswordMode.Event:
                    label3.Visible = false;
                    numericUpDown1.Visible = false;
                    label3.Enabled = false;
                    numericUpDown1.Enabled = false;

                    label2.Visible = false;
                    comboBox2.Visible = false;
                    label2.Enabled = false;
                    comboBox2.Enabled = false;

                    comboBox3.Enabled = true;
                    comboBox3.Visible = true;
                    break;
                default:
                    label3.Visible = false;
                    numericUpDown1.Visible = false;
                    label3.Enabled = false;
                    numericUpDown1.Enabled = false;

                    label2.Visible = false;
                    comboBox2.Visible = false;
                    label2.Enabled = false;
                    comboBox2.Enabled = false;

                    comboBox3.Visible = false;
                    comboBox3.Enabled = false;
                    break;
            }

            CheckReady(null, null);
        }
        private void CheckReady(object sender, EventArgs e)
        {
            if (comboBox1.SelectedIndex == (int)PasswordMode.Pokémon)
                button2.Enabled = Enum.IsDefined(typeof(Pokémon), comboBox2.Text);
            else if (comboBox1.SelectedIndex == (int)PasswordMode.Event)
                button2.Enabled = comboBox3.SelectedIndex != -1;
            else
                button2.Enabled = false;
        }
        private void button2_Click(object sender, EventArgs e)
        {
            Password Password = new Password();
            Password.Mode = (PasswordMode)comboBox1.SelectedIndex;

            switch (Password.Mode)
            {
                case PasswordMode.Pokémon:
                    Password.Flag = (int)numericUpDown1.Value;
                    Password.Pokémon = (Pokémon)Enum.Parse(typeof(Pokémon), comboBox2.Text);
                    break;
                case PasswordMode.Event:
                    Password.Flag = comboBox3.SelectedIndex;
                    break;
            }
            Password.ScramblePattern = Randomizer.Next(32);

            textBox1.Text = Password.GetPassword((PasswordRegion)RegionRadioButtons.SelectedIndex);
        }

        private void Form1_Shown(object sender, EventArgs e)
        {
            if (MessageBox.Show("Passwords generated with this utility should be clearly labeled as such.\nPlease do NOT submit passwords generated with this to sites like GameFAQs.", "PassConqueror " + Version, MessageBoxButtons.OKCancel, MessageBoxIcon.Warning) == DialogResult.Cancel)
                this.Close();
            else
                this.Enabled = true;
        }

        private void TryPortPassword(object sender, EventArgs e)
        {
            try
            {
                // Try porting the current password
                string PassString = textBox1.Text;
                Password Password = new Password(ref PassString, CurrentRegion);
                textBox1.Text = Password.GetPassword((PasswordRegion)RegionRadioButtons.SelectedIndex);
            }
            catch
            {
                // If it didn't work, do nothing
            }
            CurrentRegion = (PasswordRegion)RegionRadioButtons.SelectedIndex;
        }
    }

    public class InvalidPasswordException : Exception
    {
        public InvalidPasswordException() { }
        public InvalidPasswordException(string message) : base(message) { }
        public InvalidPasswordException(string message, Exception inner) : base(message, inner) { }
        protected InvalidPasswordException(
          System.Runtime.Serialization.SerializationInfo info,
          System.Runtime.Serialization.StreamingContext context)
            : base(info, context) { }
    }
}