﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace CheatDBChangelogGenerator
{
    public partial class MainWindow : Form
    {
        public MainWindow()
        {
            InitializeComponent();
        }

        string originalFilePath;
        string newFilePath;

        private void loadOriginal(object sender, EventArgs e)
        {
            if (openUsrcheatDialog.ShowDialog() != DialogResult.OK)
                return;

            originalFilePath = openUsrcheatDialog.FileName;
            textBox1.Text = originalFilePath;
            textBox1.SelectionStart = textBox1.Text.Length - 1;

            checkReady(null, null);
        }
        private void loadChanged(object sender, EventArgs e)
        {
            if (openUsrcheatDialog.ShowDialog() != DialogResult.OK)
                return;

            newFilePath = openUsrcheatDialog.FileName;
            textBox2.Text = newFilePath;
            textBox2.SelectionStart = textBox2.Text.Length - 1;

            checkReady(null, null);
        }

        private void generateChangelog(object sender, EventArgs e)
        {
            if (saveChangelogDialog.ShowDialog() != DialogResult.OK)
                return;

            List<byte[]> addedList = new List<byte[]>();
            List<byte[]> removedList = new List<byte[]>();
            List<byte[]> remainList = new List<byte[]>();
            List<byte[]> changedList = new List<byte[]>();
            List<byte[]> oldDupeList = new List<byte[]>();
            List<byte[]> newDupeList = new List<byte[]>();

            List<string> addedNames = new List<string>();
            List<string> removedNames = new List<string>();
            List<string> changedNames = new List<string>();
            List<string> newDupeNames = new List<string>();

            bool checkAdded = checkBox1.Checked;
            bool checkRemoved = checkBox3.Checked;
            bool checkChanged = checkBox2.Checked;

            Refresh();

            statusText("Opening databases...");
            FileStream originalFile = new FileStream(originalFilePath, FileMode.Open, FileAccess.Read);
            FileStream newFile = new FileStream(newFilePath, FileMode.Open, FileAccess.Read);
            StreamWriter changelog = new StreamWriter(saveChangelogDialog.FileName);    // create changelog file

            statusText("Loading game headers...");
            List<byte[]> originalList = generateGameHeaderList(originalFile);
            List<byte[]> newList = generateGameHeaderList(newFile);

            statusText("Checking for duplicate games...");                        // get rid of dupes, they mess everything up
            initProgress((originalList.Count + newList.Count) * 2);
            removeFromList(ref originalList, removeDupes(ref originalList));
            oldDupeList = removeDupes(ref newList);                                  // keep the newList dupeList for the changelog
            removeFromList(ref newList, oldDupeList);



            if (checkAdded || checkChanged)
            {
                statusText("Searching for added games...");
                addedList = findExclusiveHeaders(originalList, newList);        // find headers in originalList that are not in newList
            }
            if (checkRemoved || checkChanged)
            {
                statusText("Searching for removed games...");
                removedList = findExclusiveHeaders(newList, originalList);      // find headers in newList that are not in originalList
            }
            if (checkChanged)
            {
                statusText("Getting remaining games...");
                initProgress(originalList.Count + newList.Count);
                removeFromList(ref originalList, removedList);
                removeFromList(ref newList, addedList);

                statusText("Sorting game header lists...");
                originalList.Sort(compareHeaders);
                newList.Sort(compareHeaders);

                statusText("Searching for updated games...");
                changedList = findChangedCheats(originalList, originalFile, newList, newFile);
            }



            if (oldDupeList.Count > 0)
            {
                statusText("Loading duplicate game names...");
                newDupeNames = loadGameNames(newFile, oldDupeList);
            }
            if (checkAdded)
            {
                statusText("Loading added game names...");
                addedNames = loadGameNames(newFile, addedList);
            }
            if (checkRemoved)
            {
                statusText("Loading removed game names...");
                removedNames = loadGameNames(originalFile, removedList);
            }
            if (checkChanged)
            {
                statusText("Loading updated game names...");
                changedNames = loadGameNames(newFile, changedList);
            }


            if (newDupeNames.Count > 0)
            {
                statusText("Writing duplicate game names...");
                changelog.WriteLine("Duplicate games (" + newDupeNames.Count + "):");
                foreach (string name in newDupeNames)
                    changelog.WriteLine(name);
                changelog.WriteLine();
            }
            if (checkAdded)
            {
                statusText("Writing added game names...");
                changelog.WriteLine("New games added (" + addedNames.Count + "):");
                foreach (string name in addedNames)
                    changelog.WriteLine(name);
                if (checkRemoved || checkChanged)
                    changelog.WriteLine();      // if there's more coming, place a newline
            }
            if (checkRemoved)
            {
                statusText("Writing removed game names...");
                changelog.WriteLine("Removed games (" + removedNames.Count + "):");
                foreach (string name in removedNames)
                    changelog.WriteLine(name);
                if (checkChanged)
                    changelog.WriteLine();      // if there's more coming, place a newline
            }
            if (checkChanged)
            {
                statusText("Writing changed game names...");
                changelog.WriteLine("Updated games (" + changedNames.Count + "):");
                foreach (string name in changedNames)
                    changelog.WriteLine(name);
                // changed games are at the bottom, so no need for another WriteLine()
            }



            statusText("Finishing...");
            originalFile.Close();
            newFile.Close();
            changelog.Close();

            statusText("Done.");
            toolStripProgressBar1.Value = 0;
        }

        private List<byte[]> generateGameHeaderList(FileStream filestream)
        {
            List<byte[]> gameHeaderList = new List<byte[]>();
            byte[] gameHeader;

            BinaryReader binaryreader = new BinaryReader(filestream);
            filestream.Position = 0xC;
            filestream.Position = binaryreader.ReadInt32();
            while (true)
            {
                gameHeader = binaryreader.ReadBytes(12);

                if (gameHeader[0] == 0)
                    break;

                gameHeaderList.Add(gameHeader);
                filestream.Position += 4;
            }

            return gameHeaderList;
        }
        private List<byte[]> findExclusiveHeaders(List<byte[]> originalList, List<byte[]> newList)
        {
            List<byte[]> headerList = new List<byte[]>();

            initProgress(newList.Count);

            foreach (byte[] gameHeader in newList)
            {
                if (!originalList.Any(entry => entry.Take(8).SequenceEqual(gameHeader.Take(8))))
                    headerList.Add(gameHeader);
                incProgress();
            }

            return headerList;
        }
        private void removeFromList(ref List<byte[]> headerList, List<byte[]> removeList)
        {
            byte[] gameHeader;

            for (int index = 0; index < headerList.Count; index++)
            {
                gameHeader = headerList[index];
                foreach (byte[] removeHeader in removeList)
                    if (gameHeader.Take(8).SequenceEqual(removeHeader.Take(8)))     // if the current gameHeader matches one in the removeList
                        headerList.RemoveAt(index--);
                incProgress();
            }
        }
        private List<byte[]> findChangedCheats(List<byte[]> originalList, FileStream originalFile, List<byte[]> newList, FileStream newFile)
        {
            List<byte[]> headerList = new List<byte[]>();
            byte[] originalHeader;
            byte[] newHeader;
            byte[] originalData;
            byte[] newData;

            initProgress(originalList.Count);

            for (int i = 0; i < originalList.Count; i++)
            {
                originalHeader = originalList[i];
                newHeader = newList[i];
                originalData = loadCheatData(originalFile, BitConverter.ToInt32(originalHeader, 8));       // load all cheat data for the game from originalFile
                newData = loadCheatData(newFile, BitConverter.ToInt32(newHeader, 8));                      // load all cheat data for the game from newFile

                if (originalData.Length != newData.Length)
                    headerList.Add(newHeader);
                else
                    for (int b = 0; b < originalData.Length; b++)
                        if (originalData[b] != newData[b])
                        {
                            headerList.Add(newHeader);
                            break;
                        }

                incProgress();
            }

            return headerList;
        }
        private List<byte[]> removeDupes(ref List<byte[]> headerList)
        {
            List<byte[]> dupeList = new List<byte[]>();

            foreach (byte[] header1 in headerList)
            {
                foreach (byte[] header2 in headerList)
                    if (!header1.Equals(header2))                  // if not comparing the game to itself
                        if (compareHeaders(header1, header2) == 0) // if gameID is the same
                            dupeList.Add(header1);                 // add header1 to dupeList (header2 is added as a different header1 later)
                incProgress();
            }

            return dupeList;
        }
        private byte[] loadCheatData(FileStream file, int offset)
        {
            BinaryReader fileReader = new BinaryReader(file);

            int entryCount;
            int length = 0;
            int value;

            file.Position = offset;

            length += getNameLength(file);                       // add length of name
            length = (int)Math.Ceiling((decimal)length / 4) * 4; // round up to nearest 4
            file.Position = offset + length;                     // sync FileStream position with current length

            entryCount = fileReader.ReadInt16();                 // get amount of cheat/folder entries
            length += 0x24;                                      // length of settings, master code and stuff
            file.Position = offset + length;

            for (int i = 0; i < entryCount; i++)
            {
                value = fileReader.ReadInt32();
                length += 0x4;

                if (value >> 0x1C == 0x1)   // it's a folder
                {
                    length += getNameLength(file);          // name
                    length += getNameLength(file);          // description
                    length = (int)Math.Ceiling((decimal)length / 4) * 4; // round up to nearest 4
                    file.Position = offset + length;
                }
                else // it's a cheat
                    length += value * 4;

                file.Position = offset + length;
            }

            file.Position = offset;
            byte[] buffer = new byte[length];
            file.Read(buffer, 0, length);

            return buffer;
        }
        private string loadName(FileStream file)
        {
            int offset = (int)file.Position;
            int length = 0;
            while (file.ReadByte() != 0x00)
                length++;

            byte[] name = new byte[length];
            file.Position = offset;
            file.Read(name, 0, length);
            file.Position++;

            return Encoding.UTF8.GetString(name);
        }
        private int getNameLength(FileStream file)
        {
            int offset = (int)file.Position;
            int length = 1;
            while (file.ReadByte() != 0x00)
                length++;

            return length;
        }
        private List<string> loadGameNames(FileStream cheatDB, List<byte[]> headerList)
        {
            List<string> gameNames = new List<string>();
            string name;

            initProgress(headerList.Count);

            foreach (byte[] gameHeader in headerList)
            {
                cheatDB.Position = BitConverter.ToInt32(gameHeader, 8);
                name = "- " + loadName(cheatDB);
                if (checkBox4.Checked == true)
                    name += " [" + Encoding.UTF8.GetString(gameHeader, 0, 4) + "-" + BitConverter.ToUInt32(gameHeader, 4).ToString("X8") + "]";
                gameNames.Add(name);
                incProgress();
            }

            gameNames.Sort();
            return gameNames;
        }

        private void statusText(string text)
        {
            statusLabel.Text = text;
            statusStrip.Update();
            Refresh();
        }
        private void initProgress(int value)
        {
            toolStripProgressBar1.Value = 0;
            toolStripProgressBar1.Maximum = value;
        }
        private void incProgress()
        {
            toolStripProgressBar1.Value++;
        }
        private void checkReady(object sender, EventArgs e)
        {
            if (originalFilePath != null && newFilePath != null)
            {
                if (checkBox1.Checked == true || checkBox3.Checked == true || checkBox2.Checked == true)
                {
                    statusLabel.Text = "Ready.";
                    button2.Enabled = true;
                }
                else
                {
                    statusLabel.Text = "No changes selected.";
                    button2.Enabled = false;
                }
            }
        }
        private int compareHeaders(byte[] header1, byte[] header2)
        {
            ulong gameID1 = BitConverter.ToUInt64(header1, 0);
            ulong gameID2 = BitConverter.ToUInt64(header2, 0);
            return gameID1.CompareTo(gameID2);
        }
    }
}
