/**
 * DLDI GUI Patcher for Linux
 *
 * This program is an easy to use GUI to patch multiple files at once,
 * created with GTKmm and intended voor Linux users
 *
 * Created by Lucas van Dijk, using an modified version of dldi source,
 * by Michael Chisholm (Chishm)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * @author Lucas van Dijk
 * @version $Id: uimain.cpp 18 2007-06-21 09:31:49Z luckyphp $
 * @license http://www.opensource.org/licenses/gpl-license.php
 */

#include "../include/uimain.h"

/**
 * Constructor, inits our window, sets up cout redirection, and connects some signals
 * @param window_obj The C Window object
 * @param ui_xml A reference to the glade instance
 */
DLDIWindow::DLDIWindow(GtkWindow*& window_obj, const Glib::RefPtr<Gnome::Glade::Xml>& ui_xml) : Gtk::Window(window_obj), glade_ref(ui_xml), about_dlg(0)
{
	this -> show_all();

	// Set up our own stream buffer
	// First get the text buffer from the TextView
	Gtk::TextView * textview = 0;
	this -> glade_ref -> get_widget("txtLog", textview);

	this -> stream_buffer = new TextViewStreamBuffer(textview -> get_buffer());
	std::cout.rdbuf(dynamic_cast<std::streambuf *>(this -> stream_buffer));

	// Connect button signals
	this -> connect_button("btnAdd", DLDIWindow::BUTTON_ADD);
	this -> connect_button("btnDelete", DLDIWindow::BUTTON_REMOVE);
	this -> connect_button("btnClear", DLDIWindow::BUTTON_CLEAR);
	this -> connect_button("btnPatch", DLDIWindow::BUTTON_PATCH);
	this -> connect_button("btnBrowse", DLDIWindow::BUTTON_BROWSE);

	// Connect menu signals
	this -> connect_menu("menuAddFile", DLDIWindow::MENU_ADD);
	this -> connect_menu("menuPatch", DLDIWindow::MENU_PATCH);
	this -> connect_menu("menuQuit", DLDIWindow::MENU_QUIT);
	this -> connect_menu("menuInfo", DLDIWindow::MENU_INFO);

	// Setup listview & combo
	this -> initialize_list();
	this -> initialize_combo();

	// Load last folders
	try
	{
		XMLNode folders_xml = XMLNode::openFileHelper("folders.xml", "folders");
		this -> last_dldi_folder = std::string(folders_xml.getChildNode("dldi_files").getText());
		this -> last_nds_folder = std::string(folders_xml.getChildNode("nds_files").getText());
	}
	catch(...)
	{
		this -> last_dldi_folder = ".";
		this -> last_nds_folder = ".";
	}

	// Load DLDI files
	this -> load_dldi_files(this -> last_dldi_folder);
}

/**
 * Cleans up some vars
 */
DLDIWindow::~DLDIWindow()
{
	// Write last folders to XML file
	std::ofstream xml;
	xml.open("folders.xml", ios_base::out | ios_base::trunc);
	if(xml.is_open())
	{
		std::cerr << "Saving last folders...";
		xml << "<?xml version='1.0' encoding='utf-8'?>\n<folders>\n<nds_files>" << this -> last_nds_folder << "</nds_files>\n<dldi_files>" << this -> last_dldi_folder << "</dldi_files>\n</folders>" << std::endl;
		std::cerr << " Done" << std::endl;
	}

	xml.close();

	// Clean up our loaded DLDI File
	this -> clear_dldi_files();
	this -> clear_nds_files();

	// Clear the about dialog
	if(this -> about_dlg != 0)
	{
		delete this -> about_dlg;
	}
}

/**
 * Loads all .dldi files in current directory, and puts them in the combobox
 */
void DLDIWindow::load_dldi_files(std::string search_dir)
{
	// Clear previous entries
	this -> combo_model -> clear();
	this -> clear_dldi_files();

	if(search_dir.substr(search_dir.size() - 1) != "/")
	{
		search_dir += '/';
	}

	Glib::Dir dir(search_dir);

	std::vector<std::string> files(dir.begin(), dir.end());

	// Get combo pointer
	Gtk::ComboBox * combo_drivers = 0;
	this -> glade_ref -> get_widget("cmbDriver", combo_drivers);

	Gtk::TreeModel::Row row;

	std::cout << "Loading dldi files in directory " << search_dir << std::endl;

	// Iterate through all files
	for(unsigned int i = 0; i < files.size(); i++)
	{
		// Get extension
		std::string::size_type dot_position = files[i].find_last_of('.');

		if(dot_position == std::string::npos)
		{
			// It's a directory, skip it
			continue;
		}

		std::string extension = files[i].substr(dot_position, files[i].length());

		// If it's al DLDI File...
		if(extension == ".dldi")
		{
			// .. load it and add it to the combobox
			DLDIFile * dldi_file = new DLDIFile(search_dir + files[i]);

			this -> dldi_files.push_back(dldi_file);

			std::cout << "Found DLDI File, Driver name: " << dldi_file -> get_driver_name() << std::endl;

			// Add to combo
			row = *(this -> combo_model -> append());
			row[this -> columns.text] = dldi_file -> get_driver_name();
		}
	}

	combo_drivers -> set_active(0);
	std::cout << endl;
}

/**
 * Initializes our Combo
 */
void DLDIWindow::initialize_combo()
{
	// Get combo pointer
	Gtk::ComboBox * combo_drivers = 0;
	this -> glade_ref -> get_widget("cmbDriver", combo_drivers);

	// Set model for combo box
	this -> combo_model = Gtk::ListStore::create(this -> columns);
	combo_drivers -> set_model(this -> combo_model);

	combo_drivers -> pack_start(this -> columns.text);
}

/**
 * Initializes our ListView which shows all files to patch
 *
 * This includes adding the columns
 */
void DLDIWindow::initialize_list()
{
	// Get listview pointer
	Gtk::TreeView * listview = 0;
	this -> glade_ref -> get_widget("listFiles", listview);

	// Set model for listview, use the same columns for the listview as the combo
	this -> listview_model = Gtk::ListStore::create(this -> listview_columns);
	listview -> set_model(this -> listview_model);

	listview -> append_column("Filename", this -> listview_columns.text);
	listview -> append_column("Current Driver", this -> listview_columns.driver);
}

/**
 * Connects the clicked signal for a given button
 * @param name The name of the button
 * @param id An unique ID for the button
 */
void DLDIWindow::connect_button(std::string name, Buttons id)
{
	// Try to get the button from the glade file
	Gtk::Button * button = 0;
	this -> glade_ref -> get_widget(name, button);

	// Connect the signal
	if(button)
	{
		button -> signal_clicked().connect(sigc::bind(sigc::mem_fun(*this, &DLDIWindow::OnButtonClick), id));
	}
	else
	{
		throw std::runtime_error("Widget not found");
	}
}

/**
 * Connects the activated signal for a given menu item
 * @param name The name of the menu item
 * @param id An unique ID for the menu
 */
void DLDIWindow::connect_menu(std::string name, Menus id)
{
	// Try to get the button from the glade file
	Gtk::MenuItem * menu = 0;
	this -> glade_ref -> get_widget(name, menu);

	// Connect the signal
	if(menu)
	{
		menu -> signal_activate().connect(sigc::bind(sigc::mem_fun(*this, &DLDIWindow::OnMenuClick), id));
	}
	else
	{
		throw std::runtime_error("Widget not found");
	}
}

void DLDIWindow::OnButtonClick(Buttons id)
{
	switch(id)
	{
		case DLDIWindow::BUTTON_ADD:
			this -> add_nds_file();
		break;
		case DLDIWindow::BUTTON_REMOVE:
		{
			// Get list view
			Gtk::TreeView * listview =  0;
			this -> glade_ref -> get_widget("listFiles", listview);

			Gtk::TreeModel::iterator iter = listview -> get_selection() -> get_selected();
			if(iter)
			{
				this -> listview_model -> erase(listview -> get_selection() -> get_selected());

				int row_index = ((int) iter) - 1;
				std::cerr << row_index << std::endl;
				delete this -> nds_files[row_index];

				this -> nds_files.erase(this -> nds_files.begin() + row_index);
			}
		}
		break;
		case DLDIWindow::BUTTON_CLEAR:
			this -> clear_nds_files();
			this -> listview_model -> clear();
		break;
		case DLDIWindow::BUTTON_PATCH:
			this -> patch();
		break;
		case DLDIWindow::BUTTON_BROWSE:
		{
			Gtk::FileChooserDialog dialog("Please choose a folder",
			Gtk::FILE_CHOOSER_ACTION_SELECT_FOLDER);
			dialog.set_transient_for(*this);

			//Add response buttons the the dialog:
			dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
			dialog.add_button("Select", Gtk::RESPONSE_OK);

			if(this -> last_dldi_folder != ".")
			{
				dialog.set_current_folder(this -> last_dldi_folder);
			}

			int result = dialog.run();

			if(result == Gtk::RESPONSE_OK)
			{
				this -> load_dldi_files(dialog.get_filename());
				this -> last_dldi_folder = dialog.get_filename();
			}
		}
		default:
			// Nothing
		break;
	}

}

void DLDIWindow::OnMenuClick(Menus id)
{
	switch(id)
	{
		case DLDIWindow::MENU_ADD:
			this -> add_nds_file();
		break;
		case DLDIWindow::MENU_QUIT:
			Gtk::Main::quit();
		break;
		case DLDIWindow::MENU_INFO:
		{
			if(this -> about_dlg == 0)
			{
				Glib::RefPtr<Gnome::Glade::Xml> dlg_glade_ref = Gnome::Glade::Xml::create("dldi-linux-gui.glade", "dlgAbout");

				dlg_glade_ref -> get_widget("dlgAbout", this -> about_dlg);
			}


			this -> about_dlg -> signal_response().connect(sigc::mem_fun(*this, &DLDIWindow::OnAboutDlgResponse));
			this -> about_dlg -> run();
		}
		break;
		case DLDIWindow::MENU_PATCH:
			this -> patch();
		break;
		default:
			// Nothing
		break;
	}
}

void DLDIWindow::OnAboutDlgResponse(int response)
{
	this -> about_dlg -> hide();
}

/**
 * Show a dfialog to add a .nds file
 *
 * This method shows a FileChooserDialog, and let's the user select the files
 * which he wants to patch. Then, it adds the selected files to the nds_files vector.
 */
void DLDIWindow::add_nds_file()
{
	Gtk::FileChooserDialog dialog("Select Nintendo DS binaries", Gtk::FILE_CHOOSER_ACTION_OPEN);
	dialog.set_transient_for(*this);

	// Add response buttons the the dialog:
	dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dialog.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_OK);

	// Add filters, so that only certain file types can be selected
	Gtk::FileFilter filter_nds;
	filter_nds.set_name("Nintendo DS Binaries");
	filter_nds.add_pattern("*.nds");
	filter_nds.add_pattern("*.nds.gba");
	dialog.add_filter(filter_nds);

	Gtk::FileFilter filter_any;
	filter_any.set_name("Any files");
	filter_any.add_pattern("*");
	dialog.add_filter(filter_any);

	if(this -> last_nds_folder != ".")
	{
		dialog.set_current_folder(this -> last_nds_folder);
	}

	// Show the dialog and wait for a user response:
	int result = dialog.run();

	switch(result)
	{
		case Gtk::RESPONSE_OK:
		{
			std::vector<std::string> filenames = dialog.get_filenames();

			Gtk::TreeModel::Row row;
			for(unsigned int i = 0; i < filenames.size(); i++)
			{
				// Load NdsFile
				NdsFile * nds_file = 0;
				try
				{
					nds_file = new NdsFile(filenames[i]);
					this -> nds_files.push_back(nds_file);
				}
				catch(const std::runtime_error e)
				{
					this -> show_dialog(Glib::ustring(e.what()));
					continue;
				}

				// get filename
				std::string filename = filenames[i].substr(filenames[i].find_last_of('/') + 1, filenames[i].size());

				row = *(this -> listview_model -> append());
				row[this -> listview_columns.text] = filename;
				row[this -> listview_columns.driver] = nds_file -> get_driver_name();
				row[this -> listview_columns.row_data] = filenames[i];

				std::cout << "Added file " << filename << " with current driver " << nds_file -> get_driver_name() << std::endl;
			}

			this -> last_nds_folder = dialog.get_current_folder();
		}
		break;
	}
}

void DLDIWindow::patch()
{
	std::cout << std::endl;

	// Get combo pointer
	Gtk::ComboBox * combo_drivers = 0;
	this -> glade_ref -> get_widget("cmbDriver", combo_drivers);
	int dldi_file = combo_drivers -> get_active_row_number();

	if(dldi_file < 0 || dldi_file > (int)this -> combo_model -> children().size())
	{
		this -> show_dialog("No DLDI Driver selected.");
		return;
	}

	std::vector<NdsFile *>::iterator iter;
	for(iter = this -> nds_files.begin(); iter != this -> nds_files.end(); iter++)
	{
		NdsFile * nds_file = *iter;
		try
		{
			nds_file -> patch(*(this -> dldi_files[dldi_file]));
		}
		catch(const std::runtime_error e)
		{
			Gtk::MessageDialog dialog(*this, "Error while patching " + nds_file -> get_filename().substr(nds_file -> get_filename().find_last_of('/') + 1));
			dialog.set_secondary_text(e.what());

			dialog.run();
			continue;
		}
	}

	this -> listview_model -> clear();
	this -> clear_nds_files();
	this -> show_dialog("Patched succesfully");
}

void DLDIWindow::show_dialog(Glib::ustring text)
{
	Gtk::MessageDialog dialog(*this, text);

	dialog.run();
}

void DLDIWindow::clear_dldi_files()
{
	for(unsigned int i = 0; i < this -> dldi_files.size(); i++)
	{
		delete this -> dldi_files[i];
	}

	this -> dldi_files.clear();
}

void DLDIWindow::clear_nds_files()
{
	for(unsigned int i = 0; i < this -> nds_files.size(); i++)
	{
		delete this -> nds_files[i];
	}

	this -> nds_files.clear();
}

