/**
 * DLDI GUI Patcher for Linux
 *
 * This program is an easy to use GUI to patch multiple files at once,
 * created with GTKmm and intended voor Linux users
 *
 * Created by Lucas van Dijk, using an modified version of dldi source,
 * by Michael Chisholm (Chishm)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * @author Lucas van Dijk
 * @version $Id$
 * @license http://www.opensource.org/licenses/gpl-license.php
 */

#include "../include/ndsfile.h"

/**
 * Constructor, sets our pointers to zero
 */
NdsFile::NdsFile(std::string filename)
{
	this -> filename = filename;
	this -> app_file_data = 0;

	this -> open();
}

/**
 * Destructor, cleans a bit up
 */
NdsFile::~NdsFile()
{
	if(this -> app_file_data)
	{
		delete[] this -> app_file_data;
	}

	if(this -> file.is_open())
	{
		this -> file.close();
	}
}

/**
 * Reads data from the DLDI file
 * @param offset The offset from where to read
 * @return The data read
 */
addr_t NdsFile::read_address(addr_t offset)
{
	return (addr_t)(
			(this -> dldi_section[offset + 0] << 0) |
			(this -> dldi_section[offset + 1] << 8) |
			(this -> dldi_section[offset + 2] << 16) |
			(this -> dldi_section[offset + 3] << 24)
		);
}

/**
 * Writes data to the app file data in memory
 * @param offset Position where to write
 * @param value The value to write
 */
void NdsFile::write_address(addr_t offset, addr_t value)
{
	this -> dldi_section[offset + 0] = (data_t)(value >> 0);
	this -> dldi_section[offset + 1] = (data_t)(value >> 8);
	this -> dldi_section[offset + 2] = (data_t)(value >> 16);
	this -> dldi_section[offset + 3] = (data_t)(value >> 24);
}

/**
 * Search the app's file data for a given addr_t
 *
 * This method searches the file data of the app, for a given data, and returns the position
 * @param search The data to search
 * @param length Number of bytes per character
 * @return Position of the data
 */
addr_t NdsFile::quick_find(const data_t * search, size_t length)
{
	const int* dataChunk = (const int*) this -> app_file_data;
	int searchChunk = ((const int*)search)[0];

	addr_t i;
	addr_t dataChunkEnd = (addr_t)(this -> filesize / sizeof(int));

	for ( i = 0; i < dataChunkEnd; i++)
	{
		if (dataChunk[i] == searchChunk)
		{
			// Check if we're out of bounds
			if ((i * sizeof(int) + length) > this -> filesize)
			{
				return -1;
			}

			// Compare memory
			if (memcmp (&this -> app_file_data[i * sizeof(int)], search, length) == 0)
			{
				return i * sizeof(int);
			}
		}
	}

	// Nothing found
	return -1;
}

/**
 * Opens the .nds file
 *
 * This method opens the .nds file, so the data can be read in memory and modified.
 * If anything goes wrong, an runtime_error is thrown
 */
void NdsFile::open()
{
	// Try to open the file, and put the pointer at the end
	this -> file.open(this -> filename.c_str(), ios::in | ios::out | ios::binary | ios::ate);

	if(this -> file.is_open() == false)
	{
		throw std::runtime_error("Could not open NDS File file");
	}

	// Get filesize
	this -> filesize = this -> file.tellg();

	// Initialize member
	this -> app_file_data = new data_t[this -> filesize];
	char * buffer = new char[this -> filesize];

	// Put the pointer back at the beginning
	this -> file.seekg(0, ios::beg);

	// Read it into memory
	this -> file.read(buffer, this -> filesize);

	this -> app_file_data = reinterpret_cast<data_t *>(buffer);

	if(buffer)
	{
		//delete[] buffer;
		buffer = 0;
	}

	this -> parse_info();
}

/**
 * This method parses the current driver name from the application
 *
 * It also determines the position of the DLDI section in the application.
 */
void NdsFile::parse_info()
{
	// First, find the reserved DLDI space in the app file
	this -> patch_offset = this -> quick_find (dldiMagicString, sizeof(dldiMagicString) / sizeof(char));

	if(this -> patch_offset < 0)
	{
		throw std::runtime_error("File does not have a DLDI section");
	}

	this -> dldi_section = &(this -> app_file_data[patch_offset]);
	this -> driver_name = string(reinterpret_cast<const char*>(&this -> dldi_section[DO_friendlyName]));
}


/**
 * Patches the file
 *
 * This method does the patching, it copies the data from the DLDI file
 * into the file which you want to patch.
 * Also this method throws an runtime_error when something goes wrong
 * @param dldi_file The new driver file
 */
void NdsFile::patch(DLDIFile &dldi_file)
{
	// Check if the app has enough space for the DLDI driver
	if(dldi_file[DO_driverSize] > this -> dldi_section[DO_allocatedSpace])
	{
		throw std::runtime_error("File does not have enough space for the driver");
	}

	// Find at what position the DLDI data is
	addr_t mem_offset = this -> read_address(DO_text_start);
	if(mem_offset == 0)
	{
		mem_offset = this -> read_address(DO_startup) - DO_code;
	}

	addr_t dldi_mem_offset = dldi_file.read_address(DO_text_start);
	addr_t relocation_offset = mem_offset - dldi_mem_offset;

	std::cout << "Patching " << this -> filename << " to "  << dldi_file.get_driver_name() << std::endl;

	// Find DLDI memory positions
	addr_t dldi_mem_start = dldi_file.read_address(DO_text_start);
	addr_t dldi_mem_size = (1 << dldi_file[DO_driverSize]);
	addr_t dldi_mem_end = dldi_mem_start + dldi_mem_size;

	std::cout << "Filesize:" << this -> filesize << " bytes" << std::endl;
	std::cout << "DLDI Filesize:" << dldi_file.get_filesize() << " bytes\n" << std::endl;
	std::cout << "Position in file:  0x" << std::hex << this -> patch_offset << std::dec << std::endl;
	std::cout << "Position in memory:  0x" << std::hex << mem_offset << std::dec << std::endl;
	std::cout << "Patch base address:  0x" << std::hex << dldi_mem_offset << std::dec << std::endl;
	std::cout << "Relocation offset:  0x" << std::hex << relocation_offset << std::dec << std::endl;
	std::cout << "DLDI Memory Size:  0x" << std::hex << dldi_mem_size << std::dec << std::endl;

	printf ("Old driver:          %s\n", &this -> dldi_section[DO_friendlyName]);
	printf ("New driver:          %s\n", &dldi_file[DO_friendlyName]);
	printf ("\n");
	printf ("Position in file:    0x%08X\n", this -> patch_offset);
	printf ("Position in memory:  0x%08X\n", mem_offset);
	printf ("Patch base address:  0x%08X\n", dldi_mem_offset);
	printf ("Relocation offset:   0x%08X\n", relocation_offset);
	printf("DLDI Memory size: 0x%08X\n", dldi_mem_size);
	printf ("\n");

	data_t * data_to_copy = dldi_file.get_data();
	data_to_copy[DO_allocatedSpace] = this -> dldi_section[DO_allocatedSpace];

	// Copy memory to the app
	memcpy(this -> dldi_section, data_to_copy, dldi_file.get_filesize());

	// Fix pointers to the sections in the header
	this -> write_address(DO_text_start, this -> read_address(DO_text_start) + relocation_offset);
	this -> write_address(DO_data_end, this -> read_address(DO_data_end) + relocation_offset);
	this -> write_address(DO_glue_start, this -> read_address(DO_glue_start) + relocation_offset);
	this -> write_address(DO_glue_end, this -> read_address(DO_glue_end) + relocation_offset);
	this -> write_address(DO_got_start, this -> read_address(DO_got_start) + relocation_offset);
	this -> write_address(DO_got_end, this -> read_address(DO_got_end) + relocation_offset);
	this -> write_address(DO_bss_start, this -> read_address(DO_bss_start) + relocation_offset);
	this -> write_address(DO_bss_end, this -> read_address(DO_bss_end) + relocation_offset);

	// Fix the function pointers in the header
	this -> write_address(DO_startup, this -> read_address(DO_startup) + relocation_offset);
	this -> write_address(DO_isInserted, this -> read_address(DO_isInserted) + relocation_offset);
	this -> write_address(DO_readSectors, this -> read_address(DO_readSectors) + relocation_offset);
	this -> write_address(DO_writeSectors, this -> read_address(DO_writeSectors) + relocation_offset);
	this -> write_address(DO_clearStatus, this -> read_address(DO_clearStatus) + relocation_offset);
	this -> write_address(DO_shutdown, this -> read_address(DO_shutdown) + relocation_offset);

	addr_t addrIter;

	if (dldi_file[DO_fixSections] & FIX_ALL)
	{
		// Search through and fix pointers within the data section of the file
		for (addrIter = (dldi_file.read_address(DO_text_start) - dldi_mem_start); addrIter < (dldi_file.read_address(DO_data_end) - dldi_mem_start); addrIter++)
		{
			if ((dldi_mem_start <= this -> read_address(addrIter)) && (this -> read_address(addrIter) < dldi_mem_end))
			{
				this -> write_address(addrIter, this -> read_address(addrIter) + relocation_offset);
			}
		}
	}

	if (dldi_file[DO_fixSections] & FIX_GLUE)
	{
		// Search through and fix pointers within the glue section of the file
		for (addrIter = (dldi_file.read_address(DO_glue_start) - dldi_mem_start); addrIter < (dldi_file.read_address(DO_glue_end) - dldi_mem_start); addrIter++)
		{
			if ((dldi_mem_start <= this -> read_address(addrIter)) && (this -> read_address(addrIter) < dldi_mem_end))
			{
				this -> write_address(addrIter, this -> read_address(addrIter) + relocation_offset);
			}
		}
	}

	if (dldi_file[DO_fixSections] & FIX_GOT)
	{
		// Search through and fix pointers within the Global Offset Table section of the file
		for (addrIter = (dldi_file.read_address(DO_got_start) - dldi_mem_start); addrIter < (dldi_file.read_address(DO_got_end) - dldi_mem_start); addrIter++)
		{
			if ((dldi_mem_start <= this -> read_address(addrIter)) && (this -> read_address(addrIter) < dldi_mem_end))
			{
				this -> write_address(addrIter, this -> read_address(addrIter) + relocation_offset);
			}
		}
	}

	if (dldi_file[DO_fixSections] & FIX_BSS)
	{
		// Initialise the BSS to 0
		memset (&this -> dldi_section[dldi_file.read_address(DO_bss_start) - dldi_mem_start] , 0, dldi_file.read_address(DO_bss_end) - dldi_file.read_address(DO_bss_start));
	}

	this -> file.seekp(this -> patch_offset, ios::beg);
	char * char_dldi = reinterpret_cast<char *>(this -> dldi_section);
	this -> file.write(char_dldi, dldi_mem_size);
	this -> file.close();

	std::cout << std::endl << "Patched Successfully" << std::endl << std::endl;
}

/**
 * Overloads the [] operator
 *
 * This can be used to acces a specific byte in the filedata.
 */
data_t &NdsFile::operator[](addr_t index)
{
	if(index > this -> filesize)
	{
		throw std::runtime_error("Index out of bounds");
	}

	return this -> app_file_data[index];
}

std::string NdsFile::get_driver_name()
{
	return this -> driver_name;
}

std::string NdsFile::get_filename()
{
	return this -> filename;
}
