/**
 * DLDI GUI Patcher for Linux
 *
 * This program is an easy to use GUI to patch multiple files at once,
 * created with GTKmm and intended voor Linux users
 *
 * Created by Lucas van Dijk, using an modified version of dldi source,
 * by Michael Chisholm (Chishm)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * @author Lucas van Dijk
 * @version $Id$
 * @license http://www.opensource.org/licenses/gpl-license.php
 */

#include "../include/dldifile.h"
#include "../include/dldi.h"

/**
 * Constructor, initializes some vars, and opens the DLDI File
 *
 * Usage: DLDIFile file("path/to/file.dldi");
 * file.get_driver_name();
 *
 * @param filename Path to the DLDI file
 */
DLDIFile::DLDIFile(std::string filename)
{
	this -> filename = filename;
	this -> file_data = 0;

	this -> open_file();
}

/**
 * Destructor, Cleans it up
 */
DLDIFile::~DLDIFile()
{
	if(this -> file_data)
	{
		delete[] this -> file_data;
	}
}

/**
 * Returns the driver name of the DLDI file
 */
string DLDIFile::get_driver_name()
{
	return this -> driver_name;
}

/**
 * Returns the filename of the DLDI file
 */
string DLDIFile::get_path()
{
	return this -> filename;
}

/**
 * Tries to find and open the .dldi file
 *
 * This function does not do such extensive searching like the original DLDI does,
 * but just looks in the current working directory.
 * It creates a fstream object, and puts the contents in memory, which us used by the function parse_info()
 */
void DLDIFile::open_file()
{
	if(this -> filename.length() == 0)
	{
		throw runtime_error("No filename");
	}

	// Try to open the file, and put the pointer at the end
	ifstream dldi_file(this -> filename.c_str(), ios::in | ios::binary | ios::ate);

	if(dldi_file.is_open() == false)
	{
		throw runtime_error("Could not open DLDI file");
	}

	// Get filesize
	this -> filesize = dldi_file.tellg();

	// Initialize member
	this -> file_data = new data_t[this -> filesize];
	char * buffer = new char[this -> filesize];

	// Put the pointer back at the beginning
	dldi_file.seekg(0, ios::beg);

	// Read it into memory
	dldi_file.read(buffer, this -> filesize);

	this -> file_data = reinterpret_cast<data_t *>(buffer);

	if(buffer)
	{
		// Deleting the buffer gives an double freed error :/
		//delete[] buffer;
		buffer = 0;
	}

	// And close the file
	dldi_file.close();

	// Parse the info from the file
	this -> parse_info();
}

/**
 * Parses some info from the DLDI file
 *
 * This method tries to parse some info from the DLDI file, like the version, the driver name
 * and also checks if its a valid DLDI file.
 * If something fails, an runtime_error is thrown
 */
void DLDIFile::parse_info()
{
	// Check if it's a valid DLDI File
	if (strcmp(reinterpret_cast<const char*>(dldiMagicString), reinterpret_cast<char*>(&(this -> file_data[DO_magicString]))) != 0)
	{
		throw runtime_error("Invalid DLDI File");
	}

	// Check version
	if (this -> file_data[DO_version] != DLDI_VERSION)
	{
		ostringstream text;
		text << "Incorrect DLDI file version. Excepted " << DLDI_VERSION << ", found " << this -> file_data[DO_version];
		throw runtime_error(text.str());
	}

	// Get driver name
	this -> driver_name = string(reinterpret_cast<const char*>(&this -> file_data[DO_friendlyName]));
}

/**
 * Reads data from the DLDI file
 * @param offset The offset from where to read
 * @return The data read
 */
addr_t DLDIFile::read_address(addr_t offset)
{
	return (addr_t)(
			(this -> file_data[offset + 0] << 0) |
			(this -> file_data[offset + 1] << 8) |
			(this -> file_data[offset + 2] << 16) |
			(this -> file_data[offset + 3] << 24)
		);
}

/**
 * Overloads the [] operator
 *
 * This can be used to acces a specific byte in the filedata.
 */
data_t &DLDIFile::operator[](addr_t index)
{
	if(index > this -> filesize)
	{
		throw runtime_error("Index out of bounds");
	}

	return this -> file_data[index];
}

/**
 * Returns the pointer to the filedata
 */
data_t * DLDIFile::get_data()
{
	return this -> file_data;
}

/**
 * Returns the filesize of the DLDI file
 */
fstream::pos_type DLDIFile::get_filesize()
{
	return this -> filesize;
}
