#include <nds.h>
#include <fat.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include "calc.h"
#include "lcd.h"
#include "hw_83p.h"
#include "disasm.h"
#include "browser.h"
#include "z80code.h"
#include "bitmap.h"

#include "keymap_bmp.h"		//Key map for touch screen

#include "keystrings.h"	


BITMAP keymap;

CALC_T calc;
CONFIG_T cfg;
REWIND_t rewindti;

struct DrZ80 *cpu	= &calc.cpu;
TIMER_t  *timer		= &calc.time_c;
memc     *mem		= &calc.mem_c;
KEY_t    *keys		= &calc.keys_c;
STDINT_t *stdint	= &calc.stdint_c;
LCD_t    *lcd		= &calc.lcd_c;
link_t   *link		= &calc.link;

static bool loadedrom = false;
static bool apd = true;
static bool inmenu = false;
dirlist_t *Romlist = NULL;
dirlist_t *Varlist = NULL;


char default_keys[] = {
			5,7,6,5,15,15,
			1,0,0,2,0,1,
			0,3,0,0,15,15,
			15,15,6,6,4,7};


void VblankHandler(void) {
	static int lastpressed = 0;
	static bool light_on = true;
	static bool lid_was_closed = false;
	scanKeys();
	u32 press = keysHeld();
	//if lid open
	if (!(press & KEY_LID)) {
		if ((press & (BIT(13)-1)) || lid_was_closed) {
			lastpressed = 0;
			if (!light_on) {
				REG_IPC_FIFO_TX = 3;
				light_on = true;
			}
		} else {
			if (lastpressed>=(60*45)) {
				if (light_on && (apd || inmenu)) {
					REG_IPC_FIFO_TX = 2;
					light_on = false;
				}
			} else {
				lastpressed++;
			}
		}
		lid_was_closed = false;
	} else {
		lastpressed = 0;
		lid_was_closed = true;
		if (inmenu) {
			REG_IPC_FIFO_TX = 1;
			while ((press & KEY_LID)) {
				scanKeys();
				press = keysHeld();
			}
		}
	}
}





void FileList() {
	if (Romlist) ReleaseDirList(Romlist);
	if (Varlist) ReleaseDirList(Varlist);
	Romlist = Build_DirList("*.rom\0*.bin\0*.tisv\0*.8xu\0\0");
	Varlist = Build_DirList("*.8x*\0*.83*\0*.82*\0\0");
}

void invalidateRewind(unsigned int seconds) {
	rewindti.delay	= (60*seconds);
	rewindti.count	= 0;
	rewindti.start	= 0;
	rewindti.frames	= 0;
}

void Calc_init(char *rom){
	FILE* infile;
	u8 string[9];
	unsigned long size;
	
	MEM_init_83p();
	LCD_init();
	STDINT_init_83p();
	LINK_init();
	cpu_init();
	timer->nextslice	= TimeSlice;
	timer->slices		= 0;
	timer->frames		= 0;
	timer->tstates		= 0;
	calc.fakeon = 0;
	if (!rom) return;
	infile = fopen(rom,"rb");
	if (!infile) {
		puts("Calc init file open Failed");
		return;
	}
	fread(string,1,8,infile);
	if (!memcmp(string,"**TIFL**",8)) {
		Load_8xu(infile);
		calc.rom[0x56] = 0x5A;
		calc.rom[0x57] = 0xA5;
	} else {
		fseek(infile,0,SEEK_END);
		size = ftell(infile);
		fseek(infile,0,SEEK_SET);
		if (size>(512*1024)) size = (512*1024);
		fread(calc.rom,1,size,infile);
	}
	fclose(infile);
	calc_erase_certificate();
}

void ZeroCtrlChar(const void *string){
	unsigned char *str = (unsigned char *)string;
	while(*str) {
		if ((*str)<32) (*str)=0;
		if ((*str)>127) (*str)=0;
		str++;
	}
}

void Write_Save() {
	FILE *ofile;
	int i;
	for(i=strlen(cfg.save_path)-1;i>1 && cfg.save_path[i]!='.';i--);
	if (cfg.save_path[i]!='.') {
		i=strlen(cfg.save_path);
		cfg.save_path[i] = '.';
	}
	i++;
	if (strcasecmp(&cfg.save_path[i],"TISV")) {
		cfg.save_path[i++] = 't';
		cfg.save_path[i++] = 'i';
		cfg.save_path[i++] = 's';
		cfg.save_path[i++] = 'v';
		cfg.save_path[i++] = 0;
	}
	if (strlen(cfg.save_path)<6) {
		memset(cfg.save_path,0,256);
		return;
	}
	
	ofile = fopen(cfg.save_path,"wb");
	if (!ofile) {
		puts("Failed to open for write");
		return;
	}
	fprintf(ofile,"**TIDS**%s",SAVE_VERSION);
	fwrite(&calc,sizeof(calc),1,ofile);
	fclose(ofile);
}

int Read_Save() {
	FILE *ifile;
	char header[16];
	int i;
	for(i=strlen(cfg.save_path)-1;i>1 && cfg.save_path[i]!='.';i--);
	if (cfg.save_path[i]!='.') {
		puts("Invalid name");
		return 1;
	}
	i++;
	if (strcasecmp(&cfg.save_path[i],"TISV")) {
		loadedrom = true;
		Calc_init(cfg.save_path);
	} else {
		Calc_init(NULL);
		ifile = fopen(cfg.save_path,"rb");
		if (!ifile) puts("Could not open save file");
		fread(header,12,1,ifile);
		header[12] = 0;
		if (memcmp(header,"**TIDS**",8)) return 1;
		if (memcmp(header+8,SAVE_VERSION,4)) return 1;
		fread(&calc,sizeof(calc),1,ifile);
		fclose(ifile);
		
		//Update pointers
		mem->flash			= calc.rom;
		mem->ram			= calc.ram;
		for(i=0;i<5;i++) {
			if (mem->banks[i].ram) {
				mem->banks[i].addr	= mem->ram+((mem->banks[i].page^0x01)<<14);
			} else {
				mem->banks[i].addr	= mem->flash+(mem->banks[i].page<<14);
			}
		}
		cpu->z80_rebasePC		= &DrZ80_rebasePC;
		cpu->z80_rebaseSP		= &DrZ80_rebaseSP;
		cpu->z80_read8			= &z80_read;
		cpu->z80_read16			= &z80_read16;
		cpu->z80_write8			= &z80_write;
		cpu->z80_write16		= &z80_write16;
		cpu->z80_in				= &in_83p;	// <- change if other models 
		cpu->z80_out			= &out_83p;	//    supported
		cpu->z80_irq_callback	= &DrZ80_irq_callback;
		cpu->Z80PC				= cpu->z80_rebasePC(getPC());
		cpu->z80_rebaseSP(getSP());
	}
	return 0;
}	
	
	


void Write_Config() {
	FILE *ofile;
	int i;
	ofile = fopen("fat:/tids.cfg","w");
	if (!ofile) return;
	fprintf(ofile,"<TIDS>\n");
	fprintf(ofile,"<SAVE>\n");
	fprintf(ofile,"%s\n",cfg.save_path);
	fprintf(ofile,"<KEYCFG>\n");
	for(i=0;i<12;i++) {
		fprintf(ofile,"%s= %01X%01X\n",ButtonNames[i],cfg.keys[i][0]&0x0F,cfg.keys[i][1]&0x0F);
	}
	if (cfg.skin_n) {
		fprintf(ofile,"<SKIN_N>\n");
		fprintf(ofile,"%s\n",cfg.skin_n_path);
		fprintf(ofile,"%s\n",cfg.key_n_path);
	}
	if (cfg.skin_r) {
		fprintf(ofile,"<SKIN_R>\n");
		fprintf(ofile,"%s\n",cfg.skin_r_path);
		fprintf(ofile,"%s\n",cfg.key_r_path);
	}
	if (cfg.skin_l) {
		fprintf(ofile,"<SKIN_L>\n");
		fprintf(ofile,"%s\n",cfg.skin_l_path);
		fprintf(ofile,"%s\n",cfg.key_l_path);
	}
	fprintf(ofile,"<ROTATE>\n");
	fprintf(ofile,"%d\n",cfg.rotate%3);
	if (cfg.autosave) {
		fprintf(ofile,"<AUTOSAVE>\n");
		fprintf(ofile,"%d\n\n\n",cfg.autosave);
	}
	fclose(ofile);
}

void Load_Config() {
	int i,b,readline;
	char string[256];
	char str2[256];
	char str3[256];

	FILE *ifile;

	memset(cfg.save_path,0,256);
	memcpy(cfg.keys,default_keys,24);
	cfg.skin_n = 0;
	memset(cfg.skin_n_path,0,256);
	memset(cfg.key_n_path,0,256);
	cfg.skin_r = 0;
	memset(cfg.skin_r_path,0,256);
	memset(cfg.key_r_path,0,256);
	cfg.skin_l = 0;
	memset(cfg.skin_l_path,0,256);
	memset(cfg.key_l_path,0,256);	
	cfg.rotate = 0;
	cfg.autosave = 0;
	ifile = fopen("fat:/tids.cfg","rt");
	if (!ifile) return;

	memset(string,0,256);
	fgets(string,255,ifile);

	if (!strcmp(string,"<TIDS>")) return;
	readline = 1;
	while(!feof(ifile)) {
		if (readline) {
			if (!fgets(string,255,ifile)) break;
		}
		ZeroCtrlChar(string);
		readline = 1;
		STR_SWITCH(string) {
			STR_CASE("<SAVE>"):
				fgets(cfg.save_path,255,ifile);
				ZeroCtrlChar(cfg.save_path);
				if (cfg.save_path[0]=='<') {
					memcpy(string,cfg.save_path,256);
					readline = 0;
				}
				if (readline == 0 || strlen(cfg.save_path)<1) {
					memset(cfg.save_path,0,255);
				}
				break;
			STR_CASE("<AUTOSAVE>"):
				fgets(str2,127,ifile);
				ZeroCtrlChar(str2);
				if (str2[0]=='<') {
					memcpy(string,str2,127);
					readline = 0;
				} else {
					unsigned int scans,autosave;
					scans = sscanf(str2,"%d",&autosave);
					if (scans==1) cfg.autosave = autosave%(60*60*3);
					else cfg.autosave = 0;
				}
				break;
			STR_CASE("<KEYCFG>"): {
				int scans=9,key=0x99;
				while(fgets(str2,127,ifile) && !feof(ifile) && str2[0]!='<') {
					ZeroCtrlChar(str2);
					scans = sscanf(str2,"%s %2x",str3,&key);
					if (scans == 2) {
						for(i=0;i<12;i++) {
							if (!strncasecmp(ButtonNames[i],str3,strlen(ButtonNames[i]))) {
								cfg.keys[i][0] = (key>>4)&0x0F;
								cfg.keys[i][1] = (key>>0)&0x0F;
							}
						}
					}
				}
				if (str2[0]=='<') {
					memcpy(string,str2,128);
					readline = 0;
				}
				break;
			}
			STR_CASE("<ROTATE>"): {
				int scans,read;
				fgets(str2,127,ifile);
				ZeroCtrlChar(str2);
				if (str2[0]=='<') {
					memcpy(string,str2,256);
					readline = 0;
				} else {
					scans = sscanf(str2,"%d",&read);
					if (scans==1) {
						cfg.rotate = read%3;
					}
				}
				break;
			}
			
		}
	}
	fclose(ifile);
}




void runtimed(int remaining) {
	int length,time;
	while (remaining>0) {
		if (remaining > TimeSlice) {
			length = TimeSlice;
		} else {
			length = remaining;
		}
		time = DrZ80Run(cpu,length);
		if (getHALT()) time = -(length&0x03);
		timer->tstates += (length - time);
		timer->nextslice = timer->nextslice-(length - time);
		remaining -= (length - time);
		if (timer->nextslice<0) {
			timer->nextslice = TimeSlice+timer->nextslice;
			interrupt_83p();
			timer->slices++;
			lcd->push++;
			if (lcd->push>=MAX_PUSH) {
				Push_LCD();	//Double because frame rate
				Push_LCD(); //likely means black and white
			}
			if ((timer->slices%SPF)==0) {
				timer->frames++;
			}
		}
	}
}

void TurnOn() {
	int i;
	if (!lcd->active) {
		calc.fakeon = 0;
		runtimed(CPU_SPEED/4);
		calc.fakeon = 1;
		runtimed(CPU_SPEED/2);
		calc.fakeon = 0;
		runtimed(CPU_SPEED/4);
	}
}


void Key_Config() {
	int karray[] = {0,1,10,11,3,4,5,6,7};
	int select = 0;
	int map[9];
	int i,b;
	swiWaitForVBlank();
	setupvideo();
	BG0_CR = BG_MAP_BASE(2) | BG_TILE_BASE(1) | BG_32x32 | BG_256_COLOR | BG_PRIORITY(0);
	keysSetRepeat(18,6);
	while(1) {
		for(b=0;b<9;b++) {
			for(i=0;i<50;i++) {
				if ((cfg.keys[karray[b]][0] == keylayout[i].group) &&
					(cfg.keys[karray[b]][1] == keylayout[i].bit)) {
					map[b] = i;
				}
			}
		}
		swiWaitForVBlank();
		consoleClear();
		printfxy(2,1,"DS Button");
		printfxy(14,1,"TI Key");
		for(i=0;i<9;i++) {
			if (select==i) {
				printfxy(0,(i*2)+3,">");
			}
			printfxy(2,(i*2)+3,"%s:", ButtonNames[karray[i]]);
			printfxy(12,(i*2)+3,"<");
			printfxy(14,(i*2)+3,"%s", keylayout[map[i]].Name);
			printfxy(21,(i*2)+3,">");
		}
		printfxy(24,22,"Default");
		int update = 0;
		while(!update) {
			swiWaitForVBlank();
//			scanKeys();
			int press = keysDownRepeat();
			if ( (press & KEY_SELECT)|(press & KEY_B)) {
				Write_Config();
				keysSetRepeat(15,1);
				FadeToWhite();
				consoleClear();	
				BG0_CR = 0;
				return;
			}
			if (press & KEY_START) {
				memcpy(cfg.keys,default_keys,24);
				update = 1;
			}
			if (press & KEY_DOWN) {
				select++;
				if (select>=9) select =0;
				update = 1;
			}
			if (press & KEY_UP) {
				select--;
				if (select<0) select =8;
				update = 1;
			}
			if (press & KEY_LEFT) {
				map[select]--;
				if (map[select]<00) map[select] = 49;
				cfg.keys[karray[select]][0] = keylayout[map[select]].group;
				cfg.keys[karray[select]][1] = keylayout[map[select]].bit;
				update = 1;
			}
			if (press & KEY_RIGHT) {
				map[select]++;
				if (map[select]>=50) map[select] = 0;
				cfg.keys[karray[select]][0] = keylayout[map[select]].group;
				cfg.keys[karray[select]][1] = keylayout[map[select]].bit;
				update = 1;
			}
			if (press & KEY_TOUCH) {
				touchPosition t = touchReadXY();
				if ((t.py>=20 && t.py<164) &&
					( (t.px>=92 && t.px<108) || (t.px>=164 && t.px<180) ) ) {
					
					if (t.px>=92 && t.px<108) i = -1;
					else i = 1;
					
					b = (t.py-20)/16;
					select = b;
					map[b] = map[b]+i;
					if (map[b]<00) map[b] = 49;
					if (map[b]>49) map[b] = 00;
					
					cfg.keys[karray[b]][0] = keylayout[map[b]].group;
					cfg.keys[karray[b]][1] = keylayout[map[b]].bit;
					update = 1;
				} else 	if ( (t.py>=172 && t.py<188) &&
					(t.px>=188 && t.px<252) ) {
					memcpy(cfg.keys,default_keys,24);
					update = 1;
				}
			}
		}
	}
}

void SetMenuMode() {
	videoSetMode(	MODE_5_2D |
					DISPLAY_SPR_1D_LAYOUT | 
					DISPLAY_SPR_ACTIVE |
					DISPLAY_BG3_ACTIVE |
					DISPLAY_BG2_ACTIVE |
					DISPLAY_BG0_ACTIVE);
	BG2_CR  = BG_BMP16_256x256  | BG_BMP_BASE(0x08) | BG_PRIORITY(2);
	BG2_XDX = 1<<8;
	BG2_XDY = 0;
	BG2_YDX = 0;
	BG2_YDY = 1<<8;
	BG2_CX  = 0;
	BG2_CY  = 0;
}

void Menu() {
	static int select = 0;
	int clicked = 0;
	int i,press,down,x,y;
	FadeToWhite();
	swiWaitForVBlank();
	SetMenuMode();
	FadeFromWhite();
	
	x = 8;
	y = 8;
	
	while(1) {
		swiWaitForVBlank();
		CleanOAM();
		sprite((select&0x01)?132:5,(select&0x02)?100:5,0);
		updateOAM();
//		scanKeys();
		press = keysHeld();
		down = keysDown();
		if ( (down & KEY_SELECT)|(down & KEY_B)) {
			FadeToWhite();
			return;
		}
		if (down & KEY_DOWN) {
			select ^= 0x2;
		}
		if (down & KEY_UP) {
			select ^= 0x2;
		}
		if (down & KEY_LEFT) {
			select ^= 0x1;
		}
		if (down & KEY_RIGHT) {
			select ^= 0x1;
		}
		if (down & KEY_A) {
			clicked = 1;
		}
		if (press & KEY_TOUCH) {
			touchPosition touch = touchReadXY();
			i = 0;
			if (touch.px>=128) i++;
			if (touch.py>=96) i+=2;
			select = i;
			clicked = 1;
		}
		if (clicked) {
			swiWaitForVBlank();
			CleanOAM();
			sprite((select&0x01)?132:5,(select&0x02)?100:5,0);
			updateOAM();
			for(i=0;i<4;i++) swiWaitForVBlank();
			switch(select) {
				case 0:
					Key_Config();
					break;
				case 1: {
					char *name = Browser(Varlist);
					swiWaitForVBlank();
					if (name) {		
						consoleClear();
						BG0_CR = BG_MAP_BASE(2) | BG_TILE_BASE(1) | BG_32x32 | BG_256_COLOR | BG_PRIORITY(0);
						NormalSkin();
						TurnOn();
						SendFile(name);
						Write_Save();
						Write_Config();
						swiWaitForVBlank();
						WhiteSkin();
						consoleClear();
						BG0_CR = 0;
						invalidateRewind(1);
						return;
					}
					break;
				}
				case 2: {
					loadedrom = false;
					char string[256];
					char *rom = Browser(Romlist);
					swiWaitForVBlank();
					if (rom) {
						consoleClear();
						BG0_CR = BG_MAP_BASE(2) | BG_TILE_BASE(1) | BG_32x32 | BG_256_COLOR | BG_PRIORITY(0);
						NormalSkin();
						memcpy(string,cfg.save_path,256);
						memcpy(cfg.save_path,rom,255);
						if (Read_Save()) {
							memcpy(cfg.save_path,string,256);
						}
						if (loadedrom) {
							TurnOn();
							Write_Save();
							FileList();
							loadedrom = false;
						}
						Write_Config();
						swiWaitForVBlank();
						WhiteSkin();
						consoleClear();
						BG0_CR = 0;
						return;
					}
					WhiteSkin();
					consoleClear();
					BG0_CR = 0;
					break;	
				}
				case 3:
					swiWaitForVBlank();
					setupvideo();
					DisAsm();
					break;
			}
			if (keysHeld() & KEY_SELECT) return;
			swiWaitForVBlank();
			setupvideo();
			SetMenuMode();
			FadeFromWhite();
		}
		clicked = 0;
	}

}





int main(void) {
	int i,time;
	FILE *tmpfile = NULL;
	char *rom = NULL;
	irqInit();
	irqSet(IRQ_VBLANK, VblankHandler);
	REG_IPC_FIFO_CR = IPC_FIFO_SEND_CLEAR | IPC_FIFO_ERROR | IPC_FIFO_ENABLE;
	keysSetRepeat(15,1);
	LoadSkins();
	fatInitDefault();
	Load_Config();
	FileList();
	BitmapDef(&keymap,(void*)keymap_bmp);
	tmpfile = fopen(cfg.save_path,"rb");
	
	swiWaitForVBlank();
	WhiteSkin();
	setupvideo();
	SetRotation();
	SetSkin(0);
	if (tmpfile) {
		fclose(tmpfile);
		swiWaitForVBlank();
		consoleClear();
		BG0_CR = BG_MAP_BASE(2) | BG_TILE_BASE(1) | BG_32x32 | BG_256_COLOR | BG_PRIORITY(0);
		BG0_X0 = 0;
		BG0_Y0 = 0;
		NormalSkin();
		Read_Save();
		swiWaitForVBlank();
		WhiteSkin();
		BG0_CR = 0;
	} else {
		inmenu = true;
		while(!rom) {
			while(!rom) {
				swiWaitForVBlank();
				BG0_CR = BG_MAP_BASE(0) | BG_TILE_BASE(1) | BG_32x32 | BG_256_COLOR | BG_PRIORITY(0);
				BG0_X0 = 0;
				BG0_Y0 = 0;
				NormalSkin();
				rom = Browser(Romlist);
			}
			memcpy(cfg.save_path,rom,255);
			swiWaitForVBlank();
			consoleClear();
			BG0_CR = BG_MAP_BASE(2) | BG_TILE_BASE(1) | BG_32x32 | BG_256_COLOR | BG_PRIORITY(0);
			BG0_X0 = 0;
			BG0_Y0 = 0;
			NormalSkin();
			if (Read_Save()) rom = NULL;
			swiWaitForVBlank();
			WhiteSkin();
			BG0_CR = 0;
		}
		FadeFromWhite();
		inmenu = false;
	}
	NormalSkin();
	if (loadedrom) {
		TurnOn();
		Write_Save();
		Write_Config();
		FileList();
		loadedrom = false;
	}
	
	invalidateRewind(0);

	
	apd = false;
	inmenu = false;
	while(1) {
		do {
			time = DrZ80Run(cpu,timer->nextslice);
			if (getHALT()) time = -(timer->nextslice&0x03);
			timer->tstates += (TimeSlice - time);
			timer->nextslice = TimeSlice+time;
			interrupt_83p();
			timer->slices++;
			lcd->push++;
			if (lcd->push>=MAX_PUSH) {
				Push_LCD();	//Double because frame rate
				Push_LCD(); //likely means black and white
			}
		} while (timer->slices%SPF);
		
		if (getHALT() && !lcd->active && (stdint->intactive&0x06)==0x00) {
			apd = true;
		} else {
			apd = false;
		}

		timer->frames++;

		if (!rewindti.delay) rewindti.delay--;
		else {
			rewindti.frames = (rewindti.frames+1)%20;
			if (!rewindti.frames) {
				if (rewindti.count<MAX_REWIND) rewindti.count++;
				memcpy(&rewindti.backup[rewindti.start],&calc,sizeof(BACKUP));
				rewindti.start = (rewindti.start+1)%MAX_REWIND;
			}
		}
//		scanKeys();
		keys->touch_button = 0;
		int press	= keysHeld();
		int keydown	= keysDown();
		if (press & KEY_TOUCH) {
			touchPosition touch = touchReadXY();
			RGBQUAD key;
			switch(cfg.rotate%3) {
				case 1:
					key = BitmapGetPixel24(&keymap,(touch.py*256)/192,((255-touch.px)*192)/256);
					break;
				case 2:
					key = BitmapGetPixel24(&keymap,((191-touch.py)*256)/192,(touch.px*192)/256);
					break;
				default:
					key = BitmapGetPixel24(&keymap,touch.px,touch.py);
					break;
			}
			if (key.rgbRed != 0xFF) {
				keys->touch_button	= 1;
				keys->touch_group	= key.rgbGreen>>4;
				keys->touch_bit		= key.rgbBlue>>4;
			}
		}

		//The arm7 is putting everything to sleep
		// so this code only writes the save and
		// makes sure the lid is open
		if (press & KEY_LID) {
			Write_Save();
			Write_Config();

			REG_IPC_FIFO_TX = 1;

			while ((press & KEY_LID)) {
				swiWaitForVBlank();
//				scanKeys();
				press = keysHeld();
			}
		}

		if (keydown&KEY_SELECT) {
			inmenu = true;
			Menu();
			swiWaitForVBlank();
			CleanOAM();
			updateOAM();
			setupvideo();

			FadeFromWhite();
			for(i=0;i<10;i++) {
				swiWaitForVBlank();
//				scanKeys();
			}
			inmenu = false;
		}
		if (cfg.autosave!=0) {
			if ((timer->frames%(cfg.autosave*60))==0) {
				Write_Save();
				Write_Config();
			}
		}
		if ((!(press&KEY_R)) && (!(press&KEY_L))) {
			swiWaitForVBlank();
		} else if ((press&KEY_L) && (!(press&KEY_R))) {
			if (rewindti.count) {
				rewindti.count--;
				rewindti.start--;
				if (rewindti.start<0) rewindti.start = MAX_REWIND-1;
				memcpy(&calc,&rewindti.backup[rewindti.start],sizeof(BACKUP));
				for(i=0;i<10;i++) swiWaitForVBlank();
			}
		}
		if ((keydown&KEY_L) && (press&KEY_R)) {
			swiWaitForVBlank();
			cfg.rotate = (cfg.rotate+1)%3;
			SetRotation();
		}
		ShiftSkin();
		GREYLCD();
	}
}
 
