#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <nds.h>
#include "calc.h"
#include "link.h"
#include "lcd.h"

#include <setjmp.h>

//#define DEBUG

#ifdef DEBUG
	#define debugf	printf
#else
	#define debugf	//
#endif

extern CALC_T calc;
extern struct DrZ80	*cpu;
extern TIMER_t		*timer;
extern memc			*mem;
extern KEY_t		*keys;
extern STDINT_t		*stdint;
extern LCD_t		*lcd;
extern link_t		*link;


static jmp_buf exc_pkt, exc_byte;

#define out link->pc
#define link_endian(z) (z)

#ifdef DEBUG
static void print_command_ID(uint8_t);
#endif

/* Run a number of tstates */
static void link_wait(time_t tstates) {
	runtimed(tstates);
}


void LINK_init() {
	link->calc			= 0;
	link->pc			= 0;
	link->send_count	= 0;
	link->recv_count	= 0;
	link->print			= 0;
}

/* Send a byte through the virtual link
 * On error: Throws a Byte Exception */
static void link_send(u8 byte) {
	u32 bit, i;

	for(bit = 0; bit < 8; bit++, byte >>= 1) {
		out = (byte & 1) + 1;
		
		for(i = 0; i<LINK_TIMEOUT && vlink!= 0; i++) runtimed(32);
		if (vlink != 0) longjmp(exc_byte, LERR_TIMEOUT);

		out = 0;
		for(i = 0; i < LINK_TIMEOUT && vlink != 3; i++) runtimed(32);
		if (vlink != 3) longjmp(exc_byte, LERR_TIMEOUT);
	}
	
	link->send_count++;

	if (link->print) {
		static int byteskip = 0;
		byteskip = (byteskip+1)%100;
		if (!byteskip) {
			printf("\x1b[%02dD",32);
			printf("Bytes Sent: %d   ",link->send_count);
		}
	}

}

/* Receive a byte through the virtual link
 * On error: Throws a Byte Exception */
static u_char link_recv() {
	u32 bit, i;
	u8 byte = 0;

	for (bit = 0; bit < 8; bit++) {
		byte >>= 1;
		
		for(i = 0; i < LINK_TIMEOUT && vlink == 3; i++) runtimed(32);

		if (vlink == 0) longjmp(exc_byte, LERR_LINK);
		if (vlink == 3) longjmp(exc_byte, LERR_TIMEOUT);
		
		out = vlink;
		if (out == 1) byte |= 0x80;

		for(i = 0; i < LINK_TIMEOUT && vlink == 0; i++) runtimed(32);
		if (vlink == 0) longjmp(exc_byte, LERR_TIMEOUT);
		out = 0;
	}
	
	link->recv_count++;
	return byte;
}

/* Calculate a TI Link Protocol checksum */
static uint16_t link_chksum(u_char *data, size_t length) {
	uint16_t chksum = 0;
	size_t i;
	for (i = 0; i < length; i++) chksum += data[i];
	
	return chksum;
}

/* Returns a Machine ID for the calc attached to virtual link
 * On error: Returns -1 */
inline static u_char link_target_ID() {
	return 0x23; //83+ only
}

/* Send a sequence of bytes over the virtual link
 * On error: Throws a Byte Exception */
static void link_send_bytes(void *data, size_t length) {
	size_t i;
	for (i = 0; i < length; i++) link_send(((u_char*) data)[i]);
}

/* Receive a sequence of bytes over the virtual link
 * On error: Throws a Byte Exception */
static void link_recv_bytes(void *data, size_t length) {
	size_t i;
	for (i = 0; i < length; i++) ((u_char*) data)[i] = link_recv();
}

/* Send a TI packet over the virtual link
 * On error: Throws a Packet Exception */
static void link_send_pkt(u_char command_ID, void *data) {
	TI_PKTHDR hdr;
	uint16_t data_len;

	hdr.machine_ID = link_target_ID();
	hdr.command_ID = command_ID;
#ifdef DEBUG
	printf("SEND: ");
	print_command_ID(command_ID);
	putchar('\n');
#endif
	switch (command_ID) {
		case CID_VAR: case CID_DEL: case CID_REQ: case CID_RTS: {
			uint8_t type_ID = ((TI_VARHDR *) data)->type_ID;

			if (type_ID == BackupObj) data_len = sizeof(TI_BACKUPHDR);
			else if (type_ID >= 0x22 && type_ID <= 0x28) data_len = sizeof(TI_FLASHHDR);
			else data_len = sizeof(TI_VARHDR);
			break;
		}
		case CID_DATA:
			data_len = ((TI_DATA*) data)->length;
			data = ((TI_DATA*) data)->data;
			break;
		case CID_EXIT:
			data_len = 5;
			break;
		default:
			data_len = 0;
			break;
	}

	int err;
	switch (err = setjmp(exc_byte)) {
		case 0:
			link_wait(LINK_DELAY);
			hdr.data_len = link_endian(data_len);
			link_send_bytes(&hdr, sizeof(TI_PKTHDR));
			if (data_len != 0) {
				link_send_bytes(data, data_len);
				uint16_t chksum = link_endian(link_chksum(data, data_len));
				link_send_bytes(&chksum, sizeof(chksum));
			}
			return;
		default: 
			return longjmp(exc_pkt, err);
	}
}

/* Receive a TI packet over the virtual link (blocking)
 * On error: Throws a Packet Exception */
static void link_recv_pkt(TI_PKTHDR *hdr, u_char *data) {
	int err;
	switch (err = setjmp(exc_byte)) {
		case 0:
			memset(hdr, 0, sizeof(TI_PKTHDR));
			// Receive the packet header
			hdr->machine_ID = link_recv();
			hdr->command_ID = link_recv();
			hdr->data_len = link_recv() + (link_recv() << 8);
#ifdef DEBUG
			printf("RECV %02x: ", hdr->machine_ID);
			print_command_ID(hdr->command_ID);
			putchar('\n');
#endif
			switch (hdr->command_ID) {
				case CID_VAR: case CID_DEL: case CID_REQ: case CID_RTS: {
					link_recv_bytes(data, 3);
					uint8_t type_ID = data[2];
					
					if (type_ID == BackupObj) hdr->data_len = sizeof(TI_BACKUPHDR);
					else if (type_ID >= 0x22 && type_ID <= 0x28) hdr->data_len = sizeof(TI_FLASHHDR);
					else hdr->data_len = sizeof(TI_VARHDR);
					
					link_recv_bytes(&data[3], hdr->data_len - 3);
					break; // Do checksum
				}
				case CID_DATA: case 0:
					// Receive the data
					link_recv_bytes(data, hdr->data_len);
					break; // Do checksum
				default: return;
			}
			break;
		default:
			return longjmp(exc_pkt, err);
	}
	uint16_t chksum = link_recv() + (link_recv() << 8);
	
	if (chksum != link_chksum(data, hdr->data_len)) longjmp(exc_pkt, LERR_CHKSUM);		
}

/* Send a Request To Send packet
 * On error: Throws Packet Exception */
static void link_RTS(TIVAR *tivar, int dest) {
	TI_VARHDR var_hdr;
	
	var_hdr.length = link_endian(tivar->length);
	var_hdr.type_ID = tivar->vartype;
	memset(var_hdr.name, 0, 8);
	strcpy(var_hdr.name, tivar->name);
	var_hdr.version = tivar->version;
	if (dest == SEND_RAM) {
		var_hdr.type_ID2 = 0x00;
	} else if (dest == SEND_ARC) {
		var_hdr.type_ID2 = 0x80;
	} else {
		var_hdr.type_ID2 = tivar->flag;
	}
	
	link_send_pkt(CID_RTS, &var_hdr);
}

/* Send a variable over the virtual link
 * On error: Returns an error code */
LINK_ERR link_send_var(FILE* infile,TIVAR *tivar, SEND_FLAG dest) {
	printf("Size: %d\n",tivar->length);
	printf("Name: %s\n",tivar->name);

	int err;
	switch (err = setjmp(exc_pkt)) {
		case 0: {
			TI_PKTHDR rpkt;
			u_char data[64];
			
			link_RTS(tivar, dest);
			
			link_recv_pkt(&rpkt, data);
			if (rpkt.command_ID != CID_ACK) return LERR_LINK;
			
			link_recv_pkt(&rpkt, data);
			if (rpkt.command_ID != CID_CTS) {
				if (rpkt.command_ID == CID_EXIT) return LERR_MEM;
				return LERR_LINK;
			}

			link_send_pkt(CID_ACK, NULL);
			TI_DATA s_data = {tivar->length, tivar->data};
			link->print	= 1;
			link->send_count = 0;
			link_send_pkt(CID_DATA, &s_data);
			if (link->print) printf("\n");
			link->print	= 0;
			link_recv_pkt(&rpkt, data);
			if (rpkt.command_ID != CID_ACK) return LERR_LINK;

			link_send_pkt(CID_EOT, NULL);
			return 0;
		}
		default:
			return err;
	}
}



int FindField(u8 *data,u8 Field) {
	int i;
	i=6;
	for(i=6;i<128;) {
		if (data[i++]!=0x80) return 0;
		if ((data[i]&0xF0)==Field) return i+1;
		i+=1+(data[i]&0x0F);
	}
	return 0;
}

int ReadIntelHex(FILE* ifile, intelhex_t *ihex) {
	u8	str[530];
	u32	i,size,addr,type,byte;
	memset(str,0x00,530);
	if (!fgets(str,530,ifile)) return 0;
	if (sscanf(str,":%02X%04X%02X",&size,&addr,&type) != 3) return 0;
	ihex->size		= size;
	ihex->address	= addr;
	ihex->type		= type;
	memset(ihex->data,0x00,256);
	for(i=0;i<size;i++) {
		if (sscanf(str+9+(i*2),"%02X",&byte)!=1) return 0;
		ihex->data[i] = byte;
	}
	if (sscanf(str+9+(i*2),"%02X",&byte)!=1) return 0;
	ihex->chksum = byte;
	return 1;
}


/* Send a flash application over the virtual link
 * On error: Returns an error code */
LINK_ERR link_send_app(FILE* infile) {
	intelhex_t ihex;
	int i,b;
	int	pages	= 0;
	int	curpage	= -1;
	u8	appname[9];
	u8 appheader[130];
	debugf("Start app\n");
	fseek(infile,TI_FLASH_HEADER_SIZE,SEEK_SET);
	
	// Find the first page, usually after the first line
	do {
		if (!ReadIntelHex(infile,&ihex)) {
			fclose(infile);
			return LERR_FORCELOAD;
		}
	}while(ihex.type!=2 || ihex.size != 2 || ihex.data[0]!=0 || ihex.data[1]!=0);
	
	//read app header
	for( i = 0; i < 128 ; ) {
		if ((!ReadIntelHex(infile,&ihex)) || ihex.type!=0) {
			fclose(infile);
			return LERR_FORCELOAD;
		}
		for( b = 0; (b < ihex.size) && (i < 128) ; b++ , i++ ) {
			appheader[i] = ihex.data[b];
		}
	}
	
	fseek(infile,TI_FLASH_HEADER_SIZE,SEEK_SET);
	
	// All apps require these bytes as the first
	// field.
	if (appheader[0]!=0x80 && appheader[1]!=0x0F) {
		fclose(infile);
		return LERR_FORCELOAD;
	}

	//Find App name
	if (!(i = FindField(appheader,0x40))) {
		fclose(infile);
		return LERR_FORCELOAD;
	}
	memset(appname,0x00,9);
	memcpy(appname,appheader+i,appheader[i-1]&0x0F);

	//Find pages
	if (!(i = FindField(appheader,0x80))) {
		fclose(infile);
		return LERR_FORCELOAD;
	}
	pages =  appheader[i];
	
	printf("Name: %s\n",appname);
	printf("Pages: %d\n",pages);
	
	u8 (*flash)[16384] = (u8 (*)[])mem->flash;
	int rompages = 0;
	for(curpage=0x15;curpage>=0x0A;) {
		if (flash[curpage][0]==0x80 && flash[curpage][1]==0x0F) {
		
			if (!(i = FindField(flash[curpage],0x80))) {
				fclose(infile);
				return LERR_FORCELOAD;
			}
			rompages = flash[curpage][i];
			
			if (!(i = FindField(flash[curpage],0x40))) {
				fclose(infile);
				return LERR_FORCELOAD;
			}
			
			if (memcmp(appname,&flash[curpage][i],8)==0) {
				if (pages == rompages) break;
				else {
					fclose(infile);
					return LERR_FORCELOAD;
				}
			} else {
				curpage -= rompages;
			}
		} else  {
			break;
		}
	}
	
	if (flash[curpage][0]!=0x80) {
		for(i=curpage;i>curpage-pages;i--) {
			for(b=0;b<16384;b++) {
				if (flash[i][b]!=0xFF) {
					fclose(infile);
					return LERR_FORCELOAD;
				}
			}
		}
	}
	
	for(i=curpage;i>curpage-pages;i--) {
		memset(&flash[i][0],0xFF,16384);
	}
	debugf("Start writing app\n");
	int startpage = curpage;
	int load = 1;
	while(load) {
		if (!ReadIntelHex(infile,&ihex)) {
			fclose(infile);
			return LERR_FORCELOAD;
		}
		switch(ihex.type) {
			case 0x00: {
				memcpy(flash[curpage]+(ihex.address&0x3FFF),ihex.data,ihex.size);
				break;
			}
			case 0x01: {
				fclose(infile);
				load = 0;
				break;
			}
			case 0x02: {
				if (ihex.size==2) {
					i = ((ihex.data[0]<<8)+ihex.data[1]);
					curpage = startpage-i;
					printf("Page %d of %d pages\n",i+1,pages);
				} else {
					fclose(infile);
					return LERR_FORCELOAD;
				}
				break;
			}
			default: {
				fclose(infile);
				return LERR_FORCELOAD;
			}
		}
	}
	
//	calc_erase_certificate();
	
	//Now send via link
	debugf("Start sending app\n");
	int err;
	switch (err = setjmp(exc_pkt)) {
		case 0: {
			TI_PKTHDR rpkt;
			u_char data[64];
			printf("Validating, Please wait...\n");
			// Send the version request
			link_send_pkt(CID_VER, NULL);

			// Receive the ACK
			link_recv_pkt(&rpkt, data);
			if (rpkt.command_ID != CID_ACK) return LERR_LINK;
			
			// Send the CTS
			link_send_pkt(CID_CTS, NULL);
			
			// Receive the ACK
			link_recv_pkt(&rpkt, data);
			if (rpkt.command_ID != CID_ACK) return LERR_LINK;

			// Receive the version
			link_recv_pkt(&rpkt, data);
			
			// Send the ACK
			link_send_pkt(CID_ACK, NULL);
			
			// Send the ready request
			link_send_pkt(CID_RDY, NULL);
			
			// Receive the ACK
			link_recv_pkt(&rpkt, data);
			if (rpkt.command_ID != CID_ACK) return LERR_LINK;
			
			uint16_t page, offset;
			page = 0;
			offset = 0;
			
			for(i=0;i<1;i++) {
				
//				printf("page: 00 offset: %04x \n",(0x80*i));
				
				// Send the flash header
				TI_FLASHHDR flash_hdr;
				flash_hdr.sizeLSB 	= link_endian(0x0080);
				flash_hdr.type_ID 	= FlashObj;
				flash_hdr.sizeMSB 	= link_endian(0x0000);
				flash_hdr.offset 	= link_endian((0x80*i) + 0x4000);
				flash_hdr.page 		= link_endian(page);
				link_send_pkt(CID_VAR, &flash_hdr);
				
				// Receive the ACK
				link_recv_pkt(&rpkt, data);
				if (rpkt.command_ID != CID_ACK) return LERR_LINK;
				
				// Receive the CTS
				link_recv_pkt(&rpkt, data);
				if (rpkt.command_ID != CID_CTS) {
					if (rpkt.command_ID == CID_EXIT) return LERR_MEM;
					return LERR_LINK;
				}
				
				// Send the ACK
				link_send_pkt(CID_ACK, NULL);
				
				// Send the data packet
				TI_DATA s_data = {0x0080, appheader+(0x80*i)};
				link_send_pkt(CID_DATA, &s_data);
				
				link_wait(CPU_SPEED/8);
				
				// Receive the ACK
				link_recv_pkt(&rpkt, data);
				if (rpkt.command_ID != CID_ACK) return LERR_LINK;
			}

			return 0;
			link_send_pkt(CID_EOT, NULL);
			
			link_recv_pkt(&rpkt, data);
			if (rpkt.command_ID != CID_ACK) return LERR_LINK;
			return 0;
		}
		default:
			return err;
	}
}





int SendFile(char * filename) {
	FILE *infile;
	TIVAR tivar;
	int i,b;
	u8 string[9];
	link->print	= 0;
	debugf("Send file started\n");
	
	infile = fopen(filename,"rb");
	if (infile == NULL) return 1;
	memset(string,0,9);
	fread(string,1,8,infile);

	debugf("h: %s\n",string);

	// Test if a flash file type
	if ( memcmp(string,"**TIFL**",8)==0 ) {
		debugf("flash file\n");
		//0x31 says if its an OS or App
		fseek(infile,0x31,SEEK_SET);
		switch(fgetc(infile)) {
			case 0x23: { //os
				debugf("OS\n");
				fclose(infile);
				return 1;
			}
			case 0x24: { //app
				debugf("APP\n");
				i = link_send_app(infile);
				debugf("Done sending app\n");
				return i;
			}
			default: {
				debugf("Other\n");
				fclose(infile);
				return 1;
			}
		}
	} else if ( memcmp(string,"**TI83F*",8)==0 ) {
		if (fseek(infile,TI_FILE_HEADER_SIZE,SEEK_SET)) {
			fclose(infile);
			return LERR_FILE;
		}
		if (fread(&tivar,1,TI_VAR_HEADER_SIZE,infile)!=TI_VAR_HEADER_SIZE) {
			fclose(infile);
			return LERR_FILE;
		}
		tivar.version	=  fgetc(infile);
		tivar.flag		=  fgetc(infile);
		tivar.length2	=  fgetc(infile);
		tivar.length2	+= fgetc(infile)<<8;
		if (!(tivar.data = (u8 *) malloc(tivar.length))) {
			fclose(infile);
			return;
		}
		fread(tivar.data,1,tivar.length,infile);
		fclose(infile);
		i = link_send_var(infile,&tivar,SEND_ARC);
		free(tivar.data);
		return i;
	} else if ( (memcmp(string,"**TI83**",8)==0) || 
				(memcmp(string,"**TI82**",8)==0) ) {
		if (fseek(infile,TI_FILE_HEADER_SIZE,SEEK_SET)) {
			fclose(infile);
			return LERR_FILE;
		}
		if (fread(&tivar,1,TI_VAR_HEADER_SIZE,infile)!=TI_VAR_HEADER_SIZE) {
			fclose(infile);
			return LERR_FILE;
		}
		tivar.version	=  0;
		tivar.flag		=  0;
		tivar.length2	=  fgetc(infile);
		tivar.length2	+= fgetc(infile)<<8;
		if (!(tivar.data = (u8 *) malloc(tivar.length))) {
			fclose(infile);
			return;
		}
		fread(tivar.data,1,tivar.length,infile);
		fclose(infile);
		i = link_send_var(infile,&tivar,SEND_ARC);
		free(tivar.data);
		return i;
	} else {
		debugf("something else\n");
		fclose(infile);
		return 1;
	}
}




#ifdef DEBUG
static void print_command_ID(uint8_t command_ID) {
	char buffer[256];
	switch (command_ID) {
		case CID_ACK:
			strcpy(buffer, "ACK");
			break;
		case CID_CTS:
			strcpy(buffer, "CTS");
			break;
		case CID_DATA:
			strcpy(buffer, "DATA");
			break;
		case CID_DEL:
			strcpy(buffer, "DEL");
			break;
		case CID_EOT:
			strcpy(buffer, "EOT");
			break;
		case CID_ERR:
			strcpy(buffer, "ERR");
			break;
		case CID_EXIT:
			strcpy(buffer, "SKIP/EXIT");
			break;
		case CID_RDY:
			strcpy(buffer, "RDY");
			break;
		case CID_REQ:
			strcpy(buffer, "REQ");
			break;
		case CID_RTS:
			strcpy(buffer, "RTS");
			break;
		case CID_SCR:
			strcpy(buffer, "SCR");
			break;
		case CID_VAR:
			strcpy(buffer, "VAR");
			break;
		case CID_VER:
			strcpy(buffer, "VER");
			break;
		default:
			strcpy(buffer, "error");
			break;
	}
	printf(buffer);
}
#endif






void Load_8xu(FILE* infile) {
	intelhex_t ihex;
	if (!infile) return;

	//0x31 says if its an OS or App
	fseek(infile,0x31,SEEK_SET);
	if (fgetc(infile) != 0x23) {
		puts("[0x31] != 0x23");
		return;
	}

	fseek(infile,TI_FLASH_HEADER_SIZE,SEEK_SET);
	
	// Find the first page, usually after the first line
	do {
		if (!ReadIntelHex(infile,&ihex)) return;
	}while(ihex.type!=0x02 || ihex.size != 2);
	
	int curpage = ((ihex.data[0]<<8)|ihex.data[1])&0x1F;
	u8 (*flash)[16384] = (u8 (*)[])calc.rom;
	

	printf("Loading IntelHex OS\n",curpage);
	printf("Loading page %02Xh\n",curpage);
	while(1) {
		if (!ReadIntelHex(infile,&ihex)) {
			return;
		}
		switch(ihex.type) {
			case 0x00:
				memcpy(flash[curpage]+(ihex.address&0x3FFF),ihex.data,ihex.size);
				break;
			case 0x02:
				if (ihex.size==2) curpage = ((ihex.data[0]<<8)+ihex.data[1])&0x1F;
				printf("Loading page %02Xh\n",curpage);
				break;
			default:
				return;
		}
	}
}
