/*
** nester - NES emulator
** Copyright (C) 2000  Darren Ranalli
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful, 
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
** Library General Public License for more details.  To obtain a 
** copy of the GNU Library General Public License, write to the Free 
** Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
** Any permitted reproduction of these routines, in whole or in part,
** must bear this legend.
*/

#include <string.h>
#include <math.h>
#include "NES.h"
#include "NES_screen_mgr.h"
#include "NES_ROM.h"
#include "ppu/NES_PPU.h"
#include "../pixmap.h"
#include "SNSS.h"
//#include "win32_directory.h"
#include "../GBA_globals.h"
#include "../arm7_shared.h"

#include "../debug/debug.h"

#ifndef _WIN32
#include <nds/arm9/console.h> //DSPIP2
#endif

static float  ideal_cycle_count NESTER_DTCM;   // number of cycles that should have executed so far
static uint32 emulated_cycle_count NESTER_DTCM;  // number of cycles that have executed so far

#define INLINE __inline//static

// internal memory
static uint8 RAM[0x800] NESTER_DTCM;

uint8 *NES_GetRAM(void) {
	return RAM;
}

// joypad stuff
//NES_pad* pad1 NESTER_DTCM;
//NES_pad* pad2 NESTER_DTCM;
static boolean  pad_strobe NESTER_DTCM;
static uint8 pad1_bits NESTER_DTCM;
static uint8 pad2_bits NESTER_DTCM;

nesMapper_t *NES_mapper;
uint8 NES_SaveRAM[0x2000];

uint8 NES_RGB_pal[NES_NUM_COLORS][3];

NESTER_STATIC_VAR_EXPL_Q const uint8 NES_preset_palette[NES_NUM_COLORS][3] = 
{
// include the NES palette
#include "NES_pal.h"
};

NESTER_STATIC_VAR_EXPL_Q NESTER_VAR_CONST float CYCLES_PER_LINE = ((float)((double)1364.0/(double)12.0));

//#define _COLOR_CODE_DEBUG

int NES_FakeConstructor(void)
{
	const char *x = "x";

#ifdef _COLOR_CODE_DEBUG
	int i;
	uint16 *frameBuffer = VRAM_A;
	for (i = 0; i < (SCREEN_WIDTH*SCREEN_HEIGHT); i++) 
	{
		frameBuffer[i] = RGB15(31,0,0);
	}
#endif
  
	if (1)
	{
		//scr_mgr = _screen_mgr;
		//snd_mgr = _sound_mgr;

		//SCRMGR_setParentNES(this);
	}
	else
	{
		//scr_mgr = 0;
		//snd_mgr = 0;
	}

	GBA_Log("nester nes emulator by\ndarren ranalli (c) 2000\ngba port by\nrich whitehouse\n");

	//cpu = p_cpu;
	//ppu = p_ppu;
	//apu_arm7
	//apu = p_apu;

#ifdef _COLOR_CODE_DEBUG
	for (i = 0; i < (SCREEN_WIDTH*SCREEN_HEIGHT); i++) 
	{
		frameBuffer[i] = RGB15(31,31,0);
	}
#endif

	CPUINT_FakeConstructor();

#ifdef _COLOR_CODE_DEBUG
	for (i = 0; i < (SCREEN_WIDTH*SCREEN_HEIGHT); i++) 
	{
		frameBuffer[i] = RGB15(0,31,0);
	}
#endif

	PPU_FakeConstructor();
//	apu->FakeConstructor(); apu_arm7

#ifdef _COLOR_CODE_DEBUG
	for (i = 0; i < (SCREEN_WIDTH*SCREEN_HEIGHT); i++) 
	{
		frameBuffer[i] = RGB15(0,0,31);
	}
#endif
	if (!NES_loadROM(x))
	{
		return 0;
	}

	// set up palette and assert it
	NES_calculate_palette();
	SCRMGR_assert_palette();
//	pad1 = 0;
//	pad2 = 0;

	return 1;
}

extern int CheckROMStatus(void);
int NES_loadROM(const char* fn)
{
  if (CheckROMStatus() <= 0)
  {
	  return 0;
  }
  GBA_Log("loading rom...");
//if (1) return 1;
//  ROM = new NES_ROM;
//  if (!ROM)
//  {
//	  return 0;
//  }
  ROM_OldConstructFunc();

  // set up the mapper
  NES_mapper = GetMapper();
  if(!NES_mapper)
  {
    // unsupported mapper
    //LOG("mapper #" << (int)ROM->get_mapper_num() << " not supported" << endl);
	GBA_Log("unsupported mapper");

//    delete ROM;
//    ROM = 0;

    return 0;
  }
  NES_mapper->FakeConstructor();
  GBA_Log("done\n");

//  LOG(ROM->GetRomName() << ".nes: #" << (int)ROM->get_mapper_num() << " ");
  switch(ROM_get_mirroring())
  {
    case MIRROR_HORIZ:
      LOG("H ");
      break;
    case MIRROR_VERT:
      LOG("V ");
      break;
    case MIRROR_FOUR_SCREEN:
      LOG("F ");
      break;
  }

  if(ROM_has_save_RAM())
  {
    LOG("S ");
  }

  if(ROM_has_trainer())
  {
    LOG("T ");
  }

//  LOG(16*ROM->get_num_16k_ROM_banks() << "K/" << 8*ROM->get_num_8k_VROM_banks() << "K " << endl);

  NES_Load_SaveRAM();

  NES_reset();

  //LOG("Starting emulation...\n");
  GBA_Log("starting emulation");

  return 1;
}

void NES_freeROM()
{
  NES_Save_SaveRAM();

  LOG("Freeing ROM...");
//  if(ROM)
//  {
//    delete ROM;
//    ROM = 0;
//  }
  if(NES_mapper)
  {
    //delete mapper;
	free(NES_mapper);
    NES_mapper = 0;
  }

  SCRMGR_clear(0x00);
  SCRMGR_blt();

  LOG("Done\n");
  LOG(endl);
}

const char* NES_getROMname()
{
  return ROM_GetRomName();
}

const char* NES_getROMnameExt()
{
  return ROM_GetRomNameExt();
}

const char* NES_getROMpath()
{
  return ROM_GetRomPath();
}

void NES_reset()
{
  LOG("Resetting NES\n");

  // make sure saveRAM is saved
  NES_Save_SaveRAM();

  // RAM
  memset_gba(RAM, 0x00, sizeof(RAM));

  // SaveRAM
  NES_Load_SaveRAM();

  // set up CPU
  {
    CPUINT_Context context;

    memset_gba((void*)&context, 0x00, sizeof(context));
    CPUINT_GetContext(&context);

    context.mem_page[0] = RAM;
    context.mem_page[3] = NES_SaveRAM;

    CPUINT_SetContext(&context);
  }

  // set up the trainer if present
  if(ROM_has_trainer())
  {
    LOG("  Setting trainer...");
    // trainer is located at 0x7000; SaveRAM is 0x2000 bytes at 0x6000
    memcpy_gba(&NES_SaveRAM[0x1000], ROM_get_trainer(), ROM_TRAINER_LEN);
    LOG("OK\n");
  }

  LOG("  PPU reset...");
  // reset the PPU
  PPU_reset();
  LOG("OK\n");

  //apu_arm7
  /*
  LOG("  APU reset...");
  // reset the APU
  apu->reset();
  LOG("OK\n");
  */

  if(NES_mapper)
  {
    LOG("  mapper reset...");
    // reset the mapper
    NES_mapper->Reset();
    LOG("OK\n");
  }

  LOG("  CPU reset...");
  // reset the CPU
  CPUINT_Reset();
  LOG("OK\n");

  ideal_cycle_count  = 0.0;
  emulated_cycle_count = 0;

  pad_strobe = FALSE;
  pad1_bits = 0x00;
  pad2_bits = 0x00;
}

#ifdef _SHOW_LAST_OP
#define _LASTLINEPRINT
#endif

#ifdef _DTCM_VID_BUFFER
#ifndef _DMAVIDCOPY
static inline void VidMemCpy(uint8 *bIn, uint8 *bOut)
{
	uint32 *in = (uint32 *)bIn;
	uint32 *out = (uint32 *)bOut;
	int i;
	for (i = 0; i < 64; i++) {
		in[i] = out[i];
	}
}
#endif
#endif

//#define _EMU_FRAME_BENCH
#ifdef _EMU_FRAME_BENCH
extern int GetRawTime(void);
#endif

boolean NES_emulate_frame(boolean draw)
{
  uint32 i;
  pixmap p;
  uint8* cur_line = NULL; // ptr to screen buffer
  boolean retval = draw;
#ifdef _DTCM_VID_BUFFER
#define TOP_MARGIN 8
 #ifdef _DMAVIDCOPY
  uint8 dmaChannel = 0;
 #endif
  uint8 *vramOut = (uint8 *)BG_BMP_RAM(0);
#endif
#ifdef _EMU_FRAME_BENCH
  int start, end;
  start = GetRawTime();
#endif

  //testing apu sync
  //while (g_sharedPData->apuFrames > 0);

#ifndef _WIN32
  if (g_sharedPData->nesBurn != 0)
  { //burn cycles if the arm7 says so
	  nes6502_burn(g_sharedPData->nesBurn);
	  g_sharedPData->nesBurn = 0;
  }
#endif

  NES_trim_cycle_counts();

  // do frame
  PPU_start_frame();

  if (!globals->speedHack)
  {
	if(retval)
	{
		if(!SCRMGR_lock(&p))
		retval = FALSE;
		else
		cur_line = p.data;
	}

		// LINES 0-239
		for(i = 0; i < NES_NUM_FRAME_LINES; i++)
		{
			// do one line's worth of CPU cycles
			NES_emulate_CPU_cycles(CYCLES_PER_LINE);
			NES_mapper->HSync(i);
			if(retval)
			{
				// render line
				PPU_do_scanline_and_draw(cur_line);
				// point to next line
#ifdef _DTCM_VID_BUFFER
				if (i > TOP_MARGIN)
				{
#ifdef _DMAVIDCOPY
					while (dmaBusy(dmaChannel));
					dmaCopyWordsAsynch(dmaChannel, cur_line+SIDE_MARGIN, vramOut, 256);
					dmaChannel = (dmaChannel+1)%4;
					vramOut += 512;
#else
					if (globals->hwScale || i >= globals->heightOffset)
					{
						VidMemCpy(vramOut, cur_line+SIDE_MARGIN);
						vramOut += 512;
					}
#endif
				}
#else
				cur_line += p.pitch;
#endif
			}
			else
			{
				PPU_do_scanline_and_dont_draw();
			}
		}

	if(retval)
	{
		SCRMGR_unlock();
	}
  }
  else
  { //this is the horrible hacked way that breaks things for speedup.
	NES_emulate_CPU_cycles(CYCLES_PER_LINE*NES_NUM_FRAME_LINES);

	if (retval)
	{
		SCRMGR_lock(&p);
		cur_line = p.data;

		for(i = 0; i < NES_NUM_FRAME_LINES; i++)
		{
			PPU_do_scanline_and_draw(cur_line);
#ifdef _DTCM_VID_BUFFER
			if (i > TOP_MARGIN)
			{
#ifdef _DMAVIDCOPY
				while (dmaBusy(dmaChannel));
				dmaCopyWordsAsynch(dmaChannel, cur_line+SIDE_MARGIN, vramOut, 256);
				dmaChannel = (dmaChannel+1)%4;
				vramOut += 512;
#else
				if (globals->hwScale || i >= globals->heightOffset)
				{
					VidMemCpy(vramOut, cur_line+SIDE_MARGIN);
					vramOut += 512;
				}
#endif
			}
#else
			cur_line += p.pitch;
#endif
		}

		SCRMGR_unlock();
	}
  }

  PPU_end_frame();

  for(i = 240; i <= 261; i++)
  {
    if(i == 241)
    {
      // do v-blank
      PPU_start_vblank();
      NES_mapper->VSync();
    }
    else if(i == 261)
    {
      PPU_end_vblank();
    }

    if(i == 241)
    {
      // 1 instruction between vblank flag and NMI
      NES_emulate_CPU_cycles(1.0);
      if(PPU_NMI_enabled()) CPUINT_DoNMI();
      NES_emulate_CPU_cycles((float)CYCLES_PER_LINE-(float)1.0);
      NES_mapper->HSync(i);
      continue;
    }

    NES_emulate_CPU_cycles(CYCLES_PER_LINE);
    NES_mapper->HSync(i);
  }


/*
  // NES DOES NOT DO THIS
  // HALF-LINE 262.5
  emulate_CPU_cycles(CYCLES_PER_LINE/2);
*/

  //apu_arm7
//  apu->DoFrame();

  //update cycles for arm7 logic
#ifndef _WIN32
  g_sharedPData->nesCycles = nes6502_getcycles(FALSE);
#endif

#ifdef _EMU_FRAME_BENCH
  end = GetRawTime();
  char test[128];
  sprintf(test, "\n%i     \n", end-start);
  consolePrintf(test);
#endif

  return retval;
}

void NES_onFreeze()
{
	//apu_arm7
  //apu->freeze();
}

void NES_onThaw()
{
	//apu_arm7
  //apu->thaw();
}

uint8 NES_MemoryRead(uint32 addr)
{
//  LOG("Read " << HEX(addr,4) << endl);

  if(addr < 0x2000) // RAM
  {
    return NES_ReadRAM(addr);
  }
  else if(addr < 0x4000) // low registers
  {
    return NES_ReadLowRegs(addr);
  }
  else if(addr < 0x4018) // high registers
  {
    return NES_ReadHighRegs(addr);
  }
  else if(addr < 0x6000) // mapper low
  {
    //LOG("MAPPER LOW READ: " << HEX(addr,4) << endl);
    return((uint8)(addr >> 8)); // what's good for conte is good for me
  }
  else // save RAM, or ROM (mapper 40)
  {
    return CPUINT_GetByte(addr);
  }
}

//#define _LOG_CPU

void NES_MemoryWrite(uint32 addr, uint8 data)
{
//  LOG("Write " << HEX(addr,4) << " " << HEX(data,2) << endl);

  if(addr < 0x2000) // RAM
  {
    NES_WriteRAM(addr, data);
#ifdef _LOG_CPU
	GBA_Log("ram");
#endif
  }
  else if(addr < 0x4000) // low registers
  {
    NES_WriteLowRegs(addr, data);
#ifdef _LOG_CPU
	char inf[1024];
	sprintf(inf, "lowreg %i %i", addr, data);
	GBA_Log(inf);
#endif
  }
  else if(addr < 0x4018) // high registers
  {
    NES_WriteHighRegs(addr, data);
#ifdef _LOG_CPU
	GBA_Log("highreg");
#endif
  }
  else if(addr < 0x6000) // mapper low
  {
    NES_mapper->MemoryWriteLow(addr, data);
#ifdef _LOG_CPU
	GBA_Log("lowmap");
#endif
  }
  else if(addr < 0x8000) // save RAM
  {
    NES_SaveRAM[addr - 0x6000] = data;
    NES_mapper->MemoryWriteSaveRAM(addr, data);
#ifdef _LOG_CPU
	GBA_Log("sram");
#endif
  }
  else // mapper
  {
#ifdef _LOG_CPU
	char inf[1024];
	sprintf(inf, "mapper %i %i", addr, data);
	globals->logOps = 1;
	GBA_Log(inf);
#endif
    NES_mapper->MemoryWrite(addr, data);
  }
}


uint8 NES_ReadRAM(uint32 addr)
{
  return RAM[addr & 0x7FF];
}

void NES_WriteRAM(uint32 addr, uint8 data)
{
  RAM[addr & 0x7FF] = data;
}


uint8 NES_ReadLowRegs(uint32 addr)
{
  return PPU_ReadLowRegs(addr & 0xE007);
}

void NES_WriteLowRegs(uint32 addr, uint8 data)
{
  PPU_WriteLowRegs(addr & 0xE007, data);
}

//DSPIP
INLINE uint8 DS_APURead(uint32 address)
{
#ifdef _WIN32
	return 0;
#else
   uint8 value;
   volatile vitalAPUData_t *apushare = &g_sharedPData->apuData;

   switch (address)
   {
	#define  APU_SMASK      0x4015
   case APU_SMASK:
      value = 0;
      // Return 1 in 0-5 bit pos if a channel is playing
      if (apushare->rect0Enabled && apushare->rect0VblLen)
         value |= 0x01;
      if (apushare->rect1Enabled && apushare->rect1VblLen)
         value |= 0x02;
      if (apushare->triEnabled && apushare->triVblLen)
         value |= 0x04;
      if (apushare->noiseEnabled && apushare->noiseVblLen)
         value |= 0x08;

      // bodge for timestamp queue 
      if (apushare->dmcEnabled)
         value |= 0x10;

      if (apushare->dmcIrqOccured)
         value |= 0x80;

      break;

   default:
      value = (address >> 8); // heavy capacitance on data bus 
      break;
   }

   return value;
#endif
}

//DSPIP
void DS_APUWrite(uint32 address, uint8 data)
{
#ifndef _WIN32
	volatile apuIncoming_t *apuInc = &g_sharedPData->apuIncoming[g_sharedPData->apuNumIncoming];
	apuInc->timestamp = nes6502_getcycles(FALSE);
	apuInc->address = address;
	apuInc->value = data;
	g_sharedPData->apuNumIncoming++;

	//DSPIP2
	if (g_sharedPData->apuNumIncoming >= MAX_PENDING_APU_COUNT)
	{ //process now
		consolePrintf("QUEUE OVERFLOW\n");
		while (g_sharedPData->apuNumIncoming > 0); //wait
		consolePrintf("QUEUE PROCESSED\n");
	}
#endif
}

uint8 NES_ReadHighRegs(uint32 addr)
{
  if(addr == 0x4014) // SPR-RAM DMA
  {
    LOG("Read from SPR-RAM DMA reg??" << endl);
    return PPU_Read0x4014();
  }
  else if(addr < 0x4016) // APU
  {
//    LOG("APU READ:" << HEX(addr,4) << endl);
    //return apu->Read(addr);
	  //DSPIP
	  return DS_APURead(addr);
  }
  else // joypad regs
  {
    uint8 retval;

    if(addr == 0x4016)
    {
      // joypad 1
      retval = pad1_bits & 0x01;
      pad1_bits >>= 1;
    }
    else
    {
      // joypad 2
      retval = pad2_bits & 0x01;
      pad2_bits >>= 1;
    }
    return retval;
  }
}

void NES_WriteHighRegs(uint32 addr, uint8 data)
{
  if(addr == 0x4014) // SPR-RAM DMA
  {
    PPU_Write0x4014(data);
    // sprite DMA takes XXX cycles
    CPUINT_SetDMA(512);
  }
  else if(addr < 0x4016) // APU
  {
//    LOG("APU WRITE" << endl);
    //apu->Write(addr, data);
	  //DSPIP
	  DS_APUWrite(addr, data);
  }
  else // joypad regs
  {
    // bit 0 == joypad strobe
    if(data & 0x01)
    {
      pad_strobe = TRUE;
    }
    else
    {
      if(pad_strobe)
      {
        pad_strobe = FALSE;
        // get input states
        //if(pad1) pad1_bits = pad1->get_inp_state();
        //if(pad2) pad2_bits = pad2->get_inp_state();
		pad1_bits = PAD_get_inp_state();
		pad2_bits = 0;
      }
    }
  }
}

void NES_emulate_CPU_cycles(float num_cycles)
{
  uint32 cycle_deficit;

  ideal_cycle_count += num_cycles;
  cycle_deficit = ((uint32)ideal_cycle_count) - emulated_cycle_count;
  if(cycle_deficit > 0)
  {
    emulated_cycle_count += CPUINT_Execute(cycle_deficit);
  }
}

// call every once in a while to avoid cycle count overflow
void NES_trim_cycle_counts()
{
  uint32 trim_amount;

  trim_amount = (uint32)floor(ideal_cycle_count);
  if(trim_amount > emulated_cycle_count) trim_amount = emulated_cycle_count;

  ideal_cycle_count  -= (float)trim_amount;
  emulated_cycle_count -= trim_amount;
}


void NES_Save_SaveRAM()
{
	/*
  // does the ROM use save ram?
  if(!ROM_has_save_RAM()) return;

  // has anything been written to Save RAM?
  for(uint32 i = 0; i < sizeof(SaveRAM); i++)
  {
    if(SaveRAM[i] != 0x00) break;
  }
  if(i < sizeof(SaveRAM))
  {
    FILE* fp = 0;
    char fn[_MAX_PATH];

    LOG("Saving Save RAM...");

    DIR_createNesFileName(ROM, fn, NESTER_settings.nes.preferences.saveRamDirType,
      NESTER_settings.nes.preferences.saveRamDir, ROM_GetRomName(), ".sav");

    try
    {
      fp = fopen(fn, "wb");
      if(!fp) throw "can't open save RAM file";

      if(fwrite(SaveRAM, sizeof(SaveRAM), 1, fp) != 1)
        throw "can't open save RAM file";

      fclose(fp);
      LOG("Done." << endl);

    } catch(...) {
      LOG("can't save" << endl);
      if(fp) fclose(fp);
    }
  }
  */
//rwwWRITEME: hm, maybe i can write this to sram?
}

void NES_Load_SaveRAM()
{
/*
  memset_gba(SaveRAM, 0x00, sizeof(SaveRAM));

  // does the ROM use save ram?
  if(!ROM_has_save_RAM()) return;

  {
    FILE* fp = 0;
    char fn[_MAX_PATH];

    DIR_createNesFileName(ROM, fn, NESTER_settings.nes.preferences.saveRamDirType,
      NESTER_settings.nes.preferences.saveRamDir, ROM_GetRomName(), ".sav");

    try
    {
      fp = fopen(fn, "rb");
      if(!fp) throw "none found.";

      LOG("Loading Save RAM...");

      if(fread(SaveRAM, sizeof(SaveRAM), 1, fp) != 1)
      {
        LOG("error reading Save RAM file" << endl);
        throw "error reading Save RAM file";
      }

      fclose(fp);
      LOG("Done." << endl);

    } catch(...) {
      if(fp) fclose(fp);
    }
  }
  */
//rwwWRITEME: hm, maybe i can load this from sram?
	memset_gba(NES_SaveRAM, 0x00, sizeof(NES_SaveRAM));
}

boolean NES_loadState(const char* fn)
{
//  return LoadSNSS(fn, this);
	return FALSE;
}

boolean NES_saveState(const char* fn)
{
//  return SaveSNSS(fn, this);
	return FALSE;
}

void NES_calculate_palette()
{
  if(GBAOPTION_CALCULATE_PALETTE)
  {
    int x,z;
    float tint = ((float)GBAOPTION_TINT) / 256.0f;
    float hue = 332.0f + (((float)GBAOPTION_HUE - (float)0x80) * (20.0f / 256.0f));
    float s,y;
    int cols[16] = {0,240,210,180,150,120,90,60,30,0,330,300,270,0,0,0};
    float theta;
    float br1[4] = {0.5f, 0.75f, 1.0f, 1.0f};
    float br2[4] = {0.29f, 0.45f, 0.73f, 0.9f};
    float br3[4] = {0.0f, 0.24f, 0.47f, 0.77f};
    float r,g,b;

    for(x = 0; x <= 3; x++)
    {
      for(z = 0; z <= 15; z++)
      {
        s = tint;
        y = br2[x];
        if(z == 0)
        {
          s = 0;
          y = br1[x];
        }
        else if(z == 13)
        {
          s = 0;
          y = br3[x];
        }
        else if((z == 14) || (z == 15))
        {
          s = 0;
          y = 0;
        }

        theta = 3.14159265f * (((float)(cols[z] + hue)) / 180.0f);

        r = y + (s * (float)sin(theta));
        g = y - ((27.0f / 53.0f) * s * (float)sin(theta)) + ((10.0f / 53.0f) * s * (float)cos(theta));
        b = y - (s * (float)cos(theta));

        r = r * 256.0f;
        g = g * 256.0f;
        b = b * 256.0f;

        if(r > 255.0f) r = 255.0f;
        if(g > 255.0f) g = 255.0f;
        if(b > 255.0f) b = 255.0f;

        if(r < 0.0f) r = 0.0;
        if(g < 0.0f) g = 0.0;
        if(b < 0.0f) b = 0.0;

        NES_RGB_pal[(x*16) + z][0] = (uint8)r;
        NES_RGB_pal[(x*16) + z][1] = (uint8)g;
        NES_RGB_pal[(x*16) + z][2] = (uint8)b;
      }
    }
  }
  else
  {
    memcpy_gba(NES_RGB_pal, NES_preset_palette, sizeof(NES_RGB_pal));
  }

  if(GBAOPTION_BLACK_AND_WHITE)
  {
    int i;

    for(i = 0; i < NES_NUM_COLORS; i++)
    {
      uint8 Y;
      Y = (uint8)(((float)NES_RGB_pal[i][0] * 0.299) +
                  ((float)NES_RGB_pal[i][1] * 0.587) +
                  ((float)NES_RGB_pal[i][2] * 0.114));
      NES_RGB_pal[i][0] = Y;
      NES_RGB_pal[i][1] = Y;
      NES_RGB_pal[i][2] = Y;
    }
  }
}
