/*
** nester - NES emulator
** Copyright (C) 2000  Darren Ranalli
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful, 
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
** Library General Public License for more details.  To obtain a 
** copy of the GNU Library General Public License, write to the Free 
** Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
** Any permitted reproduction of these routines, in whole or in part,
** must bear this legend.
*/

#ifndef _NES_ROM_H_
#define _NES_ROM_H_

#include <stdio.h>

#include "../types.h"
#include "ppu/NES_PPU.h"

#define ROMINLINE __inline

  struct NES_header
  {
#if !defined(__GNUC__)
    unsigned char id[3]; // 'NES'
    unsigned char ctrl_z; // control-z
    unsigned char num_16k_rom_banks;
    unsigned char num_8k_vrom_banks;
    unsigned char flags_1;
    unsigned char flags_2;
    unsigned char reserved[8];
#else
    unsigned char id[3]						__attribute__((packed)) ; // 'NES'
    unsigned char ctrl_z					__attribute__((packed)) ; // control-z
    unsigned char num_16k_rom_banks			__attribute__((packed)) ;
    unsigned char num_8k_vrom_banks			__attribute__((packed)) ;
    unsigned char flags_1					__attribute__((packed)) ;
    unsigned char flags_2					__attribute__((packed)) ;
    unsigned char reserved[8]				__attribute__((packed)) ;
#endif
  };

  extern struct NES_header ROM_header;
  extern uint8 ROM_mapper;

  extern uint8* ROM_trainer;
  extern uint8* ROM_ROM_banks;
  extern uint8* ROM_VROM_banks;

  extern char* ROM_rom_name; // filename w/out ".NES"
  extern char* ROM_rom_name_ext; // filename w/ ".NES"
  extern char* ROM_rom_path; // rom file path

  void ROM_OldConstructFunc(void);

  enum {
    ROM_TRAINER_ADDRESS = 0x7000,
    ROM_TRAINER_LEN     = 512
  };

  enum {
    ROM_MASK_VERTICAL_MIRRORING = 0x01,
    ROM_MASK_HAS_SAVE_RAM       = 0x02,
    ROM_MASK_HAS_TRAINER        = 0x04,
    ROM_MASK_4SCREEN_MIRRORING  = 0x08
  };

  static ROMINLINE mirroring_type ROM_get_mirroring()
  {
    if(ROM_header.flags_1 & ROM_MASK_4SCREEN_MIRRORING)
    {
      return MIRROR_FOUR_SCREEN;
    }
    else if(ROM_header.flags_1 & ROM_MASK_VERTICAL_MIRRORING)
    {
      return MIRROR_VERT;
    }
    else
    {
      return MIRROR_HORIZ;
    }
  }

  static ROMINLINE uint8 ROM_get_mapper_num() { return ROM_mapper; }

  static ROMINLINE boolean  ROM_has_save_RAM()  { return ROM_header.flags_1 & ROM_MASK_HAS_SAVE_RAM; }
  static ROMINLINE boolean  ROM_has_trainer()   { return ROM_header.flags_1 & ROM_MASK_HAS_TRAINER;  }

  static ROMINLINE uint8 ROM_get_num_16k_ROM_banks() { return ROM_header.num_16k_rom_banks; }
  static ROMINLINE uint8 ROM_get_num_8k_VROM_banks() { return ROM_header.num_8k_vrom_banks; }

  static ROMINLINE uint8* ROM_get_trainer()    { return ROM_trainer;     }
  static ROMINLINE uint8* ROM_get_ROM_banks()  { return ROM_ROM_banks;   }
  static ROMINLINE uint8* ROM_get_VROM_banks() { return ROM_VROM_banks;  }

  static ROMINLINE const char* ROM_GetRomName()    { return ROM_rom_name; }
  static ROMINLINE const char* ROM_GetRomNameExt() { return ROM_rom_name_ext; }
  static ROMINLINE const char* ROM_GetRomPath()    { return ROM_rom_path; }

  void ROM_GetPathInfo(const char* fn);


/*
 .NES file format
---------------------------------------------------------------------------
0-3      String "NES^Z" used to recognize .NES files.
4        Number of 16kB ROM banks.
5        Number of 8kB VROM banks.
6        bit 0     1 for vertical mirroring, 0 for horizontal mirroring
         bit 1     1 for battery-backed RAM at $6000-$7FFF
         bit 2     1 for a 512-byte trainer at $7000-$71FF
         bit 3     1 for a four-screen VRAM layout 
         bit 4-7   Four lower bits of ROM Mapper Type.
7        bit 0-3   Reserved, must be zeroes!
         bit 4-7   Four higher bits of ROM Mapper Type.
8-15     Reserved, must be zeroes!
16-...   ROM banks, in ascending order. If a trainer is present, its
         512 bytes precede the ROM bank contents.
...-EOF  VROM banks, in ascending order.
---------------------------------------------------------------------------
*/

#endif
