
#include <nds.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "_console.h"
#include "_consolewritelog.h"
#include "maindef.h"
#include "memtool.h"
#include "extmem.h"
#include "_const.h"
#include "lang.h"
#include "inifile.h"

#include "glib/glib.h"

#include "gba_nds_fat.h"
#include "zlibhelp.h"
#include "cstream_fs.h"
#include "cipk.h"
#include "ipkfilelist.h"
#include "ipkstate.h"
#include "corner.h"

#include "rect.h"
#include "component.h"

#include "proc_screen.h"

typedef struct {
  u32 Life;
  u16 *pBuf;
} TMCU;

typedef struct {
  FAT_FILE *pfilehandle;
  CStreamFS *pstream;
  CIPK *pipk;
  s32 ScreenRatio; // fix .8 fraction
  s32 ImageIndex;
  TIPKThumbnail thumb;
  TIPKImageInfo imginfo;
  u32 MCUCount;
  u32 *pMCUOffsets;
  TMCU *pMCUs;
  u32 MCULife;
  bool PreloadMCU;
} TCurrent;

static TCurrent Current;

typedef struct {
  s32 ThumbnailOffsetX,ThumbnailOffsetY;
  s32 posx,posy;
  s32 lastposx,lastposy;
  s32 boundwidth,boundheight;
} TScreen;

static TScreen Screen;

// --------------------------------

#define RatioListCount (11)
static const s32 RatioList[RatioListCount]={(s32)(0.25*0x100),(s32)(0.375*0x100),(s32)(0.5*0x100),(s32)(0.625*0x100),(s32)(0.75*0x100),(s32)(0.875*0x100),(s32)(1.0*0x100),(s32)(1.25*0x100),(s32)(1.5*0x100),(s32)(1.75*0x100),(s32)(2.0*0x100)};

static s32 GetRatioIndexFromValue(s32 RatioValue)
{
  s32 res=-1;
  
  for(s32 idx=0;idx<RatioListCount;idx++){
    if(RatioValue==RatioList[idx]) res=idx;
  }
  
  if(res==-1){
    res=GetRatioIndexFromValue(1*0x100);
  }
  
  return(res);
}

static s32 GetRatioIndexFromVector(s32 RatioValue,s32 Vector)
{
  s32 idx=GetRatioIndexFromValue(RatioValue);
  
  if(idx==-1){
    idx=GetRatioIndexFromValue(1*0x100);
    return(idx);
  }
  
  if(Vector!=0){
    idx+=Vector;
    if(idx<0) idx=0;
    if(RatioListCount<=idx) idx=RatioListCount-1;
  }
  
  return(idx);
}

// --------------------------

static void OpenCurrent(void)
{
  TCurrent *pcur=&Current;
  TIPKThumbnail *pth=&pcur->thumb;
  TIPKImageInfo *pif=&pcur->imginfo;
  
  pcur->pfilehandle=NULL;
  pcur->pstream=NULL;
  pcur->pipk=NULL;
  
  pcur->ScreenRatio=RatioList[GetRatioIndexFromValue(IPKState.ScreenRatio)];
  pcur->ImageIndex=IPKState.ImageIndex;
  
  pth->Width=0;
  pth->Height=0;
  pth->Ratio=0<<16;
  pth->pBuf=NULL;
  
  pif->Width=0;
  pif->Height=0;
  pif->MCUXCount=0;
  pif->MCUYCount=0;
  
  pcur->MCUCount=0;
  pcur->pMCUOffsets=NULL;
  
  char *pfullfn=IPK_GetFullFilename(CurrentIPKFileIndex);
  pcur->pfilehandle=FAT_fopen(pfullfn,"r");
  if(pcur->pfilehandle==NULL){
    _consolePrintf("file open error.\n[%s]\n",pfullfn);
    ShowLogHalt();
  }
  pcur->pstream=new CStreamFS(pcur->pfilehandle);
  pcur->pipk=new CIPK(pcur->pstream);
  
  if(isVertical==true){ // 128ms
    pcur->pipk->GetThumbnail(pcur->ImageIndex,EIPKTHID_192256,pth);
    }else{
    pcur->pipk->GetThumbnail(pcur->ImageIndex,EIPKTHID_256192,pth);
  }
  pcur->pipk->GetImageInfo(pcur->ImageIndex,pif);
  
  pcur->MCUCount=(u32)pif->MCUXCount*(u32)pif->MCUYCount;
  
  pcur->pMCUOffsets=(u32*)safemalloc(pcur->MCUCount*4);
  pcur->pipk->GetMCUOffsets(pcur->ImageIndex,pcur->pMCUOffsets); // 129ms
  
  pcur->pMCUs=(TMCU*)safemalloc(pcur->MCUCount*sizeof(TMCU));
  for(u32 idx=0;idx<pcur->MCUCount;idx++){
    TMCU *pMCU=&pcur->pMCUs[idx];
    pMCU->Life=0;
    pMCU->pBuf=NULL;
  }
  pcur->MCULife=1;
  
  pcur->PreloadMCU=true;
}

static void CloseCurrent(void)
{
  TCurrent *pcur=&Current;
  
  if(pcur->pipk!=NULL){
    delete pcur->pipk; pcur->pipk=NULL;
  }
  if(pcur->pstream!=NULL){
    delete pcur->pstream; pcur->pstream=NULL;
  }
  if(pcur->pfilehandle!=NULL){
    FAT_fclose(pcur->pfilehandle);
    pcur->pfilehandle=NULL;
  }
  
  TIPKThumbnail *pth=&pcur->thumb;
  if(pth->pBuf!=NULL){
    safefree(pth->pBuf); pth->pBuf=NULL;
  }
  
  if(pcur->pMCUOffsets!=NULL){
    safefree(pcur->pMCUOffsets); pcur->pMCUOffsets=NULL;
  }
  
  if(pcur->pMCUs!=NULL){
    for(u32 idx=0;idx<pcur->MCUCount;idx++){
      TMCU *pMCU=&pcur->pMCUs[idx];
      if(pMCU->pBuf!=NULL){
        safefree(pMCU->pBuf); pMCU->pBuf=NULL;
      }
    }
    safefree(pcur->pMCUs); pcur->pMCUs=NULL;
  }
  
  IPKState.ScreenRatio=pcur->ScreenRatio;
  IPKState.ImageIndex=pcur->ImageIndex;
}

static bool CurIsFirstPage(void)
{
  TCurrent *pcur=&Current;
  s32 idx=pcur->ImageIndex;
  if(idx==0) return(true);
  return(false);
}

static bool CurIsLastPage(void)
{
  TCurrent *pcur=&Current;
  s32 idx=pcur->ImageIndex;
  if(idx==((s32)pcur->pipk->GetFilesCount()-1)) return(true);
  return(false);
}

/*
#define RatioIconWidth (32)
#define RatioIconHeight (16)
static CglCanvas *pRatioIconBGCanvas;
*/

static u32 CornerIconTimeoutVsync;

static void InitCornerIcon(void)
{
  const u32 s=CornerIconSize;
  
  s32 sx=0,sy=192;
  s32 sw=16,sh=0;
  u32 idx=0;
  
  if(isVertical==true){
    sx=192; sy=0;
    sw=0; sh=16;
    }else{
    sx=0; sy=192;
    sw=16; sh=0;
  }
  
  CglCanvas *pcan=pScreenSubBM->pCanvas;
  CglB15 *pB15;
  
  pB15=Corner_GetIcon(EIT_Lite);
  pB15->pCanvas->BitBlt(pcan,sx,sy,s,s,0,0,false);
  idx++; sx+=sw; sy+=sh;
  
  pB15=Corner_GetIcon(EIT_Up);
  pB15->pCanvas->BitBlt(pcan,sx,sy,s,s,0,0,false);
  idx++; sx+=sw; sy+=sh;
  
  if(CurIsFirstPage()==false){
    pB15=Corner_GetIcon(EIT_Left);
    pB15->pCanvas->BitBlt(pcan,sx,sy,s,s,0,0,false);
    idx++; sx+=sw; sy+=sh;
  }
  
  if(CurIsLastPage()==false){
    pB15=Corner_GetIcon(EIT_Right);
    pB15->pCanvas->BitBlt(pcan,sx,sy,s,s,0,0,false);
    idx++; sx+=sw; sy+=sh;
  }
  
  pB15=Corner_GetIcon(EIT_ToggleDisplay);
  pB15->pCanvas->BitBlt(pcan,sx,sy,s,s,0,0,false);
  idx++; sx+=sw; sy+=sh;
  
  if(isVertical==true){
    pB15=Corner_GetIcon(EIT_DblScr_Vertical);
    }else{
    pB15=Corner_GetIcon(EIT_DblScr_Horizontal);
  }
  pB15->pCanvas->BitBlt(pcan,sx,sy,s,s,0,0,false);
  idx++; sx+=sw; sy+=sh;
  
  pB15=Corner_GetIcon(EIT_ReduceDown);
  pB15->pCanvas->BitBlt(pcan,sx,sy,s,s,0,0,false);
  idx++; sx+=sw; sy+=sh;
  
  pB15=Corner_GetIcon(EIT_ReduceUp);
  pB15->pCanvas->BitBlt(pcan,sx,sy,s,s,0,0,false);
  idx++; sx+=sw; sy+=sh;
  
  pScreenSubBM->UpdateOAM();
}

static void DrawCornerIcon(void)
{
  CornerIconTimeoutVsync=GlobalINI.CustomConfig.ViewMode_CornerIconTimeoutSec*60;
  
  const u32 s=CornerIconSize;
  
  TCurrent *pcur=&Current;
  
  s32 sx=0,sy=192;
  s32 sw=16,sh=0;
  u32 idx=0;
  
  if(isVertical==true){
    sx=192; sy=0;
    sw=0; sh=16;
    }else{
    sx=0; sy=192;
    sw=16; sh=0;
  }
  
  s32 leftx=AutoScreenWidth-s;
  s32 centerx=AutoScreenWidth/2;
  s32 centerx_left=centerx-16-CornerIconSize;
//  s32 centerx_center=centerx-(CornerIconSize/2);
  s32 centerx_right=centerx+16;
  s32 downy=AutoScreenHeight-s;
  
  CglCanvas *pcan=pScreenSubBM->pCanvas;
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,0,0,16);
  idx++; sx+=sw; sy+=sh;
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,leftx,0,16);
  idx++; sx+=sw; sy+=sh;
  
  if(CurIsFirstPage()==false){
    pScreenSubBM->SetSpritePos16(idx,sx,sy,0,downy,16);
    idx++; sx+=sw; sy+=sh;
  }
  
  if(CurIsLastPage()==false){
    pScreenSubBM->SetSpritePos16(idx,sx,sy,leftx,downy,16);
    idx++; sx+=sw; sy+=sh;
  }
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,centerx_left,0,16);
  idx++; sx+=sw; sy+=sh;
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,centerx_right,0,16);
  idx++; sx+=sw; sy+=sh;
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,centerx_left,downy,16);
  idx++; sx+=sw; sy+=sh;
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,centerx_right,downy,16);
  idx++; sx+=sw; sy+=sh;
  
  pScreenSubBM->UpdateOAM();
  
/*
  if(pRatioIconBGCanvas==NULL) ShowLogHalt();
  pcan->BitBlt(pRatioIconBGCanvas,0,0,RatioIconWidth,RatioIconHeight,centerx-(RatioIconWidth/2),downy-RatioIconHeight,false);
*/
  
  s32 ratio=pcur->ScreenRatio;
  
  char msg[8];
  snprintf(msg,128,"%d",ratio*100/0x100);
  pcan->SetCglFont(NULL);
  u32 tx=centerx-(pcan->GetTextWidthA(msg)/2);
  u32 ty=downy+(s-pcan->GetTextHeight());
  pcan->SetFontTextColor(GlobalTextColor15);
  for(s32 y=-1;y<=1;y++){
    for(s32 x=-1;x<=1;x++){
      if((x!=0)||(y!=0)){
        pcan->TextOutA(tx+x,ty+y,msg);
      }
    }
  }
  pcan->SetFontTextColor(GlobalBGColor15);
  pcan->TextOutA(tx,ty,msg);
}

static void HideCornerIcon(u32 p)
{
  const u32 s=CornerIconSize;
  
  s32 sx=0,sy=192;
  s32 sw=16,sh=0;
  u32 idx=0;
  
  if(isVertical==true){
    sx=192; sy=0;
    sw=0; sh=16;
    }else{
    sx=0; sy=192;
    sw=16; sh=0;
  }
  
  s32 leftx=AutoScreenWidth-s;
  s32 centerx=AutoScreenWidth/2;
  s32 centerx_left=centerx-16-CornerIconSize;
//  s32 centerx_center=centerx-(CornerIconSize/2);
  s32 centerx_right=centerx+16;
  s32 downy=AutoScreenHeight-s;
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,0-p,0-p,16);
  idx++; sx+=sw; sy+=sh;
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,leftx+p,0-p,16);
  idx++; sx+=sw; sy+=sh;
  
  if(CurIsFirstPage()==false){
    pScreenSubBM->SetSpritePos16(idx,sx,sy,0-p,downy+p,16);
    idx++; sx+=sw; sy+=sh;
  }
  
  if(CurIsLastPage()==false){
    pScreenSubBM->SetSpritePos16(idx,sx,sy,leftx+p,downy+p,16);
    idx++; sx+=sw; sy+=sh;
  }
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,centerx_left,0-p,16);
  idx++; sx+=sw; sy+=sh;
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,centerx_right,0-p,16);
  idx++; sx+=sw; sy+=sh;
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,centerx_left,downy+p,16);
  idx++; sx+=sw; sy+=sh;
  
  pScreenSubBM->SetSpritePos16(idx,sx,sy,centerx_right,downy+p,16);
  idx++; sx+=sw; sy+=sh;
  
  pScreenSubBM->UpdateOAM();
  
/*
  const u32 s=CornerIconSize;
  
  TCurrent *pcur=&Current;
  
  s32 leftx=AutoScreenWidth-s;
  s32 centerx=AutoScreenWidth/2;
  s32 centerx_left=centerx-16-CornerIconSize;
//  s32 centerx_center=centerx-(CornerIconSize/2);
  s32 centerx_right=centerx+16;
  s32 downy=AutoScreenHeight-s;
  
  CglCanvas *pcan=pScreenSubBM->pCanvas;
  CglB15 *pB15;
  
  if(pRatioIconBGCanvas==NULL) ShowLogHalt();
  pRatioIconBGCanvas->BitBlt(pcan,centerx-(RatioIconWidth/2),downy-RatioIconHeight,RatioIconWidth,RatioIconHeight,0,0,false);
*/
}

static TRect Screen_GetLastBoundBox(void)
{
  TCurrent *pcur=&Current;
  TIPKThumbnail *pth=&pcur->thumb;
  TScreen *pscr=&Screen;
  
  u32 bx=(pscr->lastposx*pth->Ratio)/0x10000;
  u32 by=(pscr->lastposy*pth->Ratio)/0x10000;
  u32 bw=(pscr->boundwidth*pth->Ratio)/0x10000;
  u32 bh=(pscr->boundheight*pth->Ratio)/0x10000;
  
  TRect r;
  
  r.x=bx;
  r.y=by;
  r.w=bw;
  r.h=bh;
  
  s32 ratio=pcur->ScreenRatio;
  
  r.x=r.x*0x100/ratio;
  r.y=r.y*0x100/ratio;
  r.w=r.w*0x100/ratio;
  r.h=r.h*0x100/ratio;
  
  return(r);
}

static TRect Screen_GetBoundBox(void)
{
  TCurrent *pcur=&Current;
  TIPKThumbnail *pth=&pcur->thumb;
  TScreen *pscr=&Screen;
  
  u32 bx=(pscr->posx*pth->Ratio)/0x10000;
  u32 by=(pscr->posy*pth->Ratio)/0x10000;
  u32 bw=(pscr->boundwidth*pth->Ratio)/0x10000;
  u32 bh=(pscr->boundheight*pth->Ratio)/0x10000;
  
  TRect r;
  
  r.x=bx;
  r.y=by;
  r.w=bw;
  r.h=bh;
  
  s32 ratio=pcur->ScreenRatio;
  
  r.x=r.x*0x100/ratio;
  r.y=r.y*0x100/ratio;
  r.w=r.w*0x100/ratio;
  r.h=r.h*0x100/ratio;
  
  return(r);
}

static bool Screen_IsInsideBoundBox(s32 x,s32 y)
{
  TRect r=Screen_GetBoundBox();
  
  TScreen *pscr=&Screen;
  r.x+=pscr->ThumbnailOffsetX;
  r.y+=pscr->ThumbnailOffsetY;
  
  if((r.x<=x)&&(x<=(r.x+r.w))){
    if((r.y<=y)&&(y<=(r.y+r.h))){
      return(true);
    }
  }
  return(false);
}

static void Screen_FitPosition(void)
{
  TCurrent *pcur=&Current;
  TIPKImageInfo *pimginfo=&pcur->imginfo;
  TScreen *pscr=&Screen;
  s32 ratio=pcur->ScreenRatio;
  
  s32 x=pscr->posx;
  s32 y=pscr->posy;
  s32 w=pscr->boundwidth;
  s32 h=pscr->boundheight;
  
  s32 limw=(pimginfo->Width*ratio/0x100)-w;
  s32 limh=(pimginfo->Height*ratio/0x100)-h;
  if(limw<x) x=limw;
  if(limh<y) y=limh;
  
  if(x<0) x=0;
  if(y<0) y=0;
  
  pscr->posx=x;
  pscr->posy=y;
}

static void Screen_GetPosFromDisplay(s32 x,s32 y,s32 *dx,s32 *dy)
{
  TCurrent *pcur=&Current;
  TIPKThumbnail *pth=&pcur->thumb;
  
  *dx=(x*0x10000)/pth->Ratio;
  *dy=(y*0x10000)/pth->Ratio;
}

static void Screen_SetPosFromDisplay(s32 x,s32 y)
{
  TScreen *pscr=&Screen;
  s32 ratio=Current.ScreenRatio;
  
  x=x*ratio/0x100;
  y=y*ratio/0x100;
  
  if(x<0) x=0;
  if(y<0) y=0;
  
  s32 posx,posy;
  
  Screen_GetPosFromDisplay(x,y,&posx,&posy);
  
  pscr->posx=posx;
  pscr->posy=posy;
  
  Screen_FitPosition();
}

static void Screen_Reset(void)
{
  TCurrent *pcur=&Current;
  s32 ratio=pcur->ScreenRatio;
  TIPKThumbnail *pth=&pcur->thumb;
  TIPKImageInfo *pimginfo=&pcur->imginfo;
  TScreen *pscr=&Screen;
  
  s32 scrw=AutoScreenWidth;
  s32 scrh=AutoScreenHeight;
  
  pscr->ThumbnailOffsetX=(scrw-pth->Width)/2;
  pscr->ThumbnailOffsetY=(scrh-pth->Height)/2;
  
  pscr->lastposx=-1;
  pscr->lastposy=-1;
  
  s32 bw=scrw;
  s32 bh=scrh;
  s32 blimx=pimginfo->Width*ratio/0x100;
  s32 blimy=pimginfo->Height*ratio/0x100;
  
  if(IPKState.DoubleScreen==true){
    if(isVertical==true){
      bw*=2;
      }else{
      bh*=2;
    }
  }
  
  if(blimx<bw) bw=blimx;
  if(blimy<bh) bh=blimy;
  
  pscr->boundwidth=bw;
  pscr->boundheight=bh;
}

static void IPK_DrawLargeThumbnail(void)
{
  if(IPKState.DoubleScreen==true) return;
  
  TCurrent *pcur=&Current;
  TIPKThumbnail *pth=&pcur->thumb;
  
  TScreen *pscr=&Screen;
  
  if((pscr->lastposx==pscr->posx)&&(pscr->lastposy==pscr->posy)) return;
  
  VRAMWriteCache_Enable();
  
  CglCanvas *pcan=pScreenSubBM->pCanvas;
  
  TRect CurRect=Screen_GetBoundBox();
  
  {
    TRect r=CurRect;
    u32 SrcBufSize=pth->Width;
    u16 *pSrcBuf=&pth->pBuf[r.x+(r.y*SrcBufSize)];
    u32 DstBufSize=pcan->GetWidth();
    u16 *pDstBuf=pcan->GetScanLine(r.y+pscr->ThumbnailOffsetY)+(r.x+pscr->ThumbnailOffsetX);
    
    for(u32 y=0;y<(u32)r.h;y++){
      if((y==0)||(y==((u32)r.h-1))){
        for(u32 x=0;x<(u32)r.w;x++){
          pDstBuf[x]=(((pSrcBuf[x]&RGB15(30,30,30))>>1)+RGB15(8,7,7)) | BIT15;
        }
        }else{
        {
          u32 x=0;
          pDstBuf[x]=(((pSrcBuf[x]&RGB15(30,30,30))>>1)+RGB15(8,7,7)) | BIT15;
        }
        for(u32 x=1;x<(u32)r.w-1;x++){
          pDstBuf[x]=(((pSrcBuf[x]&RGB15(30,30,30))>>1)+RGB15(4,3,3)) | BIT15;
        }
        {
          u32 x=(u32)r.w-1;
          pDstBuf[x]=(((pSrcBuf[x]&RGB15(30,30,30))>>1)+RGB15(8,7,7)) | BIT15;
        }
      }
      pSrcBuf+=SrcBufSize;
      pDstBuf+=DstBufSize;
    }
  }
  
  if((pscr->lastposx!=-1)&&(pscr->lastposy!=-1)){
    TRect LastRect=Screen_GetLastBoundBox();
    
    TRect *pRects;
    u32 RectCount=GetDirtyRect_OutSide(LastRect,CurRect,&pRects);
    
    for(u32 idx=0;idx<RectCount;idx++){
      TRect r=pRects[idx];
      u32 SrcBufSize=pth->Width;
      u16 *pSrcBuf=&pth->pBuf[r.x+(r.y*SrcBufSize)];
      u32 DstBufSize=pcan->GetWidth();
      u16 *pDstBuf=pcan->GetScanLine(r.y+pscr->ThumbnailOffsetY)+(r.x+pscr->ThumbnailOffsetX);
      
      for(u32 y=0;y<(u32)r.h;y++){
        MemCopy16DMA3(pSrcBuf,pDstBuf,r.w*2);
        pSrcBuf+=SrcBufSize;
        pDstBuf+=DstBufSize;
      }
    }
  }
  
  VRAMWriteCache_Disable();
  
  pscr->lastposx=pscr->posx;
  pscr->lastposy=pscr->posy;
}

static void IPK_FullDrawLargeThumbnail(void)
{
  if(IPKState.DoubleScreen==true) return;
  
  TCurrent *pcur=&Current;
  TIPKThumbnail *pth=&pcur->thumb;
  
  TScreen *pscr=&Screen;
  
  VRAMWriteCache_Enable();
  
  CglCanvas *pcan=pScreenSubBM->pCanvas;
  pcan->SetColor(GlobalBGColor15);
  pcan->FillFast(0,0,AutoScreenWidth,AutoScreenHeight);
  
  u32 srcbufw=pth->Width;
  u16 *psrcbuf=pth->pBuf;
  u32 dstbufw=pcan->GetWidth();
  u16 *pdstbuf=pcan->GetScanLine(pscr->ThumbnailOffsetY)+pscr->ThumbnailOffsetX;
  
  for(u32 py=0;py<pth->Height;py++){
    MemCopy16DMA3(psrcbuf,pdstbuf,pth->Width*2);
    psrcbuf+=srcbufw;
    pdstbuf+=dstbufw;
  }
  
  VRAMWriteCache_Disable();
  
  pscr->lastposx=-1;
  pscr->lastposy=-1;
  
  IPK_DrawLargeThumbnail();
  
  DrawCornerIcon();
}

static bool View_safemalloc_RequestFreeArea(void)
{
  TCurrent *pcur=&Current;
  TMCU *pMCUs=pcur->pMCUs;
  
  u32 lastidx=(u32)-1,lastlife=0x7fffffff;
  u32 mcucnt=pcur->MCUCount;
  for(u32 idx=0;idx<mcucnt;idx++){
    if(pMCUs[idx].pBuf!=NULL){
      u32 Life=pMCUs[idx].Life;
      if(Life!=0){
        if(Life<lastlife){
          lastidx=idx;
          lastlife=Life;
        }
      }
    }
  }
  
  if(lastidx==(u32)-1) return(false);
  
  TMCU *pMCU=&pMCUs[lastidx];
  
  pMCU->Life=0;
  safefree(pMCU->pBuf); pMCU->pBuf=NULL;
  
  return(true);
}

static inline void getrgb(u32 col,u32 *r,u32 *g,u32 *b)
{
  *r+=(col>>0)&0x1f;
  *g+=(col>>5)&0x1f;
  *b+=(col>>10)&0x1f;
}

static inline void getrgb32(u32 col,u32 *r,u32 *g,u32 *b)
{
  *r=(col>>0)&0x1f;
  *g=(col>>5)&0x1f;
  *b=(col>>10)&0x1f;
}

static inline void getrgb8(u32 col,u8 *r,u8 *g,u8 *b)
{
  *r=(col>>0)&0x1f;
  *g=(col>>5)&0x1f;
  *b=(col>>10)&0x1f;
}

static void LoadMCU(u32 MCUIndex)
{
  TCurrent *pcur=&Current;
  TMCU *pMCU=&pcur->pMCUs[MCUIndex];
  
  const u32 MCUBytes=IPKMCUWidth*IPKMCUHeight*2;
  
  if(pMCU->pBuf==NULL){
    pMCU->Life=0;
    pMCU->pBuf=(u16*)safemalloc(MCUBytes);
    if(extmem_Exists(MCUIndex)==true){
      extmem_Read(MCUIndex,pMCU->pBuf,MCUBytes);
      }else{
      pcur->pipk->GetMCUData(&pcur->imginfo,pcur->pMCUOffsets[MCUIndex],pMCU->pBuf,MCUBytes);
      if(extmem_Alloc(MCUIndex,MCUBytes)==true) extmem_Write(MCUIndex,pMCU->pBuf,MCUBytes);
    }
  }
  
  pMCU->Life=pcur->MCULife;
  pcur->MCULife++;
}

static void DrawScreen(void)
{
  TCurrent *pcur=&Current;
  const TIPKImageInfo *pif=&pcur->imginfo;
  const TScreen *pscr=&Screen;
  const s32 ratio=pcur->ScreenRatio;
  
  CglCanvas *pUpLeftCan=pScreenMain->pCanvas;
  CglCanvas *pDownRightCan=pScreenSubBM->pCanvas;
  
  const u32 MCUXCount=(u32)pif->MCUXCount;
  const u32 MCUYCount=(u32)pif->MCUYCount;
  const u32 posx=pscr->posx;
  const u32 posy=pscr->posy;
  
  const TMCU *pMCUs=pcur->pMCUs;
  
  s32 singlewidth=AutoScreenWidth;
  s32 singleheight=AutoScreenHeight;
  s32 scrwidth=AutoScreenWidth,nextwidth=0;
  s32 scrheight=AutoScreenHeight,nextheight=0;
  const u32 redMCUWidth=IPKMCUWidth*ratio/0x100;
  const u32 redMCUHeight=IPKMCUHeight*ratio/0x100;
  
  if(IPKState.DoubleScreen==true){
    if(isVertical==true){
      nextwidth=scrwidth;
      scrwidth*=2;
      }else{
      nextheight=scrheight;
      scrheight*=2;
    }
  }
  
//  PrfStart();
  
  s32 FirstMCUIndex=-1;
  
  if(pcur->PreloadMCU==true){
    for(u32 mcux=0;mcux<MCUXCount;mcux++){
      s32 drawx=(mcux*redMCUWidth)-posx;
      if((-(s32)redMCUWidth<drawx)&&(drawx<scrwidth)){
        for(u32 mcuy=0;mcuy<MCUYCount;mcuy++){
          s32 drawy=(mcuy*redMCUHeight)-posy;
          if((-(s32)redMCUHeight<drawy)&&(drawy<scrheight)){
            u32 mcuidx=mcux+(mcuy*MCUXCount);
            if(FirstMCUIndex==-1) FirstMCUIndex=mcuidx;
            LoadMCU(mcuidx);
          }
        }
      }
    }
  }
  
  if(FirstMCUIndex!=-1){
    if(pMCUs[FirstMCUIndex].pBuf==NULL){
      // メモリオーバーしたら先読みしない
      _consolePrintf("Disable PreloadMCU\n");
      pcur->PreloadMCU=false;
    }
  }
  
  VRAMWriteCache_Enable();
  
  if(ratio==(s32)(1.0*0x100)){
    CglCanvas *pSrcCan=new CglCanvas(NULL,IPKMCUWidth,IPKMCUHeight,pf15bit);
    for(u32 mcux=0;mcux<MCUXCount;mcux++){
      s32 drawx=(mcux*redMCUWidth)-posx;
      if((-(s32)IPKMCUWidth<drawx)&&(drawx<scrwidth)){
        for(u32 mcuy=0;mcuy<MCUYCount;mcuy++){
          s32 drawy=(mcuy*redMCUHeight)-posy;
          if((-(s32)IPKMCUHeight<drawy)&&(drawy<scrheight)){
            u32 mcuidx=mcux+(mcuy*MCUXCount);
            LoadMCU(mcuidx);
            pSrcCan->SetVRAMBuf(pMCUs[mcuidx].pBuf,IPKMCUWidth,IPKMCUHeight,pf15bit);
            {
              s32 dx=drawx,dy=drawy;
              s32 dw=IPKMCUWidth,dh=IPKMCUHeight;
              if(singlewidth<(dx+dw)) dw=singlewidth-dx;
              if(singleheight<(dy+dh)) dh=singleheight-dy;
              if((0<dw)&&(0<dh)){
                if((-dw<dx)&&(dx<singlewidth)&&(-dh<dy)&&(dy<singleheight)){
                  pSrcCan->BitBlt(pUpLeftCan,dx,dy,dw,dh,0,0,false);
                }
              }
            }
            if((nextwidth!=0)||(nextheight!=0)){
              s32 dx=drawx-nextwidth,dy=drawy-nextheight;
              s32 dw=IPKMCUWidth,dh=IPKMCUHeight;
              if(singlewidth<(dx+dw)) dw=singlewidth-dx;
              if(singleheight<(dy+dh)) dh=singleheight-dy;
              if((0<dw)&&(0<dh)){
                if((-dw<dx)&&(dx<singlewidth)&&(-dh<dy)&&(dy<singleheight)){
                  pSrcCan->BitBlt(pDownRightCan,dx,dy,dw,dh,0,0,false);
                }
              }
            }
          }
        }
      }
    }
    delete pSrcCan; pSrcCan=NULL;
    }else{
    CglCanvas *ptmpcan=new CglCanvas(NULL,redMCUWidth,redMCUHeight,pf15bit);
    for(u32 mcux=0;mcux<MCUXCount;mcux++){
      s32 drawx=(mcux*redMCUWidth)-posx;
      if((-(s32)redMCUWidth<drawx)&&(drawx<scrwidth)){
        for(u32 mcuy=0;mcuy<MCUYCount;mcuy++){
          s32 drawy=(mcuy*redMCUHeight)-posy;
          if((-(s32)redMCUHeight<drawy)&&(drawy<scrheight)){
            u32 mcuidx=mcux+(mcuy*MCUXCount);
            LoadMCU(mcuidx);
            u16 *pSrcBuf=pMCUs[mcuidx].pBuf;
            u16 *pDstBuf=ptmpcan->GetVRAMBuf();
            switch(ratio){
              case 0x40: { // 0.25*0x100
/*
                static int a=0;
                a++;
                PrfStart();
*/
                // 476us
                for(u32 y=0;y<redMCUHeight;y++){
                  for(u32 x=0;x<redMCUWidth;x++){
                    u32 r=0,g=0,b=0;
                    u16 *pSrcBuf0=pSrcBuf+(x*4);
                    asm volatile(
                      "mov r2,#0x1f \n"
                      "orr r2,r2,r2,lsl #16 \n"
                      
                      "ldmia %3,{r0,r1} \n"
                      "add %3,#64*2 \n"
                      "and r3,r0,r2,lsl #0 \n add %0,r3,lsr #0 \n"
                      "and r3,r0,r2,lsl #5 \n add %1,r3,lsr #5 \n"
                      "and r3,r0,r2,lsl #10 \n add %2,r3,lsr #10 \n"
                      "and r3,r1,r2,lsl #0 \n add %0,r3,lsr #0 \n"
                      "and r3,r1,r2,lsl #5 \n add %1,r3,lsr #5 \n"
                      "and r3,r1,r2,lsl #10 \n add %2,r3,lsr #10 \n"
                      
                      "ldmia %3,{r0,r1} \n"
                      "add %3,#64*2 \n"
                      "and r3,r0,r2,lsl #0 \n add %0,r3,lsr #0 \n"
                      "and r3,r0,r2,lsl #5 \n add %1,r3,lsr #5 \n"
                      "and r3,r0,r2,lsl #10 \n add %2,r3,lsr #10 \n"
                      "and r3,r1,r2,lsl #0 \n add %0,r3,lsr #0 \n"
                      "and r3,r1,r2,lsl #5 \n add %1,r3,lsr #5 \n"
                      "and r3,r1,r2,lsl #10 \n add %2,r3,lsr #10 \n"
                      
                      "ldmia %3,{r0,r1} \n"
                      "add %3,#64*2 \n"
                      "and r3,r0,r2,lsl #0 \n add %0,r3,lsr #0 \n"
                      "and r3,r0,r2,lsl #5 \n add %1,r3,lsr #5 \n"
                      "and r3,r0,r2,lsl #10 \n add %2,r3,lsr #10 \n"
                      "and r3,r1,r2,lsl #0 \n add %0,r3,lsr #0 \n"
                      "and r3,r1,r2,lsl #5 \n add %1,r3,lsr #5 \n"
                      "and r3,r1,r2,lsl #10 \n add %2,r3,lsr #10 \n"
                      
                      "ldmia %3,{r0,r1} \n"
                      "sub %3,#(64*2)*3 \n"
                      "and r3,r0,r2,lsl #0 \n add %0,r3,lsr #0 \n"
                      "and r3,r0,r2,lsl #5 \n add %1,r3,lsr #5 \n"
                      "and r3,r0,r2,lsl #10 \n add %2,r3,lsr #10 \n"
                      "and r3,r1,r2,lsl #0 \n add %0,r3,lsr #0 \n"
                      "and r3,r1,r2,lsl #5 \n add %1,r3,lsr #5 \n"
                      "and r3,r1,r2,lsl #10 \n add %2,r3,lsr #10 \n"
                      
                      "mov r2,#0xff \n"
                      "orr r2,r2,r2,lsl #8 \n"
                      
                      "and r0,%0,r2 \n add %0,r0,%0,lsr #16 \n lsr %0,%0,#4 \n"
                      "and r0,%1,r2 \n add %1,r0,%1,lsr #16 \n lsr %1,%1,#4 \n"
                      "and r0,%2,r2 \n add %2,r0,%2,lsr #16 \n lsr %2,%2,#4 \n"
                      
                      "orr %0,%0,%1,lsl #5 \n"
                      "orr %0,%0,%2,lsl #10 \n"
                      "orr %0,%0,#0x8000 \n"
                      
                      : "+r"(r),"+r"(g),"+r"(b) : "r"(pSrcBuf0)
                      : "r0","r1","r2","r3"
                    );
                    pDstBuf[x]=r;
                  }
                  pSrcBuf+=IPKMCUWidth*4;
                  pDstBuf+=redMCUWidth;
                }
//                PrfEnd(a);// ShowLogHalt();
              } break;
              case 0x80: { // 0.5*0x100
/*
                static int a=0;
                a++;
                PrfStart();
*/
                // 817us
                u16 *pSrcBuf0=pSrcBuf;
                for(u32 y=0;y<redMCUHeight;y++){
                  for(u32 x=0;x<redMCUWidth;x++){
                    u32 r=0,g=0,b=0;
                    asm volatile(
                      "reduce50per_asm: \n"
                      
                      "mov r2,#0x1f \n"
                      "orr r2,r2,r2,lsl #16 \n"
                      
                      "ldr r1,[%3,#64*2] \n"
                      "ldr r0,[%3],#2*2 \n"
                      "and r3,r1,r2,lsl #0 \n add %0,r3,lsr #0 \n"
                      "and r3,r1,r2,lsl #5 \n add %1,r3,lsr #5 \n"
                      "and r3,r1,r2,lsl #10 \n add %2,r3,lsr #10 \n"
                      "and r3,r0,r2,lsl #0 \n add %0,r3,lsr #0 \n"
                      "and r3,r0,r2,lsl #5 \n add %1,r3,lsr #5 \n"
                      "and r3,r0,r2,lsl #10 \n add %2,r3,lsr #10 \n"
                      
                      "mov r2,#0xff \n"
                      "orr r2,r2,r2,lsl #8 \n"
                      
                      "and r0,%0,r2 \n add %0,r0,%0,lsr #16 \n lsr %0,%0,#2 \n"
                      "and r0,%1,r2 \n add %1,r0,%1,lsr #16 \n lsr %1,%1,#2 \n"
                      "and r0,%2,r2 \n add %2,r0,%2,lsr #16 \n lsr %2,%2,#2 \n"
                      
                      "orr %0,%0,%1,lsl #5 \n"
                      "orr %0,%0,%2,lsl #10 \n"
                      "orr %0,%0,#0x8000 \n"
                      
                      : "+r"(r),"+r"(g),"+r"(b) : "r"(pSrcBuf0)
                      : "r0","r1","r2","r3"
                    );
                    pDstBuf[x]=r;
                  }
                  pSrcBuf0+=IPKMCUWidth*1;
                  pDstBuf+=redMCUWidth;
                }
//                PrfEnd(a);// ShowLogHalt();
              } break;
              case 0x200: { // 2.0*0x100
                u16 *pSrcBuf0=pSrcBuf;
                u16 *pDstBuf0=pDstBuf;
                for(u32 y=0;y<IPKMCUHeight-1;y++){
                  u16 *pSrcBuf1=pSrcBuf0+IPKMCUWidth;
                  u16 *pDstBuf1=pDstBuf0+redMCUWidth;
                  for(u32 x=0;x<IPKMCUWidth-1;x++){
                    u32 sr,sg,sb,tr,tg,tb;
                    u32 rx=x*2;
                    getrgb32(pSrcBuf0[x+0],&sr,&sg,&sb);
                    pDstBuf0[rx+0]=RGB15(sr,sg,sb)|BIT15;
                    getrgb32(pSrcBuf0[x+1],&tr,&tg,&tb);
                    pDstBuf0[rx+1]=RGB15((sr+tr)/2,(sg+tg)/2,(sb+tb)/2)|BIT15;
                    getrgb32(pSrcBuf1[x+0],&tr,&tg,&tb);
                    pDstBuf1[rx+0]=RGB15((sr+tr)/2,(sg+tg)/2,(sb+tb)/2)|BIT15;
                    getrgb32(pSrcBuf1[x+1],&tr,&tg,&tb);
                    pDstBuf1[rx+1]=RGB15((sr+tr)/2,(sg+tg)/2,(sb+tb)/2)|BIT15;
                  }
                  {
                    u32 r0,g0,b0,r1,g1,b1;
                    u16 c0,c1;
                    u32 x=IPKMCUWidth-1;
                    u32 rx=x*2;
                    getrgb32(pSrcBuf0[x+0],&r0,&g0,&b0);
                    getrgb32(pSrcBuf1[x+0],&r1,&g1,&b1);
                    c0=RGB15(r0,g0,b0)|BIT15;
                    c1=RGB15((r0+r1)/2,(g0+g1)/2,(b0+b1)/2)|BIT15;
                    pDstBuf0[rx+0]=c0;
                    pDstBuf0[rx+1]=c0;
                    pDstBuf1[rx+0]=c1;
                    pDstBuf1[rx+1]=c1;
                  }
                  pSrcBuf0+=IPKMCUWidth;
                  pDstBuf0+=redMCUWidth*2;
                }
                u16 *pDstBuf1=pDstBuf0+redMCUWidth;
                for(u32 x=0;x<IPKMCUWidth-1;x++){
                  u32 r0,g0,b0,r1,g1,b1;
                  u16 c0,c1;
                  u32 rx=x*2;
                  getrgb32(pSrcBuf0[x+0],&r0,&g0,&b0);
                  getrgb32(pSrcBuf0[x+1],&r1,&g1,&b1);
                  c0=RGB15(r0,g0,b0)|BIT15;
                  c1=RGB15((r0+r1)/2,(g0+g1)/2,(b0+b1)/2)|BIT15;
                  pDstBuf0[rx+0]=c0;
                  pDstBuf0[rx+1]=c1;
                  pDstBuf1[rx+0]=c0;
                  pDstBuf1[rx+1]=c1;
                }
                {
                  u32 r,g,b;
                  u16 c;
                  u32 x=IPKMCUWidth-1;
                  u32 rx=x*2;
                  getrgb32(pSrcBuf0[x+0],&r,&g,&b);
                  c=RGB15(r,g,b)|BIT15;
                  pDstBuf0[rx+0]=c;
                  pDstBuf0[rx+1]=c;
                  pDstBuf1[rx+0]=c;
                  pDstBuf1[rx+1]=c;
                }
              } break;
              default: {
                const u32 srcsize=IPKMCUWidth;
                const u32 dstsize=redMCUWidth;
                static u8 r[(srcsize+1)*(srcsize+1)],g[(srcsize+1)*(srcsize+1)],b[(srcsize+1)*(srcsize+1)];
                {
                  u16 *ps=pSrcBuf;
                  u32 idx=0;
                  for(u32 y=0;y<srcsize;y++){
                    for(u32 x=0;x<srcsize;x++){
                      getrgb8(ps[x],&r[idx],&g[idx],&b[idx]);
                      idx++;
                    }
                    u32 lidx=idx-1;
                    r[idx]=r[lidx]; g[idx]=g[lidx]; b[idx]=b[lidx];
                    idx++;
                    ps+=srcsize;
                  }
                  for(u32 x=0;x<srcsize+1;x++){
                    u32 lidx=idx-(srcsize+1);
                    r[idx]=r[lidx]; g[idx]=g[lidx]; b[idx]=b[lidx];
                    idx++;
                  }
                }
                const u32 stepdiv=0x1000;
                const u32 srcstep=(u32)(srcsize*stepdiv/dstsize);
                const u32 coldiv=stepdiv*stepdiv;
                const u32 underlimit=coldiv/32;
                u32 fsy=0;
                for(u32 y=0;y<dstsize;y++){
                  u32 sy=fsy/stepdiv;
                  u32 SrcIdx0=sy*(srcsize+1);
                  u32 ya,yia;
                  ya=(u32)(fsy&(stepdiv-1)); yia=stepdiv-ya;
                  u32 fsx=0;
                  for(u32 x=0;x<dstsize;x++){
                    u32 sx=fsx/stepdiv;
                    u32 xa,xia;
                    xa=(u32)(fsx&(stepdiv-1)); xia=stepdiv-xa;
                    
                    u32 cr=0,cg=0,cb=0;
                    u32 a;
                    
                    u32 sidx0=SrcIdx0+sx;
                    u32 sidx1=sidx0+(srcsize+1);
                    
                    a=yia*xia;
                    if(underlimit<=a){
                      cr+=r[sidx0+0]*a; cg+=g[sidx0+0]*a; cb+=b[sidx0+0]*a;
                    }
                    a=ya*xia;
                    if(underlimit<=a){
                      cr+=r[sidx1+0]*a; cg+=g[sidx1+0]*a; cb+=b[sidx1+0]*a;
                    }
                    a=yia*xa;
                    if(underlimit<=a){
                      cr+=r[sidx0+1]*a; cg+=g[sidx0+1]*a; cb+=b[sidx0+1]*a;
                    }
                    a=ya*xa;
                    if(underlimit<=a){
                      cr+=r[sidx1+1]*a; cg+=g[sidx1+1]*a; cb+=b[sidx1+1]*a;
                    }
                    
                    *pDstBuf++=RGB15(cr/coldiv,cg/coldiv,cb/coldiv)|BIT15;
                    fsx+=srcstep;
                  }
                  fsy+=srcstep;
                }
              } break;
            }
            {
              s32 dx=drawx,dy=drawy;
              s32 dw=redMCUWidth,dh=redMCUHeight;
              if(singlewidth<(dx+dw)) dw=singlewidth-dx;
              if(singleheight<(dy+dh)) dh=singleheight-dy;
              if((0<dw)&&(0<dh)){
                if((-dw<dx)&&(dx<singlewidth)&&(-dh<dy)&&(dy<singleheight)){
                  ptmpcan->BitBlt(pUpLeftCan,dx,dy,dw,dh,0,0,false);
                }
              }
            }
            if((nextwidth!=0)||(nextheight!=0)){
              s32 dx=drawx-nextwidth,dy=drawy-nextheight;
              s32 dw=redMCUWidth,dh=redMCUHeight;
              if(singlewidth<(dx+dw)) dw=singlewidth-dx;
              if(singleheight<(dy+dh)) dh=singleheight-dy;
              if((0<dw)&&(0<dh)){
                if((-dw<dx)&&(dx<singlewidth)&&(-dh<dy)&&(dy<singleheight)){
                  ptmpcan->BitBlt(pDownRightCan,dx,dy,dw,dh,0,0,false);
                }
              }
            }
          }
        }
      }
    }
    delete ptmpcan; ptmpcan=NULL;
  }
  
  VRAMWriteCache_Disable();
  
//  PrfEnd(0);
}

static void SetPosOnlyFromStartPos(EStartPos StartPos)
{
  TCurrent *pcur=&Current;
  TIPKImageInfo *pimginfo=&pcur->imginfo;
  TScreen *pscr=&Screen;
  s32 ratio=pcur->ScreenRatio;
  
  s32 w=pscr->boundwidth;
  s32 h=pscr->boundheight;
  
  s32 limw=(pimginfo->Width*ratio/0x100)-w;
  s32 limh=(pimginfo->Height*ratio/0x100)-h;
  
  s32 x=0,y=0;
  
  switch(StartPos){
    case ESP_LeftTop: x=0; y=0; break;
    case ESP_RightTop: x=limw; y=0; break;
    case ESP_LeftDown: x=0; y=limh; break;
    case ESP_RightDown: x=limw; y=limh; break;
  }
  
  pscr->posx=x;
  pscr->posy=y;
  Screen_FitPosition();
}

// --------- start pos menu

static bool Modal_StartPosMenu;

static TComponentFrame SPM_ComponentFrame;

#define SPM_ItemCount (4)
static s32 SPM_ItemIndex;
static const char SPM_ItemMsgsUTF8_eng[SPM_ItemCount][16]={"LeftTop","RightTop","LeftDown","RightDown"};
static const char SPM_ItemMsgsUTF8_jpn[SPM_ItemCount][16]={"左上","右上","左下","右下"};

static void SPM_CB_ItemDraw(u32 Index,CglCanvas *pCanvas,TRect r)
{
  pCanvas->SetCglFont(pCglFont_Large14);
  pCanvas->SetFontBGColor(GlobalTextColor15);
  pCanvas->SetFontTextColor(GlobalBGColor15);
  
  if((s32)Index==SPM_ItemIndex){
    CglB15 *pB15=Component_GetIcon(ECIT_Ok);
    pB15->pCanvas->BitBlt(pCanvas,r.x,r.y,pB15->pCanvas->GetWidth(),pB15->pCanvas->GetHeight(),0,0,true);
  }
  const char *pmsgUTF8=Lang_GetTextUTF8(SPM_ItemMsgsUTF8_eng[Index],SPM_ItemMsgsUTF8_jpn[Index]);
  pCanvas->TextOutUTF8(r.x+24,r.y,pmsgUTF8);
}

static void SPM_Start(void)
{
  SPM_ItemIndex=0;
  
  TComponentFrame *pcf=&SPM_ComponentFrame;
  ComponentFrame_Init(pcf,pScreenSubBM->pCanvas);
  
  pcf->pTitleUTF8_eng="Start pos";
  pcf->pTitleUTF8_jpn="開始位置";
  pcf->ItemCount=5;
  pcf->ItemHeight=16;
  pcf->TextOfsX=4;
  pcf->TextOfsY=1;
  s32 w=104;
  s32 h=pcf->ItemHeight*SPM_ItemCount;
  pcf->Rect=CreateRect((AutoScreenWidth-w)/2,AutoScreenHeight-32-h,w,h);
  pcf->FillAlpha=12;
  pcf->CallBack_ItemDraw=SPM_CB_ItemDraw;
  
  SPM_ItemIndex=0;
  switch(IPKState.StartPos){
    case ESP_LeftTop: SPM_ItemIndex=0; break;
    case ESP_RightTop: SPM_ItemIndex=1; break;
    case ESP_LeftDown: SPM_ItemIndex=2; break;
    case ESP_RightDown: SPM_ItemIndex=3; break;
  }
  
  ComponentFrame_Draw(pcf);
  
  Modal_StartPosMenu=true;
}

static void SPM_End(bool apply,s32 idx)
{
  Modal_StartPosMenu=false;
  
  if(apply==true){
    switch(idx){
      case 0: IPKState.StartPos=ESP_LeftTop; break;
      case 1: IPKState.StartPos=ESP_RightTop; break;
      case 2: IPKState.StartPos=ESP_LeftDown; break;
      case 3: IPKState.StartPos=ESP_RightDown; break;
    }
  }
  
  SetPosOnlyFromStartPos(IPKState.StartPos);
  
  IPK_FullDrawLargeThumbnail();
  DrawScreen();
  DrawCornerIcon();
}

static void SPM_CB_KeyDown(u32 Keys)
{
}

static void SPM_CB_KeyPress(u32 VsyncCount,u32 Keys)
{
  s32 v=0;
  
  if((Keys&KEY_UP)!=0) v=-1;
  if((Keys&KEY_DOWN)!=0) v=+1;
  
  if(v!=0){
    s32 idx=SPM_ItemIndex+v;
    if((idx<0)||(SPM_ItemCount<=idx)){
      SPM_End(false,0);
      }else{
      SPM_End(true,idx);
    }
  }
}

static void SPM_CB_KeyUp(u32 Keys)
{
}

static void SPM_CB_MouseDown(s32 x,s32 y)
{
}

static void SPM_CB_MouseMove(s32 x,s32 y,bool CanIgnore)
{
}

static void SPM_CB_MouseUp(s32 x,s32 y)
{
  s32 idx=ComponentFrame_GetIndexFromPos(&SPM_ComponentFrame,x,y);
  
  if((idx<0)||(SPM_ItemCount<=idx)){
    SPM_End(false,0);
    }else{
    SPM_End(true,idx);
  }
}

// --------- ratio menu

static bool Modal_RatioMenu;

static TComponentFrame RM_ComponentFrame;

static s32 RM_ItemIndex;

static void RM_CB_ItemDraw(u32 Index,CglCanvas *pCanvas,TRect r)
{
  pCanvas->SetCglFont(pCglFont_Large14);
  pCanvas->SetFontBGColor(GlobalTextColor15);
  pCanvas->SetFontTextColor(GlobalBGColor15);
  
  char msg[32];
  u32 iratio=RatioList[Index]*100/0x100;
  if((iratio!=50)&&(iratio!=100)&&(iratio!=200)){
    snprintf(msg,16,"(slow) %d%%",iratio);
    }else{
    snprintf(msg,16,"%d%%",iratio);
  }
  s32 dx=r.x+r.w-pCanvas->GetTextWidthA(msg);
  pCanvas->TextOutA(dx,r.y,msg);
  
  if((s32)Index==RM_ItemIndex){
    CglB15 *pB15=Component_GetIcon(ECIT_Ok);
    pB15->pCanvas->BitBlt(pCanvas,r.x,r.y,pB15->pCanvas->GetWidth(),pB15->pCanvas->GetHeight(),0,0,true);
  }
}

static void RM_Start(void)
{
  TCurrent *pcur=&Current;
  
  RM_ItemIndex=GetRatioIndexFromValue(pcur->ScreenRatio);
  if(RM_ItemIndex==-1) RM_ItemIndex=2;
  
  TComponentFrame *pcf=&RM_ComponentFrame;
  ComponentFrame_Init(pcf,pScreenSubBM->pCanvas);
  
  pcf->pTitleUTF8_eng="Ratio menu";
  pcf->pTitleUTF8_jpn="縮小率設定";
  pcf->ItemCount=RatioListCount;
  pcf->ItemHeight=16;
  pcf->TextOfsX=4;
  pcf->TextOfsY=1;
  s32 w=96;
  s32 h=pcf->ItemHeight*RatioListCount;
  s32 t=(AutoScreenHeight-h)-32;
  if(t<2) t=2;
  pcf->Rect=CreateRect((AutoScreenWidth-w)/2,t,w,h);
  pcf->FillAlpha=12;
  pcf->CallBack_ItemDraw=RM_CB_ItemDraw;
  
  ComponentFrame_Draw(pcf);
  
  Modal_RatioMenu=true;
}

static void RM_End(s32 NewRatioValue)
{
  Modal_RatioMenu=false;
  
  // 泥臭い処理（苦笑
  
  TCurrent *pcur=&Current;
  TScreen *pscr=&Screen;
  
  s32 ratio;
  
  ratio=pcur->ScreenRatio;
  s32 lastposx=(pscr->posx+(pscr->boundwidth/2))*0x100/ratio;
  s32 lastposy=(pscr->posy+(pscr->boundheight/2))*0x100/ratio;
  
  if(NewRatioValue!=0) pcur->ScreenRatio=NewRatioValue;
  Screen_Reset();
  
  ratio=pcur->ScreenRatio;
  pscr->posx=(lastposx*ratio/0x100)-(pscr->boundwidth/2);
  pscr->posy=(lastposy*ratio/0x100)-(pscr->boundheight/2);
  Screen_FitPosition();
  
  {
    CglCanvas *pcan=pScreenSubBM->pCanvas;
    pcan->SetColor(GlobalBGColor15);
    pcan->FillFast(0,0,AutoScreenWidth,AutoScreenHeight);
  }
  IPK_FullDrawLargeThumbnail();
  
  {
    CglCanvas *pcan=pScreenMain->pCanvas;
    pcan->SetColor(GlobalBGColor15);
    pcan->FillFast(0,0,AutoScreenWidth,AutoScreenHeight);
  }
  DrawScreen();
  
  DrawCornerIcon();
}

static void RM_CB_KeyDown(u32 Keys)
{
}

static void RM_CB_KeyPress(u32 VsyncCount,u32 Keys)
{
  s32 v=0;
  
  if((Keys&KEY_UP)!=0) v=-1;
  if((Keys&KEY_DOWN)!=0) v=+1;
  
  if(v!=0){
    TCurrent *pcur=&Current;
    s32 idx=GetRatioIndexFromVector(pcur->ScreenRatio,v);
    RM_End(RatioList[idx]);
  }
  
  if(((Keys&KEY_LEFT)!=0)||((Keys&KEY_RIGHT)!=0)){
    RM_End(0);
  }
}

static void RM_CB_KeyUp(u32 Keys)
{
}

static void RM_CB_MouseDown(s32 x,s32 y)
{
}

static void RM_CB_MouseMove(s32 x,s32 y,bool CanIgnore)
{
}

static void RM_CB_MouseUp(s32 x,s32 y)
{
  s32 idx=ComponentFrame_GetIndexFromPos(&RM_ComponentFrame,x,y);
  
  s32 ratio=0;
  if(idx!=-1) ratio=RatioList[idx];
  
  RM_End(ratio);
}

// ---------

static bool Modal_CommandMenu;

static TComponentFrame CM_ComponentFrame;

#define CM_ItemCount (7)
static const char CM_ItemTexts1[CM_ItemCount][16]={"","","L+↑","L+←","L+→","L+↓","L+SEL"};
static const char CM_ItemTexts2eng[CM_ItemCount][32]={"","","Back to thumbnail","Previous page","Next page","Open ratio menu","Open start pos"};
static const char CM_ItemTexts2jpn[CM_ItemCount][32]={"","","サムネイルに戻る","前のページに戻る","次のページに進む","縮小率設定メニュー","開始位置メニュー"};

static void CM_CB_ItemDraw(u32 Index,CglCanvas *pCanvas,TRect r)
{
  pCanvas->SetCglFont(pCglFont_Large14);
  pCanvas->SetFontBGColor(GlobalTextColor15);
  pCanvas->SetFontTextColor(GlobalBGColor15);
  
  TCurrent *pcur=&Current;
  
  switch(Index){
    case 0: {
      char msg[256];
      snprintf(msg,256,"%d/%d %dx%d pixels",pcur->ImageIndex+1,pcur->pipk->GetFilesCount(),pcur->imginfo.Width,pcur->imginfo.Height);
      pCanvas->TextOutA(r.x,r.y,msg);
    } break;
    case 1: {
      UnicodeChar msgw[256];
      pcur->pipk->GetFilename(pcur->ImageIndex,msgw);
      pCanvas->TextOutW(r.x,r.y,msgw);
    } break;
    default: {
      const char *pmsg;
      pmsg=CM_ItemTexts1[Index];
      pCanvas->TextOutUTF8(r.x,r.y,pmsg);
      pmsg=Lang_GetTextUTF8(CM_ItemTexts2eng[Index],CM_ItemTexts2jpn[Index]);
      if(Index!=6){
        r.x+=42;
        }else{
        r.x+=56;
      }
      pCanvas->TextOutUTF8(r.x,r.y,pmsg);
    }
  }
}

static void CB_KeyDown(u32 Keys)
{
  if(Modal_RatioMenu==true){
    RM_CB_KeyDown(Keys);
    return;
  }
  if(Modal_StartPosMenu==true){
    SPM_CB_KeyDown(Keys);
    return;
  }
  
  if(Modal_CommandMenu==true) return;
  
  if((Keys&KEY_L)!=0){
    TComponentFrame *pcf=&CM_ComponentFrame;
    ComponentFrame_Init(pcf,pScreenSubBM->pCanvas);
    
    pcf->pTitleUTF8_eng="Extend command menu";
    pcf->pTitleUTF8_jpn="拡張コマンドメニュー";
    pcf->ItemCount=CM_ItemCount;
    pcf->ItemHeight=16;
    pcf->TextOfsX=4;
    pcf->TextOfsY=1;
    s32 w=180;
    s32 h=pcf->ItemHeight*pcf->ItemCount;
    pcf->Rect=CreateRect((AutoScreenWidth-w)/2,AutoScreenHeight-24-h,w,h);
    pcf->FillAlpha=25;
    pcf->CallBack_ItemDraw=CM_CB_ItemDraw;
    
    ComponentFrame_Draw(pcf);
    
    Modal_CommandMenu=true;
  }
}

static void CB_KeyPress(u32 VsyncCount,u32 Keys)
{
  if(Modal_RatioMenu==true){
    RM_CB_KeyPress(VsyncCount,Keys);
    return;
  }
  if(Modal_StartPosMenu==true){
    SPM_CB_KeyPress(VsyncCount,Keys);
    return;
  }
  
  if(Modal_CommandMenu==true){
    TCurrent *pcur=&Current;
    if((Keys&KEY_UP)!=0){
      NextProc=ENP_Thumbnail;
    }
    if((Keys&KEY_LEFT)!=0){
      if(CurIsFirstPage()==false){
        pcur->ImageIndex--;
        NextProc=ENP_View;
      }
    }
    if((Keys&KEY_RIGHT)!=0){
      if(CurIsLastPage()==false){
        pcur->ImageIndex++;
        NextProc=ENP_View;
      }
    }
    if((Keys&KEY_DOWN)!=0){
      Modal_CommandMenu=false;
      RM_Start();
    }
    if((Keys&KEY_SELECT)!=0){
      Modal_CommandMenu=false;
      SPM_Start();
    }
    return;
  }
  
  if((Keys&KEY_START)!=0){
    NextProc=GetNextProc_SlideOpt();
    return;
  }
  if((Keys&KEY_SELECT)!=0){
    NextProc=ENP_Thumbnail;
    return;
  }
  
  s32 vx=0,vy=0;
  
  if((Keys&KEY_LEFT)!=0) vx=-1;
  if((Keys&KEY_RIGHT)!=0) vx=1;
  if((Keys&KEY_UP)!=0) vy=-1;
  if((Keys&KEY_DOWN)!=0) vy=1;
  
  if((vx!=0)||(vy!=0)){
    TScreen *pscr=&Screen;
    
    pscr->posx+=vx*4*VsyncCount;
    pscr->posy+=vy*4*VsyncCount;
    Screen_FitPosition();
    
    IPK_DrawLargeThumbnail();
    DrawScreen();
    DrawCornerIcon();
  }
}

static void CB_KeyUp(u32 Keys)
{
  if(Modal_RatioMenu==true){
    RM_CB_KeyUp(Keys);
    return;
  }
  if(Modal_StartPosMenu==true){
    SPM_CB_KeyUp(Keys);
    return;
  }
  
  if((Keys&KEY_L)!=0){
    if(Modal_CommandMenu==true){
      Modal_CommandMenu=false;
      IPK_FullDrawLargeThumbnail();
      DrawScreen();
      DrawCornerIcon();
    }
  }
  
}

static bool deskmf;
static ECornerItem deskCornerItem;
static s32 deskmx,deskmy;
static s32 deskpx,deskpy;

static void CB_MouseDown(s32 x,s32 y)
{
  if(Modal_RatioMenu==true){
    RM_CB_MouseDown(x,y);
    return;
  }
  if(Modal_StartPosMenu==true){
    SPM_CB_MouseDown(x,y);
    return;
  }
  
  if(Modal_CommandMenu==true) return;
  
  DrawCornerIcon();
  
  deskmf=false;
  deskmx=x;
  deskmy=y;
  
  deskCornerItem=GetInsideCornerItem(x,y);
  if((deskCornerItem==ECI_LeftUp)||(deskCornerItem==ECI_RightUp)||(deskCornerItem==ECI_CenterDown_Left)||(deskCornerItem==ECI_CenterDown_Center)||(deskCornerItem==ECI_CenterDown_Right)||(deskCornerItem==ECI_CenterTop_Left)||(deskCornerItem==ECI_CenterTop_Right)) return;
  
  if(deskCornerItem==ECI_LeftDown){
    if(CurIsFirstPage()==false) return;
  }
  if(deskCornerItem==ECI_RightDown){
    if(CurIsLastPage()==false) return;
  }
  
  deskCornerItem=ECI_None;
  
  s32 ratio=Current.ScreenRatio;
  
  if(IPKState.DoubleScreen==false){
    if(Screen_IsInsideBoundBox(x,y)==false){
      TIPKThumbnail *pth=&Current.thumb;
      TScreen *pscr=&Screen;
      s32 bx=x-pscr->ThumbnailOffsetX;
      s32 by=y-pscr->ThumbnailOffsetY;
      s32 bw=(pscr->boundwidth*pth->Ratio)/0x10000;
      s32 bh=(pscr->boundheight*pth->Ratio)/0x10000;
      bw=bw*0x100/ratio;
      bh=bh*0x100/ratio;
      Screen_SetPosFromDisplay(bx-(bw/2),by-(bh/2));
      IPK_DrawLargeThumbnail();
      DrawScreen();
      DrawCornerIcon();
    }
    
    TIPKThumbnail *pth=&Current.thumb;
    TScreen *pscr=&Screen;
    
    deskpx=(pscr->posx*pth->Ratio)/0x10000;
    deskpy=(pscr->posy*pth->Ratio)/0x10000;
    deskpx=deskpx*0x100/ratio;
    deskpy=deskpy*0x100/ratio;
    
    }else{
    TScreen *pscr=&Screen;
    
    deskpx=pscr->posx*0x100/ratio;
    deskpy=pscr->posx*0x100/ratio;
  }
  
  deskmf=true;
}

static void CB_MouseMove(s32 x,s32 y,bool CanIgnore)
{
  if(Modal_RatioMenu==true){
    RM_CB_MouseMove(x,y,CanIgnore);
    return;
  }
  if(Modal_StartPosMenu==true){
    SPM_CB_MouseMove(x,y,CanIgnore);
    return;
  }
  
  if(Modal_CommandMenu==true) return;
  
  if(deskmf==false) return;
  if(CanIgnore==true) return;
  
  if(IPKState.DoubleScreen==false){
    s32 vx=x-deskmx;
    s32 vy=y-deskmy;
    
    Screen_SetPosFromDisplay(deskpx+vx,deskpy+vy);
    }else{
    s32 vx=x-deskmx;
    s32 vy=y-deskmy;
    
    TScreen *pscr=&Screen;
    pscr->posx-=vx*2;
    pscr->posy-=vy*2;
    
    Screen_FitPosition();
    
    deskmx=x;
    deskmy=y;
  }
  
  IPK_DrawLargeThumbnail();
  DrawScreen();
}

static void CB_MouseUp(s32 x,s32 y)
{
  if(Modal_RatioMenu==true){
    RM_CB_MouseUp(x,y);
    return;
  }
  if(Modal_StartPosMenu==true){
    SPM_CB_MouseUp(x,y);
    return;
  }
  
  if(Modal_CommandMenu==true){
    TCurrent *pcur=&Current;
    s32 idx=ComponentFrame_GetIndexFromPos(&CM_ComponentFrame,x,y);
    switch(idx){
      case 0: break;
      case 1: break;
      case 2: NextProc=ENP_Thumbnail; break;
      case 3: {
        if(CurIsFirstPage()==false){
          pcur->ImageIndex--;
          NextProc=ENP_View;
        }
      } break;
      case 4: {
        if(CurIsLastPage()==false){
          pcur->ImageIndex++;
          NextProc=ENP_View;
        }
      } break;
      case 5: {
        Modal_CommandMenu=false;
        RM_Start();
      } break;
      case 6: {
        Modal_CommandMenu=false;
        SPM_Start();
      } break;
    }
    return;
  }
  
  if(deskmf==false){
    ECornerItem ECI=GetInsideCornerItem(x,y);
    if(ECI!=ECI_None){
      if(ECI!=deskCornerItem) return;
      TCurrent *pcur=&Current;
      switch(ECI){
        case ECI_LeftUp: {
          PressNDSL();
        } break;
        case ECI_RightUp: {
          NextProc=ENP_Thumbnail;
        } break;
        case ECI_LeftDown: {
          pcur->ImageIndex--;
          NextProc=ENP_View;
        } break;
        case ECI_RightDown: {
          pcur->ImageIndex++;
          NextProc=ENP_View;
        } break;
        case ECI_CenterDown_Left: {
          TCurrent *pcur=&Current;
          s32 idx=GetRatioIndexFromVector(pcur->ScreenRatio,-1);
          RM_End(RatioList[idx]);
        } break;
        case ECI_CenterDown_Center: {
          RM_Start();
        } break;
        case ECI_CenterDown_Right: {
          TCurrent *pcur=&Current;
          s32 idx=GetRatioIndexFromVector(pcur->ScreenRatio,1);
          RM_End(RatioList[idx]);
        } break;
        case ECI_CenterTop_Left: {
          ToggleDisplayMode();
          NextProc=ENP_View;
        } break;
        case ECI_CenterTop_Right: {
          bool f=IPKState.DoubleScreen;
          if((extmem_ExistMemory()==false)&&(GlobalINI.CustomConfig.ViewMode_DoubleScreenCompulsorily==false)){
            if(f==false) return;
            f=false;
            }else{
            if(f==false){
              f=true;
              }else{
              f=false;
            }
          }
          IPKState.DoubleScreen=f;
          NextProc=ENP_View;
        } break;
        default: break;
      }
      return;
    }
  }
  
  if(deskmf==false) return;
  
  CB_MouseMove(x,y,false);
  DrawCornerIcon();
  
  deskmf=false;
}

static bool Preload_enabled;
static u32 Preload_curidx;

static void CB_Start(void)
{
  // 449.216ms
  
  ScreenInit(ESS_Bitmap,IPKState.DisplayMode);
  
  {
    const u32 s=CornerIconSize;
    CglB15 *pB15=Corner_GetIcon(EIT_Mouse);
    pScreenSubBM->Mouse_Init(256-s,256-s,pB15->pCanvas);
  }
  
  {
    CglCanvas *pcan=pScreenSubBM->pCanvas;
    pcan->SetColor(GlobalBGColor15);
    pcan->FillAll();
  }
  {
    CglCanvas *pcan=pScreenMain->pCanvas;
    pcan->SetColor(GlobalBGColor15);
    pcan->FillAll();
  }
  
  Modal_RatioMenu=false;
  Modal_StartPosMenu=false;
  Modal_CommandMenu=false;
  deskmf=false;
  deskCornerItem=ECI_None;
  
  OpenCurrent(); // 277.440ms
  
  Screen_Reset();
  
  extmem_Init();
  extmem_SetCount(Current.MCUCount);
  Preload_enabled=extmem_ExistMemory();
  Preload_curidx=0;
  
  safemalloc_RequestFreeArea=View_safemalloc_RequestFreeArea;
  
  pScreenMain->SetWhiteOut(16);
  pScreenSubBM->SetAlpha(0);
  
/*
  pRatioIconBGCanvas=new CglCanvas(NULL,RatioIconWidth,RatioIconHeight,pf15bit);
  DrawCornerIcon(); // true
*/
  InitCornerIcon();
  
  SetPosOnlyFromStartPos(IPKState.StartPos);
  IPK_FullDrawLargeThumbnail();
  DrawScreen(); // 150.400ms
  DrawCornerIcon();
  
  s32 blend=0;
  while(1){
    WaitForVBlank(true);
    blend++;
    pScreenMain->SetWhiteOut(16-blend);
    pScreenSubBM->SetAlpha(blend);
    if(blend==16) break;
  }
  
//  pScreenSubBM->SetAlpha(8); // for Debug.
//  pScreenSubBM->SetAlpha(15); // for Debug.
}

static void CB_VsyncUpdate(u32 VsyncCount)
{
  bool ProcTimeout=true;
  
  if(Modal_RatioMenu==true) ProcTimeout=false;
  if(Modal_StartPosMenu==true) ProcTimeout=false;
  if(Modal_CommandMenu==true) ProcTimeout=false;
  if(deskmf==true) ProcTimeout=false;
  
  if(ProcTimeout==true){
    if(CornerIconTimeoutVsync!=0){
      CornerIconTimeoutVsync--;
      if(CornerIconTimeoutVsync<CornerIconSize) HideCornerIcon(CornerIconSize-CornerIconTimeoutVsync);
    }
  }
  
  if(2<=VsyncCount) return;
  
  if(Preload_enabled==false) return;
  
  extern volatile bool VBlankPassed;
  
  TCurrent *pcur=&Current;
  
  while(VBlankPassed==false){
    u32 idx=Preload_curidx;
    if(extmem_Exists(idx)==false){
      const u32 MCUBytes=IPKMCUWidth*IPKMCUHeight*2;
      if(extmem_Alloc(idx,IPKMCUWidth*IPKMCUHeight*2)==false){
        _consolePrintf("Preload: extmem free area overflow.\n stopped at %d/%d.\n",idx,pcur->MCUCount);
        Preload_enabled=false;
        return;
        }else{
        u16 *pBuf=(u16*)safemalloc(MCUBytes);
        pcur->pipk->GetMCUData(&pcur->imginfo,pcur->pMCUOffsets[idx],pBuf,MCUBytes);
        extmem_Write(idx,pBuf,MCUBytes);
        safefree(pBuf); pBuf=NULL;
      }
    }
    Preload_curidx++;
    if(Preload_curidx==pcur->MCUCount){
      _consolePrintf("Preload: Full loaded.\n");
      Preload_enabled=false;
      return;
    }
  }
}

static void CB_End(void)
{
  s32 blend=16;
  while(1){
    WaitForVBlank(true);
    blend--;
    pScreenMain->SetWhiteOut(16-blend);
    pScreenSubBM->SetAlpha(blend);
    if(blend==0) break;
  }
  
  safemalloc_RequestFreeArea=NULL;
  
/*
  delete pRatioIconBGCanvas; pRatioIconBGCanvas=NULL;
*/
  
  CloseCurrent();
  
  extmem_Free();
  
  ScreenEnd(ESS_Bitmap);
}

void ProcView_SetCallBack(void)
{
  CallBack_Start=CB_Start;
  CallBack_VsyncUpdate=CB_VsyncUpdate;
  CallBack_End=CB_End;
  CallBack_KeyDown=CB_KeyDown;
  CallBack_KeyPress=CB_KeyPress;
  CallBack_KeyUp=CB_KeyUp;
  CallBack_MouseDown=CB_MouseDown;
  CallBack_MouseMove=CB_MouseMove;
  CallBack_MouseUp=CB_MouseUp;
}

