
#define MaxFilenameLength (256)

typedef u16 UnicodeChar;

#define chkdsk_MsgOfsY (8+(16*6))

enum EFatalErrorType {EFET_DuplicatedArea,EFET_BrokenEntry,EFET_IlligalClusterLink,EFET_BrokenDirectoryLink,EFET_BrokenUnicode,EFET_FileSizeError};

static void chkdsk_DuplicateCluster_ShowFatalError(EFatalErrorType FatalErrorType)
{
  CglCanvas *pCanvas=pScreenMain->pCanvas;
  
  pCanvas->FillFull(GlobalBGColor15);
  
  u32 x=8,y=8;
  
  pCanvas->TextOutUTF8(x,y,"! ディスクに致命的なエラーを発見しました !");
  y+=16;
  y+=16;
  switch(FatalErrorType){
    case EFET_DuplicatedArea: {
      pCanvas->TextOutUTF8(x,y,"複数のファイルが重なった領域を使用しています。");
      y+=16;
      pCanvas->TextOutUTF8(x,y,"いくつかのファイルは既に破壊されています。");
      y+=16;
    } break;
    case EFET_BrokenEntry: {
      pCanvas->TextOutUTF8(x,y,"ディレクトリエントリが異常です。");
      y+=16;
      pCanvas->TextOutUTF8(x,y,"一つ以上のファイルが破損しています。");
      y+=16;
    } break;
    case EFET_IlligalClusterLink: {
      pCanvas->TextOutUTF8(x,y,"クラスタリンクが規格外で途切れました。");
      y+=16;
      pCanvas->TextOutUTF8(x,y,"ファイルが破損している可能性があります。");
      y+=16;
    } break;
    case EFET_BrokenDirectoryLink: {
      pCanvas->TextOutUTF8(x,y,"ディレクトリエントリが異常です。");
      y+=16;
      pCanvas->TextOutUTF8(x,y,"フォルダが破壊されています。");
      y+=16;
    } break;
    case EFET_BrokenUnicode: {
      pCanvas->TextOutUTF8(x,y,"ディレクトリエントリが異常です。");
      y+=16;
      pCanvas->TextOutUTF8(x,y,"長いファイル名が破壊されています。");
      y+=16;
    } break;
    case EFET_FileSizeError: {
      pCanvas->TextOutUTF8(x,y,"ディレクトリエントリが示すサイズと、");
      y+=16;
      pCanvas->TextOutUTF8(x,y,"実際のファイルサイズが違います。");
      y+=16;
    } break;
    default: {
      pCanvas->TextOutUTF8(x,y,"未定義エラー");
      y+=16;
    } break;
  }
  y+=16;
  pCanvas->TextOutUTF8(x,y,"回復手段：");
  y+=16;
  pCanvas->TextOutUTF8(x,y,"全てのファイルをHDDにバックアップして、");
  y+=16;
  pCanvas->TextOutUTF8(x,y,"フォーマット後にコピーし直して下さい。");
  y+=16;
  y+=16;
  pCanvas->TextOutUTF8(x,y,"アプリケーションを停止しました。");
  
  ShowLogHalt();
}

static void chkdsk_DuplicateCluster_ShowBufferOverflow(void)
{
  CglCanvas *pCanvas=pScreenMain->pCanvas;
  
  pCanvas->FillFull(GlobalBGColor15);
  
  u32 x=8,y=8;
  
  pCanvas->TextOutUTF8(x,y,"扱えないフォーマットのディスクです。");
  y+=16;
  y+=16;
  pCanvas->TextOutUTF8(x,y,"ディスクサイズが大きすぎるか、");
  y+=16;
  pCanvas->TextOutUTF8(x,y,"クラスタサイズが小さすぎます。");
  y+=16;
  y+=16;
  pCanvas->TextOutUTF8(x,y,"回復手段：");
  y+=16;
  pCanvas->TextOutUTF8(x,y,"もっと大きなクラスタサイズで、");
  y+=16;
  pCanvas->TextOutUTF8(x,y,"フォーマットを試みて下さい。");
  y+=16;
  y+=16;
  pCanvas->TextOutUTF8(x,y,"アプリケーションを停止しました。");
  
  ShowLogHalt();
}

#define pClusterFlagsSize (1*1024*1024)
static u8 *pClusterFlags;

static void chkdsk_DuplicateCluster_Init(void)
{
  pClusterFlags=(u8*)safemalloc(pClusterFlagsSize);
  
  for(u32 idx=0;idx<pClusterFlagsSize;idx++){
    pClusterFlags[idx]=0;
  }
}

static void chkdsk_DuplicateCluster_SetFlag(u32 Cluster)
{
  if((pClusterFlagsSize*8)<=Cluster){
    _consolePrint("FileSysterm: Fatal error. Cluster check buffer overflow.\n");
    chkdsk_DuplicateCluster_ShowBufferOverflow();
    return;
  }
  
  u32 flags=pClusterFlags[Cluster/8];
  
  u32 bit=1<<(Cluster&7);
  
  if((flags&bit)!=0){
    _consolePrint("FileSysterm: Fatal error. Duplicate cluster link finded.\n");
    chkdsk_DuplicateCluster_ShowFatalError(EFET_DuplicatedArea);
    return;
  }
  
  flags|=bit;
  
  pClusterFlags[Cluster/8]=flags;
}

static void chkdsk_DuplicateCluster_Free(void)
{
  if(pClusterFlags!=NULL){
    safefree(pClusterFlags); pClusterFlags=NULL;
  }
}

#define CLUSTER_FREE	0x0000
#define	CLUSTER_EOF	0x0FFFFFFF

static void chkdsk_DuplicateCluster_FillUsedCluster(const char *pPath,const char *pAlias,const u32 FileSize)
{
  u32 ClusterSize=FAT_GetFS_SecPerClus()*512;
  u32 UsedClusterCount=(FileSize+(ClusterSize-1))/ClusterSize;
  u32 ClusterCount=0;
  
  FAT_FILE *pfile;
  
  {
    char fullfn[1024];
    if((pPath[0]=='/')&&(pPath[1]==0)){
      snprintf(fullfn,256,"/%s",pAlias);
      }else{
      snprintf(fullfn,256,"%s/%s",pPath,pAlias);
    }
    pfile=FAT_fopen(fullfn,"r");
    if(pfile==NULL){
      _consolePrint("FileSysterm: Fatal error. File open error.\n");
      chkdsk_DuplicateCluster_ShowFatalError(EFET_BrokenEntry);
      return;
    }
  }
  
  u32 CurClus=pfile->firstCluster;
  
  if(CurClus!=CLUSTER_FREE){
    while(1){
      chkdsk_DuplicateCluster_SetFlag(CurClus);
      
      ClusterCount++;
      
      u32 NextClus=FAT_NextCluster(CurClus);
      if(NextClus==CLUSTER_EOF) break;
      
      CurClus=NextClus;
      
      if(CurClus==CLUSTER_FREE){
        _consolePrint("FileSysterm: Fatal error. Can not search cluster link.\n");
        chkdsk_DuplicateCluster_ShowFatalError(EFET_IlligalClusterLink);
        return;
      }
    }
  }
  
  FAT_fclose(pfile);
  
  if(UsedClusterCount!=ClusterCount){
    _consolePrintf("%s/%s\n",pPath,pAlias);
    _consolePrintf("Entry clusters: %d, Actual clusters:%d.\n",UsedClusterCount,ClusterCount);
    _consolePrint("A size that the directory entry shows and an actual size of the file are different.\n");
    chkdsk_DuplicateCluster_ShowFatalError(EFET_FileSizeError);
  }
}

static void chkdsk_DuplicateCluster_ins(const char *pPath)
{
  if(FAT_CWD(pPath)==false){
    _consolePrintf("FileSysterm: Fatal error. FAT_CWD(%s)==false.\n",pPath);
    chkdsk_DuplicateCluster_ShowFatalError(EFET_BrokenDirectoryLink);
    return;
  }
  
  u32 FolderCount=0;
  char *pFolderNameAlias=(char*)safemalloc(512*(12+1));
  u32 FileCount=0;
  char *pFileNameAlias=(char*)safemalloc(512*(12+1));
  u32 *pFileSize=(u32*)safemalloc(512*4);
  
  {
    static char afn[MaxFilenameLength];
    static UnicodeChar ufn[MaxFilenameLength];
    
    u32 FAT_FileType=FAT_FindFirstFile(afn);
    
    while(FAT_FileType!=FT_NONE){
      switch(FAT_FileType){
        case FT_NONE: break;
        case FT_DIR: {
          if((strcmp(afn,".")!=0)&&(strcmp(afn,"..")!=0)){
            if(FAT_GetLongFilenameUnicode(ufn,MaxFilenameLength)==false){
              _consolePrintf("FileSysterm: Fatal error. Unicode filename read error.\n Alias='%s'\n",afn);
              chkdsk_DuplicateCluster_ShowFatalError(EFET_BrokenUnicode);
              return;
            }
            strcpy(&pFolderNameAlias[FolderCount*(12+1)],afn);
            FolderCount++;
          }
        } break;
        case FT_FILE: {
          if(FAT_GetLongFilenameUnicode(ufn,MaxFilenameLength)==false){
            _consolePrintf("FileSysterm: Fatal error. Unicode filename read error.\n Alias='%s'\n",afn);
            chkdsk_DuplicateCluster_ShowFatalError(EFET_BrokenUnicode);
            return;
          }
          strcpy(&pFileNameAlias[FileCount*(12+1)],afn);
          pFileSize[FileCount]=FAT_GetFileSize();
          FileCount++;
        } break;
      }
      
      FAT_FileType=FAT_FindNextFile(afn);
    }
  }
  
  for(u32 idx=0;idx<FileCount;idx++){
    chkdsk_DuplicateCluster_FillUsedCluster(pPath,&pFileNameAlias[idx*(12+1)],pFileSize[idx]);
  }
  
  for(u32 idx=0;idx<FolderCount;idx++){
    char str[MaxFilenameLength];
    if((pPath[0]=='/')&&(pPath[1]==0)){
      snprintf(str,256,"/%s",&pFolderNameAlias[idx*(12+1)]);
      }else{
      snprintf(str,256,"%s/%s",pPath,&pFolderNameAlias[idx*(12+1)]);
    }
    chkdsk_DuplicateCluster_ins(str);
  }
  
  if(pFolderNameAlias!=NULL){
    safefree(pFolderNameAlias); pFolderNameAlias=NULL;
  }
  if(pFileNameAlias!=NULL){
    safefree(pFileNameAlias); pFileNameAlias=NULL;
  }
  if(pFileSize!=NULL){
    safefree(pFileSize); pFileSize=NULL;
  }
}

static void chkdsk_DuplicateCluster(void)
{
  {
    CglCanvas *pCanvas=pScreenMain->pCanvas;
    pCanvas->TextOutUTF8(16,chkdsk_MsgOfsY+(16*0),"重複クラスタと断片化を調べています。");
  }
  
  chkdsk_DuplicateCluster_Init();
  
  chkdsk_DuplicateCluster_ins("/");
  
  chkdsk_DuplicateCluster_Free();
}

static void chkdsk_WriteTest_ShowError(void)
{
  CglCanvas *pCanvas=pScreenMain->pCanvas;
  
  pCanvas->FillFull(GlobalBGColor15);
  
  u32 x=8,y=8;
  
  pCanvas->TextOutUTF8(x,y,"ディスクアクセステストに失敗しました。");
  y+=16;
  y+=16;
  pCanvas->TextOutUTF8(x,y,"・ディスク空き容量が極端に少ない");
  y+=16;
  pCanvas->TextOutUTF8(x,y,"・書き込みに対応していないDLDIドライバ");
  y+=16;
  pCanvas->TextOutUTF8(x,y,"・CF/SDとアダプタの相性、接触不良");
  y+=16;
  pCanvas->TextOutUTF8(x,y,"・ディスク管理エリアが破損している");
  y+=16;
  y+=16;
  pCanvas->TextOutUTF8(x,y,"などの原因が考えられます。");
  y+=16;
  y+=16;
  pCanvas->TextOutUTF8(x,y,"安全のため、アプリケーションを停止しました。");
  
  ShowLogHalt();
}

#define chkdsk_TempFilename "_CHKDSK_.$$$"

#include "disc_io.h"
#include "mediatype.h"

extern LPIO_INTERFACE active_interface;

static void chkdsk_WriteTest(void)
{
  {
    CglCanvas *pCanvas=pScreenMain->pCanvas;
    pCanvas->TextOutUTF8(16,chkdsk_MsgOfsY+(16*2),"ディスクの読み書きテストをしています。");
  }
  
  if((active_interface->ul_Features & FEATURE_MEDIUM_CANWRITE)==0){
    _consolePrintf("\nnot support disk write function.\nAdapter = %s\n",DIMediaName);
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  const u32 TestSize=(16*1024)/4;
  u32 *pTestBuf=(u32*)safemalloc(TestSize*4);
  
  if(pTestBuf==NULL){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  _consolePrintf("Disk read-write check. [%s] / ",chkdsk_TempFilename);
  
  FAT_FILE *pfile;
  
  _consolePrint("Open file for write. / ");
  pfile=FAT_fopen(chkdsk_TempFilename,"w");
  if(pfile==NULL){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  for(u32 idx=0;idx<TestSize;idx++){
    pTestBuf[idx]=idx;
  }
  
  _consolePrint("File write. / ");
  if(FAT_fwrite(pTestBuf,4,TestSize,pfile)!=(TestSize*4)){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  _consolePrint("Check position. / ");
  if(FAT_ftell(pfile)!=(TestSize*4)){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  _consolePrint("File seek. / ");
  if(FAT_fseek(pfile,0,SEEK_SET)!=0){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  _consolePrint("Check position. / ");
  if(FAT_ftell(pfile)!=0){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  _consolePrint("File close. / ");
  if(FAT_fclose(pfile)==false){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  for(u32 idx=0;idx<TestSize;idx++){
    pTestBuf[idx]=0;
  }
  
  _consolePrint("Open file for read. / ");
  pfile=FAT_fopen(chkdsk_TempFilename,"r");
  if(pfile==NULL){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  _consolePrint("File read. / ");
  if(FAT_fread(pTestBuf,4,TestSize,pfile)!=(TestSize*4)){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  _consolePrint("Check position. / ");
  if(FAT_ftell(pfile)!=(TestSize*4)){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  _consolePrint("Check values. / ");
  for(u32 idx=0;idx<TestSize;idx++){
    if(pTestBuf[idx]!=idx){
      chkdsk_WriteTest_ShowError();
      return;
    }
  }
  
  _consolePrint("File close. / ");
  if(FAT_fclose(pfile)==false){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  _consolePrint("Remove file. / ");
  if(FAT_remove(chkdsk_TempFilename)!=0){
    chkdsk_WriteTest_ShowError();
    return;
  }
  
  _consolePrint("Succeeded.\n");
  
  safefree(pTestBuf);
}

