
#include <stdio.h>
#include <NDS.h>

#include "plugin.h"
#include "plugin_def.h"

#include "libmp3/mad.h"

// --------------------------------------------------------------------

void abort(void)
{
  _consolePrintf("ABORT!!\n");
  ShowLogHalt();
  while(1);
}

static int FileHandle;
static s32 FileSize;
static s32 FileOffset;

typedef struct mad_decoder Tmad_decoder;

static Tmad_decoder StaticMadDecoder;

#define SamplesPerFrame (1152*1)

// --------------------------------------------------------------------

static s16 *outlbuf,*outrbuf;
static u32 outofs;

#define INPUT_BUFFER_SIZE (2*1024)
static u8 *InputBuffer=NULL;

static
enum mad_flow input(void *data,
                    struct mad_stream *stream)
{
  size_t ReadSize,Remaining;
  u8 *ReadStart;
  
  if(stream->next_frame!=NULL)
  {
    Remaining=stream->bufend-stream->next_frame;
    memmove(InputBuffer,stream->next_frame,Remaining);
    ReadStart=InputBuffer+Remaining;
    ReadSize=INPUT_BUFFER_SIZE-Remaining;
    
    }else{
    ReadSize=INPUT_BUFFER_SIZE;
    ReadStart=InputBuffer;
    Remaining=0;
  }
  
  if((FileOffset+ReadSize)>=FileSize){
    ReadSize=FileSize-FileOffset;
  }
  
  if(ReadSize<=0) return MAD_FLOW_STOP;
  
  fread(ReadStart,1,ReadSize,FileHandle);
  FileOffset+=ReadSize;
  
//  _consolePrintf("r%d->%d\n",Remaining,ReadSize);
  
  mad_stream_buffer(stream,InputBuffer,ReadSize+Remaining);
  
  stream->error=0;
  
  return MAD_FLOW_CONTINUE;
}

static inline
signed int scale(mad_fixed_t sample)
{
  /* round */
  sample += (1L << (MAD_F_FRACBITS - 16));

  /* clip */
  if (sample >= MAD_F_ONE)
    sample = MAD_F_ONE - 1;
  else if (sample < -MAD_F_ONE)
    sample = -MAD_F_ONE;

  /* quantize */
  return sample >> (MAD_F_FRACBITS + 1 - 16);
}

static
enum mad_flow output(void *data,
                     struct mad_header const *header,
                     struct mad_pcm *pcm)
{
  unsigned int nchannels, nsamples;
  mad_fixed_t const *left_ch, *right_ch;

  /* pcm->samplerate contains the sampling frequency */

  nchannels = pcm->channels;
  nsamples  = pcm->length;
  left_ch   = pcm->samples[0];
  right_ch  = pcm->samples[1];

  if(nsamples!=SamplesPerFrame){
//    _consolePrintf("nsamples==%d!=%d\n",nsamples,SamplesPerFrame);
//    return MAD_FLOW_STOP;
  }
  
  if((outlbuf==NULL)||(outrbuf==NULL)){
    outofs+=nsamples;
    return MAD_FLOW_CONTINUE;
  }
  
  if (nchannels == 1) {
    while (nsamples--) {
      signed int sample;

      sample = scale(*left_ch++);
      outlbuf[outofs]=sample;
      outrbuf[outofs]=sample;
      outofs++;
    }
    return MAD_FLOW_CONTINUE;
  }

  if (nchannels == 2) {
    while (nsamples--) {
      signed int sample;

      sample = scale(*left_ch++);
      outlbuf[outofs]=sample;
      sample = scale(*right_ch++);
      outrbuf[outofs]=sample;
      outofs++;
    }
    return MAD_FLOW_CONTINUE;
  }

  return MAD_FLOW_STOP;
}

static
enum mad_flow error(void *data,
                    struct mad_stream *stream,
                    struct mad_frame *frame)
{
  _consolePrintf("decoding error 0x%04x (%s) at byte offset %u\n",
          stream->error, mad_stream_errorstr(stream),
          FileOffset);

  /* return MAD_FLOW_BREAK here to stop decoding (and propagate an error) */

  return MAD_FLOW_CONTINUE;//MAD_FLOW_STOP;
}

// --------------------------------------------------------------------

#define madusr_Return_OK (0)
#define madusr_Return_NG (-1)

#define madusr_ExecReturn_Next (0)
#define madusr_ExecReturn_End (-1)
#define madusr_ExecReturn_Fail (-2)

static
int madusr_decode_init(struct mad_decoder *decoder)
{
  decoder->sync = NULL;
  
  /* configure input, output, and error functions */
  
  mad_decoder_init(decoder, NULL, input, 0 /* header */, 0 /* filter */, output, error, 0 /* message */);
  
  if (decoder->input_func == 0) return madusr_Return_NG;
  if (decoder->output_func == 0) return madusr_Return_NG;
  if (decoder->error_func==0) return madusr_Return_NG;
  
  struct mad_stream *stream;
  struct mad_frame *frame;
  struct mad_synth *synth;
  
  decoder->sync = safemalloc(sizeof(*decoder->sync));
  if (decoder->sync == NULL) return madusr_Return_NG;
  
  stream = &decoder->sync->stream;
  frame  = &decoder->sync->frame;
  synth  = &decoder->sync->synth;

  mad_stream_init(stream);
  mad_frame_init(frame);
  mad_synth_init(synth);

  mad_stream_options(stream, decoder->options);
  
  if(decoder->input_func(decoder->cb_data, stream)!=MAD_FLOW_CONTINUE) return madusr_Return_NG;
  
  return madusr_Return_OK;
}

//#include "../_consoleWriteLog.h"

static
int madusr_decode_exec(struct mad_decoder *decoder)
{
  enum mad_flow (*error_func)(void *, struct mad_stream *, struct mad_frame *);
  void *error_data;
  
  struct mad_stream *stream;
  struct mad_frame *frame;
  struct mad_synth *synth;

  error_func = decoder->error_func;
  error_data = decoder->cb_data;
  
  stream = &decoder->sync->stream;
  frame  = &decoder->sync->frame;
  synth  = &decoder->sync->synth;

  if(stream->error == MAD_ERROR_BUFLEN){
    return madusr_ExecReturn_Fail;
  }
  
//  PrfStart();
  
  while(1){
    if (decoder->header_func) {
      if (mad_header_decode(&frame->header, stream) == -1) {
        if (!MAD_RECOVERABLE(stream->error)){
          if(stream->error==MAD_ERROR_BUFLEN){
            if(decoder->input_func(decoder->cb_data, stream)!=MAD_FLOW_CONTINUE) return madusr_ExecReturn_Fail;
            continue;
          }
          return madusr_ExecReturn_Fail;
        }
  
        if(stream->error==MAD_ERROR_BUFLEN){
          if(decoder->input_func(decoder->cb_data, stream)!=MAD_FLOW_CONTINUE) return madusr_ExecReturn_Fail;
          continue;
        }
        
        switch (error_func(error_data, stream, frame)) {
          case MAD_FLOW_STOP:
            return madusr_ExecReturn_End;
          case MAD_FLOW_BREAK:
            return madusr_ExecReturn_Fail;
          case MAD_FLOW_IGNORE:
          case MAD_FLOW_CONTINUE:
          default:
            return madusr_ExecReturn_Next;
        }
      }
  
      switch (decoder->header_func(decoder->cb_data, &frame->header)) {
        case MAD_FLOW_STOP:
          return madusr_ExecReturn_End;
        case MAD_FLOW_BREAK:
          return madusr_ExecReturn_Fail;
        case MAD_FLOW_IGNORE:
          return madusr_ExecReturn_Next;
        case MAD_FLOW_CONTINUE:
          continue;
          break;
      }
    }
    break;
  }
  
//  PrfEnd(1);
//  PrfStart();
  
  while(1){
    if (mad_frame_decode(frame, stream) == -1) {
    
      if (!MAD_RECOVERABLE(stream->error)){
        if(stream->error==MAD_ERROR_BUFLEN){
          if(decoder->input_func(decoder->cb_data, stream)!=MAD_FLOW_CONTINUE) return madusr_ExecReturn_Fail;
          continue;
        }
        return madusr_ExecReturn_Fail;
      }
      
      switch (error_func(error_data, stream, frame)) {
        case MAD_FLOW_STOP:
          return madusr_ExecReturn_End;
        case MAD_FLOW_BREAK:
          return madusr_ExecReturn_Fail;
        case MAD_FLOW_IGNORE:
          return madusr_ExecReturn_Next;
        case MAD_FLOW_CONTINUE:
          continue;
        default:
          break;
      }
    }
    break;
  }
  
//  PrfEnd(2);
//  PrfStart();
  
  if((outlbuf==NULL)||(outrbuf==NULL)){
    outofs+=synth->pcm.length;
    return madusr_ExecReturn_Next;
  }
  
  if (decoder->filter_func) {
    switch (decoder->filter_func(decoder->cb_data, stream, frame)) {
      case MAD_FLOW_STOP:
        return madusr_ExecReturn_End;
      case MAD_FLOW_BREAK:
        return madusr_ExecReturn_Fail;
      case MAD_FLOW_IGNORE:
        return madusr_ExecReturn_Next;
      case MAD_FLOW_CONTINUE:
        break;
    }
  }

//  PrfEnd(3);
//  PrfStart();
  
  mad_synth_frame(synth, frame);

//  PrfEnd(4);
//  PrfStart();
  
  if (decoder->output_func) {
    switch (decoder->output_func(decoder->cb_data, &frame->header, &synth->pcm)) {
      case MAD_FLOW_STOP:
        return madusr_ExecReturn_End;
      case MAD_FLOW_BREAK:
        return madusr_ExecReturn_Fail;
      case MAD_FLOW_IGNORE:
      case MAD_FLOW_CONTINUE:
        break;
    }
  }
  
//  PrfEnd(5);
  
  return madusr_ExecReturn_Next;
}

static
void madusr_decode_free(struct mad_decoder *decoder)
{
  struct mad_stream *stream;
  struct mad_frame *frame;
  struct mad_synth *synth;
  
  if(decoder->sync!=NULL){
    stream = &decoder->sync->stream;
    frame  = &decoder->sync->frame;
    synth  = &decoder->sync->synth;
    
    mad_synth_finish(synth);
    mad_frame_finish(frame);
    mad_stream_finish(stream);
    
    safefree(decoder->sync);
    decoder->sync = NULL;
  }
  
  mad_decoder_finish(decoder);
}

// --------------------------------------------------------------------

typedef struct {
  bool Enabled;
  char title[31];
  char artist[31];
  char album[31];
  char year[5];
  char comment[31];
  byte genre;
} TID3Tag;

static TID3Tag ID3Tag;

const char* GetGenreStr(u8 Genre)
{
  // NOTE: The spelling of these genre names is identical to those found in
  // Winamp and mp3info.
  static const char *genre_names="Blues_Classic Rock_Country_Dance_Disco_Funk_Grunge_Hip-Hop_Jazz_Metal_New Age_Oldies_Other_Pop_R&B_Rap_Reggae_Rock_Techno_Industrial_Alternative_Ska_Death Metal_Pranks_Soundtrack_Euro-Techno_Ambient_Trip-Hop_Vocal_Jazz+Funk_Fusion_Trance_Classical_Instrumental_Acid_House_Game_Sound Clip_Gospel_Noise_Alt. Rock_Bass_Soul_Punk_Space_Meditative_Instrumental Pop_Instrumental Rock_Ethnic_Gothic_Darkwave_Techno-Industrial_Electronic_Pop-Folk_Eurodance_Dream_Southern Rock_Comedy_Cult_Gangsta Rap_Top 40_Christian Rap_Pop/Funk_Jungle_Native American_Cabaret_New Wave_Psychedelic_Rave_Showtunes_Trailer_Lo-Fi_Tribal_Acid Punk_Acid Jazz_Polka_Retro_Musical_Rock & Roll_Hard Rock_Folk_Folk/Rock_National Folk_Swing_Fast-Fusion_Bebob_Latin_Revival_Celtic_Bluegrass_Avantgarde_Gothic Rock_Progressive Rock_Psychedelic Rock_Symphonic Rock_Slow Rock_Big Band_Chorus_Easy Listening_Acoustic_Humour_Speech_Chanson_Opera_Chamber Music_Sonata_Symphony_Booty Bass_Primus_Porn Groove_Satire_Slow Jam_Club_Tango_Samba_Folklore_Ballad_Power Ballad_Rhythmic Soul_Freestyle_Duet_Punk Rock_Drum Solo_A Cappella_Euro-House_Dance Hall_Goa_Drum & Bass_Club-House_Hardcore_Terror_Indie_BritPop_Negerpunk_Polsk Punk_Beat_Christian Gangsta Rap_Heavy Metal_Black Metal_Crossover_Contemporary Christian_Christian Rock_Merengue_Salsa_Thrash Metal_Anime_JPop_Synthpop";
  static const char *genre_names_none="not found genre.";
  
  const char *res=genre_names;
  
  while(*res!=0){
    if(Genre==0) return(res);
    if(*res=='_') Genre--;
    res++;
  }
  return(genre_names_none);
}

void ReadID3TAG(void)
{
  if(FileSize<128){
    ID3Tag.Enabled=false;
    return;
  }
  
  char buf[128];
  
  fseek(FileHandle,FileSize-128,SEEK_SET);
  fread(buf,1,128,FileHandle);
  fseek(FileHandle,0,SEEK_SET);
  
  if((buf[0]!='T')||(buf[1]!='A')||(buf[2]!='G')){
    ID3Tag.Enabled=false;
    return;
  }
  
  ID3Tag.Enabled=true;
  
  MemCopy8CPU(&buf[3],ID3Tag.title,30);
  ID3Tag.title[30]=0;
  
  MemCopy8CPU(&buf[33],ID3Tag.artist,30);
  ID3Tag.artist[30]=0;
  
  MemCopy8CPU(&buf[63],ID3Tag.album,30);
  ID3Tag.album[30]=0;
  
  MemCopy8CPU(&buf[93],ID3Tag.year,4);
  ID3Tag.year[4]=0;
  
  MemCopy8CPU(&buf[97],ID3Tag.comment,30);
  ID3Tag.comment[30]=0;
  
  ID3Tag.genre=(byte)buf[127];
}

// --------------------------------------------------------------------

void FreeMP3(void);

static s16 *DecBufL=NULL,*DecBufR=NULL;
static bool DecBufRefresh=false;

bool StartMP3(int _FileHandle)
{
  FileHandle=_FileHandle;
  FileOffset=0;
  
  fseek(FileHandle,0,SEEK_END);
  FileSize=ftell(FileHandle);
  fseek(FileHandle,0,SEEK_SET);
  
  ReadID3TAG();
  
  InputBuffer=(u8*)safemalloc(INPUT_BUFFER_SIZE+MAD_BUFFER_GUARD);
  if(InputBuffer==NULL){
    _consolePrintf("InputBuffer out of memory.\n");
    FreeMP3();
    return(false);
  }
  
  _consolePrintf("libmad init.\n");
  
  int result;
  
  result=madusr_decode_init(&StaticMadDecoder);
  if(result==madusr_Return_NG){
    _consolePrintf("madusr_decode_init()==madusr_Return_NG.\n");
    FreeMP3();
    return(false);
  }
  
  outlbuf=NULL;
  outrbuf=NULL;
  outofs=0;
  
  result=madusr_decode_exec(&StaticMadDecoder);
  if(result!=madusr_ExecReturn_Next){
    _consolePrintf("madusr_decode_exec()==%d!=madusr_ExecReturn_Next.\n",result);
    FreeMP3();
    return(false);
  }
  
  struct mad_frame *frame;
  
  frame  = &StaticMadDecoder.sync->frame;
  
  _consolePrintf("\n");
  
  _consolePrintf("Format=");
  switch(frame->header.layer){
    case MAD_LAYER_I:
      _consolePrintf("Layer I\n");
      break;
    case MAD_LAYER_II:
      _consolePrintf("Layer II\n");
      break;
    case MAD_LAYER_III:
      _consolePrintf("Layer III\n");
      break;
    default:
      _consolePrintf("unknown layer.\n");
      FreeMP3();
      return(false);
      break;
  }

  _consolePrintf("ChannelMode=");
  switch(frame->header.mode){
    case MAD_MODE_SINGLE_CHANNEL:
      _consolePrintf("SingleChannel\n");
      break;
    case MAD_MODE_DUAL_CHANNEL:
      _consolePrintf("DualChannel\n");
      break;
    case MAD_MODE_JOINT_STEREO:
      _consolePrintf("JointStereo\n");
      break;
    case MAD_MODE_STEREO:
      _consolePrintf("Normal LR Stereo\n");
      break;
    default:
      _consolePrintf("unknown ChannelMode.\n");
      FreeMP3();
      return(false);
      break;
  }
  
  _consolePrintf("BitRate=%dkbps\n",frame->header.bitrate/1000);
  _consolePrintf("SampleRate=%dHz\n",frame->header.samplerate);
  
  _consolePrintf("SamplesPerFrame=%d\n",SamplesPerFrame);
  
  DecBufL=(s16*)safemalloc(SamplesPerFrame*2*2);
  DecBufR=(s16*)safemalloc(SamplesPerFrame*2*2);
  DecBufRefresh=true;
  
  return(true);
}

u32 UpdateMP3(s16 *lbuf,s16 *rbuf)
{
  if(DecBufRefresh==true){
    DecBufRefresh=false;
    
    int result;
    
    outlbuf=DecBufL;
    outrbuf=DecBufR;
    outofs=0;
    
    result=madusr_ExecReturn_Next;
    
    while((result==madusr_ExecReturn_Next)&&(outofs<(SamplesPerFrame*2))){
      result=madusr_decode_exec(&StaticMadDecoder);
    }
    
    if(result!=madusr_ExecReturn_Next){
      if(result==madusr_ExecReturn_End){
//        _consolePrintf("ExecReturn_End\n");
        return(0);
      }
      if(result==madusr_ExecReturn_Fail){
//        _consolePrintf("ExecReturn_Fail\n");
        return(0);
      }
    }
    
    MemCopy16DMA3(&DecBufL[SamplesPerFrame*0],lbuf,SamplesPerFrame*2);
    MemCopy16DMA3(&DecBufR[SamplesPerFrame*0],rbuf,SamplesPerFrame*2);
    
    }else{
    DecBufRefresh=true;
    
    MemCopy16DMA3(&DecBufL[SamplesPerFrame*1],lbuf,SamplesPerFrame*2);
    MemCopy16DMA3(&DecBufR[SamplesPerFrame*1],rbuf,SamplesPerFrame*2);
  }
  
  return(SamplesPerFrame);
}

void FreeMP3(void)
{
  madusr_decode_free(&StaticMadDecoder);
  if(InputBuffer!=NULL){
    safefree(InputBuffer); InputBuffer=NULL;
  }
  if(DecBufL!=NULL){
    safefree(DecBufL); DecBufL=NULL;
  }
  if(DecBufR!=NULL){
    safefree(DecBufR); DecBufR=NULL;
  }
  DecBufRefresh=false;
}

s32 MP3_GetFileSize(void)
{
  return(FileSize);
}

s32 MP3_GetFileOffset(void)
{
  return(FileOffset);
}

void MP3_SetFileOffset(s32 ofs)
{
  if(FileSize<=ofs) ofs=FileSize-1;
  if(ofs<0) ofs=0;
  
  ofs&=~1; // 16bit align
  
  FileOffset=ofs;
  if(FileHandle!=0) fseek(FileHandle,FileOffset,SEEK_SET);
  
  struct mad_stream *stream=&StaticMadDecoder.sync->stream;
  
  size_t ReadSize,Remaining;
  u8 *ReadStart;
  
  ReadSize=INPUT_BUFFER_SIZE;
  ReadStart=InputBuffer;
  Remaining=0;
  
  if((FileOffset+ReadSize)>=FileSize){
    ReadSize=FileSize-FileOffset;
  }
  
  if(0<ReadSize){
    fread(ReadStart,1,ReadSize,FileHandle);
    FileOffset+=ReadSize;
    mad_stream_buffer(stream,InputBuffer,ReadSize+Remaining);
  }
}

u32 MP3_GetBitRate(void)
{
  struct mad_frame *frame;
  
  frame  = &StaticMadDecoder.sync->frame;
  
  return(frame->header.bitrate);
}

u32 MP3_GetChannelCount(void)
{
  struct mad_frame *frame;
  
  frame  = &StaticMadDecoder.sync->frame;
  
  switch(frame->header.mode){
    case MAD_MODE_SINGLE_CHANNEL:
      return(1);
      break;
    case MAD_MODE_DUAL_CHANNEL:
      return(2);
      break;
    case MAD_MODE_JOINT_STEREO:
      return(2);
      break;
    case MAD_MODE_STEREO:
      return(2);
      break;
    default:
      return(1);
      break;
  }
}

u32 MP3_GetSampleRate(void)
{
  struct mad_frame *frame;
  
  frame  = &StaticMadDecoder.sync->frame;
  
  return(frame->header.samplerate);
}

u32 MP3_GetSamplePerFrame(void)
{
  return(SamplesPerFrame);
}

int MP3_GetInfoIndexCount(void)
{
  if(ID3Tag.Enabled==false){
    return(1);
    }else{
    return(5);
  }
}

bool MP3_GetInfoStrL(int idx,char *str,int len)
{
  if(ID3Tag.Enabled==false){
    if(idx==0){
      snprintf(str,len,"ID3TAGv1 not found.");
      return(true);
    }
    return(false);
  }
  
  switch(idx){
    case 0: snprintf(str,len,"Title=%s",ID3Tag.title); return(true); break;
    case 1: snprintf(str,len,"Artist=%s",ID3Tag.artist); return(true); break;
    case 2: snprintf(str,len,"Album=%s",ID3Tag.album); return(true); break;
    case 3: snprintf(str,len,"Comment=%s",ID3Tag.comment); return(true); break;
    case 4: snprintf(str,len,"%s %s",ID3Tag.year,GetGenreStr(ID3Tag.genre)); return(true); break;
  }
  return(false);
}

bool MP3_GetInfoStrW(int idx,UnicodeChar *str,int len)
{
  return(false);
}

bool MP3_GetInfoStrUTF8(int idx,char *str,int len)
{
  return(false);
}

