/*---------------------------------------------------------------------------------
	$Id: interrupts.c,v 1.10 2007/02/06 12:00:32 wntrmute Exp $

	Copyright (C) 2005
		Dave Murphy (WinterMute)

	This software is provided 'as-is', without any express or implied
	warranty.  In no event will the authors be held liable for any
	damages arising from the use of this software.

	Permission is granted to anyone to use this software for any
	purpose, including commercial applications, and to alter it and
	redistribute it freely, subject to the following restrictions:

	1.	The origin of this software must not be misrepresented; you
		must not claim that you wrote the original software. If you use
		this software in a product, an acknowledgment in the product
		documentation would be appreciated but is not required.
	2.	Altered source versions must be plainly marked as such, and
		must not be misrepresented as being the original software.
	3.	This notice may not be removed or altered from any source
		distribution.

	$Log: interrupts.c,v $
	Revision 1.10  2007/02/06 12:00:32  wntrmute
	use uint32 for irqEnable
	
	Revision 1.9  2006/06/16 07:35:14  wntrmute
	more moves to REG_ values
	implement SetYtrigger helper function
	set bits in REG_DISPSTAT for IRQ_VCOUNT
	
	Revision 1.8  2006/02/21 00:12:35  wntrmute
	clear interrupts and enable REG_IME in irqInit
	
	Revision 1.7  2005/12/12 13:03:29  wntrmute
	set/clear LCD interrupts in irqEnable/Disable
	
	Revision 1.6  2005/11/07 01:11:08  wntrmute
	more doxygen
	moved IntrMain declaration to C file
	
	Revision 1.5  2005/09/20 05:00:24  wntrmute
	use new register definitions
	
	Revision 1.4  2005/09/04 16:28:04  wntrmute
	place irqTable in ewram

	Revision 1.3  2005/09/03 17:09:35  wntrmute
	added interworking aware interrupt dispatcher


---------------------------------------------------------------------------------*/

#include <nds/interrupts.h>
#include <nds/system.h>

extern "C" {
extern void IntrMain();	// Prototype for assembly interrupt dispatcher
}

//---------------------------------------------------------------------------------
void irqDummy(void) {}
//---------------------------------------------------------------------------------


#ifdef ARM9
#define INT_TABLE_SECTION __attribute__((section(".dtcm")))
#else
#define INT_TABLE_SECTION
#endif

extern "C" {
extern struct IntTable irqTable[];
}
__attribute__ ((section (".dtcm"))) struct IntTable irqTable[MAX_INTERRUPTS] INT_TABLE_SECTION;

//---------------------------------------------------------------------------------
void irqSet(int mask, IntFn handler) {
//---------------------------------------------------------------------------------
	if (!mask) return;

	int i;

	for	(i=0;i<MAX_INTERRUPTS;i++)
		if	(!irqTable[i].mask || irqTable[i].mask == mask) break;

	if ( i == MAX_INTERRUPTS ) return;

	irqTable[i].handler	= handler;
	irqTable[i].mask	= mask;

	if(mask & IRQ_VBLANK)
		REG_DISPSTAT |= DISP_VBLANK_IRQ ;
	if(mask & IRQ_HBLANK)
		REG_DISPSTAT |= DISP_HBLANK_IRQ ;

	REG_IE |= mask;
}

void irqSet_u32(int mask, u32 handler) {
  irqSet(mask,(VoidFunctionPointer)handler);
}

//---------------------------------------------------------------------------------
void irqInit() {
//---------------------------------------------------------------------------------
	int i;

	// Set all interrupts to dummy functions.
	for(i = 0; i < MAX_INTERRUPTS; i ++)
	{
		irqTable[i].handler = irqDummy;
		irqTable[i].mask = 0;
	}  

	IRQ_HANDLER = IntrMain;

	REG_IE	= 0;			// disable all interrupts
	REG_IF	= IRQ_ALL;		// clear all pending interrupts
//	REG_IME = 1;			// enable global interrupt

}


//---------------------------------------------------------------------------------
void irqClear(int mask) {
//---------------------------------------------------------------------------------
	int i = 0;

	for	(i=0;i<MAX_INTERRUPTS;i++)
		if	(irqTable[i].mask == mask) break;

	if ( i == MAX_INTERRUPTS ) return;

	irqTable[i].handler	= irqDummy;

	if (mask & IRQ_VBLANK)
		REG_DISPSTAT &= ~DISP_VBLANK_IRQ ;
	if (mask & IRQ_HBLANK)
		REG_DISPSTAT &= ~DISP_HBLANK_IRQ ;
	if (mask & IRQ_VCOUNT)
		REG_DISPSTAT &= ~DISP_YTRIGGER_IRQ;

	REG_IE &= ~mask;
}


//---------------------------------------------------------------------------------
void irqInitHandler(IntFn handler) {
//---------------------------------------------------------------------------------
	REG_IME = 0;
	REG_IF = ~0;
	REG_IE = 0;

	IRQ_HANDLER = handler;

//	REG_IME = 1;
}

//---------------------------------------------------------------------------------
void irqEnable(uint32 irq) {
//---------------------------------------------------------------------------------
	if (irq & IRQ_VBLANK)
		REG_DISPSTAT |= DISP_VBLANK_IRQ ;
	if (irq & IRQ_HBLANK)
		REG_DISPSTAT |= DISP_HBLANK_IRQ ;
	if (irq & IRQ_VCOUNT)
		REG_DISPSTAT |= DISP_YTRIGGER_IRQ;

	REG_IE |= irq;
//	REG_IME = 1;
}

//---------------------------------------------------------------------------------
void irqDisable(uint32 irq) {
//---------------------------------------------------------------------------------
	if (irq & IRQ_VBLANK)
		REG_DISPSTAT &= ~DISP_VBLANK_IRQ ;
	if (irq & IRQ_HBLANK)
		REG_DISPSTAT &= ~DISP_HBLANK_IRQ ;
	if (irq & IRQ_VCOUNT)
		REG_DISPSTAT &= ~DISP_YTRIGGER_IRQ;

	REG_IE &= ~irq;
}

