
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <NDS.h>

#include "_console.h"
#include "_consoleWriteLog.h"
#include "_const.h"
#include "memtool.h"
#include "shell.h"
#include "inifile.h"
#include "arm9tcm.h"

#include "sndeff.h"

#include "procstate.h"

#include "ipc6.h"
#include "strpcm.h"

#include "sndeff_dfs.h"

typedef struct {
  u32 ofs,smpcnt;
  u16 chs,smprate;
} TWaveHeader;

typedef struct {
  EWAVFN WAVFN;
  u32 FileOffset,Channels,SampleRate,SamplesCount;
  bool AlreadyLoaded;
  bool Loaded;
  u8 *lbuf,*rbuf;
} TSound;

#define SoundsCount (WAVFN_Count)
static TSound Sounds[SoundsCount];

static const s32 StreamPCM_BufSize=1024;

typedef struct {
  u32 DataOffset;
  u32 Freq;
  u32 BufCount;
  u32 Channels;
  u32 CurrentPos;
} TStreamPCM;
static TStreamPCM StreamPCM;

static void StreamPCM_Init(void)
{
  TStreamPCM *psp=&StreamPCM;
  psp->DataOffset=0;
  psp->Freq=0;
  psp->BufCount=0;
  psp->Channels=0;
  psp->CurrentPos=0;
}

static void StreamPCM_Free(void)
{
  strpcmStop();
  StreamPCM_Init();
}

static void StreamPCM_Start(void)
{
  TStreamPCM *psp=&StreamPCM;
  strpcmStart(true,psp->Freq,StreamPCM_BufSize,psp->Channels,SPF_PCMx1);
}

static void Sound_Load(EWAVFN WAVFN)
{
  TSound *psnd=&Sounds[WAVFN];
  if(psnd->Loaded==true) return;
  psnd->Loaded=true;
  
  if(VerboseDebugLog==true) _consolePrintf("SE: ID%d, %dHz, %dsmps, %dchs. ofs=%dbyte.\n",psnd->WAVFN,psnd->SampleRate,psnd->SamplesCount,psnd->Channels,psnd->FileOffset);
  if((psnd->SamplesCount&3)!=0) StopFatalError(0,"sndeff.dat file size alignment error. %dsmps.",psnd->SamplesCount);
  
  SndEff_DFS_SetOffset(psnd->FileOffset);
  
  psnd->lbuf=(u8*)safemalloc32_chkmem(&MM_System,psnd->SamplesCount);
  SndEff_DFS_Read32bit(psnd->lbuf,psnd->SamplesCount);
  DCache_CleanRangeOverrun(psnd->lbuf,psnd->SamplesCount);
  
  if(psnd->Channels==2){
    psnd->rbuf=(u8*)safemalloc32_chkmem(&MM_System,psnd->SamplesCount);
    SndEff_DFS_Read32bit(psnd->rbuf,psnd->SamplesCount);
    DCache_CleanRangeOverrun(psnd->rbuf,psnd->SamplesCount);
  }
}

static void Sound_Free(EWAVFN WAVFN)
{
  TSound *psnd=&Sounds[WAVFN];
  
  if(psnd->Loaded==false) return;
  
  const u8 *lbuf=psnd->lbuf,*rbuf=psnd->rbuf;
  
  FIFO_TX32(FIFOCMD_ToARM7_StopSoundBlock);
  FIFO_TX32((u32)lbuf);
  FIFO_TX32((u32)rbuf);
  
  psnd->Loaded=false;
  
  if(psnd->lbuf!=NULL){
    safefree(&MM_System,psnd->lbuf); psnd->lbuf=NULL;
  }
  if(psnd->rbuf!=NULL){
    safefree(&MM_System,psnd->rbuf); psnd->rbuf=NULL;
  }
}

static void Sound_Open_ins_SetAlreadyLoaded(EWAVFN WAVFN)
{
  TSound *psnd=&Sounds[WAVFN];
  psnd->AlreadyLoaded=true;
  
  Sound_Load(WAVFN);
}

void Sound_Open(void)
{
  StreamPCM_Init();
  
  {
    FAT_FILE *pf=Shell_FAT_fopen_Internal_WithCheckExists(SndEffDatFilename);
    SndEff_DFS_Init(pf);
    FAT2_fclose(pf);
  }
    
  u32 cnt;
  SndEff_DFS_SetOffset(0);
  SndEff_DFS_Read32bit(&cnt,4);
  if(SoundsCount!=cnt) StopFatalError(0,"SndEff.dat types error. %d!=%d",SoundsCount,cnt);
  
  for(u32 idx=0;idx<SoundsCount;idx++){
    TSound *psnd=&Sounds[idx];
    TWaveHeader WaveHeader;
    SndEff_DFS_Read32bit(&WaveHeader,sizeof(TWaveHeader));
    psnd->WAVFN=(EWAVFN)idx;
    psnd->FileOffset=WaveHeader.ofs;
    psnd->Channels=WaveHeader.chs;
    psnd->SampleRate=WaveHeader.smprate;
    psnd->SamplesCount=WaveHeader.smpcnt;
    psnd->AlreadyLoaded=false;
    psnd->Loaded=false;
    psnd->lbuf=NULL;
    psnd->rbuf=NULL;
  }
  
  Sound_Open_ins_SetAlreadyLoaded(WAVFN_ClickButton);
  Sound_Open_ins_SetAlreadyLoaded(WAVFN_AutoFlag);
  Sound_Open_ins_SetAlreadyLoaded(WAVFN_AutoOpen);
  Sound_Open_ins_SetAlreadyLoaded(WAVFN_Change);
  Sound_Open_ins_SetAlreadyLoaded(WAVFN_Open);
  Sound_Open_ins_SetAlreadyLoaded(WAVFN_Zoom);
  Sound_Open_ins_SetAlreadyLoaded(WAVFN_GiveUp);
  Sound_Open_ins_SetAlreadyLoaded(WAVFN_GameStart);
  
  if(VerboseDebugLog==true) _consolePrintf("Wave header loaded. (%d,%d)\n",sizeof(TWaveHeader),cnt);
}

void Sound_Close(void)
{
  for(u32 idx=0;idx<SoundsCount;idx++){
    Sound_Free((EWAVFN)idx);
  }
  
  StreamPCM_Free();
  
  SndEff_DFS_Free();
}

static void PlaySoundBlock(TSound *psnd,u32 Volume,bool isLoop)
{
  const u8 *lbuf=psnd->lbuf,*rbuf=psnd->rbuf;
  
  if(127<Volume) Volume=127;
  
  FIFO_TX32(FIFOCMD_ToARM7_PlaySoundBlock);
  FIFO_TX32(isLoop);
  FIFO_TX32(psnd->SampleRate);
  FIFO_TX32((u32)lbuf);
  FIFO_TX32((u32)rbuf);
  FIFO_TX32(psnd->SamplesCount);
  FIFO_TX32(Volume);
  FIFO_TX32(1);
}

void SoundSE_Start(EWAVFN WAVFN)
{
  if(ProcState.System.ClickSound==false) return;
  
  u32 Volume=GetSEVolume128();
  if(Volume==0) return;
  
  if(Volume!=0) Volume+=32;
  
  if(WAVFN==WAVFN_License) Volume*=0.5;
  if(WAVFN==WAVFN_Setting) Volume*=0.5;
  if(WAVFN==WAVFN_ClickButton) Volume*=0.5;
  if(WAVFN==WAVFN_GiveUp) Volume*=0.75;
  if(WAVFN==WAVFN_Change) Volume*=0.5;
  if(WAVFN==WAVFN_Clear1) Volume/=3;
  if(WAVFN==WAVFN_Clear2) Volume/=3;
  if(WAVFN==WAVFN_AutoFlag) Volume/=4;
  if(WAVFN==WAVFN_AutoOpen) Volume/=16;
  if(WAVFN==WAVFN_GameStart) Volume/=2;
  if(WAVFN==WAVFN_HighScore) Volume/=2;
  if(WAVFN==WAVFN_PanelOpen) Volume/=4;
  if(WAVFN==WAVFN_PanelClose) Volume/=4;
  
  Sound_Load(WAVFN);
  
  PlaySoundBlock(&Sounds[WAVFN],Volume,false);
}

void SoundSE_StartVol(EWAVFN WAVFN,u32 _Volume)
{
  if(ProcState.System.ClickSound==false) return;
  
  u32 Volume=GetSEVolume128();
  if(Volume==0) return;
  
  if(Volume!=0) Volume+=32;
  
  Volume=(Volume*_Volume)/128;
  
  Sound_Load(WAVFN);
  
  PlaySoundBlock(&Sounds[WAVFN],Volume,false);
}

void SoundSE_Stop(void)
{
  for(u32 idx=0;idx<SoundsCount;idx++){
    TSound *psnd=&Sounds[idx];
    if(psnd->AlreadyLoaded==false){
      Sound_Free((EWAVFN)idx);
    }
  }
}

void SoundBGM_Start(EWAVFN WAVFN)
{
  strpcm_ExclusivePause=false;
  
  u32 Volume64=GetBGMVolume128()/2;
  IPC6->strpcmAudioVolume64=Volume64/3;
  
  if(ProcState.System.ClickSound==false) return;
  
  StreamPCM_Free();
  
  TStreamPCM *psp=&StreamPCM;
  
  {
    TSound *psnd=&Sounds[WAVFN];
    if(VerboseDebugLog==true) _consolePrintf("BGM: ID%d, %dHz, %dsmps, %dchs. ofs=%dbyte.\n",psnd->WAVFN,psnd->SampleRate,psnd->SamplesCount,psnd->Channels,psnd->FileOffset);
    if((psnd->SamplesCount&3)!=0) StopFatalError(0,"sndeff.dat file size alignment error. %dsmps.",psnd->SamplesCount);
    
    psp->DataOffset=psnd->FileOffset;
    psp->Freq=psnd->SampleRate;
    psp->Channels=psnd->Channels;
    psp->BufCount=psnd->SamplesCount;
    psp->CurrentPos=0;
  }
  
  StreamPCM_Start();
  
  while(SoundSE_MainVBlankHandler()==false){
  }
}

void SoundBGM_Stop(void)
{
  StreamPCM_Free();
}

void SoundSE_IRQVBlankHandler(void)
{
}

bool SoundSE_MainVBlankHandler(void)
{
  TStreamPCM *psp=&StreamPCM;
  if(psp->DataOffset==0) return(true);
  
  if(strpcmRingBufWriteIndex==strpcmRingBufReadIndex) return(true);
  
  const u32 BufSize=StreamPCM_BufSize;
  
  u32 *pLRBuf=&strpcmRingLRBuf[BufSize*strpcmRingBufWriteIndex];
  
  if(IPC6->PanelOpened==false){
    for(u32 idx=0;idx<BufSize;idx++){
      pLRBuf[idx]=0;
    }
    }else{
    u32 CurrentPos=psp->CurrentPos;
    const u32 DataOffset=psp->DataOffset;
    const u32 BufCount=psp->BufCount;
    
    u8 lbuf[BufSize],rbuf[BufSize];
    
    u32 Remain=BufCount-CurrentPos;
    if(BufSize<Remain) Remain=BufSize;
    
    {
      SndEff_DFS_SetOffset(DataOffset+CurrentPos);
      SndEff_DFS_Read32bit(lbuf,Remain);
      SndEff_DFS_SetOffset(DataOffset+BufCount+CurrentPos);
      SndEff_DFS_Read32bit(rbuf,Remain);
    }
    
    if(BufSize==Remain){
      u8 *plbuf=lbuf,*prbuf=rbuf;
      for(u32 idx=0;idx<Remain;idx++){
        u32 smp32=0;
        smp32|=(u32)*plbuf++<<8;
        smp32|=(u32)*prbuf++<<24;
        *pLRBuf++=smp32;
      }
      CurrentPos+=Remain;
      if(CurrentPos==BufCount) CurrentPos=0;
      }else{
      u8 *plbuf=lbuf,*prbuf=rbuf;
      for(u32 idx=0;idx<Remain;idx++){
        u32 smp32=0;
        smp32|=(u32)*plbuf++<<8;
        smp32|=(u32)*prbuf++<<24;
        *pLRBuf++=smp32;
      }
      CurrentPos=0;
      Remain=BufSize-Remain;
      SndEff_DFS_SetOffset(DataOffset+CurrentPos);
      SndEff_DFS_Read32bit(lbuf,Remain);
      SndEff_DFS_SetOffset(DataOffset+BufCount+CurrentPos);
      SndEff_DFS_Read32bit(rbuf,Remain);
      plbuf=lbuf,prbuf=rbuf;
      for(u32 idx=0;idx<Remain;idx++){
        u32 smp32=0;
        smp32|=(u32)*plbuf++<<8;
        smp32|=(u32)*prbuf++<<24;
        *pLRBuf++=smp32;
      }
      CurrentPos+=Remain;
      if(CurrentPos==BufCount) CurrentPos=0;
    }
    
    psp->CurrentPos=CurrentPos;
  }
  
  strpcmRingBufWriteIndex++;
  if(strpcmRingBufWriteIndex==strpcmRingBufCount) strpcmRingBufWriteIndex=0;
  
//  _consolePrintf("R%d,W%d\n",strpcmRingBufReadIndex,strpcmRingBufWriteIndex);
  
  return(false);
}

