//AGB_Rogue was adapted from IBM(TM) PC Rogue version 1.48
//by Donnie Russell, Copyright (C) 2007



#include <string.h> //strlen

#include "gba.h"
#include "inc.h"



//brightness level of text colors; 0-256
#define BRIGHTNESS_LEVEL  128



//...........................................................................
//these are big and should go in ewram
VAR_IN_EWRAM unsigned short savewin[80*25];
VAR_IN_EWRAM unsigned short textwin[80*25];
VAR_IN_EWRAM unsigned char _flags[80*(25-3)];
VAR_IN_EWRAM unsigned char _level[80*(25-3)];



extern unsigned long *VideoFrame; //from fast.c
//...........................................................................



/////////////////////////////////////////////////////////////////////////////
//from fast.c
CODE_IN_IWRAM void IntHandler(void);
CODE_IN_IWRAM void SelectVideoFrame(int frame);
CODE_IN_IWRAM void Clear(void);
CODE_IN_IWRAM void SaveRestoreScreen(int mode);
CODE_IN_IWRAM void PutChar_4x6(int ch, int attr, int x, int y);
CODE_IN_IWRAM void PutChar_6x8(int ch, int attr, int x, int y);
CODE_IN_IWRAM void DrawTile(int x, int y, int tile);
CODE_IN_IWRAM void DrawBlankTile(int x, int y);
CODE_IN_IWRAM void CopyEllipse(int x, int y, int rx, int ry, int step);
CODE_IN_IWRAM void DrawMinimapRoutine(unsigned short *p, int hx, int hy);



void get_hero_position(int *x, int *y); //from rogue.c
unsigned int rangeMT(unsigned int lower, unsigned int upper); //from rogue.c



void Refresh(void);
/////////////////////////////////////////////////////////////////////////////



//***************************************************************************
void WaitForVBlank(void)
{
  while (REG_VCOUNT!=160) ;
  while (REG_VCOUNT==160) ;
}
//***************************************************************************



//***************************************************************************
void StopSound(void)
{
  int i, j;

  REG_TM0CNT = 0; //timer 0 off
  REG_TM1CNT = 0; //timer 1 off
  REG_DMA1CNT = 0; //dma 1 off

  i = 0; for (j=0; j<100; j++) i++; //wait a little bit
}



void PlaySound(const signed char *pData, unsigned int length, unsigned int freq)
{
  StopSound();

  REG_SOUNDCNT_H |= DSA_FIFO_RESET; //reset fifo a

  REG_TM0D = 65536-((1<<24)/freq);
  REG_TM0CNT = TIME_ENABLE;

  REG_TM1D = 65536-length;
  REG_TM1CNT = TIME_OVERFLOW|TIME_IRQ_ENABLE|TIME_ENABLE;

  REG_DMA1SAD = (u32)pData;
  REG_DMA1DAD = (u32)REG_FIFO_A;
  REG_DMA1CNT = DST_FIXED|SRC_INCREMENT|DMA_REPEAT|TRANSFER_32_BITS|START_ON_FIFO_EMPTY|DMA_ENABLE;
}



void play_sound(int n)
{
  int i;

  PlaySoundByNumber(n);

  switch (n)
  {
    case HIT1_SOUND:
    case HIT2_SOUND:
    case MISS1_SOUND:
    case MISS2_SOUND:
      for (i=0; i<4; i++) WaitForVBlank();
    break;
  }
}
//***************************************************************************



//***************************************************************************
#define CM_X  9
#define CM_Y  4



char *cm_string1[11] =
{
  "Game", "Help", "Level", "Object", "Weapon", "Armor", "Food", "Potion", "Scroll", "Ring", "Wand"
};

char *cm_string2[28] =
{
  "Use Macro", "Define Macro", "Tiles On/Off", "Version", "Save & Halt", "Quit",
  "Last Message", "Discoveries", "Identify Trap", "Commands", "Symbols",
  "Descend", "Ascend", "Rest", "Search",
  "Inventory", "Drop", "Throw", "Call",
  "Wield",
  "Take Off", "Wear",
  "Eat",
  "Quaff",
  "Read",
  "Remove", "Put On",
  "Zap"
};

char *cm_char2 = "GF!vSQoD^?/><.sidtcwTWeqrRPz";

const int cm_ind[11] = {0, 6, 11, 15, 19, 20, 22, 23, 24, 25, 27};

const int cm_num[11] = {6, 5, 4, 4, 1, 2, 1, 1, 1, 2, 1};



int command_menu_flag = 0;



void PrintMenuString(char *p, int attr, int x, int y)
{
  while (*p) PutChar_6x8(*p++, attr, x++, y);
}



int command_menu(void)
{
  int i, j, ch;
  char right_arrow_string[2] = {16, 0};

  static int depth = 0, pos1 = 0, pos2 = 0;

  command_menu_flag = 1;

                               for (i=CM_X-1; i<=CM_X+22; i++) PutChar_6x8(i==CM_X-1?218:(i==CM_X+22?191:196), YELLOW, i, CM_Y-1 );
  for (j=CM_Y; j<CM_Y+11; j++) for (i=CM_X-1; i<=CM_X+22; i++) PutChar_6x8(i==CM_X-1?179:(i==CM_X+22?179: 32), YELLOW, i, j      );
                               for (i=CM_X-1; i<=CM_X+22; i++) PutChar_6x8(i==CM_X-1?192:(i==CM_X+22?217:196), YELLOW, i, CM_Y+11);

  while (1)
  {
    for (i=0; i<11; i++) PrintMenuString(cm_string1[i], i==pos1?YELLOW:GREY, CM_X+1, CM_Y+i);

    PrintMenuString(right_arrow_string, LRED, CM_X+(depth==0?0:1+6+1), CM_Y+(depth==0?pos1:pos1+pos2));
    for (i=0; i<cm_num[pos1]; i++) PrintMenuString(cm_string2[cm_ind[pos1]+i], depth==0?GREY:(i==pos2?YELLOW:GREY), CM_X+1+6+1+1, CM_Y+pos1+i);

    ch = get_input();

    PrintMenuString(" ", LRED, CM_X+(depth==0?0:1+6+1), CM_Y+(depth==0?pos1:pos1+pos2));
    for (i=0; i<cm_num[pos1]; i++) PrintMenuString("             ", GREY, CM_X+1+6+1+1, CM_Y+pos1+i);



    if (ch==27) break; //escape

    if (ch=='\b' || ch==' ' || ch==13)
    {
      if (depth==0) ch = 27; else ch = cm_char2[cm_ind[pos1]+pos2];
      break;
    }

    //up down left right
    if (ch=='k') {if (depth==0) {if (pos1>0   ) pos1--;} else {if (pos2>0             ) pos2--; else if (pos1>0   ) {pos1--; pos2 = cm_num[pos1]-1;}}}
    if (ch=='j') {if (depth==0) {if (pos1<11-1) pos1++;} else {if (pos2<cm_num[pos1]-1) pos2++; else if (pos1<11-1) {pos1++; pos2 = 0             ;}}}
    if (ch=='h' && depth==1) depth = 0;
    if (ch=='l' && depth==0) {depth = 1; pos2 = 0;}
  }

  command_menu_flag = 0;

  Refresh();

  return ch;
}
//***************************************************************************



//***************************************************************************
int display_credits_flag = 0;



//x,TITLE_DATA_W,CLIENTW must be multiple of 4
void DrawTitle(void)
{
  unsigned long *s, *d;
  int x, y, i, j;

  x = (CLIENTW-TITLE_DATA_W)>>1;
  y = 6;

  s = (unsigned long *)TitleData+(TITLE_DATA_W>>2)*(TITLE_DATA_H-1);
  d = (unsigned long *)VideoFrame+(CLIENTW>>2)*y+(x>>2);

  j = TITLE_DATA_H; while (j--)
  {
    i = TITLE_DATA_W>>2; while (i--) *d++ = *s++;

    s += (-TITLE_DATA_W-TITLE_DATA_W)>>2;
    d += (CLIENTW-TITLE_DATA_W)>>2;
  }
}



void CenterLargeText(char *p, int attr, int y)
{
  int x;

  x = (40-strlen(p))>>1;

  while (*p) PutChar_6x8(*p++, attr, x++, y);
}



void display_credits(void)
{
  DrawTitle();

  CenterLargeText("Rogue Copyright (C) 1981 by", LMAGENTA, 7);
  CenterLargeText("Michael Toy, Ken Arnold,", WHITE, 8);
  CenterLargeText("and Glenn Wichman", WHITE, 9);

  CenterLargeText("IBM(TM) PC Rogue Copyright (C) 1983 by", LMAGENTA, 11);
  CenterLargeText("Mel Sibony and Jon Lane", WHITE, 12);

  CenterLargeText("AGB_Rogue Copyright (C) 2007 by", LMAGENTA, 14);
  CenterLargeText("Donnie Russell", WHITE, 15);
}
//***************************************************************************



//***************************************************************************
void DrawMinimap(void)
{
  int hx, hy;

  get_hero_position(&hx, &hy);
  hx -= 15/2; if (hx<0) hx = 0; if (hx>80-15) hx = 80-15;
  hy -=  8/2; if (hy<0) hy = 0; if (hy>22- 8) hy = 22- 8;

  DrawMinimapRoutine(textwin+80*1, hx, hy);
}
//***************************************************************************



//***************************************************************************
#define COLOR16(r,g,b)  (((r)>>3)+(((g)>>3)<<5)+(((b)>>3)<<10))

#define COLOR16_R(color)  (( (color)     &31)<<3)
#define COLOR16_G(color)  ((((color)>> 5)&31)<<3)
#define COLOR16_B(color)  ((((color)>>10)&31)<<3)

//color1 shifted to color2 by value (0-256)
#define SHIFT_COLOR16(color1,color2,value)  COLOR16( \
        COLOR16_R(color1)+(((COLOR16_R(color2)-COLOR16_R(color1))*(value))>>8), \
        COLOR16_G(color1)+(((COLOR16_G(color2)-COLOR16_G(color1))*(value))>>8), \
        COLOR16_B(color1)+(((COLOR16_B(color2)-COLOR16_B(color1))*(value))>>8))



const struct {int r, g, b;} rgb_of_attr[] = { {  0,   0,   0}, {  0,   0, 128}, {  0, 128,   0}, {  0, 128, 128}, {128,   0,   0}, {128,   0, 128}, {128,  64,   0}, {192, 192, 192}, {128, 128, 128}, {  0,   0, 255}, {  0, 255,   0}, {  0, 255, 255}, {255,   0,   0}, {255,   0, 255}, {255, 255,   0}, {255, 255, 255} };



//0 is always black
unsigned long GetColor(int i)
{
  if (i==0) return COLOR16(0, 0, 0);
  //shift color to white by brightness level
  return SHIFT_COLOR16(COLOR16(rgb_of_attr[i].r, rgb_of_attr[i].g, rgb_of_attr[i].b), COLOR16(255, 255, 255), BRIGHTNESS_LEVEL);
}
//***************************************************************************



//***************************************************************************
void PlotTile(int ch, int a, int x, int y)
{
  int t = -1;



  y--;
  if (x<0 || x>=15 || y<0 || y>=8) return;
  x <<= 4;
  y <<= 4;
  y += 6+2;



  if (ch>='A' && ch<='Z') t = 0+(ch-'A'); //monsters
  else switch (ch)
  {
    case 32: break;

    case 0x01: t = 26; break; //player
    case 0xc9: t = 27; break; //ulwall
    case 0xbb: t = 28; break; //urwall
    case 0xc8: t = 29; break; //llwall
    case 0xbc: t = 30; break; //lrwall
    case 0xcd: t = 31; break; //hwall
    case 0xba: t = 32; break; //vwall
    case 0xfa: t = 33; break; //floor
    case 0xb1: t = 34; break; //passage
    case 0xce: t = 35; break; //door
    case 0xf0: t = 36; break; //stairs
    case 0x04: t = 37; break; //trap
    case 0x0c: t = 38; break; //amulet
    case 0x08: t = 39; break; //armor
    case 0x05: t = 40; break; //food
    case 0x0f: t = 41; break; //gold
    case 0xad: t = 42; break; //potion
    case 0x09: t = 43; break; //ring
    case 0x0d: t = 44; break; //scroll
    case 0xe7: t = 45; break; //stick
    case 0x18: t = 46; break; //weapon
    case  '$': t = 47; break; //magic
    case  '+': t = 48; break; //bmagic
    case  '*': t = 48; break; //missile (same tile as bmagic)

    case '\\': t = 49; if (a==RED) t += 4; break; //bolt
    case  '/': t = 50; if (a==RED) t += 4; break; //bolt
    case  '-': t = 51; if (a==RED) t += 4; break; //bolt
    case  '|': t = 52; if (a==RED) t += 4; break; //bolt
  }

  if (t>=0) DrawTile(x, y, t); else DrawBlankTile(x, y);
}
//***************************************************************************



//***************************************************************************
#define SELECTOR_X           15
#define SELECTOR_Y           5
#define SELECTOR_W           10
#define SELECTOR_H           10
#define SELECTOR_DESC_X      15
#define SELECTOR_DESC_Y      4
#define SELECTOR_NUM         95

#define SELECTOR_TIMEOUT     (60*10) //10 seconds (approx.)

#define INPUT_DELAY          20
#define INPUT_REPEAT         2



static int TileType = 1; //0: text, 1: tiles
static int TileMode = 0;
static int TileOffsetX = 0;
static int TileOffsetY = 0;

static int SelectPos = 0;
static int RepeatSelection = 0;
static int DefinableKey = 's';



const char CommandDesc[10*95] =
"  Space   "
"TilesOnOff"
"          "
"          "
"          "
"          "
"          "
"          "
"          "
"          "
"          "
"  Throw   "
"          "
" Zap Wand "
"   Rest   "
" Symbols  "
"          "
"          "
"          "
"          "
"          "
"          "
"          "
"          "
"          "
"          "
"          "
"          "
"  Ascend  "
"          "
" Descend  "
" Commands "
"          "
"          "
"  Run SW  "
"          "
" Discover "
"          "
"Def Macro "
"Use Macro "
" Run West "
"          "
"Run South "
"Run North "
" Run East "
"          "
"  Run SE  "
"          "
"PutOn Ring"
"   Quit   "
"RemoveRing"
"Save&Halt "
"TakeOffArm"
"  Run NE  "
"          "
"Wear Armor"
"          "
"  Run NW  "
"          "
"          "
"          "
"          "
"Ident Trap"
"          "
"          "
"  Repeat  "
"  Go SW   "
"   Call   "
"   Drop   "
" Eat Food "
"   Find   "
"          "
" Go West  "
"Inventory "
" Go South "
" Go North "
" Go East  "
"          "
"  Go SE   "
" Last Msg "
"          "
"QuaffPot'n"
"ReadScroll"
"  Search  "
"  Throw   "
"  Go NE   "
" Version  "
"Wield Weap"
"          "
"  Go NW   "
" Zap Wand "
"          "
"          "
"          "
"          ";



void Refresh(void)
{
  unsigned short *p;
  int x, y, c;



  if (command_menu_flag) return;



  if (TileMode==0)
  {
    p = textwin;

    if (display_credits_flag && TileType)
    {
      for (y=0; y<25; y++) {for (x=0; x<60; x++) {c = *p++; if ((c&255)!=32 || y>=23) PutChar_4x6(c&255, c>>8, x, y);} p += 80-60;}

      display_credits();
    }
    else
    {
      for (y=0; y<25; y++) {for (x=0; x<60; x++) {c = *p++; PutChar_4x6(c&255, c>>8, x, y);} p += 80-60;}
    }

    return;
  }



  p = textwin; for (y=0; y<1; y++) {for (x=0; x<60; x++) {c = *p++; PutChar_4x6(c&255, c>>8, x, y);} p += 80-60;}

  p = textwin+80*(1+TileOffsetY)+TileOffsetX;
  if (TileType==0)
  {
    for (y=1; y<23; y++) {for (x=0; x<60; x++) {c = *p++; PutChar_4x6(c&255, c>>8, x, y);} p += 80-60;}
  }
  else
  {
    for (y=1; y<9; y++) {for (x=0; x<15; x++) {c = *p++; PlotTile(c&255, c>>8, x, y);} p += 80-15;}

    DrawMinimap();
  }

  p = textwin+80*23; for (y=23; y<25; y++) {for (x=0; x<40; x++) {c = *p++; PutChar_4x6(c&255, c>>8, x, y);} p += 80-40;}
}



//returns 1 if refresh needed
int UpdateTileOffset(void)
{
  int hx, hy, x, y;

  get_hero_position(&hx, &hy);

  x = TileOffsetX;
  y = TileOffsetY;

  if (TileType==0)
  {
         if (hx<80*1/3) TileOffsetX = 0;
    else if (hx<80*2/3) TileOffsetX = (80-60)/2;
    else                TileOffsetX = (80-60)/1;

    TileOffsetY = 0;

    if (x!=TileOffsetX || y!=TileOffsetY) return 1;
    return 0;
  }
  else
  {
    TileOffsetX = hx-15/2; if (TileOffsetX<0) TileOffsetX = 0; if (TileOffsetX>80-15) TileOffsetX = 80-15;
    TileOffsetY = hy- 8/2; if (TileOffsetY<0) TileOffsetY = 0; if (TileOffsetY>22- 8) TileOffsetY = 22- 8;

    return 1;
  }
}



//if specified button is pressed, wait for its release
int PressRelease(int f)
{
  if ((REG_P1&f)==0) //pressed
  {
    while ((REG_P1&f)==0) ; //wait for release
    return 1;
  }
  return 0;
}



void DrawSelectChar(int i)
{
  int c;

  if (i==SelectPos) c = WHITE|BLACK_ON_BLUE; else c = BLACK|BLACK_ON_GREY;
  PutChar_6x8(32+i, c, SELECTOR_X+i%SELECTOR_W, SELECTOR_Y+i/SELECTOR_W);

  if (i==SelectPos)
  {
    const char *p;
    int x;

    p = CommandDesc+10*i;
    for (x=0; x<10; x++) PutChar_6x8(*p++, WHITE|BLACK_ON_GREEN, SELECTOR_DESC_X+x, SELECTOR_DESC_Y);
  }
}



int SelectChar(void)
{
  int timeout, oldpos, counter, ch, i, key;

  RepeatSelection = 0;

  timeout = 0; oldpos = -1; counter = 0; ch = 0;

  SaveRestoreScreen(0); //save screen

  for (i=0; i<SELECTOR_NUM; i++) DrawSelectChar(i);

  while (1)
  {
    timeout++; if (timeout==SELECTOR_TIMEOUT) break;

    WaitForVBlank();

    if (SelectPos!=oldpos)
    {
      if (oldpos!=-1) DrawSelectChar(oldpos);
      DrawSelectChar(SelectPos);
      oldpos = SelectPos;
    }

    key = 0;
    if ((REG_P1&KEY_LEFT )==0) key = 1;
    if ((REG_P1&KEY_RIGHT)==0) key = 2;
    if ((REG_P1&KEY_UP   )==0) key = 3;
    if ((REG_P1&KEY_DOWN )==0) key = 4;

    if (key==0) counter = 0; else {timeout = 0; if (counter==0) counter++; else if (counter==INPUT_DELAY) counter -= INPUT_REPEAT; else {counter++; key = 0;}}

    if (key==1) {SelectPos--;     if (SelectPos<0            ) SelectPos = SELECTOR_NUM-1;}
    if (key==2) {SelectPos++;     if (SelectPos>=SELECTOR_NUM) SelectPos = 0;}
    if (key==3) {SelectPos -= 10; if (SelectPos<0            ) SelectPos += 100; if (SelectPos>=SELECTOR_NUM) SelectPos -= 10;}
    if (key==4) {SelectPos += 10; if (SelectPos>=SELECTOR_NUM) SelectPos -= 100; if (SelectPos<0            ) SelectPos += 10;}

    if (PressRelease(KEY_A     )) {ch = 32; RepeatSelection = 1; break;}
    if (PressRelease(KEY_B     )) {ch = 32+SelectPos; RepeatSelection = 1; break;}
    if (PressRelease(KEY_L     )) {break;}
    if (PressRelease(KEY_R     )) {ch = 32+SelectPos; break;}
    if (PressRelease(KEY_SELECT)) DefinableKey = 32+SelectPos;
    if (PressRelease(KEY_START )) {ch = 13; break;}
  }

  SaveRestoreScreen(1); //restore screen

  return ch;
}



int GetNormalDirKey(void)
{
  int b;

  b = REG_P1;

  if ((b&(KEY_LEFT |KEY_UP  ))==0) return 'y';
  if ((b&(KEY_RIGHT|KEY_UP  ))==0) return 'u';
  if ((b&(KEY_LEFT |KEY_DOWN))==0) return 'b';
  if ((b&(KEY_RIGHT|KEY_DOWN))==0) return 'n';

  if ((b&KEY_LEFT            )==0) return 'h';
  if ((b&KEY_RIGHT           )==0) return 'l';
  if ((b&KEY_UP              )==0) return 'k';
  if ((b&KEY_DOWN            )==0) return 'j';

  return 0;
}



int GetDiagonalDirKey(void)
{
  int b;

  b = REG_P1;

  if ((b&(KEY_LEFT |KEY_UP  ))==0) return 'y';
  if ((b&(KEY_RIGHT|KEY_UP  ))==0) return 'u';
  if ((b&(KEY_LEFT |KEY_DOWN))==0) return 'b';
  if ((b&(KEY_RIGHT|KEY_DOWN))==0) return 'n';

  if ((b&KEY_LEFT            )==0) return 'y';
  if ((b&KEY_RIGHT           )==0) return 'n';
  if ((b&KEY_UP              )==0) return 'u';
  if ((b&KEY_DOWN            )==0) return 'b';

  return 0;
}



int GetRunDirKey(void)
{
  int b;

  b = REG_P1;

  if ((b&(KEY_LEFT |KEY_UP  ))==0) return 'Y';
  if ((b&(KEY_RIGHT|KEY_UP  ))==0) return 'U';
  if ((b&(KEY_LEFT |KEY_DOWN))==0) return 'B';
  if ((b&(KEY_RIGHT|KEY_DOWN))==0) return 'N';

  if ((b&KEY_LEFT            )==0) return 'H';
  if ((b&KEY_RIGHT           )==0) return 'L';
  if ((b&KEY_UP              )==0) return 'K';
  if ((b&KEY_DOWN            )==0) return 'J';

  return 0;
}



int SelectRoutine(void)
{
  int c, x, y, ch, temp;

  c = WHITE|BLACK_ON_GREEN;
  x = 18;
  y = 8;
  ch = DefinableKey;

  SaveRestoreScreen(0); //save screen

  while (1)
  {
    PutChar_6x8(ch=='y'?92:32, c, x+0, y+0);
    PutChar_6x8(ch=='k'?24:32, c, x+1, y+0);
    PutChar_6x8(ch=='u'?47:32, c, x+2, y+0);
    PutChar_6x8(ch=='h'?27:32, c, x+0, y+1);
    PutChar_6x8(ch           , c, x+1, y+1);
    PutChar_6x8(ch=='l'?26:32, c, x+2, y+1);
    PutChar_6x8(ch=='b'?47:32, c, x+0, y+2);
    PutChar_6x8(ch=='j'?25:32, c, x+1, y+2);
    PutChar_6x8(ch=='n'?92:32, c, x+2, y+2);

    temp = GetDiagonalDirKey(); if (temp) ch = temp;

    //pressed
    if ((REG_P1&KEY_A)==0) ch = 0;
    if ((REG_P1&KEY_B)==0) ch = DefinableKey;

    //released
    if (REG_P1&KEY_SELECT) break;
  }

  while ((REG_P1&(KEY_LEFT|KEY_RIGHT|KEY_UP|KEY_DOWN))!=(KEY_LEFT|KEY_RIGHT|KEY_UP|KEY_DOWN)) ; //wait for direction pad release

  SaveRestoreScreen(1); //restore screen

  return ch;
}



int StartRoutine(void)
{
  int c, x, y, ch, temp;

  c = WHITE|BLACK_ON_GREEN;
  x = 18;
  y = 8;
  ch = 13;

  SaveRestoreScreen(0); //save screen

  while (1)
  {
    PutChar_6x8(ch=='Y'?92:32, c, x+0, y+0);
    PutChar_6x8(ch=='K'?24:32, c, x+1, y+0);
    PutChar_6x8(ch=='U'?47:32, c, x+2, y+0);
    PutChar_6x8(ch=='H'?27:ch==13? 17:32, c, x+0, y+1);
    PutChar_6x8(           ch==13?196:ch, c, x+1, y+1);
    PutChar_6x8(ch=='L'?26:ch==13?217:32, c, x+2, y+1);
    PutChar_6x8(ch=='B'?47:32, c, x+0, y+2);
    PutChar_6x8(ch=='J'?25:32, c, x+1, y+2);
    PutChar_6x8(ch=='N'?92:32, c, x+2, y+2);

    temp = GetRunDirKey(); if (temp) ch = temp;

    //pressed
    if ((REG_P1&KEY_A)==0) ch = 0;
    if ((REG_P1&KEY_B)==0) ch = 13;

    //released
    if (REG_P1&KEY_START) break;
  }

  while ((REG_P1&(KEY_LEFT|KEY_RIGHT|KEY_UP|KEY_DOWN))!=(KEY_LEFT|KEY_RIGHT|KEY_UP|KEY_DOWN)) ; //wait for direction pad release

  SaveRestoreScreen(1); //restore screen

  return ch;
}



int GetInput(int max_iterations)
{
  int iterations, ch;

  static int counter = 0;



  if (UpdateTileOffset()) Refresh();



  iterations = 0; ch = 0;

  while (1)
  {
    WaitForVBlank();

    if (RepeatSelection) ch = SelectChar();
    else if (PressRelease(KEY_L)) ch = SelectChar();
    else
    {
      ch = GetNormalDirKey();

      if (ch==0) counter = 0; else {if (counter==0) counter++; else if (counter==INPUT_DELAY) counter -= INPUT_REPEAT; else {counter++; ch = 0;}}

      if (PressRelease(KEY_A    )) ch = 32; //space
      if (PressRelease(KEY_B    )) ch =  8; //backspace
      if (PressRelease(KEY_R    )) ch = 27; //escape

      if ((REG_P1&KEY_SELECT)==0) ch = SelectRoutine();
      if ((REG_P1&KEY_START )==0) ch =  StartRoutine();
    }

    if (ch) break;

    if (max_iterations) {iterations++; if (iterations==max_iterations) break;}
  }

  return ch;
}
//***************************************************************************



//***************************************************************************
//up to 256 palette entries
void SetPalette(void)
{
  int i;

  for (i=0; i<16; i++) BG_PALETTE[   i] =     GetColor(i);
  for (i=0; i<16; i++) BG_PALETTE[16+i] = TitlePalette[i];
  for (i=0; i<16; i++) BG_PALETTE[32+i] =  TilePalette[i];
}



//dir:  0 fade from black  1 fade to black
void FadePalette(int dir)
{
  int a, b, i;

  for (a=0; a<=256; a+=4)
  {
    WaitForVBlank();

    b = dir?a:256-a;

    for (i=0; i<16; i++) BG_PALETTE[   i] = SHIFT_COLOR16(    GetColor(i), COLOR16(0,0,0), b);
    for (i=0; i<16; i++) BG_PALETTE[16+i] = SHIFT_COLOR16(TitlePalette[i], COLOR16(0,0,0), b);
    for (i=0; i<16; i++) BG_PALETTE[32+i] = SHIFT_COLOR16( TilePalette[i], COLOR16(0,0,0), b);
  }
}



void InitInterface(void)
{
  REG_TM3CNT = TIME_FREQUENCY_SYSTEM|TIME_ENABLE; //start timer 3 (at system frequency)
  REG_TM3D = 0; //set timer 3 to 0

  REG_DISPCNT = MODE_4|BG2_ENABLE; //set display mode

  Clear();

  SetPalette();



  REG_IME = 0; //disable interrupts
  REG_INTERRUPT = (u32)IntHandler;
  REG_IE = BIT(4); //enable irq for timer 1 overflow
  REG_IME = 1; //enable interrupts



  REG_SOUNDCNT_X = SND_ENABLED;
  REG_SOUNDCNT_L = 0;
  REG_SOUNDCNT_H = SND_OUTPUT_RATIO_100|DSA_OUTPUT_RATIO_100|
                   DSA_OUTPUT_TO_RIGHT|DSA_OUTPUT_TO_LEFT|
                   DSA_TIMER0|DSA_FIFO_RESET;
}
//***************************************************************************



//***************************************************************************
int ch_attr, xpos, ypos;



void startup(void)
{
  ch_attr = GREY;
  xpos = 0;
  ypos = 0;

  InitInterface();
}



int get_input(void)
{
  return GetInput(0);
}



void tick_pause(void)
{
  int ch;

  ch = GetInput(2);
}



void move(int y, int x)
{
  if (x<0 || x>79) return;
  if (y<0 || y>24) return;

  xpos = x;
  ypos = y;
}



void getrc(int *y, int *x)
{
  *x = xpos;
  *y = ypos;
}



void set_attr(int a)
{
  ch_attr = a;
}



int get_attr(void)
{
  return ch_attr;
}



void putchr(int i)
{
  int x, y;

  textwin[ypos*80+xpos] = i|(ch_attr<<8);

  if (TileType==0)
  {
    x = xpos; y = ypos;
    if (TileMode && y>=1 && y<=22) {x -= TileOffsetX; y -= TileOffsetY; if (y<1 || y>22) return;}
    PutChar_4x6(i, ch_attr, x, y);
  }
}



int curch(void)
{
  return textwin[ypos*80+xpos];
}



void wclear(void)
{
  int i;

  move(0, 0);
  for (i = 0; i<80*25; i++) textwin[i] = 32|(GREY<<8);

  Clear();
}



void wdump(void)
{
  int savedx, savedy, x, y;

  getrc(&savedy, &savedx);

  for (y=0; y<25; y++) for (x=0; x<80; x++) {move(y, x); savewin[y*80+x] = curch();}

  move(savedy, savedx);
}



void wrestor(void)
{
  int savedx, savedy, saveda, x, y, i;

  getrc(&savedy, &savedx); saveda = get_attr();

  Clear();
  for (y=0; y<25; y++) for (x=0; x<80; x++) {move(y, x); i = savewin[y*80+x]; set_attr(i>>8); putchr(i&255);}

  set_attr(saveda); move(savedy, savedx);
}



void clrtoeol(void)
{
  int x, y, i, a;

  getrc(&y, &x);

  a = get_attr();
  set_attr(GREY);
  for (i = x; i<80; i++) {move(y, i); putchr(32);}
  set_attr(a);

  move(y, x);
}



void set_tile_mode(int mode)
{
  TileMode = mode;
}



unsigned short get_seed(void)
{
  //get random number seed from timer 3
  return REG_TM3D;
}
//***************************************************************************



//***************************************************************************
int curtain_flag = 0;
int transition_type = 0;



void drop_curtain(void)
{
  unsigned long *d;
  int i, j;

  d = (unsigned long *)VIDEO_FRAME0;

  j = CLIENTH; while (j--)
  {
    if (j&1) WaitForVBlank();

    i = CLIENTW>>2; while (i--) *d++ = ((unsigned long)GREEN<<24)|((unsigned long)GREEN<<16)|((unsigned long)GREEN<<8)|(unsigned long)GREEN;
  }

  SelectVideoFrame(1);

  curtain_flag = 1;
}



void raise_curtain(void)
{
  unsigned long *s, *d;
  int i, j;

  UpdateTileOffset(); Refresh();

  s = (unsigned long *)VIDEO_FRAME1+((CLIENTW*CLIENTH-1)>>2);
  d = (unsigned long *)VIDEO_FRAME0+((CLIENTW*CLIENTH-1)>>2);

  j = CLIENTH; while (j--)
  {
    if (j&1) WaitForVBlank();

    i = CLIENTW>>2; while (i--) *d-- = *s--;
  }

  SelectVideoFrame(0);

  curtain_flag = 0;
}



void start_transition(void)
{
  if (curtain_flag) return;

  transition_type = rangeMT(0, 1);

  if (transition_type==0)
  {
    FadePalette(1);
  }
  else
  {
    SelectVideoFrame(1);
  }
}



void end_transition(void)
{
  if (curtain_flag) return;

  UpdateTileOffset(); Refresh();

  if (transition_type==0)
  {
    FadePalette(0);
  }
  else
  {
    int i = 1;

    while (1)
    {
      WaitForVBlank();

      CopyEllipse(CLIENTW/2, CLIENTH/2, i, i, ((2*ELLIPSE_RAD+1)<<16)/(2*i+1));

      i += 1; if (i>CLIENTW/2+32) break;
    }

    SelectVideoFrame(0);
  }
}



void toggle_tiles(void)
{
  TileType = 1-TileType;

  Clear();

  UpdateTileOffset(); Refresh();
}
//***************************************************************************



//***************************************************************************
//cave image must be same size as screen
void display_cave_exit(void)
{
  unsigned long *s, *d;
  int i, j;

  Clear();
  for (i=0; i<256; i++) BG_PALETTE[i] = CavePalette[i];

  s = (unsigned long *)CaveData+(CAVE_DATA_W>>2)*(CAVE_DATA_H-1);
  d = (unsigned long *)VideoFrame;
  j = CAVE_DATA_H; while (j--)
  {
    i = CAVE_DATA_W>>2; while (i--) *d++ = *s++;
    s += (-CAVE_DATA_W-CAVE_DATA_W)>>2;
  }

  while (1)
  {
    if (PressRelease(KEY_A     )) break;
    if (PressRelease(KEY_B     )) break;
    if (PressRelease(KEY_START )) break;
  }

  Clear();
  SetPalette();
}
//***************************************************************************
