#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gccore.h>
#include <malloc.h>

#include "IOSPatcher.h"
#include "wad.h"
#include "tools.h"

#define IOS15version 1031
#define IOS36version 3607
#define IOS37version 5662



// Prevent IOS36 loading at startup
s32 __IOS_LoadStartupIOS()
{
	return 0;
}

s32 __u8Cmp(const void *a, const void *b)
{
	return *(u8 *)a-*(u8 *)b;
}

u8 *get_ioslist(u32 *cnt)
{
	u64 *buf = 0;
	s32 i, res;
	u32 tcnt = 0, icnt;
	u8 *ioses = NULL;
	
	//Get stored IOS versions.
	res = ES_GetNumTitles(&tcnt);
	if(res < 0)
	{
		printf("ES_GetNumTitles: Error! (result = %d)\n", res);
		return 0;
	}
	buf = memalign(32, sizeof(u64) * tcnt);
	res = ES_GetTitles(buf, tcnt);
	if(res < 0)
	{
		printf("ES_GetTitles: Error! (result = %d)\n", res);
		if (buf) free(buf);
		return 0;
	}

	icnt = 0;
	for(i = 0; i < tcnt; i++)
	{
		if(*((u32 *)(&(buf[i]))) == 1 && (u32)buf[i] > 2 && (u32)buf[i] < 0x100)
		{
			icnt++;
			ioses = (u8 *)realloc(ioses, sizeof(u8) * icnt);
			ioses[icnt - 1] = (u8)buf[i];
		}
	}

	ioses = (u8 *)malloc(sizeof(u8) * icnt);
	icnt = 0;
	
	for(i = 0; i < tcnt; i++)
	{
		if(*((u32 *)(&(buf[i]))) == 1 && (u32)buf[i] > 2 && (u32)buf[i] < 0x100)
		{
			icnt++;
			ioses[icnt - 1] = (u8)buf[i];
		}
	}
	free(buf);
	qsort(ioses, icnt, 1, __u8Cmp);

	*cnt = icnt;
	return ioses;
}


int ios_selectionmenu(int default_ios)
{
	u32 pressed;
	u32 pressedGC;
	int selection = 0;
	u32 ioscount;
	u8 *list = get_ioslist(&ioscount);
	
	int i;
	for (i=0;i<ioscount;i++)
	{
		// Default to default_ios if found, else the loaded IOS
		if (list[i] == default_ios)
		{
			selection = i;
			break;
		}
		if (list[i] == IOS_GetVersion())
		{
			selection = i;
		}
	}	
	
	while (true)
	{
		printf("\x1B[%d;%dH",2,0);	// move console cursor to y/x
		printf("Select which IOS to load:       \b\b\b\b\b\b");
		
		set_highlight(true);
		printf("IOS%u\n", list[selection]);
		set_highlight(false);
		
		printf("Press B to continue without IOS Reload\n");
		
		waitforbuttonpress(&pressed, &pressedGC);
		
		if (pressed == WPAD_BUTTON_LEFT || pressedGC == PAD_BUTTON_LEFT)
		{	
			if (selection > 0)
			{
				selection--;
			} else
			{
				selection = ioscount - 1;
			}
		}
		if (pressed == WPAD_BUTTON_RIGHT || pressedGC == PAD_BUTTON_RIGHT)
		{
			if (selection < ioscount -1	)
			{
				selection++;
			} else
			{
				selection = 0;
			}
		}
		if (pressed == WPAD_BUTTON_A || pressedGC == PAD_BUTTON_A) break;
		if (pressed == WPAD_BUTTON_B || pressedGC == PAD_BUTTON_B)
		{
			return 0;
		}
		
	}
	return list[selection];
}


void show_boot2_info()
{
	int ret;
	printf("\x1b[2J");
	printheadline();
	printf("\n");

	printf("Retrieving boot2 version...\n");
	u32 boot2version = 0;
	ret = ES_GetBoot2Version(&boot2version);
	if (ret < 0)
	{
		printf("Could not get boot2 version. It's possible your Wii is\n");
		printf("a boot2v4+ Wii, maybe not.\n");
	} else
	{
		printf("Your boot2 version is: %u\n", boot2version);
		if (boot2version < 4)
		{
			printf("This means you should not have problems.\n");
		}
	}	
	
	printf("\n");
	printf("Boot2v4 is an indicator for the 'new' Wii hardware revision that prevents\n");
	printf("the execution of some old IOS. These Wiis are often called LU64+ Wiis or\n");
	printf("'unsoftmoddable' Wiis. You MUST NOT downgrade one of these Wiis and be\n");
	printf("EXTRA careful when messing with ANYTHING on them.\n");
	printf("The downgraded IOS15 you get with the Trucha Bug Restorer should work\n");
	printf("on these Wiis and not harm Wiis in general.\n");
	printf("\n");
	printf("If you updated your Wii via wifi to 4.2 or higher, your boot2 got\n");
	printf("updated by this and you can't use it as indicator for this.\n");
	printf("\n");
	printf("Press any button to return to the menu\n");
	waitforbuttonpress(NULL, NULL);
}


int iosinstallmenu(u32 ios, u32 revision)
{
	u32 pressed;
	u32 pressedGC;
	u32 maxoptions = 0;
	int selection = 0;
	int destselect = 0;
	int ret;
	int i;
	bool options[4] = { true, false, false, false };

	char *optionsstring[4] = {	"Patch hash check (trucha): ",
								"Patch ES_Identify:         ",
								"Patch nand permissions:    ",
								"Patch version check:       "};
								
	u32 destination[3] = { ios, 200+ios, 249 };
	
	while (true)
	{
		printf("\x1b[2J");
		printheadline();
		printf("\n");
		
		set_highlight(selection == 0);
		if (options[0] || options[1] || options[2] || options[3] || ios != destination[destselect])
		{
			printf("Install patched IOS%u\n", ios);
		} else
		{
			printf("Install IOS%u\n", ios);
		}
		set_highlight(false);

		printf("Install IOS to slot:       ");
		set_highlight(selection == 1);
		printf("%u\n", destination[destselect]);
		set_highlight(false);
		
		if (ios == 36)
		{
			maxoptions = 4;
		} else
		{
			maxoptions = 1;
		}
		
		for (i=0;i < maxoptions;i++)
		{
			printf(optionsstring[i]);
			set_highlight(selection == i+2);
			printf("%s\n", options[i] ? "yes" : "no");
			set_highlight(false);
		}
		printf("\n");
		
		waitforbuttonpress(&pressed, &pressedGC);
		
		if (pressed == WPAD_BUTTON_LEFT || pressedGC == PAD_BUTTON_LEFT)
		{	
			if (selection == 1)
			{
				if (destselect > 0)
				{
					destselect--;
				} else
				{
					destselect = 2;
				}
			}
			if (selection > 1)
			{
				options[selection-2] = !options[selection-2];
			}
		}

		if (pressed == WPAD_BUTTON_RIGHT || pressedGC == PAD_BUTTON_RIGHT)
		{	
			if (selection == 1)
			{
				if (destselect < 2)
				{
					destselect++;
				} else
				{
					destselect = 0;
				}
			}
			if (selection > 1)
			{
				options[selection-2] = !options[selection-2];
			}
		}
		
		if (pressed == WPAD_BUTTON_UP || pressedGC == PAD_BUTTON_UP)
		{
			if (selection > 0)
			{
				selection--;
			} else
			{
				selection = maxoptions+1;
			}
		}

		if (pressed == WPAD_BUTTON_DOWN || pressedGC == PAD_BUTTON_DOWN)
		{
			if (selection < maxoptions+1)
			{
				selection++;
			} else
			{
				selection = 0;
			}
		}

		if ((pressed == WPAD_BUTTON_A || pressedGC == PAD_BUTTON_A) && selection == 0)
		{
			if (destination[destselect] == 36 || destination[destselect] == 37)
			{
				ret = Install_patched_IOS(ios, revision, options[0], options[1], options[2], options[3], destination[destselect], revision);
			} else
			{
				ret = Install_patched_IOS(ios, revision, options[0], options[1], options[2], options[3], destination[destselect], 1);
			}
			if (ret < 0)
			{
				printf("IOS%u installation failed.\n", ios);
				return -1;
			}
			return 0;
		}
		
		if (pressed == WPAD_BUTTON_B || pressedGC == PAD_BUTTON_B)
		{
			printf("Installation cancelled\n");
			return 0;
		}		
	}	

}

int mainmenu()
{
	u32 pressed;
	u32 pressedGC;
	int selection = 0;
	int ret;
	int i;
	char *optionsstring[6] = { "IOS36 menu", "IOS37 menu", "Downgrade IOS15", "Restore IOS15", "Show boot2 info", "Exit" };
	
	while (true)
	{
		printf("\x1b[2J");

		printheadline();
		printf("\n");
		
		for (i=0;i < 6;i++)
		{
			set_highlight(selection == i);
			printf("%s\n", optionsstring[i]);
			set_highlight(false);
		}
		printf("\n");
		
		waitforbuttonpress(&pressed, &pressedGC);
		
		if (pressed == WPAD_BUTTON_UP || pressedGC == PAD_BUTTON_UP)
		{
			if (selection > 0)
			{
				selection--;
			} else
			{
				selection = 5;
			}
		}

		if (pressed == WPAD_BUTTON_DOWN || pressedGC == PAD_BUTTON_DOWN)
		{
			if (selection < 5)
			{
				selection++;
			} else
			{
				selection = 0;
			}
		}

		if (pressed == WPAD_BUTTON_A || pressedGC == PAD_BUTTON_A)
		{
			if (selection == 0)
			{
				return iosinstallmenu(36, IOS36version);
			}
			
			if (selection == 1)
			{
				return iosinstallmenu(37, IOS37version);
			}

			if (selection == 2)
			{
				ret = Downgrade_IOS(15, IOS15version, 257);
				if (ret < 0)
				{
					printf("Downgrade failed\n");
					return -1;
				}
				return 0;
			}
			
			if (selection == 3)
			{
				ret = install_unpatched_IOS(15, IOS15version);
				if (ret < 0)
				{
					printf("IOS15 Restore failed\n");
					return -1;
				}
				printf("IOS15 restored.\n");
				return 0;
			}
			
			if (selection == 4)
			{
				show_boot2_info();
			}

			if (selection == 5)
			{
				return 0;
			}
		}
		
		if (pressed == WPAD_BUTTON_B || pressedGC == PAD_BUTTON_B)
		{
			return 0;
		}		
	}	
}

int main(int argc, char* argv[])
{
	int ret;
	Init_Console();
	printf("\x1b[%u;%um", 37, false);

	PAD_Init();
	WPAD_Init();
	WPAD_SetDataFormat(WPAD_CHAN_0, WPAD_FMT_BTNS_ACC_IR);					

	printheadline();
	
	ret = ios_selectionmenu(36);
	if (ret != 0)
	{
		WPAD_Shutdown();
		IOS_ReloadIOS(ret);
		PAD_Init();
		WPAD_Init();
		WPAD_SetDataFormat(WPAD_CHAN_0, WPAD_FMT_BTNS_ACC_IR);					
	}

	printf("\n");
	
	printf("This application can brick your Wii, use it at your own risk!\n\n");
	printf("Skip the IOS15 downgrade if you don't need it! It's for example not needed\n");
	printf("on system menu <3.3, if a cIOS is installed or this app is booted with\n");
	printf("cBoot2.\n\n");
	printf("Don't forget to restore IOS15 after using it.\n\n");
	printf("See readme for instructions and details.\n\n");
	
	time_t t = time(NULL) + 7;
	while (time(NULL) < t)
	{
		WPAD_ScanPads();
		PAD_ScanPads();
		if(WPAD_ButtonsDown(0) || PAD_ButtonsDown(0)) 
		{
			printf("Don't be impatient, press any button to exit...\n");
			waitforbuttonpress(NULL, NULL);
			Reboot();
		}
	}
	
	printf("Press 1 or Start to start the application...\n");

	u32 pressed;
	u32 pressedGC;	
	waitforbuttonpress(&pressed, &pressedGC);
	if (pressed != WPAD_BUTTON_1 && pressedGC != PAD_BUTTON_START)
	{
		printf("Other button pressed, press any button to exit...\n");
		waitforbuttonpress(NULL, NULL);
		Reboot();
	}

	mainmenu();
	
	Close_SD();
	Close_USB();
	
	printf("Press any button to exit...\n");
	waitforbuttonpress(NULL, NULL);
	
	Reboot();

	return 0;
}
