#include "stdafx.h"
#include "SeqTrack.h"
#include "SeqEvent.h"
#include "VGMSeq.h"
#include "MidiFile.h"
#include "ScaleConversion.h"

//  ********
//  SeqTrack
//  ********


SeqTrack::SeqTrack(VGMSeq* parentFile, ULONG offset, ULONG length)
: VGMContainerItem(parentFile, offset, length), parentSeq(parentFile),/* pMidiTrack(NULL),*/ bInLoop(FALSE)/*mastVol(255),*/
{
	pMidiTrack = NULL;
	vol = 100;
	expression = 127;
	prevPan = 64;
	channelGroup = 0;
	transpose = 0;
	cDrumNote = -1;
	cKeyCorrection = 0;
	bDetermineTrackLengthEventByEvent = false;

	swprintf(numberedName, sizeof(numberedName)/sizeof(numberedName[0]), L"Track %d", parentSeq->aTracks.size()+1);
	name = numberedName;
	AddContainer<SeqEvent>(aEvents);
}

SeqTrack::~SeqTrack()
{
	DeleteVect<SeqEvent>(aEvents);
}


/*void SeqTrack::AddToUI(VGMItem* parent, VGMFile* theVGMFile)
{
	pRoot->UI_AddItem(vgmfile, this, parent, name);
	for (UINT i=0; i<aEvents.size(); i++)
		aEvents[i]->AddToUI(this);
}*/


int SeqTrack::ReadEvent(void)
{
	return false;		//by default, don't add any events, just stop immediately.
}

int SeqTrack::LoadTrack(int trackNum, U32 stopOffset)
{
	if (!LoadTrackInit(trackNum))
		return false;
	if (!LoadTrackMainLoop(stopOffset))
		return false;
	return true;
}

int SeqTrack::LoadTrackInit(int trackNum)
{
	pMidiTrack = parentSeq->midi.AddTrack();
	SetChannelAndGroupFromTrkNum(trackNum);

	if (parentSeq->bWriteInitialTempo && trackNum == 0)
		pMidiTrack->AddTempoBPM(parentSeq->tempoBPM);
	if (parentSeq->bAlwaysWriteInitialVol)
		AddVolNoItem(parentSeq->initialVol);
	if (parentSeq->bAlwaysWriteInitialExpression)
		AddExpressionNoItem(parentSeq->initialExpression);
	if (parentSeq->bAlwaysWriteInitialPitchBendRange)
		AddPitchBendRangeNoItem(parentSeq->initialPitchBendRangeSemiTones, parentSeq->initialPitchBendRangeCents);

	return true;
}

int SeqTrack::LoadTrackMainLoop(U32 stopOffset)
{
	bInLoop = false;
	curOffset = dwOffset;	//start at beginning of track
	while ((curOffset < stopOffset) && ReadEvent())
		;
	if (unLength == 0)			//if unLength has not been changed from default value of 0
		unLength = curOffset-dwOffset;

	return true;
}

void SeqTrack::SetChannelAndGroupFromTrkNum(int theTrackNum)
{
	if (theTrackNum > 39)
		theTrackNum += 3;
	else if (theTrackNum > 23)
		theTrackNum += 2;					//compensate for channel 10 - drum track.  we'll skip it, by default
	else if (theTrackNum > 8)
		theTrackNum++;						//''
	channel = theTrackNum%16;
	channelGroup = theTrackNum/16;
	pMidiTrack->SetChannelGroup(channelGroup);
}

ULONG SeqTrack::GetDelta(void)
{
	return pMidiTrack->GetDelta();
}

void SeqTrack::SetDelta(ULONG NewDelta)
{
	pMidiTrack->SetDelta(NewDelta);
}

void SeqTrack::AddDelta(ULONG AddDelta)
{
	pMidiTrack->AddDelta(AddDelta);
}

void SeqTrack::SubtractDelta(ULONG SubtractDelta)
{
	pMidiTrack->SubtractDelta(SubtractDelta);
}

void SeqTrack::ResetDelta(void)
{
	pMidiTrack->ResetDelta();
}

ULONG SeqTrack::ReadVarLen(ULONG& offset)
{
    register ULONG value;
    register UCHAR c;

    if ( (value = GetByte(offset++)) & 0x80 )
    {
       value &= 0x7F;
       do
       {
         value = (value << 7) + ((c = GetByte(offset++)) & 0x7F);
       } while (c & 0x80);
    }

    return value;
}

void SeqTrack::AddControllerSlide(U32 offset, U32 length, U32 dur, BYTE& prevVal, BYTE targVal, 
								  void (MidiTrack::*insertFunc)(BYTE, BYTE, U32))
{
	double valInc = (double)((double)(targVal-prevVal)/(double)dur);
	char newVal = -1;
	for (int i=0; i<dur; i++)
	{
		char prevValInSlide = newVal;
		newVal=ceil(prevVal+(valInc*(i+1)));
		//only create an event if the pan value has changed since the last iteration
		if (prevValInSlide != newVal)
		{
			if (newVal < 0)
				newVal = 0;
			if (newVal > 0x7F)
				newVal = 0x7F;
			(pMidiTrack->*insertFunc)(channel, newVal, GetDelta()+i);
		}
	}
	prevVal = targVal;
}


ULONG SeqTrack::offsetInQuestion;

struct SeqTrack::IsEventAtOffset : unary_function< SeqEvent, bool >
{
	inline bool operator()( const SeqEvent* theEvent ) const
	{
		return (theEvent->dwOffset == offsetInQuestion);
	}
};


bool SeqTrack::IsOffsetUsed(ULONG offset)
{
	offsetInQuestion = offset;
	vector<SeqEvent*>::iterator iter = find_if(aEvents.begin(), aEvents.end(), IsEventAtOffset());
	return (iter != aEvents.end());
}


void SeqTrack::AddEvent(SeqEvent* pSeqEvent)
{
	//if (!bInLoop)
		aEvents.push_back(pSeqEvent);
	//else
	//	delete pSeqEvent;
	if (bDetermineTrackLengthEventByEvent)
	{
		ULONG length = pSeqEvent->dwOffset + pSeqEvent->unLength - dwOffset;
		if (unLength < length)
			unLength = length;
	}
}

void SeqTrack::AddGenericEvent(ULONG offset, ULONG length, const wchar_t* sEventName, BYTE color)
{
	if (!IsOffsetUsed(offset))
	{
		SeqEvent* newEvent = new SeqEvent(this, offset, length, sEventName);
		newEvent->color = color;
		AddEvent(newEvent);
	}
}


void SeqTrack::AddUnknown(ULONG offset, ULONG length, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
	{
		SeqEvent* newEvent = new SeqEvent(this, offset, length, sEventName);
		newEvent->color = CLR_UNKNOWN;
		AddEvent(newEvent);
	}
}

void SeqTrack::AddSetOctave(ULONG offset, ULONG length, BYTE newOctave,  const wchar_t* sEventName)
{
	octave = newOctave; 
	if (!IsOffsetUsed(offset))
		AddEvent(new SetOctaveSeqEvent(this, newOctave, offset, length, sEventName));
}

void SeqTrack::AddIncrementOctave(ULONG offset, ULONG length, const wchar_t* sEventName)
{
	octave++;
	AddEvent(new SeqEvent(this, offset, length, sEventName));
}

void SeqTrack::AddDecrementOctave(ULONG offset, ULONG length, const wchar_t* sEventName)
{
	octave--;
	if (!IsOffsetUsed(offset))
		AddEvent(new SeqEvent(this, offset, length, sEventName));
}

void SeqTrack::AddRest(ULONG offset, ULONG length, UINT restTime,  const wchar_t* sEventName)
{
	AddDelta(restTime);
	if (!IsOffsetUsed(offset))
		AddEvent(new RestSeqEvent(this, restTime, offset, length, sEventName));
}

void SeqTrack::AddHold(ULONG offset, ULONG length, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new SeqEvent(this, offset, length, sEventName));
}

void SeqTrack::AddNoteOn(ULONG offset, ULONG length, char key, char vel, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new NoteOnSeqEvent(this, key, vel, offset, length, sEventName));
	AddNoteOnNoItem(key, vel);
}

void SeqTrack::AddNoteOnNoItem(char key, char vel)
{
	BYTE finalVel = vel;
	if (parentSeq->bUseLinearAmplitudeScale)
		finalVel = Convert7bitPercentVolValToStdMidiVal(vel);

	if (cDrumNote == -1)
	{
		pMidiTrack->AddNoteOn(channel, key+cKeyCorrection+transpose+parentSeq->globTranspose, finalVel);
	}
	else
		AddPercNoteOnNoItem(cDrumNote, finalVel);
	prevKey = key;
	prevVel = vel;
	return;
}


void SeqTrack::AddPercNoteOn(ULONG offset, ULONG length, char key, char vel, const wchar_t* sEventName)
{
	BYTE origChan = channel;
	channel = 9;
	char origDrumNote = cDrumNote;
	cDrumNote = -1;
//	DrumAssoc* pDrumAssoc = parentSeq->GetDrumAssoc(key);
//	if (pDrumAssoc)
//		key = pDrumAssoc->GMDrumNote;
	AddNoteOn(offset, length, key-transpose-parentSeq->globTranspose, vel, sEventName);
	cDrumNote = origDrumNote;
	channel = origChan;
}

void SeqTrack::AddPercNoteOnNoItem(char key, char vel)
{
	BYTE origChan = channel;
	channel = 9;
	char origDrumNote = cDrumNote;
	cDrumNote = -1;
//	DrumAssoc* pDrumAssoc = parentSeq->GetDrumAssoc(key);
//	if (pDrumAssoc)
//		key = pDrumAssoc->GMDrumNote;
	AddNoteOnNoItem(key-transpose-parentSeq->globTranspose, vel);
	cDrumNote = origDrumNote;
	channel = origChan;
}

void SeqTrack::InsertNoteOn(ULONG offset, ULONG length, char key, char vel, ULONG absTime, const wchar_t* sEventName)
{
	BYTE finalVel = vel;
	if (parentSeq->bUseLinearAmplitudeScale)
		finalVel = Convert7bitPercentVolValToStdMidiVal(vel);

	if (!IsOffsetUsed(offset))
		AddEvent(new NoteOnSeqEvent(this, key, vel, offset, length, sEventName));
	pMidiTrack->InsertNoteOn(channel, key+cKeyCorrection+transpose+parentSeq->globTranspose, finalVel, absTime);
	prevKey = key;
	prevVel = vel;
}

void SeqTrack::AddNoteOff(ULONG offset, ULONG length, char key, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new NoteOffSeqEvent(this, key, offset, length, sEventName));
	AddNoteOffNoItem(key);
}

void SeqTrack::AddNoteOffNoItem(char key)
{
	if (cDrumNote == -1)
	{
		pMidiTrack->AddNoteOff(channel, key+cKeyCorrection+transpose+parentSeq->globTranspose);
	}
	else
		AddPercNoteOffNoItem(cDrumNote);
	return;
}


void SeqTrack::AddPercNoteOff(ULONG offset, ULONG length, char key,const wchar_t* sEventName)
{
	BYTE origChan = channel;
	channel = 9;
	char origDrumNote = cDrumNote;
	cDrumNote = -1;
//	DrumAssoc* pDrumAssoc = parentSeq->GetDrumAssoc(key);
//	if (pDrumAssoc)
//		key = pDrumAssoc->GMDrumNote;
	AddNoteOff(offset, length, key-transpose-parentSeq->globTranspose, sEventName);
	cDrumNote = origDrumNote;
	channel = origChan;
}

void SeqTrack::AddPercNoteOffNoItem(char key)
{
	BYTE origChan = channel;
	channel = 9;
	char origDrumNote = cDrumNote;
	cDrumNote = -1;
//	DrumAssoc* pDrumAssoc = parentSeq->GetDrumAssoc(key);
//	if (pDrumAssoc)
//		key = pDrumAssoc->GMDrumNote;
	AddNoteOffNoItem(key-transpose-parentSeq->globTranspose);
	cDrumNote = origDrumNote;
	channel = origChan;
}

void SeqTrack::InsertNoteOff(ULONG offset, ULONG length, char key, ULONG absTime, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new NoteOffSeqEvent(this, key, offset, length, sEventName));
	pMidiTrack->InsertNoteOff(channel, key+cKeyCorrection+transpose+parentSeq->globTranspose, absTime);
}

void SeqTrack::AddNoteByDur(ULONG offset, ULONG length, char key, char vel, UINT dur, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new DurNoteSeqEvent(this, key, vel, dur, offset, length, sEventName));
	AddNoteByDurNoItem(key, vel, dur);
}

void SeqTrack::AddNoteByDurNoItem(char key, char vel, UINT dur)
{
	BYTE finalVel = vel;
	if (parentSeq->bUseLinearAmplitudeScale)
		finalVel = Convert7bitPercentVolValToStdMidiVal(vel);

	if (cDrumNote == -1)
	{
		pMidiTrack->AddNoteByDur(channel, key+cKeyCorrection+transpose+parentSeq->globTranspose, finalVel, dur);
	}
	else
		AddPercNoteByDurNoItem(cDrumNote, vel, dur);
	prevKey = key;
	prevVel = vel;
	return;
}

void SeqTrack::AddPercNoteByDur(ULONG offset, ULONG length, char key, char vel, UINT dur, const wchar_t* sEventName)
{
	BYTE origChan = channel;
	channel = 9;
	char origDrumNote = cDrumNote;
	cDrumNote = -1;
//	DrumAssoc* pDrumAssoc = parentSeq->GetDrumAssoc(key);
//	if (pDrumAssoc)
//		key = pDrumAssoc->GMDrumNote;
	AddNoteByDur(offset, length, key-transpose-parentSeq->globTranspose, vel, dur, sEventName);
	cDrumNote = origDrumNote;
	channel = origChan;
}

void SeqTrack::AddPercNoteByDurNoItem(char key, char vel, UINT dur)
{
	BYTE origChan = channel;
	channel = 9;
	char origDrumNote = cDrumNote;
	cDrumNote = -1;
//	DrumAssoc* pDrumAssoc = parentSeq->GetDrumAssoc(key);
//	if (pDrumAssoc)
//		key = pDrumAssoc->GMDrumNote;
	AddNoteByDurNoItem(key-transpose-parentSeq->globTranspose, vel, dur);
	cDrumNote = origDrumNote;
	channel = origChan;
}

/*void SeqTrack::AddNoteByDur(ULONG offset, ULONG length, char key, char vel, UINT dur, BYTE chan, const wchar_t* sEventName)
{
	BYTE origChan = channel;
	channel = chan;
	AddNoteByDur(offset, length, key, vel, dur, selectMsg, sEventName);
	channel = origChan;
}*/

void SeqTrack::InsertNoteByDur(ULONG offset, ULONG length, char key, char vel, UINT dur, ULONG absTime, const wchar_t* sEventName)
{
	BYTE finalVel = vel;
	if (parentSeq->bUseLinearAmplitudeScale)
		finalVel = Convert7bitPercentVolValToStdMidiVal(vel);

	if (!IsOffsetUsed(offset))
		AddEvent(new DurNoteSeqEvent(this, key, vel, dur, offset, length, sEventName));
	pMidiTrack->InsertNoteByDur(channel, key+cKeyCorrection+transpose+parentSeq->globTranspose, finalVel, dur, absTime);
	prevKey = key;
	prevVel = vel;
}

//void SeqTrack::MakePrevDurNoteEnd()
//{
//	//if (mode == MODE_MIDI && pMidiTrack->prevDurEvent && pMidiTrack->prevDurEvent->AbsTime + pMidiTrack->prevDurEvent->dur > GetDelta())
//	if (pMidiTrack->prevDurEvent)
//		pMidiTrack->prevDurEvent->duration = GetDelta() - pMidiTrack->prevDurEvent->AbsTime;
//}

void SeqTrack::MakePrevDurNoteEnd()
{
	if (pMidiTrack->prevDurNoteOff)
		pMidiTrack->prevDurNoteOff->AbsTime = GetDelta();
}

void SeqTrack::AddVol(ULONG offset, ULONG length, BYTE newVol, const wchar_t* sEventName)
{	
	if (!IsOffsetUsed(offset))
		AddEvent(new VolSeqEvent(this, newVol, offset, length, sEventName));
	//pMidiTrack->AddMastVolMarker(channel, mastVol, PRIORITY_HIGHEST);		//highest priority so that other mast vol events overlapping are the ones to take effect
	//pMidiTrack->AddVolMarker(channel, vol);
	AddVolNoItem(newVol);
}

void SeqTrack::AddVolNoItem(BYTE newVol)
{
	BYTE finalVol = newVol;
	if (parentSeq->bUseLinearAmplitudeScale)
		finalVol = Convert7bitPercentVolValToStdMidiVal(newVol);

	pMidiTrack->AddVol(channel, finalVol);
	vol = newVol;
	return;
}

void SeqTrack::AddVolSlide(ULONG offset, ULONG length, ULONG dur, BYTE targVol, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new VolSlideSeqEvent(this, targVol, dur, offset, length, sEventName));
	AddControllerSlide(offset, length, dur, vol, targVol, &MidiTrack::InsertVol);
}

void SeqTrack::InsertVol(ULONG offset, ULONG length, BYTE newVol, ULONG absTime, const wchar_t* sEventName)
{
	BYTE finalVol = newVol;
	if (parentSeq->bUseLinearAmplitudeScale)
		finalVol = Convert7bitPercentVolValToStdMidiVal(newVol);

	if (!IsOffsetUsed(offset))
		AddEvent(new VolSeqEvent(this, newVol, offset, length, sEventName));
	//pMidiTrack->AddMastVolMarker(channel, mastVol, PRIORITY_HIGHEST);		//highest priority so that other mast vol events overlapping are the ones to take effect
	//pMidiTrack->InsertVolMarker(channel, vol, absTime);
	pMidiTrack->InsertVol(channel, finalVol, absTime);
	vol = newVol;
}

void SeqTrack::AddExpression(ULONG offset, ULONG length, BYTE level, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new ExpressionSeqEvent(this, level, offset, length, sEventName));
	AddExpressionNoItem(level);
}

void SeqTrack::AddExpressionNoItem(BYTE level)
{
	BYTE finalExpression = level;
	if (parentSeq->bUseLinearAmplitudeScale)
		finalExpression = Convert7bitPercentVolValToStdMidiVal(level);

	pMidiTrack->AddExpression(channel, finalExpression);
	expression = level;
}

void SeqTrack::AddExpressionSlide(ULONG offset, ULONG length, ULONG dur, BYTE targExpr, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new ExpressionSlideSeqEvent(this, targExpr, dur, offset, length, sEventName));
	AddControllerSlide(offset, length, dur, expression, targExpr, &MidiTrack::InsertExpression);
}

void SeqTrack::InsertExpression(ULONG offset, ULONG length, BYTE level, ULONG absTime, const wchar_t* sEventName)
{
	BYTE finalExpression = level;
	if (parentSeq->bUseLinearAmplitudeScale)
		finalExpression = Convert7bitPercentVolValToStdMidiVal(level);

	if (!IsOffsetUsed(offset))
		AddEvent(new ExpressionSeqEvent(this, level, offset, length, sEventName));
	pMidiTrack->InsertExpression(channel, finalExpression, absTime);
	expression = level;
}


void SeqTrack::AddMasterVol(ULONG offset, ULONG length, BYTE newVol, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new MastVolSeqEvent(this, newVol, offset, length, sEventName));
	AddMasterVolNoItem(newVol);
}

void SeqTrack::AddMasterVolNoItem(BYTE newVol)
{
	BYTE finalVol = newVol;
	if (parentSeq->bUseLinearAmplitudeScale)
		finalVol = Convert7bitPercentVolValToStdMidiVal(newVol);

	//mastVol = newVol;
	pMidiTrack->AddMasterVol(channel, finalVol);
	//pMidiTrack->AddMastVolMarker(channel, mastVol);
	//pMidiTrack->AddVol(channel);
}
/*
void SeqTrack::AddMastVolSlide(ULONG offset, ULONG length, ULONG dur, char targMastVol, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new MastVolSlideSeqEvent(this, targMastVol, dur, offset, length, sEventName));
	//pMidiTrack->AddMastVolMarker(channel, mastVol, PRIORITY_HIGHEST);		//highest priority so that other mast vol events overlapping are the ones to take effect

	double volInc = (double)((double)(targMastVol-vol)/(double)dur);
	char newVol;
	for (int i=0; i<dur; i++)
	{
		newVol=ceil(vol+(volInc*(i+1)));
		if (newVol < 0)
			newVol = 0;
		if (newVol > 0x7F)
			newVol = 0x7F;
//			pMidiTrack->InsertVolMarker(channel, newVol, GetDelta()+i);
//			pMidiTrack->InsertVol(channel, GetDelta()+i);
	}
	vol = targVol;
}*/

/*
void SeqTrack::AddMastVolSlide(ULONG offset, ULONG length, ULONG dur, BYTE targVol, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new MastVolSlideSeqEvent(this, targVol, dur, offset, length, sEventName), CLR_SETVOLUME);
	double volInc = (targVol-mastVol)/dur;
	BYTE newVol;
	for (int i=0; i<dur; i++)
	{
		newVol=ceil(mastVol+(volInc*(i+1)));
		pMidiTrack->InsertMastVolMarker(channel, newVol, GetDelta()+i);
		pMidiTrack->InsertVol(channel, GetDelta()+i);
	}
	mastVol = targVol;
}

void SeqTrack::InsertMastVol(ULONG offset, ULONG length, BYTE newVol, ULONG absTime, const wchar_t* sEventName)
{
	mastVol = newVol;
	if (!IsOffsetUsed(offset))
		AddEvent(new MastVolSeqEvent(this, newVol, offset, length, sEventName), CLR_SETVOLUME);
	pMidiTrack->InsertMastVolMarker(channel, mastVol, absTime);
	pMidiTrack->AddVol(channel);
}*/



void SeqTrack::AddPan(ULONG offset, ULONG length, BYTE pan, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new PanSeqEvent(this, pan, offset, length, sEventName));
	pMidiTrack->AddPan(channel, pan);
	prevPan = pan;
}

void SeqTrack::AddPanSlide(ULONG offset, ULONG length, ULONG dur, BYTE targPan, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new PanSlideSeqEvent(this, targPan, dur, offset, length, sEventName));
	AddControllerSlide(offset, length, dur, prevPan, targPan, &MidiTrack::InsertPan);
}


void SeqTrack::InsertPan(ULONG offset, ULONG length, BYTE pan, ULONG absTime, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new PanSeqEvent(this, pan, offset, length, sEventName));
	pMidiTrack->InsertPan(channel, pan, absTime);
}

void SeqTrack::AddPitchBendMidiFormat(ULONG offset, ULONG length, BYTE lo, BYTE hi, const wchar_t* sEventName)
{
	AddPitchBend(offset, length, lo+(hi<<7)-0x2000, sEventName);
}

void SeqTrack::AddPitchBend(ULONG offset, ULONG length, SHORT bend, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new PitchBendSeqEvent(this, bend, offset, length, sEventName));
	pMidiTrack->AddPitchBend(channel, bend);
}

void SeqTrack::AddPitchBendRange(ULONG offset, ULONG length, BYTE semitones, BYTE cents, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new PitchBendRangeSeqEvent(this, semitones, cents, offset, length, sEventName));
	pMidiTrack->AddPitchBendRange(channel, semitones, cents);
}

void SeqTrack::AddPitchBendRangeNoItem(BYTE semitones, BYTE cents)
{
	pMidiTrack->AddPitchBendRange(channel, semitones, cents);
}

void SeqTrack::AddTranspose(ULONG offset, ULONG length, int transpose, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new TransposeSeqEvent(this, transpose, offset, length, sEventName));
	pMidiTrack->AddTranspose(channel, transpose);
}


void SeqTrack::AddModulation(ULONG offset, ULONG length, BYTE depth, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new ModulationSeqEvent(this, depth, offset, length, sEventName));
	pMidiTrack->AddModulation(channel, depth);
}

void SeqTrack::AddBreath(ULONG offset, ULONG length, BYTE depth, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new BreathSeqEvent(this, depth, offset, length, sEventName));
	pMidiTrack->AddBreath(channel, depth);
}

void SeqTrack::AddSustainEvent(ULONG offset, ULONG length, bool bOn, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new SustainSeqEvent(this, bOn, offset, length, sEventName));
	pMidiTrack->AddSustain(channel, bOn);
}

void SeqTrack::InsertSustainEvent(ULONG offset, ULONG length, bool bOn, ULONG absTime, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new SustainSeqEvent(this, bOn, offset, length, sEventName));
	pMidiTrack->InsertSustain(channel, bOn, absTime);
}

void SeqTrack::AddPortamento(ULONG offset, ULONG length, bool bOn, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new PortamentoSeqEvent(this, bOn, offset, length, sEventName));
	pMidiTrack->AddPortamento(channel, bOn);
}

void SeqTrack::InsertPortamento(ULONG offset, ULONG length, bool bOn, ULONG absTime, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new PortamentoSeqEvent(this, bOn, offset, length, sEventName));
	pMidiTrack->InsertPortamento(channel, bOn, absTime);
}

void SeqTrack::AddPortamentoTime(ULONG offset, ULONG length, BYTE time, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new PortamentoTimeSeqEvent(this, time, offset, length, sEventName));
	pMidiTrack->AddPortamentoTime(channel, time);
}

void SeqTrack::InsertPortamentoTime(ULONG offset, ULONG length, BYTE time, ULONG absTime, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new PortamentoTimeSeqEvent(this, time, offset, length, sEventName));
	pMidiTrack->InsertPortamentoTime(channel, time, absTime);
}



/*void InsertNoteOnEvent(char key, char vel, ULONG absTime);
void AddNoteOffEvent(char key);
void InsertNoteOffEvent(char key, char vel, ULONG absTime);
void AddNoteByDur(char key, char vel);
void InsertNoteByDur(char key, char vel, ULONG absTime);
void AddVolumeEvent(BYTE vol);
void InsertVolumeEvent(BYTE vol, ULONG absTime);
void AddExpression(BYTE expression);
void InsertExpression(BYTE expression, ULONG absTime);
void AddPanEvent(BYTE pan);
void InsertPanEvent(BYTE pan, ULONG absTime);*/

void SeqTrack::AddProgramChange(ULONG offset, ULONG length, BYTE progNum, const wchar_t* sEventName)
{
/*	InstrAssoc* pInstrAssoc = parentSeq->GetInstrAssoc(progNum);
	if (pInstrAssoc)
	{
		if (pInstrAssoc->drumNote == -1)		//if this program uses a drum note
		{
			progNum = pInstrAssoc->GMProgNum;
			cKeyCorrection = pInstrAssoc->keyCorrection;
			cDrumNote = -1;
		}
		else
			cDrumNote = pInstrAssoc->drumNote;
	}
	else
		cDrumNote = -1;
*/
	if (!IsOffsetUsed(offset))
		AddEvent(new ProgChangeSeqEvent(this, progNum, offset, length, sEventName));
//	if (cDrumNote == -1)
//	{
		pMidiTrack->AddProgramChange(channel, progNum);
//	}

}

void SeqTrack::AddProgramChange(ULONG offset, ULONG length, BYTE progNum, BYTE chan, const wchar_t* sEventName)
{
	//if (selectMsg = NULL)
	//	selectMsg.Forma
	BYTE origChan = channel;
	channel = chan;
	AddProgramChange(offset, length, progNum, sEventName);
	channel = origChan;
}

void SeqTrack::AddTempo(ULONG offset, ULONG length, ULONG microsPerQuarter, const wchar_t* sEventName)
{
	parentSeq->tempoBPM = ((double)60000000/microsPerQuarter);
	if (!IsOffsetUsed(offset))
		AddEvent(new TempoSeqEvent(this, parentSeq->tempoBPM, offset, length, sEventName));
	pMidiTrack->AddTempo(microsPerQuarter);
}

void SeqTrack::AddTempoSlide(ULONG offset, ULONG length, ULONG dur, ULONG targMicrosPerQuarter, const wchar_t* sEventName)
{
	AddTempoBPMSlide(offset, length, dur, ((double)60000000/targMicrosPerQuarter), sEventName);
}

void SeqTrack::AddTempoBPM(ULONG offset, ULONG length, double bpm, const wchar_t* sEventName)
{
	parentSeq->tempoBPM = bpm;
	if (!IsOffsetUsed(offset))
		AddEvent(new TempoSeqEvent(this, bpm, offset, length, sEventName));
	pMidiTrack->AddTempoBPM(bpm);
}

void SeqTrack::AddTempoBPMSlide(ULONG offset, ULONG length, ULONG dur, double targBPM, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new TempoSlideSeqEvent(this, targBPM, dur, offset, length, sEventName));
	//pMidiTrack->AddMastVolMarker(channel, mastVol, PRIORITY_HIGHEST);		//highest priority so that other mast vol events overlapping are the ones to take effect
	double tempoInc = (targBPM-parentSeq->tempoBPM)/((double)dur);
	double newTempo;
	for (int i=0; i<dur; i++)
	{
		newTempo=parentSeq->tempoBPM+(tempoInc*(i+1));
		pMidiTrack->InsertTempoBPM(newTempo, GetDelta()+i);
	}
	parentSeq->tempoBPM = targBPM;
}

void SeqTrack::AddTimeSig(ULONG offset, ULONG length, BYTE numer, BYTE denom, BYTE ticksPerQuarter,  const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new TimeSigSeqEvent(this, numer, denom, ticksPerQuarter, offset, length, sEventName));
	pMidiTrack->AddTimeSig(numer, denom, ticksPerQuarter);
}

void SeqTrack::InsertTimeSig(ULONG offset, ULONG length, BYTE numer, BYTE denom, BYTE ticksPerQuarter,ULONG absTime,const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new TimeSigSeqEvent(this, numer, denom, ticksPerQuarter, offset, length, sEventName));
	pMidiTrack->InsertTimeSig(numer, denom, ticksPerQuarter, absTime);
}

void SeqTrack::AddEndOfTrack(ULONG offset, ULONG length, const wchar_t* sEventName)
{
	if (!IsOffsetUsed(offset))
		AddEvent(new TrackEndSeqEvent(this, offset, length, sEventName));
	pMidiTrack->AddEndOfTrack();
}





//HELPER FUNCTIONS

//BYTE SeqTrack::Convert7bitPercentVolValToStdMidiVal(BYTE percentVal)
//{
//	//In standard MIDI, the attenuation for volume in db is 40*log10(127/val) == 20*log10(127^2/val^2). (dls1 spec page 14)
//	//Here, the scale is different.  We get rid of the exponents
//	// so it's just 20*log10(127/val).
//	//Therefore, we must convert from one scale to the other.
//	//The equation for the first line is self-evident.
//	//For the second line, I simply solved db = 40*log10(127/val) for val.
//	//The result is val = 127*10^(-0.025*db).  So, by plugging in the original val into
//	//the original scale equation, we get a db that we can plug into that second equation to get
//	//the new val.
//
//	double origAttenInDB = 20*log10((127/(double)percentVal));
//	return 127*pow(10.0,-0.025*origAttenInDB);
//}