#include "stdafx.h"
#include "QSoundSeq.h"
#include "QSoundFormat.h"

DECLARE_FORMAT(QSound);

const BYTE delta_table[3][7] = { 2, 4, 8, 0x10, 0x20, 0x40, 0x80,
								  3, 6, 0xC, 0x18, 0x30, 0x60, 0xC0,
								  0, 9, 0x12, 0x24, 0x48, 0x90, 0 };
//octave_table provides the note value for the start of each octave.
//wholly unnecessary for me include it, but i'm following the original driver code verbatim for now
const BYTE octave_table[] = { 0x00, 0x0C, 0x18, 0x24, 0x30, 0x3C, 0x48, 0x54,
						   0x18, 0x24, 0x30, 0x3C, 0x48, 0x54, 0x60, 0x6C };


// *********
// QSoundSeq
// *********

QSoundSeq::QSoundSeq(RawFile* file, ULONG offset, float fmtVersion)
: VGMSeq(QSoundFormat::name, file, offset), fmt_version(fmtVersion)
{
	//UseReverb();
	//AlwaysWriteInitialVol(127);
}

QSoundSeq::~QSoundSeq(void)
{
}

int QSoundSeq::GetHeaderInfo(void)
{
	SetPPQN(0x30);

	wostringstream	theName;
	theName << L"QSound Seq";
	name = theName.str();
	return true;		//successful
}


int QSoundSeq::GetTrackPointers(void)
{
	for (int i=0; i<16; i++)
	{
		UINT offset = GetShortBE(dwOffset+1+i*2);
		if (offset == 0)
			break;
		if (GetShortBE(offset+dwOffset) == 0xE017)	//Rest, EndTrack (used by empty tracks)
			continue;
		QSoundTrack *newTrack = new QSoundTrack(this, offset+dwOffset);
		aTracks.push_back(newTrack);
	}
	if (aTracks.size() == 0)
		return false;
	// if the first track is in some completely separate location from the track ptrs
	// make the start of the first track the beginning of the sequence
	if (aTracks[0]->dwOffset - dwOffset > 0x21)
		dwOffset = aTracks[0]->dwOffset;
	else
		this->AddHeader(dwOffset, aTracks[0]->dwOffset - dwOffset, L"Track Pointers");

	return true;
}

int QSoundSeq::LoadTracks(void)
{
	for (UINT i=0; i<nNumTracks; i++)
	{
		if (!aTracks[i]->LoadTrack(i, 0xFFFFFFFF))
			return false;
	}
	aTracks[0]->InsertTimeSig(0, 0, 4, 4, GetPPQN(), 0);
}

// *************
// QSoundTrack
// *************


QSoundTrack::QSoundTrack(QSoundSeq* parentSeq, long offset, long length)
: SeqTrack(parentSeq, offset, length), curDeltaTable(0), noteState(0)
{
	memset(loop, 0, sizeof(loop));
}


int QSoundTrack::ReadEvent(void)
{
	ULONG beginOffset = curOffset;
	BYTE status_byte = GetByte(curOffset++);

	UINT value1;

	if (status_byte >= 0x20)
	{
		if ( (noteState & 0x30) == 0)
			curDeltaTable = 1;
		//else if (noteState & 0x20)
		else if ((noteState & 0x10) == 0)
			curDeltaTable = 0;
		else
		{
			noteState &= ~(1 << 4);		//RES 4		at 0xBD6 in sfa2
			curDeltaTable = 2;
		}

		//effectively, use the highest 3 bits of the status byte as index to delta_table.
		//those highest 3 bits are used as though lowest 3 bits, of course
		value1 = delta_table[curDeltaTable][((status_byte>>5)&7)-1];	// this code starts at BB3
		//rest_time += value1;
		

		if ((status_byte & 0x1F) != 0)		//if it's not a rest
		{
			UINT tempDuration = (float)(dur/127)*value1;
			if (dur == 0)
				tempDuration = value1;
			if (tempDuration > value1)
				tempDuration = value1;

			key = (status_byte & 0x1F) + octave_table[noteState & 0x0F];
			//if (mode == MODE_MIDI && pMidiTrack->prevDurEvent && pMidiTrack->prevDurEvent->AbsTime + pMidiTrack->prevDurEvent->duration > GetDelta())
			if (pMidiTrack->prevDurNoteOff && pMidiTrack->prevDurNoteOff->AbsTime > GetDelta())
				MakePrevDurNoteEnd();
			AddNoteByDur(beginOffset, curOffset-beginOffset, key, vel, dur);
		}
		else			//it's a rest
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Rest"), BG_CLR_CYAN);
		AddDelta(value1);
	}
	else
	{
		int loopNum;
		switch (status_byte)
		{
		case 0x00 :
			noteState ^= 0x20;
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Note State xor 0x20"), BG_CLR_WHEAT);
			break;
		case 0x01 :
			noteState ^= 0x40;
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Note State xor 0x40"), BG_CLR_WHEAT);
			break;
		case 0x02 :
			noteState |= (1 << 4);
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Note State |= 0x10"), BG_CLR_WHEAT);
			break;
		case 0x03 :
			noteState ^= 8;
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Note State xor 8"), BG_CLR_WHEAT);
			break;
		case 0x04 :
			noteState &= 0x97;
			noteState |= GetByte(curOffset++);
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Change Note State (& 0x97)"), BG_CLR_WHEAT);
			break;
		case 0x05 :
			{
				float fmt_version = ((QSoundSeq*)parentSeq)->fmt_version;
				//if (isEqual(fmt_version, 1.71))
				
				if (((QSoundSeq*)parentSeq)->fmt_version + F_EPSILON >= 1.71){
					BYTE tempo = GetByte(curOffset++);
					AddTempoBPM(beginOffset, curOffset-beginOffset, tempo);
				}
				else
				{

					USHORT tempo = GetShortBE(curOffset);
					double fTempo = tempo/3.25;
					curOffset+=2;
					AddTempoBPM(beginOffset, curOffset-beginOffset, fTempo);
					//RecordMidiSetTempoBPM(current_delta_time, flValue1, hFile);
				}
			}
			break;
		case 0x06 :
			dur = GetByte(curOffset++);
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Set Duration"), BG_CLR_STEEL);
			break;
		case 0x07 :
			vel = GetByte(curOffset++);			//expression value
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Set Velocity"), BG_CLR_STEEL);
			break;
		case 0x08 :
			{
				BYTE progNum = GetByte(curOffset++);
				if (((QSoundSeq*)parentSeq)->fmt_version + F_EPSILON < 1.16)
				{
					pMidiTrack->AddBankSelect(channel, 0);				
					pMidiTrack->AddBankSelectFine(channel, progNum/128); 
					AddProgramChange(beginOffset, curOffset-beginOffset, progNum%128);
				}
				else
					AddProgramChange(beginOffset, curOffset-beginOffset, progNum);
			}
			break;
		case 0x09 :					//effectively sets the octave
			noteState &= 0xF8;
			noteState |= GetByte(curOffset++);
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Set Octave"), BG_CLR_STEEL);
			break;
		case 0x0A :
			curOffset++;
			AddUnknown(beginOffset, curOffset-beginOffset);
			break;
		case 0x0B :
			transpose = GetByte(curOffset++);
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Transpose?"), BG_CLR_STEEL);
			break;
		case 0x0C :
			curOffset++;
			AddUnknown(beginOffset, curOffset-beginOffset);
			break;
		case 0x0D :				//portamento speed, i think.  Clearly pitch related
			curOffset++;
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Portamento"), CLR_PITCHBEND);
			break;
		case 0x0E :				//loop
			loopNum = 0;
			goto theLoop;
		case 0x0F :
			loopNum = 1;
			goto theLoop;
		case 0x10 :
			loopNum = 2;
			goto theLoop;
		case 0x11 :
			loopNum = 3;
theLoop:	if (loop[loopNum] == 0)						//first time hitting loop
			{
				bInLoop = true;
				loop[loopNum] = GetByte(curOffset++);	//set the number of times to loop
			}
			else										//already engaged in loop
			{
				loop[loopNum]--;						//decrement loop counter
				curOffset++;
				if (loop[loopNum] == 0)
				{
					bInLoop = false;
					curOffset+=2;
					AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Loop"), BG_CLR_CHEDDAR);
					break;
				}
			}
			{
				short jump = (GetByte(curOffset++)<<8) + GetByte(curOffset++);
				curOffset += jump;
			}
			break;
		case 0x12 :				
			loopNum = 0;
			goto loopBreak;
		case 0x13 :
			loopNum = 1;
			goto loopBreak;
		case 0x14 :
			loopNum = 2;
			goto loopBreak;
		case 0x15 :
			loopNum = 3;
loopBreak:	if (loop[loopNum]-1 == 0)
			{
				bInLoop = false;
				loop[loopNum] = 0;
				noteState &= 0x97;
				noteState |= GetByte(curOffset++);
				{
					short jump = (GetByte(curOffset++)<<8) + GetByte(curOffset++);
					AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Loop Break"), BG_CLR_CHEDDAR);
					curOffset += jump;
				}
			}
			else
				curOffset += 3;
			break;
		case 0x16 :
			curOffset+=2;
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Loop Always"), BG_CLR_CHEDDAR);
			break;
		case 0x17 :
			MakePrevDurNoteEnd();
			AddEndOfTrack(beginOffset, curOffset-beginOffset);
			return false;
		case 0x18 :				// pan
			{
				//the pan value is b/w 0 and 0x20.  0 - hard left, 0x10 - center, 0x20 - hard right
				BYTE pan = GetByte(curOffset++);
				pan *= 4;
				if (pan == 0x80) pan = 0x7f;
				this->AddPan(beginOffset, curOffset-beginOffset, pan);
			}
			break;
		case 0x19 :
			curOffset++;
			AddUnknown(beginOffset, curOffset-beginOffset);
			break;
		case 0x1A :	//master(?) vol
			{
				//curOffset++;
				vol = GetByte(curOffset++);
				AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Master Volume"), CLR_UNKNOWN);
				//this->AddMasterVol(beginOffset, curOffset-beginOffset, vol);
				//AddVolume(beginOffset, curOffset-beginOffset, vol);
			}
			//AddUnknown(beginOffset, curOffset-beginOffset);
			break;
		case 0x1B :				//Vibrato freq
			curOffset++;
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Vibrato"), CLR_MODULATION);
			break;
		case 0x1C :
			curOffset++;
			AddUnknown(beginOffset, curOffset-beginOffset);
			break;
		case 0x1D :
			curOffset++;
			AddUnknown(beginOffset, curOffset-beginOffset);
			break;
		case 0x1E :
			curOffset++;
			AddUnknown(beginOffset, curOffset-beginOffset);
			break;
		case 0x1F :
			{
				U8 value = GetByte(curOffset++);
				if (((QSoundSeq*)parentSeq)->fmt_version + F_EPSILON < 1.16)
				{
					pMidiTrack->AddBankSelect(channel, 0);				
					pMidiTrack->AddBankSelectFine(channel, 2+(value/128));
					AddProgramChange(beginOffset, curOffset-beginOffset, value%128);
				}
				else
				{
					pMidiTrack->AddBankSelect(channel, 0);				
					pMidiTrack->AddBankSelectFine(channel, value);
					AddGenericEvent(beginOffset, curOffset-beginOffset, _T("Bank Change"), CLR_PROGCHANGE);
				}
					
			}
			break;
		case 0x20 :
			curOffset++;
			AddUnknown(beginOffset, curOffset-beginOffset);
			break;
		case 0x21 :
			curOffset++;
			AddUnknown(beginOffset, curOffset-beginOffset);
			break;
		case 0x22 :
			curOffset++;
			AddUnknown(beginOffset, curOffset-beginOffset);
			break;
		case 0x23 :
			curOffset++;
			AddUnknown(beginOffset, curOffset-beginOffset);
			break;
		default :
			AddGenericEvent(beginOffset, curOffset-beginOffset, _T("UNKNOWN"), BG_CLR_RED);
		}
	}
	return true;
}
