/* Max Drive Save Converter
   Copyright (C) 2005 ASk, EVASiON & TheImp

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA

*/

#include "convert.h"

int main(int argc, char *argv[]) {


    FILE *input, *output;
    char *new_filename = NULL;
    char *temp;

    /* ASk */
    long flen;
    byte *fbuf = NULL;
    byte *converted_fbuf = NULL;

    if (argc != 3) {
        usage(argv[0]);
        return -1;
    }
    

    printf( "+---------------------------------------------------+\n"
            "| MAX Drive Save Converter by ASk & EVASiON         |\n"
            "| GCI and SAV file formats reverse engineered by    |\n"
            "| Timothy Wilson (TheImp)                           |\n"
            "| Modified by Buzbee on 12-8-2009  v1.03            |\n"
            "+---------------------------------------------------+\n\n");
    /* first test for possible errors */



    if (strcmp(argv[1], "-s2g") != 0)
    { 
     if (strcmp(argv[1], "-g2s") != 0)
     {
      if (strcmp(argv[1], "-c2g") != 0)
      {
       if (strcmp(argv[1], "-g2c") != 0)
        {
          usage(argv[0]);
          return -1;
        }
       }
      }
     } 

    if ((input = fopen(argv[2], "rb")) == NULL) {
        fprintf(stderr, "Could not open input file %s!\n", argv[2]);
        return -1;
    }

    /* ASk - read the file to buffer, replace the needed bytes */
    fseek(input, 0, SEEK_END);
    flen = ftell(input); /* works fine since it's in binary mode */
    fbuf = malloc(flen*sizeof(byte));
    fseek(input, 0, SEEK_SET);
    fread(fbuf, flen,1,input);
    fclose(input);


    new_filename =  malloc(strlen(argv[2])+1);

    temp = strrchr(argv[2],'.');
    strncpy(new_filename, argv[2], temp - argv[2]);
    new_filename[temp-argv[2]]='\0';
    

    
    if(strcmp(argv[1], "-s2g") == 0) {
        converted_fbuf = sav2gci(fbuf, flen);
        flen = flen - SAV_HEADER_LENGTH;
        strcat(new_filename, ".gci");
    }
    if(strcmp(argv[1], "-c2g") == 0) {
        converted_fbuf = gcs2gci(fbuf, flen);
        flen = flen - SAV_HEADER_LENGTH2;
        strcat(new_filename, ".gci");
    }
    if(strcmp(argv[1], "-g2c") == 0) {
        converted_fbuf = gci2gcs(fbuf, flen);
        flen = flen + SAV_HEADER_LENGTH2;
        strcat(new_filename, ".gcs");
    }
    if(strcmp(argv[1], "-g2s") == 0) {
        converted_fbuf = gci2sav(fbuf, flen);
        flen = flen + SAV_HEADER_LENGTH;
        strcat(new_filename, ".sav");
    }

    if(converted_fbuf == NULL)
    {
        fprintf(stderr, "Error occured during conversion process");
        free(fbuf);
        free(converted_fbuf);
        free(new_filename);
        exit(-1);
    }



    /* we read and changed, now we can be almost sure that it will succeed
    unless we run into problems out of our reach
    */
   
 
    /* check for file existence */
    if((output = fopen(new_filename, "r")) != NULL)
    {
        /* file exists, backup */
        char *backup_filename = (char *)malloc((strlen(new_filename) + 128)*sizeof(char));
        time_t timer;
        struct tm *today;

        fclose(output);
        time(&timer);
        today = localtime(&timer);
        snprintf(backup_filename, strlen(new_filename) + 127,
            "%s-%02d.%02d.%02d-%02d.%02d.%04d",
            new_filename, today->tm_hour, today->tm_min, today->tm_sec,
            today->tm_mday, today->tm_mon, 1900+today->tm_year);
        rename(new_filename, backup_filename);
        fprintf(stdout, "Backed up existing file %s as %s\n",
            new_filename, backup_filename);
        free(backup_filename);


    }
    
    if ((output = fopen(new_filename, "wb")) == NULL) {
        fprintf(stderr, "Could not open output file %s!\n", new_filename);
        free(fbuf);
        free(converted_fbuf);
        free(new_filename);
        return -1;
    }

    fwrite(converted_fbuf, 1, flen, output);

    fclose(output);

    printf("Succesfully created %s\n", new_filename);

    free(fbuf);
    free(converted_fbuf);
    free(new_filename);

    return 0;
}


void usage(char *prog) {
    fprintf(stderr, "+---------------------------------------------------+\n"
                    "| MAX Drive Save Converter by ASk & EVASiON         |\n"
                    "| GCI and SAV file formats reverse engineered by    |\n"
                    "| Timothy Wilson (TheImp)                           |\n"
                    "| Modified by Buzbee on 12-8-2009  v1.03            |\n"
                    "+---------------------------------------------------+\n\n"
			        "Usage: %s [-direction] <original file> \n"
                    " -direction values:\n"
                    "   -g2s - gci to sav converter\n"
                    "   -g2c - gci to gcs converter\n"
                    "   -c2g - gcs to gci converter\n"
                    "   -s2g - sav to gci converter\n", prog);
}

int is_big_endian()
{
    long one= 1;
    return !(*((char *)(&one)));
}


byte* gci2sav(byte *buf, int buflen)
{
    byte *sav;
    int i;
    int base_offset;


    

    sav = (byte *)malloc((buflen + SAV_HEADER_LENGTH)*sizeof(byte));
    if(sav == NULL)
    {
        fprintf(stderr, "Memory allocation failed\n");     
        return NULL;
    }
   

    /* stage 1+2 */
    for(i=0; i<0x80; i++)
        sav[i] = 0;

    memcpy(sav, SAV_HEADER, strlen(SAV_HEADER));
    memcpy(sav + SAV_HEADER_LENGTH, buf, buflen);

    /* stage 3 */
    /* find out if our system is little-endian, and if yes, convert */
    if(!is_big_endian()) {
       base_offset = 
           (sav[SAV_HEADER_LENGTH + 0x3C] << 24) +
           (sav[SAV_HEADER_LENGTH + 0x3D] << 16) +
           (sav[SAV_HEADER_LENGTH + 0x3E] << 8)  +
            sav[SAV_HEADER_LENGTH + 0x3F];
    }
    else {
        base_offset = (int)(*(sav + SAV_HEADER_LENGTH + 0x3C));
    }

    /* stage 4+5 */
    base_offset = base_offset + SAV_HEADER_LENGTH + 0x40;
    memcpy(sav + 0x10, sav + base_offset, 0x20);

    /* stage 6+7 */
    base_offset += 0x20;
    memcpy(sav + 0x48, sav + base_offset, 0x20);

    /* step 8 */

    SWAP(sav[SAV_HEADER_LENGTH + 0x06],sav[SAV_HEADER_LENGTH + 0x07]);
  
    for(i=0; i<10; i++)
    {
        SWAP(sav[SAV_HEADER_LENGTH + 0x2C + i*2],sav[SAV_HEADER_LENGTH + 0x2C + i*2 + 1]);
    }

    return sav;    
}

byte *sav2gci(byte *buf, int buflen)
{
    byte *gci;
    int i;

    /* check if a file is SAV */
    if(memcmp(buf, SAV_HEADER, strlen(SAV_HEADER)))
    {
        fprintf(stderr, "Not a valid SAV save file\n");
        return NULL;
    }
    
    gci = (byte *)malloc((buflen - 0x80)*sizeof(byte));
    
    if(gci == NULL)
    {
        fprintf(stderr, "Memory allocation failed\n");     
        return NULL;
    }

    
    
    /* stage 1 */
    memcpy(gci, buf+SAV_HEADER_LENGTH, buflen-SAV_HEADER_LENGTH);

    /* stage 2 */
    
    SWAP(gci[0x06],gci[0x07]);

    for(i=0; i<10; i++)
    {
        SWAP(gci[0x2C + i*2],gci[0x2C + i*2 + 1]);
    }

    return gci;
}

byte *gcs2gci(byte *buf, int buflen)
{
    byte *gci;
    int i;

    /* check if a file is GCS */
    if(memcmp(buf, SAV_HEADER2, strlen(SAV_HEADER2)))
    {
        fprintf(stderr, "Not a valid GCS save file\n");
        return NULL;
    }
    
    gci = (byte *)malloc((buflen - 0x110)*sizeof(byte));
    
    if(gci == NULL)
    {
        fprintf(stderr, "Memory allocation failed\n");     
        return NULL;
    }

    /* stage 1 */
    memcpy(gci, buf+SAV_HEADER_LENGTH2, buflen-SAV_HEADER_LENGTH2);

    return gci;
}

byte* gci2gcs(byte *buf, int buflen)
{
    byte *gcs;
    int i;
    int base_offset;


    

    gcs = (byte *)malloc((buflen + SAV_HEADER_LENGTH2)*sizeof(byte));
    if(gcs == NULL)
    {
        fprintf(stderr, "Memory allocation failed\n");     
        return NULL;
    }
   

    /* stage 1+2 */
    for(i=0; i<0x110; i++)
        gcs[i] = 0;

    memcpy(gcs, SAV_HEADER2, strlen(SAV_HEADER2));
    memcpy(gcs + SAV_HEADER_LENGTH2, buf, buflen);
    
    /* stage 3 */
    /* find out if our system is little-endian, and if yes, convert */
    if(!is_big_endian()) {
       base_offset = 
           (gcs[SAV_HEADER_LENGTH2 + 0x3C] << 24) +
           (gcs[SAV_HEADER_LENGTH2 + 0x3D] << 16) +
           (gcs[SAV_HEADER_LENGTH2 + 0x3E] << 8)  +
            gcs[SAV_HEADER_LENGTH2 + 0x3F];
    }
    else {
        base_offset = (int)(*(gcs + SAV_HEADER_LENGTH2 + 0x3C));
    }

    /* stage 4+5 */
    base_offset = base_offset + SAV_HEADER_LENGTH2 + 0x40;
    memcpy(gcs + 0x10, gcs + base_offset, 0x20);

   
    return gcs;    
}
