/*
 * NDS SAVES to SLOT-2 v1.0 - by Marc R. (2011)
 *
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License 
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License along 
 * with this program; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
 
#include <nds.h>
#include <nds/interrupts.h>
#include <nds/card.h>
#include <nds/arm9/console.h>
#include <fat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/dir.h>
#include <sys/unistd.h>
#include <dirent.h>
#include <algorithm>
#include "fat.h"
#include "dsCard.h"
#include "auxspi.h"


#define BLACK			30
#define DARK_RED		31
#define DARK_GREEN		32
#define DARK_YELLOW		33
#define DARK_BLUE		34
#define DARK_MAGENTA	35
#define DARK_CYAN		36
#define DARK_WHITE		37
#define GREY			40
#define RED				41
#define GREEN			42
#define YELLOW			43
#define BLUE			44
#define MAGENTA			45
#define CYAN			46
#define WHITE			47

#define BLOCKSIZE	256
#define SAV_PATH	"/NDS_Backup"
#define MENU_ITEMS	16


//Card info
char gameId[4+1];
char gameTitle[12+1];	
int saveSize;
int saveSizeK;
uint8 saveType;
bool with_infrared;

//FAT files
int nFiles=0;
char* fileNames[256];



void PAD_Pause(){
	iprintf(" (press A to continue)\n\n");
	for(;;){
		scanKeys();
		if (keysDown() & KEY_A)
			break;
	}
}

bool PAD_Ask(){
	bool ret=false;
	iprintf(" [A] Continue    [B] Cancel\n\n");
	for(;;){
		scanKeys();
		if (keysDown() & KEY_A){
			ret=true;
			break;
		}else if (keysDown() & KEY_B)
			break;
	}
	return ret;
}


void CONSOLE_SetColor(u8 color){
	printf("\x1b[%um", color);
	fflush(stdout);
}

void CONSOLE_PrintHeader(int showCardInfo){
	consoleClear();
	iprintf("    NDS SAVES to SLOT-2 v1.0\n");
	if(showCardInfo){
		iprintf("  Game:  %s - %s\n", gameId, gameTitle);
		iprintf("  Save:  %dKB", saveSizeK);
		if(with_infrared)
			iprintf(" (IR detected)");
		iprintf("\n");
	}
	iprintf("---------------------- by Marc -\n");
	//iprintf(" Save type:  %d\n", saveType);
}

bool swap_cart(){
	int i;
	sNDSHeader header1;
	sNDSHeader header2;
	
	iprintf(" Insert a SLOT-1 game card.\n Reinsert it if it's needed.\n");
	PAD_Pause();

	// identify hardware
	sysSetBusOwners(true, true);
	cardReadHeader((u8*)&header1);
	cardReadHeader((u8*)&header2);

	//read header twice to check if it's encrypted (if it's the case, it should return different junk chars)
	while(memcmp((u8*)&header1, (u8*)&header2, 32) != 0) {
		// If not, the card needs ejected and reinserted into the DS
		iprintf(" Reinsert SLOT-1 card.\n");
		PAD_Pause();
		cardReadHeader((u8*)&header1);
		cardReadHeader((u8*)&header2);
	}

	for(i=0; i<4; i++)
		gameId[i]=header1.gameCode[i];
	gameId[4]='\0';
	for(i=0; i<12; i++)
		gameTitle[i]=header1.gameTitle[i];
	gameTitle[12]='\0';
	

	// Check IR-enabled game. If the save size returns something nonzero, we have
	// found an IR-enabled game If there is no IR device present, "size2" is always zero.
	with_infrared = true;
	uint32 size2 = auxspi_save_size_log_2();
	if (size2) {		
		saveSize=auxspi_save_size();
		saveType=auxspi_save_type();
		saveSizeK=saveSize/1024;
		return true;
	}

	// Check if it's a Flash Card, which has no save chip. Therefore, trying to
	// identify this without IR-specific functions again returns zero.
	with_infrared = false;
	uint32 size1 = auxspi_save_size_log_2();
	if (!size1) {
		return false;
	}

	// No special hardware found, so it must be a regular game.
	saveSize=auxspi_save_size();
	saveType=auxspi_save_type();
	saveSizeK=saveSize/1024;
	return true;
}


/*void __print_Header_Info(u8* buffer){
	iprintf("HDR: %d %d %d %d %d %d %d %d\n", buffer[0], buffer[1], buffer[2], buffer[3], buffer[4], buffer[5], buffer[6], buffer[7]);
	iprintf("     %d %d %d %d %d %d %d %d\n\n", buffer[8], buffer[9], buffer[10], buffer[11], buffer[12], buffer[13], buffer[14], buffer[15]);
}*/





int hwBackupGBA(char* fileName){
	int i;
	char filePath[256];
	u8* buffer;


	sprintf(filePath, SAV_PATH "/%s", fileName);

	/*uint8 size = auxspi_save_size_log_2();
	int size_blocks = 1 << max(0, (int8(size) - 18)); // ... in units of 0x40000 bytes - that's 256 kB
	if (size < 15)
		size_blocks = 1;
	else
		size_blocks = 1 << (uint8(size) - 15);*/

	// Show some info
	CONSOLE_PrintHeader(1);
	iprintf(" Backup savegame to:\n");
	iprintf(" %s\n\n", fileName);


	if(saveSize%BLOCKSIZE != 0){
		iprintf("ERROR! Invalid size.\n");
		PAD_Pause();
		return -1;
	}


	if(PAD_Ask()){
		iprintf(" Reading...\n");
		buffer=(u8*)malloc(saveSize);

		//Set to 00
		for(i=0; i<saveSize; i++)
			buffer[i]=0x00;

		//Copy savedata to byte buffer
		for(i=0; i<saveSize; i+=BLOCKSIZE)
			auxspi_read_data(i, (u8*)&buffer[i], BLOCKSIZE, saveType);	

		//__print_Header_Info(buffer);

		iprintf(" Saving... ");
		i=FAT_Save_File(filePath, buffer, saveSize);
		iprintf("%d bytes saved\n", i);
		if(i==-1){
			iprintf("ERROR!\nFile was not created.\n");
		}else if(i!=saveSize){
			iprintf("ERROR!\nSizes don't match.\n");
			free(buffer);
		}else{
			iprintf("OK!\n");
			fileNames[nFiles]=fileName;
			nFiles++;
			free(buffer);
		}
	}else{
		iprintf(" Canceled.\n");
	}

	PAD_Pause();
	
	return 0;
}

char* buildNewFileName(){
	char* fileName;
	
	int hours, seconds, minutes, day, month, year;
	time_t unixTime = time(NULL);
	struct tm* timeStruct = gmtime((const time_t *)&unixTime);

	hours = timeStruct->tm_hour;
	minutes = timeStruct->tm_min;
	seconds = timeStruct->tm_sec;
	day = timeStruct->tm_mday;
	month = timeStruct->tm_mon;
	year = timeStruct->tm_year +1900;

	fileName=(char*)malloc(64);
	sprintf(fileName, "%s_%d%02i%02i_%02i%02i%02i.sav", gameTitle, year, month+1, day, hours, minutes, seconds);
	return fileName;
}

int hwRestoreGBA(const char* fileName){
	int i, ret=0;
	char filePath[256];
	sprintf(filePath, SAV_PATH "/%s", fileName);

	u8* buffer=NULL;

	// Show some info
	CONSOLE_PrintHeader(1);
	iprintf(" Restore savegame from:\n");
	iprintf(" %s\n\n", fileName);

	if(PAD_Ask()){
		ret=FAT_Read_File(filePath, (void**)&buffer);
		if(ret==-1){
			iprintf(" ERROR!\nFile not found.\n");
		}else if(ret!=saveSize){
			iprintf(" ERROR!\nSizes don't match.\n");
		}else{
			//iprintf(" (%d bytes read)\n", ret);
			//__print_Header_Info(buffer);
			//iprintf(" Ready to restore card savedata.\n");

			if (saveType==3){
				iprintf(" Formatting... ");
				auxspi_erase();
				iprintf(" OK!\n");
			}

			iprintf(" Writing to flash chip... ");

			//Copy savedata from byte buffer
			for(i=0; i<saveSize; i+=BLOCKSIZE)
				auxspi_write_data(i, (u8*)&buffer[i], BLOCKSIZE, saveType);
	
			iprintf("OK!\n");
		}

	}else{
		iprintf(" Canceled.\n");
	}

	PAD_Pause();

	return ret;
}

/*void hwEraseGBA(){
	int i, size;
	uint8 type;
	char filename[128];

	size=auxspi_save_size();
	type=auxspi_save_type();

	sprintf(filename, "/%s.sav", gameTitle);
	
	// Show some info
	CONSOLE_PrintHeader(1);


	iprintf("Ready to format card savedata.\n");
	PAD_Pause();
	iprintf("Formatting flash chip... ");
	if (type==3){
		auxspi_erase();
	}

	//Copy savedata from byte buffer
	for(i=0; i<size; i++)
		auxspi_write_data(i, 0, 1, type);

	iprintf("OK!\n");
	PAD_Pause();
}*/







int __showMenu(){
	int i, j, first, ret=0;

	first=0;
	for(;;){
		CONSOLE_SetColor(DARK_WHITE);
		CONSOLE_PrintHeader(1);
		CONSOLE_SetColor(GREY);
		
		for(i=first; i<first+MENU_ITEMS && i<nFiles+1; i++){
			if(i==ret){
				CONSOLE_SetColor(YELLOW);
			}else{
				CONSOLE_SetColor(GREY);
			}
			if(i==0){
				iprintf(" <Backup savegame>\n");
			}else{
				if(strlen(fileNames[i-1])<30){
					iprintf(" %s\n", fileNames[i-1]);
				}else{
					iprintf(" ");
					for(j=0; j<27; j++){
						iprintf("%c", fileNames[i-1][j]);
					}
					iprintf("...\n");
				}
			}
		}
		CONSOLE_SetColor(WHITE);
		//iprintf(" first: %d  ret: %d\n", first, ret);
		iprintf("\n\n [A] ");
		if(ret==0){
			iprintf("Backup savegame");
		}else{
			iprintf("Restore savegame");
		}
		
		swiWaitForVBlank();

		scanKeys();
		if(keysDown() & KEY_UP){
			ret--;
			if(ret==-1)
				ret=nFiles;
		}else if(keysDown() & KEY_DOWN){
			ret++;
			if(ret==nFiles+1)
				ret=0;
		}else if(keysDown() & KEY_LEFT){
			ret-=MENU_ITEMS;
			if(ret<0)
				ret=0;
		}else if(keysDown() & KEY_RIGHT){
			ret+=MENU_ITEMS;
			if(ret>nFiles+1)
				ret=nFiles;
		}else if(keysDown() & KEY_A){
			break;
		}
		while(ret<first){
			first--;
		}
		while(ret>=first+MENU_ITEMS){
			first++;
		}
	}
	return ret;
}

int main(int argc, char **argv){
	int option;

	consoleDemoInit();

	//Set background to white
	//BG_PALETTE_SUB[0] = RGB15(28,30,31);
	//Set text to red
	//BG_PALETTE_SUB[255] = RGB15(31,0,0);


	CONSOLE_PrintHeader(0);

	if(swap_cart()){

		if(FAT_InitSLOT2()){
			FAT_MakeDir(SAV_PATH);
			nFiles=FAT_ReadDir(SAV_PATH, fileNames);
			if(nFiles>=0){
				for(;;){
					option=__showMenu();
					if(option==0){
						hwBackupGBA(buildNewFileName());
					}else{
						hwRestoreGBA(fileNames[option-1]);
					}
				}
			}else{
				iprintf("ERROR: I can't open %s\n", SAV_PATH);
			}

		}else{
			iprintf("ERROR initialising FAT.\n");
			iprintf("  Incompatible SLOT-2 cart?\n");
			iprintf("  No valid DLDI patch detected?\n");
		}
	}else{
		iprintf(" ERROR: I can't read card.");
	}


	iprintf("\n\n (turn off your NDS)");
	while(1) {
		swiWaitForVBlank();
	}

	return 0;
}
