package org.pictods;

import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Properties;
import java.util.Vector;
import java.util.zip.Deflater;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageWriteParam;
import javax.imageio.stream.ImageOutputStream;

import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.util.FileComparator;
import org.util.IProgressMonitor;
import org.util.Log;
import org.util.Status;
import org.util.StreamGobbler;
import org.util.Zip;

//import com.adobe.acrobat.PDFDocument;
import com.sun.imageio.plugins.jpeg.JPEGImageWriter;

public class Pictods {
	
	private File tmpDir = new File("../bin/pictodstmp");
	private static final int BUFFER = 256;
	private static PictodsSettings pictodsSettings;
	private static final String VERSION = "210";
	private static final String EOL = "\r\n";
	
	public Pictods() {
		this(new PictodsSettings());
	}

	public Pictods(PictodsSettings pathSet) {
		pictodsSettings = pathSet;
//		Log.setOutputStream(System.out);
		try {
			Log.setOutputStream(new FileOutputStream("pictodslog.txt"));
		} catch (FileNotFoundException e) {
			Log.debug(e);
		}
		Log.debug = true;
	}

	public static void main(String[] args) {
		String param;
		File configFile;
		NDSSettings ndsSettings = new NDSSettings();
		UserSettings userSettings = new UserSettings();
		PictodsSettings tmpPictodsSettings = new PictodsSettings();;
		Pictods instance = new Pictods();
		
		if (args.length == 0) {
			System.out.println(
					"\nWelcome to PictoDS command line !\n\nexample of command line:\n\n" +
					"a simple one:\n" + 
					"java -cp org.pictods.jar org.pictods.Pictods c:\\test\\test.cbz\nIn this case the conf and properties files will be the default ones.\n\n" +
					"a litle bit more complex:\n" +
					"java -cp org.pictods.jar org.pictods.Pictods -userconf:c:\\ComicBookDS.default.ini" +
					" -pictodsconf:c:\\pictods.properties" +
					" -ndsconf:c:\\ComicBookDS_book.test1.ini c:\\test\\test1.cbz" +
					" -ndsconf:c:\\ComicBookDS_book.test2.ini c:\\test\\test2.cbz");
			return;
		}
		
		for (int i = 0; i < args.length; i++) {
			if (args[i].toLowerCase().startsWith("-ndsconf:")) {
				param = args[i].substring("-ndsconf:".length());
				configFile = new File(param);
				if (configFile.exists()) {
					instance.loadNdsSettings(param, ndsSettings);
				}
				else {
					System.out.println(configFile + " not found. Using default one.");
				}
			}
			else if (args[i].toLowerCase().startsWith("-userconf:")) {
				param = args[i].substring("-userconf:".length());
				configFile = new File(param);
				if (configFile.exists()) {
					instance.loadUserSettings(param, userSettings);
				}
				else {
					System.out.println(configFile + " not found. Using default one.");
				}
			}
			else if (args[i].toLowerCase().startsWith("-pictodsconf:")) {
				param = args[i].substring("-pictodsconf:".length());
				configFile = new File(param);
				if (configFile.exists()) {
					tmpPictodsSettings = instance.loadPictoDSSettings(param);
				}
				else {
					System.out.println(configFile + " not found. Using default one.");
				}
			}
			else {
				ndsSettings = new NDSSettings(ndsSettings, getOutputName(args[i]));
				ndsSettings.path = tmpPictodsSettings.outputPath + "/" + clearFileName(ndsSettings.name) + ".cbds";
				instance.fileToNds(args[i], ndsSettings.path, ndsSettings, userSettings, tmpPictodsSettings, new StubProgressMonitor(), null);
			}
		}
	}
	
	public static final String getOutputName(String filename) {
		filename = filename.replace('\\', '/');
		int end = filename.lastIndexOf('/');
		if (end != -1)
			filename = filename.substring(end + 1);
		end = filename.lastIndexOf('.');
		if (end != -1)
			filename = filename.substring(0, end);
		return filename;
	}

	public static final String clearFileName(String str) {
		str = str.replace('/', ' ');
		str = str.replace('\\', ' ');
		str = str.replace('<', ' ');
		str = str.replace('>', ' ');
		str = str.replace('*', ' ');
		str = str.replace('?', ' ');
		str = str.replace(':', ' ');
		str = str.replace('|', ' ');
		str = str.replace('"', ' ');
		return str;
	}

	public Status imagesToNds(File dir, NDSSettings set, IProgressMonitor monitor, boolean recursive) {
		Log.debug("enter imagesToNds with: " + dir.getAbsolutePath());
		Status result;
		monitor.setSelection(33);
		result = createResizedImages(dir, set, monitor, recursive);
		if (result.code == Status.ERROR)
			return result;
		if (monitor.isCanceled())
			return Status.OK_STATUS;
		monitor.setSelection(66);
		// create cbds.ini and then zip to cbds file
		try {
			packageCBDS(set);
		} catch (IOException e) {
			Log.debug(e);
			result = new Status(e, e.getMessage(), Status.ERROR);
		}
		monitor.setSelection(100);
		return result;
	}
	
	private void packageCBDS(NDSSettings set) throws IOException {
		saveNdsSettings(tmpDir.getAbsolutePath() + "/ComicBookDS_book.ini", set);
		zipTmpDir(set.path);
	}
	
	private void zipTmpDir(String zipFileName) throws IOException {
        Zip.zipDir(tmpDir, zipFileName, Deflater.NO_COMPRESSION); 
	}
	
	private Status imageDirToNds(String dirname, NDSSettings set, IProgressMonitor monitor) {
		File file = new File(dirname);
		if (set.name == null || set.name.length() == 0)
			set.name = file.getName();
		return imagesToNds(file, set, monitor, false);
	}
	
	private Status zipToNds(String zipname, NDSSettings set, IProgressMonitor monitor) {
		File[] files = extractZipToTmpDirectory(zipname, monitor);
		if (files != null && files.length > 0) {
			if (set.name == null || set.name.length() == 0) {
				set.name = new File(zipname).getName();
				int end = set.name.lastIndexOf('.');
				set.name = set.name.substring(0, end);
			}
			return imagesToNds(tmpDir, set, monitor, true);
		}
		return Status.OK_STATUS;
	}
	
	private Status rarToNds(String rarname, NDSSettings set, IProgressMonitor monitor) {
		Status result;
		result = extractRarToTmpDirectory(rarname, monitor);
		if (result.code == Status.OK) {
			Log.debug("extraction seems ok");			
			if (set.name == null || set.name.length() == 0) {
				set.name = new File(rarname).getName();
				int end = set.name.lastIndexOf('.');
				set.name = set.name.substring(0, end);
			}
			return imagesToNds(tmpDir, set, monitor, true);
		}
		return result;
	}
	
	private Status pdfToNds(String pdfname, NDSSettings set, IProgressMonitor monitor) {
		Status result;
		result = extractPdfToTmpDirectory(pdfname, monitor);
		if (result.code == Status.OK) {
			Log.debug("extraction seems ok");			
			if (set.name == null || set.name.length() == 0) {
				set.name = new File(pdfname).getName();
				int end = set.name.lastIndexOf('.');
				set.name = set.name.substring(0, end);
			}
			return imagesToNds(tmpDir, set, monitor, true);
		}
		return result;
	}

	public Status fileToNds(
			String filename, String destname, NDSSettings settings, UserSettings usettings, PictodsSettings psettings, 
			IProgressMonitor monitor, IPictoDSJobListener listener) {
		Status result;
		File file = new File(filename);
		String tmp;
		if (file.exists()) {
			tmp = filename.toLowerCase();
			monitor.setMaximum(100);
			monitor.setSelection(0);
			createTmpDirectory();
			if (file.isDirectory()) {
				result = imageDirToNds(filename, settings, monitor);
			}
			else if (isZipFile(tmp)) {
				result = zipToNds(filename, settings, monitor);
			}
			else if (isRarFile(tmp)) {
				result = rarToNds(filename, settings, monitor);
			}
			else if (isPdfFile(tmp)) {
				result = pdfToNds(filename, settings, monitor);
			}
			else {
				result = new Status(null, "'" + filename + "' format not supported", Status.ERROR);
			}
			clean();
		}
		else {
			result =  new Status(null, "'" + filename + "' does not exist", Status.ERROR);
		}
		
		if (listener != null)
			listener.jobFinished(result);
		return result;
	}

	private File[] extractZipToTmpDirectory(String zipfilename, IProgressMonitor monitor) {
		Vector<File> result = new Vector<File>(10);
		byte[] data = new byte[BUFFER];
		int count;
		try {
			BufferedOutputStream dest = null;
			ZipFile zis = new ZipFile(zipfilename);
			ZipEntry entry;
			InputStream stream;
			Enumeration enu = zis.getEntries();
			float i = 0.0f, nbFiles = 0.0f;
			while (enu.hasMoreElements()) {
				enu.nextElement();
				nbFiles++;
			}
			enu = zis.getEntries();
			while (enu.hasMoreElements()) {
				entry = (ZipEntry) enu.nextElement();
				if (entry.isDirectory()) {
					continue;
				}
				stream = zis.getInputStream(entry);
				String filename = entry.getName();
				File file = createFile(filename);
				if (file == null)
					continue;
				result.add(file);
				dest = new BufferedOutputStream(new FileOutputStream(file), BUFFER);

				if (monitor.isCanceled())
					return null;
				monitor.setSelection((int) ((i/nbFiles)*33.0f));
				monitor.setText("Extracting " + filename);

				while ((count = stream.read(data, 0, BUFFER)) != -1) {
					dest.write(data, 0, count);
				}
				dest.flush();
				dest.close();
				stream.close();
				i++;
			}
		} catch (IOException e) {
			Log.debug(e);
			return null;
		}

		return (File[]) result.toArray(new File[result.size()]);
	}
	
	private Status extractRarToTmpDirectory(String rarfilename, IProgressMonitor monitor) {
		try {
			Log.debug("enter extractRarToTmpDirectory with: " + rarfilename);
			if (pictodsSettings.unrarPath.equals("")) {
				return new Status(null, "Can't UNRAR file: " + rarfilename + ".\nPlease, define a command line in pictods path option to enable RAR functions.", Status.ERROR);
			}
			monitor.setText("Extracting " + rarfilename);
			String cmd = replaceAll(pictodsSettings.unrarPath, "%filename", rarfilename);
			cmd = replaceAll(cmd, "%outputdirectoryname", tmpDir.getName());
//			String cmd = "\"" + pictodsSettings.unrarPath + "\" " +
//				unrarOptions + " " +
//				"\"" + rarfilename + "\" " +
//				tmpDir.getName(); 
			Process pr = Runtime.getRuntime().exec(cmd, null, tmpDir.getParentFile()); 
			Log.debug("exec: " + cmd + " from " + tmpDir.getParentFile());
            StreamGobbler errorGobbler = new 
                StreamGobbler(pr.getErrorStream(), "ERROR");            
            StreamGobbler inputGobbler = new 
                StreamGobbler(pr.getInputStream(), "INPUT");
            errorGobbler.start();
            inputGobbler.start();
            Log.debug("exit extractRarToTmpDirectory");
            return pr.waitFor() == 0? Status.OK_STATUS:new Status(null, "Unrar could not extract: " + rarfilename, Status.ERROR);
		} catch (Exception e) {
			Log.debug(e);
			return new Status(e, e.getMessage(), Status.ERROR);
		}
	}

	private Status extractPdfToTmpDirectory(String pdffilename, IProgressMonitor monitor) {
		try {
			Log.debug("enter extractPdfToTmpDirectory with: " + pdffilename);
			if (pictodsSettings.pdf2jpgPath.equals("")) {
				return new Status(null, "Can't extract pdf file: " + pdffilename + ".\nPlease, define a command line in pictods path option to enable PDF functions.", Status.ERROR);
			}
			monitor.setText("Extracting " + pdffilename);
			String cmd = replaceAll(pictodsSettings.pdf2jpgPath, "%filename", pdffilename);
			cmd = replaceAll(cmd, "%outputdirectoryname", tmpDir.getName());
			Process pr = Runtime.getRuntime().exec(cmd, null, tmpDir.getParentFile()); 
			Log.debug("exec: " + cmd + " from " + tmpDir.getParentFile());
            StreamGobbler errorGobbler = new 
                StreamGobbler(pr.getErrorStream(), "ERROR");            
            StreamGobbler inputGobbler = new 
                StreamGobbler(pr.getInputStream(), "INPUT");
            errorGobbler.start();
            inputGobbler.start();
            Log.debug("exit extractPdfToTmpDirectory");
            return pr.waitFor() == 0? Status.OK_STATUS:new Status(null, "PDF could not extract: " + pdffilename, Status.ERROR);
		} catch (Exception e) {
			Log.debug(e);
			return new Status(e, e.getMessage(), Status.ERROR);
		}
	}
	
	private String replaceAll(String original, String token, String replace) {
		int i = 0;
		while (true) {
			int idx = original.indexOf(token, i);
			i = idx;
			if (idx == -1)
				return original;
			original = original.substring(0, idx) + replace + original.substring(idx + token.length());
		}
	}

	private File createFile(String filename) throws IOException {
		File result;
		int idx = filename.lastIndexOf('/');
		if (idx == -1) {
			result = new File(tmpDir.getCanonicalPath() + '/' + filename);
		}
		else {
			if (idx + 1 == filename.length()) // directory
				result = null;
			else {
				result = new File(tmpDir.getCanonicalPath() + filename.substring(idx));
			}
		}
		return result;
	}
	
	private BufferedImage createResizedImage(BufferedImage src, int width, int height, int maxsize, boolean keepratio) {
		BufferedImage dest;

		int srcWidth = src.getWidth();
		int srcHeight = src.getHeight();
		if (srcWidth > width || srcHeight > height) {
			if (keepratio) {
				double ratioW, ratioH;
				ratioW = (double) srcWidth / (double) width;
				ratioH = (double) srcHeight / (double) height;
				if (ratioW > ratioH) {
					width = (int) ((double) srcWidth / ratioW);
					height = (int) ((double) srcHeight / ratioW);
				}
				else {
					width = (int) ((double) srcWidth / ratioH);
					height = (int) ((double) srcHeight / ratioH);
				}
			}
		}
		else {
			width = srcWidth;
			height = srcHeight;
		}
		if (maxsize != 0 && width * height > maxsize) {
			double sqrt = Math.sqrt(
					((double) maxsize)
					/ 
					((double) (srcWidth * srcHeight)));
			width = (int) (((double) width) * sqrt);
			height = (int) (((double) height) * sqrt);
		}
		
		Image img;
		if (width != srcWidth || height != srcHeight) {
			img = src.getScaledInstance(width, height, Image.SCALE_AREA_AVERAGING);
		}
		else {
			img = src.getSubimage(0, 0, width, height);
		}
		dest = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
		Graphics2D g = (Graphics2D) dest.getGraphics();
		g.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
		g.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY);
		g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		g.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
		g.drawImage(img, 0, 0, width, height, null);
		g.dispose();
		return dest;
	}
	
	private void createFile(String path, String content) throws IOException {
		File out = new File(path);
		out.createNewFile();
		FileOutputStream fout = new FileOutputStream(out);
		fout.write(content.getBytes());
		fout.write(EOL.getBytes());
		fout.close();
	}
	
	private Status createResizedImages(File dir, NDSSettings ndsSettings, IProgressMonitor monitor, boolean recursive) {
		File[] children = dir.listFiles();
		Log.debug("enter createResizedImages");

		Arrays.sort(children, new FileComparator());
		File child = null;
		File outputI, outputON, outputOR, outputTN, outputTR, outputN;
		try {
			outputI = new File(tmpDir.getCanonicalPath() + "/IMAGE");
			outputI.mkdir();
			outputON = new File(tmpDir.getCanonicalPath() + "/SMALL_N");
			outputON.mkdir();
			outputOR = new File(tmpDir.getCanonicalPath() + "/SMALL_R");
			outputOR.mkdir();
			outputTN = new File(tmpDir.getCanonicalPath() + "/THMB_N");
			outputTN.mkdir();
			outputTR = new File(tmpDir.getCanonicalPath() + "/THMB_R");
			outputTR.mkdir();
			outputN = new File(tmpDir.getCanonicalPath() + "/NAME");
			outputN.mkdir();
		} catch (IOException e) {
			Log.debug(e);
			return new Status(e, e.getMessage(), Status.ERROR);
		}

		AffineTransformOp affineTransOp;

		int originalWidth, originalHeight, nbFiles = children.length;
		
	    Iterator it = ImageIO.getImageWritersByFormatName("jpeg");
	    JPEGImageWriter iw = null;
	    ImageWriteParam param = null;
	    ImageOutputStream stream;
	    String filename;
	    while (it.hasNext()) {
			Object element = (Object) it.next();
			if (element instanceof JPEGImageWriter) {
				iw = (JPEGImageWriter) element;
				param = iw.getDefaultWriteParam();
				param.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
			}
		}
	    if (iw == null) {
	    	Log.debug("no jpeg writer found");
	    	return new Status(null, "No jpeg writer found", Status.ERROR);
	    }
		int pagenumber = 1;
		for (int i = 0; i < nbFiles; i++) {
			child = children[i];
			filename = child.getName();
			Log.debug("loop on " + filename);

			if (monitor.isCanceled())
				return Status.OK_STATUS;
		
			String outfile = pagenumber + ".jpg";
			if (child.isDirectory())
				continue;
			BufferedImage originalBuffer;
			BufferedImage originalRotatedBuffer;

			try {
				originalBuffer = ImageIO.read(child);
			} catch (Exception e) {
				if (dir.equals(tmpDir))
					child.delete();
				Log.debug("Could not read: " + child);
				Log.debug(e);
				continue;
			}
			if (dir.equals(tmpDir))
				child.delete();
			
			if (originalBuffer == null) {
				Log.debug("Could not read: " + child);
				continue;
			}
			
			// create name file
			try {
				String name = filename;
				int idx = filename.lastIndexOf('.');
				if (idx != -1 && filename.length() - idx <= 5) {
					name = filename.substring(0, idx);
				}
				createFile(outputN.getAbsolutePath() + "/" + pagenumber + ".txt", name);
			} catch (IOException e) {
				Log.debug("Can't create " + outputN.getAbsolutePath() + "/" + pagenumber);
			}
			pagenumber++;
			
			monitor.setText("Resizing " + filename);
			monitor.setSelection((int) (33.0f + (((float) i) / ((float) nbFiles))*33.0f));

			originalWidth = originalBuffer.getWidth();
			originalHeight = originalBuffer.getHeight();

			BufferedImage src = new BufferedImage(originalWidth, originalHeight, BufferedImage.TYPE_INT_RGB);
			Graphics2D g = (Graphics2D) src.getGraphics();
			g.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
			g.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY);
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			g.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
			g.drawImage(originalBuffer, 0, 0, originalWidth, originalHeight, null);
			g.dispose();

			// create rotated image
			originalRotatedBuffer = new BufferedImage(originalHeight, originalWidth, BufferedImage.TYPE_INT_RGB);
			affineTransOp = new AffineTransformOp(	
					AffineTransform.getRotateInstance(Math.toRadians(90), ((double) originalHeight) / 2.0f, ((double) originalHeight) / 2.0f),
					AffineTransformOp.TYPE_BILINEAR);
			affineTransOp.filter(src, originalRotatedBuffer);

			// create base image
			param.setCompressionQuality(((float) ndsSettings.iQuality) / 100.0f);
			BufferedImage baseI = createResizedImage(originalBuffer, ndsSettings.iWidth, ndsSettings.iHeight, ndsSettings.iSize, true);
			try {
				//ImageIO.write(baseI, "jpeg", new File(outputI.getAbsolutePath() + "/" + filename));
				stream = ImageIO.createImageOutputStream(new File(outputI.getAbsolutePath() + "/" + outfile));
				iw.setOutput(stream);
				iw.write(null, new IIOImage(baseI, null, null), param);
				stream.close();
			} catch (IOException e) {
				Log.debug(e);
			    continue;
			}
			
			// create overview image
			param.setCompressionQuality(((float) ndsSettings.oQuality) / 100.0f);
			BufferedImage on = createResizedImage(baseI, ndsSettings.oWidth, ndsSettings.oHeight, ndsSettings.oSize, true);
			try {
			    //ImageIO.write(on, "jpeg", new File(outputON.getAbsolutePath() + "/" + filename));
				stream = ImageIO.createImageOutputStream(new File(outputON.getAbsolutePath() + "/" + outfile));
				iw.setOutput(stream);
				iw.write(null, new IIOImage(on, null, null), param);
				stream.close();
			} catch (IOException e) {
				Log.debug(e);
			    continue;
			}
			BufferedImage or = createResizedImage(originalRotatedBuffer, ndsSettings.oWidth, ndsSettings.oHeight, ndsSettings.oSize, true);
			try {
			    //ImageIO.write(or, "jpeg", new File(outputOR.getAbsolutePath() + "/" + filename));
				stream = ImageIO.createImageOutputStream(new File(outputOR.getAbsolutePath() + "/" + outfile));
				iw.setOutput(stream);
				iw.write(null, new IIOImage(or, null, null), param);
				stream.close();
			} catch (IOException e) {
				Log.debug(e);
			    continue;
			}

			// create thumnails image
			param.setCompressionQuality(((float) ndsSettings.thQuality) / 100.0f);
			BufferedImage tn = createResizedImage(on, ndsSettings.thWidth, ndsSettings.thHeight, ndsSettings.thSize, true);
			try {
			    //ImageIO.write(tn, "jpeg", new File(outputTN.getAbsolutePath() + "/" + filename));
				stream = ImageIO.createImageOutputStream(new File(outputTN.getAbsolutePath() + "/" + outfile));
				iw.setOutput(stream);
				iw.write(null, new IIOImage(tn, null, null), param);
				stream.close();
			} catch (IOException e) {
				Log.debug(e);
			    continue;
			}
			BufferedImage tr = createResizedImage(or, ndsSettings.thWidth, ndsSettings.thHeight, ndsSettings.thSize, true);
			try {
			    //ImageIO.write(tr, "jpeg", new File(outputTR.getAbsolutePath() + "/" + filename));
				stream = ImageIO.createImageOutputStream(new File(outputTR.getAbsolutePath() + "/" + outfile));
				iw.setOutput(stream);
				iw.write(null, new IIOImage(tr, null, null), param);
				stream.close();
			} catch (IOException e) {
				Log.debug(e);
			    continue;
			}
		}
		ndsSettings.nbPages = pagenumber - 1;
		return Status.OK_STATUS;
	}
	
	private void createTmpDirectory() {
		if (tmpDir.exists()) {
			clean();
		}
		tmpDir.mkdir();
	}

	private boolean isZipFile(String filename) {
		return filename.endsWith(".zip") || filename.endsWith(".cbz");
	}
	
	private boolean isRarFile(String filename) {
		return filename.endsWith(".rar") || filename.endsWith(".cbr");
	}

	private boolean isPdfFile(String filename) {
		return filename.endsWith(".pdf");
	}

	private void deleteDir(File dir) {
		File[] children = dir.listFiles();
		if (children == null)
			return;
		File child;
		for (int i = 0; i < children.length; i++) {
			child = children[i]; 
			if (child.isDirectory())
				deleteDir(child);
			child.delete();
		}
	}
	
	public void clean() {
		deleteDir(tmpDir);
		tmpDir.delete();
	}
	
	private static final class StubProgressMonitor implements IProgressMonitor {

		public void cancel() {
		}

		public int getSelection() {
			return 0;
		}

		public boolean isCanceled() {
			return false;
		}

		public void setMaximum(int max) {
		}

		public void setSelection(int sel) {
		}

		public void setText(String text) {
		}
		
	}
	
	public void loadNdsSettings(String file, NDSSettings ndsSettings) {
		FileInputStream fis;
		Properties props;
		try {
			fis = new FileInputStream(file);
			props = new Properties();
			props.load(fis);
		} catch (IOException e) {
			Log.debug(e);
			return;
		}
		try {
			ndsSettings.name = (String) props.get("CbCredits1");
			ndsSettings.author = (String) props.get("CbCredits2");
			ndsSettings.provider = (String) props.get("CbCredits3");
			ndsSettings.manga = 
				Integer.parseInt((String) props.get("LeftToRight")) == 0?1:0;
			ndsSettings.nbPages = Integer.parseInt((String) props.get("NbPages"));
			ndsSettings.iHeight = Integer.parseInt((String) props.get("iHeight"));
			ndsSettings.iQuality = Integer.parseInt((String) props.get("iQuality"));
			ndsSettings.iWidth = Integer.parseInt((String) props.get("iWidth"));
			ndsSettings.iSize = Integer.parseInt((String) props.get("iSize"));
			ndsSettings.oHeight = Integer.parseInt((String) props.get("oHeight"));
			ndsSettings.oQuality = Integer.parseInt((String) props.get("oQuality"));
			ndsSettings.oWidth = Integer.parseInt((String) props.get("oWidth"));
			ndsSettings.oSize = Integer.parseInt((String) props.get("oSize"));
			ndsSettings.thHeight = Integer.parseInt((String) props.get("thHeight"));
			ndsSettings.thQuality = Integer.parseInt((String) props.get("thQuality"));
			ndsSettings.thWidth = Integer.parseInt((String) props.get("thWidth"));
			ndsSettings.thSize = Integer.parseInt((String) props.get("thSize"));
		} catch (Exception e) {
			Log.debug(e);
		}
		try {
			fis.close();
		} catch (IOException e) {
			Log.debug(e);
		}
	}

	public void saveNdsSettings(String file, NDSSettings ndsSettings) {
		FileOutputStream fos;
		
		try {
			fos = new FileOutputStream(file);
		} catch (FileNotFoundException e) {
			Log.debug(e);
			return;
		}

		try {
			fos.write(COMMENT_NDSINI_INTRO.getBytes());
			if (ndsSettings.name != null) {
				fos.write(COMMENT_NDSINI_CBCREDITS1.getBytes());
				fos.write(("CbCredits1 = " + ndsSettings.name + "\r\n").getBytes());
			}
			fos.write(COMMENT_NDSINI_CBCREDITS2.getBytes());
			fos.write(("CbCredits2 = " + ndsSettings.author + "\r\n").getBytes());
			fos.write(COMMENT_NDSINI_CBCREDITS3.getBytes());
			fos.write(("CbCredits3 = " + ndsSettings.provider + "\r\n").getBytes());
			fos.write(COMMENT_NDSINI_LEFTTORIGHT.getBytes());
			fos.write(("LeftToRight = " + (ndsSettings.manga == 0?"1":"0") + "\r\n").getBytes());
			fos.write(COMMENT_NDSINI_NBPAGES.getBytes());
			fos.write(("NbPages = " + ndsSettings.nbPages + "\r\n").getBytes());
			fos.write(COMMENT_NDSINI_VERSION.getBytes());
			fos.write("Version = 200\r\n".getBytes());
			fos.write(("iHeight = " + ndsSettings.iHeight + "\r\n").getBytes());
			fos.write(("iQuality = " + ndsSettings.iQuality + "\r\n").getBytes());
			fos.write(("iSize = " + ndsSettings.iSize + "\r\n").getBytes());
			fos.write(("iWidth = " + ndsSettings.iWidth + "\r\n").getBytes());
			fos.write(("oHeight = " + ndsSettings.oHeight + "\r\n").getBytes());
			fos.write(("oQuality = " + ndsSettings.oQuality + "\r\n").getBytes());
			fos.write(("oSize = " + ndsSettings.oSize + "\r\n").getBytes());
			fos.write(("oWidth = " + ndsSettings.oWidth + "\r\n").getBytes());
			fos.write(("thHeight = " + ndsSettings.thHeight + "\r\n").getBytes());
			fos.write(("thQuality = " + ndsSettings.thQuality + "\r\n").getBytes());
			fos.write(("thSize = " + ndsSettings.thSize + "\r\n").getBytes());
			fos.write(("thWidth = " + ndsSettings.thWidth + "\r\n").getBytes());
		} catch (IOException e) {
			Log.debug(e);
		}
		try {
			fos.close();
		} catch (IOException e) {
			Log.debug(e);
		}
	}
	
	public void loadUserSettings(String file, UserSettings userSettings) {
		FileInputStream fis;
		Properties props;
		try {
			fis = new FileInputStream(file);
			props = new Properties();
			props.load(fis);
		} catch (IOException e) {
			Log.debug(e);
			return;
		}
		try {
			userSettings.scrollIncrement = Integer.parseInt((String) props.get("ScrollIncrement"));
			userSettings.zoomIncrement = Integer.parseInt((String) props.get("ZoomIncrement"));
			userSettings.iconAnim = Integer.parseInt((String) props.get("IconsAnimation"));
			userSettings.iconLock = Integer.parseInt((String) props.get("LockIcons"));
			userSettings.landscape = Integer.parseInt((String) props.get("LandscapeMode"));
			userSettings.lefty = Integer.parseInt((String) props.get("LeftHandedMode"));
			userSettings.screenspace = Integer.parseInt((String) props.get("ReflectScreenSpace"));
			userSettings.scrollChange = Integer.parseInt((String) props.get("ScrollCanChangePage"));
			userSettings.transitionAnim = Integer.parseInt((String) props.get("TransitionsAnimation"));
			userSettings.soundsEffect = Integer.parseInt((String) props.get("SoundEffects"));
			userSettings.startupSound = Integer.parseInt((String) props.get("StartupSound"));
			userSettings.resizeFilter = Integer.parseInt((String) props.get("UseResizeFilter"));
			userSettings.waitForKeyRelease = Integer.parseInt((String) props.get("WaitForKeyRelease"));
			userSettings.progressiveScroll = Integer.parseInt((String) props.get("ProgressiveScroll"));
		} catch (Exception e) {
			Log.debug(e);
		}
		try {
			fis.close();
		} catch (IOException e) {
			Log.debug(e);
		}
	}
	
	public static void SaveUserSettings(String path, UserSettings userSettings) {
		FileOutputStream fos;
		
		try {
			fos = new FileOutputStream(path);
		} catch (FileNotFoundException e) {
			Log.debug(e);
			return;
		}

		try {
			fos.write(COMMENT_USERINI_INTRO.getBytes());
			fos.write(COMMENT_USERINI_SCROLLINCR.getBytes());
			fos.write(("ScrollIncrement = " + userSettings.scrollIncrement + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_ZOOMINCR.getBytes());
			fos.write(("ZoomIncrement = " + userSettings.zoomIncrement + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_ICONSANIM.getBytes());
			fos.write(("IconsAnimation = " + userSettings.iconAnim + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_LOCKICONS.getBytes());
			fos.write(("LockIcons = " + userSettings.iconLock + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_LANDSCAPE.getBytes());
			fos.write(("LandscapeMode = " + userSettings.landscape + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_LEFTY.getBytes());
			fos.write(("LeftHandedMode = " + userSettings.lefty + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_SCREENSPACE.getBytes());
			fos.write(("ReflectScreenSpace = " + userSettings.screenspace + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_SCROLLCHANGE.getBytes());
			fos.write(("ScrollCanChangePage = " + userSettings.scrollChange + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_TRANSITIONANIM.getBytes());
			fos.write(("TransitionsAnimation = " + userSettings.transitionAnim + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_SOUNDSFX.getBytes());
			fos.write(("SoundEffects = " + userSettings.soundsEffect + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_STARTUPSOUND.getBytes());
			fos.write(("StartupSound = " + userSettings.startupSound + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_RESIZEFILTER.getBytes());
			fos.write(("UseResizeFilter = " + userSettings.resizeFilter + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_WAITFORKEYRELEASE.getBytes());
			fos.write(("WaitForKeyRelease = " + userSettings.waitForKeyRelease + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_PROGRESSIVESCROLL.getBytes());
			fos.write(("ProgressiveScroll = " + userSettings.progressiveScroll + "\r\n").getBytes());
			fos.write(COMMENT_USERINI_KEYS.getBytes());
		} catch (IOException e) {
			Log.debug(e);
		}
		try {
			fos.close();
		} catch (IOException e) {
			Log.debug(e);
		}
	}

	public static void SaveUserSettings(UserSettings userSettings) {
		SaveUserSettings(pictodsSettings.outputPath + "/ComicBookDS.ini", userSettings);
	}
	
	public PictodsSettings loadPictoDSSettings(String file) {
		FileInputStream fis;
		Properties props;
		try {
			fis = new FileInputStream(file);
			props = new Properties();
			props.load(fis);
		} catch (IOException e) {
			Log.debug(e);
			return pictodsSettings;
		}
		try {
			pictodsSettings.inputPath = (String) props.get("pictodsSettings.inputPath");
			if (pictodsSettings.inputPath == null)
				pictodsSettings.inputPath = "";
			pictodsSettings.outputPath = (String) props.get("pictodsSettings.outputPath");
			if (pictodsSettings.outputPath == null || pictodsSettings.outputPath.trim().equals("")) { 
				pictodsSettings.outputPath = new File("../cbds").getCanonicalPath();				
			}
			pictodsSettings.unrarPath = (String) props.get("pictodsSettings.unrarPath");
			pictodsSettings.explorerPath = (String) props.get("pictodsSettings.explorerPath");
			pictodsSettings.pdf2jpgPath = (String) props.get("pictodsSettings.pdf2jpgPath");
		} catch (Exception e) {
			Log.debug(e);
		}
		try {
			fis.close();
		} catch (IOException e) {
			Log.debug(e);
		}
		return pictodsSettings;
	}

	public void savePictoDSSettings(String file) {
		FileOutputStream fos;
		
		try {
			fos = new FileOutputStream(file);
		} catch (FileNotFoundException e) {
			Log.debug(e);
			return;
		}

		Properties props = new Properties();
		props.put("pictodsSettings.inputPath", pictodsSettings.inputPath);
		props.put("pictodsSettings.outputPath", pictodsSettings.outputPath);
		props.put("pictodsSettings.unrarPath", pictodsSettings.unrarPath);
		props.put("pictodsSettings.pdf2jpgPath", pictodsSettings.pdf2jpgPath);
		props.put("pictodsSettings.explorerPath", pictodsSettings.explorerPath);
		//props.put("pictodsSettings.desmumePath", pictodsSettings.desmumePath);
		props.put("VERSION", VERSION);

		try {
			props.store(fos, null);
		} catch (IOException e) {
			Log.debug(e);
		}
		try {
			fos.close();
		} catch (IOException e) {
			Log.debug(e);
		}
	}
	
	/*public static void print(
			JFrame frm,
			String strFile)
	{
		Canvas foo = new Canvas();
		frm.add(foo);
		try
		{
			PDFDocument doc = new PDFDocument(new java.io.File(strFile));
			//JobAttributes jobAttrs = new JobAttributes();

			//jobAttrs.setSides(JobAttributes.SidesType.ONE_SIDED);
			//PageAttributes pageAttrs = new PageAttributes();
//			pageAttrs.setPrintQuality(PageAttributes.PrintQualityType.DRAFT);
			//pageAttrs.setPrintQuality(PageAttributes.PrintQualityType.NORMAL);
//			pageAttrs.setPrintQuality(PageAttributes.PrintQualityType.HIGH);
			//pageAttrs.setColor(PageAttributes.ColorType.COLOR);

//			PrintJob job = Toolkit.getDefaultToolkit().getPrintJob(
//					frm,
//					"The print",
//					jobAttrs,
//					pageAttrs);

//			draw pages
			//int nFinalHeight = (int) job.getPageDimension().getHeight() - 36;
			//int nPageWidth = (int) job.getPageDimension().getWidth() - 36;
//			System.out.println("H:" + nFinalHeight +", W:"+nPageWidth);
			
		    Iterator it = ImageIO.getImageWritersByFormatName("jpeg");
		    JPEGImageWriter iw = null;
		    ImageWriteParam param = null;
		    ImageOutputStream stream;
		    while (it.hasNext()) {
				Object element = (Object) it.next();
				if (element instanceof JPEGImageWriter) {
					iw = (JPEGImageWriter) element;
					param = iw.getDefaultWriteParam();
					param.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
				}
			}
			
			frm.setVisible(true);
			for (int nFile = 0; nFile < doc.getNumPages(); nFile++)
			{
				int nScale = 2;
				int x = (int) doc.getPageSize(nFile).x*nScale;
				int y = (int) doc.getPageSize(nFile).y*nScale;
				//double nFinalWidth = doc.getPageSize(nFile).x / doc.getPageSize(nFile).y * nFinalHeight;

				int nColorModel = java.awt.image.BufferedImage.TYPE_INT_RGB;
				BufferedImage image = new BufferedImage(
						x,
						y,
						nColorModel);
				image.getGraphics().fillRect(0,0,x,y);
				com.adobe.acrobat.sidecar.AffineTransform af = new com.adobe.acrobat.sidecar.AffineTransform(nScale,0,0,nScale,0,0);

				try {
					doc.drawPage(nFile, image, af, null, frm);
					image.flush();
					stream = ImageIO.createImageOutputStream(new File("d:/toto" + nFile + ".jpg"));
					iw.setOutput(stream);
					iw.write(null, new IIOImage(image, null, null), param);
					stream.close();
				} catch (Exception e) {
					System.out.println("tototto");
					Log.debug(e);
				    continue;
				}
				
			}
			frm.setVisible(false);

			//job.end();
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
	}*/	
	
	
	private static final String COMMENT_NDSINI_INTRO = "; ComicBookDS ini file\r\n";
	private static final String COMMENT_NDSINI_VERSION = "; The compatibility version of this comic book\r\n";
	private static final String COMMENT_NDSINI_NBPAGES = "; The number of pages contained in this comic book\r\n";
	private static final String COMMENT_NDSINI_LEFTTORIGHT = "; Left To Right reading mode [RightToLeft=0,LeftToRight=1]\r\n";
	private static final String COMMENT_NDSINI_CBCREDITS1 = "; 1st line displayed in the credits (usually the ComicBook name)\r\n";
	private static final String COMMENT_NDSINI_CBCREDITS2 = "; 2nd line displayed in the credits (usually the ComicBook author)\r\n";
	private static final String COMMENT_NDSINI_CBCREDITS3 = "; 3rd line displayed in the credits (usually the ComicBook provider)\r\n";
	
	private static final String COMMENT_USERINI_INTRO = "\r\n";
	private static final String COMMENT_USERINI_LANDSCAPE = "; Set the initial Orientation to Landscape Mode [No=0,Yes=1]\r\n";
	private static final String COMMENT_USERINI_LEFTY = "; Use the Left Handed configuration [RightHanded=0,LeftHanded=1]\r\n";
	private static final String COMMENT_USERINI_SCREENSPACE = "; Reflects the screen space between the top and bottom images [Yes=1,No=0]\r\n";
	private static final String COMMENT_USERINI_RESIZEFILTER = "; Use filters to resize the image (nicer but require a second image display)\r\n";
	private static final String COMMENT_USERINI_SCROLLCHANGE = "; A scroll action (with key presses) while viewing the last image part triggers a page change [Yes=1,No=0]\r\n";
	private static final String COMMENT_USERINI_LOCKICONS = "; Locks the icons (they remain visible on screen) [Yes=1,No=0]\r\n";
	private static final String COMMENT_USERINI_TRANSITIONANIM = "; Animate the transitions [Yes=1,No=0]\r\n";
	private static final String COMMENT_USERINI_ICONSANIM = "; Animate the icons [Yes=1,No=0]\r\n";
	private static final String COMMENT_USERINI_SCROLLINCR = "; Defines the Scroll Increment in pixels\r\n";
	private static final String COMMENT_USERINI_ZOOMINCR = "; Defines the Zoom Increment (The higher the value is, the quicker the zoom will be)\r\n";
	private static final String COMMENT_USERINI_SOUNDSFX = "; Sound effects will be performed on user interaction [No=0,Yes=1]\r\n";
	private static final String COMMENT_USERINI_STARTUPSOUND = "; Plays a welcome sound (\"welcome to comicbookds\") at application startup [No=0,Yes=1]\r\n";
	private static final String COMMENT_USERINI_WAITFORKEYRELEASE = "; In thumbnails and library modes: Wait for the key to be released before switching to next element [No=0,Yes=1]\r\n";
	private static final String COMMENT_USERINI_PROGRESSIVESCROLL = "; One pixel scroll at the begining, then quicker until the max value (ScrollIncrement) [No=0,Yes=1]\r\n";
	private static final String COMMENT_USERINI_KEYS = 
		";----------------------------------------------------------------------------------\r\n" +
		"; Here are the different actions to be assigned to a keypress :\r\n" +
		";    ScrollRight\r\n" +
		";    ScrollLeft\r\n" +
		";    ScrollUp\r\n" +
		";    ScrollDown\r\n" +
		";    ZoomIn\r\n" +
		";    ZoomOut\r\n" +
		";    NoZoom\r\n" +
		";    ZoomPrev\r\n" +
		";    RightPage\r\n" +
		";    LeftPage\r\n" +
		";    UpPage\r\n" +
		";    DownPage\r\n" +
		";    RightThmbPage\r\n" +
		";    LeftThmbPage\r\n" +
		";    ChangeMode // Thmbs->Overview->Dual->\r\n" +
		";    ChangeViewMode\r\n" +
		";    ChangeThmbMode\r\n" +
		";    SetModeLibrary  // Library->prevmode->\r\n" +
		";    SetModeThumbnails\r\n" +
		";    SetModeOverview\r\n" +
		";    SetModeDualScreen\r\n" +
		";    ChangeOrientation\r\n" +
		";    ChangeHand\r\n" +
		";    ChangeReading\r\n" +
		";    ChangeScreenSpace\r\n" +
		";    ChangeLight\r\n" +
		";    SetDefaultConf\r\n" +
		";    Save\r\n" +
		";----------------------------------------------------------------------------------\r\n" +
		";RightHanded_Landscape_View\r\n" +
		"RightHanded_Landscape_View.KEY_Right  = ScrollRight\r\n" +
		"RightHanded_Landscape_View.KEY_Down   = ScrollDown\r\n" +
		"RightHanded_Landscape_View.KEY_Up     = ScrollUp\r\n" +
		"RightHanded_Landscape_View.KEY_Left   = ScrollLeft\r\n" +
		"RightHanded_Landscape_View.KEY_A      = ChangeViewMode\r\n" +
		"RightHanded_Landscape_View.KEY_B      = ChangeThmbMode\r\n" +
		"RightHanded_Landscape_View.KEY_X      = ZoomIn\r\n" +
		"RightHanded_Landscape_View.KEY_Y      = ZoomOut\r\n" +
		"RightHanded_Landscape_View.KEY_R      = ChangeOrientation\r\n" +
		"RightHanded_Landscape_View.KEY_L      = ChangeMode\r\n" +
		"RightHanded_Landscape_View.KEY_Select = ChangeLight\r\n" +
		";RightHanded_Portrait_View\r\n" +
		"RightHanded_Portrait_View.KEY_Right  = ScrollUp\r\n" +
		"RightHanded_Portrait_View.KEY_Down   = ScrollRight\r\n" +
		"RightHanded_Portrait_View.KEY_Up     = ScrollLeft\r\n" +
		"RightHanded_Portrait_View.KEY_Left   = ScrollDown\r\n" +
		"RightHanded_Portrait_View.KEY_A      = ChangeViewMode\r\n" +
		"RightHanded_Portrait_View.KEY_B      = ChangeThmbMode\r\n" +
		"RightHanded_Portrait_View.KEY_X      = ZoomIn\r\n" +
		"RightHanded_Portrait_View.KEY_Y      = ZoomOut\r\n" +
		"RightHanded_Portrait_View.KEY_R      = ChangeOrientation\r\n" +
		"RightHanded_Portrait_View.KEY_L      = ChangeMode\r\n" +
		"RightHanded_Portrait_View.KEY_Select = ChangeLight\r\n" +
		";RightHanded_Landscape_Thmb\r\n" +
		"RightHanded_Landscape_Thmb.KEY_Right  = RightPage\r\n" +
		"RightHanded_Landscape_Thmb.KEY_Down   = DownPage\r\n" +
		"RightHanded_Landscape_Thmb.KEY_Up     = UpPage\r\n" +
		"RightHanded_Landscape_Thmb.KEY_Left   = LeftPage\r\n" +
		"RightHanded_Landscape_Thmb.KEY_A      = ChangeViewMode\r\n" +
		"RightHanded_Landscape_Thmb.KEY_B      = ChangeThmbMode\r\n" +
		"RightHanded_Landscape_Thmb.KEY_X      = RightThmbPage\r\n" +
		"RightHanded_Landscape_Thmb.KEY_Y      = LeftThmbPage\r\n" +
		"RightHanded_Landscape_Thmb.KEY_R      = ChangeOrientation\r\n" +
		"RightHanded_Landscape_Thmb.KEY_L      = ChangeMode\r\n" +
		"RightHanded_Landscape_Thmb.KEY_Select = ChangeLight\r\n" +
		";RightHanded_Portrait_Thmb\r\n" +
		"RightHanded_Portrait_Thmb.KEY_Right  = UpPage\r\n" +
		"RightHanded_Portrait_Thmb.KEY_Down   = RightPage\r\n" +
		"RightHanded_Portrait_Thmb.KEY_Up     = LeftPage\r\n" +
		"RightHanded_Portrait_Thmb.KEY_Left   = DownPage\r\n" +
		"RightHanded_Portrait_Thmb.KEY_A      = ChangeViewMode\r\n" +
		"RightHanded_Portrait_Thmb.KEY_B      = ChangeThmbMode\r\n" +
		"RightHanded_Portrait_Thmb.KEY_X      = LeftThmbPage\r\n" +
		"RightHanded_Portrait_Thmb.KEY_Y      = RightThmbPage\r\n" +
		"RightHanded_Portrait_Thmb.KEY_R      = ChangeOrientation\r\n" +
		"RightHanded_Portrait_Thmb.KEY_L      = ChangeMode\r\n" +
		"RightHanded_Portrait_Thmb.KEY_Select = ChangeLight\r\n" +
		";LeftHanded_Landscape_View\r\n" +
		"LeftHanded_Landscape_View.KEY_Right  = ZoomOut\r\n" +
		"LeftHanded_Landscape_View.KEY_Down   = ChangeThmbMode\r\n" +
		"LeftHanded_Landscape_View.KEY_Up     = ZoomIn\r\n" +
		"LeftHanded_Landscape_View.KEY_Left   = ChangeViewMode\r\n" +
		"LeftHanded_Landscape_View.KEY_A      = ScrollRight\r\n" +
		"LeftHanded_Landscape_View.KEY_B      = ScrollDown\r\n" +
		"LeftHanded_Landscape_View.KEY_X      = ScrollUp\r\n" +
		"LeftHanded_Landscape_View.KEY_Y      = ScrollLeft\r\n" +
		"LeftHanded_Landscape_View.KEY_R      = ChangeMode\r\n" +
		"LeftHanded_Landscape_View.KEY_L      = ChangeOrientation\r\n" +
		"LeftHanded_Landscape_View.KEY_Select = ChangeLight\r\n" +
		";LeftHanded_Portrait_View\r\n" +
		"LeftHanded_Portrait_View.KEY_Right  = ZoomOut\r\n" +
		"LeftHanded_Portrait_View.KEY_Down   = ChangeThmbMode\r\n" +
		"LeftHanded_Portrait_View.KEY_Up     = ZoomIn\r\n" +
		"LeftHanded_Portrait_View.KEY_Left   = ChangeViewMode\r\n" +
		"LeftHanded_Portrait_View.KEY_A      = ScrollDown\r\n" +
		"LeftHanded_Portrait_View.KEY_B      = ScrollLeft\r\n" +
		"LeftHanded_Portrait_View.KEY_X      = ScrollRight\r\n" +
		"LeftHanded_Portrait_View.KEY_Y      = ScrollUp\r\n" +
		"LeftHanded_Portrait_View.KEY_R      = ChangeMode\r\n" +
		"LeftHanded_Portrait_View.KEY_L      = ChangeOrientation\r\n" +
		"LeftHanded_Portrait_View.KEY_Select = ChangeLight\r\n" +
		";LeftHanded_Landscape_Thmb\r\n" +
		"LeftHanded_Landscape_Thmb.KEY_Right  = RightThmbPage\r\n" +
		"LeftHanded_Landscape_Thmb.KEY_Down   = ChangeThmbMode\r\n" +
		"LeftHanded_Landscape_Thmb.KEY_Up     = LeftThmbPage\r\n" +
		"LeftHanded_Landscape_Thmb.KEY_Left   = ChangeViewMode\r\n" +
		"LeftHanded_Landscape_Thmb.KEY_A      = RightPage\r\n" +
		"LeftHanded_Landscape_Thmb.KEY_B      = DownPage\r\n" +
		"LeftHanded_Landscape_Thmb.KEY_X      = UpPage\r\n" +
		"LeftHanded_Landscape_Thmb.KEY_Y      = LeftPage\r\n" +
		"LeftHanded_Landscape_Thmb.KEY_R      = ChangeMode\r\n" +
		"LeftHanded_Landscape_Thmb.KEY_L      = ChangeOrientation\r\n" +
		"LeftHanded_Landscape_Thmb.KEY_Select = ChangeLight\r\n" +
		";LeftHanded_Portrait_Thmb\r\n" +
		"LeftHanded_Portrait_Thmb.KEY_Right  = LeftThmbPage\r\n" +
		"LeftHanded_Portrait_Thmb.KEY_Down   = ChangeThmbMode\r\n" +
		"LeftHanded_Portrait_Thmb.KEY_Up     = RightThmbPage\r\n" +
		"LeftHanded_Portrait_Thmb.KEY_Left   = ChangeViewMode\r\n" +
		"LeftHanded_Portrait_Thmb.KEY_A      = DownPage\r\n" +
		"LeftHanded_Portrait_Thmb.KEY_B      = LeftPage\r\n" +
		"LeftHanded_Portrait_Thmb.KEY_X      = RightPage\r\n" +
		"LeftHanded_Portrait_Thmb.KEY_Y      = UpPage\r\n" +
		"LeftHanded_Portrait_Thmb.KEY_R      = ChangeMode\r\n" +
		"LeftHanded_Portrait_Thmb.KEY_L      = ChangeOrientation\r\n" +
		"LeftHanded_Portrait_Thmb.KEY_Select = ChangeLight\r\n";
	
	
}

